import { getParents } from "../../dag/utils";
import { solve } from "../../simplex";
import { bigrams } from "../../utils";
/** @internal */
function buildOperator(options) {
    // TODO optimize this for disconnected graphs by breaking them apart, solving
    // each, then mushing them back together
    function optCall(layers) {
        // check for large input
        const numVars = layers.reduce((t, l) => t + (l.length * Math.max(l.length - 1, 0)) / 2, 0);
        const numEdges = layers.reduce((t, l) => t + l.reduce((s, n) => s + n.nchildren(), 0), 0);
        if (options.large !== "large" && numVars > 1200) {
            throw new Error(`size of dag to decrossOpt is too large and will likely crash instead of complete, enable "large" graphs to run anyway`);
        }
        else if (options.large !== "large" &&
            options.large !== "medium" &&
            (numVars > 400 || numEdges > 100)) {
            throw new Error(`size of dag to decrossOpt is too large and will likely not complete, enable "medium" graphs to run anyway`);
        }
        const distanceConstraints = [];
        for (const [topLayer, bottomLayer] of bigrams(layers)) {
            const withParents = new Set(topLayer.flatMap((node) => node.children()));
            const topUnconstrained = bottomLayer.filter((node) => !withParents.has(node));
            const topGroups = topLayer
                .map((node) => node.children())
                .filter((cs) => cs.length > 1);
            distanceConstraints.push([topUnconstrained, topGroups]);
            const bottomUnconstrained = topLayer.filter((n) => !n.nchildren());
            const parents = getParents(topLayer);
            const bottomGroups = [...parents.values()];
            distanceConstraints.push([bottomUnconstrained, bottomGroups]);
        }
        // NOTE distance cost for an unconstrained node ina group can't violate
        // all pairs at once, so cost is ~(n/2)^2 not n(n-1)/2
        const maxDistCost = distanceConstraints.reduce((cost, [unc, gs]) => gs.reduce((t, cs) => t + cs.length * cs.length, 0) * unc.length, 0) / 4;
        const distWeight = 1 / (maxDistCost + 1);
        // add small value to objective for preserving the original order of nodes
        const preserveWeight = distWeight / (numVars + 1);
        // initialize model
        const variables = {};
        const constraints = {};
        const ints = {};
        // map every node to an id for quick access, if one nodes id is less than
        // another it must come before it on the layer, or in a previous layer
        const inds = new Map();
        {
            let i = 0;
            for (const layer of layers) {
                for (const node of layer) {
                    inds.set(node, i++);
                }
            }
        }
        /** create a key from nodes */
        function key(...nodes) {
            return nodes
                .map((n) => inds.get(n))
                .sort((a, b) => a - b)
                .join(" => ");
        }
        function perms(layer) {
            // add variables for each pair of bottom later nodes indicating if they
            // should be flipped
            for (const [i, n1] of layer.slice(0, layer.length - 1).entries()) {
                for (const n2 of layer.slice(i + 1)) {
                    const pair = key(n1, n2);
                    ints[pair] = 1;
                    constraints[pair] = {
                        max: 1,
                    };
                    variables[pair] = {
                        // add small value to objective for preserving the original order of nodes
                        opt: -preserveWeight,
                        [pair]: 1,
                    };
                }
            }
            // add constraints to enforce triangle inequality, e.g. that if a -> b is 1
            // and b -> c is 1 then a -> c must also be one
            for (const [i, n1] of layer.slice(0, layer.length - 1).entries()) {
                for (const [j, n2] of layer.slice(i + 1).entries()) {
                    for (const n3 of layer.slice(i + j + 2)) {
                        const pair1 = key(n1, n2);
                        const pair2 = key(n1, n3);
                        const pair3 = key(n2, n3);
                        const triangle = key(n1, n2, n3);
                        const triangleUp = triangle + "+";
                        constraints[triangleUp] = {
                            max: 1,
                        };
                        variables[pair1][triangleUp] = 1;
                        variables[pair2][triangleUp] = -1;
                        variables[pair3][triangleUp] = 1;
                        const triangleDown = triangle + "-";
                        constraints[triangleDown] = {
                            min: 0,
                        };
                        variables[pair1][triangleDown] = 1;
                        variables[pair2][triangleDown] = -1;
                        variables[pair3][triangleDown] = 1;
                    }
                }
            }
        }
        function cross(layer) {
            for (const [i, p1] of layer.slice(0, layer.length - 1).entries()) {
                for (const p2 of layer.slice(i + 1)) {
                    const pairp = key(p1, p2);
                    for (const c1 of p1.ichildren()) {
                        for (const c2 of p2.ichildren()) {
                            if (c1 === c2) {
                                continue;
                            }
                            const pairc = key(c1, c2);
                            const slack = `slack (${pairp}) (${pairc})`;
                            const slackUp = `${slack} +`;
                            const slackDown = `${slack} -`;
                            variables[slack] = {
                                opt: 1,
                                [slackUp]: 1,
                                [slackDown]: 1,
                            };
                            const sign = Math.sign(inds.get(c1) - inds.get(c2));
                            const flip = Math.max(sign, 0);
                            constraints[slackUp] = {
                                min: flip,
                            };
                            variables[pairp][slackUp] = 1;
                            variables[pairc][slackUp] = sign;
                            constraints[slackDown] = {
                                min: -flip,
                            };
                            variables[pairp][slackDown] = -1;
                            variables[pairc][slackDown] = -sign;
                        }
                    }
                }
            }
        }
        function distance(unconstrained, groups) {
            for (const node of unconstrained) {
                for (const group of groups) {
                    for (const [i, start] of group.entries()) {
                        for (const end of group.slice(i + 1)) {
                            // want to minimize node being between start and end
                            // NOTE we don't sort because we care which is in the center
                            const base = [start, node, end]
                                .map((n) => inds.get(n))
                                .join(" => ");
                            const slack = `dist ${base}`;
                            const normal = `${slack} normal`;
                            const reversed = `${slack} reversed`;
                            variables[slack] = {
                                opt: distWeight,
                                [normal]: 1,
                                [reversed]: 1,
                            };
                            let pos = 0;
                            for (const [n1, n2] of [
                                [start, node],
                                [start, end],
                                [node, end],
                            ]) {
                                const pair = key(n1, n2);
                                const sign = Math.sign(inds.get(n1) - inds.get(n2));
                                pos += +(sign > 0);
                                variables[pair][normal] = -sign;
                                variables[pair][reversed] = sign;
                            }
                            constraints[normal] = {
                                min: 1 - pos,
                            };
                            constraints[reversed] = {
                                min: pos - 2,
                            };
                        }
                    }
                }
            }
        }
        // add variables and permutation invariants
        for (const layer of layers) {
            perms(layer);
        }
        // add crossing minimization
        for (const layer of layers.slice(0, layers.length - 1)) {
            cross(layer);
        }
        // add distance minimization
        if (options.dist) {
            for (const [unconstrained, groups] of distanceConstraints) {
                distance(unconstrained, groups);
            }
        }
        // solve objective
        // NOTE bundling sets this to undefined, and we need it to be settable
        const ordering = solve("opt", "min", variables, constraints, ints);
        // sort layers
        for (const layer of layers) {
            layer.sort((n1, n2) => ordering[key(n1, n2)] || -1);
        }
    }
    function large(val) {
        if (val === undefined) {
            return options.large;
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, options), { large: val }));
        }
    }
    optCall.large = large;
    function dist(val) {
        if (val === undefined) {
            return options.dist;
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, options), { dist: val }));
        }
    }
    optCall.dist = dist;
    return optCall;
}
/**
 * Create a default {@link OptOperator}, bundled as {@link decrossOpt}.
 */
export function opt(...args) {
    if (args.length) {
        throw new Error(`got arguments to opt(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ large: "small", dist: false });
}
