var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { entries, flatMap } from "../../iters";
import { solve } from "../../simplex";
import { bigrams } from "../../utils";
import { componentMap, splitComponentLayers } from "./utils";
/**
 * a function for creating a {@link ConstAccessor}
 */
export function createConstAccessor(value) {
    const [a, b, c] = value;
    if (a <= 0 || b <= 0 || c <= 0) {
        throw new Error("const accessors should return non-negative values");
    }
    const accessor = () => value;
    accessor.value = value;
    return accessor;
}
/**
 * If an accessor is a const accessor
 */
function isConstAccessor(acc) {
    return ("value" in acc &&
        typeof acc.value === "object" &&
        acc.value.length === 3 &&
        acc.value.every((v) => typeof v === "number"));
}
/**
 * create a cached weight accessor given the weight accessor and the layered dag
 *
 * This short circuits the constant operator, verifies the weights are
 * positive, and makes sure that we only call it once for each pair of nodes.
 */
function createCachedWeightAccessor(layers, weight) {
    if (isConstAccessor(weight)) {
        const [two, one, zero] = weight.value;
        if (two <= 0 || one <= 0 || zero <= 0) {
            throw new Error(`simplex weights must be positive, but got: ${two}, ${one}, ${zero}`);
        }
        const cached = (par, child) => {
            const count = +("node" in par.data) + +("node" in child.data);
            /* istanbul ignore next */
            switch (count) {
                case 0:
                    return zero;
                case 1:
                    return one;
                case 2:
                    return two;
                default:
                    /* istanbul ignore next */
                    throw new Error("internal error");
            }
        };
        return cached;
    }
    else {
        const cache = new Map();
        for (const node of flatMap(layers, (l) => l)) {
            if ("node" in node.data) {
                const rawNode = node.data.node;
                const targets = new Map();
                for (const link of rawNode.ichildLinks()) {
                    const { target } = link;
                    const vals = weight(link);
                    const [zero, one, two] = vals;
                    if (zero <= 0 || one <= 0 || two <= 0) {
                        throw new Error(`simplex weights must be positive, but got: ${zero}, ${one}, ${two}`);
                    }
                    targets.set(target, vals);
                }
                cache.set(rawNode, targets);
            }
        }
        const cached = (par, child) => {
            // NOTE this structure is to make sure type script does inference about
            // the sugi data appropriately
            if ("link" in par.data) {
                const { source, target } = par.data.link;
                const [, one, two] = cache.get(source).get(target);
                return "link" in child.data ? two : one;
            }
            else if ("link" in child.data) {
                const { source, target } = child.data.link;
                const [, val] = cache.get(source).get(target);
                return val;
            }
            else {
                const [val] = cache.get(par.data.node).get(child.data.node);
                return val;
            }
        };
        return cached;
    }
}
function buildOperator(opts) {
    function simplexComponent(layers, nodeSize) {
        var _a;
        const variables = {};
        const constraints = {};
        const cachedWeight = createCachedWeightAccessor(layers, opts.weight);
        // initialize ids and non-slack variables
        const ids = new Map();
        for (const [i, node] of entries(flatMap(layers, (l) => l))) {
            const id = i.toString();
            ids.set(node, id);
            variables[id] = {};
        }
        /** get node id */
        function n(node) {
            return ids.get(node);
        }
        // layer order constraints
        for (const layer of layers) {
            for (const [left, right] of bigrams(layer)) {
                const lid = n(left);
                const rid = n(right);
                const cons = `layer ${lid} -> ${rid}`;
                const width = (nodeSize(left) + nodeSize(right)) / 2;
                constraints[cons] = { min: width };
                variables[lid][cons] = -1;
                variables[rid][cons] = 1;
            }
        }
        // minimize weighted difference
        for (const node of flatMap(layers, (l) => l)) {
            const nid = n(node);
            for (const child of node.ichildren()) {
                const cid = n(child);
                const slack = `link ${nid} -> ${cid}`;
                const pcons = `${slack} parent`;
                constraints[pcons] = { min: 0 };
                const ccons = `${slack} child`;
                constraints[ccons] = { min: 0 };
                variables[nid][pcons] = 1;
                variables[nid][ccons] = -1;
                variables[cid][pcons] = -1;
                variables[cid][ccons] = 1;
                const weight = cachedWeight(node, child);
                variables[slack] = { opt: weight, [pcons]: 1, [ccons]: 1 };
            }
        }
        const assignment = solve("opt", "min", variables, constraints);
        let offset = 0;
        let width = 0;
        for (const layer of layers) {
            for (const node of layer) {
                node.x = (_a = assignment[n(node)]) !== null && _a !== void 0 ? _a : 0;
            }
            const first = layer[0];
            offset = Math.min(offset, first.x - nodeSize(first) / 2);
            const last = layer[layer.length - 1];
            width = Math.max(width, last.x + nodeSize(last) / 2);
        }
        for (const node of flatMap(layers, (l) => l)) {
            node.x -= offset;
        }
        return width - offset;
    }
    function simplexCall(layers, nodeSize) {
        // split components
        const compMap = componentMap(layers);
        const components = splitComponentLayers(layers, compMap);
        // layout each component and get width
        const widths = components.map((compon) => simplexComponent(compon, nodeSize));
        // center components
        const maxWidth = Math.max(...widths);
        if (maxWidth <= 0) {
            throw new Error("must assign nonzero width to at least one node");
        }
        for (const [i, compon] of components.entries()) {
            const offset = (maxWidth - widths[i]) / 2;
            for (const layer of compon) {
                for (const node of layer) {
                    node.x += offset;
                }
            }
        }
        return maxWidth;
    }
    function weight(val) {
        if (val === undefined) {
            return opts.weight;
        }
        else {
            const { weight: _ } = opts, rest = __rest(opts, ["weight"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { weight: val }));
        }
    }
    simplexCall.weight = weight;
    return simplexCall;
}
/**
 * Create a default {@link SimplexOperator}, bundled as {@link coordSimplex}.
 */
export function simplex(...args) {
    if (args.length) {
        throw new Error(`got arguments to simplex(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({
        weight: createConstAccessor([1, 2, 8]),
    });
}
