import FastPriorityQueue from "fastpriorityqueue";
import { map } from "../iters";
import { listMultimapPush } from "../utils";
/**
 * get a mapping from a the children of a set of nodes to their unique parents
 */
export function getParents(parentNodes) {
    const parents = new Map();
    for (const par of parentNodes) {
        for (const child of par.ichildren()) {
            listMultimapPush(parents, child, par);
        }
    }
    return parents;
}
/**
 * get a mapping from a the children of a set of nodes to their parents with counts
 */
export function getParentCounts(parentNodes) {
    const parents = new Map();
    for (const par of parentNodes) {
        for (const [child, count] of par.ichildrenCounts()) {
            listMultimapPush(parents, child, [par, count]);
        }
    }
    return parents;
}
/** convert a dag into a dot string */
export function dot(dag, id) {
    const links = [
        ...map(dag.ilinks(), ({ source, target }) => `    "${id(source)}" -> "${id(target)}"`),
    ];
    return `digraph {\n${links.join("\n")}\n}`;
}
/**
 * iterate over nodes in a dag
 *
 * Parents will be returned before children. Nodes without priority will be
 * returned first, then nodes with the lowest priority.
 */
export function* before(dag, priority) {
    var _a, _b;
    const numBefore = new Map();
    for (const node of dag) {
        for (const child of node.ichildren()) {
            numBefore.set(child, (numBefore.get(child) || 0) + 1);
        }
    }
    const prioritized = new FastPriorityQueue(([, a], [, b]) => a < b);
    const unprioritized = [];
    function push(node) {
        const pri = priority(node);
        if (pri === undefined) {
            unprioritized.push(node);
        }
        else {
            prioritized.add([node, pri]);
        }
    }
    for (const node of dag.iroots()) {
        push(node);
    }
    let node;
    while ((node = (_a = unprioritized.pop()) !== null && _a !== void 0 ? _a : (_b = prioritized.poll()) === null || _b === void 0 ? void 0 : _b[0])) {
        yield node;
        for (const child of node.ichildren()) {
            const before = numBefore.get(child);
            if (before > 1) {
                numBefore.set(child, before - 1);
            }
            else {
                push(child);
            }
        }
    }
}
