import { jsxs, jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { forwardRef } from 'react';
import { isIconName } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { useStyles2 } from '../../themes/ThemeContext.js';
import 'micro-memoize';
import '@emotion/react';
import { getFocusStyles, getMouseFocusStyles, mediaUp } from '../../themes/mixins.js';
import '../../utils/skeleton.js';
import { getPropertiesForVariant } from '../Button/Button.js';
import '../Button/ButtonGroup.js';
import { Icon } from '../Icon/Icon.js';
import { Tooltip } from '../Tooltip/Tooltip.js';

const ToolbarButton = forwardRef(
  ({
    tooltip,
    icon,
    iconSize,
    className,
    children,
    imgSrc,
    imgAlt,
    fullWidth,
    isOpen,
    narrow,
    variant = "default",
    iconOnly,
    "aria-label": ariaLabel,
    isHighlighted,
    ...rest
  }, ref) => {
    const styles = useStyles2(getStyles);
    const buttonStyles = cx(
      {
        [styles.button]: true,
        [styles.buttonFullWidth]: fullWidth,
        [styles.narrow]: narrow
      },
      styles[variant],
      className
    );
    const contentStyles = cx({
      [styles.content]: true,
      [styles.contentWithIcon]: !!icon,
      [styles.contentWithRightIcon]: isOpen !== void 0
    });
    const body = /* @__PURE__ */ jsxs(
      "button",
      {
        ref,
        className: buttonStyles,
        "aria-label": getButtonAriaLabel(ariaLabel, tooltip),
        "aria-expanded": isOpen,
        ...rest,
        children: [
          renderIcon(icon, iconSize),
          imgSrc && /* @__PURE__ */ jsx("img", { className: styles.img, src: imgSrc, alt: imgAlt != null ? imgAlt : "" }),
          children && !iconOnly && /* @__PURE__ */ jsx("div", { className: contentStyles, children }),
          isOpen === false && /* @__PURE__ */ jsx(Icon, { name: "angle-down" }),
          isOpen === true && /* @__PURE__ */ jsx(Icon, { name: "angle-up" }),
          isHighlighted && /* @__PURE__ */ jsx("div", { className: styles.highlight })
        ]
      }
    );
    return tooltip ? /* @__PURE__ */ jsx(Tooltip, { ref, content: tooltip, placement: "bottom", children: body }) : body;
  }
);
ToolbarButton.displayName = "ToolbarButton";
function getButtonAriaLabel(ariaLabel, tooltip) {
  return ariaLabel ? ariaLabel : tooltip ? selectors.components.PageToolbar.item(tooltip) : void 0;
}
function renderIcon(icon, iconSize) {
  if (!icon) {
    return null;
  }
  if (isIconName(icon)) {
    return /* @__PURE__ */ jsx(Icon, { name: icon, size: `${iconSize ? iconSize : "lg"}` });
  }
  return icon;
}
const getStyles = (theme) => {
  const primaryVariant = getPropertiesForVariant(theme, "primary", "solid");
  const destructiveVariant = getPropertiesForVariant(theme, "destructive", "solid");
  const defaultOld = css({
    color: theme.colors.text.primary,
    background: theme.colors.secondary.main,
    "&:hover": {
      color: theme.colors.text.primary,
      background: theme.colors.secondary.shade,
      border: `1px solid ${theme.colors.border.medium}`
    }
  });
  return {
    button: css({
      label: "toolbar-button",
      position: "relative",
      display: "flex",
      alignItems: "center",
      height: theme.spacing(theme.components.height.md),
      padding: theme.spacing(0, 1),
      borderRadius: theme.shape.radius.default,
      lineHeight: `${theme.components.height.md * theme.spacing.gridSize - 2}px`,
      fontWeight: theme.typography.fontWeightMedium,
      border: `1px solid ${theme.colors.secondary.border}`,
      whiteSpace: "nowrap",
      [theme.transitions.handleMotion("no-preference", "reduce")]: {
        transition: theme.transitions.create(["background", "box-shadow", "border-color", "color"], {
          duration: theme.transitions.duration.short
        })
      },
      [theme.breakpoints.down("md")]: {
        width: "auto !important"
      },
      "&:focus, &:focus-visible": {
        ...getFocusStyles(theme),
        zIndex: 1
      },
      "&:focus:not(:focus-visible)": getMouseFocusStyles(),
      "&[disabled], &:disabled": {
        cursor: "not-allowed",
        opacity: theme.colors.action.disabledOpacity,
        background: theme.colors.action.disabledBackground,
        boxShadow: "none",
        "&:hover": {
          color: theme.colors.text.disabled,
          background: theme.colors.action.disabledBackground,
          boxShadow: "none"
        }
      }
    }),
    default: css({
      color: theme.colors.text.secondary,
      background: "transparent",
      border: `1px solid transparent`,
      "&:hover": {
        color: theme.colors.text.primary,
        background: theme.colors.action.hover
      }
    }),
    canvas: defaultOld,
    active: cx(
      defaultOld,
      css({
        "&::before": {
          display: "block",
          content: '" "',
          position: "absolute",
          left: 0,
          right: 0,
          height: "2px",
          bottom: 0,
          borderRadius: theme.shape.radius.default,
          backgroundImage: theme.colors.gradients.brandHorizontal
        }
      })
    ),
    primary: css(primaryVariant),
    destructive: css(destructiveVariant),
    narrow: css({
      padding: theme.spacing(0, 0.5)
    }),
    img: css({
      width: "16px",
      height: "16px",
      marginRight: theme.spacing(1)
    }),
    buttonFullWidth: css({
      flexGrow: 1
    }),
    content: css({
      display: "flex",
      flexGrow: 1
    }),
    contentWithIcon: css({
      display: "none",
      paddingLeft: theme.spacing(1),
      [`@media ${mediaUp(theme.v1.breakpoints.md)}`]: {
        display: "block"
      }
    }),
    contentWithRightIcon: css({
      paddingRight: theme.spacing(0.5)
    }),
    highlight: css({
      backgroundColor: theme.colors.success.main,
      borderRadius: theme.shape.radius.circle,
      width: "6px",
      height: "6px",
      position: "absolute",
      top: "-3px",
      right: "-3px",
      zIndex: 1
    })
  };
};

export { ToolbarButton };
//# sourceMappingURL=ToolbarButton.js.map
