"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var initialize_1 = require("../../initialize");
var testUtils_1 = require("../../testUtils");
describe('api.events', function () {
    function createAPI(_a) {
        var _b = _a === void 0 ? { dedupe: true } : _a, dedupe = _b.dedupe;
        var transport = new testUtils_1.MockTransport();
        var config = (0, testUtils_1.mockConfig)({
            dedupe: dedupe,
            transports: [transport],
        });
        var api = (0, initialize_1.initializeFaro)(config).api;
        return [api, transport];
    }
    describe('pushEvent', function () {
        var api;
        var transport;
        beforeEach(function () {
            var _a;
            _a = createAPI(), api = _a[0], transport = _a[1];
        });
        describe('Filtering', function () {
            it('filters the same event', function () {
                api.pushEvent('test', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
            });
            it("doesn't filter events with same name and partially same values", function () {
                api.pushEvent('test', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test', {
                    a: '1',
                    b: '2',
                });
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter events with different name and same values", function () {
                api.pushEvent('test1', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test2', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(2);
            });
            it("filters an event and doesn't filter the next different one", function () {
                api.pushEvent('test1', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test1', {
                    a: '1',
                });
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test2', {
                    b: '1',
                });
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter when dedupe is false", function () {
                var _a;
                _a = createAPI({ dedupe: false }), api = _a[0], transport = _a[1];
                api.pushEvent('test');
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test');
                expect(transport.items).toHaveLength(2);
            });
            it("doesn't filter when skipDedupe is true", function () {
                api.pushEvent('test');
                expect(transport.items).toHaveLength(1);
                api.pushEvent('test', {}, undefined, { skipDedupe: true });
                expect(transport.items).toHaveLength(2);
            });
            it('uses traceId and spanId from custom context', function () {
                var _a;
                var spanContext = {
                    traceId: 'my-trace-id',
                    spanId: 'my-span-id',
                };
                var mockGetTraceContext = jest.fn();
                jest.spyOn(api, 'getTraceContext').mockImplementationOnce(mockGetTraceContext);
                api.pushEvent('test', undefined, undefined, { spanContext: spanContext });
                expect(mockGetTraceContext).not.toHaveBeenCalled();
                expect(((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload).trace).toStrictEqual({
                    trace_id: 'my-trace-id',
                    span_id: 'my-span-id',
                });
            });
            it('Sets the timestamp to the provided custom timestamp', function () {
                var _a;
                api.pushEvent('test', undefined, undefined, { timestampOverwriteMs: 123 });
                expect(transport.items).toHaveLength(1);
                expect(((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload).timestamp).toBe('1970-01-01T00:00:00.123Z');
            });
            it('stringifies all values in the attributes object', function () {
                var _a;
                api.pushEvent('test', {
                    // @ts-expect-error
                    a: 1,
                    b: 'foo',
                    // @ts-expect-error
                    c: true,
                    // @ts-expect-error
                    d: { e: 'bar' },
                    // @ts-expect-error
                    g: null,
                    // @ts-expect-error
                    h: undefined,
                    // @ts-expect-error
                    i: [1, 2, 3],
                });
                // @ts-expect-error
                expect((_a = transport.items[0]) === null || _a === void 0 ? void 0 : _a.payload.attributes).toStrictEqual({
                    a: '1',
                    b: 'foo',
                    c: 'true',
                    d: '{"e":"bar"}',
                    g: 'null',
                    h: 'undefined',
                    i: '[1,2,3]',
                });
            });
        });
    });
});
//# sourceMappingURL=initialize.test.js.map