#!/usr/bin/env python3
#
# The purpose of this file is to produce rst output interspersed into
# the the text of the example code

import sys
import re
import tempfile
import shutil

import apsw.ext

from typing import Any, TextIO

# start of each section in example-code.py
section_re = r"""###\s+(?P<section>\b\w+\b):\s+(?P<desc>.*)\s*"""

# magic string from output to recognise new section
section_marker = "!@#$%^&*()(-:"

# table output
table_re = r"\s*print\(apsw\.ext\.format_query_table"

# magic string to recognise table
table_marker = "@!#$%^&*()(-:"

# start of rst output
header = """\
.. Automatically generated by example2rst.py.  Do not edit this file

.. currentmodule:: apsw

Example/Tour
============

This code demonstrates usage of APSW.  It gives you a good overview of
all the things that can be done.  Also included is output so you can
see what gets printed when you run the code.

There are also specific examples in the classes, functions,
and attribute documentation.

.. code-block:: python
"""

def get_output(filename: str):
    code: list[str] = []
    for num, line in enumerate(open(filename, "rt")):
        mo = re.match(section_re, line)
        if mo:
            code.append(f"print('{ section_marker }{ mo.group('section') }')")
            continue
        mo = re.match(table_re, line)
        if mo:
            code.append(f"print('{ table_marker }<{ num }\\n', apsw.ext.format_query_table"+line[len(mo[0]):])
        else:
            code.append(line.rstrip())

    output: dict[str, list[str]] = {}
    tables: dict[int, list[str]] = {}
    cur_section = None

    def my_print(*args: Any) -> None:
        nonlocal output, cur_section, tables
        s = " ".join(str(a) for a in args)
        if s.startswith(section_marker):
            cur_section = s[len(section_marker):]
            assert cur_section not in output
            output[cur_section] = []
            print("SECTION", cur_section)
            return

        if s.startswith(table_marker):
            in_table = int(s.split("<")[1].split("\n")[0])
            lines = s.split("\n")[1:]
            lines[0] = lines[0][1:] # remove leading space
            tables[in_table]= lines
            s = "\n".join(lines)
        else:
            assert cur_section is not None
            output[cur_section].extend(s.split("\n"))

        print(s)

    exec(compile("\n".join(code), "example-code.py", "exec"), {"print": my_print})

    return output, tables


def gen_rst(filename: str, outfile: TextIO, output: dict[str, list[str]], tables) -> None:
    print(header, file=outfile)
    cur_section = None
    seen_blank = True

    with open(filename, "rt") as source:
        for num, line in enumerate(source):
            mo = re.match(section_re, line)
            if mo:
                if cur_section and output[cur_section]:
                    print("\n\n.. code-block:: output\n", file=outfile)
                    for l in output[cur_section]:
                        print("    " + l, file=outfile)
                    print("", file=outfile)
                cur_section = mo.group("section")
                print(f"\n.. index:: { mo.group('desc') } (example code)\n", file=outfile)
                print(f".. _example_{ cur_section }:\n", file=outfile)
                print(mo.group("desc"), file=outfile)
                print("-" * len(mo.group("desc")), file=outfile)
                print("", file=outfile)
                seen_blank = False
                continue

            if not seen_blank:
                if not line.strip():
                    seen_blank = True
                    print("\n.. code-block:: python\n", file=outfile)
                else:
                    l = line.rstrip()
                    l = l.lstrip("#")[1:]
                    print(l, file=outfile)
                continue
            print("   " + line.rstrip(), file=outfile)

            if num in tables:
                print("\n\n.. code-block:: output\n", file=outfile)
                for l in tables[num]:
                    print("    " + l, file=outfile)
                print("\n\n.. code-block:: python\n", file=outfile)


    f.flush()


if __name__ == "__main__":
    try:
        output, tables = get_output("example-code.py")
    except:
        apsw.ext.print_augmented_traceback(*sys.exc_info())
        raise
    with tempfile.NamedTemporaryFile("wt", prefix="example2rst") as f:
        gen_rst("example-code.py", f, output, tables)
        shutil.copy(f.name, "doc/example.rst")
