"use strict";
/**
 * npm_install_proxy -- localhost NPM registry to `npm install` without network
 *
 * Copyright (C) 2020  SUSE LLC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.Registry = void 0;
const url_1 = require("url");
const utils_1 = require("./utils");
;
class Registry {
    constructor() {
        this.pkgs = [];
        this.backends = [];
        this.requestHandler = { url: new url_1.URL('file://') };
    }
    addBackend(backend) {
        this.backends.push(backend);
    }
    set serviceProvider(service) {
        this.requestHandler = service;
    }
    fetchPackages() {
        let package_summary = [];
        for (let i = 0; i < this.pkgs.length; i = i + 1) {
            let p = this.pkgs[i];
            let current_pkg = package_summary.filter((f => f.name === p.name));
            if (current_pkg.length === 0) {
                current_pkg = [{ 'name': p.name, 'versions': {} }];
                current_pkg[0].versions[p.version] = this.requestHandler.url + [p.name, p.version].join('/');
                package_summary.push(current_pkg[0]);
            }
            else {
                let versions = current_pkg[0].versions;
                versions[p.version] = this.requestHandler.url + [p.name, p.version].join('/');
            }
        }
        return package_summary;
    }
    fetchVersions(pkg_name) {
        let obj = {
            name: pkg_name,
            versions: {}
        };
        for (let i = 0; i < this.pkgs.length; i = i + 1) {
            if (this.pkgs[i].name === pkg_name) {
                const pkg = JSON.parse(JSON.stringify(this.pkgs[i]));
                pkg['dist']['tarball'] = this.requestHandler.url + '-/' + (0, utils_1.baseTarballName)(pkg.dist.tarball);
                obj.versions[pkg.version] = pkg;
            }
        }
        if (Object.keys(obj.versions).length === 0)
            throw new Error("not found");
        return obj;
    }
    fetchPkgVersion(pkg_name, version) {
        const obj = this.fetchVersions(pkg_name);
        const versions = Object.keys(obj['versions']).sort();
        if (!versions.includes(version)) {
            const non_standard_version = version.match(/[^\d\.]/);
            if (non_standard_version == null)
                throw new Error("not found");
            const new_version = versions[versions.length - 1];
            console.warn("pkg: %s is asking for non-standard version '%s'. Fallback to version %s", pkg_name, version, new_version);
            return obj['versions'][new_version];
        }
        return obj['versions'][version];
    }
    static verifyPkgJsonType(pkg_json) {
        if (!(pkg_json instanceof Object))
            return false;
        const keys = Object.keys(pkg_json);
        return keys.includes('name') &&
            keys.includes('version') &&
            keys.includes('dist');
    }
    commitPkgJson(pkg_json) {
        if (!Registry.verifyPkgJsonType(pkg_json))
            return 0;
        this.pkgs.push(pkg_json);
        return 1;
    }
    processPathWithBackend(backend, path) {
        return backend.extractPkgJson(path).then(jsons => {
            let processed_packages = 0;
            jsons.forEach(pkg_json => processed_packages += this.commitPkgJson(pkg_json));
            return processed_packages;
        }).catch(err => 0);
    }
    register(path) {
        let backend_processors = Promise.resolve(0);
        for (let i = 0; i < this.backends.length; ++i) {
            backend_processors = backend_processors.then(imported => {
                if (imported > 0)
                    return imported;
                return this.processPathWithBackend(this.backends[i], path);
            });
        }
        if (this.backends.length === 0)
            console.log("No backends");
        return backend_processors;
    }
    archiveFile(file) {
        const files = this.pkgs.filter(json => ((0, utils_1.baseTarballName)(json.dist.tarball) === file));
        if (files.length !== 1)
            return null;
        return files[0].dist.tarball.substring(2);
    }
}
exports.Registry = Registry;
;
//# sourceMappingURL=registry.js.map