/*
 * livepatch_bsc1234892
 *
 * Fix for CVE-2024-53173, bsc#1234892
 *
 *  Upstream commit:
 *  2fdb05dc0931 ("NFSv4.0: Fix a use-after-free problem in the asynchronous open()")
 *
 *  SLE12-SP5 commit:
 *  a7e3c22f0677df583d33bdd5c9344a08250837df
 *
 *  SLE15-SP3 commit:
 *  a94e553d68dc263090e96d42b9012db1826d767c
 *
 *  SLE15-SP4 and -SP5 commit:
 *  f801b5bdd060f9e5f2f1f51c05fa5a4fea53b773
 *
 *  SLE15-SP6 commit:
 *  9d06142475ccaf4094ca8fb270be42048efec952
 *
 *  SLE MICRO-6-0 commit:
 *  9d06142475ccaf4094ca8fb270be42048efec952
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Marco Crivellari <marco.crivellari@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from fs/nfs/nfs4proc.c */
#include <linux/mm.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/ratelimit.h>
#include <linux/printk.h>
#include <linux/slab.h>
#include <linux/sunrpc/clnt.h>
#include <linux/nfs.h>
#include <linux/nfs4.h>
#include <linux/nfs_fs.h>

/* klp-ccp: from include/linux/nfs_page.h */
#define _LINUX_NFS_PAGE_H

/* klp-ccp: from include/uapi/linux/nfs_mount.h */
#define _LINUX_NFS_MOUNT_H

/* klp-ccp: from fs/nfs/nfs4proc.c */
#include <linux/mount.h>
#include <linux/module.h>

/* klp-ccp: from include/linux/xattr.h */
#define _LINUX_XATTR_H

/* klp-ccp: from fs/nfs/nfs4proc.c */
#include <linux/utsname.h>

/* klp-ccp: from include/linux/iversion.h */
#define _LINUX_IVERSION_H

/* klp-ccp: from fs/nfs/nfs4_fs.h */
#if IS_ENABLED(CONFIG_NFS_V4)

#include <linux/seqlock.h>

struct nfs4_opendata {
	struct kref kref;
	struct nfs_openargs o_arg;
	struct nfs_openres o_res;
	struct nfs_open_confirmargs c_arg;
	struct nfs_open_confirmres c_res;
	struct nfs4_string owner_name;
	struct nfs4_string group_name;
	struct nfs4_label *a_label;
	struct nfs_fattr f_attr;
	struct dentry *dir;
	struct dentry *dentry;
	struct nfs4_state_owner *owner;
	struct nfs4_state *state;
	struct iattr attrs;
	struct nfs4_layoutget *lgp;
	unsigned long timestamp;
	bool rpc_done;
	bool file_created;
	bool is_recover;
	bool cancelled;
	int rpc_status;
};

extern void nfs4_put_state_owner(struct nfs4_state_owner *);

extern void nfs4_put_open_state(struct nfs4_state *);
extern void nfs4_close_state(struct nfs4_state *, fmode_t);

extern void nfs_release_seqid(struct nfs_seqid *seqid);
extern void nfs_free_seqid(struct nfs_seqid *seqid);

#else /* CONFIG_NFS_V4 */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_NFS_V4 */

/* klp-ccp: from fs/nfs/internal.h */
#include <linux/security.h>
#include <linux/crc32.h>

#include <linux/nfs_page.h>
#include <linux/wait_bit.h>

extern void nfs_sb_deactive(struct super_block *sb);

/* klp-ccp: from fs/nfs/iostat.h */
#include <linux/percpu.h>
#include <linux/cache.h>

/* klp-ccp: from fs/nfs/pnfs.h */
#include <linux/refcount.h>
#include <linux/nfs_fs.h>
#include <linux/nfs_page.h>
#include <linux/workqueue.h>

#ifdef CONFIG_NFS_V4_1

void nfs4_lgopen_release(struct nfs4_layoutget *lgp);

#else  /* CONFIG_NFS_V4_1 */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_NFS_V4_1 */

/* klp-ccp: from fs/nfs/netns.h */
#include <linux/nfs4.h>
#include <net/net_namespace.h>

#include <linux/sunrpc/stats.h>

/* klp-ccp: from fs/nfs/nfs4idmap.h */
#include <linux/uidgid.h>

void nfs_fattr_free_names(struct nfs_fattr *);

/* klp-ccp: from fs/nfs/nfs4session.h */
#define NFS4_MAX_SLOT_TABLE (1024U)

#if IS_ENABLED(CONFIG_NFS_V4)

struct nfs4_slot {
	struct nfs4_slot_table	*table;
	struct nfs4_slot	*next;
	unsigned long		generation;
	u32			slot_nr;
	u32		 	seq_nr;
	u32		 	seq_nr_last_acked;
	u32		 	seq_nr_highest_sent;
	unsigned int		privileged : 1,
				seq_done : 1;
};

#define SLOT_TABLE_SZ DIV_ROUND_UP(NFS4_MAX_SLOT_TABLE, BITS_PER_LONG)
struct nfs4_slot_table {
	struct nfs4_session *session;		/* Parent session */
	struct nfs4_slot *slots;		/* seqid per slot */
	unsigned long   used_slots[SLOT_TABLE_SZ]; /* used/unused bitmap */
	spinlock_t	slot_tbl_lock;
	struct rpc_wait_queue	slot_tbl_waitq;	/* allocators may wait here */
	wait_queue_head_t	slot_waitq;	/* Completion wait on slot */
	u32		max_slots;		/* # slots in table */
	u32		max_slotid;		/* Max allowed slotid value */
	u32		highest_used_slotid;	/* sent to server on each SEQ.
						 * op for dynamic resizing */
	u32		target_highest_slotid;	/* Server max_slot target */
	u32		server_highest_slotid;	/* Server highest slotid */
	s32		d_target_highest_slotid; /* Derivative */
	s32		d2_target_highest_slotid; /* 2nd derivative */
	unsigned long	generation;		/* Generation counter for
						   target_highest_slotid */
	struct completion complete;
	unsigned long	slot_tbl_state;
};

extern void nfs4_free_slot(struct nfs4_slot_table *tbl, struct nfs4_slot *slot);

bool nfs41_wake_and_assign_slot(struct nfs4_slot_table *tbl,
		struct nfs4_slot *slot);

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif /* IS_ENABLED(CONFIG_NFS_V4) */

/* klp-ccp: from fs/nfs/fscache.h */
#include <linux/nfs_fs.h>
#include <linux/nfs_mount.h>

#include <linux/iversion.h>

/* klp-ccp: from fs/nfs/nfs42.h */
#include <linux/xattr.h>


/* klp-ccp: from fs/nfs/nfs4proc.c */
static void nfs40_sequence_free_slot(struct nfs4_sequence_res *res)
{
	struct nfs4_slot *slot = res->sr_slot;
	struct nfs4_slot_table *tbl;

	tbl = slot->table;
	spin_lock(&tbl->slot_tbl_lock);
	if (!nfs41_wake_and_assign_slot(tbl, slot))
		nfs4_free_slot(tbl, slot);
	spin_unlock(&tbl->slot_tbl_lock);

	res->sr_slot = NULL;
}

#if defined(CONFIG_NFS_V4_1)

extern void nfs41_release_slot(struct nfs4_slot *slot);

static void nfs41_sequence_free_slot(struct nfs4_sequence_res *res)
{
	nfs41_release_slot(res->sr_slot);
	res->sr_slot = NULL;
}

static void nfs4_sequence_free_slot(struct nfs4_sequence_res *res)
{
	if (res->sr_slot != NULL) {
		if (res->sr_slot->table->session != NULL)
			nfs41_sequence_free_slot(res);
		else
			nfs40_sequence_free_slot(res);
	}
}

#else	/* !CONFIG_NFS_V4_1 */
#error "klp-ccp: non-taken branch"
#endif	/* !CONFIG_NFS_V4_1 */

static void nfs4_opendata_free(struct kref *kref)
{
	struct nfs4_opendata *p = container_of(kref,
			struct nfs4_opendata, kref);
	struct super_block *sb = p->dentry->d_sb;

	nfs4_lgopen_release(p->lgp);
	nfs_free_seqid(p->o_arg.seqid);
	nfs4_sequence_free_slot(&p->o_res.seq_res);
	if (p->state != NULL)
		nfs4_put_open_state(p->state);
	nfs4_put_state_owner(p->owner);

	nfs4_label_free(p->a_label);
	nfs4_label_free(p->f_attr.label);

	dput(p->dir);
	dput(p->dentry);
	nfs_sb_deactive(sb);
	nfs_fattr_free_names(&p->f_attr);
	kfree(p->f_attr.mdsthreshold);
	kfree(p);
}

static void nfs4_opendata_put(struct nfs4_opendata *p)
{
	if (p != NULL)
		kref_put(&p->kref, nfs4_opendata_free);
}

extern struct nfs4_state *
nfs4_opendata_to_nfs4_state(struct nfs4_opendata *data);

void klpp_nfs4_open_release(void *calldata)
{
	struct nfs4_opendata *data = calldata;
	struct nfs4_state *state = NULL;

	/* In case of error, no cleanup! */
	if (data->rpc_status != 0 || !data->rpc_done) {
		nfs_release_seqid(data->o_arg.seqid);
		goto out_free;
	}
	/* If this request hasn't been cancelled, do nothing */
	if (!data->cancelled)
		goto out_free;
	/* In case we need an open_confirm, no cleanup! */
	if (data->o_res.rflags & NFS4_OPEN_RESULT_CONFIRM)
		goto out_free;
	state = nfs4_opendata_to_nfs4_state(data);
	if (!IS_ERR(state))
		nfs4_close_state(state, data->o_arg.fmode);
out_free:
	nfs4_opendata_put(data);
}


#include "livepatch_bsc1234892.h"

#include <linux/livepatch.h>

extern typeof(nfs41_release_slot) nfs41_release_slot
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs41_release_slot);
extern typeof(nfs41_wake_and_assign_slot) nfs41_wake_and_assign_slot
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs41_wake_and_assign_slot);
extern typeof(nfs4_close_state) nfs4_close_state
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_close_state);
extern typeof(nfs4_free_slot) nfs4_free_slot
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_free_slot);
extern typeof(nfs4_lgopen_release) nfs4_lgopen_release
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_lgopen_release);
extern typeof(nfs4_opendata_to_nfs4_state) nfs4_opendata_to_nfs4_state
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_opendata_to_nfs4_state);
extern typeof(nfs4_put_open_state) nfs4_put_open_state
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_put_open_state);
extern typeof(nfs4_put_state_owner) nfs4_put_state_owner
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs4_put_state_owner);
extern typeof(nfs_fattr_free_names) nfs_fattr_free_names
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs_fattr_free_names);
extern typeof(nfs_free_seqid) nfs_free_seqid
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs_free_seqid);
extern typeof(nfs_release_seqid) nfs_release_seqid
	 KLP_RELOC_SYMBOL(nfsv4, nfsv4, nfs_release_seqid);
extern typeof(nfs_sb_deactive) nfs_sb_deactive
	 KLP_RELOC_SYMBOL(nfsv4, nfs, nfs_sb_deactive);
