//
// Copyright (c) 2017 Joey <majunjiev@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package ovirtsdk

import (
	"fmt"
	"time"
)

// AffinityGroup An affinity group represents a group of virtual machines with a defined relationship.
type AffinityGroup struct {
	Struct
	cluster     *Cluster
	comment     *string
	description *string
	enforcing   *bool
	hostLabels  *AffinityLabelSlice
	hosts       *HostSlice
	hostsRule   *AffinityRule
	id          *string
	name        *string
	positive    *bool
	priority    *float64
	vmLabels    *AffinityLabelSlice
	vms         *VmSlice
	vmsRule     *AffinityRule
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *AffinityGroup) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *AffinityGroup) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *AffinityGroup) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *AffinityGroup) SetComment(attr string) {
	p.comment = &attr
}

func (p *AffinityGroup) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *AffinityGroup) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *AffinityGroup) SetDescription(attr string) {
	p.description = &attr
}

func (p *AffinityGroup) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *AffinityGroup) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *AffinityGroup) SetEnforcing(attr bool) {
	p.enforcing = &attr
}

func (p *AffinityGroup) Enforcing() (bool, bool) {
	if p.enforcing != nil {
		return *p.enforcing, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityGroup) MustEnforcing() bool {
	if p.enforcing == nil {
		panic("the enforcing must not be nil, please use Enforcing() function instead")
	}
	return *p.enforcing
}

func (p *AffinityGroup) SetHostLabels(attr *AffinityLabelSlice) {
	p.hostLabels = attr
}

func (p *AffinityGroup) HostLabels() (*AffinityLabelSlice, bool) {
	if p.hostLabels != nil {
		return p.hostLabels, true
	}
	return nil, false
}

func (p *AffinityGroup) MustHostLabels() *AffinityLabelSlice {
	if p.hostLabels == nil {
		panic("the hostLabels must not be nil, please use HostLabels() function instead")
	}
	return p.hostLabels
}

func (p *AffinityGroup) SetHosts(attr *HostSlice) {
	p.hosts = attr
}

func (p *AffinityGroup) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *AffinityGroup) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("the hosts must not be nil, please use Hosts() function instead")
	}
	return p.hosts
}

// SetHostsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) SetHostsRule(attr *AffinityRule) {
	p.hostsRule = attr
}

// HostsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) HostsRule() (*AffinityRule, bool) {
	if p.hostsRule != nil {
		return p.hostsRule, true
	}
	return nil, false
}

// MustHostsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) MustHostsRule() *AffinityRule {
	if p.hostsRule == nil {
		panic("the hostsRule must not be nil, please use HostsRule() function instead")
	}
	return p.hostsRule
}

func (p *AffinityGroup) SetId(attr string) {
	p.id = &attr
}

func (p *AffinityGroup) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *AffinityGroup) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *AffinityGroup) SetName(attr string) {
	p.name = &attr
}

func (p *AffinityGroup) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *AffinityGroup) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *AffinityGroup) SetPositive(attr bool) {
	p.positive = &attr
}

func (p *AffinityGroup) Positive() (bool, bool) {
	if p.positive != nil {
		return *p.positive, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityGroup) MustPositive() bool {
	if p.positive == nil {
		panic("the positive must not be nil, please use Positive() function instead")
	}
	return *p.positive
}

func (p *AffinityGroup) SetPriority(attr float64) {
	p.priority = &attr
}

func (p *AffinityGroup) Priority() (float64, bool) {
	if p.priority != nil {
		return *p.priority, true
	}
	var zero float64
	return zero, false
}

func (p *AffinityGroup) MustPriority() float64 {
	if p.priority == nil {
		panic("the priority must not be nil, please use Priority() function instead")
	}
	return *p.priority
}

func (p *AffinityGroup) SetVmLabels(attr *AffinityLabelSlice) {
	p.vmLabels = attr
}

func (p *AffinityGroup) VmLabels() (*AffinityLabelSlice, bool) {
	if p.vmLabels != nil {
		return p.vmLabels, true
	}
	return nil, false
}

func (p *AffinityGroup) MustVmLabels() *AffinityLabelSlice {
	if p.vmLabels == nil {
		panic("the vmLabels must not be nil, please use VmLabels() function instead")
	}
	return p.vmLabels
}

func (p *AffinityGroup) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *AffinityGroup) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityGroup) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

// SetVmsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) SetVmsRule(attr *AffinityRule) {
	p.vmsRule = attr
}

// VmsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) VmsRule() (*AffinityRule, bool) {
	if p.vmsRule != nil {
		return p.vmsRule, true
	}
	return nil, false
}

// MustVmsRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
func (p *AffinityGroup) MustVmsRule() *AffinityRule {
	if p.vmsRule == nil {
		panic("the vmsRule must not be nil, please use VmsRule() function instead")
	}
	return p.vmsRule
}

// AffinityLabel The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
type AffinityLabel struct {
	Struct
	comment                  *string
	description              *string
	hasImplicitAffinityGroup *bool
	hosts                    *HostSlice
	id                       *string
	name                     *string
	readOnly                 *bool
	vms                      *VmSlice
}

func (p *AffinityLabel) SetComment(attr string) {
	p.comment = &attr
}

func (p *AffinityLabel) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *AffinityLabel) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *AffinityLabel) SetDescription(attr string) {
	p.description = &attr
}

func (p *AffinityLabel) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *AffinityLabel) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *AffinityLabel) SetHasImplicitAffinityGroup(attr bool) {
	p.hasImplicitAffinityGroup = &attr
}

func (p *AffinityLabel) HasImplicitAffinityGroup() (bool, bool) {
	if p.hasImplicitAffinityGroup != nil {
		return *p.hasImplicitAffinityGroup, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityLabel) MustHasImplicitAffinityGroup() bool {
	if p.hasImplicitAffinityGroup == nil {
		panic("the hasImplicitAffinityGroup must not be nil, please use HasImplicitAffinityGroup() function instead")
	}
	return *p.hasImplicitAffinityGroup
}

func (p *AffinityLabel) SetHosts(attr *HostSlice) {
	p.hosts = attr
}

func (p *AffinityLabel) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *AffinityLabel) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("the hosts must not be nil, please use Hosts() function instead")
	}
	return p.hosts
}

func (p *AffinityLabel) SetId(attr string) {
	p.id = &attr
}

func (p *AffinityLabel) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *AffinityLabel) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *AffinityLabel) SetName(attr string) {
	p.name = &attr
}

func (p *AffinityLabel) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *AffinityLabel) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *AffinityLabel) SetReadOnly(attr bool) {
	p.readOnly = &attr
}

func (p *AffinityLabel) ReadOnly() (bool, bool) {
	if p.readOnly != nil {
		return *p.readOnly, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityLabel) MustReadOnly() bool {
	if p.readOnly == nil {
		panic("the readOnly must not be nil, please use ReadOnly() function instead")
	}
	return *p.readOnly
}

func (p *AffinityLabel) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *AffinityLabel) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityLabel) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

// AffinityRule Generic rule definition for affinity group. Each supported resource type (virtual machine, host) is controlled
// by a separate rule. This allows expressing of rules like: no affinity between defined virtual machines, but hard
// affinity between defined virtual machines and virtual hosts.
type AffinityRule struct {
	Struct
	enabled   *bool
	enforcing *bool
	positive  *bool
}

func (p *AffinityRule) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *AffinityRule) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityRule) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

func (p *AffinityRule) SetEnforcing(attr bool) {
	p.enforcing = &attr
}

func (p *AffinityRule) Enforcing() (bool, bool) {
	if p.enforcing != nil {
		return *p.enforcing, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityRule) MustEnforcing() bool {
	if p.enforcing == nil {
		panic("the enforcing must not be nil, please use Enforcing() function instead")
	}
	return *p.enforcing
}

func (p *AffinityRule) SetPositive(attr bool) {
	p.positive = &attr
}

func (p *AffinityRule) Positive() (bool, bool) {
	if p.positive != nil {
		return *p.positive, true
	}
	var zero bool
	return zero, false
}

func (p *AffinityRule) MustPositive() bool {
	if p.positive == nil {
		panic("the positive must not be nil, please use Positive() function instead")
	}
	return *p.positive
}

// Agent Type representing a fence agent.
type Agent struct {
	Struct
	address        *string
	comment        *string
	concurrent     *bool
	description    *string
	encryptOptions *bool
	host           *Host
	id             *string
	name           *string
	options        *OptionSlice
	order          *int64
	password       *string
	port           *int64
	type_          *string
	username       *string
}

func (p *Agent) SetAddress(attr string) {
	p.address = &attr
}

func (p *Agent) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *Agent) SetComment(attr string) {
	p.comment = &attr
}

func (p *Agent) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Agent) SetConcurrent(attr bool) {
	p.concurrent = &attr
}

func (p *Agent) Concurrent() (bool, bool) {
	if p.concurrent != nil {
		return *p.concurrent, true
	}
	var zero bool
	return zero, false
}

func (p *Agent) MustConcurrent() bool {
	if p.concurrent == nil {
		panic("the concurrent must not be nil, please use Concurrent() function instead")
	}
	return *p.concurrent
}

func (p *Agent) SetDescription(attr string) {
	p.description = &attr
}

func (p *Agent) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Agent) SetEncryptOptions(attr bool) {
	p.encryptOptions = &attr
}

func (p *Agent) EncryptOptions() (bool, bool) {
	if p.encryptOptions != nil {
		return *p.encryptOptions, true
	}
	var zero bool
	return zero, false
}

func (p *Agent) MustEncryptOptions() bool {
	if p.encryptOptions == nil {
		panic("the encryptOptions must not be nil, please use EncryptOptions() function instead")
	}
	return *p.encryptOptions
}

// SetHost Type representing a host.
func (p *Agent) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *Agent) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *Agent) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *Agent) SetId(attr string) {
	p.id = &attr
}

func (p *Agent) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Agent) SetName(attr string) {
	p.name = &attr
}

func (p *Agent) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Agent) SetOptions(attr *OptionSlice) {
	p.options = attr
}

func (p *Agent) Options() (*OptionSlice, bool) {
	if p.options != nil {
		return p.options, true
	}
	return nil, false
}

func (p *Agent) MustOptions() *OptionSlice {
	if p.options == nil {
		panic("the options must not be nil, please use Options() function instead")
	}
	return p.options
}

func (p *Agent) SetOrder(attr int64) {
	p.order = &attr
}

func (p *Agent) Order() (int64, bool) {
	if p.order != nil {
		return *p.order, true
	}
	var zero int64
	return zero, false
}

func (p *Agent) MustOrder() int64 {
	if p.order == nil {
		panic("the order must not be nil, please use Order() function instead")
	}
	return *p.order
}

func (p *Agent) SetPassword(attr string) {
	p.password = &attr
}

func (p *Agent) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *Agent) SetPort(attr int64) {
	p.port = &attr
}

func (p *Agent) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *Agent) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *Agent) SetType(attr string) {
	p.type_ = &attr
}

func (p *Agent) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *Agent) SetUsername(attr string) {
	p.username = &attr
}

func (p *Agent) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *Agent) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

// AgentConfiguration Deprecated Agent configuration settings.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
// The deployment of OpenStack hosts can be done by Red Hat OpenStack Platform Director or TripleO.
type AgentConfiguration struct {
	Struct
	address         *string
	brokerType      *MessageBrokerType
	networkMappings *string
	password        *string
	port            *int64
	username        *string
}

func (p *AgentConfiguration) SetAddress(attr string) {
	p.address = &attr
}

func (p *AgentConfiguration) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *AgentConfiguration) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

// SetBrokerType Deprecated Message Broker type.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
func (p *AgentConfiguration) SetBrokerType(attr MessageBrokerType) {
	p.brokerType = &attr
}

// BrokerType Deprecated Message Broker type.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
func (p *AgentConfiguration) BrokerType() (MessageBrokerType, bool) {
	if p.brokerType != nil {
		return *p.brokerType, true
	}
	var zero MessageBrokerType
	return zero, false
}

// MustBrokerType Deprecated Message Broker type.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
func (p *AgentConfiguration) MustBrokerType() MessageBrokerType {
	if p.brokerType == nil {
		panic("the brokerType must not be nil, please use BrokerType() function instead")
	}
	return *p.brokerType
}

func (p *AgentConfiguration) SetNetworkMappings(attr string) {
	p.networkMappings = &attr
}

func (p *AgentConfiguration) NetworkMappings() (string, bool) {
	if p.networkMappings != nil {
		return *p.networkMappings, true
	}
	var zero string
	return zero, false
}

func (p *AgentConfiguration) MustNetworkMappings() string {
	if p.networkMappings == nil {
		panic("the networkMappings must not be nil, please use NetworkMappings() function instead")
	}
	return *p.networkMappings
}

func (p *AgentConfiguration) SetPassword(attr string) {
	p.password = &attr
}

func (p *AgentConfiguration) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *AgentConfiguration) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *AgentConfiguration) SetPort(attr int64) {
	p.port = &attr
}

func (p *AgentConfiguration) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *AgentConfiguration) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *AgentConfiguration) SetUsername(attr string) {
	p.username = &attr
}

func (p *AgentConfiguration) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *AgentConfiguration) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

// Api This type contains the information returned by the root service of the API.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <api>
//   <link rel="hosts" href="/ovirt-engine/api/hosts"/>
//   <link rel="vms" href="/ovirt-engine/api/vms"/>
//   ...
//   <product_info>
//     <name>oVirt Engine</name>
//     <vendor>ovirt.org</vendor>
//     <version>
//       <build>0</build>
//       <full_version>4.1.0_master</full_version>
//       <major>4</major>
//       <minor>1</minor>
//       <revision>0</revision>
//     </version>
//   </product_info>
//   <special_objects>
//     <link rel="templates/blank" href="..."/>
//     <link rel="tags/root" href="..."/>
//   </special_objects>
//   <summary>
//     <vms>
//       <total>10</total>
//       <active>3</active>
//     </vms>
//     <hosts>
//       <total>2</total>
//       <active>2</active>
//     </hosts>
//     <users>
//       <total>8</total>
//       <active>2</active>
//     </users>
//     <storage_domains>
//       <total>2</total>
//       <active>2</active>
//     </storage_domains>
//   </summary>
//   <time>2016-12-12T12:22:25.866+01:00</time>
// </api>
// ----
type Api struct {
	Struct
	authenticatedUser *User
	effectiveUser     *User
	productInfo       *ProductInfo
	specialObjects    *SpecialObjects
	summary           *ApiSummary
	time              *time.Time
}

// SetAuthenticatedUser Represents a user in the system.
func (p *Api) SetAuthenticatedUser(attr *User) {
	p.authenticatedUser = attr
}

// AuthenticatedUser Represents a user in the system.
func (p *Api) AuthenticatedUser() (*User, bool) {
	if p.authenticatedUser != nil {
		return p.authenticatedUser, true
	}
	return nil, false
}

// MustAuthenticatedUser Represents a user in the system.
func (p *Api) MustAuthenticatedUser() *User {
	if p.authenticatedUser == nil {
		panic("the authenticatedUser must not be nil, please use AuthenticatedUser() function instead")
	}
	return p.authenticatedUser
}

// SetEffectiveUser Represents a user in the system.
func (p *Api) SetEffectiveUser(attr *User) {
	p.effectiveUser = attr
}

// EffectiveUser Represents a user in the system.
func (p *Api) EffectiveUser() (*User, bool) {
	if p.effectiveUser != nil {
		return p.effectiveUser, true
	}
	return nil, false
}

// MustEffectiveUser Represents a user in the system.
func (p *Api) MustEffectiveUser() *User {
	if p.effectiveUser == nil {
		panic("the effectiveUser must not be nil, please use EffectiveUser() function instead")
	}
	return p.effectiveUser
}

// SetProductInfo Product information.
//
// The entry point contains a `product_info` element to help an API user determine the legitimacy of the
// {product-name} environment. This includes the name of the product, the `vendor` and the `version`.
//
// Verify a genuine {product-name} environment
//
// The follow elements identify a genuine {product-name} environment:
//
// [source]
// ----
// <api>
// ...
// <product_info>
//   <name>oVirt Engine</name>
//   <vendor>ovirt.org</vendor>
//   <version>
//     <build>0</build>
//     <full_version>4.1.0_master</full_version>
//     <major>4</major>
//     <minor>1</minor>
//     <revision>0</revision>
//   </version>
// </product_info>
// ...
// </api>
// ----
func (p *Api) SetProductInfo(attr *ProductInfo) {
	p.productInfo = attr
}

// ProductInfo Product information.
//
// The entry point contains a `product_info` element to help an API user determine the legitimacy of the
// {product-name} environment. This includes the name of the product, the `vendor` and the `version`.
//
// Verify a genuine {product-name} environment
//
// The follow elements identify a genuine {product-name} environment:
//
// [source]
// ----
// <api>
// ...
// <product_info>
//   <name>oVirt Engine</name>
//   <vendor>ovirt.org</vendor>
//   <version>
//     <build>0</build>
//     <full_version>4.1.0_master</full_version>
//     <major>4</major>
//     <minor>1</minor>
//     <revision>0</revision>
//   </version>
// </product_info>
// ...
// </api>
// ----
func (p *Api) ProductInfo() (*ProductInfo, bool) {
	if p.productInfo != nil {
		return p.productInfo, true
	}
	return nil, false
}

// MustProductInfo Product information.
//
// The entry point contains a `product_info` element to help an API user determine the legitimacy of the
// {product-name} environment. This includes the name of the product, the `vendor` and the `version`.
//
// Verify a genuine {product-name} environment
//
// The follow elements identify a genuine {product-name} environment:
//
// [source]
// ----
// <api>
// ...
// <product_info>
//   <name>oVirt Engine</name>
//   <vendor>ovirt.org</vendor>
//   <version>
//     <build>0</build>
//     <full_version>4.1.0_master</full_version>
//     <major>4</major>
//     <minor>1</minor>
//     <revision>0</revision>
//   </version>
// </product_info>
// ...
// </api>
// ----
func (p *Api) MustProductInfo() *ProductInfo {
	if p.productInfo == nil {
		panic("the productInfo must not be nil, please use ProductInfo() function instead")
	}
	return p.productInfo
}

// SetSpecialObjects This type contains references to special objects, such as blank templates and the root of a hierarchy of tags.
func (p *Api) SetSpecialObjects(attr *SpecialObjects) {
	p.specialObjects = attr
}

// SpecialObjects This type contains references to special objects, such as blank templates and the root of a hierarchy of tags.
func (p *Api) SpecialObjects() (*SpecialObjects, bool) {
	if p.specialObjects != nil {
		return p.specialObjects, true
	}
	return nil, false
}

// MustSpecialObjects This type contains references to special objects, such as blank templates and the root of a hierarchy of tags.
func (p *Api) MustSpecialObjects() *SpecialObjects {
	if p.specialObjects == nil {
		panic("the specialObjects must not be nil, please use SpecialObjects() function instead")
	}
	return p.specialObjects
}

// SetSummary A summary containing the total number of relevant objects, such as virtual machines, hosts, and storage domains.
func (p *Api) SetSummary(attr *ApiSummary) {
	p.summary = attr
}

// Summary A summary containing the total number of relevant objects, such as virtual machines, hosts, and storage domains.
func (p *Api) Summary() (*ApiSummary, bool) {
	if p.summary != nil {
		return p.summary, true
	}
	return nil, false
}

// MustSummary A summary containing the total number of relevant objects, such as virtual machines, hosts, and storage domains.
func (p *Api) MustSummary() *ApiSummary {
	if p.summary == nil {
		panic("the summary must not be nil, please use Summary() function instead")
	}
	return p.summary
}

func (p *Api) SetTime(attr time.Time) {
	p.time = &attr
}

func (p *Api) Time() (time.Time, bool) {
	if p.time != nil {
		return *p.time, true
	}
	var zero time.Time
	return zero, false
}

func (p *Api) MustTime() time.Time {
	if p.time == nil {
		panic("the time must not be nil, please use Time() function instead")
	}
	return *p.time
}

// ApiSummary A summary containing the total number of relevant objects, such as virtual machines, hosts, and storage domains.
type ApiSummary struct {
	Struct
	hosts          *ApiSummaryItem
	storageDomains *ApiSummaryItem
	users          *ApiSummaryItem
	vms            *ApiSummaryItem
}

// SetHosts This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) SetHosts(attr *ApiSummaryItem) {
	p.hosts = attr
}

// Hosts This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) Hosts() (*ApiSummaryItem, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

// MustHosts This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) MustHosts() *ApiSummaryItem {
	if p.hosts == nil {
		panic("the hosts must not be nil, please use Hosts() function instead")
	}
	return p.hosts
}

// SetStorageDomains This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) SetStorageDomains(attr *ApiSummaryItem) {
	p.storageDomains = attr
}

// StorageDomains This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) StorageDomains() (*ApiSummaryItem, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

// MustStorageDomains This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) MustStorageDomains() *ApiSummaryItem {
	if p.storageDomains == nil {
		panic("the storageDomains must not be nil, please use StorageDomains() function instead")
	}
	return p.storageDomains
}

// SetUsers This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) SetUsers(attr *ApiSummaryItem) {
	p.users = attr
}

// Users This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) Users() (*ApiSummaryItem, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

// MustUsers This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) MustUsers() *ApiSummaryItem {
	if p.users == nil {
		panic("the users must not be nil, please use Users() function instead")
	}
	return p.users
}

// SetVms This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) SetVms(attr *ApiSummaryItem) {
	p.vms = attr
}

// Vms This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) Vms() (*ApiSummaryItem, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

// MustVms This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
func (p *ApiSummary) MustVms() *ApiSummaryItem {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

// ApiSummaryItem This type contains an item of the API summary. Each item contains the total and active number of some kind of
// object.
type ApiSummaryItem struct {
	Struct
	active *int64
	total  *int64
}

func (p *ApiSummaryItem) SetActive(attr int64) {
	p.active = &attr
}

func (p *ApiSummaryItem) Active() (int64, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero int64
	return zero, false
}

func (p *ApiSummaryItem) MustActive() int64 {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *ApiSummaryItem) SetTotal(attr int64) {
	p.total = &attr
}

func (p *ApiSummaryItem) Total() (int64, bool) {
	if p.total != nil {
		return *p.total, true
	}
	var zero int64
	return zero, false
}

func (p *ApiSummaryItem) MustTotal() int64 {
	if p.total == nil {
		panic("the total must not be nil, please use Total() function instead")
	}
	return *p.total
}

// Application Represents an application installed on a virtual machine. Applications are reported by the guest agent, if you
// deploy one on the virtual machine operating system.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/vms/123/applications/456
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <application href="/ovirt-engine/api/vms/123/applications/456" id="456">
//   <name>application-test-1.0.0-0.el7</name>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </application>
// ----
type Application struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
	vm          *Vm
}

func (p *Application) SetComment(attr string) {
	p.comment = &attr
}

func (p *Application) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Application) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Application) SetDescription(attr string) {
	p.description = &attr
}

func (p *Application) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Application) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Application) SetId(attr string) {
	p.id = &attr
}

func (p *Application) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Application) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Application) SetName(attr string) {
	p.name = &attr
}

func (p *Application) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Application) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetVm Represents a virtual machine.
func (p *Application) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Application) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Application) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type AuthorizedKey struct {
	Struct
	comment     *string
	description *string
	id          *string
	key         *string
	name        *string
	user        *User
}

func (p *AuthorizedKey) SetComment(attr string) {
	p.comment = &attr
}

func (p *AuthorizedKey) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *AuthorizedKey) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *AuthorizedKey) SetDescription(attr string) {
	p.description = &attr
}

func (p *AuthorizedKey) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *AuthorizedKey) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *AuthorizedKey) SetId(attr string) {
	p.id = &attr
}

func (p *AuthorizedKey) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *AuthorizedKey) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *AuthorizedKey) SetKey(attr string) {
	p.key = &attr
}

func (p *AuthorizedKey) Key() (string, bool) {
	if p.key != nil {
		return *p.key, true
	}
	var zero string
	return zero, false
}

func (p *AuthorizedKey) MustKey() string {
	if p.key == nil {
		panic("the key must not be nil, please use Key() function instead")
	}
	return *p.key
}

func (p *AuthorizedKey) SetName(attr string) {
	p.name = &attr
}

func (p *AuthorizedKey) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *AuthorizedKey) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetUser Represents a user in the system.
func (p *AuthorizedKey) SetUser(attr *User) {
	p.user = attr
}

// User Represents a user in the system.
func (p *AuthorizedKey) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

// MustUser Represents a user in the system.
func (p *AuthorizedKey) MustUser() *User {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return p.user
}

type Backup struct {
	Struct
	comment          *string
	creationDate     *time.Time
	description      *string
	disks            *DiskSlice
	fromCheckpointId *string
	host             *Host
	id               *string
	name             *string
	phase            *BackupPhase
	toCheckpointId   *string
	vm               *Vm
}

func (p *Backup) SetComment(attr string) {
	p.comment = &attr
}

func (p *Backup) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Backup) SetCreationDate(attr time.Time) {
	p.creationDate = &attr
}

func (p *Backup) CreationDate() (time.Time, bool) {
	if p.creationDate != nil {
		return *p.creationDate, true
	}
	var zero time.Time
	return zero, false
}

func (p *Backup) MustCreationDate() time.Time {
	if p.creationDate == nil {
		panic("the creationDate must not be nil, please use CreationDate() function instead")
	}
	return *p.creationDate
}

func (p *Backup) SetDescription(attr string) {
	p.description = &attr
}

func (p *Backup) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Backup) SetDisks(attr *DiskSlice) {
	p.disks = attr
}

func (p *Backup) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *Backup) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("the disks must not be nil, please use Disks() function instead")
	}
	return p.disks
}

func (p *Backup) SetFromCheckpointId(attr string) {
	p.fromCheckpointId = &attr
}

func (p *Backup) FromCheckpointId() (string, bool) {
	if p.fromCheckpointId != nil {
		return *p.fromCheckpointId, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustFromCheckpointId() string {
	if p.fromCheckpointId == nil {
		panic("the fromCheckpointId must not be nil, please use FromCheckpointId() function instead")
	}
	return *p.fromCheckpointId
}

// SetHost Type representing a host.
func (p *Backup) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *Backup) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *Backup) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *Backup) SetId(attr string) {
	p.id = &attr
}

func (p *Backup) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Backup) SetName(attr string) {
	p.name = &attr
}

func (p *Backup) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Backup) SetPhase(attr BackupPhase) {
	p.phase = &attr
}

func (p *Backup) Phase() (BackupPhase, bool) {
	if p.phase != nil {
		return *p.phase, true
	}
	var zero BackupPhase
	return zero, false
}

func (p *Backup) MustPhase() BackupPhase {
	if p.phase == nil {
		panic("the phase must not be nil, please use Phase() function instead")
	}
	return *p.phase
}

func (p *Backup) SetToCheckpointId(attr string) {
	p.toCheckpointId = &attr
}

func (p *Backup) ToCheckpointId() (string, bool) {
	if p.toCheckpointId != nil {
		return *p.toCheckpointId, true
	}
	var zero string
	return zero, false
}

func (p *Backup) MustToCheckpointId() string {
	if p.toCheckpointId == nil {
		panic("the toCheckpointId must not be nil, please use ToCheckpointId() function instead")
	}
	return *p.toCheckpointId
}

// SetVm Represents a virtual machine.
func (p *Backup) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Backup) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Backup) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type Balance struct {
	Struct
	comment              *string
	description          *string
	id                   *string
	name                 *string
	schedulingPolicy     *SchedulingPolicy
	schedulingPolicyUnit *SchedulingPolicyUnit
}

func (p *Balance) SetComment(attr string) {
	p.comment = &attr
}

func (p *Balance) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Balance) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Balance) SetDescription(attr string) {
	p.description = &attr
}

func (p *Balance) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Balance) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Balance) SetId(attr string) {
	p.id = &attr
}

func (p *Balance) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Balance) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Balance) SetName(attr string) {
	p.name = &attr
}

func (p *Balance) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Balance) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Balance) SetSchedulingPolicy(attr *SchedulingPolicy) {
	p.schedulingPolicy = attr
}

func (p *Balance) SchedulingPolicy() (*SchedulingPolicy, bool) {
	if p.schedulingPolicy != nil {
		return p.schedulingPolicy, true
	}
	return nil, false
}

func (p *Balance) MustSchedulingPolicy() *SchedulingPolicy {
	if p.schedulingPolicy == nil {
		panic("the schedulingPolicy must not be nil, please use SchedulingPolicy() function instead")
	}
	return p.schedulingPolicy
}

func (p *Balance) SetSchedulingPolicyUnit(attr *SchedulingPolicyUnit) {
	p.schedulingPolicyUnit = attr
}

func (p *Balance) SchedulingPolicyUnit() (*SchedulingPolicyUnit, bool) {
	if p.schedulingPolicyUnit != nil {
		return p.schedulingPolicyUnit, true
	}
	return nil, false
}

func (p *Balance) MustSchedulingPolicyUnit() *SchedulingPolicyUnit {
	if p.schedulingPolicyUnit == nil {
		panic("the schedulingPolicyUnit must not be nil, please use SchedulingPolicyUnit() function instead")
	}
	return p.schedulingPolicyUnit
}

type Bios struct {
	Struct
	bootMenu *BootMenu
	type_    *BiosType
}

// SetBootMenu Represents boot menu configuration
// for virtual machines and templates.
func (p *Bios) SetBootMenu(attr *BootMenu) {
	p.bootMenu = attr
}

// BootMenu Represents boot menu configuration
// for virtual machines and templates.
func (p *Bios) BootMenu() (*BootMenu, bool) {
	if p.bootMenu != nil {
		return p.bootMenu, true
	}
	return nil, false
}

// MustBootMenu Represents boot menu configuration
// for virtual machines and templates.
func (p *Bios) MustBootMenu() *BootMenu {
	if p.bootMenu == nil {
		panic("the bootMenu must not be nil, please use BootMenu() function instead")
	}
	return p.bootMenu
}

// SetType Type representing a chipset and a BIOS type combination.
func (p *Bios) SetType(attr BiosType) {
	p.type_ = &attr
}

// Type Type representing a chipset and a BIOS type combination.
func (p *Bios) Type() (BiosType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero BiosType
	return zero, false
}

// MustType Type representing a chipset and a BIOS type combination.
func (p *Bios) MustType() BiosType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type BlockStatistic struct {
	Struct
	statistics *StatisticSlice
}

func (p *BlockStatistic) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *BlockStatistic) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *BlockStatistic) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// Bonding Represents a network interfaces bond.
type Bonding struct {
	Struct
	activeSlave  *HostNic
	adPartnerMac *Mac
	options      *OptionSlice
	slaves       *HostNicSlice
}

// SetActiveSlave Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *Bonding) SetActiveSlave(attr *HostNic) {
	p.activeSlave = attr
}

// ActiveSlave Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *Bonding) ActiveSlave() (*HostNic, bool) {
	if p.activeSlave != nil {
		return p.activeSlave, true
	}
	return nil, false
}

// MustActiveSlave Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *Bonding) MustActiveSlave() *HostNic {
	if p.activeSlave == nil {
		panic("the activeSlave must not be nil, please use ActiveSlave() function instead")
	}
	return p.activeSlave
}

// SetAdPartnerMac Represents a MAC address of a virtual network interface.
func (p *Bonding) SetAdPartnerMac(attr *Mac) {
	p.adPartnerMac = attr
}

// AdPartnerMac Represents a MAC address of a virtual network interface.
func (p *Bonding) AdPartnerMac() (*Mac, bool) {
	if p.adPartnerMac != nil {
		return p.adPartnerMac, true
	}
	return nil, false
}

// MustAdPartnerMac Represents a MAC address of a virtual network interface.
func (p *Bonding) MustAdPartnerMac() *Mac {
	if p.adPartnerMac == nil {
		panic("the adPartnerMac must not be nil, please use AdPartnerMac() function instead")
	}
	return p.adPartnerMac
}

func (p *Bonding) SetOptions(attr *OptionSlice) {
	p.options = attr
}

func (p *Bonding) Options() (*OptionSlice, bool) {
	if p.options != nil {
		return p.options, true
	}
	return nil, false
}

func (p *Bonding) MustOptions() *OptionSlice {
	if p.options == nil {
		panic("the options must not be nil, please use Options() function instead")
	}
	return p.options
}

func (p *Bonding) SetSlaves(attr *HostNicSlice) {
	p.slaves = attr
}

func (p *Bonding) Slaves() (*HostNicSlice, bool) {
	if p.slaves != nil {
		return p.slaves, true
	}
	return nil, false
}

func (p *Bonding) MustSlaves() *HostNicSlice {
	if p.slaves == nil {
		panic("the slaves must not be nil, please use Slaves() function instead")
	}
	return p.slaves
}

// Bookmark Represents a bookmark in the system.
type Bookmark struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
	value       *string
}

func (p *Bookmark) SetComment(attr string) {
	p.comment = &attr
}

func (p *Bookmark) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Bookmark) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Bookmark) SetDescription(attr string) {
	p.description = &attr
}

func (p *Bookmark) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Bookmark) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Bookmark) SetId(attr string) {
	p.id = &attr
}

func (p *Bookmark) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Bookmark) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Bookmark) SetName(attr string) {
	p.name = &attr
}

func (p *Bookmark) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Bookmark) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Bookmark) SetValue(attr string) {
	p.value = &attr
}

func (p *Bookmark) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *Bookmark) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

// Boot Configuration of the boot sequence of a virtual machine.
type Boot struct {
	Struct
	devices []BootDevice
}

func (p *Boot) SetDevices(attr []BootDevice) {
	p.devices = attr
}

func (p *Boot) Devices() ([]BootDevice, bool) {
	if p.devices != nil {
		return p.devices, true
	}
	return nil, false
}

func (p *Boot) MustDevices() []BootDevice {
	if p.devices == nil {
		panic("the devices must not be nil, please use Devices() function instead")
	}
	return p.devices
}

// BootMenu Represents boot menu configuration
// for virtual machines and templates.
type BootMenu struct {
	Struct
	enabled *bool
}

func (p *BootMenu) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *BootMenu) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *BootMenu) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

type BrickProfileDetail struct {
	Struct
	brick          *GlusterBrick
	profileDetails *ProfileDetailSlice
}

func (p *BrickProfileDetail) SetBrick(attr *GlusterBrick) {
	p.brick = attr
}

func (p *BrickProfileDetail) Brick() (*GlusterBrick, bool) {
	if p.brick != nil {
		return p.brick, true
	}
	return nil, false
}

func (p *BrickProfileDetail) MustBrick() *GlusterBrick {
	if p.brick == nil {
		panic("the brick must not be nil, please use Brick() function instead")
	}
	return p.brick
}

func (p *BrickProfileDetail) SetProfileDetails(attr *ProfileDetailSlice) {
	p.profileDetails = attr
}

func (p *BrickProfileDetail) ProfileDetails() (*ProfileDetailSlice, bool) {
	if p.profileDetails != nil {
		return p.profileDetails, true
	}
	return nil, false
}

func (p *BrickProfileDetail) MustProfileDetails() *ProfileDetailSlice {
	if p.profileDetails == nil {
		panic("the profileDetails must not be nil, please use ProfileDetails() function instead")
	}
	return p.profileDetails
}

type Cdrom struct {
	Struct
	comment      *string
	description  *string
	file         *File
	id           *string
	instanceType *InstanceType
	name         *string
	template     *Template
	vm           *Vm
	vms          *VmSlice
}

func (p *Cdrom) SetComment(attr string) {
	p.comment = &attr
}

func (p *Cdrom) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Cdrom) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Cdrom) SetDescription(attr string) {
	p.description = &attr
}

func (p *Cdrom) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Cdrom) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Cdrom) SetFile(attr *File) {
	p.file = attr
}

func (p *Cdrom) File() (*File, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *Cdrom) MustFile() *File {
	if p.file == nil {
		panic("the file must not be nil, please use File() function instead")
	}
	return p.file
}

func (p *Cdrom) SetId(attr string) {
	p.id = &attr
}

func (p *Cdrom) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Cdrom) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Cdrom) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Cdrom) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Cdrom) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *Cdrom) SetName(attr string) {
	p.name = &attr
}

func (p *Cdrom) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Cdrom) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Cdrom) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Cdrom) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Cdrom) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetVm Represents a virtual machine.
func (p *Cdrom) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Cdrom) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Cdrom) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *Cdrom) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Cdrom) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Cdrom) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

type Certificate struct {
	Struct
	comment      *string
	content      *string
	description  *string
	id           *string
	name         *string
	organization *string
	subject      *string
}

func (p *Certificate) SetComment(attr string) {
	p.comment = &attr
}

func (p *Certificate) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Certificate) SetContent(attr string) {
	p.content = &attr
}

func (p *Certificate) Content() (string, bool) {
	if p.content != nil {
		return *p.content, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustContent() string {
	if p.content == nil {
		panic("the content must not be nil, please use Content() function instead")
	}
	return *p.content
}

func (p *Certificate) SetDescription(attr string) {
	p.description = &attr
}

func (p *Certificate) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Certificate) SetId(attr string) {
	p.id = &attr
}

func (p *Certificate) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Certificate) SetName(attr string) {
	p.name = &attr
}

func (p *Certificate) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Certificate) SetOrganization(attr string) {
	p.organization = &attr
}

func (p *Certificate) Organization() (string, bool) {
	if p.organization != nil {
		return *p.organization, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustOrganization() string {
	if p.organization == nil {
		panic("the organization must not be nil, please use Organization() function instead")
	}
	return *p.organization
}

func (p *Certificate) SetSubject(attr string) {
	p.subject = &attr
}

func (p *Certificate) Subject() (string, bool) {
	if p.subject != nil {
		return *p.subject, true
	}
	var zero string
	return zero, false
}

func (p *Certificate) MustSubject() string {
	if p.subject == nil {
		panic("the subject must not be nil, please use Subject() function instead")
	}
	return *p.subject
}

type Checkpoint struct {
	Struct
	comment      *string
	creationDate *time.Time
	description  *string
	disks        *DiskSlice
	id           *string
	name         *string
	parentId     *string
	state        *CheckpointState
	vm           *Vm
}

func (p *Checkpoint) SetComment(attr string) {
	p.comment = &attr
}

func (p *Checkpoint) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Checkpoint) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Checkpoint) SetCreationDate(attr time.Time) {
	p.creationDate = &attr
}

func (p *Checkpoint) CreationDate() (time.Time, bool) {
	if p.creationDate != nil {
		return *p.creationDate, true
	}
	var zero time.Time
	return zero, false
}

func (p *Checkpoint) MustCreationDate() time.Time {
	if p.creationDate == nil {
		panic("the creationDate must not be nil, please use CreationDate() function instead")
	}
	return *p.creationDate
}

func (p *Checkpoint) SetDescription(attr string) {
	p.description = &attr
}

func (p *Checkpoint) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Checkpoint) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Checkpoint) SetDisks(attr *DiskSlice) {
	p.disks = attr
}

func (p *Checkpoint) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *Checkpoint) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("the disks must not be nil, please use Disks() function instead")
	}
	return p.disks
}

func (p *Checkpoint) SetId(attr string) {
	p.id = &attr
}

func (p *Checkpoint) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Checkpoint) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Checkpoint) SetName(attr string) {
	p.name = &attr
}

func (p *Checkpoint) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Checkpoint) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Checkpoint) SetParentId(attr string) {
	p.parentId = &attr
}

func (p *Checkpoint) ParentId() (string, bool) {
	if p.parentId != nil {
		return *p.parentId, true
	}
	var zero string
	return zero, false
}

func (p *Checkpoint) MustParentId() string {
	if p.parentId == nil {
		panic("the parentId must not be nil, please use ParentId() function instead")
	}
	return *p.parentId
}

func (p *Checkpoint) SetState(attr CheckpointState) {
	p.state = &attr
}

func (p *Checkpoint) State() (CheckpointState, bool) {
	if p.state != nil {
		return *p.state, true
	}
	var zero CheckpointState
	return zero, false
}

func (p *Checkpoint) MustState() CheckpointState {
	if p.state == nil {
		panic("the state must not be nil, please use State() function instead")
	}
	return *p.state
}

// SetVm Represents a virtual machine.
func (p *Checkpoint) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Checkpoint) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Checkpoint) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

// CloudInit Deprecated type to specify _cloud-init_ configuration.
//
// This type has been deprecated and replaced by alternative attributes inside the
// <<types/initialization, Initialization>> type. See the
// <<types/initialization/attributes/cloud_init, cloud_init>> attribute documentation for details.
type CloudInit struct {
	Struct
	authorizedKeys       *AuthorizedKeySlice
	files                *FileSlice
	host                 *Host
	networkConfiguration *NetworkConfiguration
	regenerateSshKeys    *bool
	timezone             *string
	users                *UserSlice
}

func (p *CloudInit) SetAuthorizedKeys(attr *AuthorizedKeySlice) {
	p.authorizedKeys = attr
}

func (p *CloudInit) AuthorizedKeys() (*AuthorizedKeySlice, bool) {
	if p.authorizedKeys != nil {
		return p.authorizedKeys, true
	}
	return nil, false
}

func (p *CloudInit) MustAuthorizedKeys() *AuthorizedKeySlice {
	if p.authorizedKeys == nil {
		panic("the authorizedKeys must not be nil, please use AuthorizedKeys() function instead")
	}
	return p.authorizedKeys
}

func (p *CloudInit) SetFiles(attr *FileSlice) {
	p.files = attr
}

func (p *CloudInit) Files() (*FileSlice, bool) {
	if p.files != nil {
		return p.files, true
	}
	return nil, false
}

func (p *CloudInit) MustFiles() *FileSlice {
	if p.files == nil {
		panic("the files must not be nil, please use Files() function instead")
	}
	return p.files
}

// SetHost Type representing a host.
func (p *CloudInit) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *CloudInit) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *CloudInit) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *CloudInit) SetNetworkConfiguration(attr *NetworkConfiguration) {
	p.networkConfiguration = attr
}

func (p *CloudInit) NetworkConfiguration() (*NetworkConfiguration, bool) {
	if p.networkConfiguration != nil {
		return p.networkConfiguration, true
	}
	return nil, false
}

func (p *CloudInit) MustNetworkConfiguration() *NetworkConfiguration {
	if p.networkConfiguration == nil {
		panic("the networkConfiguration must not be nil, please use NetworkConfiguration() function instead")
	}
	return p.networkConfiguration
}

func (p *CloudInit) SetRegenerateSshKeys(attr bool) {
	p.regenerateSshKeys = &attr
}

func (p *CloudInit) RegenerateSshKeys() (bool, bool) {
	if p.regenerateSshKeys != nil {
		return *p.regenerateSshKeys, true
	}
	var zero bool
	return zero, false
}

func (p *CloudInit) MustRegenerateSshKeys() bool {
	if p.regenerateSshKeys == nil {
		panic("the regenerateSshKeys must not be nil, please use RegenerateSshKeys() function instead")
	}
	return *p.regenerateSshKeys
}

func (p *CloudInit) SetTimezone(attr string) {
	p.timezone = &attr
}

func (p *CloudInit) Timezone() (string, bool) {
	if p.timezone != nil {
		return *p.timezone, true
	}
	var zero string
	return zero, false
}

func (p *CloudInit) MustTimezone() string {
	if p.timezone == nil {
		panic("the timezone must not be nil, please use Timezone() function instead")
	}
	return *p.timezone
}

func (p *CloudInit) SetUsers(attr *UserSlice) {
	p.users = attr
}

func (p *CloudInit) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *CloudInit) MustUsers() *UserSlice {
	if p.users == nil {
		panic("the users must not be nil, please use Users() function instead")
	}
	return p.users
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
type Cluster struct {
	Struct
	affinityGroups                   *AffinityGroupSlice
	ballooningEnabled                *bool
	biosType                         *BiosType
	comment                          *string
	cpu                              *Cpu
	cpuProfiles                      *CpuProfileSlice
	customSchedulingPolicyProperties *PropertySlice
	dataCenter                       *DataCenter
	description                      *string
	display                          *Display
	enabledFeatures                  *ClusterFeatureSlice
	errorHandling                    *ErrorHandling
	externalNetworkProviders         *ExternalProviderSlice
	fencingPolicy                    *FencingPolicy
	fipsMode                         *FipsMode
	firewallType                     *FirewallType
	glusterHooks                     *GlusterHookSlice
	glusterService                   *bool
	glusterTunedProfile              *string
	glusterVolumes                   *GlusterVolumeSlice
	haReservation                    *bool
	id                               *string
	ksm                              *Ksm
	logMaxMemoryUsedThreshold        *int64
	logMaxMemoryUsedThresholdType    *LogMaxMemoryUsedThresholdType
	macPool                          *MacPool
	maintenanceReasonRequired        *bool
	managementNetwork                *Network
	memoryPolicy                     *MemoryPolicy
	migration                        *MigrationOptions
	name                             *string
	networkFilters                   *NetworkFilterSlice
	networks                         *NetworkSlice
	optionalReason                   *bool
	permissions                      *PermissionSlice
	requiredRngSources               []RngSource
	schedulingPolicy                 *SchedulingPolicy
	serialNumber                     *SerialNumber
	supportedVersions                *VersionSlice
	switchType                       *SwitchType
	threadsAsCores                   *bool
	trustedService                   *bool
	tunnelMigration                  *bool
	version                          *Version
	virtService                      *bool
	vncEncryption                    *bool
}

func (p *Cluster) SetAffinityGroups(attr *AffinityGroupSlice) {
	p.affinityGroups = attr
}

func (p *Cluster) AffinityGroups() (*AffinityGroupSlice, bool) {
	if p.affinityGroups != nil {
		return p.affinityGroups, true
	}
	return nil, false
}

func (p *Cluster) MustAffinityGroups() *AffinityGroupSlice {
	if p.affinityGroups == nil {
		panic("the affinityGroups must not be nil, please use AffinityGroups() function instead")
	}
	return p.affinityGroups
}

func (p *Cluster) SetBallooningEnabled(attr bool) {
	p.ballooningEnabled = &attr
}

func (p *Cluster) BallooningEnabled() (bool, bool) {
	if p.ballooningEnabled != nil {
		return *p.ballooningEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustBallooningEnabled() bool {
	if p.ballooningEnabled == nil {
		panic("the ballooningEnabled must not be nil, please use BallooningEnabled() function instead")
	}
	return *p.ballooningEnabled
}

// SetBiosType Type representing a chipset and a BIOS type combination.
func (p *Cluster) SetBiosType(attr BiosType) {
	p.biosType = &attr
}

// BiosType Type representing a chipset and a BIOS type combination.
func (p *Cluster) BiosType() (BiosType, bool) {
	if p.biosType != nil {
		return *p.biosType, true
	}
	var zero BiosType
	return zero, false
}

// MustBiosType Type representing a chipset and a BIOS type combination.
func (p *Cluster) MustBiosType() BiosType {
	if p.biosType == nil {
		panic("the biosType must not be nil, please use BiosType() function instead")
	}
	return *p.biosType
}

func (p *Cluster) SetComment(attr string) {
	p.comment = &attr
}

func (p *Cluster) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Cluster) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Cluster) SetCpu(attr *Cpu) {
	p.cpu = attr
}

func (p *Cluster) Cpu() (*Cpu, bool) {
	if p.cpu != nil {
		return p.cpu, true
	}
	return nil, false
}

func (p *Cluster) MustCpu() *Cpu {
	if p.cpu == nil {
		panic("the cpu must not be nil, please use Cpu() function instead")
	}
	return p.cpu
}

func (p *Cluster) SetCpuProfiles(attr *CpuProfileSlice) {
	p.cpuProfiles = attr
}

func (p *Cluster) CpuProfiles() (*CpuProfileSlice, bool) {
	if p.cpuProfiles != nil {
		return p.cpuProfiles, true
	}
	return nil, false
}

func (p *Cluster) MustCpuProfiles() *CpuProfileSlice {
	if p.cpuProfiles == nil {
		panic("the cpuProfiles must not be nil, please use CpuProfiles() function instead")
	}
	return p.cpuProfiles
}

func (p *Cluster) SetCustomSchedulingPolicyProperties(attr *PropertySlice) {
	p.customSchedulingPolicyProperties = attr
}

func (p *Cluster) CustomSchedulingPolicyProperties() (*PropertySlice, bool) {
	if p.customSchedulingPolicyProperties != nil {
		return p.customSchedulingPolicyProperties, true
	}
	return nil, false
}

func (p *Cluster) MustCustomSchedulingPolicyProperties() *PropertySlice {
	if p.customSchedulingPolicyProperties == nil {
		panic("the customSchedulingPolicyProperties must not be nil, please use CustomSchedulingPolicyProperties() function instead")
	}
	return p.customSchedulingPolicyProperties
}

func (p *Cluster) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Cluster) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Cluster) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Cluster) SetDescription(attr string) {
	p.description = &attr
}

func (p *Cluster) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Cluster) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDisplay Represents a graphic console configuration.
func (p *Cluster) SetDisplay(attr *Display) {
	p.display = attr
}

// Display Represents a graphic console configuration.
func (p *Cluster) Display() (*Display, bool) {
	if p.display != nil {
		return p.display, true
	}
	return nil, false
}

// MustDisplay Represents a graphic console configuration.
func (p *Cluster) MustDisplay() *Display {
	if p.display == nil {
		panic("the display must not be nil, please use Display() function instead")
	}
	return p.display
}

func (p *Cluster) SetEnabledFeatures(attr *ClusterFeatureSlice) {
	p.enabledFeatures = attr
}

func (p *Cluster) EnabledFeatures() (*ClusterFeatureSlice, bool) {
	if p.enabledFeatures != nil {
		return p.enabledFeatures, true
	}
	return nil, false
}

func (p *Cluster) MustEnabledFeatures() *ClusterFeatureSlice {
	if p.enabledFeatures == nil {
		panic("the enabledFeatures must not be nil, please use EnabledFeatures() function instead")
	}
	return p.enabledFeatures
}

func (p *Cluster) SetErrorHandling(attr *ErrorHandling) {
	p.errorHandling = attr
}

func (p *Cluster) ErrorHandling() (*ErrorHandling, bool) {
	if p.errorHandling != nil {
		return p.errorHandling, true
	}
	return nil, false
}

func (p *Cluster) MustErrorHandling() *ErrorHandling {
	if p.errorHandling == nil {
		panic("the errorHandling must not be nil, please use ErrorHandling() function instead")
	}
	return p.errorHandling
}

func (p *Cluster) SetExternalNetworkProviders(attr *ExternalProviderSlice) {
	p.externalNetworkProviders = attr
}

func (p *Cluster) ExternalNetworkProviders() (*ExternalProviderSlice, bool) {
	if p.externalNetworkProviders != nil {
		return p.externalNetworkProviders, true
	}
	return nil, false
}

func (p *Cluster) MustExternalNetworkProviders() *ExternalProviderSlice {
	if p.externalNetworkProviders == nil {
		panic("the externalNetworkProviders must not be nil, please use ExternalNetworkProviders() function instead")
	}
	return p.externalNetworkProviders
}

// SetFencingPolicy Type representing a cluster fencing policy.
func (p *Cluster) SetFencingPolicy(attr *FencingPolicy) {
	p.fencingPolicy = attr
}

// FencingPolicy Type representing a cluster fencing policy.
func (p *Cluster) FencingPolicy() (*FencingPolicy, bool) {
	if p.fencingPolicy != nil {
		return p.fencingPolicy, true
	}
	return nil, false
}

// MustFencingPolicy Type representing a cluster fencing policy.
func (p *Cluster) MustFencingPolicy() *FencingPolicy {
	if p.fencingPolicy == nil {
		panic("the fencingPolicy must not be nil, please use FencingPolicy() function instead")
	}
	return p.fencingPolicy
}

// SetFipsMode Representation of the FIPS mode to the cluster.
func (p *Cluster) SetFipsMode(attr FipsMode) {
	p.fipsMode = &attr
}

// FipsMode Representation of the FIPS mode to the cluster.
func (p *Cluster) FipsMode() (FipsMode, bool) {
	if p.fipsMode != nil {
		return *p.fipsMode, true
	}
	var zero FipsMode
	return zero, false
}

// MustFipsMode Representation of the FIPS mode to the cluster.
func (p *Cluster) MustFipsMode() FipsMode {
	if p.fipsMode == nil {
		panic("the fipsMode must not be nil, please use FipsMode() function instead")
	}
	return *p.fipsMode
}

// SetFirewallType Describes all firewall types supported by the system.
func (p *Cluster) SetFirewallType(attr FirewallType) {
	p.firewallType = &attr
}

// FirewallType Describes all firewall types supported by the system.
func (p *Cluster) FirewallType() (FirewallType, bool) {
	if p.firewallType != nil {
		return *p.firewallType, true
	}
	var zero FirewallType
	return zero, false
}

// MustFirewallType Describes all firewall types supported by the system.
func (p *Cluster) MustFirewallType() FirewallType {
	if p.firewallType == nil {
		panic("the firewallType must not be nil, please use FirewallType() function instead")
	}
	return *p.firewallType
}

func (p *Cluster) SetGlusterHooks(attr *GlusterHookSlice) {
	p.glusterHooks = attr
}

func (p *Cluster) GlusterHooks() (*GlusterHookSlice, bool) {
	if p.glusterHooks != nil {
		return p.glusterHooks, true
	}
	return nil, false
}

func (p *Cluster) MustGlusterHooks() *GlusterHookSlice {
	if p.glusterHooks == nil {
		panic("the glusterHooks must not be nil, please use GlusterHooks() function instead")
	}
	return p.glusterHooks
}

func (p *Cluster) SetGlusterService(attr bool) {
	p.glusterService = &attr
}

func (p *Cluster) GlusterService() (bool, bool) {
	if p.glusterService != nil {
		return *p.glusterService, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustGlusterService() bool {
	if p.glusterService == nil {
		panic("the glusterService must not be nil, please use GlusterService() function instead")
	}
	return *p.glusterService
}

func (p *Cluster) SetGlusterTunedProfile(attr string) {
	p.glusterTunedProfile = &attr
}

func (p *Cluster) GlusterTunedProfile() (string, bool) {
	if p.glusterTunedProfile != nil {
		return *p.glusterTunedProfile, true
	}
	var zero string
	return zero, false
}

func (p *Cluster) MustGlusterTunedProfile() string {
	if p.glusterTunedProfile == nil {
		panic("the glusterTunedProfile must not be nil, please use GlusterTunedProfile() function instead")
	}
	return *p.glusterTunedProfile
}

func (p *Cluster) SetGlusterVolumes(attr *GlusterVolumeSlice) {
	p.glusterVolumes = attr
}

func (p *Cluster) GlusterVolumes() (*GlusterVolumeSlice, bool) {
	if p.glusterVolumes != nil {
		return p.glusterVolumes, true
	}
	return nil, false
}

func (p *Cluster) MustGlusterVolumes() *GlusterVolumeSlice {
	if p.glusterVolumes == nil {
		panic("the glusterVolumes must not be nil, please use GlusterVolumes() function instead")
	}
	return p.glusterVolumes
}

func (p *Cluster) SetHaReservation(attr bool) {
	p.haReservation = &attr
}

func (p *Cluster) HaReservation() (bool, bool) {
	if p.haReservation != nil {
		return *p.haReservation, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustHaReservation() bool {
	if p.haReservation == nil {
		panic("the haReservation must not be nil, please use HaReservation() function instead")
	}
	return *p.haReservation
}

func (p *Cluster) SetId(attr string) {
	p.id = &attr
}

func (p *Cluster) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Cluster) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Cluster) SetKsm(attr *Ksm) {
	p.ksm = attr
}

func (p *Cluster) Ksm() (*Ksm, bool) {
	if p.ksm != nil {
		return p.ksm, true
	}
	return nil, false
}

func (p *Cluster) MustKsm() *Ksm {
	if p.ksm == nil {
		panic("the ksm must not be nil, please use Ksm() function instead")
	}
	return p.ksm
}

func (p *Cluster) SetLogMaxMemoryUsedThreshold(attr int64) {
	p.logMaxMemoryUsedThreshold = &attr
}

func (p *Cluster) LogMaxMemoryUsedThreshold() (int64, bool) {
	if p.logMaxMemoryUsedThreshold != nil {
		return *p.logMaxMemoryUsedThreshold, true
	}
	var zero int64
	return zero, false
}

func (p *Cluster) MustLogMaxMemoryUsedThreshold() int64 {
	if p.logMaxMemoryUsedThreshold == nil {
		panic("the logMaxMemoryUsedThreshold must not be nil, please use LogMaxMemoryUsedThreshold() function instead")
	}
	return *p.logMaxMemoryUsedThreshold
}

// SetLogMaxMemoryUsedThresholdType Describes all maximum memory threshold types supported by the system.
func (p *Cluster) SetLogMaxMemoryUsedThresholdType(attr LogMaxMemoryUsedThresholdType) {
	p.logMaxMemoryUsedThresholdType = &attr
}

// LogMaxMemoryUsedThresholdType Describes all maximum memory threshold types supported by the system.
func (p *Cluster) LogMaxMemoryUsedThresholdType() (LogMaxMemoryUsedThresholdType, bool) {
	if p.logMaxMemoryUsedThresholdType != nil {
		return *p.logMaxMemoryUsedThresholdType, true
	}
	var zero LogMaxMemoryUsedThresholdType
	return zero, false
}

// MustLogMaxMemoryUsedThresholdType Describes all maximum memory threshold types supported by the system.
func (p *Cluster) MustLogMaxMemoryUsedThresholdType() LogMaxMemoryUsedThresholdType {
	if p.logMaxMemoryUsedThresholdType == nil {
		panic("the logMaxMemoryUsedThresholdType must not be nil, please use LogMaxMemoryUsedThresholdType() function instead")
	}
	return *p.logMaxMemoryUsedThresholdType
}

// SetMacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *Cluster) SetMacPool(attr *MacPool) {
	p.macPool = attr
}

// MacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *Cluster) MacPool() (*MacPool, bool) {
	if p.macPool != nil {
		return p.macPool, true
	}
	return nil, false
}

// MustMacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *Cluster) MustMacPool() *MacPool {
	if p.macPool == nil {
		panic("the macPool must not be nil, please use MacPool() function instead")
	}
	return p.macPool
}

func (p *Cluster) SetMaintenanceReasonRequired(attr bool) {
	p.maintenanceReasonRequired = &attr
}

func (p *Cluster) MaintenanceReasonRequired() (bool, bool) {
	if p.maintenanceReasonRequired != nil {
		return *p.maintenanceReasonRequired, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustMaintenanceReasonRequired() bool {
	if p.maintenanceReasonRequired == nil {
		panic("the maintenanceReasonRequired must not be nil, please use MaintenanceReasonRequired() function instead")
	}
	return *p.maintenanceReasonRequired
}

// SetManagementNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Cluster) SetManagementNetwork(attr *Network) {
	p.managementNetwork = attr
}

// ManagementNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Cluster) ManagementNetwork() (*Network, bool) {
	if p.managementNetwork != nil {
		return p.managementNetwork, true
	}
	return nil, false
}

// MustManagementNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Cluster) MustManagementNetwork() *Network {
	if p.managementNetwork == nil {
		panic("the managementNetwork must not be nil, please use ManagementNetwork() function instead")
	}
	return p.managementNetwork
}

// SetMemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *Cluster) SetMemoryPolicy(attr *MemoryPolicy) {
	p.memoryPolicy = attr
}

// MemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *Cluster) MemoryPolicy() (*MemoryPolicy, bool) {
	if p.memoryPolicy != nil {
		return p.memoryPolicy, true
	}
	return nil, false
}

// MustMemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *Cluster) MustMemoryPolicy() *MemoryPolicy {
	if p.memoryPolicy == nil {
		panic("the memoryPolicy must not be nil, please use MemoryPolicy() function instead")
	}
	return p.memoryPolicy
}

// SetMigration The type for migration options.
func (p *Cluster) SetMigration(attr *MigrationOptions) {
	p.migration = attr
}

// Migration The type for migration options.
func (p *Cluster) Migration() (*MigrationOptions, bool) {
	if p.migration != nil {
		return p.migration, true
	}
	return nil, false
}

// MustMigration The type for migration options.
func (p *Cluster) MustMigration() *MigrationOptions {
	if p.migration == nil {
		panic("the migration must not be nil, please use Migration() function instead")
	}
	return p.migration
}

func (p *Cluster) SetName(attr string) {
	p.name = &attr
}

func (p *Cluster) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Cluster) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Cluster) SetNetworkFilters(attr *NetworkFilterSlice) {
	p.networkFilters = attr
}

func (p *Cluster) NetworkFilters() (*NetworkFilterSlice, bool) {
	if p.networkFilters != nil {
		return p.networkFilters, true
	}
	return nil, false
}

func (p *Cluster) MustNetworkFilters() *NetworkFilterSlice {
	if p.networkFilters == nil {
		panic("the networkFilters must not be nil, please use NetworkFilters() function instead")
	}
	return p.networkFilters
}

func (p *Cluster) SetNetworks(attr *NetworkSlice) {
	p.networks = attr
}

func (p *Cluster) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *Cluster) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("the networks must not be nil, please use Networks() function instead")
	}
	return p.networks
}

func (p *Cluster) SetOptionalReason(attr bool) {
	p.optionalReason = &attr
}

func (p *Cluster) OptionalReason() (bool, bool) {
	if p.optionalReason != nil {
		return *p.optionalReason, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustOptionalReason() bool {
	if p.optionalReason == nil {
		panic("the optionalReason must not be nil, please use OptionalReason() function instead")
	}
	return *p.optionalReason
}

func (p *Cluster) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Cluster) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Cluster) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Cluster) SetRequiredRngSources(attr []RngSource) {
	p.requiredRngSources = attr
}

func (p *Cluster) RequiredRngSources() ([]RngSource, bool) {
	if p.requiredRngSources != nil {
		return p.requiredRngSources, true
	}
	return nil, false
}

func (p *Cluster) MustRequiredRngSources() []RngSource {
	if p.requiredRngSources == nil {
		panic("the requiredRngSources must not be nil, please use RequiredRngSources() function instead")
	}
	return p.requiredRngSources
}

func (p *Cluster) SetSchedulingPolicy(attr *SchedulingPolicy) {
	p.schedulingPolicy = attr
}

func (p *Cluster) SchedulingPolicy() (*SchedulingPolicy, bool) {
	if p.schedulingPolicy != nil {
		return p.schedulingPolicy, true
	}
	return nil, false
}

func (p *Cluster) MustSchedulingPolicy() *SchedulingPolicy {
	if p.schedulingPolicy == nil {
		panic("the schedulingPolicy must not be nil, please use SchedulingPolicy() function instead")
	}
	return p.schedulingPolicy
}

func (p *Cluster) SetSerialNumber(attr *SerialNumber) {
	p.serialNumber = attr
}

func (p *Cluster) SerialNumber() (*SerialNumber, bool) {
	if p.serialNumber != nil {
		return p.serialNumber, true
	}
	return nil, false
}

func (p *Cluster) MustSerialNumber() *SerialNumber {
	if p.serialNumber == nil {
		panic("the serialNumber must not be nil, please use SerialNumber() function instead")
	}
	return p.serialNumber
}

func (p *Cluster) SetSupportedVersions(attr *VersionSlice) {
	p.supportedVersions = attr
}

func (p *Cluster) SupportedVersions() (*VersionSlice, bool) {
	if p.supportedVersions != nil {
		return p.supportedVersions, true
	}
	return nil, false
}

func (p *Cluster) MustSupportedVersions() *VersionSlice {
	if p.supportedVersions == nil {
		panic("the supportedVersions must not be nil, please use SupportedVersions() function instead")
	}
	return p.supportedVersions
}

// SetSwitchType Describes all switch types supported by the Manager.
func (p *Cluster) SetSwitchType(attr SwitchType) {
	p.switchType = &attr
}

// SwitchType Describes all switch types supported by the Manager.
func (p *Cluster) SwitchType() (SwitchType, bool) {
	if p.switchType != nil {
		return *p.switchType, true
	}
	var zero SwitchType
	return zero, false
}

// MustSwitchType Describes all switch types supported by the Manager.
func (p *Cluster) MustSwitchType() SwitchType {
	if p.switchType == nil {
		panic("the switchType must not be nil, please use SwitchType() function instead")
	}
	return *p.switchType
}

func (p *Cluster) SetThreadsAsCores(attr bool) {
	p.threadsAsCores = &attr
}

func (p *Cluster) ThreadsAsCores() (bool, bool) {
	if p.threadsAsCores != nil {
		return *p.threadsAsCores, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustThreadsAsCores() bool {
	if p.threadsAsCores == nil {
		panic("the threadsAsCores must not be nil, please use ThreadsAsCores() function instead")
	}
	return *p.threadsAsCores
}

func (p *Cluster) SetTrustedService(attr bool) {
	p.trustedService = &attr
}

func (p *Cluster) TrustedService() (bool, bool) {
	if p.trustedService != nil {
		return *p.trustedService, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustTrustedService() bool {
	if p.trustedService == nil {
		panic("the trustedService must not be nil, please use TrustedService() function instead")
	}
	return *p.trustedService
}

func (p *Cluster) SetTunnelMigration(attr bool) {
	p.tunnelMigration = &attr
}

func (p *Cluster) TunnelMigration() (bool, bool) {
	if p.tunnelMigration != nil {
		return *p.tunnelMigration, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustTunnelMigration() bool {
	if p.tunnelMigration == nil {
		panic("the tunnelMigration must not be nil, please use TunnelMigration() function instead")
	}
	return *p.tunnelMigration
}

func (p *Cluster) SetVersion(attr *Version) {
	p.version = attr
}

func (p *Cluster) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *Cluster) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

func (p *Cluster) SetVirtService(attr bool) {
	p.virtService = &attr
}

func (p *Cluster) VirtService() (bool, bool) {
	if p.virtService != nil {
		return *p.virtService, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustVirtService() bool {
	if p.virtService == nil {
		panic("the virtService must not be nil, please use VirtService() function instead")
	}
	return *p.virtService
}

func (p *Cluster) SetVncEncryption(attr bool) {
	p.vncEncryption = &attr
}

func (p *Cluster) VncEncryption() (bool, bool) {
	if p.vncEncryption != nil {
		return *p.vncEncryption, true
	}
	var zero bool
	return zero, false
}

func (p *Cluster) MustVncEncryption() bool {
	if p.vncEncryption == nil {
		panic("the vncEncryption must not be nil, please use VncEncryption() function instead")
	}
	return *p.vncEncryption
}

// ClusterFeature Type represents an additional feature that is available at a cluster level.
type ClusterFeature struct {
	Struct
	clusterLevel *ClusterLevel
	comment      *string
	description  *string
	id           *string
	name         *string
}

// SetClusterLevel Describes the capabilities supported by a specific cluster level.
func (p *ClusterFeature) SetClusterLevel(attr *ClusterLevel) {
	p.clusterLevel = attr
}

// ClusterLevel Describes the capabilities supported by a specific cluster level.
func (p *ClusterFeature) ClusterLevel() (*ClusterLevel, bool) {
	if p.clusterLevel != nil {
		return p.clusterLevel, true
	}
	return nil, false
}

// MustClusterLevel Describes the capabilities supported by a specific cluster level.
func (p *ClusterFeature) MustClusterLevel() *ClusterLevel {
	if p.clusterLevel == nil {
		panic("the clusterLevel must not be nil, please use ClusterLevel() function instead")
	}
	return p.clusterLevel
}

func (p *ClusterFeature) SetComment(attr string) {
	p.comment = &attr
}

func (p *ClusterFeature) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ClusterFeature) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ClusterFeature) SetDescription(attr string) {
	p.description = &attr
}

func (p *ClusterFeature) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ClusterFeature) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *ClusterFeature) SetId(attr string) {
	p.id = &attr
}

func (p *ClusterFeature) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ClusterFeature) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ClusterFeature) SetName(attr string) {
	p.name = &attr
}

func (p *ClusterFeature) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ClusterFeature) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// ClusterLevel Describes the capabilities supported by a specific cluster level.
type ClusterLevel struct {
	Struct
	clusterFeatures *ClusterFeatureSlice
	comment         *string
	cpuTypes        *CpuTypeSlice
	description     *string
	id              *string
	name            *string
	permits         *PermitSlice
}

func (p *ClusterLevel) SetClusterFeatures(attr *ClusterFeatureSlice) {
	p.clusterFeatures = attr
}

func (p *ClusterLevel) ClusterFeatures() (*ClusterFeatureSlice, bool) {
	if p.clusterFeatures != nil {
		return p.clusterFeatures, true
	}
	return nil, false
}

func (p *ClusterLevel) MustClusterFeatures() *ClusterFeatureSlice {
	if p.clusterFeatures == nil {
		panic("the clusterFeatures must not be nil, please use ClusterFeatures() function instead")
	}
	return p.clusterFeatures
}

func (p *ClusterLevel) SetComment(attr string) {
	p.comment = &attr
}

func (p *ClusterLevel) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ClusterLevel) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ClusterLevel) SetCpuTypes(attr *CpuTypeSlice) {
	p.cpuTypes = attr
}

func (p *ClusterLevel) CpuTypes() (*CpuTypeSlice, bool) {
	if p.cpuTypes != nil {
		return p.cpuTypes, true
	}
	return nil, false
}

func (p *ClusterLevel) MustCpuTypes() *CpuTypeSlice {
	if p.cpuTypes == nil {
		panic("the cpuTypes must not be nil, please use CpuTypes() function instead")
	}
	return p.cpuTypes
}

func (p *ClusterLevel) SetDescription(attr string) {
	p.description = &attr
}

func (p *ClusterLevel) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ClusterLevel) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *ClusterLevel) SetId(attr string) {
	p.id = &attr
}

func (p *ClusterLevel) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ClusterLevel) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ClusterLevel) SetName(attr string) {
	p.name = &attr
}

func (p *ClusterLevel) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ClusterLevel) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ClusterLevel) SetPermits(attr *PermitSlice) {
	p.permits = attr
}

func (p *ClusterLevel) Permits() (*PermitSlice, bool) {
	if p.permits != nil {
		return p.permits, true
	}
	return nil, false
}

func (p *ClusterLevel) MustPermits() *PermitSlice {
	if p.permits == nil {
		panic("the permits must not be nil, please use Permits() function instead")
	}
	return p.permits
}

type Configuration struct {
	Struct
	data  *string
	type_ *ConfigurationType
}

func (p *Configuration) SetData(attr string) {
	p.data = &attr
}

func (p *Configuration) Data() (string, bool) {
	if p.data != nil {
		return *p.data, true
	}
	var zero string
	return zero, false
}

func (p *Configuration) MustData() string {
	if p.data == nil {
		panic("the data must not be nil, please use Data() function instead")
	}
	return *p.data
}

// SetType Configuration format types.
func (p *Configuration) SetType(attr ConfigurationType) {
	p.type_ = &attr
}

// Type Configuration format types.
func (p *Configuration) Type() (ConfigurationType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero ConfigurationType
	return zero, false
}

// MustType Configuration format types.
func (p *Configuration) MustType() ConfigurationType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// Console Representation for serial console device.
type Console struct {
	Struct
	enabled *bool
}

func (p *Console) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *Console) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *Console) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

type Core struct {
	Struct
	index  *int64
	socket *int64
}

func (p *Core) SetIndex(attr int64) {
	p.index = &attr
}

func (p *Core) Index() (int64, bool) {
	if p.index != nil {
		return *p.index, true
	}
	var zero int64
	return zero, false
}

func (p *Core) MustIndex() int64 {
	if p.index == nil {
		panic("the index must not be nil, please use Index() function instead")
	}
	return *p.index
}

func (p *Core) SetSocket(attr int64) {
	p.socket = &attr
}

func (p *Core) Socket() (int64, bool) {
	if p.socket != nil {
		return *p.socket, true
	}
	var zero int64
	return zero, false
}

func (p *Core) MustSocket() int64 {
	if p.socket == nil {
		panic("the socket must not be nil, please use Socket() function instead")
	}
	return *p.socket
}

type Cpu struct {
	Struct
	architecture *Architecture
	cores        *CoreSlice
	cpuTune      *CpuTune
	level        *int64
	mode         *CpuMode
	name         *string
	speed        *float64
	topology     *CpuTopology
	type_        *string
}

func (p *Cpu) SetArchitecture(attr Architecture) {
	p.architecture = &attr
}

func (p *Cpu) Architecture() (Architecture, bool) {
	if p.architecture != nil {
		return *p.architecture, true
	}
	var zero Architecture
	return zero, false
}

func (p *Cpu) MustArchitecture() Architecture {
	if p.architecture == nil {
		panic("the architecture must not be nil, please use Architecture() function instead")
	}
	return *p.architecture
}

func (p *Cpu) SetCores(attr *CoreSlice) {
	p.cores = attr
}

func (p *Cpu) Cores() (*CoreSlice, bool) {
	if p.cores != nil {
		return p.cores, true
	}
	return nil, false
}

func (p *Cpu) MustCores() *CoreSlice {
	if p.cores == nil {
		panic("the cores must not be nil, please use Cores() function instead")
	}
	return p.cores
}

func (p *Cpu) SetCpuTune(attr *CpuTune) {
	p.cpuTune = attr
}

func (p *Cpu) CpuTune() (*CpuTune, bool) {
	if p.cpuTune != nil {
		return p.cpuTune, true
	}
	return nil, false
}

func (p *Cpu) MustCpuTune() *CpuTune {
	if p.cpuTune == nil {
		panic("the cpuTune must not be nil, please use CpuTune() function instead")
	}
	return p.cpuTune
}

func (p *Cpu) SetLevel(attr int64) {
	p.level = &attr
}

func (p *Cpu) Level() (int64, bool) {
	if p.level != nil {
		return *p.level, true
	}
	var zero int64
	return zero, false
}

func (p *Cpu) MustLevel() int64 {
	if p.level == nil {
		panic("the level must not be nil, please use Level() function instead")
	}
	return *p.level
}

func (p *Cpu) SetMode(attr CpuMode) {
	p.mode = &attr
}

func (p *Cpu) Mode() (CpuMode, bool) {
	if p.mode != nil {
		return *p.mode, true
	}
	var zero CpuMode
	return zero, false
}

func (p *Cpu) MustMode() CpuMode {
	if p.mode == nil {
		panic("the mode must not be nil, please use Mode() function instead")
	}
	return *p.mode
}

func (p *Cpu) SetName(attr string) {
	p.name = &attr
}

func (p *Cpu) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Cpu) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Cpu) SetSpeed(attr float64) {
	p.speed = &attr
}

func (p *Cpu) Speed() (float64, bool) {
	if p.speed != nil {
		return *p.speed, true
	}
	var zero float64
	return zero, false
}

func (p *Cpu) MustSpeed() float64 {
	if p.speed == nil {
		panic("the speed must not be nil, please use Speed() function instead")
	}
	return *p.speed
}

func (p *Cpu) SetTopology(attr *CpuTopology) {
	p.topology = attr
}

func (p *Cpu) Topology() (*CpuTopology, bool) {
	if p.topology != nil {
		return p.topology, true
	}
	return nil, false
}

func (p *Cpu) MustTopology() *CpuTopology {
	if p.topology == nil {
		panic("the topology must not be nil, please use Topology() function instead")
	}
	return p.topology
}

func (p *Cpu) SetType(attr string) {
	p.type_ = &attr
}

func (p *Cpu) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *Cpu) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type CpuProfile struct {
	Struct
	cluster     *Cluster
	comment     *string
	description *string
	id          *string
	name        *string
	permissions *PermissionSlice
	qos         *Qos
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *CpuProfile) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *CpuProfile) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *CpuProfile) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *CpuProfile) SetComment(attr string) {
	p.comment = &attr
}

func (p *CpuProfile) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *CpuProfile) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *CpuProfile) SetDescription(attr string) {
	p.description = &attr
}

func (p *CpuProfile) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *CpuProfile) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *CpuProfile) SetId(attr string) {
	p.id = &attr
}

func (p *CpuProfile) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *CpuProfile) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *CpuProfile) SetName(attr string) {
	p.name = &attr
}

func (p *CpuProfile) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *CpuProfile) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *CpuProfile) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *CpuProfile) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *CpuProfile) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

// SetQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *CpuProfile) SetQos(attr *Qos) {
	p.qos = attr
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *CpuProfile) Qos() (*Qos, bool) {
	if p.qos != nil {
		return p.qos, true
	}
	return nil, false
}

// MustQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *CpuProfile) MustQos() *Qos {
	if p.qos == nil {
		panic("the qos must not be nil, please use Qos() function instead")
	}
	return p.qos
}

type CpuTopology struct {
	Struct
	cores   *int64
	sockets *int64
	threads *int64
}

func (p *CpuTopology) SetCores(attr int64) {
	p.cores = &attr
}

func (p *CpuTopology) Cores() (int64, bool) {
	if p.cores != nil {
		return *p.cores, true
	}
	var zero int64
	return zero, false
}

func (p *CpuTopology) MustCores() int64 {
	if p.cores == nil {
		panic("the cores must not be nil, please use Cores() function instead")
	}
	return *p.cores
}

func (p *CpuTopology) SetSockets(attr int64) {
	p.sockets = &attr
}

func (p *CpuTopology) Sockets() (int64, bool) {
	if p.sockets != nil {
		return *p.sockets, true
	}
	var zero int64
	return zero, false
}

func (p *CpuTopology) MustSockets() int64 {
	if p.sockets == nil {
		panic("the sockets must not be nil, please use Sockets() function instead")
	}
	return *p.sockets
}

func (p *CpuTopology) SetThreads(attr int64) {
	p.threads = &attr
}

func (p *CpuTopology) Threads() (int64, bool) {
	if p.threads != nil {
		return *p.threads, true
	}
	var zero int64
	return zero, false
}

func (p *CpuTopology) MustThreads() int64 {
	if p.threads == nil {
		panic("the threads must not be nil, please use Threads() function instead")
	}
	return *p.threads
}

type CpuTune struct {
	Struct
	vcpuPins *VcpuPinSlice
}

func (p *CpuTune) SetVcpuPins(attr *VcpuPinSlice) {
	p.vcpuPins = attr
}

func (p *CpuTune) VcpuPins() (*VcpuPinSlice, bool) {
	if p.vcpuPins != nil {
		return p.vcpuPins, true
	}
	return nil, false
}

func (p *CpuTune) MustVcpuPins() *VcpuPinSlice {
	if p.vcpuPins == nil {
		panic("the vcpuPins must not be nil, please use VcpuPins() function instead")
	}
	return p.vcpuPins
}

// CpuType Describes a supported CPU type.
type CpuType struct {
	Struct
	architecture *Architecture
	level        *int64
	name         *string
}

func (p *CpuType) SetArchitecture(attr Architecture) {
	p.architecture = &attr
}

func (p *CpuType) Architecture() (Architecture, bool) {
	if p.architecture != nil {
		return *p.architecture, true
	}
	var zero Architecture
	return zero, false
}

func (p *CpuType) MustArchitecture() Architecture {
	if p.architecture == nil {
		panic("the architecture must not be nil, please use Architecture() function instead")
	}
	return *p.architecture
}

func (p *CpuType) SetLevel(attr int64) {
	p.level = &attr
}

func (p *CpuType) Level() (int64, bool) {
	if p.level != nil {
		return *p.level, true
	}
	var zero int64
	return zero, false
}

func (p *CpuType) MustLevel() int64 {
	if p.level == nil {
		panic("the level must not be nil, please use Level() function instead")
	}
	return *p.level
}

func (p *CpuType) SetName(attr string) {
	p.name = &attr
}

func (p *CpuType) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *CpuType) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// CustomProperty Custom property representation.
type CustomProperty struct {
	Struct
	name   *string
	regexp *string
	value  *string
}

func (p *CustomProperty) SetName(attr string) {
	p.name = &attr
}

func (p *CustomProperty) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *CustomProperty) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *CustomProperty) SetRegexp(attr string) {
	p.regexp = &attr
}

func (p *CustomProperty) Regexp() (string, bool) {
	if p.regexp != nil {
		return *p.regexp, true
	}
	var zero string
	return zero, false
}

func (p *CustomProperty) MustRegexp() string {
	if p.regexp == nil {
		panic("the regexp must not be nil, please use Regexp() function instead")
	}
	return *p.regexp
}

func (p *CustomProperty) SetValue(attr string) {
	p.value = &attr
}

func (p *CustomProperty) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *CustomProperty) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

type DataCenter struct {
	Struct
	clusters          *ClusterSlice
	comment           *string
	description       *string
	id                *string
	iscsiBonds        *IscsiBondSlice
	local             *bool
	macPool           *MacPool
	name              *string
	networks          *NetworkSlice
	permissions       *PermissionSlice
	qoss              *QosSlice
	quotaMode         *QuotaModeType
	quotas            *QuotaSlice
	status            *DataCenterStatus
	storageDomains    *StorageDomainSlice
	storageFormat     *StorageFormat
	supportedVersions *VersionSlice
	version           *Version
}

func (p *DataCenter) SetClusters(attr *ClusterSlice) {
	p.clusters = attr
}

func (p *DataCenter) Clusters() (*ClusterSlice, bool) {
	if p.clusters != nil {
		return p.clusters, true
	}
	return nil, false
}

func (p *DataCenter) MustClusters() *ClusterSlice {
	if p.clusters == nil {
		panic("the clusters must not be nil, please use Clusters() function instead")
	}
	return p.clusters
}

func (p *DataCenter) SetComment(attr string) {
	p.comment = &attr
}

func (p *DataCenter) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *DataCenter) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *DataCenter) SetDescription(attr string) {
	p.description = &attr
}

func (p *DataCenter) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *DataCenter) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *DataCenter) SetId(attr string) {
	p.id = &attr
}

func (p *DataCenter) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *DataCenter) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *DataCenter) SetIscsiBonds(attr *IscsiBondSlice) {
	p.iscsiBonds = attr
}

func (p *DataCenter) IscsiBonds() (*IscsiBondSlice, bool) {
	if p.iscsiBonds != nil {
		return p.iscsiBonds, true
	}
	return nil, false
}

func (p *DataCenter) MustIscsiBonds() *IscsiBondSlice {
	if p.iscsiBonds == nil {
		panic("the iscsiBonds must not be nil, please use IscsiBonds() function instead")
	}
	return p.iscsiBonds
}

func (p *DataCenter) SetLocal(attr bool) {
	p.local = &attr
}

func (p *DataCenter) Local() (bool, bool) {
	if p.local != nil {
		return *p.local, true
	}
	var zero bool
	return zero, false
}

func (p *DataCenter) MustLocal() bool {
	if p.local == nil {
		panic("the local must not be nil, please use Local() function instead")
	}
	return *p.local
}

// SetMacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *DataCenter) SetMacPool(attr *MacPool) {
	p.macPool = attr
}

// MacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *DataCenter) MacPool() (*MacPool, bool) {
	if p.macPool != nil {
		return p.macPool, true
	}
	return nil, false
}

// MustMacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
func (p *DataCenter) MustMacPool() *MacPool {
	if p.macPool == nil {
		panic("the macPool must not be nil, please use MacPool() function instead")
	}
	return p.macPool
}

func (p *DataCenter) SetName(attr string) {
	p.name = &attr
}

func (p *DataCenter) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *DataCenter) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *DataCenter) SetNetworks(attr *NetworkSlice) {
	p.networks = attr
}

func (p *DataCenter) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *DataCenter) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("the networks must not be nil, please use Networks() function instead")
	}
	return p.networks
}

func (p *DataCenter) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *DataCenter) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *DataCenter) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *DataCenter) SetQoss(attr *QosSlice) {
	p.qoss = attr
}

func (p *DataCenter) Qoss() (*QosSlice, bool) {
	if p.qoss != nil {
		return p.qoss, true
	}
	return nil, false
}

func (p *DataCenter) MustQoss() *QosSlice {
	if p.qoss == nil {
		panic("the qoss must not be nil, please use Qoss() function instead")
	}
	return p.qoss
}

func (p *DataCenter) SetQuotaMode(attr QuotaModeType) {
	p.quotaMode = &attr
}

func (p *DataCenter) QuotaMode() (QuotaModeType, bool) {
	if p.quotaMode != nil {
		return *p.quotaMode, true
	}
	var zero QuotaModeType
	return zero, false
}

func (p *DataCenter) MustQuotaMode() QuotaModeType {
	if p.quotaMode == nil {
		panic("the quotaMode must not be nil, please use QuotaMode() function instead")
	}
	return *p.quotaMode
}

func (p *DataCenter) SetQuotas(attr *QuotaSlice) {
	p.quotas = attr
}

func (p *DataCenter) Quotas() (*QuotaSlice, bool) {
	if p.quotas != nil {
		return p.quotas, true
	}
	return nil, false
}

func (p *DataCenter) MustQuotas() *QuotaSlice {
	if p.quotas == nil {
		panic("the quotas must not be nil, please use Quotas() function instead")
	}
	return p.quotas
}

func (p *DataCenter) SetStatus(attr DataCenterStatus) {
	p.status = &attr
}

func (p *DataCenter) Status() (DataCenterStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero DataCenterStatus
	return zero, false
}

func (p *DataCenter) MustStatus() DataCenterStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *DataCenter) SetStorageDomains(attr *StorageDomainSlice) {
	p.storageDomains = attr
}

func (p *DataCenter) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *DataCenter) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("the storageDomains must not be nil, please use StorageDomains() function instead")
	}
	return p.storageDomains
}

// SetStorageFormat Type which represents a format of <<types/storage_domain, storage domain>>.
func (p *DataCenter) SetStorageFormat(attr StorageFormat) {
	p.storageFormat = &attr
}

// StorageFormat Type which represents a format of <<types/storage_domain, storage domain>>.
func (p *DataCenter) StorageFormat() (StorageFormat, bool) {
	if p.storageFormat != nil {
		return *p.storageFormat, true
	}
	var zero StorageFormat
	return zero, false
}

// MustStorageFormat Type which represents a format of <<types/storage_domain, storage domain>>.
func (p *DataCenter) MustStorageFormat() StorageFormat {
	if p.storageFormat == nil {
		panic("the storageFormat must not be nil, please use StorageFormat() function instead")
	}
	return *p.storageFormat
}

func (p *DataCenter) SetSupportedVersions(attr *VersionSlice) {
	p.supportedVersions = attr
}

func (p *DataCenter) SupportedVersions() (*VersionSlice, bool) {
	if p.supportedVersions != nil {
		return p.supportedVersions, true
	}
	return nil, false
}

func (p *DataCenter) MustSupportedVersions() *VersionSlice {
	if p.supportedVersions == nil {
		panic("the supportedVersions must not be nil, please use SupportedVersions() function instead")
	}
	return p.supportedVersions
}

func (p *DataCenter) SetVersion(attr *Version) {
	p.version = attr
}

func (p *DataCenter) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *DataCenter) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// Device A device wraps links to potential parents of a device.
type Device struct {
	Struct
	comment      *string
	description  *string
	id           *string
	instanceType *InstanceType
	name         *string
	template     *Template
	vm           *Vm
	vms          *VmSlice
}

func (p *Device) SetComment(attr string) {
	p.comment = &attr
}

func (p *Device) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Device) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Device) SetDescription(attr string) {
	p.description = &attr
}

func (p *Device) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Device) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Device) SetId(attr string) {
	p.id = &attr
}

func (p *Device) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Device) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Device) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Device) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Device) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *Device) SetName(attr string) {
	p.name = &attr
}

func (p *Device) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Device) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Device) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Device) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Device) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetVm Represents a virtual machine.
func (p *Device) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Device) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Device) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *Device) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Device) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Device) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

// Disk Represents a virtual disk device.
type Disk struct {
	Struct
	active              *bool
	actualSize          *int64
	alias               *string
	backup              *DiskBackup
	backupMode          *DiskBackupMode
	bootable            *bool
	comment             *string
	contentType         *DiskContentType
	description         *string
	diskProfile         *DiskProfile
	diskSnapshots       *DiskSnapshotSlice
	externalDisk        *string
	format              *DiskFormat
	id                  *string
	imageId             *string
	initialSize         *int64
	instanceType        *InstanceType
	interface_          *DiskInterface
	logicalName         *string
	lunStorage          *HostStorage
	name                *string
	openstackVolumeType *OpenStackVolumeType
	permissions         *PermissionSlice
	propagateErrors     *bool
	provisionedSize     *int64
	qcowVersion         *QcowVersion
	quota               *Quota
	readOnly            *bool
	sgio                *ScsiGenericIO
	shareable           *bool
	snapshot            *Snapshot
	sparse              *bool
	statistics          *StatisticSlice
	status              *DiskStatus
	storageDomain       *StorageDomain
	storageDomains      *StorageDomainSlice
	storageType         *DiskStorageType
	template            *Template
	totalSize           *int64
	usesScsiReservation *bool
	vm                  *Vm
	vms                 *VmSlice
	wipeAfterDelete     *bool
}

func (p *Disk) SetActive(attr bool) {
	p.active = &attr
}

func (p *Disk) Active() (bool, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustActive() bool {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *Disk) SetActualSize(attr int64) {
	p.actualSize = &attr
}

func (p *Disk) ActualSize() (int64, bool) {
	if p.actualSize != nil {
		return *p.actualSize, true
	}
	var zero int64
	return zero, false
}

func (p *Disk) MustActualSize() int64 {
	if p.actualSize == nil {
		panic("the actualSize must not be nil, please use ActualSize() function instead")
	}
	return *p.actualSize
}

func (p *Disk) SetAlias(attr string) {
	p.alias = &attr
}

func (p *Disk) Alias() (string, bool) {
	if p.alias != nil {
		return *p.alias, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustAlias() string {
	if p.alias == nil {
		panic("the alias must not be nil, please use Alias() function instead")
	}
	return *p.alias
}

// SetBackup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *Disk) SetBackup(attr DiskBackup) {
	p.backup = &attr
}

// Backup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *Disk) Backup() (DiskBackup, bool) {
	if p.backup != nil {
		return *p.backup, true
	}
	var zero DiskBackup
	return zero, false
}

// MustBackup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *Disk) MustBackup() DiskBackup {
	if p.backup == nil {
		panic("the backup must not be nil, please use Backup() function instead")
	}
	return *p.backup
}

// SetBackupMode Represents an enumeration of backup modes
func (p *Disk) SetBackupMode(attr DiskBackupMode) {
	p.backupMode = &attr
}

// BackupMode Represents an enumeration of backup modes
func (p *Disk) BackupMode() (DiskBackupMode, bool) {
	if p.backupMode != nil {
		return *p.backupMode, true
	}
	var zero DiskBackupMode
	return zero, false
}

// MustBackupMode Represents an enumeration of backup modes
func (p *Disk) MustBackupMode() DiskBackupMode {
	if p.backupMode == nil {
		panic("the backupMode must not be nil, please use BackupMode() function instead")
	}
	return *p.backupMode
}

func (p *Disk) SetBootable(attr bool) {
	p.bootable = &attr
}

func (p *Disk) Bootable() (bool, bool) {
	if p.bootable != nil {
		return *p.bootable, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustBootable() bool {
	if p.bootable == nil {
		panic("the bootable must not be nil, please use Bootable() function instead")
	}
	return *p.bootable
}

func (p *Disk) SetComment(attr string) {
	p.comment = &attr
}

func (p *Disk) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

// SetContentType The actual content residing on the disk.
func (p *Disk) SetContentType(attr DiskContentType) {
	p.contentType = &attr
}

// ContentType The actual content residing on the disk.
func (p *Disk) ContentType() (DiskContentType, bool) {
	if p.contentType != nil {
		return *p.contentType, true
	}
	var zero DiskContentType
	return zero, false
}

// MustContentType The actual content residing on the disk.
func (p *Disk) MustContentType() DiskContentType {
	if p.contentType == nil {
		panic("the contentType must not be nil, please use ContentType() function instead")
	}
	return *p.contentType
}

func (p *Disk) SetDescription(attr string) {
	p.description = &attr
}

func (p *Disk) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Disk) SetDiskProfile(attr *DiskProfile) {
	p.diskProfile = attr
}

func (p *Disk) DiskProfile() (*DiskProfile, bool) {
	if p.diskProfile != nil {
		return p.diskProfile, true
	}
	return nil, false
}

func (p *Disk) MustDiskProfile() *DiskProfile {
	if p.diskProfile == nil {
		panic("the diskProfile must not be nil, please use DiskProfile() function instead")
	}
	return p.diskProfile
}

func (p *Disk) SetDiskSnapshots(attr *DiskSnapshotSlice) {
	p.diskSnapshots = attr
}

func (p *Disk) DiskSnapshots() (*DiskSnapshotSlice, bool) {
	if p.diskSnapshots != nil {
		return p.diskSnapshots, true
	}
	return nil, false
}

func (p *Disk) MustDiskSnapshots() *DiskSnapshotSlice {
	if p.diskSnapshots == nil {
		panic("the diskSnapshots must not be nil, please use DiskSnapshots() function instead")
	}
	return p.diskSnapshots
}

func (p *Disk) SetExternalDisk(attr string) {
	p.externalDisk = &attr
}

func (p *Disk) ExternalDisk() (string, bool) {
	if p.externalDisk != nil {
		return *p.externalDisk, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustExternalDisk() string {
	if p.externalDisk == nil {
		panic("the externalDisk must not be nil, please use ExternalDisk() function instead")
	}
	return *p.externalDisk
}

// SetFormat The underlying storage format of disks.
func (p *Disk) SetFormat(attr DiskFormat) {
	p.format = &attr
}

// Format The underlying storage format of disks.
func (p *Disk) Format() (DiskFormat, bool) {
	if p.format != nil {
		return *p.format, true
	}
	var zero DiskFormat
	return zero, false
}

// MustFormat The underlying storage format of disks.
func (p *Disk) MustFormat() DiskFormat {
	if p.format == nil {
		panic("the format must not be nil, please use Format() function instead")
	}
	return *p.format
}

func (p *Disk) SetId(attr string) {
	p.id = &attr
}

func (p *Disk) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Disk) SetImageId(attr string) {
	p.imageId = &attr
}

func (p *Disk) ImageId() (string, bool) {
	if p.imageId != nil {
		return *p.imageId, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustImageId() string {
	if p.imageId == nil {
		panic("the imageId must not be nil, please use ImageId() function instead")
	}
	return *p.imageId
}

func (p *Disk) SetInitialSize(attr int64) {
	p.initialSize = &attr
}

func (p *Disk) InitialSize() (int64, bool) {
	if p.initialSize != nil {
		return *p.initialSize, true
	}
	var zero int64
	return zero, false
}

func (p *Disk) MustInitialSize() int64 {
	if p.initialSize == nil {
		panic("the initialSize must not be nil, please use InitialSize() function instead")
	}
	return *p.initialSize
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Disk) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Disk) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Disk) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

// SetInterface The underlying storage interface of disks communication with controller.
func (p *Disk) SetInterface(attr DiskInterface) {
	p.interface_ = &attr
}

// Interface The underlying storage interface of disks communication with controller.
func (p *Disk) Interface() (DiskInterface, bool) {
	if p.interface_ != nil {
		return *p.interface_, true
	}
	var zero DiskInterface
	return zero, false
}

// MustInterface The underlying storage interface of disks communication with controller.
func (p *Disk) MustInterface() DiskInterface {
	if p.interface_ == nil {
		panic("the interface_ must not be nil, please use Interface() function instead")
	}
	return *p.interface_
}

func (p *Disk) SetLogicalName(attr string) {
	p.logicalName = &attr
}

func (p *Disk) LogicalName() (string, bool) {
	if p.logicalName != nil {
		return *p.logicalName, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustLogicalName() string {
	if p.logicalName == nil {
		panic("the logicalName must not be nil, please use LogicalName() function instead")
	}
	return *p.logicalName
}

func (p *Disk) SetLunStorage(attr *HostStorage) {
	p.lunStorage = attr
}

func (p *Disk) LunStorage() (*HostStorage, bool) {
	if p.lunStorage != nil {
		return p.lunStorage, true
	}
	return nil, false
}

func (p *Disk) MustLunStorage() *HostStorage {
	if p.lunStorage == nil {
		panic("the lunStorage must not be nil, please use LunStorage() function instead")
	}
	return p.lunStorage
}

func (p *Disk) SetName(attr string) {
	p.name = &attr
}

func (p *Disk) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Disk) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Disk) SetOpenstackVolumeType(attr *OpenStackVolumeType) {
	p.openstackVolumeType = attr
}

func (p *Disk) OpenstackVolumeType() (*OpenStackVolumeType, bool) {
	if p.openstackVolumeType != nil {
		return p.openstackVolumeType, true
	}
	return nil, false
}

func (p *Disk) MustOpenstackVolumeType() *OpenStackVolumeType {
	if p.openstackVolumeType == nil {
		panic("the openstackVolumeType must not be nil, please use OpenstackVolumeType() function instead")
	}
	return p.openstackVolumeType
}

func (p *Disk) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Disk) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Disk) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Disk) SetPropagateErrors(attr bool) {
	p.propagateErrors = &attr
}

func (p *Disk) PropagateErrors() (bool, bool) {
	if p.propagateErrors != nil {
		return *p.propagateErrors, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustPropagateErrors() bool {
	if p.propagateErrors == nil {
		panic("the propagateErrors must not be nil, please use PropagateErrors() function instead")
	}
	return *p.propagateErrors
}

func (p *Disk) SetProvisionedSize(attr int64) {
	p.provisionedSize = &attr
}

func (p *Disk) ProvisionedSize() (int64, bool) {
	if p.provisionedSize != nil {
		return *p.provisionedSize, true
	}
	var zero int64
	return zero, false
}

func (p *Disk) MustProvisionedSize() int64 {
	if p.provisionedSize == nil {
		panic("the provisionedSize must not be nil, please use ProvisionedSize() function instead")
	}
	return *p.provisionedSize
}

// SetQcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *Disk) SetQcowVersion(attr QcowVersion) {
	p.qcowVersion = &attr
}

// QcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *Disk) QcowVersion() (QcowVersion, bool) {
	if p.qcowVersion != nil {
		return *p.qcowVersion, true
	}
	var zero QcowVersion
	return zero, false
}

// MustQcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *Disk) MustQcowVersion() QcowVersion {
	if p.qcowVersion == nil {
		panic("the qcowVersion must not be nil, please use QcowVersion() function instead")
	}
	return *p.qcowVersion
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *Disk) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *Disk) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *Disk) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

func (p *Disk) SetReadOnly(attr bool) {
	p.readOnly = &attr
}

func (p *Disk) ReadOnly() (bool, bool) {
	if p.readOnly != nil {
		return *p.readOnly, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustReadOnly() bool {
	if p.readOnly == nil {
		panic("the readOnly must not be nil, please use ReadOnly() function instead")
	}
	return *p.readOnly
}

// SetSgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *Disk) SetSgio(attr ScsiGenericIO) {
	p.sgio = &attr
}

// Sgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *Disk) Sgio() (ScsiGenericIO, bool) {
	if p.sgio != nil {
		return *p.sgio, true
	}
	var zero ScsiGenericIO
	return zero, false
}

// MustSgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *Disk) MustSgio() ScsiGenericIO {
	if p.sgio == nil {
		panic("the sgio must not be nil, please use Sgio() function instead")
	}
	return *p.sgio
}

func (p *Disk) SetShareable(attr bool) {
	p.shareable = &attr
}

func (p *Disk) Shareable() (bool, bool) {
	if p.shareable != nil {
		return *p.shareable, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustShareable() bool {
	if p.shareable == nil {
		panic("the shareable must not be nil, please use Shareable() function instead")
	}
	return *p.shareable
}

// SetSnapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *Disk) SetSnapshot(attr *Snapshot) {
	p.snapshot = attr
}

// Snapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *Disk) Snapshot() (*Snapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

// MustSnapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *Disk) MustSnapshot() *Snapshot {
	if p.snapshot == nil {
		panic("the snapshot must not be nil, please use Snapshot() function instead")
	}
	return p.snapshot
}

func (p *Disk) SetSparse(attr bool) {
	p.sparse = &attr
}

func (p *Disk) Sparse() (bool, bool) {
	if p.sparse != nil {
		return *p.sparse, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustSparse() bool {
	if p.sparse == nil {
		panic("the sparse must not be nil, please use Sparse() function instead")
	}
	return *p.sparse
}

func (p *Disk) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *Disk) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *Disk) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// SetStatus Current status representation for disk.
func (p *Disk) SetStatus(attr DiskStatus) {
	p.status = &attr
}

// Status Current status representation for disk.
func (p *Disk) Status() (DiskStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero DiskStatus
	return zero, false
}

// MustStatus Current status representation for disk.
func (p *Disk) MustStatus() DiskStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Disk) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Disk) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Disk) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

func (p *Disk) SetStorageDomains(attr *StorageDomainSlice) {
	p.storageDomains = attr
}

func (p *Disk) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *Disk) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("the storageDomains must not be nil, please use StorageDomains() function instead")
	}
	return p.storageDomains
}

func (p *Disk) SetStorageType(attr DiskStorageType) {
	p.storageType = &attr
}

func (p *Disk) StorageType() (DiskStorageType, bool) {
	if p.storageType != nil {
		return *p.storageType, true
	}
	var zero DiskStorageType
	return zero, false
}

func (p *Disk) MustStorageType() DiskStorageType {
	if p.storageType == nil {
		panic("the storageType must not be nil, please use StorageType() function instead")
	}
	return *p.storageType
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Disk) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Disk) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Disk) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *Disk) SetTotalSize(attr int64) {
	p.totalSize = &attr
}

func (p *Disk) TotalSize() (int64, bool) {
	if p.totalSize != nil {
		return *p.totalSize, true
	}
	var zero int64
	return zero, false
}

func (p *Disk) MustTotalSize() int64 {
	if p.totalSize == nil {
		panic("the totalSize must not be nil, please use TotalSize() function instead")
	}
	return *p.totalSize
}

func (p *Disk) SetUsesScsiReservation(attr bool) {
	p.usesScsiReservation = &attr
}

func (p *Disk) UsesScsiReservation() (bool, bool) {
	if p.usesScsiReservation != nil {
		return *p.usesScsiReservation, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustUsesScsiReservation() bool {
	if p.usesScsiReservation == nil {
		panic("the usesScsiReservation must not be nil, please use UsesScsiReservation() function instead")
	}
	return *p.usesScsiReservation
}

// SetVm Represents a virtual machine.
func (p *Disk) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Disk) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Disk) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *Disk) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Disk) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Disk) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

func (p *Disk) SetWipeAfterDelete(attr bool) {
	p.wipeAfterDelete = &attr
}

func (p *Disk) WipeAfterDelete() (bool, bool) {
	if p.wipeAfterDelete != nil {
		return *p.wipeAfterDelete, true
	}
	var zero bool
	return zero, false
}

func (p *Disk) MustWipeAfterDelete() bool {
	if p.wipeAfterDelete == nil {
		panic("the wipeAfterDelete must not be nil, please use WipeAfterDelete() function instead")
	}
	return *p.wipeAfterDelete
}

// DiskAttachment Describes how a disk is attached to a virtual machine.
type DiskAttachment struct {
	Struct
	active              *bool
	bootable            *bool
	comment             *string
	description         *string
	disk                *Disk
	id                  *string
	interface_          *DiskInterface
	logicalName         *string
	name                *string
	passDiscard         *bool
	readOnly            *bool
	template            *Template
	usesScsiReservation *bool
	vm                  *Vm
}

func (p *DiskAttachment) SetActive(attr bool) {
	p.active = &attr
}

func (p *DiskAttachment) Active() (bool, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero bool
	return zero, false
}

func (p *DiskAttachment) MustActive() bool {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *DiskAttachment) SetBootable(attr bool) {
	p.bootable = &attr
}

func (p *DiskAttachment) Bootable() (bool, bool) {
	if p.bootable != nil {
		return *p.bootable, true
	}
	var zero bool
	return zero, false
}

func (p *DiskAttachment) MustBootable() bool {
	if p.bootable == nil {
		panic("the bootable must not be nil, please use Bootable() function instead")
	}
	return *p.bootable
}

func (p *DiskAttachment) SetComment(attr string) {
	p.comment = &attr
}

func (p *DiskAttachment) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *DiskAttachment) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *DiskAttachment) SetDescription(attr string) {
	p.description = &attr
}

func (p *DiskAttachment) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *DiskAttachment) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDisk Represents a virtual disk device.
func (p *DiskAttachment) SetDisk(attr *Disk) {
	p.disk = attr
}

// Disk Represents a virtual disk device.
func (p *DiskAttachment) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

// MustDisk Represents a virtual disk device.
func (p *DiskAttachment) MustDisk() *Disk {
	if p.disk == nil {
		panic("the disk must not be nil, please use Disk() function instead")
	}
	return p.disk
}

func (p *DiskAttachment) SetId(attr string) {
	p.id = &attr
}

func (p *DiskAttachment) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *DiskAttachment) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInterface The underlying storage interface of disks communication with controller.
func (p *DiskAttachment) SetInterface(attr DiskInterface) {
	p.interface_ = &attr
}

// Interface The underlying storage interface of disks communication with controller.
func (p *DiskAttachment) Interface() (DiskInterface, bool) {
	if p.interface_ != nil {
		return *p.interface_, true
	}
	var zero DiskInterface
	return zero, false
}

// MustInterface The underlying storage interface of disks communication with controller.
func (p *DiskAttachment) MustInterface() DiskInterface {
	if p.interface_ == nil {
		panic("the interface_ must not be nil, please use Interface() function instead")
	}
	return *p.interface_
}

func (p *DiskAttachment) SetLogicalName(attr string) {
	p.logicalName = &attr
}

func (p *DiskAttachment) LogicalName() (string, bool) {
	if p.logicalName != nil {
		return *p.logicalName, true
	}
	var zero string
	return zero, false
}

func (p *DiskAttachment) MustLogicalName() string {
	if p.logicalName == nil {
		panic("the logicalName must not be nil, please use LogicalName() function instead")
	}
	return *p.logicalName
}

func (p *DiskAttachment) SetName(attr string) {
	p.name = &attr
}

func (p *DiskAttachment) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *DiskAttachment) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *DiskAttachment) SetPassDiscard(attr bool) {
	p.passDiscard = &attr
}

func (p *DiskAttachment) PassDiscard() (bool, bool) {
	if p.passDiscard != nil {
		return *p.passDiscard, true
	}
	var zero bool
	return zero, false
}

func (p *DiskAttachment) MustPassDiscard() bool {
	if p.passDiscard == nil {
		panic("the passDiscard must not be nil, please use PassDiscard() function instead")
	}
	return *p.passDiscard
}

func (p *DiskAttachment) SetReadOnly(attr bool) {
	p.readOnly = &attr
}

func (p *DiskAttachment) ReadOnly() (bool, bool) {
	if p.readOnly != nil {
		return *p.readOnly, true
	}
	var zero bool
	return zero, false
}

func (p *DiskAttachment) MustReadOnly() bool {
	if p.readOnly == nil {
		panic("the readOnly must not be nil, please use ReadOnly() function instead")
	}
	return *p.readOnly
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskAttachment) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskAttachment) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskAttachment) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *DiskAttachment) SetUsesScsiReservation(attr bool) {
	p.usesScsiReservation = &attr
}

func (p *DiskAttachment) UsesScsiReservation() (bool, bool) {
	if p.usesScsiReservation != nil {
		return *p.usesScsiReservation, true
	}
	var zero bool
	return zero, false
}

func (p *DiskAttachment) MustUsesScsiReservation() bool {
	if p.usesScsiReservation == nil {
		panic("the usesScsiReservation must not be nil, please use UsesScsiReservation() function instead")
	}
	return *p.usesScsiReservation
}

// SetVm Represents a virtual machine.
func (p *DiskAttachment) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *DiskAttachment) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *DiskAttachment) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type DiskProfile struct {
	Struct
	comment       *string
	description   *string
	id            *string
	name          *string
	permissions   *PermissionSlice
	qos           *Qos
	storageDomain *StorageDomain
}

func (p *DiskProfile) SetComment(attr string) {
	p.comment = &attr
}

func (p *DiskProfile) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *DiskProfile) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *DiskProfile) SetDescription(attr string) {
	p.description = &attr
}

func (p *DiskProfile) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *DiskProfile) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *DiskProfile) SetId(attr string) {
	p.id = &attr
}

func (p *DiskProfile) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *DiskProfile) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *DiskProfile) SetName(attr string) {
	p.name = &attr
}

func (p *DiskProfile) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *DiskProfile) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *DiskProfile) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *DiskProfile) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *DiskProfile) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

// SetQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *DiskProfile) SetQos(attr *Qos) {
	p.qos = attr
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *DiskProfile) Qos() (*Qos, bool) {
	if p.qos != nil {
		return p.qos, true
	}
	return nil, false
}

// MustQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *DiskProfile) MustQos() *Qos {
	if p.qos == nil {
		panic("the qos must not be nil, please use Qos() function instead")
	}
	return p.qos
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskProfile) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskProfile) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskProfile) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

type DiskSnapshot struct {
	Struct
	active              *bool
	actualSize          *int64
	alias               *string
	backup              *DiskBackup
	backupMode          *DiskBackupMode
	bootable            *bool
	comment             *string
	contentType         *DiskContentType
	description         *string
	disk                *Disk
	diskProfile         *DiskProfile
	diskSnapshots       *DiskSnapshotSlice
	externalDisk        *string
	format              *DiskFormat
	id                  *string
	imageId             *string
	initialSize         *int64
	instanceType        *InstanceType
	interface_          *DiskInterface
	logicalName         *string
	lunStorage          *HostStorage
	name                *string
	openstackVolumeType *OpenStackVolumeType
	parent              *DiskSnapshot
	permissions         *PermissionSlice
	propagateErrors     *bool
	provisionedSize     *int64
	qcowVersion         *QcowVersion
	quota               *Quota
	readOnly            *bool
	sgio                *ScsiGenericIO
	shareable           *bool
	snapshot            *Snapshot
	sparse              *bool
	statistics          *StatisticSlice
	status              *DiskStatus
	storageDomain       *StorageDomain
	storageDomains      *StorageDomainSlice
	storageType         *DiskStorageType
	template            *Template
	totalSize           *int64
	usesScsiReservation *bool
	vm                  *Vm
	vms                 *VmSlice
	wipeAfterDelete     *bool
}

func (p *DiskSnapshot) SetActive(attr bool) {
	p.active = &attr
}

func (p *DiskSnapshot) Active() (bool, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustActive() bool {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *DiskSnapshot) SetActualSize(attr int64) {
	p.actualSize = &attr
}

func (p *DiskSnapshot) ActualSize() (int64, bool) {
	if p.actualSize != nil {
		return *p.actualSize, true
	}
	var zero int64
	return zero, false
}

func (p *DiskSnapshot) MustActualSize() int64 {
	if p.actualSize == nil {
		panic("the actualSize must not be nil, please use ActualSize() function instead")
	}
	return *p.actualSize
}

func (p *DiskSnapshot) SetAlias(attr string) {
	p.alias = &attr
}

func (p *DiskSnapshot) Alias() (string, bool) {
	if p.alias != nil {
		return *p.alias, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustAlias() string {
	if p.alias == nil {
		panic("the alias must not be nil, please use Alias() function instead")
	}
	return *p.alias
}

// SetBackup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *DiskSnapshot) SetBackup(attr DiskBackup) {
	p.backup = &attr
}

// Backup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *DiskSnapshot) Backup() (DiskBackup, bool) {
	if p.backup != nil {
		return *p.backup, true
	}
	var zero DiskBackup
	return zero, false
}

// MustBackup Represents an enumeration of the backup mechanism
// that is enabled on the disk.
func (p *DiskSnapshot) MustBackup() DiskBackup {
	if p.backup == nil {
		panic("the backup must not be nil, please use Backup() function instead")
	}
	return *p.backup
}

// SetBackupMode Represents an enumeration of backup modes
func (p *DiskSnapshot) SetBackupMode(attr DiskBackupMode) {
	p.backupMode = &attr
}

// BackupMode Represents an enumeration of backup modes
func (p *DiskSnapshot) BackupMode() (DiskBackupMode, bool) {
	if p.backupMode != nil {
		return *p.backupMode, true
	}
	var zero DiskBackupMode
	return zero, false
}

// MustBackupMode Represents an enumeration of backup modes
func (p *DiskSnapshot) MustBackupMode() DiskBackupMode {
	if p.backupMode == nil {
		panic("the backupMode must not be nil, please use BackupMode() function instead")
	}
	return *p.backupMode
}

func (p *DiskSnapshot) SetBootable(attr bool) {
	p.bootable = &attr
}

func (p *DiskSnapshot) Bootable() (bool, bool) {
	if p.bootable != nil {
		return *p.bootable, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustBootable() bool {
	if p.bootable == nil {
		panic("the bootable must not be nil, please use Bootable() function instead")
	}
	return *p.bootable
}

func (p *DiskSnapshot) SetComment(attr string) {
	p.comment = &attr
}

func (p *DiskSnapshot) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

// SetContentType The actual content residing on the disk.
func (p *DiskSnapshot) SetContentType(attr DiskContentType) {
	p.contentType = &attr
}

// ContentType The actual content residing on the disk.
func (p *DiskSnapshot) ContentType() (DiskContentType, bool) {
	if p.contentType != nil {
		return *p.contentType, true
	}
	var zero DiskContentType
	return zero, false
}

// MustContentType The actual content residing on the disk.
func (p *DiskSnapshot) MustContentType() DiskContentType {
	if p.contentType == nil {
		panic("the contentType must not be nil, please use ContentType() function instead")
	}
	return *p.contentType
}

func (p *DiskSnapshot) SetDescription(attr string) {
	p.description = &attr
}

func (p *DiskSnapshot) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDisk Represents a virtual disk device.
func (p *DiskSnapshot) SetDisk(attr *Disk) {
	p.disk = attr
}

// Disk Represents a virtual disk device.
func (p *DiskSnapshot) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

// MustDisk Represents a virtual disk device.
func (p *DiskSnapshot) MustDisk() *Disk {
	if p.disk == nil {
		panic("the disk must not be nil, please use Disk() function instead")
	}
	return p.disk
}

func (p *DiskSnapshot) SetDiskProfile(attr *DiskProfile) {
	p.diskProfile = attr
}

func (p *DiskSnapshot) DiskProfile() (*DiskProfile, bool) {
	if p.diskProfile != nil {
		return p.diskProfile, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustDiskProfile() *DiskProfile {
	if p.diskProfile == nil {
		panic("the diskProfile must not be nil, please use DiskProfile() function instead")
	}
	return p.diskProfile
}

func (p *DiskSnapshot) SetDiskSnapshots(attr *DiskSnapshotSlice) {
	p.diskSnapshots = attr
}

func (p *DiskSnapshot) DiskSnapshots() (*DiskSnapshotSlice, bool) {
	if p.diskSnapshots != nil {
		return p.diskSnapshots, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustDiskSnapshots() *DiskSnapshotSlice {
	if p.diskSnapshots == nil {
		panic("the diskSnapshots must not be nil, please use DiskSnapshots() function instead")
	}
	return p.diskSnapshots
}

func (p *DiskSnapshot) SetExternalDisk(attr string) {
	p.externalDisk = &attr
}

func (p *DiskSnapshot) ExternalDisk() (string, bool) {
	if p.externalDisk != nil {
		return *p.externalDisk, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustExternalDisk() string {
	if p.externalDisk == nil {
		panic("the externalDisk must not be nil, please use ExternalDisk() function instead")
	}
	return *p.externalDisk
}

// SetFormat The underlying storage format of disks.
func (p *DiskSnapshot) SetFormat(attr DiskFormat) {
	p.format = &attr
}

// Format The underlying storage format of disks.
func (p *DiskSnapshot) Format() (DiskFormat, bool) {
	if p.format != nil {
		return *p.format, true
	}
	var zero DiskFormat
	return zero, false
}

// MustFormat The underlying storage format of disks.
func (p *DiskSnapshot) MustFormat() DiskFormat {
	if p.format == nil {
		panic("the format must not be nil, please use Format() function instead")
	}
	return *p.format
}

func (p *DiskSnapshot) SetId(attr string) {
	p.id = &attr
}

func (p *DiskSnapshot) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *DiskSnapshot) SetImageId(attr string) {
	p.imageId = &attr
}

func (p *DiskSnapshot) ImageId() (string, bool) {
	if p.imageId != nil {
		return *p.imageId, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustImageId() string {
	if p.imageId == nil {
		panic("the imageId must not be nil, please use ImageId() function instead")
	}
	return *p.imageId
}

func (p *DiskSnapshot) SetInitialSize(attr int64) {
	p.initialSize = &attr
}

func (p *DiskSnapshot) InitialSize() (int64, bool) {
	if p.initialSize != nil {
		return *p.initialSize, true
	}
	var zero int64
	return zero, false
}

func (p *DiskSnapshot) MustInitialSize() int64 {
	if p.initialSize == nil {
		panic("the initialSize must not be nil, please use InitialSize() function instead")
	}
	return *p.initialSize
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *DiskSnapshot) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *DiskSnapshot) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *DiskSnapshot) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

// SetInterface The underlying storage interface of disks communication with controller.
func (p *DiskSnapshot) SetInterface(attr DiskInterface) {
	p.interface_ = &attr
}

// Interface The underlying storage interface of disks communication with controller.
func (p *DiskSnapshot) Interface() (DiskInterface, bool) {
	if p.interface_ != nil {
		return *p.interface_, true
	}
	var zero DiskInterface
	return zero, false
}

// MustInterface The underlying storage interface of disks communication with controller.
func (p *DiskSnapshot) MustInterface() DiskInterface {
	if p.interface_ == nil {
		panic("the interface_ must not be nil, please use Interface() function instead")
	}
	return *p.interface_
}

func (p *DiskSnapshot) SetLogicalName(attr string) {
	p.logicalName = &attr
}

func (p *DiskSnapshot) LogicalName() (string, bool) {
	if p.logicalName != nil {
		return *p.logicalName, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustLogicalName() string {
	if p.logicalName == nil {
		panic("the logicalName must not be nil, please use LogicalName() function instead")
	}
	return *p.logicalName
}

func (p *DiskSnapshot) SetLunStorage(attr *HostStorage) {
	p.lunStorage = attr
}

func (p *DiskSnapshot) LunStorage() (*HostStorage, bool) {
	if p.lunStorage != nil {
		return p.lunStorage, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustLunStorage() *HostStorage {
	if p.lunStorage == nil {
		panic("the lunStorage must not be nil, please use LunStorage() function instead")
	}
	return p.lunStorage
}

func (p *DiskSnapshot) SetName(attr string) {
	p.name = &attr
}

func (p *DiskSnapshot) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *DiskSnapshot) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *DiskSnapshot) SetOpenstackVolumeType(attr *OpenStackVolumeType) {
	p.openstackVolumeType = attr
}

func (p *DiskSnapshot) OpenstackVolumeType() (*OpenStackVolumeType, bool) {
	if p.openstackVolumeType != nil {
		return p.openstackVolumeType, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustOpenstackVolumeType() *OpenStackVolumeType {
	if p.openstackVolumeType == nil {
		panic("the openstackVolumeType must not be nil, please use OpenstackVolumeType() function instead")
	}
	return p.openstackVolumeType
}

func (p *DiskSnapshot) SetParent(attr *DiskSnapshot) {
	p.parent = attr
}

func (p *DiskSnapshot) Parent() (*DiskSnapshot, bool) {
	if p.parent != nil {
		return p.parent, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustParent() *DiskSnapshot {
	if p.parent == nil {
		panic("the parent must not be nil, please use Parent() function instead")
	}
	return p.parent
}

func (p *DiskSnapshot) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *DiskSnapshot) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *DiskSnapshot) SetPropagateErrors(attr bool) {
	p.propagateErrors = &attr
}

func (p *DiskSnapshot) PropagateErrors() (bool, bool) {
	if p.propagateErrors != nil {
		return *p.propagateErrors, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustPropagateErrors() bool {
	if p.propagateErrors == nil {
		panic("the propagateErrors must not be nil, please use PropagateErrors() function instead")
	}
	return *p.propagateErrors
}

func (p *DiskSnapshot) SetProvisionedSize(attr int64) {
	p.provisionedSize = &attr
}

func (p *DiskSnapshot) ProvisionedSize() (int64, bool) {
	if p.provisionedSize != nil {
		return *p.provisionedSize, true
	}
	var zero int64
	return zero, false
}

func (p *DiskSnapshot) MustProvisionedSize() int64 {
	if p.provisionedSize == nil {
		panic("the provisionedSize must not be nil, please use ProvisionedSize() function instead")
	}
	return *p.provisionedSize
}

// SetQcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *DiskSnapshot) SetQcowVersion(attr QcowVersion) {
	p.qcowVersion = &attr
}

// QcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *DiskSnapshot) QcowVersion() (QcowVersion, bool) {
	if p.qcowVersion != nil {
		return *p.qcowVersion, true
	}
	var zero QcowVersion
	return zero, false
}

// MustQcowVersion The QCOW version specifies to the qemu which qemu version the volume supports.
//
// This field can be updated using the update API and will be reported only for QCOW volumes,
// it is determined by the storage domain's version which the disk is created on.
// Storage domains with version lower than V4 support QCOW2 version 2 volumes,
// while V4 storage domains also support QCOW2 version 3.
// For more information about features of the different QCOW versions, see http://wiki.qemu.org/Features/Qcow3[here].
func (p *DiskSnapshot) MustQcowVersion() QcowVersion {
	if p.qcowVersion == nil {
		panic("the qcowVersion must not be nil, please use QcowVersion() function instead")
	}
	return *p.qcowVersion
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *DiskSnapshot) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *DiskSnapshot) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *DiskSnapshot) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

func (p *DiskSnapshot) SetReadOnly(attr bool) {
	p.readOnly = &attr
}

func (p *DiskSnapshot) ReadOnly() (bool, bool) {
	if p.readOnly != nil {
		return *p.readOnly, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustReadOnly() bool {
	if p.readOnly == nil {
		panic("the readOnly must not be nil, please use ReadOnly() function instead")
	}
	return *p.readOnly
}

// SetSgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *DiskSnapshot) SetSgio(attr ScsiGenericIO) {
	p.sgio = &attr
}

// Sgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *DiskSnapshot) Sgio() (ScsiGenericIO, bool) {
	if p.sgio != nil {
		return *p.sgio, true
	}
	var zero ScsiGenericIO
	return zero, false
}

// MustSgio When a direct LUN disk is using SCSI passthrough the privileged I/O policy is determined by this enum.
func (p *DiskSnapshot) MustSgio() ScsiGenericIO {
	if p.sgio == nil {
		panic("the sgio must not be nil, please use Sgio() function instead")
	}
	return *p.sgio
}

func (p *DiskSnapshot) SetShareable(attr bool) {
	p.shareable = &attr
}

func (p *DiskSnapshot) Shareable() (bool, bool) {
	if p.shareable != nil {
		return *p.shareable, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustShareable() bool {
	if p.shareable == nil {
		panic("the shareable must not be nil, please use Shareable() function instead")
	}
	return *p.shareable
}

// SetSnapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *DiskSnapshot) SetSnapshot(attr *Snapshot) {
	p.snapshot = attr
}

// Snapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *DiskSnapshot) Snapshot() (*Snapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

// MustSnapshot Represents a snapshot object.
//
// Example XML representation:
//
// [source,xml]
// ----
// <snapshot id="456" href="/ovirt-engine/api/vms/123/snapshots/456">
//   <actions>
//     <link rel="restore" href="/ovirt-engine/api/vms/123/snapshots/456/restore"/>
//   </actions>
//   <vm id="123" href="/ovirt-engine/api/vms/123"/>
//   <description>Virtual Machine 1 - Snapshot A</description>
//   <type>active</type>
//   <date>2010-08-16T14:24:29</date>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
func (p *DiskSnapshot) MustSnapshot() *Snapshot {
	if p.snapshot == nil {
		panic("the snapshot must not be nil, please use Snapshot() function instead")
	}
	return p.snapshot
}

func (p *DiskSnapshot) SetSparse(attr bool) {
	p.sparse = &attr
}

func (p *DiskSnapshot) Sparse() (bool, bool) {
	if p.sparse != nil {
		return *p.sparse, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustSparse() bool {
	if p.sparse == nil {
		panic("the sparse must not be nil, please use Sparse() function instead")
	}
	return *p.sparse
}

func (p *DiskSnapshot) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *DiskSnapshot) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// SetStatus Current status representation for disk.
func (p *DiskSnapshot) SetStatus(attr DiskStatus) {
	p.status = &attr
}

// Status Current status representation for disk.
func (p *DiskSnapshot) Status() (DiskStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero DiskStatus
	return zero, false
}

// MustStatus Current status representation for disk.
func (p *DiskSnapshot) MustStatus() DiskStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskSnapshot) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskSnapshot) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *DiskSnapshot) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

func (p *DiskSnapshot) SetStorageDomains(attr *StorageDomainSlice) {
	p.storageDomains = attr
}

func (p *DiskSnapshot) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("the storageDomains must not be nil, please use StorageDomains() function instead")
	}
	return p.storageDomains
}

func (p *DiskSnapshot) SetStorageType(attr DiskStorageType) {
	p.storageType = &attr
}

func (p *DiskSnapshot) StorageType() (DiskStorageType, bool) {
	if p.storageType != nil {
		return *p.storageType, true
	}
	var zero DiskStorageType
	return zero, false
}

func (p *DiskSnapshot) MustStorageType() DiskStorageType {
	if p.storageType == nil {
		panic("the storageType must not be nil, please use StorageType() function instead")
	}
	return *p.storageType
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskSnapshot) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskSnapshot) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *DiskSnapshot) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *DiskSnapshot) SetTotalSize(attr int64) {
	p.totalSize = &attr
}

func (p *DiskSnapshot) TotalSize() (int64, bool) {
	if p.totalSize != nil {
		return *p.totalSize, true
	}
	var zero int64
	return zero, false
}

func (p *DiskSnapshot) MustTotalSize() int64 {
	if p.totalSize == nil {
		panic("the totalSize must not be nil, please use TotalSize() function instead")
	}
	return *p.totalSize
}

func (p *DiskSnapshot) SetUsesScsiReservation(attr bool) {
	p.usesScsiReservation = &attr
}

func (p *DiskSnapshot) UsesScsiReservation() (bool, bool) {
	if p.usesScsiReservation != nil {
		return *p.usesScsiReservation, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustUsesScsiReservation() bool {
	if p.usesScsiReservation == nil {
		panic("the usesScsiReservation must not be nil, please use UsesScsiReservation() function instead")
	}
	return *p.usesScsiReservation
}

// SetVm Represents a virtual machine.
func (p *DiskSnapshot) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *DiskSnapshot) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *DiskSnapshot) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *DiskSnapshot) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *DiskSnapshot) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *DiskSnapshot) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

func (p *DiskSnapshot) SetWipeAfterDelete(attr bool) {
	p.wipeAfterDelete = &attr
}

func (p *DiskSnapshot) WipeAfterDelete() (bool, bool) {
	if p.wipeAfterDelete != nil {
		return *p.wipeAfterDelete, true
	}
	var zero bool
	return zero, false
}

func (p *DiskSnapshot) MustWipeAfterDelete() bool {
	if p.wipeAfterDelete == nil {
		panic("the wipeAfterDelete must not be nil, please use WipeAfterDelete() function instead")
	}
	return *p.wipeAfterDelete
}

// Display Represents a graphic console configuration.
type Display struct {
	Struct
	address             *string
	allowOverride       *bool
	certificate         *Certificate
	copyPasteEnabled    *bool
	disconnectAction    *string
	fileTransferEnabled *bool
	keyboardLayout      *string
	monitors            *int64
	port                *int64
	proxy               *string
	securePort          *int64
	singleQxlPci        *bool
	smartcardEnabled    *bool
	type_               *DisplayType
}

func (p *Display) SetAddress(attr string) {
	p.address = &attr
}

func (p *Display) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *Display) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *Display) SetAllowOverride(attr bool) {
	p.allowOverride = &attr
}

func (p *Display) AllowOverride() (bool, bool) {
	if p.allowOverride != nil {
		return *p.allowOverride, true
	}
	var zero bool
	return zero, false
}

func (p *Display) MustAllowOverride() bool {
	if p.allowOverride == nil {
		panic("the allowOverride must not be nil, please use AllowOverride() function instead")
	}
	return *p.allowOverride
}

func (p *Display) SetCertificate(attr *Certificate) {
	p.certificate = attr
}

func (p *Display) Certificate() (*Certificate, bool) {
	if p.certificate != nil {
		return p.certificate, true
	}
	return nil, false
}

func (p *Display) MustCertificate() *Certificate {
	if p.certificate == nil {
		panic("the certificate must not be nil, please use Certificate() function instead")
	}
	return p.certificate
}

func (p *Display) SetCopyPasteEnabled(attr bool) {
	p.copyPasteEnabled = &attr
}

func (p *Display) CopyPasteEnabled() (bool, bool) {
	if p.copyPasteEnabled != nil {
		return *p.copyPasteEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *Display) MustCopyPasteEnabled() bool {
	if p.copyPasteEnabled == nil {
		panic("the copyPasteEnabled must not be nil, please use CopyPasteEnabled() function instead")
	}
	return *p.copyPasteEnabled
}

func (p *Display) SetDisconnectAction(attr string) {
	p.disconnectAction = &attr
}

func (p *Display) DisconnectAction() (string, bool) {
	if p.disconnectAction != nil {
		return *p.disconnectAction, true
	}
	var zero string
	return zero, false
}

func (p *Display) MustDisconnectAction() string {
	if p.disconnectAction == nil {
		panic("the disconnectAction must not be nil, please use DisconnectAction() function instead")
	}
	return *p.disconnectAction
}

func (p *Display) SetFileTransferEnabled(attr bool) {
	p.fileTransferEnabled = &attr
}

func (p *Display) FileTransferEnabled() (bool, bool) {
	if p.fileTransferEnabled != nil {
		return *p.fileTransferEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *Display) MustFileTransferEnabled() bool {
	if p.fileTransferEnabled == nil {
		panic("the fileTransferEnabled must not be nil, please use FileTransferEnabled() function instead")
	}
	return *p.fileTransferEnabled
}

func (p *Display) SetKeyboardLayout(attr string) {
	p.keyboardLayout = &attr
}

func (p *Display) KeyboardLayout() (string, bool) {
	if p.keyboardLayout != nil {
		return *p.keyboardLayout, true
	}
	var zero string
	return zero, false
}

func (p *Display) MustKeyboardLayout() string {
	if p.keyboardLayout == nil {
		panic("the keyboardLayout must not be nil, please use KeyboardLayout() function instead")
	}
	return *p.keyboardLayout
}

func (p *Display) SetMonitors(attr int64) {
	p.monitors = &attr
}

func (p *Display) Monitors() (int64, bool) {
	if p.monitors != nil {
		return *p.monitors, true
	}
	var zero int64
	return zero, false
}

func (p *Display) MustMonitors() int64 {
	if p.monitors == nil {
		panic("the monitors must not be nil, please use Monitors() function instead")
	}
	return *p.monitors
}

func (p *Display) SetPort(attr int64) {
	p.port = &attr
}

func (p *Display) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *Display) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *Display) SetProxy(attr string) {
	p.proxy = &attr
}

func (p *Display) Proxy() (string, bool) {
	if p.proxy != nil {
		return *p.proxy, true
	}
	var zero string
	return zero, false
}

func (p *Display) MustProxy() string {
	if p.proxy == nil {
		panic("the proxy must not be nil, please use Proxy() function instead")
	}
	return *p.proxy
}

func (p *Display) SetSecurePort(attr int64) {
	p.securePort = &attr
}

func (p *Display) SecurePort() (int64, bool) {
	if p.securePort != nil {
		return *p.securePort, true
	}
	var zero int64
	return zero, false
}

func (p *Display) MustSecurePort() int64 {
	if p.securePort == nil {
		panic("the securePort must not be nil, please use SecurePort() function instead")
	}
	return *p.securePort
}

func (p *Display) SetSingleQxlPci(attr bool) {
	p.singleQxlPci = &attr
}

func (p *Display) SingleQxlPci() (bool, bool) {
	if p.singleQxlPci != nil {
		return *p.singleQxlPci, true
	}
	var zero bool
	return zero, false
}

func (p *Display) MustSingleQxlPci() bool {
	if p.singleQxlPci == nil {
		panic("the singleQxlPci must not be nil, please use SingleQxlPci() function instead")
	}
	return *p.singleQxlPci
}

func (p *Display) SetSmartcardEnabled(attr bool) {
	p.smartcardEnabled = &attr
}

func (p *Display) SmartcardEnabled() (bool, bool) {
	if p.smartcardEnabled != nil {
		return *p.smartcardEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *Display) MustSmartcardEnabled() bool {
	if p.smartcardEnabled == nil {
		panic("the smartcardEnabled must not be nil, please use SmartcardEnabled() function instead")
	}
	return *p.smartcardEnabled
}

// SetType Represents an enumeration of the protocol used
// to connect to the graphic console of the virtual
// machine.
func (p *Display) SetType(attr DisplayType) {
	p.type_ = &attr
}

// Type Represents an enumeration of the protocol used
// to connect to the graphic console of the virtual
// machine.
func (p *Display) Type() (DisplayType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero DisplayType
	return zero, false
}

// MustType Represents an enumeration of the protocol used
// to connect to the graphic console of the virtual
// machine.
func (p *Display) MustType() DisplayType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// Dns Represents the DNS resolver configuration.
type Dns struct {
	Struct
	searchDomains *HostSlice
	servers       *HostSlice
}

func (p *Dns) SetSearchDomains(attr *HostSlice) {
	p.searchDomains = attr
}

func (p *Dns) SearchDomains() (*HostSlice, bool) {
	if p.searchDomains != nil {
		return p.searchDomains, true
	}
	return nil, false
}

func (p *Dns) MustSearchDomains() *HostSlice {
	if p.searchDomains == nil {
		panic("the searchDomains must not be nil, please use SearchDomains() function instead")
	}
	return p.searchDomains
}

func (p *Dns) SetServers(attr *HostSlice) {
	p.servers = attr
}

func (p *Dns) Servers() (*HostSlice, bool) {
	if p.servers != nil {
		return p.servers, true
	}
	return nil, false
}

func (p *Dns) MustServers() *HostSlice {
	if p.servers == nil {
		panic("the servers must not be nil, please use Servers() function instead")
	}
	return p.servers
}

// DnsResolverConfiguration Represents the DNS resolver configuration.
type DnsResolverConfiguration struct {
	Struct
	nameServers []string
}

func (p *DnsResolverConfiguration) SetNameServers(attr []string) {
	p.nameServers = attr
}

func (p *DnsResolverConfiguration) NameServers() ([]string, bool) {
	if p.nameServers != nil {
		return p.nameServers, true
	}
	return nil, false
}

func (p *DnsResolverConfiguration) MustNameServers() []string {
	if p.nameServers == nil {
		panic("the nameServers must not be nil, please use NameServers() function instead")
	}
	return p.nameServers
}

// Domain This type represents a directory service domain.
type Domain struct {
	Struct
	comment     *string
	description *string
	groups      *GroupSlice
	id          *string
	name        *string
	user        *User
	users       *UserSlice
}

func (p *Domain) SetComment(attr string) {
	p.comment = &attr
}

func (p *Domain) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Domain) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Domain) SetDescription(attr string) {
	p.description = &attr
}

func (p *Domain) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Domain) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Domain) SetGroups(attr *GroupSlice) {
	p.groups = attr
}

func (p *Domain) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *Domain) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("the groups must not be nil, please use Groups() function instead")
	}
	return p.groups
}

func (p *Domain) SetId(attr string) {
	p.id = &attr
}

func (p *Domain) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Domain) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Domain) SetName(attr string) {
	p.name = &attr
}

func (p *Domain) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Domain) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetUser Represents a user in the system.
func (p *Domain) SetUser(attr *User) {
	p.user = attr
}

// User Represents a user in the system.
func (p *Domain) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

// MustUser Represents a user in the system.
func (p *Domain) MustUser() *User {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return p.user
}

func (p *Domain) SetUsers(attr *UserSlice) {
	p.users = attr
}

func (p *Domain) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *Domain) MustUsers() *UserSlice {
	if p.users == nil {
		panic("the users must not be nil, please use Users() function instead")
	}
	return p.users
}

type EntityProfileDetail struct {
	Struct
	profileDetails *ProfileDetailSlice
}

func (p *EntityProfileDetail) SetProfileDetails(attr *ProfileDetailSlice) {
	p.profileDetails = attr
}

func (p *EntityProfileDetail) ProfileDetails() (*ProfileDetailSlice, bool) {
	if p.profileDetails != nil {
		return p.profileDetails, true
	}
	return nil, false
}

func (p *EntityProfileDetail) MustProfileDetails() *ProfileDetailSlice {
	if p.profileDetails == nil {
		panic("the profileDetails must not be nil, please use ProfileDetails() function instead")
	}
	return p.profileDetails
}

type ErrorHandling struct {
	Struct
	onError *MigrateOnError
}

func (p *ErrorHandling) SetOnError(attr MigrateOnError) {
	p.onError = &attr
}

func (p *ErrorHandling) OnError() (MigrateOnError, bool) {
	if p.onError != nil {
		return *p.onError, true
	}
	var zero MigrateOnError
	return zero, false
}

func (p *ErrorHandling) MustOnError() MigrateOnError {
	if p.onError == nil {
		panic("the onError must not be nil, please use OnError() function instead")
	}
	return *p.onError
}

// Event Type representing an event.
type Event struct {
	Struct
	cluster       *Cluster
	code          *int64
	comment       *string
	correlationId *string
	customData    *string
	customId      *int64
	dataCenter    *DataCenter
	description   *string
	floodRate     *int64
	host          *Host
	id            *string
	index         *int64
	logOnHost     *bool
	name          *string
	origin        *string
	severity      *LogSeverity
	storageDomain *StorageDomain
	template      *Template
	time          *time.Time
	user          *User
	vm            *Vm
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Event) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Event) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Event) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *Event) SetCode(attr int64) {
	p.code = &attr
}

func (p *Event) Code() (int64, bool) {
	if p.code != nil {
		return *p.code, true
	}
	var zero int64
	return zero, false
}

func (p *Event) MustCode() int64 {
	if p.code == nil {
		panic("the code must not be nil, please use Code() function instead")
	}
	return *p.code
}

func (p *Event) SetComment(attr string) {
	p.comment = &attr
}

func (p *Event) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Event) SetCorrelationId(attr string) {
	p.correlationId = &attr
}

func (p *Event) CorrelationId() (string, bool) {
	if p.correlationId != nil {
		return *p.correlationId, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustCorrelationId() string {
	if p.correlationId == nil {
		panic("the correlationId must not be nil, please use CorrelationId() function instead")
	}
	return *p.correlationId
}

func (p *Event) SetCustomData(attr string) {
	p.customData = &attr
}

func (p *Event) CustomData() (string, bool) {
	if p.customData != nil {
		return *p.customData, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustCustomData() string {
	if p.customData == nil {
		panic("the customData must not be nil, please use CustomData() function instead")
	}
	return *p.customData
}

func (p *Event) SetCustomId(attr int64) {
	p.customId = &attr
}

func (p *Event) CustomId() (int64, bool) {
	if p.customId != nil {
		return *p.customId, true
	}
	var zero int64
	return zero, false
}

func (p *Event) MustCustomId() int64 {
	if p.customId == nil {
		panic("the customId must not be nil, please use CustomId() function instead")
	}
	return *p.customId
}

func (p *Event) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Event) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Event) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Event) SetDescription(attr string) {
	p.description = &attr
}

func (p *Event) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Event) SetFloodRate(attr int64) {
	p.floodRate = &attr
}

func (p *Event) FloodRate() (int64, bool) {
	if p.floodRate != nil {
		return *p.floodRate, true
	}
	var zero int64
	return zero, false
}

func (p *Event) MustFloodRate() int64 {
	if p.floodRate == nil {
		panic("the floodRate must not be nil, please use FloodRate() function instead")
	}
	return *p.floodRate
}

// SetHost Type representing a host.
func (p *Event) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *Event) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *Event) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *Event) SetId(attr string) {
	p.id = &attr
}

func (p *Event) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Event) SetIndex(attr int64) {
	p.index = &attr
}

func (p *Event) Index() (int64, bool) {
	if p.index != nil {
		return *p.index, true
	}
	var zero int64
	return zero, false
}

func (p *Event) MustIndex() int64 {
	if p.index == nil {
		panic("the index must not be nil, please use Index() function instead")
	}
	return *p.index
}

func (p *Event) SetLogOnHost(attr bool) {
	p.logOnHost = &attr
}

func (p *Event) LogOnHost() (bool, bool) {
	if p.logOnHost != nil {
		return *p.logOnHost, true
	}
	var zero bool
	return zero, false
}

func (p *Event) MustLogOnHost() bool {
	if p.logOnHost == nil {
		panic("the logOnHost must not be nil, please use LogOnHost() function instead")
	}
	return *p.logOnHost
}

func (p *Event) SetName(attr string) {
	p.name = &attr
}

func (p *Event) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Event) SetOrigin(attr string) {
	p.origin = &attr
}

func (p *Event) Origin() (string, bool) {
	if p.origin != nil {
		return *p.origin, true
	}
	var zero string
	return zero, false
}

func (p *Event) MustOrigin() string {
	if p.origin == nil {
		panic("the origin must not be nil, please use Origin() function instead")
	}
	return *p.origin
}

// SetSeverity Enum representing a severity of an event.
func (p *Event) SetSeverity(attr LogSeverity) {
	p.severity = &attr
}

// Severity Enum representing a severity of an event.
func (p *Event) Severity() (LogSeverity, bool) {
	if p.severity != nil {
		return *p.severity, true
	}
	var zero LogSeverity
	return zero, false
}

// MustSeverity Enum representing a severity of an event.
func (p *Event) MustSeverity() LogSeverity {
	if p.severity == nil {
		panic("the severity must not be nil, please use Severity() function instead")
	}
	return *p.severity
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Event) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Event) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Event) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Event) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Event) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Event) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *Event) SetTime(attr time.Time) {
	p.time = &attr
}

func (p *Event) Time() (time.Time, bool) {
	if p.time != nil {
		return *p.time, true
	}
	var zero time.Time
	return zero, false
}

func (p *Event) MustTime() time.Time {
	if p.time == nil {
		panic("the time must not be nil, please use Time() function instead")
	}
	return *p.time
}

// SetUser Represents a user in the system.
func (p *Event) SetUser(attr *User) {
	p.user = attr
}

// User Represents a user in the system.
func (p *Event) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

// MustUser Represents a user in the system.
func (p *Event) MustUser() *User {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return p.user
}

// SetVm Represents a virtual machine.
func (p *Event) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Event) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Event) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type EventSubscription struct {
	Struct
	address            *string
	comment            *string
	description        *string
	event              *NotifiableEvent
	id                 *string
	name               *string
	notificationMethod *NotificationMethod
	user               *User
}

func (p *EventSubscription) SetAddress(attr string) {
	p.address = &attr
}

func (p *EventSubscription) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *EventSubscription) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *EventSubscription) SetComment(attr string) {
	p.comment = &attr
}

func (p *EventSubscription) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *EventSubscription) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *EventSubscription) SetDescription(attr string) {
	p.description = &attr
}

func (p *EventSubscription) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *EventSubscription) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetEvent Type representing a subset of events in the {product-name} server:
// those which a user may subscribe to receive a notification about.
func (p *EventSubscription) SetEvent(attr NotifiableEvent) {
	p.event = &attr
}

// Event Type representing a subset of events in the {product-name} server:
// those which a user may subscribe to receive a notification about.
func (p *EventSubscription) Event() (NotifiableEvent, bool) {
	if p.event != nil {
		return *p.event, true
	}
	var zero NotifiableEvent
	return zero, false
}

// MustEvent Type representing a subset of events in the {product-name} server:
// those which a user may subscribe to receive a notification about.
func (p *EventSubscription) MustEvent() NotifiableEvent {
	if p.event == nil {
		panic("the event must not be nil, please use Event() function instead")
	}
	return *p.event
}

func (p *EventSubscription) SetId(attr string) {
	p.id = &attr
}

func (p *EventSubscription) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *EventSubscription) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *EventSubscription) SetName(attr string) {
	p.name = &attr
}

func (p *EventSubscription) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *EventSubscription) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNotificationMethod Type representing the notification method for an event
// subscription. Currently only SMTP is supported by the API
// In the future support for SNMP notifications may be added.
func (p *EventSubscription) SetNotificationMethod(attr NotificationMethod) {
	p.notificationMethod = &attr
}

// NotificationMethod Type representing the notification method for an event
// subscription. Currently only SMTP is supported by the API
// In the future support for SNMP notifications may be added.
func (p *EventSubscription) NotificationMethod() (NotificationMethod, bool) {
	if p.notificationMethod != nil {
		return *p.notificationMethod, true
	}
	var zero NotificationMethod
	return zero, false
}

// MustNotificationMethod Type representing the notification method for an event
// subscription. Currently only SMTP is supported by the API
// In the future support for SNMP notifications may be added.
func (p *EventSubscription) MustNotificationMethod() NotificationMethod {
	if p.notificationMethod == nil {
		panic("the notificationMethod must not be nil, please use NotificationMethod() function instead")
	}
	return *p.notificationMethod
}

// SetUser Represents a user in the system.
func (p *EventSubscription) SetUser(attr *User) {
	p.user = attr
}

// User Represents a user in the system.
func (p *EventSubscription) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

// MustUser Represents a user in the system.
func (p *EventSubscription) MustUser() *User {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return p.user
}

type ExternalComputeResource struct {
	Struct
	comment              *string
	description          *string
	externalHostProvider *ExternalHostProvider
	id                   *string
	name                 *string
	provider             *string
	url                  *string
	user                 *string
}

func (p *ExternalComputeResource) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalComputeResource) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalComputeResource) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalComputeResource) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalComputeResource) SetExternalHostProvider(attr *ExternalHostProvider) {
	p.externalHostProvider = attr
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalComputeResource) ExternalHostProvider() (*ExternalHostProvider, bool) {
	if p.externalHostProvider != nil {
		return p.externalHostProvider, true
	}
	return nil, false
}

// MustExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalComputeResource) MustExternalHostProvider() *ExternalHostProvider {
	if p.externalHostProvider == nil {
		panic("the externalHostProvider must not be nil, please use ExternalHostProvider() function instead")
	}
	return p.externalHostProvider
}

func (p *ExternalComputeResource) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalComputeResource) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalComputeResource) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalComputeResource) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalComputeResource) SetProvider(attr string) {
	p.provider = &attr
}

func (p *ExternalComputeResource) Provider() (string, bool) {
	if p.provider != nil {
		return *p.provider, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustProvider() string {
	if p.provider == nil {
		panic("the provider must not be nil, please use Provider() function instead")
	}
	return *p.provider
}

func (p *ExternalComputeResource) SetUrl(attr string) {
	p.url = &attr
}

func (p *ExternalComputeResource) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *ExternalComputeResource) SetUser(attr string) {
	p.user = &attr
}

func (p *ExternalComputeResource) User() (string, bool) {
	if p.user != nil {
		return *p.user, true
	}
	var zero string
	return zero, false
}

func (p *ExternalComputeResource) MustUser() string {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return *p.user
}

type ExternalDiscoveredHost struct {
	Struct
	comment              *string
	description          *string
	externalHostProvider *ExternalHostProvider
	id                   *string
	ip                   *string
	lastReport           *string
	mac                  *string
	name                 *string
	subnetName           *string
}

func (p *ExternalDiscoveredHost) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalDiscoveredHost) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalDiscoveredHost) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalDiscoveredHost) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalDiscoveredHost) SetExternalHostProvider(attr *ExternalHostProvider) {
	p.externalHostProvider = attr
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalDiscoveredHost) ExternalHostProvider() (*ExternalHostProvider, bool) {
	if p.externalHostProvider != nil {
		return p.externalHostProvider, true
	}
	return nil, false
}

// MustExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalDiscoveredHost) MustExternalHostProvider() *ExternalHostProvider {
	if p.externalHostProvider == nil {
		panic("the externalHostProvider must not be nil, please use ExternalHostProvider() function instead")
	}
	return p.externalHostProvider
}

func (p *ExternalDiscoveredHost) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalDiscoveredHost) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalDiscoveredHost) SetIp(attr string) {
	p.ip = &attr
}

func (p *ExternalDiscoveredHost) Ip() (string, bool) {
	if p.ip != nil {
		return *p.ip, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustIp() string {
	if p.ip == nil {
		panic("the ip must not be nil, please use Ip() function instead")
	}
	return *p.ip
}

func (p *ExternalDiscoveredHost) SetLastReport(attr string) {
	p.lastReport = &attr
}

func (p *ExternalDiscoveredHost) LastReport() (string, bool) {
	if p.lastReport != nil {
		return *p.lastReport, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustLastReport() string {
	if p.lastReport == nil {
		panic("the lastReport must not be nil, please use LastReport() function instead")
	}
	return *p.lastReport
}

func (p *ExternalDiscoveredHost) SetMac(attr string) {
	p.mac = &attr
}

func (p *ExternalDiscoveredHost) Mac() (string, bool) {
	if p.mac != nil {
		return *p.mac, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustMac() string {
	if p.mac == nil {
		panic("the mac must not be nil, please use Mac() function instead")
	}
	return *p.mac
}

func (p *ExternalDiscoveredHost) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalDiscoveredHost) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalDiscoveredHost) SetSubnetName(attr string) {
	p.subnetName = &attr
}

func (p *ExternalDiscoveredHost) SubnetName() (string, bool) {
	if p.subnetName != nil {
		return *p.subnetName, true
	}
	var zero string
	return zero, false
}

func (p *ExternalDiscoveredHost) MustSubnetName() string {
	if p.subnetName == nil {
		panic("the subnetName must not be nil, please use SubnetName() function instead")
	}
	return *p.subnetName
}

// ExternalHost Represents a host provisioned by a host
// provider (such as Foreman/Satellite).
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
type ExternalHost struct {
	Struct
	address              *string
	comment              *string
	description          *string
	externalHostProvider *ExternalHostProvider
	id                   *string
	name                 *string
}

func (p *ExternalHost) SetAddress(attr string) {
	p.address = &attr
}

func (p *ExternalHost) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHost) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *ExternalHost) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalHost) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHost) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalHost) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalHost) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHost) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHost) SetExternalHostProvider(attr *ExternalHostProvider) {
	p.externalHostProvider = attr
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHost) ExternalHostProvider() (*ExternalHostProvider, bool) {
	if p.externalHostProvider != nil {
		return p.externalHostProvider, true
	}
	return nil, false
}

// MustExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHost) MustExternalHostProvider() *ExternalHostProvider {
	if p.externalHostProvider == nil {
		panic("the externalHostProvider must not be nil, please use ExternalHostProvider() function instead")
	}
	return p.externalHostProvider
}

func (p *ExternalHost) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalHost) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHost) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalHost) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalHost) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHost) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

type ExternalHostGroup struct {
	Struct
	architectureName     *string
	comment              *string
	description          *string
	domainName           *string
	externalHostProvider *ExternalHostProvider
	id                   *string
	name                 *string
	operatingSystemName  *string
	subnetName           *string
}

func (p *ExternalHostGroup) SetArchitectureName(attr string) {
	p.architectureName = &attr
}

func (p *ExternalHostGroup) ArchitectureName() (string, bool) {
	if p.architectureName != nil {
		return *p.architectureName, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustArchitectureName() string {
	if p.architectureName == nil {
		panic("the architectureName must not be nil, please use ArchitectureName() function instead")
	}
	return *p.architectureName
}

func (p *ExternalHostGroup) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalHostGroup) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalHostGroup) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalHostGroup) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *ExternalHostGroup) SetDomainName(attr string) {
	p.domainName = &attr
}

func (p *ExternalHostGroup) DomainName() (string, bool) {
	if p.domainName != nil {
		return *p.domainName, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustDomainName() string {
	if p.domainName == nil {
		panic("the domainName must not be nil, please use DomainName() function instead")
	}
	return *p.domainName
}

// SetExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHostGroup) SetExternalHostProvider(attr *ExternalHostProvider) {
	p.externalHostProvider = attr
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHostGroup) ExternalHostProvider() (*ExternalHostProvider, bool) {
	if p.externalHostProvider != nil {
		return p.externalHostProvider, true
	}
	return nil, false
}

// MustExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *ExternalHostGroup) MustExternalHostProvider() *ExternalHostProvider {
	if p.externalHostProvider == nil {
		panic("the externalHostProvider must not be nil, please use ExternalHostProvider() function instead")
	}
	return p.externalHostProvider
}

func (p *ExternalHostGroup) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalHostGroup) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalHostGroup) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalHostGroup) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalHostGroup) SetOperatingSystemName(attr string) {
	p.operatingSystemName = &attr
}

func (p *ExternalHostGroup) OperatingSystemName() (string, bool) {
	if p.operatingSystemName != nil {
		return *p.operatingSystemName, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustOperatingSystemName() string {
	if p.operatingSystemName == nil {
		panic("the operatingSystemName must not be nil, please use OperatingSystemName() function instead")
	}
	return *p.operatingSystemName
}

func (p *ExternalHostGroup) SetSubnetName(attr string) {
	p.subnetName = &attr
}

func (p *ExternalHostGroup) SubnetName() (string, bool) {
	if p.subnetName != nil {
		return *p.subnetName, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostGroup) MustSubnetName() string {
	if p.subnetName == nil {
		panic("the subnetName must not be nil, please use SubnetName() function instead")
	}
	return *p.subnetName
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
type ExternalHostProvider struct {
	Struct
	authenticationUrl      *string
	certificates           *CertificateSlice
	comment                *string
	computeResources       *ExternalComputeResourceSlice
	description            *string
	discoveredHosts        *ExternalDiscoveredHostSlice
	hostGroups             *ExternalHostGroupSlice
	hosts                  *HostSlice
	id                     *string
	name                   *string
	password               *string
	properties             *PropertySlice
	requiresAuthentication *bool
	url                    *string
	username               *string
}

func (p *ExternalHostProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *ExternalHostProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *ExternalHostProvider) SetCertificates(attr *CertificateSlice) {
	p.certificates = attr
}

func (p *ExternalHostProvider) Certificates() (*CertificateSlice, bool) {
	if p.certificates != nil {
		return p.certificates, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustCertificates() *CertificateSlice {
	if p.certificates == nil {
		panic("the certificates must not be nil, please use Certificates() function instead")
	}
	return p.certificates
}

func (p *ExternalHostProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalHostProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalHostProvider) SetComputeResources(attr *ExternalComputeResourceSlice) {
	p.computeResources = attr
}

func (p *ExternalHostProvider) ComputeResources() (*ExternalComputeResourceSlice, bool) {
	if p.computeResources != nil {
		return p.computeResources, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustComputeResources() *ExternalComputeResourceSlice {
	if p.computeResources == nil {
		panic("the computeResources must not be nil, please use ComputeResources() function instead")
	}
	return p.computeResources
}

func (p *ExternalHostProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalHostProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *ExternalHostProvider) SetDiscoveredHosts(attr *ExternalDiscoveredHostSlice) {
	p.discoveredHosts = attr
}

func (p *ExternalHostProvider) DiscoveredHosts() (*ExternalDiscoveredHostSlice, bool) {
	if p.discoveredHosts != nil {
		return p.discoveredHosts, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustDiscoveredHosts() *ExternalDiscoveredHostSlice {
	if p.discoveredHosts == nil {
		panic("the discoveredHosts must not be nil, please use DiscoveredHosts() function instead")
	}
	return p.discoveredHosts
}

func (p *ExternalHostProvider) SetHostGroups(attr *ExternalHostGroupSlice) {
	p.hostGroups = attr
}

func (p *ExternalHostProvider) HostGroups() (*ExternalHostGroupSlice, bool) {
	if p.hostGroups != nil {
		return p.hostGroups, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustHostGroups() *ExternalHostGroupSlice {
	if p.hostGroups == nil {
		panic("the hostGroups must not be nil, please use HostGroups() function instead")
	}
	return p.hostGroups
}

func (p *ExternalHostProvider) SetHosts(attr *HostSlice) {
	p.hosts = attr
}

func (p *ExternalHostProvider) Hosts() (*HostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustHosts() *HostSlice {
	if p.hosts == nil {
		panic("the hosts must not be nil, please use Hosts() function instead")
	}
	return p.hosts
}

func (p *ExternalHostProvider) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalHostProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalHostProvider) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalHostProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalHostProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *ExternalHostProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *ExternalHostProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *ExternalHostProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *ExternalHostProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *ExternalHostProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *ExternalHostProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *ExternalHostProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *ExternalHostProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *ExternalHostProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *ExternalHostProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *ExternalHostProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *ExternalHostProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

// ExternalNetworkProviderConfiguration Describes how an external network provider is provisioned on a host.
type ExternalNetworkProviderConfiguration struct {
	Struct
	comment                 *string
	description             *string
	externalNetworkProvider *ExternalProvider
	host                    *Host
	id                      *string
	name                    *string
}

func (p *ExternalNetworkProviderConfiguration) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalNetworkProviderConfiguration) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalNetworkProviderConfiguration) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalNetworkProviderConfiguration) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalNetworkProviderConfiguration) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalNetworkProviderConfiguration) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetExternalNetworkProvider Represents an external provider.
func (p *ExternalNetworkProviderConfiguration) SetExternalNetworkProvider(attr *ExternalProvider) {
	p.externalNetworkProvider = attr
}

// ExternalNetworkProvider Represents an external provider.
func (p *ExternalNetworkProviderConfiguration) ExternalNetworkProvider() (*ExternalProvider, bool) {
	if p.externalNetworkProvider != nil {
		return p.externalNetworkProvider, true
	}
	return nil, false
}

// MustExternalNetworkProvider Represents an external provider.
func (p *ExternalNetworkProviderConfiguration) MustExternalNetworkProvider() *ExternalProvider {
	if p.externalNetworkProvider == nil {
		panic("the externalNetworkProvider must not be nil, please use ExternalNetworkProvider() function instead")
	}
	return p.externalNetworkProvider
}

// SetHost Type representing a host.
func (p *ExternalNetworkProviderConfiguration) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *ExternalNetworkProviderConfiguration) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *ExternalNetworkProviderConfiguration) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *ExternalNetworkProviderConfiguration) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalNetworkProviderConfiguration) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalNetworkProviderConfiguration) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalNetworkProviderConfiguration) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalNetworkProviderConfiguration) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalNetworkProviderConfiguration) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// ExternalProvider Represents an external provider.
type ExternalProvider struct {
	Struct
	authenticationUrl      *string
	comment                *string
	description            *string
	id                     *string
	name                   *string
	password               *string
	properties             *PropertySlice
	requiresAuthentication *bool
	url                    *string
	username               *string
}

func (p *ExternalProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *ExternalProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *ExternalProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *ExternalProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ExternalProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *ExternalProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *ExternalProvider) SetId(attr string) {
	p.id = &attr
}

func (p *ExternalProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *ExternalProvider) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *ExternalProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *ExternalProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *ExternalProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *ExternalProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *ExternalProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *ExternalProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *ExternalProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *ExternalProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *ExternalProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *ExternalProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *ExternalProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *ExternalProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

// ExternalTemplateImport Describes the parameters for the template import operation from an external system.
// Currently supports OVA only.
type ExternalTemplateImport struct {
	Struct
	cluster       *Cluster
	cpuProfile    *CpuProfile
	host          *Host
	quota         *Quota
	storageDomain *StorageDomain
	template      *Template
	url           *string
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalTemplateImport) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalTemplateImport) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalTemplateImport) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *ExternalTemplateImport) SetCpuProfile(attr *CpuProfile) {
	p.cpuProfile = attr
}

func (p *ExternalTemplateImport) CpuProfile() (*CpuProfile, bool) {
	if p.cpuProfile != nil {
		return p.cpuProfile, true
	}
	return nil, false
}

func (p *ExternalTemplateImport) MustCpuProfile() *CpuProfile {
	if p.cpuProfile == nil {
		panic("the cpuProfile must not be nil, please use CpuProfile() function instead")
	}
	return p.cpuProfile
}

// SetHost Type representing a host.
func (p *ExternalTemplateImport) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *ExternalTemplateImport) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *ExternalTemplateImport) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalTemplateImport) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalTemplateImport) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalTemplateImport) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalTemplateImport) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalTemplateImport) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalTemplateImport) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *ExternalTemplateImport) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *ExternalTemplateImport) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *ExternalTemplateImport) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *ExternalTemplateImport) SetUrl(attr string) {
	p.url = &attr
}

func (p *ExternalTemplateImport) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *ExternalTemplateImport) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

// ExternalVmImport Describes the parameters for the virtual machine import operation from an external system.
type ExternalVmImport struct {
	Struct
	cluster       *Cluster
	cpuProfile    *CpuProfile
	driversIso    *File
	host          *Host
	name          *string
	password      *string
	provider      *ExternalVmProviderType
	quota         *Quota
	sparse        *bool
	storageDomain *StorageDomain
	url           *string
	username      *string
	vm            *Vm
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalVmImport) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalVmImport) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *ExternalVmImport) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *ExternalVmImport) SetCpuProfile(attr *CpuProfile) {
	p.cpuProfile = attr
}

func (p *ExternalVmImport) CpuProfile() (*CpuProfile, bool) {
	if p.cpuProfile != nil {
		return p.cpuProfile, true
	}
	return nil, false
}

func (p *ExternalVmImport) MustCpuProfile() *CpuProfile {
	if p.cpuProfile == nil {
		panic("the cpuProfile must not be nil, please use CpuProfile() function instead")
	}
	return p.cpuProfile
}

func (p *ExternalVmImport) SetDriversIso(attr *File) {
	p.driversIso = attr
}

func (p *ExternalVmImport) DriversIso() (*File, bool) {
	if p.driversIso != nil {
		return p.driversIso, true
	}
	return nil, false
}

func (p *ExternalVmImport) MustDriversIso() *File {
	if p.driversIso == nil {
		panic("the driversIso must not be nil, please use DriversIso() function instead")
	}
	return p.driversIso
}

// SetHost Type representing a host.
func (p *ExternalVmImport) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *ExternalVmImport) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *ExternalVmImport) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *ExternalVmImport) SetName(attr string) {
	p.name = &attr
}

func (p *ExternalVmImport) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ExternalVmImport) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ExternalVmImport) SetPassword(attr string) {
	p.password = &attr
}

func (p *ExternalVmImport) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *ExternalVmImport) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

// SetProvider Describes the type of external hypervisor system.
func (p *ExternalVmImport) SetProvider(attr ExternalVmProviderType) {
	p.provider = &attr
}

// Provider Describes the type of external hypervisor system.
func (p *ExternalVmImport) Provider() (ExternalVmProviderType, bool) {
	if p.provider != nil {
		return *p.provider, true
	}
	var zero ExternalVmProviderType
	return zero, false
}

// MustProvider Describes the type of external hypervisor system.
func (p *ExternalVmImport) MustProvider() ExternalVmProviderType {
	if p.provider == nil {
		panic("the provider must not be nil, please use Provider() function instead")
	}
	return *p.provider
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalVmImport) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalVmImport) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *ExternalVmImport) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

func (p *ExternalVmImport) SetSparse(attr bool) {
	p.sparse = &attr
}

func (p *ExternalVmImport) Sparse() (bool, bool) {
	if p.sparse != nil {
		return *p.sparse, true
	}
	var zero bool
	return zero, false
}

func (p *ExternalVmImport) MustSparse() bool {
	if p.sparse == nil {
		panic("the sparse must not be nil, please use Sparse() function instead")
	}
	return *p.sparse
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalVmImport) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalVmImport) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *ExternalVmImport) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

func (p *ExternalVmImport) SetUrl(attr string) {
	p.url = &attr
}

func (p *ExternalVmImport) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *ExternalVmImport) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *ExternalVmImport) SetUsername(attr string) {
	p.username = &attr
}

func (p *ExternalVmImport) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *ExternalVmImport) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

// SetVm Represents a virtual machine.
func (p *ExternalVmImport) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *ExternalVmImport) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *ExternalVmImport) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

// FencingPolicy Type representing a cluster fencing policy.
type FencingPolicy struct {
	Struct
	enabled                   *bool
	skipIfConnectivityBroken  *SkipIfConnectivityBroken
	skipIfGlusterBricksUp     *bool
	skipIfGlusterQuorumNotMet *bool
	skipIfSdActive            *SkipIfSdActive
}

func (p *FencingPolicy) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *FencingPolicy) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *FencingPolicy) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

func (p *FencingPolicy) SetSkipIfConnectivityBroken(attr *SkipIfConnectivityBroken) {
	p.skipIfConnectivityBroken = attr
}

func (p *FencingPolicy) SkipIfConnectivityBroken() (*SkipIfConnectivityBroken, bool) {
	if p.skipIfConnectivityBroken != nil {
		return p.skipIfConnectivityBroken, true
	}
	return nil, false
}

func (p *FencingPolicy) MustSkipIfConnectivityBroken() *SkipIfConnectivityBroken {
	if p.skipIfConnectivityBroken == nil {
		panic("the skipIfConnectivityBroken must not be nil, please use SkipIfConnectivityBroken() function instead")
	}
	return p.skipIfConnectivityBroken
}

func (p *FencingPolicy) SetSkipIfGlusterBricksUp(attr bool) {
	p.skipIfGlusterBricksUp = &attr
}

func (p *FencingPolicy) SkipIfGlusterBricksUp() (bool, bool) {
	if p.skipIfGlusterBricksUp != nil {
		return *p.skipIfGlusterBricksUp, true
	}
	var zero bool
	return zero, false
}

func (p *FencingPolicy) MustSkipIfGlusterBricksUp() bool {
	if p.skipIfGlusterBricksUp == nil {
		panic("the skipIfGlusterBricksUp must not be nil, please use SkipIfGlusterBricksUp() function instead")
	}
	return *p.skipIfGlusterBricksUp
}

func (p *FencingPolicy) SetSkipIfGlusterQuorumNotMet(attr bool) {
	p.skipIfGlusterQuorumNotMet = &attr
}

func (p *FencingPolicy) SkipIfGlusterQuorumNotMet() (bool, bool) {
	if p.skipIfGlusterQuorumNotMet != nil {
		return *p.skipIfGlusterQuorumNotMet, true
	}
	var zero bool
	return zero, false
}

func (p *FencingPolicy) MustSkipIfGlusterQuorumNotMet() bool {
	if p.skipIfGlusterQuorumNotMet == nil {
		panic("the skipIfGlusterQuorumNotMet must not be nil, please use SkipIfGlusterQuorumNotMet() function instead")
	}
	return *p.skipIfGlusterQuorumNotMet
}

// SetSkipIfSdActive This type represents the storage related
// configuration in the fencing policy.
func (p *FencingPolicy) SetSkipIfSdActive(attr *SkipIfSdActive) {
	p.skipIfSdActive = attr
}

// SkipIfSdActive This type represents the storage related
// configuration in the fencing policy.
func (p *FencingPolicy) SkipIfSdActive() (*SkipIfSdActive, bool) {
	if p.skipIfSdActive != nil {
		return p.skipIfSdActive, true
	}
	return nil, false
}

// MustSkipIfSdActive This type represents the storage related
// configuration in the fencing policy.
func (p *FencingPolicy) MustSkipIfSdActive() *SkipIfSdActive {
	if p.skipIfSdActive == nil {
		panic("the skipIfSdActive must not be nil, please use SkipIfSdActive() function instead")
	}
	return p.skipIfSdActive
}

type File struct {
	Struct
	comment       *string
	content       *string
	description   *string
	id            *string
	name          *string
	storageDomain *StorageDomain
	type_         *string
}

func (p *File) SetComment(attr string) {
	p.comment = &attr
}

func (p *File) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *File) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *File) SetContent(attr string) {
	p.content = &attr
}

func (p *File) Content() (string, bool) {
	if p.content != nil {
		return *p.content, true
	}
	var zero string
	return zero, false
}

func (p *File) MustContent() string {
	if p.content == nil {
		panic("the content must not be nil, please use Content() function instead")
	}
	return *p.content
}

func (p *File) SetDescription(attr string) {
	p.description = &attr
}

func (p *File) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *File) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *File) SetId(attr string) {
	p.id = &attr
}

func (p *File) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *File) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *File) SetName(attr string) {
	p.name = &attr
}

func (p *File) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *File) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *File) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *File) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *File) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

func (p *File) SetType(attr string) {
	p.type_ = &attr
}

func (p *File) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *File) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type Filter struct {
	Struct
	comment              *string
	description          *string
	id                   *string
	name                 *string
	position             *int64
	schedulingPolicyUnit *SchedulingPolicyUnit
}

func (p *Filter) SetComment(attr string) {
	p.comment = &attr
}

func (p *Filter) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Filter) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Filter) SetDescription(attr string) {
	p.description = &attr
}

func (p *Filter) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Filter) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Filter) SetId(attr string) {
	p.id = &attr
}

func (p *Filter) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Filter) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Filter) SetName(attr string) {
	p.name = &attr
}

func (p *Filter) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Filter) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Filter) SetPosition(attr int64) {
	p.position = &attr
}

func (p *Filter) Position() (int64, bool) {
	if p.position != nil {
		return *p.position, true
	}
	var zero int64
	return zero, false
}

func (p *Filter) MustPosition() int64 {
	if p.position == nil {
		panic("the position must not be nil, please use Position() function instead")
	}
	return *p.position
}

func (p *Filter) SetSchedulingPolicyUnit(attr *SchedulingPolicyUnit) {
	p.schedulingPolicyUnit = attr
}

func (p *Filter) SchedulingPolicyUnit() (*SchedulingPolicyUnit, bool) {
	if p.schedulingPolicyUnit != nil {
		return p.schedulingPolicyUnit, true
	}
	return nil, false
}

func (p *Filter) MustSchedulingPolicyUnit() *SchedulingPolicyUnit {
	if p.schedulingPolicyUnit == nil {
		panic("the schedulingPolicyUnit must not be nil, please use SchedulingPolicyUnit() function instead")
	}
	return p.schedulingPolicyUnit
}

// Floppy The underlying representation of a floppy file.
type Floppy struct {
	Struct
	comment      *string
	description  *string
	file         *File
	id           *string
	instanceType *InstanceType
	name         *string
	template     *Template
	vm           *Vm
	vms          *VmSlice
}

func (p *Floppy) SetComment(attr string) {
	p.comment = &attr
}

func (p *Floppy) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Floppy) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Floppy) SetDescription(attr string) {
	p.description = &attr
}

func (p *Floppy) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Floppy) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Floppy) SetFile(attr *File) {
	p.file = attr
}

func (p *Floppy) File() (*File, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *Floppy) MustFile() *File {
	if p.file == nil {
		panic("the file must not be nil, please use File() function instead")
	}
	return p.file
}

func (p *Floppy) SetId(attr string) {
	p.id = &attr
}

func (p *Floppy) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Floppy) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Floppy) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Floppy) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Floppy) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *Floppy) SetName(attr string) {
	p.name = &attr
}

func (p *Floppy) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Floppy) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Floppy) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Floppy) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Floppy) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetVm Represents a virtual machine.
func (p *Floppy) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Floppy) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Floppy) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *Floppy) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Floppy) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Floppy) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

type FopStatistic struct {
	Struct
	name       *string
	statistics *StatisticSlice
}

func (p *FopStatistic) SetName(attr string) {
	p.name = &attr
}

func (p *FopStatistic) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *FopStatistic) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *FopStatistic) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *FopStatistic) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *FopStatistic) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

type GlusterBrick struct {
	Struct
	brickDir       *string
	comment        *string
	description    *string
	device         *string
	fsName         *string
	glusterClients *GlusterClientSlice
	glusterVolume  *GlusterVolume
	id             *string
	instanceType   *InstanceType
	memoryPools    *GlusterMemoryPoolSlice
	mntOptions     *string
	name           *string
	pid            *int64
	port           *int64
	serverId       *string
	statistics     *StatisticSlice
	status         *GlusterBrickStatus
	template       *Template
	vm             *Vm
	vms            *VmSlice
}

func (p *GlusterBrick) SetBrickDir(attr string) {
	p.brickDir = &attr
}

func (p *GlusterBrick) BrickDir() (string, bool) {
	if p.brickDir != nil {
		return *p.brickDir, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustBrickDir() string {
	if p.brickDir == nil {
		panic("the brickDir must not be nil, please use BrickDir() function instead")
	}
	return *p.brickDir
}

func (p *GlusterBrick) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterBrick) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterBrick) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterBrick) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterBrick) SetDevice(attr string) {
	p.device = &attr
}

func (p *GlusterBrick) Device() (string, bool) {
	if p.device != nil {
		return *p.device, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustDevice() string {
	if p.device == nil {
		panic("the device must not be nil, please use Device() function instead")
	}
	return *p.device
}

func (p *GlusterBrick) SetFsName(attr string) {
	p.fsName = &attr
}

func (p *GlusterBrick) FsName() (string, bool) {
	if p.fsName != nil {
		return *p.fsName, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustFsName() string {
	if p.fsName == nil {
		panic("the fsName must not be nil, please use FsName() function instead")
	}
	return *p.fsName
}

func (p *GlusterBrick) SetGlusterClients(attr *GlusterClientSlice) {
	p.glusterClients = attr
}

func (p *GlusterBrick) GlusterClients() (*GlusterClientSlice, bool) {
	if p.glusterClients != nil {
		return p.glusterClients, true
	}
	return nil, false
}

func (p *GlusterBrick) MustGlusterClients() *GlusterClientSlice {
	if p.glusterClients == nil {
		panic("the glusterClients must not be nil, please use GlusterClients() function instead")
	}
	return p.glusterClients
}

func (p *GlusterBrick) SetGlusterVolume(attr *GlusterVolume) {
	p.glusterVolume = attr
}

func (p *GlusterBrick) GlusterVolume() (*GlusterVolume, bool) {
	if p.glusterVolume != nil {
		return p.glusterVolume, true
	}
	return nil, false
}

func (p *GlusterBrick) MustGlusterVolume() *GlusterVolume {
	if p.glusterVolume == nil {
		panic("the glusterVolume must not be nil, please use GlusterVolume() function instead")
	}
	return p.glusterVolume
}

func (p *GlusterBrick) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterBrick) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrick) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrick) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrick) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *GlusterBrick) SetMemoryPools(attr *GlusterMemoryPoolSlice) {
	p.memoryPools = attr
}

func (p *GlusterBrick) MemoryPools() (*GlusterMemoryPoolSlice, bool) {
	if p.memoryPools != nil {
		return p.memoryPools, true
	}
	return nil, false
}

func (p *GlusterBrick) MustMemoryPools() *GlusterMemoryPoolSlice {
	if p.memoryPools == nil {
		panic("the memoryPools must not be nil, please use MemoryPools() function instead")
	}
	return p.memoryPools
}

func (p *GlusterBrick) SetMntOptions(attr string) {
	p.mntOptions = &attr
}

func (p *GlusterBrick) MntOptions() (string, bool) {
	if p.mntOptions != nil {
		return *p.mntOptions, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustMntOptions() string {
	if p.mntOptions == nil {
		panic("the mntOptions must not be nil, please use MntOptions() function instead")
	}
	return *p.mntOptions
}

func (p *GlusterBrick) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterBrick) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterBrick) SetPid(attr int64) {
	p.pid = &attr
}

func (p *GlusterBrick) Pid() (int64, bool) {
	if p.pid != nil {
		return *p.pid, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterBrick) MustPid() int64 {
	if p.pid == nil {
		panic("the pid must not be nil, please use Pid() function instead")
	}
	return *p.pid
}

func (p *GlusterBrick) SetPort(attr int64) {
	p.port = &attr
}

func (p *GlusterBrick) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterBrick) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *GlusterBrick) SetServerId(attr string) {
	p.serverId = &attr
}

func (p *GlusterBrick) ServerId() (string, bool) {
	if p.serverId != nil {
		return *p.serverId, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrick) MustServerId() string {
	if p.serverId == nil {
		panic("the serverId must not be nil, please use ServerId() function instead")
	}
	return *p.serverId
}

func (p *GlusterBrick) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *GlusterBrick) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *GlusterBrick) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

func (p *GlusterBrick) SetStatus(attr GlusterBrickStatus) {
	p.status = &attr
}

func (p *GlusterBrick) Status() (GlusterBrickStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero GlusterBrickStatus
	return zero, false
}

func (p *GlusterBrick) MustStatus() GlusterBrickStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrick) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrick) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrick) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetVm Represents a virtual machine.
func (p *GlusterBrick) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *GlusterBrick) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *GlusterBrick) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *GlusterBrick) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *GlusterBrick) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *GlusterBrick) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

type GlusterBrickAdvancedDetails struct {
	Struct
	comment        *string
	description    *string
	device         *string
	fsName         *string
	glusterClients *GlusterClientSlice
	id             *string
	instanceType   *InstanceType
	memoryPools    *GlusterMemoryPoolSlice
	mntOptions     *string
	name           *string
	pid            *int64
	port           *int64
	template       *Template
	vm             *Vm
	vms            *VmSlice
}

func (p *GlusterBrickAdvancedDetails) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterBrickAdvancedDetails) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterBrickAdvancedDetails) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterBrickAdvancedDetails) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterBrickAdvancedDetails) SetDevice(attr string) {
	p.device = &attr
}

func (p *GlusterBrickAdvancedDetails) Device() (string, bool) {
	if p.device != nil {
		return *p.device, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustDevice() string {
	if p.device == nil {
		panic("the device must not be nil, please use Device() function instead")
	}
	return *p.device
}

func (p *GlusterBrickAdvancedDetails) SetFsName(attr string) {
	p.fsName = &attr
}

func (p *GlusterBrickAdvancedDetails) FsName() (string, bool) {
	if p.fsName != nil {
		return *p.fsName, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustFsName() string {
	if p.fsName == nil {
		panic("the fsName must not be nil, please use FsName() function instead")
	}
	return *p.fsName
}

func (p *GlusterBrickAdvancedDetails) SetGlusterClients(attr *GlusterClientSlice) {
	p.glusterClients = attr
}

func (p *GlusterBrickAdvancedDetails) GlusterClients() (*GlusterClientSlice, bool) {
	if p.glusterClients != nil {
		return p.glusterClients, true
	}
	return nil, false
}

func (p *GlusterBrickAdvancedDetails) MustGlusterClients() *GlusterClientSlice {
	if p.glusterClients == nil {
		panic("the glusterClients must not be nil, please use GlusterClients() function instead")
	}
	return p.glusterClients
}

func (p *GlusterBrickAdvancedDetails) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterBrickAdvancedDetails) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrickAdvancedDetails) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrickAdvancedDetails) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GlusterBrickAdvancedDetails) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *GlusterBrickAdvancedDetails) SetMemoryPools(attr *GlusterMemoryPoolSlice) {
	p.memoryPools = attr
}

func (p *GlusterBrickAdvancedDetails) MemoryPools() (*GlusterMemoryPoolSlice, bool) {
	if p.memoryPools != nil {
		return p.memoryPools, true
	}
	return nil, false
}

func (p *GlusterBrickAdvancedDetails) MustMemoryPools() *GlusterMemoryPoolSlice {
	if p.memoryPools == nil {
		panic("the memoryPools must not be nil, please use MemoryPools() function instead")
	}
	return p.memoryPools
}

func (p *GlusterBrickAdvancedDetails) SetMntOptions(attr string) {
	p.mntOptions = &attr
}

func (p *GlusterBrickAdvancedDetails) MntOptions() (string, bool) {
	if p.mntOptions != nil {
		return *p.mntOptions, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustMntOptions() string {
	if p.mntOptions == nil {
		panic("the mntOptions must not be nil, please use MntOptions() function instead")
	}
	return *p.mntOptions
}

func (p *GlusterBrickAdvancedDetails) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterBrickAdvancedDetails) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterBrickAdvancedDetails) SetPid(attr int64) {
	p.pid = &attr
}

func (p *GlusterBrickAdvancedDetails) Pid() (int64, bool) {
	if p.pid != nil {
		return *p.pid, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustPid() int64 {
	if p.pid == nil {
		panic("the pid must not be nil, please use Pid() function instead")
	}
	return *p.pid
}

func (p *GlusterBrickAdvancedDetails) SetPort(attr int64) {
	p.port = &attr
}

func (p *GlusterBrickAdvancedDetails) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterBrickAdvancedDetails) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrickAdvancedDetails) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrickAdvancedDetails) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GlusterBrickAdvancedDetails) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetVm Represents a virtual machine.
func (p *GlusterBrickAdvancedDetails) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *GlusterBrickAdvancedDetails) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *GlusterBrickAdvancedDetails) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *GlusterBrickAdvancedDetails) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *GlusterBrickAdvancedDetails) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *GlusterBrickAdvancedDetails) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

type GlusterBrickMemoryInfo struct {
	Struct
	memoryPools *GlusterMemoryPoolSlice
}

func (p *GlusterBrickMemoryInfo) SetMemoryPools(attr *GlusterMemoryPoolSlice) {
	p.memoryPools = attr
}

func (p *GlusterBrickMemoryInfo) MemoryPools() (*GlusterMemoryPoolSlice, bool) {
	if p.memoryPools != nil {
		return p.memoryPools, true
	}
	return nil, false
}

func (p *GlusterBrickMemoryInfo) MustMemoryPools() *GlusterMemoryPoolSlice {
	if p.memoryPools == nil {
		panic("the memoryPools must not be nil, please use MemoryPools() function instead")
	}
	return p.memoryPools
}

type GlusterClient struct {
	Struct
	bytesRead    *int64
	bytesWritten *int64
	clientPort   *int64
	hostName     *string
}

func (p *GlusterClient) SetBytesRead(attr int64) {
	p.bytesRead = &attr
}

func (p *GlusterClient) BytesRead() (int64, bool) {
	if p.bytesRead != nil {
		return *p.bytesRead, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterClient) MustBytesRead() int64 {
	if p.bytesRead == nil {
		panic("the bytesRead must not be nil, please use BytesRead() function instead")
	}
	return *p.bytesRead
}

func (p *GlusterClient) SetBytesWritten(attr int64) {
	p.bytesWritten = &attr
}

func (p *GlusterClient) BytesWritten() (int64, bool) {
	if p.bytesWritten != nil {
		return *p.bytesWritten, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterClient) MustBytesWritten() int64 {
	if p.bytesWritten == nil {
		panic("the bytesWritten must not be nil, please use BytesWritten() function instead")
	}
	return *p.bytesWritten
}

func (p *GlusterClient) SetClientPort(attr int64) {
	p.clientPort = &attr
}

func (p *GlusterClient) ClientPort() (int64, bool) {
	if p.clientPort != nil {
		return *p.clientPort, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterClient) MustClientPort() int64 {
	if p.clientPort == nil {
		panic("the clientPort must not be nil, please use ClientPort() function instead")
	}
	return *p.clientPort
}

func (p *GlusterClient) SetHostName(attr string) {
	p.hostName = &attr
}

func (p *GlusterClient) HostName() (string, bool) {
	if p.hostName != nil {
		return *p.hostName, true
	}
	var zero string
	return zero, false
}

func (p *GlusterClient) MustHostName() string {
	if p.hostName == nil {
		panic("the hostName must not be nil, please use HostName() function instead")
	}
	return *p.hostName
}

type GlusterHook struct {
	Struct
	checksum       *string
	cluster        *Cluster
	comment        *string
	conflictStatus *int64
	conflicts      *string
	content        *string
	contentType    *HookContentType
	description    *string
	glusterCommand *string
	id             *string
	name           *string
	serverHooks    *GlusterServerHookSlice
	stage          *HookStage
	status         *GlusterHookStatus
}

func (p *GlusterHook) SetChecksum(attr string) {
	p.checksum = &attr
}

func (p *GlusterHook) Checksum() (string, bool) {
	if p.checksum != nil {
		return *p.checksum, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustChecksum() string {
	if p.checksum == nil {
		panic("the checksum must not be nil, please use Checksum() function instead")
	}
	return *p.checksum
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterHook) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterHook) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterHook) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *GlusterHook) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterHook) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterHook) SetConflictStatus(attr int64) {
	p.conflictStatus = &attr
}

func (p *GlusterHook) ConflictStatus() (int64, bool) {
	if p.conflictStatus != nil {
		return *p.conflictStatus, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterHook) MustConflictStatus() int64 {
	if p.conflictStatus == nil {
		panic("the conflictStatus must not be nil, please use ConflictStatus() function instead")
	}
	return *p.conflictStatus
}

func (p *GlusterHook) SetConflicts(attr string) {
	p.conflicts = &attr
}

func (p *GlusterHook) Conflicts() (string, bool) {
	if p.conflicts != nil {
		return *p.conflicts, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustConflicts() string {
	if p.conflicts == nil {
		panic("the conflicts must not be nil, please use Conflicts() function instead")
	}
	return *p.conflicts
}

func (p *GlusterHook) SetContent(attr string) {
	p.content = &attr
}

func (p *GlusterHook) Content() (string, bool) {
	if p.content != nil {
		return *p.content, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustContent() string {
	if p.content == nil {
		panic("the content must not be nil, please use Content() function instead")
	}
	return *p.content
}

// SetContentType Represents content type of hook script.
func (p *GlusterHook) SetContentType(attr HookContentType) {
	p.contentType = &attr
}

// ContentType Represents content type of hook script.
func (p *GlusterHook) ContentType() (HookContentType, bool) {
	if p.contentType != nil {
		return *p.contentType, true
	}
	var zero HookContentType
	return zero, false
}

// MustContentType Represents content type of hook script.
func (p *GlusterHook) MustContentType() HookContentType {
	if p.contentType == nil {
		panic("the contentType must not be nil, please use ContentType() function instead")
	}
	return *p.contentType
}

func (p *GlusterHook) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterHook) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterHook) SetGlusterCommand(attr string) {
	p.glusterCommand = &attr
}

func (p *GlusterHook) GlusterCommand() (string, bool) {
	if p.glusterCommand != nil {
		return *p.glusterCommand, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustGlusterCommand() string {
	if p.glusterCommand == nil {
		panic("the glusterCommand must not be nil, please use GlusterCommand() function instead")
	}
	return *p.glusterCommand
}

func (p *GlusterHook) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterHook) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *GlusterHook) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterHook) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterHook) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterHook) SetServerHooks(attr *GlusterServerHookSlice) {
	p.serverHooks = attr
}

func (p *GlusterHook) ServerHooks() (*GlusterServerHookSlice, bool) {
	if p.serverHooks != nil {
		return p.serverHooks, true
	}
	return nil, false
}

func (p *GlusterHook) MustServerHooks() *GlusterServerHookSlice {
	if p.serverHooks == nil {
		panic("the serverHooks must not be nil, please use ServerHooks() function instead")
	}
	return p.serverHooks
}

// SetStage Type represents a stage of volume event at which hook executes.
func (p *GlusterHook) SetStage(attr HookStage) {
	p.stage = &attr
}

// Stage Type represents a stage of volume event at which hook executes.
func (p *GlusterHook) Stage() (HookStage, bool) {
	if p.stage != nil {
		return *p.stage, true
	}
	var zero HookStage
	return zero, false
}

// MustStage Type represents a stage of volume event at which hook executes.
func (p *GlusterHook) MustStage() HookStage {
	if p.stage == nil {
		panic("the stage must not be nil, please use Stage() function instead")
	}
	return *p.stage
}

func (p *GlusterHook) SetStatus(attr GlusterHookStatus) {
	p.status = &attr
}

func (p *GlusterHook) Status() (GlusterHookStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero GlusterHookStatus
	return zero, false
}

func (p *GlusterHook) MustStatus() GlusterHookStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

type GlusterMemoryPool struct {
	Struct
	allocCount  *int64
	coldCount   *int64
	comment     *string
	description *string
	hotCount    *int64
	id          *string
	maxAlloc    *int64
	maxStdalloc *int64
	name        *string
	paddedSize  *int64
	poolMisses  *int64
	type_       *string
}

func (p *GlusterMemoryPool) SetAllocCount(attr int64) {
	p.allocCount = &attr
}

func (p *GlusterMemoryPool) AllocCount() (int64, bool) {
	if p.allocCount != nil {
		return *p.allocCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustAllocCount() int64 {
	if p.allocCount == nil {
		panic("the allocCount must not be nil, please use AllocCount() function instead")
	}
	return *p.allocCount
}

func (p *GlusterMemoryPool) SetColdCount(attr int64) {
	p.coldCount = &attr
}

func (p *GlusterMemoryPool) ColdCount() (int64, bool) {
	if p.coldCount != nil {
		return *p.coldCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustColdCount() int64 {
	if p.coldCount == nil {
		panic("the coldCount must not be nil, please use ColdCount() function instead")
	}
	return *p.coldCount
}

func (p *GlusterMemoryPool) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterMemoryPool) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterMemoryPool) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterMemoryPool) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterMemoryPool) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterMemoryPool) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterMemoryPool) SetHotCount(attr int64) {
	p.hotCount = &attr
}

func (p *GlusterMemoryPool) HotCount() (int64, bool) {
	if p.hotCount != nil {
		return *p.hotCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustHotCount() int64 {
	if p.hotCount == nil {
		panic("the hotCount must not be nil, please use HotCount() function instead")
	}
	return *p.hotCount
}

func (p *GlusterMemoryPool) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterMemoryPool) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterMemoryPool) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *GlusterMemoryPool) SetMaxAlloc(attr int64) {
	p.maxAlloc = &attr
}

func (p *GlusterMemoryPool) MaxAlloc() (int64, bool) {
	if p.maxAlloc != nil {
		return *p.maxAlloc, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustMaxAlloc() int64 {
	if p.maxAlloc == nil {
		panic("the maxAlloc must not be nil, please use MaxAlloc() function instead")
	}
	return *p.maxAlloc
}

func (p *GlusterMemoryPool) SetMaxStdalloc(attr int64) {
	p.maxStdalloc = &attr
}

func (p *GlusterMemoryPool) MaxStdalloc() (int64, bool) {
	if p.maxStdalloc != nil {
		return *p.maxStdalloc, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustMaxStdalloc() int64 {
	if p.maxStdalloc == nil {
		panic("the maxStdalloc must not be nil, please use MaxStdalloc() function instead")
	}
	return *p.maxStdalloc
}

func (p *GlusterMemoryPool) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterMemoryPool) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterMemoryPool) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterMemoryPool) SetPaddedSize(attr int64) {
	p.paddedSize = &attr
}

func (p *GlusterMemoryPool) PaddedSize() (int64, bool) {
	if p.paddedSize != nil {
		return *p.paddedSize, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustPaddedSize() int64 {
	if p.paddedSize == nil {
		panic("the paddedSize must not be nil, please use PaddedSize() function instead")
	}
	return *p.paddedSize
}

func (p *GlusterMemoryPool) SetPoolMisses(attr int64) {
	p.poolMisses = &attr
}

func (p *GlusterMemoryPool) PoolMisses() (int64, bool) {
	if p.poolMisses != nil {
		return *p.poolMisses, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterMemoryPool) MustPoolMisses() int64 {
	if p.poolMisses == nil {
		panic("the poolMisses must not be nil, please use PoolMisses() function instead")
	}
	return *p.poolMisses
}

func (p *GlusterMemoryPool) SetType(attr string) {
	p.type_ = &attr
}

func (p *GlusterMemoryPool) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *GlusterMemoryPool) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type GlusterServerHook struct {
	Struct
	checksum    *string
	comment     *string
	contentType *HookContentType
	description *string
	host        *Host
	id          *string
	name        *string
	status      *GlusterHookStatus
}

func (p *GlusterServerHook) SetChecksum(attr string) {
	p.checksum = &attr
}

func (p *GlusterServerHook) Checksum() (string, bool) {
	if p.checksum != nil {
		return *p.checksum, true
	}
	var zero string
	return zero, false
}

func (p *GlusterServerHook) MustChecksum() string {
	if p.checksum == nil {
		panic("the checksum must not be nil, please use Checksum() function instead")
	}
	return *p.checksum
}

func (p *GlusterServerHook) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterServerHook) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterServerHook) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

// SetContentType Represents content type of hook script.
func (p *GlusterServerHook) SetContentType(attr HookContentType) {
	p.contentType = &attr
}

// ContentType Represents content type of hook script.
func (p *GlusterServerHook) ContentType() (HookContentType, bool) {
	if p.contentType != nil {
		return *p.contentType, true
	}
	var zero HookContentType
	return zero, false
}

// MustContentType Represents content type of hook script.
func (p *GlusterServerHook) MustContentType() HookContentType {
	if p.contentType == nil {
		panic("the contentType must not be nil, please use ContentType() function instead")
	}
	return *p.contentType
}

func (p *GlusterServerHook) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterServerHook) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterServerHook) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetHost Type representing a host.
func (p *GlusterServerHook) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *GlusterServerHook) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *GlusterServerHook) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *GlusterServerHook) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterServerHook) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterServerHook) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *GlusterServerHook) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterServerHook) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterServerHook) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterServerHook) SetStatus(attr GlusterHookStatus) {
	p.status = &attr
}

func (p *GlusterServerHook) Status() (GlusterHookStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero GlusterHookStatus
	return zero, false
}

func (p *GlusterServerHook) MustStatus() GlusterHookStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

type GlusterVolume struct {
	Struct
	bricks          *GlusterBrickSlice
	cluster         *Cluster
	comment         *string
	description     *string
	disperseCount   *int64
	id              *string
	name            *string
	options         *OptionSlice
	redundancyCount *int64
	replicaCount    *int64
	statistics      *StatisticSlice
	status          *GlusterVolumeStatus
	stripeCount     *int64
	transportTypes  []TransportType
	volumeType      *GlusterVolumeType
}

func (p *GlusterVolume) SetBricks(attr *GlusterBrickSlice) {
	p.bricks = attr
}

func (p *GlusterVolume) Bricks() (*GlusterBrickSlice, bool) {
	if p.bricks != nil {
		return p.bricks, true
	}
	return nil, false
}

func (p *GlusterVolume) MustBricks() *GlusterBrickSlice {
	if p.bricks == nil {
		panic("the bricks must not be nil, please use Bricks() function instead")
	}
	return p.bricks
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterVolume) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterVolume) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *GlusterVolume) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *GlusterVolume) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterVolume) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolume) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterVolume) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterVolume) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolume) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterVolume) SetDisperseCount(attr int64) {
	p.disperseCount = &attr
}

func (p *GlusterVolume) DisperseCount() (int64, bool) {
	if p.disperseCount != nil {
		return *p.disperseCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterVolume) MustDisperseCount() int64 {
	if p.disperseCount == nil {
		panic("the disperseCount must not be nil, please use DisperseCount() function instead")
	}
	return *p.disperseCount
}

func (p *GlusterVolume) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterVolume) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolume) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *GlusterVolume) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterVolume) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolume) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterVolume) SetOptions(attr *OptionSlice) {
	p.options = attr
}

func (p *GlusterVolume) Options() (*OptionSlice, bool) {
	if p.options != nil {
		return p.options, true
	}
	return nil, false
}

func (p *GlusterVolume) MustOptions() *OptionSlice {
	if p.options == nil {
		panic("the options must not be nil, please use Options() function instead")
	}
	return p.options
}

func (p *GlusterVolume) SetRedundancyCount(attr int64) {
	p.redundancyCount = &attr
}

func (p *GlusterVolume) RedundancyCount() (int64, bool) {
	if p.redundancyCount != nil {
		return *p.redundancyCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterVolume) MustRedundancyCount() int64 {
	if p.redundancyCount == nil {
		panic("the redundancyCount must not be nil, please use RedundancyCount() function instead")
	}
	return *p.redundancyCount
}

func (p *GlusterVolume) SetReplicaCount(attr int64) {
	p.replicaCount = &attr
}

func (p *GlusterVolume) ReplicaCount() (int64, bool) {
	if p.replicaCount != nil {
		return *p.replicaCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterVolume) MustReplicaCount() int64 {
	if p.replicaCount == nil {
		panic("the replicaCount must not be nil, please use ReplicaCount() function instead")
	}
	return *p.replicaCount
}

func (p *GlusterVolume) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *GlusterVolume) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *GlusterVolume) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

func (p *GlusterVolume) SetStatus(attr GlusterVolumeStatus) {
	p.status = &attr
}

func (p *GlusterVolume) Status() (GlusterVolumeStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero GlusterVolumeStatus
	return zero, false
}

func (p *GlusterVolume) MustStatus() GlusterVolumeStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *GlusterVolume) SetStripeCount(attr int64) {
	p.stripeCount = &attr
}

func (p *GlusterVolume) StripeCount() (int64, bool) {
	if p.stripeCount != nil {
		return *p.stripeCount, true
	}
	var zero int64
	return zero, false
}

func (p *GlusterVolume) MustStripeCount() int64 {
	if p.stripeCount == nil {
		panic("the stripeCount must not be nil, please use StripeCount() function instead")
	}
	return *p.stripeCount
}

func (p *GlusterVolume) SetTransportTypes(attr []TransportType) {
	p.transportTypes = attr
}

func (p *GlusterVolume) TransportTypes() ([]TransportType, bool) {
	if p.transportTypes != nil {
		return p.transportTypes, true
	}
	return nil, false
}

func (p *GlusterVolume) MustTransportTypes() []TransportType {
	if p.transportTypes == nil {
		panic("the transportTypes must not be nil, please use TransportTypes() function instead")
	}
	return p.transportTypes
}

// SetVolumeType Type representing the type of Gluster Volume.
func (p *GlusterVolume) SetVolumeType(attr GlusterVolumeType) {
	p.volumeType = &attr
}

// VolumeType Type representing the type of Gluster Volume.
func (p *GlusterVolume) VolumeType() (GlusterVolumeType, bool) {
	if p.volumeType != nil {
		return *p.volumeType, true
	}
	var zero GlusterVolumeType
	return zero, false
}

// MustVolumeType Type representing the type of Gluster Volume.
func (p *GlusterVolume) MustVolumeType() GlusterVolumeType {
	if p.volumeType == nil {
		panic("the volumeType must not be nil, please use VolumeType() function instead")
	}
	return *p.volumeType
}

type GlusterVolumeProfileDetails struct {
	Struct
	brickProfileDetails *BrickProfileDetailSlice
	comment             *string
	description         *string
	id                  *string
	name                *string
	nfsProfileDetails   *NfsProfileDetailSlice
}

func (p *GlusterVolumeProfileDetails) SetBrickProfileDetails(attr *BrickProfileDetailSlice) {
	p.brickProfileDetails = attr
}

func (p *GlusterVolumeProfileDetails) BrickProfileDetails() (*BrickProfileDetailSlice, bool) {
	if p.brickProfileDetails != nil {
		return p.brickProfileDetails, true
	}
	return nil, false
}

func (p *GlusterVolumeProfileDetails) MustBrickProfileDetails() *BrickProfileDetailSlice {
	if p.brickProfileDetails == nil {
		panic("the brickProfileDetails must not be nil, please use BrickProfileDetails() function instead")
	}
	return p.brickProfileDetails
}

func (p *GlusterVolumeProfileDetails) SetComment(attr string) {
	p.comment = &attr
}

func (p *GlusterVolumeProfileDetails) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolumeProfileDetails) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GlusterVolumeProfileDetails) SetDescription(attr string) {
	p.description = &attr
}

func (p *GlusterVolumeProfileDetails) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolumeProfileDetails) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GlusterVolumeProfileDetails) SetId(attr string) {
	p.id = &attr
}

func (p *GlusterVolumeProfileDetails) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolumeProfileDetails) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *GlusterVolumeProfileDetails) SetName(attr string) {
	p.name = &attr
}

func (p *GlusterVolumeProfileDetails) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GlusterVolumeProfileDetails) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GlusterVolumeProfileDetails) SetNfsProfileDetails(attr *NfsProfileDetailSlice) {
	p.nfsProfileDetails = attr
}

func (p *GlusterVolumeProfileDetails) NfsProfileDetails() (*NfsProfileDetailSlice, bool) {
	if p.nfsProfileDetails != nil {
		return p.nfsProfileDetails, true
	}
	return nil, false
}

func (p *GlusterVolumeProfileDetails) MustNfsProfileDetails() *NfsProfileDetailSlice {
	if p.nfsProfileDetails == nil {
		panic("the nfsProfileDetails must not be nil, please use NfsProfileDetails() function instead")
	}
	return p.nfsProfileDetails
}

type GraphicsConsole struct {
	Struct
	address      *string
	comment      *string
	description  *string
	id           *string
	instanceType *InstanceType
	name         *string
	port         *int64
	protocol     *GraphicsType
	template     *Template
	tlsPort      *int64
	vm           *Vm
}

func (p *GraphicsConsole) SetAddress(attr string) {
	p.address = &attr
}

func (p *GraphicsConsole) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *GraphicsConsole) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *GraphicsConsole) SetComment(attr string) {
	p.comment = &attr
}

func (p *GraphicsConsole) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *GraphicsConsole) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *GraphicsConsole) SetDescription(attr string) {
	p.description = &attr
}

func (p *GraphicsConsole) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *GraphicsConsole) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *GraphicsConsole) SetId(attr string) {
	p.id = &attr
}

func (p *GraphicsConsole) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *GraphicsConsole) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GraphicsConsole) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GraphicsConsole) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *GraphicsConsole) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

func (p *GraphicsConsole) SetName(attr string) {
	p.name = &attr
}

func (p *GraphicsConsole) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *GraphicsConsole) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *GraphicsConsole) SetPort(attr int64) {
	p.port = &attr
}

func (p *GraphicsConsole) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *GraphicsConsole) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

// SetProtocol The graphics protocol used to connect to the graphic console.
func (p *GraphicsConsole) SetProtocol(attr GraphicsType) {
	p.protocol = &attr
}

// Protocol The graphics protocol used to connect to the graphic console.
func (p *GraphicsConsole) Protocol() (GraphicsType, bool) {
	if p.protocol != nil {
		return *p.protocol, true
	}
	var zero GraphicsType
	return zero, false
}

// MustProtocol The graphics protocol used to connect to the graphic console.
func (p *GraphicsConsole) MustProtocol() GraphicsType {
	if p.protocol == nil {
		panic("the protocol must not be nil, please use Protocol() function instead")
	}
	return *p.protocol
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GraphicsConsole) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GraphicsConsole) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *GraphicsConsole) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *GraphicsConsole) SetTlsPort(attr int64) {
	p.tlsPort = &attr
}

func (p *GraphicsConsole) TlsPort() (int64, bool) {
	if p.tlsPort != nil {
		return *p.tlsPort, true
	}
	var zero int64
	return zero, false
}

func (p *GraphicsConsole) MustTlsPort() int64 {
	if p.tlsPort == nil {
		panic("the tlsPort must not be nil, please use TlsPort() function instead")
	}
	return *p.tlsPort
}

// SetVm Represents a virtual machine.
func (p *GraphicsConsole) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *GraphicsConsole) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *GraphicsConsole) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

// Group This type represents all groups in the directory service.
type Group struct {
	Struct
	comment       *string
	description   *string
	domain        *Domain
	domainEntryId *string
	id            *string
	name          *string
	namespace     *string
	permissions   *PermissionSlice
	roles         *RoleSlice
	tags          *TagSlice
}

func (p *Group) SetComment(attr string) {
	p.comment = &attr
}

func (p *Group) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Group) SetDescription(attr string) {
	p.description = &attr
}

func (p *Group) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDomain This type represents a directory service domain.
func (p *Group) SetDomain(attr *Domain) {
	p.domain = attr
}

// Domain This type represents a directory service domain.
func (p *Group) Domain() (*Domain, bool) {
	if p.domain != nil {
		return p.domain, true
	}
	return nil, false
}

// MustDomain This type represents a directory service domain.
func (p *Group) MustDomain() *Domain {
	if p.domain == nil {
		panic("the domain must not be nil, please use Domain() function instead")
	}
	return p.domain
}

func (p *Group) SetDomainEntryId(attr string) {
	p.domainEntryId = &attr
}

func (p *Group) DomainEntryId() (string, bool) {
	if p.domainEntryId != nil {
		return *p.domainEntryId, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustDomainEntryId() string {
	if p.domainEntryId == nil {
		panic("the domainEntryId must not be nil, please use DomainEntryId() function instead")
	}
	return *p.domainEntryId
}

func (p *Group) SetId(attr string) {
	p.id = &attr
}

func (p *Group) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Group) SetName(attr string) {
	p.name = &attr
}

func (p *Group) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Group) SetNamespace(attr string) {
	p.namespace = &attr
}

func (p *Group) Namespace() (string, bool) {
	if p.namespace != nil {
		return *p.namespace, true
	}
	var zero string
	return zero, false
}

func (p *Group) MustNamespace() string {
	if p.namespace == nil {
		panic("the namespace must not be nil, please use Namespace() function instead")
	}
	return *p.namespace
}

func (p *Group) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Group) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Group) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Group) SetRoles(attr *RoleSlice) {
	p.roles = attr
}

func (p *Group) Roles() (*RoleSlice, bool) {
	if p.roles != nil {
		return p.roles, true
	}
	return nil, false
}

func (p *Group) MustRoles() *RoleSlice {
	if p.roles == nil {
		panic("the roles must not be nil, please use Roles() function instead")
	}
	return p.roles
}

func (p *Group) SetTags(attr *TagSlice) {
	p.tags = attr
}

func (p *Group) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *Group) MustTags() *TagSlice {
	if p.tags == nil {
		panic("the tags must not be nil, please use Tags() function instead")
	}
	return p.tags
}

// GuestOperatingSystem Represents an operating system installed on the virtual machine.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/vms/123
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <vm href="/ovirt-engine/api/vms/123" id="123">
// ...
//   <guest_operating_system>
//     <architecture>x86_64</architecture>
//     <codename>Maipo</codename>
//     <distribution>Red Hat Enterprise Linux Server</distribution>
//     <family>Linux</family>
//     <kernel>
//       <version>
//         <build>0</build>
//         <full_version>3.10.0-514.10.2.el7.x86_64</full_version>
//         <major>3</major>
//         <minor>10</minor>
//         <revision>514</revision>
//       </version>
//     </kernel>
//     <version>
//       <full_version>7.3</full_version>
//       <major>7</major>
//       <minor>3</minor>
//     </version>
//   </guest_operating_system>
// </vm>
// ----
type GuestOperatingSystem struct {
	Struct
	architecture *string
	codename     *string
	distribution *string
	family       *string
	kernel       *Kernel
	version      *Version
}

func (p *GuestOperatingSystem) SetArchitecture(attr string) {
	p.architecture = &attr
}

func (p *GuestOperatingSystem) Architecture() (string, bool) {
	if p.architecture != nil {
		return *p.architecture, true
	}
	var zero string
	return zero, false
}

func (p *GuestOperatingSystem) MustArchitecture() string {
	if p.architecture == nil {
		panic("the architecture must not be nil, please use Architecture() function instead")
	}
	return *p.architecture
}

func (p *GuestOperatingSystem) SetCodename(attr string) {
	p.codename = &attr
}

func (p *GuestOperatingSystem) Codename() (string, bool) {
	if p.codename != nil {
		return *p.codename, true
	}
	var zero string
	return zero, false
}

func (p *GuestOperatingSystem) MustCodename() string {
	if p.codename == nil {
		panic("the codename must not be nil, please use Codename() function instead")
	}
	return *p.codename
}

func (p *GuestOperatingSystem) SetDistribution(attr string) {
	p.distribution = &attr
}

func (p *GuestOperatingSystem) Distribution() (string, bool) {
	if p.distribution != nil {
		return *p.distribution, true
	}
	var zero string
	return zero, false
}

func (p *GuestOperatingSystem) MustDistribution() string {
	if p.distribution == nil {
		panic("the distribution must not be nil, please use Distribution() function instead")
	}
	return *p.distribution
}

func (p *GuestOperatingSystem) SetFamily(attr string) {
	p.family = &attr
}

func (p *GuestOperatingSystem) Family() (string, bool) {
	if p.family != nil {
		return *p.family, true
	}
	var zero string
	return zero, false
}

func (p *GuestOperatingSystem) MustFamily() string {
	if p.family == nil {
		panic("the family must not be nil, please use Family() function instead")
	}
	return *p.family
}

func (p *GuestOperatingSystem) SetKernel(attr *Kernel) {
	p.kernel = attr
}

func (p *GuestOperatingSystem) Kernel() (*Kernel, bool) {
	if p.kernel != nil {
		return p.kernel, true
	}
	return nil, false
}

func (p *GuestOperatingSystem) MustKernel() *Kernel {
	if p.kernel == nil {
		panic("the kernel must not be nil, please use Kernel() function instead")
	}
	return p.kernel
}

func (p *GuestOperatingSystem) SetVersion(attr *Version) {
	p.version = attr
}

func (p *GuestOperatingSystem) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *GuestOperatingSystem) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// HardwareInformation Represents hardware information of host.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/hosts/123
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <host href="/ovirt-engine/api/hosts/123" id="123">
//   ...
//   <hardware_information>
//     <family>Red Hat Enterprise Linux</family>
//     <manufacturer>Red Hat</manufacturer>
//     <product_name>RHEV Hypervisor</product_name>
//     <serial_number>01234567-89AB-CDEF-0123-456789ABCDEF</serial_number>
//     <supported_rng_sources>
//       <supported_rng_source>random</supported_rng_source>
//     </supported_rng_sources>
//     <uuid>12345678-9ABC-DEF0-1234-56789ABCDEF0</uuid>
//     <version>1.2-34.5.el7ev</version>
//   </hardware_information>
//   ...
// </application>
// ----
type HardwareInformation struct {
	Struct
	family              *string
	manufacturer        *string
	productName         *string
	serialNumber        *string
	supportedRngSources []RngSource
	uuid                *string
	version             *string
}

func (p *HardwareInformation) SetFamily(attr string) {
	p.family = &attr
}

func (p *HardwareInformation) Family() (string, bool) {
	if p.family != nil {
		return *p.family, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustFamily() string {
	if p.family == nil {
		panic("the family must not be nil, please use Family() function instead")
	}
	return *p.family
}

func (p *HardwareInformation) SetManufacturer(attr string) {
	p.manufacturer = &attr
}

func (p *HardwareInformation) Manufacturer() (string, bool) {
	if p.manufacturer != nil {
		return *p.manufacturer, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustManufacturer() string {
	if p.manufacturer == nil {
		panic("the manufacturer must not be nil, please use Manufacturer() function instead")
	}
	return *p.manufacturer
}

func (p *HardwareInformation) SetProductName(attr string) {
	p.productName = &attr
}

func (p *HardwareInformation) ProductName() (string, bool) {
	if p.productName != nil {
		return *p.productName, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustProductName() string {
	if p.productName == nil {
		panic("the productName must not be nil, please use ProductName() function instead")
	}
	return *p.productName
}

func (p *HardwareInformation) SetSerialNumber(attr string) {
	p.serialNumber = &attr
}

func (p *HardwareInformation) SerialNumber() (string, bool) {
	if p.serialNumber != nil {
		return *p.serialNumber, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustSerialNumber() string {
	if p.serialNumber == nil {
		panic("the serialNumber must not be nil, please use SerialNumber() function instead")
	}
	return *p.serialNumber
}

func (p *HardwareInformation) SetSupportedRngSources(attr []RngSource) {
	p.supportedRngSources = attr
}

func (p *HardwareInformation) SupportedRngSources() ([]RngSource, bool) {
	if p.supportedRngSources != nil {
		return p.supportedRngSources, true
	}
	return nil, false
}

func (p *HardwareInformation) MustSupportedRngSources() []RngSource {
	if p.supportedRngSources == nil {
		panic("the supportedRngSources must not be nil, please use SupportedRngSources() function instead")
	}
	return p.supportedRngSources
}

func (p *HardwareInformation) SetUuid(attr string) {
	p.uuid = &attr
}

func (p *HardwareInformation) Uuid() (string, bool) {
	if p.uuid != nil {
		return *p.uuid, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustUuid() string {
	if p.uuid == nil {
		panic("the uuid must not be nil, please use Uuid() function instead")
	}
	return *p.uuid
}

func (p *HardwareInformation) SetVersion(attr string) {
	p.version = &attr
}

func (p *HardwareInformation) Version() (string, bool) {
	if p.version != nil {
		return *p.version, true
	}
	var zero string
	return zero, false
}

func (p *HardwareInformation) MustVersion() string {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return *p.version
}

// HighAvailability Type representing high availability of a virtual machine.
type HighAvailability struct {
	Struct
	enabled  *bool
	priority *int64
}

func (p *HighAvailability) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *HighAvailability) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *HighAvailability) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

func (p *HighAvailability) SetPriority(attr int64) {
	p.priority = &attr
}

func (p *HighAvailability) Priority() (int64, bool) {
	if p.priority != nil {
		return *p.priority, true
	}
	var zero int64
	return zero, false
}

func (p *HighAvailability) MustPriority() int64 {
	if p.priority == nil {
		panic("the priority must not be nil, please use Priority() function instead")
	}
	return *p.priority
}

// Hook Represents a hook.
type Hook struct {
	Struct
	comment     *string
	description *string
	eventName   *string
	host        *Host
	id          *string
	md5         *string
	name        *string
}

func (p *Hook) SetComment(attr string) {
	p.comment = &attr
}

func (p *Hook) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Hook) SetDescription(attr string) {
	p.description = &attr
}

func (p *Hook) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Hook) SetEventName(attr string) {
	p.eventName = &attr
}

func (p *Hook) EventName() (string, bool) {
	if p.eventName != nil {
		return *p.eventName, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustEventName() string {
	if p.eventName == nil {
		panic("the eventName must not be nil, please use EventName() function instead")
	}
	return *p.eventName
}

// SetHost Type representing a host.
func (p *Hook) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *Hook) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *Hook) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *Hook) SetId(attr string) {
	p.id = &attr
}

func (p *Hook) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Hook) SetMd5(attr string) {
	p.md5 = &attr
}

func (p *Hook) Md5() (string, bool) {
	if p.md5 != nil {
		return *p.md5, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustMd5() string {
	if p.md5 == nil {
		panic("the md5 must not be nil, please use Md5() function instead")
	}
	return *p.md5
}

func (p *Hook) SetName(attr string) {
	p.name = &attr
}

func (p *Hook) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Hook) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// Host Type representing a host.
type Host struct {
	Struct
	address                               *string
	affinityLabels                        *AffinityLabelSlice
	agents                                *AgentSlice
	autoNumaStatus                        *AutoNumaStatus
	certificate                           *Certificate
	cluster                               *Cluster
	comment                               *string
	cpu                                   *Cpu
	description                           *string
	devicePassthrough                     *HostDevicePassthrough
	devices                               *HostDeviceSlice
	display                               *Display
	externalHostProvider                  *ExternalHostProvider
	externalNetworkProviderConfigurations *ExternalNetworkProviderConfigurationSlice
	externalStatus                        *ExternalStatus
	hardwareInformation                   *HardwareInformation
	hooks                                 *HookSlice
	hostedEngine                          *HostedEngine
	id                                    *string
	iscsi                                 *IscsiDetails
	katelloErrata                         *KatelloErratumSlice
	kdumpStatus                           *KdumpStatus
	ksm                                   *Ksm
	libvirtVersion                        *Version
	maxSchedulingMemory                   *int64
	memory                                *int64
	name                                  *string
	networkAttachments                    *NetworkAttachmentSlice
	networkOperationInProgress            *bool
	nics                                  *HostNicSlice
	numaNodes                             *NumaNodeSlice
	numaSupported                         *bool
	os                                    *OperatingSystem
	overrideIptables                      *bool
	permissions                           *PermissionSlice
	port                                  *int64
	powerManagement                       *PowerManagement
	protocol                              *HostProtocol
	reinstallationRequired                *bool
	rootPassword                          *string
	seLinux                               *SeLinux
	spm                                   *Spm
	ssh                                   *Ssh
	statistics                            *StatisticSlice
	status                                *HostStatus
	statusDetail                          *string
	storageConnectionExtensions           *StorageConnectionExtensionSlice
	storages                              *HostStorageSlice
	summary                               *VmSummary
	tags                                  *TagSlice
	transparentHugePages                  *TransparentHugePages
	type_                                 *HostType
	unmanagedNetworks                     *UnmanagedNetworkSlice
	updateAvailable                       *bool
	version                               *Version
	vgpuPlacement                         *VgpuPlacement
}

func (p *Host) SetAddress(attr string) {
	p.address = &attr
}

func (p *Host) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *Host) SetAffinityLabels(attr *AffinityLabelSlice) {
	p.affinityLabels = attr
}

func (p *Host) AffinityLabels() (*AffinityLabelSlice, bool) {
	if p.affinityLabels != nil {
		return p.affinityLabels, true
	}
	return nil, false
}

func (p *Host) MustAffinityLabels() *AffinityLabelSlice {
	if p.affinityLabels == nil {
		panic("the affinityLabels must not be nil, please use AffinityLabels() function instead")
	}
	return p.affinityLabels
}

func (p *Host) SetAgents(attr *AgentSlice) {
	p.agents = attr
}

func (p *Host) Agents() (*AgentSlice, bool) {
	if p.agents != nil {
		return p.agents, true
	}
	return nil, false
}

func (p *Host) MustAgents() *AgentSlice {
	if p.agents == nil {
		panic("the agents must not be nil, please use Agents() function instead")
	}
	return p.agents
}

func (p *Host) SetAutoNumaStatus(attr AutoNumaStatus) {
	p.autoNumaStatus = &attr
}

func (p *Host) AutoNumaStatus() (AutoNumaStatus, bool) {
	if p.autoNumaStatus != nil {
		return *p.autoNumaStatus, true
	}
	var zero AutoNumaStatus
	return zero, false
}

func (p *Host) MustAutoNumaStatus() AutoNumaStatus {
	if p.autoNumaStatus == nil {
		panic("the autoNumaStatus must not be nil, please use AutoNumaStatus() function instead")
	}
	return *p.autoNumaStatus
}

func (p *Host) SetCertificate(attr *Certificate) {
	p.certificate = attr
}

func (p *Host) Certificate() (*Certificate, bool) {
	if p.certificate != nil {
		return p.certificate, true
	}
	return nil, false
}

func (p *Host) MustCertificate() *Certificate {
	if p.certificate == nil {
		panic("the certificate must not be nil, please use Certificate() function instead")
	}
	return p.certificate
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Host) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Host) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Host) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *Host) SetComment(attr string) {
	p.comment = &attr
}

func (p *Host) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Host) SetCpu(attr *Cpu) {
	p.cpu = attr
}

func (p *Host) Cpu() (*Cpu, bool) {
	if p.cpu != nil {
		return p.cpu, true
	}
	return nil, false
}

func (p *Host) MustCpu() *Cpu {
	if p.cpu == nil {
		panic("the cpu must not be nil, please use Cpu() function instead")
	}
	return p.cpu
}

func (p *Host) SetDescription(attr string) {
	p.description = &attr
}

func (p *Host) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Host) SetDevicePassthrough(attr *HostDevicePassthrough) {
	p.devicePassthrough = attr
}

func (p *Host) DevicePassthrough() (*HostDevicePassthrough, bool) {
	if p.devicePassthrough != nil {
		return p.devicePassthrough, true
	}
	return nil, false
}

func (p *Host) MustDevicePassthrough() *HostDevicePassthrough {
	if p.devicePassthrough == nil {
		panic("the devicePassthrough must not be nil, please use DevicePassthrough() function instead")
	}
	return p.devicePassthrough
}

func (p *Host) SetDevices(attr *HostDeviceSlice) {
	p.devices = attr
}

func (p *Host) Devices() (*HostDeviceSlice, bool) {
	if p.devices != nil {
		return p.devices, true
	}
	return nil, false
}

func (p *Host) MustDevices() *HostDeviceSlice {
	if p.devices == nil {
		panic("the devices must not be nil, please use Devices() function instead")
	}
	return p.devices
}

// SetDisplay Represents a graphic console configuration.
func (p *Host) SetDisplay(attr *Display) {
	p.display = attr
}

// Display Represents a graphic console configuration.
func (p *Host) Display() (*Display, bool) {
	if p.display != nil {
		return p.display, true
	}
	return nil, false
}

// MustDisplay Represents a graphic console configuration.
func (p *Host) MustDisplay() *Display {
	if p.display == nil {
		panic("the display must not be nil, please use Display() function instead")
	}
	return p.display
}

// SetExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *Host) SetExternalHostProvider(attr *ExternalHostProvider) {
	p.externalHostProvider = attr
}

// ExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *Host) ExternalHostProvider() (*ExternalHostProvider, bool) {
	if p.externalHostProvider != nil {
		return p.externalHostProvider, true
	}
	return nil, false
}

// MustExternalHostProvider Represents an external host provider,
// such as Foreman or Satellite.
//
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite
// for more details on Red Hat Satellite.
func (p *Host) MustExternalHostProvider() *ExternalHostProvider {
	if p.externalHostProvider == nil {
		panic("the externalHostProvider must not be nil, please use ExternalHostProvider() function instead")
	}
	return p.externalHostProvider
}

func (p *Host) SetExternalNetworkProviderConfigurations(attr *ExternalNetworkProviderConfigurationSlice) {
	p.externalNetworkProviderConfigurations = attr
}

func (p *Host) ExternalNetworkProviderConfigurations() (*ExternalNetworkProviderConfigurationSlice, bool) {
	if p.externalNetworkProviderConfigurations != nil {
		return p.externalNetworkProviderConfigurations, true
	}
	return nil, false
}

func (p *Host) MustExternalNetworkProviderConfigurations() *ExternalNetworkProviderConfigurationSlice {
	if p.externalNetworkProviderConfigurations == nil {
		panic("the externalNetworkProviderConfigurations must not be nil, please use ExternalNetworkProviderConfigurations() function instead")
	}
	return p.externalNetworkProviderConfigurations
}

// SetExternalStatus Represents an external status.
// This status is currently used for <<types/host, hosts>>
// and <<types/storage_domain, storage domains>>, and allows an external
// system to update status of objects it is aware of.
func (p *Host) SetExternalStatus(attr ExternalStatus) {
	p.externalStatus = &attr
}

// ExternalStatus Represents an external status.
// This status is currently used for <<types/host, hosts>>
// and <<types/storage_domain, storage domains>>, and allows an external
// system to update status of objects it is aware of.
func (p *Host) ExternalStatus() (ExternalStatus, bool) {
	if p.externalStatus != nil {
		return *p.externalStatus, true
	}
	var zero ExternalStatus
	return zero, false
}

// MustExternalStatus Represents an external status.
// This status is currently used for <<types/host, hosts>>
// and <<types/storage_domain, storage domains>>, and allows an external
// system to update status of objects it is aware of.
func (p *Host) MustExternalStatus() ExternalStatus {
	if p.externalStatus == nil {
		panic("the externalStatus must not be nil, please use ExternalStatus() function instead")
	}
	return *p.externalStatus
}

// SetHardwareInformation Represents hardware information of host.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/hosts/123
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <host href="/ovirt-engine/api/hosts/123" id="123">
//   ...
//   <hardware_information>
//     <family>Red Hat Enterprise Linux</family>
//     <manufacturer>Red Hat</manufacturer>
//     <product_name>RHEV Hypervisor</product_name>
//     <serial_number>01234567-89AB-CDEF-0123-456789ABCDEF</serial_number>
//     <supported_rng_sources>
//       <supported_rng_source>random</supported_rng_source>
//     </supported_rng_sources>
//     <uuid>12345678-9ABC-DEF0-1234-56789ABCDEF0</uuid>
//     <version>1.2-34.5.el7ev</version>
//   </hardware_information>
//   ...
// </application>
// ----
func (p *Host) SetHardwareInformation(attr *HardwareInformation) {
	p.hardwareInformation = attr
}

// HardwareInformation Represents hardware information of host.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/hosts/123
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <host href="/ovirt-engine/api/hosts/123" id="123">
//   ...
//   <hardware_information>
//     <family>Red Hat Enterprise Linux</family>
//     <manufacturer>Red Hat</manufacturer>
//     <product_name>RHEV Hypervisor</product_name>
//     <serial_number>01234567-89AB-CDEF-0123-456789ABCDEF</serial_number>
//     <supported_rng_sources>
//       <supported_rng_source>random</supported_rng_source>
//     </supported_rng_sources>
//     <uuid>12345678-9ABC-DEF0-1234-56789ABCDEF0</uuid>
//     <version>1.2-34.5.el7ev</version>
//   </hardware_information>
//   ...
// </application>
// ----
func (p *Host) HardwareInformation() (*HardwareInformation, bool) {
	if p.hardwareInformation != nil {
		return p.hardwareInformation, true
	}
	return nil, false
}

// MustHardwareInformation Represents hardware information of host.
//
// To get that information send a request like this:
//
// ....
// GET /ovirt-engine/api/hosts/123
// ....
//
// The result will be like this:
//
// [source,xml]
// ----
// <host href="/ovirt-engine/api/hosts/123" id="123">
//   ...
//   <hardware_information>
//     <family>Red Hat Enterprise Linux</family>
//     <manufacturer>Red Hat</manufacturer>
//     <product_name>RHEV Hypervisor</product_name>
//     <serial_number>01234567-89AB-CDEF-0123-456789ABCDEF</serial_number>
//     <supported_rng_sources>
//       <supported_rng_source>random</supported_rng_source>
//     </supported_rng_sources>
//     <uuid>12345678-9ABC-DEF0-1234-56789ABCDEF0</uuid>
//     <version>1.2-34.5.el7ev</version>
//   </hardware_information>
//   ...
// </application>
// ----
func (p *Host) MustHardwareInformation() *HardwareInformation {
	if p.hardwareInformation == nil {
		panic("the hardwareInformation must not be nil, please use HardwareInformation() function instead")
	}
	return p.hardwareInformation
}

func (p *Host) SetHooks(attr *HookSlice) {
	p.hooks = attr
}

func (p *Host) Hooks() (*HookSlice, bool) {
	if p.hooks != nil {
		return p.hooks, true
	}
	return nil, false
}

func (p *Host) MustHooks() *HookSlice {
	if p.hooks == nil {
		panic("the hooks must not be nil, please use Hooks() function instead")
	}
	return p.hooks
}

func (p *Host) SetHostedEngine(attr *HostedEngine) {
	p.hostedEngine = attr
}

func (p *Host) HostedEngine() (*HostedEngine, bool) {
	if p.hostedEngine != nil {
		return p.hostedEngine, true
	}
	return nil, false
}

func (p *Host) MustHostedEngine() *HostedEngine {
	if p.hostedEngine == nil {
		panic("the hostedEngine must not be nil, please use HostedEngine() function instead")
	}
	return p.hostedEngine
}

func (p *Host) SetId(attr string) {
	p.id = &attr
}

func (p *Host) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Host) SetIscsi(attr *IscsiDetails) {
	p.iscsi = attr
}

func (p *Host) Iscsi() (*IscsiDetails, bool) {
	if p.iscsi != nil {
		return p.iscsi, true
	}
	return nil, false
}

func (p *Host) MustIscsi() *IscsiDetails {
	if p.iscsi == nil {
		panic("the iscsi must not be nil, please use Iscsi() function instead")
	}
	return p.iscsi
}

func (p *Host) SetKatelloErrata(attr *KatelloErratumSlice) {
	p.katelloErrata = attr
}

func (p *Host) KatelloErrata() (*KatelloErratumSlice, bool) {
	if p.katelloErrata != nil {
		return p.katelloErrata, true
	}
	return nil, false
}

func (p *Host) MustKatelloErrata() *KatelloErratumSlice {
	if p.katelloErrata == nil {
		panic("the katelloErrata must not be nil, please use KatelloErrata() function instead")
	}
	return p.katelloErrata
}

func (p *Host) SetKdumpStatus(attr KdumpStatus) {
	p.kdumpStatus = &attr
}

func (p *Host) KdumpStatus() (KdumpStatus, bool) {
	if p.kdumpStatus != nil {
		return *p.kdumpStatus, true
	}
	var zero KdumpStatus
	return zero, false
}

func (p *Host) MustKdumpStatus() KdumpStatus {
	if p.kdumpStatus == nil {
		panic("the kdumpStatus must not be nil, please use KdumpStatus() function instead")
	}
	return *p.kdumpStatus
}

func (p *Host) SetKsm(attr *Ksm) {
	p.ksm = attr
}

func (p *Host) Ksm() (*Ksm, bool) {
	if p.ksm != nil {
		return p.ksm, true
	}
	return nil, false
}

func (p *Host) MustKsm() *Ksm {
	if p.ksm == nil {
		panic("the ksm must not be nil, please use Ksm() function instead")
	}
	return p.ksm
}

func (p *Host) SetLibvirtVersion(attr *Version) {
	p.libvirtVersion = attr
}

func (p *Host) LibvirtVersion() (*Version, bool) {
	if p.libvirtVersion != nil {
		return p.libvirtVersion, true
	}
	return nil, false
}

func (p *Host) MustLibvirtVersion() *Version {
	if p.libvirtVersion == nil {
		panic("the libvirtVersion must not be nil, please use LibvirtVersion() function instead")
	}
	return p.libvirtVersion
}

func (p *Host) SetMaxSchedulingMemory(attr int64) {
	p.maxSchedulingMemory = &attr
}

func (p *Host) MaxSchedulingMemory() (int64, bool) {
	if p.maxSchedulingMemory != nil {
		return *p.maxSchedulingMemory, true
	}
	var zero int64
	return zero, false
}

func (p *Host) MustMaxSchedulingMemory() int64 {
	if p.maxSchedulingMemory == nil {
		panic("the maxSchedulingMemory must not be nil, please use MaxSchedulingMemory() function instead")
	}
	return *p.maxSchedulingMemory
}

func (p *Host) SetMemory(attr int64) {
	p.memory = &attr
}

func (p *Host) Memory() (int64, bool) {
	if p.memory != nil {
		return *p.memory, true
	}
	var zero int64
	return zero, false
}

func (p *Host) MustMemory() int64 {
	if p.memory == nil {
		panic("the memory must not be nil, please use Memory() function instead")
	}
	return *p.memory
}

func (p *Host) SetName(attr string) {
	p.name = &attr
}

func (p *Host) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Host) SetNetworkAttachments(attr *NetworkAttachmentSlice) {
	p.networkAttachments = attr
}

func (p *Host) NetworkAttachments() (*NetworkAttachmentSlice, bool) {
	if p.networkAttachments != nil {
		return p.networkAttachments, true
	}
	return nil, false
}

func (p *Host) MustNetworkAttachments() *NetworkAttachmentSlice {
	if p.networkAttachments == nil {
		panic("the networkAttachments must not be nil, please use NetworkAttachments() function instead")
	}
	return p.networkAttachments
}

func (p *Host) SetNetworkOperationInProgress(attr bool) {
	p.networkOperationInProgress = &attr
}

func (p *Host) NetworkOperationInProgress() (bool, bool) {
	if p.networkOperationInProgress != nil {
		return *p.networkOperationInProgress, true
	}
	var zero bool
	return zero, false
}

func (p *Host) MustNetworkOperationInProgress() bool {
	if p.networkOperationInProgress == nil {
		panic("the networkOperationInProgress must not be nil, please use NetworkOperationInProgress() function instead")
	}
	return *p.networkOperationInProgress
}

func (p *Host) SetNics(attr *HostNicSlice) {
	p.nics = attr
}

func (p *Host) Nics() (*HostNicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *Host) MustNics() *HostNicSlice {
	if p.nics == nil {
		panic("the nics must not be nil, please use Nics() function instead")
	}
	return p.nics
}

func (p *Host) SetNumaNodes(attr *NumaNodeSlice) {
	p.numaNodes = attr
}

func (p *Host) NumaNodes() (*NumaNodeSlice, bool) {
	if p.numaNodes != nil {
		return p.numaNodes, true
	}
	return nil, false
}

func (p *Host) MustNumaNodes() *NumaNodeSlice {
	if p.numaNodes == nil {
		panic("the numaNodes must not be nil, please use NumaNodes() function instead")
	}
	return p.numaNodes
}

func (p *Host) SetNumaSupported(attr bool) {
	p.numaSupported = &attr
}

func (p *Host) NumaSupported() (bool, bool) {
	if p.numaSupported != nil {
		return *p.numaSupported, true
	}
	var zero bool
	return zero, false
}

func (p *Host) MustNumaSupported() bool {
	if p.numaSupported == nil {
		panic("the numaSupported must not be nil, please use NumaSupported() function instead")
	}
	return *p.numaSupported
}

// SetOs Information describing the operating system. This is used for both virtual machines and hosts.
func (p *Host) SetOs(attr *OperatingSystem) {
	p.os = attr
}

// Os Information describing the operating system. This is used for both virtual machines and hosts.
func (p *Host) Os() (*OperatingSystem, bool) {
	if p.os != nil {
		return p.os, true
	}
	return nil, false
}

// MustOs Information describing the operating system. This is used for both virtual machines and hosts.
func (p *Host) MustOs() *OperatingSystem {
	if p.os == nil {
		panic("the os must not be nil, please use Os() function instead")
	}
	return p.os
}

func (p *Host) SetOverrideIptables(attr bool) {
	p.overrideIptables = &attr
}

func (p *Host) OverrideIptables() (bool, bool) {
	if p.overrideIptables != nil {
		return *p.overrideIptables, true
	}
	var zero bool
	return zero, false
}

func (p *Host) MustOverrideIptables() bool {
	if p.overrideIptables == nil {
		panic("the overrideIptables must not be nil, please use OverrideIptables() function instead")
	}
	return *p.overrideIptables
}

func (p *Host) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Host) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Host) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Host) SetPort(attr int64) {
	p.port = &attr
}

func (p *Host) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *Host) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *Host) SetPowerManagement(attr *PowerManagement) {
	p.powerManagement = attr
}

func (p *Host) PowerManagement() (*PowerManagement, bool) {
	if p.powerManagement != nil {
		return p.powerManagement, true
	}
	return nil, false
}

func (p *Host) MustPowerManagement() *PowerManagement {
	if p.powerManagement == nil {
		panic("the powerManagement must not be nil, please use PowerManagement() function instead")
	}
	return p.powerManagement
}

// SetProtocol The protocol used by the engine to communicate with
// a host.
//
// WARNING: Since version 4.1 of the engine the protocol
// is always set to `stomp` since `xml` was removed.
func (p *Host) SetProtocol(attr HostProtocol) {
	p.protocol = &attr
}

// Protocol The protocol used by the engine to communicate with
// a host.
//
// WARNING: Since version 4.1 of the engine the protocol
// is always set to `stomp` since `xml` was removed.
func (p *Host) Protocol() (HostProtocol, bool) {
	if p.protocol != nil {
		return *p.protocol, true
	}
	var zero HostProtocol
	return zero, false
}

// MustProtocol The protocol used by the engine to communicate with
// a host.
//
// WARNING: Since version 4.1 of the engine the protocol
// is always set to `stomp` since `xml` was removed.
func (p *Host) MustProtocol() HostProtocol {
	if p.protocol == nil {
		panic("the protocol must not be nil, please use Protocol() function instead")
	}
	return *p.protocol
}

func (p *Host) SetReinstallationRequired(attr bool) {
	p.reinstallationRequired = &attr
}

func (p *Host) ReinstallationRequired() (bool, bool) {
	if p.reinstallationRequired != nil {
		return *p.reinstallationRequired, true
	}
	var zero bool
	return zero, false
}

func (p *Host) MustReinstallationRequired() bool {
	if p.reinstallationRequired == nil {
		panic("the reinstallationRequired must not be nil, please use ReinstallationRequired() function instead")
	}
	return *p.reinstallationRequired
}

func (p *Host) SetRootPassword(attr string) {
	p.rootPassword = &attr
}

func (p *Host) RootPassword() (string, bool) {
	if p.rootPassword != nil {
		return *p.rootPassword, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustRootPassword() string {
	if p.rootPassword == nil {
		panic("the rootPassword must not be nil, please use RootPassword() function instead")
	}
	return *p.rootPassword
}

// SetSeLinux Represents SELinux in the system.
func (p *Host) SetSeLinux(attr *SeLinux) {
	p.seLinux = attr
}

// SeLinux Represents SELinux in the system.
func (p *Host) SeLinux() (*SeLinux, bool) {
	if p.seLinux != nil {
		return p.seLinux, true
	}
	return nil, false
}

// MustSeLinux Represents SELinux in the system.
func (p *Host) MustSeLinux() *SeLinux {
	if p.seLinux == nil {
		panic("the seLinux must not be nil, please use SeLinux() function instead")
	}
	return p.seLinux
}

func (p *Host) SetSpm(attr *Spm) {
	p.spm = attr
}

func (p *Host) Spm() (*Spm, bool) {
	if p.spm != nil {
		return p.spm, true
	}
	return nil, false
}

func (p *Host) MustSpm() *Spm {
	if p.spm == nil {
		panic("the spm must not be nil, please use Spm() function instead")
	}
	return p.spm
}

func (p *Host) SetSsh(attr *Ssh) {
	p.ssh = attr
}

func (p *Host) Ssh() (*Ssh, bool) {
	if p.ssh != nil {
		return p.ssh, true
	}
	return nil, false
}

func (p *Host) MustSsh() *Ssh {
	if p.ssh == nil {
		panic("the ssh must not be nil, please use Ssh() function instead")
	}
	return p.ssh
}

func (p *Host) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *Host) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *Host) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// SetStatus Type representing a host status.
func (p *Host) SetStatus(attr HostStatus) {
	p.status = &attr
}

// Status Type representing a host status.
func (p *Host) Status() (HostStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero HostStatus
	return zero, false
}

// MustStatus Type representing a host status.
func (p *Host) MustStatus() HostStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *Host) SetStatusDetail(attr string) {
	p.statusDetail = &attr
}

func (p *Host) StatusDetail() (string, bool) {
	if p.statusDetail != nil {
		return *p.statusDetail, true
	}
	var zero string
	return zero, false
}

func (p *Host) MustStatusDetail() string {
	if p.statusDetail == nil {
		panic("the statusDetail must not be nil, please use StatusDetail() function instead")
	}
	return *p.statusDetail
}

func (p *Host) SetStorageConnectionExtensions(attr *StorageConnectionExtensionSlice) {
	p.storageConnectionExtensions = attr
}

func (p *Host) StorageConnectionExtensions() (*StorageConnectionExtensionSlice, bool) {
	if p.storageConnectionExtensions != nil {
		return p.storageConnectionExtensions, true
	}
	return nil, false
}

func (p *Host) MustStorageConnectionExtensions() *StorageConnectionExtensionSlice {
	if p.storageConnectionExtensions == nil {
		panic("the storageConnectionExtensions must not be nil, please use StorageConnectionExtensions() function instead")
	}
	return p.storageConnectionExtensions
}

func (p *Host) SetStorages(attr *HostStorageSlice) {
	p.storages = attr
}

func (p *Host) Storages() (*HostStorageSlice, bool) {
	if p.storages != nil {
		return p.storages, true
	}
	return nil, false
}

func (p *Host) MustStorages() *HostStorageSlice {
	if p.storages == nil {
		panic("the storages must not be nil, please use Storages() function instead")
	}
	return p.storages
}

// SetSummary Type containing information related to virtual machines on a particular host.
func (p *Host) SetSummary(attr *VmSummary) {
	p.summary = attr
}

// Summary Type containing information related to virtual machines on a particular host.
func (p *Host) Summary() (*VmSummary, bool) {
	if p.summary != nil {
		return p.summary, true
	}
	return nil, false
}

// MustSummary Type containing information related to virtual machines on a particular host.
func (p *Host) MustSummary() *VmSummary {
	if p.summary == nil {
		panic("the summary must not be nil, please use Summary() function instead")
	}
	return p.summary
}

func (p *Host) SetTags(attr *TagSlice) {
	p.tags = attr
}

func (p *Host) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *Host) MustTags() *TagSlice {
	if p.tags == nil {
		panic("the tags must not be nil, please use Tags() function instead")
	}
	return p.tags
}

// SetTransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *Host) SetTransparentHugePages(attr *TransparentHugePages) {
	p.transparentHugePages = attr
}

// TransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *Host) TransparentHugePages() (*TransparentHugePages, bool) {
	if p.transparentHugePages != nil {
		return p.transparentHugePages, true
	}
	return nil, false
}

// MustTransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *Host) MustTransparentHugePages() *TransparentHugePages {
	if p.transparentHugePages == nil {
		panic("the transparentHugePages must not be nil, please use TransparentHugePages() function instead")
	}
	return p.transparentHugePages
}

// SetType This enumerated type is used to determine which type of operating system is used by the host.
func (p *Host) SetType(attr HostType) {
	p.type_ = &attr
}

// Type This enumerated type is used to determine which type of operating system is used by the host.
func (p *Host) Type() (HostType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero HostType
	return zero, false
}

// MustType This enumerated type is used to determine which type of operating system is used by the host.
func (p *Host) MustType() HostType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *Host) SetUnmanagedNetworks(attr *UnmanagedNetworkSlice) {
	p.unmanagedNetworks = attr
}

func (p *Host) UnmanagedNetworks() (*UnmanagedNetworkSlice, bool) {
	if p.unmanagedNetworks != nil {
		return p.unmanagedNetworks, true
	}
	return nil, false
}

func (p *Host) MustUnmanagedNetworks() *UnmanagedNetworkSlice {
	if p.unmanagedNetworks == nil {
		panic("the unmanagedNetworks must not be nil, please use UnmanagedNetworks() function instead")
	}
	return p.unmanagedNetworks
}

func (p *Host) SetUpdateAvailable(attr bool) {
	p.updateAvailable = &attr
}

func (p *Host) UpdateAvailable() (bool, bool) {
	if p.updateAvailable != nil {
		return *p.updateAvailable, true
	}
	var zero bool
	return zero, false
}

func (p *Host) MustUpdateAvailable() bool {
	if p.updateAvailable == nil {
		panic("the updateAvailable must not be nil, please use UpdateAvailable() function instead")
	}
	return *p.updateAvailable
}

func (p *Host) SetVersion(attr *Version) {
	p.version = attr
}

func (p *Host) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *Host) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// SetVgpuPlacement The vGPU placement strategy.
//
// It can either put vGPUs on the first available physical
// cards, or spread them over multiple physical cards.
func (p *Host) SetVgpuPlacement(attr VgpuPlacement) {
	p.vgpuPlacement = &attr
}

// VgpuPlacement The vGPU placement strategy.
//
// It can either put vGPUs on the first available physical
// cards, or spread them over multiple physical cards.
func (p *Host) VgpuPlacement() (VgpuPlacement, bool) {
	if p.vgpuPlacement != nil {
		return *p.vgpuPlacement, true
	}
	var zero VgpuPlacement
	return zero, false
}

// MustVgpuPlacement The vGPU placement strategy.
//
// It can either put vGPUs on the first available physical
// cards, or spread them over multiple physical cards.
func (p *Host) MustVgpuPlacement() VgpuPlacement {
	if p.vgpuPlacement == nil {
		panic("the vgpuPlacement must not be nil, please use VgpuPlacement() function instead")
	}
	return *p.vgpuPlacement
}

type HostDevice struct {
	Struct
	capability       *string
	comment          *string
	description      *string
	driver           *string
	host             *Host
	id               *string
	iommuGroup       *int64
	mDevTypes        *MDevTypeSlice
	name             *string
	parentDevice     *HostDevice
	physicalFunction *HostDevice
	placeholder      *bool
	product          *Product
	vendor           *Vendor
	virtualFunctions *int64
	vm               *Vm
}

func (p *HostDevice) SetCapability(attr string) {
	p.capability = &attr
}

func (p *HostDevice) Capability() (string, bool) {
	if p.capability != nil {
		return *p.capability, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustCapability() string {
	if p.capability == nil {
		panic("the capability must not be nil, please use Capability() function instead")
	}
	return *p.capability
}

func (p *HostDevice) SetComment(attr string) {
	p.comment = &attr
}

func (p *HostDevice) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *HostDevice) SetDescription(attr string) {
	p.description = &attr
}

func (p *HostDevice) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *HostDevice) SetDriver(attr string) {
	p.driver = &attr
}

func (p *HostDevice) Driver() (string, bool) {
	if p.driver != nil {
		return *p.driver, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustDriver() string {
	if p.driver == nil {
		panic("the driver must not be nil, please use Driver() function instead")
	}
	return *p.driver
}

// SetHost Type representing a host.
func (p *HostDevice) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *HostDevice) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *HostDevice) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *HostDevice) SetId(attr string) {
	p.id = &attr
}

func (p *HostDevice) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *HostDevice) SetIommuGroup(attr int64) {
	p.iommuGroup = &attr
}

func (p *HostDevice) IommuGroup() (int64, bool) {
	if p.iommuGroup != nil {
		return *p.iommuGroup, true
	}
	var zero int64
	return zero, false
}

func (p *HostDevice) MustIommuGroup() int64 {
	if p.iommuGroup == nil {
		panic("the iommuGroup must not be nil, please use IommuGroup() function instead")
	}
	return *p.iommuGroup
}

func (p *HostDevice) SetMDevTypes(attr *MDevTypeSlice) {
	p.mDevTypes = attr
}

func (p *HostDevice) MDevTypes() (*MDevTypeSlice, bool) {
	if p.mDevTypes != nil {
		return p.mDevTypes, true
	}
	return nil, false
}

func (p *HostDevice) MustMDevTypes() *MDevTypeSlice {
	if p.mDevTypes == nil {
		panic("the mDevTypes must not be nil, please use MDevTypes() function instead")
	}
	return p.mDevTypes
}

func (p *HostDevice) SetName(attr string) {
	p.name = &attr
}

func (p *HostDevice) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *HostDevice) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *HostDevice) SetParentDevice(attr *HostDevice) {
	p.parentDevice = attr
}

func (p *HostDevice) ParentDevice() (*HostDevice, bool) {
	if p.parentDevice != nil {
		return p.parentDevice, true
	}
	return nil, false
}

func (p *HostDevice) MustParentDevice() *HostDevice {
	if p.parentDevice == nil {
		panic("the parentDevice must not be nil, please use ParentDevice() function instead")
	}
	return p.parentDevice
}

func (p *HostDevice) SetPhysicalFunction(attr *HostDevice) {
	p.physicalFunction = attr
}

func (p *HostDevice) PhysicalFunction() (*HostDevice, bool) {
	if p.physicalFunction != nil {
		return p.physicalFunction, true
	}
	return nil, false
}

func (p *HostDevice) MustPhysicalFunction() *HostDevice {
	if p.physicalFunction == nil {
		panic("the physicalFunction must not be nil, please use PhysicalFunction() function instead")
	}
	return p.physicalFunction
}

func (p *HostDevice) SetPlaceholder(attr bool) {
	p.placeholder = &attr
}

func (p *HostDevice) Placeholder() (bool, bool) {
	if p.placeholder != nil {
		return *p.placeholder, true
	}
	var zero bool
	return zero, false
}

func (p *HostDevice) MustPlaceholder() bool {
	if p.placeholder == nil {
		panic("the placeholder must not be nil, please use Placeholder() function instead")
	}
	return *p.placeholder
}

func (p *HostDevice) SetProduct(attr *Product) {
	p.product = attr
}

func (p *HostDevice) Product() (*Product, bool) {
	if p.product != nil {
		return p.product, true
	}
	return nil, false
}

func (p *HostDevice) MustProduct() *Product {
	if p.product == nil {
		panic("the product must not be nil, please use Product() function instead")
	}
	return p.product
}

func (p *HostDevice) SetVendor(attr *Vendor) {
	p.vendor = attr
}

func (p *HostDevice) Vendor() (*Vendor, bool) {
	if p.vendor != nil {
		return p.vendor, true
	}
	return nil, false
}

func (p *HostDevice) MustVendor() *Vendor {
	if p.vendor == nil {
		panic("the vendor must not be nil, please use Vendor() function instead")
	}
	return p.vendor
}

func (p *HostDevice) SetVirtualFunctions(attr int64) {
	p.virtualFunctions = &attr
}

func (p *HostDevice) VirtualFunctions() (int64, bool) {
	if p.virtualFunctions != nil {
		return *p.virtualFunctions, true
	}
	var zero int64
	return zero, false
}

func (p *HostDevice) MustVirtualFunctions() int64 {
	if p.virtualFunctions == nil {
		panic("the virtualFunctions must not be nil, please use VirtualFunctions() function instead")
	}
	return *p.virtualFunctions
}

// SetVm Represents a virtual machine.
func (p *HostDevice) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *HostDevice) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *HostDevice) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type HostDevicePassthrough struct {
	Struct
	enabled *bool
}

func (p *HostDevicePassthrough) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *HostDevicePassthrough) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *HostDevicePassthrough) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

// HostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
type HostNic struct {
	Struct
	adAggregatorId                *int64
	baseInterface                 *string
	bonding                       *Bonding
	bootProtocol                  *BootProtocol
	bridged                       *bool
	checkConnectivity             *bool
	comment                       *string
	customConfiguration           *bool
	description                   *string
	host                          *Host
	id                            *string
	ip                            *Ip
	ipv6                          *Ip
	ipv6BootProtocol              *BootProtocol
	mac                           *Mac
	mtu                           *int64
	name                          *string
	network                       *Network
	networkLabels                 *NetworkLabelSlice
	overrideConfiguration         *bool
	physicalFunction              *HostNic
	properties                    *PropertySlice
	qos                           *Qos
	speed                         *int64
	statistics                    *StatisticSlice
	status                        *NicStatus
	virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration
	vlan                          *Vlan
}

func (p *HostNic) SetAdAggregatorId(attr int64) {
	p.adAggregatorId = &attr
}

func (p *HostNic) AdAggregatorId() (int64, bool) {
	if p.adAggregatorId != nil {
		return *p.adAggregatorId, true
	}
	var zero int64
	return zero, false
}

func (p *HostNic) MustAdAggregatorId() int64 {
	if p.adAggregatorId == nil {
		panic("the adAggregatorId must not be nil, please use AdAggregatorId() function instead")
	}
	return *p.adAggregatorId
}

func (p *HostNic) SetBaseInterface(attr string) {
	p.baseInterface = &attr
}

func (p *HostNic) BaseInterface() (string, bool) {
	if p.baseInterface != nil {
		return *p.baseInterface, true
	}
	var zero string
	return zero, false
}

func (p *HostNic) MustBaseInterface() string {
	if p.baseInterface == nil {
		panic("the baseInterface must not be nil, please use BaseInterface() function instead")
	}
	return *p.baseInterface
}

// SetBonding Represents a network interfaces bond.
func (p *HostNic) SetBonding(attr *Bonding) {
	p.bonding = attr
}

// Bonding Represents a network interfaces bond.
func (p *HostNic) Bonding() (*Bonding, bool) {
	if p.bonding != nil {
		return p.bonding, true
	}
	return nil, false
}

// MustBonding Represents a network interfaces bond.
func (p *HostNic) MustBonding() *Bonding {
	if p.bonding == nil {
		panic("the bonding must not be nil, please use Bonding() function instead")
	}
	return p.bonding
}

// SetBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) SetBootProtocol(attr BootProtocol) {
	p.bootProtocol = &attr
}

// BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) BootProtocol() (BootProtocol, bool) {
	if p.bootProtocol != nil {
		return *p.bootProtocol, true
	}
	var zero BootProtocol
	return zero, false
}

// MustBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) MustBootProtocol() BootProtocol {
	if p.bootProtocol == nil {
		panic("the bootProtocol must not be nil, please use BootProtocol() function instead")
	}
	return *p.bootProtocol
}

func (p *HostNic) SetBridged(attr bool) {
	p.bridged = &attr
}

func (p *HostNic) Bridged() (bool, bool) {
	if p.bridged != nil {
		return *p.bridged, true
	}
	var zero bool
	return zero, false
}

func (p *HostNic) MustBridged() bool {
	if p.bridged == nil {
		panic("the bridged must not be nil, please use Bridged() function instead")
	}
	return *p.bridged
}

func (p *HostNic) SetCheckConnectivity(attr bool) {
	p.checkConnectivity = &attr
}

func (p *HostNic) CheckConnectivity() (bool, bool) {
	if p.checkConnectivity != nil {
		return *p.checkConnectivity, true
	}
	var zero bool
	return zero, false
}

func (p *HostNic) MustCheckConnectivity() bool {
	if p.checkConnectivity == nil {
		panic("the checkConnectivity must not be nil, please use CheckConnectivity() function instead")
	}
	return *p.checkConnectivity
}

func (p *HostNic) SetComment(attr string) {
	p.comment = &attr
}

func (p *HostNic) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *HostNic) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *HostNic) SetCustomConfiguration(attr bool) {
	p.customConfiguration = &attr
}

func (p *HostNic) CustomConfiguration() (bool, bool) {
	if p.customConfiguration != nil {
		return *p.customConfiguration, true
	}
	var zero bool
	return zero, false
}

func (p *HostNic) MustCustomConfiguration() bool {
	if p.customConfiguration == nil {
		panic("the customConfiguration must not be nil, please use CustomConfiguration() function instead")
	}
	return *p.customConfiguration
}

func (p *HostNic) SetDescription(attr string) {
	p.description = &attr
}

func (p *HostNic) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *HostNic) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetHost Type representing a host.
func (p *HostNic) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *HostNic) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *HostNic) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *HostNic) SetId(attr string) {
	p.id = &attr
}

func (p *HostNic) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *HostNic) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetIp Represents the IP configuration of a network interface.
func (p *HostNic) SetIp(attr *Ip) {
	p.ip = attr
}

// Ip Represents the IP configuration of a network interface.
func (p *HostNic) Ip() (*Ip, bool) {
	if p.ip != nil {
		return p.ip, true
	}
	return nil, false
}

// MustIp Represents the IP configuration of a network interface.
func (p *HostNic) MustIp() *Ip {
	if p.ip == nil {
		panic("the ip must not be nil, please use Ip() function instead")
	}
	return p.ip
}

// SetIpv6 Represents the IP configuration of a network interface.
func (p *HostNic) SetIpv6(attr *Ip) {
	p.ipv6 = attr
}

// Ipv6 Represents the IP configuration of a network interface.
func (p *HostNic) Ipv6() (*Ip, bool) {
	if p.ipv6 != nil {
		return p.ipv6, true
	}
	return nil, false
}

// MustIpv6 Represents the IP configuration of a network interface.
func (p *HostNic) MustIpv6() *Ip {
	if p.ipv6 == nil {
		panic("the ipv6 must not be nil, please use Ipv6() function instead")
	}
	return p.ipv6
}

// SetIpv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) SetIpv6BootProtocol(attr BootProtocol) {
	p.ipv6BootProtocol = &attr
}

// Ipv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) Ipv6BootProtocol() (BootProtocol, bool) {
	if p.ipv6BootProtocol != nil {
		return *p.ipv6BootProtocol, true
	}
	var zero BootProtocol
	return zero, false
}

// MustIpv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *HostNic) MustIpv6BootProtocol() BootProtocol {
	if p.ipv6BootProtocol == nil {
		panic("the ipv6BootProtocol must not be nil, please use Ipv6BootProtocol() function instead")
	}
	return *p.ipv6BootProtocol
}

// SetMac Represents a MAC address of a virtual network interface.
func (p *HostNic) SetMac(attr *Mac) {
	p.mac = attr
}

// Mac Represents a MAC address of a virtual network interface.
func (p *HostNic) Mac() (*Mac, bool) {
	if p.mac != nil {
		return p.mac, true
	}
	return nil, false
}

// MustMac Represents a MAC address of a virtual network interface.
func (p *HostNic) MustMac() *Mac {
	if p.mac == nil {
		panic("the mac must not be nil, please use Mac() function instead")
	}
	return p.mac
}

func (p *HostNic) SetMtu(attr int64) {
	p.mtu = &attr
}

func (p *HostNic) Mtu() (int64, bool) {
	if p.mtu != nil {
		return *p.mtu, true
	}
	var zero int64
	return zero, false
}

func (p *HostNic) MustMtu() int64 {
	if p.mtu == nil {
		panic("the mtu must not be nil, please use Mtu() function instead")
	}
	return *p.mtu
}

func (p *HostNic) SetName(attr string) {
	p.name = &attr
}

func (p *HostNic) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *HostNic) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *HostNic) SetNetwork(attr *Network) {
	p.network = attr
}

// Network The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *HostNic) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

// MustNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *HostNic) MustNetwork() *Network {
	if p.network == nil {
		panic("the network must not be nil, please use Network() function instead")
	}
	return p.network
}

func (p *HostNic) SetNetworkLabels(attr *NetworkLabelSlice) {
	p.networkLabels = attr
}

func (p *HostNic) NetworkLabels() (*NetworkLabelSlice, bool) {
	if p.networkLabels != nil {
		return p.networkLabels, true
	}
	return nil, false
}

func (p *HostNic) MustNetworkLabels() *NetworkLabelSlice {
	if p.networkLabels == nil {
		panic("the networkLabels must not be nil, please use NetworkLabels() function instead")
	}
	return p.networkLabels
}

func (p *HostNic) SetOverrideConfiguration(attr bool) {
	p.overrideConfiguration = &attr
}

func (p *HostNic) OverrideConfiguration() (bool, bool) {
	if p.overrideConfiguration != nil {
		return *p.overrideConfiguration, true
	}
	var zero bool
	return zero, false
}

func (p *HostNic) MustOverrideConfiguration() bool {
	if p.overrideConfiguration == nil {
		panic("the overrideConfiguration must not be nil, please use OverrideConfiguration() function instead")
	}
	return *p.overrideConfiguration
}

// SetPhysicalFunction Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *HostNic) SetPhysicalFunction(attr *HostNic) {
	p.physicalFunction = attr
}

// PhysicalFunction Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *HostNic) PhysicalFunction() (*HostNic, bool) {
	if p.physicalFunction != nil {
		return p.physicalFunction, true
	}
	return nil, false
}

// MustPhysicalFunction Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *HostNic) MustPhysicalFunction() *HostNic {
	if p.physicalFunction == nil {
		panic("the physicalFunction must not be nil, please use PhysicalFunction() function instead")
	}
	return p.physicalFunction
}

func (p *HostNic) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *HostNic) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *HostNic) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

// SetQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *HostNic) SetQos(attr *Qos) {
	p.qos = attr
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *HostNic) Qos() (*Qos, bool) {
	if p.qos != nil {
		return p.qos, true
	}
	return nil, false
}

// MustQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *HostNic) MustQos() *Qos {
	if p.qos == nil {
		panic("the qos must not be nil, please use Qos() function instead")
	}
	return p.qos
}

func (p *HostNic) SetSpeed(attr int64) {
	p.speed = &attr
}

func (p *HostNic) Speed() (int64, bool) {
	if p.speed != nil {
		return *p.speed, true
	}
	var zero int64
	return zero, false
}

func (p *HostNic) MustSpeed() int64 {
	if p.speed == nil {
		panic("the speed must not be nil, please use Speed() function instead")
	}
	return *p.speed
}

func (p *HostNic) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *HostNic) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *HostNic) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// SetStatus Network interface card status.
func (p *HostNic) SetStatus(attr NicStatus) {
	p.status = &attr
}

// Status Network interface card status.
func (p *HostNic) Status() (NicStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero NicStatus
	return zero, false
}

// MustStatus Network interface card status.
func (p *HostNic) MustStatus() NicStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

// SetVirtualFunctionsConfiguration Describes the virtual functions configuration of an SR-IOV-enabled physical function NIC.
func (p *HostNic) SetVirtualFunctionsConfiguration(attr *HostNicVirtualFunctionsConfiguration) {
	p.virtualFunctionsConfiguration = attr
}

// VirtualFunctionsConfiguration Describes the virtual functions configuration of an SR-IOV-enabled physical function NIC.
func (p *HostNic) VirtualFunctionsConfiguration() (*HostNicVirtualFunctionsConfiguration, bool) {
	if p.virtualFunctionsConfiguration != nil {
		return p.virtualFunctionsConfiguration, true
	}
	return nil, false
}

// MustVirtualFunctionsConfiguration Describes the virtual functions configuration of an SR-IOV-enabled physical function NIC.
func (p *HostNic) MustVirtualFunctionsConfiguration() *HostNicVirtualFunctionsConfiguration {
	if p.virtualFunctionsConfiguration == nil {
		panic("the virtualFunctionsConfiguration must not be nil, please use VirtualFunctionsConfiguration() function instead")
	}
	return p.virtualFunctionsConfiguration
}

// SetVlan Type representing a Virtual LAN (VLAN) type.
func (p *HostNic) SetVlan(attr *Vlan) {
	p.vlan = attr
}

// Vlan Type representing a Virtual LAN (VLAN) type.
func (p *HostNic) Vlan() (*Vlan, bool) {
	if p.vlan != nil {
		return p.vlan, true
	}
	return nil, false
}

// MustVlan Type representing a Virtual LAN (VLAN) type.
func (p *HostNic) MustVlan() *Vlan {
	if p.vlan == nil {
		panic("the vlan must not be nil, please use Vlan() function instead")
	}
	return p.vlan
}

// HostNicVirtualFunctionsConfiguration Describes the virtual functions configuration of an SR-IOV-enabled physical function NIC.
type HostNicVirtualFunctionsConfiguration struct {
	Struct
	allNetworksAllowed          *bool
	maxNumberOfVirtualFunctions *int64
	numberOfVirtualFunctions    *int64
}

func (p *HostNicVirtualFunctionsConfiguration) SetAllNetworksAllowed(attr bool) {
	p.allNetworksAllowed = &attr
}

func (p *HostNicVirtualFunctionsConfiguration) AllNetworksAllowed() (bool, bool) {
	if p.allNetworksAllowed != nil {
		return *p.allNetworksAllowed, true
	}
	var zero bool
	return zero, false
}

func (p *HostNicVirtualFunctionsConfiguration) MustAllNetworksAllowed() bool {
	if p.allNetworksAllowed == nil {
		panic("the allNetworksAllowed must not be nil, please use AllNetworksAllowed() function instead")
	}
	return *p.allNetworksAllowed
}

func (p *HostNicVirtualFunctionsConfiguration) SetMaxNumberOfVirtualFunctions(attr int64) {
	p.maxNumberOfVirtualFunctions = &attr
}

func (p *HostNicVirtualFunctionsConfiguration) MaxNumberOfVirtualFunctions() (int64, bool) {
	if p.maxNumberOfVirtualFunctions != nil {
		return *p.maxNumberOfVirtualFunctions, true
	}
	var zero int64
	return zero, false
}

func (p *HostNicVirtualFunctionsConfiguration) MustMaxNumberOfVirtualFunctions() int64 {
	if p.maxNumberOfVirtualFunctions == nil {
		panic("the maxNumberOfVirtualFunctions must not be nil, please use MaxNumberOfVirtualFunctions() function instead")
	}
	return *p.maxNumberOfVirtualFunctions
}

func (p *HostNicVirtualFunctionsConfiguration) SetNumberOfVirtualFunctions(attr int64) {
	p.numberOfVirtualFunctions = &attr
}

func (p *HostNicVirtualFunctionsConfiguration) NumberOfVirtualFunctions() (int64, bool) {
	if p.numberOfVirtualFunctions != nil {
		return *p.numberOfVirtualFunctions, true
	}
	var zero int64
	return zero, false
}

func (p *HostNicVirtualFunctionsConfiguration) MustNumberOfVirtualFunctions() int64 {
	if p.numberOfVirtualFunctions == nil {
		panic("the numberOfVirtualFunctions must not be nil, please use NumberOfVirtualFunctions() function instead")
	}
	return *p.numberOfVirtualFunctions
}

type HostStorage struct {
	Struct
	address                *string
	comment                *string
	description            *string
	driverOptions          *PropertySlice
	driverSensitiveOptions *PropertySlice
	host                   *Host
	id                     *string
	logicalUnits           *LogicalUnitSlice
	mountOptions           *string
	name                   *string
	nfsRetrans             *int64
	nfsTimeo               *int64
	nfsVersion             *NfsVersion
	overrideLuns           *bool
	password               *string
	path                   *string
	port                   *int64
	portal                 *string
	target                 *string
	type_                  *StorageType
	username               *string
	vfsType                *string
	volumeGroup            *VolumeGroup
}

func (p *HostStorage) SetAddress(attr string) {
	p.address = &attr
}

func (p *HostStorage) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *HostStorage) SetComment(attr string) {
	p.comment = &attr
}

func (p *HostStorage) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *HostStorage) SetDescription(attr string) {
	p.description = &attr
}

func (p *HostStorage) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *HostStorage) SetDriverOptions(attr *PropertySlice) {
	p.driverOptions = attr
}

func (p *HostStorage) DriverOptions() (*PropertySlice, bool) {
	if p.driverOptions != nil {
		return p.driverOptions, true
	}
	return nil, false
}

func (p *HostStorage) MustDriverOptions() *PropertySlice {
	if p.driverOptions == nil {
		panic("the driverOptions must not be nil, please use DriverOptions() function instead")
	}
	return p.driverOptions
}

func (p *HostStorage) SetDriverSensitiveOptions(attr *PropertySlice) {
	p.driverSensitiveOptions = attr
}

func (p *HostStorage) DriverSensitiveOptions() (*PropertySlice, bool) {
	if p.driverSensitiveOptions != nil {
		return p.driverSensitiveOptions, true
	}
	return nil, false
}

func (p *HostStorage) MustDriverSensitiveOptions() *PropertySlice {
	if p.driverSensitiveOptions == nil {
		panic("the driverSensitiveOptions must not be nil, please use DriverSensitiveOptions() function instead")
	}
	return p.driverSensitiveOptions
}

// SetHost Type representing a host.
func (p *HostStorage) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *HostStorage) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *HostStorage) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *HostStorage) SetId(attr string) {
	p.id = &attr
}

func (p *HostStorage) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *HostStorage) SetLogicalUnits(attr *LogicalUnitSlice) {
	p.logicalUnits = attr
}

func (p *HostStorage) LogicalUnits() (*LogicalUnitSlice, bool) {
	if p.logicalUnits != nil {
		return p.logicalUnits, true
	}
	return nil, false
}

func (p *HostStorage) MustLogicalUnits() *LogicalUnitSlice {
	if p.logicalUnits == nil {
		panic("the logicalUnits must not be nil, please use LogicalUnits() function instead")
	}
	return p.logicalUnits
}

func (p *HostStorage) SetMountOptions(attr string) {
	p.mountOptions = &attr
}

func (p *HostStorage) MountOptions() (string, bool) {
	if p.mountOptions != nil {
		return *p.mountOptions, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustMountOptions() string {
	if p.mountOptions == nil {
		panic("the mountOptions must not be nil, please use MountOptions() function instead")
	}
	return *p.mountOptions
}

func (p *HostStorage) SetName(attr string) {
	p.name = &attr
}

func (p *HostStorage) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *HostStorage) SetNfsRetrans(attr int64) {
	p.nfsRetrans = &attr
}

func (p *HostStorage) NfsRetrans() (int64, bool) {
	if p.nfsRetrans != nil {
		return *p.nfsRetrans, true
	}
	var zero int64
	return zero, false
}

func (p *HostStorage) MustNfsRetrans() int64 {
	if p.nfsRetrans == nil {
		panic("the nfsRetrans must not be nil, please use NfsRetrans() function instead")
	}
	return *p.nfsRetrans
}

func (p *HostStorage) SetNfsTimeo(attr int64) {
	p.nfsTimeo = &attr
}

func (p *HostStorage) NfsTimeo() (int64, bool) {
	if p.nfsTimeo != nil {
		return *p.nfsTimeo, true
	}
	var zero int64
	return zero, false
}

func (p *HostStorage) MustNfsTimeo() int64 {
	if p.nfsTimeo == nil {
		panic("the nfsTimeo must not be nil, please use NfsTimeo() function instead")
	}
	return *p.nfsTimeo
}

func (p *HostStorage) SetNfsVersion(attr NfsVersion) {
	p.nfsVersion = &attr
}

func (p *HostStorage) NfsVersion() (NfsVersion, bool) {
	if p.nfsVersion != nil {
		return *p.nfsVersion, true
	}
	var zero NfsVersion
	return zero, false
}

func (p *HostStorage) MustNfsVersion() NfsVersion {
	if p.nfsVersion == nil {
		panic("the nfsVersion must not be nil, please use NfsVersion() function instead")
	}
	return *p.nfsVersion
}

func (p *HostStorage) SetOverrideLuns(attr bool) {
	p.overrideLuns = &attr
}

func (p *HostStorage) OverrideLuns() (bool, bool) {
	if p.overrideLuns != nil {
		return *p.overrideLuns, true
	}
	var zero bool
	return zero, false
}

func (p *HostStorage) MustOverrideLuns() bool {
	if p.overrideLuns == nil {
		panic("the overrideLuns must not be nil, please use OverrideLuns() function instead")
	}
	return *p.overrideLuns
}

func (p *HostStorage) SetPassword(attr string) {
	p.password = &attr
}

func (p *HostStorage) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *HostStorage) SetPath(attr string) {
	p.path = &attr
}

func (p *HostStorage) Path() (string, bool) {
	if p.path != nil {
		return *p.path, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustPath() string {
	if p.path == nil {
		panic("the path must not be nil, please use Path() function instead")
	}
	return *p.path
}

func (p *HostStorage) SetPort(attr int64) {
	p.port = &attr
}

func (p *HostStorage) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *HostStorage) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *HostStorage) SetPortal(attr string) {
	p.portal = &attr
}

func (p *HostStorage) Portal() (string, bool) {
	if p.portal != nil {
		return *p.portal, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustPortal() string {
	if p.portal == nil {
		panic("the portal must not be nil, please use Portal() function instead")
	}
	return *p.portal
}

func (p *HostStorage) SetTarget(attr string) {
	p.target = &attr
}

func (p *HostStorage) Target() (string, bool) {
	if p.target != nil {
		return *p.target, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustTarget() string {
	if p.target == nil {
		panic("the target must not be nil, please use Target() function instead")
	}
	return *p.target
}

// SetType Type representing a storage domain type.
func (p *HostStorage) SetType(attr StorageType) {
	p.type_ = &attr
}

// Type Type representing a storage domain type.
func (p *HostStorage) Type() (StorageType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero StorageType
	return zero, false
}

// MustType Type representing a storage domain type.
func (p *HostStorage) MustType() StorageType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *HostStorage) SetUsername(attr string) {
	p.username = &attr
}

func (p *HostStorage) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

func (p *HostStorage) SetVfsType(attr string) {
	p.vfsType = &attr
}

func (p *HostStorage) VfsType() (string, bool) {
	if p.vfsType != nil {
		return *p.vfsType, true
	}
	var zero string
	return zero, false
}

func (p *HostStorage) MustVfsType() string {
	if p.vfsType == nil {
		panic("the vfsType must not be nil, please use VfsType() function instead")
	}
	return *p.vfsType
}

func (p *HostStorage) SetVolumeGroup(attr *VolumeGroup) {
	p.volumeGroup = attr
}

func (p *HostStorage) VolumeGroup() (*VolumeGroup, bool) {
	if p.volumeGroup != nil {
		return p.volumeGroup, true
	}
	return nil, false
}

func (p *HostStorage) MustVolumeGroup() *VolumeGroup {
	if p.volumeGroup == nil {
		panic("the volumeGroup must not be nil, please use VolumeGroup() function instead")
	}
	return p.volumeGroup
}

type HostedEngine struct {
	Struct
	active            *bool
	configured        *bool
	globalMaintenance *bool
	localMaintenance  *bool
	score             *int64
}

func (p *HostedEngine) SetActive(attr bool) {
	p.active = &attr
}

func (p *HostedEngine) Active() (bool, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero bool
	return zero, false
}

func (p *HostedEngine) MustActive() bool {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *HostedEngine) SetConfigured(attr bool) {
	p.configured = &attr
}

func (p *HostedEngine) Configured() (bool, bool) {
	if p.configured != nil {
		return *p.configured, true
	}
	var zero bool
	return zero, false
}

func (p *HostedEngine) MustConfigured() bool {
	if p.configured == nil {
		panic("the configured must not be nil, please use Configured() function instead")
	}
	return *p.configured
}

func (p *HostedEngine) SetGlobalMaintenance(attr bool) {
	p.globalMaintenance = &attr
}

func (p *HostedEngine) GlobalMaintenance() (bool, bool) {
	if p.globalMaintenance != nil {
		return *p.globalMaintenance, true
	}
	var zero bool
	return zero, false
}

func (p *HostedEngine) MustGlobalMaintenance() bool {
	if p.globalMaintenance == nil {
		panic("the globalMaintenance must not be nil, please use GlobalMaintenance() function instead")
	}
	return *p.globalMaintenance
}

func (p *HostedEngine) SetLocalMaintenance(attr bool) {
	p.localMaintenance = &attr
}

func (p *HostedEngine) LocalMaintenance() (bool, bool) {
	if p.localMaintenance != nil {
		return *p.localMaintenance, true
	}
	var zero bool
	return zero, false
}

func (p *HostedEngine) MustLocalMaintenance() bool {
	if p.localMaintenance == nil {
		panic("the localMaintenance must not be nil, please use LocalMaintenance() function instead")
	}
	return *p.localMaintenance
}

func (p *HostedEngine) SetScore(attr int64) {
	p.score = &attr
}

func (p *HostedEngine) Score() (int64, bool) {
	if p.score != nil {
		return *p.score, true
	}
	var zero int64
	return zero, false
}

func (p *HostedEngine) MustScore() int64 {
	if p.score == nil {
		panic("the score must not be nil, please use Score() function instead")
	}
	return *p.score
}

// Icon Icon of virtual machine or template.
type Icon struct {
	Struct
	comment     *string
	data        *string
	description *string
	id          *string
	mediaType   *string
	name        *string
}

func (p *Icon) SetComment(attr string) {
	p.comment = &attr
}

func (p *Icon) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Icon) SetData(attr string) {
	p.data = &attr
}

func (p *Icon) Data() (string, bool) {
	if p.data != nil {
		return *p.data, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustData() string {
	if p.data == nil {
		panic("the data must not be nil, please use Data() function instead")
	}
	return *p.data
}

func (p *Icon) SetDescription(attr string) {
	p.description = &attr
}

func (p *Icon) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Icon) SetId(attr string) {
	p.id = &attr
}

func (p *Icon) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Icon) SetMediaType(attr string) {
	p.mediaType = &attr
}

func (p *Icon) MediaType() (string, bool) {
	if p.mediaType != nil {
		return *p.mediaType, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustMediaType() string {
	if p.mediaType == nil {
		panic("the mediaType must not be nil, please use MediaType() function instead")
	}
	return *p.mediaType
}

func (p *Icon) SetName(attr string) {
	p.name = &attr
}

func (p *Icon) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Icon) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// Identified This interface is the base model for all types that represent objects with an identifier.
type Identified struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
}

func (p *Identified) SetComment(attr string) {
	p.comment = &attr
}

func (p *Identified) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Identified) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Identified) SetDescription(attr string) {
	p.description = &attr
}

func (p *Identified) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Identified) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Identified) SetId(attr string) {
	p.id = &attr
}

func (p *Identified) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Identified) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Identified) SetName(attr string) {
	p.name = &attr
}

func (p *Identified) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Identified) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// Image Represents an image entity.
type Image struct {
	Struct
	comment       *string
	description   *string
	id            *string
	name          *string
	size          *int64
	storageDomain *StorageDomain
	type_         *ImageFileType
}

func (p *Image) SetComment(attr string) {
	p.comment = &attr
}

func (p *Image) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Image) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Image) SetDescription(attr string) {
	p.description = &attr
}

func (p *Image) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Image) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Image) SetId(attr string) {
	p.id = &attr
}

func (p *Image) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Image) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Image) SetName(attr string) {
	p.name = &attr
}

func (p *Image) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Image) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Image) SetSize(attr int64) {
	p.size = &attr
}

func (p *Image) Size() (int64, bool) {
	if p.size != nil {
		return *p.size, true
	}
	var zero int64
	return zero, false
}

func (p *Image) MustSize() int64 {
	if p.size == nil {
		panic("the size must not be nil, please use Size() function instead")
	}
	return *p.size
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Image) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Image) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Image) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

// SetType Represents the file type of an image.
func (p *Image) SetType(attr ImageFileType) {
	p.type_ = &attr
}

// Type Represents the file type of an image.
func (p *Image) Type() (ImageFileType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero ImageFileType
	return zero, false
}

// MustType Represents the file type of an image.
func (p *Image) MustType() ImageFileType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// ImageTransfer This type contains information regarding an image transfer being performed.
type ImageTransfer struct {
	Struct
	active            *bool
	backup            *Backup
	comment           *string
	description       *string
	direction         *ImageTransferDirection
	disk              *Disk
	format            *DiskFormat
	host              *Host
	id                *string
	image             *Image
	inactivityTimeout *int64
	name              *string
	phase             *ImageTransferPhase
	proxyUrl          *string
	shallow           *bool
	snapshot          *DiskSnapshot
	timeoutPolicy     *ImageTransferTimeoutPolicy
	transferUrl       *string
	transferred       *int64
}

func (p *ImageTransfer) SetActive(attr bool) {
	p.active = &attr
}

func (p *ImageTransfer) Active() (bool, bool) {
	if p.active != nil {
		return *p.active, true
	}
	var zero bool
	return zero, false
}

func (p *ImageTransfer) MustActive() bool {
	if p.active == nil {
		panic("the active must not be nil, please use Active() function instead")
	}
	return *p.active
}

func (p *ImageTransfer) SetBackup(attr *Backup) {
	p.backup = attr
}

func (p *ImageTransfer) Backup() (*Backup, bool) {
	if p.backup != nil {
		return p.backup, true
	}
	return nil, false
}

func (p *ImageTransfer) MustBackup() *Backup {
	if p.backup == nil {
		panic("the backup must not be nil, please use Backup() function instead")
	}
	return p.backup
}

func (p *ImageTransfer) SetComment(attr string) {
	p.comment = &attr
}

func (p *ImageTransfer) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *ImageTransfer) SetDescription(attr string) {
	p.description = &attr
}

func (p *ImageTransfer) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDirection The <<types/image_transfer, image transfer>> direction for a transfer.
//
// When adding a new transfer, the user can choose whether the transfer will be to an image, choosing `upload`,
// or to transfer from an image- choosing `download` as an ImageTransferDirection.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) SetDirection(attr ImageTransferDirection) {
	p.direction = &attr
}

// Direction The <<types/image_transfer, image transfer>> direction for a transfer.
//
// When adding a new transfer, the user can choose whether the transfer will be to an image, choosing `upload`,
// or to transfer from an image- choosing `download` as an ImageTransferDirection.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) Direction() (ImageTransferDirection, bool) {
	if p.direction != nil {
		return *p.direction, true
	}
	var zero ImageTransferDirection
	return zero, false
}

// MustDirection The <<types/image_transfer, image transfer>> direction for a transfer.
//
// When adding a new transfer, the user can choose whether the transfer will be to an image, choosing `upload`,
// or to transfer from an image- choosing `download` as an ImageTransferDirection.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) MustDirection() ImageTransferDirection {
	if p.direction == nil {
		panic("the direction must not be nil, please use Direction() function instead")
	}
	return *p.direction
}

// SetDisk Represents a virtual disk device.
func (p *ImageTransfer) SetDisk(attr *Disk) {
	p.disk = attr
}

// Disk Represents a virtual disk device.
func (p *ImageTransfer) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

// MustDisk Represents a virtual disk device.
func (p *ImageTransfer) MustDisk() *Disk {
	if p.disk == nil {
		panic("the disk must not be nil, please use Disk() function instead")
	}
	return p.disk
}

// SetFormat The underlying storage format of disks.
func (p *ImageTransfer) SetFormat(attr DiskFormat) {
	p.format = &attr
}

// Format The underlying storage format of disks.
func (p *ImageTransfer) Format() (DiskFormat, bool) {
	if p.format != nil {
		return *p.format, true
	}
	var zero DiskFormat
	return zero, false
}

// MustFormat The underlying storage format of disks.
func (p *ImageTransfer) MustFormat() DiskFormat {
	if p.format == nil {
		panic("the format must not be nil, please use Format() function instead")
	}
	return *p.format
}

// SetHost Type representing a host.
func (p *ImageTransfer) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *ImageTransfer) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *ImageTransfer) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *ImageTransfer) SetId(attr string) {
	p.id = &attr
}

func (p *ImageTransfer) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetImage Represents an image entity.
func (p *ImageTransfer) SetImage(attr *Image) {
	p.image = attr
}

// Image Represents an image entity.
func (p *ImageTransfer) Image() (*Image, bool) {
	if p.image != nil {
		return p.image, true
	}
	return nil, false
}

// MustImage Represents an image entity.
func (p *ImageTransfer) MustImage() *Image {
	if p.image == nil {
		panic("the image must not be nil, please use Image() function instead")
	}
	return p.image
}

func (p *ImageTransfer) SetInactivityTimeout(attr int64) {
	p.inactivityTimeout = &attr
}

func (p *ImageTransfer) InactivityTimeout() (int64, bool) {
	if p.inactivityTimeout != nil {
		return *p.inactivityTimeout, true
	}
	var zero int64
	return zero, false
}

func (p *ImageTransfer) MustInactivityTimeout() int64 {
	if p.inactivityTimeout == nil {
		panic("the inactivityTimeout must not be nil, please use InactivityTimeout() function instead")
	}
	return *p.inactivityTimeout
}

func (p *ImageTransfer) SetName(attr string) {
	p.name = &attr
}

func (p *ImageTransfer) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetPhase A list of possible phases for an <<types/image_transfer, image transfer>> entity. Each of these values
// defines a specific point in a transfer flow.
//
// Please refer to <<services/image_transfer, image transfer>> for more
// information.
func (p *ImageTransfer) SetPhase(attr ImageTransferPhase) {
	p.phase = &attr
}

// Phase A list of possible phases for an <<types/image_transfer, image transfer>> entity. Each of these values
// defines a specific point in a transfer flow.
//
// Please refer to <<services/image_transfer, image transfer>> for more
// information.
func (p *ImageTransfer) Phase() (ImageTransferPhase, bool) {
	if p.phase != nil {
		return *p.phase, true
	}
	var zero ImageTransferPhase
	return zero, false
}

// MustPhase A list of possible phases for an <<types/image_transfer, image transfer>> entity. Each of these values
// defines a specific point in a transfer flow.
//
// Please refer to <<services/image_transfer, image transfer>> for more
// information.
func (p *ImageTransfer) MustPhase() ImageTransferPhase {
	if p.phase == nil {
		panic("the phase must not be nil, please use Phase() function instead")
	}
	return *p.phase
}

func (p *ImageTransfer) SetProxyUrl(attr string) {
	p.proxyUrl = &attr
}

func (p *ImageTransfer) ProxyUrl() (string, bool) {
	if p.proxyUrl != nil {
		return *p.proxyUrl, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustProxyUrl() string {
	if p.proxyUrl == nil {
		panic("the proxyUrl must not be nil, please use ProxyUrl() function instead")
	}
	return *p.proxyUrl
}

func (p *ImageTransfer) SetShallow(attr bool) {
	p.shallow = &attr
}

func (p *ImageTransfer) Shallow() (bool, bool) {
	if p.shallow != nil {
		return *p.shallow, true
	}
	var zero bool
	return zero, false
}

func (p *ImageTransfer) MustShallow() bool {
	if p.shallow == nil {
		panic("the shallow must not be nil, please use Shallow() function instead")
	}
	return *p.shallow
}

func (p *ImageTransfer) SetSnapshot(attr *DiskSnapshot) {
	p.snapshot = attr
}

func (p *ImageTransfer) Snapshot() (*DiskSnapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *ImageTransfer) MustSnapshot() *DiskSnapshot {
	if p.snapshot == nil {
		panic("the snapshot must not be nil, please use Snapshot() function instead")
	}
	return p.snapshot
}

// SetTimeoutPolicy The <<types/image_transfer, image transfer>> timeout policy.
//
// Define how the system handles a transfer when the client is inactive
// for inactivityTimeout seconds.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) SetTimeoutPolicy(attr ImageTransferTimeoutPolicy) {
	p.timeoutPolicy = &attr
}

// TimeoutPolicy The <<types/image_transfer, image transfer>> timeout policy.
//
// Define how the system handles a transfer when the client is inactive
// for inactivityTimeout seconds.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) TimeoutPolicy() (ImageTransferTimeoutPolicy, bool) {
	if p.timeoutPolicy != nil {
		return *p.timeoutPolicy, true
	}
	var zero ImageTransferTimeoutPolicy
	return zero, false
}

// MustTimeoutPolicy The <<types/image_transfer, image transfer>> timeout policy.
//
// Define how the system handles a transfer when the client is inactive
// for inactivityTimeout seconds.
//
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
func (p *ImageTransfer) MustTimeoutPolicy() ImageTransferTimeoutPolicy {
	if p.timeoutPolicy == nil {
		panic("the timeoutPolicy must not be nil, please use TimeoutPolicy() function instead")
	}
	return *p.timeoutPolicy
}

func (p *ImageTransfer) SetTransferUrl(attr string) {
	p.transferUrl = &attr
}

func (p *ImageTransfer) TransferUrl() (string, bool) {
	if p.transferUrl != nil {
		return *p.transferUrl, true
	}
	var zero string
	return zero, false
}

func (p *ImageTransfer) MustTransferUrl() string {
	if p.transferUrl == nil {
		panic("the transferUrl must not be nil, please use TransferUrl() function instead")
	}
	return *p.transferUrl
}

func (p *ImageTransfer) SetTransferred(attr int64) {
	p.transferred = &attr
}

func (p *ImageTransfer) Transferred() (int64, bool) {
	if p.transferred != nil {
		return *p.transferred, true
	}
	var zero int64
	return zero, false
}

func (p *ImageTransfer) MustTransferred() int64 {
	if p.transferred == nil {
		panic("the transferred must not be nil, please use Transferred() function instead")
	}
	return *p.transferred
}

type Initialization struct {
	Struct
	activeDirectoryOu        *string
	authorizedSshKeys        *string
	cloudInit                *CloudInit
	cloudInitNetworkProtocol *CloudInitNetworkProtocol
	configuration            *Configuration
	customScript             *string
	dnsSearch                *string
	dnsServers               *string
	domain                   *string
	hostName                 *string
	inputLocale              *string
	nicConfigurations        *NicConfigurationSlice
	orgName                  *string
	regenerateIds            *bool
	regenerateSshKeys        *bool
	rootPassword             *string
	systemLocale             *string
	timezone                 *string
	uiLanguage               *string
	userLocale               *string
	userName                 *string
	windowsLicenseKey        *string
}

func (p *Initialization) SetActiveDirectoryOu(attr string) {
	p.activeDirectoryOu = &attr
}

func (p *Initialization) ActiveDirectoryOu() (string, bool) {
	if p.activeDirectoryOu != nil {
		return *p.activeDirectoryOu, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustActiveDirectoryOu() string {
	if p.activeDirectoryOu == nil {
		panic("the activeDirectoryOu must not be nil, please use ActiveDirectoryOu() function instead")
	}
	return *p.activeDirectoryOu
}

func (p *Initialization) SetAuthorizedSshKeys(attr string) {
	p.authorizedSshKeys = &attr
}

func (p *Initialization) AuthorizedSshKeys() (string, bool) {
	if p.authorizedSshKeys != nil {
		return *p.authorizedSshKeys, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustAuthorizedSshKeys() string {
	if p.authorizedSshKeys == nil {
		panic("the authorizedSshKeys must not be nil, please use AuthorizedSshKeys() function instead")
	}
	return *p.authorizedSshKeys
}

// SetCloudInit Deprecated type to specify _cloud-init_ configuration.
//
// This type has been deprecated and replaced by alternative attributes inside the
// <<types/initialization, Initialization>> type. See the
// <<types/initialization/attributes/cloud_init, cloud_init>> attribute documentation for details.
func (p *Initialization) SetCloudInit(attr *CloudInit) {
	p.cloudInit = attr
}

// CloudInit Deprecated type to specify _cloud-init_ configuration.
//
// This type has been deprecated and replaced by alternative attributes inside the
// <<types/initialization, Initialization>> type. See the
// <<types/initialization/attributes/cloud_init, cloud_init>> attribute documentation for details.
func (p *Initialization) CloudInit() (*CloudInit, bool) {
	if p.cloudInit != nil {
		return p.cloudInit, true
	}
	return nil, false
}

// MustCloudInit Deprecated type to specify _cloud-init_ configuration.
//
// This type has been deprecated and replaced by alternative attributes inside the
// <<types/initialization, Initialization>> type. See the
// <<types/initialization/attributes/cloud_init, cloud_init>> attribute documentation for details.
func (p *Initialization) MustCloudInit() *CloudInit {
	if p.cloudInit == nil {
		panic("the cloudInit must not be nil, please use CloudInit() function instead")
	}
	return p.cloudInit
}

// SetCloudInitNetworkProtocol Defines the values for the cloud-init protocol.
// This protocol decides how the cloud-init network
// parameters are formatted before being passed to
// the virtual machine in order to be processed by cloud-init.
//
// Protocols supported are cloud-init version dependent.
// For more information, see
// http://cloudinit.readthedocs.io/en/latest/topics/network-config.html#network-configuration-sources[Network Configuration Sources]
func (p *Initialization) SetCloudInitNetworkProtocol(attr CloudInitNetworkProtocol) {
	p.cloudInitNetworkProtocol = &attr
}

// CloudInitNetworkProtocol Defines the values for the cloud-init protocol.
// This protocol decides how the cloud-init network
// parameters are formatted before being passed to
// the virtual machine in order to be processed by cloud-init.
//
// Protocols supported are cloud-init version dependent.
// For more information, see
// http://cloudinit.readthedocs.io/en/latest/topics/network-config.html#network-configuration-sources[Network Configuration Sources]
func (p *Initialization) CloudInitNetworkProtocol() (CloudInitNetworkProtocol, bool) {
	if p.cloudInitNetworkProtocol != nil {
		return *p.cloudInitNetworkProtocol, true
	}
	var zero CloudInitNetworkProtocol
	return zero, false
}

// MustCloudInitNetworkProtocol Defines the values for the cloud-init protocol.
// This protocol decides how the cloud-init network
// parameters are formatted before being passed to
// the virtual machine in order to be processed by cloud-init.
//
// Protocols supported are cloud-init version dependent.
// For more information, see
// http://cloudinit.readthedocs.io/en/latest/topics/network-config.html#network-configuration-sources[Network Configuration Sources]
func (p *Initialization) MustCloudInitNetworkProtocol() CloudInitNetworkProtocol {
	if p.cloudInitNetworkProtocol == nil {
		panic("the cloudInitNetworkProtocol must not be nil, please use CloudInitNetworkProtocol() function instead")
	}
	return *p.cloudInitNetworkProtocol
}

func (p *Initialization) SetConfiguration(attr *Configuration) {
	p.configuration = attr
}

func (p *Initialization) Configuration() (*Configuration, bool) {
	if p.configuration != nil {
		return p.configuration, true
	}
	return nil, false
}

func (p *Initialization) MustConfiguration() *Configuration {
	if p.configuration == nil {
		panic("the configuration must not be nil, please use Configuration() function instead")
	}
	return p.configuration
}

func (p *Initialization) SetCustomScript(attr string) {
	p.customScript = &attr
}

func (p *Initialization) CustomScript() (string, bool) {
	if p.customScript != nil {
		return *p.customScript, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustCustomScript() string {
	if p.customScript == nil {
		panic("the customScript must not be nil, please use CustomScript() function instead")
	}
	return *p.customScript
}

func (p *Initialization) SetDnsSearch(attr string) {
	p.dnsSearch = &attr
}

func (p *Initialization) DnsSearch() (string, bool) {
	if p.dnsSearch != nil {
		return *p.dnsSearch, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustDnsSearch() string {
	if p.dnsSearch == nil {
		panic("the dnsSearch must not be nil, please use DnsSearch() function instead")
	}
	return *p.dnsSearch
}

func (p *Initialization) SetDnsServers(attr string) {
	p.dnsServers = &attr
}

func (p *Initialization) DnsServers() (string, bool) {
	if p.dnsServers != nil {
		return *p.dnsServers, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustDnsServers() string {
	if p.dnsServers == nil {
		panic("the dnsServers must not be nil, please use DnsServers() function instead")
	}
	return *p.dnsServers
}

func (p *Initialization) SetDomain(attr string) {
	p.domain = &attr
}

func (p *Initialization) Domain() (string, bool) {
	if p.domain != nil {
		return *p.domain, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustDomain() string {
	if p.domain == nil {
		panic("the domain must not be nil, please use Domain() function instead")
	}
	return *p.domain
}

func (p *Initialization) SetHostName(attr string) {
	p.hostName = &attr
}

func (p *Initialization) HostName() (string, bool) {
	if p.hostName != nil {
		return *p.hostName, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustHostName() string {
	if p.hostName == nil {
		panic("the hostName must not be nil, please use HostName() function instead")
	}
	return *p.hostName
}

func (p *Initialization) SetInputLocale(attr string) {
	p.inputLocale = &attr
}

func (p *Initialization) InputLocale() (string, bool) {
	if p.inputLocale != nil {
		return *p.inputLocale, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustInputLocale() string {
	if p.inputLocale == nil {
		panic("the inputLocale must not be nil, please use InputLocale() function instead")
	}
	return *p.inputLocale
}

func (p *Initialization) SetNicConfigurations(attr *NicConfigurationSlice) {
	p.nicConfigurations = attr
}

func (p *Initialization) NicConfigurations() (*NicConfigurationSlice, bool) {
	if p.nicConfigurations != nil {
		return p.nicConfigurations, true
	}
	return nil, false
}

func (p *Initialization) MustNicConfigurations() *NicConfigurationSlice {
	if p.nicConfigurations == nil {
		panic("the nicConfigurations must not be nil, please use NicConfigurations() function instead")
	}
	return p.nicConfigurations
}

func (p *Initialization) SetOrgName(attr string) {
	p.orgName = &attr
}

func (p *Initialization) OrgName() (string, bool) {
	if p.orgName != nil {
		return *p.orgName, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustOrgName() string {
	if p.orgName == nil {
		panic("the orgName must not be nil, please use OrgName() function instead")
	}
	return *p.orgName
}

func (p *Initialization) SetRegenerateIds(attr bool) {
	p.regenerateIds = &attr
}

func (p *Initialization) RegenerateIds() (bool, bool) {
	if p.regenerateIds != nil {
		return *p.regenerateIds, true
	}
	var zero bool
	return zero, false
}

func (p *Initialization) MustRegenerateIds() bool {
	if p.regenerateIds == nil {
		panic("the regenerateIds must not be nil, please use RegenerateIds() function instead")
	}
	return *p.regenerateIds
}

func (p *Initialization) SetRegenerateSshKeys(attr bool) {
	p.regenerateSshKeys = &attr
}

func (p *Initialization) RegenerateSshKeys() (bool, bool) {
	if p.regenerateSshKeys != nil {
		return *p.regenerateSshKeys, true
	}
	var zero bool
	return zero, false
}

func (p *Initialization) MustRegenerateSshKeys() bool {
	if p.regenerateSshKeys == nil {
		panic("the regenerateSshKeys must not be nil, please use RegenerateSshKeys() function instead")
	}
	return *p.regenerateSshKeys
}

func (p *Initialization) SetRootPassword(attr string) {
	p.rootPassword = &attr
}

func (p *Initialization) RootPassword() (string, bool) {
	if p.rootPassword != nil {
		return *p.rootPassword, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustRootPassword() string {
	if p.rootPassword == nil {
		panic("the rootPassword must not be nil, please use RootPassword() function instead")
	}
	return *p.rootPassword
}

func (p *Initialization) SetSystemLocale(attr string) {
	p.systemLocale = &attr
}

func (p *Initialization) SystemLocale() (string, bool) {
	if p.systemLocale != nil {
		return *p.systemLocale, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustSystemLocale() string {
	if p.systemLocale == nil {
		panic("the systemLocale must not be nil, please use SystemLocale() function instead")
	}
	return *p.systemLocale
}

func (p *Initialization) SetTimezone(attr string) {
	p.timezone = &attr
}

func (p *Initialization) Timezone() (string, bool) {
	if p.timezone != nil {
		return *p.timezone, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustTimezone() string {
	if p.timezone == nil {
		panic("the timezone must not be nil, please use Timezone() function instead")
	}
	return *p.timezone
}

func (p *Initialization) SetUiLanguage(attr string) {
	p.uiLanguage = &attr
}

func (p *Initialization) UiLanguage() (string, bool) {
	if p.uiLanguage != nil {
		return *p.uiLanguage, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustUiLanguage() string {
	if p.uiLanguage == nil {
		panic("the uiLanguage must not be nil, please use UiLanguage() function instead")
	}
	return *p.uiLanguage
}

func (p *Initialization) SetUserLocale(attr string) {
	p.userLocale = &attr
}

func (p *Initialization) UserLocale() (string, bool) {
	if p.userLocale != nil {
		return *p.userLocale, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustUserLocale() string {
	if p.userLocale == nil {
		panic("the userLocale must not be nil, please use UserLocale() function instead")
	}
	return *p.userLocale
}

func (p *Initialization) SetUserName(attr string) {
	p.userName = &attr
}

func (p *Initialization) UserName() (string, bool) {
	if p.userName != nil {
		return *p.userName, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustUserName() string {
	if p.userName == nil {
		panic("the userName must not be nil, please use UserName() function instead")
	}
	return *p.userName
}

func (p *Initialization) SetWindowsLicenseKey(attr string) {
	p.windowsLicenseKey = &attr
}

func (p *Initialization) WindowsLicenseKey() (string, bool) {
	if p.windowsLicenseKey != nil {
		return *p.windowsLicenseKey, true
	}
	var zero string
	return zero, false
}

func (p *Initialization) MustWindowsLicenseKey() string {
	if p.windowsLicenseKey == nil {
		panic("the windowsLicenseKey must not be nil, please use WindowsLicenseKey() function instead")
	}
	return *p.windowsLicenseKey
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
type InstanceType struct {
	Struct
	bios                         *Bios
	cdroms                       *CdromSlice
	cluster                      *Cluster
	comment                      *string
	console                      *Console
	cpu                          *Cpu
	cpuProfile                   *CpuProfile
	cpuShares                    *int64
	creationTime                 *time.Time
	customCompatibilityVersion   *Version
	customCpuModel               *string
	customEmulatedMachine        *string
	customProperties             *CustomPropertySlice
	deleteProtected              *bool
	description                  *string
	diskAttachments              *DiskAttachmentSlice
	display                      *Display
	domain                       *Domain
	graphicsConsoles             *GraphicsConsoleSlice
	highAvailability             *HighAvailability
	id                           *string
	initialization               *Initialization
	io                           *Io
	largeIcon                    *Icon
	lease                        *StorageDomainLease
	memory                       *int64
	memoryPolicy                 *MemoryPolicy
	migration                    *MigrationOptions
	migrationDowntime            *int64
	multiQueuesEnabled           *bool
	name                         *string
	nics                         *NicSlice
	origin                       *string
	os                           *OperatingSystem
	permissions                  *PermissionSlice
	placementPolicy              *VmPlacementPolicy
	quota                        *Quota
	rngDevice                    *RngDevice
	serialNumber                 *SerialNumber
	smallIcon                    *Icon
	soundcardEnabled             *bool
	sso                          *Sso
	startPaused                  *bool
	stateless                    *bool
	status                       *TemplateStatus
	storageDomain                *StorageDomain
	storageErrorResumeBehaviour  *VmStorageErrorResumeBehaviour
	tags                         *TagSlice
	timeZone                     *TimeZone
	tpmEnabled                   *bool
	tunnelMigration              *bool
	type_                        *VmType
	usb                          *Usb
	version                      *TemplateVersion
	virtioScsi                   *VirtioScsi
	virtioScsiMultiQueuesEnabled *bool
	vm                           *Vm
	watchdogs                    *WatchdogSlice
}

func (p *InstanceType) SetBios(attr *Bios) {
	p.bios = attr
}

func (p *InstanceType) Bios() (*Bios, bool) {
	if p.bios != nil {
		return p.bios, true
	}
	return nil, false
}

func (p *InstanceType) MustBios() *Bios {
	if p.bios == nil {
		panic("the bios must not be nil, please use Bios() function instead")
	}
	return p.bios
}

func (p *InstanceType) SetCdroms(attr *CdromSlice) {
	p.cdroms = attr
}

func (p *InstanceType) Cdroms() (*CdromSlice, bool) {
	if p.cdroms != nil {
		return p.cdroms, true
	}
	return nil, false
}

func (p *InstanceType) MustCdroms() *CdromSlice {
	if p.cdroms == nil {
		panic("the cdroms must not be nil, please use Cdroms() function instead")
	}
	return p.cdroms
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *InstanceType) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *InstanceType) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *InstanceType) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *InstanceType) SetComment(attr string) {
	p.comment = &attr
}

func (p *InstanceType) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

// SetConsole Representation for serial console device.
func (p *InstanceType) SetConsole(attr *Console) {
	p.console = attr
}

// Console Representation for serial console device.
func (p *InstanceType) Console() (*Console, bool) {
	if p.console != nil {
		return p.console, true
	}
	return nil, false
}

// MustConsole Representation for serial console device.
func (p *InstanceType) MustConsole() *Console {
	if p.console == nil {
		panic("the console must not be nil, please use Console() function instead")
	}
	return p.console
}

func (p *InstanceType) SetCpu(attr *Cpu) {
	p.cpu = attr
}

func (p *InstanceType) Cpu() (*Cpu, bool) {
	if p.cpu != nil {
		return p.cpu, true
	}
	return nil, false
}

func (p *InstanceType) MustCpu() *Cpu {
	if p.cpu == nil {
		panic("the cpu must not be nil, please use Cpu() function instead")
	}
	return p.cpu
}

func (p *InstanceType) SetCpuProfile(attr *CpuProfile) {
	p.cpuProfile = attr
}

func (p *InstanceType) CpuProfile() (*CpuProfile, bool) {
	if p.cpuProfile != nil {
		return p.cpuProfile, true
	}
	return nil, false
}

func (p *InstanceType) MustCpuProfile() *CpuProfile {
	if p.cpuProfile == nil {
		panic("the cpuProfile must not be nil, please use CpuProfile() function instead")
	}
	return p.cpuProfile
}

func (p *InstanceType) SetCpuShares(attr int64) {
	p.cpuShares = &attr
}

func (p *InstanceType) CpuShares() (int64, bool) {
	if p.cpuShares != nil {
		return *p.cpuShares, true
	}
	var zero int64
	return zero, false
}

func (p *InstanceType) MustCpuShares() int64 {
	if p.cpuShares == nil {
		panic("the cpuShares must not be nil, please use CpuShares() function instead")
	}
	return *p.cpuShares
}

func (p *InstanceType) SetCreationTime(attr time.Time) {
	p.creationTime = &attr
}

func (p *InstanceType) CreationTime() (time.Time, bool) {
	if p.creationTime != nil {
		return *p.creationTime, true
	}
	var zero time.Time
	return zero, false
}

func (p *InstanceType) MustCreationTime() time.Time {
	if p.creationTime == nil {
		panic("the creationTime must not be nil, please use CreationTime() function instead")
	}
	return *p.creationTime
}

func (p *InstanceType) SetCustomCompatibilityVersion(attr *Version) {
	p.customCompatibilityVersion = attr
}

func (p *InstanceType) CustomCompatibilityVersion() (*Version, bool) {
	if p.customCompatibilityVersion != nil {
		return p.customCompatibilityVersion, true
	}
	return nil, false
}

func (p *InstanceType) MustCustomCompatibilityVersion() *Version {
	if p.customCompatibilityVersion == nil {
		panic("the customCompatibilityVersion must not be nil, please use CustomCompatibilityVersion() function instead")
	}
	return p.customCompatibilityVersion
}

func (p *InstanceType) SetCustomCpuModel(attr string) {
	p.customCpuModel = &attr
}

func (p *InstanceType) CustomCpuModel() (string, bool) {
	if p.customCpuModel != nil {
		return *p.customCpuModel, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustCustomCpuModel() string {
	if p.customCpuModel == nil {
		panic("the customCpuModel must not be nil, please use CustomCpuModel() function instead")
	}
	return *p.customCpuModel
}

func (p *InstanceType) SetCustomEmulatedMachine(attr string) {
	p.customEmulatedMachine = &attr
}

func (p *InstanceType) CustomEmulatedMachine() (string, bool) {
	if p.customEmulatedMachine != nil {
		return *p.customEmulatedMachine, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustCustomEmulatedMachine() string {
	if p.customEmulatedMachine == nil {
		panic("the customEmulatedMachine must not be nil, please use CustomEmulatedMachine() function instead")
	}
	return *p.customEmulatedMachine
}

func (p *InstanceType) SetCustomProperties(attr *CustomPropertySlice) {
	p.customProperties = attr
}

func (p *InstanceType) CustomProperties() (*CustomPropertySlice, bool) {
	if p.customProperties != nil {
		return p.customProperties, true
	}
	return nil, false
}

func (p *InstanceType) MustCustomProperties() *CustomPropertySlice {
	if p.customProperties == nil {
		panic("the customProperties must not be nil, please use CustomProperties() function instead")
	}
	return p.customProperties
}

func (p *InstanceType) SetDeleteProtected(attr bool) {
	p.deleteProtected = &attr
}

func (p *InstanceType) DeleteProtected() (bool, bool) {
	if p.deleteProtected != nil {
		return *p.deleteProtected, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustDeleteProtected() bool {
	if p.deleteProtected == nil {
		panic("the deleteProtected must not be nil, please use DeleteProtected() function instead")
	}
	return *p.deleteProtected
}

func (p *InstanceType) SetDescription(attr string) {
	p.description = &attr
}

func (p *InstanceType) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *InstanceType) SetDiskAttachments(attr *DiskAttachmentSlice) {
	p.diskAttachments = attr
}

func (p *InstanceType) DiskAttachments() (*DiskAttachmentSlice, bool) {
	if p.diskAttachments != nil {
		return p.diskAttachments, true
	}
	return nil, false
}

func (p *InstanceType) MustDiskAttachments() *DiskAttachmentSlice {
	if p.diskAttachments == nil {
		panic("the diskAttachments must not be nil, please use DiskAttachments() function instead")
	}
	return p.diskAttachments
}

// SetDisplay Represents a graphic console configuration.
func (p *InstanceType) SetDisplay(attr *Display) {
	p.display = attr
}

// Display Represents a graphic console configuration.
func (p *InstanceType) Display() (*Display, bool) {
	if p.display != nil {
		return p.display, true
	}
	return nil, false
}

// MustDisplay Represents a graphic console configuration.
func (p *InstanceType) MustDisplay() *Display {
	if p.display == nil {
		panic("the display must not be nil, please use Display() function instead")
	}
	return p.display
}

// SetDomain This type represents a directory service domain.
func (p *InstanceType) SetDomain(attr *Domain) {
	p.domain = attr
}

// Domain This type represents a directory service domain.
func (p *InstanceType) Domain() (*Domain, bool) {
	if p.domain != nil {
		return p.domain, true
	}
	return nil, false
}

// MustDomain This type represents a directory service domain.
func (p *InstanceType) MustDomain() *Domain {
	if p.domain == nil {
		panic("the domain must not be nil, please use Domain() function instead")
	}
	return p.domain
}

func (p *InstanceType) SetGraphicsConsoles(attr *GraphicsConsoleSlice) {
	p.graphicsConsoles = attr
}

func (p *InstanceType) GraphicsConsoles() (*GraphicsConsoleSlice, bool) {
	if p.graphicsConsoles != nil {
		return p.graphicsConsoles, true
	}
	return nil, false
}

func (p *InstanceType) MustGraphicsConsoles() *GraphicsConsoleSlice {
	if p.graphicsConsoles == nil {
		panic("the graphicsConsoles must not be nil, please use GraphicsConsoles() function instead")
	}
	return p.graphicsConsoles
}

// SetHighAvailability Type representing high availability of a virtual machine.
func (p *InstanceType) SetHighAvailability(attr *HighAvailability) {
	p.highAvailability = attr
}

// HighAvailability Type representing high availability of a virtual machine.
func (p *InstanceType) HighAvailability() (*HighAvailability, bool) {
	if p.highAvailability != nil {
		return p.highAvailability, true
	}
	return nil, false
}

// MustHighAvailability Type representing high availability of a virtual machine.
func (p *InstanceType) MustHighAvailability() *HighAvailability {
	if p.highAvailability == nil {
		panic("the highAvailability must not be nil, please use HighAvailability() function instead")
	}
	return p.highAvailability
}

func (p *InstanceType) SetId(attr string) {
	p.id = &attr
}

func (p *InstanceType) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *InstanceType) SetInitialization(attr *Initialization) {
	p.initialization = attr
}

func (p *InstanceType) Initialization() (*Initialization, bool) {
	if p.initialization != nil {
		return p.initialization, true
	}
	return nil, false
}

func (p *InstanceType) MustInitialization() *Initialization {
	if p.initialization == nil {
		panic("the initialization must not be nil, please use Initialization() function instead")
	}
	return p.initialization
}

func (p *InstanceType) SetIo(attr *Io) {
	p.io = attr
}

func (p *InstanceType) Io() (*Io, bool) {
	if p.io != nil {
		return p.io, true
	}
	return nil, false
}

func (p *InstanceType) MustIo() *Io {
	if p.io == nil {
		panic("the io must not be nil, please use Io() function instead")
	}
	return p.io
}

// SetLargeIcon Icon of virtual machine or template.
func (p *InstanceType) SetLargeIcon(attr *Icon) {
	p.largeIcon = attr
}

// LargeIcon Icon of virtual machine or template.
func (p *InstanceType) LargeIcon() (*Icon, bool) {
	if p.largeIcon != nil {
		return p.largeIcon, true
	}
	return nil, false
}

// MustLargeIcon Icon of virtual machine or template.
func (p *InstanceType) MustLargeIcon() *Icon {
	if p.largeIcon == nil {
		panic("the largeIcon must not be nil, please use LargeIcon() function instead")
	}
	return p.largeIcon
}

// SetLease Represents a lease residing on a storage domain.
//
// A lease is a http://www.ovirt.org/develop/developer-guide/vdsm/sanlock[Sanlock] resource residing on a special
// volume on the storage domain, this Sanlock resource is used to provide storage base locking.
func (p *InstanceType) SetLease(attr *StorageDomainLease) {
	p.lease = attr
}

// Lease Represents a lease residing on a storage domain.
//
// A lease is a http://www.ovirt.org/develop/developer-guide/vdsm/sanlock[Sanlock] resource residing on a special
// volume on the storage domain, this Sanlock resource is used to provide storage base locking.
func (p *InstanceType) Lease() (*StorageDomainLease, bool) {
	if p.lease != nil {
		return p.lease, true
	}
	return nil, false
}

// MustLease Represents a lease residing on a storage domain.
//
// A lease is a http://www.ovirt.org/develop/developer-guide/vdsm/sanlock[Sanlock] resource residing on a special
// volume on the storage domain, this Sanlock resource is used to provide storage base locking.
func (p *InstanceType) MustLease() *StorageDomainLease {
	if p.lease == nil {
		panic("the lease must not be nil, please use Lease() function instead")
	}
	return p.lease
}

func (p *InstanceType) SetMemory(attr int64) {
	p.memory = &attr
}

func (p *InstanceType) Memory() (int64, bool) {
	if p.memory != nil {
		return *p.memory, true
	}
	var zero int64
	return zero, false
}

func (p *InstanceType) MustMemory() int64 {
	if p.memory == nil {
		panic("the memory must not be nil, please use Memory() function instead")
	}
	return *p.memory
}

// SetMemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *InstanceType) SetMemoryPolicy(attr *MemoryPolicy) {
	p.memoryPolicy = attr
}

// MemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *InstanceType) MemoryPolicy() (*MemoryPolicy, bool) {
	if p.memoryPolicy != nil {
		return p.memoryPolicy, true
	}
	return nil, false
}

// MustMemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
func (p *InstanceType) MustMemoryPolicy() *MemoryPolicy {
	if p.memoryPolicy == nil {
		panic("the memoryPolicy must not be nil, please use MemoryPolicy() function instead")
	}
	return p.memoryPolicy
}

// SetMigration The type for migration options.
func (p *InstanceType) SetMigration(attr *MigrationOptions) {
	p.migration = attr
}

// Migration The type for migration options.
func (p *InstanceType) Migration() (*MigrationOptions, bool) {
	if p.migration != nil {
		return p.migration, true
	}
	return nil, false
}

// MustMigration The type for migration options.
func (p *InstanceType) MustMigration() *MigrationOptions {
	if p.migration == nil {
		panic("the migration must not be nil, please use Migration() function instead")
	}
	return p.migration
}

func (p *InstanceType) SetMigrationDowntime(attr int64) {
	p.migrationDowntime = &attr
}

func (p *InstanceType) MigrationDowntime() (int64, bool) {
	if p.migrationDowntime != nil {
		return *p.migrationDowntime, true
	}
	var zero int64
	return zero, false
}

func (p *InstanceType) MustMigrationDowntime() int64 {
	if p.migrationDowntime == nil {
		panic("the migrationDowntime must not be nil, please use MigrationDowntime() function instead")
	}
	return *p.migrationDowntime
}

func (p *InstanceType) SetMultiQueuesEnabled(attr bool) {
	p.multiQueuesEnabled = &attr
}

func (p *InstanceType) MultiQueuesEnabled() (bool, bool) {
	if p.multiQueuesEnabled != nil {
		return *p.multiQueuesEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustMultiQueuesEnabled() bool {
	if p.multiQueuesEnabled == nil {
		panic("the multiQueuesEnabled must not be nil, please use MultiQueuesEnabled() function instead")
	}
	return *p.multiQueuesEnabled
}

func (p *InstanceType) SetName(attr string) {
	p.name = &attr
}

func (p *InstanceType) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *InstanceType) SetNics(attr *NicSlice) {
	p.nics = attr
}

func (p *InstanceType) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *InstanceType) MustNics() *NicSlice {
	if p.nics == nil {
		panic("the nics must not be nil, please use Nics() function instead")
	}
	return p.nics
}

func (p *InstanceType) SetOrigin(attr string) {
	p.origin = &attr
}

func (p *InstanceType) Origin() (string, bool) {
	if p.origin != nil {
		return *p.origin, true
	}
	var zero string
	return zero, false
}

func (p *InstanceType) MustOrigin() string {
	if p.origin == nil {
		panic("the origin must not be nil, please use Origin() function instead")
	}
	return *p.origin
}

// SetOs Information describing the operating system. This is used for both virtual machines and hosts.
func (p *InstanceType) SetOs(attr *OperatingSystem) {
	p.os = attr
}

// Os Information describing the operating system. This is used for both virtual machines and hosts.
func (p *InstanceType) Os() (*OperatingSystem, bool) {
	if p.os != nil {
		return p.os, true
	}
	return nil, false
}

// MustOs Information describing the operating system. This is used for both virtual machines and hosts.
func (p *InstanceType) MustOs() *OperatingSystem {
	if p.os == nil {
		panic("the os must not be nil, please use Os() function instead")
	}
	return p.os
}

func (p *InstanceType) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *InstanceType) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *InstanceType) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *InstanceType) SetPlacementPolicy(attr *VmPlacementPolicy) {
	p.placementPolicy = attr
}

func (p *InstanceType) PlacementPolicy() (*VmPlacementPolicy, bool) {
	if p.placementPolicy != nil {
		return p.placementPolicy, true
	}
	return nil, false
}

func (p *InstanceType) MustPlacementPolicy() *VmPlacementPolicy {
	if p.placementPolicy == nil {
		panic("the placementPolicy must not be nil, please use PlacementPolicy() function instead")
	}
	return p.placementPolicy
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *InstanceType) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *InstanceType) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *InstanceType) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

// SetRngDevice Random number generator (RNG) device model.
func (p *InstanceType) SetRngDevice(attr *RngDevice) {
	p.rngDevice = attr
}

// RngDevice Random number generator (RNG) device model.
func (p *InstanceType) RngDevice() (*RngDevice, bool) {
	if p.rngDevice != nil {
		return p.rngDevice, true
	}
	return nil, false
}

// MustRngDevice Random number generator (RNG) device model.
func (p *InstanceType) MustRngDevice() *RngDevice {
	if p.rngDevice == nil {
		panic("the rngDevice must not be nil, please use RngDevice() function instead")
	}
	return p.rngDevice
}

func (p *InstanceType) SetSerialNumber(attr *SerialNumber) {
	p.serialNumber = attr
}

func (p *InstanceType) SerialNumber() (*SerialNumber, bool) {
	if p.serialNumber != nil {
		return p.serialNumber, true
	}
	return nil, false
}

func (p *InstanceType) MustSerialNumber() *SerialNumber {
	if p.serialNumber == nil {
		panic("the serialNumber must not be nil, please use SerialNumber() function instead")
	}
	return p.serialNumber
}

// SetSmallIcon Icon of virtual machine or template.
func (p *InstanceType) SetSmallIcon(attr *Icon) {
	p.smallIcon = attr
}

// SmallIcon Icon of virtual machine or template.
func (p *InstanceType) SmallIcon() (*Icon, bool) {
	if p.smallIcon != nil {
		return p.smallIcon, true
	}
	return nil, false
}

// MustSmallIcon Icon of virtual machine or template.
func (p *InstanceType) MustSmallIcon() *Icon {
	if p.smallIcon == nil {
		panic("the smallIcon must not be nil, please use SmallIcon() function instead")
	}
	return p.smallIcon
}

func (p *InstanceType) SetSoundcardEnabled(attr bool) {
	p.soundcardEnabled = &attr
}

func (p *InstanceType) SoundcardEnabled() (bool, bool) {
	if p.soundcardEnabled != nil {
		return *p.soundcardEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustSoundcardEnabled() bool {
	if p.soundcardEnabled == nil {
		panic("the soundcardEnabled must not be nil, please use SoundcardEnabled() function instead")
	}
	return *p.soundcardEnabled
}

func (p *InstanceType) SetSso(attr *Sso) {
	p.sso = attr
}

func (p *InstanceType) Sso() (*Sso, bool) {
	if p.sso != nil {
		return p.sso, true
	}
	return nil, false
}

func (p *InstanceType) MustSso() *Sso {
	if p.sso == nil {
		panic("the sso must not be nil, please use Sso() function instead")
	}
	return p.sso
}

func (p *InstanceType) SetStartPaused(attr bool) {
	p.startPaused = &attr
}

func (p *InstanceType) StartPaused() (bool, bool) {
	if p.startPaused != nil {
		return *p.startPaused, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustStartPaused() bool {
	if p.startPaused == nil {
		panic("the startPaused must not be nil, please use StartPaused() function instead")
	}
	return *p.startPaused
}

func (p *InstanceType) SetStateless(attr bool) {
	p.stateless = &attr
}

func (p *InstanceType) Stateless() (bool, bool) {
	if p.stateless != nil {
		return *p.stateless, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustStateless() bool {
	if p.stateless == nil {
		panic("the stateless must not be nil, please use Stateless() function instead")
	}
	return *p.stateless
}

// SetStatus Type representing a status of a virtual machine template.
func (p *InstanceType) SetStatus(attr TemplateStatus) {
	p.status = &attr
}

// Status Type representing a status of a virtual machine template.
func (p *InstanceType) Status() (TemplateStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero TemplateStatus
	return zero, false
}

// MustStatus Type representing a status of a virtual machine template.
func (p *InstanceType) MustStatus() TemplateStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *InstanceType) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *InstanceType) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *InstanceType) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

// SetStorageErrorResumeBehaviour If the storage, on which this virtual machine has some disks gets
// unresponsive, the virtual machine gets paused.
//
// This are the possible options, what should happen with the virtual machine
// in the moment the storage gets available again.
func (p *InstanceType) SetStorageErrorResumeBehaviour(attr VmStorageErrorResumeBehaviour) {
	p.storageErrorResumeBehaviour = &attr
}

// StorageErrorResumeBehaviour If the storage, on which this virtual machine has some disks gets
// unresponsive, the virtual machine gets paused.
//
// This are the possible options, what should happen with the virtual machine
// in the moment the storage gets available again.
func (p *InstanceType) StorageErrorResumeBehaviour() (VmStorageErrorResumeBehaviour, bool) {
	if p.storageErrorResumeBehaviour != nil {
		return *p.storageErrorResumeBehaviour, true
	}
	var zero VmStorageErrorResumeBehaviour
	return zero, false
}

// MustStorageErrorResumeBehaviour If the storage, on which this virtual machine has some disks gets
// unresponsive, the virtual machine gets paused.
//
// This are the possible options, what should happen with the virtual machine
// in the moment the storage gets available again.
func (p *InstanceType) MustStorageErrorResumeBehaviour() VmStorageErrorResumeBehaviour {
	if p.storageErrorResumeBehaviour == nil {
		panic("the storageErrorResumeBehaviour must not be nil, please use StorageErrorResumeBehaviour() function instead")
	}
	return *p.storageErrorResumeBehaviour
}

func (p *InstanceType) SetTags(attr *TagSlice) {
	p.tags = attr
}

func (p *InstanceType) Tags() (*TagSlice, bool) {
	if p.tags != nil {
		return p.tags, true
	}
	return nil, false
}

func (p *InstanceType) MustTags() *TagSlice {
	if p.tags == nil {
		panic("the tags must not be nil, please use Tags() function instead")
	}
	return p.tags
}

// SetTimeZone Time zone representation.
func (p *InstanceType) SetTimeZone(attr *TimeZone) {
	p.timeZone = attr
}

// TimeZone Time zone representation.
func (p *InstanceType) TimeZone() (*TimeZone, bool) {
	if p.timeZone != nil {
		return p.timeZone, true
	}
	return nil, false
}

// MustTimeZone Time zone representation.
func (p *InstanceType) MustTimeZone() *TimeZone {
	if p.timeZone == nil {
		panic("the timeZone must not be nil, please use TimeZone() function instead")
	}
	return p.timeZone
}

func (p *InstanceType) SetTpmEnabled(attr bool) {
	p.tpmEnabled = &attr
}

func (p *InstanceType) TpmEnabled() (bool, bool) {
	if p.tpmEnabled != nil {
		return *p.tpmEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustTpmEnabled() bool {
	if p.tpmEnabled == nil {
		panic("the tpmEnabled must not be nil, please use TpmEnabled() function instead")
	}
	return *p.tpmEnabled
}

func (p *InstanceType) SetTunnelMigration(attr bool) {
	p.tunnelMigration = &attr
}

func (p *InstanceType) TunnelMigration() (bool, bool) {
	if p.tunnelMigration != nil {
		return *p.tunnelMigration, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustTunnelMigration() bool {
	if p.tunnelMigration == nil {
		panic("the tunnelMigration must not be nil, please use TunnelMigration() function instead")
	}
	return *p.tunnelMigration
}

// SetType Type representing what the virtual machine is optimized for.
func (p *InstanceType) SetType(attr VmType) {
	p.type_ = &attr
}

// Type Type representing what the virtual machine is optimized for.
func (p *InstanceType) Type() (VmType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero VmType
	return zero, false
}

// MustType Type representing what the virtual machine is optimized for.
func (p *InstanceType) MustType() VmType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// SetUsb Configuration of the USB device of a virtual machine.
func (p *InstanceType) SetUsb(attr *Usb) {
	p.usb = attr
}

// Usb Configuration of the USB device of a virtual machine.
func (p *InstanceType) Usb() (*Usb, bool) {
	if p.usb != nil {
		return p.usb, true
	}
	return nil, false
}

// MustUsb Configuration of the USB device of a virtual machine.
func (p *InstanceType) MustUsb() *Usb {
	if p.usb == nil {
		panic("the usb must not be nil, please use Usb() function instead")
	}
	return p.usb
}

// SetVersion Type representing a version of a virtual machine template.
func (p *InstanceType) SetVersion(attr *TemplateVersion) {
	p.version = attr
}

// Version Type representing a version of a virtual machine template.
func (p *InstanceType) Version() (*TemplateVersion, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

// MustVersion Type representing a version of a virtual machine template.
func (p *InstanceType) MustVersion() *TemplateVersion {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// SetVirtioScsi Type representing the support of virtio-SCSI.
// If it supported we use virtio driver for SCSI guest device.
func (p *InstanceType) SetVirtioScsi(attr *VirtioScsi) {
	p.virtioScsi = attr
}

// VirtioScsi Type representing the support of virtio-SCSI.
// If it supported we use virtio driver for SCSI guest device.
func (p *InstanceType) VirtioScsi() (*VirtioScsi, bool) {
	if p.virtioScsi != nil {
		return p.virtioScsi, true
	}
	return nil, false
}

// MustVirtioScsi Type representing the support of virtio-SCSI.
// If it supported we use virtio driver for SCSI guest device.
func (p *InstanceType) MustVirtioScsi() *VirtioScsi {
	if p.virtioScsi == nil {
		panic("the virtioScsi must not be nil, please use VirtioScsi() function instead")
	}
	return p.virtioScsi
}

func (p *InstanceType) SetVirtioScsiMultiQueuesEnabled(attr bool) {
	p.virtioScsiMultiQueuesEnabled = &attr
}

func (p *InstanceType) VirtioScsiMultiQueuesEnabled() (bool, bool) {
	if p.virtioScsiMultiQueuesEnabled != nil {
		return *p.virtioScsiMultiQueuesEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *InstanceType) MustVirtioScsiMultiQueuesEnabled() bool {
	if p.virtioScsiMultiQueuesEnabled == nil {
		panic("the virtioScsiMultiQueuesEnabled must not be nil, please use VirtioScsiMultiQueuesEnabled() function instead")
	}
	return *p.virtioScsiMultiQueuesEnabled
}

// SetVm Represents a virtual machine.
func (p *InstanceType) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *InstanceType) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *InstanceType) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *InstanceType) SetWatchdogs(attr *WatchdogSlice) {
	p.watchdogs = attr
}

func (p *InstanceType) Watchdogs() (*WatchdogSlice, bool) {
	if p.watchdogs != nil {
		return p.watchdogs, true
	}
	return nil, false
}

func (p *InstanceType) MustWatchdogs() *WatchdogSlice {
	if p.watchdogs == nil {
		panic("the watchdogs must not be nil, please use Watchdogs() function instead")
	}
	return p.watchdogs
}

type Io struct {
	Struct
	threads *int64
}

func (p *Io) SetThreads(attr int64) {
	p.threads = &attr
}

func (p *Io) Threads() (int64, bool) {
	if p.threads != nil {
		return *p.threads, true
	}
	var zero int64
	return zero, false
}

func (p *Io) MustThreads() int64 {
	if p.threads == nil {
		panic("the threads must not be nil, please use Threads() function instead")
	}
	return *p.threads
}

// Ip Represents the IP configuration of a network interface.
type Ip struct {
	Struct
	address *string
	gateway *string
	netmask *string
	version *IpVersion
}

func (p *Ip) SetAddress(attr string) {
	p.address = &attr
}

func (p *Ip) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *Ip) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *Ip) SetGateway(attr string) {
	p.gateway = &attr
}

func (p *Ip) Gateway() (string, bool) {
	if p.gateway != nil {
		return *p.gateway, true
	}
	var zero string
	return zero, false
}

func (p *Ip) MustGateway() string {
	if p.gateway == nil {
		panic("the gateway must not be nil, please use Gateway() function instead")
	}
	return *p.gateway
}

func (p *Ip) SetNetmask(attr string) {
	p.netmask = &attr
}

func (p *Ip) Netmask() (string, bool) {
	if p.netmask != nil {
		return *p.netmask, true
	}
	var zero string
	return zero, false
}

func (p *Ip) MustNetmask() string {
	if p.netmask == nil {
		panic("the netmask must not be nil, please use Netmask() function instead")
	}
	return *p.netmask
}

// SetVersion Defines the values for the IP protocol version.
func (p *Ip) SetVersion(attr IpVersion) {
	p.version = &attr
}

// Version Defines the values for the IP protocol version.
func (p *Ip) Version() (IpVersion, bool) {
	if p.version != nil {
		return *p.version, true
	}
	var zero IpVersion
	return zero, false
}

// MustVersion Defines the values for the IP protocol version.
func (p *Ip) MustVersion() IpVersion {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return *p.version
}

// IpAddressAssignment Represents an IP address assignment for a network device.
//
// For a static boot protocol assignment, subnet mask and IP address
// (and optinally default gateway) must be provided in the IP configuration.
type IpAddressAssignment struct {
	Struct
	assignmentMethod *BootProtocol
	ip               *Ip
}

// SetAssignmentMethod Defines the options of the IP address assignment method to a NIC.
func (p *IpAddressAssignment) SetAssignmentMethod(attr BootProtocol) {
	p.assignmentMethod = &attr
}

// AssignmentMethod Defines the options of the IP address assignment method to a NIC.
func (p *IpAddressAssignment) AssignmentMethod() (BootProtocol, bool) {
	if p.assignmentMethod != nil {
		return *p.assignmentMethod, true
	}
	var zero BootProtocol
	return zero, false
}

// MustAssignmentMethod Defines the options of the IP address assignment method to a NIC.
func (p *IpAddressAssignment) MustAssignmentMethod() BootProtocol {
	if p.assignmentMethod == nil {
		panic("the assignmentMethod must not be nil, please use AssignmentMethod() function instead")
	}
	return *p.assignmentMethod
}

// SetIp Represents the IP configuration of a network interface.
func (p *IpAddressAssignment) SetIp(attr *Ip) {
	p.ip = attr
}

// Ip Represents the IP configuration of a network interface.
func (p *IpAddressAssignment) Ip() (*Ip, bool) {
	if p.ip != nil {
		return p.ip, true
	}
	return nil, false
}

// MustIp Represents the IP configuration of a network interface.
func (p *IpAddressAssignment) MustIp() *Ip {
	if p.ip == nil {
		panic("the ip must not be nil, please use Ip() function instead")
	}
	return p.ip
}

type IscsiBond struct {
	Struct
	comment            *string
	dataCenter         *DataCenter
	description        *string
	id                 *string
	name               *string
	networks           *NetworkSlice
	storageConnections *StorageConnectionSlice
}

func (p *IscsiBond) SetComment(attr string) {
	p.comment = &attr
}

func (p *IscsiBond) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *IscsiBond) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *IscsiBond) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *IscsiBond) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *IscsiBond) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *IscsiBond) SetDescription(attr string) {
	p.description = &attr
}

func (p *IscsiBond) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *IscsiBond) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *IscsiBond) SetId(attr string) {
	p.id = &attr
}

func (p *IscsiBond) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *IscsiBond) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *IscsiBond) SetName(attr string) {
	p.name = &attr
}

func (p *IscsiBond) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *IscsiBond) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *IscsiBond) SetNetworks(attr *NetworkSlice) {
	p.networks = attr
}

func (p *IscsiBond) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *IscsiBond) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("the networks must not be nil, please use Networks() function instead")
	}
	return p.networks
}

func (p *IscsiBond) SetStorageConnections(attr *StorageConnectionSlice) {
	p.storageConnections = attr
}

func (p *IscsiBond) StorageConnections() (*StorageConnectionSlice, bool) {
	if p.storageConnections != nil {
		return p.storageConnections, true
	}
	return nil, false
}

func (p *IscsiBond) MustStorageConnections() *StorageConnectionSlice {
	if p.storageConnections == nil {
		panic("the storageConnections must not be nil, please use StorageConnections() function instead")
	}
	return p.storageConnections
}

type IscsiDetails struct {
	Struct
	address         *string
	diskId          *string
	initiator       *string
	lunMapping      *int64
	password        *string
	paths           *int64
	port            *int64
	portal          *string
	productId       *string
	serial          *string
	size            *int64
	status          *string
	storageDomainId *string
	target          *string
	username        *string
	vendorId        *string
	volumeGroupId   *string
}

func (p *IscsiDetails) SetAddress(attr string) {
	p.address = &attr
}

func (p *IscsiDetails) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *IscsiDetails) SetDiskId(attr string) {
	p.diskId = &attr
}

func (p *IscsiDetails) DiskId() (string, bool) {
	if p.diskId != nil {
		return *p.diskId, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustDiskId() string {
	if p.diskId == nil {
		panic("the diskId must not be nil, please use DiskId() function instead")
	}
	return *p.diskId
}

func (p *IscsiDetails) SetInitiator(attr string) {
	p.initiator = &attr
}

func (p *IscsiDetails) Initiator() (string, bool) {
	if p.initiator != nil {
		return *p.initiator, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustInitiator() string {
	if p.initiator == nil {
		panic("the initiator must not be nil, please use Initiator() function instead")
	}
	return *p.initiator
}

func (p *IscsiDetails) SetLunMapping(attr int64) {
	p.lunMapping = &attr
}

func (p *IscsiDetails) LunMapping() (int64, bool) {
	if p.lunMapping != nil {
		return *p.lunMapping, true
	}
	var zero int64
	return zero, false
}

func (p *IscsiDetails) MustLunMapping() int64 {
	if p.lunMapping == nil {
		panic("the lunMapping must not be nil, please use LunMapping() function instead")
	}
	return *p.lunMapping
}

func (p *IscsiDetails) SetPassword(attr string) {
	p.password = &attr
}

func (p *IscsiDetails) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *IscsiDetails) SetPaths(attr int64) {
	p.paths = &attr
}

func (p *IscsiDetails) Paths() (int64, bool) {
	if p.paths != nil {
		return *p.paths, true
	}
	var zero int64
	return zero, false
}

func (p *IscsiDetails) MustPaths() int64 {
	if p.paths == nil {
		panic("the paths must not be nil, please use Paths() function instead")
	}
	return *p.paths
}

func (p *IscsiDetails) SetPort(attr int64) {
	p.port = &attr
}

func (p *IscsiDetails) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *IscsiDetails) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *IscsiDetails) SetPortal(attr string) {
	p.portal = &attr
}

func (p *IscsiDetails) Portal() (string, bool) {
	if p.portal != nil {
		return *p.portal, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustPortal() string {
	if p.portal == nil {
		panic("the portal must not be nil, please use Portal() function instead")
	}
	return *p.portal
}

func (p *IscsiDetails) SetProductId(attr string) {
	p.productId = &attr
}

func (p *IscsiDetails) ProductId() (string, bool) {
	if p.productId != nil {
		return *p.productId, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustProductId() string {
	if p.productId == nil {
		panic("the productId must not be nil, please use ProductId() function instead")
	}
	return *p.productId
}

func (p *IscsiDetails) SetSerial(attr string) {
	p.serial = &attr
}

func (p *IscsiDetails) Serial() (string, bool) {
	if p.serial != nil {
		return *p.serial, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustSerial() string {
	if p.serial == nil {
		panic("the serial must not be nil, please use Serial() function instead")
	}
	return *p.serial
}

func (p *IscsiDetails) SetSize(attr int64) {
	p.size = &attr
}

func (p *IscsiDetails) Size() (int64, bool) {
	if p.size != nil {
		return *p.size, true
	}
	var zero int64
	return zero, false
}

func (p *IscsiDetails) MustSize() int64 {
	if p.size == nil {
		panic("the size must not be nil, please use Size() function instead")
	}
	return *p.size
}

func (p *IscsiDetails) SetStatus(attr string) {
	p.status = &attr
}

func (p *IscsiDetails) Status() (string, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustStatus() string {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *IscsiDetails) SetStorageDomainId(attr string) {
	p.storageDomainId = &attr
}

func (p *IscsiDetails) StorageDomainId() (string, bool) {
	if p.storageDomainId != nil {
		return *p.storageDomainId, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustStorageDomainId() string {
	if p.storageDomainId == nil {
		panic("the storageDomainId must not be nil, please use StorageDomainId() function instead")
	}
	return *p.storageDomainId
}

func (p *IscsiDetails) SetTarget(attr string) {
	p.target = &attr
}

func (p *IscsiDetails) Target() (string, bool) {
	if p.target != nil {
		return *p.target, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustTarget() string {
	if p.target == nil {
		panic("the target must not be nil, please use Target() function instead")
	}
	return *p.target
}

func (p *IscsiDetails) SetUsername(attr string) {
	p.username = &attr
}

func (p *IscsiDetails) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

func (p *IscsiDetails) SetVendorId(attr string) {
	p.vendorId = &attr
}

func (p *IscsiDetails) VendorId() (string, bool) {
	if p.vendorId != nil {
		return *p.vendorId, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustVendorId() string {
	if p.vendorId == nil {
		panic("the vendorId must not be nil, please use VendorId() function instead")
	}
	return *p.vendorId
}

func (p *IscsiDetails) SetVolumeGroupId(attr string) {
	p.volumeGroupId = &attr
}

func (p *IscsiDetails) VolumeGroupId() (string, bool) {
	if p.volumeGroupId != nil {
		return *p.volumeGroupId, true
	}
	var zero string
	return zero, false
}

func (p *IscsiDetails) MustVolumeGroupId() string {
	if p.volumeGroupId == nil {
		panic("the volumeGroupId must not be nil, please use VolumeGroupId() function instead")
	}
	return *p.volumeGroupId
}

// Job Represents a job, which monitors execution of a flow in the system.
// A job can contain multiple steps in a hierarchic structure.
// The steps can be processed in parallel, depends on the implementation of the flow.
type Job struct {
	Struct
	autoCleared *bool
	comment     *string
	description *string
	endTime     *time.Time
	external    *bool
	id          *string
	lastUpdated *time.Time
	name        *string
	owner       *User
	startTime   *time.Time
	status      *JobStatus
	steps       *StepSlice
}

func (p *Job) SetAutoCleared(attr bool) {
	p.autoCleared = &attr
}

func (p *Job) AutoCleared() (bool, bool) {
	if p.autoCleared != nil {
		return *p.autoCleared, true
	}
	var zero bool
	return zero, false
}

func (p *Job) MustAutoCleared() bool {
	if p.autoCleared == nil {
		panic("the autoCleared must not be nil, please use AutoCleared() function instead")
	}
	return *p.autoCleared
}

func (p *Job) SetComment(attr string) {
	p.comment = &attr
}

func (p *Job) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Job) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Job) SetDescription(attr string) {
	p.description = &attr
}

func (p *Job) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Job) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Job) SetEndTime(attr time.Time) {
	p.endTime = &attr
}

func (p *Job) EndTime() (time.Time, bool) {
	if p.endTime != nil {
		return *p.endTime, true
	}
	var zero time.Time
	return zero, false
}

func (p *Job) MustEndTime() time.Time {
	if p.endTime == nil {
		panic("the endTime must not be nil, please use EndTime() function instead")
	}
	return *p.endTime
}

func (p *Job) SetExternal(attr bool) {
	p.external = &attr
}

func (p *Job) External() (bool, bool) {
	if p.external != nil {
		return *p.external, true
	}
	var zero bool
	return zero, false
}

func (p *Job) MustExternal() bool {
	if p.external == nil {
		panic("the external must not be nil, please use External() function instead")
	}
	return *p.external
}

func (p *Job) SetId(attr string) {
	p.id = &attr
}

func (p *Job) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Job) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Job) SetLastUpdated(attr time.Time) {
	p.lastUpdated = &attr
}

func (p *Job) LastUpdated() (time.Time, bool) {
	if p.lastUpdated != nil {
		return *p.lastUpdated, true
	}
	var zero time.Time
	return zero, false
}

func (p *Job) MustLastUpdated() time.Time {
	if p.lastUpdated == nil {
		panic("the lastUpdated must not be nil, please use LastUpdated() function instead")
	}
	return *p.lastUpdated
}

func (p *Job) SetName(attr string) {
	p.name = &attr
}

func (p *Job) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Job) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetOwner Represents a user in the system.
func (p *Job) SetOwner(attr *User) {
	p.owner = attr
}

// Owner Represents a user in the system.
func (p *Job) Owner() (*User, bool) {
	if p.owner != nil {
		return p.owner, true
	}
	return nil, false
}

// MustOwner Represents a user in the system.
func (p *Job) MustOwner() *User {
	if p.owner == nil {
		panic("the owner must not be nil, please use Owner() function instead")
	}
	return p.owner
}

func (p *Job) SetStartTime(attr time.Time) {
	p.startTime = &attr
}

func (p *Job) StartTime() (time.Time, bool) {
	if p.startTime != nil {
		return *p.startTime, true
	}
	var zero time.Time
	return zero, false
}

func (p *Job) MustStartTime() time.Time {
	if p.startTime == nil {
		panic("the startTime must not be nil, please use StartTime() function instead")
	}
	return *p.startTime
}

// SetStatus Represents the status of the job.
func (p *Job) SetStatus(attr JobStatus) {
	p.status = &attr
}

// Status Represents the status of the job.
func (p *Job) Status() (JobStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero JobStatus
	return zero, false
}

// MustStatus Represents the status of the job.
func (p *Job) MustStatus() JobStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *Job) SetSteps(attr *StepSlice) {
	p.steps = attr
}

func (p *Job) Steps() (*StepSlice, bool) {
	if p.steps != nil {
		return p.steps, true
	}
	return nil, false
}

func (p *Job) MustSteps() *StepSlice {
	if p.steps == nil {
		panic("the steps must not be nil, please use Steps() function instead")
	}
	return p.steps
}

// KatelloErratum Type representing a Katello erratum.
type KatelloErratum struct {
	Struct
	comment     *string
	description *string
	host        *Host
	id          *string
	issued      *time.Time
	name        *string
	packages    *PackageSlice
	severity    *string
	solution    *string
	summary     *string
	title       *string
	type_       *string
	vm          *Vm
}

func (p *KatelloErratum) SetComment(attr string) {
	p.comment = &attr
}

func (p *KatelloErratum) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *KatelloErratum) SetDescription(attr string) {
	p.description = &attr
}

func (p *KatelloErratum) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetHost Type representing a host.
func (p *KatelloErratum) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *KatelloErratum) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *KatelloErratum) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *KatelloErratum) SetId(attr string) {
	p.id = &attr
}

func (p *KatelloErratum) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *KatelloErratum) SetIssued(attr time.Time) {
	p.issued = &attr
}

func (p *KatelloErratum) Issued() (time.Time, bool) {
	if p.issued != nil {
		return *p.issued, true
	}
	var zero time.Time
	return zero, false
}

func (p *KatelloErratum) MustIssued() time.Time {
	if p.issued == nil {
		panic("the issued must not be nil, please use Issued() function instead")
	}
	return *p.issued
}

func (p *KatelloErratum) SetName(attr string) {
	p.name = &attr
}

func (p *KatelloErratum) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *KatelloErratum) SetPackages(attr *PackageSlice) {
	p.packages = attr
}

func (p *KatelloErratum) Packages() (*PackageSlice, bool) {
	if p.packages != nil {
		return p.packages, true
	}
	return nil, false
}

func (p *KatelloErratum) MustPackages() *PackageSlice {
	if p.packages == nil {
		panic("the packages must not be nil, please use Packages() function instead")
	}
	return p.packages
}

func (p *KatelloErratum) SetSeverity(attr string) {
	p.severity = &attr
}

func (p *KatelloErratum) Severity() (string, bool) {
	if p.severity != nil {
		return *p.severity, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustSeverity() string {
	if p.severity == nil {
		panic("the severity must not be nil, please use Severity() function instead")
	}
	return *p.severity
}

func (p *KatelloErratum) SetSolution(attr string) {
	p.solution = &attr
}

func (p *KatelloErratum) Solution() (string, bool) {
	if p.solution != nil {
		return *p.solution, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustSolution() string {
	if p.solution == nil {
		panic("the solution must not be nil, please use Solution() function instead")
	}
	return *p.solution
}

func (p *KatelloErratum) SetSummary(attr string) {
	p.summary = &attr
}

func (p *KatelloErratum) Summary() (string, bool) {
	if p.summary != nil {
		return *p.summary, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustSummary() string {
	if p.summary == nil {
		panic("the summary must not be nil, please use Summary() function instead")
	}
	return *p.summary
}

func (p *KatelloErratum) SetTitle(attr string) {
	p.title = &attr
}

func (p *KatelloErratum) Title() (string, bool) {
	if p.title != nil {
		return *p.title, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustTitle() string {
	if p.title == nil {
		panic("the title must not be nil, please use Title() function instead")
	}
	return *p.title
}

func (p *KatelloErratum) SetType(attr string) {
	p.type_ = &attr
}

func (p *KatelloErratum) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *KatelloErratum) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// SetVm Represents a virtual machine.
func (p *KatelloErratum) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *KatelloErratum) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *KatelloErratum) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

type Kernel struct {
	Struct
	version *Version
}

func (p *Kernel) SetVersion(attr *Version) {
	p.version = attr
}

func (p *Kernel) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *Kernel) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

type Ksm struct {
	Struct
	enabled          *bool
	mergeAcrossNodes *bool
}

func (p *Ksm) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *Ksm) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *Ksm) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

func (p *Ksm) SetMergeAcrossNodes(attr bool) {
	p.mergeAcrossNodes = &attr
}

func (p *Ksm) MergeAcrossNodes() (bool, bool) {
	if p.mergeAcrossNodes != nil {
		return *p.mergeAcrossNodes, true
	}
	var zero bool
	return zero, false
}

func (p *Ksm) MustMergeAcrossNodes() bool {
	if p.mergeAcrossNodes == nil {
		panic("the mergeAcrossNodes must not be nil, please use MergeAcrossNodes() function instead")
	}
	return *p.mergeAcrossNodes
}

// LinkLayerDiscoveryProtocolElement Represents an information element received by Link Layer Discovery Protocol (LLDP).
// IEEE 802.1AB defines type, length, value (TLV) as a "short, variable length encoding of an information element".
// This type represents such an information element.
//
// The attribute `name` is a human-readable string used to describe what the value is about, and may not be unique.
// The name is redundant, because it could be created from `type` and the optional `oui` and `subtype`.
// The purpose of `name` is to simplify the reading of the information element.
// The `name` of a property is exactly the same string which is used in IEEE 802.1AB chapter 8.
//
// Organizationally-specific information elements have the `type` of `127` and the attributes
// `oui` and `subtype`.
//
// For example, the XML representation of an information element may look like this:
//
// [source,xml]
// ----
// <link_layer_discovery_protocol_element>
//   <name>Port VLAN Id</name>
//   <oui>32962</oui>
//   <properties>
//     <property>
//       <name>vlan id</name>
//       <value>488</value>
//     </property>
//     <property>
//       <name>vlan name</name>
//       <value>v2-0488-03-0505</value>
//     </property>
//   </properties>
//   <subtype>3</subtype>
//   <type>127</type>
// </link_layer_discovery_protocol_element>
// ----
type LinkLayerDiscoveryProtocolElement struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
	oui         *int64
	properties  *PropertySlice
	subtype     *int64
	type_       *int64
}

func (p *LinkLayerDiscoveryProtocolElement) SetComment(attr string) {
	p.comment = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *LinkLayerDiscoveryProtocolElement) SetDescription(attr string) {
	p.description = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *LinkLayerDiscoveryProtocolElement) SetId(attr string) {
	p.id = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *LinkLayerDiscoveryProtocolElement) SetName(attr string) {
	p.name = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *LinkLayerDiscoveryProtocolElement) SetOui(attr int64) {
	p.oui = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Oui() (int64, bool) {
	if p.oui != nil {
		return *p.oui, true
	}
	var zero int64
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustOui() int64 {
	if p.oui == nil {
		panic("the oui must not be nil, please use Oui() function instead")
	}
	return *p.oui
}

func (p *LinkLayerDiscoveryProtocolElement) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *LinkLayerDiscoveryProtocolElement) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *LinkLayerDiscoveryProtocolElement) SetSubtype(attr int64) {
	p.subtype = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Subtype() (int64, bool) {
	if p.subtype != nil {
		return *p.subtype, true
	}
	var zero int64
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustSubtype() int64 {
	if p.subtype == nil {
		panic("the subtype must not be nil, please use Subtype() function instead")
	}
	return *p.subtype
}

func (p *LinkLayerDiscoveryProtocolElement) SetType(attr int64) {
	p.type_ = &attr
}

func (p *LinkLayerDiscoveryProtocolElement) Type() (int64, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero int64
	return zero, false
}

func (p *LinkLayerDiscoveryProtocolElement) MustType() int64 {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type LogicalUnit struct {
	Struct
	address           *string
	discardMaxSize    *int64
	discardZeroesData *bool
	diskId            *string
	id                *string
	lunMapping        *int64
	password          *string
	paths             *int64
	port              *int64
	portal            *string
	productId         *string
	serial            *string
	size              *int64
	status            *LunStatus
	storageDomainId   *string
	target            *string
	username          *string
	vendorId          *string
	volumeGroupId     *string
}

func (p *LogicalUnit) SetAddress(attr string) {
	p.address = &attr
}

func (p *LogicalUnit) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *LogicalUnit) SetDiscardMaxSize(attr int64) {
	p.discardMaxSize = &attr
}

func (p *LogicalUnit) DiscardMaxSize() (int64, bool) {
	if p.discardMaxSize != nil {
		return *p.discardMaxSize, true
	}
	var zero int64
	return zero, false
}

func (p *LogicalUnit) MustDiscardMaxSize() int64 {
	if p.discardMaxSize == nil {
		panic("the discardMaxSize must not be nil, please use DiscardMaxSize() function instead")
	}
	return *p.discardMaxSize
}

func (p *LogicalUnit) SetDiscardZeroesData(attr bool) {
	p.discardZeroesData = &attr
}

func (p *LogicalUnit) DiscardZeroesData() (bool, bool) {
	if p.discardZeroesData != nil {
		return *p.discardZeroesData, true
	}
	var zero bool
	return zero, false
}

func (p *LogicalUnit) MustDiscardZeroesData() bool {
	if p.discardZeroesData == nil {
		panic("the discardZeroesData must not be nil, please use DiscardZeroesData() function instead")
	}
	return *p.discardZeroesData
}

func (p *LogicalUnit) SetDiskId(attr string) {
	p.diskId = &attr
}

func (p *LogicalUnit) DiskId() (string, bool) {
	if p.diskId != nil {
		return *p.diskId, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustDiskId() string {
	if p.diskId == nil {
		panic("the diskId must not be nil, please use DiskId() function instead")
	}
	return *p.diskId
}

func (p *LogicalUnit) SetId(attr string) {
	p.id = &attr
}

func (p *LogicalUnit) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *LogicalUnit) SetLunMapping(attr int64) {
	p.lunMapping = &attr
}

func (p *LogicalUnit) LunMapping() (int64, bool) {
	if p.lunMapping != nil {
		return *p.lunMapping, true
	}
	var zero int64
	return zero, false
}

func (p *LogicalUnit) MustLunMapping() int64 {
	if p.lunMapping == nil {
		panic("the lunMapping must not be nil, please use LunMapping() function instead")
	}
	return *p.lunMapping
}

func (p *LogicalUnit) SetPassword(attr string) {
	p.password = &attr
}

func (p *LogicalUnit) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *LogicalUnit) SetPaths(attr int64) {
	p.paths = &attr
}

func (p *LogicalUnit) Paths() (int64, bool) {
	if p.paths != nil {
		return *p.paths, true
	}
	var zero int64
	return zero, false
}

func (p *LogicalUnit) MustPaths() int64 {
	if p.paths == nil {
		panic("the paths must not be nil, please use Paths() function instead")
	}
	return *p.paths
}

func (p *LogicalUnit) SetPort(attr int64) {
	p.port = &attr
}

func (p *LogicalUnit) Port() (int64, bool) {
	if p.port != nil {
		return *p.port, true
	}
	var zero int64
	return zero, false
}

func (p *LogicalUnit) MustPort() int64 {
	if p.port == nil {
		panic("the port must not be nil, please use Port() function instead")
	}
	return *p.port
}

func (p *LogicalUnit) SetPortal(attr string) {
	p.portal = &attr
}

func (p *LogicalUnit) Portal() (string, bool) {
	if p.portal != nil {
		return *p.portal, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustPortal() string {
	if p.portal == nil {
		panic("the portal must not be nil, please use Portal() function instead")
	}
	return *p.portal
}

func (p *LogicalUnit) SetProductId(attr string) {
	p.productId = &attr
}

func (p *LogicalUnit) ProductId() (string, bool) {
	if p.productId != nil {
		return *p.productId, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustProductId() string {
	if p.productId == nil {
		panic("the productId must not be nil, please use ProductId() function instead")
	}
	return *p.productId
}

func (p *LogicalUnit) SetSerial(attr string) {
	p.serial = &attr
}

func (p *LogicalUnit) Serial() (string, bool) {
	if p.serial != nil {
		return *p.serial, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustSerial() string {
	if p.serial == nil {
		panic("the serial must not be nil, please use Serial() function instead")
	}
	return *p.serial
}

func (p *LogicalUnit) SetSize(attr int64) {
	p.size = &attr
}

func (p *LogicalUnit) Size() (int64, bool) {
	if p.size != nil {
		return *p.size, true
	}
	var zero int64
	return zero, false
}

func (p *LogicalUnit) MustSize() int64 {
	if p.size == nil {
		panic("the size must not be nil, please use Size() function instead")
	}
	return *p.size
}

func (p *LogicalUnit) SetStatus(attr LunStatus) {
	p.status = &attr
}

func (p *LogicalUnit) Status() (LunStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero LunStatus
	return zero, false
}

func (p *LogicalUnit) MustStatus() LunStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *LogicalUnit) SetStorageDomainId(attr string) {
	p.storageDomainId = &attr
}

func (p *LogicalUnit) StorageDomainId() (string, bool) {
	if p.storageDomainId != nil {
		return *p.storageDomainId, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustStorageDomainId() string {
	if p.storageDomainId == nil {
		panic("the storageDomainId must not be nil, please use StorageDomainId() function instead")
	}
	return *p.storageDomainId
}

func (p *LogicalUnit) SetTarget(attr string) {
	p.target = &attr
}

func (p *LogicalUnit) Target() (string, bool) {
	if p.target != nil {
		return *p.target, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustTarget() string {
	if p.target == nil {
		panic("the target must not be nil, please use Target() function instead")
	}
	return *p.target
}

func (p *LogicalUnit) SetUsername(attr string) {
	p.username = &attr
}

func (p *LogicalUnit) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

func (p *LogicalUnit) SetVendorId(attr string) {
	p.vendorId = &attr
}

func (p *LogicalUnit) VendorId() (string, bool) {
	if p.vendorId != nil {
		return *p.vendorId, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustVendorId() string {
	if p.vendorId == nil {
		panic("the vendorId must not be nil, please use VendorId() function instead")
	}
	return *p.vendorId
}

func (p *LogicalUnit) SetVolumeGroupId(attr string) {
	p.volumeGroupId = &attr
}

func (p *LogicalUnit) VolumeGroupId() (string, bool) {
	if p.volumeGroupId != nil {
		return *p.volumeGroupId, true
	}
	var zero string
	return zero, false
}

func (p *LogicalUnit) MustVolumeGroupId() string {
	if p.volumeGroupId == nil {
		panic("the volumeGroupId must not be nil, please use VolumeGroupId() function instead")
	}
	return *p.volumeGroupId
}

// MDevType Mediated device is a software device that allows to divide physical device's resources.
//
// See https://libvirt.org/drvnodedev.html#MDEV[Libvirt-MDEV] for further details.
type MDevType struct {
	Struct
	availableInstances *int64
	description        *string
	humanReadableName  *string
	name               *string
}

func (p *MDevType) SetAvailableInstances(attr int64) {
	p.availableInstances = &attr
}

func (p *MDevType) AvailableInstances() (int64, bool) {
	if p.availableInstances != nil {
		return *p.availableInstances, true
	}
	var zero int64
	return zero, false
}

func (p *MDevType) MustAvailableInstances() int64 {
	if p.availableInstances == nil {
		panic("the availableInstances must not be nil, please use AvailableInstances() function instead")
	}
	return *p.availableInstances
}

func (p *MDevType) SetDescription(attr string) {
	p.description = &attr
}

func (p *MDevType) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *MDevType) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *MDevType) SetHumanReadableName(attr string) {
	p.humanReadableName = &attr
}

func (p *MDevType) HumanReadableName() (string, bool) {
	if p.humanReadableName != nil {
		return *p.humanReadableName, true
	}
	var zero string
	return zero, false
}

func (p *MDevType) MustHumanReadableName() string {
	if p.humanReadableName == nil {
		panic("the humanReadableName must not be nil, please use HumanReadableName() function instead")
	}
	return *p.humanReadableName
}

func (p *MDevType) SetName(attr string) {
	p.name = &attr
}

func (p *MDevType) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *MDevType) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// Mac Represents a MAC address of a virtual network interface.
type Mac struct {
	Struct
	address *string
}

func (p *Mac) SetAddress(attr string) {
	p.address = &attr
}

func (p *Mac) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *Mac) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

// MacPool Represents a MAC address pool.
//
// Example of an XML representation of a MAC address pool:
//
// [source,xml]
// ----
// <mac_pool href="/ovirt-engine/api/macpools/123" id="123">
//   <name>Default</name>
//   <description>Default MAC pool</description>
//   <allow_duplicates>false</allow_duplicates>
//   <default_pool>true</default_pool>
//   <ranges>
//     <range>
//       <from>00:1A:4A:16:01:51</from>
//       <to>00:1A:4A:16:01:E6</to>
//     </range>
//   </ranges>
// </mac_pool>
// ----
type MacPool struct {
	Struct
	allowDuplicates *bool
	comment         *string
	defaultPool     *bool
	description     *string
	id              *string
	name            *string
	permissions     *PermissionSlice
	ranges          *RangeSlice
}

func (p *MacPool) SetAllowDuplicates(attr bool) {
	p.allowDuplicates = &attr
}

func (p *MacPool) AllowDuplicates() (bool, bool) {
	if p.allowDuplicates != nil {
		return *p.allowDuplicates, true
	}
	var zero bool
	return zero, false
}

func (p *MacPool) MustAllowDuplicates() bool {
	if p.allowDuplicates == nil {
		panic("the allowDuplicates must not be nil, please use AllowDuplicates() function instead")
	}
	return *p.allowDuplicates
}

func (p *MacPool) SetComment(attr string) {
	p.comment = &attr
}

func (p *MacPool) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *MacPool) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *MacPool) SetDefaultPool(attr bool) {
	p.defaultPool = &attr
}

func (p *MacPool) DefaultPool() (bool, bool) {
	if p.defaultPool != nil {
		return *p.defaultPool, true
	}
	var zero bool
	return zero, false
}

func (p *MacPool) MustDefaultPool() bool {
	if p.defaultPool == nil {
		panic("the defaultPool must not be nil, please use DefaultPool() function instead")
	}
	return *p.defaultPool
}

func (p *MacPool) SetDescription(attr string) {
	p.description = &attr
}

func (p *MacPool) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *MacPool) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *MacPool) SetId(attr string) {
	p.id = &attr
}

func (p *MacPool) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *MacPool) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *MacPool) SetName(attr string) {
	p.name = &attr
}

func (p *MacPool) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *MacPool) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *MacPool) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *MacPool) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *MacPool) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *MacPool) SetRanges(attr *RangeSlice) {
	p.ranges = attr
}

func (p *MacPool) Ranges() (*RangeSlice, bool) {
	if p.ranges != nil {
		return p.ranges, true
	}
	return nil, false
}

func (p *MacPool) MustRanges() *RangeSlice {
	if p.ranges == nil {
		panic("the ranges must not be nil, please use Ranges() function instead")
	}
	return p.ranges
}

type MemoryOverCommit struct {
	Struct
	percent *int64
}

func (p *MemoryOverCommit) SetPercent(attr int64) {
	p.percent = &attr
}

func (p *MemoryOverCommit) Percent() (int64, bool) {
	if p.percent != nil {
		return *p.percent, true
	}
	var zero int64
	return zero, false
}

func (p *MemoryOverCommit) MustPercent() int64 {
	if p.percent == nil {
		panic("the percent must not be nil, please use Percent() function instead")
	}
	return *p.percent
}

// MemoryPolicy Logical grouping of memory-related properties of virtual machine-like entities.
type MemoryPolicy struct {
	Struct
	ballooning           *bool
	guaranteed           *int64
	max                  *int64
	overCommit           *MemoryOverCommit
	transparentHugePages *TransparentHugePages
}

func (p *MemoryPolicy) SetBallooning(attr bool) {
	p.ballooning = &attr
}

func (p *MemoryPolicy) Ballooning() (bool, bool) {
	if p.ballooning != nil {
		return *p.ballooning, true
	}
	var zero bool
	return zero, false
}

func (p *MemoryPolicy) MustBallooning() bool {
	if p.ballooning == nil {
		panic("the ballooning must not be nil, please use Ballooning() function instead")
	}
	return *p.ballooning
}

func (p *MemoryPolicy) SetGuaranteed(attr int64) {
	p.guaranteed = &attr
}

func (p *MemoryPolicy) Guaranteed() (int64, bool) {
	if p.guaranteed != nil {
		return *p.guaranteed, true
	}
	var zero int64
	return zero, false
}

func (p *MemoryPolicy) MustGuaranteed() int64 {
	if p.guaranteed == nil {
		panic("the guaranteed must not be nil, please use Guaranteed() function instead")
	}
	return *p.guaranteed
}

func (p *MemoryPolicy) SetMax(attr int64) {
	p.max = &attr
}

func (p *MemoryPolicy) Max() (int64, bool) {
	if p.max != nil {
		return *p.max, true
	}
	var zero int64
	return zero, false
}

func (p *MemoryPolicy) MustMax() int64 {
	if p.max == nil {
		panic("the max must not be nil, please use Max() function instead")
	}
	return *p.max
}

func (p *MemoryPolicy) SetOverCommit(attr *MemoryOverCommit) {
	p.overCommit = attr
}

func (p *MemoryPolicy) OverCommit() (*MemoryOverCommit, bool) {
	if p.overCommit != nil {
		return p.overCommit, true
	}
	return nil, false
}

func (p *MemoryPolicy) MustOverCommit() *MemoryOverCommit {
	if p.overCommit == nil {
		panic("the overCommit must not be nil, please use OverCommit() function instead")
	}
	return p.overCommit
}

// SetTransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *MemoryPolicy) SetTransparentHugePages(attr *TransparentHugePages) {
	p.transparentHugePages = attr
}

// TransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *MemoryPolicy) TransparentHugePages() (*TransparentHugePages, bool) {
	if p.transparentHugePages != nil {
		return p.transparentHugePages, true
	}
	return nil, false
}

// MustTransparentHugePages Type representing a transparent huge pages (THP) support.
func (p *MemoryPolicy) MustTransparentHugePages() *TransparentHugePages {
	if p.transparentHugePages == nil {
		panic("the transparentHugePages must not be nil, please use TransparentHugePages() function instead")
	}
	return p.transparentHugePages
}

type Method struct {
	Struct
	id *SsoMethod
}

func (p *Method) SetId(attr SsoMethod) {
	p.id = &attr
}

func (p *Method) Id() (SsoMethod, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero SsoMethod
	return zero, false
}

func (p *Method) MustId() SsoMethod {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// MigrationBandwidth Defines the bandwidth used by migration.
type MigrationBandwidth struct {
	Struct
	assignmentMethod *MigrationBandwidthAssignmentMethod
	customValue      *int64
}

// SetAssignmentMethod Defines how the migration bandwidth is assigned.
func (p *MigrationBandwidth) SetAssignmentMethod(attr MigrationBandwidthAssignmentMethod) {
	p.assignmentMethod = &attr
}

// AssignmentMethod Defines how the migration bandwidth is assigned.
func (p *MigrationBandwidth) AssignmentMethod() (MigrationBandwidthAssignmentMethod, bool) {
	if p.assignmentMethod != nil {
		return *p.assignmentMethod, true
	}
	var zero MigrationBandwidthAssignmentMethod
	return zero, false
}

// MustAssignmentMethod Defines how the migration bandwidth is assigned.
func (p *MigrationBandwidth) MustAssignmentMethod() MigrationBandwidthAssignmentMethod {
	if p.assignmentMethod == nil {
		panic("the assignmentMethod must not be nil, please use AssignmentMethod() function instead")
	}
	return *p.assignmentMethod
}

func (p *MigrationBandwidth) SetCustomValue(attr int64) {
	p.customValue = &attr
}

func (p *MigrationBandwidth) CustomValue() (int64, bool) {
	if p.customValue != nil {
		return *p.customValue, true
	}
	var zero int64
	return zero, false
}

func (p *MigrationBandwidth) MustCustomValue() int64 {
	if p.customValue == nil {
		panic("the customValue must not be nil, please use CustomValue() function instead")
	}
	return *p.customValue
}

// MigrationOptions The type for migration options.
type MigrationOptions struct {
	Struct
	autoConverge *InheritableBoolean
	bandwidth    *MigrationBandwidth
	compressed   *InheritableBoolean
	encrypted    *InheritableBoolean
	policy       *MigrationPolicy
}

// SetAutoConverge Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) SetAutoConverge(attr InheritableBoolean) {
	p.autoConverge = &attr
}

// AutoConverge Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) AutoConverge() (InheritableBoolean, bool) {
	if p.autoConverge != nil {
		return *p.autoConverge, true
	}
	var zero InheritableBoolean
	return zero, false
}

// MustAutoConverge Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) MustAutoConverge() InheritableBoolean {
	if p.autoConverge == nil {
		panic("the autoConverge must not be nil, please use AutoConverge() function instead")
	}
	return *p.autoConverge
}

// SetBandwidth Defines the bandwidth used by migration.
func (p *MigrationOptions) SetBandwidth(attr *MigrationBandwidth) {
	p.bandwidth = attr
}

// Bandwidth Defines the bandwidth used by migration.
func (p *MigrationOptions) Bandwidth() (*MigrationBandwidth, bool) {
	if p.bandwidth != nil {
		return p.bandwidth, true
	}
	return nil, false
}

// MustBandwidth Defines the bandwidth used by migration.
func (p *MigrationOptions) MustBandwidth() *MigrationBandwidth {
	if p.bandwidth == nil {
		panic("the bandwidth must not be nil, please use Bandwidth() function instead")
	}
	return p.bandwidth
}

// SetCompressed Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) SetCompressed(attr InheritableBoolean) {
	p.compressed = &attr
}

// Compressed Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) Compressed() (InheritableBoolean, bool) {
	if p.compressed != nil {
		return *p.compressed, true
	}
	var zero InheritableBoolean
	return zero, false
}

// MustCompressed Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) MustCompressed() InheritableBoolean {
	if p.compressed == nil {
		panic("the compressed must not be nil, please use Compressed() function instead")
	}
	return *p.compressed
}

// SetEncrypted Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) SetEncrypted(attr InheritableBoolean) {
	p.encrypted = &attr
}

// Encrypted Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) Encrypted() (InheritableBoolean, bool) {
	if p.encrypted != nil {
		return *p.encrypted, true
	}
	var zero InheritableBoolean
	return zero, false
}

// MustEncrypted Enum representing the boolean value that can be either set, or inherited from a higher level.
// The inheritance order is virtual machine -> cluster -> engine-config.
func (p *MigrationOptions) MustEncrypted() InheritableBoolean {
	if p.encrypted == nil {
		panic("the encrypted must not be nil, please use Encrypted() function instead")
	}
	return *p.encrypted
}

// SetPolicy A policy describing how the migration is treated, such as convergence or
// how many parallel migrations are allowed.
func (p *MigrationOptions) SetPolicy(attr *MigrationPolicy) {
	p.policy = attr
}

// Policy A policy describing how the migration is treated, such as convergence or
// how many parallel migrations are allowed.
func (p *MigrationOptions) Policy() (*MigrationPolicy, bool) {
	if p.policy != nil {
		return p.policy, true
	}
	return nil, false
}

// MustPolicy A policy describing how the migration is treated, such as convergence or
// how many parallel migrations are allowed.
func (p *MigrationOptions) MustPolicy() *MigrationPolicy {
	if p.policy == nil {
		panic("the policy must not be nil, please use Policy() function instead")
	}
	return p.policy
}

// MigrationPolicy A policy describing how the migration is treated, such as convergence or
// how many parallel migrations are allowed.
type MigrationPolicy struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
}

func (p *MigrationPolicy) SetComment(attr string) {
	p.comment = &attr
}

func (p *MigrationPolicy) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *MigrationPolicy) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *MigrationPolicy) SetDescription(attr string) {
	p.description = &attr
}

func (p *MigrationPolicy) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *MigrationPolicy) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *MigrationPolicy) SetId(attr string) {
	p.id = &attr
}

func (p *MigrationPolicy) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *MigrationPolicy) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *MigrationPolicy) SetName(attr string) {
	p.name = &attr
}

func (p *MigrationPolicy) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *MigrationPolicy) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// Network The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
type Network struct {
	Struct
	cluster                         *Cluster
	comment                         *string
	dataCenter                      *DataCenter
	description                     *string
	display                         *bool
	dnsResolverConfiguration        *DnsResolverConfiguration
	externalProvider                *OpenStackNetworkProvider
	externalProviderPhysicalNetwork *Network
	id                              *string
	ip                              *Ip
	mtu                             *int64
	name                            *string
	networkLabels                   *NetworkLabelSlice
	permissions                     *PermissionSlice
	portIsolation                   *bool
	profileRequired                 *bool
	qos                             *Qos
	required                        *bool
	status                          *NetworkStatus
	stp                             *bool
	usages                          []NetworkUsage
	vdsmName                        *string
	vlan                            *Vlan
	vnicProfiles                    *VnicProfileSlice
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Network) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Network) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Network) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *Network) SetComment(attr string) {
	p.comment = &attr
}

func (p *Network) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Network) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Network) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Network) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Network) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Network) SetDescription(attr string) {
	p.description = &attr
}

func (p *Network) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Network) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Network) SetDisplay(attr bool) {
	p.display = &attr
}

func (p *Network) Display() (bool, bool) {
	if p.display != nil {
		return *p.display, true
	}
	var zero bool
	return zero, false
}

func (p *Network) MustDisplay() bool {
	if p.display == nil {
		panic("the display must not be nil, please use Display() function instead")
	}
	return *p.display
}

// SetDnsResolverConfiguration Represents the DNS resolver configuration.
func (p *Network) SetDnsResolverConfiguration(attr *DnsResolverConfiguration) {
	p.dnsResolverConfiguration = attr
}

// DnsResolverConfiguration Represents the DNS resolver configuration.
func (p *Network) DnsResolverConfiguration() (*DnsResolverConfiguration, bool) {
	if p.dnsResolverConfiguration != nil {
		return p.dnsResolverConfiguration, true
	}
	return nil, false
}

// MustDnsResolverConfiguration Represents the DNS resolver configuration.
func (p *Network) MustDnsResolverConfiguration() *DnsResolverConfiguration {
	if p.dnsResolverConfiguration == nil {
		panic("the dnsResolverConfiguration must not be nil, please use DnsResolverConfiguration() function instead")
	}
	return p.dnsResolverConfiguration
}

func (p *Network) SetExternalProvider(attr *OpenStackNetworkProvider) {
	p.externalProvider = attr
}

func (p *Network) ExternalProvider() (*OpenStackNetworkProvider, bool) {
	if p.externalProvider != nil {
		return p.externalProvider, true
	}
	return nil, false
}

func (p *Network) MustExternalProvider() *OpenStackNetworkProvider {
	if p.externalProvider == nil {
		panic("the externalProvider must not be nil, please use ExternalProvider() function instead")
	}
	return p.externalProvider
}

// SetExternalProviderPhysicalNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Network) SetExternalProviderPhysicalNetwork(attr *Network) {
	p.externalProviderPhysicalNetwork = attr
}

// ExternalProviderPhysicalNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Network) ExternalProviderPhysicalNetwork() (*Network, bool) {
	if p.externalProviderPhysicalNetwork != nil {
		return p.externalProviderPhysicalNetwork, true
	}
	return nil, false
}

// MustExternalProviderPhysicalNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Network) MustExternalProviderPhysicalNetwork() *Network {
	if p.externalProviderPhysicalNetwork == nil {
		panic("the externalProviderPhysicalNetwork must not be nil, please use ExternalProviderPhysicalNetwork() function instead")
	}
	return p.externalProviderPhysicalNetwork
}

func (p *Network) SetId(attr string) {
	p.id = &attr
}

func (p *Network) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Network) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetIp Represents the IP configuration of a network interface.
func (p *Network) SetIp(attr *Ip) {
	p.ip = attr
}

// Ip Represents the IP configuration of a network interface.
func (p *Network) Ip() (*Ip, bool) {
	if p.ip != nil {
		return p.ip, true
	}
	return nil, false
}

// MustIp Represents the IP configuration of a network interface.
func (p *Network) MustIp() *Ip {
	if p.ip == nil {
		panic("the ip must not be nil, please use Ip() function instead")
	}
	return p.ip
}

func (p *Network) SetMtu(attr int64) {
	p.mtu = &attr
}

func (p *Network) Mtu() (int64, bool) {
	if p.mtu != nil {
		return *p.mtu, true
	}
	var zero int64
	return zero, false
}

func (p *Network) MustMtu() int64 {
	if p.mtu == nil {
		panic("the mtu must not be nil, please use Mtu() function instead")
	}
	return *p.mtu
}

func (p *Network) SetName(attr string) {
	p.name = &attr
}

func (p *Network) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Network) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Network) SetNetworkLabels(attr *NetworkLabelSlice) {
	p.networkLabels = attr
}

func (p *Network) NetworkLabels() (*NetworkLabelSlice, bool) {
	if p.networkLabels != nil {
		return p.networkLabels, true
	}
	return nil, false
}

func (p *Network) MustNetworkLabels() *NetworkLabelSlice {
	if p.networkLabels == nil {
		panic("the networkLabels must not be nil, please use NetworkLabels() function instead")
	}
	return p.networkLabels
}

func (p *Network) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Network) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Network) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Network) SetPortIsolation(attr bool) {
	p.portIsolation = &attr
}

func (p *Network) PortIsolation() (bool, bool) {
	if p.portIsolation != nil {
		return *p.portIsolation, true
	}
	var zero bool
	return zero, false
}

func (p *Network) MustPortIsolation() bool {
	if p.portIsolation == nil {
		panic("the portIsolation must not be nil, please use PortIsolation() function instead")
	}
	return *p.portIsolation
}

func (p *Network) SetProfileRequired(attr bool) {
	p.profileRequired = &attr
}

func (p *Network) ProfileRequired() (bool, bool) {
	if p.profileRequired != nil {
		return *p.profileRequired, true
	}
	var zero bool
	return zero, false
}

func (p *Network) MustProfileRequired() bool {
	if p.profileRequired == nil {
		panic("the profileRequired must not be nil, please use ProfileRequired() function instead")
	}
	return *p.profileRequired
}

// SetQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *Network) SetQos(attr *Qos) {
	p.qos = attr
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *Network) Qos() (*Qos, bool) {
	if p.qos != nil {
		return p.qos, true
	}
	return nil, false
}

// MustQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *Network) MustQos() *Qos {
	if p.qos == nil {
		panic("the qos must not be nil, please use Qos() function instead")
	}
	return p.qos
}

func (p *Network) SetRequired(attr bool) {
	p.required = &attr
}

func (p *Network) Required() (bool, bool) {
	if p.required != nil {
		return *p.required, true
	}
	var zero bool
	return zero, false
}

func (p *Network) MustRequired() bool {
	if p.required == nil {
		panic("the required must not be nil, please use Required() function instead")
	}
	return *p.required
}

func (p *Network) SetStatus(attr NetworkStatus) {
	p.status = &attr
}

func (p *Network) Status() (NetworkStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero NetworkStatus
	return zero, false
}

func (p *Network) MustStatus() NetworkStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *Network) SetStp(attr bool) {
	p.stp = &attr
}

func (p *Network) Stp() (bool, bool) {
	if p.stp != nil {
		return *p.stp, true
	}
	var zero bool
	return zero, false
}

func (p *Network) MustStp() bool {
	if p.stp == nil {
		panic("the stp must not be nil, please use Stp() function instead")
	}
	return *p.stp
}

func (p *Network) SetUsages(attr []NetworkUsage) {
	p.usages = attr
}

func (p *Network) Usages() ([]NetworkUsage, bool) {
	if p.usages != nil {
		return p.usages, true
	}
	return nil, false
}

func (p *Network) MustUsages() []NetworkUsage {
	if p.usages == nil {
		panic("the usages must not be nil, please use Usages() function instead")
	}
	return p.usages
}

func (p *Network) SetVdsmName(attr string) {
	p.vdsmName = &attr
}

func (p *Network) VdsmName() (string, bool) {
	if p.vdsmName != nil {
		return *p.vdsmName, true
	}
	var zero string
	return zero, false
}

func (p *Network) MustVdsmName() string {
	if p.vdsmName == nil {
		panic("the vdsmName must not be nil, please use VdsmName() function instead")
	}
	return *p.vdsmName
}

// SetVlan Type representing a Virtual LAN (VLAN) type.
func (p *Network) SetVlan(attr *Vlan) {
	p.vlan = attr
}

// Vlan Type representing a Virtual LAN (VLAN) type.
func (p *Network) Vlan() (*Vlan, bool) {
	if p.vlan != nil {
		return p.vlan, true
	}
	return nil, false
}

// MustVlan Type representing a Virtual LAN (VLAN) type.
func (p *Network) MustVlan() *Vlan {
	if p.vlan == nil {
		panic("the vlan must not be nil, please use Vlan() function instead")
	}
	return p.vlan
}

func (p *Network) SetVnicProfiles(attr *VnicProfileSlice) {
	p.vnicProfiles = attr
}

func (p *Network) VnicProfiles() (*VnicProfileSlice, bool) {
	if p.vnicProfiles != nil {
		return p.vnicProfiles, true
	}
	return nil, false
}

func (p *Network) MustVnicProfiles() *VnicProfileSlice {
	if p.vnicProfiles == nil {
		panic("the vnicProfiles must not be nil, please use VnicProfiles() function instead")
	}
	return p.vnicProfiles
}

// NetworkAttachment Describes how a host connects to a network.
//
// An XML representation of a network attachment on a host:
//
// [source,xml]
// ----
// <network_attachment href="/ovirt-engine/api/hosts/123/nics/456/networkattachments/789" id="789">
//   <network href="/ovirt-engine/api/networks/234" id="234"/>
//   <host_nic href="/ovirt-engine/api/hosts/123/nics/123" id="123"/>
//   <in_sync>true</in_sync>
//   <ip_address_assignments>
//     <ip_address_assignment>
//       <assignment_method>static</assignment_method>
//       <ip>
//         <address>192.168.122.39</address>
//         <gateway>192.168.122.1</gateway>
//         <netmask>255.255.255.0</netmask>
//         <version>v4</version>
//       </ip>
//     </ip_address_assignment>
//   </ip_address_assignments>
//   <reported_configurations>
//     <reported_configuration>
//       <name>mtu</name>
//       <expected_value>1500</expected_value>
//       <actual_value>1500</actual_value>
//       <in_sync>true</in_sync>
//     </reported_configuration>
//     <reported_configuration>
//       <name>bridged</name>
//       <expected_value>true</expected_value>
//       <actual_value>true</actual_value>
//       <in_sync>true</in_sync>
//     </reported_configuration>
//     ...
//   </reported_configurations>
// </network_attachment>
// ----
//
// The network element, with either a `name` or an `id`, is required in order to attach a network
// to a network interface card (NIC).
//
// For example, to attach a network to a host network interface card, send a request like this:
//
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/nics/456/networkattachments
// ----
//
// With a request body like this:
//
// [source,xml]
// ----
// <networkattachment>
//   <network id="234"/>
// </networkattachment>
// ----
//
// To attach a network to a host, send a request like this:
//
// [source]
// ----
// POST /ovirt-engine/api/hosts/123/networkattachments
// ----
//
// With a request body like this:
//
// [source,xml]
// ----
// <network_attachment>
//   <network id="234"/>
//   <host_nic id="456"/>
// </network_attachment>
// ----
//
// The `ip_address_assignments` and `properties` elements are updatable post-creation.
//
// For example, to update a network attachment, send a request like this:
//
// [source]
// ----
// PUT /ovirt-engine/api/hosts/123/nics/456/networkattachments/789
// ----
//
// With a request body like this:
//
// [source,xml]
// ----
// <network_attachment>
//   <ip_address_assignments>
//     <ip_address_assignment>
//       <assignment_method>static</assignment_method>
//       <ip>
//         <address>7.1.1.1</address>
//         <gateway>7.1.1.2</gateway>
//         <netmask>255.255.255.0</netmask>
//         <version>v4</version>
//       </ip>
//     </ip_address_assignment>
//   </ip_address_assignments>
// </network_attachment>
// ----
//
// To detach a network from the network interface card send a request like this:
//
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/nics/456/networkattachments/789
// ----
//
// IMPORTANT: Changes to network attachment configuration must be explicitly committed.
//
// An XML representation of a network attachment's `properties` sub-collection:
//
// [source, xml]
// ----
// <network_attachment>
//   <properties>
//     <property>
//       <name>bridge_opts</name>
//       <value>
//         forward_delay=1500 group_fwd_mask=0x0 multicast_snooping=1
//       </value>
//     </property>
//   </properties>
//   ...
// </network_attachment>
// ----
type NetworkAttachment struct {
	Struct
	comment                  *string
	description              *string
	dnsResolverConfiguration *DnsResolverConfiguration
	host                     *Host
	hostNic                  *HostNic
	id                       *string
	inSync                   *bool
	ipAddressAssignments     *IpAddressAssignmentSlice
	name                     *string
	network                  *Network
	properties               *PropertySlice
	qos                      *Qos
	reportedConfigurations   *ReportedConfigurationSlice
}

func (p *NetworkAttachment) SetComment(attr string) {
	p.comment = &attr
}

func (p *NetworkAttachment) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *NetworkAttachment) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *NetworkAttachment) SetDescription(attr string) {
	p.description = &attr
}

func (p *NetworkAttachment) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *NetworkAttachment) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDnsResolverConfiguration Represents the DNS resolver configuration.
func (p *NetworkAttachment) SetDnsResolverConfiguration(attr *DnsResolverConfiguration) {
	p.dnsResolverConfiguration = attr
}

// DnsResolverConfiguration Represents the DNS resolver configuration.
func (p *NetworkAttachment) DnsResolverConfiguration() (*DnsResolverConfiguration, bool) {
	if p.dnsResolverConfiguration != nil {
		return p.dnsResolverConfiguration, true
	}
	return nil, false
}

// MustDnsResolverConfiguration Represents the DNS resolver configuration.
func (p *NetworkAttachment) MustDnsResolverConfiguration() *DnsResolverConfiguration {
	if p.dnsResolverConfiguration == nil {
		panic("the dnsResolverConfiguration must not be nil, please use DnsResolverConfiguration() function instead")
	}
	return p.dnsResolverConfiguration
}

// SetHost Type representing a host.
func (p *NetworkAttachment) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *NetworkAttachment) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *NetworkAttachment) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

// SetHostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkAttachment) SetHostNic(attr *HostNic) {
	p.hostNic = attr
}

// HostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkAttachment) HostNic() (*HostNic, bool) {
	if p.hostNic != nil {
		return p.hostNic, true
	}
	return nil, false
}

// MustHostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkAttachment) MustHostNic() *HostNic {
	if p.hostNic == nil {
		panic("the hostNic must not be nil, please use HostNic() function instead")
	}
	return p.hostNic
}

func (p *NetworkAttachment) SetId(attr string) {
	p.id = &attr
}

func (p *NetworkAttachment) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *NetworkAttachment) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *NetworkAttachment) SetInSync(attr bool) {
	p.inSync = &attr
}

func (p *NetworkAttachment) InSync() (bool, bool) {
	if p.inSync != nil {
		return *p.inSync, true
	}
	var zero bool
	return zero, false
}

func (p *NetworkAttachment) MustInSync() bool {
	if p.inSync == nil {
		panic("the inSync must not be nil, please use InSync() function instead")
	}
	return *p.inSync
}

func (p *NetworkAttachment) SetIpAddressAssignments(attr *IpAddressAssignmentSlice) {
	p.ipAddressAssignments = attr
}

func (p *NetworkAttachment) IpAddressAssignments() (*IpAddressAssignmentSlice, bool) {
	if p.ipAddressAssignments != nil {
		return p.ipAddressAssignments, true
	}
	return nil, false
}

func (p *NetworkAttachment) MustIpAddressAssignments() *IpAddressAssignmentSlice {
	if p.ipAddressAssignments == nil {
		panic("the ipAddressAssignments must not be nil, please use IpAddressAssignments() function instead")
	}
	return p.ipAddressAssignments
}

func (p *NetworkAttachment) SetName(attr string) {
	p.name = &attr
}

func (p *NetworkAttachment) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NetworkAttachment) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkAttachment) SetNetwork(attr *Network) {
	p.network = attr
}

// Network The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkAttachment) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

// MustNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkAttachment) MustNetwork() *Network {
	if p.network == nil {
		panic("the network must not be nil, please use Network() function instead")
	}
	return p.network
}

func (p *NetworkAttachment) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *NetworkAttachment) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *NetworkAttachment) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

// SetQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *NetworkAttachment) SetQos(attr *Qos) {
	p.qos = attr
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *NetworkAttachment) Qos() (*Qos, bool) {
	if p.qos != nil {
		return p.qos, true
	}
	return nil, false
}

// MustQos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
func (p *NetworkAttachment) MustQos() *Qos {
	if p.qos == nil {
		panic("the qos must not be nil, please use Qos() function instead")
	}
	return p.qos
}

func (p *NetworkAttachment) SetReportedConfigurations(attr *ReportedConfigurationSlice) {
	p.reportedConfigurations = attr
}

func (p *NetworkAttachment) ReportedConfigurations() (*ReportedConfigurationSlice, bool) {
	if p.reportedConfigurations != nil {
		return p.reportedConfigurations, true
	}
	return nil, false
}

func (p *NetworkAttachment) MustReportedConfigurations() *ReportedConfigurationSlice {
	if p.reportedConfigurations == nil {
		panic("the reportedConfigurations must not be nil, please use ReportedConfigurations() function instead")
	}
	return p.reportedConfigurations
}

type NetworkConfiguration struct {
	Struct
	dns  *Dns
	nics *NicSlice
}

// SetDns Represents the DNS resolver configuration.
func (p *NetworkConfiguration) SetDns(attr *Dns) {
	p.dns = attr
}

// Dns Represents the DNS resolver configuration.
func (p *NetworkConfiguration) Dns() (*Dns, bool) {
	if p.dns != nil {
		return p.dns, true
	}
	return nil, false
}

// MustDns Represents the DNS resolver configuration.
func (p *NetworkConfiguration) MustDns() *Dns {
	if p.dns == nil {
		panic("the dns must not be nil, please use Dns() function instead")
	}
	return p.dns
}

func (p *NetworkConfiguration) SetNics(attr *NicSlice) {
	p.nics = attr
}

func (p *NetworkConfiguration) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *NetworkConfiguration) MustNics() *NicSlice {
	if p.nics == nil {
		panic("the nics must not be nil, please use Nics() function instead")
	}
	return p.nics
}

// NetworkFilter Network filters filter packets sent to and from the virtual machine's NIC according to defined rules.
//
// There are several types of network filters supported based on libvirt.
// For more details about the different network filters see https://libvirt.org/firewall.html[here].
//
// In addition to libvirt's network filters, there are two additional network filters:
// The first is called `vdsm-no-mac-spoofing` and is composed of `no-mac-spoofing` and `no-arp-mac-spoofing`.
// The second is called `ovirt-no-filter` and is used when no network filter is to be defined for the virtual machine's NIC.
// The `ovirt-no-filter` network filter is only used for internal implementation, and
// does not exist on the NICs.
//
// This is a example of the XML representation:
//
// [source,xml]
// ----
// <network_filter id="00000019-0019-0019-0019-00000000026c">
//   <name>example-filter</name>
//   <version>
//     <major>4</major>
//     <minor>0</minor>
//     <build>-1</build>
//     <revision>-1</revision>
//   </version>
// </network_filter>
// ----
//
// If any part of the version is not present, it is represented by -1.
type NetworkFilter struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
	version     *Version
}

func (p *NetworkFilter) SetComment(attr string) {
	p.comment = &attr
}

func (p *NetworkFilter) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilter) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *NetworkFilter) SetDescription(attr string) {
	p.description = &attr
}

func (p *NetworkFilter) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilter) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *NetworkFilter) SetId(attr string) {
	p.id = &attr
}

func (p *NetworkFilter) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilter) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *NetworkFilter) SetName(attr string) {
	p.name = &attr
}

func (p *NetworkFilter) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilter) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *NetworkFilter) SetVersion(attr *Version) {
	p.version = attr
}

func (p *NetworkFilter) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *NetworkFilter) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// NetworkFilterParameter Parameter for the <<types/network_filter,network filter>>.
//
// See https://libvirt.org/formatnwfilter.html#nwfconceptsvars[Libvirt-Filters] for further details.
// This is a example of the XML representation:
//
// [source,xml]
// ----
// <network_filter_parameter id="123">
//   <name>IP</name>
//   <value>10.0.1.2</value>
// </network_filter_parameter>
// ----
type NetworkFilterParameter struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
	nic         *Nic
	value       *string
}

func (p *NetworkFilterParameter) SetComment(attr string) {
	p.comment = &attr
}

func (p *NetworkFilterParameter) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilterParameter) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *NetworkFilterParameter) SetDescription(attr string) {
	p.description = &attr
}

func (p *NetworkFilterParameter) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilterParameter) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *NetworkFilterParameter) SetId(attr string) {
	p.id = &attr
}

func (p *NetworkFilterParameter) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilterParameter) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *NetworkFilterParameter) SetName(attr string) {
	p.name = &attr
}

func (p *NetworkFilterParameter) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilterParameter) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNic Represents a virtual machine NIC.
//
// For example, the XML representation of a NIC will look like this:
//
// [source,xml]
// ----
// <nic href="/ovirt-engine/api/vms/123/nics/456" id="456">
//   <name>nic1</name>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <interface>virtio</interface>
//   <linked>true</linked>
//   <mac>
//     <address>02:00:00:00:00:00</address>
//   </mac>
//   <plugged>true</plugged>
//   <vnic_profile href="/ovirt-engine/api/vnicprofiles/789" id="789"/>
// </nic>
// ----
func (p *NetworkFilterParameter) SetNic(attr *Nic) {
	p.nic = attr
}

// Nic Represents a virtual machine NIC.
//
// For example, the XML representation of a NIC will look like this:
//
// [source,xml]
// ----
// <nic href="/ovirt-engine/api/vms/123/nics/456" id="456">
//   <name>nic1</name>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <interface>virtio</interface>
//   <linked>true</linked>
//   <mac>
//     <address>02:00:00:00:00:00</address>
//   </mac>
//   <plugged>true</plugged>
//   <vnic_profile href="/ovirt-engine/api/vnicprofiles/789" id="789"/>
// </nic>
// ----
func (p *NetworkFilterParameter) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

// MustNic Represents a virtual machine NIC.
//
// For example, the XML representation of a NIC will look like this:
//
// [source,xml]
// ----
// <nic href="/ovirt-engine/api/vms/123/nics/456" id="456">
//   <name>nic1</name>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <interface>virtio</interface>
//   <linked>true</linked>
//   <mac>
//     <address>02:00:00:00:00:00</address>
//   </mac>
//   <plugged>true</plugged>
//   <vnic_profile href="/ovirt-engine/api/vnicprofiles/789" id="789"/>
// </nic>
// ----
func (p *NetworkFilterParameter) MustNic() *Nic {
	if p.nic == nil {
		panic("the nic must not be nil, please use Nic() function instead")
	}
	return p.nic
}

func (p *NetworkFilterParameter) SetValue(attr string) {
	p.value = &attr
}

func (p *NetworkFilterParameter) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *NetworkFilterParameter) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

// NetworkLabel Represents a label which can be added to a host network interface and to a network.
// The label binds the network to the host network interface by the label `id`.
type NetworkLabel struct {
	Struct
	comment     *string
	description *string
	hostNic     *HostNic
	id          *string
	name        *string
	network     *Network
}

func (p *NetworkLabel) SetComment(attr string) {
	p.comment = &attr
}

func (p *NetworkLabel) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *NetworkLabel) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *NetworkLabel) SetDescription(attr string) {
	p.description = &attr
}

func (p *NetworkLabel) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *NetworkLabel) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetHostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkLabel) SetHostNic(attr *HostNic) {
	p.hostNic = attr
}

// HostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkLabel) HostNic() (*HostNic, bool) {
	if p.hostNic != nil {
		return p.hostNic, true
	}
	return nil, false
}

// MustHostNic Represents a host NIC.
//
// For example, the XML representation of a host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>eth0</name>
//   <boot_protocol>static</boot_protocol>
//   <bridged>true</bridged>
//   <custom_configuration>true</custom_configuration>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <ipv6>
//     <gateway>::</gateway>
//     <version>v6</version>
//   </ipv6>
//   <ipv6_boot_protocol>none</ipv6_boot_protocol>
//   <mac>
//     <address>52:54:00:0c:79:1d</address>
//   </mac>
//   <mtu>1500</mtu>
//   <status>up</status>
// </host_nic>
// ----
//
// A bonded interface is represented as a <<types/host_nic, HostNic>> object
// containing the `bonding` and `slaves` attributes.
//
// For example, the XML representation of a bonded host NIC looks like this:
//
// [source,xml]
// ----
// <host_nic href="/ovirt-engine/api/hosts/123/nics/456" id="456">
//   <name>bond0</name>
//   <mac address="00:00:00:00:00:00"/>
//   <ip>
//     <address>192.168.122.39</address>
//     <gateway>192.168.122.1</gateway>
//     <netmask>255.255.255.0</netmask>
//     <version>v4</version>
//   </ip>
//   <boot_protocol>dhcp</boot_protocol>
//   <bonding>
//     <options>
//       <option>
//         <name>mode</name>
//         <value>4</value>
//         <type>Dynamic link aggregation (802.3ad)</type>
//       </option>
//       <option>
//         <name>miimon</name>
//         <value>100</value>
//       </option>
//     </options>
//     <slaves>
//       <host_nic id="123"/>
//       <host_nic id="456"/>
//     </slaves>
//   </bonding>
//   <mtu>1500</mtu>
//   <bridged>true</bridged>
//   <custom_configuration>false</custom_configuration>
// </host_nic>
// ----
func (p *NetworkLabel) MustHostNic() *HostNic {
	if p.hostNic == nil {
		panic("the hostNic must not be nil, please use HostNic() function instead")
	}
	return p.hostNic
}

func (p *NetworkLabel) SetId(attr string) {
	p.id = &attr
}

func (p *NetworkLabel) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *NetworkLabel) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *NetworkLabel) SetName(attr string) {
	p.name = &attr
}

func (p *NetworkLabel) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NetworkLabel) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkLabel) SetNetwork(attr *Network) {
	p.network = attr
}

// Network The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkLabel) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

// MustNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *NetworkLabel) MustNetwork() *Network {
	if p.network == nil {
		panic("the network must not be nil, please use Network() function instead")
	}
	return p.network
}

type NfsProfileDetail struct {
	Struct
	nfsServerIp    *string
	profileDetails *ProfileDetailSlice
}

func (p *NfsProfileDetail) SetNfsServerIp(attr string) {
	p.nfsServerIp = &attr
}

func (p *NfsProfileDetail) NfsServerIp() (string, bool) {
	if p.nfsServerIp != nil {
		return *p.nfsServerIp, true
	}
	var zero string
	return zero, false
}

func (p *NfsProfileDetail) MustNfsServerIp() string {
	if p.nfsServerIp == nil {
		panic("the nfsServerIp must not be nil, please use NfsServerIp() function instead")
	}
	return *p.nfsServerIp
}

func (p *NfsProfileDetail) SetProfileDetails(attr *ProfileDetailSlice) {
	p.profileDetails = attr
}

func (p *NfsProfileDetail) ProfileDetails() (*ProfileDetailSlice, bool) {
	if p.profileDetails != nil {
		return p.profileDetails, true
	}
	return nil, false
}

func (p *NfsProfileDetail) MustProfileDetails() *ProfileDetailSlice {
	if p.profileDetails == nil {
		panic("the profileDetails must not be nil, please use ProfileDetails() function instead")
	}
	return p.profileDetails
}

// Nic Represents a virtual machine NIC.
//
// For example, the XML representation of a NIC will look like this:
//
// [source,xml]
// ----
// <nic href="/ovirt-engine/api/vms/123/nics/456" id="456">
//   <name>nic1</name>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <interface>virtio</interface>
//   <linked>true</linked>
//   <mac>
//     <address>02:00:00:00:00:00</address>
//   </mac>
//   <plugged>true</plugged>
//   <vnic_profile href="/ovirt-engine/api/vnicprofiles/789" id="789"/>
// </nic>
// ----
type Nic struct {
	Struct
	bootProtocol                   *BootProtocol
	comment                        *string
	description                    *string
	id                             *string
	instanceType                   *InstanceType
	interface_                     *NicInterface
	linked                         *bool
	mac                            *Mac
	name                           *string
	network                        *Network
	networkAttachments             *NetworkAttachmentSlice
	networkFilterParameters        *NetworkFilterParameterSlice
	networkLabels                  *NetworkLabelSlice
	onBoot                         *bool
	plugged                        *bool
	reportedDevices                *ReportedDeviceSlice
	statistics                     *StatisticSlice
	synced                         *bool
	template                       *Template
	virtualFunctionAllowedLabels   *NetworkLabelSlice
	virtualFunctionAllowedNetworks *NetworkSlice
	vm                             *Vm
	vms                            *VmSlice
	vnicProfile                    *VnicProfile
}

// SetBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *Nic) SetBootProtocol(attr BootProtocol) {
	p.bootProtocol = &attr
}

// BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *Nic) BootProtocol() (BootProtocol, bool) {
	if p.bootProtocol != nil {
		return *p.bootProtocol, true
	}
	var zero BootProtocol
	return zero, false
}

// MustBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *Nic) MustBootProtocol() BootProtocol {
	if p.bootProtocol == nil {
		panic("the bootProtocol must not be nil, please use BootProtocol() function instead")
	}
	return *p.bootProtocol
}

func (p *Nic) SetComment(attr string) {
	p.comment = &attr
}

func (p *Nic) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Nic) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Nic) SetDescription(attr string) {
	p.description = &attr
}

func (p *Nic) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Nic) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Nic) SetId(attr string) {
	p.id = &attr
}

func (p *Nic) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Nic) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Nic) SetInstanceType(attr *InstanceType) {
	p.instanceType = attr
}

// InstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Nic) InstanceType() (*InstanceType, bool) {
	if p.instanceType != nil {
		return p.instanceType, true
	}
	return nil, false
}

// MustInstanceType Describes the hardware configuration of virtual machines.
//
// For example `medium` instance type includes 1 virtual CPU and 4 GiB of memory. It is a top-level
// entity (e.g. not bound to any data center or cluster). The attributes that are used for instance
// types and are common to virtual machine and template types are:
//
// - `console`
// - `cpu`
// - `custom_cpu_model`
// - `custom_emulated_machine`
// - `display`
// - `high_availability`
// - `io`
// - `memory`
// - `memory_policy`
// - `migration`
// - `migration_downtime`
// - `os`
// - `rng_device`
// - `soundcard_enabled`
// - `usb`
// - `virtio_scsi`
//
// When creating a virtual machine from both an instance type and a template, the virtual machine
// will inherit the hardware configurations from the instance type
//
// NOTE: An instance type inherits it's attributes from the template entity although most template
// attributes are not used in instance types.
func (p *Nic) MustInstanceType() *InstanceType {
	if p.instanceType == nil {
		panic("the instanceType must not be nil, please use InstanceType() function instead")
	}
	return p.instanceType
}

// SetInterface Defines the options for an emulated virtual network interface device model.
func (p *Nic) SetInterface(attr NicInterface) {
	p.interface_ = &attr
}

// Interface Defines the options for an emulated virtual network interface device model.
func (p *Nic) Interface() (NicInterface, bool) {
	if p.interface_ != nil {
		return *p.interface_, true
	}
	var zero NicInterface
	return zero, false
}

// MustInterface Defines the options for an emulated virtual network interface device model.
func (p *Nic) MustInterface() NicInterface {
	if p.interface_ == nil {
		panic("the interface_ must not be nil, please use Interface() function instead")
	}
	return *p.interface_
}

func (p *Nic) SetLinked(attr bool) {
	p.linked = &attr
}

func (p *Nic) Linked() (bool, bool) {
	if p.linked != nil {
		return *p.linked, true
	}
	var zero bool
	return zero, false
}

func (p *Nic) MustLinked() bool {
	if p.linked == nil {
		panic("the linked must not be nil, please use Linked() function instead")
	}
	return *p.linked
}

// SetMac Represents a MAC address of a virtual network interface.
func (p *Nic) SetMac(attr *Mac) {
	p.mac = attr
}

// Mac Represents a MAC address of a virtual network interface.
func (p *Nic) Mac() (*Mac, bool) {
	if p.mac != nil {
		return p.mac, true
	}
	return nil, false
}

// MustMac Represents a MAC address of a virtual network interface.
func (p *Nic) MustMac() *Mac {
	if p.mac == nil {
		panic("the mac must not be nil, please use Mac() function instead")
	}
	return p.mac
}

func (p *Nic) SetName(attr string) {
	p.name = &attr
}

func (p *Nic) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Nic) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Nic) SetNetwork(attr *Network) {
	p.network = attr
}

// Network The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Nic) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

// MustNetwork The type for a logical network.
//
// An example of the JSON representation of a logical network:
//
// [source,json]
// ----
// {
//   "network" : [ {
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "stp" : "false",
//     "mtu" : "0",
//     "usages" : {
//       "usage" : [ "vm" ]
//     },
//     "name" : "ovirtmgmt",
//     "description" : "Management Network",
//     "href" : "/ovirt-engine/api/networks/456",
//     "id" : "456",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/networks/456/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/vnicprofiles",
//       "rel" : "vnicprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/networks/456/labels",
//       "rel" : "labels"
//     } ]
//   } ]
// }
// ----
//
// An example of the XML representation of the same logical network:
//
// [source,xml]
// ----
// <network href="/ovirt-engine/api/networks/456" id="456">
//   <name>ovirtmgmt</name>
//   <description>Management Network</description>
//   <link href="/ovirt-engine/api/networks/456/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/networks/456/vnicprofiles" rel="vnicprofiles"/>
//   <link href="/ovirt-engine/api/networks/456/labels" rel="labels"/>
//   <data_center href="/ovirt-engine/api/datacenters/123" id="123"/>
//   <stp>false</stp>
//   <mtu>0</mtu>
//   <usages>
//     <usage>vm</usage>
//   </usages>
// </network>
// ----
func (p *Nic) MustNetwork() *Network {
	if p.network == nil {
		panic("the network must not be nil, please use Network() function instead")
	}
	return p.network
}

func (p *Nic) SetNetworkAttachments(attr *NetworkAttachmentSlice) {
	p.networkAttachments = attr
}

func (p *Nic) NetworkAttachments() (*NetworkAttachmentSlice, bool) {
	if p.networkAttachments != nil {
		return p.networkAttachments, true
	}
	return nil, false
}

func (p *Nic) MustNetworkAttachments() *NetworkAttachmentSlice {
	if p.networkAttachments == nil {
		panic("the networkAttachments must not be nil, please use NetworkAttachments() function instead")
	}
	return p.networkAttachments
}

func (p *Nic) SetNetworkFilterParameters(attr *NetworkFilterParameterSlice) {
	p.networkFilterParameters = attr
}

func (p *Nic) NetworkFilterParameters() (*NetworkFilterParameterSlice, bool) {
	if p.networkFilterParameters != nil {
		return p.networkFilterParameters, true
	}
	return nil, false
}

func (p *Nic) MustNetworkFilterParameters() *NetworkFilterParameterSlice {
	if p.networkFilterParameters == nil {
		panic("the networkFilterParameters must not be nil, please use NetworkFilterParameters() function instead")
	}
	return p.networkFilterParameters
}

func (p *Nic) SetNetworkLabels(attr *NetworkLabelSlice) {
	p.networkLabels = attr
}

func (p *Nic) NetworkLabels() (*NetworkLabelSlice, bool) {
	if p.networkLabels != nil {
		return p.networkLabels, true
	}
	return nil, false
}

func (p *Nic) MustNetworkLabels() *NetworkLabelSlice {
	if p.networkLabels == nil {
		panic("the networkLabels must not be nil, please use NetworkLabels() function instead")
	}
	return p.networkLabels
}

func (p *Nic) SetOnBoot(attr bool) {
	p.onBoot = &attr
}

func (p *Nic) OnBoot() (bool, bool) {
	if p.onBoot != nil {
		return *p.onBoot, true
	}
	var zero bool
	return zero, false
}

func (p *Nic) MustOnBoot() bool {
	if p.onBoot == nil {
		panic("the onBoot must not be nil, please use OnBoot() function instead")
	}
	return *p.onBoot
}

func (p *Nic) SetPlugged(attr bool) {
	p.plugged = &attr
}

func (p *Nic) Plugged() (bool, bool) {
	if p.plugged != nil {
		return *p.plugged, true
	}
	var zero bool
	return zero, false
}

func (p *Nic) MustPlugged() bool {
	if p.plugged == nil {
		panic("the plugged must not be nil, please use Plugged() function instead")
	}
	return *p.plugged
}

func (p *Nic) SetReportedDevices(attr *ReportedDeviceSlice) {
	p.reportedDevices = attr
}

func (p *Nic) ReportedDevices() (*ReportedDeviceSlice, bool) {
	if p.reportedDevices != nil {
		return p.reportedDevices, true
	}
	return nil, false
}

func (p *Nic) MustReportedDevices() *ReportedDeviceSlice {
	if p.reportedDevices == nil {
		panic("the reportedDevices must not be nil, please use ReportedDevices() function instead")
	}
	return p.reportedDevices
}

func (p *Nic) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *Nic) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *Nic) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

func (p *Nic) SetSynced(attr bool) {
	p.synced = &attr
}

func (p *Nic) Synced() (bool, bool) {
	if p.synced != nil {
		return *p.synced, true
	}
	var zero bool
	return zero, false
}

func (p *Nic) MustSynced() bool {
	if p.synced == nil {
		panic("the synced must not be nil, please use Synced() function instead")
	}
	return *p.synced
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Nic) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Nic) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Nic) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

func (p *Nic) SetVirtualFunctionAllowedLabels(attr *NetworkLabelSlice) {
	p.virtualFunctionAllowedLabels = attr
}

func (p *Nic) VirtualFunctionAllowedLabels() (*NetworkLabelSlice, bool) {
	if p.virtualFunctionAllowedLabels != nil {
		return p.virtualFunctionAllowedLabels, true
	}
	return nil, false
}

func (p *Nic) MustVirtualFunctionAllowedLabels() *NetworkLabelSlice {
	if p.virtualFunctionAllowedLabels == nil {
		panic("the virtualFunctionAllowedLabels must not be nil, please use VirtualFunctionAllowedLabels() function instead")
	}
	return p.virtualFunctionAllowedLabels
}

func (p *Nic) SetVirtualFunctionAllowedNetworks(attr *NetworkSlice) {
	p.virtualFunctionAllowedNetworks = attr
}

func (p *Nic) VirtualFunctionAllowedNetworks() (*NetworkSlice, bool) {
	if p.virtualFunctionAllowedNetworks != nil {
		return p.virtualFunctionAllowedNetworks, true
	}
	return nil, false
}

func (p *Nic) MustVirtualFunctionAllowedNetworks() *NetworkSlice {
	if p.virtualFunctionAllowedNetworks == nil {
		panic("the virtualFunctionAllowedNetworks must not be nil, please use VirtualFunctionAllowedNetworks() function instead")
	}
	return p.virtualFunctionAllowedNetworks
}

// SetVm Represents a virtual machine.
func (p *Nic) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Nic) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Nic) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

func (p *Nic) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Nic) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Nic) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

// SetVnicProfile A vNIC profile is a collection of settings that can be applied to individual <<types/nic,NIC>>.
func (p *Nic) SetVnicProfile(attr *VnicProfile) {
	p.vnicProfile = attr
}

// VnicProfile A vNIC profile is a collection of settings that can be applied to individual <<types/nic,NIC>>.
func (p *Nic) VnicProfile() (*VnicProfile, bool) {
	if p.vnicProfile != nil {
		return p.vnicProfile, true
	}
	return nil, false
}

// MustVnicProfile A vNIC profile is a collection of settings that can be applied to individual <<types/nic,NIC>>.
func (p *Nic) MustVnicProfile() *VnicProfile {
	if p.vnicProfile == nil {
		panic("the vnicProfile must not be nil, please use VnicProfile() function instead")
	}
	return p.vnicProfile
}

// NicConfiguration The type describes the configuration of a virtual network interface.
type NicConfiguration struct {
	Struct
	bootProtocol     *BootProtocol
	ip               *Ip
	ipv6             *Ip
	ipv6BootProtocol *BootProtocol
	name             *string
	onBoot           *bool
}

// SetBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) SetBootProtocol(attr BootProtocol) {
	p.bootProtocol = &attr
}

// BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) BootProtocol() (BootProtocol, bool) {
	if p.bootProtocol != nil {
		return *p.bootProtocol, true
	}
	var zero BootProtocol
	return zero, false
}

// MustBootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) MustBootProtocol() BootProtocol {
	if p.bootProtocol == nil {
		panic("the bootProtocol must not be nil, please use BootProtocol() function instead")
	}
	return *p.bootProtocol
}

// SetIp Represents the IP configuration of a network interface.
func (p *NicConfiguration) SetIp(attr *Ip) {
	p.ip = attr
}

// Ip Represents the IP configuration of a network interface.
func (p *NicConfiguration) Ip() (*Ip, bool) {
	if p.ip != nil {
		return p.ip, true
	}
	return nil, false
}

// MustIp Represents the IP configuration of a network interface.
func (p *NicConfiguration) MustIp() *Ip {
	if p.ip == nil {
		panic("the ip must not be nil, please use Ip() function instead")
	}
	return p.ip
}

// SetIpv6 Represents the IP configuration of a network interface.
func (p *NicConfiguration) SetIpv6(attr *Ip) {
	p.ipv6 = attr
}

// Ipv6 Represents the IP configuration of a network interface.
func (p *NicConfiguration) Ipv6() (*Ip, bool) {
	if p.ipv6 != nil {
		return p.ipv6, true
	}
	return nil, false
}

// MustIpv6 Represents the IP configuration of a network interface.
func (p *NicConfiguration) MustIpv6() *Ip {
	if p.ipv6 == nil {
		panic("the ipv6 must not be nil, please use Ipv6() function instead")
	}
	return p.ipv6
}

// SetIpv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) SetIpv6BootProtocol(attr BootProtocol) {
	p.ipv6BootProtocol = &attr
}

// Ipv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) Ipv6BootProtocol() (BootProtocol, bool) {
	if p.ipv6BootProtocol != nil {
		return *p.ipv6BootProtocol, true
	}
	var zero BootProtocol
	return zero, false
}

// MustIpv6BootProtocol Defines the options of the IP address assignment method to a NIC.
func (p *NicConfiguration) MustIpv6BootProtocol() BootProtocol {
	if p.ipv6BootProtocol == nil {
		panic("the ipv6BootProtocol must not be nil, please use Ipv6BootProtocol() function instead")
	}
	return *p.ipv6BootProtocol
}

func (p *NicConfiguration) SetName(attr string) {
	p.name = &attr
}

func (p *NicConfiguration) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NicConfiguration) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *NicConfiguration) SetOnBoot(attr bool) {
	p.onBoot = &attr
}

func (p *NicConfiguration) OnBoot() (bool, bool) {
	if p.onBoot != nil {
		return *p.onBoot, true
	}
	var zero bool
	return zero, false
}

func (p *NicConfiguration) MustOnBoot() bool {
	if p.onBoot == nil {
		panic("the onBoot must not be nil, please use OnBoot() function instead")
	}
	return *p.onBoot
}

// NumaNode Represents a physical NUMA node.
//
// Example XML representation:
//
// [source,xml]
// ----
// <host_numa_node href="/ovirt-engine/api/hosts/0923f1ea/numanodes/007cf1ab" id="007cf1ab">
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>65536</memory>
//   <node_distance>40 20 40 10</node_distance>
//   <host href="/ovirt-engine/api/hosts/0923f1ea" id="0923f1ea"/>
// </host_numa_node>
// ----
type NumaNode struct {
	Struct
	comment      *string
	cpu          *Cpu
	description  *string
	host         *Host
	id           *string
	index        *int64
	memory       *int64
	name         *string
	nodeDistance *string
	statistics   *StatisticSlice
}

func (p *NumaNode) SetComment(attr string) {
	p.comment = &attr
}

func (p *NumaNode) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *NumaNode) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *NumaNode) SetCpu(attr *Cpu) {
	p.cpu = attr
}

func (p *NumaNode) Cpu() (*Cpu, bool) {
	if p.cpu != nil {
		return p.cpu, true
	}
	return nil, false
}

func (p *NumaNode) MustCpu() *Cpu {
	if p.cpu == nil {
		panic("the cpu must not be nil, please use Cpu() function instead")
	}
	return p.cpu
}

func (p *NumaNode) SetDescription(attr string) {
	p.description = &attr
}

func (p *NumaNode) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *NumaNode) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetHost Type representing a host.
func (p *NumaNode) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *NumaNode) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *NumaNode) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *NumaNode) SetId(attr string) {
	p.id = &attr
}

func (p *NumaNode) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *NumaNode) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *NumaNode) SetIndex(attr int64) {
	p.index = &attr
}

func (p *NumaNode) Index() (int64, bool) {
	if p.index != nil {
		return *p.index, true
	}
	var zero int64
	return zero, false
}

func (p *NumaNode) MustIndex() int64 {
	if p.index == nil {
		panic("the index must not be nil, please use Index() function instead")
	}
	return *p.index
}

func (p *NumaNode) SetMemory(attr int64) {
	p.memory = &attr
}

func (p *NumaNode) Memory() (int64, bool) {
	if p.memory != nil {
		return *p.memory, true
	}
	var zero int64
	return zero, false
}

func (p *NumaNode) MustMemory() int64 {
	if p.memory == nil {
		panic("the memory must not be nil, please use Memory() function instead")
	}
	return *p.memory
}

func (p *NumaNode) SetName(attr string) {
	p.name = &attr
}

func (p *NumaNode) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *NumaNode) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *NumaNode) SetNodeDistance(attr string) {
	p.nodeDistance = &attr
}

func (p *NumaNode) NodeDistance() (string, bool) {
	if p.nodeDistance != nil {
		return *p.nodeDistance, true
	}
	var zero string
	return zero, false
}

func (p *NumaNode) MustNodeDistance() string {
	if p.nodeDistance == nil {
		panic("the nodeDistance must not be nil, please use NodeDistance() function instead")
	}
	return *p.nodeDistance
}

func (p *NumaNode) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *NumaNode) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *NumaNode) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

// NumaNodePin Represents the pinning of a virtual NUMA node to a physical NUMA node.
type NumaNodePin struct {
	Struct
	hostNumaNode *NumaNode
	index        *int64
	pinned       *bool
}

// SetHostNumaNode Represents a physical NUMA node.
//
// Example XML representation:
//
// [source,xml]
// ----
// <host_numa_node href="/ovirt-engine/api/hosts/0923f1ea/numanodes/007cf1ab" id="007cf1ab">
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>65536</memory>
//   <node_distance>40 20 40 10</node_distance>
//   <host href="/ovirt-engine/api/hosts/0923f1ea" id="0923f1ea"/>
// </host_numa_node>
// ----
func (p *NumaNodePin) SetHostNumaNode(attr *NumaNode) {
	p.hostNumaNode = attr
}

// HostNumaNode Represents a physical NUMA node.
//
// Example XML representation:
//
// [source,xml]
// ----
// <host_numa_node href="/ovirt-engine/api/hosts/0923f1ea/numanodes/007cf1ab" id="007cf1ab">
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>65536</memory>
//   <node_distance>40 20 40 10</node_distance>
//   <host href="/ovirt-engine/api/hosts/0923f1ea" id="0923f1ea"/>
// </host_numa_node>
// ----
func (p *NumaNodePin) HostNumaNode() (*NumaNode, bool) {
	if p.hostNumaNode != nil {
		return p.hostNumaNode, true
	}
	return nil, false
}

// MustHostNumaNode Represents a physical NUMA node.
//
// Example XML representation:
//
// [source,xml]
// ----
// <host_numa_node href="/ovirt-engine/api/hosts/0923f1ea/numanodes/007cf1ab" id="007cf1ab">
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>65536</memory>
//   <node_distance>40 20 40 10</node_distance>
//   <host href="/ovirt-engine/api/hosts/0923f1ea" id="0923f1ea"/>
// </host_numa_node>
// ----
func (p *NumaNodePin) MustHostNumaNode() *NumaNode {
	if p.hostNumaNode == nil {
		panic("the hostNumaNode must not be nil, please use HostNumaNode() function instead")
	}
	return p.hostNumaNode
}

func (p *NumaNodePin) SetIndex(attr int64) {
	p.index = &attr
}

func (p *NumaNodePin) Index() (int64, bool) {
	if p.index != nil {
		return *p.index, true
	}
	var zero int64
	return zero, false
}

func (p *NumaNodePin) MustIndex() int64 {
	if p.index == nil {
		panic("the index must not be nil, please use Index() function instead")
	}
	return *p.index
}

func (p *NumaNodePin) SetPinned(attr bool) {
	p.pinned = &attr
}

func (p *NumaNodePin) Pinned() (bool, bool) {
	if p.pinned != nil {
		return *p.pinned, true
	}
	var zero bool
	return zero, false
}

func (p *NumaNodePin) MustPinned() bool {
	if p.pinned == nil {
		panic("the pinned must not be nil, please use Pinned() function instead")
	}
	return *p.pinned
}

type OpenStackImage struct {
	Struct
	comment                *string
	description            *string
	id                     *string
	name                   *string
	openstackImageProvider *OpenStackImageProvider
}

func (p *OpenStackImage) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackImage) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImage) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackImage) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackImage) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImage) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackImage) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackImage) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImage) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackImage) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackImage) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImage) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackImage) SetOpenstackImageProvider(attr *OpenStackImageProvider) {
	p.openstackImageProvider = attr
}

func (p *OpenStackImage) OpenstackImageProvider() (*OpenStackImageProvider, bool) {
	if p.openstackImageProvider != nil {
		return p.openstackImageProvider, true
	}
	return nil, false
}

func (p *OpenStackImage) MustOpenstackImageProvider() *OpenStackImageProvider {
	if p.openstackImageProvider == nil {
		panic("the openstackImageProvider must not be nil, please use OpenstackImageProvider() function instead")
	}
	return p.openstackImageProvider
}

type OpenStackImageProvider struct {
	Struct
	authenticationUrl      *string
	certificates           *CertificateSlice
	comment                *string
	description            *string
	id                     *string
	images                 *OpenStackImageSlice
	name                   *string
	password               *string
	properties             *PropertySlice
	requiresAuthentication *bool
	tenantName             *string
	url                    *string
	username               *string
}

func (p *OpenStackImageProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *OpenStackImageProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *OpenStackImageProvider) SetCertificates(attr *CertificateSlice) {
	p.certificates = attr
}

func (p *OpenStackImageProvider) Certificates() (*CertificateSlice, bool) {
	if p.certificates != nil {
		return p.certificates, true
	}
	return nil, false
}

func (p *OpenStackImageProvider) MustCertificates() *CertificateSlice {
	if p.certificates == nil {
		panic("the certificates must not be nil, please use Certificates() function instead")
	}
	return p.certificates
}

func (p *OpenStackImageProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackImageProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackImageProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackImageProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackImageProvider) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackImageProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackImageProvider) SetImages(attr *OpenStackImageSlice) {
	p.images = attr
}

func (p *OpenStackImageProvider) Images() (*OpenStackImageSlice, bool) {
	if p.images != nil {
		return p.images, true
	}
	return nil, false
}

func (p *OpenStackImageProvider) MustImages() *OpenStackImageSlice {
	if p.images == nil {
		panic("the images must not be nil, please use Images() function instead")
	}
	return p.images
}

func (p *OpenStackImageProvider) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackImageProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackImageProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *OpenStackImageProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *OpenStackImageProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *OpenStackImageProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *OpenStackImageProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *OpenStackImageProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *OpenStackImageProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackImageProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *OpenStackImageProvider) SetTenantName(attr string) {
	p.tenantName = &attr
}

func (p *OpenStackImageProvider) TenantName() (string, bool) {
	if p.tenantName != nil {
		return *p.tenantName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustTenantName() string {
	if p.tenantName == nil {
		panic("the tenantName must not be nil, please use TenantName() function instead")
	}
	return *p.tenantName
}

func (p *OpenStackImageProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *OpenStackImageProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *OpenStackImageProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *OpenStackImageProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackImageProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

type OpenStackNetwork struct {
	Struct
	comment                  *string
	description              *string
	id                       *string
	name                     *string
	openstackNetworkProvider *OpenStackNetworkProvider
}

func (p *OpenStackNetwork) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackNetwork) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetwork) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackNetwork) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackNetwork) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetwork) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackNetwork) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackNetwork) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetwork) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackNetwork) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackNetwork) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetwork) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackNetwork) SetOpenstackNetworkProvider(attr *OpenStackNetworkProvider) {
	p.openstackNetworkProvider = attr
}

func (p *OpenStackNetwork) OpenstackNetworkProvider() (*OpenStackNetworkProvider, bool) {
	if p.openstackNetworkProvider != nil {
		return p.openstackNetworkProvider, true
	}
	return nil, false
}

func (p *OpenStackNetwork) MustOpenstackNetworkProvider() *OpenStackNetworkProvider {
	if p.openstackNetworkProvider == nil {
		panic("the openstackNetworkProvider must not be nil, please use OpenstackNetworkProvider() function instead")
	}
	return p.openstackNetworkProvider
}

type OpenStackNetworkProvider struct {
	Struct
	agentConfiguration     *AgentConfiguration
	authenticationUrl      *string
	autoSync               *bool
	certificates           *CertificateSlice
	comment                *string
	description            *string
	externalPluginType     *string
	id                     *string
	name                   *string
	networks               *OpenStackNetworkSlice
	password               *string
	pluginType             *NetworkPluginType
	projectDomainName      *string
	projectName            *string
	properties             *PropertySlice
	readOnly               *bool
	requiresAuthentication *bool
	subnets                *OpenStackSubnetSlice
	tenantName             *string
	type_                  *OpenStackNetworkProviderType
	unmanaged              *bool
	url                    *string
	userDomainName         *string
	username               *string
}

// SetAgentConfiguration Deprecated Agent configuration settings.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
// The deployment of OpenStack hosts can be done by Red Hat OpenStack Platform Director or TripleO.
func (p *OpenStackNetworkProvider) SetAgentConfiguration(attr *AgentConfiguration) {
	p.agentConfiguration = attr
}

// AgentConfiguration Deprecated Agent configuration settings.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
// The deployment of OpenStack hosts can be done by Red Hat OpenStack Platform Director or TripleO.
func (p *OpenStackNetworkProvider) AgentConfiguration() (*AgentConfiguration, bool) {
	if p.agentConfiguration != nil {
		return p.agentConfiguration, true
	}
	return nil, false
}

// MustAgentConfiguration Deprecated Agent configuration settings.
//
// Ignored, because the deployment of OpenStack Neutron agent is dropped since {product-name} 4.4.0.
// The deployment of OpenStack hosts can be done by Red Hat OpenStack Platform Director or TripleO.
func (p *OpenStackNetworkProvider) MustAgentConfiguration() *AgentConfiguration {
	if p.agentConfiguration == nil {
		panic("the agentConfiguration must not be nil, please use AgentConfiguration() function instead")
	}
	return p.agentConfiguration
}

func (p *OpenStackNetworkProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *OpenStackNetworkProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *OpenStackNetworkProvider) SetAutoSync(attr bool) {
	p.autoSync = &attr
}

func (p *OpenStackNetworkProvider) AutoSync() (bool, bool) {
	if p.autoSync != nil {
		return *p.autoSync, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackNetworkProvider) MustAutoSync() bool {
	if p.autoSync == nil {
		panic("the autoSync must not be nil, please use AutoSync() function instead")
	}
	return *p.autoSync
}

func (p *OpenStackNetworkProvider) SetCertificates(attr *CertificateSlice) {
	p.certificates = attr
}

func (p *OpenStackNetworkProvider) Certificates() (*CertificateSlice, bool) {
	if p.certificates != nil {
		return p.certificates, true
	}
	return nil, false
}

func (p *OpenStackNetworkProvider) MustCertificates() *CertificateSlice {
	if p.certificates == nil {
		panic("the certificates must not be nil, please use Certificates() function instead")
	}
	return p.certificates
}

func (p *OpenStackNetworkProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackNetworkProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackNetworkProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackNetworkProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackNetworkProvider) SetExternalPluginType(attr string) {
	p.externalPluginType = &attr
}

func (p *OpenStackNetworkProvider) ExternalPluginType() (string, bool) {
	if p.externalPluginType != nil {
		return *p.externalPluginType, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustExternalPluginType() string {
	if p.externalPluginType == nil {
		panic("the externalPluginType must not be nil, please use ExternalPluginType() function instead")
	}
	return *p.externalPluginType
}

func (p *OpenStackNetworkProvider) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackNetworkProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackNetworkProvider) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackNetworkProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackNetworkProvider) SetNetworks(attr *OpenStackNetworkSlice) {
	p.networks = attr
}

func (p *OpenStackNetworkProvider) Networks() (*OpenStackNetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *OpenStackNetworkProvider) MustNetworks() *OpenStackNetworkSlice {
	if p.networks == nil {
		panic("the networks must not be nil, please use Networks() function instead")
	}
	return p.networks
}

func (p *OpenStackNetworkProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *OpenStackNetworkProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

// SetPluginType Network plug-in type.
//
// Specifies the provider driver implementation on the host.
//
// Since version 4.2 of the {engine-name}, this type has been deprecated in favour of the `external_plugin_type`
// attribute of the `OpenStackNetworkProvider` type.
func (p *OpenStackNetworkProvider) SetPluginType(attr NetworkPluginType) {
	p.pluginType = &attr
}

// PluginType Network plug-in type.
//
// Specifies the provider driver implementation on the host.
//
// Since version 4.2 of the {engine-name}, this type has been deprecated in favour of the `external_plugin_type`
// attribute of the `OpenStackNetworkProvider` type.
func (p *OpenStackNetworkProvider) PluginType() (NetworkPluginType, bool) {
	if p.pluginType != nil {
		return *p.pluginType, true
	}
	var zero NetworkPluginType
	return zero, false
}

// MustPluginType Network plug-in type.
//
// Specifies the provider driver implementation on the host.
//
// Since version 4.2 of the {engine-name}, this type has been deprecated in favour of the `external_plugin_type`
// attribute of the `OpenStackNetworkProvider` type.
func (p *OpenStackNetworkProvider) MustPluginType() NetworkPluginType {
	if p.pluginType == nil {
		panic("the pluginType must not be nil, please use PluginType() function instead")
	}
	return *p.pluginType
}

func (p *OpenStackNetworkProvider) SetProjectDomainName(attr string) {
	p.projectDomainName = &attr
}

func (p *OpenStackNetworkProvider) ProjectDomainName() (string, bool) {
	if p.projectDomainName != nil {
		return *p.projectDomainName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustProjectDomainName() string {
	if p.projectDomainName == nil {
		panic("the projectDomainName must not be nil, please use ProjectDomainName() function instead")
	}
	return *p.projectDomainName
}

func (p *OpenStackNetworkProvider) SetProjectName(attr string) {
	p.projectName = &attr
}

func (p *OpenStackNetworkProvider) ProjectName() (string, bool) {
	if p.projectName != nil {
		return *p.projectName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustProjectName() string {
	if p.projectName == nil {
		panic("the projectName must not be nil, please use ProjectName() function instead")
	}
	return *p.projectName
}

func (p *OpenStackNetworkProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *OpenStackNetworkProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *OpenStackNetworkProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *OpenStackNetworkProvider) SetReadOnly(attr bool) {
	p.readOnly = &attr
}

func (p *OpenStackNetworkProvider) ReadOnly() (bool, bool) {
	if p.readOnly != nil {
		return *p.readOnly, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackNetworkProvider) MustReadOnly() bool {
	if p.readOnly == nil {
		panic("the readOnly must not be nil, please use ReadOnly() function instead")
	}
	return *p.readOnly
}

func (p *OpenStackNetworkProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *OpenStackNetworkProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackNetworkProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *OpenStackNetworkProvider) SetSubnets(attr *OpenStackSubnetSlice) {
	p.subnets = attr
}

func (p *OpenStackNetworkProvider) Subnets() (*OpenStackSubnetSlice, bool) {
	if p.subnets != nil {
		return p.subnets, true
	}
	return nil, false
}

func (p *OpenStackNetworkProvider) MustSubnets() *OpenStackSubnetSlice {
	if p.subnets == nil {
		panic("the subnets must not be nil, please use Subnets() function instead")
	}
	return p.subnets
}

func (p *OpenStackNetworkProvider) SetTenantName(attr string) {
	p.tenantName = &attr
}

func (p *OpenStackNetworkProvider) TenantName() (string, bool) {
	if p.tenantName != nil {
		return *p.tenantName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustTenantName() string {
	if p.tenantName == nil {
		panic("the tenantName must not be nil, please use TenantName() function instead")
	}
	return *p.tenantName
}

// SetType The OpenStack network provider can either be implemented by OpenStack Neutron, in which case the Neutron
// agent is automatically installed on the hosts, or it can be an external provider implementing the
// OpenStack API, in which case the virtual interface driver is a custom solution installed manually.
func (p *OpenStackNetworkProvider) SetType(attr OpenStackNetworkProviderType) {
	p.type_ = &attr
}

// Type The OpenStack network provider can either be implemented by OpenStack Neutron, in which case the Neutron
// agent is automatically installed on the hosts, or it can be an external provider implementing the
// OpenStack API, in which case the virtual interface driver is a custom solution installed manually.
func (p *OpenStackNetworkProvider) Type() (OpenStackNetworkProviderType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero OpenStackNetworkProviderType
	return zero, false
}

// MustType The OpenStack network provider can either be implemented by OpenStack Neutron, in which case the Neutron
// agent is automatically installed on the hosts, or it can be an external provider implementing the
// OpenStack API, in which case the virtual interface driver is a custom solution installed manually.
func (p *OpenStackNetworkProvider) MustType() OpenStackNetworkProviderType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *OpenStackNetworkProvider) SetUnmanaged(attr bool) {
	p.unmanaged = &attr
}

func (p *OpenStackNetworkProvider) Unmanaged() (bool, bool) {
	if p.unmanaged != nil {
		return *p.unmanaged, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackNetworkProvider) MustUnmanaged() bool {
	if p.unmanaged == nil {
		panic("the unmanaged must not be nil, please use Unmanaged() function instead")
	}
	return *p.unmanaged
}

func (p *OpenStackNetworkProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *OpenStackNetworkProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *OpenStackNetworkProvider) SetUserDomainName(attr string) {
	p.userDomainName = &attr
}

func (p *OpenStackNetworkProvider) UserDomainName() (string, bool) {
	if p.userDomainName != nil {
		return *p.userDomainName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustUserDomainName() string {
	if p.userDomainName == nil {
		panic("the userDomainName must not be nil, please use UserDomainName() function instead")
	}
	return *p.userDomainName
}

func (p *OpenStackNetworkProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *OpenStackNetworkProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackNetworkProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

type OpenStackProvider struct {
	Struct
	authenticationUrl      *string
	comment                *string
	description            *string
	id                     *string
	name                   *string
	password               *string
	properties             *PropertySlice
	requiresAuthentication *bool
	tenantName             *string
	url                    *string
	username               *string
}

func (p *OpenStackProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *OpenStackProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *OpenStackProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackProvider) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackProvider) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *OpenStackProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *OpenStackProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *OpenStackProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *OpenStackProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *OpenStackProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *OpenStackProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *OpenStackProvider) SetTenantName(attr string) {
	p.tenantName = &attr
}

func (p *OpenStackProvider) TenantName() (string, bool) {
	if p.tenantName != nil {
		return *p.tenantName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustTenantName() string {
	if p.tenantName == nil {
		panic("the tenantName must not be nil, please use TenantName() function instead")
	}
	return *p.tenantName
}

func (p *OpenStackProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *OpenStackProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *OpenStackProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *OpenStackProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

type OpenStackSubnet struct {
	Struct
	cidr             *string
	comment          *string
	description      *string
	dnsServers       []string
	gateway          *string
	id               *string
	ipVersion        *string
	name             *string
	openstackNetwork *OpenStackNetwork
}

func (p *OpenStackSubnet) SetCidr(attr string) {
	p.cidr = &attr
}

func (p *OpenStackSubnet) Cidr() (string, bool) {
	if p.cidr != nil {
		return *p.cidr, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustCidr() string {
	if p.cidr == nil {
		panic("the cidr must not be nil, please use Cidr() function instead")
	}
	return *p.cidr
}

func (p *OpenStackSubnet) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackSubnet) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackSubnet) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackSubnet) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackSubnet) SetDnsServers(attr []string) {
	p.dnsServers = attr
}

func (p *OpenStackSubnet) DnsServers() ([]string, bool) {
	if p.dnsServers != nil {
		return p.dnsServers, true
	}
	return nil, false
}

func (p *OpenStackSubnet) MustDnsServers() []string {
	if p.dnsServers == nil {
		panic("the dnsServers must not be nil, please use DnsServers() function instead")
	}
	return p.dnsServers
}

func (p *OpenStackSubnet) SetGateway(attr string) {
	p.gateway = &attr
}

func (p *OpenStackSubnet) Gateway() (string, bool) {
	if p.gateway != nil {
		return *p.gateway, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustGateway() string {
	if p.gateway == nil {
		panic("the gateway must not be nil, please use Gateway() function instead")
	}
	return *p.gateway
}

func (p *OpenStackSubnet) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackSubnet) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackSubnet) SetIpVersion(attr string) {
	p.ipVersion = &attr
}

func (p *OpenStackSubnet) IpVersion() (string, bool) {
	if p.ipVersion != nil {
		return *p.ipVersion, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustIpVersion() string {
	if p.ipVersion == nil {
		panic("the ipVersion must not be nil, please use IpVersion() function instead")
	}
	return *p.ipVersion
}

func (p *OpenStackSubnet) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackSubnet) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackSubnet) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackSubnet) SetOpenstackNetwork(attr *OpenStackNetwork) {
	p.openstackNetwork = attr
}

func (p *OpenStackSubnet) OpenstackNetwork() (*OpenStackNetwork, bool) {
	if p.openstackNetwork != nil {
		return p.openstackNetwork, true
	}
	return nil, false
}

func (p *OpenStackSubnet) MustOpenstackNetwork() *OpenStackNetwork {
	if p.openstackNetwork == nil {
		panic("the openstackNetwork must not be nil, please use OpenstackNetwork() function instead")
	}
	return p.openstackNetwork
}

type OpenStackVolumeProvider struct {
	Struct
	authenticationKeys     *OpenstackVolumeAuthenticationKeySlice
	authenticationUrl      *string
	certificates           *CertificateSlice
	comment                *string
	dataCenter             *DataCenter
	description            *string
	id                     *string
	name                   *string
	password               *string
	properties             *PropertySlice
	requiresAuthentication *bool
	tenantName             *string
	url                    *string
	username               *string
	volumeTypes            *OpenStackVolumeTypeSlice
}

func (p *OpenStackVolumeProvider) SetAuthenticationKeys(attr *OpenstackVolumeAuthenticationKeySlice) {
	p.authenticationKeys = attr
}

func (p *OpenStackVolumeProvider) AuthenticationKeys() (*OpenstackVolumeAuthenticationKeySlice, bool) {
	if p.authenticationKeys != nil {
		return p.authenticationKeys, true
	}
	return nil, false
}

func (p *OpenStackVolumeProvider) MustAuthenticationKeys() *OpenstackVolumeAuthenticationKeySlice {
	if p.authenticationKeys == nil {
		panic("the authenticationKeys must not be nil, please use AuthenticationKeys() function instead")
	}
	return p.authenticationKeys
}

func (p *OpenStackVolumeProvider) SetAuthenticationUrl(attr string) {
	p.authenticationUrl = &attr
}

func (p *OpenStackVolumeProvider) AuthenticationUrl() (string, bool) {
	if p.authenticationUrl != nil {
		return *p.authenticationUrl, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustAuthenticationUrl() string {
	if p.authenticationUrl == nil {
		panic("the authenticationUrl must not be nil, please use AuthenticationUrl() function instead")
	}
	return *p.authenticationUrl
}

func (p *OpenStackVolumeProvider) SetCertificates(attr *CertificateSlice) {
	p.certificates = attr
}

func (p *OpenStackVolumeProvider) Certificates() (*CertificateSlice, bool) {
	if p.certificates != nil {
		return p.certificates, true
	}
	return nil, false
}

func (p *OpenStackVolumeProvider) MustCertificates() *CertificateSlice {
	if p.certificates == nil {
		panic("the certificates must not be nil, please use Certificates() function instead")
	}
	return p.certificates
}

func (p *OpenStackVolumeProvider) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackVolumeProvider) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackVolumeProvider) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *OpenStackVolumeProvider) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *OpenStackVolumeProvider) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *OpenStackVolumeProvider) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackVolumeProvider) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackVolumeProvider) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackVolumeProvider) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackVolumeProvider) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackVolumeProvider) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackVolumeProvider) SetPassword(attr string) {
	p.password = &attr
}

func (p *OpenStackVolumeProvider) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *OpenStackVolumeProvider) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *OpenStackVolumeProvider) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *OpenStackVolumeProvider) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

func (p *OpenStackVolumeProvider) SetRequiresAuthentication(attr bool) {
	p.requiresAuthentication = &attr
}

func (p *OpenStackVolumeProvider) RequiresAuthentication() (bool, bool) {
	if p.requiresAuthentication != nil {
		return *p.requiresAuthentication, true
	}
	var zero bool
	return zero, false
}

func (p *OpenStackVolumeProvider) MustRequiresAuthentication() bool {
	if p.requiresAuthentication == nil {
		panic("the requiresAuthentication must not be nil, please use RequiresAuthentication() function instead")
	}
	return *p.requiresAuthentication
}

func (p *OpenStackVolumeProvider) SetTenantName(attr string) {
	p.tenantName = &attr
}

func (p *OpenStackVolumeProvider) TenantName() (string, bool) {
	if p.tenantName != nil {
		return *p.tenantName, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustTenantName() string {
	if p.tenantName == nil {
		panic("the tenantName must not be nil, please use TenantName() function instead")
	}
	return *p.tenantName
}

func (p *OpenStackVolumeProvider) SetUrl(attr string) {
	p.url = &attr
}

func (p *OpenStackVolumeProvider) Url() (string, bool) {
	if p.url != nil {
		return *p.url, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustUrl() string {
	if p.url == nil {
		panic("the url must not be nil, please use Url() function instead")
	}
	return *p.url
}

func (p *OpenStackVolumeProvider) SetUsername(attr string) {
	p.username = &attr
}

func (p *OpenStackVolumeProvider) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeProvider) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

func (p *OpenStackVolumeProvider) SetVolumeTypes(attr *OpenStackVolumeTypeSlice) {
	p.volumeTypes = attr
}

func (p *OpenStackVolumeProvider) VolumeTypes() (*OpenStackVolumeTypeSlice, bool) {
	if p.volumeTypes != nil {
		return p.volumeTypes, true
	}
	return nil, false
}

func (p *OpenStackVolumeProvider) MustVolumeTypes() *OpenStackVolumeTypeSlice {
	if p.volumeTypes == nil {
		panic("the volumeTypes must not be nil, please use VolumeTypes() function instead")
	}
	return p.volumeTypes
}

type OpenStackVolumeType struct {
	Struct
	comment                 *string
	description             *string
	id                      *string
	name                    *string
	openstackVolumeProvider *OpenStackVolumeProvider
	properties              *PropertySlice
}

func (p *OpenStackVolumeType) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenStackVolumeType) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeType) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenStackVolumeType) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenStackVolumeType) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeType) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenStackVolumeType) SetId(attr string) {
	p.id = &attr
}

func (p *OpenStackVolumeType) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeType) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenStackVolumeType) SetName(attr string) {
	p.name = &attr
}

func (p *OpenStackVolumeType) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenStackVolumeType) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenStackVolumeType) SetOpenstackVolumeProvider(attr *OpenStackVolumeProvider) {
	p.openstackVolumeProvider = attr
}

func (p *OpenStackVolumeType) OpenstackVolumeProvider() (*OpenStackVolumeProvider, bool) {
	if p.openstackVolumeProvider != nil {
		return p.openstackVolumeProvider, true
	}
	return nil, false
}

func (p *OpenStackVolumeType) MustOpenstackVolumeProvider() *OpenStackVolumeProvider {
	if p.openstackVolumeProvider == nil {
		panic("the openstackVolumeProvider must not be nil, please use OpenstackVolumeProvider() function instead")
	}
	return p.openstackVolumeProvider
}

func (p *OpenStackVolumeType) SetProperties(attr *PropertySlice) {
	p.properties = attr
}

func (p *OpenStackVolumeType) Properties() (*PropertySlice, bool) {
	if p.properties != nil {
		return p.properties, true
	}
	return nil, false
}

func (p *OpenStackVolumeType) MustProperties() *PropertySlice {
	if p.properties == nil {
		panic("the properties must not be nil, please use Properties() function instead")
	}
	return p.properties
}

type OpenstackVolumeAuthenticationKey struct {
	Struct
	comment                 *string
	creationDate            *time.Time
	description             *string
	id                      *string
	name                    *string
	openstackVolumeProvider *OpenStackVolumeProvider
	usageType               *OpenstackVolumeAuthenticationKeyUsageType
	uuid                    *string
	value                   *string
}

func (p *OpenstackVolumeAuthenticationKey) SetComment(attr string) {
	p.comment = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OpenstackVolumeAuthenticationKey) SetCreationDate(attr time.Time) {
	p.creationDate = &attr
}

func (p *OpenstackVolumeAuthenticationKey) CreationDate() (time.Time, bool) {
	if p.creationDate != nil {
		return *p.creationDate, true
	}
	var zero time.Time
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustCreationDate() time.Time {
	if p.creationDate == nil {
		panic("the creationDate must not be nil, please use CreationDate() function instead")
	}
	return *p.creationDate
}

func (p *OpenstackVolumeAuthenticationKey) SetDescription(attr string) {
	p.description = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OpenstackVolumeAuthenticationKey) SetId(attr string) {
	p.id = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *OpenstackVolumeAuthenticationKey) SetName(attr string) {
	p.name = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *OpenstackVolumeAuthenticationKey) SetOpenstackVolumeProvider(attr *OpenStackVolumeProvider) {
	p.openstackVolumeProvider = attr
}

func (p *OpenstackVolumeAuthenticationKey) OpenstackVolumeProvider() (*OpenStackVolumeProvider, bool) {
	if p.openstackVolumeProvider != nil {
		return p.openstackVolumeProvider, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKey) MustOpenstackVolumeProvider() *OpenStackVolumeProvider {
	if p.openstackVolumeProvider == nil {
		panic("the openstackVolumeProvider must not be nil, please use OpenstackVolumeProvider() function instead")
	}
	return p.openstackVolumeProvider
}

func (p *OpenstackVolumeAuthenticationKey) SetUsageType(attr OpenstackVolumeAuthenticationKeyUsageType) {
	p.usageType = &attr
}

func (p *OpenstackVolumeAuthenticationKey) UsageType() (OpenstackVolumeAuthenticationKeyUsageType, bool) {
	if p.usageType != nil {
		return *p.usageType, true
	}
	var zero OpenstackVolumeAuthenticationKeyUsageType
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustUsageType() OpenstackVolumeAuthenticationKeyUsageType {
	if p.usageType == nil {
		panic("the usageType must not be nil, please use UsageType() function instead")
	}
	return *p.usageType
}

func (p *OpenstackVolumeAuthenticationKey) SetUuid(attr string) {
	p.uuid = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Uuid() (string, bool) {
	if p.uuid != nil {
		return *p.uuid, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustUuid() string {
	if p.uuid == nil {
		panic("the uuid must not be nil, please use Uuid() function instead")
	}
	return *p.uuid
}

func (p *OpenstackVolumeAuthenticationKey) SetValue(attr string) {
	p.value = &attr
}

func (p *OpenstackVolumeAuthenticationKey) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *OpenstackVolumeAuthenticationKey) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

// OperatingSystem Information describing the operating system. This is used for both virtual machines and hosts.
type OperatingSystem struct {
	Struct
	boot                  *Boot
	cmdline               *string
	customKernelCmdline   *string
	initrd                *string
	kernel                *string
	reportedKernelCmdline *string
	type_                 *string
	version               *Version
}

// SetBoot Configuration of the boot sequence of a virtual machine.
func (p *OperatingSystem) SetBoot(attr *Boot) {
	p.boot = attr
}

// Boot Configuration of the boot sequence of a virtual machine.
func (p *OperatingSystem) Boot() (*Boot, bool) {
	if p.boot != nil {
		return p.boot, true
	}
	return nil, false
}

// MustBoot Configuration of the boot sequence of a virtual machine.
func (p *OperatingSystem) MustBoot() *Boot {
	if p.boot == nil {
		panic("the boot must not be nil, please use Boot() function instead")
	}
	return p.boot
}

func (p *OperatingSystem) SetCmdline(attr string) {
	p.cmdline = &attr
}

func (p *OperatingSystem) Cmdline() (string, bool) {
	if p.cmdline != nil {
		return *p.cmdline, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustCmdline() string {
	if p.cmdline == nil {
		panic("the cmdline must not be nil, please use Cmdline() function instead")
	}
	return *p.cmdline
}

func (p *OperatingSystem) SetCustomKernelCmdline(attr string) {
	p.customKernelCmdline = &attr
}

func (p *OperatingSystem) CustomKernelCmdline() (string, bool) {
	if p.customKernelCmdline != nil {
		return *p.customKernelCmdline, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustCustomKernelCmdline() string {
	if p.customKernelCmdline == nil {
		panic("the customKernelCmdline must not be nil, please use CustomKernelCmdline() function instead")
	}
	return *p.customKernelCmdline
}

func (p *OperatingSystem) SetInitrd(attr string) {
	p.initrd = &attr
}

func (p *OperatingSystem) Initrd() (string, bool) {
	if p.initrd != nil {
		return *p.initrd, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustInitrd() string {
	if p.initrd == nil {
		panic("the initrd must not be nil, please use Initrd() function instead")
	}
	return *p.initrd
}

func (p *OperatingSystem) SetKernel(attr string) {
	p.kernel = &attr
}

func (p *OperatingSystem) Kernel() (string, bool) {
	if p.kernel != nil {
		return *p.kernel, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustKernel() string {
	if p.kernel == nil {
		panic("the kernel must not be nil, please use Kernel() function instead")
	}
	return *p.kernel
}

func (p *OperatingSystem) SetReportedKernelCmdline(attr string) {
	p.reportedKernelCmdline = &attr
}

func (p *OperatingSystem) ReportedKernelCmdline() (string, bool) {
	if p.reportedKernelCmdline != nil {
		return *p.reportedKernelCmdline, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustReportedKernelCmdline() string {
	if p.reportedKernelCmdline == nil {
		panic("the reportedKernelCmdline must not be nil, please use ReportedKernelCmdline() function instead")
	}
	return *p.reportedKernelCmdline
}

func (p *OperatingSystem) SetType(attr string) {
	p.type_ = &attr
}

func (p *OperatingSystem) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystem) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *OperatingSystem) SetVersion(attr *Version) {
	p.version = attr
}

func (p *OperatingSystem) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *OperatingSystem) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

// OperatingSystemInfo Represents a guest operating system.
type OperatingSystemInfo struct {
	Struct
	architecture *Architecture
	comment      *string
	description  *string
	id           *string
	largeIcon    *Icon
	name         *string
	smallIcon    *Icon
}

func (p *OperatingSystemInfo) SetArchitecture(attr Architecture) {
	p.architecture = &attr
}

func (p *OperatingSystemInfo) Architecture() (Architecture, bool) {
	if p.architecture != nil {
		return *p.architecture, true
	}
	var zero Architecture
	return zero, false
}

func (p *OperatingSystemInfo) MustArchitecture() Architecture {
	if p.architecture == nil {
		panic("the architecture must not be nil, please use Architecture() function instead")
	}
	return *p.architecture
}

func (p *OperatingSystemInfo) SetComment(attr string) {
	p.comment = &attr
}

func (p *OperatingSystemInfo) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystemInfo) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *OperatingSystemInfo) SetDescription(attr string) {
	p.description = &attr
}

func (p *OperatingSystemInfo) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystemInfo) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *OperatingSystemInfo) SetId(attr string) {
	p.id = &attr
}

func (p *OperatingSystemInfo) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystemInfo) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

// SetLargeIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) SetLargeIcon(attr *Icon) {
	p.largeIcon = attr
}

// LargeIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) LargeIcon() (*Icon, bool) {
	if p.largeIcon != nil {
		return p.largeIcon, true
	}
	return nil, false
}

// MustLargeIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) MustLargeIcon() *Icon {
	if p.largeIcon == nil {
		panic("the largeIcon must not be nil, please use LargeIcon() function instead")
	}
	return p.largeIcon
}

func (p *OperatingSystemInfo) SetName(attr string) {
	p.name = &attr
}

func (p *OperatingSystemInfo) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *OperatingSystemInfo) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetSmallIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) SetSmallIcon(attr *Icon) {
	p.smallIcon = attr
}

// SmallIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) SmallIcon() (*Icon, bool) {
	if p.smallIcon != nil {
		return p.smallIcon, true
	}
	return nil, false
}

// MustSmallIcon Icon of virtual machine or template.
func (p *OperatingSystemInfo) MustSmallIcon() *Icon {
	if p.smallIcon == nil {
		panic("the smallIcon must not be nil, please use SmallIcon() function instead")
	}
	return p.smallIcon
}

type Option struct {
	Struct
	name  *string
	type_ *string
	value *string
}

func (p *Option) SetName(attr string) {
	p.name = &attr
}

func (p *Option) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Option) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Option) SetType(attr string) {
	p.type_ = &attr
}

func (p *Option) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *Option) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *Option) SetValue(attr string) {
	p.value = &attr
}

func (p *Option) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *Option) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

// Package Type representing a package.
//
// This is an example of the package element:
//
// [source,xml]
// ----
// <package>
//   <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
// </package>
// ----
type Package struct {
	Struct
	name *string
}

func (p *Package) SetName(attr string) {
	p.name = &attr
}

func (p *Package) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Package) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

type Payload struct {
	Struct
	files    *FileSlice
	type_    *VmDeviceType
	volumeId *string
}

func (p *Payload) SetFiles(attr *FileSlice) {
	p.files = attr
}

func (p *Payload) Files() (*FileSlice, bool) {
	if p.files != nil {
		return p.files, true
	}
	return nil, false
}

func (p *Payload) MustFiles() *FileSlice {
	if p.files == nil {
		panic("the files must not be nil, please use Files() function instead")
	}
	return p.files
}

func (p *Payload) SetType(attr VmDeviceType) {
	p.type_ = &attr
}

func (p *Payload) Type() (VmDeviceType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero VmDeviceType
	return zero, false
}

func (p *Payload) MustType() VmDeviceType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *Payload) SetVolumeId(attr string) {
	p.volumeId = &attr
}

func (p *Payload) VolumeId() (string, bool) {
	if p.volumeId != nil {
		return *p.volumeId, true
	}
	var zero string
	return zero, false
}

func (p *Payload) MustVolumeId() string {
	if p.volumeId == nil {
		panic("the volumeId must not be nil, please use VolumeId() function instead")
	}
	return *p.volumeId
}

// Permission Type represents a permission.
type Permission struct {
	Struct
	cluster       *Cluster
	comment       *string
	dataCenter    *DataCenter
	description   *string
	disk          *Disk
	group         *Group
	host          *Host
	id            *string
	name          *string
	role          *Role
	storageDomain *StorageDomain
	template      *Template
	user          *User
	vm            *Vm
	vmPool        *VmPool
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Permission) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Permission) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *Permission) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *Permission) SetComment(attr string) {
	p.comment = &attr
}

func (p *Permission) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Permission) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Permission) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Permission) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Permission) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Permission) SetDescription(attr string) {
	p.description = &attr
}

func (p *Permission) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Permission) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

// SetDisk Represents a virtual disk device.
func (p *Permission) SetDisk(attr *Disk) {
	p.disk = attr
}

// Disk Represents a virtual disk device.
func (p *Permission) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

// MustDisk Represents a virtual disk device.
func (p *Permission) MustDisk() *Disk {
	if p.disk == nil {
		panic("the disk must not be nil, please use Disk() function instead")
	}
	return p.disk
}

// SetGroup This type represents all groups in the directory service.
func (p *Permission) SetGroup(attr *Group) {
	p.group = attr
}

// Group This type represents all groups in the directory service.
func (p *Permission) Group() (*Group, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

// MustGroup This type represents all groups in the directory service.
func (p *Permission) MustGroup() *Group {
	if p.group == nil {
		panic("the group must not be nil, please use Group() function instead")
	}
	return p.group
}

// SetHost Type representing a host.
func (p *Permission) SetHost(attr *Host) {
	p.host = attr
}

// Host Type representing a host.
func (p *Permission) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

// MustHost Type representing a host.
func (p *Permission) MustHost() *Host {
	if p.host == nil {
		panic("the host must not be nil, please use Host() function instead")
	}
	return p.host
}

func (p *Permission) SetId(attr string) {
	p.id = &attr
}

func (p *Permission) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Permission) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Permission) SetName(attr string) {
	p.name = &attr
}

func (p *Permission) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Permission) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetRole Represents a system role.
func (p *Permission) SetRole(attr *Role) {
	p.role = attr
}

// Role Represents a system role.
func (p *Permission) Role() (*Role, bool) {
	if p.role != nil {
		return p.role, true
	}
	return nil, false
}

// MustRole Represents a system role.
func (p *Permission) MustRole() *Role {
	if p.role == nil {
		panic("the role must not be nil, please use Role() function instead")
	}
	return p.role
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Permission) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Permission) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *Permission) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

// SetTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Permission) SetTemplate(attr *Template) {
	p.template = attr
}

// Template The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Permission) Template() (*Template, bool) {
	if p.template != nil {
		return p.template, true
	}
	return nil, false
}

// MustTemplate The type that represents a virtual machine template.
// Templates allow for a rapid instantiation of virtual machines with common configuration and disk states.
func (p *Permission) MustTemplate() *Template {
	if p.template == nil {
		panic("the template must not be nil, please use Template() function instead")
	}
	return p.template
}

// SetUser Represents a user in the system.
func (p *Permission) SetUser(attr *User) {
	p.user = attr
}

// User Represents a user in the system.
func (p *Permission) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

// MustUser Represents a user in the system.
func (p *Permission) MustUser() *User {
	if p.user == nil {
		panic("the user must not be nil, please use User() function instead")
	}
	return p.user
}

// SetVm Represents a virtual machine.
func (p *Permission) SetVm(attr *Vm) {
	p.vm = attr
}

// Vm Represents a virtual machine.
func (p *Permission) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

// MustVm Represents a virtual machine.
func (p *Permission) MustVm() *Vm {
	if p.vm == nil {
		panic("the vm must not be nil, please use Vm() function instead")
	}
	return p.vm
}

// SetVmPool Type represeting a virtual machines pool.
func (p *Permission) SetVmPool(attr *VmPool) {
	p.vmPool = attr
}

// VmPool Type represeting a virtual machines pool.
func (p *Permission) VmPool() (*VmPool, bool) {
	if p.vmPool != nil {
		return p.vmPool, true
	}
	return nil, false
}

// MustVmPool Type represeting a virtual machines pool.
func (p *Permission) MustVmPool() *VmPool {
	if p.vmPool == nil {
		panic("the vmPool must not be nil, please use VmPool() function instead")
	}
	return p.vmPool
}

// Permit Type represents a permit.
type Permit struct {
	Struct
	administrative *bool
	comment        *string
	description    *string
	id             *string
	name           *string
	role           *Role
}

func (p *Permit) SetAdministrative(attr bool) {
	p.administrative = &attr
}

func (p *Permit) Administrative() (bool, bool) {
	if p.administrative != nil {
		return *p.administrative, true
	}
	var zero bool
	return zero, false
}

func (p *Permit) MustAdministrative() bool {
	if p.administrative == nil {
		panic("the administrative must not be nil, please use Administrative() function instead")
	}
	return *p.administrative
}

func (p *Permit) SetComment(attr string) {
	p.comment = &attr
}

func (p *Permit) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Permit) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Permit) SetDescription(attr string) {
	p.description = &attr
}

func (p *Permit) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Permit) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Permit) SetId(attr string) {
	p.id = &attr
}

func (p *Permit) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Permit) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Permit) SetName(attr string) {
	p.name = &attr
}

func (p *Permit) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Permit) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetRole Represents a system role.
func (p *Permit) SetRole(attr *Role) {
	p.role = attr
}

// Role Represents a system role.
func (p *Permit) Role() (*Role, bool) {
	if p.role != nil {
		return p.role, true
	}
	return nil, false
}

// MustRole Represents a system role.
func (p *Permit) MustRole() *Role {
	if p.role == nil {
		panic("the role must not be nil, please use Role() function instead")
	}
	return p.role
}

type PmProxy struct {
	Struct
	type_ *PmProxyType
}

func (p *PmProxy) SetType(attr PmProxyType) {
	p.type_ = &attr
}

func (p *PmProxy) Type() (PmProxyType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero PmProxyType
	return zero, false
}

func (p *PmProxy) MustType() PmProxyType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

type PortMirroring struct {
	Struct
}

type PowerManagement struct {
	Struct
	address            *string
	agents             *AgentSlice
	automaticPmEnabled *bool
	enabled            *bool
	kdumpDetection     *bool
	options            *OptionSlice
	password           *string
	pmProxies          *PmProxySlice
	status             *PowerManagementStatus
	type_              *string
	username           *string
}

func (p *PowerManagement) SetAddress(attr string) {
	p.address = &attr
}

func (p *PowerManagement) Address() (string, bool) {
	if p.address != nil {
		return *p.address, true
	}
	var zero string
	return zero, false
}

func (p *PowerManagement) MustAddress() string {
	if p.address == nil {
		panic("the address must not be nil, please use Address() function instead")
	}
	return *p.address
}

func (p *PowerManagement) SetAgents(attr *AgentSlice) {
	p.agents = attr
}

func (p *PowerManagement) Agents() (*AgentSlice, bool) {
	if p.agents != nil {
		return p.agents, true
	}
	return nil, false
}

func (p *PowerManagement) MustAgents() *AgentSlice {
	if p.agents == nil {
		panic("the agents must not be nil, please use Agents() function instead")
	}
	return p.agents
}

func (p *PowerManagement) SetAutomaticPmEnabled(attr bool) {
	p.automaticPmEnabled = &attr
}

func (p *PowerManagement) AutomaticPmEnabled() (bool, bool) {
	if p.automaticPmEnabled != nil {
		return *p.automaticPmEnabled, true
	}
	var zero bool
	return zero, false
}

func (p *PowerManagement) MustAutomaticPmEnabled() bool {
	if p.automaticPmEnabled == nil {
		panic("the automaticPmEnabled must not be nil, please use AutomaticPmEnabled() function instead")
	}
	return *p.automaticPmEnabled
}

func (p *PowerManagement) SetEnabled(attr bool) {
	p.enabled = &attr
}

func (p *PowerManagement) Enabled() (bool, bool) {
	if p.enabled != nil {
		return *p.enabled, true
	}
	var zero bool
	return zero, false
}

func (p *PowerManagement) MustEnabled() bool {
	if p.enabled == nil {
		panic("the enabled must not be nil, please use Enabled() function instead")
	}
	return *p.enabled
}

func (p *PowerManagement) SetKdumpDetection(attr bool) {
	p.kdumpDetection = &attr
}

func (p *PowerManagement) KdumpDetection() (bool, bool) {
	if p.kdumpDetection != nil {
		return *p.kdumpDetection, true
	}
	var zero bool
	return zero, false
}

func (p *PowerManagement) MustKdumpDetection() bool {
	if p.kdumpDetection == nil {
		panic("the kdumpDetection must not be nil, please use KdumpDetection() function instead")
	}
	return *p.kdumpDetection
}

func (p *PowerManagement) SetOptions(attr *OptionSlice) {
	p.options = attr
}

func (p *PowerManagement) Options() (*OptionSlice, bool) {
	if p.options != nil {
		return p.options, true
	}
	return nil, false
}

func (p *PowerManagement) MustOptions() *OptionSlice {
	if p.options == nil {
		panic("the options must not be nil, please use Options() function instead")
	}
	return p.options
}

func (p *PowerManagement) SetPassword(attr string) {
	p.password = &attr
}

func (p *PowerManagement) Password() (string, bool) {
	if p.password != nil {
		return *p.password, true
	}
	var zero string
	return zero, false
}

func (p *PowerManagement) MustPassword() string {
	if p.password == nil {
		panic("the password must not be nil, please use Password() function instead")
	}
	return *p.password
}

func (p *PowerManagement) SetPmProxies(attr *PmProxySlice) {
	p.pmProxies = attr
}

func (p *PowerManagement) PmProxies() (*PmProxySlice, bool) {
	if p.pmProxies != nil {
		return p.pmProxies, true
	}
	return nil, false
}

func (p *PowerManagement) MustPmProxies() *PmProxySlice {
	if p.pmProxies == nil {
		panic("the pmProxies must not be nil, please use PmProxies() function instead")
	}
	return p.pmProxies
}

func (p *PowerManagement) SetStatus(attr PowerManagementStatus) {
	p.status = &attr
}

func (p *PowerManagement) Status() (PowerManagementStatus, bool) {
	if p.status != nil {
		return *p.status, true
	}
	var zero PowerManagementStatus
	return zero, false
}

func (p *PowerManagement) MustStatus() PowerManagementStatus {
	if p.status == nil {
		panic("the status must not be nil, please use Status() function instead")
	}
	return *p.status
}

func (p *PowerManagement) SetType(attr string) {
	p.type_ = &attr
}

func (p *PowerManagement) Type() (string, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero string
	return zero, false
}

func (p *PowerManagement) MustType() string {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

func (p *PowerManagement) SetUsername(attr string) {
	p.username = &attr
}

func (p *PowerManagement) Username() (string, bool) {
	if p.username != nil {
		return *p.username, true
	}
	var zero string
	return zero, false
}

func (p *PowerManagement) MustUsername() string {
	if p.username == nil {
		panic("the username must not be nil, please use Username() function instead")
	}
	return *p.username
}

type Product struct {
	Struct
	comment     *string
	description *string
	id          *string
	name        *string
}

func (p *Product) SetComment(attr string) {
	p.comment = &attr
}

func (p *Product) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Product) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Product) SetDescription(attr string) {
	p.description = &attr
}

func (p *Product) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Product) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Product) SetId(attr string) {
	p.id = &attr
}

func (p *Product) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Product) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Product) SetName(attr string) {
	p.name = &attr
}

func (p *Product) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Product) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// ProductInfo Product information.
//
// The entry point contains a `product_info` element to help an API user determine the legitimacy of the
// {product-name} environment. This includes the name of the product, the `vendor` and the `version`.
//
// Verify a genuine {product-name} environment
//
// The follow elements identify a genuine {product-name} environment:
//
// [source]
// ----
// <api>
// ...
// <product_info>
//   <name>oVirt Engine</name>
//   <vendor>ovirt.org</vendor>
//   <version>
//     <build>0</build>
//     <full_version>4.1.0_master</full_version>
//     <major>4</major>
//     <minor>1</minor>
//     <revision>0</revision>
//   </version>
// </product_info>
// ...
// </api>
// ----
type ProductInfo struct {
	Struct
	instanceId *string
	name       *string
	vendor     *string
	version    *Version
}

func (p *ProductInfo) SetInstanceId(attr string) {
	p.instanceId = &attr
}

func (p *ProductInfo) InstanceId() (string, bool) {
	if p.instanceId != nil {
		return *p.instanceId, true
	}
	var zero string
	return zero, false
}

func (p *ProductInfo) MustInstanceId() string {
	if p.instanceId == nil {
		panic("the instanceId must not be nil, please use InstanceId() function instead")
	}
	return *p.instanceId
}

func (p *ProductInfo) SetName(attr string) {
	p.name = &attr
}

func (p *ProductInfo) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *ProductInfo) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *ProductInfo) SetVendor(attr string) {
	p.vendor = &attr
}

func (p *ProductInfo) Vendor() (string, bool) {
	if p.vendor != nil {
		return *p.vendor, true
	}
	var zero string
	return zero, false
}

func (p *ProductInfo) MustVendor() string {
	if p.vendor == nil {
		panic("the vendor must not be nil, please use Vendor() function instead")
	}
	return *p.vendor
}

func (p *ProductInfo) SetVersion(attr *Version) {
	p.version = attr
}

func (p *ProductInfo) Version() (*Version, bool) {
	if p.version != nil {
		return p.version, true
	}
	return nil, false
}

func (p *ProductInfo) MustVersion() *Version {
	if p.version == nil {
		panic("the version must not be nil, please use Version() function instead")
	}
	return p.version
}

type ProfileDetail struct {
	Struct
	blockStatistics *BlockStatisticSlice
	duration        *int64
	fopStatistics   *FopStatisticSlice
	profileType     *string
	statistics      *StatisticSlice
}

func (p *ProfileDetail) SetBlockStatistics(attr *BlockStatisticSlice) {
	p.blockStatistics = attr
}

func (p *ProfileDetail) BlockStatistics() (*BlockStatisticSlice, bool) {
	if p.blockStatistics != nil {
		return p.blockStatistics, true
	}
	return nil, false
}

func (p *ProfileDetail) MustBlockStatistics() *BlockStatisticSlice {
	if p.blockStatistics == nil {
		panic("the blockStatistics must not be nil, please use BlockStatistics() function instead")
	}
	return p.blockStatistics
}

func (p *ProfileDetail) SetDuration(attr int64) {
	p.duration = &attr
}

func (p *ProfileDetail) Duration() (int64, bool) {
	if p.duration != nil {
		return *p.duration, true
	}
	var zero int64
	return zero, false
}

func (p *ProfileDetail) MustDuration() int64 {
	if p.duration == nil {
		panic("the duration must not be nil, please use Duration() function instead")
	}
	return *p.duration
}

func (p *ProfileDetail) SetFopStatistics(attr *FopStatisticSlice) {
	p.fopStatistics = attr
}

func (p *ProfileDetail) FopStatistics() (*FopStatisticSlice, bool) {
	if p.fopStatistics != nil {
		return p.fopStatistics, true
	}
	return nil, false
}

func (p *ProfileDetail) MustFopStatistics() *FopStatisticSlice {
	if p.fopStatistics == nil {
		panic("the fopStatistics must not be nil, please use FopStatistics() function instead")
	}
	return p.fopStatistics
}

func (p *ProfileDetail) SetProfileType(attr string) {
	p.profileType = &attr
}

func (p *ProfileDetail) ProfileType() (string, bool) {
	if p.profileType != nil {
		return *p.profileType, true
	}
	var zero string
	return zero, false
}

func (p *ProfileDetail) MustProfileType() string {
	if p.profileType == nil {
		panic("the profileType must not be nil, please use ProfileType() function instead")
	}
	return *p.profileType
}

func (p *ProfileDetail) SetStatistics(attr *StatisticSlice) {
	p.statistics = attr
}

func (p *ProfileDetail) Statistics() (*StatisticSlice, bool) {
	if p.statistics != nil {
		return p.statistics, true
	}
	return nil, false
}

func (p *ProfileDetail) MustStatistics() *StatisticSlice {
	if p.statistics == nil {
		panic("the statistics must not be nil, please use Statistics() function instead")
	}
	return p.statistics
}

type Property struct {
	Struct
	name  *string
	value *string
}

func (p *Property) SetName(attr string) {
	p.name = &attr
}

func (p *Property) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Property) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Property) SetValue(attr string) {
	p.value = &attr
}

func (p *Property) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *Property) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

type ProxyTicket struct {
	Struct
	value *string
}

func (p *ProxyTicket) SetValue(attr string) {
	p.value = &attr
}

func (p *ProxyTicket) Value() (string, bool) {
	if p.value != nil {
		return *p.value, true
	}
	var zero string
	return zero, false
}

func (p *ProxyTicket) MustValue() string {
	if p.value == nil {
		panic("the value must not be nil, please use Value() function instead")
	}
	return *p.value
}

// Qos This type represents the attributes to define Quality of service (QoS).
//
// For storage the `type` is <<types/qos_type, storage>>, the attributes `max_throughput`, `max_read_throughput`,
// `max_write_throughput`, `max_iops`, `max_read_iops` and `max_write_iops` are relevant.
//
// For resources with computing capabilities the `type` is <<types/qos_type, cpu>>, the attribute `cpu_limit` is
// relevant.
//
// For virtual machines networks the `type` is <<types/qos_type, network>>, the attributes `inbound_average`,
// `inbound_peak`, `inbound_burst`, `outbound_average`, `outbound_peak` and `outbound_burst` are relevant.
//
// For host networks the `type` is <<types/qos_type, hostnetwork>>, the attributes `outbound_average_linkshare`,
// `outbound_average_upperlimit` and `outbound_average_realtime` are relevant.
type Qos struct {
	Struct
	comment                   *string
	cpuLimit                  *int64
	dataCenter                *DataCenter
	description               *string
	id                        *string
	inboundAverage            *int64
	inboundBurst              *int64
	inboundPeak               *int64
	maxIops                   *int64
	maxReadIops               *int64
	maxReadThroughput         *int64
	maxThroughput             *int64
	maxWriteIops              *int64
	maxWriteThroughput        *int64
	name                      *string
	outboundAverage           *int64
	outboundAverageLinkshare  *int64
	outboundAverageRealtime   *int64
	outboundAverageUpperlimit *int64
	outboundBurst             *int64
	outboundPeak              *int64
	type_                     *QosType
}

func (p *Qos) SetComment(attr string) {
	p.comment = &attr
}

func (p *Qos) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Qos) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Qos) SetCpuLimit(attr int64) {
	p.cpuLimit = &attr
}

func (p *Qos) CpuLimit() (int64, bool) {
	if p.cpuLimit != nil {
		return *p.cpuLimit, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustCpuLimit() int64 {
	if p.cpuLimit == nil {
		panic("the cpuLimit must not be nil, please use CpuLimit() function instead")
	}
	return *p.cpuLimit
}

func (p *Qos) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Qos) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Qos) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Qos) SetDescription(attr string) {
	p.description = &attr
}

func (p *Qos) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Qos) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Qos) SetId(attr string) {
	p.id = &attr
}

func (p *Qos) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Qos) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Qos) SetInboundAverage(attr int64) {
	p.inboundAverage = &attr
}

func (p *Qos) InboundAverage() (int64, bool) {
	if p.inboundAverage != nil {
		return *p.inboundAverage, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustInboundAverage() int64 {
	if p.inboundAverage == nil {
		panic("the inboundAverage must not be nil, please use InboundAverage() function instead")
	}
	return *p.inboundAverage
}

func (p *Qos) SetInboundBurst(attr int64) {
	p.inboundBurst = &attr
}

func (p *Qos) InboundBurst() (int64, bool) {
	if p.inboundBurst != nil {
		return *p.inboundBurst, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustInboundBurst() int64 {
	if p.inboundBurst == nil {
		panic("the inboundBurst must not be nil, please use InboundBurst() function instead")
	}
	return *p.inboundBurst
}

func (p *Qos) SetInboundPeak(attr int64) {
	p.inboundPeak = &attr
}

func (p *Qos) InboundPeak() (int64, bool) {
	if p.inboundPeak != nil {
		return *p.inboundPeak, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustInboundPeak() int64 {
	if p.inboundPeak == nil {
		panic("the inboundPeak must not be nil, please use InboundPeak() function instead")
	}
	return *p.inboundPeak
}

func (p *Qos) SetMaxIops(attr int64) {
	p.maxIops = &attr
}

func (p *Qos) MaxIops() (int64, bool) {
	if p.maxIops != nil {
		return *p.maxIops, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxIops() int64 {
	if p.maxIops == nil {
		panic("the maxIops must not be nil, please use MaxIops() function instead")
	}
	return *p.maxIops
}

func (p *Qos) SetMaxReadIops(attr int64) {
	p.maxReadIops = &attr
}

func (p *Qos) MaxReadIops() (int64, bool) {
	if p.maxReadIops != nil {
		return *p.maxReadIops, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxReadIops() int64 {
	if p.maxReadIops == nil {
		panic("the maxReadIops must not be nil, please use MaxReadIops() function instead")
	}
	return *p.maxReadIops
}

func (p *Qos) SetMaxReadThroughput(attr int64) {
	p.maxReadThroughput = &attr
}

func (p *Qos) MaxReadThroughput() (int64, bool) {
	if p.maxReadThroughput != nil {
		return *p.maxReadThroughput, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxReadThroughput() int64 {
	if p.maxReadThroughput == nil {
		panic("the maxReadThroughput must not be nil, please use MaxReadThroughput() function instead")
	}
	return *p.maxReadThroughput
}

func (p *Qos) SetMaxThroughput(attr int64) {
	p.maxThroughput = &attr
}

func (p *Qos) MaxThroughput() (int64, bool) {
	if p.maxThroughput != nil {
		return *p.maxThroughput, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxThroughput() int64 {
	if p.maxThroughput == nil {
		panic("the maxThroughput must not be nil, please use MaxThroughput() function instead")
	}
	return *p.maxThroughput
}

func (p *Qos) SetMaxWriteIops(attr int64) {
	p.maxWriteIops = &attr
}

func (p *Qos) MaxWriteIops() (int64, bool) {
	if p.maxWriteIops != nil {
		return *p.maxWriteIops, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxWriteIops() int64 {
	if p.maxWriteIops == nil {
		panic("the maxWriteIops must not be nil, please use MaxWriteIops() function instead")
	}
	return *p.maxWriteIops
}

func (p *Qos) SetMaxWriteThroughput(attr int64) {
	p.maxWriteThroughput = &attr
}

func (p *Qos) MaxWriteThroughput() (int64, bool) {
	if p.maxWriteThroughput != nil {
		return *p.maxWriteThroughput, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustMaxWriteThroughput() int64 {
	if p.maxWriteThroughput == nil {
		panic("the maxWriteThroughput must not be nil, please use MaxWriteThroughput() function instead")
	}
	return *p.maxWriteThroughput
}

func (p *Qos) SetName(attr string) {
	p.name = &attr
}

func (p *Qos) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Qos) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Qos) SetOutboundAverage(attr int64) {
	p.outboundAverage = &attr
}

func (p *Qos) OutboundAverage() (int64, bool) {
	if p.outboundAverage != nil {
		return *p.outboundAverage, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundAverage() int64 {
	if p.outboundAverage == nil {
		panic("the outboundAverage must not be nil, please use OutboundAverage() function instead")
	}
	return *p.outboundAverage
}

func (p *Qos) SetOutboundAverageLinkshare(attr int64) {
	p.outboundAverageLinkshare = &attr
}

func (p *Qos) OutboundAverageLinkshare() (int64, bool) {
	if p.outboundAverageLinkshare != nil {
		return *p.outboundAverageLinkshare, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundAverageLinkshare() int64 {
	if p.outboundAverageLinkshare == nil {
		panic("the outboundAverageLinkshare must not be nil, please use OutboundAverageLinkshare() function instead")
	}
	return *p.outboundAverageLinkshare
}

func (p *Qos) SetOutboundAverageRealtime(attr int64) {
	p.outboundAverageRealtime = &attr
}

func (p *Qos) OutboundAverageRealtime() (int64, bool) {
	if p.outboundAverageRealtime != nil {
		return *p.outboundAverageRealtime, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundAverageRealtime() int64 {
	if p.outboundAverageRealtime == nil {
		panic("the outboundAverageRealtime must not be nil, please use OutboundAverageRealtime() function instead")
	}
	return *p.outboundAverageRealtime
}

func (p *Qos) SetOutboundAverageUpperlimit(attr int64) {
	p.outboundAverageUpperlimit = &attr
}

func (p *Qos) OutboundAverageUpperlimit() (int64, bool) {
	if p.outboundAverageUpperlimit != nil {
		return *p.outboundAverageUpperlimit, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundAverageUpperlimit() int64 {
	if p.outboundAverageUpperlimit == nil {
		panic("the outboundAverageUpperlimit must not be nil, please use OutboundAverageUpperlimit() function instead")
	}
	return *p.outboundAverageUpperlimit
}

func (p *Qos) SetOutboundBurst(attr int64) {
	p.outboundBurst = &attr
}

func (p *Qos) OutboundBurst() (int64, bool) {
	if p.outboundBurst != nil {
		return *p.outboundBurst, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundBurst() int64 {
	if p.outboundBurst == nil {
		panic("the outboundBurst must not be nil, please use OutboundBurst() function instead")
	}
	return *p.outboundBurst
}

func (p *Qos) SetOutboundPeak(attr int64) {
	p.outboundPeak = &attr
}

func (p *Qos) OutboundPeak() (int64, bool) {
	if p.outboundPeak != nil {
		return *p.outboundPeak, true
	}
	var zero int64
	return zero, false
}

func (p *Qos) MustOutboundPeak() int64 {
	if p.outboundPeak == nil {
		panic("the outboundPeak must not be nil, please use OutboundPeak() function instead")
	}
	return *p.outboundPeak
}

// SetType This type represents the kind of resource the <<types/qos,Quality of service (QoS)>> can be assigned to.
func (p *Qos) SetType(attr QosType) {
	p.type_ = &attr
}

// Type This type represents the kind of resource the <<types/qos,Quality of service (QoS)>> can be assigned to.
func (p *Qos) Type() (QosType, bool) {
	if p.type_ != nil {
		return *p.type_, true
	}
	var zero QosType
	return zero, false
}

// MustType This type represents the kind of resource the <<types/qos,Quality of service (QoS)>> can be assigned to.
func (p *Qos) MustType() QosType {
	if p.type_ == nil {
		panic("the type_ must not be nil, please use Type() function instead")
	}
	return *p.type_
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
type Quota struct {
	Struct
	clusterHardLimitPct *int64
	clusterSoftLimitPct *int64
	comment             *string
	dataCenter          *DataCenter
	description         *string
	disks               *DiskSlice
	id                  *string
	name                *string
	permissions         *PermissionSlice
	quotaClusterLimits  *QuotaClusterLimitSlice
	quotaStorageLimits  *QuotaStorageLimitSlice
	storageHardLimitPct *int64
	storageSoftLimitPct *int64
	users               *UserSlice
	vms                 *VmSlice
}

func (p *Quota) SetClusterHardLimitPct(attr int64) {
	p.clusterHardLimitPct = &attr
}

func (p *Quota) ClusterHardLimitPct() (int64, bool) {
	if p.clusterHardLimitPct != nil {
		return *p.clusterHardLimitPct, true
	}
	var zero int64
	return zero, false
}

func (p *Quota) MustClusterHardLimitPct() int64 {
	if p.clusterHardLimitPct == nil {
		panic("the clusterHardLimitPct must not be nil, please use ClusterHardLimitPct() function instead")
	}
	return *p.clusterHardLimitPct
}

func (p *Quota) SetClusterSoftLimitPct(attr int64) {
	p.clusterSoftLimitPct = &attr
}

func (p *Quota) ClusterSoftLimitPct() (int64, bool) {
	if p.clusterSoftLimitPct != nil {
		return *p.clusterSoftLimitPct, true
	}
	var zero int64
	return zero, false
}

func (p *Quota) MustClusterSoftLimitPct() int64 {
	if p.clusterSoftLimitPct == nil {
		panic("the clusterSoftLimitPct must not be nil, please use ClusterSoftLimitPct() function instead")
	}
	return *p.clusterSoftLimitPct
}

func (p *Quota) SetComment(attr string) {
	p.comment = &attr
}

func (p *Quota) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *Quota) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *Quota) SetDataCenter(attr *DataCenter) {
	p.dataCenter = attr
}

func (p *Quota) DataCenter() (*DataCenter, bool) {
	if p.dataCenter != nil {
		return p.dataCenter, true
	}
	return nil, false
}

func (p *Quota) MustDataCenter() *DataCenter {
	if p.dataCenter == nil {
		panic("the dataCenter must not be nil, please use DataCenter() function instead")
	}
	return p.dataCenter
}

func (p *Quota) SetDescription(attr string) {
	p.description = &attr
}

func (p *Quota) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *Quota) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *Quota) SetDisks(attr *DiskSlice) {
	p.disks = attr
}

func (p *Quota) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *Quota) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("the disks must not be nil, please use Disks() function instead")
	}
	return p.disks
}

func (p *Quota) SetId(attr string) {
	p.id = &attr
}

func (p *Quota) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *Quota) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *Quota) SetName(attr string) {
	p.name = &attr
}

func (p *Quota) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *Quota) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

func (p *Quota) SetPermissions(attr *PermissionSlice) {
	p.permissions = attr
}

func (p *Quota) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *Quota) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("the permissions must not be nil, please use Permissions() function instead")
	}
	return p.permissions
}

func (p *Quota) SetQuotaClusterLimits(attr *QuotaClusterLimitSlice) {
	p.quotaClusterLimits = attr
}

func (p *Quota) QuotaClusterLimits() (*QuotaClusterLimitSlice, bool) {
	if p.quotaClusterLimits != nil {
		return p.quotaClusterLimits, true
	}
	return nil, false
}

func (p *Quota) MustQuotaClusterLimits() *QuotaClusterLimitSlice {
	if p.quotaClusterLimits == nil {
		panic("the quotaClusterLimits must not be nil, please use QuotaClusterLimits() function instead")
	}
	return p.quotaClusterLimits
}

func (p *Quota) SetQuotaStorageLimits(attr *QuotaStorageLimitSlice) {
	p.quotaStorageLimits = attr
}

func (p *Quota) QuotaStorageLimits() (*QuotaStorageLimitSlice, bool) {
	if p.quotaStorageLimits != nil {
		return p.quotaStorageLimits, true
	}
	return nil, false
}

func (p *Quota) MustQuotaStorageLimits() *QuotaStorageLimitSlice {
	if p.quotaStorageLimits == nil {
		panic("the quotaStorageLimits must not be nil, please use QuotaStorageLimits() function instead")
	}
	return p.quotaStorageLimits
}

func (p *Quota) SetStorageHardLimitPct(attr int64) {
	p.storageHardLimitPct = &attr
}

func (p *Quota) StorageHardLimitPct() (int64, bool) {
	if p.storageHardLimitPct != nil {
		return *p.storageHardLimitPct, true
	}
	var zero int64
	return zero, false
}

func (p *Quota) MustStorageHardLimitPct() int64 {
	if p.storageHardLimitPct == nil {
		panic("the storageHardLimitPct must not be nil, please use StorageHardLimitPct() function instead")
	}
	return *p.storageHardLimitPct
}

func (p *Quota) SetStorageSoftLimitPct(attr int64) {
	p.storageSoftLimitPct = &attr
}

func (p *Quota) StorageSoftLimitPct() (int64, bool) {
	if p.storageSoftLimitPct != nil {
		return *p.storageSoftLimitPct, true
	}
	var zero int64
	return zero, false
}

func (p *Quota) MustStorageSoftLimitPct() int64 {
	if p.storageSoftLimitPct == nil {
		panic("the storageSoftLimitPct must not be nil, please use StorageSoftLimitPct() function instead")
	}
	return *p.storageSoftLimitPct
}

func (p *Quota) SetUsers(attr *UserSlice) {
	p.users = attr
}

func (p *Quota) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *Quota) MustUsers() *UserSlice {
	if p.users == nil {
		panic("the users must not be nil, please use Users() function instead")
	}
	return p.users
}

func (p *Quota) SetVms(attr *VmSlice) {
	p.vms = attr
}

func (p *Quota) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *Quota) MustVms() *VmSlice {
	if p.vms == nil {
		panic("the vms must not be nil, please use Vms() function instead")
	}
	return p.vms
}

type QuotaClusterLimit struct {
	Struct
	cluster     *Cluster
	comment     *string
	description *string
	id          *string
	memoryLimit *float64
	memoryUsage *float64
	name        *string
	quota       *Quota
	vcpuLimit   *int64
	vcpuUsage   *int64
}

// SetCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *QuotaClusterLimit) SetCluster(attr *Cluster) {
	p.cluster = attr
}

// Cluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *QuotaClusterLimit) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

// MustCluster Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *QuotaClusterLimit) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("the cluster must not be nil, please use Cluster() function instead")
	}
	return p.cluster
}

func (p *QuotaClusterLimit) SetComment(attr string) {
	p.comment = &attr
}

func (p *QuotaClusterLimit) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *QuotaClusterLimit) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *QuotaClusterLimit) SetDescription(attr string) {
	p.description = &attr
}

func (p *QuotaClusterLimit) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *QuotaClusterLimit) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *QuotaClusterLimit) SetId(attr string) {
	p.id = &attr
}

func (p *QuotaClusterLimit) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *QuotaClusterLimit) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *QuotaClusterLimit) SetMemoryLimit(attr float64) {
	p.memoryLimit = &attr
}

func (p *QuotaClusterLimit) MemoryLimit() (float64, bool) {
	if p.memoryLimit != nil {
		return *p.memoryLimit, true
	}
	var zero float64
	return zero, false
}

func (p *QuotaClusterLimit) MustMemoryLimit() float64 {
	if p.memoryLimit == nil {
		panic("the memoryLimit must not be nil, please use MemoryLimit() function instead")
	}
	return *p.memoryLimit
}

func (p *QuotaClusterLimit) SetMemoryUsage(attr float64) {
	p.memoryUsage = &attr
}

func (p *QuotaClusterLimit) MemoryUsage() (float64, bool) {
	if p.memoryUsage != nil {
		return *p.memoryUsage, true
	}
	var zero float64
	return zero, false
}

func (p *QuotaClusterLimit) MustMemoryUsage() float64 {
	if p.memoryUsage == nil {
		panic("the memoryUsage must not be nil, please use MemoryUsage() function instead")
	}
	return *p.memoryUsage
}

func (p *QuotaClusterLimit) SetName(attr string) {
	p.name = &attr
}

func (p *QuotaClusterLimit) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *QuotaClusterLimit) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaClusterLimit) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaClusterLimit) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaClusterLimit) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

func (p *QuotaClusterLimit) SetVcpuLimit(attr int64) {
	p.vcpuLimit = &attr
}

func (p *QuotaClusterLimit) VcpuLimit() (int64, bool) {
	if p.vcpuLimit != nil {
		return *p.vcpuLimit, true
	}
	var zero int64
	return zero, false
}

func (p *QuotaClusterLimit) MustVcpuLimit() int64 {
	if p.vcpuLimit == nil {
		panic("the vcpuLimit must not be nil, please use VcpuLimit() function instead")
	}
	return *p.vcpuLimit
}

func (p *QuotaClusterLimit) SetVcpuUsage(attr int64) {
	p.vcpuUsage = &attr
}

func (p *QuotaClusterLimit) VcpuUsage() (int64, bool) {
	if p.vcpuUsage != nil {
		return *p.vcpuUsage, true
	}
	var zero int64
	return zero, false
}

func (p *QuotaClusterLimit) MustVcpuUsage() int64 {
	if p.vcpuUsage == nil {
		panic("the vcpuUsage must not be nil, please use VcpuUsage() function instead")
	}
	return *p.vcpuUsage
}

type QuotaStorageLimit struct {
	Struct
	comment       *string
	description   *string
	id            *string
	limit         *int64
	name          *string
	quota         *Quota
	storageDomain *StorageDomain
	usage         *float64
}

func (p *QuotaStorageLimit) SetComment(attr string) {
	p.comment = &attr
}

func (p *QuotaStorageLimit) Comment() (string, bool) {
	if p.comment != nil {
		return *p.comment, true
	}
	var zero string
	return zero, false
}

func (p *QuotaStorageLimit) MustComment() string {
	if p.comment == nil {
		panic("the comment must not be nil, please use Comment() function instead")
	}
	return *p.comment
}

func (p *QuotaStorageLimit) SetDescription(attr string) {
	p.description = &attr
}

func (p *QuotaStorageLimit) Description() (string, bool) {
	if p.description != nil {
		return *p.description, true
	}
	var zero string
	return zero, false
}

func (p *QuotaStorageLimit) MustDescription() string {
	if p.description == nil {
		panic("the description must not be nil, please use Description() function instead")
	}
	return *p.description
}

func (p *QuotaStorageLimit) SetId(attr string) {
	p.id = &attr
}

func (p *QuotaStorageLimit) Id() (string, bool) {
	if p.id != nil {
		return *p.id, true
	}
	var zero string
	return zero, false
}

func (p *QuotaStorageLimit) MustId() string {
	if p.id == nil {
		panic("the id must not be nil, please use Id() function instead")
	}
	return *p.id
}

func (p *QuotaStorageLimit) SetLimit(attr int64) {
	p.limit = &attr
}

func (p *QuotaStorageLimit) Limit() (int64, bool) {
	if p.limit != nil {
		return *p.limit, true
	}
	var zero int64
	return zero, false
}

func (p *QuotaStorageLimit) MustLimit() int64 {
	if p.limit == nil {
		panic("the limit must not be nil, please use Limit() function instead")
	}
	return *p.limit
}

func (p *QuotaStorageLimit) SetName(attr string) {
	p.name = &attr
}

func (p *QuotaStorageLimit) Name() (string, bool) {
	if p.name != nil {
		return *p.name, true
	}
	var zero string
	return zero, false
}

func (p *QuotaStorageLimit) MustName() string {
	if p.name == nil {
		panic("the name must not be nil, please use Name() function instead")
	}
	return *p.name
}

// SetQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaStorageLimit) SetQuota(attr *Quota) {
	p.quota = attr
}

// Quota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaStorageLimit) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

// MustQuota Represents a quota object.
//
// An example XML representation of a quota:
//
// [source,xml]
// ----
// <quota href="/ovirt-engine/api/datacenters/7044934e/quotas/dcad5ddc" id="dcad5ddc">
//   <name>My Quota</name>
//   <description>A quota for my oVirt environment</description>
//   <cluster_hard_limit_pct>0</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>0</cluster_soft_limit_pct>
//   <data_center href="/ovirt-engine/api/datacenters/7044934e" id="7044934e"/>
//   <storage_hard_limit_pct>0</storage_hard_limit_pct>
//   <storage_soft_limit_pct>0</storage_soft_limit_pct>
// </quota>
// ----
func (p *QuotaStorageLimit) MustQuota() *Quota {
	if p.quota == nil {
		panic("the quota must not be nil, please use Quota() function instead")
	}
	return p.quota
}

// SetStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *QuotaStorageLimit) SetStorageDomain(attr *StorageDomain) {
	p.storageDomain = attr
}

// StorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *QuotaStorageLimit) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

// MustStorageDomain Storage domain.
//
// An XML representation of a NFS storage domain with identifier `123`:
//
// [source,xml]
// ----
// <storage_domain href="/ovirt-engine/api/storagedomains/123" id="123">
//   <name>mydata</name>
//   <description>My data</description>
//   <available>38654705664</available>
//   <committed>1073741824</committed>
//   <critical_space_action_blocker>5</critical_space_action_blocker>
//   <external_status>ok</external_status>
//   <master>true</master>
//   <storage>
//     <address>mynfs.example.com</address>
//     <nfs_version>v3</nfs_version>
//     <path>/exports/mydata</path>
//     <type>nfs</type>
//   </storage>
//   <storage_format>v3</storage_format>
//   <type>data</type>
//   <used>13958643712</used>
//   <warning_low_space_indicator>10</warning_low_space_indicator>
//   <wipe_after_delete>false</wipe_after_delete>
//   <data_centers>
//     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
//   </data_centers>
// </storage_domain>
// ----
func (p *QuotaStorageLimit) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("the storageDomain must not be nil, please use StorageDomain() function instead")
	}
	return p.storageDomain
}

func (p *QuotaStorageLimit) SetUsage(attr float64) {
	p.usage = &attr
}

func (p *QuotaStorageLimit) Usage() (float64, bool) {
	if p.usage != nil {
		return *p.usage, true
	}
	var zero float64
	return zero, false
}

func (p *QuotaStorageLimit) MustUsage() float64 {
	if p.usage == nil {
		panic("the usage must not be nil, please use Usage() function instead")
	}
	return *p.usage
}

type Range struct {
	Struct
	from *string
	to   *string
}

func (p *Range) SetFrom(attr string) {
	p.from = &attr
}

func (p *Range) From() (string, bool) {
	if p.from != nil {
		return *p.from, true
	}
	var zero string
	return zero, false
}

func (p *Range) MustFrom() string {
	if p.from == nil {
		panic("the from must not be nil, please use From() function instead")
	}
	return *p.from
}

func (p *Range) SetTo(attr string) {
	p.to = &attr
}

func (p *Range) To() (string, bool) {
	if p.to != nil {
		return *p.to, true
	}
	var zero string
	return zero, false
}

func (p *Range) MustTo() string {
	if p.to == nil {
		panic("the to must not be nil, please use To() function instead")
	}
	return *p.to
}

// Rate Determines maximum speed of consumption of bytes from random number generator device.
type Rate struct {
	Struct
	bytes  *int64
	period *int64
}

func (p *Rate) SetBytes(attr int64) {
	p.bytes = &attr
}

func (p *Rate) Bytes() (int64, bool) {
	if p.bytes != nil {
		return *p.bytes, true
	}
	var zero int64
	return zero, false
}

func (p *Rate) MustBytes() int64 {
	if p.bytes == nil {
		panic("the bytes must not be nil, please use Bytes() function instead")
	}
	return *p.bytes
}

func (p *Rate) SetPeriod(attr int64) {
	p.period = &attr
}

func (p *Rate) Period() (int64, bool) {
	if p.period != nil {
		return *p.period, true
	}
	var zero int64
	return zero, false
}

func (p *Rate) MustPeriod() int64 {
	if p.period == nil {
		panic("the period must not be nil, please use Period() function instead")
	}
	return *p.period
}

// RegistrationAffinityGroupMapping This type describes how to map affinity groups as part of the object registration. An object can be
// a virtual machine, template, etc.
//
// An example of an XML representation using this mapping:
//
// [source,xml]
// ----
// <action>
//   <registration_configuration>
//     <affinity_group_mappings>
//      <registration_affinity_group_mapping>
//        <from>
//          <name>affinity</name>
//        </from>
//        <to>
//          <name>affinity2</name>
//        </to>
//      </registration_affinity_group_mapping>
//     </affinity_group_mappings>
//   </registration_configuration>
// </action>
// ----
type RegistrationAffinityGroupMapping struct {
	Struct
	from *AffinityGroup
	to   *AffinityGroup
}

// SetFrom An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) SetFrom(attr *AffinityGroup) {
	p.from = attr
}

// From An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) From() (*AffinityGroup, bool) {
	if p.from != nil {
		return p.from, true
	}
	return nil, false
}

// MustFrom An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) MustFrom() *AffinityGroup {
	if p.from == nil {
		panic("the from must not be nil, please use From() function instead")
	}
	return p.from
}

// SetTo An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) SetTo(attr *AffinityGroup) {
	p.to = attr
}

// To An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) To() (*AffinityGroup, bool) {
	if p.to != nil {
		return p.to, true
	}
	return nil, false
}

// MustTo An affinity group represents a group of virtual machines with a defined relationship.
func (p *RegistrationAffinityGroupMapping) MustTo() *AffinityGroup {
	if p.to == nil {
		panic("the to must not be nil, please use To() function instead")
	}
	return p.to
}

// RegistrationAffinityLabelMapping This type describes how to map affinity labels as part of the object registration. An object can be
// a virtual machine, template, etc.
//
// An example of an XML representation using mapping:
//
// [source,xml]
// ----
// <action>
//   <registration_configuration>
//     <affinity_label_mappings>
//      <registration_affinity_label_mapping>
//        <from>
//          <name>affinity_label</name>
//        </from>
//        <to>
//          <name>affinity_label2</name>
//        </to>
//      </registration_affinity_label_mapping>
//     </affinity_label_mappings>
//   </registration_configuration>
// </action>
// ----
type RegistrationAffinityLabelMapping struct {
	Struct
	from *AffinityLabel
	to   *AffinityLabel
}

// SetFrom The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) SetFrom(attr *AffinityLabel) {
	p.from = attr
}

// From The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) From() (*AffinityLabel, bool) {
	if p.from != nil {
		return p.from, true
	}
	return nil, false
}

// MustFrom The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) MustFrom() *AffinityLabel {
	if p.from == nil {
		panic("the from must not be nil, please use From() function instead")
	}
	return p.from
}

// SetTo The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) SetTo(attr *AffinityLabel) {
	p.to = attr
}

// To The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) To() (*AffinityLabel, bool) {
	if p.to != nil {
		return p.to, true
	}
	return nil, false
}

// MustTo The affinity label can influence virtual machine scheduling.
// It is most frequently used to create a sub-cluster from the available hosts.
func (p *RegistrationAffinityLabelMapping) MustTo() *AffinityLabel {
	if p.to == nil {
		panic("the to must not be nil, please use To() function instead")
	}
	return p.to
}

// RegistrationClusterMapping This type describes how to map clusters as part of the object registration. An object can be
// a virtual machine, template, etc.
//
// An example of an XML representation using this mapping:
//
// [source,xml]
// ----
// <action>
//   <registration_configuration>
//     <cluster_mappings>
//       <registration_cluster_mapping>
//         <from>
//           <name>myoriginalcluster</name>
//         </from>
//         <to>
//           <name>mynewcluster</name>
//         </to>
//       </registration_cluster_mapping>
//     </cluster_mappings>
//   </registration_configuration>
// </action>
// ----
type RegistrationClusterMapping struct {
	Struct
	from *Cluster
	to   *Cluster
}

// SetFrom Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) SetFrom(attr *Cluster) {
	p.from = attr
}

// From Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) From() (*Cluster, bool) {
	if p.from != nil {
		return p.from, true
	}
	return nil, false
}

// MustFrom Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) MustFrom() *Cluster {
	if p.from == nil {
		panic("the from must not be nil, please use From() function instead")
	}
	return p.from
}

// SetTo Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) SetTo(attr *Cluster) {
	p.to = attr
}

// To Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) To() (*Cluster, bool) {
	if p.to != nil {
		return p.to, true
	}
	return nil, false
}

// MustTo Type representation of a cluster.
//
// A JSON representation of a cluster:
//
// [source]
// ----
// {
//   "cluster" : [ {
//     "ballooning_enabled" : "false",
//     "cpu" : {
//       "architecture" : "x86_64",
//       "type" : "Intel SandyBridge Family"
//     },
//     "custom_scheduling_policy_properties" : {
//       "property" : [ {
//         "name" : "HighUtilization",
//         "value" : "80"
//       }, {
//         "name" : "CpuOverCommitDurationMinutes",
//         "value" : "2"
//       } ]
//     },
//     "error_handling" : {
//       "on_error" : "migrate"
//     },
//     "fencing_policy" : {
//       "enabled" : "true",
//       "skip_if_connectivity_broken" : {
//         "enabled" : "false",
//         "threshold" : "50"
//       },
//       "skip_if_gluster_bricks_up" : "false",
//       "skip_if_gluster_quorum_not_met" : "false",
//       "skip_if_sd_active" : {
//         "enabled" : "false"
//       }
//     },
//     "gluster_service" : "false",
//     "firewall_type" : "iptables",
//     "ha_reservation" : "false",
//     "ksm" : {
//       "enabled" : "true",
//       "merge_across_nodes" : "true"
//     },
//     "memory_policy" : {
//       "over_commit" : {
//         "percent" : "100"
//       },
//       "transparent_hugepages" : {
//         "enabled" : "true"
//       }
//     },
//     "migration" : {
//       "auto_converge" : "inherit",
//       "bandwidth" : {
//         "assignment_method" : "auto"
//       },
//       "compressed" : "inherit",
//       "policy" : {
//         "id" : "00000000-0000-0000-0000-000000000000"
//       }
//     },
//     "required_rng_sources" : {
//       "required_rng_source" : [ "random" ]
//     },
//     "switch_type" : "legacy",
//     "threads_as_cores" : "false",
//     "trusted_service" : "false",
//     "tunnel_migration" : "false",
//     "version" : {
//       "major" : "4",
//       "minor" : "1"
//     },
//     "virt_service" : "true",
//     "data_center" : {
//       "href" : "/ovirt-engine/api/datacenters/123",
//       "id" : "123"
//     },
//     "mac_pool" : {
//       "href" : "/ovirt-engine/api/macpools/456",
//       "id" : "456"
//     },
//     "scheduling_policy" : {
//       "href" : "/ovirt-engine/api/schedulingpolicies/789",
//       "id" : "789"
//     },
//     "actions" : {
//       "link" : [ {
//         "href" : "/ovirt-engine/api/clusters/234/resetemulatedmachine",
//         "rel" : "resetemulatedmachine"
//       } ]
//     },
//     "name" : "Default",
//     "description" : "The default server cluster",
//     "href" : "/ovirt-engine/api/clusters/234",
//     "id" : "234",
//     "link" : [ {
//       "href" : "/ovirt-engine/api/clusters/234/permissions",
//       "rel" : "permissions"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/cpuprofiles",
//       "rel" : "cpuprofiles"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networkfilters",
//       "rel" : "networkfilters"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/networks",
//       "rel" : "networks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/affinitygroups",
//       "rel" : "affinitygroups"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glusterhooks",
//       "rel" : "glusterhooks"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/glustervolumes",
//       "rel" : "glustervolumes"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/enabledfeatures",
//       "rel" : "enabledfeatures"
//     }, {
//       "href" : "/ovirt-engine/api/clusters/234/externalnetworkproviders",
//       "rel" : "externalnetworkproviders"
//     } ]
//   } ]
// }
// ----
func (p *RegistrationClusterMapping) MustTo() *Cluster {
	if p.to == nil {
		panic("the to must not be nil, please use To() function instead")
	}
	return p.to
}

// RegistrationConfiguration This type describes how an object (virtual machine, template,
// etc) is registered, and is used for the implementation of disaster recovery solutions.
//
// Each mapping contained in this type can be used to map
// objects in the original system to corresponding objects
// in the system where the virtual machine or template is being registered.
// For example, there could be a primary setup with a virtual machine
// configured on cluster A, and an active secondary setup with cluster B.
// Cluster B is compatible with that virtual machine, and in case of a
// disaster recovery scenario the storage domain can be
// imported to the secondary setup, and the user can register the
// virtual machine to cluster B.
//
// In that case, we can automate the recovery process by defining a cluster mapping.
// After the entity is registered, its OVF will indicate it belongs to
// cluster A, but the mapping will indicate that cluster A will
// be replaced with cluster B.
// {engine-name} should do the switch and register the virtual machine to cluster B
// in the secondary site.
//
// Cluster mapping is just one example, there are different types of mappings:
//
// - Cluster mapping.
// - LUN mapping.
// - Role mapping.
// - Domain mapping.
// - Permissions mapping.
// - Affinity Group mapping.
// - Affinity Label mapping.
// - Virtual NIC profile mapping.
//
// Each mapping will be used for its specific OVF's data once the register operation
// takes place in the {engine-name}.
//
// An example of an XML representation using the mapping:
//
// [source,xml]
// ----
// <action>
//   <registration_configuration>
//     <cluster_mappings>
//       <registration_cluster_mapping>
//         <from>
//           <name>myoriginalcluster</name>
//         </from>
//         <to>
//           <name>mynewcluster</name>
//         </to>
//       </registration_cluster_mapping>
//     </cluster_mappings>
//     <role_mappings>
//       <registration_role_mapping>
//         <from>
//           <name>SuperUser</name>
//         </from>
//         <to>
//           <name>UserVmRunTimeManager</name>
//         </to>
//       </registration_role_mapping>
//     </role_mappings>
//     <domain_mappings>
//       <registration_domain_mapping>
//         <from>
//           <name>redhat</name>
//         </from>
//         <to>
//           <name>internal</name>
//         </to>
//       </registration_domain_mapping>
//     </domain_mappings>
//     <lun_mappings>
//      <registration_lun_mapping>
//        <from id="111">
//        </from>
//        <to id="222">
//          <alias>weTestLun</alias>
//          <lun_storage>
//            <type>iscsi</type>
//            <logical_units>
//               <logical_unit id="36001405fb1ddb4b91e44078f1fffcfef">
//                  <address>44.33.11.22</address>
//                  <port>3260</port>
//                  <portal>1</portal>
//                  <target>iqn.2017-11.com.name.redhat:444</target>
//               </logical_unit>
//            </logical_units>
//          </lun_storage>
//        </to>
//      </registration_lun_mapping>
//     </lun_mappings>
//     <affinity_group_mappings>
//      <registration_affinity_group_mapping>
//        <from>
//          <name>affinity</name>
//        </from>
//        <to>
//          <name>affinity2</name>
//        </to>
//      </registration_affinity_group_mapping>
//     </affinity_group_mappings>
//     <affinity_label_mappings>
//      <registration_affinity_label_mapping>
//        <from>
//          <name>affinity_label</name>
//        </from>
//        <to>
//          <name>affinity_label2</name>
//        </to>
//      </registration_affinity_label_mapping>
//     </affinity_label_mappings>
//     <vnic_profile_mappings>
//       <registration_vnic_profile_mapping>
//         <from>
//           <name>gold</name>
//           <network>
//             <name>red</name>
//           </network>
//         </from>
//         <to id="738dd914-8ec8-4a8b-8628-34672a5d449b"/>
//       </registration_vnic_profile_mapping>
//       <registration_vnic_profile_mapping>
//         <from>
//           <name>silver</name>
//           <network>
//             <name>blue</name>
//           </network>
//         </from>
//         <to>
//           <name>copper</name>
//           <network>
//             <name>orange</name>
//           </network>
//         </to>
//       </registration_vnic_profile_mapping>
//     </vnic_profile_mappings>
//   </registration_configuration>
// </action>
// ----
type RegistrationConfiguration struct {
	Struct
	affinityGroupMappings *RegistrationAffinityGroupMappingSlice
	affinityLabelMappings *RegistrationAffinityLabelMappingSlice
	clusterMappings       *RegistrationClusterMappingSlice
	domainMappings        *RegistrationDomainMappingSlice
	lunMappings           *RegistrationLunMappingSlice
	roleMappings          *RegistrationRoleMappingSlice
	vnicProfileMappings   *RegistrationVnicProfileMappingSlice
}

func (p *RegistrationConfiguration) SetAffinityGroupMappings(attr *RegistrationAffinityGroupMappingSlice) {
	p.affinityGroupMappings = attr
}

func (p *RegistrationConfiguration) AffinityGroupMappings() (*RegistrationAffinityGroupMappingSlice, bool) {
	if p.affinityGroupMappings != nil {
		return p.affinityGroupMappings, true
	}
	return nil, false
}

func (p *RegistrationConfiguration) MustAffinityGroupMappings() *RegistrationAffinityGroupMappingSlice {
	if p.affinityGroupMappings == nil {
		panic("the affinityGroupMappings must not be nil, please use AffinityGroupMappings() function instead")
	}
	return p.affinityGroupMappings
}

func (p *RegistrationConfiguration) SetAffinityLabelMappings(attr *RegistrationAffinityLabelMappingSlice) {
	p.affinityLabelMappings = attr
}

func (p *RegistrationConfiguration) AffinityLabelMappings() (*RegistrationAffinityLabelMappingSlice, bool) {
	if p.affinityLabelMappings != nil {
		return p.affinityLabelMappings, true
	}
	return nil, false
}

func (p *RegistrationConfiguration) MustAffinityLabelMappings() *RegistrationAffinityLabelMappingSlice {
	if p.affinityLabelMappings == nil {
		panic("the affinityLabelMappings must not be nil, please use AffinityLabelMappings() function instead")
	}
	return p.affinityLabelMappings
}

func (p *RegistrationConfiguration) SetClusterMappings(attr *RegistrationClusterMappingSlice) {
	p.clusterMappings = attr
}

func (p *RegistrationConfiguration) ClusterMappings() (*RegistrationClusterMappingSlice, bool) {
	if p.clusterMappings != nil {
		return p.clusterMappings, true
	}
	return nil, false
}

func (p *RegistrationConfiguration) MustClusterMappings() *RegistrationClusterMappingSlice {
	if p.clusterMappings == nil {
		panic("the clusterMappings must not be nil, please use ClusterMappings() function instead")
	}
	return p.clusterMappings
}

func (p *RegistrationConfiguration) SetDomainMappings(attr *RegistrationDomainMappingSlice) {
	p.domainMappings = attr
}

func (p *RegistrationConfiguration) DomainMappings() (*RegistrationDomainMappingSlice, bool) {
	if p.domainMappings != nil {
		return p.domainMappings, true
	}
	return nil, false
}

func (p *RegistrationConfiguration) MustDomainMappings() *RegistrationDomainMappingSlice {
	if p.domainMappings == nil {
		panic("the domainMappings must not be nil, please use DomainMappings() function instead")
	}
	return p.domainMappings
}

func (p *RegistrationConfiguration) SetLunMappings(attr *RegistrationLunMappingSlice) {
	p.lunMappings = attr
}

func (p *RegistrationConfiguration) LunMappings() (*RegistrationLunMappingSlice, bool) {
	if p.lunMappings != nil {
		return p.lunMappings, true
	}
	return nil, false
}

func (p *RegistrationConfiguration) MustLunMappings()