/*
 * livepatch_bsc1236244
 *
 * Fix for CVE-2024-53208, bsc#1236244
 *
 *  Upstream commit:
 *  0b882940665c ("Bluetooth: MGMT: Fix slab-use-after-free Read in set_powered_sync")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  Not affected
 *
 *  SLE15-SP6 commit:
 *  0e6f9cb7fa5473e70449f2ea7f600c1497b398d2
 *
 *  SLE MICRO-6-0 commit:
 *  0e6f9cb7fa5473e70449f2ea7f600c1497b398d2
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Ali Abdallah <ali.abdallah@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_BT)

#if !IS_MODULE(CONFIG_BT)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <linux/module.h>

/* klp-ccp: from include/asm-generic/unaligned.h */
#define __ASM_GENERIC_UNALIGNED_H

/* klp-ccp: from net/bluetooth/mgmt.c */
#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>
#include <net/bluetooth/hci_sock.h>

#include <net/bluetooth/mgmt.h>

/* klp-ccp: from net/bluetooth/hci_request.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt_util.h */
struct mgmt_pending_cmd {
	struct list_head list;
	u16 opcode;
	int index;
	void *param;
	size_t param_len;
	struct sock *sk;
	struct sk_buff *skb;
	void *user_data;
	int (*cmd_complete)(struct mgmt_pending_cmd *cmd, u8 status);
};

int mgmt_send_event(u16 event, struct hci_dev *hdev, unsigned short channel,
		    void *data, u16 data_len, int flag, struct sock *skip_sk);
int mgmt_cmd_status(struct sock *sk, u16 index, u16 cmd, u8 status);
int mgmt_cmd_complete(struct sock *sk, u16 index, u16 cmd, u8 status,
		      void *rp, size_t rp_len);

struct mgmt_pending_cmd *mgmt_pending_find(unsigned short channel, u16 opcode,
					   struct hci_dev *hdev);

void mgmt_pending_remove(struct mgmt_pending_cmd *cmd);

/* klp-ccp: from net/bluetooth/eir.h */
#include <asm/unaligned.h>

/* klp-ccp: from net/bluetooth/mgmt.c */
extern u8 mgmt_status(int err);

static int mgmt_limited_event(u16 event, struct hci_dev *hdev, void *data,
			      u16 len, int flag, struct sock *skip_sk)
{
	return mgmt_send_event(event, hdev, HCI_CHANNEL_CONTROL, data, len,
			       flag, skip_sk);
}

extern u32 get_current_settings(struct hci_dev *hdev);

static struct mgmt_pending_cmd *pending_find(u16 opcode, struct hci_dev *hdev)
{
	return mgmt_pending_find(HCI_CHANNEL_CONTROL, opcode, hdev);
}

static int send_settings_rsp(struct sock *sk, u16 opcode, struct hci_dev *hdev);

static int send_settings_rsp(struct sock *sk, u16 opcode, struct hci_dev *hdev)
{
	__le32 settings = cpu_to_le32(get_current_settings(hdev));

	return mgmt_cmd_complete(sk, hdev->id, opcode, 0, &settings,
				 sizeof(settings));
}

extern void restart_le_actions(struct hci_dev *hdev);

static int new_settings(struct hci_dev *hdev, struct sock *skip)
{
	__le32 ev = cpu_to_le32(get_current_settings(hdev));

	return mgmt_limited_event(MGMT_EV_NEW_SETTINGS, hdev, &ev,
				  sizeof(ev), HCI_MGMT_SETTING_EVENTS, skip);
}

void klpp_mgmt_set_powered_complete(struct hci_dev *hdev, void *data, int err)
{
	struct mgmt_pending_cmd *cmd = data;
	struct mgmt_mode *cp;

	/* Make sure cmd still outstanding. */
	if (err == -ECANCELED ||
	    cmd != pending_find(MGMT_OP_SET_POWERED, hdev))
		return;

	cp = cmd->param;

	bt_dev_dbg(hdev, "err %d", err);

	if (!err) {
		if (cp->val) {
			hci_dev_lock(hdev);
			restart_le_actions(hdev);
			hci_update_passive_scan(hdev);
			hci_dev_unlock(hdev);
		}

		send_settings_rsp(cmd->sk, cmd->opcode, hdev);

		/* Only call new_setting for power on as power off is deferred
		 * to hdev->power_off work which does call hci_dev_do_close.
		 */
		if (cp->val)
			new_settings(hdev, cmd->sk);
	} else {
		mgmt_cmd_status(cmd->sk, hdev->id, MGMT_OP_SET_POWERED,
				mgmt_status(err));
	}

	mgmt_pending_remove(cmd);
}

int klpp_set_powered_sync(struct hci_dev *hdev, void *data)
{
	struct mgmt_pending_cmd *cmd = data;
	struct mgmt_mode *cp;

	/* Make sure cmd still outstanding. */
	if (cmd != pending_find(MGMT_OP_SET_POWERED, hdev))
		return -ECANCELED;

	cp = cmd->param;

	BT_DBG("%s", hdev->name);

	return hci_set_powered_sync(hdev, cp->val);
}


#include "livepatch_bsc1236244.h"

#include <linux/livepatch.h>

extern typeof(get_current_settings) get_current_settings
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, get_current_settings);
extern typeof(hci_set_powered_sync) hci_set_powered_sync
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, hci_set_powered_sync);
extern typeof(hci_update_passive_scan) hci_update_passive_scan
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, hci_update_passive_scan);
extern typeof(mgmt_cmd_complete) mgmt_cmd_complete
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_cmd_complete);
extern typeof(mgmt_cmd_status) mgmt_cmd_status
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_cmd_status);
extern typeof(mgmt_pending_find) mgmt_pending_find
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_pending_find);
extern typeof(mgmt_pending_remove) mgmt_pending_remove
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_pending_remove);
extern typeof(mgmt_send_event) mgmt_send_event
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_send_event);
extern typeof(mgmt_status) mgmt_status
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, mgmt_status);
extern typeof(restart_le_actions) restart_le_actions
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, restart_le_actions);

#endif /* IS_ENABLED(CONFIG_BT) */
