/*
Copyright © 2022 - 2025 SUSE LLC

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package test

import (
	"fmt"
	"strings"

	"github.com/gobuffalo/flect"
	apiextensionsv1 "k8s.io/apiextensions-apiserver/pkg/apis/apiextensions/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/utils/ptr"
)

var (
	managementGroupVersion = schema.GroupVersion{Group: "management.cattle.io", Version: "v3"}
	clusterGroupVersion    = schema.GroupVersion{Group: "cluster.x-k8s.io", Version: "v1beta1"}
	fleetGroupVersion      = schema.GroupVersion{Group: "fleet.cattle.io", Version: "v1alpha1"}

	// fakeSettingKind is the Kind for the Setting object.
	fakeSettingKind = "Setting"
	// fakeSettingCRD is a fake Setting CRD.
	fakeSettingCRD = generateCRD(managementGroupVersion.WithKind(fakeSettingKind), apiextensionsv1.ClusterScoped)

	// fakeMachineKind is the Kind for the Machine object.
	fakeMachineKind = "Machine"
	// fakeMachineCRD is a fake Machine CRD.
	fakeMachineCRD = generateCRD(clusterGroupVersion.WithKind(fakeMachineKind), apiextensionsv1.NamespaceScoped)

	// fakeBundleKind is the Kind for the Bundle object.
	fakeBundleKind = "Bundle"
	// fakeBundleCRD is a fake Bundle CRD.
	fakeBundleCRD = generateCRD(fleetGroupVersion.WithKind(fakeBundleKind), apiextensionsv1.NamespaceScoped)
)

func generateCRD(gvk schema.GroupVersionKind, scope apiextensionsv1.ResourceScope) *apiextensionsv1.CustomResourceDefinition {
	return &apiextensionsv1.CustomResourceDefinition{
		TypeMeta: metav1.TypeMeta{
			APIVersion: apiextensionsv1.SchemeGroupVersion.String(),
			Kind:       "CustomResourceDefinition",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: fmt.Sprintf("%s.%s", flect.Pluralize(strings.ToLower(gvk.Kind)), gvk.Group),
		},
		Spec: apiextensionsv1.CustomResourceDefinitionSpec{
			Group: gvk.Group,
			Scope: scope,
			Names: apiextensionsv1.CustomResourceDefinitionNames{
				Kind:   gvk.Kind,
				Plural: flect.Pluralize(strings.ToLower(gvk.Kind)),
			},
			Versions: []apiextensionsv1.CustomResourceDefinitionVersion{
				{
					Name:    gvk.Version,
					Served:  true,
					Storage: true,
					Subresources: &apiextensionsv1.CustomResourceSubresources{
						Status: &apiextensionsv1.CustomResourceSubresourceStatus{},
					},
					Schema: &apiextensionsv1.CustomResourceValidation{
						OpenAPIV3Schema: &apiextensionsv1.JSONSchemaProps{
							Type: "object",
							Properties: map[string]apiextensionsv1.JSONSchemaProps{
								"spec": {
									Type:                   "object",
									XPreserveUnknownFields: ptr.To(true),
								},
								"value": {
									Type:                   "string",
									XPreserveUnknownFields: ptr.To(true),
								},
								"status": {
									Type:                   "object",
									XPreserveUnknownFields: ptr.To(true),
								},
							},
						},
					},
				},
			},
		},
	}
}
