/*
 * Copyright (C) 2020 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "compare.h"
#include "lc_kmac.h"
#include "lc_rng.h"
#include "math_helper.h"
#include "visibility.h"

#include "sha3_c.h"
#include "sha3_arm_asm.h"
#include "sha3_arm_ce.h"
#include "sha3_arm_neon.h"
#include "sha3_avx2.h"
#include "sha3_avx512.h"

#define LC_EXEC_ONE_TEST(sha3_impl)                                            \
	if (sha3_impl)                                                         \
	ret += _kmac_256_xof_tester(sha3_impl, #sha3_impl)

static int _kmac_256_xof_tester(const struct lc_hash *cshake_256,
				const char *name)
{
	static const uint8_t msg1[] = {
		0x6F, 0x50, 0xA7, 0xC3, 0x48, 0xCE, 0xA5, 0x10, 0x6A,
		0xBE, 0x32, 0xE4, 0xF0, 0x9E, 0x7B, 0xC6, 0x0E, 0x5F,
		0x8F, 0xE1, 0x17, 0xF9, 0x41, 0x29, 0x73, 0xC2, 0xAC,
		0x0E, 0xD6, 0x87, 0xCD, 0x41, 0x99, 0xB7, 0xCD, 0x5B,
		0x89, 0xA4, 0x02, 0x82, 0xD8, 0x54, 0x51
	};
	static const uint8_t key1[] = {
		0x04, 0xBB, 0xB3, 0xF4, 0x84, 0x74, 0x25, 0x97, 0x72, 0xD8,
		0xF0, 0x78, 0x3C, 0xAC, 0x31, 0x67, 0x4B, 0x50, 0x7D, 0x64,
		0xBB, 0xC3, 0xED, 0x98, 0xE4, 0x23, 0xEF, 0xEC, 0xA6, 0xD1,
		0x68, 0xD1, 0x8F, 0x36, 0xED, 0x5A, 0xDB, 0x0E, 0xFD, 0x8C,
		0x3A, 0x43, 0x91, 0x2F, 0x32, 0x9C, 0xF0, 0x4B, 0x75, 0x4A,
		0xD3, 0xEA, 0xAA, 0xE4, 0x88, 0xF2, 0x15, 0x8F, 0x02, 0x82,
		0x01, 0x60, 0xDB, 0x03, 0x08, 0x23, 0x14, 0x2D, 0xF7, 0xA6,
		0xB2, 0x1F, 0x3B, 0x28, 0x48, 0x44, 0xB5, 0x03, 0x28, 0xE6,
		0xA5, 0xF1, 0x4C, 0x81, 0xD4, 0x70, 0xF5, 0xA4, 0x64, 0xE4,
		0x00, 0x8D, 0x2D, 0x38, 0xB4, 0x83, 0x87
	};
	static const uint8_t cust1[] = {
		0xCD, 0xD2, 0x11, 0x8F, 0xB7, 0xF9, 0xFC, 0x88, 0xF2, 0x96,
		0x88, 0xA2, 0xF9, 0x40, 0x70, 0xE5, 0x2D, 0xA5, 0xDA, 0xAD,
		0xD3, 0x12, 0xFB, 0x9F, 0xA4, 0xBF, 0xC6, 0x15, 0x77, 0xFE,
		0xBA, 0xF7, 0x89, 0x56, 0x8A, 0xF0, 0x81, 0xB0, 0x79, 0xDE,
		0x69, 0xC6, 0x1D, 0x0E, 0x0F, 0xB4, 0xC7, 0x2F, 0xE0, 0xC2,
		0x95, 0xA3, 0xF3, 0xF7, 0x3D, 0x57, 0xC3, 0x92, 0x40, 0x31,
		0xD8, 0x65, 0x4E, 0x10, 0xA4, 0x97, 0x57, 0xC0
	};
	static const uint8_t exp1[] = { 0xB1, 0x37, 0x64, 0xCF, 0xCF, 0x8E,
					0x3A, 0x67, 0xE0, 0x76, 0x05, 0x66,
					0x5D, 0x21, 0xFE, 0x99, 0xCC, 0x7E,
					0xC7, 0xA9, 0xE9, 0x71, 0xC4, 0x3D,
					0x01, 0xFD, 0x66 };

	static const uint8_t msg2[] = {
		0x94, 0xF2, 0x1B, 0x18, 0x48, 0x43, 0xED, 0x35, 0xA0, 0x5F,
		0xA5, 0x72, 0x1A, 0x4F, 0xBD, 0xAA, 0xA3, 0xFE, 0xA1, 0xD2,
		0x2D, 0xEF, 0xA4, 0x31, 0xD9, 0x92, 0x1F, 0x60, 0xD5, 0x7E,
		0x22, 0xA3, 0x41, 0x49, 0x25, 0xFA, 0x92, 0x49, 0xFF, 0x2A,
		0x7E, 0xC3, 0xFC, 0xE8, 0x0F, 0x2B, 0x17, 0x9F, 0x55, 0xCB,
		0xBE, 0x45, 0xE7, 0xFB, 0xC7, 0xBA, 0xEC, 0xAE, 0x61, 0x48,
		0x6D, 0x6B, 0xF8, 0x9D, 0x47, 0xB4, 0x9E, 0x31, 0xEE, 0x13,
		0x75, 0x62, 0x43, 0x89, 0xCD, 0xF4, 0xA9, 0x90, 0xDF, 0x3B,
		0xDB, 0x8D, 0xCB, 0xCD, 0x1C, 0x69, 0x57, 0xE0, 0x7B, 0x14,
		0x0D, 0x08, 0xAC, 0x43, 0xB3, 0x4B, 0xB7, 0xA6, 0x78, 0xEE,
		0x66, 0xF2, 0x2D, 0x21, 0x98, 0xDA, 0xAF, 0x3F, 0xDB, 0x42,
		0xCE, 0xF9, 0xB6, 0x0E, 0x8B, 0x20, 0xB6, 0xB0, 0x43, 0xC0,
		0xA3, 0x05, 0x24, 0x1B, 0x34, 0x0E, 0x3D, 0xF3, 0xC8, 0xA8,
		0x37, 0x78, 0x8A, 0x96, 0x49, 0x66, 0xAE, 0x82, 0x7E, 0x60,
		0x2A, 0xFB, 0xE5, 0xA0, 0x30, 0x42, 0x16, 0xE9, 0x36, 0x7C,
		0xF4, 0xEE, 0x1E, 0x03, 0x84, 0xB0, 0x59, 0xDB, 0xA3, 0x17,
		0x3A, 0xD9, 0x3A, 0xC0, 0xE3, 0xFE, 0x3C, 0xCD, 0xF7, 0xC2,
		0x9D, 0xAB, 0x9F, 0x62, 0xC3, 0x28, 0xB5, 0x7D, 0xF7, 0xD8,
		0x25, 0x77, 0xCC, 0x03, 0x69, 0x1A, 0x76, 0xC6, 0x63, 0x40,
		0x29, 0x4B, 0x2E, 0x1D, 0xC2, 0xED, 0x23, 0x9B, 0x1E, 0x27,
		0x1A, 0x3C, 0x60, 0x03, 0x7F, 0x5E, 0x09, 0x95, 0xC9, 0xE8,
		0xF1, 0x82, 0x26, 0xD1, 0x34, 0xFD, 0xC7, 0x5E, 0x5C, 0x01,
		0x26, 0x56, 0x09, 0x57, 0xC6, 0x1C, 0x09, 0xA8, 0xF7, 0x8D,
		0x10, 0xA7, 0x00, 0x1E, 0xAA, 0x5A, 0x74, 0x93, 0x62, 0xC6,
		0x1B, 0x9D, 0x1D, 0x6D, 0x98, 0xEF, 0x67, 0x37, 0xC5, 0xBA,
		0x7B, 0x34, 0x42, 0x1E, 0x45, 0xC5, 0x96, 0xD5, 0x97, 0xE1,
		0x24, 0x25, 0x32, 0xE7, 0x55, 0x8A, 0x72, 0x8E, 0x09, 0xFA,
		0xB7, 0x42, 0x92, 0x9F, 0x20, 0x6C, 0x1C, 0xA0, 0x2D, 0x5D,
		0xFE, 0x2D, 0xCD, 0x2A, 0x74, 0x66, 0x31, 0x73, 0xAC, 0xB4,
		0x54, 0x2C, 0xD6, 0xF4, 0x65, 0x3C, 0x09, 0x39, 0x0D, 0xC6,
		0x4D, 0x66, 0x65, 0x92, 0xE4, 0xEF, 0x9A, 0x74, 0x92, 0xF2,
		0x79, 0xEB, 0x01, 0x6D, 0xD6, 0x89, 0x61, 0x3C, 0x21, 0x63,
		0x50, 0xBF, 0xF4, 0x6F, 0x8C, 0x9C, 0x32, 0x2D, 0x13, 0xB2,
		0x5F, 0x96, 0x8C, 0x2F, 0xCB, 0xD4, 0xB7, 0x2E, 0x66, 0x95,
		0xC9, 0x82, 0x33, 0xA1, 0x23, 0xE1, 0xC8, 0x08, 0x22, 0x78,
		0xEE, 0x68, 0xD0, 0x9A, 0xF5, 0x18, 0x3A, 0x1D, 0xC1, 0x4B,
		0x85, 0x64, 0xF7, 0xC5, 0x94, 0x9D, 0xCD, 0xCA, 0x75, 0xF8,
		0x02, 0x50, 0xA5, 0xF1, 0xFD, 0x46, 0xD8, 0x2F, 0x8C, 0x4E,
		0xE8, 0xB7, 0x91, 0x0B, 0x77, 0xF4, 0x34, 0x59, 0xFA, 0x40,
		0xCD, 0xB9, 0x7E, 0x0B, 0x87, 0xC7, 0xC9, 0x0B, 0xD1, 0xB0,
		0x2F, 0xB1, 0xBF, 0xB3, 0x84, 0x66, 0x5C, 0x04, 0xF3, 0xF0,
		0x59, 0x07, 0xD6, 0x3F, 0x57, 0x0C, 0x2B, 0xC0, 0xD0, 0x66,
		0x4D, 0xFC, 0x7C, 0x35, 0xAC, 0x86, 0xB6, 0x02, 0xF6, 0xF0,
		0x36, 0x31, 0x37, 0x63, 0x8E, 0x3F, 0xC4, 0xBD, 0xCC, 0x48,
		0x5A, 0x82, 0x56, 0x76, 0x76, 0x79, 0x40, 0x35, 0x0D, 0x7D,
		0xB8, 0x00, 0x3C, 0x08, 0x13, 0xBD, 0xF0, 0x84, 0x71, 0x50,
		0xAF, 0xB7, 0x86, 0x49, 0x66, 0xCF, 0x81, 0x17, 0x75, 0x65,
		0x8A, 0xE3, 0xC0, 0x0D, 0x6F, 0x61, 0xBA, 0x6C, 0xB9, 0x1D,
		0xD7, 0x66, 0xB9, 0x7C, 0x3A, 0x9D, 0x00, 0xB9, 0x7A, 0x03,
		0xA5, 0x4E, 0xA0, 0x8C, 0x56, 0xC9, 0xE3, 0x10, 0xE3, 0x30,
		0x97, 0x17, 0x9B, 0xC7, 0xF9, 0xF2, 0x9D, 0x3D, 0xC2, 0x8E,
		0xC1, 0x95, 0xD3, 0x11, 0xDD, 0x05, 0x80, 0x6A, 0x5A, 0x4A,
		0xFF, 0x19, 0x2F, 0x11, 0x2C, 0x84, 0xD5, 0x5E, 0x4A, 0x31,
		0xF0, 0x7C, 0x02, 0x8D, 0x2F, 0xB4, 0x6C, 0x39, 0xE6, 0x94,
		0xE7, 0xFF, 0xD2, 0xAA, 0x15, 0xD1, 0xD3, 0x76, 0xEA, 0xA1,
		0xFA, 0xD3, 0x8D, 0xE6, 0xBD, 0xC6, 0xCB, 0x08, 0x01, 0xE0,
		0x86, 0x84, 0x3D, 0x94, 0x38, 0xD6, 0x67, 0xA6, 0x88, 0xC8,
		0xA6, 0x7B, 0x3E, 0xE7, 0x30, 0xBC, 0xE0, 0xFA, 0x19, 0x2D,
		0x65, 0x5C, 0xE3, 0x87, 0xC5, 0x28, 0x53, 0x6A, 0x33, 0xB7,
		0x7C, 0x12, 0xB5, 0x24, 0x7C, 0x17, 0xBC, 0xB2, 0xA4, 0x2E,
		0xFF, 0x38, 0xD3, 0xCE, 0x28, 0xC0, 0xFC, 0xE0, 0x50, 0xF1,
		0x7E, 0x48, 0x1B, 0xB6, 0x05, 0x03, 0x9C, 0xF9, 0x47, 0xDD,
		0x05, 0x67, 0xDE, 0x5C, 0x96, 0xCF, 0x3B, 0x3B, 0xC8, 0x81,
		0xAD, 0x5F, 0x43, 0x34, 0xE3, 0x6A, 0x60, 0x7F, 0x81, 0x0D,
		0x06, 0x58, 0xD1, 0x12, 0x14, 0x12, 0xE2, 0xF0, 0x60, 0x44,
		0x98, 0x6A, 0x12, 0xE8, 0x63, 0xE3, 0xA2, 0x33, 0x4D, 0xA9,
		0xCD, 0x03, 0x39, 0x22, 0x04, 0x93, 0x5A, 0xFD, 0xA6, 0x65,
		0x47, 0x35, 0x63, 0x59, 0xC5, 0x68, 0x4A, 0x81, 0x8F, 0xE3,
		0x79, 0x09, 0x3E, 0x01, 0xA4, 0xB1, 0xF2, 0x9D, 0x32, 0x8A,
		0xD0, 0xA7, 0xB5, 0xE4, 0x66, 0xCC, 0xC1, 0x1C, 0x94, 0xA4,
		0xF6, 0x5C, 0xFA, 0x9E, 0x45, 0x3C, 0x1B, 0x04, 0x85, 0x95,
		0x6E, 0x73, 0xDB, 0x02, 0x8E, 0x34, 0xCF, 0xFB, 0x3F, 0x98,
		0x37, 0x50, 0x59, 0xA1, 0x04, 0x8D, 0x72, 0x7F, 0x5D, 0x4B,
		0xFE, 0xB8, 0xF5, 0x6B, 0xE6, 0x07, 0x3D, 0x0E, 0xFC, 0xE3,
		0x4A, 0x29, 0xEE, 0x8C, 0x2B, 0xD4, 0x35, 0x27, 0x63, 0xC4,
		0x17, 0xC0, 0xD5, 0x8A, 0x0F, 0xDD, 0x64, 0x38, 0xAC, 0x42,
		0x20, 0x66, 0x2D, 0x74, 0x21, 0x9A, 0x48, 0x10, 0x2A, 0xF1,
		0xF9, 0xE7, 0xBB, 0x14, 0x23, 0x46, 0x2E, 0xBD, 0xD9, 0xB1,
		0xCF, 0x92, 0xE5, 0x8B, 0xC0, 0x20, 0x52, 0x3D, 0x7B, 0x11,
		0xF9, 0x36, 0xF5, 0xB8, 0xF7, 0x6D, 0x70, 0x1C, 0x4A, 0x05,
		0x86, 0x04, 0xBA, 0x48, 0x50, 0xD8, 0xE0, 0x27, 0xC8, 0x8B,
		0xEE, 0x5D, 0x6E, 0xA2, 0xFA, 0x17, 0xD4, 0x9D, 0x91, 0x6B,
		0x9F, 0xCE, 0x15, 0xE5, 0x3A, 0x49, 0xBF, 0x9C, 0x3B, 0xEB,
		0xD1, 0x1C, 0x1B, 0x48, 0x54, 0xCA, 0xA9, 0xB9, 0x29, 0x51,
		0x8C, 0xF2, 0x0C, 0x91, 0xF6, 0x34, 0x80, 0xB5, 0x54, 0x33,
		0x59, 0x41, 0x39, 0x65, 0x4D, 0xDA, 0xCF, 0x64, 0x02, 0x79,
		0x29, 0x6C, 0x19, 0xD5, 0x1C, 0xE5, 0x92, 0xEF, 0x58, 0x4E,
		0x4A, 0x2D, 0xDF, 0xCB, 0xD1, 0xEC, 0xA9, 0xE4, 0x49, 0x3D,
		0xC2, 0xE9, 0x09, 0x55, 0xC0, 0xED, 0x02, 0xAD, 0x37, 0xAC,
		0x19, 0xF4, 0xE7, 0x08, 0x1C, 0x5D, 0xDE, 0xB6, 0x75, 0x3F,
		0x71, 0x42, 0x9E, 0x5D, 0x88, 0x1A, 0x2F, 0x3B, 0x13, 0x84,
		0x86, 0xE8, 0x58, 0xE9, 0x82, 0x46, 0xCA, 0xF8, 0x8B, 0x91,
		0x25, 0x03, 0x31, 0xF7, 0xAC, 0xD3, 0x47, 0x0E, 0x07, 0xE5,
		0x71, 0x30, 0xCD, 0xA1, 0x26, 0x45, 0x30, 0xD4, 0x4E, 0x4C,
		0x7F, 0x27, 0xB1, 0x68, 0xA3, 0xC5, 0xC7, 0x2E, 0x17, 0x23,
		0x42, 0x4F, 0xE6, 0x42, 0xE1, 0x98, 0x87, 0xCE, 0x31, 0xD7,
		0x66, 0xAF, 0x62, 0xC5, 0xED, 0x2A, 0xAE, 0x1F, 0x18, 0x9A,
		0x6A, 0x0D, 0x48, 0x72, 0xD4, 0x06, 0x9B, 0xB9, 0xC5, 0xDE,
		0xAA, 0xAD, 0xBE, 0x6A, 0xAA, 0x5E, 0x5A, 0x7B, 0xC1, 0x3E,
		0xDF, 0x9A, 0xE1, 0x96, 0xBE, 0x54, 0x2D, 0x4D, 0x22, 0x69,
		0xA3, 0x71, 0xDC, 0xD5, 0x8E, 0xED, 0xAE, 0x41, 0xBE, 0xA2,
		0xD6, 0x66, 0x69, 0x13, 0x6D, 0x25, 0x03, 0x90, 0x12, 0x0F,
		0x93, 0x52, 0x91, 0xF4, 0x70, 0xF8, 0xFA, 0xF6, 0xB4, 0x93,
		0xA9, 0x72, 0x20, 0x2F, 0xF1, 0x18, 0xD3, 0xC1, 0xFA, 0xBC,
		0x6D, 0xEE, 0xCD, 0x53, 0x33, 0x4F, 0x93, 0xDC, 0x8C, 0x73,
		0xE7, 0xD6, 0xC0, 0x42, 0x98, 0x95, 0x8A, 0xAF, 0x97, 0x88,
		0xA9, 0x1B, 0x30, 0x8C, 0xF5, 0x0A, 0xB6, 0xFD, 0xF0, 0xFA,
		0x98, 0xEB, 0x67, 0xD2, 0x51, 0x2C, 0xE7, 0x1C, 0xA3, 0x07,
		0x81, 0xA4, 0x12, 0xE5, 0x1F, 0x9E, 0x7C, 0xD6, 0x2C, 0xE1,
		0x13, 0x6B, 0xAC, 0x61, 0x8F, 0xBF, 0x1B, 0xB5, 0x6F, 0xD6,
		0x56, 0x14, 0x81, 0x9F, 0x58, 0x06, 0x75, 0x3D, 0x19, 0xC5,
		0x77, 0x9D, 0x1B, 0x44, 0xEA, 0x38, 0xDB, 0x4B, 0x41, 0x64,
		0xE1, 0x0D, 0x12, 0x71, 0x99, 0xF1, 0xF1, 0xDC, 0x0F, 0x6A,
		0xF2, 0x5F, 0xB9, 0xBF, 0x4B, 0x6B, 0xDD, 0xED, 0x31, 0x60,
		0x03, 0x5D, 0xE3, 0x3E, 0x53, 0xAD, 0x5D, 0x9E, 0x3D, 0x9E,
		0x2A, 0x0D, 0x14, 0x28, 0x95, 0xA2, 0x80, 0x5E, 0x8C, 0x89,
		0x67, 0x78, 0x9B, 0xFC, 0x04, 0xCA, 0x4A, 0x78, 0xE2, 0x28,
		0xD7, 0x06, 0x59, 0x56, 0x8D, 0x77, 0xB0, 0x92, 0xCF, 0x26,
		0x7A, 0x58, 0x55, 0xAA, 0x67, 0xF6, 0x29, 0x04, 0x37, 0x2E,
		0x93, 0xB4, 0xF3, 0x8A, 0xB6, 0x91, 0xD5, 0x93, 0xD1, 0x22,
		0xD3, 0xD7, 0xF7, 0x93, 0x49, 0x92, 0x04, 0x3E, 0x2C, 0xCD,
		0xD0, 0x14, 0x13, 0x80, 0x14, 0x1E, 0x0A, 0x70, 0x90, 0xD5,
		0xED, 0x1E, 0x50, 0xD9, 0xBB, 0x21, 0x98, 0x5E, 0x94, 0xBA,
		0x44, 0x80, 0xA6, 0xE2, 0xA7, 0x8F, 0x76, 0x5F, 0x1B, 0xF8,
		0xCC, 0x16, 0x22, 0x87, 0x07, 0x4E, 0xD5, 0x8C, 0xE9, 0x4B,
		0x4B, 0x19, 0x14, 0x77, 0x05, 0x7B, 0x66, 0x6D, 0xF2, 0x64,
		0x8B, 0xD1, 0x54, 0x4F, 0x54, 0xD8, 0x62, 0xE8, 0x0C, 0xFD,
		0x69, 0x55, 0x89, 0xA5, 0x4D, 0xF2, 0x99, 0xB6, 0x47, 0xAD,
		0x86, 0x23, 0x30, 0xE9, 0x09, 0x98, 0x58, 0x09, 0xCB, 0xB4,
		0x8C, 0xE0, 0xB5, 0xA4, 0x9E, 0x1A, 0x09, 0xBC, 0x3E, 0x4E,
		0xFC, 0x5C, 0xAF, 0x19, 0x34, 0x55, 0xB1, 0x6D, 0x81, 0xD1,
		0x2F, 0xF3, 0x4D, 0xA7, 0x09, 0x29, 0xEB, 0x5E, 0x0B, 0x92,
		0x26, 0x5F, 0x44, 0xBC, 0x16, 0x9D, 0xAA, 0x3B, 0x13, 0xCE,
		0xC3, 0xC9, 0xAA, 0x95, 0x55, 0x8F, 0x4C, 0x1B, 0x47, 0x15,
		0x2B, 0x7F, 0x69, 0x10, 0x70, 0xA3, 0x7D, 0x60, 0xDF, 0xB1,
		0xE9, 0xCD, 0x85, 0xE8, 0xE3, 0x52, 0xEB, 0xFC, 0x35, 0x7B,
		0x6A, 0xDD, 0xED, 0xEA, 0xE1, 0x9F, 0x97, 0xCB, 0x07, 0xE9,
		0xDA, 0xD5, 0x2B, 0x90, 0x7B, 0x0A, 0xB2, 0x30, 0xB8, 0x4C,
		0x18, 0x15, 0x18, 0x00, 0xB9, 0xA5, 0x68, 0x41, 0x60, 0x5D,
		0x78, 0x27, 0xF1, 0x72, 0xA6, 0xA6, 0x8A, 0x1B, 0x41, 0x28,
		0xEF, 0x95, 0x75, 0xF8, 0xB2, 0xA2, 0x73, 0xEE, 0x2A, 0x44,
		0xAF, 0x63, 0x24, 0x26, 0xEE, 0x10, 0x84, 0x77, 0x05, 0x22,
		0xC8, 0x86, 0x60, 0x7D, 0xE1, 0x8D, 0x77, 0x9E, 0x7F, 0xA6,
		0x4E, 0x62, 0x94, 0xDC, 0x72, 0x06, 0x25, 0x9C, 0x78, 0xBF,
		0xC8, 0x31, 0xC8, 0xC1, 0xDA, 0x5A, 0xB5, 0x68, 0x11, 0x25,
		0x4C, 0xD8, 0x7D, 0xD5, 0x05, 0x62, 0x9C, 0x9B, 0x94, 0x2A,
		0x61, 0x3B, 0x1F, 0xF3, 0x05, 0x8A, 0xBE, 0xD9, 0x8E, 0xFF,
		0x35, 0xD0, 0x50, 0x83, 0xF8, 0xCD, 0xE9, 0xFD, 0x18, 0x76,
		0xB7, 0xD8, 0xDA, 0x9E, 0x99, 0x53, 0xEC, 0x52, 0xB5, 0x01,
		0x06, 0x5B, 0x6D, 0xC8, 0x42, 0xC0, 0xFF, 0xB1, 0xBE, 0xF3,
		0xD0, 0x6D, 0x40, 0xEB, 0xC3, 0xE6, 0x2C, 0x6A, 0x27, 0xBA,
		0x63, 0xA3, 0xF6, 0x02, 0xF3, 0xA3, 0xFC, 0x1E, 0xA4, 0xC4,
		0x7D, 0x8E, 0x46, 0xCF, 0xEB, 0xF9, 0xB6, 0x7D, 0xED, 0x1F,
		0x88, 0x4C, 0xA0, 0x0E, 0x78, 0xAB, 0x7C, 0xD6, 0xBD, 0x3D,
		0x18, 0x6A, 0xE1, 0xFB, 0xD0, 0xF2, 0x55, 0x24, 0x45, 0x30,
		0xE0, 0x87, 0xFD, 0x5F, 0x01, 0x49, 0x18, 0xBD, 0xC2, 0x81,
		0x23, 0x44, 0xD4, 0x08, 0xB6, 0x33, 0x95, 0x4B, 0x26, 0x39,
		0xD3, 0xDD, 0xA6, 0xA2, 0x23, 0x7F, 0xD8, 0x5E, 0x8D, 0xAE,
		0x7B, 0xDA, 0xC1, 0xC2, 0x28, 0x3F, 0x80, 0x36, 0x5A, 0x9F,
		0xE4, 0x50, 0xB4, 0x64, 0xE0, 0x3C, 0xEA, 0xB9, 0x5A, 0x5D,
		0xA0, 0xD0, 0xFF, 0x07, 0xF2, 0x09, 0xC7, 0x7E, 0x95, 0x5C,
		0xE9, 0xFC, 0x0D, 0x77, 0xE5, 0x04, 0x80, 0xBF, 0xC2, 0x10,
		0x91, 0x36, 0x1B, 0x26, 0x40, 0xC6, 0xE7, 0x6E, 0x20, 0xB9,
		0xDB, 0xC1, 0x5B, 0x9B, 0x25, 0x60, 0xDF, 0x42, 0x81, 0xEA,
		0x50, 0x12, 0x6C, 0x76, 0xC5, 0x25, 0x17, 0x9B, 0xD2, 0x01,
		0xF6, 0xC5, 0x7A, 0xB5, 0xF8, 0xC6, 0x31, 0xDE, 0x4A, 0x74,
		0xF4, 0x17, 0x90, 0x02, 0xD2, 0xAE, 0x22, 0x25, 0x12, 0xB8,
		0x05, 0x76, 0x84, 0x3C, 0xC6, 0x8A, 0xFD, 0x4C, 0x04, 0x8B,
		0x3B, 0xDB, 0x85, 0xB0, 0xBC, 0xAE, 0x6F, 0x7D, 0xC0, 0x38,
		0x19, 0xE5, 0xB9, 0x06, 0x27, 0x16, 0x9C, 0x69, 0xC3, 0xCA,
		0xB0, 0x52, 0x77, 0x58, 0xB2, 0xB2, 0x74, 0xEA, 0x33, 0xB9,
		0xFF, 0x4A, 0xC4, 0x04, 0x7A, 0x1B, 0x2E, 0x2C, 0x6C, 0x95,
		0x78, 0xD3, 0x77, 0x89, 0x10, 0xB8, 0x29, 0xF9, 0x26, 0xFA,
		0x43, 0xFC, 0x97, 0xC8, 0x8F, 0xB6, 0xF7, 0xB4, 0xDD, 0x28,
		0x43, 0xE8, 0x51, 0x37, 0xC2, 0x29, 0x9D, 0xE3, 0xFF, 0x4D,
		0x11, 0xE0, 0x66, 0x5A, 0x93, 0x40, 0x08, 0x6E, 0x0C, 0x90,
		0xDA, 0xB7, 0xD2, 0x84, 0xF0, 0xC2, 0x74, 0x61, 0xE7, 0x7A,
		0x69, 0x3F, 0x0D, 0xD2, 0xF0, 0x0C, 0x13, 0x7F, 0xF8, 0x1C,
		0x48, 0x04, 0xB1, 0xF9, 0x7F, 0x34, 0xBE, 0x0B, 0x2A, 0x76,
		0x30, 0x60, 0xF4, 0xEA, 0xFA, 0x34, 0xF4, 0x64, 0xB7, 0xBA,
		0x46, 0xE5, 0xE7, 0xB0, 0x47, 0x90, 0x14, 0x25, 0x66, 0xBD,
		0xB6, 0x38, 0xCF, 0x00, 0x7A, 0xD5, 0xEC, 0xC7, 0xE9, 0xA8,
		0x7C, 0xBA, 0x47, 0xCE, 0x2B, 0xB0, 0xD0, 0x21, 0xF7, 0x56,
		0xC9, 0xCA, 0xB8, 0xC7, 0x1D, 0xC3, 0xD9, 0x9D, 0xC1, 0x68,
		0xDC, 0x3B, 0xC6, 0x72, 0x45, 0x5B, 0x11, 0xC4, 0x0C, 0xF2,
		0x8A, 0xBE, 0x2E, 0x70, 0x81, 0x63, 0xA9, 0xC1, 0x4D, 0x08,
		0xF9, 0xF1, 0x60, 0x2C, 0x2F, 0xFB, 0x94, 0xE0, 0xC5, 0x87,
		0xC8, 0x76, 0x86, 0x17, 0x18, 0xFE, 0x1E, 0xF0, 0x8F, 0x78,
		0xB9, 0xCA, 0x13, 0x58, 0x83, 0xF9, 0xC6, 0x4D, 0xA7, 0xE2,
		0x4A, 0xA2, 0x13, 0x48, 0x80, 0xF4, 0x39, 0xE5, 0x62, 0xD4,
		0xD4, 0xCA, 0x8F, 0x1F, 0x4A, 0x0E, 0xB8, 0xF4, 0xBD, 0x29,
		0x46, 0xF3, 0x63, 0x61, 0x50, 0x79, 0xE6, 0x9A, 0x11, 0x92,
		0xD0, 0x69, 0xAB, 0x54, 0xA7, 0x13, 0x85, 0x3F, 0x0C, 0xD6,
		0x27, 0x64, 0xE4, 0xF2, 0xDA, 0x09, 0x43, 0xF9, 0xF2, 0x95,
		0x8C, 0x96, 0xB6, 0x52, 0x36, 0xF2, 0xB5, 0x98, 0x88, 0x8C,
		0xCA, 0x61, 0x49, 0x47, 0xC7, 0xAF, 0xD4, 0xD2, 0x18, 0x3D,
		0xA3, 0x45, 0x8B, 0x74, 0x7C, 0x7D, 0xE5, 0x18, 0xFA, 0xD8,
		0x05, 0xB1, 0xFC, 0x5B, 0xD9, 0x98, 0x18, 0x7F, 0x27, 0xC0,
		0x50, 0x28, 0x85, 0x97, 0x4A, 0xA4, 0xE1, 0xCA, 0x0B, 0xD5,
		0xE2, 0x49, 0xAA, 0xEF, 0xF0, 0x66, 0x2E, 0xCF, 0xD2, 0x5E,
		0xB2, 0x05, 0xC1, 0x0D, 0x23, 0x15, 0x06, 0x2B, 0x11, 0xD3,
		0x84, 0xB7, 0x1C, 0xC3, 0x2C, 0x94, 0xAC, 0xB9, 0x43, 0x4E,
		0xDB, 0x98, 0x1E, 0xBC, 0xE6, 0x29, 0xC6, 0x02, 0x87, 0xEB,
		0x6B, 0x9C, 0x36, 0x43, 0x83, 0x7D, 0xBD, 0x31, 0x50, 0x8D,
		0xF2, 0xBD, 0x62, 0x2A, 0x91, 0xC9, 0x72, 0x70, 0xF1, 0x78,
		0x52, 0x9A, 0x6B, 0x09, 0xA1, 0x35, 0x83, 0x7E, 0x3F, 0x74,
		0xA3, 0xF9, 0xE8, 0x74, 0xFD, 0x15, 0x80, 0x8F, 0x33, 0x1E,
		0x0C, 0xFE, 0x54, 0xAF, 0x57, 0x7C, 0xCE, 0xA1, 0x3A, 0x25,
		0x9D, 0x69, 0xFB, 0x70, 0x5F, 0x68, 0xB2, 0xB6, 0xD3, 0xB7,
		0x9B, 0xC9, 0x30, 0xA5, 0x3D, 0x4A, 0x3A, 0x30, 0xCB, 0xCB,
		0xEE, 0x10, 0xDA, 0xEC, 0xE8, 0xB1, 0x02, 0x1D, 0xC0, 0x74,
		0xED, 0x1E, 0x48, 0x6F, 0x19, 0xE0, 0x43, 0x22, 0xA4, 0xC9,
		0xE1, 0x6F, 0x9F, 0x0E, 0x0B, 0xF5, 0xAF, 0x57, 0xCD, 0x18,
		0xF1, 0xBB, 0xC0, 0x83, 0x95, 0xB7, 0x4A, 0x3F, 0x92, 0x04,
		0x31, 0x93, 0x26, 0x73, 0xC6, 0xA6, 0xBF, 0xE0, 0xFC, 0x0A,
		0x62, 0x2B, 0xEC, 0x8C, 0x4E, 0x84, 0xC7, 0x25, 0x2C, 0x7B,
		0xB5, 0x34, 0xB1, 0xE6, 0xE8, 0xB3, 0xC5, 0x07, 0x4F, 0x15,
		0xEB, 0xA6, 0x4E, 0x91, 0x55, 0x30, 0xF3, 0x67, 0x33, 0x37,
		0x70, 0x65, 0x8B, 0x7E, 0xFE, 0xD9, 0x6E, 0x86, 0x8A, 0xE5,
		0xCC, 0x35, 0xE6, 0xE1, 0x95, 0x87, 0xC3, 0x19, 0xDA, 0xF9,
		0xDE, 0x2F, 0x45, 0x87, 0x88, 0x61, 0x05, 0x5B, 0x13, 0xD9,
		0x69, 0x94, 0x5B, 0x12, 0x5D, 0x4A, 0xBC, 0x74, 0xFF, 0xC2,
		0xAD, 0xA4, 0x37, 0x95, 0x9D, 0x74, 0x07, 0x54, 0xEC, 0x69,
		0x09, 0x5C, 0xFB, 0x8E, 0x4A, 0xCD, 0xB0, 0x48, 0xF3, 0xFD,
		0x9C, 0x1C, 0x2B, 0xC1, 0x7A, 0x6A, 0xAA, 0xEE, 0x36, 0x81,
		0xF1, 0xCC, 0x76, 0xD6, 0x88, 0x28, 0xFD, 0xF5, 0xD0, 0x7B,
		0x5E, 0xA2, 0x77, 0x3E, 0x9B, 0xE4, 0xFE, 0x05, 0x00, 0x93,
		0xCC, 0xAD, 0x41, 0x76, 0x99, 0x35, 0xCD, 0x34, 0x8A, 0x6E,
		0xF0, 0x66, 0xF7, 0x63, 0xA7, 0xA4, 0x41, 0xFA, 0x94, 0x4A,
		0x61, 0x7C, 0xF6, 0x34, 0x73, 0x36, 0x52, 0xF8, 0xF6, 0x05,
		0xD8, 0x9A, 0x55, 0x2E, 0xE5, 0x98, 0x71, 0xB7, 0xD1, 0x1C,
		0x73, 0x5F, 0x72, 0xC7, 0xB0, 0xD2, 0xD8, 0xC9, 0x36, 0x91,
		0xE9, 0xD6, 0x9F, 0x55, 0xC1, 0xB1, 0x46, 0xB3, 0x2D, 0x6A,
		0x52, 0x40, 0x94, 0xFD, 0x35, 0xA0, 0x39, 0xD0, 0x2D, 0xDE,
		0x36, 0x2A, 0xD1, 0x33, 0xA0, 0x23, 0x2E, 0x3D, 0xE7, 0xA1,
		0xA0, 0xA4, 0x38, 0x8A, 0x02, 0x0D, 0x3B, 0x8E, 0xFE, 0x13,
		0x77, 0xA7, 0xE2, 0xA2, 0x50, 0x13, 0x82, 0xA0, 0x04, 0x98,
		0x74, 0x79, 0x6E, 0x6F, 0x1C, 0xEC, 0xE4, 0xDA, 0xE4, 0xC0,
		0xC4, 0xBD, 0x5B, 0x10, 0x0D, 0xA2, 0x54, 0x16, 0x9D, 0x39,
		0x39, 0xAE, 0x4A, 0x9C, 0x4D, 0x52, 0x17, 0xB5, 0x0E, 0x85,
		0xE3, 0xF2, 0xFA, 0x42, 0xB1, 0xAE, 0x47, 0x8D, 0xA1, 0x9F,
		0x47, 0x74, 0x3D, 0x48, 0xAF, 0xC5, 0xFA, 0xFA, 0x58, 0x5E,
		0x99, 0x66, 0x93, 0xFF, 0xDB, 0x26, 0x27, 0xB6, 0xFC, 0xBD,
		0x8C, 0x2B, 0x4F, 0x21, 0x08, 0x74, 0x82, 0x55, 0x0A, 0x65,
		0x01, 0xED, 0x7D, 0x30, 0x38, 0x14, 0xD4, 0x71, 0x52, 0x4A,
		0x10, 0xA3, 0xE1, 0x5D, 0xA6, 0xEC, 0x67, 0x7D, 0x98, 0xBC,
		0x14, 0x49, 0x7A, 0x1E, 0xD0, 0x7F, 0x89, 0x12, 0xBA, 0x02,
		0x0F, 0x0B, 0x9D, 0x39, 0xBC, 0x3F, 0x52, 0xFA, 0x7D, 0x74,
		0xE2, 0x3B, 0x00, 0x77, 0xDC, 0x5C, 0xAB, 0xD8, 0x33, 0xC2,
		0x76, 0x92, 0x61, 0xA8, 0x9F, 0xBE, 0x2D, 0xCF, 0x2A, 0xB5,
		0x88, 0xB7, 0xAD, 0x52, 0xC2, 0xED, 0x00, 0x8C, 0x36, 0xDA,
		0xAF, 0xDD, 0x10, 0xF9, 0x64, 0xB3, 0x08, 0xA8, 0xEE, 0xFA,
		0x35, 0x11, 0xC9, 0x67, 0x54, 0xD2, 0xF8, 0x9E, 0xC6, 0xF3,
		0x1A, 0x01, 0xD7, 0x8D, 0x6E, 0x47, 0xEF, 0x08, 0x20, 0xCA,
		0x56, 0xE0, 0xEA, 0xFD, 0x2A, 0xCB, 0x64, 0x59, 0x46, 0x66,
		0xB1, 0x47, 0xD1, 0x01, 0x43, 0xF6, 0xFE, 0x9D, 0x5B, 0xE2,
		0xB8, 0x57, 0x6C, 0x20, 0xD8, 0xD9, 0xA8, 0x75, 0x53, 0xAA,
		0x1A, 0xB9, 0xF0, 0xC0, 0xFB, 0x78, 0x08, 0xC8, 0x05, 0xB0,
		0xB4, 0xB0, 0xF5, 0x33, 0x13, 0xF2, 0xDB, 0x7F, 0x03, 0xC1,
		0xF1, 0x09, 0xC2, 0x10, 0x8B, 0x52, 0x58, 0xF2, 0xC5, 0x30,
		0x7B, 0x19, 0x0A, 0x3B, 0x99, 0xC5, 0x3A, 0xAD, 0x81, 0x46,
		0xB8, 0xF0, 0xA7, 0x1D, 0xF2, 0x43, 0xAA, 0xC5, 0x13, 0x9D,
		0x0A, 0x23, 0x8F, 0x5D, 0xE6, 0xB3, 0x7F, 0x2A, 0x6D, 0x7D,
		0xBE, 0xC0, 0xB9, 0xAC, 0xCD, 0xA7, 0xF3, 0x9F, 0xF9, 0x27,
		0x90, 0xE5, 0xA3, 0x92, 0x1F, 0xD8, 0xBD, 0x41, 0x3A, 0xF8,
		0xD0, 0x17, 0xB3, 0x7F, 0xDC, 0x48, 0xC1, 0xFD, 0xDA, 0x93,
		0xEC, 0xCA, 0x93, 0x25, 0xD8, 0xD7, 0xC9, 0x65, 0x5E, 0x2D,
		0x3B, 0x7A, 0xDD, 0xC6, 0x1D, 0xE6, 0x6C, 0x50, 0x5B, 0xC1,
		0x46, 0x6B, 0xF5, 0x00, 0xA2, 0x97, 0x98, 0x82, 0x11, 0x2B,
		0xD4, 0x2F, 0x19, 0xD6, 0xED, 0xAD, 0xCC, 0x10, 0xE2, 0x7A,
		0x99, 0x04, 0xAD, 0x30, 0xF9, 0x72, 0xAF, 0x4C, 0x01, 0xEE,
		0x74, 0x96, 0x2D, 0xF4, 0x69, 0xDA, 0x78, 0x5F, 0x35, 0xBB,
		0x52, 0x62, 0x69, 0x0B, 0x5D, 0x35, 0xED, 0x8C, 0x12, 0x1E,
		0xD5, 0xE0, 0xE6, 0x88, 0xB5, 0xB0, 0x46, 0x6E, 0x60, 0xD8,
		0x52, 0xEA, 0xE4, 0x7B, 0x37, 0x32, 0x64, 0x06, 0x63, 0x00,
		0xCC, 0xA2, 0xF0, 0xC8, 0x03, 0x4F, 0xB5, 0xF3, 0x9E, 0x5A,
		0x4A, 0x01, 0x36, 0xE6, 0x6A, 0xAC, 0xC8, 0x92, 0xFB, 0xCD,
		0xDA, 0x44, 0x02, 0x26, 0x83, 0x55, 0xB1, 0x1F, 0xFF, 0xFC,
		0x8F, 0x38, 0x8E, 0x49, 0x04, 0x40, 0x8C, 0xAF, 0x1A, 0xB1,
		0x7E, 0x21, 0xB3, 0x63, 0xFE, 0xD6, 0x40, 0x96, 0x9B, 0x37,
		0x2F, 0xF4, 0x3F, 0xF1, 0x4A, 0xD1, 0xE2, 0x7D, 0x78, 0x1E,
		0x00, 0x33, 0x14, 0x51, 0xBD, 0x49, 0xD7, 0x76, 0x7C, 0x06,
		0xE3, 0xF8, 0x87, 0x3A, 0xBF, 0x25, 0x16, 0xE4, 0xA3, 0xFC,
		0x73, 0xB1, 0x53, 0x13, 0x4E, 0x4D, 0x81, 0x66, 0xC0, 0x81,
		0x46, 0x6F, 0xA7, 0xA1, 0xEB, 0x3D, 0x46, 0xC9, 0x47, 0xA1,
		0xD2, 0xB3, 0x47, 0xEE, 0x4B, 0x7E, 0x42, 0xC0, 0x57, 0x0A,
		0x52, 0x28, 0xED, 0x1F, 0xC0, 0xEB, 0x70, 0x5A, 0x1B, 0x05,
		0xEB, 0xAA, 0x2A, 0xC4, 0xF4, 0x2E, 0x76, 0xC9, 0xFB, 0x16,
		0x2F, 0x69, 0x0D, 0x91, 0x1C, 0x33, 0x9D, 0xC4, 0x6E, 0x6A,
		0x52, 0x6C, 0x10, 0xEB, 0x46, 0x3B, 0x9E, 0x1E, 0x6B, 0xE6,
		0x54, 0x76, 0x32, 0x55, 0x33, 0xEE, 0xE7, 0xE5, 0xB3, 0xB9,
		0xE8, 0x95, 0x2D, 0x67, 0xF5, 0x99, 0x64, 0x8B, 0x3E, 0x2B,
		0xD2, 0xC4, 0xD0, 0x4E, 0x5B, 0x1E, 0x98, 0x11, 0xA9, 0xA7,
		0xE8, 0x5F, 0x5E, 0x28, 0x7E, 0x25, 0x02, 0x71, 0x1C, 0x5A,
		0x5E, 0x70, 0x89, 0x3F, 0xB8, 0xAE, 0x8A, 0xA8, 0xFC, 0xF0,
		0xE3, 0x86, 0x7E, 0xC9, 0x0F, 0xD2, 0x9B, 0x35, 0xC7, 0x4F,
		0x49, 0x46, 0x55, 0x3C, 0xF9, 0xA1, 0x43, 0xB8, 0x10, 0xF2,
		0x89, 0x47, 0x11, 0xBC, 0xF4, 0xE2, 0x97, 0x6C, 0x83, 0x22,
		0x6E, 0xE1, 0x58, 0xFF, 0xE4, 0xE1, 0x00, 0xA1, 0xE0, 0x8F,
		0x17, 0x8B, 0xB8, 0x2E, 0x17, 0x53, 0x52, 0x20, 0xAB, 0x3C,
		0xD0, 0xF9, 0x8A, 0x54, 0xEF, 0xC7, 0x67, 0x7E, 0x6B, 0xEF,
		0x62, 0xB3, 0xC1, 0x80, 0x7D, 0x60, 0x9A, 0xA7, 0x5D, 0xA5,
		0xDD, 0x13, 0xB1, 0x9A, 0x47, 0x78, 0x0E, 0x79, 0x3B, 0x60,
		0xD1, 0xF9, 0x1F, 0x44, 0xA2, 0xEB, 0x42, 0xF2, 0xA7, 0x41,
		0x87, 0xDE, 0xB5, 0x91, 0x2A, 0x5F, 0x04, 0xC7, 0xA7, 0x53,
		0xB4, 0x24, 0x1A, 0xCB, 0x81, 0x73, 0xE4, 0xCC, 0x47, 0x96,
		0x08, 0xEF, 0x5D, 0x20, 0x81, 0x8C, 0x12, 0xC5, 0x47, 0x25,
		0xB4, 0x4E, 0xCB, 0x77, 0x70, 0x5F, 0x87, 0x96, 0x67, 0x8E,
		0x0F, 0x17, 0x68, 0x99, 0x81, 0x53, 0x85, 0x31, 0x0A, 0xEA,
		0x11, 0xF6, 0x40, 0x72, 0xB9, 0xF8, 0x3D, 0x3D, 0x27, 0x91,
		0xED, 0x14, 0xDE, 0x0F, 0xED, 0x35, 0x77, 0xA0, 0xFC, 0x8C,
		0x5E, 0xF6, 0xC3, 0x87, 0x09, 0xC1, 0xE7, 0x9B, 0x5E, 0xA8,
		0x1E, 0x00, 0xDA, 0x25, 0x34, 0xA1, 0x83, 0x42, 0x18, 0x6D,
		0xEB, 0xE5, 0x8C, 0x2B, 0x16, 0x19, 0xD9, 0x4F, 0x34, 0xDF,
		0xFC, 0xA4, 0x31, 0xE5, 0x10, 0x46, 0x98, 0xC4, 0xB5, 0xD1,
		0x66, 0x29, 0x22, 0xB3, 0xBD, 0x04, 0x2D, 0xB7, 0x5F, 0x21,
		0x14, 0x73, 0x44, 0x2E, 0x71, 0x6E, 0x3C, 0x5C, 0x4B, 0xBF,
		0x93, 0x4D, 0xC4, 0x81, 0x5C, 0xDB, 0x04, 0x07, 0xDD, 0x94,
		0xC3, 0x98, 0xB6, 0x5D, 0x9C, 0x5C, 0x53, 0xC8, 0x87, 0x61,
		0xD8, 0x6C, 0x9D, 0xF8, 0x60, 0x58, 0x80, 0x36, 0xEC, 0xED,
		0xDA, 0x5B, 0x5C, 0x6C, 0x2E, 0x03, 0x85, 0x76, 0x7C, 0x45,
		0x66, 0x06, 0x24, 0xFD, 0x1A, 0x75, 0xCB, 0xC4, 0x64, 0x5E,
		0x5B, 0x16, 0xB7, 0xA8, 0x2B, 0xC2, 0x50, 0x4A, 0xF8, 0xD9,
		0x21, 0x6A, 0x0E, 0x01, 0x91, 0xDD, 0x0E, 0x82, 0x22, 0x01,
		0xBA, 0xAF, 0xDF, 0x3F, 0x0A, 0x04, 0x2F, 0xEE, 0x42, 0x58,
		0x31, 0xA1, 0xA0, 0xFB, 0xEE, 0x6C, 0x38, 0x6B, 0xEE, 0x51,
		0x05, 0x40, 0x26, 0x5B, 0x85, 0x70, 0x0A, 0x53, 0xEB, 0x5C,
		0xCE, 0xFD, 0x66, 0x05, 0x59, 0x4A, 0x82, 0xF0, 0x13, 0x19,
		0x13, 0x55, 0x8A, 0x1E, 0x34, 0xA6, 0xDA, 0x19, 0x54, 0x7D,
		0x3A, 0xDC, 0x7F, 0x88, 0x07, 0x14, 0xFA, 0x62, 0x30, 0x1E,
		0x1E, 0x42, 0x44, 0xCD, 0x77, 0x66, 0x47, 0x7B, 0x14, 0x43,
		0xE9, 0x72, 0x48, 0x6D, 0xD9, 0x0C, 0x55, 0x21, 0xBD, 0xB5,
		0x8F, 0x29, 0x96, 0x30, 0x2A, 0x46, 0x64, 0xAD, 0x80, 0x6E,
		0x8E, 0x6C, 0x63, 0x5C, 0x87, 0xED, 0x62, 0xE0, 0x40, 0xF2,
		0x5B, 0x6A, 0x42, 0xEB, 0xAD, 0x64, 0x23, 0xE4, 0x35, 0x87,
		0x82, 0x88, 0xCE, 0x94, 0x7A, 0xA5, 0x30, 0xA6, 0xAD, 0xB8,
		0x59, 0xA3, 0xE5, 0x57, 0x2E, 0x96, 0xD7, 0x6B, 0x51, 0x0E,
		0x93, 0xD8, 0xAA, 0x8C, 0x49, 0x0A, 0xCE, 0xC6, 0xD6, 0xD8,
		0xFE, 0xBD, 0x50, 0x66, 0xC5, 0xBD, 0x02, 0x3B, 0xC7, 0xD9,
		0xFB, 0xF6, 0xB8, 0xEA, 0x7F, 0xA5, 0x74, 0x11, 0xB1, 0x84,
		0x3D, 0x21, 0xA1, 0x86, 0xED, 0x48, 0x46, 0xD1, 0xCF, 0x42,
		0x7F, 0x2B, 0x9C, 0x66, 0xA2, 0xD8, 0x49, 0x59, 0x68, 0xE0,
		0x1B, 0x15, 0x8D, 0xF3, 0x95, 0x81, 0x20, 0xF0, 0x5D, 0x19,
		0x4C, 0x4D, 0xF2, 0x73, 0x3E, 0xA0, 0x96, 0x99, 0xC0, 0x2E,
		0x9D, 0x1D, 0x50, 0x54, 0xCD, 0x91, 0x78, 0xD8, 0xDA, 0xDC,
		0xBF, 0xC0, 0x5F, 0xA4, 0xF7, 0x7B, 0xC2, 0x38, 0x20, 0xAE,
		0x76, 0xE3, 0xEF, 0x20, 0x96, 0x4C, 0xBD, 0x14, 0x3F, 0x7C,
		0xB4, 0x08, 0x1A, 0xC7, 0x84, 0x12, 0x90, 0x4F, 0x2C, 0xB5,
		0x30, 0x97, 0xD3, 0x4C, 0x3A, 0xE3, 0x9A, 0xAF, 0x88, 0x7D,
		0x1D, 0x4B, 0xA2, 0xDA, 0xF5, 0x22, 0x30, 0xE1, 0xA3, 0xF9,
		0x46, 0xA8, 0x70, 0xAE, 0x28, 0xF9, 0xA3, 0xF9, 0x53, 0x80,
		0xF9, 0xDF, 0x83, 0xC2, 0x88, 0x6C, 0x06, 0x1F, 0x01, 0x3A,
		0x46, 0x03, 0xBC, 0xAF, 0x2B, 0xDD, 0x02, 0xBE, 0xC1, 0xD8,
		0x93, 0x8A, 0x2E, 0xAF, 0x77, 0xB8, 0x03, 0xD9, 0x6C, 0x3B,
		0x1D, 0xE0, 0xB2, 0x45, 0x2D, 0x48, 0x28, 0xA5, 0x99, 0xB6,
		0xFA, 0xC4, 0x6C, 0xD2, 0xA8, 0xBE, 0x60, 0x73, 0x88, 0x95,
		0x35, 0xC1, 0xD9, 0x06, 0xC5, 0x27, 0x28, 0xAC, 0x67, 0x21,
		0x23, 0x00, 0xE4, 0x6E, 0x0D, 0xEF, 0xC1, 0x1D, 0x22, 0xB3,
		0x72, 0x38, 0x80, 0x9B, 0xDE, 0x59, 0xB2, 0x57, 0x5B, 0xDF,
		0xDA, 0x26, 0xDA, 0x50, 0xAA, 0x2B, 0xCF, 0x74, 0x63, 0x50,
		0xFC, 0x9B, 0xDD, 0x19, 0x52, 0xD8, 0x1D, 0x1E, 0x29, 0x04,
		0x9D, 0x3B, 0x9A, 0x62, 0xE5, 0xDF, 0xED, 0xA9, 0x7A, 0x29,
		0x77, 0x24, 0x47, 0x45, 0x3B, 0x75, 0xD9, 0x01, 0x8C, 0x37,
		0xF3, 0x5E, 0x59, 0xBD, 0x16, 0x58, 0x18, 0x6D, 0x44, 0x59,
		0x82, 0x29, 0x7B, 0xA3, 0x24, 0x16, 0x9B, 0xB3, 0x52, 0x36,
		0x63, 0xC3, 0xD3, 0xBC, 0x0D, 0x12, 0xF4, 0xF5, 0x87, 0x96,
		0x32, 0xA0, 0x47, 0xA5, 0xA8, 0xC2, 0x51, 0x36, 0x16, 0x37,
		0x16, 0x50, 0x6C, 0x4D, 0xE5, 0x84, 0x36, 0x3C, 0x3B, 0xC3,
		0x0B, 0x0B, 0xA2, 0xD4, 0xDC, 0x9B, 0x21, 0xAA, 0xF3, 0x2C,
		0x90, 0x2E, 0xBE, 0xB1, 0xA7, 0x68, 0xAD, 0xB1, 0xDF, 0x5F,
		0xD9, 0xA7, 0xBE, 0xF4, 0x0B, 0x3F, 0x5F, 0xF4, 0x67, 0x00,
		0x3E, 0x84, 0x24, 0x46, 0x47, 0xD9, 0x08, 0xF5, 0x56, 0x87,
		0x22, 0x2A, 0xDD, 0xD5, 0x2C, 0x21, 0x90, 0xCB, 0x42, 0x4D,
		0x84, 0x75, 0xC0, 0x5F, 0x09, 0x53, 0xE4, 0xAA, 0xC9, 0x33,
		0x23, 0x43, 0xD5, 0xD0, 0x01, 0xBC, 0x50, 0x4F, 0x7C, 0xE9,
		0xB6, 0x32, 0x26, 0xA0, 0xBF, 0xE2, 0xFB, 0xA3, 0x9E, 0xD4,
		0x73, 0x82, 0x38, 0x3D, 0xD1, 0x40, 0x41, 0x75, 0x95, 0xC4,
		0xC2, 0xBE, 0x27, 0x7D, 0x27, 0xE1, 0x0D, 0xD9, 0x78, 0xCB,
		0xF6, 0x03, 0x4E, 0x7A, 0x8B, 0x55, 0xB5, 0x01, 0xD4, 0x9E,
		0x18, 0xA6, 0x69, 0xF1, 0xB1, 0x4B, 0x25, 0xE3, 0xA9, 0x79,
		0x87, 0x47, 0x52, 0xCB, 0x63, 0x41, 0x87, 0xDC, 0xA2, 0x94,
		0x81, 0x13, 0x49, 0x08, 0xC6, 0x24, 0xCE, 0x5D, 0x21, 0x7A,
		0xBA, 0xC8, 0xBC, 0xE8, 0x8F, 0x26, 0xFF, 0xAF, 0x5A, 0xC9,
		0x76, 0x8F, 0xEC, 0xC2, 0xD3, 0xBC, 0x28, 0x1C, 0xD8, 0xAC,
		0xD0, 0x70, 0xCA, 0x91, 0x87, 0xA7, 0xCD, 0x69, 0xA0, 0x0B,
		0x15, 0x9B, 0x73, 0x96, 0x3C, 0x5B, 0x04, 0x7B, 0xE7, 0x3B,
		0xED, 0x58, 0x4D, 0x6C, 0x25, 0x08, 0xBB, 0x17, 0x71, 0x67,
		0x30, 0xE3, 0xF8, 0x94, 0x11, 0xF6, 0x24, 0xC0, 0x63, 0x77,
		0x62, 0xED, 0x07, 0xA7, 0x36, 0x9B, 0x98, 0xA7, 0xD9, 0x45,
		0x24, 0x58, 0x00, 0xD0, 0x40, 0xE1, 0x67, 0xEF, 0x52, 0xEB,
		0x1F, 0x2F, 0x56, 0xA8, 0xB3, 0x4C, 0x37, 0x0B, 0x3D, 0x15,
		0xA0, 0x28, 0xD4, 0xE7, 0x0C, 0x2E, 0x3A, 0xB4, 0x5B, 0x77,
		0xF3, 0xDA, 0x46, 0xC8, 0x10, 0xE4, 0x2F, 0x2F, 0xA0, 0x4F,
		0x0B, 0xC5, 0xE9, 0x37, 0x5F, 0x6F, 0x1C, 0x42, 0x5C, 0x92,
		0x13, 0x8C, 0x82, 0x1F, 0x0E, 0x34, 0x3C, 0x30, 0x00, 0xE1,
		0x31, 0x72, 0xB4, 0x3B, 0x22, 0x7E, 0x9E, 0x72, 0xEB, 0x10,
		0xDE, 0x4A, 0x98, 0xC0, 0x34, 0xAF, 0x5C, 0x11, 0x17, 0x0D,
		0x0F, 0x6B, 0x27, 0x0F, 0xB3, 0x16, 0x58, 0xE4, 0x3B, 0xFB,
		0xF4, 0x0E, 0x7A, 0x38, 0x0F, 0x40, 0x74, 0x45, 0x48, 0x76,
		0x7C, 0x2B, 0xAD, 0x83, 0xA2, 0xC6, 0xA3, 0xDA, 0x04, 0x4A,
		0x2F, 0x31, 0x3E, 0x7B, 0x19, 0x40, 0xB3, 0x28, 0x80, 0x6B,
		0xF4, 0x1B, 0x9E, 0x99, 0x93, 0x81, 0x11, 0x3B, 0x9A, 0x39,
		0xF9, 0x8B, 0x42, 0x56, 0x88, 0x29, 0xAC, 0xDA, 0x38, 0x33,
		0x82, 0x4D, 0x00, 0xDA, 0x94, 0x88, 0x90, 0x8B, 0x73, 0xA9,
		0x79, 0x6D, 0xD1, 0x8B, 0xCD, 0x05, 0x54, 0x5A, 0x5E, 0xDB,
		0xFA, 0x25, 0x28, 0x04, 0x24, 0xC3, 0xB9, 0xCE, 0x06, 0x8C,
		0xEA, 0x2D, 0x96, 0x51, 0xF6, 0xC2, 0x20, 0xF3, 0xF3, 0xF1,
		0x5C, 0x0B, 0x60, 0xC2, 0x3D, 0x1D, 0x4F, 0xFB, 0x25, 0x8F,
		0xB6, 0x96, 0x07, 0xCE, 0x85, 0x78, 0x0F, 0x21, 0xCD, 0x87,
		0x7A, 0xEC, 0x97, 0xD4, 0x6C, 0xB7, 0x8B, 0x20, 0x86, 0xF2,
		0x95, 0x0A, 0xF0, 0x65, 0xA9, 0x80, 0x9E, 0x37, 0x25, 0xF2,
		0x4B, 0x33, 0xBB, 0x9E, 0x61, 0x56, 0xE4, 0x21, 0x72, 0x7D,
		0xB5, 0xB6, 0xFE, 0xE4, 0x57, 0x06, 0x8B, 0xFF, 0xFA, 0x63,
		0x0F, 0x9A, 0x2F, 0xE0, 0x2D, 0x17, 0xED, 0xEC, 0xEF, 0xF8,
		0xF6, 0xC5, 0xD7, 0xCB, 0x6B, 0xD2, 0xF2, 0x22, 0xD3, 0x66,
		0x1B, 0x9A, 0xBA, 0xB6, 0xD3, 0xC3, 0x22, 0xDE, 0x52, 0x99,
		0x78, 0x95, 0xFC, 0xFD, 0xBD, 0xB6, 0x05, 0xAD, 0xC0, 0x35,
		0xBD, 0x00, 0xBC, 0x2C, 0x33, 0x88, 0x7A, 0x20, 0xC2, 0xAA,
		0x9F, 0x03, 0x94, 0xF7, 0x61, 0xCD, 0x51, 0x34, 0x8E, 0xEE,
		0x4C, 0xAB, 0xEC, 0xAC, 0x25, 0x2A, 0x4D, 0x51, 0xEC, 0x2E,
		0x7C, 0x92, 0x26, 0x68, 0x5E, 0x7C, 0x68, 0x0B, 0xEB, 0x61,
		0xB2, 0x6E, 0x8A, 0x3F, 0x45, 0x9D, 0xBF, 0x4A, 0x4F, 0xAA,
		0xE5, 0x82, 0x3F, 0xE7, 0x9D, 0x1C, 0x7D, 0x24, 0x38, 0x1F,
		0xE9, 0x2C, 0x8E, 0x4D, 0xEA, 0xE5, 0xA0, 0x54, 0x41, 0xB0,
		0x77, 0x36, 0xB7, 0xCD, 0xD5, 0xE9, 0xC7, 0x6A, 0xD0, 0x5D,
		0x49, 0x37, 0x7E, 0x7B, 0x7B, 0x66, 0x77, 0xC7, 0xBC, 0xA0,
		0x2D, 0xE0, 0x42, 0xA2, 0x2E, 0x87, 0x58, 0x52, 0xA9, 0x16,
		0xFA, 0x09, 0x81, 0xCD, 0x25, 0x61, 0xE8, 0xB7, 0xAE, 0x59,
		0xBC, 0xAA, 0x04, 0x7B, 0xC1, 0x18, 0xB3, 0xE7, 0x9F, 0x93,
		0xFB, 0x9B, 0x5D, 0xC3, 0x7E, 0x9C, 0x97, 0xA0, 0x84, 0x83,
		0x43, 0x84, 0x14, 0xD5, 0xD8, 0x0B, 0x50, 0x32, 0x62, 0x1A,
		0xFA, 0x73, 0x62, 0x19, 0x64, 0xAB, 0x7C, 0xDA, 0x42, 0xB3,
		0x0B, 0x9B, 0xD6, 0xBE, 0xEB, 0xD3, 0x14, 0xD1, 0xFF, 0x87,
		0x7C, 0xE3, 0x2A, 0x11, 0x72, 0x20, 0x56, 0x81, 0x55, 0x04,
		0xCC, 0xBF, 0x83, 0x75, 0x51, 0xE7, 0x6F, 0x71, 0x95, 0x7D,
		0x24, 0x14, 0x83, 0xAF, 0x94, 0xB6, 0xA5, 0x6B, 0x56, 0x2E,
		0x0F, 0x1C, 0xF1, 0xE5, 0x90, 0x4C, 0x5A, 0xD2, 0x8A, 0xF9,
		0x29, 0xA1, 0xEE, 0xC5, 0x62, 0xA0, 0x52, 0x7C, 0x2A, 0x42,
		0x02, 0x23, 0xEE, 0xF1, 0x6D, 0x0D, 0x2F, 0x84, 0xE3, 0xFD,
		0x1A, 0x80, 0x22, 0x44, 0xC9, 0x2F, 0x60, 0xD5, 0xF2, 0x7B,
		0x29, 0x71, 0x94, 0x84, 0x6E, 0x91, 0x21, 0xB2, 0x33, 0x78,
		0xB4, 0xBF, 0xEC, 0x79, 0x72, 0xD8, 0x4B, 0x51, 0xBB, 0x83,
		0xC8, 0x83, 0x44, 0x1F, 0x7D, 0x98, 0x32, 0x10, 0x77, 0x96,
		0x18, 0x06, 0x3C, 0x16, 0xBC, 0x75, 0xFC, 0x3A, 0x42, 0x28,
		0x90, 0xF7, 0x08, 0x27, 0x1D, 0x98, 0x19, 0x55, 0x54, 0x41,
		0xC5, 0x78, 0xDC, 0x2E, 0x92, 0xE5, 0xE6, 0x90, 0xBE, 0xB1,
		0xB9, 0xB1, 0xA9, 0x3C, 0xA0, 0xD1, 0x58, 0x05, 0xCF, 0xC7,
		0x1A, 0x11, 0xCA, 0x97, 0xB1, 0x5A, 0x69, 0x2B, 0xB0, 0x3C,
		0x4A, 0xEC, 0xC1, 0x21, 0xD2, 0xB8, 0xD8, 0x94, 0x1B, 0x8B,
		0x3C, 0x16, 0x4C, 0x0D, 0xBF, 0x41, 0x51, 0x71, 0x52, 0xCD,
		0xD4, 0x63, 0x42, 0x16, 0x62, 0x98, 0x31, 0xB4, 0xC2, 0xAD,
		0x71, 0x23, 0xA6, 0x0A, 0xBF, 0x79, 0xF3, 0xFB, 0xAE, 0xF2,
		0x9F, 0x2C, 0x68, 0x12, 0xDD, 0x7D, 0xAA, 0xFC, 0x4B, 0x74,
		0x07, 0xDA, 0x4E, 0x57, 0x47, 0x09, 0x66, 0xB5, 0x1C, 0x80,
		0xAB, 0x77, 0xC0, 0x84, 0x9D, 0x19, 0x93, 0xF6, 0x0A, 0x0B,
		0xB5, 0x59, 0x4C, 0xA1, 0x25, 0x1B, 0x6D, 0xA8, 0xCC, 0x7F,
		0x19, 0xDD, 0x2B, 0x83, 0x31, 0x0F, 0x3D, 0x3A, 0x01, 0xC9,
		0x84, 0xA9, 0x54, 0xFA, 0x94, 0xC4, 0x48, 0xD6, 0xBB, 0xA9,
		0xFC, 0xC4, 0xDA, 0x77, 0xF7, 0x39, 0xEA, 0x8B, 0x91, 0x48,
		0xD3, 0xEB, 0x5F, 0x71, 0x7B, 0x40, 0x3B, 0x09, 0x5A, 0x59,
		0xC3, 0x90, 0xBB, 0xE4, 0x82, 0xAD, 0x92, 0x36, 0x2C, 0xB6,
		0x64, 0x40, 0x94, 0x2B, 0xF1, 0x95, 0x1D, 0x67, 0x8C, 0x8C,
		0x68, 0x00, 0xCA, 0xBC, 0xAD, 0x07, 0x57, 0xE7, 0x31, 0xC1,
		0x7A, 0xAC, 0x16, 0x4F, 0xD4, 0xEE, 0x59, 0xF8, 0xDE, 0x84,
		0xC5, 0x52, 0x8C, 0x5B, 0x05, 0x80, 0xF2, 0xB6, 0x59, 0xAD,
		0x01, 0xB6, 0xB6, 0x70, 0xA8, 0x98, 0x7E, 0x14, 0xDC, 0x60,
		0x31, 0x58, 0x89, 0xB9, 0x11, 0x69, 0xC1, 0x87, 0xE1, 0x4A,
		0x42, 0x1F, 0x1D, 0x43, 0xFC, 0x10, 0x99, 0x48, 0xD1, 0x8A,
		0x67, 0x88, 0x3F, 0xBB, 0x7A, 0x8C, 0x16, 0xD7, 0xD6, 0xB6,
		0x39, 0x9B, 0x17, 0xFC, 0xC7, 0xC2, 0x63, 0x25, 0xA4, 0xD6,
		0x0B, 0x5A, 0x3F, 0x99, 0x68, 0x7B, 0x43, 0xCC, 0x48, 0x7F,
		0x4E, 0xB8, 0xB0, 0xF4, 0x5C, 0xF9, 0x1A, 0xEC, 0xDF, 0xB5,
		0x40, 0xCC, 0xE1, 0x20, 0xE4, 0xAA, 0x9B, 0x60, 0x10, 0x44,
		0x8F, 0xE7, 0x00, 0x81, 0x7F, 0xC4, 0x82, 0x5A, 0xA7, 0x59,
		0x42, 0x56, 0xA3, 0xB8, 0xD6, 0x1D, 0x46, 0xB2, 0x5D, 0xD8,
		0x07, 0xF3, 0x0D, 0xA4, 0x57, 0x8A, 0x1F, 0xC1, 0xF3, 0x10,
		0xD5, 0x85, 0x23, 0xEC, 0x37, 0x25, 0xD5, 0xF6, 0xA7, 0xBA,
		0xD1, 0x21, 0xE0, 0xEC, 0x5A, 0x97, 0x57, 0x01, 0xE2, 0xD8,
		0xBA, 0x70, 0xE6, 0x34, 0x31, 0xC7, 0x0D, 0xD6, 0x78, 0x26,
		0x7E, 0x54, 0x7C, 0x88, 0x46, 0x9C, 0xEE, 0xFA, 0xB9, 0xB9,
		0xF5, 0x01, 0x23, 0x43, 0x0C, 0x38, 0xBA, 0x1A, 0x7B, 0xCF,
		0xB0, 0x9E, 0x16, 0x70, 0x98, 0x1B, 0x22, 0x12, 0xE6, 0xF0,
		0xDE, 0xCB, 0xF8, 0xCA, 0xFD, 0xDE, 0x5A, 0x9A, 0x07, 0xD5,
		0x67, 0x07, 0x9F, 0xF9, 0x48, 0x53, 0x8C, 0x08, 0x46, 0x17,
		0xB7, 0xEC, 0xA1, 0x70, 0xF1, 0x7C, 0x32, 0xD6, 0x22, 0x4D,
		0x72, 0x56, 0x02, 0x58, 0xB2, 0x3E, 0x54, 0xAF, 0x4C, 0x52,
		0x50, 0x72, 0x06, 0x1D, 0x40, 0xA2, 0x1B, 0x49, 0x47, 0xBB,
		0x69, 0x94, 0x69, 0xE9, 0x49, 0x91, 0x09, 0x4D, 0x7A, 0x49,
		0x30, 0xAA, 0x0C, 0x0E, 0xF2, 0xA8, 0xAA, 0xBA, 0x42, 0x90,
		0x0E, 0xF5, 0x9D, 0xAF, 0x01, 0x00, 0x7D, 0xAB, 0xBD, 0x6B,
		0xC6, 0x63, 0x81, 0x2F, 0x5C, 0x65, 0xFE, 0xAC, 0x13, 0xF5,
		0xCE, 0x63, 0xA5, 0xE9, 0x02, 0x52, 0x72, 0x84, 0xC8, 0xFE,
		0x0D, 0xE1, 0xAD, 0xC8, 0xEB, 0xC5, 0xF9, 0x67, 0x96, 0x5C,
		0x86, 0xD8, 0xB0, 0x1F, 0x32, 0x33, 0xFC, 0x7F, 0x92, 0xBB,
		0xA6, 0xB7, 0x9C, 0xF0, 0x58, 0xF9, 0xBB, 0x97, 0x17, 0x98,
		0x5D, 0x04, 0x84, 0x6E, 0x8B, 0xF9, 0x96, 0x7D, 0xAF, 0xEE,
		0x20, 0xA5, 0x64, 0xAF, 0xE5, 0x27, 0x5D, 0x60, 0x80, 0x19,
		0xEA, 0xD0, 0x88, 0x4D, 0x6C, 0x98, 0xDB, 0x43, 0x46, 0x9A,
		0xB4, 0x1A, 0x0D, 0xFB, 0x0B, 0x6C, 0xFF, 0x0E, 0x80, 0xB3,
		0x4C, 0xE7, 0x4F, 0x1B, 0x08, 0x12, 0x50, 0x77, 0x33, 0x9E,
		0x94, 0x6F, 0xB1, 0x24, 0x93, 0x69, 0x70, 0x83, 0x7C, 0x0B,
		0xAC, 0xE3, 0x64, 0xFF, 0x0A, 0x29, 0xB8, 0xB4, 0x27, 0x19,
		0xC5, 0xF2, 0xD0, 0x32, 0x3E, 0x7E, 0x39, 0xF9, 0xD0, 0xE8,
		0xA3, 0x88, 0x27, 0xAB, 0x99, 0xBE, 0x44, 0xC5, 0xAF, 0x32,
		0x85, 0xC5, 0x00, 0x50, 0xE5, 0x8B, 0x65, 0x88, 0x14, 0xD9,
		0xFB, 0xF3, 0xFF, 0x5B, 0x4A, 0x2E, 0x2E, 0xF5, 0xEA, 0x43,
		0x9C, 0xF6, 0x49, 0x7C, 0x5B, 0x92, 0x6F, 0xCA, 0x88, 0x1E,
		0x02, 0xB3, 0xA8, 0x26, 0x74, 0x10, 0x20, 0xFE, 0x28, 0x29,
		0xC1, 0xD5, 0xEF, 0x1A, 0x65, 0xFF, 0x1A, 0x97, 0x75, 0x22,
		0xDD, 0x46, 0x79, 0x57, 0x59, 0xFA, 0xD5, 0x20, 0x4E, 0x0D,
		0xB4, 0x27, 0x5D, 0x03, 0xF3, 0x60, 0xF3, 0x1A, 0xF2, 0xBF,
		0x50, 0x0B, 0x3C, 0x63, 0x9A, 0xF5, 0xF0, 0x1F, 0x9D, 0xAA,
		0x9F, 0x24, 0xF1, 0xE6, 0x22, 0xF6, 0x14, 0x9B, 0x40, 0x67,
		0xD0, 0x71, 0x09, 0x64, 0x8C, 0x5C, 0xCA, 0xE0, 0x39, 0x02,
		0xA9, 0xB9, 0xBB, 0x67, 0x08, 0x00, 0x3D, 0x8D, 0xF3, 0x9A,
		0x62, 0xA5, 0x92, 0x58, 0x72, 0x41, 0x83, 0x28, 0xE3, 0x2C,
		0x63, 0x9C, 0xD3, 0xAD, 0x27, 0x58, 0x0C, 0x7A, 0xF1, 0x8E,
		0x90, 0x52, 0xC4, 0x2F, 0x02, 0x11, 0x65, 0x75, 0x0E, 0x06,
		0x23, 0x78, 0x38, 0x14, 0x15, 0xF7, 0xC9, 0x16, 0xDE, 0x7D,
		0xEA, 0x5B, 0x32, 0xFE, 0x32, 0xAB, 0xCB, 0x20, 0xDD, 0x16,
		0x07, 0x60, 0x06, 0x2B, 0xAF, 0x77, 0x9D, 0x5C, 0x6E, 0x51,
		0xBE, 0x6E, 0x84, 0x16, 0x2A, 0x96, 0x48, 0x9F, 0xCE, 0x04,
		0x6F, 0x13, 0xDA, 0xF7, 0xB9, 0x40, 0xCA, 0x62, 0x8C, 0x70,
		0x62, 0x24, 0xE4, 0x93, 0x1D, 0x4E, 0x0F, 0x80, 0xB4, 0xE4,
		0xFC, 0x39, 0x29, 0xB9, 0x74, 0xA4, 0x9B, 0xFD, 0x60, 0xBE,
		0x1C, 0x2E, 0x44, 0xC1, 0xAA, 0x24, 0xAE, 0x15, 0xAC, 0x63,
		0x5B, 0x99, 0x95, 0x97, 0x0A, 0x33, 0x13, 0x1F, 0x13, 0x8B,
		0x55, 0x65, 0xCD, 0x13, 0xC5, 0x19, 0x37, 0x88, 0x0C, 0x66,
		0xA3, 0x26, 0xFC, 0x21, 0x08, 0x6C, 0x57, 0xF7, 0xB3, 0x31,
		0xD9, 0x64, 0xE6, 0x81, 0xF7, 0xC2, 0x53, 0x60, 0x42, 0x81,
		0x0A, 0x3F, 0x23, 0x5C, 0xE0, 0x25, 0x9B, 0x9C, 0xB8, 0x5D,
		0x56, 0x88, 0x7A, 0x92, 0xE5, 0x53, 0x5C, 0xE9, 0xF8, 0x4C,
		0xB5, 0x1D, 0x5F, 0x37, 0xE8, 0x36, 0x8B, 0x68, 0xB0, 0x7E,
		0x77, 0x8D, 0x11, 0x87, 0xF8, 0x88, 0x79, 0x58, 0x0A, 0x01,
		0xB9, 0x0D, 0x14, 0x12, 0x68, 0xC1, 0x9B, 0x71, 0xEA, 0xA8,
		0x40, 0xCA, 0xEE, 0x53, 0x08, 0x03, 0xC8, 0x9C, 0xB5, 0xEB,
		0x27, 0x54, 0xA9, 0xCC, 0x6F, 0xBE, 0x49, 0x15, 0x04, 0xCF,
		0xFB, 0x9C, 0xB3, 0xE7, 0x9F, 0xBD, 0x21, 0x27, 0xD4, 0x9D,
		0x22, 0xB2, 0xBD, 0x5A, 0x4A, 0xE6, 0x4C, 0x5A, 0x9C, 0xFD,
		0x58, 0x75, 0xF5, 0xBE, 0xE8, 0x59, 0xCA, 0x35, 0x39, 0x3C,
		0x5F, 0xBD, 0x23, 0xA7, 0x02, 0x8B, 0xB1, 0x3B, 0xAE, 0xA5,
		0x00, 0x20, 0x11, 0xAB, 0xAA, 0x0F, 0xAE, 0x74, 0x02, 0xA9,
		0xFB, 0xA8, 0xD3, 0x28, 0x77, 0xFE, 0x28, 0x64, 0x22, 0x4D,
		0x63, 0x21, 0x48, 0xE5, 0x48, 0xCC, 0xC2, 0x6B, 0xCF, 0x4D,
		0x90, 0x48, 0x03, 0x70, 0x25, 0xD0, 0x73, 0x1D, 0x23, 0x0B,
		0xA7, 0x2E, 0x57, 0x81, 0x22, 0xA7, 0x8A, 0xE5, 0x13, 0xD1,
		0xD9, 0x41, 0x83, 0xB1, 0x15, 0x26, 0xF1, 0x84, 0xC8, 0xCD,
		0x1B, 0x55, 0xA3, 0x5A, 0x2B, 0x1E, 0x50, 0x5D, 0x0E, 0x8B,
		0xCF, 0x01, 0xA5, 0xA6, 0xFF, 0x79, 0x48, 0x39, 0x6A, 0x11,
		0x9A, 0xDD, 0xC0, 0x72, 0xA8, 0xBB, 0x66, 0x54, 0xCE, 0x33,
		0x5B, 0x43, 0x1F, 0xC9, 0xB1, 0xA6, 0xA7, 0xA0, 0xBB, 0x8D,
		0xB9, 0x24, 0xC3, 0xD3, 0x44, 0x81, 0xFC, 0xE9, 0x06, 0xE4,
		0x59, 0x4C, 0xC8, 0xE8, 0x96, 0xCD, 0xFB, 0x4A, 0xFA, 0xE6,
		0xAB, 0x0A, 0x82, 0xD1, 0x5F, 0x14, 0x82, 0xA2, 0x8F, 0x05,
		0x5D, 0xA2, 0x86, 0x32, 0xF9, 0x30, 0xAA, 0xBC, 0xBD, 0xA2,
		0xA2, 0x12, 0x5F, 0x50, 0x04, 0xBA, 0x57, 0xA1, 0x6B, 0x41,
		0xAC, 0x05, 0x42, 0x41, 0xC0, 0xB3, 0x55, 0x6D, 0x90, 0x02,
		0x23, 0x9E, 0xFB, 0x65, 0x35, 0x5E, 0x07, 0x1F, 0x89, 0x02,
		0x0E, 0xA8, 0x50, 0x3C, 0x28, 0x6C, 0xBE, 0xD7, 0x3F, 0xEF,
		0x16, 0xA6, 0x41, 0x8B, 0x6B, 0x93, 0x00, 0x8D, 0x52, 0x83,
		0x33, 0xB9, 0x64, 0x0D, 0xDF, 0xCA, 0x7E, 0x1D, 0x28, 0x43,
		0xA9, 0x6D, 0x48, 0xAC, 0xCC, 0x18, 0x98, 0x42, 0xCF, 0x13,
		0x37, 0x88, 0x26, 0xF4, 0x0F, 0x03, 0x69, 0x7A, 0x83, 0x5E,
		0xA2, 0xB1, 0x8D, 0xD1, 0xAF, 0xDB, 0xC0, 0xE8, 0x7F, 0xF8,
		0x7F, 0xF5, 0x60, 0x65, 0xE4, 0x26, 0xA9, 0xAA, 0x05, 0xEF,
		0xF1, 0xA3, 0x3E, 0xDF, 0x55, 0xE2, 0x94, 0x25, 0xF6, 0x5E,
		0xC2, 0xFE, 0x12, 0x89, 0xA9, 0x8D, 0x94, 0xF1, 0x57, 0xD2,
		0x07, 0x89, 0xE6, 0x0E, 0xF2, 0x8C, 0x6B, 0xBA, 0x1A, 0x06,
		0x61, 0x2B, 0x78, 0x96, 0xE7, 0x4C, 0x3E, 0x21, 0xC3, 0x06,
		0xAD, 0xC8, 0x72, 0x65, 0x06, 0x72, 0x82, 0xCC, 0x20, 0xFE,
		0x2D, 0x0B, 0xD6, 0x7D, 0x85, 0xD8, 0xEF, 0x7D, 0x01, 0x19,
		0xE9, 0x41, 0xED, 0x69, 0xC4, 0xC8, 0x70, 0x95, 0x55, 0xCA,
		0x8B, 0xC1, 0xFC, 0x2D, 0x4F, 0x8D, 0x62, 0xD5, 0x8B, 0xF4,
		0xA6, 0x55, 0xFC, 0x07, 0x70, 0x26, 0xF4, 0xE8, 0x2B, 0x3C,
		0x17, 0xCA, 0xE3, 0xBF, 0x34, 0xFD, 0x55, 0xD5
	};
	static const uint8_t key2[] = {
		0x3B, 0x91, 0x66, 0x28, 0x21, 0x1C, 0x05, 0x9D, 0x31, 0xCA,
		0x66, 0xB1, 0x15, 0x06, 0x48, 0xE4, 0xD8, 0x1E, 0x3E, 0x43,
		0x3E, 0xFF, 0x5A, 0xD7, 0xBD, 0x76, 0xE7, 0x91, 0x5F, 0x2D,
		0xF8, 0xB6, 0x3E, 0x77, 0xDC, 0xD4, 0xD9, 0x7E, 0x73, 0x3A,
		0x4A, 0xAF, 0x25, 0x4F, 0xC0, 0xE9, 0xB3, 0xFE, 0x5C, 0x42,
		0xED, 0x2D, 0xB3, 0x94, 0x4C, 0x17, 0x66, 0x3F, 0xA7, 0x94,
		0x57, 0x50, 0x4D, 0x56, 0x1A, 0x26, 0x9A, 0xC8, 0x05, 0xA5,
		0x27, 0xC6, 0x4B, 0xF1, 0xD7, 0x8B
	};
	static const uint8_t exp2[] = { 0xC1, 0xD1, 0x49, 0x37, 0xB7, 0xB4,
					0xAC, 0x66, 0x8C, 0x16, 0x05, 0x16,
					0xBA, 0x9C, 0x03, 0x8A, 0x3C, 0x65,
					0x7A, 0xAB, 0xCE, 0x59, 0xD1, 0xB6,
					0x74, 0x43, 0x9D };
	uint8_t act1[sizeof(exp1)];
	uint8_t act2[sizeof(exp2)];
	struct lc_kmac_ctx *ctx;
	size_t i;
	int ret;
	LC_KMAC_KDF_DRNG_CTX_ON_STACK(kmac_rng, cshake_256);

	printf("kmac ctx %s (%s implementation) len %lu, re-init ctx len %lu\n",
	       name, cshake_256 == lc_cshake256_c ? "C" : "accelerated",
	       LC_KMAC_CTX_SIZE(cshake_256),
	       LC_KMAC_CTX_SIZE_REINIT(cshake_256));

	if (lc_kmac_alloc(cshake_256, &ctx, LC_KMAC_FLAGS_SUPPORT_REINIT))
		return 1;

	lc_kmac_init(ctx, key1, sizeof(key1), cust1, sizeof(cust1));
	lc_kmac_update(ctx, msg1, sizeof(msg1));
	lc_kmac_final_xof(ctx, act1, sizeof(act1));
	ret = lc_compare(act1, exp1, sizeof(act1), "KMAC256 XOF 1");

	/* no zeroization to test reinit */

	if (ret)
		goto out;

	lc_kmac_reinit(ctx);
	lc_kmac_update(ctx, msg1, sizeof(msg1));
	lc_kmac_final_xof(ctx, act1, sizeof(act1));
	ret = lc_compare(act1, exp1, sizeof(act1), "KMAC256 XOF 1 reinit");
	lc_kmac_zero(ctx);

	if (ret)
		goto out;

	lc_kmac_xof(cshake_256, key2, sizeof(key2), NULL, 0, msg2, sizeof(msg2),
		    act2, sizeof(act2));
	ret = lc_compare(act2, exp2, sizeof(act2), "KMAC256 XOF 2");
	lc_kmac_zero(ctx);

	/*
	 * Verify that subsequent calls to the "RNG" of KMAC returns the same
	 * data as one common KMAC call. This shows that the RNG version
	 * can be inserted into any cipher implementation to generate the
	 * same data as if one KMAC call would be made to generate the entire
	 * requested buffer that is handed down to the wrapping cipher.
	 */
	/* Iterate through block sizes */
	for (i = 1; i <= sizeof(exp1); i++) {
		size_t j = 0;

		/* Reinitialize the KMAC context */
		lc_rng_zero(kmac_rng);
		if (lc_rng_seed(kmac_rng, key1, sizeof(key1), cust1,
				sizeof(cust1))) {
			printf("KMAC256 RNG stack failed\n");
			return 1;
		}

		/* Ensure a KMAC update */
		if (lc_rng_generate(kmac_rng, msg1, sizeof(msg1), NULL, 0)) {
			printf("KMAC256 RNG generate stack failed\n");
			return 1;
		}

		/*
		 * Fill the entire requested buffer size with the given block
		 * size.
		 */
		while (j < sizeof(exp1)) {
			size_t todo = min_size(i, sizeof(exp1) - j);

			if (lc_rng_generate(kmac_rng, NULL, 0, act1 + j,
					    todo)) {
				printf("KMAC256 RNG generate stack failed\n");
				return 1;
			}

			j += todo;
		}
		ret += lc_compare(act1, exp1, sizeof(exp1),
				  "KMAC256 RNG generate");
	}

	lc_rng_zero(kmac_rng);

out:
	lc_kmac_zero_free(ctx);
	return ret;
}

static int kmac_xof_tester(void)
{
	int ret = 0;

	LC_EXEC_ONE_TEST(lc_cshake256);
	LC_EXEC_ONE_TEST(lc_cshake256_c);
	LC_EXEC_ONE_TEST(lc_cshake256_arm_asm);
	LC_EXEC_ONE_TEST(lc_cshake256_arm_ce);
	LC_EXEC_ONE_TEST(lc_cshake256_arm_neon);
	LC_EXEC_ONE_TEST(lc_cshake256_avx2);
	LC_EXEC_ONE_TEST(lc_cshake256_avx512);

	return ret;
}

LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	(void)argc;
	(void)argv;
	return kmac_xof_tester();
}
