/*
Copyright 2022 The OpenVEX Authors
SPDX-License-Identifier: Apache-2.0
*/

package cmd

import (
	"context"
	"errors"
	"fmt"
	"io"
	"os"

	"github.com/google/go-containerregistry/pkg/name"
	"github.com/openvex/vexctl/pkg/ctl"
	"github.com/spf13/cobra"
)

type attestOptions struct {
	outFileOption
	attach bool
	sign   bool
	refs   []string
}

func (o *attestOptions) AddFlags(cmd *cobra.Command) {
	o.outFileOption.AddFlags(cmd)
	cmd.PersistentFlags().BoolVarP(
		&o.attach,
		"attach",
		"a",
		false,
		"attach the generated attestation to an image (implies --sign)",
	)

	cmd.PersistentFlags().BoolVarP(
		&o.sign,
		"sign",
		"s",
		false,
		"sign the attestation with sigstore",
	)

	cmd.PersistentFlags().StringArrayVarP(
		&o.refs,
		"refs",
		"r",
		[]string{},
		"list of image references to attach the attestation to",
	)
}

// Validate checks if the options are sane
func (o *attestOptions) Validate() error {
	var sErr error
	for _, ref := range o.refs {
		if _, err := name.ParseReference(ref); err != nil {
			sErr = fmt.Errorf("parsing reference: %w", err)
			break
		}
	}

	if o.attach {
		o.sign = true
	}

	return errors.Join(
		sErr, o.outFileOption.Validate(),
	)
}

func addAttest(parentCmd *cobra.Command) {
	opts := attestOptions{}
	attestCmd := &cobra.Command{
		Short: fmt.Sprintf("%s attest: generate a VEX  attestation", appname),
		Long: fmt.Sprintf(`%s attest: generate an VEX attestation

The attach subcommand lets users wrap OpenVEX documents in in-toto attestations.
Attestations generated by %s can be signed with sigstore and attached to
container images stored in an OCI registry.

In its simplest form, %s will create an attestation from an OpenVEX file and
write it to stdout:

  %s attest data.vex.json

Without any more arguments, container images and other products that have hashes
defined in their VEX statements, will be read by %s and transferred to
the attestation's subjects when required.

If the products are purls of type oci:, they will be converted to image
references as is customary in the sigstore tooling. For example:

If a product identified by the following package URL is found in a statement:

  pkg:oci/kube-apiserver?repository_url=registry.k8s.io&tag=v1.26.0

It will be transferred to the attestation subjects as:

  registry.k8s.io/kube-apiserver:v1.26.0

Any oci purls and image references not specifying a digest will trigger a
network lookup to read the image digest from the registry.

Please note that purls of types other than oci: and other strings which are not
valid image references will not be included in the resulting attestation unless 
the product has hashes associated with it. 

Signing Attestations
--------------------

Passing the --sign flag will trigger the cosign signing flow, either asking for
credentials from the user or trying to get them from the environment:

  %s attest --sign data.vex.json

When signing an attestation, the standard sigstore signing flow will be triggered
if credentials are not found in the environment. Refer to the sigstore
documentation for details.

Attaching Attestations
----------------------

The --attach flag will attach the resulting attestation to the OCI registry of
all subjects that parse as image references. If this behavior fails, try defining
--refs to specify which images to attach the attestation to.

%s will use the credentials from the user's environment to authenticate to the
registry, this means that if you can write to the registry, attaching should work.

Note: --attach always implies --sign as sigstore does not support attaching
unsigned attestations.

Specifying Images to Attest
---------------------------

If any further positional arguments are defined, they will be interpreted as
products/image references. %s will generate and attach (if applicable) the
attestation only for those images, not transferring other products in the
VEX document as in-toto subjects. 

For example, the following invocation will only attest and attach vex.json
to user/test, even if the OpenVEX document has product entries for other images:

%s attest --attach vex.json user/test


`, appname, appname, appname, appname, appname, appname, appname, appname, appname),
		Use:               "attest",
		SilenceUsage:      false,
		SilenceErrors:     false,
		PersistentPreRunE: initLogging,
		RunE: func(cmd *cobra.Command, args []string) error {
			if len(args) == 0 {
				return errors.New("not enough arguments")
			}

			if err := opts.Validate(); err != nil {
				return fmt.Errorf("validating options: %w", err)
			}

			cmd.SilenceUsage = true
			ctx := context.Background()

			vexctl := ctl.New()
			vexctl.Options.Sign = opts.sign

			attestation, err := vexctl.Attest(args[0], args[1:])
			if err != nil {
				return fmt.Errorf("generating attestation: %w", err)
			}

			if opts.attach {
				if err := vexctl.Attach(ctx, attestation); err != nil {
					return fmt.Errorf("attaching attestation: %w", err)
				}
			}

			var out io.Writer = os.Stdout
			if opts.outFileOption.outFilePath != "" {
				out, err = os.Create(opts.outFilePath)
				if err != nil {
					return fmt.Errorf("opening attestation file: %w", err)
				}
			}
			if err := attestation.ToJSON(out); err != nil {
				return fmt.Errorf("marshaling attestation to json")
			}

			return nil
		},
	}
	opts.AddFlags(attestCmd)
	parentCmd.AddCommand(attestCmd)
}
