package system

import (
	"fmt"
	"os"
	"testing"
)

var limitsSampleText = `# yadi yadi yada
# /etc/security/limits.conf
*               hard    nproc           8000
*               soft    nproc           4000
root            hard    nproc           16000
root            soft    nproc           8000

some random soft nofile 12345

*               hard    nofile          32000
*               soft    nofile          16000
root            hard    nofile          64000
root            soft    nofile          32000
@dba            hard    memlock         unlimited
`
var limitsMatchText = `# yadi yadi yada
# /etc/security/limits.conf
* hard nproc 1234
* soft nproc 4000
root hard nproc 16000
root soft nproc 8000

some random soft nofile 12345

* hard nofile 32000
* soft nofile 16000
root hard nofile 64000
root soft nofile 32000
@dba hard memlock unlimited
@sapsys soft nofile 65535
@dba soft memlock unlimited
`

var dropInMatchText = `### /etc/security/limits.d/saptune-@sapsys-nofile-soft.conf
### file autogenerated by saptune!
### requested by Note 471101
###
### Please do NOT change or delete!
###

@sapsys soft nofile 65535
`

func TestSecLimits(t *testing.T) {
	// Parse the sample text
	limits := ParseSecLimits(limitsSampleText)

	// Read keys
	if value, exists := limits.Get("*", "soft", "nproc"); !exists || value != "4000" {
		t.Fatal(value, exists)
	}
	if value, exists := limits.Get("@dba", "hard", "memlock"); !exists || value != "unlimited" {
		t.Fatal(value, exists)
	}
	value := limits.GetOr0("@dba", "hard", "memlock")
	if value != SecurityLimitUnlimitedValue {
		t.Error(value)
	}
	if value.String() != "unlimited" {
		t.Errorf("value should be 'unlimited', but is '%+v'\n", value)
	}
	value = limits.GetOr0("root", "soft", "nproc")
	if value != 8000 {
		t.Error(value)
	}
	if value.String() != "8000" {
		t.Errorf("value should be '8000', but is '%+v'\n", value)
	}
	if value, exists := limits.Get("does_not_exist", "soft", "nproc"); exists {
		t.Fatal(value, exists)
	}
	if value := limits.GetOr0("does_not_exist", "soft", "nproc"); value != 0 {
		t.Fatal(value)
	}

	// Write keys
	limits.Set("*", "hard", "nproc", "1234")
	limits.Set("@sapsys", "soft", "nofile", "65535")
	limits.Set("@dba", "soft", "memlock", "unlimited")

	// The converted back text should carry new value for nproc and new entry
	if txt := limits.ToText(); txt != limitsMatchText {
		fmt.Println("==============")
		fmt.Println(txt)
		fmt.Println("==============")
		t.Fatal("failed to convert back into text")
	}
	lim := []string{"@sapsys", "soft", "nofile", "0"}
	dropInFile := "/etc/security/limits.d/saptune-@sapsys-nofile-soft.conf"
	noteID := "471101"
	if txt := limits.ToDropIn(lim, noteID, dropInFile); txt != dropInMatchText {
		fmt.Println("==============")
		fmt.Println(txt)
		fmt.Println("==============")
		fmt.Println(dropInMatchText)
		fmt.Println("==============")
		t.Fatal("failed to convert back into text")
	}

	// Apply keys to drop in file
	err := limits.ApplyDropIn(lim, noteID)
	if err != nil {
		t.Fatal(err)
	}

	newDropInLimits, err := ParseSecLimitsFile(dropInFile)
	if err != nil {
		t.Fatal(err)
	}
	if value, exists := newDropInLimits.Get("@sapsys", "soft", "nofile"); !exists || value != "65535" {
		t.Fatal(value, exists)
	}

	// Apply keys to system file /etc/security/limits.conf
	err = limits.Apply()
	if err != nil {
		t.Fatal(err)
	}
	newLimits, err := ParseSecLimitsFile("file_does_not_exist")
	if err != nil {
		t.Fatal(err)
	}
	if value, exists := newLimits.Get("*", "hard", "nproc"); !exists || value != "1234" {
		t.Fatal(value, exists)
	}
	if value, exists := newLimits.Get("@dba", "soft", "memlock"); !exists || value != "unlimited" {
		t.Fatal(value, exists)
	}

	// no limits file for input available
	_ = CopyFile("/etc/security/limits.conf", "/etc/security/limits.conf_OrG")
	os.Remove("/etc/security/limits.conf")
	noLimits, err := ParseSecLimitsFile("file_does_not_exist")
	if err == nil && noLimits != nil {
		t.Error("should return an error and not 'nil'")
	}
	_ = CopyFile("/etc/security/limits.conf_OrG", "/etc/security/limits.conf")
	os.Remove("/etc/security/limits.conf_OrG")
	os.Remove(dropInFile)

	tstlim := []string{"tst1", "tst2", "tst3"}
	os.Rename("/etc/security/limits.d", "/etc/security/limits.d_OrG")
	err = limits.ApplyDropIn(tstlim, "tstnote")
	if err != nil {
		t.Error(err)
	}
	os.RemoveAll("/etc/security/limits.d")
	os.Rename("/etc/security/limits.d_OrG", "/etc/security/limits.d")
}
