/// <reference types="node" />
import { Transform } from '../transform';
import { Node } from './base';
/**
 * Character/sequence to match.
 *
 * May have following types:
 *
 * * `number` - for single character
 * * `string` - for printable character sequence
 * * `Buffer` - for raw byte sequence
 */
export declare type MatchSingleValue = string | number | Buffer;
/**
 * Convenience type for passing several characters/sequences to match methods.
 */
export declare type MatchValue = MatchSingleValue | ReadonlyArray<MatchSingleValue>;
/**
 * A map from characters/sequences to `.select()`'s values. Used for specifying
 * the value to be passed to `.select()'`s targets.
 */
export interface IMatchSelect {
    readonly [key: string]: number;
}
/**
 * This node matches characters/sequences and forwards the execution according
 * to matched character with optional attached value (See `.select()`).
 */
export declare class Match extends Node {
    private transformFn;
    /**
     * Set character transformation function.
     *
     * @param transform  Transformation to apply. Can be created with
     *                   `builder.transform.*()` methods.
     */
    transform(transformFn: Transform): this;
    /**
     * Match sequence/character and forward execution to `next` on success,
     * consuming matched bytes of the input.
     *
     * No value is attached on such execution forwarding, and the target node
     * **must not** be an `Invoke` node with a callback expecting the value.
     *
     * @param value  Sequence/character to be matched
     * @param next   Target node to be executed on success.
     */
    match(value: MatchValue, next: Node): this;
    /**
     * Match character and forward execution to `next` on success
     * without consuming one byte of the input.
     *
     * No value is attached on such execution forwarding, and the target node
     * **must not** be an `Invoke` with a callback expecting the value.
     *
     * @param value  Character to be matched
     * @param next   Target node to be executed on success.
     */
    peek(value: MatchValue, next: Node): this;
    /**
     * Match character/sequence and forward execution to `next` on success
     * consumed matched bytes of the input.
     *
     * Value is attached on such execution forwarding, and the target node
     * **must** be an `Invoke` with a callback expecting the value.
     *
     * Possible signatures:
     *
     * * `.select(key, value [, next ])`
     * * `.select({ key: value } [, next])`
     *
     * @param keyOrMap     Either a sequence to match, or a map from sequences to
     *                     values
     * @param valueOrNext  Either an integer value to be forwarded to the target
     *                     node, or an otherwise node
     * @param next         Convenience param. Same as calling `.otherwise(...)`
     */
    select(keyOrMap: MatchSingleValue | IMatchSelect, valueOrNext?: number | Node, next?: Node): this;
    /**
     * Get tranformation function
     */
    getTransform(): Transform | undefined;
}
