/*
 * Copyright (C) 2024 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef SPHINCS_TESTER_VECTORS_SHAKE_128S_H
#define SPHINCS_TESTER_VECTORS_SHAKE_128S_H

#ifdef __cplusplus
extern "C" {
#endif

struct lc_sphincs_test {
	uint8_t seed[LC_SPX_SEEDBYTES];
	uint8_t msg[33];
	uint8_t pk[sizeof(struct lc_sphincs_pk)];
	uint8_t sk[sizeof(struct lc_sphincs_sk)];
	uint8_t sig[sizeof(struct lc_sphincs_sig)];
};

static const struct lc_sphincs_test tests[] =
{
	{
		.seed = {
			0x7c, 0x99, 0x35, 0xa0, 0xb0, 0x76, 0x94, 0xaa,
			0x0c, 0x6d, 0x10, 0xe4, 0xdb, 0x6b, 0x1a, 0xdd,
			0x2f, 0xd8, 0x1a, 0x25, 0xcc, 0xb1, 0x48, 0x03,
			0x2d, 0xcd, 0x73, 0x99, 0x36, 0x73, 0x7f, 0x2d,
			0xb5, 0x05, 0xd7, 0xcf, 0xad, 0x1b, 0x49, 0x74,
			0x99, 0x32, 0x3c, 0x86, 0x86, 0x32, 0x5e, 0x47,
		},
		.msg = {
			0xD8, 0x1C, 0x4D, 0x8D, 0x73, 0x4F, 0xCB, 0xFB,
			0xEA, 0xDE, 0x3D, 0x3F, 0x8A, 0x03, 0x9F, 0xAA,
			0x2A, 0x2C, 0x99, 0x57, 0xE8, 0x35, 0xAD, 0x55,
			0xB2, 0x2E, 0x75, 0xBF, 0x57, 0xBB, 0x55, 0x6A,
			0xC8
		},
		.pk = {
			0xB5, 0x05, 0xD7, 0xCF, 0xAD, 0x1B, 0x49, 0x74,
			0x99, 0x32, 0x3C, 0x86, 0x86, 0x32, 0x5E, 0x47,
			0xAC, 0x52, 0x49, 0x02, 0xFC, 0x81, 0xF5, 0x03,
			0x2B, 0xC2, 0x7B, 0x17, 0xD9, 0x26, 0x1E, 0xBD
		},
		.sk = {
			0x7C, 0x99, 0x35, 0xA0, 0xB0, 0x76, 0x94, 0xAA,
			0x0C, 0x6D, 0x10, 0xE4, 0xDB, 0x6B, 0x1A, 0xDD,
			0x2F, 0xD8, 0x1A, 0x25, 0xCC, 0xB1, 0x48, 0x03,
			0x2D, 0xCD, 0x73, 0x99, 0x36, 0x73, 0x7F, 0x2D,
			0xB5, 0x05, 0xD7, 0xCF, 0xAD, 0x1B, 0x49, 0x74,
			0x99, 0x32, 0x3C, 0x86, 0x86, 0x32, 0x5E, 0x47,
			0xAC, 0x52, 0x49, 0x02, 0xFC, 0x81, 0xF5, 0x03,
			0x2B, 0xC2, 0x7B, 0x17, 0xD9, 0x26, 0x1E, 0xBD
		},
		.sig = {
			0xe1, 0x72, 0x9e, 0x58, 0x11, 0x4b, 0xd4, 0x9c,
			0x65, 0xee, 0xc1, 0xf6, 0xd8, 0xdb, 0x77, 0x0e,
			0x3b, 0xf9, 0xbf, 0xbb, 0x33, 0xf3, 0xb0, 0xfb,
			0x75, 0x38, 0x67, 0x14, 0xef, 0x03, 0x84, 0xb4,
			0xad, 0x2e, 0x59, 0xda, 0x60, 0xad, 0x51, 0xc6,
			0x5d, 0x8e, 0x96, 0x69, 0xc2, 0x49, 0xbc, 0x21,
			0x12, 0xf0, 0x8e, 0x26, 0xe8, 0x89, 0x75, 0xd5,
			0x6e, 0xbb, 0x30, 0x92, 0x62, 0xc9, 0x21, 0xef,
			0x2a, 0x87, 0x04, 0x51, 0x21, 0x34, 0x13, 0x6b,
			0xf8, 0x9f, 0x2d, 0xa5, 0xbd, 0xe5, 0x80, 0xe1,
			0x44, 0x91, 0xf4, 0x09, 0x5f, 0x3c, 0xe1, 0x4d,
			0x5c, 0xad, 0x05, 0x24, 0xa8, 0x41, 0x99, 0xd0,
			0x96, 0x0e, 0x6d, 0x3b, 0xd5, 0x06, 0x55, 0x90,
			0xda, 0xc2, 0x1f, 0x44, 0xf6, 0x76, 0x83, 0xfd,
			0x02, 0x32, 0x23, 0xf9, 0x1b, 0x7b, 0xa4, 0xf9,
			0x6e, 0xf3, 0x6d, 0x34, 0xd3, 0xf0, 0xf6, 0x99,
			0xda, 0xea, 0x60, 0xd8, 0xd3, 0x9e, 0xe7, 0x33,
			0x73, 0xda, 0x5c, 0xd8, 0x76, 0x84, 0x82, 0x8a,
			0x35, 0xb9, 0x7a, 0xd7, 0x13, 0xd4, 0xf1, 0x48,
			0x90, 0xd9, 0xbf, 0x68, 0xb9, 0x4b, 0xb1, 0x48,
			0x5c, 0xe8, 0xc2, 0x6d, 0xfb, 0x2e, 0x48, 0x0b,
			0x13, 0x36, 0xed, 0xfb, 0x4d, 0x2d, 0xd3, 0xb8,
			0xb1, 0x12, 0xff, 0xb4, 0x74, 0xf2, 0xc7, 0x44,
			0xd4, 0xa6, 0x4a, 0x03, 0xf9, 0x43, 0x87, 0x69,
			0x30, 0x27, 0xbe, 0x67, 0xa0, 0x90, 0x0d, 0xa2,
			0x65, 0x79, 0x93, 0x7d, 0x9a, 0x55, 0x62, 0xcb,
			0x76, 0x16, 0x09, 0xf5, 0xc2, 0xbf, 0x4a, 0xb9,
			0x49, 0x5b, 0xd6, 0x25, 0x93, 0xe9, 0xf0, 0xa9,
			0xc8, 0xf1, 0x73, 0x71, 0x01, 0xe0, 0xe5, 0x4f,
			0xd5, 0x3a, 0x98, 0x32, 0xb1, 0x55, 0x2a, 0xf8,
			0x2d, 0x83, 0xeb, 0xb6, 0x0d, 0x60, 0x51, 0xc9,
			0xd1, 0xe8, 0xec, 0x1a, 0x3f, 0xec, 0xf6, 0xc3,
			0xcc, 0x5d, 0x19, 0x63, 0xce, 0x64, 0xe9, 0x3c,
			0x4f, 0x7c, 0xf4, 0xed, 0x4d, 0xac, 0x5c, 0x57,
			0xd1, 0x24, 0xb6, 0xd2, 0xf9, 0x4d, 0x71, 0x03,
			0x35, 0x8c, 0xc2, 0xcb, 0xc9, 0xca, 0x83, 0x46,
			0xff, 0x95, 0x58, 0x9c, 0x92, 0x50, 0x36, 0xf6,
			0x71, 0x43, 0x0e, 0x0e, 0xe1, 0x04, 0x3f, 0xcd,
			0x71, 0x73, 0xd4, 0x74, 0xd1, 0xc7, 0xcd, 0x77,
			0x75, 0x1d, 0xfb, 0xfb, 0xc1, 0x15, 0x2d, 0x10,
			0x51, 0x26, 0xc1, 0xcb, 0xc3, 0x74, 0xa0, 0x0f,
			0x31, 0x55, 0x40, 0x6b, 0x27, 0x95, 0x45, 0xf0,
			0x09, 0xa3, 0x3e, 0xa2, 0xbb, 0x00, 0xcc, 0xb3,
			0x98, 0xd6, 0x2d, 0x3f, 0xd2, 0x32, 0x08, 0xd2,
			0x86, 0x3b, 0x32, 0x33, 0x79, 0xbd, 0x28, 0x5a,
			0xf8, 0x12, 0x10, 0xbf, 0xb6, 0xe9, 0x47, 0x2e,
			0xcf, 0x6b, 0x29, 0xcf, 0x5b, 0xe1, 0xba, 0xc4,
			0xd2, 0xd2, 0xbc, 0x10, 0xb2, 0x5b, 0x95, 0x75,
			0x34, 0x82, 0x07, 0x48, 0x47, 0x64, 0x2e, 0x45,
			0x08, 0xf3, 0x84, 0x29, 0x18, 0xef, 0x08, 0x76,
			0xcd, 0xdc, 0x2c, 0x7f, 0xaf, 0xa5, 0x68, 0x65,
			0x22, 0xd4, 0x56, 0x5c, 0xfb, 0x9e, 0x76, 0x1b,
			0xa0, 0x45, 0x47, 0x38, 0xf8, 0x19, 0x81, 0x6c,
			0x14, 0x1d, 0xe9, 0x12, 0x93, 0x1b, 0x18, 0x03,
			0x73, 0xfa, 0x30, 0x6a, 0x19, 0x94, 0xab, 0xdc,
			0x0e, 0xf4, 0xd8, 0xa2, 0x88, 0xca, 0xd7, 0xcf,
			0x92, 0x91, 0x12, 0xa1, 0xc7, 0x28, 0x8d, 0xe8,
			0x5f, 0x3c, 0x98, 0xe3, 0x97, 0x2e, 0x27, 0x16,
			0x82, 0xa4, 0xf5, 0x2e, 0xae, 0xed, 0xd1, 0xc0,
			0x13, 0x17, 0xce, 0x9b, 0xf7, 0x27, 0x77, 0x31,
			0x30, 0xd2, 0xa8, 0x6e, 0x8f, 0xd1, 0xf4, 0x05,
			0x31, 0xf7, 0xd9, 0x04, 0x6d, 0x3a, 0x52, 0x8d,
			0x4c, 0x0f, 0x60, 0x8c, 0x51, 0x6f, 0xf8, 0x80,
			0x96, 0x7e, 0xfb, 0x4c, 0x87, 0xa1, 0x98, 0x16,
			0x39, 0x29, 0x6e, 0x38, 0xa4, 0xaf, 0xbe, 0x72,
			0xdc, 0x13, 0x54, 0x4c, 0xf3, 0x62, 0xef, 0x7b,
			0xc1, 0x50, 0x3c, 0x90, 0xe5, 0xc4, 0x7a, 0x3e,
			0xac, 0x06, 0xf8, 0x09, 0xc2, 0xfe, 0xd2, 0xb7,
			0xc6, 0x17, 0xe3, 0xb1, 0xda, 0x7c, 0x26, 0x54,
			0x38, 0x31, 0xfa, 0xd7, 0x9c, 0x4f, 0xf3, 0xf6,
			0xc8, 0xdd, 0x2e, 0x37, 0xe0, 0x68, 0x3d, 0xd9,
			0x85, 0x32, 0xb1, 0x44, 0xf5, 0x8c, 0x9d, 0xb3,
			0xca, 0x32, 0x70, 0xd7, 0x78, 0x0f, 0x60, 0x7a,
			0xb1, 0x3c, 0xfa, 0x61, 0x60, 0x77, 0xd9, 0x2a,
			0x4e, 0x24, 0x55, 0xb3, 0xe1, 0x49, 0xc0, 0xa9,
			0xc5, 0x59, 0x7b, 0x4b, 0x20, 0x74, 0xad, 0x3e,
			0x2b, 0x3f, 0x8f, 0x72, 0x7a, 0x2d, 0xde, 0x16,
			0x71, 0x1c, 0xa5, 0x32, 0xe8, 0xaa, 0x00, 0x37,
			0x72, 0x85, 0xc5, 0xaf, 0x9e, 0xa1, 0xcf, 0x48,
			0xf6, 0x50, 0x9e, 0xb2, 0x83, 0x50, 0x3e, 0xec,
			0x80, 0xe5, 0xd5, 0xd3, 0xb7, 0x72, 0x53, 0x48,
			0xad, 0x22, 0xeb, 0xf2, 0x41, 0x1e, 0x34, 0x18,
			0xcd, 0x53, 0x6d, 0xc2, 0x20, 0xc1, 0xb6, 0x40,
			0x66, 0xb1, 0xd7, 0x2f, 0x06, 0x82, 0xbd, 0xe6,
			0x1c, 0x31, 0xe0, 0x8d, 0x0a, 0x52, 0x81, 0xf4,
			0x35, 0xbe, 0x89, 0x6c, 0xa4, 0xb4, 0x2b, 0x12,
			0x85, 0x22, 0x8b, 0xb1, 0x7c, 0x77, 0xdb, 0x8b,
			0xab, 0x88, 0xda, 0x38, 0x04, 0xa9, 0x8f, 0x7c,
			0xd5, 0xa8, 0xc8, 0x06, 0xe4, 0xfd, 0xc1, 0x83,
			0x29, 0xbf, 0xa6, 0x72, 0x2a, 0x94, 0x9c, 0x29,
			0x97, 0x54, 0x76, 0x9f, 0x92, 0x2c, 0x57, 0x20,
			0xf3, 0x7d, 0x45, 0x2f, 0x1b, 0x5f, 0xc4, 0x20,
			0x22, 0x84, 0x7a, 0x92, 0xeb, 0xe6, 0x3a, 0xda,
			0xfc, 0x71, 0x81, 0xd1, 0x84, 0xc0, 0x72, 0xd5,
			0xe6, 0x34, 0x02, 0x29, 0x2e, 0x47, 0x4b, 0xe5,
			0xf2, 0xc6, 0x2f, 0x03, 0xc4, 0xba, 0xfc, 0x32,
			0x6a, 0xc0, 0x8b, 0x4c, 0xd2, 0xc0, 0xfb, 0x2b,
			0x39, 0x3d, 0xc8, 0x9d, 0xa6, 0xc3, 0xfe, 0x4f,
			0x82, 0x9b, 0xeb, 0x1c, 0x94, 0x59, 0x58, 0xc6,
			0xb1, 0xd7, 0xd9, 0x50, 0x73, 0xc4, 0x8a, 0x9e,
			0x48, 0x1f, 0x8d, 0xf3, 0x65, 0x87, 0xa2, 0xd4,
			0x45, 0xd0, 0xcd, 0x30, 0xfe, 0x98, 0xf1, 0xdf,
			0xb9, 0x90, 0x6c, 0xf1, 0xaf, 0xb4, 0x93, 0xf6,
			0xcd, 0xcf, 0x2c, 0xd7, 0x0e, 0x27, 0x64, 0x11,
			0x36, 0xb7, 0xc1, 0xc2, 0xc7, 0x34, 0x48, 0x64,
			0xb7, 0x1d, 0x0b, 0x7d, 0x83, 0xf0, 0x48, 0xd2,
			0xb0, 0x64, 0xc2, 0xfa, 0xb1, 0x1f, 0xfa, 0xf7,
			0xc8, 0x8f, 0xcd, 0xa4, 0xf0, 0x24, 0x96, 0x55,
			0x28, 0x23, 0x56, 0xbd, 0x95, 0xf2, 0xbf, 0x45,
			0x6b, 0xb8, 0xdb, 0x99, 0x6e, 0x0b, 0xd5, 0x51,
			0x96, 0x88, 0x3a, 0xc7, 0xf0, 0x4e, 0x80, 0x54,
			0x6d, 0x1e, 0xcb, 0x34, 0xf6, 0x10, 0x75, 0x0c,
			0x25, 0xb9, 0x7b, 0xc5, 0xac, 0xc0, 0x9a, 0x25,
			0xde, 0xc8, 0x46, 0x6d, 0x30, 0x07, 0x37, 0x61,
			0xb2, 0x5f, 0x16, 0x98, 0xab, 0x86, 0xab, 0x6a,
			0x8f, 0xc6, 0xda, 0x5a, 0x39, 0xff, 0x7c, 0x43,
			0x51, 0x4c, 0x50, 0x94, 0x19, 0x19, 0x9b, 0x57,
			0x41, 0xa4, 0x69, 0x44, 0xd2, 0xac, 0xfe, 0x47,
			0xd3, 0x49, 0x33, 0x40, 0xaf, 0xf4, 0x96, 0x2d,
			0x21, 0xaf, 0xce, 0x3f, 0x1d, 0x08, 0x46, 0x8f,
			0xfe, 0x78, 0xb5, 0x7a, 0xc7, 0xe6, 0x3b, 0xcb,
			0x4f, 0x9d, 0x5a, 0xfa, 0x0a, 0xfb, 0x29, 0xb9,
			0xd8, 0xe0, 0x83, 0xd0, 0xc5, 0x99, 0x2f, 0xa1,
			0x5b, 0x2d, 0xa7, 0x58, 0xaa, 0xd6, 0xcf, 0x4a,
			0x6e, 0x35, 0x3f, 0xbd, 0xde, 0x2b, 0x43, 0xfb,
			0x89, 0x27, 0x47, 0xfe, 0x97, 0xa4, 0xdb, 0x7c,
			0xb8, 0x83, 0x6a, 0xfd, 0x6c, 0xf6, 0x74, 0xa4,
			0x93, 0xdc, 0xb0, 0x40, 0xf7, 0x0a, 0x91, 0xed,
			0x6e, 0xc0, 0xe3, 0x47, 0xda, 0x67, 0xba, 0xbf,
			0x6b, 0xc3, 0xa6, 0x4c, 0xd3, 0xfc, 0x7e, 0x00,
			0xed, 0xb1, 0xf4, 0xa3, 0x6b, 0xf7, 0x17, 0xb9,
			0x5d, 0x66, 0xba, 0x7b, 0x16, 0x3b, 0xf2, 0x27,
			0xe8, 0xda, 0xd5, 0xe7, 0x7f, 0x09, 0xea, 0x57,
			0x18, 0x03, 0x66, 0xcd, 0xfe, 0x1f, 0xd5, 0x51,
			0x09, 0xf3, 0x61, 0xee, 0x86, 0xce, 0x54, 0xea,
			0xbe, 0xa2, 0x21, 0xd2, 0x9c, 0x4c, 0xd8, 0x2a,
			0x3d, 0x0f, 0xfa, 0x14, 0x7e, 0x98, 0x71, 0x9e,
			0x93, 0xa4, 0xe7, 0xf7, 0xdc, 0x45, 0x63, 0x69,
			0x69, 0x79, 0xb9, 0xae, 0x8c, 0x72, 0x20, 0xf0,
			0xd8, 0x44, 0xb7, 0xd4, 0x80, 0x67, 0x7e, 0x56,
			0x8a, 0xb1, 0xd7, 0x87, 0x31, 0x83, 0x57, 0xae,
			0xd3, 0x46, 0xd7, 0xe7, 0x42, 0xe7, 0x98, 0xc4,
			0x46, 0xb6, 0x7d, 0x40, 0xab, 0xfb, 0x10, 0x47,
			0x66, 0xa0, 0x74, 0x3a, 0x30, 0x65, 0xf8, 0xcd,
			0x8b, 0x4d, 0xf1, 0xa4, 0x96, 0xe6, 0x31, 0x75,
			0x84, 0x4e, 0x73, 0xb9, 0x0f, 0x96, 0x0b, 0xac,
			0xce, 0xa3, 0xde, 0x33, 0xdb, 0x8b, 0x3a, 0x63,
			0x34, 0x7d, 0xdb, 0x7b, 0x25, 0x5d, 0xf2, 0xa5,
			0xca, 0xf8, 0xf0, 0x38, 0xab, 0x5b, 0xb2, 0x14,
			0x18, 0x37, 0x0a, 0x85, 0x3c, 0x27, 0xbc, 0x44,
			0xa2, 0x77, 0x62, 0x31, 0x12, 0xf8, 0x0d, 0xfb,
			0x15, 0xa8, 0xdf, 0x34, 0x3b, 0x35, 0x4d, 0x56,
			0x9b, 0x6d, 0x76, 0x52, 0xce, 0x99, 0xcd, 0x7c,
			0x7b, 0xa7, 0xe3, 0xe9, 0xe5, 0x1a, 0x6b, 0xf9,
			0x4b, 0x23, 0x72, 0x5c, 0x88, 0x7f, 0x8a, 0x3b,
			0xd4, 0x49, 0x2f, 0x2b, 0xb7, 0x98, 0x58, 0xde,
			0x0d, 0x57, 0x04, 0xe8, 0x4f, 0x37, 0x15, 0x01,
			0xfd, 0x69, 0x49, 0x32, 0xf8, 0x78, 0xb2, 0x99,
			0x88, 0xd9, 0x66, 0xbb, 0xf7, 0x8d, 0xcf, 0xfc,
			0x15, 0xfc, 0xa0, 0xe6, 0xcd, 0xa4, 0x9f, 0xc0,
			0x2e, 0xf2, 0x3c, 0x24, 0x4c, 0xb0, 0xbc, 0x4b,
			0xa1, 0xcd, 0x1a, 0x29, 0x4d, 0xc7, 0x04, 0x63,
			0x4e, 0xb0, 0x0e, 0xb8, 0xe0, 0x3e, 0x8f, 0x1a,
			0xd4, 0xb8, 0x8d, 0x2e, 0x74, 0xae, 0x02, 0x4f,
			0xe3, 0x0d, 0x49, 0x15, 0x82, 0x37, 0x43, 0x26,
			0xd8, 0x0c, 0xeb, 0xc2, 0x87, 0x8d, 0x35, 0xbf,
			0xea, 0xc6, 0x9e, 0x61, 0x4f, 0x52, 0x1c, 0x51,
			0xb3, 0x88, 0xd5, 0xa2, 0x67, 0xf1, 0x13, 0x4a,
			0x86, 0x70, 0xeb, 0xa6, 0x53, 0xb2, 0x1e, 0x73,
			0xbc, 0xd8, 0x7f, 0xff, 0x2e, 0x29, 0x65, 0xcb,
			0xe3, 0x2b, 0x45, 0x32, 0x39, 0xad, 0xf5, 0x94,
			0x54, 0x2a, 0x83, 0xde, 0xc7, 0x7b, 0x3b, 0xac,
			0xe2, 0x34, 0x59, 0x8e, 0xdc, 0xfd, 0x57, 0xc4,
			0x9c, 0xde, 0xd7, 0x6f, 0xe6, 0xcc, 0x22, 0xd8,
			0x07, 0xeb, 0xba, 0x8d, 0xee, 0x0c, 0x5e, 0x43,
			0x61, 0x7c, 0x88, 0x16, 0x23, 0x61, 0x3b, 0x75,
			0xe6, 0x68, 0x49, 0xc5, 0x18, 0x1e, 0x49, 0xb6,
			0x70, 0x85, 0x13, 0xc7, 0x1a, 0xa1, 0x06, 0x6e,
			0x41, 0x4b, 0xe5, 0xe8, 0x7a, 0x8c, 0x6f, 0x8e,
			0xf0, 0x5e, 0x18, 0x97, 0x48, 0x56, 0x96, 0xcd,
			0x00, 0xcc, 0xf4, 0x07, 0x63, 0x3f, 0x9d, 0xf6,
			0xad, 0xa6, 0x12, 0x65, 0xf6, 0x82, 0xd3, 0x0f,
			0x9d, 0x51, 0x9d, 0xf0, 0x14, 0x8c, 0x6d, 0x71,
			0x16, 0xce, 0xbc, 0xd9, 0xbf, 0x81, 0xc5, 0x92,
			0x38, 0xf6, 0xf1, 0x77, 0xf2, 0x78, 0x59, 0x98,
			0x6e, 0x74, 0xda, 0xfa, 0xa4, 0x82, 0x44, 0x1d,
			0x25, 0xf7, 0xf7, 0x85, 0x72, 0x33, 0x20, 0x82,
			0x87, 0x82, 0x5a, 0xb8, 0x64, 0xe5, 0x98, 0x5b,
			0xf0, 0xb0, 0x9d, 0x3d, 0x21, 0xa2, 0x13, 0x59,
			0x78, 0xba, 0x4f, 0xcc, 0x94, 0x67, 0x68, 0x8c,
			0xfd, 0x2b, 0x6f, 0xa2, 0x31, 0xcb, 0xcd, 0x00,
			0xb7, 0x28, 0x43, 0x36, 0xa1, 0x1e, 0xe1, 0xc3,
			0x5f, 0x22, 0x84, 0x15, 0xac, 0x41, 0x3c, 0x44,
			0x0b, 0xfc, 0xda, 0x6f, 0x73, 0xd4, 0x1e, 0xe5,
			0x06, 0x5e, 0xbe, 0xb7, 0xcc, 0x4d, 0x20, 0x68,
			0x25, 0xaa, 0x4b, 0xf3, 0x2b, 0x47, 0xf8, 0x72,
			0x21, 0xc6, 0x8b, 0x04, 0x9c, 0x22, 0xbe, 0xd0,
			0x60, 0x5c, 0x7e, 0x31, 0x24, 0x76, 0x5f, 0x86,
			0xaf, 0x29, 0x1f, 0x4c, 0x9a, 0x6c, 0xdd, 0xda,
			0x9d, 0x0a, 0x84, 0x05, 0x1a, 0xea, 0xfe, 0x02,
			0x20, 0xea, 0xa0, 0x6e, 0x81, 0x01, 0x55, 0x71,
			0xdc, 0x8d, 0x4c, 0xd2, 0x01, 0xfc, 0x76, 0xed,
			0x51, 0x19, 0xf0, 0x12, 0xaf, 0xb5, 0xc6, 0x5b,
			0x46, 0x7e, 0xd9, 0x6c, 0xe4, 0xd7, 0x58, 0x9b,
			0xd1, 0xcb, 0xd4, 0x17, 0xcb, 0x3b, 0xd4, 0x5c,
			0xf7, 0x3f, 0x43, 0xc3, 0x4e, 0xdd, 0x56, 0x1f,
			0x88, 0x91, 0x2c, 0x27, 0x61, 0xa0, 0x26, 0xbe,
			0x0c, 0xad, 0x0a, 0xbd, 0x35, 0x6b, 0xe3, 0xbb,
			0x12, 0xdc, 0x9d, 0xae, 0x08, 0x45, 0x74, 0x1c,
			0xd5, 0x20, 0x65, 0x77, 0xae, 0xdb, 0xaf, 0xd1,
			0x79, 0x6f, 0xe4, 0xdb, 0xbe, 0x1a, 0xa5, 0x0c,
			0x16, 0xbd, 0x2e, 0x05, 0x98, 0x57, 0xf9, 0x69,
			0xe5, 0xc7, 0xe7, 0x25, 0x76, 0x6f, 0xe2, 0xa5,
			0xb8, 0x52, 0x05, 0x3e, 0xa5, 0x7a, 0x3b, 0x2a,
			0xd3, 0xaa, 0x68, 0x4e, 0x81, 0xd6, 0x3e, 0xa2,
			0xa8, 0x5a, 0x83, 0x63, 0x93, 0xc7, 0x68, 0x21,
			0x05, 0x53, 0xdc, 0xa1, 0x07, 0xf2, 0x12, 0x05,
			0x87, 0x5a, 0x35, 0xc7, 0x7d, 0x6a, 0xfb, 0x11,
			0x54, 0x2f, 0xa3, 0xcf, 0x79, 0x3a, 0xbb, 0x44,
			0x8d, 0xe2, 0xaf, 0xe7, 0x90, 0xb4, 0xd3, 0x59,
			0xaf, 0xa3, 0x3a, 0xae, 0x4f, 0xa9, 0xcc, 0x2f,
			0x0c, 0x42, 0x9c, 0x2c, 0x5d, 0x66, 0xf5, 0x60,
			0x0c, 0xc5, 0x30, 0x2c, 0xc8, 0x89, 0xba, 0xda,
			0x01, 0xfd, 0xcf, 0x3d, 0xd2, 0x33, 0xf9, 0x41,
			0xf7, 0x18, 0xad, 0x80, 0xfe, 0x33, 0xd5, 0x83,
			0x67, 0x62, 0x98, 0xa6, 0xc5, 0x70, 0x83, 0xcc,
			0x7e, 0x51, 0x71, 0xe2, 0xfb, 0x2e, 0x4f, 0xc7,
			0xc9, 0x1f, 0x35, 0xa9, 0x7d, 0x9c, 0xe2, 0x5c,
			0x2d, 0x3b, 0x8e, 0x42, 0x03, 0x28, 0xc2, 0xcb,
			0x51, 0x62, 0xf8, 0x5c, 0x76, 0xcd, 0x3d, 0x59,
			0xb2, 0xb3, 0x27, 0x68, 0xf0, 0xa9, 0xbd, 0xbf,
			0x92, 0x3a, 0xd8, 0x9a, 0xd0, 0xd2, 0x1e, 0x74,
			0x6a, 0x06, 0x58, 0x75, 0x94, 0xec, 0xa0, 0x31,
			0x0b, 0x6c, 0xaa, 0xba, 0x84, 0xae, 0x6c, 0xe2,
			0x48, 0x3b, 0xd9, 0xe7, 0x41, 0x34, 0x32, 0xc8,
			0x57, 0x00, 0xf8, 0xe4, 0x1b, 0x40, 0x41, 0x08,
			0x63, 0xfd, 0x54, 0x79, 0x25, 0x00, 0x24, 0xec,
			0xc7, 0x57, 0x3d, 0xac, 0x7a, 0x80, 0x31, 0x3c,
			0x76, 0x8d, 0xbd, 0x0b, 0x3a, 0xe3, 0x8f, 0x55,
			0x20, 0xca, 0xb1, 0x73, 0x85, 0xc1, 0x1e, 0x65,
			0xac, 0xda, 0x27, 0x6f, 0x2b, 0xc4, 0xb1, 0x58,
			0x7e, 0x16, 0xe8, 0x81, 0x48, 0xfd, 0xe2, 0xd3,
			0x55, 0x1b, 0x79, 0xca, 0x2d, 0xf8, 0xe8, 0xf3,
			0x0a, 0xad, 0xec, 0x15, 0x9b, 0x9a, 0x28, 0x54,
			0x17, 0xe6, 0x04, 0x89, 0x6d, 0x6b, 0xbd, 0x9d,
			0x3b, 0xfb, 0x31, 0x7a, 0xf5, 0xa0, 0xa5, 0x84,
			0x7f, 0x89, 0x1d, 0x9f, 0x88, 0x40, 0xb0, 0x4d,
			0xba, 0x83, 0x6e, 0x4c, 0xb0, 0xd2, 0x5a, 0x6f,
			0x0c, 0xb6, 0xe3, 0xd5, 0xb7, 0x86, 0xd4, 0x5a,
			0x9f, 0x5c, 0x15, 0xd3, 0x52, 0x9e, 0xf7, 0x75,
			0xd9, 0xff, 0xd1, 0xe6, 0x0b, 0x19, 0x67, 0xd2,
			0xd9, 0xb0, 0x16, 0xc7, 0x6f, 0xf4, 0x33, 0x86,
			0x98, 0x89, 0x9e, 0x89, 0x58, 0xfb, 0xab, 0xbe,
			0x64, 0xc3, 0x25, 0xca, 0xde, 0x07, 0xbe, 0x2f,
			0xcc, 0x19, 0x1a, 0xb0, 0x3f, 0xd9, 0x81, 0xee,
			0xde, 0xf1, 0xb5, 0x20, 0xcb, 0xa0, 0xc6, 0x83,
			0xb8, 0x65, 0x0e, 0x2d, 0x9a, 0x41, 0x2b, 0x9c,
			0x4c, 0x50, 0xb2, 0xf8, 0xd6, 0x0d, 0xcc, 0xe9,
			0xd4, 0x38, 0x06, 0x3c, 0x70, 0xca, 0x72, 0x0b,
			0xf0, 0xfb, 0xed, 0x80, 0xe8, 0x40, 0x85, 0xf6,
			0xb1, 0xff, 0x06, 0xff, 0xd7, 0x81, 0xb5, 0x42,
			0x06, 0xa7, 0x6c, 0xdb, 0x62, 0x05, 0x98, 0x88,
			0x44, 0xaf, 0x25, 0xb4, 0x32, 0xda, 0xd7, 0xd6,
			0xa1, 0xb1, 0xf9, 0x3b, 0xc3, 0x73, 0xe6, 0x14,
			0xaa, 0x85, 0x16, 0xb7, 0xe9, 0x76, 0x67, 0xe8,
			0xb9, 0xd7, 0x35, 0x36, 0xd5, 0xec, 0x3c, 0x18,
			0x58, 0x81, 0x58, 0x3e, 0x34, 0x8f, 0x89, 0x01,
			0x09, 0x41, 0x7c, 0xd8, 0x02, 0x05, 0x4f, 0x58,
			0xbb, 0xc8, 0xdb, 0x64, 0x14, 0x64, 0x9f, 0x53,
			0xb1, 0x2d, 0x37, 0x55, 0x48, 0x58, 0xc7, 0xe8,
			0xa3, 0x98, 0x31, 0x6d, 0x98, 0x67, 0x5c, 0x91,
			0x7f, 0x08, 0xb5, 0x49, 0x33, 0xa6, 0x23, 0x79,
			0x04, 0xd2, 0x3c, 0x98, 0xa8, 0xfe, 0xb8, 0xf7,
			0xb7, 0xc3, 0xb3, 0xd9, 0x79, 0x57, 0x29, 0x7b,
			0x59, 0x55, 0x23, 0x92, 0xdf, 0xfd, 0xcf, 0x74,
			0xad, 0x10, 0xdb, 0x3a, 0xd2, 0x87, 0x6f, 0x16,
			0x3d, 0x1e, 0xb0, 0x9f, 0x64, 0xd8, 0x9f, 0x9c,
			0xe6, 0xb8, 0x46, 0xad, 0x7c, 0xcf, 0xab, 0x29,
			0x9f, 0x65, 0xe6, 0x5f, 0x6f, 0x2d, 0x89, 0x6d,
			0xbb, 0x42, 0xda, 0x28, 0x88, 0xb4, 0x72, 0xc3,
			0x67, 0xee, 0xac, 0x34, 0x7a, 0x60, 0x8d, 0x24,
			0x96, 0xf5, 0xa0, 0xe1, 0xc0, 0xc3, 0xee, 0x66,
			0x45, 0x34, 0x79, 0xc1, 0xf5, 0xa3, 0xdd, 0xe7,
			0x78, 0x4a, 0x99, 0xbb, 0x6f, 0xa7, 0x2b, 0x98,
			0xef, 0xe0, 0xf8, 0x8d, 0x21, 0x85, 0xc1, 0xb1,
			0x69, 0xb5, 0x7b, 0xb8, 0x64, 0x79, 0x61, 0x29,
			0x57, 0x11, 0xc0, 0x56, 0x0a, 0x61, 0x99, 0xcc,
			0x58, 0x51, 0x4b, 0x82, 0xb7, 0xe6, 0x3e, 0xc2,
			0xa7, 0xe1, 0x83, 0x6c, 0x76, 0x28, 0x9f, 0x4e,
			0xa9, 0xaf, 0xd1, 0x4a, 0x91, 0x17, 0xf7, 0x98,
			0x53, 0x10, 0x7b, 0x82, 0x78, 0x49, 0x09, 0xfa,
			0xa4, 0x93, 0x52, 0x6d, 0x2a, 0x66, 0xc0, 0xe5,
			0xc0, 0x6e, 0x80, 0x60, 0xf0, 0x2e, 0x11, 0x01,
			0xd7, 0x59, 0xa3, 0xab, 0xb9, 0x3b, 0xf3, 0xbe,
			0xe5, 0xcb, 0x63, 0x98, 0xc7, 0xac, 0xbe, 0x6b,
			0xc0, 0x34, 0xa3, 0xe7, 0xd5, 0x8d, 0x44, 0xbb,
			0x09, 0xe9, 0x0d, 0x44, 0xd2, 0x2b, 0x59, 0x25,
			0xca, 0x1f, 0xcc, 0xf6, 0x23, 0x3a, 0xf5, 0x2a,
			0x06, 0x8f, 0x5c, 0x05, 0x54, 0x18, 0x10, 0x83,
			0xf0, 0xd1, 0xc5, 0x89, 0x51, 0x90, 0x46, 0x3d,
			0xe9, 0xf3, 0xd1, 0xbc, 0xb9, 0x72, 0x41, 0x64,
			0x91, 0xb2, 0x4a, 0x8d, 0x1d, 0xde, 0x0a, 0x98,
			0xe3, 0x4c, 0xae, 0xa7, 0x73, 0x25, 0xd9, 0xc7,
			0x96, 0x0a, 0xe5, 0xd4, 0xac, 0x3d, 0xa2, 0x6e,
			0xa1, 0x3a, 0x56, 0x50, 0x0b, 0xd1, 0x69, 0x83,
			0xc2, 0xcf, 0xde, 0x3e, 0xd0, 0x64, 0xf8, 0x13,
			0xdb, 0x18, 0x2e, 0x51, 0x30, 0xb9, 0xd3, 0x93,
			0xf9, 0x91, 0xae, 0xf2, 0x04, 0x9f, 0x01, 0x4e,
			0x9a, 0xf8, 0xaf, 0x0c, 0x1e, 0xb4, 0x36, 0x7b,
			0x49, 0xeb, 0x5b, 0xa3, 0x34, 0x6f, 0xba, 0xe1,
			0x54, 0x98, 0x39, 0x7e, 0x15, 0x31, 0x96, 0xfd,
			0x19, 0x8e, 0xa7, 0x07, 0xe5, 0x95, 0xa6, 0xa8,
			0xe5, 0xce, 0x92, 0xb9, 0xa2, 0x16, 0x6f, 0xa6,
			0x15, 0xcb, 0xa3, 0x79, 0x32, 0xf0, 0x4d, 0xc8,
			0x22, 0x0a, 0xb3, 0xaa, 0x77, 0x39, 0x5d, 0xd1,
			0xd5, 0x1c, 0x57, 0xcf, 0x0b, 0x43, 0x61, 0xe1,
			0x27, 0xad, 0x6a, 0x20, 0xd2, 0xfa, 0xa0, 0x3d,
			0xdd, 0x3e, 0xc8, 0x44, 0x78, 0x40, 0x34, 0xff,
			0x36, 0xd8, 0x69, 0x6b, 0x27, 0x06, 0xd6, 0xba,
			0x0d, 0xca, 0x62, 0xd7, 0xca, 0xab, 0xcf, 0x37,
			0xd0, 0xae, 0x55, 0x1f, 0xf8, 0x0a, 0x19, 0x85,
			0x1b, 0xe1, 0x50, 0x7e, 0x6b, 0x92, 0x8f, 0x92,
			0xc7, 0xc6, 0x29, 0x66, 0x2f, 0xdd, 0xf3, 0xbe,
			0x5c, 0x2b, 0xc3, 0xad, 0xf0, 0x89, 0xf9, 0x48,
			0xef, 0xa2, 0xa7, 0x9c, 0x41, 0x63, 0x1c, 0xb3,
			0xe9, 0x31, 0x43, 0x1d, 0xfe, 0x0a, 0x64, 0x12,
			0x3a, 0xc7, 0x6b, 0x26, 0xaa, 0x43, 0xcb, 0xce,
			0x2c, 0x54, 0xd1, 0x13, 0x09, 0x38, 0xaa, 0x2c,
			0xdf, 0x25, 0xf0, 0x4b, 0xba, 0xc9, 0x8f, 0xd6,
			0x9b, 0x21, 0x44, 0x5a, 0x36, 0x12, 0xc0, 0x0c,
			0x6e, 0xf1, 0x9c, 0x21, 0xd6, 0x0c, 0x5e, 0xc0,
			0x95, 0x01, 0x6a, 0x59, 0x98, 0xc3, 0x41, 0xf8,
			0x5a, 0xe7, 0xca, 0x89, 0x01, 0xa0, 0xbb, 0xc4,
			0x80, 0xf4, 0x87, 0x7b, 0x7e, 0x56, 0xd8, 0xf5,
			0x8b, 0x4e, 0xdb, 0x2b, 0xed, 0x99, 0xd8, 0xaf,
			0x2b, 0x60, 0x07, 0x60, 0x99, 0xf8, 0x93, 0xaa,
			0xbe, 0x9d, 0x98, 0x99, 0x5f, 0x4e, 0x5e, 0xcb,
			0x26, 0x66, 0x48, 0x75, 0xdf, 0x84, 0x04, 0xc5,
			0x5d, 0xad, 0x67, 0x12, 0x93, 0xba, 0x0c, 0x73,
			0xaa, 0xd6, 0xc9, 0xad, 0x58, 0xec, 0x54, 0x3e,
			0x2c, 0xde, 0x2c, 0x1e, 0xc4, 0x4d, 0xcc, 0x47,
			0x1b, 0x94, 0x1e, 0x4a, 0x0c, 0x25, 0xf8, 0xf3,
			0xe8, 0x67, 0xfb, 0xfb, 0x51, 0x02, 0x13, 0xe6,
			0x52, 0x9c, 0x04, 0x9b, 0x82, 0x8c, 0xe6, 0x8e,
			0xc5, 0x72, 0x0e, 0x9e, 0x62, 0x6b, 0x83, 0x63,
			0xc1, 0xc2, 0x75, 0x50, 0x0a, 0xeb, 0x2b, 0x9c,
			0x8c, 0x6f, 0xfd, 0x1f, 0xd1, 0xb6, 0xee, 0xc3,
			0xb9, 0x9f, 0x25, 0x85, 0x7f, 0x97, 0x61, 0x51,
			0x3b, 0x92, 0x66, 0x4a, 0x58, 0xe7, 0x60, 0x1f,
			0x8f, 0xcf, 0x16, 0x4d, 0x02, 0xc0, 0xee, 0x4a,
			0x76, 0x88, 0xf9, 0x33, 0x83, 0xe2, 0x6d, 0x31,
			0x1c, 0x92, 0xcd, 0x33, 0x3a, 0xd8, 0x43, 0x74,
			0xdd, 0x40, 0x55, 0x85, 0x47, 0x17, 0xd6, 0x8c,
			0xe9, 0x32, 0x75, 0x86, 0x70, 0x05, 0x08, 0x41,
			0x33, 0x5b, 0x48, 0x67, 0x6e, 0x14, 0x23, 0xcd,
			0x01, 0x6a, 0x7d, 0x5b, 0x36, 0x95, 0x66, 0x47,
			0x6c, 0x4c, 0x66, 0xb1, 0xa3, 0x7f, 0x62, 0xcd,
			0xfc, 0xdc, 0xdb, 0x0f, 0x7c, 0xc4, 0x28, 0x26,
			0xcf, 0xac, 0x4a, 0xb9, 0x19, 0x95, 0xca, 0xd6,
			0x70, 0x08, 0x3d, 0x8a, 0x0e, 0x88, 0x1b, 0xea,
			0xb5, 0x95, 0x6f, 0x28, 0xa1, 0x01, 0xcc, 0xf3,
			0x3f, 0x2e, 0xc3, 0x48, 0xaa, 0x02, 0x14, 0x0f,
			0x88, 0xf9, 0x20, 0xf1, 0x4c, 0xd4, 0xc8, 0x8c,
			0x40, 0xa2, 0xe3, 0x1c, 0x27, 0x49, 0x77, 0x59,
			0x0e, 0x8e, 0x65, 0xdf, 0xb2, 0xa8, 0xf1, 0x2e,
			0xb4, 0x65, 0x33, 0x5b, 0x1f, 0xbf, 0xe7, 0x19,
			0xf9, 0xfe, 0x19, 0x6f, 0x0f, 0x3e, 0x16, 0xd0,
			0x39, 0x8f, 0x10, 0x6a, 0x46, 0x35, 0xca, 0xae,
			0x4a, 0x1a, 0x85, 0xee, 0x91, 0x3e, 0xc1, 0x0f,
			0x70, 0x25, 0xb9, 0x9d, 0xbb, 0x9c, 0x75, 0xcb,
			0x63, 0x78, 0xa2, 0x63, 0x36, 0x52, 0xea, 0xff,
			0x73, 0x45, 0x5b, 0xab, 0xd2, 0x85, 0x62, 0x06,
			0x63, 0x1b, 0x03, 0xcb, 0x1b, 0x1d, 0xaf, 0xd4,
			0x3d, 0x13, 0x65, 0x68, 0xdc, 0xad, 0x9a, 0x64,
			0xd0, 0x8b, 0xbb, 0xb1, 0xd8, 0xf4, 0xcc, 0x42,
			0x1d, 0x6d, 0x77, 0xed, 0xc9, 0x8b, 0x1e, 0x2c,
			0x5e, 0xa5, 0x5b, 0xe6, 0x12, 0x94, 0x8f, 0x91,
			0x4f, 0x70, 0x6e, 0x29, 0x70, 0x72, 0x85, 0xec,
			0x71, 0x2b, 0xc1, 0x9e, 0x1a, 0xd9, 0x70, 0xd7,
			0x6a, 0x5a, 0xee, 0x58, 0xe5, 0x9d, 0x8c, 0xe4,
			0x8b, 0x91, 0xd2, 0x9b, 0x8e, 0x77, 0x9b, 0xc1,
			0xbe, 0xb7, 0x44, 0x80, 0x3f, 0x0f, 0x0a, 0xbd,
			0x4a, 0xe8, 0xe9, 0x20, 0xd0, 0x6f, 0x04, 0xb9,
			0xc6, 0xac, 0x01, 0xff, 0x80, 0x2d, 0x1e, 0x53,
			0xc1, 0x55, 0x4c, 0xbe, 0xfc, 0xb1, 0x0d, 0x13,
			0x9b, 0xde, 0xf8, 0x3e, 0xce, 0x75, 0x5f, 0x5d,
			0xa7, 0x02, 0x8d, 0x17, 0x3d, 0xaa, 0xfe, 0xde,
			0xc5, 0x7f, 0xee, 0xe4, 0xfb, 0x06, 0xfe, 0xa1,
			0x6d, 0x67, 0x44, 0x2d, 0xa6, 0xfc, 0x0f, 0xb7,
			0x13, 0x75, 0xe5, 0xd2, 0x08, 0x89, 0x0a, 0x87,
			0x73, 0xf0, 0x0d, 0xd9, 0xe2, 0x19, 0xd8, 0xe1,
			0xcc, 0x3f, 0x27, 0x9d, 0x4c, 0x22, 0xae, 0xdb,
			0x32, 0x63, 0xfa, 0x10, 0x26, 0xaf, 0xd5, 0xa6,
			0x4b, 0x0f, 0xe6, 0xac, 0x1b, 0x17, 0xee, 0xa3,
			0xb6, 0x53, 0x61, 0xee, 0x3d, 0x25, 0x22, 0xb9,
			0x6c, 0xac, 0xd7, 0x1b, 0x46, 0x41, 0x08, 0xb2,
			0xd7, 0xe5, 0xb8, 0xce, 0xea, 0xe8, 0xd4, 0xc4,
			0xfc, 0xa6, 0x49, 0xc7, 0x37, 0x98, 0x3f, 0x81,
			0x47, 0xb3, 0xc1, 0x7d, 0xb2, 0xd3, 0x3c, 0x04,
			0x00, 0x64, 0x8d, 0x03, 0x50, 0x85, 0xd8, 0x61,
			0x13, 0x47, 0x31, 0x8a, 0x31, 0xdc, 0x9b, 0xd7,
			0x16, 0xb1, 0x29, 0x42, 0x70, 0x8f, 0xa4, 0x44,
			0xb5, 0x75, 0xc1, 0xb7, 0xfc, 0xc9, 0x4f, 0xf6,
			0x94, 0x5c, 0x80, 0xcb, 0x78, 0x41, 0xc9, 0x32,
			0xa6, 0x43, 0xbe, 0xe6, 0xcf, 0x4e, 0x62, 0x23,
			0xe8, 0xa0, 0xe2, 0x80, 0xd8, 0x76, 0x3a, 0x55,
			0x60, 0x31, 0x49, 0xf3, 0x8e, 0x73, 0x40, 0x8d,
			0xb5, 0xbe, 0x59, 0xe2, 0x26, 0xc3, 0xaa, 0x6f,
			0xa2, 0x0d, 0x05, 0x3c, 0x85, 0x73, 0xa6, 0x06,
			0x33, 0x87, 0x4c, 0xff, 0xd9, 0x67, 0x22, 0x15,
			0xd2, 0x95, 0xf0, 0xe8, 0x48, 0x19, 0xe7, 0xbf,
			0xab, 0xf4, 0xf6, 0xb8, 0xeb, 0x82, 0xc0, 0xf8,
			0x1c, 0x68, 0x88, 0x72, 0xa1, 0xa7, 0x00, 0xf2,
			0xa3, 0x41, 0x4c, 0xb2, 0xc9, 0x8e, 0x14, 0xc3,
			0x83, 0x08, 0x99, 0x01, 0x06, 0x8a, 0x11, 0xf6,
			0x04, 0x78, 0xf7, 0x50, 0xe7, 0xa7, 0x18, 0x99,
			0x17, 0x36, 0xf3, 0x06, 0x21, 0x20, 0x3b, 0x8b,
			0x38, 0x1a, 0xe6, 0x93, 0x43, 0x4c, 0x94, 0xf9,
			0xc9, 0x4a, 0xae, 0xcf, 0x98, 0x35, 0x69, 0x63,
			0x97, 0xdf, 0x46, 0xa9, 0x8b, 0xbd, 0x62, 0x83,
			0x5c, 0x9e, 0xe9, 0xe5, 0xa8, 0x97, 0xd1, 0xe2,
			0xc7, 0x2e, 0x48, 0x56, 0xdc, 0x5b, 0x8e, 0x0d,
			0xbf, 0x4e, 0xd9, 0xd2, 0x79, 0x85, 0xa6, 0xa5,
			0x51, 0xc6, 0xab, 0x50, 0x64, 0x5f, 0xdd, 0xe6,
			0x57, 0x60, 0xcc, 0xfe, 0x18, 0x0d, 0x2f, 0x9f,
			0x20, 0xd7, 0xc1, 0xc0, 0x1e, 0xc6, 0x04, 0x23,
			0x79, 0x3d, 0x68, 0x14, 0xf5, 0x3a, 0x91, 0x6c,
			0x52, 0x62, 0x22, 0x69, 0x45, 0xfc, 0x0a, 0x55,
			0x66, 0x8b, 0x15, 0xf2, 0xc7, 0x47, 0x20, 0xdf,
			0x2c, 0x4e, 0x92, 0xdf, 0xdc, 0x1d, 0x00, 0x45,
			0x29, 0xf4, 0xcc, 0x0b, 0xbf, 0xb5, 0x85, 0x2d,
			0x00, 0xe3, 0x26, 0x5b, 0x7d, 0x91, 0xe1, 0x15,
			0x97, 0xb2, 0x25, 0xbe, 0xf7, 0xeb, 0xe8, 0x07,
			0xd8, 0x30, 0x61, 0x2e, 0x0d, 0x3a, 0x2a, 0xa0,
			0x57, 0x5b, 0x05, 0x69, 0xb2, 0x5c, 0x6a, 0x52,
			0x11, 0xf1, 0xf2, 0x1d, 0xe4, 0x57, 0x9f, 0x67,
			0xdb, 0xf0, 0xe4, 0xff, 0x2d, 0x3b, 0x03, 0x0f,
			0x3a, 0x67, 0xea, 0x1f, 0x72, 0x89, 0xc9, 0x76,
			0xd0, 0x2d, 0xed, 0xde, 0x54, 0xdd, 0x9e, 0xc9,
			0x03, 0xbc, 0x69, 0x8e, 0xe7, 0x4d, 0x88, 0xa2,
			0xc3, 0x31, 0x80, 0x29, 0xeb, 0x4c, 0xe6, 0xa7,
			0x45, 0xa3, 0xc3, 0xdf, 0x44, 0x97, 0x90, 0x76,
			0x0e, 0x96, 0xa1, 0x69, 0x7f, 0x1f, 0xeb, 0x02,
			0x9c, 0xe4, 0x1f, 0xc4, 0x87, 0x3a, 0x3f, 0x1b,
			0xf2, 0x14, 0x61, 0x33, 0x8a, 0x06, 0x38, 0xb4,
			0xc8, 0xb3, 0x87, 0x2d, 0x16, 0x9d, 0xe7, 0xa6,
			0x65, 0xe1, 0x4e, 0x83, 0x2d, 0x6a, 0x6a, 0xb3,
			0x6d, 0x7a, 0xad, 0x4d, 0x0a, 0xe6, 0x2d, 0xaa,
			0x90, 0x17, 0xb8, 0xf0, 0xc1, 0x14, 0x36, 0xb9,
			0xc2, 0x3c, 0x02, 0xc9, 0xc5, 0x9d, 0xd4, 0xe8,
			0x08, 0x8d, 0x9b, 0xd5, 0x56, 0x32, 0x4d, 0xdd,
			0xb4, 0x91, 0x7b, 0x0f, 0xbf, 0x4b, 0x7b, 0xbb,
			0x78, 0x94, 0xe3, 0x3b, 0xa1, 0x36, 0x49, 0x36,
			0x9f, 0x7c, 0x55, 0xef, 0x3c, 0xcf, 0xbe, 0x23,
			0x43, 0x48, 0x29, 0xef, 0xcb, 0x24, 0x05, 0x97,
			0x3a, 0xa3, 0x6e, 0xe3, 0x77, 0x73, 0x83, 0xee,
			0xa5, 0x40, 0xa3, 0xf9, 0xd1, 0xb6, 0x5d, 0xdc,
			0x8e, 0xb6, 0x22, 0x74, 0x31, 0xb7, 0xc6, 0x3f,
			0x1b, 0xb7, 0xbd, 0x81, 0x8a, 0x12, 0xf3, 0x33,
			0x5b, 0x0c, 0x34, 0x76, 0x35, 0x84, 0x3d, 0x5b,
			0x19, 0x49, 0xcd, 0x75, 0x6a, 0x4e, 0x61, 0x03,
			0x4a, 0xb0, 0x20, 0xc6, 0x82, 0x05, 0xfc, 0xba,
			0x7e, 0xec, 0x71, 0x3c, 0x08, 0x08, 0xc8, 0xb5,
			0x4f, 0xb7, 0x32, 0x30, 0xac, 0x14, 0x8a, 0x21,
			0x72, 0xed, 0xfc, 0x92, 0x67, 0x96, 0xfc, 0x09,
			0x65, 0xbe, 0x77, 0xc5, 0x66, 0xdc, 0x64, 0x5d,
			0xd5, 0x36, 0xda, 0x7b, 0xa5, 0x04, 0x4f, 0x46,
			0xc0, 0xbd, 0xf3, 0xc8, 0x8c, 0x34, 0x74, 0x6b,
			0xad, 0xc6, 0xba, 0xc6, 0xd6, 0x61, 0xfd, 0x95,
			0x6b, 0x36, 0x1c, 0x35, 0x1f, 0xde, 0xcb, 0xc5,
			0xa3, 0x4f, 0xe4, 0x6d, 0x98, 0xab, 0xda, 0x0b,
			0xec, 0xcf, 0x15, 0xf2, 0xa2, 0x3f, 0x57, 0x6c,
			0x16, 0x22, 0x40, 0xcc, 0xd8, 0xdd, 0x58, 0x18,
			0x31, 0x43, 0x80, 0x06, 0xc0, 0x17, 0xa4, 0x20,
			0x63, 0x2b, 0x47, 0xda, 0x96, 0xb7, 0x46, 0x5f,
			0xa5, 0x83, 0xd7, 0x4e, 0x33, 0xec, 0x43, 0x1c,
			0xf0, 0x8a, 0x12, 0x9b, 0x37, 0x51, 0xee, 0x41,
			0xc0, 0x36, 0x2e, 0x81, 0x1d, 0x5c, 0x0b, 0x6b,
			0x85, 0x30, 0x74, 0x99, 0x80, 0x35, 0xb0, 0xc5,
			0x4e, 0x21, 0xc1, 0x60, 0x70, 0xf7, 0xe4, 0x10,
			0xb5, 0x93, 0x01, 0x6b, 0xae, 0x52, 0xfe, 0x17,
			0x63, 0x53, 0xe3, 0xc2, 0xde, 0x0d, 0x29, 0x6c,
			0x72, 0x3b, 0xdc, 0xa3, 0xe6, 0x25, 0x1e, 0x33,
			0xa9, 0x2f, 0xb7, 0xee, 0x6c, 0xd5, 0xe2, 0x18,
			0x39, 0xcc, 0x06, 0x7b, 0x30, 0x33, 0xd5, 0xfe,
			0x53, 0x75, 0xf4, 0x3e, 0xd2, 0x4a, 0xc3, 0x62,
			0x44, 0x59, 0x45, 0x8d, 0x8d, 0xde, 0x7b, 0xa0,
			0x36, 0x3a, 0x3a, 0x42, 0xec, 0xed, 0x2d, 0x2a,
			0x1e, 0xdf, 0x86, 0x22, 0x48, 0x4f, 0x4d, 0x30,
			0xa4, 0xbe, 0xd1, 0xf2, 0x95, 0xa9, 0x27, 0x4a,
			0xfc, 0x74, 0xa8, 0x82, 0x77, 0xbb, 0x9e, 0xa1,
			0xb9, 0xf1, 0xf0, 0xc8, 0xed, 0xe8, 0xe4, 0xa3,
			0x94, 0xc8, 0xad, 0x3f, 0x03, 0x35, 0xbb, 0x28,
			0x6e, 0x35, 0x43, 0x5b, 0x4a, 0x1a, 0xe4, 0xce,
			0x59, 0x0c, 0xab, 0xa5, 0x90, 0x0a, 0x9d, 0x86,
			0xf3, 0x45, 0x7d, 0x30, 0x11, 0x13, 0x39, 0xc6,
			0x3b, 0x91, 0x7d, 0x95, 0x4c, 0x8b, 0xb6, 0xa6,
			0xd7, 0xd9, 0xa3, 0xbe, 0xff, 0xba, 0x4a, 0x5c,
			0xcb, 0x70, 0x44, 0x28, 0xb5, 0x04, 0x86, 0xfe,
			0x12, 0x4c, 0x92, 0x38, 0x94, 0xf2, 0x74, 0x17,
			0x39, 0xf8, 0xa7, 0xbd, 0xb6, 0x90, 0xb0, 0x04,
			0xd1, 0x80, 0x35, 0x67, 0x14, 0x54, 0xf0, 0xc6,
			0x3a, 0x60, 0xb0, 0x8c, 0x88, 0x75, 0xaa, 0xb8,
			0xdb, 0xc6, 0x7f, 0x12, 0x80, 0x25, 0x9e, 0x06,
			0xc7, 0x1f, 0xab, 0x3f, 0xe3, 0x35, 0x81, 0xfa,
			0xcd, 0xd8, 0xe3, 0x5b, 0x01, 0x67, 0x4a, 0xb0,
			0x83, 0x15, 0xe8, 0x27, 0xbc, 0xd2, 0x1c, 0x4c,
			0x58, 0x59, 0x81, 0xec, 0xb6, 0x37, 0x4a, 0xa4,
			0x83, 0xd3, 0x05, 0x60, 0xb2, 0x01, 0x45, 0x7c,
			0x56, 0x9b, 0xf8, 0x73, 0x9b, 0xe8, 0x2a, 0x55,
			0x6f, 0xa1, 0xcb, 0x6c, 0xbd, 0x08, 0xd2, 0x2e,
			0x49, 0x19, 0x4d, 0x43, 0x39, 0x60, 0x3b, 0x45,
			0x37, 0xdd, 0x9f, 0xda, 0xf4, 0x97, 0x06, 0xf1,
			0x73, 0xc1, 0xd7, 0x49, 0xd5, 0x67, 0xd5, 0xca,
			0x07, 0xf7, 0x6c, 0xff, 0x85, 0x06, 0x7b, 0xba,
			0xa4, 0xda, 0x8f, 0x60, 0xdf, 0x28, 0x6f, 0xaa,
			0xa0, 0x6a, 0xe1, 0xf2, 0x52, 0x75, 0xde, 0x85,
			0x9d, 0xad, 0xe7, 0x46, 0x31, 0xd6, 0xfd, 0x41,
			0x62, 0x11, 0x00, 0xe9, 0x12, 0x13, 0xbc, 0x01,
			0x7d, 0x11, 0xae, 0x55, 0xaf, 0xb6, 0xd1, 0xf2,
			0x21, 0x82, 0xb8, 0xaa, 0xbe, 0x53, 0xfb, 0xd1,
			0x3e, 0x01, 0x89, 0x27, 0x28, 0x20, 0x69, 0x09,
			0xe1, 0x68, 0xc9, 0x87, 0xd3, 0x83, 0x57, 0xc2,
			0x3b, 0x58, 0xfb, 0x3c, 0x69, 0x4c, 0x3c, 0x17,
			0xff, 0x4c, 0xad, 0xb1, 0xb4, 0x45, 0x7c, 0x0b,
			0x0d, 0xba, 0xe1, 0x95, 0x93, 0x08, 0xfe, 0xf3,
			0x8d, 0x15, 0xad, 0xe3, 0x27, 0x06, 0xaf, 0xda,
			0x5c, 0xf0, 0x18, 0x95, 0xe9, 0x17, 0x55, 0xb2,
			0x69, 0x62, 0x53, 0x5d, 0x6a, 0xbc, 0x85, 0x84,
			0xbc, 0xf9, 0xd1, 0xea, 0x37, 0x65, 0x2b, 0xa9,
			0x5f, 0xf4, 0xdb, 0x20, 0x69, 0x23, 0xc9, 0xa8,
			0xfa, 0xaa, 0x05, 0xb6, 0x17, 0x6f, 0x0d, 0x89,
			0x64, 0x56, 0x91, 0x9e, 0xcb, 0xb2, 0xb7, 0x9a,
			0x05, 0x72, 0xba, 0x6d, 0xc9, 0x36, 0xbe, 0x3e,
			0x92, 0x4a, 0x85, 0xf8, 0xfa, 0x5b, 0x63, 0xfd,
			0x93, 0x20, 0x07, 0xb1, 0x9d, 0x88, 0xcf, 0x30,
			0xf0, 0x05, 0x58, 0x1d, 0x24, 0x71, 0xa2, 0xa2,
			0xaa, 0x5a, 0xb8, 0x31, 0x7d, 0xbd, 0xf5, 0xe9,
			0x67, 0x54, 0x66, 0x1f, 0x8d, 0x8a, 0x8d, 0x23,
			0x3e, 0x30, 0x98, 0xeb, 0xfc, 0xb3, 0x8e, 0x6c,
			0xe5, 0xc1, 0x5f, 0xe8, 0xcd, 0x3e, 0x3d, 0xfe,
			0x8d, 0x7d, 0xf0, 0x83, 0x41, 0x51, 0x51, 0xe0,
			0xcf, 0x9b, 0x1b, 0x09, 0x0d, 0x85, 0x72, 0xb3,
			0x9c, 0x8c, 0x5c, 0x39, 0x3d, 0x53, 0x67, 0xf6,
			0xd4, 0x45, 0x8e, 0xd2, 0xd1, 0x48, 0xe1, 0xe4,
			0x15, 0x5e, 0x49, 0x1b, 0xf5, 0x9d, 0xc0, 0xd8,
			0x83, 0x57, 0xd8, 0xe4, 0x61, 0xb7, 0x24, 0x4d,
			0xca, 0x92, 0x29, 0x3d, 0x39, 0xb6, 0x29, 0xc5,
			0x36, 0x29, 0x80, 0xdf, 0x03, 0x0c, 0x6c, 0x4d,
			0xef, 0x94, 0xd9, 0xb2, 0x03, 0x2b, 0xab, 0x94,
			0x5a, 0x73, 0x48, 0x4d, 0xc3, 0x4d, 0x30, 0x79,
			0x5f, 0xd3, 0x33, 0x3d, 0x0e, 0xbe, 0x1b, 0x19,
			0x19, 0xa8, 0xa8, 0xc7, 0x08, 0x8b, 0xe6, 0x11,
			0x20, 0x06, 0x12, 0xbc, 0x80, 0x83, 0xe4, 0xe0,
			0x3c, 0xcd, 0x35, 0x16, 0xd3, 0x81, 0x38, 0x05,
			0x62, 0x66, 0xcf, 0xab, 0x2c, 0x3a, 0x3e, 0xb8,
			0xfd, 0x0b, 0x86, 0xea, 0xe7, 0x7a, 0xbd, 0x22,
			0x61, 0x46, 0x1b, 0x10, 0x53, 0x24, 0x89, 0x37,
			0xac, 0x5d, 0x8d, 0x96, 0x27, 0x5e, 0x5d, 0x35,
			0xe1, 0x89, 0xe4, 0xce, 0x7e, 0x92, 0x79, 0x9b,
			0xa1, 0xae, 0xd7, 0x6b, 0x2d, 0xd7, 0xa1, 0xa8,
			0xce, 0x23, 0xd9, 0x9d, 0xed, 0xce, 0x35, 0x14,
			0x10, 0xb2, 0x06, 0x4b, 0xf3, 0x8f, 0x49, 0xd3,
			0x66, 0x96, 0xd3, 0xe3, 0x09, 0xc2, 0xc2, 0xea,
			0x99, 0xce, 0xf8, 0x50, 0x56, 0x16, 0xda, 0x7d,
			0xb2, 0x70, 0xf7, 0x54, 0xa7, 0xf7, 0xb5, 0x1a,
			0x38, 0x40, 0xe6, 0x2a, 0x81, 0x74, 0xf6, 0x5c,
			0x9b, 0xd0, 0xd2, 0xe7, 0x5e, 0x3f, 0x36, 0xd7,
			0x9b, 0xf8, 0x02, 0x4f, 0xd6, 0x0e, 0xe5, 0xa6,
			0xaf, 0x2c, 0xe5, 0xfc, 0xaf, 0x7e, 0x18, 0xca,
			0xc8, 0x0b, 0x94, 0x18, 0x7f, 0xe7, 0xc5, 0x0f,
			0xfc, 0xd7, 0x16, 0xc6, 0xc5, 0x15, 0xf0, 0x1f,
			0x72, 0xa9, 0x87, 0x6c, 0x64, 0x82, 0xcb, 0x60,
			0xa7, 0x8c, 0x6b, 0x3b, 0xe6, 0xda, 0xbf, 0x8a,
			0x03, 0x39, 0x91, 0x91, 0x37, 0x3f, 0x46, 0xbd,
			0xc2, 0x21, 0xd0, 0x2f, 0xef, 0xc3, 0x52, 0xbc,
			0x94, 0x6e, 0x66, 0x12, 0xfb, 0x8e, 0xed, 0x11,
			0xd1, 0xbc, 0x88, 0xca, 0x93, 0x64, 0x93, 0x1b,
			0x86, 0x16, 0x4e, 0x56, 0x2d, 0xaf, 0x62, 0x08,
			0x94, 0x01, 0x0f, 0x44, 0x4f, 0xfc, 0x59, 0x51,
			0x0e, 0x0f, 0x90, 0x66, 0x90, 0x0e, 0x83, 0x50,
			0x0a, 0x1e, 0x52, 0xc4, 0x16, 0x45, 0xce, 0xda,
			0xe4, 0x79, 0xa2, 0x73, 0x29, 0xed, 0xf6, 0xae,
			0x0c, 0x4f, 0x9b, 0x3c, 0xb7, 0x68, 0x63, 0x84,
			0xd0, 0xf9, 0xeb, 0x71, 0xe4, 0xcd, 0x84, 0x58,
			0x4d, 0xd1, 0x2b, 0x1f, 0xa1, 0x66, 0xb1, 0x68,
			0x6a, 0xeb, 0x9d, 0x9b, 0x58, 0x96, 0x4e, 0xdf,
			0x45, 0x96, 0x48, 0x54, 0x8a, 0xfd, 0x76, 0x6b,
			0x92, 0xf2, 0x4f, 0xef, 0xd0, 0x50, 0x7b, 0x4a,
			0xf3, 0xba, 0xfb, 0xcc, 0x66, 0x72, 0x00, 0xbf,
			0x16, 0x24, 0x63, 0x37, 0xaf, 0xa0, 0x05, 0xb8,
			0x8b, 0x52, 0xae, 0x99, 0x4d, 0xd4, 0x13, 0xb4,
			0xb0, 0x78, 0x60, 0x4e, 0x38, 0x29, 0x34, 0x68,
			0xed, 0x02, 0x0d, 0x97, 0x26, 0x3e, 0x0f, 0xb3,
			0xef, 0x7c, 0x0c, 0x7a, 0x00, 0xde, 0xa4, 0x9c,
			0x54, 0xc7, 0x96, 0xc2, 0xdc, 0xe4, 0x1e, 0x79,
			0xfd, 0x9e, 0x5a, 0x35, 0x07, 0x94, 0x10, 0xaf,
			0xe3, 0x8b, 0x42, 0xd8, 0xc6, 0xb1, 0x44, 0xd6,
			0x93, 0x77, 0xb2, 0xc1, 0x2a, 0x22, 0x8b, 0xcb,
			0xb4, 0xbb, 0xe8, 0x7d, 0x73, 0x89, 0xe8, 0xfe,
			0xd1, 0xa1, 0xef, 0xdf, 0x97, 0xd6, 0x99, 0x45,
			0xf7, 0x2a, 0x66, 0x86, 0x49, 0xd7, 0x49, 0xd1,
			0x3c, 0xe2, 0x91, 0xc2, 0x79, 0xe7, 0x1e, 0xa7,
			0xc3, 0x36, 0x8f, 0x01, 0x67, 0x88, 0xf2, 0xf8,
			0x26, 0x1d, 0xda, 0x89, 0x52, 0xea, 0xe9, 0x7f,
			0x10, 0x83, 0xd3, 0x48, 0xfc, 0x0b, 0x4c, 0x21,
			0xb4, 0x6f, 0x0c, 0xa8, 0x27, 0x52, 0x97, 0x4a,
			0xb6, 0x3f, 0x8c, 0x96, 0x3b, 0xb5, 0xfb, 0xf6,
			0xb5, 0xe0, 0xc7, 0xe8, 0xd6, 0xec, 0x24, 0xbf,
			0x39, 0x64, 0x0a, 0xe6, 0x1a, 0xad, 0xb6, 0x47,
			0xe5, 0x3c, 0x7e, 0xaa, 0x0f, 0xd5, 0x7b, 0x62,
			0x2c, 0xdc, 0x23, 0xcf, 0xe3, 0x29, 0x5b, 0x66,
			0x8f, 0x0d, 0x38, 0xe3, 0xc8, 0x8a, 0x5c, 0x2b,
			0x36, 0x7e, 0x9a, 0x3c, 0x55, 0x3d, 0xf6, 0x9e,
			0x93, 0x5e, 0x71, 0x28, 0x2a, 0x1b, 0xc8, 0x96,
			0xc5, 0xcd, 0xf7, 0xfc, 0x04, 0x80, 0x60, 0x90,
			0x86, 0xc9, 0xc8, 0xc1, 0x42, 0xd8, 0x99, 0xc4,
			0xde, 0xb6, 0x8c, 0x61, 0xc9, 0x08, 0xc9, 0xc2,
			0xb5, 0xc6, 0x80, 0x17, 0x57, 0x53, 0x26, 0x96,
			0xde, 0x68, 0x0c, 0x1c, 0x7f, 0xfd, 0xec, 0xdc,
			0x8a, 0x77, 0x6a, 0x36, 0x80, 0xa8, 0x87, 0x26,
			0x49, 0xee, 0x97, 0x91, 0x4a, 0x7f, 0xb7, 0xed,
			0xfe, 0x15, 0x82, 0xd2, 0x13, 0x5a, 0x2d, 0x16,
			0x7e, 0x57, 0x65, 0xfb, 0x05, 0x6f, 0x49, 0x3e,
			0x0b, 0x2f, 0x52, 0x4c, 0x78, 0x20, 0x52, 0x79,
			0x2e, 0x44, 0xc9, 0xf1, 0x03, 0xe4, 0x5a, 0xaa,
			0xb6, 0x48, 0xea, 0x63, 0x99, 0x1e, 0xe6, 0x46,
			0x28, 0x93, 0xa3, 0xe1, 0x88, 0xa5, 0x3d, 0xad,
			0x82, 0x66, 0x85, 0x7a, 0x77, 0x68, 0xaa, 0x41,
			0x72, 0x41, 0xa7, 0x9f, 0xe6, 0xc3, 0x86, 0x8b,
			0xc4, 0x6a, 0x10, 0xbe, 0x91, 0x8f, 0x50, 0x20,
			0x28, 0x50, 0xa3, 0x4c, 0xaa, 0x0d, 0x09, 0x9a,
			0xd0, 0x87, 0x47, 0xe5, 0x9c, 0x2e, 0x96, 0x7b,
			0x0c, 0x2f, 0xd9, 0x79, 0xdd, 0xb9, 0x10, 0x4c,
			0x96, 0x1d, 0x3b, 0x8c, 0x88, 0x59, 0x30, 0xb1,
			0x09, 0x7c, 0x92, 0x87, 0xce, 0x95, 0x91, 0x51,
			0x37, 0x62, 0xcb, 0x9c, 0x95, 0xd5, 0xc1, 0xac,
			0x0d, 0x21, 0x51, 0x69, 0x34, 0xd9, 0x9b, 0x30,
			0x1f, 0x95, 0x60, 0x1d, 0x34, 0x6d, 0x52, 0xbe,
			0xd4, 0x36, 0x3d, 0x15, 0xca, 0x3c, 0x95, 0x3c,
			0x75, 0x56, 0x3e, 0x88, 0xc9, 0x7d, 0xa9, 0x9e,
			0xa8, 0xb4, 0x8b, 0x5d, 0x7e, 0x98, 0x90, 0x60,
			0x8b, 0x10, 0x34, 0x36, 0x37, 0x01, 0xc0, 0xa9,
			0x8e, 0xc6, 0x72, 0x9c, 0x09, 0x34, 0x7b, 0xa4,
			0xba, 0x3c, 0x01, 0x7b, 0xb3, 0x25, 0xca, 0x27,
			0x8e, 0x5f, 0xe5, 0x0f, 0x51, 0x15, 0x3f, 0xe2,
			0xf3, 0x1f, 0x21, 0xd9, 0x28, 0x82, 0xaf, 0xcb,
			0x30, 0xfa, 0xa5, 0x8b, 0xcd, 0x21, 0x6e, 0xe9,
			0xd5, 0xbe, 0x86, 0x5c, 0xf7, 0x9d, 0x7b, 0xdd,
			0x75, 0xc3, 0x94, 0x46, 0x50, 0xec, 0xa0, 0x9c,
			0xd4, 0x55, 0x90, 0xee, 0x30, 0xc4, 0xac, 0x8b,
			0xa6, 0x39, 0x20, 0x15, 0xf1, 0xf1, 0xe7, 0x37,
			0x4b, 0x2a, 0x26, 0x89, 0x35, 0xe2, 0x8c, 0x86,
			0x70, 0x40, 0x24, 0x80, 0x3f, 0x41, 0xd7, 0x84,
			0x96, 0xbd, 0x86, 0x67, 0x0f, 0x13, 0x32, 0x81,
			0x94, 0xf2, 0xbd, 0x3b, 0x4e, 0xbb, 0xe8, 0x7f,
			0x17, 0x3d, 0x29, 0xc8, 0x4c, 0xd9, 0xf6, 0x8c,
			0x6d, 0xc6, 0x5a, 0xf5, 0x98, 0x50, 0xb1, 0xfa,
			0xec, 0x34, 0xa1, 0xd8, 0xde, 0x9c, 0xf3, 0xf1,
			0x8b, 0x3d, 0x07, 0x8f, 0x45, 0x1e, 0x41, 0x4e,
			0x9c, 0xf9, 0x11, 0x86, 0x85, 0xff, 0x17, 0x90,
			0x56, 0x81, 0x97, 0xab, 0x8d, 0xd6, 0x37, 0xa6,
			0xbc, 0x0e, 0xdf, 0x9b, 0xe3, 0x2d, 0xf8, 0x89,
			0xe9, 0xae, 0x00, 0xe8, 0x84, 0x8e, 0x22, 0x53,
			0xf0, 0xe6, 0x7d, 0x24, 0x3d, 0x61, 0x79, 0xfe,
			0x8a, 0xe4, 0x79, 0xa3, 0xae, 0x7f, 0xef, 0x86,
			0x8f, 0x89, 0x94, 0x18, 0x59, 0x3c, 0xb9, 0x70,
			0x96, 0x26, 0x84, 0xcf, 0x34, 0xb6, 0x0a, 0xf8,
			0x6f, 0xcd, 0xb8, 0x4b, 0x1a, 0xdf, 0xd6, 0x27,
			0xf2, 0xcc, 0x7c, 0x0e, 0x98, 0x91, 0xe0, 0x32,
			0x2f, 0x82, 0xec, 0xbe, 0x3f, 0xdd, 0x8f, 0xab,
			0x2e, 0x5d, 0x38, 0xf5, 0x50, 0x24, 0x4e, 0x17,
			0xd0, 0x1f, 0x66, 0xb0, 0xd7, 0x06, 0xe5, 0x6a,
			0x18, 0xd5, 0xb0, 0x04, 0xbc, 0x72, 0x43, 0xd2,
			0xe2, 0x5b, 0xb6, 0x15, 0x6d, 0x09, 0x8a, 0x6d,
			0x76, 0x62, 0x95, 0x0b, 0x5d, 0x41, 0x22, 0xd9,
			0xf7, 0x65, 0xa9, 0x8e, 0xc1, 0xa5, 0xbc, 0x1c,
			0xeb, 0xa5, 0xb2, 0x3d, 0x9d, 0xd7, 0x01, 0x4b,
			0xd2, 0x61, 0xca, 0x29, 0x79, 0x14, 0xd2, 0xba,
			0xda, 0x8d, 0x21, 0x58, 0x70, 0xa3, 0x25, 0x00,
			0x56, 0xee, 0x0a, 0x92, 0xe6, 0x8e, 0xb8, 0xd9,
			0x56, 0x35, 0x9f, 0xd5, 0xb5, 0x8c, 0x36, 0xfe,
			0xc5, 0xf9, 0xcb, 0x0f, 0x6d, 0xf0, 0x56, 0x54,
			0x90, 0xc3, 0x2a, 0x7e, 0xc5, 0x77, 0x52, 0x71,
			0xd3, 0xb0, 0xbb, 0x07, 0x11, 0x31, 0xc7, 0xa0,
			0x20, 0x0f, 0x6b, 0xbe, 0x0f, 0x5f, 0x57, 0x74,
			0xb6, 0xd4, 0x2e, 0x69, 0x8d, 0xe2, 0x74, 0xee,
			0x23, 0x19, 0xf2, 0xc4, 0x5c, 0x8a, 0x1e, 0x91,
			0xcd, 0x49, 0x94, 0x76, 0x2f, 0xdd, 0x17, 0x7e,
			0xf3, 0x04, 0x20, 0xb8, 0x5f, 0xa5, 0x0a, 0x91,
			0xba, 0xe0, 0x34, 0x6f, 0xe0, 0xe6, 0x91, 0x19,
			0xf1, 0x42, 0x57, 0x3a, 0xaf, 0x6b, 0xdf, 0xfd,
			0x23, 0xb7, 0x15, 0x5f, 0x29, 0x41, 0x2f, 0x41,
			0x29, 0x6b, 0x18, 0xdd, 0x1d, 0xb0, 0x91, 0xbc,
			0x66, 0x2b, 0x12, 0x53, 0x77, 0xa1, 0xc7, 0x47,
			0xeb, 0xa4, 0xd2, 0xa6, 0x6d, 0x0d, 0x91, 0xcb,
			0x9f, 0x27, 0x20, 0x3c, 0xa4, 0x99, 0x37, 0x13,
			0x46, 0xc1, 0x17, 0x69, 0xbe, 0x83, 0x11, 0x9b,
			0xd4, 0xbf, 0xf8, 0x6a, 0xdb, 0xc7, 0xa7, 0x35,
			0x5a, 0x22, 0x92, 0x95, 0x3f, 0xf2, 0x41, 0x74,
			0xe0, 0x28, 0x55, 0x12, 0x07, 0x50, 0x7d, 0xf2,
			0x24, 0xe6, 0xe4, 0xf7, 0xaf, 0xec, 0x43, 0xba,
			0x30, 0x17, 0x82, 0xf8, 0x81, 0xee, 0x1a, 0xa3,
			0xc3, 0xe0, 0xf4, 0x86, 0x57, 0x6e, 0xcd, 0xe5,
			0x24, 0xe0, 0xe7, 0xe5, 0x5f, 0x43, 0xdb, 0xe7,
			0x1a, 0x7e, 0xb8, 0x51, 0xfc, 0x4d, 0xe8, 0x7a,
			0x4d, 0x85, 0xf4, 0xb6, 0xf9, 0x0f, 0x60, 0x8d,
			0x6f, 0x30, 0xdb, 0xd0, 0x28, 0x5c, 0x9b, 0x1c,
			0x38, 0xed, 0xc9, 0x1b, 0x6e, 0x09, 0xe5, 0xeb,
			0x09, 0x55, 0x8b, 0x12, 0xcf, 0xd0, 0xda, 0xae,
			0x9b, 0x16, 0xac, 0x0d, 0x06, 0xc8, 0xe5, 0x82,
			0x87, 0x21, 0xdc, 0x96, 0x28, 0xef, 0x5b, 0x7d,
			0x1c, 0x01, 0xc4, 0x09, 0x5f, 0x50, 0x58, 0xfc,
			0x1d, 0x72, 0xbc, 0xe9, 0xa0, 0xa7, 0xd9, 0xd9,
			0x5f, 0x7d, 0xfc, 0x51, 0x84, 0xcf, 0x20, 0xdc,
			0xb4, 0x92, 0x94, 0xf4, 0xd6, 0xb9, 0xe2, 0xd6,
			0xa5, 0xc2, 0x50, 0x6f, 0x21, 0x2b, 0x8b, 0x48,
			0x74, 0xea, 0xce, 0x95, 0x07, 0x18, 0x62, 0xe2,
			0xa2, 0x50, 0xd5, 0x0f, 0x10, 0x76, 0x7d, 0x8a,
			0x81, 0xa0, 0xa2, 0xeb, 0xa1, 0x6a, 0xeb, 0x0a,
			0x59, 0x8c, 0x83, 0x3f, 0xc8, 0xbf, 0x68, 0x65,
			0x94, 0x54, 0xaf, 0xa1, 0x5f, 0xbb, 0x99, 0x80,
			0xc8, 0x4a, 0x76, 0x59, 0xad, 0x1d, 0x64, 0x42,
			0x52, 0xed, 0x66, 0x5d, 0x67, 0xf8, 0xc9, 0x22,
			0xbf, 0x51, 0xbb, 0x4d, 0x75, 0x8f, 0x30, 0x73,
			0xba, 0xc1, 0xb7, 0xe2, 0x95, 0xe8, 0x70, 0xda,
			0x55, 0xb4, 0x6f, 0xd9, 0xf0, 0xbe, 0x66, 0x4a,
			0x15, 0x8e, 0x0e, 0x18, 0xfd, 0xf5, 0x6d, 0xe9,
			0x67, 0x45, 0xaf, 0x97, 0x7d, 0xdb, 0x33, 0x39,
			0x9e, 0x8f, 0xa4, 0x9e, 0x75, 0x3b, 0x5e, 0xcb,
			0x1a, 0x65, 0x5a, 0xe4, 0x2e, 0xfb, 0x43, 0x4a,
			0x80, 0x75, 0xf7, 0xd2, 0xdb, 0x7d, 0x53, 0x73,
			0x51, 0x72, 0x8a, 0x1f, 0x56, 0x2a, 0x39, 0xca,
			0x9c, 0xb3, 0x00, 0xf8, 0x18, 0xb4, 0xd5, 0x06,
			0x27, 0x27, 0x8b, 0x60, 0x2a, 0x63, 0x76, 0xf0,
			0x5c, 0x0c, 0x67, 0x70, 0x59, 0x98, 0x4c, 0x79,
			0x54, 0xea, 0xda, 0xda, 0x79, 0x66, 0xb5, 0x1e,
			0xec, 0xc3, 0xf2, 0x9b, 0x10, 0xa5, 0xc7, 0xe8,
			0xe4, 0xa0, 0x1b, 0xde, 0x8b, 0xf2, 0x13, 0x20,
			0x68, 0x94, 0x7d, 0xfc, 0x01, 0xed, 0x05, 0x0b,
			0x56, 0x40, 0xce, 0xd2, 0x56, 0xe3, 0x0f, 0x9a,
			0x5a, 0x8f, 0xcd, 0xdd, 0xeb, 0xa8, 0x4d, 0x87,
			0xd6, 0x9c, 0xcb, 0x5e, 0xd2, 0x80, 0x86, 0x8f,
			0xd0, 0x0a, 0xaf, 0xa1, 0x0c, 0x3b, 0xaa, 0x4a,
			0x31, 0x72, 0x83, 0x63, 0x8d, 0x73, 0x33, 0xbf,
			0x77, 0xdd, 0x13, 0xc6, 0x2d, 0xe6, 0xc8, 0x65,
			0x24, 0x08, 0x28, 0x87, 0x7f, 0x21, 0xa9, 0x04,
			0x4d, 0xc6, 0xa7, 0x34, 0x86, 0xe0, 0x9a, 0x7b,
			0x63, 0xd7, 0x3f, 0x9e, 0xbd, 0xbc, 0x7a, 0x94,
			0x39, 0xb4, 0x7b, 0x8a, 0x1f, 0x12, 0x58, 0x8a,
			0xe9, 0x7f, 0x89, 0x3a, 0xbe, 0x37, 0xa8, 0x19,
			0x5e, 0x4c, 0x14, 0x98, 0xf0, 0x4a, 0xb6, 0xdc,
			0x6e, 0xba, 0x5f, 0x70, 0x7a, 0xd3, 0xdd, 0xec,
			0x53, 0xa0, 0x46, 0x8a, 0x6c, 0x51, 0xe3, 0x80,
			0x84, 0x30, 0xd8, 0x50, 0x8f, 0xb0, 0x42, 0x85,
			0xe0, 0xb2, 0x81, 0x2e, 0x2a, 0x78, 0x60, 0xbe,
			0xe7, 0x62, 0x39, 0xf7, 0x62, 0x88, 0x27, 0xfc,
			0x9e, 0xbb, 0x1a, 0x53, 0x3a, 0x72, 0x45, 0xc4,
			0x5f, 0xa7, 0xc9, 0xda, 0xbd, 0xde, 0xb7, 0xaf,
			0x79, 0x88, 0x11, 0x4e, 0x3c, 0x81, 0xfd, 0x20,
			0xd8, 0xb1, 0x9d, 0x5c, 0x7b, 0x2b, 0x92, 0x15,
			0xff, 0xf2, 0x06, 0xa8, 0x30, 0xe9, 0x50, 0x9d,
			0x0d, 0x40, 0xb4, 0xb1, 0xef, 0x0b, 0x7e, 0x05,
			0x6e, 0xdb, 0xc9, 0x08, 0xda, 0xee, 0x69, 0x80,
			0x70, 0x10, 0xb2, 0x14, 0xb3, 0xab, 0x71, 0x50,
			0x11, 0xe5, 0x9c, 0x47, 0xc6, 0xfe, 0x01, 0xae,
			0x93, 0x12, 0x4d, 0x2d, 0xcb, 0x9a, 0x97, 0xe7,
			0xc3, 0xef, 0x12, 0x45, 0x89, 0x97, 0xf9, 0x7c,
			0x6c, 0x37, 0x48, 0x93, 0x3d, 0x47, 0xf2, 0x50,
			0x95, 0xfd, 0xf0, 0x40, 0x00, 0xfb, 0x80, 0x68,
			0x4b, 0xb1, 0xfb, 0x4c, 0x97, 0xde, 0x5d, 0x55,
			0x69, 0x41, 0x84, 0xfc, 0x59, 0x9c, 0x32, 0xc7,
			0x64, 0x84, 0xbd, 0x09, 0x29, 0x30, 0xf3, 0x36,
			0x08, 0xc9, 0x93, 0x33, 0xa6, 0x35, 0xb5, 0xfe,
			0xae, 0x4f, 0x7d, 0x02, 0x3f, 0x37, 0x8b, 0xe9,
			0xe8, 0x81, 0x64, 0x7b, 0xc5, 0x5c, 0xd8, 0xec,
			0x6d, 0x47, 0x46, 0x99, 0x37, 0x8d, 0x5b, 0x32,
			0xe7, 0x05, 0x48, 0xfe, 0x75, 0x9d, 0x29, 0x3b,
			0x9a, 0x33, 0xda, 0x98, 0xa4, 0x44, 0xf7, 0x1d,
			0xb0, 0x3e, 0x12, 0x4e, 0x4c, 0x0d, 0x09, 0x0c,
			0x09, 0x2d, 0x2b, 0x57, 0xb6, 0x0c, 0xe7, 0x27,
			0x0a, 0xcc, 0x3f, 0x1d, 0xa1, 0x86, 0x2b, 0x31,
			0x72, 0x23, 0xc4, 0x55, 0x87, 0x5b, 0x45, 0x5b,
			0x41, 0x51, 0xef, 0xae, 0xe9, 0x5a, 0xde, 0x52,
			0xf7, 0xbc, 0x51, 0xaf, 0x3f, 0xe2, 0x24, 0x04,
			0x49, 0xf0, 0xbe, 0x82, 0x04, 0x1c, 0x68, 0x3b,
			0xa8, 0x1a, 0x3e, 0x91, 0xaa, 0x82, 0xcd, 0xed,
			0x21, 0x6c, 0x7e, 0x8f, 0x18, 0x28, 0x83, 0x76,
			0xdf, 0x16, 0xcf, 0xac, 0x0d, 0xc6, 0xc0, 0x68,
			0xfa, 0x81, 0xad, 0x0d, 0x76, 0x1e, 0x32, 0xe2,
			0x1a, 0x24, 0x0a, 0x98, 0xb7, 0x77, 0x11, 0x36,
			0x1a, 0xb2, 0x9a, 0xcf, 0x8e, 0x7e, 0x65, 0xfb,
			0x31, 0x5c, 0xd6, 0x29, 0xbf, 0xf2, 0x04, 0x20,
			0x96, 0xac, 0x02, 0xb3, 0x4f, 0x98, 0xea, 0x49,
			0x87, 0x40, 0xd4, 0xb3, 0x4d, 0xca, 0xbb, 0xce,
			0x18, 0x8e, 0x21, 0x32, 0x46, 0x93, 0x95, 0xe1,
			0x8b, 0xce, 0x9d, 0xcb, 0x2e, 0xdb, 0xc3, 0x61,
			0x62, 0x2d, 0xc9, 0x5d, 0xb8, 0x40, 0x56, 0x04,
			0x04, 0xe9, 0xb2, 0xc4, 0x1d, 0xfa, 0x07, 0xc2,
			0x14, 0xc0, 0x9c, 0xc9, 0x6c, 0x73, 0x06, 0xb7,
			0x40, 0xa3, 0x11, 0x8f, 0xab, 0x18, 0x70, 0x6d,
			0x39, 0xfc, 0x8f, 0x01, 0x87, 0x9b, 0x08, 0xd3,
			0x2c, 0xbc, 0x23, 0x6c, 0xa5, 0x66, 0xfb, 0x6f,
			0x7a, 0x6d, 0x94, 0x39, 0x7a, 0xda, 0x4a, 0xd5,
			0x5a, 0xdd, 0x3e, 0x0a, 0xb6, 0xff, 0xa3, 0xe5,
			0x86, 0xfe, 0xd1, 0x78, 0xe3, 0x30, 0xf4, 0x70,
			0x13, 0x1a, 0x6e, 0x28, 0xda, 0x8a, 0x19, 0xef,
			0x4b, 0x1f, 0x9d, 0xc2, 0xc9, 0x3b, 0x7a, 0x26,
			0xcf, 0x21, 0xe6, 0xd5, 0x81, 0x25, 0x01, 0x74,
			0x35, 0x62, 0xed, 0x72, 0x5c, 0x25, 0xdc, 0xb6,
			0x31, 0x18, 0x96, 0x52, 0x7f, 0x9b, 0xa0, 0x45,
			0xb0, 0xd0, 0x6b, 0x4a, 0xec, 0x76, 0x74, 0x54,
			0x27, 0x3c, 0x9b, 0xde, 0xed, 0x60, 0x76, 0x1b,
			0x80, 0xec, 0x2b, 0xc1, 0xc6, 0xa7, 0x80, 0xd5,
			0xd6, 0x2e, 0xbf, 0xcf, 0x5d, 0x78, 0x10, 0x4e,
			0xaa, 0x04, 0x7e, 0x5d, 0xb5, 0xcc, 0x7f, 0xc5,
			0x65, 0x8d, 0x28, 0x16, 0xfb, 0xc2, 0x27, 0x8b,
			0x14, 0xba, 0x9c, 0x16, 0x71, 0x4c, 0xf2, 0x2e,
			0x13, 0xee, 0x2d, 0x65, 0xa6, 0xdb, 0xdb, 0xf1,
			0xc1, 0xc9, 0x66, 0x41, 0x4f, 0x8a, 0x08, 0xc1,
			0xda, 0x0e, 0xb7, 0x96, 0xc0, 0x1c, 0xb9, 0x83,
			0xb4, 0x58, 0x33, 0x80, 0x00, 0x51, 0x57, 0xdd,
			0xab, 0xed, 0x64, 0x42, 0x57, 0x8f, 0xe7, 0xae,
			0x37, 0xa8, 0xc0, 0x2f, 0x8e, 0xbf, 0x3a, 0x2f,
			0xda, 0x3a, 0xee, 0x14, 0x47, 0xff, 0x1a, 0x40,
			0xd8, 0x0d, 0x3f, 0x49, 0xf0, 0x35, 0x1a, 0x4a,
			0x4e, 0x44, 0x43, 0x35, 0x69, 0xda, 0xb0, 0x5f,
			0xe2, 0x43, 0xa8, 0x37, 0x4f, 0xab, 0xb1, 0x5a,
			0x08, 0x6d, 0x6a, 0x46, 0xfb, 0x97, 0x1c, 0x24,
			0x4d, 0x32, 0x2d, 0x3d, 0x7a, 0x34, 0x71, 0x3a,
			0xa7, 0xc2, 0x93, 0xa7, 0xce, 0x2a, 0x96, 0x55,
			0x90, 0x6e, 0xa2, 0x8e, 0x77, 0x1a, 0x50, 0x07,
			0xf3, 0x82, 0xd5, 0xf2, 0x52, 0x82, 0xec, 0xa9,
			0x37, 0x8a, 0xf8, 0x11, 0xc3, 0x5b, 0x95, 0xc5,
			0xa3, 0x8d, 0x2e, 0x45, 0xdf, 0x8c, 0x21, 0x9e,
			0xd7, 0xe6, 0xdd, 0x93, 0x96, 0xc5, 0x7c, 0x8b,
			0xa7, 0x1b, 0x2f, 0x41, 0x19, 0xd4, 0x50, 0x3e,
			0x2f, 0xee, 0xe3, 0x96, 0xcd, 0x6b, 0xd5, 0x02,
			0xea, 0x4a, 0xea, 0xf8, 0xcb, 0xd9, 0x8e, 0x87,
			0x88, 0x7d, 0xaf, 0xc2, 0x2e, 0xe3, 0xd8, 0x46,
			0xef, 0x26, 0xf2, 0xa6, 0xc7, 0x15, 0xf8, 0x59,
			0x50, 0xae, 0xba, 0xce, 0xc2, 0x01, 0xee, 0x4b,
			0xbd, 0x1b, 0xef, 0x2b, 0xf0, 0x62, 0xa7, 0x6c,
			0x2c, 0x17, 0xbf, 0x1f, 0x04, 0x6e, 0xd4, 0x6d,
			0x72, 0x0c, 0x01, 0xee, 0xd5, 0x71, 0x1a, 0xf7,
			0xf2, 0x1a, 0xdc, 0xd3, 0x78, 0xf2, 0x15, 0x46,
			0x78, 0xc1, 0x43, 0x42, 0x62, 0x59, 0x13, 0x7e,
			0x47, 0xd5, 0x6b, 0x62, 0xc2, 0x9f, 0x9b, 0x8b,
			0xc0, 0x19, 0xa1, 0x68, 0x22, 0xeb, 0xe3, 0x29,
			0x77, 0x0a, 0xfc, 0x1f, 0xda, 0x8b, 0x73, 0xbc,
			0xb0, 0xca, 0xcb, 0xff, 0x50, 0x98, 0x11, 0x21,
			0xe8, 0xbe, 0x55, 0x93, 0x83, 0x8c, 0x29, 0x60,
			0xf1, 0x07, 0x81, 0x2e, 0x4d, 0x33, 0x92, 0xdf,
			0xc2, 0x55, 0xf1, 0x6b, 0x88, 0x54, 0xd3, 0x95,
			0x1a, 0x2f, 0x8d, 0x26, 0x3f, 0xa8, 0x5a, 0xde,
			0xca, 0xc3, 0xbd, 0x75, 0x73, 0xe1, 0x6f, 0xf2,
			0x9c, 0x48, 0xd0, 0x47, 0xd0, 0xc7, 0xad, 0xfa,
			0xd0, 0x1c, 0xa4, 0xbe, 0x8a, 0x6d, 0x92, 0x4a,
			0xe9, 0x14, 0x82, 0xb2, 0xb7, 0x16, 0x95, 0x69,
			0x1f, 0xb7, 0x50, 0x01, 0x98, 0x44, 0x04, 0xfe,
			0x64, 0x59, 0x8b, 0xf0, 0x2d, 0x04, 0x62, 0xf8,
			0x7f, 0xec, 0xfe, 0xc7, 0x6a, 0x4b, 0xc7, 0x1f,
			0xa7, 0x1e, 0x01, 0x89, 0xec, 0x7a, 0x49, 0x33,
			0xb3, 0x81, 0x13, 0x6b, 0x12, 0x36, 0xb1, 0xb9,
			0x3b, 0xa5, 0x72, 0xc2, 0x67, 0xe6, 0x3b, 0xf3,
			0x56, 0x6e, 0x53, 0x55, 0x42, 0xe7, 0xec, 0x3a,
			0x7b, 0xf2, 0x16, 0x6a, 0xa6, 0x4b, 0x72, 0xb1,
			0x21, 0xe9, 0xa9, 0x8f, 0x88, 0x6d, 0xc1, 0x09,
			0xbc, 0x2e, 0x3d, 0xd1, 0xb8, 0x6e, 0x67, 0x1b,
			0x8a, 0x56, 0x11, 0x64, 0x7c, 0x2c, 0xa9, 0x08,
			0xef, 0x4a, 0xcc, 0x9e, 0xf4, 0xd5, 0x82, 0xa7,
			0x44, 0x07, 0x3f, 0x4e, 0xa8, 0x65, 0x30, 0x57,
			0x92, 0xc8, 0x3d, 0x70, 0xda, 0xe4, 0xb0, 0x1b,
			0x92, 0x5d, 0xb2, 0xa4, 0x65, 0xa4, 0x57, 0x3a,
			0x4a, 0x79, 0x4d, 0xcd, 0x63, 0x1b, 0xc2, 0x37,
			0xf1, 0xe7, 0x6a, 0x01, 0x6a, 0x54, 0xd6, 0xf5,
			0x38, 0x59, 0xc9, 0x75, 0xb4, 0x63, 0x1a, 0xb1,
			0xb0, 0x5f, 0x04, 0xfe, 0x5a, 0x0c, 0x5f, 0xa1,
			0x6e, 0x04, 0x6d, 0x00, 0xaf, 0x95, 0x9c, 0x4d,
			0xe4, 0x11, 0xe0, 0x04, 0xcb, 0x2d, 0xd2, 0x3e,
			0xf0, 0x90, 0xbb, 0xfd, 0x93, 0x0b, 0xb8, 0xc3,
			0x52, 0x3d, 0x15, 0xff, 0xd7, 0x5f, 0x38, 0x7e,
			0xd6, 0xb6, 0x57, 0xad, 0x1a, 0xb3, 0xd6, 0xb5,
			0x22, 0xe4, 0xc6, 0xd4, 0x9d, 0xf2, 0x46, 0x14,
			0x42, 0x79, 0x4e, 0x4f, 0xf3, 0x5b, 0x62, 0x83,
			0x78, 0x0c, 0x5d, 0x62, 0xa8, 0x55, 0xf0, 0x6f,
			0xf9, 0x3c, 0x54, 0xad, 0x85, 0xea, 0xa4, 0x36,
			0xd2, 0x35, 0x3b, 0x38, 0xc0, 0x5e, 0x32, 0xf0,
			0x5b, 0x1d, 0xf2, 0x0e, 0xc0, 0xd4, 0x80, 0xb5,
			0x1e, 0x76, 0x3f, 0xbf, 0x31, 0x48, 0x3e, 0xe9,
			0xc8, 0xb6, 0xeb, 0xd3, 0x6f, 0xef, 0x10, 0xb8,
			0x28, 0x42, 0x4f, 0xad, 0x68, 0x87, 0xf3, 0x93,
			0x53, 0x3e, 0x17, 0x60, 0x3b, 0x9f, 0x22, 0xe7,
			0xc4, 0xa4, 0xd5, 0xfd, 0x61, 0xcc, 0x7e, 0x26,
			0xf6, 0x9f, 0xcb, 0xdc, 0x25, 0xee, 0x4d, 0x96,
			0xbb, 0xb5, 0x5f, 0xfa, 0xd6, 0x6c, 0x47, 0xcf,
			0x2b, 0x8a, 0x42, 0x7c, 0x59, 0x13, 0xe5, 0x7a,
			0xea, 0x5f, 0x8f, 0x0f, 0xd2, 0xc1, 0x21, 0xdb,
			0xa3, 0xf4, 0x55, 0x13, 0x63, 0xea, 0x48, 0x9a,
			0x11, 0x3d, 0x7c, 0x5a, 0x92, 0xcf, 0x51, 0x1d,
			0xf0, 0x38, 0x0a, 0x22, 0x06, 0x8f, 0x9e, 0x15,
			0x9c, 0xc1, 0x93, 0x98, 0x1c, 0xcd, 0x21, 0xa3,
			0xa8, 0xd5, 0x52, 0xf3, 0xc8, 0x21, 0x09, 0x42,
			0x6a, 0x42, 0x15, 0xbc, 0xcc, 0xa8, 0x69, 0x94,
			0x27, 0x3e, 0x68, 0x97, 0x42, 0x05, 0x36, 0x0e,
			0x8c, 0xd6, 0xca, 0x90, 0x3b, 0xdc, 0xe4, 0x76,
			0x21, 0x42, 0x9e, 0xe9, 0x18, 0x75, 0x22, 0x93,
			0xf7, 0x91, 0x65, 0x2e, 0xff, 0x65, 0xf3, 0x6d,
			0xba, 0x49, 0x2c, 0xb3, 0xf3, 0x0d, 0xe1, 0x23,
			0xc7, 0x4a, 0xa3, 0x21, 0xeb, 0x6e, 0xa8, 0xd0,
			0x27, 0xa0, 0x22, 0x03, 0x44, 0xaf, 0xa5, 0x05,
			0xae, 0x6a, 0x20, 0x94, 0xba, 0x93, 0x4e, 0x5c,
			0xb9, 0xab, 0x01, 0xf5, 0x34, 0xb4, 0x8a, 0xe3,
			0x4f, 0x9c, 0xd5, 0x7b, 0x2f, 0xa1, 0x62, 0xf2,
			0x37, 0x4d, 0xd8, 0xe7, 0x40, 0xd3, 0x95, 0x78,
			0x67, 0xc4, 0x1b, 0xe1, 0xbe, 0x7e, 0x58, 0x82,
			0x6e, 0xe0, 0x89, 0x91, 0xbf, 0x9a, 0x09, 0xee,
			0xa7, 0xd3, 0x59, 0xd8, 0xbe, 0xe9, 0xff, 0x1a,
			0x3b, 0x36, 0x79, 0x39, 0xf3, 0x62, 0xd3, 0x89,
			0xf8, 0x71, 0x1b, 0x6b, 0x69, 0x38, 0x7a, 0x4d,
			0xde, 0x7e, 0x54, 0x01, 0x1e, 0xfe, 0xd6, 0x77,
			0xb8, 0x6d, 0x08, 0x20, 0x4c, 0x84, 0xda, 0xf1,
			0x7f, 0x45, 0xe9, 0x83, 0x25, 0x38, 0x69, 0xcc,
			0x53, 0x34, 0x5c, 0x7f, 0x16, 0x5e, 0xc1, 0x94,
			0x57, 0xd6, 0xb1, 0xa2, 0x44, 0x04, 0xed, 0x5e,
			0x46, 0xec, 0xd2, 0xe1, 0x58, 0x71, 0x84, 0x11,
			0x8d, 0xd5, 0xc6, 0xff, 0x23, 0x56, 0xb3, 0x86,
			0xe6, 0xff, 0xbb, 0x6e, 0x48, 0x96, 0x37, 0xc3,
			0xee, 0x13, 0xfa, 0x26, 0x8d, 0x10, 0xbb, 0x91,
			0xe6, 0x8b, 0x46, 0x30, 0x1f, 0xb5, 0xd7, 0x3d,
			0x85, 0x26, 0x74, 0x5f, 0x5a, 0x91, 0xe3, 0x70,
			0xe1, 0x6a, 0xff, 0x0b, 0x1f, 0x91, 0xbc, 0x6d,
			0x65, 0x6b, 0x8e, 0x9c, 0x5e, 0x4f, 0x82, 0x81,
			0x39, 0x22, 0xad, 0x25, 0xb5, 0x6a, 0x7c, 0x9a,
			0x3d, 0x6f, 0x78, 0x0a, 0x52, 0x7c, 0x77, 0xc1,
			0x6f, 0x63, 0x08, 0xc4, 0xfd, 0x3a, 0x6a, 0x01,
			0x34, 0xfb, 0xb6, 0xf1, 0x61, 0xe5, 0x21, 0x9b,
			0xc4, 0x07, 0x85, 0x43, 0x80, 0xbe, 0x8a, 0x88,
			0xf6, 0x3e, 0x75, 0x74, 0xe3, 0x16, 0x14, 0x29,
			0x06, 0xf2, 0x90, 0xd1, 0x29, 0xe5, 0xfe, 0xd7,
			0x93, 0x77, 0xca, 0x07, 0xe2, 0x0c, 0x5f, 0x5c,
			0x97, 0x0f, 0xcf, 0xb4, 0x16, 0x54, 0x50, 0xa0,
			0x71, 0xea, 0xa5, 0xce, 0x15, 0x9f, 0xe8, 0x4e,
			0x1f, 0x48, 0x81, 0x46, 0x77, 0x90, 0xde, 0x47,
			0x01, 0xbb, 0x8e, 0x32, 0x63, 0xdb, 0x83, 0x2d,
			0x01, 0x70, 0x7c, 0xd9, 0x5e, 0x0b, 0x7b, 0xd0,
			0xdd, 0x10, 0x33, 0xbd, 0xfc, 0x8d, 0xfd, 0x31,
			0x38, 0x33, 0xb9, 0x5f, 0x60, 0xd1, 0x64, 0x59,
			0x7f, 0xa5, 0xe2, 0xa7, 0xe1, 0x68, 0x09, 0xc6,
			0x24, 0x9e, 0x13, 0xa0, 0x75, 0xb3, 0x4b, 0x5d,
			0xb4, 0xc5, 0x11, 0x59, 0xc4, 0x11, 0x20, 0x03,
			0x39, 0x9f, 0xe7, 0xeb, 0x93, 0x4d, 0x5f, 0xaa,
			0x3a, 0xe9, 0x66, 0xd6, 0x03, 0x34, 0x23, 0x94,
			0x85, 0x0e, 0x1d, 0xda, 0x87, 0x4a, 0xcf, 0xa2,
			0x05, 0x2d, 0x71, 0xce, 0xd7, 0x74, 0x11, 0xf6,
			0xd8, 0xf4, 0xe4, 0x07, 0x6d, 0x02, 0x81, 0xb7,
			0x7c, 0x99, 0x95, 0xaa, 0xce, 0x40, 0x90, 0x29,
			0xa4, 0xf7, 0x7b, 0x02, 0x93, 0xda, 0x1a, 0x9f,
			0x2a, 0xa4, 0x25, 0xde, 0x36, 0xf2, 0x2a, 0xbe,
			0xe6, 0xad, 0x11, 0x8b, 0x8d, 0xb0, 0x07, 0x5e,
			0xdc, 0x93, 0x8c, 0xc4, 0x3f, 0xba, 0xb2, 0x7c,
			0xee, 0x42, 0xdc, 0x15, 0xae, 0xb5, 0xbb, 0x92,
			0x69, 0xd0, 0x9b, 0xe8, 0x03, 0xb2, 0xf7, 0xf8,
			0x8c, 0xa5, 0x5f, 0xf2, 0x04, 0xc0, 0x0b, 0xc0,
			0x10, 0x5a, 0x34, 0xbc, 0x3d, 0xad, 0x0c, 0x18,
			0x2c, 0x8c, 0x46, 0x0e, 0xa6, 0x70, 0x7f, 0x5b,
			0x65, 0xde, 0x9a, 0x51, 0x98, 0xc4, 0xb0, 0x82,
			0x39, 0xbc, 0x9a, 0xf1, 0x8b, 0x00, 0xdf, 0xa9,
			0x06, 0x48, 0x26, 0x2e, 0x84, 0xef, 0xb6, 0xbe,
			0xc2, 0x21, 0x72, 0xb1, 0xb5, 0x25, 0xe9, 0xfc,
			0x07, 0xbe, 0x28, 0x84, 0x37, 0xec, 0x45, 0xfb,
			0xa8, 0xae, 0x37, 0xa2, 0xfd, 0x44, 0xe3, 0x98,
			0x8e, 0xdd, 0xfa, 0xa3, 0x02, 0xdf, 0x02, 0x05,
			0x59, 0xf7, 0x7f, 0x78, 0x3f, 0xcd, 0xda, 0xb9,
			0x25, 0x3d, 0xcb, 0x51, 0xfc, 0xfc, 0xfd, 0xca,
			0x32, 0xe8, 0x7e, 0xbb, 0x56, 0x29, 0x8b, 0x1d,
			0x9f, 0x94, 0x20, 0x82, 0x57, 0xe8, 0x31, 0xff,
			0x97, 0x45, 0xa5, 0xef, 0x63, 0x4e, 0x4f, 0xdc,
			0x38, 0x41, 0x65, 0x1e, 0x72, 0xdb, 0x4f, 0xf8,
			0x95, 0x4e, 0x3e, 0x6f, 0x0b, 0x1b, 0x5c, 0xbb,
			0x2f, 0xc9, 0x95, 0x6c, 0x41, 0x4d, 0x8c, 0xa4,
			0x4f, 0xe3, 0xd1, 0x12, 0x35, 0x5e, 0xfd, 0x89,
		},
	}
};

#ifdef __cplusplus
}
#endif

#endif /* SPHINCS_TESTER_VECTORS_SHAKE_128S_H */
