/*
 * Copyright (C) 2022 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "compare.h"
#include "ext_headers_internal.h"
#include "lc_ascon_lightweight.h"
#include "test_helper_common.h"
#include "visibility.h"

static int ascon_tester_one(const uint8_t *pt, size_t ptlen,
			    const uint8_t *nonce, size_t noncelen,
			    const uint8_t *aad, size_t aadlen,
			    const uint8_t *key, size_t keylen,
			    const uint8_t *exp_ct, const uint8_t *exp_tag,
			    size_t exp_tag_len)
{
	struct lc_aead_ctx *al_heap = NULL;
	ssize_t ret;
	int ret_checked = 0;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wvla"
	uint8_t out_enc[ptlen];
	uint8_t out_dec[ptlen];
	uint8_t tag[exp_tag_len];
#pragma GCC diagnostic pop
	LC_AL_CTX_ON_STACK(al);

	/* One shot encryption with pt ptr != ct ptr */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return 1;
	lc_aead_encrypt(al, pt, out_enc, ptlen, aad, aadlen, tag, exp_tag_len);
	lc_aead_zero(al);

	ret_checked +=
		lc_compare(out_enc, exp_ct, ptlen,
			   "Ascon lightweight crypt: Encryption, ciphertext");
	ret_checked += lc_compare(tag, exp_tag, exp_tag_len,
				  "Ascon lightweight crypt: Encryption, tag");

	//bin2print(out_enc, ptlen, stderr, "out_enc");
	//bin2print(tag, exp_tag_len, stderr, "tag");

	/* One shot encryption with pt ptr == ct ptr */
	if (lc_al_alloc(&al_heap))
		return 1;

	if (lc_aead_setkey(al_heap, key, keylen, nonce, noncelen))
		return 1;

	memcpy(out_enc, pt, ptlen);
	lc_aead_encrypt(al_heap, out_enc, out_enc, ptlen, aad, aadlen, tag,
			exp_tag_len);
	lc_aead_zero_free(al_heap);

	ret_checked += lc_compare(out_enc, exp_ct, ptlen,
				  "Ascon lightweight: Encryption, ciphertext");
	ret_checked += lc_compare(tag, exp_tag, exp_tag_len,
				  "Ascon lightweight: Encryption, tag");

	/* One shot decryption with pt ptr != ct ptr */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return -EFAULT;
	ret = lc_aead_decrypt(al, out_enc, out_dec, ptlen, aad, aadlen, tag,
			      exp_tag_len);
	//bin2print(out_dec, ptlen, stderr, "out_enc");
	lc_aead_zero(al);
	if (ret < 0)
		ret_checked += 1;

	ret_checked +=
		lc_compare(out_dec, pt, ptlen,
			   "Ascon lightweight crypt: Decryption, plaintext");

	/* Check authentication error */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return -EFAULT;

	out_enc[0] = (uint8_t)((out_enc[0] + 1) & 0xff);
	ret = lc_aead_decrypt(al, out_enc, out_dec, ptlen, aad, aadlen, tag,
			      exp_tag_len);
	lc_aead_zero(al);
	if (ret != -EBADMSG)
		ret_checked += 1;

	return ret_checked;
}

static int ascon_tester_128(void)
{
	/*
	 * Vector 1089 from genkat_crypto_aead_asconaead128_ref generated by
	 * code https://github.com/ascon/ascon-c
	 */
	static const uint8_t pt[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				      0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
				      0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
				      0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
				      0x1C, 0x1D, 0x1E, 0x1F };
	static const uint8_t aad[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				       0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
				       0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
				       0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
				       0x1C, 0x1D, 0x1E, 0x1F };
	static const uint8_t key[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
				       0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
				       0x0C, 0x0D, 0x0E, 0x0F };
	static const uint8_t nonce[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
					 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
					 0x0C, 0x0D, 0x0E, 0x0F };
	static const uint8_t exp_ct[] = { 0x4C, 0x08, 0x6D, 0x27, 0xA3, 0xB5,
					  0x1A, 0x23, 0x33, 0xCF, 0xC7, 0xF2,
					  0x21, 0x72, 0xA9, 0xBC, 0xAD, 0x88,
					  0xB8, 0xD4, 0xD7, 0x7E, 0x50, 0x62,
					  0x2D, 0x78, 0x83, 0x45, 0xFA, 0x7B,
					  0xEE, 0x44 };
	static const uint8_t exp_tag[] = { 0x68, 0x91, 0x5D, 0x3F, 0x94, 0x22,
					   0x28, 0x9F, 0x23, 0x49, 0xD6, 0xA3,
					   0xB4, 0x16, 0x03, 0x97 };
	int ret;

	printf("Ascon lightweight 128 crypt ctx len %u, state len %u\n",
	       (unsigned int)LC_AL_CTX_SIZE, (unsigned int)LC_AL_STATE_SIZE);

	ret = ascon_tester_one(pt, sizeof(pt), nonce, sizeof(nonce), aad,
			       sizeof(aad), key, sizeof(key), exp_ct, exp_tag,
			       sizeof(exp_tag));

	return ret;
}

static int ascon_tester_128_non_aligned(void)
{
	static const uint8_t pt[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				      0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
				      0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
				      0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
				      0x1C, 0x1D, 0x1E };
	static const uint8_t key[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
				       0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
				       0x0C, 0x0D, 0x0E, 0x0F };
	static const uint8_t exp_ct[] = { 0x14, 0x1B, 0x8B, 0x25, 0xE5, 0x9E,
					  0x0D, 0x01, 0xB1, 0x14, 0x22, 0xD9,
					  0x40, 0x48, 0xC8, 0x04, 0x5E, 0xE7,
					  0x74, 0x4D, 0xD0, 0x04, 0x0B, 0xF0,
					  0x8B, 0x6C, 0xED, 0x0E, 0x4F, 0x25,
					  0x5B };
	static const uint8_t exp_tag[] = { 0x19, 0x6B, 0x88, 0x31, 0x37, 0xE0,
					   0xD1, 0xCA, 0xF3, 0x3A, 0xFE, 0xBB,
					   0x4B, 0x9E, 0x72, 0xC4 };

	return ascon_tester_one(pt, sizeof(pt), key, sizeof(key), pt,
				sizeof(pt), key, sizeof(key), exp_ct, exp_tag,
				sizeof(exp_tag));
}

LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	int ret = 0;
	(void)argc;
	(void)argv;

	ret += ascon_tester_128();
	ret += ascon_tester_128_non_aligned();

	ret = test_validate_status(ret, LC_ALG_STATUS_ASCON_AEAD_128, 1);
	ret += test_print_status();

	return ret;
}
