import os
import platform
import tempfile
import unittest
from configparser import RawConfigParser

from keylime import config
from keylime.cmd import convert_config
from keylime.common.algorithms import Hash
from keylime.mba import mba

try:
    from keylime.mba.elparsing import tpm_bootlog_enrich
except Exception:
    unittest.skip(f"tpm_bootlog_enrich not available, architecture ({platform.machine()}) not supported")

TEMPLATES_DIR = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "templates"))


@unittest.skipIf(platform.machine() in ["ppc64le", "s390x"], "ppc64le and s390x are not supported")
class TestMBAParsing(unittest.TestCase):
    def test_parse_bootlog(self):
        """Test parsing binary measured boot event log"""
        # This test requires the verifier configuration file, so let's create
        # one with the default values to use, so that we do not depend on the
        # configuration files existing in the test system.
        with tempfile.TemporaryDirectory() as config_dir:
            # Let's write the config file for the verifier.
            verifier_config = convert_config.process_versions(["verifier"], TEMPLATES_DIR, RawConfigParser(), True)
            convert_config.output(["verifier"], verifier_config, TEMPLATES_DIR, config_dir)

            # As we want to use a config file from a different location, the
            # proper way would be to define an environment variable for the
            # module of interest, e.g. in our case it would be the
            # KEYLIME_VERIFIER_CONFIG variable. However, the config module
            # reads such env vars at first load, and there is no clean way
            # to have it re-read them, so for this test we will override it
            # manually.
            config.CONFIG_ENV["verifier"] = os.path.abspath(os.path.join(config_dir, "verifier.conf"))

            mba.load_imports()
            # Use the file that triggered https://github.com/keylime/keylime/issues/1153
            mb_log_path = os.path.abspath(os.path.join(os.path.dirname(__file__), "data/mb_log.b64"))
            with open(mb_log_path, encoding="utf-8") as f:
                # Read the base64 input and remove the newlines
                b64 = "".join(f.read().splitlines())
                pcr_hashes, boot_aggregates, measurement_data, failure = mba.bootlog_parse(b64, Hash.SHA256)

                self.assertFalse(
                    failure,
                    f"Parsing of measured boot log failed with: {list(map(lambda x: x.context, failure.events))}",
                )
                self.assertTrue(isinstance(pcr_hashes, dict))
                self.assertTrue(isinstance(boot_aggregates, dict))
                self.assertTrue(isinstance(measurement_data, dict))

    def test_vendor_db_enrichment_actual_hex(self):
        """Test vendor_db enrichment with actual hex string from real vendor_db event.

        Different versions of tpm2_eventlog may provide vendor_db data in different formats.
        This test uses actual hex data and verifies it gets normalized to signature list format.
        """
        # Actual vendor_db hex string from real EV_EFI_VARIABLE_AUTHORITY event.
        actual_vendor_db_hex = "dbed230279908843af772d65b1c35d3b308203943082027ca00302010202090083730d2b7280d15a300d06092a864886f70d01010b0500305f31163014060355040a0c0d526564204861742c20496e632e3121301f06035504030c18526564204861742053656375726520426f6f7420434120353122302006092a864886f70d0109011613736563616c657274407265646861742e636f6d301e170d3230303630393038313533365a170d3338303131383038313533365a305f31163014060355040a0c0d526564204861742c20496e432e3121301f06035504030c18526564204861742053656375726520426f6f7420434120353122302006092a864886f70d0109011613736563616c657274407265646861742e636f6d30820122300d06092a864886f70d01010105000382010f003082010a0282010100cebaea41171c81a18809bfa1d4a9fa532e9d9ebcfc3b289c3052a00bf4000f36c88341f6a9c915496564d5b2769e58c12e1eeacf93386b47d6ba92c5f800e777a55769df41b1c4905b2d20c174aa038680b6a459efa988445e5240d47715a104859ceff3c69ff30f0fd68446e466dc266ad6d88a6e474acae34c431574997a06328ce033bfe5f846673dea0e943bbf3ddd8bf67f308c45540ba4de23355a997305d880e765141a07302c7386b02da3a636a64d815d91a767bbea3b5b828a9ccf83da31d1543416bc1907172a944ef0cecf0dbaf4fbe4d44889238b8cdc8e4513d77aa8d5e5840313520206c2d590763ab5d7b89d7ab0c9d09869fb8e0d01f5850203010001a3533051301d0603551d0e04160414cc6fa5e72868ba494e939bbd680b9144769a9f8f301f0603551d23041830168014cc6fa5e72868ba494e939bbd680b9144769a9f8f300f0603551d130101ff040530030101ff300d06092a864886f70d01010b050003820101001de75e426a66cc723e9b5cc9afa3ca542eed64abc0b917be27a91e58b1593c4d1174d1971a520584058ad9f085c8f5ec8f9ce9e7086dbb3acbfa6f3c33e6784d75bddfc095729f0350d2752a7cb481e08762945cefcf6bda3ae3bf6e18743455500c22518eaa5830bebd3e304db697b5131b6daf6c183b714a09a18917a7e718f56d51b1d310c80ed6e43219024b1ab2d2dc29a326951d0106e452697806d3304444b07577cc54ade46e2222ff5dff93060cf9983a9c39b70c81d0f3f807a7098b6f9c8ae1adfc419850a65f0bbaa57f1cfc838d06592e9e6ebff43ec31a746625948a5dbf21b6139b9f67f87edc421f4c0edd88737d8c95d03f77c190b864f1"

        # Expected parsed format - certificate data without the GUID prefix.
        expected_cert_data = actual_vendor_db_hex[32:]  # Skip first 32 chars (16 bytes GUID).
        expected_parsed_format = [
            {"SignatureOwner": "0223eddb-9079-4388-af77-2d65b1c35d3b", "SignatureData": expected_cert_data}
        ]

        # Test event matching the structure from the debug logs.
        test_event = {
            "VariableName": "d719b2cb-3d3a-4596-a3bc-dad00e67656f",
            "UnicodeNameLength": 9,
            "VariableDataLength": len(actual_vendor_db_hex) // 2,
            "UnicodeName": "vendor_db",
            "VariableData": actual_vendor_db_hex,
        }

        # Apply vendor_db enrichment.
        tpm_bootlog_enrich.enrich_vendor_db_authority_variable(test_event)  # type: ignore[reportPossiblyUnboundVariable]

        # Verify that VariableData gets normalized to signature list format.
        self.assertIsInstance(test_event["VariableData"], list)
        self.assertEqual(len(test_event["VariableData"]), 1)

        signature = test_event["VariableData"][0]
        self.assertIn("SignatureOwner", signature)
        self.assertIn("SignatureData", signature)
        # pylint: disable=invalid-sequence-index
        self.assertEqual(signature["SignatureOwner"], expected_parsed_format[0]["SignatureOwner"])
        self.assertEqual(signature["SignatureData"], expected_parsed_format[0]["SignatureData"])
        # pylint: enable=invalid-sequence-index

    def test_vendor_db_enrichment_multiple_certificates_real_data(self):
        """Test vendor_db enrichment with real data containing multiple certificates.

        This test uses actual hex data from a real secureboot db variable containing
        multiple Microsoft certificates to verify correct parsing of complex vendor_db data.
        """
        # Real hex string from secureboot db variable with multiple certificates.
        # The format of db is similar to vendor_db, so we use it here to test tpm_bootlog_enrich with multiple certificates.
        real_vendor_db_hex = "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"

        # Test event structure matching EV_EFI_VARIABLE_AUTHORITY format.
        test_event = {
            "VariableName": "d719b2cb-3d3a-4596-a3bc-dad00e67656f",
            "UnicodeNameLength": 9,
            "VariableDataLength": len(real_vendor_db_hex) // 2,
            "UnicodeName": "vendor_db",
            "VariableData": real_vendor_db_hex,
        }

        # Apply vendor_db enrichment
        tpm_bootlog_enrich.enrich_vendor_db_authority_variable(test_event)  # type: ignore[reportPossiblyUnboundVariable]

        # Verify enrichment results.
        self.assertIsInstance(test_event["VariableData"], list)
        # Real data should contain multiple certificates (4 in this case based on the structure).
        self.assertGreater(len(test_event["VariableData"]), 1, "Real vendor_db should contain multiple certificates")

        # Verify each certificate has the required structure.
        for i, signature in enumerate(test_event["VariableData"]):
            with self.subTest(certificate=i):
                self.assertIn("SignatureOwner", signature, f"Certificate {i} missing SignatureOwner")
                self.assertIn("SignatureData", signature, f"Certificate {i} missing SignatureData")

                # Verify SignatureOwner is a valid GUID format.
                guid = signature["SignatureOwner"]
                self.assertRegex(
                    guid,
                    r"^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$",
                    f"Certificate {i} has invalid GUID format: {guid}",
                )

                # Verify SignatureData is hex string and not empty.
                cert_data = signature["SignatureData"]
                self.assertIsInstance(cert_data, str, f"Certificate {i} SignatureData should be string")
                self.assertGreater(len(cert_data), 0, f"Certificate {i} SignatureData should not be empty")
                # Verify it's valid hex.
                try:
                    bytes.fromhex(cert_data)
                except ValueError:
                    self.fail(f"Certificate {i} SignatureData is not valid hex: {cert_data[:100]}...")

    def test_vendor_db_enrichment_preserves_signature_lists(self):
        """Test that enrichment preserves VariableData that's already in signature list format"""
        # VariableData that is already in the expected signature list format.
        signature_list_format = [
            {
                "SignatureOwner": "0223eddb-9079-4388-af77-2d65b1c35d3b",
                "SignatureData": "308203943082027ca00302010202090083730d2b7280d15a300d06092a864886f70d01010b0500305f31163014060355040a0c0d526564204861742c20496e632e3121301f06035504030c18526564204861742053656375726520426f6f7420434120353122302006092a864886f70d0109011613736563616c657274407265646861742e636f6d301e170d3230303630393038313533365a170d3338303131383038313533365a305f31163014060355040a0c0d526564204861742c20496e632e3121301f06035504030c18526564204861742053656375726520426f6f7420434120353122302006092a864886f70d0109011613736563616c657274407265646861742e636f6d30820122300d06092a864886f70d01010105000382010f003082010a0282010100cebaea41171c81a18809bfa1d4a9fa532e9d9ebcfc3b289c3052a00bf4000f36c88341f6a9c915496564d5b2769e58c12e1eeacf93386b47d6ba92c5f800e777a55769df41b1c4905b2d20c174aa038680b6a459efa988445e5240d47715a104859ceff3c69ff30f0fd68446e466dc266ad6d88a6e474acae34c431574997a06328ce033bfe5f846673dea0e943bbf3ddd8bf67f308c45540ba4de23355a997305d880e765141a07302c7386b02da3a636a64d815d91a767bbea3b5b828a9ccf83da31d1543416bc1907172a944ef0cecf0dbaf4fbe4d44889238b8cdc8e4513d77aa8d5e5840313520206c2d590763ab5d7b89d7ab0c9d09869fb8e0d01f5850203010001a3533051301d0603551d0e04160414cc6fa5e72868ba494e939bbd680b9144769a9f8f301f0603551d23041830168014cc6fa5e72868ba494e939bbd680b9144769a9f8f300f0603551d130101ff040530030101ff300d06092a864886f70d01010b050003820101001de75e426a66cc723e9b5cc9afa3ca542eed64abc0b917be27a91e58b1593c4d1174d1971a520584058ad9f085c8f5ec8f9ce9e7086dbb3acbfa6f3c33e6784d75bddfc095729f0350d2752a7cb481e08762945cefcf6bda3ae3bf6e18743455500c22518eaa5830bebd3e304db697b5131b6daf6c183b714a09a18917a7e718f56d51b1d310c80ed6e43219024b1ab2d2dc29a326951d0106e452697806d3304444b07577cc54ade46e2222ff5dff93060cf9983a9c39b70c81d0f3f807a7098b6f9c8ae1adfc419850a65f0bbaa57f1cfc838d06592e9e6ebff43ec31a746625948a5dbf21b6139b9f67f87edc421f4c0edd88737d8c95d03f77c190b864f1",
            }
        ]

        # Test event with VariableData already in signature list format.
        test_event = {
            "VariableName": "d719b2cb-3d3a-4596-a3bc-dad00e67656f",
            "UnicodeName": "vendor_db",
            "VariableData": signature_list_format.copy(),
        }

        original_data = test_event["VariableData"].copy()

        # Apply enrichment
        tpm_bootlog_enrich.enrich_vendor_db_authority_variable(test_event)  # type: ignore[reportPossiblyUnboundVariable]

        # Verify that VariableData in signature list format remains unchanged
        self.assertEqual(test_event["VariableData"], original_data)


if __name__ == "__main__":
    unittest.main()
