use std::fmt;
use thiserror::Error;

/// Errors thrown by the serde integration.
#[derive(Debug, Error)]
pub enum Error {
    /// Custom error message.
    #[error("{0}")]
    Custom(String),

    /// Error generated when a sequence has too many items.
    #[error("sequence has too many items, limit is 2^32")]
    TooManyItems,

    /// Error generated by the binary reader or writer.
    #[error(transparent)]
    Binary(#[from] binary_stream::BinaryError),

    /// Error generated converting to slices.
    #[error(transparent)]
    TryFromSlice(#[from] std::array::TryFromSliceError),

    /// Generic error type for user space errors.
    #[error(transparent)]
    Boxed(#[from] Box<dyn std::error::Error + Send + Sync>),
}

impl serde::ser::Error for Error {
    #[cold]
    fn custom<T: fmt::Display>(msg: T) -> Self {
        Self::Custom(msg.to_string())
    }
}

impl serde::de::Error for Error {
    #[cold]
    fn custom<T: fmt::Display>(msg: T) -> Self {
        Self::Custom(msg.to_string())
    }
}
