#!/usr/bin/env bash
set -euo pipefail

: "${PLAYWRIGHT_WS_PORT:=3000}"
: "${PLAYWRIGHT_WS_HOST:=127.0.0.1}"
: "${PLAYWRIGHT_SOCKET_PATH:=/var/run/playwright/playwright.sock}"
: "${PLAYWRIGHT_SERVER_ARGS:=}"

echo "[entrypoint] Using TCP endpoint ${PLAYWRIGHT_WS_HOST}:${PLAYWRIGHT_WS_PORT}"
echo "[entrypoint] Unix socket at ${PLAYWRIGHT_SOCKET_PATH}"

# Clean up any stale socket from a previous run
rm -f "${PLAYWRIGHT_SOCKET_PATH}"

# Start the Playwright browser server in the background.
# This uses the Playwright CLI installed globally in the image.
playwright run-server \
  --port "${PLAYWRIGHT_WS_PORT}" \
  --host "${PLAYWRIGHT_WS_HOST}" \
  ${PLAYWRIGHT_SERVER_ARGS} &

PW_PID=$!
echo "[entrypoint] Playwright server PID: ${PW_PID}"

# Ensure we clean up the Playwright server on container shutdown
cleanup() {
  echo "[entrypoint] Caught signal, stopping Playwright (PID ${PW_PID})"
  kill "${PW_PID}" 2>/dev/null || true
  exit 0
}

trap cleanup TERM INT

# socat will:
#   - listen on the Unix socket
#   - connect each incoming client to the TCP WS endpoint
# This keeps the container in the foreground.
echo "[entrypoint] Starting socat proxy on ${PLAYWRIGHT_SOCKET_PATH} -> ${PLAYWRIGHT_WS_HOST}:${PLAYWRIGHT_WS_PORT}"

exec socat \
  UNIX-LISTEN:"${PLAYWRIGHT_SOCKET_PATH}",fork,reuseaddr \
  TCP:"${PLAYWRIGHT_WS_HOST}":"${PLAYWRIGHT_WS_PORT}"
