use thiserror::Error;

/// Error generated reading and writing binary data.
#[derive(Debug, Error)]
pub enum BinaryError {
    /// Error generated attempted to read past the end of file.
    #[error("attempt to read past EOF")]
    ReadPastEof,

    /// Error generated trying to read the char type.
    #[error("invalid character read from stream")]
    InvalidChar,

    /// Error generated converting between integers.
    #[error(transparent)]
    TryFromInt(#[from] std::num::TryFromIntError),

    /// Error generated converting to UTF-8.
    #[error(transparent)]
    Utf8Error(#[from] std::string::FromUtf8Error),

    /// Error generated by input / output.
    #[error(transparent)]
    Io(#[from] std::io::Error),

    /// Error generated converting to slices.
    #[error(transparent)]
    TryFromSlice(#[from] std::array::TryFromSliceError),

    /// Custom error message.
    #[error("{0}")]
    Custom(String),

    /// Generic error type for user space errors.
    #[error(transparent)]
    Boxed(#[from] Box<dyn std::error::Error + Send + Sync>),
}
