#!/usr/bin/env python3
"""
Himmelblau Dockerfile generator

Usage:
  python gen_dockerfiles.py [--out ./dockerfiles] [--only debian12,ubuntu22.04]

This follows a config-driven pattern inspired by Samba's bootstrap/config.py:
- deb family => cargo deb chain with per-package feature flags and --deb-revision=<distro>
- rpm/zypper family => cargo build + strip + cargo generate-rpm chain
"""

import argparse
import os

GENERATED_MARKER = """\
#
# This file is generated by gen_dockerfiles.py
# Do not edit by hand — update the generator or config instead.
#

"""

# ---- Package config ----------------------------------------------------------

COMMON = [
    "git",
    "curl",
    "wget",
    "make",
    "pkg-config",
    "jq",
    "ca-certificates",
    "libtool",
    "autoconf",
    "gettext",
    "checkpolicy",
    "policycoreutils",
    "python3",
    "systemd",
]

PKG_PAIRS = [
    ("build-essential", "@development-tools"),
    ("gcc", "gcc"),
    ("g++", "gcc-c++"),
    ("libssl-dev", "openssl-devel"),
    ("libdbus-1-dev", "dbus-devel"),
    ("libkrb5-dev", "krb5-devel"),
    ("libpam0g-dev", "pam-devel"),
    ("libcap-dev", "libcap-devel"),
    ("libudev-dev", "libudev-devel"),
    ("cmake", "cmake"),
    ("libtss2-dev tpm-udev", "tpm2-tss-devel"),
    ("libclang-dev", "clang"),
    ("libpcre2-dev", "pcre2-devel"),
    ("libsqlite3-dev", "sqlite-devel"),
    ("libunistring-dev", "libunistring-devel"),
]

SELINUX_PKGS = ["policycoreutils-devel", "selinux-policy-targeted"]
AUTHSELECT_PKGS = ["authselect"]

DEB_PKGS = COMMON + [p for p, _ in PKG_PAIRS if p]
RPM_PKGS = COMMON + AUTHSELECT_PKGS + [q for _, q in PKG_PAIRS if q]

APT_BOOTSTRAP = """\
RUN apt-get update && apt-get install -y \\\n    {pkgs} \\\n && rm -rf /var/lib/apt/lists/*\n"""

DNF_BOOTSTRAP = """\
RUN dnf -y update && dnf -y install \\\n    {pkgs} \\\n && dnf clean all\n"""

ZYPPER_BOOTSTRAP = """\
RUN zypper --non-interactive refresh && \\\n    zypper --non-interactive update && \\\n    zypper --non-interactive install --no-recommends \\\n        {pkgs} && \\\n    zypper clean --all\n"""

FAMILIES = {
    "deb": {
        "bootstrap": APT_BOOTSTRAP,
        "pkgs": DEB_PKGS,
        "env": "ENV DEBIAN_FRONTEND=noninteractive HIMMELBLAU_ALLOW_MISSING_SELINUX=1",
    },
    "rpm": {
        "bootstrap": DNF_BOOTSTRAP,
        "pkgs": RPM_PKGS,
        "env": None,
    },
    "zypper": {
        "bootstrap": ZYPPER_BOOTSTRAP,
        "pkgs": RPM_PKGS,
        "env": None,
    },
}

# ---- Final command builders --------------------------------------------------

PACKAGES = [
    # (crate name, crate source, needs_tpm_feature)
    ("himmelblaud", "src/daemon", True),
    ("nss_himmelblau", "src/nss", True),
    ("pam_himmelblau", "src/pam", True),
    ("sshd-config", "src/sshd-config", False),
    ("sso", "src/sso", True),
    ("qr-greeter", "src/qr-greeter", False),
    ("selinux", "src/selinux", False),
    ("o365", "src/o365", False),
]

CMD_TAB = "     "
CMD_SEP = f" && \ \n{CMD_TAB}"


def build_deb_final_cmd(features: list, distro_slug: str) -> str:
    parts = []
    for pkg, _, needs_tpm in PACKAGES:
        if pkg == "selinux":  # Debian doesn't use selinux
            continue
        if not needs_tpm and "tpm" in features:
            features.remove("tpm")
        feat_str = f" --features {','.join(features)}" if features else ""
        if "pam" in pkg or "nss" in pkg:
            feat_str += " --multiarch=same"
        parts.append(f"cargo deb{feat_str} --deb-revision={distro_slug} -p {pkg}")
    gen_servicefiles = "make deb-servicefiles"
    return f'CMD ["/bin/sh", "-c", \\\n{CMD_TAB}"{gen_servicefiles} && {CMD_SEP.join(parts)} "]'


def build_rpm_final_cmd(features: list, selinux: bool) -> str:
    feat_str = f" --features {','.join(features)}" if features else ""
    build = f"cargo build --release{feat_str} && \\ \n{CMD_TAB}"
    strip = CMD_SEP.join(
        ["strip -s target/release/%s" % s for s in ["*.so", "aad-tool", "himmelblaud", "himmelblaud_tasks", "broker"]]
    )
    if selinux:
        pkgs = PACKAGES
    else:
        pkgs = [pkg for pkg in PACKAGES if pkg[0] != "selinux"]
    rpms = CMD_SEP.join([f"cargo generate-rpm -p {s}" for _, s, _ in pkgs])
    gen_servicefiles = "make rpm-servicefiles"
    gen_authselect = "(authselect select minimal --force || authselect select local --force) && make authselect"
    return f'CMD ["/bin/sh", "-c", \\\n{CMD_TAB}"{gen_servicefiles} && {build}{strip} && {gen_authselect} && \\\n{CMD_TAB}{rpms}"]'


# ---- Distro targets ----------------------------------------------------------

DISTS = {
    # ---- Debian family ----
    "debian12": {
        "family": "deb",
        "image": "debian:12",
        "replace": {
            "@development-tools": "",
        },
        "tpm": True,
    },
    "debian13": {
        "family": "deb",
        "image": "debian:13",
        "replace": {
            "@development-tools": "",
        },
        "tpm": True,
    },
    "ubuntu22.04": {
        "family": "deb",
        "image": "ubuntu:22.04",
        "tpm": True,
    },
    "ubuntu24.04": {
        "family": "deb",
        "image": "ubuntu:24.04",
        "tpm": True,
    },
    "test": {
        "family": "deb",
        "image": "ubuntu:24.04",
        "tpm": False,
    },
    # ---- Fedora family ----
    "fedora42": {
        "family": "rpm",
        "image": "fedora:42",
        "tpm": True,
        "selinux": True,
    },
    "fedora43": {
        "family": "rpm",
        "image": "fedora:43",
        "tpm": True,
        "selinux": True,
    },
    "rawhide": {
        "family": "rpm",
        "image": "fedora:rawhide",
        "extra_prep": [
            "RUN dnf -y --refresh update glibc glibc-common glibc-minimal-langpack systemd-libs systemd-standalone-sysusers rpm-libs rpm && dnf -y --refresh update"
        ],
        "tpm": True,
        "selinux": True,
    },
    # ---- Rocky family ----
    "rocky8": {
        "family": "rpm",
        "image": "rockylinux/rockylinux:8",
        "extra_prep": [
            "RUN dnf -y install 'dnf-command(config-manager)' && dnf config-manager --set-enabled powertools"
        ],
        "replace": {
            "build-essential": '"@Development Tools"',
            "@development-tools": "",
        },
        "tpm": False,
        "selinux": True,
    },
    "rocky9": {
        "family": "rpm",
        "image": "rockylinux/rockylinux:9",
        "extra_prep": [
            "RUN dnf -y install 'dnf-command(config-manager)' && dnf config-manager --set-enabled crb",
            "RUN sed -i -e 's|$rltype||g' /etc/yum.repos.d/rocky*.repo",
        ],
        "replace": {
            "build-essential": '"@Development Tools"',
            "@development-tools": "",
            "curl": "",  # avoid the curl/curl-minimal install conflict
        },
        "tpm": True,
        "selinux": True,
    },
    "rocky10": {
        "family": "rpm",
        "image": "rockylinux/rockylinux:10",
        "extra_prep": [
            "RUN dnf install -y 'dnf-command(config-manager)' && dnf config-manager --set-enabled crb",
            "RUN sed -i -e 's|$rltype||g' /etc/yum.repos.d/rocky*.repo",
        ],
        "replace": {
            "build-essential": '"@Development Tools"',
            "@development-tools": "",
        },
        "tpm": True,
        "selinux": True,
    },
    # ---- SUSE family ----
    "sle15sp6": {
        "family": "zypper",
        "image": "registry.suse.com/suse/sle15:15.6",
        "scc": True,
        "scc_vers": "15.6",
        "replace": {
            "build-essential": "",
            "@development-tools": "",
            "dbus-devel": "dbus-1-devel",
            "tpm2-tss-devel": "tpm2-0-tss-devel",
            "sqlite-devel": "sqlite3-devel",
            "policycoreutils-devel": "",
            "selinux-policy-targeted": "",
        },
        "tpm": True,
    },
    "sle15sp7": {
        "family": "zypper",
        "image": "registry.suse.com/suse/sle15:15.7",
        "scc": True,
        "scc_vers": "15.7",
        "replace": {
            "build-essential": "",
            "@development-tools": "",
            "dbus-devel": "dbus-1-devel",
            "tpm2-tss-devel": "tpm2-0-tss-devel",
            "sqlite-devel": "sqlite3-devel",
            "clang": "clang7",
            "policycoreutils-devel": "",
            "selinux-policy-targeted": "",
        },
        "tpm": True,
    },
    "sle16": {
        "family": "zypper",
        "image": "registry.suse.com/bci/bci-sle16-kernel-module-devel:16.0",
        "scc": True,
        "scc_vers": "16.0",
        "extra_prep": [
            # Temporary patch for broken SLE libudev1 version in the base image
            "RUN zypper in -y --oldpackage libudev1-257.7-160000.2.2.x86_64",
            # Temporary authselect build, since it hasn't landed in PackageHub yet
            "RUN zypper ar -e https://download.opensuse.org/repositories/home:/dmulder:/branches:/authselect/16.0/home:dmulder:branches:authselect.repo",
            "RUN zypper --non-interactive --gpg-auto-import-keys refresh home_dmulder_branches_authselect"
        ],
        "replace": {
            "build-essential": "",
            "@development-tools": "",
            "dbus-devel": "dbus-1-devel",
            "tpm2-tss-devel": "tpm2-0-tss-devel",
            "sqlite-devel": "sqlite3-devel",
            "selinux-policy-targeted": "selinux-tools selinux-policy-devel",
        },
        "tpm": True,
        "selinux": True,
    },
    "tumbleweed": {
        "family": "zypper",
        "image": "opensuse/tumbleweed:latest",
        "extra_prep": [
            "RUN zypper ar -e https://download.opensuse.org/repositories/security:/idm/openSUSE_Tumbleweed/security:idm.repo",
            "RUN zypper --non-interactive --gpg-auto-import-keys refresh security_idm"
        ],
        "replace": {
            "build-effective": "",
            "@development-tools": "",
            "dbus-devel": "dbus-1-devel",
            "tpm2-tss-devel": "tpm2-0-tss-devel",
            "sqlite-devel": "sqlite3-devel",
        },
        "tpm": True,
        "selinux": True,
    },
}

DOCKERFILE_TPL = """\
{GENERATED_MARKER}FROM {base_image}

{env}
{sle_connect}
# Install essential build dependencies
{bootstrap}

# Set environment for Rust
ENV PATH="/root/.cargo/bin:${{PATH}}"

# Project layout
VOLUME /himmelblau

# Change directory to the repository
WORKDIR /himmelblau

# Install Rust (latest stable)
RUN curl https://sh.rustup.rs -sSf | sh -s -- -y && \\
    cargo install cargo-deb cargo-generate-rpm

{selinux_enabled}
# Build the project and create the packages
{final_cmd}
"""

SLE_CONNECT_TPL = """\
# Install SUSEConnect and dependencies for registration
RUN zypper --non-interactive refresh && \\
    zypper --non-interactive install --no-recommends \\
        SUSEConnect \\
        ca-certificates \\
        suse-build-key && \\
    zypper clean --all

RUN --mount=type=secret,id=scc_regcode,dst=/run/secrets/scc_regcode \\
    set -e && \\
    source /run/secrets/scc_regcode && \\
    SUSEConnect --email "$email" --regcode "$regcode" && \\
    SUSEConnect -p PackageHub/%s/x86_64
"""


def build_pkg_list(dist_cfg, selinux):
    fam = FAMILIES[dist_cfg["family"]]
    pkgs = list(fam["pkgs"])
    rep = dist_cfg.get("replace", {})
    out = []
    if selinux:
        pkgs += SELINUX_PKGS
    for p in pkgs:
        q = rep.get(p, p)
        if q:
            out.append(q)
    out = sorted(set(out))
    sep = " \\\n        "
    return sep.join(out)


def render(dist_name, dist_cfg):
    fam = FAMILIES[dist_cfg["family"]]
    selinux = bool(dist_cfg.get("selinux", False))
    pkgs = build_pkg_list(dist_cfg, selinux)
    bootstrap = fam["bootstrap"].format(pkgs=pkgs).rstrip()
    env = fam["env"] or ""
    sle_connect = SLE_CONNECT_TPL % dist_cfg.get("scc_vers") if dist_cfg.get("scc") else ""

    # Features
    tpm = bool(dist_cfg.get("tpm", False))
    features = []
    if tpm:
        features.append("tpm")

    final_cmd = ""
    if dist_cfg["family"] == "deb" and dist_name != "test":
        final_cmd = build_deb_final_cmd(features, dist_name)
    elif dist_name == "test":
        final_cmd = "CMD cargo test"
    else:
        final_cmd = build_rpm_final_cmd(features, selinux)

    blocks = []
    if dist_cfg.get("extra_prep"):
        blocks.extend(dist_cfg["extra_prep"])
    extra = "\n".join(blocks) + ("\n" if blocks else "")

    df = DOCKERFILE_TPL.format(
        GENERATED_MARKER=GENERATED_MARKER,
        base_image=dist_cfg["image"],
        env=env,
        bootstrap=(extra + bootstrap),
        sle_connect=("\n" + sle_connect + "\n" if sle_connect else ""),
        selinux_enabled=("ENV HIMMELBLAU_ALLOW_MISSING_SELINUX=1" if not selinux else ""),
        final_cmd=final_cmd,
    )
    return df


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--out", default="./dockerfiles", help="Output directory")
    ap.add_argument("--only", default="", help="Comma-separated list of dists to render")
    args = ap.parse_args()

    if args.only:
        want = {x.strip() for x in args.only.split(",") if x.strip()}
    else:
        want = set(DISTS.keys())

    os.makedirs(args.out, exist_ok=True)

    written = []
    for name in sorted(want):
        if name not in DISTS:
            print(f"[skip] unknown dist: {name}")
            continue
        df = render(name, DISTS[name])
        path = os.path.join(args.out, f"Dockerfile.{name}")
        with open(path, "w", encoding="utf-8") as f:
            f.write(df)
        written.append(path)

    print(f"Wrote {len(written)} Dockerfiles:")
    for p in written:
        print("  -", p)


if __name__ == "__main__":
    main()
