import { getParents } from "../../dag/utils";
import { solve } from "../../simplex";
/** @internal */
function buildOperator(options) {
    function optCall(topLayer, bottomLayer, topDown) {
        // check if input is too large
        const reordered = topDown ? bottomLayer : topLayer;
        const numVars = (reordered.length * Math.max(reordered.length - 1, 0)) / 2;
        const numEdges = topLayer.reduce((t, n) => t + n.nchildren(), 0);
        if (options.large !== "large" && numVars > 1200) {
            throw new Error(`size of dag to twolayerOpt is too large and will likely crash, enable "large" dags to run anyway`);
        }
        else if (options.large !== "large" &&
            options.large !== "medium" &&
            (numVars > 400 || numEdges > 100)) {
            throw new Error(`size of dag to twolayerOpt is too large and will likely not finish, enable "medium" dags to run anyway`);
        }
        // initialize model
        const variables = {};
        const constraints = {};
        const ints = {};
        // initialize map to create ids for labeling constraints
        const inds = new Map(reordered.map((node, i) => [node, i]));
        /** create key from nodes */
        function key(...nodes) {
            return nodes
                .map((n) => inds.get(n))
                .sort((a, b) => a - b)
                .join(" => ");
        }
        let unconstrained, groups;
        if (topDown) {
            const withParents = new Set(topLayer.flatMap((node) => node.children()));
            unconstrained = bottomLayer.filter((node) => !withParents.has(node));
            groups = topLayer
                .map((node) => node.children())
                .filter((cs) => cs.length > 1);
        }
        else {
            unconstrained = topLayer.filter((n) => !n.nchildren());
            const parents = getParents(topLayer);
            groups = [...parents.values()];
        }
        // NOTE distance cost for an unconstrained node ina group can't violate
        // all pairs at once, so the cost is ~(n/2)^2 not n(n-1)/2
        const groupSize = groups.reduce((t, cs) => t + cs.length * cs.length, 0);
        const maxDistCost = (groupSize * unconstrained.length) / 4;
        const distWeight = 1 / (maxDistCost + 1);
        // add small value to objective for preserving the original order of nodes
        const preserveWeight = distWeight / (numVars + 1);
        // need a function that returns whether one child originally came before
        // another, which means we need a reverse map from node to original index
        const cinds = new Map(bottomLayer.map((node, i) => [node, i]));
        // add variables for each pair of bottom later nodes indicating if they
        // should be flipped
        for (const [i, n1] of reordered.slice(0, reordered.length - 1).entries()) {
            for (const n2 of reordered.slice(i + 1)) {
                const pair = key(n1, n2);
                ints[pair] = 1;
                constraints[pair] = {
                    max: 1,
                };
                variables[pair] = {
                    opt: -preserveWeight,
                    [pair]: 1,
                };
            }
        }
        // add constraints to enforce triangle inequality, e.g. that if a -> b is 1
        // and b -> c is 1 then a -> c must also be one
        for (const [i, n1] of reordered.slice(0, reordered.length - 1).entries()) {
            for (const [j, n2] of reordered.slice(i + 1).entries()) {
                for (const n3 of reordered.slice(i + j + 2)) {
                    const pair1 = key(n1, n2);
                    const pair2 = key(n1, n3);
                    const pair3 = key(n2, n3);
                    const triangle = key(n1, n2, n3);
                    const triangleUp = triangle + "+";
                    constraints[triangleUp] = {
                        max: 1,
                    };
                    variables[pair1][triangleUp] = 1;
                    variables[pair2][triangleUp] = -1;
                    variables[pair3][triangleUp] = 1;
                    const triangleDown = triangle + "-";
                    constraints[triangleDown] = {
                        min: 0,
                    };
                    variables[pair1][triangleDown] = 1;
                    variables[pair2][triangleDown] = -1;
                    variables[pair3][triangleDown] = 1;
                }
            }
        }
        // add crossing minimization
        for (const [i, p1] of topLayer.slice(0, topLayer.length - 1).entries()) {
            for (const p2 of topLayer.slice(i + 1)) {
                for (const c1 of p1.ichildren()) {
                    for (const c2 of p2.ichildren()) {
                        if (c1 === c2) {
                            continue;
                        }
                        const pair = topDown ? key(c1, c2) : key(p1, p2);
                        variables[pair].opt += Math.sign(cinds.get(c1) - cinds.get(c2));
                    }
                }
            }
        }
        // add distance minimization
        if (options.dist) {
            // NOTE this works by looking at triples of nodes with a common ancestor
            // (parent / child) and an unconstrained node. We add a slack variable
            // responsible for the cost to the objective with a weight such that if
            // all constraints are violated, it's still less than one crossing. We
            // then add constraints that say the slack variable must be one if the
            // unconstrained node is inside of the two nodes with a common ancestor.
            for (const node of unconstrained) {
                for (const group of groups) {
                    for (const [i, start] of group.entries()) {
                        for (const end of group.slice(i + 1)) {
                            // want to minimize node being between start and end
                            // NOTE we don't sort because we care which is in the center
                            const base = [start, node, end]
                                .map((n) => inds.get(n))
                                .join(" => ");
                            const slack = `dist ${base}`;
                            const normal = `${slack} normal`;
                            const reversed = `${slack} reversed`;
                            variables[slack] = {
                                opt: distWeight,
                                [normal]: 1,
                                [reversed]: 1,
                            };
                            let pos = 0;
                            for (const [n1, n2] of [
                                [start, node],
                                [start, end],
                                [node, end],
                            ]) {
                                const pair = key(n1, n2);
                                const sign = Math.sign(inds.get(n1) - inds.get(n2));
                                pos += +(sign > 0);
                                variables[pair][normal] = -sign;
                                variables[pair][reversed] = sign;
                            }
                            constraints[normal] = {
                                min: 1 - pos,
                            };
                            constraints[reversed] = {
                                min: pos - 2,
                            };
                        }
                    }
                }
            }
        }
        // solve objective
        // NOTE bundling sets this to undefined, and we need it to be settable
        const ordering = solve("opt", "min", variables, constraints, ints);
        // sort layers
        reordered.sort((n1, n2) => ordering[key(n1, n2)] || -1);
    }
    function large(val) {
        if (val === undefined) {
            return options.large;
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, options), { large: val }));
        }
    }
    optCall.large = large;
    function dist(val) {
        if (val === undefined) {
            return options.dist;
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, options), { dist: val }));
        }
    }
    optCall.dist = dist;
    return optCall;
}
/**
 * Create a default {@link OptOperator}, bundled as {@link twolayerOpt}.
 */
export function opt(...args) {
    if (args.length) {
        throw new Error(`got arguments to opt(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ large: "small", dist: false });
}
