/**
 * An {@link GreedyOperator} that calls another {@link TwolayerOperator} before
 * greedily swapping nodes to minimize crossings.
 *
 * @packageDocumentation
 */
import { TwolayerOperator } from ".";
/** the node datum of a set of operators */
export declare type OpNodeDatum<Op extends TwolayerOperator> = Op extends TwolayerOperator<infer D, never> ? D : never;
/** the link datum of a set of operators */
export declare type OpLinkDatum<Op extends TwolayerOperator> = Op extends TwolayerOperator<never, infer L> ? L : never;
/**
 * A {@link TwolayerOperator} that first calls a base twolayer operator, then
 * greedily swaps nodes to minimize crossings.
 *
 * This may be faster than {@link OptOperator}, but should produce better
 * layouts than {@link AggOperator}.
 *
 * Create with {@link greedy}.
 */
export interface GreedyOperator<Op extends TwolayerOperator = TwolayerOperator> extends TwolayerOperator<OpNodeDatum<Op>, OpLinkDatum<Op>> {
    /**
     * Set the {@link TwolayerOperator} for this operator.
     *
     * This operator will first call its base operator, and the greedily swap
     * nodes to minimize edge crossings. To only greedily minimize edge
     * crossings, set base to a no op.
     */
    base<NewOp extends TwolayerOperator>(val: NewOp): GreedyOperator<NewOp>;
    /**
     * Get the current base operator.
     */
    base(): Op;
    /**
     * Set whether this operator should scan to find swaps.
     *
     * Using the scan method takes longer (quadratic in layer size, versus
     * linear), but produces fewer crossings.
     */
    scan(val: boolean): GreedyOperator<Op>;
    /**
     * Get the current base operator.
     */
    scan(): boolean;
}
/** default greedy operator */
export declare type DefaultGreedyOperator = GreedyOperator<TwolayerOperator<unknown, unknown>>;
/**
 * Create a default {@link GreedyOperator}, bundled as {@link twolayerGreedy}.
 */
export declare function greedy(...args: never[]): DefaultGreedyOperator;
