/**
 * A topological layout using {@link GridOperator}.
 *
 * @packageDocumentation
 */
import { Dag, DagNode } from "../dag";
import { Up } from "../utils";
import { LaneOperator } from "./lane";
import { GreedyOperator } from "./lane/greedy";
/**
 * an operator for assigning rank priorities to nodes for a grid layout
 *
 * Nodes with lower ranks will be chose first, so for a rank order to work, a
 * node must have a lower rank than it's children.
 *
 * @remarks
 *
 * This is currently implemented naively, which means that nodes without a rank are considered to have the lowest rank, and will come before any nodes given an explicit rank. In addition, if the ranks are inconsistent with the layout of the dag, then they will be silently ignored.
 *
 * Both of these behaviors may change in the future i.e. where unranked means
 * no condition, and an error will be thrown if the ranks are inconsistent.
 */
export interface RankOperator<N = never, L = never> {
    (node: DagNode<N, L>): number | undefined;
}
/** all operators for the grid layout */
export interface Operators<N = never, L = never> {
    /** the operator for assigning nodes to a lane */
    lane: LaneOperator<N, L>;
    /** the operator for assigning nodes a rank */
    rank: RankOperator<N, L>;
}
/** the typed dag specified by a set of operators */
export declare type OpDag<Op extends Operators> = Op extends Operators<infer N, infer L> ? Dag<N, L> : never;
/**
 * The return from calling {@link GridOperator}
 *
 * This is the final width and height of the laid out dag.
 */
export interface GridInfo {
    /** the total weight after layout */
    width: number;
    /** the total height after layout */
    height: number;
}
/**
 * A simple grid based topological layout operator.
 *
 * This layout algorithm constructs a topological representation of the dag
 * meant for visualization. The nodes are topologically ordered and then nodes
 * are put into lanes such that an edge can travel horizontally to the lane of
 * a child node, and then down without intersecting to that child.
 *
 * This layout produces good representations when you want a compressed layout
 * where each node gets its only line formation.
 *
 * Create with {@link grid}.
 *
 * @example
 * <img alt="grid example" src="media://grid-greedy-topdown.png" width="200">
 *
 * @example
 * ```typescript
 * const data = [["parent", "child"], ...];
 * const create = connect();
 * const dag = create(data);
 * const layout = grid();
 * const { width, height } = layout(dag);
 * for (const node of dag) {
 *   console.log(node.x, node.y);
 * }
 * ```
 */
export interface GridOperator<Ops extends Operators> {
    /** Layout the input dag */
    (dag: OpDag<Ops>): GridInfo;
    /**
     * Set the lane operator to the given {@link LaneOperator} and returns a new
     * version of this operator. (default: {@link GreedyOperator})
     */
    lane<NewLane extends LaneOperator>(val: NewLane): GridOperator<Up<Ops, {
        /** new lane */
        lane: NewLane;
    }>>;
    /** Get the current lane operator */
    lane(): Ops["lane"];
    /**
     * Set the rank operator to the given {@link RankOperator} and returns a new
     * version of this operator. (default: () =\> undefined)
     */
    rank<NewRank extends RankOperator>(val: NewRank): GridOperator<Up<Ops, {
        /** new rank */
        rank: NewRank;
    }>>;
    /** Get the current lane operator */
    rank(): Ops["rank"];
    /**
     * Sets this grid layout's node size to the specified two-element array of
     * numbers [ *width*, *height* ] and returns a new operator. These sizes are
     * effectively the grid size, e.g. the spacing between adjacent lanes or rows
     * in the grid. (default: [1, 1])
     *
     * Note, due to the way that edges are meant to rendered, edges won't
     * intersect with nodes if width is half of the actual node width.
     */
    nodeSize(val: readonly [number, number]): GridOperator<Ops>;
    /** Get the current node size. */
    nodeSize(): [number, number];
    /**
     * Update the grid layout size
     *
     * Sets this grid layout's node size to the specified two-element array of
     * numbers [ *width*, *height* ] and returns a new operator. After the
     * initial layout, if size is not null, the dag will be rescaled so that it
     * fits in width x height. (default: null)
     */
    size(val: null | readonly [number, number]): GridOperator<Ops>;
    /** Get the current size. */
    size(): null | [number, number];
}
/** the default grid operator */
export declare type DefaultGridOperator = GridOperator<{
    /** default lane: greedy */
    lane: GreedyOperator;
    /** default rank: none */
    rank: RankOperator<unknown, unknown>;
}>;
/**
 * Create a new {@link GridOperator} with default settings.
 */
export declare function grid(...args: never[]): DefaultGridOperator;
