/**
 * Three built-in methods exist to turn existing raw data into {@link Dag}s:
 * 1. {@link HierarchyOperator} - when the data already has a dag structure.
 * 2. {@link StratifyOperator} - when the dag has a tabular structure, referencing parents by id.
 * 3. {@link ConnectOperator} - when the dag has a link structure and is specified as pairs of nodes.
 *
 * @packageDocumentation
 */
import { Dag } from ".";
import { Up } from "../utils";
/***********
 * General *
 ***********/
/**
 * The interface for getting a node id from data. The function must return an
 * appropriate unique string id for given datum. This operator will only be
 * called once for each input.
 *
 * `i` will increment in the order data are processed.
 *
 * This is used in {@link StratifyOperator.id}, {@link
 * ConnectOperator.sourceId}, and {@link ConnectOperator.targetId}.
 */
export interface IdOperator<Datum = never> {
    (d: Datum, i: number): string;
}
/***********
 * Connect *
 ***********/
/**
 * An operator that creates node data from an id
 *
 * The index passed in is the edge index where the id is first seen.
 */
export interface IdNodeDatumOperator<D = unknown> {
    (id: string, index: number): D;
}
/**
 * The default node data on dags built using {@link ConnectOperator}
 */
export interface ConnectDatum {
    /** the id */
    id: string;
}
/**
 * The operators that parametrize {@link ConnectOperator}
 */
export interface ConnectOperators<N = unknown, L = never> {
    /** the source id operator */
    sourceId: IdOperator<L>;
    /** the target id operator */
    targetId: IdOperator<L>;
    /** the node datum operator */
    nodeDatum: IdNodeDatumOperator<N>;
}
/**
 * The constraint applied to data passed into {@link ConnectOperator}
 * conditioned on its operators.
 */
declare type ConnectLinkDatum<Ops extends ConnectOperators> = Ops extends ConnectOperators<unknown, infer L> ? L : never;
export interface ConnectOperator<NodeDatum, Ops extends ConnectOperators<NodeDatum>> {
    /**
     * An operator that constructs a {@link Dag} from link data.
     *
     * Create a default connect operator with {@link connect}. The accessor for the
     * {@link sourceId | source id string}, {@link targetId | target id string},
     * and whether to allow {@link single} nodes can all be modified.
     *
     * Links in the dag will have the same data as the objects passed in, and nodes
     * will have the ids referenced as either the source or the target.
     *
     * @example
     * ```typescript
     * const data = [ ["parent", "child"] ];
     * const create = connect();
     * const dag = create(data);
     * ```
     *
     * @example
     * ```json
     * [
     *   ["Eve", "Cain"],
     *   ["Eve", "Seth"],
     *   ["Seth", "Enos"],
     *   ["Seth", "Noam"],
     *   ["Eve", "Abel"],
     *   ["Eve", "Awan"]
     * ]
     * ```
     */
    <L extends ConnectLinkDatum<Ops>>(data: readonly L[]): Dag<NodeDatum, L>;
    /**
     * Sets the source accessor to the given {@link IdOperator} and returns this
     * ConnectOperator. This should return the source id of the link
     * data. The default accessor is:
     *
     * ```js
     * function sourceAccessor(link) {
     *   return link[0];
     * }
     * ```
     */
    sourceId<NewId extends IdOperator>(id: NewId): ConnectOperator<NodeDatum, Up<Ops, {
        /** the new source id */
        sourceId: NewId;
    }>>;
    /** Gets the current sourceId accessor. */
    sourceId(): Ops["sourceId"];
    /**
     * Sets the target accessor to the given {@link IdOperator} and returns this
     * ConnectOperator. This should return the target id of the link
     * data. The default accessor is:
     *
     * ```js
     * function sourceAccessor(link) {
     *   return link[1];
     * }
     * ```
     */
    targetId<NewId extends IdOperator>(id: NewId): ConnectOperator<NodeDatum, Up<Ops, {
        /** the new target id */
        targetId: NewId;
    }>>;
    /** Gets the current targetId accessor. */
    targetId(): Ops["targetId"];
    /**
     * Sets the id node datum  accessor to the given {@link IdNodeDatumOperator} and returns this
     * ConnectOperator. This function allows you to decide what data to attach to nodes created via the connect method. The default simple wraps it in an object with an `id` field.
     * data. The default accessor is:
     *
     * ```js
     * function nodeDatumAccessor(id) {
     *   return { id };
     * }
     * ```
     */
    nodeDatum<NewNodeDatum, NewNodeDatumOp extends IdNodeDatumOperator<NewNodeDatum>>(data: NewNodeDatumOp & IdNodeDatumOperator<NewNodeDatum>): ConnectOperator<NewNodeDatum, Up<Ops, {
        /** the new node datum */
        nodeDatum: NewNodeDatumOp;
    }>>;
    /** Get the current id node datum operator */
    nodeDatum(): Ops["nodeDatum"];
    /**
     * Sets the allowance for single nodes. If enabled and the source id equals
     * the target id, then a single node with no parents will be created.
     * Otherwise a self loop will be created which will result in an error. Note
     * only single nodes without parents or children need to be specified this
     * way, otherwise any other connection to a node will create it. (default: false)
     */
    single(val: boolean): ConnectOperator<NodeDatum, Ops>;
    /** get the current single node setting. */
    single(): boolean;
    /**
     * Sets whether edges should be reversed to remove cycles
     *
     * If true, after creating the initial "graph" of nodes, an algorithm will
     * run to reverse enough edges to make the resulting data structure a valid
     * dag. It does this by reversing links in the original dag. Note that since
     * the structure is still a dag, old parents could now be positioned as
     * children, and instead the link will have `reversed` set to true. It is up
     * to the user to decide how to handle these links. (default: false)
     */
    decycle(val: boolean): ConnectOperator<NodeDatum, Ops>;
    /** get the current decycle setting */
    decycle(): boolean;
}
/** default interface for tuples that start with a string */
export interface ZeroString {
    /** the zero property */
    readonly 0: string;
}
/** default interface for functions whose second element is a string */
export interface OneString {
    /** the one property */
    readonly 1: string;
}
/** the default connect operator */
export declare type DefaultConnectOperator = ConnectOperator<ConnectDatum, {
    /** the default source id operator */
    sourceId: IdOperator<ZeroString>;
    /** the default target id operator */
    targetId: IdOperator<OneString>;
    /** the default node datum operator */
    nodeDatum: IdNodeDatumOperator<ConnectDatum>;
}>;
/**
 * Creates a new {@link ConnectOperator} with the default settings. This is
 * bundled as {@link dagConnect}
 */
export declare function connect(...args: never[]): DefaultConnectOperator;
/*************
 * Hierarchy *
 *************/
/**
 * The interface for getting child data from node data. This function must
 * return data for every child given the data for the current node. `i` will
 * increment for each node processed.
 *
 * Can be modified with {@link children}.
 */
export interface ChildrenOperator<NodeDatum> {
    (d: NodeDatum, i: number): readonly NodeDatum[] | undefined;
}
/**
 * The interface for getting children data and associated link data from node
 * data. This function must return data for every child of the given node, and
 * data for link between the two. `i` will increment for each node processed.
 *
 * Can be modified with {@link childrenData}.
 */
export interface ChildrenDataOperator<NodeDatum, LinkDatum = unknown> {
    (d: NodeDatum, i: number): readonly (readonly [NodeDatum, LinkDatum])[] | undefined;
}
/**
 * What gets returned by {@link childrenData}() when {@link children} is set.
 */
export interface WrappedChildrenOperator<NodeDatum, Children extends ChildrenOperator<NodeDatum>> extends ChildrenDataOperator<NodeDatum, undefined> {
    /** the wrapped children operator */
    wrapped: Children;
}
/**
 * What gets returned by {@link children}() when {@link childrenData} is set.
 */
export interface WrappedChildrenDataOperator<NodeDatum, ChildrenData extends ChildrenDataOperator<NodeDatum>> extends ChildrenOperator<NodeDatum> {
    /** the wrapped children data operator */
    wrapped: ChildrenData;
}
/** the hierarchy operator operators */
export interface HierarchyOperators<NodeDatum, LinkDatum = unknown> {
    /** the children operator */
    children: ChildrenOperator<NodeDatum>;
    /** the children data operator */
    childrenData: ChildrenDataOperator<NodeDatum, LinkDatum>;
}
/** a hierarchy operator with children */
export declare type ChildrenHierarchyOperator<NodeDatum, Children extends ChildrenOperator<NodeDatum>> = HierarchyOperator<NodeDatum, undefined, {
    /** new children */
    children: Children;
    /** new children data */
    childrenData: WrappedChildrenOperator<NodeDatum, Children>;
}>;
/** a hierarchy operator with children data specified */
export declare type ChildrenDataHierarchyOperator<NodeDatum, LinkDatum, ChildrenData extends ChildrenDataOperator<NodeDatum, LinkDatum>> = HierarchyOperator<NodeDatum, LinkDatum, {
    /** new children */
    children: WrappedChildrenDataOperator<NodeDatum, ChildrenData>;
    /** new children data */
    childrenData: ChildrenData;
}>;
export interface HierarchyOperator<NodeDatum, LinkDatum, Ops extends HierarchyOperators<NodeDatum, LinkDatum>> {
    /**
     * An operator that constructs a {@link Dag} from hierarchy data.
     *
     * A default operator can be created with {@link hierarchy}. How to access a
     * piece of data's {@link children} or {@link childrenData | children with
     * associated link data} can be altered. Similarly you can disable whether to
     * check that all initial nodes are actually {@link roots}.
     *
     * Data passed in will become node data, if {@link childrenData} is specified,
     * then link data will also be included. This method uses object identity, so
     * for two nodes to point to the same object, they must both return the same
     * object in their children.
     *
     * @example
     * ```typescript
     * const data = { id: "parent", children: [{ id: "child" }] };
     * const create = hierarchy();
     * const dag = create(data);
     * ```
     *
     * @example
     * ```json
     * {
     *   "id": "Eve",
     *     "children": [
     *     {
     *       "id": "Cain"
     *     },
     *     {
     *       "id": "Seth",
     *       "children": [
     *       {
     *         "id": "Enos"
     *       },
     *       {
     *         "id": "Noam"
     *       }
     *       ]
     *     },
     *     {
     *       "id": "Abel"
     *     },
     *     {
     *       "id": "Awan",
     *       "children": [
     *       {
     *         "id": "Enoch"
     *       }
     *       ]
     *     },
     *     {
     *       "id": "Azura"
     *     }
     *   ]
     * }
     * ```
     *
     * The children of each node will be further traversed until the entire dag
     * is explored. Unless {@link roots} is set to false, all initial roots must
     * be roots, i.e. they cann't occur in an call to children.
     *
     */
    (...data: readonly NodeDatum[]): Dag<NodeDatum, LinkDatum>;
    /**
     * Sets the children accessor to the given {@link ChildrenOperator} and
     * returns a new hierarchy operator. The default operator is:
     *
     * ```js
     * function children(d) {
     *   return d.children;
     * }
     * ```
     */
    children<NewDatum, NewChildren extends ChildrenOperator<NewDatum>>(ids: NewChildren & ChildrenOperator<NewDatum>): ChildrenHierarchyOperator<NewDatum, NewChildren>;
    /**
     * Gets the current {@link ChildrenOperator}, If {@link childrenData} was specified,
     * this will return a {@link WrappedChildrenOperator | wrapped version} that
     * returns only the children of that operator.
     */
    children(): Ops["children"];
    /**
     * Sets the childrenData accessor to the given {@link ChildrenDataOperator} and
     * returns a new hierarchy operator.
     */
    childrenData<NewDatum, NewLink, NewChildrenData extends ChildrenDataOperator<NewDatum, NewLink>>(data: NewChildrenData & ChildrenDataOperator<NewDatum, NewLink>): ChildrenDataHierarchyOperator<NewDatum, NewLink, NewChildrenData>;
    /**
     * Get the current childrenData operator. If {@link children} was specified, this
     * will return a {@link WrappedChildrenDataOperator | wrapped version} that
     * returns undefined data.
     */
    childrenData(): Ops["childrenData"];
    /**
     * Specify if only roots should be passed in, if true, hierarchy will throw
     * an error if a non-root was passed initially. (default: true)
     */
    roots(val: boolean): HierarchyOperator<NodeDatum, LinkDatum, Ops>;
    /** get the current roots value. */
    roots(): boolean;
    /**
     * Sets whether edges should be reversed to remove cycles
     *
     * If true, after creating the initial "graph" of nodes, an algorithm will
     * run to reverse enough edges to make the resulting data structure a valid
     * dag. It does this by reversing links in the original dag. Note that since
     * the structure is still a dag, old parents could now be positioned as
     * children, and instead the link will have `reversed` set to true. It is up
     * to the user to decide how to handle these links.
     *
     * Also note that by default, all inputs still need to be roots, if a "root"
     * node is in a cycle then {@link HierarchyOperator.roots} must also be set
     * to `false`. (default: false)
     */
    decycle(val: boolean): HierarchyOperator<NodeDatum, LinkDatum, Ops>;
    /** get the current decycle setting */
    decycle(): boolean;
}
/** an object with children */
export interface HasChildren {
    /** the children */
    readonly children?: readonly HasChildren[] | undefined;
}
/** the default hierarchy operator */
export declare type DefaultHierarchyOperator = ChildrenHierarchyOperator<HasChildren, ChildrenOperator<HasChildren>>;
/**
 * Constructs a new {@link HierarchyOperator} with default settings. This is
 * bundled as {@link dagHierarchy}.
 */
export declare function hierarchy(...args: never[]): DefaultHierarchyOperator;
/************
 * Stratify *
 ************/
/**
 * The interface for getting the parent ids from data. This must return an
 * array of the ids of every parent of this node. `i` will increment in the
 * order nodes are processed.
 *
 * This can be modified with the {@link StratifyOperator.parentIds} method.
 */
export interface ParentIdsOperator<NodeDatum = never> {
    (d: NodeDatum, i: number): readonly string[] | undefined;
}
/** the node datum of a parent ids operator */
export declare type ParIdsNodeDatum<P extends ParentIdsOperator> = P extends ParentIdsOperator<infer N> ? N : never;
/**
 * The interface for getting the parent ids and link data from the current node
 * data. This must return an array of parent ids coupled with data (i.e. an
 * array of two element arrays of the form ["Parent Id", data]) for the link
 * between this node and the parent id.
 *
 * This can be modified with the {@link StratifyOperator.parentData} method.
 */
export interface ParentDataOperator<NodeDatum = never, LinkDatum = unknown> {
    (d: NodeDatum, i: number): readonly (readonly [string, LinkDatum])[] | undefined;
}
/** the node datum of a parent data operator */
export declare type ParDataNodeDatum<P extends ParentDataOperator> = P extends ParentDataOperator<infer N> ? N : never;
declare type StratifyNodeDatum<Ops extends StratifyOperators> = Ops extends StratifyOperators<infer N> ? N : never;
/**
 * What gets returned by {@link StratifyOperator.parentData}() when {@link StratifyOperator.parentIds} is set.
 */
export interface WrappedParentIdsOperator<ParentIds extends ParentIdsOperator> extends ParentDataOperator<ParIdsNodeDatum<ParentIds>, undefined> {
    /** the wrapped parent ids operator */
    wrapped: ParentIds;
}
/**
 * What gets returned by {@link StratifyOperator.parentIds}() when {@link StratifyOperator.parentData} is set.
 */
export interface WrappedParentDataOperator<ParentData extends ParentDataOperator> extends ParentIdsOperator<ParDataNodeDatum<ParentData>> {
    /** the wrapped parent data operator */
    wrapped: ParentData;
}
/** the operators for the stratify operator */
export interface StratifyOperators<NodeDatum = never, LinkDatum = unknown> {
    /** the id operator */
    id: IdOperator<NodeDatum>;
    /** the parent ids operator */
    parentIds: ParentIdsOperator<NodeDatum>;
    /** the parent data operator */
    parentData: ParentDataOperator<NodeDatum, LinkDatum>;
}
/** the id stratify operator */
export declare type IdsStratifyOperator<Ops extends StratifyOperators, ParentIds extends ParentIdsOperator> = StratifyOperator<undefined, Up<Ops, {
    /** new parent ids */
    parentIds: ParentIds;
    /** new parent data */
    parentData: WrappedParentIdsOperator<ParentIds>;
}>>;
/** a stratify operator with parent data specified */
export declare type DataStratifyOperator<LinkDatum, Ops extends StratifyOperators, ParentData extends ParentDataOperator<never, LinkDatum>> = StratifyOperator<LinkDatum, Up<Ops, {
    /** new parent data */
    parentData: ParentData;
    /** new parent ids */
    parentIds: WrappedParentDataOperator<ParentData>;
}>>;
export interface StratifyOperator<LinkDatum, Ops extends StratifyOperators<never, LinkDatum>> {
    /**
     * The operator that constructs a {@link Dag} from stratified tabularesque
     * data.
     *
     * Create a default operator with {@link stratify}. The accessors for a node's
     * {@link id} or {@link parentIds | parents' ids} can be altered, or {@link
     * parentData} can be specified to specify parent ids and attach data to the
     * edge for that parent.
     *
     * @example
     * ```typescript
     * const data = [{ id: "parent" }, { id: "child", parents: ["parent"] }];
     * const create = stratify().parentIds(({ parents }) => parents);
     * const dag = create(data);
     * ```
     *
     * @example
     * ```json
     * [
     *   {
     *     "id": "Eve"
     *   },
     *   {
     *     "id": "Cain",
     *     "parentIds": ["Eve"]
     *   },
     *   {
     *     "id": "Seth",
     *     "parentIds": ["Eve"]
     *   },
     *   {
     *     "id": "Enos",
     *     "parentIds": ["Seth"]
     *   },
     *   {
     *     "id": "Noam",
     *     "parentIds": ["Seth"]
     *   },
     *   {
     *     "id": "Abel",
     *     "parentIds": ["Eve"]
     *   },
     *   {
     *     "id": "Awan",
     *     "parentIds": ["Eve"]
     *   },
     *   {
     *     "id": "Enoch",
     *     "parentIds": ["Eve"]
     *   },
     *   {
     *     "id": "Azura",
     *     "parentIds": ["Eve"]
     *   }
     * ]
     * ```
     */
    <N extends StratifyNodeDatum<Ops>>(data: readonly N[]): Dag<N, LinkDatum>;
    /**
     * Sets the id accessor to the given {@link IdOperator} and returns a
     * StratifyOperator. The default operator is:
     *
     * ```js
     * function id(d) {
     *   return d.id;
     * }
     * ```
     */
    id<NewId extends IdOperator>(id: NewId): StratifyOperator<LinkDatum, Up<Ops, {
        /** the new id */
        id: NewId;
    }>>;
    /**
     * Gets the current id accessor.
     */
    id(): Ops["id"];
    /**
     * Sets the parentIds accessor to the given {@link ParentIdsOperator}
     * and returns an update operator. The default operator is:
     *
     * ```js
     * function parentIds(d) {
     *   return d.parentIds;
     * }
     * ```
     */
    parentIds<NewParentIds extends ParentIdsOperator>(ids: NewParentIds): IdsStratifyOperator<Ops, NewParentIds>;
    /**
     * Gets the current parent ids accessor.  If {@link parentData} was passed, the
     * returned function will {@link WrappedParentDataOperator | wrap} that to
     * just return the ids.
     */
    parentIds(): Ops["parentIds"];
    /**
     * Sets the parentData accessor to the given {@link ParentDataOperator} and
     * returns an updated operator.
     */
    parentData<NewLinkDatum, NewParentData extends ParentDataOperator<never, NewLinkDatum>>(data: NewParentData & ParentDataOperator<never, NewLinkDatum>): DataStratifyOperator<NewLinkDatum, Ops, NewParentData>;
    /**
     * Gets the current parentData accessor. If {@link parentIds} was passed, this
     * will {@link WrappedParentIdsOperator | wrap} that to just return the ids
     * with `undefined` data.
     */
    parentData(): Ops["parentData"];
    /**
     * Sets whether edges should be reversed to remove cycles
     *
     * If true, after creating the initial "graph" of nodes, an algorithm will
     * run to reverse enough edges to make the resulting data structure a valid
     * dag. It does this by reversing links in the original dag. Note that since
     * the structure is still a dag, old parents could now be positioned as
     * children, and instead the link will have `reversed` set to true. It is up
     * to the user to decide how to handle these links. (default: false)
     */
    decycle(val: boolean): StratifyOperator<LinkDatum, Ops>;
    /** get the current decycle setting */
    decycle(): boolean;
}
/** default interface for types with an id */
export interface HasId {
    /** the id */
    readonly id: string;
}
/** default interface for data types with parent ids */
export interface HasParentIds {
    /** the parent ids */
    readonly parentIds?: readonly string[] | undefined;
}
/** the default stratify operator */
export declare type DefaultStratifyOperator = IdsStratifyOperator<{
    /** the id operator */
    id: IdOperator<HasId>;
    /** the parent id operator */
    parentIds: ParentIdsOperator;
    /** the parent data operator */
    parentData: ParentDataOperator;
}, ParentIdsOperator<HasParentIds>>;
/**
 * Constructs a new {@link StratifyOperator} with the default settings. This is
 * bundled as {@link dagStratify}.
 */
export declare function stratify(...args: never[]): DefaultStratifyOperator;
export {};
