import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { cx } from '@emotion/css';
import { useVirtualizer } from '@tanstack/react-virtual';
import { useCombobox } from 'downshift';
import { useMemo, useId } from 'react';
import '@grafana/data';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import 'micro-memoize';
import '@emotion/react';
import 'tinycolor2';
import '../../utils/skeleton.mjs';
import { t } from '../../utils/i18n.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { AutoSizeInput } from '../Input/AutoSizeInput.mjs';
import { Input } from '../Input/Input.mjs';
import { Portal } from '../Portal/Portal.mjs';
import { ScrollContainer } from '../ScrollContainer/ScrollContainer.mjs';
import { AsyncError, NotFoundError } from './MessageRows.mjs';
import { itemToString } from './filter.mjs';
import { getComboboxStyles, MENU_OPTION_HEIGHT_DESCRIPTION, MENU_OPTION_HEIGHT } from './getComboboxStyles.mjs';
import { useComboboxFloat } from './useComboboxFloat.mjs';
import { useOptions } from './useOptions.mjs';

const noop = () => {
};
const VIRTUAL_OVERSCAN_ITEMS = 4;
const Combobox = (props) => {
  const {
    options: allOptions,
    onChange,
    value: valueProp,
    placeholder: placeholderProp,
    isClearable = false,
    createCustomValue = false,
    id,
    width,
    minWidth,
    maxWidth,
    "aria-labelledby": ariaLabelledBy,
    autoFocus,
    onBlur,
    disabled,
    loading,
    invalid
  } = props;
  const value = typeof valueProp === "object" ? valueProp == null ? undefined : valueProp.value : valueProp;
  const {
    options: filteredOptions,
    updateOptions,
    asyncLoading,
    asyncError
  } = useOptions(props.options, createCustomValue);
  const isAsync = typeof allOptions === "function";
  const selectedItemIndex = useMemo(() => {
    if (isAsync) {
      return null;
    }
    if (valueProp === undefined || valueProp === null) {
      return null;
    }
    const index = allOptions.findIndex((option) => option.value === value);
    if (index === -1) {
      return null;
    }
    return index;
  }, [valueProp, allOptions, value, isAsync]);
  const selectedItem = useMemo(() => {
    if (valueProp === undefined || valueProp === null) {
      return null;
    }
    if (selectedItemIndex !== null && !isAsync) {
      return allOptions[selectedItemIndex];
    }
    return typeof valueProp === "object" ? valueProp : { value: valueProp, label: valueProp.toString() };
  }, [selectedItemIndex, isAsync, valueProp, allOptions]);
  const menuId = `downshift-${useId().replace(/:/g, "--")}-menu`;
  const labelId = `downshift-${useId().replace(/:/g, "--")}-label`;
  const styles = useStyles2(getComboboxStyles);
  const virtualizerOptions = {
    count: filteredOptions.length,
    getScrollElement: () => scrollRef.current,
    estimateSize: (index) => filteredOptions[index].description ? MENU_OPTION_HEIGHT_DESCRIPTION : MENU_OPTION_HEIGHT,
    overscan: VIRTUAL_OVERSCAN_ITEMS
  };
  const rowVirtualizer = useVirtualizer(virtualizerOptions);
  const {
    isOpen,
    highlightedIndex,
    getInputProps,
    getMenuProps,
    getItemProps,
    selectItem
  } = useCombobox({
    menuId,
    labelId,
    inputId: id,
    items: filteredOptions,
    itemToString,
    selectedItem,
    // Don't change downshift state in the onBlahChange handlers. Instead, use the stateReducer to make changes.
    // Downshift calls change handlers on the render after so you can get sync/flickering issues if you change its state
    // in them.
    // Instead, stateReducer is called in the same tick as state changes, before that state is committed and rendered.
    onSelectedItemChange: ({ selectedItem: selectedItem2 }) => {
      onChange(selectedItem2);
    },
    defaultHighlightedIndex: selectedItemIndex != null ? selectedItemIndex : 0,
    scrollIntoView: () => {
    },
    onIsOpenChange: ({ isOpen: isOpen2, inputValue }) => {
      if (isOpen2 && inputValue === "") {
        updateOptions(inputValue);
      }
    },
    onHighlightedIndexChange: ({ highlightedIndex: highlightedIndex2, type }) => {
      if (type !== useCombobox.stateChangeTypes.MenuMouseLeave) {
        rowVirtualizer.scrollToIndex(highlightedIndex2);
      }
    },
    onStateChange: ({ inputValue: newInputValue, type, selectedItem: newSelectedItem }) => {
      switch (type) {
        case useCombobox.stateChangeTypes.InputChange:
          updateOptions(newInputValue != null ? newInputValue : "");
          break;
      }
    },
    stateReducer(state, actionAndChanges) {
      let { changes } = actionAndChanges;
      const menuBeingOpened = state.isOpen === false && changes.isOpen === true;
      const menuBeingClosed = state.isOpen === true && changes.isOpen === false;
      if (menuBeingOpened && changes.inputValue === state.inputValue) {
        changes = {
          ...changes,
          inputValue: ""
        };
      }
      if (menuBeingClosed) {
        if (changes.selectedItem) {
          changes = {
            ...changes,
            inputValue: itemToString(changes.selectedItem)
          };
        } else if (changes.inputValue !== "") {
          changes = {
            ...changes,
            inputValue: ""
          };
        }
      }
      return changes;
    }
  });
  const { inputRef, floatingRef, floatStyles, scrollRef } = useComboboxFloat(filteredOptions, isOpen);
  const isAutoSize = width === "auto";
  const InputComponent = isAutoSize ? AutoSizeInput : Input;
  const suffixIcon = asyncLoading ? "spinner" : (
    // If it's loading, show loading icon. Otherwise, icon indicating menu state
    isOpen ? "search" : "angle-down"
  );
  const placeholder = (isOpen ? itemToString(selectedItem) : null) || placeholderProp;
  return /* @__PURE__ */ jsxs("div", { className: isAutoSize ? styles.addaptToParent : undefined, children: [
    /* @__PURE__ */ jsx(
      InputComponent,
      {
        width: isAutoSize ? undefined : width,
        ...isAutoSize ? { minWidth, maxWidth } : {},
        autoFocus,
        onBlur,
        disabled,
        loading,
        invalid,
        className: styles.input,
        suffix: /* @__PURE__ */ jsxs(Fragment, { children: [
          !!value && value === (selectedItem == null ? undefined : selectedItem.value) && isClearable && /* @__PURE__ */ jsx(
            Icon,
            {
              name: "times",
              className: styles.clear,
              title: t("combobox.clear.title", "Clear value"),
              tabIndex: 0,
              role: "button",
              onClick: () => {
                selectItem(null);
              },
              onKeyDown: (e) => {
                if (e.key === "Enter" || e.key === " ") {
                  selectItem(null);
                }
              }
            }
          ),
          /* @__PURE__ */ jsx(Icon, { name: suffixIcon })
        ] }),
        ...getInputProps({
          ref: inputRef,
          /*  Empty onCall to avoid TS error
           *  See issue here: https://github.com/downshift-js/downshift/issues/718
           *  Downshift repo: https://github.com/downshift-js/downshift/tree/master
           */
          onChange: noop,
          "aria-labelledby": ariaLabelledBy,
          // Label should be handled with the Field component
          placeholder
        })
      }
    ),
    /* @__PURE__ */ jsx(Portal, { children: /* @__PURE__ */ jsx(
      "div",
      {
        className: cx(styles.menu, !isOpen && styles.menuClosed),
        style: {
          ...floatStyles
        },
        ...getMenuProps({
          ref: floatingRef,
          "aria-labelledby": ariaLabelledBy
        }),
        children: isOpen && /* @__PURE__ */ jsxs(ScrollContainer, { showScrollIndicators: true, maxHeight: "inherit", ref: scrollRef, padding: 0.5, children: [
          !asyncError && /* @__PURE__ */ jsx("ul", { style: { height: rowVirtualizer.getTotalSize() }, className: styles.menuUlContainer, children: rowVirtualizer.getVirtualItems().map((virtualRow) => {
            var _a;
            const item = filteredOptions[virtualRow.index];
            return /* @__PURE__ */ jsx(
              "li",
              {
                "data-index": virtualRow.index,
                className: cx(
                  styles.optionBasic,
                  styles.option,
                  selectedItem && item.value === selectedItem.value && styles.optionSelected,
                  highlightedIndex === virtualRow.index && styles.optionFocused
                ),
                style: {
                  height: virtualRow.size,
                  transform: `translateY(${virtualRow.start}px)`
                },
                ...getItemProps({
                  item,
                  index: virtualRow.index
                }),
                children: /* @__PURE__ */ jsxs("div", { className: styles.optionBody, children: [
                  /* @__PURE__ */ jsx("span", { className: styles.optionLabel, children: (_a = item.label) != null ? _a : item.value }),
                  item.description && /* @__PURE__ */ jsx("span", { className: styles.optionDescription, children: item.description })
                ] })
              },
              `${item.value}-${virtualRow.index}`
            );
          }) }),
          /* @__PURE__ */ jsxs("div", { "aria-live": "polite", children: [
            asyncError && /* @__PURE__ */ jsx(AsyncError, {}),
            filteredOptions.length === 0 && !asyncError && /* @__PURE__ */ jsx(NotFoundError, {})
          ] })
        ] })
      }
    ) })
  ] });
};

export { Combobox, VIRTUAL_OVERSCAN_ITEMS };
//# sourceMappingURL=Combobox.mjs.map
