/*
 * livepatch_bsc1235062
 *
 * Fix for CVE-2024-56605, bsc#1235062
 *
 *  Upstream commit:
 *  7c4f78cdb8e7 ("Bluetooth: L2CAP: do not leave dangling sk pointer on error in l2cap_sock_create()")
 *
 *  SLE12-SP5 commit:
 *  6ac1393bc69ad2def6d67511ac2152fe4aa22e58
 *
 *  SLE15-SP3 commit:
 *  20f98a1dccdbe1cbec4572120239070d8d0bcede
 *
 *  SLE15-SP4 and -SP5 commit:
 *  c461209d99bef5301941c51b29154648b21d3447
 *
 *  SLE15-SP6 commit:
 *  d023f1f3e97051defc7bdb07558756ae07c2736f
 *
 *  SLE MICRO-6-0 commit:
 *  d023f1f3e97051defc7bdb07558756ae07c2736f
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo Mezzela <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_BT)

#if !IS_MODULE(CONFIG_BT)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from net/bluetooth/l2cap_sock.c */
#include <linux/module.h>
#include <linux/export.h>
#include <linux/filter.h>
#include <linux/sched/signal.h>

#include <net/bluetooth/bluetooth.h>

#include <net/bluetooth/l2cap.h>

extern struct bt_sock_list l2cap_sk_list;

extern const struct proto_ops l2cap_sock_ops;
extern void l2cap_sock_init(struct sock *sk, struct sock *parent);
static struct sock *klpp_l2cap_sock_alloc(struct net *net, struct socket *sock,
				     int proto, gfp_t prio, int kern);

struct l2cap_chan *klpp_l2cap_sock_new_connection_cb(struct l2cap_chan *chan)
{
	struct sock *sk, *parent = chan->data;

	lock_sock(parent);

	/* Check for backlog size */
	if (sk_acceptq_is_full(parent)) {
		BT_DBG("backlog full %d", parent->sk_ack_backlog);
		release_sock(parent);
		return NULL;
	}

	sk = klpp_l2cap_sock_alloc(sock_net(parent), NULL, BTPROTO_L2CAP,
			      GFP_ATOMIC, 0);
	if (!sk) {
		release_sock(parent);
		return NULL;
        }

	bt_sock_reclassify_lock(sk, BTPROTO_L2CAP);

	l2cap_sock_init(sk, parent);

	bt_accept_enqueue(parent, sk, false);

	release_sock(parent);

	return l2cap_pi(sk)->chan;
}

extern void l2cap_sock_destruct(struct sock *sk);

void l2cap_sock_init(struct sock *sk, struct sock *parent);

extern struct proto l2cap_proto;

static struct sock *klpp_l2cap_sock_alloc(struct net *net, struct socket *sock,
				     int proto, gfp_t prio, int kern)
{
	struct sock *sk;
	struct l2cap_chan *chan;

	sk = sk_alloc(net, PF_BLUETOOTH, prio, &l2cap_proto, kern);
	if (!sk)
		return NULL;

	sock_init_data(sock, sk);
	INIT_LIST_HEAD(&bt_sk(sk)->accept_q);

	sk->sk_destruct = l2cap_sock_destruct;
	sk->sk_sndtimeo = L2CAP_CONN_TIMEOUT;

	sock_reset_flag(sk, SOCK_ZAPPED);

	sk->sk_protocol = proto;
	sk->sk_state = BT_OPEN;

	chan = l2cap_chan_create();
	if (!chan) {
		sk_free(sk);
		sock->sk = NULL;
		return NULL;
	}

	l2cap_chan_hold(chan);

	l2cap_pi(sk)->chan = chan;

	return sk;
}

int klpp_l2cap_sock_create(struct net *net, struct socket *sock, int protocol,
			     int kern)
{
	struct sock *sk;

	BT_DBG("sock %p", sock);

	sock->state = SS_UNCONNECTED;

	if (sock->type != SOCK_SEQPACKET && sock->type != SOCK_STREAM &&
	    sock->type != SOCK_DGRAM && sock->type != SOCK_RAW)
		return -ESOCKTNOSUPPORT;

	if (sock->type == SOCK_RAW && !kern && !capable(CAP_NET_RAW))
		return -EPERM;

	sock->ops = &l2cap_sock_ops;

	sk = klpp_l2cap_sock_alloc(net, sock, protocol, GFP_ATOMIC, kern);
	if (!sk)
		return -ENOMEM;

	l2cap_sock_init(sk, NULL);
	bt_sock_link(&l2cap_sk_list, sk);
	return 0;
}

extern const struct proto_ops l2cap_sock_ops;


#include "livepatch_bsc1235062.h"

#include <linux/livepatch.h>

extern typeof(bt_accept_enqueue) bt_accept_enqueue
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, bt_accept_enqueue);
extern typeof(bt_sock_link) bt_sock_link
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, bt_sock_link);
extern typeof(bt_sock_reclassify_lock) bt_sock_reclassify_lock
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, bt_sock_reclassify_lock);
extern typeof(l2cap_chan_create) l2cap_chan_create
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_chan_create);
extern typeof(l2cap_chan_hold) l2cap_chan_hold
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_chan_hold);
extern typeof(l2cap_proto) l2cap_proto
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_proto);
extern typeof(l2cap_sk_list) l2cap_sk_list
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_sk_list);
extern typeof(l2cap_sock_destruct) l2cap_sock_destruct
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_sock_destruct);
extern typeof(l2cap_sock_init) l2cap_sock_init
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_sock_init);
extern typeof(l2cap_sock_ops) l2cap_sock_ops
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, l2cap_sock_ops);

#endif /* IS_ENABLED(CONFIG_BT) */
