/*
 * livepatch_bsc1231676
 *
 * Fix for CVE-2024-47674, bsc#1231676
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo Mezzela <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from mm/memory.c */
#include <linux/kernel_stat.h>
#include <linux/mm.h>

/* klp-ccp: from mm/memory.c */
#include <linux/mm_inline.h>
#include <linux/sched/mm.h>
#include <linux/sched/coredump.h>

#include <linux/sched/task.h>

#include <linux/swap.h>
#include <linux/highmem.h>
#include <linux/pagemap.h>
#include <linux/memremap.h>
#include <linux/kmsan.h>

/* klp-ccp: from include/linux/rmap.h */
#define _LINUX_RMAP_H

/* klp-ccp: from mm/memory.c */
#include <linux/rmap.h>
#include <linux/export.h>

#include <linux/init.h>

#include <linux/writeback.h>
#include <linux/memcontrol.h>
#include <linux/mmu_notifier.h>
#include <linux/swapops.h>
#include <linux/elf.h>
#include <linux/gfp.h>

#include <linux/string.h>

#include <linux/userfaultfd_k.h>

#include <linux/numa.h>

/* klp-ccp: from include/linux/ptrace.h */
#define _LINUX_PTRACE_H

/* klp-ccp: from mm/memory.c */
#include <linux/ptrace.h>

#include <linux/uaccess.h>

#include <asm/tlbflush.h>

/* klp-ccp: from mm/internal.h */
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/rmap.h>
#include <linux/tracepoint-defs.h>

/* klp-ccp: from mm/swap.h */
#ifdef CONFIG_SWAP
#include <linux/blk_types.h> /* for bio_end_io_t */

#else /* CONFIG_SWAP */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_SWAP */


#ifndef __HAVE_ARCH_PFN_MODIFY_ALLOWED

static inline bool klpr_pfn_modify_allowed(unsigned long pfn, pgprot_t prot)
{
	return true;
}
#define pfn_modify_allowed klpr_pfn_modify_allowed

#endif /* !_HAVE_ARCH_PFN_MODIFY_ALLOWED */

/* klp-ccp: from include/linux/mm.h */
#ifdef __PAGETABLE_P4D_FOLDED
static inline int klpr___p4d_alloc(struct mm_struct *mm, pgd_t *pgd,
						unsigned long address)
{
	return 0;
}
#define __p4d_alloc klpr___p4d_alloc
#endif

/* klp-ccp: from mm/memory.c */
int __pte_alloc(struct mm_struct *mm, pmd_t *pmd);

void zap_page_range_single(struct vm_area_struct *vma, unsigned long address,
		unsigned long size, struct zap_details *details);

static int remap_pte_range(struct mm_struct *mm, pmd_t *pmd,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pte_t *pte, *mapped_pte;
	spinlock_t *ptl;
	int err = 0;

	mapped_pte = pte = pte_alloc_map_lock(mm, pmd, addr, &ptl);
	if (!pte)
		return -ENOMEM;
	arch_enter_lazy_mmu_mode();
	do {
		BUG_ON(!pte_none(*pte));
		if (!pfn_modify_allowed(pfn, prot)) {
			err = -EACCES;
			break;
		}
		set_pte_at(mm, addr, pte, pte_mkspecial(pfn_pte(pfn, prot)));
		pfn++;
	} while (pte++, addr += PAGE_SIZE, addr != end);
	arch_leave_lazy_mmu_mode();
	pte_unmap_unlock(mapped_pte, ptl);
	return err;
}

static inline int remap_pmd_range(struct mm_struct *mm, pud_t *pud,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pmd_t *pmd;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	pmd = pmd_alloc(mm, pud, addr);
	if (!pmd)
		return -ENOMEM;
	VM_BUG_ON(pmd_trans_huge(*pmd));
	do {
		next = pmd_addr_end(addr, end);
		err = remap_pte_range(mm, pmd, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pmd++, addr = next, addr != end);
	return 0;
}

static inline int remap_pud_range(struct mm_struct *mm, p4d_t *p4d,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	pud_t *pud;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	pud = pud_alloc(mm, p4d, addr);
	if (!pud)
		return -ENOMEM;
	do {
		next = pud_addr_end(addr, end);
		err = remap_pmd_range(mm, pud, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pud++, addr = next, addr != end);
	return 0;
}

static inline int remap_p4d_range(struct mm_struct *mm, pgd_t *pgd,
			unsigned long addr, unsigned long end,
			unsigned long pfn, pgprot_t prot)
{
	p4d_t *p4d;
	unsigned long next;
	int err;

	pfn -= addr >> PAGE_SHIFT;
	p4d = p4d_alloc(mm, pgd, addr);
	if (!p4d)
		return -ENOMEM;
	do {
		next = p4d_addr_end(addr, end);
		err = remap_pud_range(mm, p4d, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (p4d++, addr = next, addr != end);
	return 0;
}

static int remap_pfn_range_internal(struct vm_area_struct *vma, unsigned long addr,
		unsigned long pfn, unsigned long size, pgprot_t prot)
{
	pgd_t *pgd;
	unsigned long next;
	unsigned long end = addr + PAGE_ALIGN(size);
	struct mm_struct *mm = vma->vm_mm;
	int err;

	if (WARN_ON_ONCE(!PAGE_ALIGNED(addr)))
		return -EINVAL;

	/*
	 * Physically remapped pages are special. Tell the
	 * rest of the world about it:
	 *   VM_IO tells people not to look at these pages
	 *	(accesses can have side effects).
	 *   VM_PFNMAP tells the core MM that the base pages are just
	 *	raw PFN mappings, and do not have a "struct page" associated
	 *	with them.
	 *   VM_DONTEXPAND
	 *      Disable vma merging and expanding with mremap().
	 *   VM_DONTDUMP
	 *      Omit vma from core dump, even when VM_IO turned off.
	 *
	 * There's a horrible special case to handle copy-on-write
	 * behaviour that some programs depend on. We mark the "original"
	 * un-COW'ed pages by matching them up with "vma->vm_pgoff".
	 * See vm_normal_page() for details.
	 */
	if (is_cow_mapping(vma->vm_flags)) {
		if (addr != vma->vm_start || end != vma->vm_end)
			return -EINVAL;
		vma->vm_pgoff = pfn;
	}

	vm_flags_set(vma, VM_IO | VM_PFNMAP | VM_DONTEXPAND | VM_DONTDUMP);

	BUG_ON(addr >= end);
	pfn -= addr >> PAGE_SHIFT;
	pgd = pgd_offset(mm, addr);
	flush_cache_range(vma, addr, end);
	do {
		next = pgd_addr_end(addr, end);
		err = remap_p4d_range(mm, pgd, addr, next,
				pfn + (addr >> PAGE_SHIFT), prot);
		if (err)
			return err;
	} while (pgd++, addr = next, addr != end);

	return 0;
}

int klpp_remap_pfn_range_notrack(struct vm_area_struct *vma, unsigned long addr,
		unsigned long pfn, unsigned long size, pgprot_t prot)
{
	int error = remap_pfn_range_internal(vma, addr, pfn, size, prot);

	if (!error)
		return 0;

	/*
	 * A partial pfn range mapping is dangerous: it does not
	 * maintain page reference counts, and callers may free
	 * pages due to the error. So zap it early.
	 */
	zap_page_range_single(vma, addr, size, NULL);
	return error;
}

#include "livepatch_bsc1231676.h"

#include <linux/livepatch.h>

#ifndef __PAGETABLE_P4D_FOLDED
extern typeof(__p4d_alloc) __p4d_alloc
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, __p4d_alloc);
#endif
extern typeof(__pmd_alloc) __pmd_alloc
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, __pmd_alloc);
extern typeof(__pte_alloc) __pte_alloc
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, __pte_alloc);
extern typeof(__pud_alloc) __pud_alloc
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, __pud_alloc);
#ifdef __HAVE_ARCH_PFN_MODIFY_ALLOWED
extern typeof(pfn_modify_allowed) pfn_modify_allowed
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, pfn_modify_allowed);
#endif
extern typeof(zap_page_range_single) zap_page_range_single
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, zap_page_range_single);
#if defined(CONFIG_S390)
extern typeof(ptep_set_pte_at) ptep_set_pte_at
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, ptep_set_pte_at);
#endif
#if defined(CONFIG_PPC64)
extern typeof(__flush_tlb_pending) __flush_tlb_pending
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, __flush_tlb_pending);
extern typeof(ppc64_tlb_batch) ppc64_tlb_batch
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, ppc64_tlb_batch);
extern typeof(set_pte_at) set_pte_at
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, set_pte_at);
extern typeof(pmd_page) pmd_page
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, pmd_page);
#endif
