/*
 * livepatch_bsc1233227
 *
 * Fix for CVE-2024-50250, bsc#1233227
 *
 *  Upstream commit:
 *  95472274b6fe ("fsdax: remove zeroing code from dax_unshare_iter")
 *  50793801fc7f ("fsdax: dax_unshare_iter needs to copy entire blocks")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 commit:
 *  Not affected
 *
 *  SLE15-SP4 and -SP5 commit:
 *  Not affected
 *
 *  SLE15-SP6 commit:
 *  4172bd79c6e97765a47d4026282b188049a811f5
 *  94457ab1d0e6501294923c9cadbab843d3d4824c
 *
 *  SLE MICRO-6-0 commit:
 *  4172bd79c6e97765a47d4026282b188049a811f5
 *  94457ab1d0e6501294923c9cadbab843d3d4824c
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Lidong Zhong <lidong.zhong@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


#define __KERNEL__ 1

/* klp-ccp: from fs/dax.c */
#include <linux/atomic.h>
#include <linux/blkdev.h>
#include <linux/buffer_head.h>
#include <linux/dax.h>

/* klp-ccp: from fs/dax.c */
#include <linux/fs.h>
#include <linux/highmem.h>

#include <linux/mm.h>
#include <linux/mutex.h>

#include <linux/sched.h>
#include <linux/sched/signal.h>
#include <linux/uio.h>
#include <linux/vmstat.h>

#include <linux/sizes.h>
#include <linux/mmu_notifier.h>
#include <linux/iomap.h>

/* klp-ccp: from fs/dax.c */
extern int dax_iomap_direct_access(const struct iomap *iomap, loff_t pos,
		size_t size, void **kaddr, pfn_t *pfnp);

static s64 klpp_dax_unshare_iter(struct iomap_iter *iter)
{
	struct iomap *iomap = &iter->iomap;
	const struct iomap *srcmap = iomap_iter_srcmap(iter);
	loff_t copy_pos = iter->pos;
	u64 copy_len = iomap_length(iter);
	u32 mod;
	int id = 0;
	s64 ret = 0;
	void *daddr = NULL, *saddr = NULL;

	/* don't bother with blocks that are not shared to start with */
	if (!(iomap->flags & IOMAP_F_SHARED))
		return iomap_length(iter);

	/*
	 * Extend the file range to be aligned to fsblock/pagesize, because
	 * we need to copy entire blocks, not just the byte range specified.
	 * Invalidate the mapping because we're about to CoW.
	 */
	mod = offset_in_page(copy_pos);
	if (mod) {
		copy_len += mod;
		copy_pos -= mod;
	}

	mod = offset_in_page(copy_pos + copy_len);
	if (mod)
		copy_len += PAGE_SIZE - mod;

	invalidate_inode_pages2_range(iter->inode->i_mapping,
				      copy_pos >> PAGE_SHIFT,
				      (copy_pos + copy_len - 1) >> PAGE_SHIFT);

	id = dax_read_lock();
	ret = dax_iomap_direct_access(iomap, copy_pos, copy_len, &daddr, NULL);
	if (ret < 0)
		goto out_unlock;

	ret = dax_iomap_direct_access(srcmap, copy_pos, copy_len, &saddr, NULL);
	if (ret < 0)
		goto out_unlock;

	if (copy_mc_to_kernel(daddr, saddr, copy_len) == 0)
		ret = iomap_length(iter);
	else
		ret = -EIO;

out_unlock:
	dax_read_unlock(id);
	return dax_mem2blk_err(ret);
}

int klpp_dax_file_unshare(struct inode *inode, loff_t pos, loff_t len,
		const struct iomap_ops *ops)
{
	struct iomap_iter iter = {
		.inode		= inode,
		.pos		= pos,
		.len		= len,
		.flags		= IOMAP_WRITE | IOMAP_UNSHARE | IOMAP_DAX,
	};
	int ret;

	while ((ret = iomap_iter(&iter, ops)) > 0)
		iter.processed = klpp_dax_unshare_iter(&iter);
	return ret;
}

typeof(klpp_dax_file_unshare) klpp_dax_file_unshare;


#include "livepatch_bsc1233227.h"

#include <linux/livepatch.h>

extern typeof(dax_iomap_direct_access) dax_iomap_direct_access
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, dax_iomap_direct_access);
extern typeof(iomap_iter) iomap_iter
	 KLP_RELOC_SYMBOL(vmlinux, vmlinux, iomap_iter);
