/*
 *	string related utilities
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *		Clemens Famulla-Conrad
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#ifndef __NETCONTROL_SUTILS_H
#define __NETCONTROL_SUTILS_H
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <sys/types.h>
#include <stdarg.h>


#define NC_STRINGBUF_INIT		{ .len = 0, .string = NULL }
#define NC_STRING_ARRAY_INIT            { .count = 0, .data = NULL }
#define NC_VAR_ARRAY_INIT               { .count = 0, .data = NULL }


typedef struct nc_stringbuf {
	size_t		len;
	char *		string;
} nc_stringbuf_t;

typedef struct nc_string_array {
	unsigned int	count;
	char **		data;
} nc_string_array_t;


typedef struct nc_intmap {
	const char *	name;
	unsigned int	value;
} nc_intmap_t;

typedef struct nc_variable {
	char *		name;
	char *		value;
} nc_var_t;

typedef struct nc_var_array {
	unsigned int	count;
	nc_var_t *	data;
} nc_var_array_t;

int		nc_string_dup(char **, const char *);
int		nc_string_set(char **, const char *, size_t);
void		nc_string_free(char **);
char *		nc_string_strip_spaces(char *str);

size_t		nc_string_len(const char *);
int /* bool */	nc_string_empty(const char *);
int /* bool */	nc_string_eq(const char *a, const char *b);
int		nc_string_prefix_eq(const char *prefix, const char *str);
int		nc_string_suffix_eq(const char *suffix, const char *str);

void		nc_stringbuf_init(nc_stringbuf_t *);
void		nc_stringbuf_destroy(nc_stringbuf_t *);
int		nc_stringbuf_empty(const nc_stringbuf_t *);
void		nc_stringbuf_clear(nc_stringbuf_t *);
int		nc_stringbuf_grow(nc_stringbuf_t *, size_t);
int		nc_stringbuf_puts_n(nc_stringbuf_t *, const char *, size_t);
int		nc_stringbuf_puts(nc_stringbuf_t *, const char *);
int		nc_stringbuf_putc(nc_stringbuf_t *, char);
int		nc_stringbuf_printf(nc_stringbuf_t *, const char *, ...);
int		nc_stringbuf_vprintf(nc_stringbuf_t *, const char *, va_list ap);
void		nc_stringbuf_move(nc_stringbuf_t *dst, nc_stringbuf_t *src);
void		nc_stringbuf_trim_empty_lines(nc_stringbuf_t *);
const char *	nc_stringbuf_join(nc_stringbuf_t *, const nc_string_array_t *, const char *);

void		nc_string_array_init(nc_string_array_t *);
void		nc_string_array_destroy(nc_string_array_t *);
int		nc_string_array_copy(nc_string_array_t *dst, const nc_string_array_t *src);
void		nc_string_array_move(nc_string_array_t *dst, nc_string_array_t *src);
int		nc_string_array_append(nc_string_array_t *, const char *);
int		nc_string_array_insert(nc_string_array_t *, unsigned int, const char *);
int		nc_string_array_remove_index(nc_string_array_t *, unsigned int);
int		nc_string_array_remove_match(nc_string_array_t *, const char *, unsigned int);
int		nc_string_array_set(nc_string_array_t *, unsigned int, const char *);
int		nc_string_array_get(nc_string_array_t *, unsigned int, char **);
int		nc_string_array_index(const nc_string_array_t *, const char *);
int /* bool */	nc_string_array_is_uniq(const nc_string_array_t *);

void		nc_var_array_init(nc_var_array_t *);
void		nc_var_array_destroy(nc_var_array_t *);
int		nc_var_array_copy(nc_var_array_t *, const nc_var_array_t *);
nc_var_t *	nc_var_array_get(const nc_var_array_t *, const char *name);
int		nc_var_array_set(nc_var_array_t *, const char *name, const char *value);
int		nc_var_array_set_uint(nc_var_array_t *, const char *, unsigned int);
int		nc_var_array_set_ulong(nc_var_array_t *, const char *, unsigned long);
int		nc_var_array_set_double(nc_var_array_t *, const char *, double);

int             nc_parse_int(const char *input, int *result, int base);
int             nc_parse_uint(const char *input, unsigned int *result, int base);
int             nc_parse_long(const char *input, long *result, int base);
int             nc_parse_ulong(const char *input, unsigned long *result, int base);

int		nc_parse_int_mapped(const char *, const struct nc_intmap *,
			                    unsigned int *);
const char *	nc_format_int_mapped(unsigned int, const nc_intmap_t *);

int		nc_parse_hex(const char *string, unsigned char *data,
			             unsigned int datasize);
const char *	nc_format_hex(const unsigned char *data, unsigned int datalen,
		              char *namebuf, size_t namelen);

int		nc_parse_bool(const char *input, int *result);

#endif /* __NETCONTROL_SUTILS_H */
