/*
 *	Routines for bonding interface handling
 *
 *	Copyright (C) 2009, 2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if_arp.h>
#include <assert.h>

#include <sutils.h>
#include <sysfs.h>
#include <logging.h>
#include <bonding.h>


/*
 * Add a slave device to the bond
 */
int
nc_bonding_add_slave(nc_bonding_t *bonding, const char *ifname)
{
	if(nc_string_array_index(&bonding->slave_names, ifname) < 0)
		return nc_string_array_append(&bonding->slave_names, ifname);
	return -1;
}

/*
 * Bind the bonding device
 * Look up the interface config of all slaves, and record them
 * in the bonding info.
 */
int
nc_bonding_bind(nc_interface_t *ifp, nc_handle_t *nh, nc_var_array_t *errs)
{
	nc_stringbuf_t err = NC_STRINGBUF_INIT;
	unsigned int i = 0;

	if(!ifp || !ifp->name || !nh || !errs)
		return -1;
	if(!ifp->bonding)
		return -1;
	if(ifp->type != NC_IFTYPE_BOND)
		return -1;

	for (i = 0; i < ifp->bonding->slave_names.count; ++i) {
		const char *ifname = ifp->bonding->slave_names.data[i];
		nc_interface_t *slave;

		slave = nc_interface_by_name(nh, ifname);
		if (slave == NULL) {
			nc_trace("no interface for bonding %s slave %s -- missed config?",
				ifp->name, ifname);
			continue;
		}
		if(slave->invalid) {
			ifp->invalid = 1;
			nc_stringbuf_printf(&err,
				"Bonding refers invalid slave %s",
				ifname);
			nc_var_array_set(errs, ifp->name, err.string);
			nc_stringbuf_destroy(&err);
			return 1;
		}

		if(!nc_string_len(slave->parent)) {
			nc_string_dup(&slave->parent, ifp->name);
			continue;
		}

		if(!nc_string_eq(slave->parent, ifp->name)) {
			ifp->invalid = 1;
			nc_stringbuf_printf(&err,
				"Bonding slave %s refers to %s as parent",
				ifname, slave->parent);
			nc_var_array_set(errs, ifp->name, err.string);
			nc_stringbuf_destroy(&err);
			return 1;
		} else {
			nc_trace("Bonding %s slave %s parent is already set",
				ifp->name, slave->name);
		}
	}

	if (ifp->bonding->primary_slave) {
		const char *ifname = ifp->bonding->primary_slave;
		nc_interface_t *slave = NULL;
		int pos;

		pos = nc_string_array_index(&ifp->bonding->slave_names, ifname);
		if(pos == -1) {
			ifp->invalid = 1;
			nc_stringbuf_printf(&err,
				"Bonding primary slave %s refers a not existing slave",
				ifname);
			nc_var_array_set(errs, ifp->name, err.string);
			nc_stringbuf_destroy(&err);
			return 1;
		} else {
			slave = nc_interface_by_name(nh, ifname);
			if (slave == NULL) {
				nc_trace("no interface for bonding %s primary %s -- missed config?",
					ifp->name, ifname);
			}
		}
	} else {
		/* first slave will be primary */
	}

	return 0;
}

/*
 * Clone the bonding config of a device
 */
nc_bonding_t *
nc_bonding_clone(const nc_bonding_t *src)
{
	nc_bonding_t *dst;

	dst = calloc(1, sizeof(nc_bonding_t));
	if (!dst)
		return NULL;

#define xstrdup(str)	((str) ? strdup(str) : NULL)
#define C(member)	dst->member = src->member
#define D(member, clone_fn)	\
		do { \
			if (src->member) { \
				dst->member = clone_fn(src->member); \
				if (!dst->member) \
					goto failed; \
			} \
		} while (0)
	D(module_opts, xstrdup);
	C(mode);
	C(monitoring);
	C(arpmon);
	C(miimon);
	D(primary_slave, xstrdup);

	if (nc_var_array_copy(&dst->extra_options, &src->extra_options) < 0)
		goto failed;

	if (nc_string_array_copy(&dst->slave_names, &src->slave_names) < 0)
		goto failed;
#undef C
#undef D
#undef xstrdup

	return dst;

failed:
	nc_error("Error clonding bonding configuration");
	nc_bonding_free(dst);
	return NULL;
}

/*
 * Reinitialize the bonding configuration
 *
 * - The default bonding mode is balance-rr
 * - The default monitoring mode is ARP (unless miimon=... is set)
 */
static void
nc_bonding_clear_options(nc_bonding_t *bonding)
{
	bonding->mode = NC_BOND_MODE_BALANCE_RR;
	bonding->monitoring = NC_BOND_MONITOR_ARP;
	bonding->miimon.carrier_detect = NC_BOND_CARRIER_DETECT_NETIF;

	bonding->lacp_rate = NC_BOND_LACP_RATE_SLOW;
	bonding->xmit_hash_policy = NC_BOND_XMIT_HASH_POLICY_LAYER2;

	nc_string_array_destroy(&bonding->arpmon.targets);
	nc_var_array_destroy(&bonding->extra_options);
}
static void
nc_bonding_clear_slaves(nc_bonding_t *bonding)
{
	if(bonding->primary_slave) {
		nc_string_free(&bonding->primary_slave);
	}
	bonding->slave_type = NC_IFTYPE_UNKNOWN;
	nc_string_array_destroy(&bonding->slave_names);
}
static void
nc_bonding_clear(nc_bonding_t *bonding)
{
	nc_bonding_clear_options(bonding);
	nc_bonding_clear_slaves(bonding);
}

nc_bonding_t *
nc_bonding_new(void)
{
	nc_bonding_t *bonding;

	bonding = calloc(1, sizeof(*bonding));
	assert(bonding);
	if(bonding) {
		nc_bonding_clear(bonding);
	}
	return bonding;
}

/*
 * Free bonding configuration
 */
void
nc_bonding_free(nc_bonding_t *bonding)
{
	nc_bonding_clear(bonding);
	free(bonding->module_opts);
	free(bonding);
}

/*
 * Set the bonding mode, using the strings supported by the
 * module options
 */
static nc_intmap_t	__bonding_mode[] = {
	{ "balance-rr",		NC_BOND_MODE_BALANCE_RR },
	{ "active-backup",	NC_BOND_MODE_ACTIVE_BACKUP },
	{ "balance-xor",	NC_BOND_MODE_BALANCE_XOR },
	{ "broadcast",		NC_BOND_MODE_BROADCAST },
	{ "802.3ad",		NC_BOND_MODE_802_3AD },
	{ "balance-tlb",	NC_BOND_MODE_BALANCE_TLB },
	{ "balance-alb",	NC_BOND_MODE_BALANCE_ALB },
	{ NULL,			0 }
};

int
__nc_bonding_set_module_option_mode(nc_bonding_t *bonding, char *value)
{
	/* When we parse /sys/net/class/<ifname>/bonding/mode, we end up
	 * with "balance-rr 0" or similar; strip off the int value */
	value[strcspn(value, " \t\n")] = '\0';
	return nc_parse_int_mapped(value, __bonding_mode, &bonding->mode);
}

const char *
__nc_bonding_get_module_option_mode(const nc_bonding_t *bonding)
{
	return nc_format_int_mapped(bonding->mode, __bonding_mode);
}

/*
 * Set the validation mode of ARP probes.
 */
static nc_intmap_t	__arp_validate[] = {
	{ "none",		NC_BOND_VALIDATE_NONE },
	{ "active",		NC_BOND_VALIDATE_ACTIVE },
	{ "backup",		NC_BOND_VALIDATE_BACKUP },
	{ "all",		NC_BOND_VALIDATE_ALL },
	{ NULL,			0 }
};

int
__nc_bonding_set_module_option_arp_validate(nc_bonding_t *bonding, char *value)
{
	/* When we parse /sys/net/class/<ifname>/bonding/arp_validate, we end up
	 * with "none 0" or similar; strip off the int value */
	value[strcspn(value, " \t\n")] = '\0';
	return nc_parse_int_mapped(value, __arp_validate, &bonding->arpmon.validate);
}

const char *
__nc_bonding_get_module_option_arp_validate(const nc_bonding_t *bonding)
{
	return nc_format_int_mapped(bonding->arpmon.validate, __arp_validate);
}

/*
 * Set/get lacp_rate option
 */
static nc_intmap_t	__lacp_rate[] = {
	{ "slow",		NC_BOND_LACP_RATE_SLOW	},
	{ "fast",		NC_BOND_LACP_RATE_FAST	},
	{ NULL,			0			},
};
int
__nc_bonding_set_module_option_lacp_rate(nc_bonding_t *bonding, char *value)
{
	/* When we parse /sys/net/class/<ifname>/bonding/lacp_rate, we end up
	 * with "slow 0" or "fast 1"; strip off the int value */
	value[strcspn(value, " \t\n")] = '\0';
	return nc_parse_int_mapped(value, __lacp_rate, &bonding->lacp_rate);
}
const char *
__nc_bonding_get_module_option_lacp_rate(const nc_bonding_t *bonding)
{
	return nc_format_int_mapped(bonding->lacp_rate, __lacp_rate);
}

/*
 * Set/get xmit_hash_policy option
 */
static nc_intmap_t	__xmit_hash_policy[] = {
	{ "layer2",		NC_BOND_XMIT_HASH_POLICY_LAYER2		},
	{ "layer2+3",		NC_BOND_XMIT_HASH_POLICY_LAYER23	},
	{ "layer3+4",		NC_BOND_XMIT_HASH_POLICY_LAYER34	},
	{ NULL,			0					},
};
int
__nc_bonding_set_module_option_xmit_hash_policy(nc_bonding_t *bonding, char *value)
{
	/* When we parse /sys/net/class/<ifname>/bonding/lacp_rate, we end up
	 * with "slow 0" or "fast 1"; strip off the int value */
	value[strcspn(value, " \t\n")] = '\0';
	return nc_parse_int_mapped(value, __xmit_hash_policy, &bonding->xmit_hash_policy);
}
const char *
__nc_bonding_get_module_option_xmit_hash_policy(const nc_bonding_t *bonding)
{
	return nc_format_int_mapped(bonding->xmit_hash_policy, __xmit_hash_policy);
}


/*
 * Set one bonding module option/attribute
 */
static int
nc_bonding_parse_module_attribute(nc_bonding_t *bonding, const char *attr, char *value)
{
	if (!strcmp(attr, "mode")) {
		if (__nc_bonding_set_module_option_mode(bonding, value) < 0)
			return -1;
	} else if (!strcmp(attr, "miimon")) {
		if (nc_parse_uint(value, &bonding->miimon.frequency, 10) < 0)
			return -1;
		if (bonding->miimon.frequency != 0)
			bonding->monitoring = NC_BOND_MONITOR_MII;
		else
			bonding->monitoring = NC_BOND_MONITOR_ARP;
	} else if (!strcmp(attr, "updelay")) {
		if (nc_parse_uint(value, &bonding->miimon.updelay, 10) < 0)
			return -1;
	} else if (!strcmp(attr, "downdelay")) {
		if (nc_parse_uint(value, &bonding->miimon.downdelay, 10) < 0)
			return -1;
	} else if (!strcmp(attr, "use_carrier")) {
		if (nc_parse_uint(value, &bonding->miimon.carrier_detect, 10) < 0)
			return -1;
	} else if (!strcmp(attr, "arp_validate")) {
		if (__nc_bonding_set_module_option_arp_validate(bonding, value) < 0)
			return -1;
	} else if (!strcmp(attr, "arp_interval")) {
		if (nc_parse_uint(value, &bonding->arpmon.interval, 10) < 0)
			return -1;
	} else if (!strcmp(attr, "arp_ip_target")) {
		char *s, *saveptr = NULL;

		for (s = strtok_r(value, ",", &saveptr); s; s = strtok_r(NULL, ",", &saveptr)) {
			struct in_addr dummy;

			if (inet_aton(value, &dummy) == 0)
				return -1;
			nc_string_array_append(&bonding->arpmon.targets, s);
		}
	} else if (!strcmp(attr, "primary")) {
		nc_string_dup(&bonding->primary_slave, value);
	} else if (!strcmp(attr, "lacp_rate")) {
		if(__nc_bonding_set_module_option_lacp_rate(bonding, value) < 0)
			return -1;
	} else if (!strcmp(attr, "xmit_hash_policy")) {
		if(__nc_bonding_set_module_option_xmit_hash_policy(bonding, value) < 0)
			return -1;
	} else {
		if(nc_var_array_set(&bonding->extra_options, attr, value) < 0)
			return -1;
		return -2;
	}

	return 0;
}

/*
 * Get one bonding module option/attribute
 */
static int
nc_bonding_format_module_attribute(const nc_bonding_t *bonding, const char *attr,
					nc_stringbuf_t *buffer)
{
	if (!strcmp(attr, "mode")) {
		nc_stringbuf_puts(buffer, __nc_bonding_get_module_option_mode(bonding));
	} else if (!strcmp(attr, "miimon")) {
		unsigned int freq = 0;
		if (bonding->monitoring == NC_BOND_MONITOR_MII)
			freq = bonding->miimon.frequency;
		if(freq == 0)
			return 0;
		nc_stringbuf_printf(buffer, "%u", freq);
	} else if (!strcmp(attr, "updelay")) {
		if (bonding->monitoring != NC_BOND_MONITOR_MII)
			return 0;
		if(bonding->miimon.updelay == 0)
			return 0;
		nc_stringbuf_printf(buffer, "%u", bonding->miimon.updelay);
	} else if (!strcmp(attr, "downdelay")) {
		if (bonding->monitoring != NC_BOND_MONITOR_MII)
			return 0;
		if(bonding->miimon.downdelay == 0)
			return 0;
		nc_stringbuf_printf(buffer, "%u", bonding->miimon.downdelay);
	} else if (!strcmp(attr, "use_carrier")) {
		if (bonding->monitoring != NC_BOND_MONITOR_MII)
			return 0;
		if (bonding->miimon.carrier_detect == NC_BOND_CARRIER_DETECT_NETIF)
			return 0;
		nc_stringbuf_printf(buffer, "%u", bonding->miimon.carrier_detect);
	} else if (!strcmp(attr, "arp_validate")) {
		if (bonding->mode != NC_BOND_MODE_ACTIVE_BACKUP)
			return 0;
		if (bonding->monitoring != NC_BOND_MONITOR_ARP)
			return 0;
		nc_stringbuf_puts(buffer, __nc_bonding_get_module_option_arp_validate(bonding));
	} else if (!strcmp(attr, "arp_interval")) {
		if (bonding->monitoring != NC_BOND_MONITOR_ARP)
			return 0;
		if (bonding->arpmon.interval == 0)
			return 0;
		nc_stringbuf_printf(buffer, "%u", bonding->arpmon.interval);
	} else if (!strcmp(attr, "arp_ip_target")) {
		unsigned int j, max_targets = 16;
		if (bonding->monitoring != NC_BOND_MONITOR_ARP)
			return 0;
		for (j = 0; j < bonding->arpmon.targets.count; ++j) {
			if(max_targets && j >= max_targets)
				break;
			if(j > 0)
				nc_stringbuf_putc(buffer, ',');
			nc_stringbuf_puts(buffer, bonding->arpmon.targets.data[j]);
		}
	} else if (!strcmp(attr, "primary")) {
		if (!bonding->primary_slave)
			return 0;
		 if(nc_string_array_index(&bonding->slave_names, bonding->primary_slave) == -1)
			 return 0;
		nc_stringbuf_puts(buffer, bonding->primary_slave);
	} else if (!strcmp(attr, "lacp_rate")) {
		if (bonding->mode != NC_BOND_MODE_802_3AD)
			return 0;
		if (bonding->lacp_rate == NC_BOND_LACP_RATE_SLOW)
			return 0;
		nc_stringbuf_puts(buffer, __nc_bonding_get_module_option_lacp_rate(bonding));
	} else if (!strcmp(attr, "xmit_hash_policy")) {
		if (bonding->mode != NC_BOND_MODE_BALANCE_XOR &&
		    bonding->mode != NC_BOND_MODE_802_3AD)
			return 0;
		if (bonding->xmit_hash_policy == NC_BOND_XMIT_HASH_POLICY_LAYER2)
			return 0;
		nc_stringbuf_puts(buffer, __nc_bonding_get_module_option_xmit_hash_policy(bonding));
	} else {
		return -1;
	}

	return 0;
}

/*
 * Parse the module options specified for a bonding device.
 *
 *  max_bonds:Max number of bonded devices (int)
 *  num_grat_arp:Number of gratuitous ARP packets to send on failover event (int)
 *  miimon:Link check interval in milliseconds (int)
 *  updelay:Delay before considering link up, in milliseconds (int)
 *  downdelay:Delay before considering link down, in milliseconds (int)
 *  use_carrier:Use netif_carrier_ok (vs MII ioctls) in miimon; 0 for off, 1 for on (default) (int)
 *  mode:Mode of operation : 0 for balance-rr, 1 for active-backup, 2 for balance-xor, 3 for broadcast, 4 for 802.3ad, 5 for balance-tlb, 6 for balance-alb (charp)
 *  primary:Primary network device to use (charp)
 *  lacp_rate:LACPDU tx rate to request from 802.3ad partner (slow/fast) (charp)
 *  xmit_hash_policy:XOR hashing method: 0 for layer 2 (default), 1 for layer 3+4 (charp)
 *  arp_interval:arp interval in milliseconds (int)
 *  arp_ip_target:arp targets in n.n.n.n form (array of charp)
 *  arp_validate:validate src/dst of ARP probes: none (default), active, backup or all (charp)
 *  fail_over_mac:For active-backup, do not set all slaves to the same MAC.  none (default), active or follow (charp)
 */
void
nc_bonding_parse_module_options(nc_bonding_t *bonding, nc_handle_t *nh, const char *ifname)
{
	char *temp, *s, *t, *saveptr = NULL;

	nc_bonding_clear_options(bonding);
	if (!bonding->module_opts)
		return;

	temp = NULL;
	nc_string_dup(&temp, bonding->module_opts);
	for (s = strtok_r(temp, " \t", &saveptr); s; s = strtok_r(NULL, " \t", &saveptr)) {
		int rv;

		if ((t = strchr(s, '=')) == NULL) {
			if(nc_error_once_check(nh, "/parse_bonding_module_option/token/%s/%s", ifname, s)) {
				nc_info("%s: ignoring bonding module option without a value: %s", ifname, s);
			}
			continue;
		}
		nc_error_once_clear(nh, "parse_bonding_module_option/token/%s/%s", ifname, s);

		*t++ = '\0';

		rv = nc_bonding_parse_module_attribute(bonding, s, t);
		if (rv == -2) {
			/* unknown, probably not yet supported option  */
			if(nc_error_once_check(nh, "/parse_bonding_module_option/%s/%s", ifname, s)) {
				nc_debug_ifcfg("%s: ignoring unknown bonding module option %s=%s", ifname, s, t);
			}
		} else if (rv < 0) {
			/* failure to parse a supported option (value) */
			if(nc_error_once_check(nh, "/parse_bonding_module_option/%s/%s", ifname, s)) {
				nc_info("%s: unable to parse bonding module option %s=%s", ifname, s, t);
				/* we should probably return an error here */
			}
		} else {
			nc_error_once_clear(nh, "/parse_bonding_module_option/%s/%s", ifname, s);
		}
	}

	free(temp);
}

void
nc_bonding_build_module_options(nc_bonding_t *bonding)
{
	nc_stringbuf_t outbuf = NC_STRINGBUF_INIT;
	const char *attrs[] = {
		"mode",
		"miimon",
		/*
		"num_grat_arp",
		"num_unsol_na",
		"resend_igmp",
		*/

		/* active-backup mode */
		"primary",
		/*
		"primary_reselect",
		"fail_over_mac",
		*/

		/* ignored for ARP monitoring: */
		"updelay",
		"downdelay",
		"use_carrier",

		/* ignored for MII monitoring: */
		"arp_ip_target",
		"arp_interval",
		"arp_validate",

		/* 802.3ad mode */
		"lacp_rate",
		/*
		"ad_select",
		*/
	
		/* 802.3ad or balance-xor mode */
		"xmit_hash_policy",
		NULL,
	};
	unsigned int i;

	for (i = 0; attrs[i]; ++i) {
		nc_stringbuf_t value = NC_STRINGBUF_INIT;

		if (nc_bonding_format_module_attribute(bonding, attrs[i], &value) < 0)
			continue;

		/* skip empty values */
		if (value.len && value.string) {
			if (!nc_stringbuf_empty(&outbuf))
				nc_stringbuf_putc(&outbuf, ' ');

			nc_stringbuf_puts(&outbuf, attrs[i]);
			nc_stringbuf_putc(&outbuf, '=');
			nc_stringbuf_puts(&outbuf, value.string);
		}
		nc_stringbuf_destroy(&value);
	}
	for (i = 0; i < bonding->extra_options.count; ++i) {
		const char *key = bonding->extra_options.data[i].name;
		const char *val = bonding->extra_options.data[i].value;

		if(!nc_string_len(key) || !nc_string_len(val))
			continue;

		if (!nc_stringbuf_empty(&outbuf))
			nc_stringbuf_putc(&outbuf, ' ');
		nc_stringbuf_puts(&outbuf, key);
		nc_stringbuf_putc(&outbuf, '=');
		nc_stringbuf_puts(&outbuf, val);
	}

	nc_string_free(&bonding->module_opts);
	bonding->module_opts = outbuf.string;
	return;
}

/*
 * Load bonding configuration from sysfs
 */
int
nc_bonding_parse_sysfs_attrs(const char *ifname, nc_bonding_t *bonding)
{
	const char *attrs[] = {
		"mode",
		"miimon",
		"arp_validate",
		"arp_interval",
		"updelay",
		"downdelay",
		"use_carrier",
		"primary",
		"xmit_hash_policy",
		"lacp_rate",
		NULL,
	};
	char *attrval = NULL;
	unsigned int i;

	nc_bonding_clear(bonding);
	nc_sysfs_bonding_get_slaves(ifname, &bonding->slave_names);

	for (i = 0; attrs[i]; ++i) {
		const char *attrname = attrs[i];
		int rv;

		if (nc_sysfs_bonding_get_attr(ifname, attrname, &attrval) < 0) {
			nc_error("%s: cannot get bonding attribute %s", ifname, attrname);
			goto failed;
		}

		if (attrval == NULL)
			continue;

		rv = nc_bonding_parse_module_attribute(bonding, attrname, attrval);
		if (rv == -2) {
			nc_error("ignoring unknown bonding module option %s=%s", attrname, attrval);
		} else if (rv < 0) {
			nc_error("unable to parse bonding module option %s=%s", attrname, attrval);
			goto failed;
		}
	}

	nc_sysfs_bonding_get_arp_targets(ifname, &bonding->arpmon.targets);

	nc_string_free(&attrval);
	return 0;

failed:
	nc_string_free(&attrval);
	return -1;
}

/*
 * Write one sysfs attribute
 */
#if 0
static int
nc_bonding_write_one_sysfs_attr(const char *ifname, const nc_bonding_t *bonding,
			const nc_bonding_t *current, const char *attrname)
{
	char current_value[128], config_value[128];

	if (nc_bonding_format_module_attribute(current, attrname, current_value, sizeof(current_value)) < 0
	 || nc_bonding_format_module_attribute(bonding, attrname, config_value, sizeof(config_value)) < 0) {
		nc_error("%s: cannot represent attribute %s", ifname, attrname);
		return -1;
	}

	if (config_value[0] == '\0') {
		nc_debug_ifcfg("%s: attr %s ignored", ifname, attrname);
		return 0;
	}

	if (!strcmp(current_value, config_value)) {
		nc_debug_ifcfg("%s: attr %s unchanged", ifname, attrname);
		return 0;
	}

	nc_debug_ifcfg("%s: setting attr %s=%s", ifname, attrname, config_value);
	if (nc_sysfs_bonding_set_attr(ifname, attrname, config_value) < 0) {
		nc_error("%s: cannot set bonding attribute %s=%s", ifname, attrname, config_value);
		return -1;
	}

	return 0;
}
#endif

/*
 * Write bonding configuration to sysfs.
 * This happens in two stages; the first stage happens prior to enslaving interfaces,
 * the other happens afterwards.
 */
#if 0
int
nc_bonding_write_sysfs_attrs(const char *ifname, const nc_bonding_t *bonding, const nc_bonding_t *current, int stage)
{
	const char *stage0_attrs[] = {
		"mode",
		"miimon",

		/* ignored for ARP monitoring: */
		"updelay",
		"downdelay",
		"use_carrier",

		/* ignored for MII monitoring: */
		"arp_interval",
		"arp_validate",
		NULL,
	};
	const char *stage1_attrs[] = {
		"primary",
		NULL,
	};
	const char **attrs;
	unsigned int i;


	attrs = (stage == 0)? stage0_attrs : stage1_attrs;
	for (i = 0; attrs[i]; ++i) {
		if (nc_bonding_write_one_sysfs_attr(ifname, bonding, current, attrs[i]) < 0)
			return -1;
	}

	/* arp_ip_target is special, since it's a list of addrs */
	if (stage == 0 && bonding->monitoring == NC_BOND_MONITOR_ARP
	 && nc_sysfs_bonding_set_list_attr(ifname, "arp_ip_target", &bonding->arpmon.targets) < 0)
		return -1;

	return 0;
}
#endif

