
/*
 * scsires.h
 *
 * Copyright (C) 2000 Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 11/20/00 - Initial version written by Doug Ledford
 */


/*
 * A few items missing from the scsi.h header
 */

#define RESERVE_10	0x56
#define RELEASE_10	0x57
#define PERSISTENT_RESERVE_IN	0x5e
#define PERSISTENT_RESERVE_OUT	0x5f
#ifndef SCSI_TRY_RESET_DEVICE
#define SCSI_TRY_RESET_DEVICE 1
#define SCSI_TRY_RESET_BUS    2
#define SCSI_TRY_RESET_HOST   3
#endif
#ifndef SCSI_IOCTL_STONITH
#define SCSI_IOCTL_STONITH	7
#define	SCSI_IOCTL_STONITH_ENABLED	1
#define SCSI_IOCTL_STONITH_DISABLED	0
#endif

#define DEF_WAIT	10	/* how long we wait, in seconds, for a drive to
				 * become ready by default */

typedef enum {
    READ_SHARED = 0,
    WRITE_EXCLUSIVE = 1,
    READ_EXCLUSIVE = 2,
    EXCLUSIVE_ACCESS = 3
} scsires_access_mode;

typedef enum {
    NONE = 0,
    SCSI_2_LUN = 1,
    SCSI_3_LUN = 2,
    SCSI_2_EXTENT = 3,
    SCSI_3_EXTENT = 4,
    SCSI_3_PERSISTENT = 5
} scsires_reservation_type;

typedef enum {
    FALSE = 0,
    TRUE = 1
} bool;

struct scsires_extent_elem {
    scsires_access_mode mode;
    bool relative_address;
    u_int32_t first_block;
    u_int32_t length;
};

typedef struct scsires_extent_elem scsires_extent_elem_t;

struct scsires_extent {
    u_int8_t key;
    u_int8_t extent_id;
    u_int8_t num_elements;
    bool third_party;
    u_int8_t third_party_id;
    scsires_extent_elem_t *elements;
    struct scsires_extent *next;
};

typedef struct scsires_extent scsires_extent_t;

struct scsires_pr_keys {
    unsigned char key;
    char *name;
};

typedef struct scsires_pr_keys scsires_pr_keys_t;

struct scsires_sg_dev {
    int fd;
    int disk_fd;
    int scsi_rev;
    bool extents;
    bool persistent_reservations;
    bool initialized;
    scsires_reservation_type reservation_type;
    unsigned int block_size;
    unsigned int num_blocks;
    unsigned int pr_generation;
    unsigned int scsi_dev_id;
    unsigned int scsi_dev_lun;
    unsigned int scsi_dev_bus;
    unsigned int scsi_dev_host;
    unsigned int num_reservations;
    scsires_extent_t *reservations;
    char *name;
    char *sg_name;
    char vendor[9];
    char product[17];
    char revision[5];
    char *serial_number;
    char *id_nonunique;
    char *id_vendor;
    unsigned int id_eui64[2];
    unsigned int id_fcph[2];
};

typedef struct scsires_sg_dev scsires_sg_dev_t;

/*
 * This should be in scsi.h to help make things easier, but it isn't.  So,
 * this is a makeshift struct to use to send as the argument to a
 * SCSI_IOCTL_SEND_COMMAND ioctl.
 */
struct scsires_send_command {
    int outsize;		/* bytes to be written to the bus */
    int insize;			/* bytes to be read from the bus */
    unsigned char buf[512];	/* the char array we put the
				 * data into, this may need to
				 * be larger in specific cases
				 */
    int result;			/* where we'll store the ioctl
				 * return value for testing
				 */
};

typedef struct scsires_send_command scsires_send_command_t;

/*
 * Convenient macros
 */

#define WAS_OK(x) (((x).result & 0x3f) == 0)
#define STATUS(x) (((x).result >> 1) & 0x1f)
#define WAS_SENSE(x) (((x).result & 0x3f) == 0x02)
#define RES_CONFLICT(x) (((x).result & 0x3f) == 0x18)
#define SENSE_KEY(x) ((x).buf[2] & 0x07)
#define ASC(x) ((x).buf[12])
#define ASQ(x) ((x).buf[13])

#define SEND_COMMAND(x,y) \
	((y).result = ioctl((x)->fd,SCSI_IOCTL_SEND_COMMAND,&(y)))

/*
 * Now the function declarations
 */

extern scsires_sg_dev_t *scsires_init_sg_device(int, const char *, bool);
extern int scsires_init_device_size(scsires_sg_dev_t *);
extern int scsires_init_persistent_reservations(scsires_sg_dev_t *, char *);
extern int scsires_issue_reservation(scsires_sg_dev_t *, scsires_extent_t *,

				     bool);
extern int scsires_release_reservation(scsires_sg_dev_t *,

				       scsires_extent_t *);
extern int scsires_release_sg_device(scsires_sg_dev_t *);
extern void scsires_print_device_info(scsires_sg_dev_t *);
extern void scsires_print_device_capabilities_header(void);
extern void scsires_print_device_capabilities(scsires_sg_dev_t *);
extern int scsires_get_pr_generation(scsires_sg_dev_t *);
extern int scsires_wait_device_ready(scsires_sg_dev_t *, int);

/*
 * The inline functions that we provide
 */

/*
 * Function: scsires_reset_dev (inline)
 *
 * Inputs:
 *      scsires_sg_dev_t * - The device which we want reset.
 *      int - The way we want the reset performed.
 *
 * Outputs:
 *      int - 0 on success, non-zero on failure.
 *
 * Purpose:
 *      Reset the device so we can issue a reservation to a currently
 *      held device.  We have several options for how to perform the
 *      reset.  The SCSI_TRY_RESET_* defines are the possible reset
 *      types.  If we want to be sneaky so that other hosts won't know
 *      we are trying to reset the device and steal their reservation
 *      until it is too late, then you should use SCSI_TRY_RESET_DEVICE.
 *      If you wanting to make sure the other host knows we are resetting
 *      things and give them a chance to re-issue any reservations they
 *      might have, then use SCSI_TRY_RESET_BUS instead.  The bus reset
 *      is immediately detectable by all hosts on the bus, the device
 *      reset is not.
 */

static inline int
scsires_reset_dev(scsires_sg_dev_t * sg_dev, int reset_type)
{
    return (ioctl(sg_dev->fd, SG_SCSI_RESET, &reset_type));
}


/*
 * Function: scsires_retryable_error (inline)
 *
 * Inputs:
 *      scsires_sg_dev_t * - The device which we sent the command to
 *      scsires_send_command_t - The command we sent
 *      bool - if there is a reservation conflict, should we force a reset
 *      	and then retry?
 *      int - the timeout to use for waiting on a drive to become ready
 *
 * Outputs:
 *      int - 1 if we should retry the error, 0 if the errors are considered
 *      	fatal, or if there were no errors and the command is complete
 *
 * Purpose:
 * 	This is just a shorthand function to make the code easier to read.
 * 	Almost all commands sent by this program will use this code to
 * 	determine if the command should be retried.  Putting it here
 * 	makes the rest of the code *much* easier to read.
 */

static inline int
scsires_retryable_error(scsires_sg_dev_t * sg_dev, scsires_send_command_t cmd,
			bool force, int timeout)
{
    int result;

    if (WAS_SENSE(cmd) && (SENSE_KEY(cmd) == UNIT_ATTENTION)) {
	return (1);
    } else if (WAS_SENSE(cmd) && (SENSE_KEY(cmd) == NOT_READY)) {
	result = scsires_wait_device_ready(sg_dev, timeout);
	if (result == 0) {
	    return (1);
	} else if (result == 1) {
	    printf("%s: device not ready error.\n", sg_dev->name);
	    return (0);
	} else {
	    return (0);
	}
    } else if (RES_CONFLICT(cmd) && (force == TRUE)) {
	result = scsires_reset_dev(sg_dev, SCSI_TRY_RESET_BUS);
	if (result) {
	    printf("%s: reset error.\n", sg_dev->name);
	    return (0);
	}
	result = scsires_wait_device_ready(sg_dev, timeout);
	if (result == 0) {
	    return (1);
	} else if (result == 1) {
	    printf("%s: device not ready error.\n", sg_dev->name);
	    return (0);
	} else {
	    return (0);
	}
    }
    return (0);
}
