
/*
 * scan_drives.c
 *
 * Copyright (C) 2000, Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 12/4/00 - Initial version written by Doug Ledford
 */

/*
 * This is a small and simple program that uses the scsires library.  It
 * is a test program for the library as much as anything else.  This
 * program is used to check the reservation status of a drive, issue a
 * reservation on a drive if you wish, or attempt to preempt a reservation
 * on a drive.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <popt.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/ioctl.h>

#include <scsi/scsi.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

#include "scsires.h"


/*
 * Our local variable type defines and other #define's
 */
#define	ACTION_NONE	0
#define	ACTION_HOLD	1
#define	ACTION_PREEMPT	2
#define	ACTION_RELEASE	3
#define	ACTION_RESERVE	4
#define	ACTION_RESET	5
#define	ACTION_SHOW	6
#define	ACTION_END	64

#define CHAR_BUF_SIZE 512

/*
 * Global variables
 */
static int global_verbose;
static int global_force;
static int global_stonith;
static int hold_delay = 4;
static char *drive_name;
static char *test_drive_name;
static char *config_file;
static int fd[2] = { -1, -1 };
static scsires_sg_dev_t *sg_dev[2] = { NULL, NULL };
static int *action_array;
static int current_action;
static struct sigaction sa_term;

struct poptOption options[] = {
    { "config", 'c', POPT_ARG_STRING, &config_file, 0,
	"If your SCSI-3 Persistent Reservation Key config file is found "
	"somewhere besides /etc/reservation_keys, then use this option to "
	"specify the location","<path>" },
    { "drive", 'd', POPT_ARG_STRING, &drive_name, 0,
	"This is a required option.  This tells the program which drive you "
	"want reserved (or otherwise manipulated).",
	"<path>" },
    { "delay", 0, POPT_ARG_INT, &hold_delay, 0,
	"How many seconds the system should delay between each reissue of "
	"a currently held reservation (default:4)", "seconds" },
    { "force", 'f', POPT_ARG_NONE, &global_force, 0,
	"If the drive is currently reserved, then force a reset in order to "
	"break the drive free.  This is implied by the reset and preempt "
	"options.", NULL },
    { "hold", 'h', POPT_ARG_NONE, NULL, ACTION_HOLD,
	"For devices that don't have persistent reservations, the only way "
	"to be sure that your reservation hasn't been lost is to sit in a "
	"loop and re-issue your reservation every second or so.  This option "
	"tells the program to do exactly that.", NULL },
    { "preempt", 'p', POPT_ARG_NONE, NULL, ACTION_PREEMPT,
	"Attempt to steal a reservation from another initiator.  This will "
	"involve a reset/reserve cycle.  This is different from a manual "
	"reset/reserve pair in that it will *not* wait any appreciable time "
	"between the reset and the reserve command so that it has the "
	"greatest possibility of stealing the reservation from the other "
	"initiator.", NULL },
    { "release", 0, POPT_ARG_NONE, NULL, ACTION_RELEASE,
	"Just like it sounds.  Try to release any reservations we currently "
	"hold on the device.  If this option is called without a preceding "
	"reserve, then the program assumes you are wanting to release the "
	"generic reservation that this program creates from a prior instance "
	"of this program and will attempt to do so.", NULL },
    { "reserve", 0, POPT_ARG_NONE, NULL, ACTION_RESERVE,
	"Try to reserve the drive.  The program will decide which type of "
	"reservation to use based upon drive capabilities, but it will always "
	"try to reserve the entire drive.", NULL },
    { "reset", 0, POPT_ARG_NONE, NULL, ACTION_RESET,
	"Throw a bus reset at a drive.  The program will automatically delay "
	"4 seconds after the reset to make sure that any subsequent commands "
	"are not sent so quickly that another host doesn't have a chance to "
	"reissue any reservations that our reset may have perturbed.  A "
	"reset/reserve pair can be used to take a reservation away from a "
	"dead host, but would not take a reservation away from a live host.  "
	"In order to actually steal a reservation from a live host, you must "
	"use the preempt option instead.", NULL },
    { "show", 's', POPT_ARG_NONE, NULL, ACTION_SHOW,
	"Show the current status of the drive.  For drives that only support "
	"LUN based reservations, we attempt a reservation and if it succeeds, "
	"the drive is free.  If it doesn't, the drive is currently reserved.  "
	"It is not possible to be more specific than that on LUN reservation "
	"based drives :-(  For drives that support SCSI-3 persistent "
	"reservations we can query the drive for status and produce much more "
	"detailed reservation reporting.", NULL },
    { "stonith", 0, POPT_ARG_NONE, &global_stonith, 0,
	"Enable STONITH mode (Shoot The Other Node In The Head).  In this "
	"mode, if a SCSI reservation is lost and we get a SCSI reservation "
	"conflict as a return code on any SCSI command to this device, the "
	"system will be immediately rebooted.  This option requires kernel "
	"level support.  If that support is not present, a warning will be "
	"issued and non-STONITH mode operation will continue.", NULL },
    { "test", 't', POPT_ARG_STRING, &test_drive_name, 0,
	"Run a series of tests on a device in order to test if SCSI "
	"reservations are properly honored by the device.  This "
	"option requires that you have two paths to a single device on the "
	"host system and the two paths must use different initiators for "
	"this to work.  The first path is given by the drive option and the "
	"second path is part of this option.  [Currently not implemented]",
	"<path>" },
    { "verbose", 'v', POPT_ARG_NONE, &global_verbose, 0,
	"Make the output more verbose", NULL },
    { "help", '?', POPT_ARG_NONE, NULL, ACTION_END,
	"Display help information for this program", NULL },
    { NULL, 0, 0, NULL, 0, NULL, NULL }
};

/*
 * Internal function declarations
 */

static __attribute__((noreturn)) int exit_program(int);
static void print_operation(const char *name);
static void show_drive(scsires_sg_dev_t *);
static void release_drive(scsires_sg_dev_t *);
static int reserve_drive(scsires_sg_dev_t *, bool);
void sig_term_handler(int);


/*
 * The functions themselves
 */

void sig_term_handler(int signum) {
    int i;
    char *signame;

    switch(signum) {
	case SIGHUP:
	    signame = "SIGHUP";
	    break;
	case SIGINT:
	    signame = "SIGINT";
	    break;
	case SIGQUIT:
	    signame = "SIGQUIT";
	    break;
	case SIGSEGV:
	    signame = "SIGSEGV";
	    break;
	case SIGTERM:
	    signame = "SIGTERM";
	    break;
	default:
	    signame = "Unknown";
	    break;
    }
    fprintf(stderr,"Recieved %s: cleaning up and exiting.\n", signame);
    if(global_stonith) {
	i = SCSI_IOCTL_STONITH_DISABLED;
	ioctl(fd[0], SCSI_IOCTL_STONITH, &i);
    }
    if(sg_dev[1])
	release_drive(sg_dev[1]);
    release_drive(sg_dev[0]);
    exit(signum);
}

static void print_operation(const char *name)
{
    fprintf(stderr,"
Theory of Operation:

	This program is designed to be used as a locking mechanism between
nodes in a high availability cluster.  It uses STONITH as a basis for
protecting from data corruption.  SCSI Reservations should not be used
to establish quorum or do other similar tasks.  Instead, this is merely
a tool to be used by High Availability cluster software in managing the
locks between machines.

	Under normal conditions, this program should be invoked with the
drive parameter (which is required for operation) and a chain of command
options.  Each command will be executed in the order listed on the command
line.  For example:
	%s -d /dev/sda --reserve --hold --stonith
will result in the program attempting to get an initial reservation on
/dev/sda, and the program will then attempt to keep that reservation by
reissuing it on a regular basis.  The --stonith option tells the program
to do this under the stonith model.  Without this option, the program will
exit with an error code if the reservation is ever lost.

	The ordering of options on the command line is both important and
not important.  Those options that don't represent an action can be anywhere
on the command line.  Those options that do represent some action are
executed in the order presented, so their ordering is important.  Command
options are: preempt, release, reserve, reset, show, test.

	Note that test mode is not yet implemented.  When implemented it
will be exclusive of all other command options except show.  Specifying the
test command in combination with anything other than the show option will
result in the progam claiming you have made an error and exiting.

", name);
}

int
main(int argc, char *argv[])
{
    int i, local_errno, result;
    char buffer[CHAR_BUF_SIZE];
    poptContext context;

    context = poptGetContext(argv[0], argc, (const char **)argv,
			     (const struct poptOption *)&options, 0);

    if ((action_array = malloc(argc * sizeof(int))) == NULL) {
	printf("Unable to allocate action_array, exiting.\n");
	exit(1);
    }

    for (i = 0; i < argc; i++) {
	action_array[i] = ACTION_NONE;
    }

    current_action = 0;

    while( (i = poptGetNextOpt(context)) >= 0 && (i < ACTION_END))
	action_array[current_action++] = i;

    if(i == ACTION_END) {
	poptPrintHelp(context, stderr, 0);
	print_operation(argv[0]);
	exit(0);
    }

    if(i < -1) {
	poptPrintUsage(context, stderr, 0);
	exit(1);
    }

    if (action_array[0] == ACTION_NONE) {
	fprintf(stderr,"Error: You must specify at least one action.\n");
	print_operation(argv[0]);
	exit(1);
    }

    if (drive_name == NULL) {
	fprintf(stderr,"Error: You must specify a drive.\n");
	poptPrintUsage(context, stderr, 0);
	exit(1);
    }

    if (test_drive_name != NULL) {
	if( (action_array[1] != ACTION_NONE) ||
	    (action_array[0] != ACTION_NONE &&
	     action_array[0] != ACTION_SHOW) ) {
	    poptPrintHelp(context, stderr, 0);
	    print_operation(argv[0]);
	    exit(1);
	}
    }

    /*
     * OK, we've collected the options and verified that an illegal
     * combination of options hasn't been passed in.  Now we verify
     * we can read/write all the devices we need to.
     */

    if ((fd[0] = open(drive_name, O_RDONLY | O_NDELAY)) == -1) {
	local_errno = errno;
	snprintf(buffer, CHAR_BUF_SIZE, "Unable to open %s", drive_name);
	errno = local_errno;
	perror(buffer);
	exit(1);
    }
    sg_dev[0] = scsires_init_sg_device(fd[0], drive_name, global_force);

    if(test_drive_name) {
	if ((fd[1] = open(test_drive_name, O_RDONLY | O_NDELAY)) == -1) {
	    local_errno = errno;
	    snprintf(buffer, CHAR_BUF_SIZE, "Unable to open %s",
		     test_drive_name);
	    errno = local_errno;
	    perror(buffer);
	    exit(1);
	}
	sg_dev[1] = scsires_init_sg_device(fd[1], test_drive_name,
					   global_force);
    }

    for(i=0; i<2 && sg_dev[i] != NULL; i++) {
	result = scsires_init_device_size(sg_dev[i]);
	if ((sg_dev[i]->initialized == FALSE) || (result == 2)) {
	    printf("Unable to initialize sg device for %s.\n",
		   sg_dev[i]->name);
	    exit(1);
	}
	if (sg_dev[i]->reservation_type == SCSI_3_PERSISTENT) {
	    char *path = "/etc/reservation_keys";
	    if(config_file != NULL)
		path = config_file;
	    if (scsires_init_persistent_reservations
		(sg_dev[i], path)) {
		printf("%s: unable to initialize persistent "
		       "reservation state.\n", sg_dev[i]->name);
		exit(1);
	    }
	}
    }

    /*
     * If we are to use the STONITH option, then set the ioctl here.
     */

    if(global_stonith) {
	i = SCSI_IOCTL_STONITH_ENABLED;
	if(fd[0] != -1)
	    if(ioctl(fd[0], SCSI_IOCTL_STONITH, &i)) {
		fprintf(stderr,"\nThis kernel does not support the reboot "
			"on reservation conflict ioctl.\nThe stonith option "
			"will be disabled.\n");
		global_stonith = 0;
	    }
    }

    /*
     * Everything is configured at this point and we are about to drop into
     * our action loop.  Time to set up our signal handlers so we can do the
     * right thing if we get interrupted.
     */
    sa_term.sa_handler = sig_term_handler;
    sa_term.sa_flags = 0;
    sigemptyset(&sa_term.sa_mask);
    sigaddset(&sa_term.sa_mask, SIGHUP);
    sigaddset(&sa_term.sa_mask, SIGINT);
    sigaddset(&sa_term.sa_mask, SIGQUIT);
    sigaddset(&sa_term.sa_mask, SIGSEGV);
    sigaddset(&sa_term.sa_mask, SIGTERM);


    if(sigaction(SIGHUP, &sa_term, NULL) ||
       sigaction(SIGINT, &sa_term, NULL) ||
       sigaction(SIGQUIT, &sa_term, NULL) ||
       sigaction(SIGSEGV, &sa_term, NULL) ||
       sigaction(SIGTERM, &sa_term, NULL)) {
	fprintf(stderr,"\nUnable to set signal handlers for program.  This "
		"disables the\nability for the program to clean up after "
		"itself, leaving possibly\ndangerous options in effect when "
		"the program exits.  This program refuses\nto run without "
		"the ability to intercept certain signals.\n");
	exit_program(1);
    }

    /*
     * Now that we have the drives and the sg devices opened, time to see
     * what we are suppossed to be doing here :-)
     */

    i = 0;
    do {
	switch (action_array[i]) {
	    case ACTION_SHOW:
		show_drive(sg_dev[0]);
		break;
	    case ACTION_RESET:
		scsires_reset_dev(sg_dev[0], SCSI_TRY_RESET_BUS);
		sleep(4);
		break;
	    case ACTION_HOLD:
		if (sg_dev[0]->reservations == NULL) {
		    result = reserve_drive(sg_dev[0], FALSE);
		    if (result) {
			printf("%s: unable to reserve " "drive.\n",
			       sg_dev[0]->name);
			exit_program(1);
		    }
		}
		while (scsires_issue_reservation
		       (sg_dev[0], sg_dev[0]->reservations, FALSE) == 0) {
		    sleep(hold_delay);
		}
		/*
		 * Should never get here with stonith enabled
		 */
		printf("%s: reservation was lost.\n", sg_dev[0]->name);
		break;
	    case ACTION_RELEASE:
		release_drive(sg_dev[0]);
		break;
	    case ACTION_PREEMPT:
		result = reserve_drive(sg_dev[0], TRUE);
		if (result) {
		    printf("%s: error while reserving drive.\n",
			   sg_dev[0]->name);
		    exit_program(1);
		}
		break;
	    case ACTION_RESERVE:
		result = reserve_drive(sg_dev[0], FALSE);
		if (result) {
		    printf("%s: error while reserving drive.\n",
			   sg_dev[0]->name);
		    exit_program(1);
		}
		break;
	    case ACTION_NONE:
	    default:
		printf("Bad action value in array!  Shouldn't "
		       "ever get here!\n");
		exit_program(1);
		break;
	}
    } while (action_array[++i] != ACTION_NONE);

    if(sg_dev[1]) {
	/*
	 * Someone called on test mode.  Too bad it's not currently implemented.
	 */
	fprintf(stderr,"Test mode is currently not implemented.\n");
	exit_program(1);
    }

    exit_program(0);
}

void
show_drive(scsires_sg_dev_t * sg_ptr)
{
    scsires_extent_t *extent;
    int result;


    scsires_print_device_capabilities_header();
    scsires_print_device_capabilities(sg_ptr);
    printf("\n\t");

    if ((extent = malloc(sizeof(scsires_extent_t))) == NULL) {
	printf("Unable to allocate extent, exiting.\n");
	exit_program(1);
    }
    memset(extent, 0, sizeof(scsires_extent_t));
    if ((extent->elements = malloc(sizeof(scsires_extent_elem_t))) == NULL) {
	printf("Unable to allocate element, exiting.\n");
	exit_program(1);
    }
    memset(extent->elements, 0, sizeof(scsires_extent_elem_t));
    extent->extent_id = 1;
    extent->num_elements = 1;
    extent->third_party = FALSE;
    extent->elements[0].mode = WRITE_EXCLUSIVE;
    extent->elements[0].relative_address = FALSE;
    extent->elements[0].first_block = 0;
    extent->elements[0].length = 0;
    result = scsires_issue_reservation(sg_ptr, extent, FALSE);
    if (result) {
	printf("We are currently unable to reserve this device.\n\n");
    } else {
	printf("This device is available for SCSI reservations.\n\n");
	scsires_release_reservation(sg_ptr, extent);
    }
}

static int
reserve_drive(scsires_sg_dev_t * sg_ptr, bool reset)
{
    scsires_extent_t *extent;

    if ((extent = malloc(sizeof(scsires_extent_t))) == NULL) {
	printf("Unable to allocate extent, exiting.\n");
	exit_program(1);
    }
    memset(extent, 0, sizeof(scsires_extent_t));
    if ((extent->elements = malloc(sizeof(scsires_extent_elem_t))) == NULL) {
	printf("Unable to allocate element, exiting.\n");
	exit_program(1);
    }
    memset(extent->elements, 0, sizeof(scsires_extent_elem_t));
    extent->extent_id = 1;
    extent->num_elements = 1;
    extent->third_party = FALSE;
    extent->elements[0].mode = WRITE_EXCLUSIVE;
    extent->elements[0].relative_address = FALSE;
    extent->elements[0].first_block = 0;
    extent->elements[0].length = 0;
    return (scsires_issue_reservation(sg_ptr, extent, reset));

}

static void
release_drive(scsires_sg_dev_t *sg_ptr)
{
    /*
     * If we start off with no reservations, that means we were called to
     * release a drive that was reserved by a previous instance of the
     * program.  Since we know how we would have built the reservation, we
     * just build a dummy to stick into the struct and then release the
     * dummy reservation.
     */
    if (sg_ptr->reservations == NULL) {
	scsires_extent_t *e;

	e = malloc(sizeof(scsires_extent_t));
	if (e == NULL) {
	    printf("Unable to allocate " "memory.\n");
	    exit_program(1);
	}
	memset(e, 0, sizeof(scsires_extent_t));
	e->elements = malloc(sizeof(scsires_extent_elem_t));
	if (e->elements == NULL) {
	    printf("Unable to allocate " "memory.\n");
	    exit_program(1);
	}
	memset(e->elements, 0, sizeof(scsires_extent_elem_t));
	e->extent_id = 1;
	e->num_elements = 1;
	e->third_party = FALSE;
	e->elements->mode = WRITE_EXCLUSIVE;
	e->elements->relative_address = FALSE;
	sg_ptr->reservations = e;
    }

    /*
     * The scsires_release_reservation code frees the extent on a release.
     * So, as long as reservations != NULL, there are reservations to be
     * released.  As soon as it is NULL, all reservations have been released.
     */
    while (sg_ptr->reservations != NULL) {
	int result;

	result = scsires_release_reservation(sg_ptr, sg_ptr->reservations);
	if (result) {
	    printf("%s: error releasing reservation.\n", sg_ptr->name);
	    exit_program(1);
	}
    }
}

static int
exit_program(int exit_code)
{
    int i;

    i = SCSI_IOCTL_STONITH_DISABLED;
    ioctl(fd[0],SCSI_IOCTL_STONITH,&i);
    exit(exit_code);
}

