
/*
 * scan_drives.c
 *
 * Copyright (C) 2000, Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 12/4/00 - Initial version written by Doug Ledford
 */

/*
 * This is a small and simple program that uses the scsires library.  It
 * is a test program for the library as much as anything else, but it also will
 * scan all the hard drives (up to /dev/sdz) on a system and print out the
 * capabilities of each which happens to be something I needed so that I
 * could have some kind of rough idea how many SCSI-3 drives supported
 * persistent reservations.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include <errno.h>
#include <unistd.h>

#include <popt.h>

#include <sys/types.h>
#include <sys/ioctl.h>

#include <scsi/scsi.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

#include "scsires.h"

#define NUM_SCSI_DEVICES	128

int global_verbose;
int global_multipath;
int global_reservation;

struct poptOption popt_options[] = {
    { "verbose", 'v', POPT_ARG_NONE, &global_verbose, 0,
	"Print out extra information" , NULL },
    { "multipath", 'm', POPT_ARG_NONE, &global_multipath, 0,
	"Print out any multipath devices detected", NULL },
    { "reservation", 'r', POPT_ARG_NONE, &global_reservation, 0,
	"Print out the device reservation capability table", NULL },
    POPT_AUTOHELP
    { NULL, 0, 0, NULL, 0, NULL, NULL }
};

int
main(int argc, char *argv[])
{
    int fd, i, j, num_devs;
    scsires_sg_dev_t *sg_dev[NUM_SCSI_DEVICES];
    char *buffer[NUM_SCSI_DEVICES];
    poptContext context;

    context = poptGetContext(argv[0], argc, (const char **)argv,
			     (const struct poptOption *)&popt_options, 0);

    i = poptGetNextOpt(context);
    if( i < -1 ) {
	poptPrintHelp(context, stderr, 0);
	exit(1);
    }

    if(global_reservation)
	scsires_print_device_capabilities_header();

    for(i=0; i<NUM_SCSI_DEVICES; i++) {
	sg_dev[i] = NULL;
	buffer[i] = NULL;
    }

    i = 0;
    j = 0;
    do {
	if((buffer[j] == NULL) && ((buffer[j] = malloc(128)) == NULL)) {
	    fprintf(stderr, "Unable to allocate memory, exiting.\n");
	    exit(1);
	}

	if(i <= 26)
	    sprintf(buffer[j], "/dev/sd%c", i + 'a');
	else
	    sprintf(buffer[j], "/dev/sd%c%c", i / 26 + 'a', i % 26 + 'a');

	fd = open(buffer[j], O_RDONLY | O_NDELAY);
	if (fd != -1) {
	    sg_dev[j] = scsires_init_sg_device(fd, buffer[j], FALSE);
	    if (sg_dev[j] && sg_dev[j]->initialized) {
		if (global_reservation)
		    scsires_print_device_capabilities(sg_dev[j]);
		/*
		 * We are going to cheat and use the extents variable in the
		 * multipath scan, so set it false now that we've printed
		 * out the device capability header
		 */
		if (global_multipath)
		    sg_dev[j]->extents = FALSE;
		j++;
	    }
	}
    } while (++i < NUM_SCSI_DEVICES);

    num_devs = j;

    if(global_reservation)
	printf("\n");

    if(global_multipath) {
	if(global_verbose)
	    printf("\nChecking for multipath drives...(%d devices)\n",
		   num_devs);

	for(i = 0; i < num_devs - 1; i++) {
	    int multipath_array[NUM_SCSI_DEVICES], k;

	    k = 1;
	    multipath_array[0] = i;
	    multipath_array[k] = -1;
	    for(j = i + 1; j < num_devs; j++) {
		int a0, a1, b0, b1;

		if(!sg_dev[i]->initialized || !sg_dev[j]->initialized ||
		   sg_dev[i]->extents || sg_dev[j]->extents)
		    continue;

		a0 = sg_dev[i]->id_eui64[0];
		a1 = sg_dev[i]->id_eui64[1];
		b0 = sg_dev[j]->id_eui64[0];
		b1 = sg_dev[j]->id_eui64[1];
		if((a0 || a1) && (b0 || b1) && ((a0 == b0) && (a1 == b1))) {
		    if(global_verbose)
			printf("%s and %s are multiple paths to the same "
			       "device (id_eui64)\n", sg_dev[i]->name,
			       sg_dev[j]->name);
		    multipath_array[k++] = j;
		    continue;
		} else if ((a0 || a1 || b0 || b1) &&
			   ((a0 != b0) || (a1 != b1))) {
		    continue;
		}
		
		a0 = sg_dev[i]->id_fcph[0];
		a1 = sg_dev[i]->id_fcph[1];
		b0 = sg_dev[j]->id_fcph[0];
		b1 = sg_dev[j]->id_fcph[1];
		if( (a0 || a1) && (b0 || b1) && ((a0 == b0) && (a1 == b1)) ) {
		    if(global_verbose)
			printf("%s and %s are multiple paths to the same "
			       "device (id_fcph)\n", sg_dev[i]->name,
			       sg_dev[j]->name);
		    multipath_array[k++] = j;
		    continue;
		} else if ((a0 || a1 || b0 || b1) &&
			   ((a0 != b0) || (a1 != b1))) {
		    continue;
		}
		if( (sg_dev[i]->scsi_dev_id == sg_dev[j]->scsi_dev_id) &&
		    (sg_dev[i]->scsi_dev_lun == sg_dev[j]->scsi_dev_lun) &&
		    !strcmp(sg_dev[i]->vendor, sg_dev[j]->vendor) &&
		    !strcmp(sg_dev[i]->product, sg_dev[j]->product) &&
		    !strcmp(sg_dev[i]->revision, sg_dev[j]->revision) ) {
		    if( sg_dev[i]->serial_number && sg_dev[j]->serial_number &&
	   		!strcmp(sg_dev[i]->serial_number,
				sg_dev[j]->serial_number) ) {
			if(global_verbose)
			    printf("%s and %s are multiple paths to the same "
				   "device (serial_number)\n", sg_dev[i]->name,
				   sg_dev[j]->name);
			multipath_array[k++] = j;
			continue;
		    } else if (sg_dev[i]->serial_number != NULL ||
			       sg_dev[j]->serial_number != NULL) {
			continue;
		    }
		    if( sg_dev[i]->id_vendor && sg_dev[j]->id_vendor &&
	   	       !strcmp(sg_dev[i]->id_vendor, sg_dev[j]->id_vendor) ) {
			if(global_verbose)
			    printf("%s and %s are multiple paths to the same "
				   "device (id_vendor)\n", sg_dev[i]->name,
				   sg_dev[j]->name);
			multipath_array[k++] = j;
			continue;
		    } else if (sg_dev[i]->id_vendor != NULL ||
			       sg_dev[j]->id_vendor != NULL) {
			continue;
		    }
		    if( sg_dev[i]->id_nonunique && sg_dev[j]->id_nonunique &&
	   	       !strcmp(sg_dev[i]->id_nonunique,
			       sg_dev[j]->id_nonunique) ) {
			if(global_verbose)
			    printf("%s and %s are multiple paths to the same "
				   "device (id_nonunique)\n", sg_dev[i]->name,
				   sg_dev[j]->name);
			multipath_array[k++] = j;
			continue;
		    } else if (sg_dev[i]->id_vendor != NULL ||
			       sg_dev[j]->id_vendor != NULL) {
			continue;
		    }
		}
	    }

	    if(k > 1) {
		if(k == 2) {
		    printf("%s and %s are multiple paths to the same device.\n",
			   sg_dev[i]->name, sg_dev[multipath_array[1]]->name);
		} else {
		    printf("%s, ", sg_dev[i]->name);
		    for(j = 1; j < (k - 1); j++) {
			printf("%s, ", sg_dev[multipath_array[j]]->name);
		    }
		    printf("and %s are multiple paths to the same device.\n",
			   sg_dev[multipath_array[j]]->name);
		}
	    }
	}
	if(global_verbose)
	    printf("\n");
    }
    
    exit(0);
}
