/*
 * test-render-surround-view.cpp - test render surround view
 *
 *  Copyright (c) 2018 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Author: Zong Wei <wei.zong@intel.com>
 */

#include "test_common.h"
#include "test_stream.h"
#include <interface/geo_mapper.h>
#include <interface/stitcher.h>
#include <calibration_parser.h>
#include <soft/soft_video_buf_allocator.h>
#if HAVE_GLES
#include <gles/gl_video_buffer.h>
#include <gles/egl/egl_base.h>
#endif
#if HAVE_VULKAN
#include <vulkan/vk_device.h>
#endif

#if ENABLE_DNN
#include "dnn/inference/dnn_inference_utils.h"
#include "dnn/inference/dnn_inference_engine.h"
#include "dnn/inference/dnn_object_detection.h"
#endif

#include <render/render_osg_viewer.h>
#include <render/render_osg_model.h>
#include <render/render_osg_shader.h>

using namespace XCam;

#define CAR_MODEL_NAME  "Suv.osgb"

#if ENABLE_DNN
#define DNN_MODEL_NAME "pedestrian-and-vehicle-detector-adas-0001.xml"
#endif

enum SVModule {
    SVModuleNone    = 0,
    SVModuleSoft,
    SVModuleGLES,
    SVModuleVulkan
};

static const char *intrinsic_names[] = {
    "intrinsic_camera_front.txt",
    "intrinsic_camera_right.txt",
    "intrinsic_camera_rear.txt",
    "intrinsic_camera_left.txt"
};

static const char *extrinsic_names[] = {
    "extrinsic_camera_front.txt",
    "extrinsic_camera_right.txt",
    "extrinsic_camera_rear.txt",
    "extrinsic_camera_left.txt"
};

static const float viewpoints_range[] = {64.0f, 160.0f, 64.0f, 160.0f};

static const char VtxShaderCar[] = ""
                                   "precision highp float;                                        \n"
                                   "uniform mat4 osg_ModelViewProjectionMatrix;                   \n"
                                   "uniform mat4 osg_ModelViewMatrix;                             \n"
                                   "uniform mat3 osg_NormalMatrix;                                \n"
                                   "attribute vec3 osg_Normal;                                    \n"
                                   "attribute vec4 osg_Color;                                     \n"
                                   "attribute vec4 osg_Vertex;                                    \n"
                                   "varying vec4 v_color;                                         \n"
                                   "varying float diffuseLight;                                   \n"
                                   "varying float specLight;                                      \n"
                                   "attribute vec2 osg_MultiTexCoord0;                            \n"
                                   "varying vec2 texCoord0;                                       \n"
                                   "void main()                                                   \n"
                                   "{                                                             \n"
                                   "    vec4 light = vec4(0.0,100.0, 100.0, 1.0);                 \n"
                                   "    vec4 lightColorSpec = vec4(1.0, 1.0, 1.0, 1.0);           \n"
                                   "    vec4 lightColorDiffuse = vec4(1.0, 1.0, 1.0, 1.0);        \n"
                                   "    vec4 lightColorAmbient = vec4(0.3, 0.3, .3, 1.0);         \n"
                                   "    vec4 carColorAmbient = vec4(0.0, 0.0, 1.0, 1.0);          \n"
                                   "    vec4 carColorDiffuse = vec4(0.0, 0.0, 1.0, 1.0);          \n"
                                   "    vec4 carColorSpec = vec4(1.0, 1.0, 1.0, 1.0);             \n"
                                   "    vec3 tnorm = normalize(osg_NormalMatrix * osg_Normal);    \n"
                                   "    vec4 eye = osg_ModelViewMatrix * osg_Vertex;              \n"
                                   "    vec3 s = normalize(vec3(light - eye));                    \n"
                                   "    vec3 v = normalize(-eye.xyz);                             \n"
                                   "    vec3 r = reflect(-s, tnorm);                              \n"
                                   "    diffuseLight = max(0.0, dot( s, tnorm));                  \n"
                                   "    specLight = 0.0;                                          \n"
                                   "    if(diffuseLight > 0.0)                                    \n"
                                   "    {                                                         \n"
                                   "        specLight = pow(max(0.0, dot(r,v)), 10.0);            \n"
                                   "    }                                                         \n"
                                   "    texCoord0 = osg_MultiTexCoord0;                               \n"
                                   "    v_color = (specLight *  lightColorSpec * carColorSpec) + (carColorDiffuse * lightColorDiffuse * diffuseLight) + lightColorAmbient * carColorAmbient;   \n"
                                   "    gl_Position = osg_ModelViewProjectionMatrix * osg_Vertex;     \n"
                                   "}                                                \n";

static const char FrgShaderCar[] = ""
                                   "precision highp float;                                                      \n"
                                   "varying vec4 v_color;                                                       \n"
                                   "varying float diffuseLight;                                                 \n"
                                   "varying float specLight;                                                    \n"
                                   "uniform sampler2D textureWheel;                                             \n"
                                   "varying vec2 texCoord0;                                                     \n"
                                   "void main()                                                                 \n"
                                   "{                                                                           \n"
                                   "    vec4 lightColorSpec = vec4(1.0, 1.0, 1.0, 1.0);                         \n"
                                   "    vec4 lightColorDiffuse = vec4(1.0, 1.0, 1.0, 1.0);                      \n"
                                   "    vec4 lightColorAmbient = vec4(0.3, 0.3, .3, 1.0);                       \n"
                                   "    vec4 base = texture2D(textureWheel, texCoord0.st);                      \n"
                                   "    gl_FragColor = (specLight *  lightColorSpec * base) + (base * lightColorDiffuse * diffuseLight) + lightColorAmbient * base ; \n"
                                   "}                                                                           \n";

class SVStream
    : public Stream
{
public:
    explicit SVStream (const char *file_name = NULL, uint32_t width = 0, uint32_t height = 0);
    virtual ~SVStream () {}

    void set_module (SVModule module) {
        XCAM_ASSERT (module != SVModuleNone);
        _module = module;
    }

#if HAVE_VULKAN
    void set_vk_device (SmartPtr<VKDevice> &device) {
        XCAM_ASSERT (device.ptr ());
        _vk_dev = device;
    }
    SmartPtr<VKDevice> &get_vk_device () {
        return _vk_dev;
    }
#endif

    virtual XCamReturn create_buf_pool (uint32_t reserve_count, uint32_t format = V4L2_PIX_FMT_NV12);

private:
    XCAM_DEAD_COPY (SVStream);

private:
    SVModule               _module;
#if HAVE_VULKAN
    SmartPtr<VKDevice>     _vk_dev;
#endif
};
typedef std::vector<SmartPtr<SVStream>> SVStreams;

SVStream::SVStream (const char *file_name, uint32_t width, uint32_t height)
    :  Stream (file_name, width, height)
    , _module (SVModuleNone)
{
}

XCamReturn
SVStream::create_buf_pool (uint32_t reserve_count, uint32_t format)
{
    XCAM_ASSERT (get_width () && get_height ());
    XCAM_FAIL_RETURN (
        ERROR, _module != SVModuleNone, XCAM_RETURN_ERROR_PARAM,
        "invalid module, please set module first");

    VideoBufferInfo info;
    info.init (format, get_width (), get_height ());

    SmartPtr<BufferPool> pool;
    if (_module == SVModuleSoft) {
        pool = new SoftVideoBufAllocator (info);
    } else if (_module == SVModuleGLES) {
#if HAVE_GLES
        pool = new GLVideoBufferPool (info);
#endif
    } else if (_module == SVModuleVulkan) {
#if HAVE_VULKAN
        XCAM_ASSERT (_vk_dev.ptr ());
        pool = create_vk_buffer_pool (_vk_dev);
        XCAM_ASSERT (pool.ptr ());
        pool->set_video_info (info);
#endif
    }
    XCAM_ASSERT (pool.ptr ());

    if (!pool->reserve (reserve_count)) {
        XCAM_LOG_ERROR ("create buffer pool failed");
        pool.release ();
        return XCAM_RETURN_ERROR_MEM;
    }

    set_buf_pool (pool);
    return XCAM_RETURN_NO_ERROR;
}

static SmartPtr<Stitcher>
create_stitcher (const SmartPtr<SVStream> &stitch, SVModule module)
{
    SmartPtr<Stitcher> stitcher;

    if (module == SVModuleSoft) {
        stitcher = Stitcher::create_soft_stitcher ();
    } else if (module == SVModuleGLES) {
#if HAVE_GLES
        stitcher = Stitcher::create_gl_stitcher ();
#endif
    } else if (module == SVModuleVulkan) {
#if HAVE_VULKAN
        SmartPtr<VKDevice> dev = stitch->get_vk_device ();
        XCAM_ASSERT (dev.ptr ());
        stitcher = Stitcher::create_vk_stitcher (dev);
#else
        XCAM_UNUSED (stitch);
#endif
    }
    XCAM_ASSERT (stitcher.ptr ());

    return stitcher;
}

void
get_bowl_model (
    const SmartPtr<Stitcher> &stitcher,
    BowlModel::VertexMap &vertices,
    BowlModel::PointMap &points,
    BowlModel::IndexVector &indices,
    float &a,
    float &b,
    float &c,
    float resRatio,
    uint32_t image_width,
    uint32_t image_height)
{
    uint32_t res_width = image_width * resRatio;
    uint32_t res_height = image_height * resRatio;

    BowlDataConfig bowl = stitcher->get_bowl_config();
    bowl.angle_start = 0.0f;
    bowl.angle_end = 360.0f;

    a = bowl.a;
    b = bowl.b;
    c = bowl.c;

    BowlModel bowl_model(bowl, image_width, image_height);

    bowl_model.get_bowlview_vertex_model(
        vertices,
        points,
        indices,
        res_width,
        res_height);
}

static SmartPtr<RenderOsgModel>
create_surround_view_model (
    const SmartPtr<Stitcher> &stitcher,
    uint32_t texture_width,
    uint32_t texture_height)
{
    SmartPtr<RenderOsgModel> svm_model = new RenderOsgModel ("svm model", texture_width, texture_height);

    svm_model->setup_shader_program ("SVM", osg::Shader::VERTEX, VtxShaderProjectNV12Texture);
    svm_model->setup_shader_program ("SVM", osg::Shader::FRAGMENT, FrgShaderProjectNV12Texture);

    BowlModel::VertexMap vertices;
    BowlModel::PointMap points;
    BowlModel::IndexVector indices;

    float a = 0;
    float b = 0;
    float c = 0;
    float res_ratio = 0.3;
    float scaling = 1000.0f;

    get_bowl_model (stitcher, vertices, points, indices,
                    a, b, c, res_ratio, texture_width, texture_height );

    svm_model->setup_vertex_model (vertices, points, indices, a / scaling, b / scaling, c / scaling);

    return svm_model;
}

static SmartPtr<RenderOsgModel>
create_car_model (const char *name)
{
    std::string car_name;
    if (NULL != name) {
        car_name = std::string (name);
    } else {
        car_name = std::string (CAR_MODEL_NAME);
    }
    std::string car_model_path = FISHEYE_CONFIG_PATH + car_name;

    const char *env_path = std::getenv (FISHEYE_CONFIG_ENV_VAR);
    if (env_path) {
        car_model_path.clear ();
        car_model_path = std::string (env_path) + std::string("/") + car_name;
    }

    SmartPtr<RenderOsgModel> car_model = new RenderOsgModel (car_model_path.c_str(), true);

    car_model->setup_shader_program ("Car", osg::Shader::VERTEX, VtxShaderCar);
    car_model->setup_shader_program ("Car", osg::Shader::FRAGMENT, FrgShaderCar);

    float translation_x = -0.3f;
    float translation_y = 0.0f;
    float translation_z = 0.0f;
    float rotation_x = 0.0f;
    float rotation_y = 0.0f;
    float rotation_z = 1.0f;
    float rotation_degrees = -180.0;

    car_model->setup_model_matrix (
        translation_x,
        translation_y,
        translation_z,
        rotation_x,
        rotation_y,
        rotation_z,
        rotation_degrees);

    return car_model;
}

static int
run_stitcher (
    const SmartPtr<Stitcher> &stitcher,
#if ENABLE_DNN
    const SmartPtr<DnnInferenceEngine> &infer_engine,
#endif
    const SmartPtr<RenderOsgModel> &model,
    const SVStreams &ins,
    const SVStreams &outs)
{
    XCamReturn ret = XCAM_RETURN_NO_ERROR;

    Mutex mutex;

    VideoBufferList in_buffers;
    for (uint32_t i = 0; i < ins.size (); ++i) {
        CHECK (ins[i]->rewind (), "rewind buffer from file(%s) failed", ins[i]->get_file_name ());
    }

    do {
        in_buffers.clear ();

        for (uint32_t i = 0; i < ins.size (); ++i) {
            ret = ins[i]->read_buf();
            if (ret == XCAM_RETURN_BYPASS)
                break;
            CHECK (ret, "read buffer from file(%s) failed.", ins[i]->get_file_name ());
            in_buffers.push_back (ins[i]->get_buf ());
        }
        if (ret == XCAM_RETURN_BYPASS) {
            XCAM_LOG_DEBUG ("XCAM_RETURN_BYPASS \n");
            break;
        }

        {
            SmartLock locker (mutex);
            CHECK (
                stitcher->stitch_buffers (in_buffers, outs[0]->get_buf ()),
                "stitch buffer failed.");
        }

#if ENABLE_DNN
        if (infer_engine.ptr ()) {

            VideoBufferList detect_buffers;
            detect_buffers.clear ();
            detect_buffers.push_back (outs[0]->get_buf ());

            infer_engine->set_inference_data (detect_buffers);

            if (infer_engine->ready_to_start ()) {
                CHECK (
                    infer_engine->start (),
                    "inference failed!");
            }

            size_t batch_size = infer_engine->get_output_size ();
            if (batch_size != detect_buffers.size ()) {
                XCAM_LOG_DEBUG ( "Number of images: %d ", detect_buffers.size ());
                batch_size = std::min(batch_size, detect_buffers.size ());
                XCAM_LOG_DEBUG ( "Number of images to be processed is: %d ", batch_size);
            }

            uint32_t blob_size = 0;
            float* result_ptr = NULL;

            for (uint32_t batch_idx = 0; batch_idx < batch_size; batch_idx ++) {
                result_ptr = (float*)infer_engine->get_inference_results (batch_idx, blob_size);
                if (NULL == result_ptr) {
                    continue;
                }

                std::vector<Vec4i> boxes;
                std::vector<int32_t> classes;
                uint32_t image_width = infer_engine->get_input_image_width (batch_idx);
                uint32_t image_height = infer_engine->get_input_image_height (batch_idx);

                SmartPtr<DnnObjectDetection> object_detector = infer_engine.dynamic_cast_ptr<DnnObjectDetection> ();
                CHECK (
                    object_detector->get_bounding_boxes (result_ptr, batch_idx, boxes, classes),
                    "get bounding box failed!");

                SmartPtr<VideoBuffer> render_image = detect_buffers.front ();
                uint8_t* detect_image = render_image->map ();

                CHECK (
                    XCamDNN::draw_bounding_boxes (detect_image,
                                                  image_width, image_height, DnnInferImageFormatNV12,
                                                  boxes, classes, 2),
                    "Draw bounding boxes failed!" );

                render_image->unmap ();
            }
        }
#endif

        model->update_texture (outs[0]->get_buf ());

        FPS_CALCULATION (render_surround_view, XCAM_OBJ_DUR_FRAME_NUM);
    } while (true);

    return 0;
}

#if ENABLE_DNN
static SmartPtr<DnnInferenceEngine>
create_dnn_inference_engine ()
{
    XCAM_LOG_DEBUG ("0. Get inference engine configurations");

    std::string dnn_model_name = std::string (DNN_MODEL_NAME);
    std::string dnn_model_path = FISHEYE_CONFIG_PATH + dnn_model_name;

    const char *env_path = std::getenv (FISHEYE_CONFIG_ENV_VAR);
    if (env_path) {
        dnn_model_path.clear ();
        dnn_model_path = std::string (env_path) + dnn_model_name;
    }

    DnnInferConfig infer_config;
    infer_config.target_id = DnnInferDeviceCPU;
    infer_config.model_type = DnnInferObjectDetection;
    infer_config.model_filename = dnn_model_path.c_str ();

    SmartPtr<DnnInferenceEngine> infer_engine;
    infer_engine = new DnnObjectDetection (infer_config);
    if (NULL == infer_engine.ptr ()) {
        XCAM_LOG_ERROR ("create dnn object detection inference engine failed!");
        return NULL;
    }

    infer_engine->get_model_input_info (infer_config.input_infos);
    for (uint32_t i = 0; i < infer_config.input_infos.numbers; i++) {
        infer_config.input_infos.data_type[i] = DnnInferDataTypeImage;
        infer_engine->set_input_precision (i, DnnInferPrecisionU8);
        XCAM_LOG_DEBUG ("Idx %d : [%d X %d X %d] , [%d %d %d], batch size = %d", i,
                        infer_config.input_infos.width[i], infer_config.input_infos.height[i], infer_config.input_infos.channels[i],
                        infer_config.input_infos.precision[i], infer_config.input_infos.layout[i], infer_config.input_infos.data_type[i],
                        infer_config.input_infos.batch_size);
    }

    infer_engine->get_model_output_info (infer_config.output_infos);
    XCAM_LOG_DEBUG ("Output info (numbers %d) :", infer_config.output_infos.numbers);

    for (uint32_t i = 0; i < infer_config.output_infos.numbers; i++) {
        infer_engine->set_output_precision (i, DnnInferPrecisionFP32);
        XCAM_LOG_DEBUG ("Idx %d : [%d X %d X %d] , [%d %d %d], batch size = %d", i,
                        infer_config.output_infos.width[i],
                        infer_config.output_infos.height[i],
                        infer_config.output_infos.channels[i],
                        infer_config.output_infos.precision[i],
                        infer_config.output_infos.layout[i],
                        infer_config.output_infos.data_type[i],
                        infer_config.output_infos.batch_size);
    }

    if (XCAM_RETURN_NO_ERROR != infer_engine->load_model (infer_config)) {
        XCAM_LOG_ERROR ("load model failed!");
        return NULL;
    }

    return infer_engine;
}


#endif

static void usage(const char* arg0)
{
    printf ("Usage:\n"
            "%s --module MODULE --input0 input.nv12 --input1 input1.nv12 --input2 input2.nv12 ...\n"
            "\t--module            processing module, selected from: soft, gles, vulkan\n"
            "                      read calibration files from exported path $FISHEYE_CONFIG_PATH\n"
            "\t--input             input image(NV12)\n"
            "\t--in-w              optional, input width, default: 1280\n"
            "\t--in-h              optional, input height, default: 800\n"
            "\t--out-w             optional, output width, default: 1920\n"
            "\t--out-h             optional, output height, default: 640\n"
            "\t--scale-mode        optional, scaling mode for geometric mapping,\n"
            "\t                    select from [singleconst/dualconst/dualcurve], default: singleconst\n"
            "\t--fm-mode           optional, feature match mode,\n"
#if HAVE_OPENCV
            "\t                    select from [none/default/cluster/capi], default: none\n"
#else
            "\t                    select from [none], default: none\n"
#endif
            "\t--car               optional, car model name\n"
            "\t--detect            optional, pedestrian & vehicle detection, default: disable\n"
            "\t--loop              optional, how many loops need to run, default: 1\n"
            "\t--help              usage\n",
            arg0);
}

int main (int argc, char *argv[])
{
    uint32_t input_width = 1280;
    uint32_t input_height = 800;
    uint32_t output_width = 1920;
    uint32_t output_height = 640;

    SVStreams ins;
    SVStreams outs;
    PUSH_STREAM (SVStream, outs, NULL);

    const char *car_name = NULL;

    SVModule module = SVModuleGLES;
    GeoMapScaleMode scale_mode = ScaleSingleConst;
    FeatureMatchMode fm_mode = FMNone;

    bool enable_detect = false;
    int loop = 1;

    const struct option long_opts[] = {
        {"module", required_argument, NULL, 'm'},
        {"input", required_argument, NULL, 'i'},
        {"in-w", required_argument, NULL, 'w'},
        {"in-h", required_argument, NULL, 'h'},
        {"out-w", required_argument, NULL, 'W'},
        {"out-h", required_argument, NULL, 'H'},
        {"scale-mode", required_argument, NULL, 'S'},
        {"fm-mode", required_argument, NULL, 'F'},
        {"car", required_argument, NULL, 'c'},
        {"detect", required_argument, NULL, 'd'},
        {"loop", required_argument, NULL, 'L'},
        {"help", no_argument, NULL, 'e'},
        {NULL, 0, NULL, 0},
    };

    int opt = -1;
    while ((opt = getopt_long(argc, argv, "", long_opts, NULL)) != -1) {
        switch (opt) {
        case 'm':
            XCAM_ASSERT (optarg);
            if (!strcasecmp (optarg, "soft")) {
                module = SVModuleSoft;
            } else if (!strcasecmp (optarg, "gles")) {
                module = SVModuleGLES;
            } else if (!strcasecmp (optarg, "vulkan")) {
                module = SVModuleVulkan;
            }
            break;
        case 'i':
            XCAM_ASSERT (optarg);
            PUSH_STREAM (SVStream, ins, optarg);
            break;
        case 'w':
            XCAM_ASSERT (optarg);
            input_width = (uint32_t)atoi(optarg);
            break;
        case 'h':
            XCAM_ASSERT (optarg);
            input_height = (uint32_t)atoi(optarg);
            break;
        case 'W':
            XCAM_ASSERT (optarg);
            output_width = (uint32_t)atoi(optarg);
            break;
        case 'H':
            XCAM_ASSERT (optarg);
            output_height = (uint32_t)atoi(optarg);
            break;
        case 'S':
            XCAM_ASSERT (optarg);
            if (!strcasecmp (optarg, "singleconst"))
                scale_mode = ScaleSingleConst;
            else if (!strcasecmp (optarg, "dualconst"))
                scale_mode = ScaleDualConst;
            else if (!strcasecmp (optarg, "dualcurve"))
                scale_mode = ScaleDualCurve;
            else {
                XCAM_LOG_ERROR ("GeoMapScaleMode unknown mode: %s", optarg);
                usage (argv[0]);
                return -1;
            }
            break;
        case 'F':
            XCAM_ASSERT (optarg);
            if (!strcasecmp (optarg, "none"))
                fm_mode = FMNone;
#if HAVE_OPENCV
            else if (!strcasecmp (optarg, "default"))
                fm_mode = FMDefault;
            else if (!strcasecmp (optarg, "cluster"))
                fm_mode = FMCluster;
            else if (!strcasecmp (optarg, "capi"))
                fm_mode = FMCapi;
#endif
            else {
                XCAM_LOG_ERROR ("unsupported feature match mode: %s", optarg);
                usage (argv[0]);
                return -1;
            }
            break;
        case 'c':
            XCAM_ASSERT (optarg);
            car_name = optarg;
            break;
        case 'd':
            XCAM_ASSERT (optarg);
            enable_detect = (strcasecmp (optarg, "true") == 0 ? true : false);
            break;
        case 'L':
            loop = atoi(optarg);
            break;
        case 'e':
            usage (argv[0]);
            return 0;
        default:
            XCAM_LOG_ERROR ("getopt_long return unknown value: %c", opt);
            usage (argv[0]);
            return -1;
        }
    }

    if (optind < argc || argc < 2) {
        XCAM_LOG_ERROR ("unknown option %s", argv[optind]);
        usage (argv[0]);
        return -1;
    }

    CHECK_EXP (ins.size () == 4, "surrond view needs 4 input streams");
    for (uint32_t i = 0; i < ins.size (); ++i) {
        CHECK_EXP (ins[i].ptr (), "input stream is NULL, index:%d", i);
        CHECK_EXP (strlen (ins[i]->get_file_name ()), "input file name was not set, index:%d", i);
    }

    CHECK_EXP (outs.size () == 1 && outs[0].ptr (), "surrond view needs 1 output stream");

    for (uint32_t i = 0; i < ins.size (); ++i) {
        printf ("input%d file:\t\t%s\n", i, ins[i]->get_file_name ());
    }
    printf ("input width:\t\t%d\n", input_width);
    printf ("input height:\t\t%d\n", input_height);
    printf ("output width:\t\t%d\n", output_width);
    printf ("output height:\t\t%d\n", output_height);
    printf ("scaling mode:\t\t%s\n", (scale_mode == ScaleSingleConst) ? "singleconst" :
            ((scale_mode == ScaleDualConst) ? "dualconst" : "dualcurve"));
    printf ("feature match:\t\t%s\n", (fm_mode == FMNone) ? "none" :
            ((fm_mode == FMDefault ) ? "default" : ((fm_mode == FMCluster) ? "cluster" : "capi")));
    printf ("car model name:\t\t%s\n", car_name != NULL ? car_name : "Not specified, use default model");
    printf ("loop count:\t\t%d\n", loop);

    if (module == SVModuleGLES) {
#if HAVE_GLES
        if (scale_mode == ScaleDualCurve) {
            XCAM_LOG_WARNING ("GLES module does not support Dual Curve scale mode currently, change to Single Const scale mode");
            scale_mode = ScaleSingleConst;
        }
        SmartPtr<EGLBase> egl;
        egl = new EGLBase ();
        XCAM_ASSERT (egl.ptr ());
        XCAM_FAIL_RETURN (ERROR, egl->init (), -1, "init EGL failed");
#else
        if (module == SVModuleGLES) {
            XCAM_LOG_ERROR ("GLES module is unsupported");
            return -1;
        }
#endif
    }

    if (module == SVModuleVulkan) {
#if HAVE_VULKAN
        if (scale_mode != ScaleSingleConst) {
            XCAM_LOG_WARNING ("vulkan module only support singleconst scale mode currently, change to Single Const scale mode");
            scale_mode = ScaleSingleConst;
        }

        SmartPtr<VKDevice> vk_dev = VKDevice::default_device ();
        for (uint32_t i = 0; i < ins.size (); ++i) {
            ins[i]->set_vk_device (vk_dev);
        }
        XCAM_ASSERT (outs[0].ptr ());
        outs[0]->set_vk_device (vk_dev);
#else
        XCAM_LOG_ERROR ("vulkan module is unsupported");
        return -1;
#endif
    }

    for (uint32_t i = 0; i < ins.size (); ++i) {
        ins[i]->set_module (module);
        ins[i]->set_buf_size (input_width, input_height);
        CHECK (ins[i]->create_buf_pool (6), "create buffer pool failed");
        CHECK (ins[i]->open_reader ("rb"), "open input file(%s) failed", ins[i]->get_file_name ());
    }

    outs[0]->set_buf_size (output_width, output_height);

    SmartPtr<Stitcher> stitcher = create_stitcher (outs[0], module);
    XCAM_ASSERT (stitcher.ptr ());

    stitcher->set_camera_num (ins.size ());
    stitcher->set_output_size (output_width, output_height);
    stitcher->set_scale_mode (scale_mode);
    stitcher->set_fm_mode (fm_mode);

    stitcher->set_viewpoints_range (viewpoints_range);
    stitcher->set_intrinsic_names (intrinsic_names);
    stitcher->set_extrinsic_names (extrinsic_names);

    BowlDataConfig bowl;
    bowl.wall_height = 1800.0f;
    bowl.ground_length = 3000.0f;
    bowl.angle_start = 0.0f;
    bowl.angle_end = 360.0f;
    stitcher->set_bowl_config (bowl);

    SmartPtr<RenderOsgViewer> render = new RenderOsgViewer ();

    SmartPtr<RenderOsgModel> sv_model = create_surround_view_model (stitcher, output_width, output_height);
    render->add_model (sv_model);

    SmartPtr<RenderOsgModel> car_model = create_car_model (car_name);
    render->add_model (car_model);

    render->validate_model_groups ();

    render->start_render ();

#if ENABLE_DNN
    SmartPtr<DnnInferenceEngine> infer_engine;
    if (enable_detect) {
        infer_engine = create_dnn_inference_engine ();
    }
#else
    XCAM_UNUSED (enable_detect);
#endif

    while (loop--) {
#if ENABLE_DNN
        if (enable_detect) {
            CHECK_EXP (
                run_stitcher (stitcher, infer_engine, sv_model, ins, outs) == 0,
                "run stitcher failed");
        } else {
            CHECK_EXP (
                run_stitcher (stitcher, NULL, sv_model, ins, outs) == 0,
                "run stitcher failed");
        }
#else
        CHECK_EXP (
            run_stitcher (stitcher, sv_model, ins, outs) == 0,
            "run stitcher failed");
#endif
    }

    render->stop_render ();

    return 0;
}
