import { entries } from "../iters";
import { greedy } from "./greedy";
/** @internal */
function buildOperator(nodeWidth, nodeHeight, edgeGap, sizeVal) {
    function zherebkoCall(dag) {
        var _a;
        // topological sort
        const levels = [];
        let numLevels = 0;
        let last;
        for (const node of dag.idescendants("before")) {
            if (last !== undefined && last.nchildLinksTo(node) > 1) {
                ++numLevels;
            }
            levels.push([node, numLevels++]);
            last = node;
        }
        // get link indices
        const indices = greedy(levels);
        // map to coordinates
        let minIndex = 0;
        let maxIndex = 0;
        for (const inds of indices.values()) {
            for (const ind of inds) {
                if (ind !== undefined) {
                    minIndex = Math.min(minIndex, ind);
                    maxIndex = Math.max(maxIndex, ind);
                }
            }
        }
        // assign node positions
        const nodex = -minIndex * edgeGap + nodeWidth / 2;
        for (const [node, layer] of levels) {
            node.x = nodex;
            node.y = (layer + 0.5) * nodeHeight;
        }
        // assign link points
        for (const source of dag) {
            const inds = (_a = indices.get(source)) !== null && _a !== void 0 ? _a : [];
            // we iterate like this instead of ilinks to get the indices among
            // children as a way to dedup links
            for (const [index, { target, points }] of entries(source.ichildLinks())) {
                points.length = 0;
                points.push({ x: source.x, y: source.y });
                const ind = inds[index];
                if (ind !== undefined) {
                    // assumed long link
                    const x = (ind - minIndex + 0.5) * edgeGap +
                        (ind > 0 ? nodeWidth - edgeGap : 0);
                    const y1 = source.y + nodeHeight;
                    const y2 = target.y - nodeHeight;
                    if (y2 - y1 > nodeHeight / 2) {
                        points.push({ x: x, y: y1 }, { x: x, y: y2 });
                    }
                    else {
                        points.push({ x: x, y: y1 });
                    }
                }
                points.push({ x: target.x, y: target.y });
            }
        }
        const width = (maxIndex - minIndex) * edgeGap + nodeWidth;
        const height = numLevels * nodeHeight;
        if (sizeVal === null) {
            return { width, height };
        }
        else {
            // rescale to new size
            const [newWidth, newHeight] = sizeVal;
            for (const [node] of levels) {
                node.x *= newWidth / width;
                node.y *= newHeight / height;
            }
            for (const { points } of dag.ilinks()) {
                const newPoints = points.map(({ x, y }) => ({
                    x: (x * newWidth) / width,
                    y: (y * newHeight) / height,
                }));
                points.splice(0, points.length, ...newPoints);
            }
            return { width: newWidth, height: newHeight };
        }
    }
    function nodeSize(val) {
        if (val === undefined) {
            return [nodeWidth, nodeHeight, edgeGap];
        }
        else {
            const [newWidth, newHeight, newGap] = val;
            return buildOperator(newWidth, newHeight, newGap, sizeVal);
        }
    }
    zherebkoCall.nodeSize = nodeSize;
    function size(val) {
        if (val !== undefined) {
            return buildOperator(nodeWidth, nodeHeight, edgeGap, val);
        }
        else if (sizeVal === null) {
            return sizeVal;
        }
        else {
            const [width, height] = sizeVal;
            return [width, height];
        }
    }
    zherebkoCall.size = size;
    return zherebkoCall;
}
/**
 * Create a new {@link ZherebkoOperator} with default settings.
 */
export function zherebko(...args) {
    if (args.length) {
        throw new Error(`got arguments to zherebko(${args}), but constructor takes no arguments.`);
    }
    return buildOperator(1, 1, 1, null);
}
