// REQUIRES: target-is-powerpc64le
// RUN: %clang_builtins %s %librt -o %t && %run %t
#include <stdio.h>
#include "DD.h"

long double __gcc_qdiv(long double x, long double y);
int memcmp(const void *, const void *, __typeof__(sizeof(0)));

double testAccuracy();
int testEdgeCases();

int main(int argc, char *argv[]) {
	if (testEdgeCases())
		return 1;

	if (testAccuracy() > 4.0)
		return 1;

	return 0;
}

struct testVector {
	double xhi; double xlo;
	double yhi; double ylo;
	double rhi; double rlo;
};

#define INFINITY __builtin_inf()
#define HUGE 0x1.fffffffffffffp1023
#define QNAN __builtin_nan("")

const struct testVector edgeCases[] = {
{	0.0,		0.0,		0.0,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,		0.0,		0.0,	    QNAN,		0.0			},
{   0.0,		0.0,	   -0.0,		0.0,	    QNAN,		0.0			},
{  -0.0,		0.0,	   -0.0,		0.0,	    QNAN,		0.0			},

{   INFINITY,	0.0,		0.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,		0.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -0.0,		0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -0.0,		0.0,	    INFINITY,	0.0			},
{   INFINITY,	0.0,	    1.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    1.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -1.0,		0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -1.0,		0.0,	    INFINITY,	0.0			},
{   INFINITY,	0.0,	    HUGE,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    HUGE,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -HUGE,		0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -HUGE,		0.0,	    INFINITY,	0.0			},
{   INFINITY,	0.0,	    INFINITY,	0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	    INFINITY,	0.0,	   -QNAN,		0.0			},
{   INFINITY,	0.0,	   -INFINITY,	0.0,	   -QNAN,		0.0			},
{  -INFINITY,	0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{   0.0,		0.0,		INFINITY,	0.0,		0.0,		0.0			},
{   0.0,		0.0,	   -INFINITY,	0.0,	   -0.0,		0.0			},
{  -0.0,		0.0,		INFINITY,	0.0,	   -0.0,		0.0			},
{  -0.0,		0.0,	   -INFINITY,	0.0,	    0.0,		0.0			},
{   1.0,		0.0,		INFINITY,	0.0,		0.0,		0.0			},
{   1.0,		0.0,	   -INFINITY,	0.0,	   -0.0,		0.0			},
{  -1.0,		0.0,		INFINITY,	0.0,	   -0.0,		0.0			},
{  -1.0,		0.0,	   -INFINITY,	0.0,	    0.0,		0.0			},
{   HUGE,		0.0,		INFINITY,	0.0,		0.0,		0.0			},
{   HUGE,		0.0,	   -INFINITY,	0.0,	   -0.0,		0.0			},
{  -HUGE,		0.0,		INFINITY,	0.0,	   -0.0,		0.0			},
{  -HUGE,		0.0,	   -INFINITY,	0.0,	    0.0,		0.0			},

{   QNAN,		0.0,		0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    QNAN,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -QNAN,		0.0,	    QNAN,		0.0			},
{	0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
};

const int numEdgeCases = sizeof(edgeCases) / sizeof(struct testVector);


int testEdgeCases() {
	int i;
	DD a, b, c, r;
	for (i=0; i<numEdgeCases; ++i) {
		a.hi = edgeCases[i].xhi;
		a.lo = edgeCases[i].xlo;
		b.hi = edgeCases[i].yhi;
		b.lo = edgeCases[i].ylo;
		r.hi = edgeCases[i].rhi;
		r.lo = edgeCases[i].rlo;

		int error;

		DD c = { .ld = __gcc_qdiv(a.ld, b.ld) };

		if (r.hi != r.hi) {
			if (c.hi == c.hi)
				error = 1;
			else if (c.lo != 0.0)
				error = 1;
			else
				error = 0;
		}

		else if (r.hi == 0.0)
			error = memcmp(&c, &r, sizeof(DD));

		else
			error = ((c.hi != r.hi) || (c.lo != r.lo));

		if (error) {
			printf("Error on edge case %a / %a: expected (%a, %a), got (%a, %a).\n", a.hi, b.hi, r.hi, r.lo, c.hi, c.lo);
			return 1;
		}
	}

	return 0;
}


/*

 #include <stdio.h>
 #include <stdlib.h>
 #include <mpfr.h>
 #include <math.h>

 #ifdef __x86_64__
 #define randlength 2
 #else
 #define randlength 4
 #endif

 void printTest(mpfr_t a, mpfr_t b, mpfr_t c) {
 static const double infinityD = __builtin_inf();

 MPFR_DECL_INIT(tmp, 53);

 double ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 double alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);

 double bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 double blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);

 double chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 double clo = isinf(chi) ? 0.0 : mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 }

 int main(int argc, char *argv[]) {

 MPFR_DECL_INIT(a, 106);
 MPFR_DECL_INIT(b, 106);
 MPFR_DECL_INIT(c, 106);

 int exponent_range = atoi(argv[1]);

 int i;
 for (i=0; i<128; ++i) {
 mpfr_random2(a, randlength, exponent_range);
 mpfr_random2(b, randlength, exponent_range);
 mpfr_div(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(b, b, GMP_RNDN);
 mpfr_div(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(a, a, GMP_RNDN);
 mpfr_neg(b, b, GMP_RNDN);
 mpfr_div(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(b, b, GMP_RNDN);
 mpfr_div(c, a, b, GMP_RNDN);
 printTest(a, b, c);
 }
 return 0;
 }

 */

const struct testVector accuracyTests[] = {
{0x1.00001ffffff00p+419, 0x1.fffff83ff8000p+364, 0x1.00ffc00ffff00p-598, 0x1.fe0007fffff00p-654, 0x1.fe02bca42ad25p+1016, 0x1.29aa5d717c42cp+961},
{0x1.00001ffffff00p+419, 0x1.fffff83ff8000p+364, -0x1.00ffc00ffff00p-598, -0x1.fe0007fffff00p-654, -0x1.fe02bca42ad25p+1016, -0x1.29aa5d717c42cp+961},
{-0x1.00001ffffff00p+419, -0x1.fffff83ff8000p+364, 0x1.00ffc00ffff00p-598, 0x1.fe0007fffff00p-654, -0x1.fe02bca42ad25p+1016, -0x1.29aa5d717c42cp+961},
{-0x1.00001ffffff00p+419, -0x1.fffff83ff8000p+364, -0x1.00ffc00ffff00p-598, -0x1.fe0007fffff00p-654, 0x1.fe02bca42ad25p+1016, 0x1.29aa5d717c42cp+961},
{0x1.00001fffff800p-575, 0x1.fffffffe00000p-649, 0x1.3fffffff80000p-17, 0x1.ffffffff00000p-90, 0x1.9999cccd6fd72p-559, -0x1.476d58085a038p-615},
{0x1.00001fffff800p-575, 0x1.fffffffe00000p-649, -0x1.3fffffff80000p-17, -0x1.ffffffff00000p-90, -0x1.9999cccd6fd72p-559, 0x1.476d58085a038p-615},
{-0x1.00001fffff800p-575, -0x1.fffffffe00000p-649, 0x1.3fffffff80000p-17, 0x1.ffffffff00000p-90, -0x1.9999cccd6fd72p-559, 0x1.476d58085a038p-615},
{-0x1.00001fffff800p-575, -0x1.fffffffe00000p-649, -0x1.3fffffff80000p-17, -0x1.ffffffff00000p-90, 0x1.9999cccd6fd72p-559, -0x1.476d58085a038p-615},
{0x1.0000000800000p-557, -0x1.0000000000000p-629, 0x1.000003fffc002p+323, -0x1.0000000000000p+218, 0x1.fffff810081f8p-881, -0x1.00fb820825b80p-939},
{0x1.0000000800000p-557, -0x1.0000000000000p-629, -0x1.000003fffc002p+323, 0x1.0000000000000p+218, -0x1.fffff810081f8p-881, 0x1.00fb820825b80p-939},
{-0x1.0000000800000p-557, 0x1.0000000000000p-629, 0x1.000003fffc002p+323, -0x1.0000000000000p+218, -0x1.fffff810081f8p-881, 0x1.00fb820825b80p-939},
{-0x1.0000000800000p-557, 0x1.0000000000000p-629, -0x1.000003fffc002p+323, 0x1.0000000000000p+218, 0x1.fffff810081f8p-881, -0x1.00fb820825b80p-939},
{0x1.000003ffffffep+356, -0x1.ffe0080000000p+272, 0x1.0800000000000p-150, -0x1.0000000000000p-246, 0x1.f07c26c9b26c6p+505, -0x1.745d1764d6558p+450},
{0x1.000003ffffffep+356, -0x1.ffe0080000000p+272, -0x1.0800000000000p-150, 0x1.0000000000000p-246, -0x1.f07c26c9b26c6p+505, 0x1.745d1764d6558p+450},
{-0x1.000003ffffffep+356, 0x1.ffe0080000000p+272, 0x1.0800000000000p-150, -0x1.0000000000000p-246, -0x1.f07c26c9b26c6p+505, 0x1.745d1764d6558p+450},
{-0x1.000003ffffffep+356, 0x1.ffe0080000000p+272, -0x1.0800000000000p-150, 0x1.0000000000000p-246, 0x1.f07c26c9b26c6p+505, -0x1.745d1764d6558p+450},
{0x1.0000007fffc00p-14, 0x1.efe0000000000p-108, 0x1.007fffffe0000p-312, 0x0.0000000000000p+0, 0x1.ff0080bfdf705p+297, 0x1.fb017ac5608fap+243},
{0x1.0000007fffc00p-14, 0x1.efe0000000000p-108, -0x1.007fffffe0000p-312, 0x0.0000000000000p+0, -0x1.ff0080bfdf705p+297, -0x1.fb017ac5608fap+243},
{-0x1.0000007fffc00p-14, -0x1.efe0000000000p-108, 0x1.007fffffe0000p-312, 0x0.0000000000000p+0, -0x1.ff0080bfdf705p+297, -0x1.fb017ac5608fap+243},
{-0x1.0000007fffc00p-14, -0x1.efe0000000000p-108, -0x1.007fffffe0000p-312, 0x0.0000000000000p+0, 0x1.ff0080bfdf705p+297, 0x1.fb017ac5608fap+243},
{0x1.3fe0000100000p-744, -0x1.0000000000000p-817, 0x1.3fffffffffe00p+293, 0x1.fe00040000000p+218, 0x1.ffccccce60000p-1038, 0x0.0000000000000p+0},
{0x1.3fe0000100000p-744, -0x1.0000000000000p-817, -0x1.3fffffffffe00p+293, -0x1.fe00040000000p+218, -0x1.ffccccce60000p-1038, -0x0.0000000000000p+0},
{-0x1.3fe0000100000p-744, 0x1.0000000000000p-817, 0x1.3fffffffffe00p+293, 0x1.fe00040000000p+218, -0x1.ffccccce60000p-1038, -0x0.0000000000000p+0},
{-0x1.3fe0000100000p-744, 0x1.0000000000000p-817, -0x1.3fffffffffe00p+293, -0x1.fe00040000000p+218, 0x1.ffccccce60000p-1038, 0x0.0000000000000p+0},
{0x1.0001fff800000p+0, 0x1.ffffff0000ffep-54, 0x1.00ff000000800p+11, -0x1.0000000800000p-55, 0x1.fe07f601f30ffp-12, -0x1.b3714144d2a90p-69},
{0x1.0001fff800000p+0, 0x1.ffffff0000ffep-54, -0x1.00ff000000800p+11, 0x1.0000000800000p-55, -0x1.fe07f601f30ffp-12, 0x1.b3714144d2a90p-69},
{-0x1.0001fff800000p+0, -0x1.ffffff0000ffep-54, 0x1.00ff000000800p+11, -0x1.0000000800000p-55, -0x1.fe07f601f30ffp-12, 0x1.b3714144d2a90p-69},
{-0x1.0001fff800000p+0, -0x1.ffffff0000ffep-54, -0x1.00ff000000800p+11, 0x1.0000000800000p-55, 0x1.fe07f601f30ffp-12, -0x1.b3714144d2a90p-69},
{0x1.0000800000000p-381, -0x1.ffffffe400000p-444, 0x1.000001ffffffap+606, -0x1.ffe1f80400000p+531, 0x1.00007dffff046p-987, -0x1.d940000000000p-1051},
{0x1.0000800000000p-381, -0x1.ffffffe400000p-444, -0x1.000001ffffffap+606, 0x1.ffe1f80400000p+531, -0x1.00007dffff046p-987, 0x1.d940000000000p-1051},
{-0x1.0000800000000p-381, 0x1.ffffffe400000p-444, 0x1.000001ffffffap+606, -0x1.ffe1f80400000p+531, -0x1.00007dffff046p-987, 0x1.d940000000000p-1051},
{-0x1.0000800000000p-381, 0x1.ffffffe400000p-444, -0x1.000001ffffffap+606, 0x1.ffe1f80400000p+531, 0x1.00007dffff046p-987, -0x1.d940000000000p-1051},
{0x1.0000010000000p+772, -0x1.0000000000000p+667, 0x1.007fffffffc00p+750, 0x1.fffffc0000000p+679, 0x1.ff0081bf20ef5p+21, -0x1.e0f5922874ff2p-33},
{0x1.0000010000000p+772, -0x1.0000000000000p+667, -0x1.007fffffffc00p+750, -0x1.fffffc0000000p+679, -0x1.ff0081bf20ef5p+21, 0x1.e0f5922874ff2p-33},
{-0x1.0000010000000p+772, 0x1.0000000000000p+667, 0x1.007fffffffc00p+750, 0x1.fffffc0000000p+679, -0x1.ff0081bf20ef5p+21, 0x1.e0f5922874ff2p-33},
{-0x1.0000010000000p+772, 0x1.0000000000000p+667, -0x1.007fffffffc00p+750, -0x1.fffffc0000000p+679, 0x1.ff0081bf20ef5p+21, -0x1.e0f5922874ff2p-33},
{0x1.0080000000000p-547, -0x1.0000000000000p-620, 0x1.001ffffffc000p+963, 0x1.fffffffff001cp+908, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0080000000000p-547, -0x1.0000000000000p-620, -0x1.001ffffffc000p+963, -0x1.fffffffff001cp+908, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0080000000000p-547, 0x1.0000000000000p-620, 0x1.001ffffffc000p+963, 0x1.fffffffff001cp+908, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0080000000000p-547, 0x1.0000000000000p-620, -0x1.001ffffffc000p+963, -0x1.fffffffff001cp+908, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000001fffff80p-138, 0x1.fffffc001fffcp-193, 0x1.00001ffff8000p-554, 0x1.00007f001e000p-609, 0x1.ffffc40017700p+415, -0x1.9c017d7fb3e20p+358},
{0x1.000001fffff80p-138, 0x1.fffffc001fffcp-193, -0x1.00001ffff8000p-554, -0x1.00007f001e000p-609, -0x1.ffffc40017700p+415, 0x1.9c017d7fb3e20p+358},
{-0x1.000001fffff80p-138, -0x1.fffffc001fffcp-193, 0x1.00001ffff8000p-554, 0x1.00007f001e000p-609, -0x1.ffffc40017700p+415, 0x1.9c017d7fb3e20p+358},
{-0x1.000001fffff80p-138, -0x1.fffffc001fffcp-193, -0x1.00001ffff8000p-554, -0x1.00007f001e000p-609, 0x1.ffffc40017700p+415, -0x1.9c017d7fb3e20p+358},
{0x1.00f3ffffff000p-238, 0x1.fff8000000000p-306, 0x1.3dfffffffc000p-529, 0x1.ff000003ffe00p-590, 0x1.9db5efe640c6bp+290, 0x1.d0e9f9b84e302p+236},
{0x1.00f3ffffff000p-238, 0x1.fff8000000000p-306, -0x1.3dfffffffc000p-529, -0x1.ff000003ffe00p-590, -0x1.9db5efe640c6bp+290, -0x1.d0e9f9b84e302p+236},
{-0x1.00f3ffffff000p-238, -0x1.fff8000000000p-306, 0x1.3dfffffffc000p-529, 0x1.ff000003ffe00p-590, -0x1.9db5efe640c6bp+290, -0x1.d0e9f9b84e302p+236},
{-0x1.00f3ffffff000p-238, -0x1.fff8000000000p-306, -0x1.3dfffffffc000p-529, -0x1.ff000003ffe00p-590, 0x1.9db5efe640c6bp+290, 0x1.d0e9f9b84e302p+236},
{0x1.0000001ffff80p-69, 0x1.fffffdfffff8ep-123, 0x1.0000000000000p-311, -0x1.0000000000000p-381, 0x1.0000001ffff81p+242, -0x1.ffff01ffffe72p+188},
{0x1.0000001ffff80p-69, 0x1.fffffdfffff8ep-123, -0x1.0000000000000p-311, 0x1.0000000000000p-381, -0x1.0000001ffff81p+242, 0x1.ffff01ffffe72p+188},
{-0x1.0000001ffff80p-69, -0x1.fffffdfffff8ep-123, 0x1.0000000000000p-311, -0x1.0000000000000p-381, -0x1.0000001ffff81p+242, 0x1.ffff01ffffe72p+188},
{-0x1.0000001ffff80p-69, -0x1.fffffdfffff8ep-123, -0x1.0000000000000p-311, 0x1.0000000000000p-381, 0x1.0000001ffff81p+242, -0x1.ffff01ffffe72p+188},
{0x1.001fffffffe00p-160, 0x0.0000000000000p+0, 0x1.0000002000000p+499, -0x1.0000000000000p+399, 0x1.001fffdffbe00p-659, 0x1.0020ffdffbe40p-713},
{0x1.001fffffffe00p-160, 0x0.0000000000000p+0, -0x1.0000002000000p+499, 0x1.0000000000000p+399, -0x1.001fffdffbe00p-659, -0x1.0020ffdffbe40p-713},
{-0x1.001fffffffe00p-160, 0x0.0000000000000p+0, 0x1.0000002000000p+499, -0x1.0000000000000p+399, -0x1.001fffdffbe00p-659, -0x1.0020ffdffbe40p-713},
{-0x1.001fffffffe00p-160, 0x0.0000000000000p+0, -0x1.0000002000000p+499, 0x1.0000000000000p+399, 0x1.001fffdffbe00p-659, 0x1.0020ffdffbe40p-713},
{0x1.01ffffffe0000p+182, 0x1.c000000200000p+121, 0x1.0001fff800ffep-8, 0x1.c000000000000p-111, 0x1.01fdfc0bf6d61p+190, 0x1.41500c3427868p+136},
{0x1.01ffffffe0000p+182, 0x1.c000000200000p+121, -0x1.0001fff800ffep-8, -0x1.c000000000000p-111, -0x1.01fdfc0bf6d61p+190, -0x1.41500c3427868p+136},
{-0x1.01ffffffe0000p+182, -0x1.c000000200000p+121, 0x1.0001fff800ffep-8, 0x1.c000000000000p-111, -0x1.01fdfc0bf6d61p+190, -0x1.41500c3427868p+136},
{-0x1.01ffffffe0000p+182, -0x1.c000000200000p+121, -0x1.0001fff800ffep-8, -0x1.c000000000000p-111, 0x1.01fdfc0bf6d61p+190, 0x1.41500c3427868p+136},
{0x1.00000007ff800p-868, 0x1.ffff800000000p-956, 0x1.00003fe0c4000p+822, -0x1.fffc032000000p+759, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00000007ff800p-868, 0x1.ffff800000000p-956, -0x1.00003fe0c4000p+822, 0x1.fffc032000000p+759, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00000007ff800p-868, -0x1.ffff800000000p-956, 0x1.00003fe0c4000p+822, -0x1.fffc032000000p+759, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00000007ff800p-868, -0x1.ffff800000000p-956, -0x1.00003fe0c4000p+822, 0x1.fffc032000000p+759, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000001fff8008p-330, -0x1.0000000000000p-412, 0x1.000ffffff8000p-674, 0x1.ffffffffffc00p-729, 0x1.ffe005ffa005fp+343, 0x1.f43ff5ecf10d0p+289},
{0x1.000001fff8008p-330, -0x1.0000000000000p-412, -0x1.000ffffff8000p-674, -0x1.ffffffffffc00p-729, -0x1.ffe005ffa005fp+343, -0x1.f43ff5ecf10d0p+289},
{-0x1.000001fff8008p-330, 0x1.0000000000000p-412, 0x1.000ffffff8000p-674, 0x1.ffffffffffc00p-729, -0x1.ffe005ffa005fp+343, -0x1.f43ff5ecf10d0p+289},
{-0x1.000001fff8008p-330, 0x1.0000000000000p-412, -0x1.000ffffff8000p-674, -0x1.ffffffffffc00p-729, 0x1.ffe005ffa005fp+343, 0x1.f43ff5ecf10d0p+289},
{0x1.007fbfffffcf0p+664, 0x1.ffff803ffe000p+609, 0x1.0007ffff00000p+281, 0x1.fffff80000000p+211, 0x1.0077bc431e2ddp+383, -0x1.2b617a631e528p+327},
{0x1.007fbfffffcf0p+664, 0x1.ffff803ffe000p+609, -0x1.0007ffff00000p+281, -0x1.fffff80000000p+211, -0x1.0077bc431e2ddp+383, 0x1.2b617a631e528p+327},
{-0x1.007fbfffffcf0p+664, -0x1.ffff803ffe000p+609, 0x1.0007ffff00000p+281, 0x1.fffff80000000p+211, -0x1.0077bc431e2ddp+383, 0x1.2b617a631e528p+327},
{-0x1.007fbfffffcf0p+664, -0x1.ffff803ffe000p+609, -0x1.0007ffff00000p+281, -0x1.fffff80000000p+211, 0x1.0077bc431e2ddp+383, -0x1.2b617a631e528p+327},
{0x1.0000001fc4000p+296, -0x1.fe003ffe00000p+240, 0x1.01ffffffc0004p+770, -0x1.fffc100000000p+709, 0x1.fc07f05f4870ap-475, -0x1.fd9f5028f5e8ap-529},
{0x1.0000001fc4000p+296, -0x1.fe003ffe00000p+240, -0x1.01ffffffc0004p+770, 0x1.fffc100000000p+709, -0x1.fc07f05f4870ap-475, 0x1.fd9f5028f5e8ap-529},
{-0x1.0000001fc4000p+296, 0x1.fe003ffe00000p+240, 0x1.01ffffffc0004p+770, -0x1.fffc100000000p+709, -0x1.fc07f05f4870ap-475, 0x1.fd9f5028f5e8ap-529},
{-0x1.0000001fc4000p+296, 0x1.fe003ffe00000p+240, -0x1.01ffffffc0004p+770, 0x1.fffc100000000p+709, 0x1.fc07f05f4870ap-475, -0x1.fd9f5028f5e8ap-529},
{0x1.0007e3ffe0000p+601, 0x1.fe0ffffffe000p+537, 0x1.0000001000000p+631, -0x1.ffffff0000800p+567, 0x1.0007e3efdf81cp-30, 0x1.060603b67f558p-86},
{0x1.0007e3ffe0000p+601, 0x1.fe0ffffffe000p+537, -0x1.0000001000000p+631, 0x1.ffffff0000800p+567, -0x1.0007e3efdf81cp-30, -0x1.060603b67f558p-86},
{-0x1.0007e3ffe0000p+601, -0x1.fe0ffffffe000p+537, 0x1.0000001000000p+631, -0x1.ffffff0000800p+567, -0x1.0007e3efdf81cp-30, -0x1.060603b67f558p-86},
{-0x1.0007e3ffe0000p+601, -0x1.fe0ffffffe000p+537, -0x1.0000001000000p+631, 0x1.ffffff0000800p+567, 0x1.0007e3efdf81cp-30, 0x1.060603b67f558p-86},
{0x1.00000003ff800p+298, 0x1.ffffe00000000p+212, 0x1.0000200000000p+469, -0x1.ff80800000000p+404, 0x1.ffffc00806ff0p-172, -0x1.bbc0c7e7b71b0p-229},
{0x1.00000003ff800p+298, 0x1.ffffe00000000p+212, -0x1.0000200000000p+469, 0x1.ff80800000000p+404, -0x1.ffffc00806ff0p-172, 0x1.bbc0c7e7b71b0p-229},
{-0x1.00000003ff800p+298, -0x1.ffffe00000000p+212, 0x1.0000200000000p+469, -0x1.ff80800000000p+404, -0x1.ffffc00806ff0p-172, 0x1.bbc0c7e7b71b0p-229},
{-0x1.00000003ff800p+298, -0x1.ffffe00000000p+212, -0x1.0000200000000p+469, 0x1.ff80800000000p+404, 0x1.ffffc00806ff0p-172, -0x1.bbc0c7e7b71b0p-229},
{0x1.0000001000000p-64, -0x1.fffffff804000p-131, 0x1.003ffffff8000p+483, 0x1.f00000007fc00p+425, 0x1.ff80201809f98p-548, 0x1.a8143b315e430p-604},
{0x1.0000001000000p-64, -0x1.fffffff804000p-131, -0x1.003ffffff8000p+483, -0x1.f00000007fc00p+425, -0x1.ff80201809f98p-548, -0x1.a8143b315e430p-604},
{-0x1.0000001000000p-64, 0x1.fffffff804000p-131, 0x1.003ffffff8000p+483, 0x1.f00000007fc00p+425, -0x1.ff80201809f98p-548, -0x1.a8143b315e430p-604},
{-0x1.0000001000000p-64, 0x1.fffffff804000p-131, -0x1.003ffffff8000p+483, -0x1.f00000007fc00p+425, 0x1.ff80201809f98p-548, 0x1.a8143b315e430p-604},
{0x1.1ffff80000000p-10, 0x1.ffffff7ff8000p-69, 0x1.00ff000000004p-167, -0x1.ffffff0000400p-230, 0x1.1ee234ad87262p+157, -0x1.99e41e6e04f48p+103},
{0x1.1ffff80000000p-10, 0x1.ffffff7ff8000p-69, -0x1.00ff000000004p-167, 0x1.ffffff0000400p-230, -0x1.1ee234ad87262p+157, 0x1.99e41e6e04f48p+103},
{-0x1.1ffff80000000p-10, -0x1.ffffff7ff8000p-69, 0x1.00ff000000004p-167, -0x1.ffffff0000400p-230, -0x1.1ee234ad87262p+157, 0x1.99e41e6e04f48p+103},
{-0x1.1ffff80000000p-10, -0x1.ffffff7ff8000p-69, -0x1.00ff000000004p-167, 0x1.ffffff0000400p-230, 0x1.1ee234ad87262p+157, -0x1.99e41e6e04f48p+103},
{0x1.000000ffff800p+869, 0x1.fffffc0000000p+786, 0x1.0008000000000p+602, -0x1.0000000000000p+534, 0x1.fff0027feb00ap+266, 0x1.feb80a1fb901ap+212},
{0x1.000000ffff800p+869, 0x1.fffffc0000000p+786, -0x1.0008000000000p+602, 0x1.0000000000000p+534, -0x1.fff0027feb00ap+266, -0x1.feb80a1fb901ap+212},
{-0x1.000000ffff800p+869, -0x1.fffffc0000000p+786, 0x1.0008000000000p+602, -0x1.0000000000000p+534, -0x1.fff0027feb00ap+266, -0x1.feb80a1fb901ap+212},
{-0x1.000000ffff800p+869, -0x1.fffffc0000000p+786, -0x1.0008000000000p+602, 0x1.0000000000000p+534, 0x1.fff0027feb00ap+266, 0x1.feb80a1fb901ap+212},
{0x1.0000183ffffc0p+625, 0x1.fff07fcfffffcp+571, 0x1.0000010000000p-809, -0x1.0000000000000p-884, INFINITY, 0x0.0000000000000p+0},
{0x1.0000183ffffc0p+625, 0x1.fff07fcfffffcp+571, -0x1.0000010000000p-809, 0x1.0000000000000p-884, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000183ffffc0p+625, -0x1.fff07fcfffffcp+571, 0x1.0000010000000p-809, -0x1.0000000000000p-884, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000183ffffc0p+625, -0x1.fff07fcfffffcp+571, -0x1.0000010000000p-809, 0x1.0000000000000p-884, INFINITY, 0x0.0000000000000p+0},
{0x1.0001000000000p+252, -0x1.fffffff800400p+189, 0x1.0000800000000p+229, -0x1.fffff80100000p+156, 0x1.00007fffc0002p+23, -0x1.03bf7dfd41200p-39},
{0x1.0001000000000p+252, -0x1.fffffff800400p+189, -0x1.0000800000000p+229, 0x1.fffff80100000p+156, -0x1.00007fffc0002p+23, 0x1.03bf7dfd41200p-39},
{-0x1.0001000000000p+252, 0x1.fffffff800400p+189, 0x1.0000800000000p+229, -0x1.fffff80100000p+156, -0x1.00007fffc0002p+23, 0x1.03bf7dfd41200p-39},
{-0x1.0001000000000p+252, 0x1.fffffff800400p+189, -0x1.0000800000000p+229, 0x1.fffff80100000p+156, 0x1.00007fffc0002p+23, -0x1.03bf7dfd41200p-39},
{0x1.3ffff800fff81p+251, -0x1.0000000000000p+160, 0x1.1fffffff00000p-134, 0x1.fffffff000000p-211, 0x1.1c71c001e05e1p+385, -0x1.c571834f1dba0p+329},
{0x1.3ffff800fff81p+251, -0x1.0000000000000p+160, -0x1.1fffffff00000p-134, -0x1.fffffff000000p-211, -0x1.1c71c001e05e1p+385, 0x1.c571834f1dba0p+329},
{-0x1.3ffff800fff81p+251, 0x1.0000000000000p+160, 0x1.1fffffff00000p-134, 0x1.fffffff000000p-211, -0x1.1c71c001e05e1p+385, 0x1.c571834f1dba0p+329},
{-0x1.3ffff800fff81p+251, 0x1.0000000000000p+160, -0x1.1fffffff00000p-134, -0x1.fffffff000000p-211, 0x1.1c71c001e05e1p+385, -0x1.c571834f1dba0p+329},
{0x1.fffffc0000100p+380, -0x1.fffe000040000p+309, 0x1.0000800000000p+410, -0x1.c000fffffffd0p+353, 0x1.fffefc00820fcp-30, 0x1.703780ec1f1f0p-87},
{0x1.fffffc0000100p+380, -0x1.fffe000040000p+309, -0x1.0000800000000p+410, 0x1.c000fffffffd0p+353, -0x1.fffefc00820fcp-30, -0x1.703780ec1f1f0p-87},
{-0x1.fffffc0000100p+380, 0x1.fffe000040000p+309, 0x1.0000800000000p+410, -0x1.c000fffffffd0p+353, -0x1.fffefc00820fcp-30, -0x1.703780ec1f1f0p-87},
{-0x1.fffffc0000100p+380, 0x1.fffe000040000p+309, -0x1.0000800000000p+410, 0x1.c000fffffffd0p+353, 0x1.fffefc00820fcp-30, 0x1.703780ec1f1f0p-87},
{0x1.003f80ff00004p-166, -0x1.ffffc00040000p-237, 0x1.0000ffffff000p+849, 0x1.ffe000003ffe0p+791, 0x1.003e80c080400p-1015, -0x1.2000000000000p-1070},
{0x1.003f80ff00004p-166, -0x1.ffffc00040000p-237, -0x1.0000ffffff000p+849, -0x1.ffe000003ffe0p+791, -0x1.003e80c080400p-1015, 0x1.2000000000000p-1070},
{-0x1.003f80ff00004p-166, 0x1.ffffc00040000p-237, 0x1.0000ffffff000p+849, 0x1.ffe000003ffe0p+791, -0x1.003e80c080400p-1015, 0x1.2000000000000p-1070},
{-0x1.003f80ff00004p-166, 0x1.ffffc00040000p-237, -0x1.0000ffffff000p+849, -0x1.ffe000003ffe0p+791, 0x1.003e80c080400p-1015, -0x1.2000000000000p-1070},
{0x1.000003fffff80p-485, 0x1.f007ffe00fc00p-548, 0x1.3ff8007ffffc0p+163, 0x1.e0007ff000000p+97, 0x1.99a3dd0e7862fp-649, -0x1.e94ba8273c842p-703},
{0x1.000003fffff80p-485, 0x1.f007ffe00fc00p-548, -0x1.3ff8007ffffc0p+163, -0x1.e0007ff000000p+97, -0x1.99a3dd0e7862fp-649, 0x1.e94ba8273c842p-703},
{-0x1.000003fffff80p-485, -0x1.f007ffe00fc00p-548, 0x1.3ff8007ffffc0p+163, 0x1.e0007ff000000p+97, -0x1.99a3dd0e7862fp-649, 0x1.e94ba8273c842p-703},
{-0x1.000003fffff80p-485, -0x1.f007ffe00fc00p-548, -0x1.3ff8007ffffc0p+163, -0x1.e0007ff000000p+97, 0x1.99a3dd0e7862fp-649, -0x1.e94ba8273c842p-703},
{0x1.ffffffe000000p-232, -0x1.ff80000020000p-288, 0x1.0000ffffff81fp+818, 0x1.ffc7fff800000p+754, 0x1.fffe000000000p-1050, -0x0.0000000000000p+0},
{0x1.ffffffe000000p-232, -0x1.ff80000020000p-288, -0x1.0000ffffff81fp+818, -0x1.ffc7fff800000p+754, -0x1.fffe000000000p-1050, 0x0.0000000000000p+0},
{-0x1.ffffffe000000p-232, 0x1.ff80000020000p-288, 0x1.0000ffffff81fp+818, 0x1.ffc7fff800000p+754, -0x1.fffe000000000p-1050, 0x0.0000000000000p+0},
{-0x1.ffffffe000000p-232, 0x1.ff80000020000p-288, -0x1.0000ffffff81fp+818, -0x1.ffc7fff800000p+754, 0x1.fffe000000000p-1050, -0x0.0000000000000p+0},
{0x1.0000000800000p-741, -0x1.ffffc02000000p-819, 0x1.7fffe00800000p-232, -0x1.0000000000000p-337, 0x1.555571caad093p-510, -0x1.e0b116af638a0p-568},
{0x1.0000000800000p-741, -0x1.ffffc02000000p-819, -0x1.7fffe00800000p-232, 0x1.0000000000000p-337, -0x1.555571caad093p-510, 0x1.e0b116af638a0p-568},
{-0x1.0000000800000p-741, 0x1.ffffc02000000p-819, 0x1.7fffe00800000p-232, -0x1.0000000000000p-337, -0x1.555571caad093p-510, 0x1.e0b116af638a0p-568},
{-0x1.0000000800000p-741, 0x1.ffffc02000000p-819, -0x1.7fffe00800000p-232, 0x1.0000000000000p-337, 0x1.555571caad093p-510, -0x1.e0b116af638a0p-568},
{0x1.0000400000000p-741, -0x1.0000000000000p-832, 0x1.4000000000000p-919, -0x1.ffffffff00020p-977, 0x1.999a000000000p+177, 0x1.47ae66655c2a0p+120},
{0x1.0000400000000p-741, -0x1.0000000000000p-832, -0x1.4000000000000p-919, 0x1.ffffffff00020p-977, -0x1.999a000000000p+177, -0x1.47ae66655c2a0p+120},
{-0x1.0000400000000p-741, 0x1.0000000000000p-832, 0x1.4000000000000p-919, -0x1.ffffffff00020p-977, -0x1.999a000000000p+177, -0x1.47ae66655c2a0p+120},
{-0x1.0000400000000p-741, 0x1.0000000000000p-832, -0x1.4000000000000p-919, 0x1.ffffffff00020p-977, 0x1.999a000000000p+177, 0x1.47ae66655c2a0p+120},
{0x1.000000fffff80p+807, 0x1.ffffef003e000p+744, 0x1.0000010000000p+666, -0x1.8000000000020p+608, 0x1.fffffffffff00p+140, 0x1.901ffde7e1e80p+83},
{0x1.000000fffff80p+807, 0x1.ffffef003e000p+744, -0x1.0000010000000p+666, 0x1.8000000000020p+608, -0x1.fffffffffff00p+140, -0x1.901ffde7e1e80p+83},
{-0x1.000000fffff80p+807, -0x1.ffffef003e000p+744, 0x1.0000010000000p+666, -0x1.8000000000020p+608, -0x1.fffffffffff00p+140, -0x1.901ffde7e1e80p+83},
{-0x1.000000fffff80p+807, -0x1.ffffef003e000p+744, -0x1.0000010000000p+666, 0x1.8000000000020p+608, 0x1.fffffffffff00p+140, 0x1.901ffde7e1e80p+83},
{0x1.7fffffffc0000p-420, 0x1.fffc000000000p-490, 0x1.0000001fe0000p+340, 0x1.fffffc0001f00p+279, 0x1.7fffffcff0000p-760, 0x1.79017fd9cf2a6p-814},
{0x1.7fffffffc0000p-420, 0x1.fffc000000000p-490, -0x1.0000001fe0000p+340, -0x1.fffffc0001f00p+279, -0x1.7fffffcff0000p-760, -0x1.79017fd9cf2a6p-814},
{-0x1.7fffffffc0000p-420, -0x1.fffc000000000p-490, 0x1.0000001fe0000p+340, 0x1.fffffc0001f00p+279, -0x1.7fffffcff0000p-760, -0x1.79017fd9cf2a6p-814},
{-0x1.7fffffffc0000p-420, -0x1.fffc000000000p-490, -0x1.0000001fe0000p+340, -0x1.fffffc0001f00p+279, 0x1.7fffffcff0000p-760, 0x1.79017fd9cf2a6p-814},
{0x1.0000004000000p-97, -0x1.0000000000000p-155, 0x1.8000000000000p-464, -0x1.ffe7ffc001000p-529, 0x1.555555aaaaaabp+366, -0x1.6a71c9c71555ep+312},
{0x1.0000004000000p-97, -0x1.0000000000000p-155, -0x1.8000000000000p-464, 0x1.ffe7ffc001000p-529, -0x1.555555aaaaaabp+366, 0x1.6a71c9c71555ep+312},
{-0x1.0000004000000p-97, 0x1.0000000000000p-155, 0x1.8000000000000p-464, -0x1.ffe7ffc001000p-529, -0x1.555555aaaaaabp+366, 0x1.6a71c9c71555ep+312},
{-0x1.0000004000000p-97, 0x1.0000000000000p-155, -0x1.8000000000000p-464, 0x1.ffe7ffc001000p-529, 0x1.555555aaaaaabp+366, -0x1.6a71c9c71555ep+312},
{0x1.0000000800000p-953, -0x1.fffffff400000p-1028, 0x1.007fc00007800p-531, 0x1.e0000f80001e0p-589, 0x1.ff00ff5088975p-423, -0x1.eb9bace924318p-479},
{0x1.0000000800000p-953, -0x1.fffffff400000p-1028, -0x1.007fc00007800p-531, -0x1.e0000f80001e0p-589, -0x1.ff00ff5088975p-423, 0x1.eb9bace924318p-479},
{-0x1.0000000800000p-953, 0x1.fffffff400000p-1028, 0x1.007fc00007800p-531, 0x1.e0000f80001e0p-589, -0x1.ff00ff5088975p-423, 0x1.eb9bace924318p-479},
{-0x1.0000000800000p-953, 0x1.fffffff400000p-1028, -0x1.007fc00007800p-531, -0x1.e0000f80001e0p-589, 0x1.ff00ff5088975p-423, -0x1.eb9bace924318p-479},
{0x1.3fffff0000000p+486, 0x1.fffffe0000000p+423, 0x1.0000200000000p-608, -0x1.0000000000000p-713, INFINITY, 0x0.0000000000000p+0},
{0x1.3fffff0000000p+486, 0x1.fffffe0000000p+423, -0x1.0000200000000p-608, 0x1.0000000000000p-713, -INFINITY, 0x0.0000000000000p+0},
{-0x1.3fffff0000000p+486, -0x1.fffffe0000000p+423, 0x1.0000200000000p-608, -0x1.0000000000000p-713, -INFINITY, 0x0.0000000000000p+0},
{-0x1.3fffff0000000p+486, -0x1.fffffe0000000p+423, -0x1.0000200000000p-608, 0x1.0000000000000p-713, INFINITY, 0x0.0000000000000p+0},
{0x1.7fe0000000008p-874, -0x1.fffff3ff80200p-936, 0x1.000001fff8400p+934, -0x1.fffff80010000p+869, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.7fe0000000008p-874, -0x1.fffff3ff80200p-936, -0x1.000001fff8400p+934, 0x1.fffff80010000p+869, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.7fe0000000008p-874, 0x1.fffff3ff80200p-936, 0x1.000001fff8400p+934, -0x1.fffff80010000p+869, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.7fe0000000008p-874, 0x1.fffff3ff80200p-936, -0x1.000001fff8400p+934, 0x1.fffff80010000p+869, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00000fffffff8p-940, 0x1.fffffffc00000p-999, 0x1.0000007001000p+288, -0x1.e00000fff8200p+228, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00000fffffff8p-940, 0x1.fffffffc00000p-999, -0x1.0000007001000p+288, 0x1.e00000fff8200p+228, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00000fffffff8p-940, -0x1.fffffffc00000p-999, 0x1.0000007001000p+288, -0x1.e00000fff8200p+228, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00000fffffff8p-940, -0x1.fffffffc00000p-999, -0x1.0000007001000p+288, 0x1.e00000fff8200p+228, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.001fffc000400p-967, -0x1.e000000001000p-1032, 0x1.0200000000000p+224, -0x1.0000000000000p+121, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.001fffc000400p-967, -0x1.e000000001000p-1032, -0x1.0200000000000p+224, 0x1.0000000000000p+121, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.001fffc000400p-967, 0x1.e000000001000p-1032, 0x1.0200000000000p+224, -0x1.0000000000000p+121, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.001fffc000400p-967, 0x1.e000000001000p-1032, -0x1.0200000000000p+224, 0x1.0000000000000p+121, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000008000000p-97, -0x1.0000000000000p-167, 0x1.07fff80000002p-376, -0x1.c010000000000p-447, 0x1.f07c2f0b854dbp+278, 0x1.2c698df3c3460p+221},
{0x1.0000008000000p-97, -0x1.0000000000000p-167, -0x1.07fff80000002p-376, 0x1.c010000000000p-447, -0x1.f07c2f0b854dbp+278, -0x1.2c698df3c3460p+221},
{-0x1.0000008000000p-97, 0x1.0000000000000p-167, 0x1.07fff80000002p-376, -0x1.c010000000000p-447, -0x1.f07c2f0b854dbp+278, -0x1.2c698df3c3460p+221},
{-0x1.0000008000000p-97, 0x1.0000000000000p-167, -0x1.07fff80000002p-376, 0x1.c010000000000p-447, 0x1.f07c2f0b854dbp+278, 0x1.2c698df3c3460p+221},
{0x1.03ffffffe0000p+781, 0x1.ff80000000000p+692, 0x1.7ffff00000002p-601, -0x1.0000000000000p-694, INFINITY, 0x0.0000000000000p+0},
{0x1.03ffffffe0000p+781, 0x1.ff80000000000p+692, -0x1.7ffff00000002p-601, 0x1.0000000000000p-694, -INFINITY, 0x0.0000000000000p+0},
{-0x1.03ffffffe0000p+781, -0x1.ff80000000000p+692, 0x1.7ffff00000002p-601, -0x1.0000000000000p-694, -INFINITY, 0x0.0000000000000p+0},
{-0x1.03ffffffe0000p+781, -0x1.ff80000000000p+692, -0x1.7ffff00000002p-601, 0x1.0000000000000p-694, INFINITY, 0x0.0000000000000p+0},
{0x1.0008000000000p+386, -0x1.9000000000000p+285, 0x1.0000000700002p+420, -0x1.fffff80200000p+357, 0x1.0007fff8ffc7ep-34, 0x1.a00d4d7413580p-93},
{0x1.0008000000000p+386, -0x1.9000000000000p+285, -0x1.0000000700002p+420, 0x1.fffff80200000p+357, -0x1.0007fff8ffc7ep-34, -0x1.a00d4d7413580p-93},
{-0x1.0008000000000p+386, 0x1.9000000000000p+285, 0x1.0000000700002p+420, -0x1.fffff80200000p+357, -0x1.0007fff8ffc7ep-34, -0x1.a00d4d7413580p-93},
{-0x1.0008000000000p+386, 0x1.9000000000000p+285, -0x1.0000000700002p+420, 0x1.fffff80200000p+357, 0x1.0007fff8ffc7ep-34, 0x1.a00d4d7413580p-93},
{0x1.0010000000000p-808, -0x1.0000000000000p-885, 0x1.007fc00001000p-874, -0x1.ff80000400000p-930, 0x1.ff20ef50918c4p+65, 0x1.5059d3d1c2168p+9},
{0x1.0010000000000p-808, -0x1.0000000000000p-885, -0x1.007fc00001000p-874, 0x1.ff80000400000p-930, -0x1.ff20ef50918c4p+65, -0x1.5059d3d1c2168p+9},
{-0x1.0010000000000p-808, 0x1.0000000000000p-885, 0x1.007fc00001000p-874, -0x1.ff80000400000p-930, -0x1.ff20ef50918c4p+65, -0x1.5059d3d1c2168p+9},
{-0x1.0010000000000p-808, 0x1.0000000000000p-885, -0x1.007fc00001000p-874, 0x1.ff80000400000p-930, 0x1.ff20ef50918c4p+65, 0x1.5059d3d1c2168p+9},
{0x1.0000008000000p+531, -0x1.fff0000000100p+472, 0x1.7fffff8100000p-278, -0x1.fffcc00000080p-338, 0x1.55555670e38e9p+808, 0x1.9c985b0496336p+754},
{0x1.0000008000000p+531, -0x1.fff0000000100p+472, -0x1.7fffff8100000p-278, 0x1.fffcc00000080p-338, -0x1.55555670e38e9p+808, -0x1.9c985b0496336p+754},
{-0x1.0000008000000p+531, 0x1.fff0000000100p+472, 0x1.7fffff8100000p-278, -0x1.fffcc00000080p-338, -0x1.55555670e38e9p+808, -0x1.9c985b0496336p+754},
{-0x1.0000008000000p+531, 0x1.fff0000000100p+472, -0x1.7fffff8100000p-278, 0x1.fffcc00000080p-338, 0x1.55555670e38e9p+808, 0x1.9c985b0496336p+754},
{0x1.0000003ffc010p-568, -0x1.ffffc00400000p-637, 0x1.0000c00400000p+799, -0x1.ff80800000000p+718, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000003ffc010p-568, -0x1.ffffc00400000p-637, -0x1.0000c00400000p+799, 0x1.ff80800000000p+718, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000003ffc010p-568, 0x1.ffffc00400000p-637, 0x1.0000c00400000p+799, -0x1.ff80800000000p+718, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000003ffc010p-568, 0x1.ffffc00400000p-637, -0x1.0000c00400000p+799, 0x1.ff80800000000p+718, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000007fff800p-430, 0x1.ffff800007fc0p-489, 0x1.0000007fff000p+851, 0x1.ffffff8000000p+776, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000007fff800p-430, 0x1.ffff800007fc0p-489, -0x1.0000007fff000p+851, -0x1.ffffff8000000p+776, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000007fff800p-430, -0x1.ffff800007fc0p-489, 0x1.0000007fff000p+851, 0x1.ffffff8000000p+776, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000007fff800p-430, -0x1.ffff800007fc0p-489, -0x1.0000007fff000p+851, -0x1.ffffff8000000p+776, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.4000000000000p-853, -0x1.0000000000000p-935, 0x1.0000001ff8040p+392, -0x1.fff0ffffff008p+336, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.4000000000000p-853, -0x1.0000000000000p-935, -0x1.0000001ff8040p+392, 0x1.fff0ffffff008p+336, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.4000000000000p-853, 0x1.0000000000000p-935, 0x1.0000001ff8040p+392, -0x1.fff0ffffff008p+336, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.4000000000000p-853, 0x1.0000000000000p-935, -0x1.0000001ff8040p+392, 0x1.fff0ffffff008p+336, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0004000000000p+80, -0x1.fffffffe40000p+9, 0x1.000001ffff1f0p+442, 0x1.ff807ffffc000p+388, 0x1.0003fdfff8e50p-362, -0x1.1b6c795bc3930p-416},
{0x1.0004000000000p+80, -0x1.fffffffe40000p+9, -0x1.000001ffff1f0p+442, -0x1.ff807ffffc000p+388, -0x1.0003fdfff8e50p-362, 0x1.1b6c795bc3930p-416},
{-0x1.0004000000000p+80, 0x1.fffffffe40000p+9, 0x1.000001ffff1f0p+442, 0x1.ff807ffffc000p+388, -0x1.0003fdfff8e50p-362, 0x1.1b6c795bc3930p-416},
{-0x1.0004000000000p+80, 0x1.fffffffe40000p+9, -0x1.000001ffff1f0p+442, -0x1.ff807ffffc000p+388, 0x1.0003fdfff8e50p-362, -0x1.1b6c795bc3930p-416},
{0x1.003fe00030100p-112, -0x1.00003fe002000p-178, 0x1.3ffffff800200p+913, -0x1.0000000000000p+836, 0x1.99ffccd759890p-1026, -0x0.0000000000000p+0},
{0x1.003fe00030100p-112, -0x1.00003fe002000p-178, -0x1.3ffffff800200p+913, 0x1.0000000000000p+836, -0x1.99ffccd759890p-1026, 0x0.0000000000000p+0},
{-0x1.003fe00030100p-112, 0x1.00003fe002000p-178, 0x1.3ffffff800200p+913, -0x1.0000000000000p+836, -0x1.99ffccd759890p-1026, 0x0.0000000000000p+0},
{-0x1.003fe00030100p-112, 0x1.00003fe002000p-178, -0x1.3ffffff800200p+913, 0x1.0000000000000p+836, 0x1.99ffccd759890p-1026, -0x0.0000000000000p+0},
{0x1.0001ffc1ffc20p+846, -0x1.fffffc0002000p+780, 0x1.000003fffffc0p-56, 0x1.fffffe0000000p-134, 0x1.0001fbc1f7d70p+902, -0x1.f5fb3830001ccp+848},
{0x1.0001ffc1ffc20p+846, -0x1.fffffc0002000p+780, -0x1.000003fffffc0p-56, -0x1.fffffe0000000p-134, -0x1.0001fbc1f7d70p+902, 0x1.f5fb3830001ccp+848},
{-0x1.0001ffc1ffc20p+846, 0x1.fffffc0002000p+780, 0x1.000003fffffc0p-56, 0x1.fffffe0000000p-134, -0x1.0001fbc1f7d70p+902, 0x1.f5fb3830001ccp+848},
{-0x1.0001ffc1ffc20p+846, 0x1.fffffc0002000p+780, -0x1.000003fffffc0p-56, -0x1.fffffe0000000p-134, 0x1.0001fbc1f7d70p+902, -0x1.f5fb3830001ccp+848},
{0x1.0400000000000p-92, -0x1.0000000000000p-146, 0x1.0000200000000p+893, -0x1.ffffffffffc40p+834, 0x1.03ffdf8004100p-985, -0x1.103fe00000000p-1039},
{0x1.0400000000000p-92, -0x1.0000000000000p-146, -0x1.0000200000000p+893, 0x1.ffffffffffc40p+834, -0x1.03ffdf8004100p-985, 0x1.103fe00000000p-1039},
{-0x1.0400000000000p-92, 0x1.0000000000000p-146, 0x1.0000200000000p+893, -0x1.ffffffffffc40p+834, -0x1.03ffdf8004100p-985, 0x1.103fe00000000p-1039},
{-0x1.0400000000000p-92, 0x1.0000000000000p-146, -0x1.0000200000000p+893, 0x1.ffffffffffc40p+834, 0x1.03ffdf8004100p-985, -0x1.103fe00000000p-1039},
{0x1.00003ff820000p-373, -0x1.ffffffe000100p-428, 0x1.0001f80020000p-217, -0x1.ffffffc1c0040p-276, 0x1.fffc8ff6c4929p-157, 0x1.c04a167bfc32cp-212},
{0x1.00003ff820000p-373, -0x1.ffffffe000100p-428, -0x1.0001f80020000p-217, 0x1.ffffffc1c0040p-276, -0x1.fffc8ff6c4929p-157, -0x1.c04a167bfc32cp-212},
{-0x1.00003ff820000p-373, 0x1.ffffffe000100p-428, 0x1.0001f80020000p-217, -0x1.ffffffc1c0040p-276, -0x1.fffc8ff6c4929p-157, -0x1.c04a167bfc32cp-212},
{-0x1.00003ff820000p-373, 0x1.ffffffe000100p-428, -0x1.0001f80020000p-217, 0x1.ffffffc1c0040p-276, 0x1.fffc8ff6c4929p-157, 0x1.c04a167bfc32cp-212},
{0x1.007ffffffff80p+570, 0x1.fffffffe00000p+496, 0x1.0100000000000p-172, -0x1.0000000000000p-249, 0x1.ff00ff00fef11p+741, -0x1.0ef00007f0190p+685},
{0x1.007ffffffff80p+570, 0x1.fffffffe00000p+496, -0x1.0100000000000p-172, 0x1.0000000000000p-249, -0x1.ff00ff00fef11p+741, 0x1.0ef00007f0190p+685},
{-0x1.007ffffffff80p+570, -0x1.fffffffe00000p+496, 0x1.0100000000000p-172, -0x1.0000000000000p-249, -0x1.ff00ff00fef11p+741, 0x1.0ef00007f0190p+685},
{-0x1.007ffffffff80p+570, -0x1.fffffffe00000p+496, -0x1.0100000000000p-172, 0x1.0000000000000p-249, 0x1.ff00ff00fef11p+741, -0x1.0ef00007f0190p+685},
{0x1.0000000400000p-622, -0x1.e00003fff8200p-684, 0x1.000ffff802000p-88, -0x1.fffe3fffffe00p-150, 0x1.ffe00217d982cp-535, -0x1.dbfb6b1d22926p-589},
{0x1.0000000400000p-622, -0x1.e00003fff8200p-684, -0x1.000ffff802000p-88, 0x1.fffe3fffffe00p-150, -0x1.ffe00217d982cp-535, 0x1.dbfb6b1d22926p-589},
{-0x1.0000000400000p-622, 0x1.e00003fff8200p-684, 0x1.000ffff802000p-88, -0x1.fffe3fffffe00p-150, -0x1.ffe00217d982cp-535, 0x1.dbfb6b1d22926p-589},
{-0x1.0000000400000p-622, 0x1.e00003fff8200p-684, -0x1.000ffff802000p-88, 0x1.fffe3fffffe00p-150, 0x1.ffe00217d982cp-535, -0x1.dbfb6b1d22926p-589},
{0x1.00003ffffff00p+633, 0x1.ffc003ffffe00p+578, 0x1.0001ff0000000p+629, 0x1.ff8007fffffe0p+574, 0x1.fffc8206f8541p+3, 0x1.944d561338738p-51},
{0x1.00003ffffff00p+633, 0x1.ffc003ffffe00p+578, -0x1.0001ff0000000p+629, -0x1.ff8007fffffe0p+574, -0x1.fffc8206f8541p+3, -0x1.944d561338738p-51},
{-0x1.00003ffffff00p+633, -0x1.ffc003ffffe00p+578, 0x1.0001ff0000000p+629, 0x1.ff8007fffffe0p+574, -0x1.fffc8206f8541p+3, -0x1.944d561338738p-51},
{-0x1.00003ffffff00p+633, -0x1.ffc003ffffe00p+578, -0x1.0001ff0000000p+629, -0x1.ff8007fffffe0p+574, 0x1.fffc8206f8541p+3, 0x1.944d561338738p-51},
{0x1.1000000000000p+920, -0x1.ff00000001000p+855, 0x1.0200000000000p-494, -0x1.c000000000008p-550, INFINITY, 0x0.0000000000000p+0},
{0x1.1000000000000p+920, -0x1.ff00000001000p+855, -0x1.0200000000000p-494, 0x1.c000000000008p-550, -INFINITY, 0x0.0000000000000p+0},
{-0x1.1000000000000p+920, 0x1.ff00000001000p+855, 0x1.0200000000000p-494, -0x1.c000000000008p-550, -INFINITY, 0x0.0000000000000p+0},
{-0x1.1000000000000p+920, 0x1.ff00000001000p+855, -0x1.0200000000000p-494, 0x1.c000000000008p-550, INFINITY, 0x0.0000000000000p+0},
{0x1.1fffffc000000p+266, 0x1.ffffffffffff0p+209, 0x1.007fc00000800p-488, -0x1.fffff03fff000p-548, 0x1.1f708f547908ep+754, 0x1.5692f64f2b738p+699},
{0x1.1fffffc000000p+266, 0x1.ffffffffffff0p+209, -0x1.007fc00000800p-488, 0x1.fffff03fff000p-548, -0x1.1f708f547908ep+754, -0x1.5692f64f2b738p+699},
{-0x1.1fffffc000000p+266, -0x1.ffffffffffff0p+209, 0x1.007fc00000800p-488, -0x1.fffff03fff000p-548, -0x1.1f708f547908ep+754, -0x1.5692f64f2b738p+699},
{-0x1.1fffffc000000p+266, -0x1.ffffffffffff0p+209, -0x1.007fc00000800p-488, 0x1.fffff03fff000p-548, 0x1.1f708f547908ep+754, 0x1.5692f64f2b738p+699},
{0x1.00007fffff000p+951, 0x1.ffffffff00000p+879, 0x1.1fffffffc0000p+24, 0x1.fff0000000000p-44, 0x1.c71d5555b8b13p+926, -0x1.2f9a266305f00p+869},
{0x1.00007fffff000p+951, 0x1.ffffffff00000p+879, -0x1.1fffffffc0000p+24, -0x1.fff0000000000p-44, -0x1.c71d5555b8b13p+926, 0x1.2f9a266305f00p+869},
{-0x1.00007fffff000p+951, -0x1.ffffffff00000p+879, 0x1.1fffffffc0000p+24, 0x1.fff0000000000p-44, -0x1.c71d5555b8b13p+926, 0x1.2f9a266305f00p+869},
{-0x1.00007fffff000p+951, -0x1.ffffffff00000p+879, -0x1.1fffffffc0000p+24, -0x1.fff0000000000p-44, 0x1.c71d5555b8b13p+926, -0x1.2f9a266305f00p+869},
{0x1.0003ffffffffcp+259, 0x1.f000000000000p+158, 0x1.0000003ffff80p+431, 0x1.fffffefff8000p+363, 0x1.0003ffbfff07dp-172, 0x1.00e05b827c740p-231},
{0x1.0003ffffffffcp+259, 0x1.f000000000000p+158, -0x1.0000003ffff80p+431, -0x1.fffffefff8000p+363, -0x1.0003ffbfff07dp-172, -0x1.00e05b827c740p-231},
{-0x1.0003ffffffffcp+259, -0x1.f000000000000p+158, 0x1.0000003ffff80p+431, 0x1.fffffefff8000p+363, -0x1.0003ffbfff07dp-172, -0x1.00e05b827c740p-231},
{-0x1.0003ffffffffcp+259, -0x1.f000000000000p+158, -0x1.0000003ffff80p+431, -0x1.fffffefff8000p+363, 0x1.0003ffbfff07dp-172, 0x1.00e05b827c740p-231},
{0x1.00007ffc00000p-308, 0x1.ffffe00000000p-377, 0x1.3fffffc004000p+942, -0x1.ffffe00010000p+878, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00007ffc00000p-308, 0x1.ffffe00000000p-377, -0x1.3fffffc004000p+942, 0x1.ffffe00010000p+878, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00007ffc00000p-308, -0x1.ffffe00000000p-377, 0x1.3fffffc004000p+942, -0x1.ffffe00010000p+878, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00007ffc00000p-308, -0x1.ffffe00000000p-377, -0x1.3fffffc004000p+942, 0x1.ffffe00010000p+878, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0007ffffc0004p+198, -0x1.fffffe0040000p+127, 0x1.00ffffe000400p-503, -0x1.0000000000000p-595, 0x1.fe11ee50f0cdfp+700, 0x1.eab4ac047c964p+646},
{0x1.0007ffffc0004p+198, -0x1.fffffe0040000p+127, -0x1.00ffffe000400p-503, 0x1.0000000000000p-595, -0x1.fe11ee50f0cdfp+700, -0x1.eab4ac047c964p+646},
{-0x1.0007ffffc0004p+198, 0x1.fffffe0040000p+127, 0x1.00ffffe000400p-503, -0x1.0000000000000p-595, -0x1.fe11ee50f0cdfp+700, -0x1.eab4ac047c964p+646},
{-0x1.0007ffffc0004p+198, 0x1.fffffe0040000p+127, -0x1.00ffffe000400p-503, 0x1.0000000000000p-595, 0x1.fe11ee50f0cdfp+700, 0x1.eab4ac047c964p+646},
{0x1.007ffffffff00p+38, 0x1.fff8003000000p-39, 0x1.01fffffffe000p-764, 0x1.fffc000000000p-852, 0x1.fd05f417d431bp+801, -0x1.da3f8ecab4cb2p+747},
{0x1.007ffffffff00p+38, 0x1.fff8003000000p-39, -0x1.01fffffffe000p-764, -0x1.fffc000000000p-852, -0x1.fd05f417d431bp+801, 0x1.da3f8ecab4cb2p+747},
{-0x1.007ffffffff00p+38, -0x1.fff8003000000p-39, 0x1.01fffffffe000p-764, 0x1.fffc000000000p-852, -0x1.fd05f417d431bp+801, 0x1.da3f8ecab4cb2p+747},
{-0x1.007ffffffff00p+38, -0x1.fff8003000000p-39, -0x1.01fffffffe000p-764, -0x1.fffc000000000p-852, 0x1.fd05f417d431bp+801, -0x1.da3f8ecab4cb2p+747},
{0x1.00003e0000000p+844, 0x1.ffffffffe0000p+785, 0x1.07ffffff80000p+555, 0x1.8007ffffc0000p+484, 0x1.f07c9746c22cep+288, 0x1.3edfd2a2574e4p+234},
{0x1.00003e0000000p+844, 0x1.ffffffffe0000p+785, -0x1.07ffffff80000p+555, -0x1.8007ffffc0000p+484, -0x1.f07c9746c22cep+288, -0x1.3edfd2a2574e4p+234},
{-0x1.00003e0000000p+844, -0x1.ffffffffe0000p+785, 0x1.07ffffff80000p+555, 0x1.8007ffffc0000p+484, -0x1.f07c9746c22cep+288, -0x1.3edfd2a2574e4p+234},
{-0x1.00003e0000000p+844, -0x1.ffffffffe0000p+785, -0x1.07ffffff80000p+555, -0x1.8007ffffc0000p+484, 0x1.f07c9746c22cep+288, 0x1.3edfd2a2574e4p+234},
{0x1.00003ff810000p+183, -0x1.fffffffc00800p+119, 0x1.00000ffff8002p+532, -0x1.fffffffe04000p+465, 0x1.00002ff815006p-349, 0x1.6fa53168c6300p-408},
{0x1.00003ff810000p+183, -0x1.fffffffc00800p+119, -0x1.00000ffff8002p+532, 0x1.fffffffe04000p+465, -0x1.00002ff815006p-349, -0x1.6fa53168c6300p-408},
{-0x1.00003ff810000p+183, 0x1.fffffffc00800p+119, 0x1.00000ffff8002p+532, -0x1.fffffffe04000p+465, -0x1.00002ff815006p-349, -0x1.6fa53168c6300p-408},
{-0x1.00003ff810000p+183, 0x1.fffffffc00800p+119, -0x1.00000ffff8002p+532, 0x1.fffffffe04000p+465, 0x1.00002ff815006p-349, 0x1.6fa53168c6300p-408},
{0x1.0100000000000p-963, -0x1.0000000000000p-1050, 0x1.01ffffff00004p+35, -0x1.0000000000000p-55, 0x1.fe03f811da4eep-999, -0x1.4936800000000p-1053},
{0x1.0100000000000p-963, -0x1.0000000000000p-1050, -0x1.01ffffff00004p+35, 0x1.0000000000000p-55, -0x1.fe03f811da4eep-999, 0x1.4936800000000p-1053},
{-0x1.0100000000000p-963, 0x1.0000000000000p-1050, 0x1.01ffffff00004p+35, -0x1.0000000000000p-55, -0x1.fe03f811da4eep-999, 0x1.4936800000000p-1053},
{-0x1.0100000000000p-963, 0x1.0000000000000p-1050, -0x1.01ffffff00004p+35, 0x1.0000000000000p-55, 0x1.fe03f811da4eep-999, -0x1.4936800000000p-1053},
{0x1.001fffffff800p+716, 0x1.fffff00000000p+636, 0x1.07ffe00001000p-16, -0x1.c7ffc001fe010p-73, 0x1.f0ba6ac140645p+731, -0x1.85bb11db4ee88p+676},
{0x1.001fffffff800p+716, 0x1.fffff00000000p+636, -0x1.07ffe00001000p-16, 0x1.c7ffc001fe010p-73, -0x1.f0ba6ac140645p+731, 0x1.85bb11db4ee88p+676},
{-0x1.001fffffff800p+716, -0x1.fffff00000000p+636, 0x1.07ffe00001000p-16, -0x1.c7ffc001fe010p-73, -0x1.f0ba6ac140645p+731, 0x1.85bb11db4ee88p+676},
{-0x1.001fffffff800p+716, -0x1.fffff00000000p+636, -0x1.07ffe00001000p-16, 0x1.c7ffc001fe010p-73, 0x1.f0ba6ac140645p+731, -0x1.85bb11db4ee88p+676},
{0x1.000000ffffff8p+223, 0x1.ffe0000000000p+151, 0x1.1fffffff00001p+902, -0x1.0000000000000p+817, 0x1.c71c738fcd6dap-680, 0x1.62af33c1d8a00p-735},
{0x1.000000ffffff8p+223, 0x1.ffe0000000000p+151, -0x1.1fffffff00001p+902, 0x1.0000000000000p+817, -0x1.c71c738fcd6dap-680, -0x1.62af33c1d8a00p-735},
{-0x1.000000ffffff8p+223, -0x1.ffe0000000000p+151, 0x1.1fffffff00001p+902, -0x1.0000000000000p+817, -0x1.c71c738fcd6dap-680, -0x1.62af33c1d8a00p-735},
{-0x1.000000ffffff8p+223, -0x1.ffe0000000000p+151, -0x1.1fffffff00001p+902, 0x1.0000000000000p+817, 0x1.c71c738fcd6dap-680, 0x1.62af33c1d8a00p-735},
{0x1.00000007fff00p+641, 0x1.fffc1ffff87f0p+585, 0x1.0000007c00000p+146, 0x1.fff0000000000p+75, 0x1.ffffff17ffe07p+494, 0x1.17ffe1f85d360p+440},
{0x1.00000007fff00p+641, 0x1.fffc1ffff87f0p+585, -0x1.0000007c00000p+146, -0x1.fff0000000000p+75, -0x1.ffffff17ffe07p+494, -0x1.17ffe1f85d360p+440},
{-0x1.00000007fff00p+641, -0x1.fffc1ffff87f0p+585, 0x1.0000007c00000p+146, 0x1.fff0000000000p+75, -0x1.ffffff17ffe07p+494, -0x1.17ffe1f85d360p+440},
{-0x1.00000007fff00p+641, -0x1.fffc1ffff87f0p+585, -0x1.0000007c00000p+146, -0x1.fff0000000000p+75, 0x1.ffffff17ffe07p+494, 0x1.17ffe1f85d360p+440},
{0x1.fffffffe00000p+119, 0x1.e000001fff800p+55, 0x1.0000000000000p+792, -0x1.f000080000000p+707, 0x1.fffffffe00000p-673, 0x1.e0001f2000000p-737},
{0x1.fffffffe00000p+119, 0x1.e000001fff800p+55, -0x1.0000000000000p+792, 0x1.f000080000000p+707, -0x1.fffffffe00000p-673, -0x1.e0001f2000000p-737},
{-0x1.fffffffe00000p+119, -0x1.e000001fff800p+55, 0x1.0000000000000p+792, -0x1.f000080000000p+707, -0x1.fffffffe00000p-673, -0x1.e0001f2000000p-737},
{-0x1.fffffffe00000p+119, -0x1.e000001fff800p+55, -0x1.0000000000000p+792, 0x1.f000080000000p+707, 0x1.fffffffe00000p-673, 0x1.e0001f2000000p-737},
{0x1.00007fffffe00p-629, 0x1.ffffff8000000p-699, 0x1.001f03f07ff00p-553, 0x1.ffffff8000000p-626, 0x1.ffc2ff82ff538p-77, -0x1.3d82a9119b388p-132},
{0x1.00007fffffe00p-629, 0x1.ffffff8000000p-699, -0x1.001f03f07ff00p-553, -0x1.ffffff8000000p-626, -0x1.ffc2ff82ff538p-77, 0x1.3d82a9119b388p-132},
{-0x1.00007fffffe00p-629, -0x1.ffffff8000000p-699, 0x1.001f03f07ff00p-553, 0x1.ffffff8000000p-626, -0x1.ffc2ff82ff538p-77, 0x1.3d82a9119b388p-132},
{-0x1.00007fffffe00p-629, -0x1.ffffff8000000p-699, -0x1.001f03f07ff00p-553, -0x1.ffffff8000000p-626, 0x1.ffc2ff82ff538p-77, -0x1.3d82a9119b388p-132},
{0x1.00003fffffffcp-120, 0x1.ffe0000700000p-193, 0x1.001f000000040p-170, 0x1.ffc0ffffc0000p-241, 0x1.ffc287719935fp+49, 0x1.b6b0fe47ab6b8p-6},
{0x1.00003fffffffcp-120, 0x1.ffe0000700000p-193, -0x1.001f000000040p-170, -0x1.ffc0ffffc0000p-241, -0x1.ffc287719935fp+49, -0x1.b6b0fe47ab6b8p-6},
{-0x1.00003fffffffcp-120, -0x1.ffe0000700000p-193, 0x1.001f000000040p-170, 0x1.ffc0ffffc0000p-241, -0x1.ffc287719935fp+49, -0x1.b6b0fe47ab6b8p-6},
{-0x1.00003fffffffcp-120, -0x1.ffe0000700000p-193, -0x1.001f000000040p-170, -0x1.ffc0ffffc0000p-241, 0x1.ffc287719935fp+49, 0x1.b6b0fe47ab6b8p-6},
{0x1.0000100000000p+396, -0x1.0000000000000p+314, 0x1.0000020000000p-778, -0x1.0000000000000p-870, INFINITY, 0x0.0000000000000p+0},
{0x1.0000100000000p+396, -0x1.0000000000000p+314, -0x1.0000020000000p-778, 0x1.0000000000000p-870, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000100000000p+396, 0x1.0000000000000p+314, 0x1.0000020000000p-778, -0x1.0000000000000p-870, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000100000000p+396, 0x1.0000000000000p+314, -0x1.0000020000000p-778, 0x1.0000000000000p-870, INFINITY, 0x0.0000000000000p+0},
{0x1.000007c000000p-728, 0x1.fff8000000000p-784, 0x1.003fff0000200p-354, -0x1.fe0003fc00040p-413, 0x1.ff8031732328cp-375, -0x1.49885abdce908p-430},
{0x1.000007c000000p-728, 0x1.fff8000000000p-784, -0x1.003fff0000200p-354, 0x1.fe0003fc00040p-413, -0x1.ff8031732328cp-375, 0x1.49885abdce908p-430},
{-0x1.000007c000000p-728, -0x1.fff8000000000p-784, 0x1.003fff0000200p-354, -0x1.fe0003fc00040p-413, -0x1.ff8031732328cp-375, 0x1.49885abdce908p-430},
{-0x1.000007c000000p-728, -0x1.fff8000000000p-784, -0x1.003fff0000200p-354, 0x1.fe0003fc00040p-413, 0x1.ff8031732328cp-375, -0x1.49885abdce908p-430},
{0x1.00001fffcffc0p-216, -0x1.fffe000000004p-271, 0x1.03fffff800000p+515, 0x1.fffffe0000000p+447, 0x1.f81fc10b34257p-732, -0x1.5454259251f24p-786},
{0x1.00001fffcffc0p-216, -0x1.fffe000000004p-271, -0x1.03fffff800000p+515, -0x1.fffffe0000000p+447, -0x1.f81fc10b34257p-732, 0x1.5454259251f24p-786},
{-0x1.00001fffcffc0p-216, 0x1.fffe000000004p-271, 0x1.03fffff800000p+515, 0x1.fffffe0000000p+447, -0x1.f81fc10b34257p-732, 0x1.5454259251f24p-786},
{-0x1.00001fffcffc0p-216, 0x1.fffe000000004p-271, -0x1.03fffff800000p+515, -0x1.fffffe0000000p+447, 0x1.f81fc10b34257p-732, -0x1.5454259251f24p-786},
{0x1.0000008000000p-347, -0x1.ffffff8200000p-404, 0x1.000fff8000000p+341, 0x1.fffffe0000000p+279, 0x1.ffe003ffb0070p-689, -0x1.53e881ead4d3cp-744},
{0x1.0000008000000p-347, -0x1.ffffff8200000p-404, -0x1.000fff8000000p+341, -0x1.fffffe0000000p+279, -0x1.ffe003ffb0070p-689, 0x1.53e881ead4d3cp-744},
{-0x1.0000008000000p-347, 0x1.ffffff8200000p-404, 0x1.000fff8000000p+341, 0x1.fffffe0000000p+279, -0x1.ffe003ffb0070p-689, 0x1.53e881ead4d3cp-744},
{-0x1.0000008000000p-347, 0x1.ffffff8200000p-404, -0x1.000fff8000000p+341, -0x1.fffffe0000000p+279, 0x1.ffe003ffb0070p-689, -0x1.53e881ead4d3cp-744},
{0x1.00000010001f8p-866, 0x1.fffc007e00000p-940, 0x1.0003fc0000000p+17, 0x1.ffff80ffffc00p-40, 0x1.fff8083fbf610p-884, -0x1.0ce0e337685c0p-941},
{0x1.00000010001f8p-866, 0x1.fffc007e00000p-940, -0x1.0003fc0000000p+17, -0x1.ffff80ffffc00p-40, -0x1.fff8083fbf610p-884, 0x1.0ce0e337685c0p-941},
{-0x1.00000010001f8p-866, -0x1.fffc007e00000p-940, 0x1.0003fc0000000p+17, 0x1.ffff80ffffc00p-40, -0x1.fff8083fbf610p-884, 0x1.0ce0e337685c0p-941},
{-0x1.00000010001f8p-866, -0x1.fffc007e00000p-940, -0x1.0003fc0000000p+17, -0x1.ffff80ffffc00p-40, 0x1.fff8083fbf610p-884, -0x1.0ce0e337685c0p-941},
{0x1.0000000000000p-406, -0x1.0000000000000p-501, 0x1.0300000001f00p+916, 0x1.fffffff800000p+840, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000000000000p-406, -0x1.0000000000000p-501, -0x1.0300000001f00p+916, -0x1.fffffff800000p+840, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000000000p-406, 0x1.0000000000000p-501, 0x1.0300000001f00p+916, 0x1.fffffff800000p+840, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000000000p-406, 0x1.0000000000000p-501, -0x1.0300000001f00p+916, -0x1.fffffff800000p+840, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.07ffffff1fe00p-512, 0x1.fc00000000000p-609, 0x1.0040000000000p-704, -0x1.c000000400000p-765, 0x1.07be107b011fcp+192, -0x1.f7e6ed0b063d2p+138},
{0x1.07ffffff1fe00p-512, 0x1.fc00000000000p-609, -0x1.0040000000000p-704, 0x1.c000000400000p-765, -0x1.07be107b011fcp+192, 0x1.f7e6ed0b063d2p+138},
{-0x1.07ffffff1fe00p-512, -0x1.fc00000000000p-609, 0x1.0040000000000p-704, -0x1.c000000400000p-765, -0x1.07be107b011fcp+192, 0x1.f7e6ed0b063d2p+138},
{-0x1.07ffffff1fe00p-512, -0x1.fc00000000000p-609, -0x1.0040000000000p-704, 0x1.c000000400000p-765, 0x1.07be107b011fcp+192, -0x1.f7e6ed0b063d2p+138},
{0x1.0080000000000p+198, -0x1.0000000000000p+93, 0x1.fffff1ff80000p-814, 0x1.ffffc000fe000p-869, 0x1.00800703c0512p+1011, -0x1.7f7600e4b93c0p+957},
{0x1.0080000000000p+198, -0x1.0000000000000p+93, -0x1.fffff1ff80000p-814, -0x1.ffffc000fe000p-869, -0x1.00800703c0512p+1011, 0x1.7f7600e4b93c0p+957},
{-0x1.0080000000000p+198, 0x1.0000000000000p+93, 0x1.fffff1ff80000p-814, 0x1.ffffc000fe000p-869, -0x1.00800703c0512p+1011, 0x1.7f7600e4b93c0p+957},
{-0x1.0080000000000p+198, 0x1.0000000000000p+93, -0x1.fffff1ff80000p-814, -0x1.ffffc000fe000p-869, 0x1.00800703c0512p+1011, -0x1.7f7600e4b93c0p+957},
{0x1.0400000000000p+566, -0x1.0000000000000p+461, 0x1.7fffe00000008p+712, -0x1.e000080000000p+632, 0x1.5aaac78e3b4b6p-147, 0x1.ae0183122f590p-201},
{0x1.0400000000000p+566, -0x1.0000000000000p+461, -0x1.7fffe00000008p+712, 0x1.e000080000000p+632, -0x1.5aaac78e3b4b6p-147, -0x1.ae0183122f590p-201},
{-0x1.0400000000000p+566, 0x1.0000000000000p+461, 0x1.7fffe00000008p+712, -0x1.e000080000000p+632, -0x1.5aaac78e3b4b6p-147, -0x1.ae0183122f590p-201},
{-0x1.0400000000000p+566, 0x1.0000000000000p+461, -0x1.7fffe00000008p+712, 0x1.e000080000000p+632, 0x1.5aaac78e3b4b6p-147, 0x1.ae0183122f590p-201},
{0x1.000001d000000p-530, -0x1.ff80000200000p-585, 0x1.0000003fffffcp+783, 0x1.feffff8000000p+703, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000001d000000p-530, -0x1.ff80000200000p-585, -0x1.0000003fffffcp+783, -0x1.feffff8000000p+703, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.000001d000000p-530, 0x1.ff80000200000p-585, 0x1.0000003fffffcp+783, 0x1.feffff8000000p+703, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.000001d000000p-530, 0x1.ff80000200000p-585, -0x1.0000003fffffcp+783, -0x1.feffff8000000p+703, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.001ffffffffc0p+355, 0x1.fffffff03ff00p+294, 0x1.0000010000000p-724, -0x1.fffffff810000p-793, INFINITY, 0x0.0000000000000p+0},
{0x1.001ffffffffc0p+355, 0x1.fffffff03ff00p+294, -0x1.0000010000000p-724, 0x1.fffffff810000p-793, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001ffffffffc0p+355, -0x1.fffffff03ff00p+294, 0x1.0000010000000p-724, -0x1.fffffff810000p-793, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001ffffffffc0p+355, -0x1.fffffff03ff00p+294, -0x1.0000010000000p-724, 0x1.fffffff810000p-793, INFINITY, 0x0.0000000000000p+0},
{0x1.00000fffc0001p+277, -0x1.fc000007e0040p+218, 0x1.000007fffffc0p+726, 0x1.fc0003fc00000p+666, 0x1.000007ffbfc41p-449, 0x1.a2bf70aa60b60p-505},
{0x1.00000fffc0001p+277, -0x1.fc000007e0040p+218, -0x1.000007fffffc0p+726, -0x1.fc0003fc00000p+666, -0x1.000007ffbfc41p-449, -0x1.a2bf70aa60b60p-505},
{-0x1.00000fffc0001p+277, 0x1.fc000007e0040p+218, 0x1.000007fffffc0p+726, 0x1.fc0003fc00000p+666, -0x1.000007ffbfc41p-449, -0x1.a2bf70aa60b60p-505},
{-0x1.00000fffc0001p+277, 0x1.fc000007e0040p+218, -0x1.000007fffffc0p+726, -0x1.fc0003fc00000p+666, 0x1.000007ffbfc41p-449, 0x1.a2bf70aa60b60p-505},
{0x1.01ffe000003fep-455, 0x1.ffffff8000000p-521, 0x1.00fffffe00000p-662, 0x1.ffffffe7ff800p-723, 0x1.00fee120e15c4p+207, 0x1.7b11a0fd9d316p+153},
{0x1.01ffe000003fep-455, 0x1.ffffff8000000p-521, -0x1.00fffffe00000p-662, -0x1.ffffffe7ff800p-723, -0x1.00fee120e15c4p+207, -0x1.7b11a0fd9d316p+153},
{-0x1.01ffe000003fep-455, -0x1.ffffff8000000p-521, 0x1.00fffffe00000p-662, 0x1.ffffffe7ff800p-723, -0x1.00fee120e15c4p+207, -0x1.7b11a0fd9d316p+153},
{-0x1.01ffe000003fep-455, -0x1.ffffff8000000p-521, -0x1.00fffffe00000p-662, -0x1.ffffffe7ff800p-723, 0x1.00fee120e15c4p+207, 0x1.7b11a0fd9d316p+153},
{0x1.3fffffffffc00p+478, 0x1.ffffffe000000p+415, 0x1.07c0ffc200000p+108, -0x1.e000003bfffe0p+50, 0x1.3697b20f8de20p+370, -0x1.388b9803adee0p+316},
{0x1.3fffffffffc00p+478, 0x1.ffffffe000000p+415, -0x1.07c0ffc200000p+108, 0x1.e000003bfffe0p+50, -0x1.3697b20f8de20p+370, 0x1.388b9803adee0p+316},
{-0x1.3fffffffffc00p+478, -0x1.ffffffe000000p+415, 0x1.07c0ffc200000p+108, -0x1.e000003bfffe0p+50, -0x1.3697b20f8de20p+370, 0x1.388b9803adee0p+316},
{-0x1.3fffffffffc00p+478, -0x1.ffffffe000000p+415, -0x1.07c0ffc200000p+108, 0x1.e000003bfffe0p+50, 0x1.3697b20f8de20p+370, -0x1.388b9803adee0p+316},
{0x1.0000001800000p+405, 0x1.fffffc0000200p+351, 0x1.0000004000000p-957, -0x1.fff0000004000p-1024, INFINITY, 0x0.0000000000000p+0},
{0x1.0000001800000p+405, 0x1.fffffc0000200p+351, -0x1.0000004000000p-957, 0x1.fff0000004000p-1024, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000001800000p+405, -0x1.fffffc0000200p+351, 0x1.0000004000000p-957, -0x1.fff0000004000p-1024, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000001800000p+405, -0x1.fffffc0000200p+351, -0x1.0000004000000p-957, 0x1.fff0000004000p-1024, INFINITY, 0x0.0000000000000p+0},
{0x1.0000800000000p+271, -0x1.0000000000000p+208, 0x1.0fc007c004000p-537, -0x1.fffe100000000p-601, 0x1.e2544251693b5p+807, 0x1.bf6c5bc03d0b8p+751},
{0x1.0000800000000p+271, -0x1.0000000000000p+208, -0x1.0fc007c004000p-537, 0x1.fffe100000000p-601, -0x1.e2544251693b5p+807, -0x1.bf6c5bc03d0b8p+751},
{-0x1.0000800000000p+271, 0x1.0000000000000p+208, 0x1.0fc007c004000p-537, -0x1.fffe100000000p-601, -0x1.e2544251693b5p+807, -0x1.bf6c5bc03d0b8p+751},
{-0x1.0000800000000p+271, 0x1.0000000000000p+208, -0x1.0fc007c004000p-537, 0x1.fffe100000000p-601, 0x1.e2544251693b5p+807, 0x1.bf6c5bc03d0b8p+751},
{0x1.00007fffffe00p-85, 0x1.ffffffc003f00p-146, 0x1.0080000000000p-834, -0x1.0000000000000p-892, 0x1.ff017f405f904p+748, -0x1.df2057e109728p+694},
{0x1.00007fffffe00p-85, 0x1.ffffffc003f00p-146, -0x1.0080000000000p-834, 0x1.0000000000000p-892, -0x1.ff017f405f904p+748, 0x1.df2057e109728p+694},
{-0x1.00007fffffe00p-85, -0x1.ffffffc003f00p-146, 0x1.0080000000000p-834, -0x1.0000000000000p-892, -0x1.ff017f405f904p+748, 0x1.df2057e109728p+694},
{-0x1.00007fffffe00p-85, -0x1.ffffffc003f00p-146, -0x1.0080000000000p-834, 0x1.0000000000000p-892, 0x1.ff017f405f904p+748, -0x1.df2057e109728p+694},
{0x1.0000200000000p-779, -0x1.0000000000000p-835, 0x1.00000ffffff80p-484, 0x1.ffffff0007ff8p-540, 0x1.00000fffff080p-295, -0x1.77fff803e38b4p-350},
{0x1.0000200000000p-779, -0x1.0000000000000p-835, -0x1.00000ffffff80p-484, -0x1.ffffff0007ff8p-540, -0x1.00000fffff080p-295, 0x1.77fff803e38b4p-350},
{-0x1.0000200000000p-779, 0x1.0000000000000p-835, 0x1.00000ffffff80p-484, 0x1.ffffff0007ff8p-540, -0x1.00000fffff080p-295, 0x1.77fff803e38b4p-350},
{-0x1.0000200000000p-779, 0x1.0000000000000p-835, -0x1.00000ffffff80p-484, -0x1.ffffff0007ff8p-540, 0x1.00000fffff080p-295, -0x1.77fff803e38b4p-350},
{0x1.00fe000000000p-622, 0x1.001ffffff0000p-691, 0x1.00007fffff000p+682, 0x0.0000000000000p+0, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00fe000000000p-622, 0x1.001ffffff0000p-691, -0x1.00007fffff000p+682, 0x0.0000000000000p+0, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00fe000000000p-622, -0x1.001ffffff0000p-691, 0x1.00007fffff000p+682, 0x0.0000000000000p+0, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00fe000000000p-622, -0x1.001ffffff0000p-691, -0x1.00007fffff000p+682, 0x0.0000000000000p+0, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.01e0000000002p+213, -0x1.fffffffffc200p+154, 0x1.0000080000000p-358, -0x1.0000000000000p-463, 0x1.01dff7f100409p+571, 0x1.ef7ed084099b0p+517},
{0x1.01e0000000002p+213, -0x1.fffffffffc200p+154, -0x1.0000080000000p-358, 0x1.0000000000000p-463, -0x1.01dff7f100409p+571, -0x1.ef7ed084099b0p+517},
{-0x1.01e0000000002p+213, 0x1.fffffffffc200p+154, 0x1.0000080000000p-358, -0x1.0000000000000p-463, -0x1.01dff7f100409p+571, -0x1.ef7ed084099b0p+517},
{-0x1.01e0000000002p+213, 0x1.fffffffffc200p+154, -0x1.0000080000000p-358, 0x1.0000000000000p-463, 0x1.01dff7f100409p+571, 0x1.ef7ed084099b0p+517},
{0x1.000000f1fe001p+66, -0x1.0000000000000p-17, 0x1.00003fffffff0p-433, 0x1.fc00000000000p-510, 0x1.ffff81e41b892p+498, -0x1.b8b120ccd3920p+444},
{0x1.000000f1fe001p+66, -0x1.0000000000000p-17, -0x1.00003fffffff0p-433, -0x1.fc00000000000p-510, -0x1.ffff81e41b892p+498, 0x1.b8b120ccd3920p+444},
{-0x1.000000f1fe001p+66, 0x1.0000000000000p-17, 0x1.00003fffffff0p-433, 0x1.fc00000000000p-510, -0x1.ffff81e41b892p+498, 0x1.b8b120ccd3920p+444},
{-0x1.000000f1fe001p+66, 0x1.0000000000000p-17, -0x1.00003fffffff0p-433, -0x1.fc00000000000p-510, 0x1.ffff81e41b892p+498, -0x1.b8b120ccd3920p+444},
{0x1.0008000000010p+115, -0x1.f8000ffff9000p+50, 0x1.00001ffff0000p-192, 0x1.ffff80001f000p-249, 0x1.0007dfff1401ap+307, -0x1.60516bf0b366cp+253},
{0x1.0008000000010p+115, -0x1.f8000ffff9000p+50, -0x1.00001ffff0000p-192, -0x1.ffff80001f000p-249, -0x1.0007dfff1401ap+307, 0x1.60516bf0b366cp+253},
{-0x1.0008000000010p+115, 0x1.f8000ffff9000p+50, 0x1.00001ffff0000p-192, 0x1.ffff80001f000p-249, -0x1.0007dfff1401ap+307, 0x1.60516bf0b366cp+253},
{-0x1.0008000000010p+115, 0x1.f8000ffff9000p+50, -0x1.00001ffff0000p-192, -0x1.ffff80001f000p-249, 0x1.0007dfff1401ap+307, -0x1.60516bf0b366cp+253},
{0x1.0000100000000p-13, -0x1.0000000000000p-118, 0x1.0038000800000p+253, -0x1.fffffe0000080p+193, 0x1.ff903863adb04p-267, -0x1.6b2c09860d9d8p-323},
{0x1.0000100000000p-13, -0x1.0000000000000p-118, -0x1.0038000800000p+253, 0x1.fffffe0000080p+193, -0x1.ff903863adb04p-267, 0x1.6b2c09860d9d8p-323},
{-0x1.0000100000000p-13, 0x1.0000000000000p-118, 0x1.0038000800000p+253, -0x1.fffffe0000080p+193, -0x1.ff903863adb04p-267, 0x1.6b2c09860d9d8p-323},
{-0x1.0000100000000p-13, 0x1.0000000000000p-118, -0x1.0038000800000p+253, 0x1.fffffe0000080p+193, 0x1.ff903863adb04p-267, -0x1.6b2c09860d9d8p-323},
{0x1.00003fe008000p-845, -0x1.fffdff0000080p-904, 0x1.ffffffffff800p+109, 0x1.fe00000000000p+11, 0x1.00003fe008400p-955, -0x1.803dee8002680p-1014},
{0x1.00003fe008000p-845, -0x1.fffdff0000080p-904, -0x1.ffffffffff800p+109, -0x1.fe00000000000p+11, -0x1.00003fe008400p-955, 0x1.803dee8002680p-1014},
{-0x1.00003fe008000p-845, 0x1.fffdff0000080p-904, 0x1.ffffffffff800p+109, 0x1.fe00000000000p+11, -0x1.00003fe008400p-955, 0x1.803dee8002680p-1014},
{-0x1.00003fe008000p-845, 0x1.fffdff0000080p-904, -0x1.ffffffffff800p+109, -0x1.fe00000000000p+11, 0x1.00003fe008400p-955, -0x1.803dee8002680p-1014},
{0x1.1fffffff00000p-151, 0x1.ffffff8000000p-223, 0x1.0000100000000p-443, -0x1.0000000000000p-537, 0x1.1fffedff01201p+292, -0x1.1fefee00c1200p+232},
{0x1.1fffffff00000p-151, 0x1.ffffff8000000p-223, -0x1.0000100000000p-443, 0x1.0000000000000p-537, -0x1.1fffedff01201p+292, 0x1.1fefee00c1200p+232},
{-0x1.1fffffff00000p-151, -0x1.ffffff8000000p-223, 0x1.0000100000000p-443, -0x1.0000000000000p-537, -0x1.1fffedff01201p+292, 0x1.1fefee00c1200p+232},
{-0x1.1fffffff00000p-151, -0x1.ffffff8000000p-223, -0x1.0000100000000p-443, 0x1.0000000000000p-537, 0x1.1fffedff01201p+292, -0x1.1fefee00c1200p+232},
{0x1.0ffffc0000010p-901, 0x1.f000000ff0000p-970, 0x1.00003ff002010p-855, -0x1.0000020000000p-920, 0x1.0fffb8110fd73p-46, -0x1.2a08548118244p-101},
{0x1.0ffffc0000010p-901, 0x1.f000000ff0000p-970, -0x1.00003ff002010p-855, 0x1.0000020000000p-920, -0x1.0fffb8110fd73p-46, 0x1.2a08548118244p-101},
{-0x1.0ffffc0000010p-901, -0x1.f000000ff0000p-970, 0x1.00003ff002010p-855, -0x1.0000020000000p-920, -0x1.0fffb8110fd73p-46, 0x1.2a08548118244p-101},
{-0x1.0ffffc0000010p-901, -0x1.f000000ff0000p-970, -0x1.00003ff002010p-855, 0x1.0000020000000p-920, 0x1.0fffb8110fd73p-46, -0x1.2a08548118244p-101},
{0x1.00ffffffc0000p+149, 0x1.fffffe00003f0p+94, 0x1.1fffffffff020p-235, -0x1.ffff000400000p-310, 0x1.c8e38e38735a1p+383, 0x1.ba7818263c098p+329},
{0x1.00ffffffc0000p+149, 0x1.fffffe00003f0p+94, -0x1.1fffffffff020p-235, 0x1.ffff000400000p-310, -0x1.c8e38e38735a1p+383, -0x1.ba7818263c098p+329},
{-0x1.00ffffffc0000p+149, -0x1.fffffe00003f0p+94, 0x1.1fffffffff020p-235, -0x1.ffff000400000p-310, -0x1.c8e38e38735a1p+383, -0x1.ba7818263c098p+329},
{-0x1.00ffffffc0000p+149, -0x1.fffffe00003f0p+94, -0x1.1fffffffff020p-235, 0x1.ffff000400000p-310, 0x1.c8e38e38735a1p+383, 0x1.ba7818263c098p+329},
{0x1.0200000000000p+564, -0x1.0000000000000p+511, 0x1.0000700000000p-366, 0x1.ffffffe000000p-437, 0x1.01ff8f203161ep+930, 0x1.329475dfee4e0p+875},
{0x1.0200000000000p+564, -0x1.0000000000000p+511, -0x1.0000700000000p-366, -0x1.ffffffe000000p-437, -0x1.01ff8f203161ep+930, -0x1.329475dfee4e0p+875},
{-0x1.0200000000000p+564, 0x1.0000000000000p+511, 0x1.0000700000000p-366, 0x1.ffffffe000000p-437, -0x1.01ff8f203161ep+930, -0x1.329475dfee4e0p+875},
{-0x1.0200000000000p+564, 0x1.0000000000000p+511, -0x1.0000700000000p-366, -0x1.ffffffe000000p-437, 0x1.01ff8f203161ep+930, 0x1.329475dfee4e0p+875},
{0x1.3fffffc0001fcp+882, 0x1.007fffff00000p+819, 0x1.007fffff80000p-564, 0x1.fffffffe00000p-632, INFINITY, 0x0.0000000000000p+0},
{0x1.3fffffc0001fcp+882, 0x1.007fffff00000p+819, -0x1.007fffff80000p-564, -0x1.fffffffe00000p-632, -INFINITY, 0x0.0000000000000p+0},
{-0x1.3fffffc0001fcp+882, -0x1.007fffff00000p+819, 0x1.007fffff80000p-564, 0x1.fffffffe00000p-632, -INFINITY, 0x0.0000000000000p+0},
{-0x1.3fffffc0001fcp+882, -0x1.007fffff00000p+819, -0x1.007fffff80000p-564, -0x1.fffffffe00000p-632, INFINITY, 0x0.0000000000000p+0},
{0x1.00000ffffffd0p-606, -0x1.ffffff8200000p-672, 0x1.00000c0000004p+258, -0x1.ff80000800000p+190, 0x1.000003ffffcccp-864, 0x1.02fbf43288800p-927},
{0x1.00000ffffffd0p-606, -0x1.ffffff8200000p-672, -0x1.00000c0000004p+258, 0x1.ff80000800000p+190, -0x1.000003ffffcccp-864, -0x1.02fbf43288800p-927},
{-0x1.00000ffffffd0p-606, 0x1.ffffff8200000p-672, 0x1.00000c0000004p+258, -0x1.ff80000800000p+190, -0x1.000003ffffcccp-864, -0x1.02fbf43288800p-927},
{-0x1.00000ffffffd0p-606, 0x1.ffffff8200000p-672, -0x1.00000c0000004p+258, 0x1.ff80000800000p+190, 0x1.000003ffffcccp-864, 0x1.02fbf43288800p-927},
{0x1.18003ffff8000p-770, 0x1.ffffff8000000p-836, 0x1.1ffff80000040p-581, -0x1.0000000000000p-671, 0x1.f1c79c0c99a42p-190, -0x1.672e152d5a5bep-244},
{0x1.18003ffff8000p-770, 0x1.ffffff8000000p-836, -0x1.1ffff80000040p-581, 0x1.0000000000000p-671, -0x1.f1c79c0c99a42p-190, 0x1.672e152d5a5bep-244},
{-0x1.18003ffff8000p-770, -0x1.ffffff8000000p-836, 0x1.1ffff80000040p-581, -0x1.0000000000000p-671, -0x1.f1c79c0c99a42p-190, 0x1.672e152d5a5bep-244},
{-0x1.18003ffff8000p-770, -0x1.ffffff8000000p-836, -0x1.1ffff80000040p-581, 0x1.0000000000000p-671, 0x1.f1c79c0c99a42p-190, -0x1.672e152d5a5bep-244},
{0x1.0ffffff9ffff0p+390, 0x1.f800008200000p+320, 0x1.000000ffffff8p+202, 0x1.ff0000007ffc0p+143, 0x1.0ffffeea0000ap+188, -0x1.21eba43b6d104p+133},
{0x1.0ffffff9ffff0p+390, 0x1.f800008200000p+320, -0x1.000000ffffff8p+202, -0x1.ff0000007ffc0p+143, -0x1.0ffffeea0000ap+188, 0x1.21eba43b6d104p+133},
{-0x1.0ffffff9ffff0p+390, -0x1.f800008200000p+320, 0x1.000000ffffff8p+202, 0x1.ff0000007ffc0p+143, -0x1.0ffffeea0000ap+188, 0x1.21eba43b6d104p+133},
{-0x1.0ffffff9ffff0p+390, -0x1.f800008200000p+320, -0x1.000000ffffff8p+202, -0x1.ff0000007ffc0p+143, 0x1.0ffffeea0000ap+188, -0x1.21eba43b6d104p+133},
{0x1.1fffffe000000p-899, 0x1.ffff800001ffep-953, 0x1.0000008000000p-864, -0x1.fffffc0000008p-920, 0x1.1fffff5000006p-35, 0x1.1ffee480048f8p-90},
{0x1.1fffffe000000p-899, 0x1.ffff800001ffep-953, -0x1.0000008000000p-864, 0x1.fffffc0000008p-920, -0x1.1fffff5000006p-35, -0x1.1ffee480048f8p-90},
{-0x1.1fffffe000000p-899, -0x1.ffff800001ffep-953, 0x1.0000008000000p-864, -0x1.fffffc0000008p-920, -0x1.1fffff5000006p-35, -0x1.1ffee480048f8p-90},
{-0x1.1fffffe000000p-899, -0x1.ffff800001ffep-953, -0x1.0000008000000p-864, 0x1.fffffc0000008p-920, 0x1.1fffff5000006p-35, 0x1.1ffee480048f8p-90},
{0x1.0000200000000p-148, -0x1.fffffff804000p-215, 0x1.0000080000000p+818, -0x1.0000000000000p+723, 0x1.000017ffff400p-966, 0x1.6ffff480c0400p-1028},
{0x1.0000200000000p-148, -0x1.fffffff804000p-215, -0x1.0000080000000p+818, 0x1.0000000000000p+723, -0x1.000017ffff400p-966, -0x1.6ffff480c0400p-1028},
{-0x1.0000200000000p-148, 0x1.fffffff804000p-215, 0x1.0000080000000p+818, -0x1.0000000000000p+723, -0x1.000017ffff400p-966, -0x1.6ffff480c0400p-1028},
{-0x1.0000200000000p-148, 0x1.fffffff804000p-215, -0x1.0000080000000p+818, 0x1.0000000000000p+723, 0x1.000017ffff400p-966, 0x1.6ffff480c0400p-1028},
{0x1.03ffff0000010p+113, -0x1.fffc010000000p+37, 0x1.0000020000000p+293, -0x1.0000000000000p+188, 0x1.03fffcf800071p-180, -0x1.c7fff471f0000p-249},
{0x1.03ffff0000010p+113, -0x1.fffc010000000p+37, -0x1.0000020000000p+293, 0x1.0000000000000p+188, -0x1.03fffcf800071p-180, 0x1.c7fff471f0000p-249},
{-0x1.03ffff0000010p+113, 0x1.fffc010000000p+37, 0x1.0000020000000p+293, -0x1.0000000000000p+188, -0x1.03fffcf800071p-180, 0x1.c7fff471f0000p-249},
{-0x1.03ffff0000010p+113, 0x1.fffc010000000p+37, -0x1.0000020000000p+293, 0x1.0000000000000p+188, 0x1.03fffcf800071p-180, -0x1.c7fff471f0000p-249},
{0x1.7ffffffc00000p-873, 0x1.fff80000fc000p-934, 0x1.0010000000000p+21, -0x1.ffffffffc4000p-46, 0x1.7fe8017be8418p-894, -0x1.01d7f400a641ep-948},
{0x1.7ffffffc00000p-873, 0x1.fff80000fc000p-934, -0x1.0010000000000p+21, 0x1.ffffffffc4000p-46, -0x1.7fe8017be8418p-894, 0x1.01d7f400a641ep-948},
{-0x1.7ffffffc00000p-873, -0x1.fff80000fc000p-934, 0x1.0010000000000p+21, -0x1.ffffffffc4000p-46, -0x1.7fe8017be8418p-894, 0x1.01d7f400a641ep-948},
{-0x1.7ffffffc00000p-873, -0x1.fff80000fc000p-934, -0x1.0010000000000p+21, 0x1.ffffffffc4000p-46, 0x1.7fe8017be8418p-894, -0x1.01d7f400a641ep-948},
{0x1.000007fffc004p-531, -0x1.8000004000000p-590, 0x1.7fe000001ff00p+464, 0x1.f000000000000p+363, 0x1.5571d426e1829p-996, -0x1.e917b80000000p-1050},
{0x1.000007fffc004p-531, -0x1.8000004000000p-590, -0x1.7fe000001ff00p+464, -0x1.f000000000000p+363, -0x1.5571d426e1829p-996, 0x1.e917b80000000p-1050},
{-0x1.000007fffc004p-531, 0x1.8000004000000p-590, 0x1.7fe000001ff00p+464, 0x1.f000000000000p+363, -0x1.5571d426e1829p-996, 0x1.e917b80000000p-1050},
{-0x1.000007fffc004p-531, 0x1.8000004000000p-590, -0x1.7fe000001ff00p+464, -0x1.f000000000000p+363, 0x1.5571d426e1829p-996, -0x1.e917b80000000p-1050},
{0x1.01ffffffffc00p+132, 0x0.0000000000000p+0, 0x1.0ffffff000000p+156, 0x1.fffffff000000p+86, 0x1.e5a5a5c236733p-25, -0x1.9d54b36e3ce32p-79},
{0x1.01ffffffffc00p+132, 0x0.0000000000000p+0, -0x1.0ffffff000000p+156, -0x1.fffffff000000p+86, -0x1.e5a5a5c236733p-25, 0x1.9d54b36e3ce32p-79},
{-0x1.01ffffffffc00p+132, 0x0.0000000000000p+0, 0x1.0ffffff000000p+156, 0x1.fffffff000000p+86, -0x1.e5a5a5c236733p-25, 0x1.9d54b36e3ce32p-79},
{-0x1.01ffffffffc00p+132, 0x0.0000000000000p+0, -0x1.0ffffff000000p+156, -0x1.fffffff000000p+86, 0x1.e5a5a5c236733p-25, -0x1.9d54b36e3ce32p-79},
{0x1.0000010000000p-428, -0x1.ffffffffe4000p-490, 0x1.0000020000000p-498, -0x1.ff80000100000p-571, 0x1.fffffe0000040p+69, -0x1.006005ff51e00p+9},
{0x1.0000010000000p-428, -0x1.ffffffffe4000p-490, -0x1.0000020000000p-498, 0x1.ff80000100000p-571, -0x1.fffffe0000040p+69, 0x1.006005ff51e00p+9},
{-0x1.0000010000000p-428, 0x1.ffffffffe4000p-490, 0x1.0000020000000p-498, -0x1.ff80000100000p-571, -0x1.fffffe0000040p+69, 0x1.006005ff51e00p+9},
{-0x1.0000010000000p-428, 0x1.ffffffffe4000p-490, -0x1.0000020000000p-498, 0x1.ff80000100000p-571, 0x1.fffffe0000040p+69, -0x1.006005ff51e00p+9},
{0x1.0008000000000p+313, -0x1.0000000000000p+208, 0x1.000fffff80000p+721, 0x1.fffffffc00000p+665, 0x1.fff00100efe90p-409, -0x1.07a38c84ab908p-464},
{0x1.0008000000000p+313, -0x1.0000000000000p+208, -0x1.000fffff80000p+721, -0x1.fffffffc00000p+665, -0x1.fff00100efe90p-409, 0x1.07a38c84ab908p-464},
{-0x1.0008000000000p+313, 0x1.0000000000000p+208, 0x1.000fffff80000p+721, 0x1.fffffffc00000p+665, -0x1.fff00100efe90p-409, 0x1.07a38c84ab908p-464},
{-0x1.0008000000000p+313, 0x1.0000000000000p+208, -0x1.000fffff80000p+721, -0x1.fffffffc00000p+665, 0x1.fff00100efe90p-409, -0x1.07a38c84ab908p-464},
{0x1.007fffffffc00p+944, 0x1.ffffc00000000p+879, 0x1.1ffffffff8000p-208, 0x1.fe001ffffffe0p-264, INFINITY, 0x0.0000000000000p+0},
{0x1.007fffffffc00p+944, 0x1.ffffc00000000p+879, -0x1.1ffffffff8000p-208, -0x1.fe001ffffffe0p-264, -INFINITY, 0x0.0000000000000p+0},
{-0x1.007fffffffc00p+944, -0x1.ffffc00000000p+879, 0x1.1ffffffff8000p-208, 0x1.fe001ffffffe0p-264, -INFINITY, 0x0.0000000000000p+0},
{-0x1.007fffffffc00p+944, -0x1.ffffc00000000p+879, -0x1.1ffffffff8000p-208, -0x1.fe001ffffffe0p-264, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000fffe00p+879, 0x1.3fffff87fc000p+819, 0x1.00003ffffc000p-310, 0x1.e003ff8000000p-375, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000fffe00p+879, 0x1.3fffff87fc000p+819, -0x1.00003ffffc000p-310, -0x1.e003ff8000000p-375, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000fffe00p+879, -0x1.3fffff87fc000p+819, 0x1.00003ffffc000p-310, 0x1.e003ff8000000p-375, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000fffe00p+879, -0x1.3fffff87fc000p+819, -0x1.00003ffffc000p-310, -0x1.e003ff8000000p-375, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000800000p-526, -0x1.0000000000000p-631, 0x1.7ffff80000000p-359, 0x1.ffffffffe0000p-415, 0x1.55555c7c71ed4p-168, 0x1.30ff0fb9aebd0p-225},
{0x1.0000000800000p-526, -0x1.0000000000000p-631, -0x1.7ffff80000000p-359, -0x1.ffffffffe0000p-415, -0x1.55555c7c71ed4p-168, -0x1.30ff0fb9aebd0p-225},
{-0x1.0000000800000p-526, 0x1.0000000000000p-631, 0x1.7ffff80000000p-359, 0x1.ffffffffe0000p-415, -0x1.55555c7c71ed4p-168, -0x1.30ff0fb9aebd0p-225},
{-0x1.0000000800000p-526, 0x1.0000000000000p-631, -0x1.7ffff80000000p-359, -0x1.ffffffffe0000p-415, 0x1.55555c7c71ed4p-168, 0x1.30ff0fb9aebd0p-225},
{0x1.00fffffff8000p-357, 0x1.fffe000000000p-431, 0x1.0000100000000p-423, -0x1.fffffffffc100p-478, 0x1.00ffefeff9010p+66, 0x1.1cfbfe2017250p+12},
{0x1.00fffffff8000p-357, 0x1.fffe000000000p-431, -0x1.0000100000000p-423, 0x1.fffffffffc100p-478, -0x1.00ffefeff9010p+66, -0x1.1cfbfe2017250p+12},
{-0x1.00fffffff8000p-357, -0x1.fffe000000000p-431, 0x1.0000100000000p-423, -0x1.fffffffffc100p-478, -0x1.00ffefeff9010p+66, -0x1.1cfbfe2017250p+12},
{-0x1.00fffffff8000p-357, -0x1.fffe000000000p-431, -0x1.0000100000000p-423, 0x1.fffffffffc100p-478, 0x1.00ffefeff9010p+66, 0x1.1cfbfe2017250p+12},
{0x1.0007ffffffff0p-319, 0x1.fffffc0000000p-402, 0x1.0003800000080p-931, -0x1.0000000000000p-1036, 0x1.00047ff0402e2p+612, -0x1.8ad99d893a930p+555},
{0x1.0007ffffffff0p-319, 0x1.fffffc0000000p-402, -0x1.0003800000080p-931, 0x1.0000000000000p-1036, -0x1.00047ff0402e2p+612, 0x1.8ad99d893a930p+555},
{-0x1.0007ffffffff0p-319, -0x1.fffffc0000000p-402, 0x1.0003800000080p-931, -0x1.0000000000000p-1036, -0x1.00047ff0402e2p+612, 0x1.8ad99d893a930p+555},
{-0x1.0007ffffffff0p-319, -0x1.fffffc0000000p-402, -0x1.0003800000080p-931, 0x1.0000000000000p-1036, 0x1.00047ff0402e2p+612, -0x1.8ad99d893a930p+555},
{0x1.000001fffff80p-279, 0x1.ff00000000000p-363, 0x1.0000020000000p-640, -0x1.8002000000000p-716, 0x1.fffffffffff00p+360, 0x1.0181fefcfc000p+293},
{0x1.000001fffff80p-279, 0x1.ff00000000000p-363, -0x1.0000020000000p-640, 0x1.8002000000000p-716, -0x1.fffffffffff00p+360, -0x1.0181fefcfc000p+293},
{-0x1.000001fffff80p-279, -0x1.ff00000000000p-363, 0x1.0000020000000p-640, -0x1.8002000000000p-716, -0x1.fffffffffff00p+360, -0x1.0181fefcfc000p+293},
{-0x1.000001fffff80p-279, -0x1.ff00000000000p-363, -0x1.0000020000000p-640, 0x1.8002000000000p-716, 0x1.fffffffffff00p+360, 0x1.0181fefcfc000p+293},
{0x1.0003fffff0000p+964, 0x1.ffff800007f80p+904, 0x1.000007ffffe00p+308, -0x1.fe0c000004000p+250, 0x1.0003f7ffd0600p+656, 0x1.4dc26b9fd5b00p+601},
{0x1.0003fffff0000p+964, 0x1.ffff800007f80p+904, -0x1.000007ffffe00p+308, 0x1.fe0c000004000p+250, -0x1.0003f7ffd0600p+656, -0x1.4dc26b9fd5b00p+601},
{-0x1.0003fffff0000p+964, -0x1.ffff800007f80p+904, 0x1.000007ffffe00p+308, -0x1.fe0c000004000p+250, -0x1.0003f7ffd0600p+656, -0x1.4dc26b9fd5b00p+601},
{-0x1.0003fffff0000p+964, -0x1.ffff800007f80p+904, -0x1.000007ffffe00p+308, 0x1.fe0c000004000p+250, 0x1.0003f7ffd0600p+656, 0x1.4dc26b9fd5b00p+601},
{0x1.00000003f0000p+137, 0x1.ffffffe000000p+59, 0x1.0100000000000p+110, -0x1.fffffff000080p+50, 0x1.fe01fe09d629dp+26, 0x1.9a55be319d934p-28},
{0x1.00000003f0000p+137, 0x1.ffffffe000000p+59, -0x1.0100000000000p+110, 0x1.fffffff000080p+50, -0x1.fe01fe09d629dp+26, -0x1.9a55be319d934p-28},
{-0x1.00000003f0000p+137, -0x1.ffffffe000000p+59, 0x1.0100000000000p+110, -0x1.fffffff000080p+50, -0x1.fe01fe09d629dp+26, -0x1.9a55be319d934p-28},
{-0x1.00000003f0000p+137, -0x1.ffffffe000000p+59, -0x1.0100000000000p+110, 0x1.fffffff000080p+50, 0x1.fe01fe09d629dp+26, 0x1.9a55be319d934p-28},
{0x1.007fffffffc00p+472, 0x1.fffffe0000000p+390, 0x1.0000003fffff8p-373, 0x1.fffff80000000p-444, 0x1.007fffbfdfc09p+845, 0x1.202ee977b9860p+787},
{0x1.007fffffffc00p+472, 0x1.fffffe0000000p+390, -0x1.0000003fffff8p-373, -0x1.fffff80000000p-444, -0x1.007fffbfdfc09p+845, -0x1.202ee977b9860p+787},
{-0x1.007fffffffc00p+472, -0x1.fffffe0000000p+390, 0x1.0000003fffff8p-373, 0x1.fffff80000000p-444, -0x1.007fffbfdfc09p+845, -0x1.202ee977b9860p+787},
{-0x1.007fffffffc00p+472, -0x1.fffffe0000000p+390, -0x1.0000003fffff8p-373, -0x1.fffff80000000p-444, 0x1.007fffbfdfc09p+845, 0x1.202ee977b9860p+787},
{0x1.00000003ff000p+517, 0x1.ffffff0000000p+455, 0x1.0000ffff80000p+482, 0x1.e0000000fffc0p+423, 0x1.fffe000afdf40p+34, 0x1.a97dc87838c28p-22},
{0x1.00000003ff000p+517, 0x1.ffffff0000000p+455, -0x1.0000ffff80000p+482, -0x1.e0000000fffc0p+423, -0x1.fffe000afdf40p+34, -0x1.a97dc87838c28p-22},
{-0x1.00000003ff000p+517, -0x1.ffffff0000000p+455, 0x1.0000ffff80000p+482, 0x1.e0000000fffc0p+423, -0x1.fffe000afdf40p+34, -0x1.a97dc87838c28p-22},
{-0x1.00000003ff000p+517, -0x1.ffffff0000000p+455, -0x1.0000ffff80000p+482, -0x1.e0000000fffc0p+423, 0x1.fffe000afdf40p+34, 0x1.a97dc87838c28p-22},
{0x1.007ffffffc000p-870, 0x1.ffffff8000000p-937, 0x1.0003fffffffe0p-380, 0x1.fffffff000780p-440, 0x1.007bfe1003c1fp-490, 0x1.d0655eb0f91c0p-546},
{0x1.007ffffffc000p-870, 0x1.ffffff8000000p-937, -0x1.0003fffffffe0p-380, -0x1.fffffff000780p-440, -0x1.007bfe1003c1fp-490, -0x1.d0655eb0f91c0p-546},
{-0x1.007ffffffc000p-870, -0x1.ffffff8000000p-937, 0x1.0003fffffffe0p-380, 0x1.fffffff000780p-440, -0x1.007bfe1003c1fp-490, -0x1.d0655eb0f91c0p-546},
{-0x1.007ffffffc000p-870, -0x1.ffffff8000000p-937, -0x1.0003fffffffe0p-380, -0x1.fffffff000780p-440, 0x1.007bfe1003c1fp-490, 0x1.d0655eb0f91c0p-546},
{0x1.07fffffe0003cp+175, 0x1.ffe000007f000p+110, 0x1.1ffff00800000p-290, -0x1.0000000000000p-387, 0x1.d5556f57b59dcp+464, 0x1.ea436d67e8132p+410},
{0x1.07fffffe0003cp+175, 0x1.ffe000007f000p+110, -0x1.1ffff00800000p-290, 0x1.0000000000000p-387, -0x1.d5556f57b59dcp+464, -0x1.ea436d67e8132p+410},
{-0x1.07fffffe0003cp+175, -0x1.ffe000007f000p+110, 0x1.1ffff00800000p-290, -0x1.0000000000000p-387, -0x1.d5556f57b59dcp+464, -0x1.ea436d67e8132p+410},
{-0x1.07fffffe0003cp+175, -0x1.ffe000007f000p+110, -0x1.1ffff00800000p-290, 0x1.0000000000000p-387, 0x1.d5556f57b59dcp+464, 0x1.ea436d67e8132p+410},
{0x1.000003fff8000p+401, 0x1.ffff00000003cp+346, 0x1.000001ffff020p+721, -0x1.fff0000200000p+651, 0x1.000001fff8fa0p-320, 0x1.04037c2a010d6p-374},
{0x1.000003fff8000p+401, 0x1.ffff00000003cp+346, -0x1.000001ffff020p+721, 0x1.fff0000200000p+651, -0x1.000001fff8fa0p-320, -0x1.04037c2a010d6p-374},
{-0x1.000003fff8000p+401, -0x1.ffff00000003cp+346, 0x1.000001ffff020p+721, -0x1.fff0000200000p+651, -0x1.000001fff8fa0p-320, -0x1.04037c2a010d6p-374},
{-0x1.000003fff8000p+401, -0x1.ffff00000003cp+346, -0x1.000001ffff020p+721, 0x1.fff0000200000p+651, 0x1.000001fff8fa0p-320, 0x1.04037c2a010d6p-374},
{0x1.00ffe00000000p-249, 0x1.fe00000000000p-309, 0x1.0000100000000p+516, -0x1.ffff800010000p+447, 0x1.00ffcff003010p-765, -0x1.01fef05060e00p-825},
{0x1.00ffe00000000p-249, 0x1.fe00000000000p-309, -0x1.0000100000000p+516, 0x1.ffff800010000p+447, -0x1.00ffcff003010p-765, 0x1.01fef05060e00p-825},
{-0x1.00ffe00000000p-249, -0x1.fe00000000000p-309, 0x1.0000100000000p+516, -0x1.ffff800010000p+447, -0x1.00ffcff003010p-765, 0x1.01fef05060e00p-825},
{-0x1.00ffe00000000p-249, -0x1.fe00000000000p-309, -0x1.0000100000000p+516, 0x1.ffff800010000p+447, 0x1.00ffcff003010p-765, -0x1.01fef05060e00p-825},
{0x1.fffffff000008p-375, -0x1.ffffff1000000p-452, 0x1.0001fffffff80p-412, 0x1.fffe000000000p-497, 0x1.fffbfff800208p+37, -0x1.83fe387f7d700p-21},
{0x1.fffffff000008p-375, -0x1.ffffff1000000p-452, -0x1.0001fffffff80p-412, -0x1.fffe000000000p-497, -0x1.fffbfff800208p+37, 0x1.83fe387f7d700p-21},
{-0x1.fffffff000008p-375, 0x1.ffffff1000000p-452, 0x1.0001fffffff80p-412, 0x1.fffe000000000p-497, -0x1.fffbfff800208p+37, 0x1.83fe387f7d700p-21},
{-0x1.fffffff000008p-375, 0x1.ffffff1000000p-452, -0x1.0001fffffff80p-412, -0x1.fffe000000000p-497, 0x1.fffbfff800208p+37, -0x1.83fe387f7d700p-21},
};

const int numAccuracyTests = sizeof(accuracyTests) / sizeof(struct testVector);

#include <math.h>

double testAccuracy() {
	int i;
	DD a, b, c, r;

	double worstUlps = 0.5;

	for (i=0; i<numAccuracyTests; ++i) {
		a.hi = accuracyTests[i].xhi;
		a.lo = accuracyTests[i].xlo;
		b.hi = accuracyTests[i].yhi;
		b.lo = accuracyTests[i].ylo;
		r.hi = accuracyTests[i].rhi;
		r.lo = accuracyTests[i].rlo;

		DD c = { .ld = __gcc_qdiv(a.ld, b.ld) };

		double error = __builtin_fabs(((r.hi - c.hi) + r.lo) - c.lo);

		if (error != 0.0) {

			int exponent = ilogb(r.hi);
			exponent = (exponent < -1022 ? -1022 : exponent);
			double ulpError = scalbn(error, 106 - exponent);

			if (ulpError > worstUlps) {
#ifdef PRINT_ACCURACY_INFORMATION
				printf("New worst rounding error for (%a,%a) / (%a,%a):\n", a.hi, a.lo, b.hi, b.lo);
				printf("\tExpected: (%a,%a)\n", r.hi, r.lo);
				printf("\tComputed: (%a,%a)\n", c.hi, c.lo);
				printf("\tError: %f ulps\n", ulpError);
#endif
				worstUlps = ulpError;
			}
		}
	}

	return worstUlps;
}
























