/** @file feature_cached_string.h
 * Stores a map of strings and their hash values.
 * 
 * The feature allows to reduce a size of the payload being generated by HawkTracer by
 * generating a special event (Mapping Event) that contains a mapping between string value
 * and the number, so the user can use the hash value when sends the string.
 */
#ifndef HAWKTRACER_FEATURE_CACHED_STRING_H
#define HAWKTRACER_FEATURE_CACHED_STRING_H

#include <hawktracer/timeline.h>

HT_DECLS_BEGIN

/**
 * Enables a feature for the timeline.
 * 
 * @param timeline the timeline.
 * @param thread_safe wether the feature should be thread-safe (i.e. will be used across different threads) or not.
 * @return #HT_ERR_OK if enabling the feature completes successfully; otherwise, appropriate error code.
 */
HT_API HT_ErrorCode ht_feature_cached_string_enable(HT_Timeline* timeline, HT_Boolean thread_safe);

/**
 * Adds a mapping.
 * 
 * The function also pushes a mapping event to the timeline.
 * 
 * @param timeline the timeline.
 * @param label a string to map. The string must be a static string (i.e. must be valid for the whole period of the program).
 * For dynamic strings, use ht_feature_cached_string_add_mapping_dynamic()
 * @return hash of the string passed as argument.
 */
HT_API uintptr_t ht_feature_cached_string_add_mapping(HT_Timeline* timeline, const char* label);

/**
 * Pushes mapping events of all the strings registered in the feature.
 * 
 * @param timeline the timeline.
 */
HT_API void ht_feature_cached_string_push_map(HT_Timeline* timeline);

HT_API uintptr_t ht_feature_cached_string_add_mapping_dynamic(HT_Timeline* timeline, const char* label);

HT_DECLS_END


#endif /* HAWKTRACER_FEATURE_CACHED_STRING_H */
