/*
 *	internal handle
 *
 *	Copyright (C) 2009-2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <net/if_arp.h>

#include <sutils.h>
#include <logging.h>
#include <ntypes.h>
#include <nutils.h>
#include <handle.h>
#include <vlan.h>
#include <bridge.h>
#include <bonding.h>


#define NC_IFACE_ARRAY_CHUNK	16

#define DEFAULT_ADDRCONF_IPV4	(NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC))
#define DEFAULT_ADDRCONF_IPV6	(NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC) | \
				 NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF))

extern nc_handle_t *		nc__suse_system_init(const char *, nc_service_t *);
extern nc_handle_t *		nc__suse_config_init(const char *, nc_service_t *);

static nc_interface_t *		__nc_interface_new(const char *name, unsigned int index);
static void			__nc_interface_free(nc_interface_t *ifp);

struct indirect_ns_map {
	const char *		path;
	nc_handle_t *		(*init)(const char *, nc_service_t *);
};

static struct indirect_ns_map	__namespace_map[] = {
	{
		.path	= "/system",
		.init	= nc__suse_system_init,
	},
	{
		.path	= "/config",
		.init	= nc__suse_config_init,
	},
	{
		.path	= NULL,
		.init	= NULL,
	}
};

nc_handle_t *
nc_indirect_open(const char *ns_path, const char *root_dir, nc_service_t *service)
{
	struct indirect_ns_map *ns;

	if(!ns_path)
		return NULL;

	for(ns=__namespace_map; ns && ns->path && ns->init; ns++) {
		if(strcmp(ns->path, ns_path))
			continue;
		if(ns->init) {
			return ns->init(root_dir, service);
		}
	}
	return NULL;
}

void
nc_close(nc_handle_t *nh)
{
	if(nh) {
		if(nh->op && nh->op->close)
			nh->op->close(nh);

		nc_route_array_destroy(&nh->routes);
		nc_interface_array_destroy(&nh->ifaces);

		nc_string_free(&nh->err_last);
		nc_string_array_destroy(&nh->err_once);

		nh->op = NULL;
		free(nh);
	}
}

void
__nc_error_detail_fmt(const char *func, const char *file, long long line,
	                    nc_handle_t *nh, const char *fmt, ...)
{
	nc_stringbuf_t  buf = NC_STRINGBUF_INIT;
	if(nh) {
		if(fmt && *fmt) {
			va_list ap;

			va_start(ap, fmt);
			nc_stringbuf_vprintf(&buf, fmt, ap);
			va_end(ap);
		}
		if(nh->err_last)
			free(nh->err_last);
		nh->err_last = buf.string;

		if(nh->err_last) {
			__nc_error(func,file,line,"%s", nh->err_last);
		}
	}
}

void
nc_error_detail_move(nc_handle_t *nh, char **errstr)
{
	if(nh && errstr) {
		nc_string_free(errstr);

		if(nh->err_last) {
			*errstr = nh->err_last;
			nh->err_last = NULL;
		}
	}
}

void
nc_error_detail_clear(nc_handle_t *nh)
{
	if(nh) {
		nc_string_free(&nh->err_last);
	}
}

int /* bool */
nc_error_once_check(nc_handle_t *nh, const char *once_ctx, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list        ap;

	if(!nh || !once_ctx)
		return 1;

	va_start(ap, once_ctx);
	if(nc_stringbuf_vprintf(&buf, once_ctx, ap) < 0) {
		va_end(ap);
		return 1;
	}
	va_end(ap);

	if(nc_string_array_index(&nh->err_once, buf.string) == -1) {
		nc_string_array_append(&nh->err_once, buf.string);
		nc_stringbuf_destroy(&buf);
		return 1;
	} else {
		nc_stringbuf_destroy(&buf);
		return 0;
	}
}

void
nc_error_once_clear(nc_handle_t *nh, const char *once_ctx, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list        ap;

	if(!nh || !once_ctx)
		return;

	va_start(ap, once_ctx);
	if(nc_stringbuf_vprintf(&buf, once_ctx, ap) < 0) {
		va_end(ap);
		return;
	}
	va_end(ap);

	nc_string_array_remove_match(&nh->err_once, buf.string, 0);
	nc_stringbuf_destroy(&buf);
}

void
nc_bad_reference(nc_handle_t *nh, const nc_interface_t *referrer, const char *ifname)
{
	(void)nh;
	nc_warn("%s references unknown interface %s", referrer->name, ifname);
}

static nc_interface_t *
__nc_interface_new(const char *name, unsigned int index)
{
	nc_interface_t *ifp;

	ifp = calloc(1, sizeof(*ifp));
	if (!ifp)
		return NULL;

	ifp->users = 1;
	ifp->type = NC_IFTYPE_UNKNOWN;
	ifp->hwaddr.type = ARPHRD_VOID;

	ifp->ifindex = index;
	if(name && nc_string_dup(&ifp->name, name) < 0) {
		free(ifp);
		return NULL;
	}

	ifp->ipv4.family = AF_INET;
	ifp->ipv4.enabled = 1;
	ifp->ipv4.addrconf = DEFAULT_ADDRCONF_IPV4;

	ifp->ipv6.family = AF_INET6;
	ifp->ipv6.enabled = 1;
	ifp->ipv6.addrconf = DEFAULT_ADDRCONF_IPV6;

	return ifp;
}

void
nc_interface_reset(nc_interface_t *ifp)
{
	if (ifp) {
		nc_address_array_destroy(&ifp->addrs);
		nc_route_array_destroy(&ifp->routes);
		if(ifp->vlan) {
			nc_vlan_free(ifp->vlan);
			ifp->vlan = NULL;
		}
		if(ifp->bridge) {
			nc_bridge_free(ifp->bridge);
			ifp->bridge = NULL;
		}
		if(ifp->bonding) {
			nc_bonding_free(ifp->bonding);
			ifp->bonding = NULL;
		}
		nc_string_free(&ifp->config.origin);
		nc_string_free(&ifp->config.uuid);
		nc_string_free(&ifp->parent);
		nc_string_free(&ifp->qdisc);
		nc_string_free(&ifp->kind);
		ifp->seq = 0;
		ifp->modified = 0;
		ifp->deleted = 0;
		ifp->invalid = 0;
		ifp->type = NC_IFTYPE_UNKNOWN;
		ifp->ifflags = 0;
		ifp->hwaddr.len = 0;
		ifp->hwaddr.type = ARPHRD_VOID;
		ifp->mtu = 0;
		ifp->metric = 0;
		ifp->txqlen = 0;
		ifp->master = 0;
		ifp->startmode =  NC_STARTMODE_OFF;
		memset(&ifp->ipv4, 0, sizeof(ifp->ipv4));
		ifp->ipv4.family = AF_INET;
		ifp->ipv4.enabled = 1;
		ifp->ipv4.addrconf = DEFAULT_ADDRCONF_IPV4;
		memset(&ifp->ipv6, 0, sizeof(ifp->ipv6));
		ifp->ipv6.family = AF_INET6;
		ifp->ipv6.enabled = 1;
		ifp->ipv6.addrconf = DEFAULT_ADDRCONF_IPV6;
	}
}

static void
__nc_interface_free(nc_interface_t *ifp)
{
	assert(ifp != NULL && ifp->users == 0);
	if(ifp) {
		nc_address_array_destroy(&ifp->addrs);
		nc_route_array_destroy(&ifp->routes);
		if(ifp->vlan) {
			nc_vlan_free(ifp->vlan);
		}
		if(ifp->bridge) {
			nc_bridge_free(ifp->bridge);
		}
		if(ifp->bonding) {
			nc_bonding_free(ifp->bonding);
		}
		nc_string_free(&ifp->config.origin);
		nc_string_free(&ifp->config.uuid);
		nc_string_free(&ifp->parent);
		nc_string_free(&ifp->qdisc);
		nc_string_free(&ifp->kind);
		nc_string_free(&ifp->name);
		memset(ifp, 0, sizeof(*ifp));
		free(ifp);
	}
}


nc_interface_t *
nc_interface_new(const char *name, unsigned int index)
{
	nc_interface_t *ifp;
	ifp = __nc_interface_new(name, index);
	return ifp;
}

nc_interface_t *
nc_interface_clone(const nc_interface_t *ofp)
{
	nc_interface_t *ifp;

	assert(ofp != NULL);

	ifp = __nc_interface_new(ofp->name, ofp->ifindex);
	if(!ifp)
		goto failed;

#define xstrdup(str)  ((str) ? strdup(str) : NULL)
#define C(member)       ifp->member = ofp->member
#define D(member, clone_fn)     \
			do { \
				if (ofp->member) { \
					ifp->member = clone_fn(ofp->member); \
					if (!ifp->member) \
						goto failed; \
				} \
			} while (0)

	C(type);
	C(ifflags);
	/* C(iftype); // arp_type */

	/* uuid */
	C(hwaddr);
	if(nc_address_array_copy(&ifp->addrs, &ofp->addrs) < 0)
		goto failed;
	if(nc_route_array_copy(&ifp->routes, &ofp->routes) < 0)
		goto failed;

	C(mtu);
	C(metric);
	C(txqlen);
	C(master);
	D(qdisc, xstrdup);
	D(kind, xstrdup);

	C(ipv4.enabled);
	C(ipv4.forwarding);
	C(ipv4.addrconf);
	C(ipv6.enabled);
	C(ipv6.forwarding);
	C(ipv6.addrconf);

	D(parent, xstrdup);
	D(vlan, nc_vlan_clone);
	D(bridge, nc_bridge_clone);
	D(bonding, nc_bonding_clone);

	C(config.origin);
	C(config.uuid);
	C(startmode);
#undef C
#undef D
#undef xstrdup

	return ifp;

failed:
	nc_error("Failed to clone interface data for interface %s", ofp->name);
	if (ifp)
		nc_interface_free(ifp);
	return NULL;
}

nc_interface_t *
nc_interface_ref(nc_interface_t *ifp)
{
	if(!ifp)
		return NULL;
	assert(ifp->users != 0);
	if(!ifp->users)
		return NULL;
	ifp->users++;
	return ifp;
}

int
nc_interface_free(nc_interface_t *ifp)
{
	assert(ifp && ifp->users);
	if(!ifp->users) {
		nc_error("nc_interface_free: bad reference");
		return 0;
	}
	ifp->users--;
	if (ifp->users == 0) {
		__nc_interface_free(ifp);
		return 0;
	}
	return ifp->users;
}

nc_interface_t *
nc_interface_first(nc_handle_t *nh, unsigned int *pos)
{
	if(!nh || !pos)
		return NULL;
	return nc_interface_array_first(&nh->ifaces, pos);
}

nc_interface_t *
nc_interface_next(nc_handle_t *nh, unsigned int *pos)
{
	if(!nh || !pos)
		return NULL;
	return nc_interface_array_next(&nh->ifaces, pos);
}

nc_interface_t *
nc_interface_find(nc_handle_t *nh, unsigned int *pos,
			int ifindex, const char *ifname,
			int modified, int deleted)
{
	if(!nh)
		return NULL;
	return nc_interface_array_find(&nh->ifaces, pos,
			ifindex, ifname, modified, deleted);
}

nc_interface_t *
nc_interface_by_name(nc_handle_t *nh, const char *ifname)
{
	unsigned int i;

	if(!nh || !ifname)
		return NULL;

	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *ifp = nh->ifaces.data[i];
		if(!ifp || !ifp->name)
			continue;

		if (nc_string_eq(ifp->name, ifname))
			return ifp;
	}
	return NULL;
}

nc_interface_t *
nc_interface_by_ifname(nc_handle_t *nh, const char *ifname)
{
	return nc_interface_by_name(nh, ifname);
}

nc_interface_t *
nc_interface_by_ifindex(nc_handle_t *nh, unsigned int ifindex)
{
	unsigned int i;

	if(!nh)
		return NULL;

	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *ifp = nh->ifaces.data[i];
		if(!ifp || !ifp->name)
			continue;

		if (ifp->ifindex == ifindex)
			return ifp;
	}
	return NULL;
}

int
nc_interface_device_is_up(const nc_interface_t *ifp)
{
	/* device is administratively up (IFF_UP) */
	return ifp ? (ifp->ifflags & NC_IFF_DEVICE_UP) : 0;
}

int
nc_interface_link_is_up(const nc_interface_t *ifp)
{
	/* carrier detected/inherited (IFF_LOWER_UP) */
	return ifp ? (ifp->ifflags & NC_IFF_LINK_UP) : 0;
}

int
nc_interface_network_is_up(const nc_interface_t *ifp)
{
	/* network setup run finished */
	return ifp ? (ifp->ifflags & NC_IFF_NETWORK_UP) : 0;
}

int
nc_interfaces_add(nc_handle_t *nh, nc_interface_t *ifp)
{
	if(!nh || !ifp)
		return -1;

	return nc_interface_array_append(&nh->ifaces, ifp);
}

void
nc_interfaces_clear(nc_handle_t *nh)
{
	if(nh) {
		nc_interface_array_destroy(&nh->ifaces);
	}
}

int
nc_interfaces_refresh(nc_handle_t *nh)
{
	if(nh && nh->op && nh->op->interfaces_refresh) {
		return nh->op->interfaces_refresh(nh);
	}
	return -1;
}

static inline void
nc_ovs_bridge_bind(nc_interface_t *ifp, nc_handle_t *nh)
{
	unsigned int i;

	for (i = 0; i < ifp->bridge->ports.count; ++i) {
		nc_bridge_port_t *port = ifp->bridge->ports.data[i];
		const char       *ifname = port->name;
		nc_interface_t   *iface;


		if (!(iface = nc_interface_by_name(nh, ifname)))
			continue;

		if (iface->invalid)
			iface->invalid = 1;
		else
			/* in fact, it is always ovs-system */
			nc_string_dup(&iface->parent, ifp->name);
	}
}

static inline void
nc_team_bind(nc_interface_t *ifp, nc_handle_t *nh)
{
	unsigned int i;

	for (i = 0; i < ifp->bonding->slave_names.count; ++i) {
		const char *ifname = ifp->bonding->slave_names.data[i];
		nc_interface_t   *iface;

		if (!(iface = nc_interface_by_name(nh, ifname)))
			continue;

		if (iface->invalid)
			iface->invalid = 1;
		else
			nc_string_dup(&iface->parent, ifp->name);
	}
}

int
nc_interfaces_create_topology(nc_handle_t *nh, nc_var_array_t *errs)
{
	unsigned int i;

	if(!nh || !errs)
		return -1;

	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *ifp = nh->ifaces.data[i];
		if(ifp && ifp->parent) {
			nc_string_free(&ifp->parent);
		}
	}

	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *ifp = nh->ifaces.data[i];

		if(!ifp || !ifp->name || ifp->invalid)
			continue;

		if (ifp->type == NC_IFTYPE_BOND  && ifp->bonding) {
			if(nc_bonding_bind(ifp, nh, errs) < 0)
				return -1;
		} else
		if(ifp->type == NC_IFTYPE_BRIDGE && ifp->bridge) {
			if(nc_bridge_bind(ifp, nh, errs) < 0)
				return -1;
		} else
		if (ifp->type == NC_IFTYPE_VLAN && ifp->vlan) {
			if(nc_vlan_bind(ifp, nh, errs) < 0)
				return -1;
		} else
		if (ifp->type == NC_IFTYPE_TEAM) {
			if (ifp->bonding)
				nc_team_bind(ifp, nh);
		} else
		if (ifp->type == NC_IFTYPE_OVS) {
			if (ifp->bridge)
				nc_ovs_bridge_bind(ifp, nh);
		}
	}
	return 0;
}

int
nc_interfaces_configure(nc_handle_t *nh, const nc_interface_t *cfg)
{
	int ret = -1;

	if(!cfg || !cfg->name) {
		nc_error("invalid reference to interface configuration");
		return ret;
	}

	if(nh && nh->op && nh->op->interfaces_configure) {
		nc_interface_t *nfp, *ifp;
		unsigned int i;

		/*
		 * cfg should refer the outer interface we're going to configure.
		 * Let's verify it has been also added to the interface repo.
		 */
		nfp=NULL;
		for(ifp=nc_interface_first(nh, &i); !nfp && ifp; ifp=nc_interface_next(nh, &i)) {
			if(ifp == cfg) {
				nfp = ifp;
				if(!nfp->modified) {
					nc_warn("interface %s configuration "
						"not marked modified - fixed", nfp->name);
				}
				nfp->modified = 1;
			}
		}
		if(!nfp) {
			/*
			 * It is not in the handle?! Try to fix this.
			 */
			nfp = nc_interface_clone(cfg);
			if (!nfp) {
				nc_error("unable to clone interface reference %s", cfg->name);
					return -1;
			}
			nfp->modified = 1;
			nc_interface_array_append(&nh->ifaces, nfp);
		}

		ret = nh->op->interfaces_configure(nh, nfp);

		/* finally, cleanup interfaces array a bit */
		for(ifp=nc_interface_first(nh, &i); ifp; ifp=nc_interface_next(nh, &i)) {
			if(ret) {
				/* reset on failure */
				ifp->deleted  = 0;
			}
			else if(ifp->deleted) {
				/* OK, remove it */
				nc_interface_array_remove_index(&nh->ifaces, i--);
			}
		}
	} else {
		nc_error("handle not capable to configure an interface");
	}
	return ret;
}


int
nc_interfaces_delete(nc_handle_t *nh, const nc_interface_t *cfg)
{
	unsigned int i;
	int ret = -1;

	if(!cfg) {
		nc_error("invalid reference to interface configuration");
		return ret;
	}

	if(nh && nh->op && nh->op->interfaces_delete) {
		nc_interface_t *ifp;

		/* a little bit paranoia ... */
		for(i = 0; i < nh->ifaces.count; ++i) {
			nc_interface_t *ifp = nh->ifaces.data[i];
			if(ifp) {
				ifp->deleted  = 0;
				ifp->modified = 0;
			}
		}

		/* now find and mark requested interface */
		ifp = nc_interface_by_name(nh, cfg->name);
		if(ifp) {
			ifp->deleted = 1;

			ret = nh->op->interfaces_delete(nh, ifp);
		}

		/* finally, cleanup interfaces array */
		for(i = 0; i < nh->ifaces.count; ++i) {
			nc_interface_t *ifp = nh->ifaces.data[i];

			if(!ifp || !ifp->name) {
				/* some crap */
				nc_interface_array_remove_index(&nh->ifaces, i--);
				continue;
			}
			if(ret) {
				/* revert on failute */
				ifp->deleted  = 0;
				ifp->modified = 0;
			} else
			if(ifp->deleted) {
				/* OK, remove it */
				nc_interface_array_remove_index(&nh->ifaces, i--);
			}
		}
	} else {
		nc_error("handle not capable to delete interface configuration");
	}
	return ret;
}

int
nc_interface_status(nc_handle_t *nh, nc_interface_t *ifp)
{
	int ret = -1;
	if(nh && nh->op && nh->op->interface_status) {
		if(ifp && ifp->name && nc_interface_by_name(nh, ifp->name)) {
			ret = nh->op->interface_status(nh, ifp);
		      	if(ret < 0) {
				nc_error("Cannot get status of interface %s",
					ifp->name);
			}
		} else {
			nc_error("can't check status of invalid interface %s reference",
				(ifp && ifp->name ? ifp->name : NULL));
		}
	} else {
		nc_error("handle not capable to check status of an interface");
	}
	return ret;
}

int
nc_interface_start(nc_handle_t *nh, nc_interface_t *ifp)
{
	int ret = -1;
	if(nh && nh->op && nh->op->interface_start) {
		if(ifp && ifp->name && nc_interface_by_name(nh, ifp->name)) {
			ret = nh->op->interface_start(nh, ifp);
			if(ret == 0) {
				nc_info("interface %s started", ifp->name);
			} else {
				nc_error("start of interface %s failed",
					ifp->name);
			}
		} else {
			nc_error("can't start invalid interface %s reference",
				(ifp && ifp->name ? ifp->name : NULL));
		}
	} else {
		nc_error("handle not capable to start interface");
	}
	return ret;
}

int
nc_interface_stop(nc_handle_t *nh, nc_interface_t *ifp)
{
	int ret = -1;
	if(nh && nh->op && nh->op->interface_stop) {
		if(ifp && ifp->name && nc_interface_by_name(nh, ifp->name)) {
			ret = nh->op->interface_stop(nh, ifp);
			if(ret == 0) {
				nc_info("interface %s stopped", ifp->name);
			} else {
				nc_error("stop of interface %s failed",
					ifp->name);
			}
		} else {
			nc_error("can't start invalid interface %s reference",
				(ifp && ifp->name ? ifp->name : NULL));
		}
	} else {
		nc_error("handle not capable to stop interface");
	}
	return ret;
}

nc_interface_array_t *
nc_interface_array_new(void)
{
	return (nc_interface_array_t *)calloc(1, sizeof(nc_interface_array_t));
}

void
nc_interface_array_free(nc_interface_array_t *array)
{
	nc_interface_array_destroy(array);
	free(array);
}

void
nc_interface_array_init(nc_interface_array_t *array)
{
	if(array) {
		memset(array, 0, sizeof(*array));
	}
}

void
nc_interface_array_destroy(nc_interface_array_t *array)
{
	if(array) {
		while(array->count--) {
			nc_interface_free(array->data[array->count]);
		}
		free(array->data);
		nc_interface_array_init(array);
	}
}

static int
__nc_interface_array_realloc(nc_interface_array_t *array, unsigned int newsize)
{
	nc_interface_t **newdata;
	unsigned int i;

	newsize = (newsize + NC_IFACE_ARRAY_CHUNK);
	newdata = realloc(array->data, newsize * sizeof(nc_interface_t *));
	assert(newdata != NULL);
	if (!newdata)
		return -1;
	
	array->data = newdata;
	for (i = array->count; i < newsize; ++i) {
		array->data[i] = NULL;
	}
	return 0;
}

int
nc_interface_array_append(nc_interface_array_t *array, nc_interface_t *ifp)
{
	if(!array || !ifp)
		return -1;

	if((array->count % NC_IFACE_ARRAY_CHUNK) == 0) {
		if(__nc_interface_array_realloc(array, array->count) != 0)
			return -1;
	}
	array->data[array->count++] = ifp;
	return 0;
}

nc_interface_t *
nc_interface_array_first(nc_interface_array_t *array, unsigned int *pos)
{
	if(!array || !pos)
		return NULL;

	for(*pos = 0; *pos < array->count; ++*pos) {
		nc_interface_t *ifp = array->data[*pos];
		if(!ifp || !ifp->name || ifp->invalid)
			continue;
		return ifp;
	}

	return NULL;
}

nc_interface_t *
nc_interface_array_next(nc_interface_array_t *array, unsigned int *pos)
{
	if(!array || !pos)
		return NULL;

	for(++*pos; *pos < array->count; ++*pos) {
		nc_interface_t *ifp = array->data[*pos];
		if(!ifp || !ifp->name || ifp->invalid)
			continue;
		return ifp;
	}
	return NULL;
}

nc_interface_t *
nc_interface_array_find(nc_interface_array_t *array, unsigned int *pos,
		       	int ifindex, const char *ifname,
		       	int modified, int deleted)
{
	nc_interface_t *ifp = NULL;
	unsigned int idx = 0;

	if(!array)
		return NULL;

	if(pos)
		idx = *pos;

	if(idx < array->count) {
		ifp = array->data[idx];
		if(!ifp || !ifp->name || ifp->invalid)
			ifp = nc_interface_array_next(array, &idx);
	}
	for( ; ifp ; ifp = nc_interface_array_next(array, &idx)) {
		if (ifindex > -1 && ((unsigned int)ifindex != ifp->ifindex))
			continue;

		if (ifname && !nc_string_eq(ifp->name, ifname))
			continue;

		if (modified > -1 && (!(unsigned int)modified != !ifp->modified))
			continue;

		if (deleted  > -1 && (!(unsigned int)deleted  != !ifp->deleted))
			continue;

		break;
	}

	if(pos)
		*pos = idx;
	return ifp;
}

int
nc_interface_array_index_by_name(const nc_interface_array_t *array, const char *ifname)
{
	unsigned int i;

	if(!array || !ifname)
		return -1;

	for (i = 0; i < array->count; ++i) {
		nc_interface_t *ifp = array->data[i];
		if(ifp && nc_string_eq(ifp->name, ifname))
			return i;
	}
	return -1;
}

int
nc_interface_array_remove_index(nc_interface_array_t *array, unsigned int pos)
{
	if (pos >= array->count)
		return -1;

	nc_interface_free(array->data[pos]);

	/* Note: this also copies the NULL pointer following the last element */
	memmove(&array->data[pos], &array->data[pos + 1],
		(array->count - pos) * sizeof(nc_interface_t *));
	array->count--;

	/* Don't bother with shrinking the array. It's not worth the trouble */
	return 0;
}

