/*
 * Copyright (c) 2005, 2006 SUSE LINUX Products GmbH, Germany
 * Copyright (c) 2002, 2003, 2004 SuSE Linux AG, Germany.
 * Copyright (c) 1999, 2000 SuSE GmbH, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <security/pam_modules.h>
#if defined (HAVE_SECURITY_PAM_EXT_H)
#include <security/pam_ext.h>
#endif


#include "public.h"

int
c2n (char c)
{
  if (c == '.')
    return 0;
  else if (c == '/')
    return 1;
  else if (c >= '0' && c <= '9')
    return 2 + (c - '0');
  else if (c >= 'A' && c <= 'Z')
    return 12 + (c - 'A');
  else if (c >= 'a' && c <= 'z')
    return 38 + (c - 'a');
  else return -1;
}

long
str2week (char *date)
{
  if (date == NULL || strlen (date) == 0)
    return -1;

  if (strlen (date) == 1)
    return c2n (date[0]);
  else
    return c2n (date[0]) + (c2n (date[1]) * 64);
}

/* write message to user */
int
__write_message (pam_handle_t *pamh, int flags attribute_unused, int msg_style,
		 const char *fmt,...)
{
  int retval;
  va_list ap;
#if defined (HAVE_PAM_VPROMPT)
  va_start (ap, fmt);
  retval = pam_vprompt (pamh, msg_style, NULL, fmt, ap);
  va_end (ap);
#else
  struct pam_message msg[1], *pmsg[1];
  struct pam_response *resp=NULL;
  struct pam_conv *conv;
  void *conv_void;
  char buffer[512];

  va_start (ap, fmt);
  vsnprintf (buffer, sizeof (buffer), fmt, ap);
  va_end (ap);

  pmsg[0] = &msg[0];
  msg[0].msg_style = msg_style;
  msg[0].msg = buffer;

  retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv_void);
  conv = (struct pam_conv *) conv_void;
  if (retval == PAM_SUCCESS)
    {
      retval = conv->conv (1, (const struct pam_message **)pmsg,
			   &resp, conv->appdata_ptr);
      if (retval != PAM_SUCCESS)
	return retval;
    }
  else
    return retval;

  msg[0].msg = NULL;
  if (resp)
    _pam_drop_reply(resp, 1);
#endif

  return retval;
}

/* prompt user for a using conversation calls */
int
__get_authtok (pam_handle_t *pamh, int not_set_pass)
{
  int retval;
  char *cresp = NULL;
#if defined (HAVE_PAM_PROMPT)

  retval = pam_prompt (pamh, PAM_PROMPT_ECHO_OFF, &cresp, _("Password: "));

  if (retval != PAM_SUCCESS)
    {
      _pam_drop (cresp);
      if (retval == PAM_CONV_AGAIN)
        retval = PAM_INCOMPLETE;
      return retval;
    }
#else
  struct pam_message msg[1], *pmsg[1];
  struct pam_response *resp = NULL;
  struct pam_conv *conv;
  void *conv_void;

  /* set up conversation call */

  pmsg[0] = &msg[0];
  msg[0].msg_style = PAM_PROMPT_ECHO_OFF;
  msg[0].msg = _("Password: ");

  retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv_void);
  conv = (struct pam_conv *) conv_void;
  if (retval == PAM_SUCCESS)
    {
      retval = conv->conv (1, (const struct pam_message **)pmsg,
			   &resp, conv->appdata_ptr);
      if (retval != PAM_SUCCESS)
	return retval;
    }
  else
    return retval;

  if (cresp)
    cresp = strdup (resp->resp ? resp->resp : ""); /* checked for free.  */
  _pam_drop_reply (resp, 1);
#endif

  if (cresp)
    {
      if (!not_set_pass)
        pam_set_item (pamh, PAM_AUTHTOK, cresp);

      _pam_drop (cresp);

      return PAM_SUCCESS;
    }

  return PAM_CONV_ERR;
}

#if !defined(HAVE_PAM_SYSLOG)
/* syslogging function for errors and other information */
void
pam_syslog (pam_handle_t *pamh attribute_unused, int err,
	    const char *format,...)
{
  va_list args;

  va_start (args, format);
  vsyslog (err, format, args);
  va_end (args);
}
#endif
