/*
 * Copyright (c) 2005, 2006, 2008 SUSE LINUX Products GmbH Nuernberg, Germany.
 * Copyright (c) 2002, 2003, 2004 SuSE Linux AG Nuernberg, Germany.
 * Copyright (c) 2000, 2001 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#define PAM_SM_PASSWORD

#include <pwd.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <shadow.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>

#if defined (HAVE_CRACK_H)
#include <crack.h>
#else
extern char *FascistCheck (const char *pw, const char *dictpath);
#endif

#include <security/pam_modules.h>
#if defined (HAVE_SECURITY_PAM_EXT_H)
#include <security/pam_ext.h>
#endif

#include "public.h"
#include "getuser.h"


#define OLD_PASSWORD_PROMPT _("Old Password: ")
#define NEW_PASSWORD_PROMPT _("New Password: ")
#define AGAIN_PASSWORD_PROMPT _("Reenter New Password: ")
#define MISTYPED_PASSWORD _("Passwords do not match.")
#define OLD_PASSWORDS_FILE "/etc/security/opasswd"

static void
__free_cstring (pam_handle_t *pamh attribute_unused, void *data,
		int status attribute_unused)
{
  /* NB these macros handle NULL pointers correctly */
  _pam_overwrite (data);
  _pam_drop (data);
}

static void
__save_authtok (pam_handle_t *pamh, char *pass)
{
  pam_set_data (pamh, "pwcheck/authtok", pass, __free_cstring);
}

static char *
__get_saved_authtok (pam_handle_t *pamh)
{
  int retval;
  void *pass;

  retval = pam_get_data (pamh, "pwcheck/authtok", (const void **) &pass);
  if (retval != PAM_SUCCESS)
    pass = NULL;
  return (char *)pass;
}

static char *
str_lower (char *string)
{
  char *cp;

  for (cp = string; *cp; cp++)
    *cp = tolower (*cp);
  return string;
}

/* shouldn't be a palindrome - like `R A D A R' or `M A D A M' */
static int
palindrome (const char *new)
{
  int i, j;

  i = strlen (new);

  for (j = 0; j < i; j++)
    if (new[i - j - 1] != new[j])
      return 0;

  return 1;
}

/* more than half of the characters should different ones. */
static int
similar (const char *old, const char *new)
{
  int i, j;

  /*
   * XXX - sometimes this fails when changing from a simple password
   * to a really long one (MD5).  For now, I just return success if
   * the new password is long enough.  Please feel free to suggest
   * something better...  --marekm
   */
  if (strlen (new) >= 8)
    return 0;

  for (i = j = 0; new[i] && old[i]; i++)
    if (strchr (new, old[i]))
      j++;

  if (i >= j * 2)
    return 0;

  return 1;
}

/* a nice mix of characters. */
static int
simple (const char *new)
{
  int digits = 0;
  int uppers = 0;
  int lowers = 0;
  int others = 0;
  int size;
  int i;

  for (i = 0; new[i]; i++)
    {
      if (isdigit (new[i]))
	digits++;
      else if (isupper (new[i]))
	uppers++;
      else if (islower (new[i]))
	lowers++;
      else
	others++;
    }

  /*
   * The scam is this - a password of only one character type
   * must be 8 letters long.  Two types, 7, and so on.
   */

  size = 9;
  if (digits)
    size--;
  if (uppers)
    size--;
  if (lowers)
    size--;
  if (others)
    size--;

  if (size <= i)
    return 0;

  return 1;
}

static const char *
obscure_check (pam_handle_t *pamh, const char *oldpass,
	       const char *newpass, options_t *options)
{
  const char *cmiscptr = NULL;
  char *oldmono, *newmono, *wrapped;

  if (options->debug)
    pam_syslog (pamh, LOG_DEBUG, "entered obscure_check");

  if (strcmp (oldpass, newpass) == 0)
    {
      cmiscptr = _("You must choose a new password.");
      return cmiscptr;
    }

  newmono = str_lower (strdup (newpass)); /* checked for free.  */
  oldmono = str_lower (strdup (oldpass)); /* checked for free.  */
  wrapped = malloc (strlen (oldmono) * 2 + 1); /* checked for free.  */
  strcpy (wrapped, oldmono);
  strcat (wrapped, oldmono);

  if (palindrome (newmono))
    cmiscptr = _("Bad password: a palindrome");
  else if (strcmp (oldmono, newmono) == 0)
    cmiscptr = _("Bad password: case changes only");
  else if (similar (oldmono, newmono))
    cmiscptr = _("Bad password: too similar");
  else if (simple (newpass))
    cmiscptr = _("Bad password: too simple");
  else if (strstr (wrapped, newmono))
    cmiscptr = _("Bad password: rotated");

  memset (newmono, 0, strlen (newmono));
  memset (oldmono, 0, strlen (oldmono));
  memset (wrapped, 0, strlen (wrapped));
  free (newmono);
  free (oldmono);
  free (wrapped);

  return cmiscptr;
}

static int
password_check (pam_handle_t *pamh, int flags, const char *user, uid_t uid,
		const char *oldpass, char *newpass, options_t *options)
{
  const char *msg = NULL;

  if (options->debug)
    pam_syslog (pamh, LOG_DEBUG, "entered password_check");

  if (strlen (newpass) < options->minlen)
    msg = _("Bad password: too short");

  if (msg == NULL && options->maxlen != 0)
    {
      int maxlen = options->maxlen;

      if ((int)strlen (newpass) > maxlen)
	{
	  __write_message (pamh, flags, PAM_ERROR_MSG,
			   _("Password will be truncated to %d characters."),
			   maxlen);

	  newpass[maxlen] = '\0';
	}
    }

  if (msg == NULL && !options->no_obscure_checks && strlen (newpass) > 0)
    msg = obscure_check (pamh, oldpass, newpass, options);

  if (msg == NULL && options->use_cracklib)
    {
      if (msg == NULL && options->use_cracklib)
	{
	  const char *cp = FascistCheck (newpass,
				   options->cracklib_path);

	  if (options->debug)
	    pam_syslog (pamh, LOG_DEBUG, "run cracklib");

	  if (cp)
	    {
#define MSG_BAD_PASSWORD _("Bad password: ")
	      char *tmp = alloca (strlen (cp) + strlen (MSG_BAD_PASSWORD) + 1);
	      strcpy (tmp, MSG_BAD_PASSWORD);
	      strcat (tmp, cp);
	      msg = tmp;
	    }
	}
    }

  if (msg == NULL && options->remember)
    {
      if (options->debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "look in old password file");

      if (check_old_password (pamh, flags, user, newpass) != PAM_SUCCESS)
	msg = _("Password has been used already. Choose another.");
    }

  if (msg)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG, msg);

      /* This is only a warning if root changes the password, else
	 return an error. The first case (getuid()) is, if a user
	 calls passwd to change a password. The second case is, that
         a user or root is required to change the password. In this
	 case allow root to use every password the administrator wishes,
	 else show the error message. Since a password change request is
	 always done when the program runs as root, we have to check for
	 the userid of the password we try to change to find out if we
	 change the root password. */
      if (getuid() || options->enforce_for_root ||
	  ((flags & PAM_CHANGE_EXPIRED_AUTHTOK) && uid != 0))
	return 1;
      else
	return 0;
    }
  else
    return 0;
}


PAM_EXTERN int
pam_sm_chauthtok (pam_handle_t *pamh, int flags, int argc, const char **argv)
{
  user_t *data;
  char *user, *newpass;
  const char *oldpass;
  void *user_void, *oldpass_void, *newpass_void;
  int retval, trys;
  options_t options;

  memset (&options, 0, sizeof (options));

  if (get_options (pamh, &options, argc, argv) != 0)
    {
      pam_syslog (pamh, LOG_ERR, _("Cannot get options"));
      return PAM_BUF_ERR;
    }

  if (options.debug)
    pam_syslog (pamh, LOG_DEBUG, "pam_sm_chauthtok entered");

  retval = pam_get_item (pamh, PAM_USER, (const void **) &user_void);
  user = (char *) user_void;
  if (retval != PAM_SUCCESS)
    return retval;

  if (user == NULL || strlen (user) == 0)
    {
      if (options.debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "User is not known to system");

      /* The app is supposed to get us the username! */
      return PAM_USER_UNKNOWN;
    }

  if (flags & PAM_PRELIM_CHECK)
    {
      if (options.debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "pam_sm_chauthtok(PAM_PRELIM_CHECK)");

      if (options.use_cracklib && options.cracklib_path)
	{
	  /* Check for passwd dictionary */
	  struct stat st;
	  const char *dictpath = options.cracklib_path;
	  char *cp;

	  cp = alloca (strlen (dictpath) + 5);
	  strcpy (cp, dictpath);
	  strcat (cp, ".pwd");

	  if (!stat (cp, &st) && st.st_size)
	    return PAM_SUCCESS;
	  else
	    {
	      __write_message (pamh, flags, PAM_ERROR_MSG,
			       _("Dictionary path %s is invalid"), dictpath);
	      pam_syslog (pamh, LOG_ERR,
			  "dict path '%s' is invalid",
			 dictpath);
	      return PAM_ABORT;
	    }
	}

      return PAM_SUCCESS;
    }


  /* Now we have all the initial information we need from the app to
     set things up (we assume that getting the username succeeded...) */
  data = __do_getpwnam (pamh, user);
  if (data == NULL)
    {
      if (options.debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "user not found in passwd database.");

      return PAM_USER_UNKNOWN;
    }

  /* Don't make this checks if we change the root password.
     If PAM_CHANGE_EXPIRED_AUTHTOK is set, we always run as root. So
     add additional check, if we change root password. If yes, be quiet.  */
  if (((flags & PAM_CHANGE_EXPIRED_AUTHTOK) && data->pwd->pw_uid != 0) ||
      getuid () != 0 )
    {
      if (data->is_tooearly)
	{
	  if (options.debug)
	    pam_syslog (pamh, LOG_DEBUG,
			"too early to change password.");

	  free_user_t (data);
	  return PAM_AUTHTOK_ERR;
	}
      else if (data->is_expired)
	{
	  if (options.debug)
	    pam_syslog (pamh, LOG_DEBUG,
			"account is already expired.");

	  free_user_t (data);
	  return PAM_ACCT_EXPIRED; /* If their account has expired, we
				      can't auth them to change their
				      password */
	}
    }

  if ((data->use_shadow || data->use_hp_aging) && !data->is_expiring &&
      (flags & PAM_CHANGE_EXPIRED_AUTHTOK))
    {
      if (options.debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "we can only change expiring accounts.");

      free_user_t (data);
      return PAM_SUCCESS;
    }

  retval = pam_get_item (pamh, PAM_OLDAUTHTOK, (const void **) &oldpass_void);
  oldpass = (char *) oldpass_void;
  if (retval != PAM_SUCCESS)
    {
      if (options.debug)
	pam_syslog (pamh, LOG_DEBUG,
		    "cannot get old auth token.");
      return retval;
    }
  else if (options.debug)
    {
      if (oldpass)
	pam_syslog (pamh, LOG_DEBUG, "get old auth token");
      else
	pam_syslog (pamh, LOG_DEBUG, "old auth token not set");
    }

  if (!oldpass)
    oldpass = ""; /* We are root, we don't know the users password.  */

  /* If we haven't been given a password yet, prompt for one... */
  trys = 0;
  retval = pam_get_item (pamh, PAM_AUTHTOK, (const void **) &newpass_void);
  newpass = (char *) newpass_void;
  if (retval != PAM_SUCCESS)
    return retval;
  if (options.debug)
    {
      if (newpass)
	pam_syslog (pamh, LOG_DEBUG, "got new auth token");
      else
	pam_syslog (pamh, LOG_DEBUG, "new auth token not set");
    }

  if (options.use_authtok || newpass)
    {
      if (newpass == NULL) /* We are not allowed to ask for a new password */
	{
	  free_user_t (data);
	  return PAM_AUTHTOK_ERR;
	}

      if (password_check (pamh, flags, user, data->pwd->pw_uid,
			  oldpass, newpass, &options))
	{
          /* We are only here, because old password was set.
             So overwrite it, else it will be stored! */
          pam_set_item (pamh, PAM_AUTHTOK, (void *) NULL);
          __save_authtok (pamh, NULL);
          free_user_t (data);
          return PAM_AUTHTOK_ERR;
	}
    }
  else
    {
      while ((newpass == NULL) && (trys++ < options.tries))
	{
	  int failed = 0;

	  if ((newpass = __get_saved_authtok (pamh)) != NULL)
	    goto checkit;

	  retval = __get_passwd (pamh, NEW_PASSWORD_PROMPT, &newpass);
	  if (retval != PAM_SUCCESS)
	    {
	      free_user_t (data);
	      return retval;
	    }

	  if (newpass == NULL)
	    {
	      /* We want to abort the password change */
	      __write_message (pamh, flags, PAM_ERROR_MSG,
			       _("Password change aborted."));
	      free_user_t (data);
	      return PAM_AUTHTOK_ERR;
	    }

	  /* Remember new password - the next __get_passwd call
	   * might return PAM_INCOMPLETE.
	   * We save it here so we don't get confused about when
	   * to free newpass, and when to clear the saved data. */
	  __save_authtok (pamh, newpass);

checkit:
	  failed = password_check (pamh, flags, user, data->pwd->pw_uid,
				   oldpass, newpass, &options);

	  if (failed)
	    {
	      if (newpass != NULL)
		{
		  __save_authtok (pamh, NULL);
		  newpass = NULL;
		}
	      if (trys >= options.tries)
		{
		  free_user_t (data);
		  return PAM_MAXTRIES;
		}
	      failed = 0;
	    }
	  else
	    {
	      char *new2;

	      retval = __get_passwd (pamh, AGAIN_PASSWORD_PROMPT, &new2);
	      if (retval != PAM_SUCCESS)
		{
		  free_user_t (data);
		  return retval;
		}

	      if (new2 == NULL)
		{			/* Aborting password change... */
		  __write_message (pamh, flags, PAM_ERROR_MSG,
				   _("Password change aborted."));
		  free_user_t (data);
		  return PAM_AUTHTOK_ERR;
		}

	      failed = (strcmp (newpass, new2) != 0);

	      _pam_overwrite (new2);
	      _pam_drop (new2);

	      if (failed)
	        {
		  __write_message (pamh, flags, PAM_ERROR_MSG, MISTYPED_PASSWORD);
		  __save_authtok (pamh, NULL);
		  newpass = NULL;
		}
	    }
	}
      /* end while (newpass == NULL) */

      if (newpass == NULL)
	{
	  free_user_t (data);
	  if (options.debug)
	    pam_syslog (pamh, LOG_DEBUG,
			"newpass == NULL, return PAM_MAXTRIES");
	  return PAM_MAXTRIES;   /* They didn't seem to enter the right
				    password for three tries - error */
	}
    }

  pam_set_item (pamh, PAM_AUTHTOK, (void *) newpass);

  save_old_password (pamh, flags, user, newpass, options.remember);

  free_user_t (data);

  return PAM_SUCCESS;
}

#ifdef PAM_STATIC
/* static module data */
struct pam_module _pam_pwcheck_modstruct = {
  "pam_pwcheck",
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  pam_sm_chauthtok
};
#endif
