/*
 * Copyright (c) 2005, 2006, 2011 SUSE LINUX Products GmbH Nuernberg, Germany.
 * Copyright (c) 2003, 2004 SuSE Linux AG Nuernberg, Germany.
 * Copyright (c) 2002 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <pwd.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#if defined (HAVE_CRYPT_H)
#include <crypt.h>
#endif

#include <security/pam_modules.h>

#include "public.h"

#ifndef RANDOM_DEVICE
#define RANDOM_DEVICE "/dev/urandom"
#endif

#define OLD_PASSWORDS_FILE "/etc/security/opasswd"
#define TMP_PASSWORDS_FILE OLD_PASSWORDS_FILE".tmpXXXXXX"

typedef struct {
  char *user;
  char *uid;
  int count;
  char *old_passwords;
} opwd;

static int
read_loop (int fd, char *buffer, int count)
{
  int offset, block;

  offset = 0;
  while (count > 0)
    {
      block = read (fd, &buffer[offset], count);

      if (block < 0)
        {
          if (errno == EINTR)
            continue;
          return block;
        }
      if (!block)
        return offset;

      offset += block;
      count -= block;
    }

  return offset;
}

static char *
make_crypt_salt (pam_handle_t *pamh, int flags)
{
#define CRYPT_GENSALT_OUTPUT_SIZE (7 + 22 + 1)
  int fd;
  char entropy[16];
  char *retval;
  char output[CRYPT_GENSALT_OUTPUT_SIZE];

  fd = open (RANDOM_DEVICE, O_RDONLY);
  if (fd < 0)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG,
                       _("Cannot open %s for reading: %s"),
                       RANDOM_DEVICE, strerror (errno));
      return NULL;
    }

  if (read_loop (fd, entropy, sizeof(entropy)) != sizeof(entropy))
    {
      close (fd);
      __write_message (pamh, flags, PAM_ERROR_MSG,
                       _("Unable to obtain entropy from %s"),
                       RANDOM_DEVICE);
      return NULL;
    }

  close (fd);

  retval = crypt_gensalt_r ("$1$", 0, entropy, sizeof (entropy),
			     output, sizeof(output));

  memset (entropy, 0, sizeof (entropy));

  if (!retval)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG,
                       _("Unable to generate a salt. "
                         "Check your crypt settings."));
      return NULL;
    }

  return strdup (retval);
}

static char *
create_hash (const char *oldpass, pam_handle_t *pamh, int flags)
{
  char *salt;

  salt = make_crypt_salt (pamh, flags);
  if (salt != NULL)
    {
      struct crypt_data output;
      char *newpassword;

      memset (&output, 0, sizeof (output));

      newpassword = crypt_r (oldpass, salt, &output);
      free (salt);

      return strdup (newpassword);
    }

  return NULL;
}

static int
parse_entry (char *line, opwd *data)
{
  const char delimiters[] = ":";
  char *endptr;

  data->user = strsep (&line, delimiters);
  data->uid = strsep (&line, delimiters);
  data->count = strtol (strsep (&line, delimiters), &endptr, 10);
  if (endptr != NULL && *endptr != '\0')
    {
      fprintf (stderr, "endptr = '%c'\n", *endptr);
      return 1;
    }
  data->old_passwords = strsep (&line, delimiters);

  return 0;
}

/* Check, if the new password is already in the opasswd file.  */
int
check_old_password (pam_handle_t *pamh attribute_unused, int flags attribute_unused,
		    const char *user, const char *newpass)
{
  int retval = PAM_SUCCESS;
  FILE *oldpf;
  char *buf = NULL;
  size_t buflen = 0;
  opwd entry;
  int found = 0;

  if ((oldpf = fopen (OLD_PASSWORDS_FILE, "r")) == NULL)
    {
#if 0
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       "Cannot open %s: %m", OLD_PASSWORDS_FILE);
#endif
      return PAM_SUCCESS;
    }

  while (!feof (oldpf))
    {
      char *cp, *tmp;
#if defined(HAVE_GETLINE)
      ssize_t n = getline (&buf, &buflen, oldpf);
#elif defined (HAVE_GETDELIM)
      ssize_t n = getdelim (&buf, &buflen, '\n', oldpf);
#else
      ssize_t n;

      if (buf == NULL)
        {
          buflen = 8096;
          buf = malloc (buflen);
	  if (buf == NULL)
	    return PAM_BUF_ERR;
        }
      buf[0] = '\0';
      fgets (buf, buflen - 1, oldpf);
      if (buf != NULL)
        n = strlen (buf);
      else
        n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */
      cp = buf;

      if (n < 1)
        break;

      tmp = strchr (cp, '#');  /* remove comments */
      if (tmp)
        *tmp = '\0';
      while (isspace ((int)*cp))    /* remove spaces and tabs */
        ++cp;
      if (*cp == '\0')        /* ignore empty lines */
        continue;

      if (cp[strlen (cp) - 1] == '\n')
        cp[strlen (cp) - 1] = '\0';

      if (strncasecmp (cp, user, strlen (user)) == 0 &&
          cp[strlen (user)] == ':')
        {
          /* We found the line we needed */
	  if (parse_entry (cp, &entry) == 0)
	    {
	      found = 1;
	      break;
	    }
	}
    }

  fclose (oldpf);

  if (found)
    {
      const char delimiters[] = ",";
      struct crypt_data output;
      char *running;
      char *oldpass;

      memset (&output, 0, sizeof (output));

      running = strdupa (entry.old_passwords);
      if (running == NULL)
	return PAM_BUF_ERR;

      do {
	oldpass = strsep (&running, delimiters);
	if (oldpass && strlen (oldpass) > 0 &&
	    strcmp (crypt_r (newpass, oldpass, &output), oldpass) == 0)
	  {
	    retval = PAM_AUTHTOK_ERR;
	    break;
	  }
      } while (oldpass != NULL);
    }

  if (buf)
    free (buf);

  return retval;
}

int
save_old_password (pam_handle_t *pamh, int flags,
		   const char *user, const char *oldpass, int howmany)
{
  char opasswd_tmp[] = TMP_PASSWORDS_FILE;
  struct stat opasswd_stat;
  FILE *oldpf, *newpf;
  int newpf_fd;
  int do_create = 0;
  int retval = PAM_SUCCESS;
  char *buf = NULL;
  size_t buflen = 0;
  int found = 0;
  char *md5_pass;

  if (howmany <= 0)
    return PAM_SUCCESS;

  if (oldpass == NULL)
    return PAM_SUCCESS;

  md5_pass = create_hash (oldpass, pamh, flags);
  if (md5_pass == NULL)
    return PAM_AUTHTOK_ERR;
  else
    {
      char *t = strdupa (md5_pass);
      free (md5_pass);
      md5_pass = t;
    }

  if ((oldpf = fopen (OLD_PASSWORDS_FILE, "r")) == NULL)
    {
      if (errno == ENOENT)
	{
	  __write_message (pamh, flags, PAM_ERROR_MSG,
			   _("Creating %s"), OLD_PASSWORDS_FILE);
	  do_create = 1;
	}
      else
	{
	// translator: filename, ERRNO
	  __write_message (pamh, flags, PAM_ERROR_MSG,
			   _("Cannot open %s: %m"), OLD_PASSWORDS_FILE);
	  return PAM_AUTHTOK_ERR;
	}
    }
  else if (fstat (fileno (oldpf), &opasswd_stat) < 0)
    {
	// translator: filename, ERRNO
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       _("Cannot stat %s: %m"), OLD_PASSWORDS_FILE);
      fclose (oldpf);
      return PAM_AUTHTOK_ERR;
    }

  /* Open a temp passwd file */
  newpf_fd = mkstemp (opasswd_tmp);
  if (newpf_fd == -1)
    {
	// translator: filename, ERRNO
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       _("Cannot create %s temp file: %m"), OLD_PASSWORDS_FILE);
      fclose (oldpf);
      return PAM_AUTHTOK_ERR;
    }
  if (do_create)
    {
      if (fchmod (newpf_fd, S_IRUSR|S_IWUSR) != 0)
	__write_message (pamh, flags, PAM_ERROR_MSG,
			 _("Cannot set permissions of %s temp file: %m"),
			 OLD_PASSWORDS_FILE);
      if (fchown (newpf_fd, 0, 0) != 0)
	__write_message (pamh, flags, PAM_ERROR_MSG,
			 _("Cannot set owner/group of %s temp file: %m"),
			 OLD_PASSWORDS_FILE);
    }
  else
    {
      if (fchmod (newpf_fd, opasswd_stat.st_mode) != 0)
	__write_message (pamh, flags, PAM_ERROR_MSG,
			 _("Cannot set permissions of %s temp file: %m"),
			 OLD_PASSWORDS_FILE);
      if (fchown (newpf_fd, opasswd_stat.st_uid, opasswd_stat.st_gid) != 0)
	__write_message (pamh, flags, PAM_ERROR_MSG,
			 _("Cannot set owner/group of %s temp file: %m"),
			 OLD_PASSWORDS_FILE);
    }
  newpf = fdopen (newpf_fd, "w+");
  if (newpf == NULL)
    {
	// translator: filename, ERRNO
      __write_message (pamh, flags, PAM_ERROR_MSG,
		       _("Cannot open %s: %m"), opasswd_tmp);
      fclose (oldpf);
      close (newpf_fd);
      retval = PAM_AUTHTOK_ERR;
      goto error_opasswd;
    }

  if (!do_create)
    while (!feof (oldpf))
      {
	char *cp, *tmp, *save;
#if defined(HAVE_GETLINE)
	ssize_t n = getline (&buf, &buflen, oldpf);
#elif defined (HAVE_GETDELIM)
	ssize_t n = getdelim (&buf, &buflen, '\n', oldpf);
#else
	ssize_t n;

	if (buf == NULL)
	  {
	    buflen = 8096;
	    buf = malloc (buflen);
	    if (buf == NULL)
	      return PAM_BUF_ERR;

	  }
	buf[0] = '\0';
	fgets (buf, buflen - 1, oldpf);
	if (buf != NULL)
	  n = strlen (buf);
	else
	  n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */

	cp = buf;
	save = strdup (buf); /* Copy to write the original data back.  */
	if (save == NULL)
	  return PAM_BUF_ERR;

	if (n < 1)
	  break;

	tmp = strchr (cp, '#');  /* remove comments */
	if (tmp)
	  *tmp = '\0';
	while (isspace ((int)*cp))    /* remove spaces and tabs */
	  ++cp;
	if (*cp == '\0')        /* ignore empty lines */
	  goto write_data;

	if (cp[strlen (cp) - 1] == '\n')
	  cp[strlen (cp) - 1] = '\0';

	if (strncasecmp (cp, user, strlen (user)) == 0 &&
	    cp[strlen (user)] == ':')
	  {
	    /* We found the line we needed */
	    opwd entry;

	    if (parse_entry (cp, &entry) == 0)
	      {
		char *out;
		/* increase count.  */
		entry.count++;

		/* check that we don't remember to many passwords.  */
		while (entry.count > howmany)
		  {
		    char *p = strpbrk (entry.old_passwords, ",");
		    if (p != NULL)
		      entry.old_passwords = ++p;
		    entry.count--;
		  }

		if (entry.old_passwords == NULL)
		  asprintf (&out, "%s:%s:%d:%s\n",
			    entry.user, entry.uid, entry.count,
			    md5_pass);
		else
		  asprintf (&out, "%s:%s:%d:%s,%s\n",
			    entry.user, entry.uid, entry.count,
			    entry.old_passwords, md5_pass);

		if (fputs (out, newpf) < 0)
		  {
		    free (out);
		    retval = PAM_AUTHTOK_ERR;
		    fclose (oldpf);
		    fclose (newpf);
		    goto error_opasswd;
		  }
		free (out);
		found = 1;
	      }
	  }
	else
	  {
	  write_data:
	    if (fputs (save, newpf) < 0)
	      {
		retval = PAM_AUTHTOK_ERR;
		fclose (oldpf);
		fclose (newpf);
		goto error_opasswd;
	      }
	  }
	free (save);
      }

  if (!found)
    {
      int pw_buflen = 256;
      char *pw_buffer = alloca (pw_buflen);
      struct passwd pw_resultbuf;
      struct passwd *pw = NULL;

      /* Get password file entry... */
      while (getpwnam_r (user, &pw_resultbuf, pw_buffer, pw_buflen, &pw) != 0
	     && errno == ERANGE)
	{
	  errno = 0;
	  pw_buflen += 256;
	  pw_buffer = alloca (pw_buflen);
	}

      if (pw == NULL)
	{
	  retval = PAM_AUTHTOK_ERR;
	  if (oldpf)
	    fclose (oldpf);
	  fclose (newpf);
	  goto error_opasswd;
	}
      else
	{
	  char *out;

	  asprintf (&out, "%s:%d:1:%s\n", user, pw->pw_uid, md5_pass);
	  if (fputs (out, newpf) < 0)
	    {
	      free (out);
	      retval = PAM_AUTHTOK_ERR;
	      if (oldpf)
		fclose (oldpf);
	      fclose (newpf);
	      goto error_opasswd;
	    }
	  free (out);
	}
    }

  if (oldpf)
    if (fclose (oldpf) != 0)
      {
	__write_message (pamh, flags, PAM_ERROR_MSG,
			 _("Error while closing old opasswd file: %s"),
			 strerror (errno));
	retval = PAM_AUTHTOK_ERR;
	fclose (newpf);
	goto error_opasswd;
      }

  if (fclose (newpf) != 0)
    {
      __write_message (pamh, flags, PAM_ERROR_MSG,
                       _("Error while closing temporary opasswd file: %s"),
		       strerror (errno));
      retval = PAM_AUTHTOK_ERR;
      goto error_opasswd;
    }

  unlink (OLD_PASSWORDS_FILE".old");
  if (link (OLD_PASSWORDS_FILE, OLD_PASSWORDS_FILE".old") != 0 &&
      errno != ENOENT)
    __write_message (pamh, flags, PAM_ERROR_MSG,
		     _("Cannot create backup file of %s: %m"),
		     OLD_PASSWORDS_FILE);
  rename (opasswd_tmp, OLD_PASSWORDS_FILE);
 error_opasswd:
  unlink (opasswd_tmp);

  return retval;
}
