/*
 * Copyright (c) 2005 SUSE LINUX Products GmbH, Nuernberg, Germany
 * Copyright (c) 2000, 2001, 2002 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <pwd.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <alloca.h>
#include <sys/stat.h>
#define PAM_SM_AUTH
#include <security/pam_modules.h>
#ifdef HAVE_SECURITY_PAM_EXT_H
#include <security/pam_ext.h>
#else
#include "pam_ext.h"
#endif

#include "public.h"

const char *dotfiles[] = {".profile", ".bashrc", ".bash_logout",
			  ".bash_login", ".bash_history", ".cshrc", ".zshrc",
			  ".xinitrc", NULL};

int
pam_sm_authenticate (pam_handle_t *pamh, int flags, int argc,
		     const char **argv)
{
  int retval;
  int pw_buflen = 256;
  char *pw_buffer = alloca (pw_buflen);
  struct passwd pw_resultbuf;
  struct passwd *pw = NULL;
  const char *name = NULL;
  struct stat stbuf;
  int i;
  /* Flags for options */
  int abort_on_writeable = 0;
  int debug_flag = 0;

  /* Parse parameters for module */
  for ( ; argc-- > 0; argv++)
    {
      if (strcasecmp (*argv, "abort") == 0)
	abort_on_writeable = 1;
      else if (strcasecmp (*argv, "debug") == 0)
        debug_flag = 1;
      else
	pam_syslog (pamh, LOG_ERR, "auth: Unknown option: %s", *argv);
    }

  /* get the user name */
  if ((retval = pam_get_user (pamh, &name, NULL)) != PAM_SUCCESS)
    return retval;

  if (name == NULL || *name == '\0' || !isalnum(*name))
    {
      if (name)
        pam_syslog (pamh, LOG_ERR, "bad username [%s]", name);
      return PAM_USER_UNKNOWN;
    }

  while (getpwnam_r (name, &pw_resultbuf, pw_buffer, pw_buflen, &pw) != 0
         && errno == ERANGE)
    {
      errno = 0;
      pw_buflen += 256;
      pw_buffer = alloca (pw_buflen);
    }

  if (pw == NULL)
    {
      if (debug_flag)
	pam_syslog (pamh, LOG_DEBUG, "user [%s] not found", name);

      return PAM_USER_UNKNOWN;
    }

  if (stat (pw->pw_dir, &stbuf) < 0)
    {
      int save_err = errno;
      pam_syslog (pamh, LOG_ERR, "Error on stat homedirectory (%s): %s",
		  pw->pw_dir, strerror (save_err));
      pam_error (pamh, _("Error on stat homedirectory: %s"),
		 strerror (save_err));
      return PAM_PERM_DENIED;
    }
  if (stbuf.st_mode & S_IWOTH)
    {

      pam_syslog (pamh, LOG_WARNING, "Directory %s is world-writeable",
		  pw->pw_dir);
      pam_error (pamh, _("Directory \"%s\" is world-writeable!"), pw->pw_dir);
      if (abort_on_writeable)
	return PAM_PERM_DENIED;
    }
  if (stbuf.st_uid != pw->pw_uid && stbuf.st_uid != 0)
    {
      pam_syslog (pamh, LOG_WARNING, "Directory %s is not owned by %s",
		  pw->pw_dir, pw->pw_name);
      pam_error (pamh, _("Directory \"%s\" is not owned by \"%s\"!"),
		 pw->pw_dir, pw->pw_name);
      if (abort_on_writeable)
	return PAM_PERM_DENIED;
    }

  i = 0;
  while (dotfiles[i] != NULL)
    {
      char *file = alloca (strlen (pw->pw_dir) + strlen (dotfiles[i]) + 2);

      strcpy (file, pw->pw_dir);
      strcat (file, "/");
      strcat (file, dotfiles[i]);
      if (stat (file, &stbuf) >= 0)
	{
	  if (stbuf.st_mode & S_IWOTH)
	    {
	      pam_syslog (pamh, LOG_WARNING,
			  "File %s is world-writeable", file);
	      pam_error (pamh, _("File \"%s\" is world-writeable!"), file);
	      if (abort_on_writeable)
		return PAM_PERM_DENIED;
	    }
	  if (stbuf.st_uid != pw->pw_uid && stbuf.st_uid != 0)
	    {
	      pam_syslog (pamh, LOG_WARNING,
			  "File %s is not owned by %s",
			  file, pw->pw_name);
	      pam_error (pamh, _("File \"%s\" is not owned by \"%s\"!"),
			 file, pw->pw_name);
	      if (abort_on_writeable)
		return PAM_PERM_DENIED;
	    }
	}
      ++i;
    }

  return PAM_SUCCESS;
}

int
pam_sm_setcred (pam_handle_t *pamh, int flags,
		int argc __attribute__ ((unused)),
		const char **argv __attribute__ ((unused)))
{
  return PAM_SUCCESS;
}

int
pam_sm_open_session (pam_handle_t *pamh, int flags, int argc,
                     const char **argv)
{
  return pam_sm_authenticate (pamh, flags, argc, argv);
}

int
pam_sm_close_session (pam_handle_t * pamh, int flags,
                      int argc, const char **argv)
{
  return PAM_SUCCESS;
}

/* static module data */
#ifdef PAM_STATIC
struct pam_module _pam_homecheck_modstruct =
{
  "pam_homecheck",
  pam_sm_authenticate,
  pam_sm_setcred,
  NULL /* pam_sm_acct_mgmt */,
  pam_sm_open_session,
  pam_sm_close_session,
  NULL /* pam_sm_chauthtok */
};
#endif
