/* packet-rtp-midi.c
 *
 * Routines for RFC 4695/6295 RTP-MIDI dissection
 * Copyright 2006-2012, Tobias Erichsen <t.erichsen@gmx.de>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>'
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * RTP-MIDI ( RFC 4695/6295 ) is a payload type for transmitting MIDI-data over
 * RTP-packets.  This payload type can be used on an RTP-conversation that has been
 * established either by means of a session establishment-protocol like SIP or
 * something more lightweight like the Apple network-midi session establishment.
 *
 * This dissector is built based on the RTP-MIDI-specification RFC 4695 from
 * August 13, 2008, the superseding version RFC 6295 and the MIDI-specification 96.1
 *
 * Here are some links:
 *
 * http://www.cs.berkeley.edu/~lazzaro/rtpmidi/
 * https://tools.ietf.org/html/rfc4695
 * https://tools.ietf.org/html/rfc6295
 * http://www.midi.org/
 *
 * 2012-02-24
 * - Initial version of dissector
 *
 * Open topics:
 * - reassembly of large sysex-fields going over multiple RTP-packets - this would be a major task, though!
 * - MIDI Show-Control
 * - MIDI Machine-Control
 *
 */

#include "config.h"

#include <epan/packet.h>
#include <epan/exceptions.h>
#include <epan/prefs.h>

void proto_register_rtp_midi(void);

/* Definitions for protocol name during dissector-register */
#define RTP_MIDI_DISSECTOR_NAME						"RFC 4695/6295 RTP-MIDI"
#define RTP_MIDI_DISSECTOR_SHORTNAME					"RTP-MIDI"
#define RTP_MIDI_DISSECTOR_ABBREVIATION					"rtpmidi"

#define RTP_MIDI_NO_RUNNING_STATUS					0xffff

/*
 * MIDI Controller Numbers
 */
#define RTP_MIDI_CTRL_BANK_SELECT_MSB					0
#define RTP_MIDI_CTRL_MODULATION_WHEEL_OR_LEVER_MSB			1
#define RTP_MIDI_CTRL_BREATH_CONTROLLER_MSB				2
/* #define RTP_MIDI_CTRL						3 */
#define RTP_MIDI_CTRL_FOOT_CONTROLLER_MSB				4
#define RTP_MIDI_CTRL_PORTAMENTO_TIME_MSB				5
#define RTP_MIDI_CTRL_DATA_ENTRY_MSB					6
#define RTP_MIDI_CTRL_CHANNEL_VOLUME_MSB				7
#define RTP_MIDI_CTRL_BALANCE_MSB					8
/* #define RTP_MIDI_CTRL						9 */
#define RTP_MIDI_CTRL_PAN_MSB						10
#define RTP_MIDI_CTRL_EXPRESSION_CONTROLLER_MSB				11
#define RTP_MIDI_CTRL_EFFECT_CONTROL_1_MSB				12
#define RTP_MIDI_CTRL_EFFECT_CONTROL_2_MSB				13
/* #define RTP_MIDI_CTRL_						14
#define RTP_MIDI_CTRL_							15 */
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_1_MSB			16
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_2_MSB			17
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_3_MSB			18
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_4_MSB			19
/* #define RTP_MIDI_CTRL_						20
#define RTP_MIDI_CTRL_							21
#define RTP_MIDI_CTRL_							22
#define RTP_MIDI_CTRL_							23
#define RTP_MIDI_CTRL_							24
#define RTP_MIDI_CTRL_							25
#define RTP_MIDI_CTRL_							26
#define RTP_MIDI_CTRL_							27
#define RTP_MIDI_CTRL_							28
#define RTP_MIDI_CTRL_							29
#define RTP_MIDI_CTRL_							30
#define RTP_MIDI_CTRL_							31 */
#define RTP_MIDI_CTRL_BANK_SELECT_LSB					32
#define RTP_MIDI_CTRL_MODULATION_WHEEL_OR_LEVER_LSB			33
#define RTP_MIDI_CTRL_BREATH_CONTROLLER_LSB				34
/* #define RTP_MIDI_CTRL_						35 */
#define RTP_MIDI_CTRL_FOOT_CONTROLLER_LSB				36
#define RTP_MIDI_CTRL_PORTAMENTO_TIME_LSB				37
#define RTP_MIDI_CTRL_DATA_ENTRY_LSB					38
#define RTP_MIDI_CTRL_CHANNEL_VOLUME_LSB				39
#define RTP_MIDI_CTRL_BALANCE_LSB					40
/* #define RTP_MIDI_CTRL_						41 */
#define RTP_MIDI_CTRL_PAN_LSB						42
#define RTP_MIDI_CTRL_EXPRESSION_CONTROLLER_LSB				43
#define RTP_MIDI_CTRL_EFFECT_CONTROL_1_LSB				44
#define RTP_MIDI_CTRL_EFFECT_CONTROL_2_LSB				45
/* #define RTP_MIDI_CTRL_						46
#define RTP_MIDI_CTRL_							47 */
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_1_LSB			48
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_2_LSB			49
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_3_LSB			50
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_4_LSB			51
/* #define RTP_MIDI_CTRL_						52
#define RTP_MIDI_CTRL_							53
#define RTP_MIDI_CTRL_							54
#define RTP_MIDI_CTRL_							55
#define RTP_MIDI_CTRL_							56
#define RTP_MIDI_CTRL_							57
#define RTP_MIDI_CTRL_							58
#define RTP_MIDI_CTRL_							59
#define RTP_MIDI_CTRL_							60
#define RTP_MIDI_CTRL_							61
#define RTP_MIDI_CTRL_							62
#define RTP_MIDI_CTRL_							63 */
#define RTP_MIDI_CTRL_DAMPER_PEDAL					64
#define RTP_MIDI_CTRL_PORTAMENTO_ON_OFF					65
#define RTP_MIDI_CTRL_SUSTENUTO						66
#define RTP_MIDI_CTRL_SOFT_PEDAL					67
#define RTP_MIDI_CTRL_LEGATO_FOOTSWITCH					68
#define RTP_MIDI_CTRL_HOLD_2						69
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_1				70
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_2				71
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_3				72
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_4				73
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_5				74
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_6				75
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_7				76
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_8				77
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_9				78
#define RTP_MIDI_CTRL_SOUND_CONTROLLER_10				79
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_5			80
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_6			81
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_7			82
#define RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_8			83
#define RTP_MIDI_CTRL_PORTAMENTO_CONTROL				84
/* #define RTP_MIDI_CTRL_						85
#define RTP_MIDI_CTRL_							86
#define RTP_MIDI_CTRL_							87
#define RTP_MIDI_CTRL_							88
#define RTP_MIDI_CTRL_							89
#define RTP_MIDI_CTRL_							90 */
#define RTP_MIDI_CTRL_EFFECTS_1_DEPTH					91
#define RTP_MIDI_CTRL_EFFECTS_2_DEPTH					92
#define RTP_MIDI_CTRL_EFFECTS_3_DEPTH					93
#define RTP_MIDI_CTRL_EFFECTS_4_DEPTH					94
#define RTP_MIDI_CTRL_EFFECTS_5_DEPTH					95
#define RTP_MIDI_CTRL_DATA_INCREMENT					96
#define RTP_MIDI_CTRL_DATA_DECREMENT					97
#define RTP_MIDI_CTRL_NON_REGISTERED_PARAM_LSB				98
#define RTP_MIDI_CTRL_NON_REGISTERED_PARAM_MSB				99
#define RTP_MIDI_CTRL_REGISTERED_PARAM_LSB				100
#define RTP_MIDI_CTRL_REGISTERED_PARAM_MSB				101
/* #define RTP_MIDI_CTRL_						102
#define RTP_MIDI_CTRL_							103
#define RTP_MIDI_CTRL_							104
#define RTP_MIDI_CTRL_							105
#define RTP_MIDI_CTRL_							106
#define RTP_MIDI_CTRL_							107
#define RTP_MIDI_CTRL_							108
#define RTP_MIDI_CTRL_							109
#define RTP_MIDI_CTRL_							110
#define RTP_MIDI_CTRL_							111
#define RTP_MIDI_CTRL_							112
#define RTP_MIDI_CTRL_							113
#define RTP_MIDI_CTRL_							114
#define RTP_MIDI_CTRL_							115
#define RTP_MIDI_CTRL_							116
#define RTP_MIDI_CTRL_							117
#define RTP_MIDI_CTRL_							118
#define RTP_MIDI_CTRL_							119 */
/* MIDI Channel Mode Messages */
#define RTP_MIDI_CTRL_ALL_SOUND_OFF					120
#define RTP_MIDI_CTRL_RESET_ALL_CONTROLLERS				121
#define RTP_MIDI_CTRL_LOCAL_CONTROL_ON_OFF				122
#define RTP_MIDI_CTRL_ALL_NOTES_OFF					123
#define RTP_MIDI_CTRL_OMNI_MODE_OFF					124
#define RTP_MIDI_CTRL_OMNI_MODE_ON					125
#define RTP_MIDI_CTRL_MONO_MODE_ON					126
#define RTP_MIDI_CTRL_POLY_MODE_ON					127


/*
 * MIDI Status Bytes (Channel Voice Messages)
 */
#define RTP_MIDI_STATUS_CHANNEL_NOTE_OFF				0x08		/* 0x8n		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_NOTE_ON					0x09		/* 0x9n		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_POLYPHONIC_KEY_PRESSURE			0x0a		/* 0xan		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_CONTROL_CHANGE				0x0b		/* 0xbn		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_PROGRAM_CHANGE				0x0c		/* 0xcn		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_CHANNEL_PRESSURE			0x0d		/* 0xdn		n->channel */
#define RTP_MIDI_STATUS_CHANNEL_PITCH_BEND_CHANGE			0x0e		/* 0xen		n->channel */

/*
 * MIDI-Channels
 */
#define RTP_MIDI_CHANNEL_1						0x00
#define RTP_MIDI_CHANNEL_2						0x01
#define RTP_MIDI_CHANNEL_3						0x02
#define RTP_MIDI_CHANNEL_4						0x03
#define RTP_MIDI_CHANNEL_5						0x04
#define RTP_MIDI_CHANNEL_6						0x05
#define RTP_MIDI_CHANNEL_7						0x06
#define RTP_MIDI_CHANNEL_8						0x07
#define RTP_MIDI_CHANNEL_9						0x08
#define RTP_MIDI_CHANNEL_10						0x09
#define RTP_MIDI_CHANNEL_11						0x0a
#define RTP_MIDI_CHANNEL_12						0x0b
#define RTP_MIDI_CHANNEL_13						0x0c
#define RTP_MIDI_CHANNEL_14						0x0d
#define RTP_MIDI_CHANNEL_15						0x0e
#define RTP_MIDI_CHANNEL_16						0x0f
#define RTP_MIDI_CHANNEL_MASK						0x0f


/*
 * MIDI Status Bytes (System Exclusive Messages, System Common Messages, System Realtime Messages )
 */
#define RTP_MIDI_STATUS_COMMON_SYSEX_START				0xf0
#define RTP_MIDI_STATUS_COMMON_MTC_QUARTER_FRAME			0xf1
#define RTP_MIDI_STATUS_COMMON_SONG_POSITION_POINTER			0xf2
#define RTP_MIDI_STATUS_COMMON_SONG_SELECT				0xf3
#define RTP_MIDI_STATUS_COMMON_UNDEFINED_F4				0xf4
#define RTP_MIDI_STATUS_COMMON_UNDEFINED_F5				0xf5
#define RTP_MIDI_STATUS_COMMON_TUNE_REQUEST				0xf6
#define RTP_MIDI_STATUS_COMMON_SYSEX_END				0xf7
#define RTP_MIDI_STATUS_COMMON_REALTIME_TIMING_CLOCK			0xf8
#define RTP_MIDI_STATUS_COMMON_REALTIME_MIDI_TICK			0xf9		/* Spec says undefined */
#define RTP_MIDI_STATUS_COMMON_REALTIME_START				0xfa
#define RTP_MIDI_STATUS_COMMON_REALTIME_CONTINUE			0xfb
#define RTP_MIDI_STATUS_COMMON_REALTIME_STOP				0xfc
#define RTP_MIDI_STATUS_COMMON_REALTIME_UNDEFINED_FD			0xfd
#define RTP_MIDI_STATUS_COMMON_REALTIME_ACTIVE_SENSING			0xfe
#define RTP_MIDI_STATUS_COMMON_REALTIME_SYSTEM_RESET			0xff


/*
 * MIDI Universal Non-Realtime System Exclusive Messages (sysex-manu-id=0x7e)
 */
#define RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_HEADER			0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DATA_PACKET			0x02
#define RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_REQUEST			0x03
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC					0x04
#define RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_EXTENSIONS		0x05
#define RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_INFORMATION			0x06
#define RTP_MIDI_SYSEX_COMMON_NRT_FILE_DUMP				0x07
#define RTP_MIDI_SYSEX_COMMON_NRT_MIDI_TUNING_STANDARD			0x08
#define RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_MIDI				0x09
#define RTP_MIDI_SYSEX_COMMON_NRT_DOWNLOADABLE_SOUNDS			0x0a
#define RTP_MIDI_SYSEX_COMMON_NRT_END_OF_FILE				0x7b
#define RTP_MIDI_SYSEX_COMMON_NRT_WAIT					0x7c
#define RTP_MIDI_SYSEX_COMMON_NRT_CANCEL				0x7d
#define RTP_MIDI_SYSEX_COMMON_NRT_NAK					0x7e
#define RTP_MIDI_SYSEX_COMMON_NRT_ACK					0x7f


/*
 * MIDI Universal Realtime System Exclusive Messages (sysex-manu-id=0x7f)
 */
#define RTP_MIDI_SYSEX_COMMON_RT_MIDI_TIME_CODE				0x01
#define RTP_MIDI_SYSEX_COMMON_RT_MIDI_SHOW_CONTROL			0x02
#define RTP_MIDI_SYSEX_COMMON_RT_NOTATION_INFORMATION			0x03
#define RTP_MIDI_SYSEX_COMMON_RT_DEVICE_CONTROL				0x04
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUEING				0x05
#define RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_COMMAND		0x06
#define RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_RESPONSE		0x07
#define RTP_MIDI_SYSEX_COMMON_RT_MIDI_TUNING_STANDARD			0x08


/* List of short MIDI-manufacturer-IDs - needs to be completed! */
#define RTP_MIDI_MANU_SHORT_ISLONG					0x00
#define RTP_MIDI_MANU_SHORT_SEQUENTIAL_CIRCUITS				0x01
#define RTP_MIDI_MANU_SHORT_BIG_BRIAR_IDP				0x02
#define RTP_MIDI_MANU_SHORT_OCTAVE_PLATEAU_VOYETRA			0x03
#define RTP_MIDI_MANU_SHORT_MOOG					0x04
#define RTP_MIDI_MANU_SHORT_PASSPORT_DESIGNS				0x05
#define RTP_MIDI_MANU_SHORT_LEXICON					0x06
#define RTP_MIDI_MANU_SHORT_KURZWEIL					0x07
#define RTP_MIDI_MANU_SHORT_FENDER					0x08
#define RTP_MIDI_MANU_SHORT_GULBRANSEN					0x09
#define RTP_MIDI_MANU_SHORT_AKG_ACOUSTICS				0x0a
#define RTP_MIDI_MANU_SHORT_VOYCE_MUSIC					0x0b
#define RTP_MIDI_MANU_SHORT_WAVEFRAME					0x0c
#define RTP_MIDI_MANU_SHORT_ADA_SIGNAL_PROCESSORS			0x0d
#define RTP_MIDI_MANU_SHORT_GARFIELD_ELECTRONICS			0x0e
#define RTP_MIDI_MANU_SHORT_ENSONIQ					0x0f
#define RTP_MIDI_MANU_SHORT_OBERHEIM_GIBSON_LABS			0x10
#define RTP_MIDI_MANU_SHORT_APPLE_COMPUTERS				0x11
#define RTP_MIDI_MANU_SHORT_GREY_MATTER_RESPONSE			0x12
#define RTP_MIDI_MANU_SHORT_DIGIDESIGN					0x13
#define RTP_MIDI_MANU_SHORT_PALMTREE_INSTRUMENTS			0x14
#define RTP_MIDI_MANU_SHORT_JL_COOPER					0x15
#define RTP_MIDI_MANU_SHORT_LOWREY					0x16
#define RTP_MIDI_MANU_SHORT_LINN_ADAMS_SMITH				0x17
#define RTP_MIDI_MANU_SHORT_EMU_SYSTEMS					0x18
#define RTP_MIDI_MANU_SHORT_HARMONY_SYSTEMS				0x19
#define RTP_MIDI_MANU_SHORT_ART						0x1a
#define RTP_MIDI_MANU_SHORT_BALDWIN					0x1b
#define RTP_MIDI_MANU_SHORT_EVENTIDE					0x1c
#define RTP_MIDI_MANU_SHORT_INVENTRONICS				0x1d
#define RTP_MIDI_MANU_SHORT_KEY_CONCEPTS				0x1e
#define RTP_MIDI_MANU_SHORT_CLARITY					0x1f
#define RTP_MIDI_MANU_SHORT_PASSAC					0x20
#define RTP_MIDI_MANU_SHORT_SIEL					0x21
#define RTP_MIDI_MANU_SHORT_SYNTHE_AXE					0x22
#define RTP_MIDI_MANU_SHORT_STEPP					0x23
#define RTP_MIDI_MANU_SHORT_HOHNER					0x24
#define RTP_MIDI_MANU_SHORT_CRUMAR_TWISTER				0x25
#define RTP_MIDI_MANU_SHORT_SOLTON					0x26
#define RTP_MIDI_MANU_SHORT_JELLINGHAUS_MS				0x27
#define RTP_MIDI_MANU_SHORT_SOUTHWORK_MUSIC_SYSTEMS			0x28
#define RTP_MIDI_MANU_SHORT_PPG						0x29
#define RTP_MIDI_MANU_SHORT_JEN						0x2a
#define RTP_MIDI_MANU_SHORT_SSL						0x2b
#define RTP_MIDI_MANU_SHORT_AUDIO_VERITRIEB				0x2c
#define RTP_MIDI_MANU_SHORT_NEVE_HINTON_INSTRUMENTS			0x2d
#define RTP_MIDI_MANU_SHORT_SOUNDTRACS					0x2e
#define RTP_MIDI_MANU_SHORT_ELKA_GENERAL_MUSIC				0x2f
#define RTP_MIDI_MANU_SHORT_DYNACORD					0x30
#define RTP_MIDI_MANU_SHORT_VISCOUNT					0x31
#define RTP_MIDI_MANU_SHORT_DRAWMER					0x32
#define RTP_MIDI_MANU_SHORT_CLAVIA_DIGITAL_INSTRUMENTS			0x33
#define RTP_MIDI_MANU_SHORT_AUDIO_ARCHITECTURE				0x34
#define RTP_MIDI_MANU_SHORT_GENERAL_MUSIC_CORP				0x35
#define RTP_MIDI_MANU_SHORT_CHEETAH					0x36
#define RTP_MIDI_MANU_SHORT_CTM						0x37
#define RTP_MIDI_MANU_SHORT_SIMMONS_UK					0x38
#define RTP_MIDI_MANU_SHORT_SOUNDCRAFT_ELECTRONICS			0x39
#define RTP_MIDI_MANU_SHORT_STEINBERG_GMBH				0x3a
#define RTP_MIDI_MANU_SHORT_WERSI					0x3b
#define RTP_MIDI_MANU_SHORT_AVAB_ELEKTRONIK_AB				0x3c
#define RTP_MIDI_MANU_SHORT_DIGIGRAM					0x3d
#define RTP_MIDI_MANU_SHORT_WALDORF					0x3e
#define RTP_MIDI_MANU_SHORT_QUASIMIDI					0x3f
#define RTP_MIDI_MANU_SHORT_KAWAI					0x40
#define RTP_MIDI_MANU_SHORT_ROLAND					0x41
#define RTP_MIDI_MANU_SHORT_KORG					0x42
#define RTP_MIDI_MANU_SHORT_YAMAHA					0x43
#define RTP_MIDI_MANU_SHORT_CASIO					0x44
#define RTP_MIDI_MANU_SHORT_MORIDAIRA					0x45
#define RTP_MIDI_MANU_SHORT_KAMIYA_STUDIO				0x46
#define RTP_MIDI_MANU_SHORT_AKAI					0x47
#define RTP_MIDI_MANU_SHORT_JAPAN_VICTOR				0x48
#define RTP_MIDI_MANU_SHORT_MEISOSHA					0x49
#define RTP_MIDI_MANU_SHORT_HOSHINO_GAKKI				0x4a
#define RTP_MIDI_MANU_SHORT_FUJITSU					0x4b
#define RTP_MIDI_MANU_SHORT_SONY					0x4c
#define RTP_MIDI_MANU_SHORT_NISSHIN_ONPA				0x4d
#define RTP_MIDI_MANU_SHORT_TEAC					0x4e
/* #define RTP_MIDI_MANU_SHORT_						0x4f */
#define RTP_MIDI_MANU_SHORT_MATSUSHITA_ELECTRIC				0x50
#define RTP_MIDI_MANU_SHORT_FOSTEX					0x51
#define RTP_MIDI_MANU_SHORT_ZOOM					0x52
#define RTP_MIDI_MANU_SHORT_MIDORI_ELECTRONICS				0x53
#define RTP_MIDI_MANU_SHORT_MATSUSHITA_COMMUNICATION			0x54
#define RTP_MIDI_MANU_SHORT_SUZUKI					0x55
#define RTP_MIDI_MANU_SHORT_FUJI					0x56
#define RTP_MIDI_MANU_SHORT_ACOUSTIC_TECHNICAL_LAB			0x57
/* #define RTP_MIDI_MANU_SHORT_						0x58 */
#define RTP_MIDI_MANU_SHORT_FAITH					0x59
#define RTP_MIDI_MANU_SHORT_INTERNET_CORPORATION			0x5a
/* #define RTP_MIDI_MANU_SHORT_						0x5b */
#define RTP_MIDI_MANU_SHORT_SEEKERS_CO					0x5c
/* #define RTP_MIDI_MANU_SHORT_						0x5d */
/* #define RTP_MIDI_MANU_SHORT_						0x5e */
#define RTP_MIDI_MANU_SHORT_SD_CARD_ASSOCIATION				0x5f
/* #define RTP_MIDI_MANU_SHORT_						0x60
#define RTP_MIDI_MANU_SHORT_						0x61
#define RTP_MIDI_MANU_SHORT_						0x62
#define RTP_MIDI_MANU_SHORT_						0x63
#define RTP_MIDI_MANU_SHORT_						0x64
#define RTP_MIDI_MANU_SHORT_						0x65
#define RTP_MIDI_MANU_SHORT_						0x66
#define RTP_MIDI_MANU_SHORT_						0x67
#define RTP_MIDI_MANU_SHORT_						0x68
#define RTP_MIDI_MANU_SHORT_						0x69
#define RTP_MIDI_MANU_SHORT_						0x6a
#define RTP_MIDI_MANU_SHORT_						0x6b
#define RTP_MIDI_MANU_SHORT_						0x6c
#define RTP_MIDI_MANU_SHORT_						0x6d
#define RTP_MIDI_MANU_SHORT_						0x6e
#define RTP_MIDI_MANU_SHORT_						0x6f
#define RTP_MIDI_MANU_SHORT_						0x70
#define RTP_MIDI_MANU_SHORT_						0x71
#define RTP_MIDI_MANU_SHORT_						0x72
#define RTP_MIDI_MANU_SHORT_						0x73
#define RTP_MIDI_MANU_SHORT_						0x74
#define RTP_MIDI_MANU_SHORT_						0x75
#define RTP_MIDI_MANU_SHORT_						0x76
#define RTP_MIDI_MANU_SHORT_						0x77
#define RTP_MIDI_MANU_SHORT_						0x78
#define RTP_MIDI_MANU_SHORT_						0x79
#define RTP_MIDI_MANU_SHORT_						0x7a
#define RTP_MIDI_MANU_SHORT_						0x7b
#define RTP_MIDI_MANU_SHORT_						0x7c */
#define RTP_MIDI_MANU_SHORT_EDUCATIONAL_USE				0x7d
#define RTP_MIDI_MANU_SHORT_NON_REALTIME_UNIVERSAL			0x7e
#define RTP_MIDI_MANU_SHORT_REALTIME_UNIVERSAL				0x7f


/* List of long MIDI-manufacturer-IDs - needs to be completed! */
/* North American Manufacturers */
/* #define RTP_MIDI_MANU_LONG_TIME_					0x0000 */
#define RTP_MIDI_MANU_LONG_TIME_WARNER_INTERACTIVE			0x0001
#define RTP_MIDI_MANU_LONG_ADVANCED_GRAVIS_COMP				0x0002
#define RTP_MIDI_MANU_LONG_MEDIA_VISION					0x0003
#define RTP_MIDI_MANU_LONG_DORNES_RESEARCH_GROUP			0x0004
#define RTP_MIDI_MANU_LONG_KMUSE					0x0005
#define RTP_MIDI_MANU_LONG_STYPHER					0x0006
#define RTP_MIDI_MANU_LONG_DIGITAL_MUSIC_CORPORATION			0x0007
#define RTP_MIDI_MANU_LONG_IOTA_SYSTEMS					0x0008
#define RTP_MIDI_MANU_LONG_NEW_ENGLAND_DIGITAL				0x0009
#define RTP_MIDI_MANU_LONG_ARTISYN					0x000a
#define RTP_MIDI_MANU_LONG_IVL_TECHNOLOGIES				0x000b
#define RTP_MIDI_MANU_LONG_SOUTHERN_MUSIC_SYSTEMS			0x000c
#define RTP_MIDI_MANU_LONG_LAKE_BUTLER_SOUND_COMPANY			0x000d
#define RTP_MIDI_MANU_LONG_ALESIS					0x000e
#define RTP_MIDI_MANU_LONG_SOUND_CREATION				0x000f
#define RTP_MIDI_MANU_LONG_DOD_ELECTRONICS				0x0010
#define RTP_MIDI_MANU_LONG_STUDER_EDITECH				0x0011
#define RTP_MIDI_MANU_LONG_SONUS					0x0012
#define RTP_MIDI_MANU_LONG_TEMPORAL_ACUITY_PRODUCTS			0x0013
#define RTP_MIDI_MANU_LONG_PERFECT_FRETWORKS				0x0014
#define RTP_MIDI_MANU_LONG_KAT						0x0015
#define RTP_MIDI_MANU_LONG_OPCODE					0x0016
#define RTP_MIDI_MANU_LONG_RANE_CORP					0x0017
#define RTP_MIDI_MANU_LONG_SPATIAL_SOUND_ANADI_INC			0x0018
#define RTP_MIDI_MANU_LONG_KMX						0x0019
#define RTP_MIDI_MANU_LONG_ALLEN_AND_HEATH_BRENNEL			0x001a
#define RTP_MIDI_MANU_LONG_PEAVEY					0x001b
#define RTP_MIDI_MANU_LONG_360_SYSTEMS					0x001c
#define RTP_MIDI_MANU_LONG_SPECTRUM_DESIGN_DEVELOPMENT			0x001d
#define RTP_MIDI_MANU_LONG_MARQUIS_MUSIC				0x001e
#define RTP_MIDI_MANU_LONG_ZETA_SYSTEMS					0x001f
#define RTP_MIDI_MANU_LONG_AXXES					0x0020
#define RTP_MIDI_MANU_LONG_ORBAN					0x0021
#define RTP_MIDI_MANU_LONG_INDIAN_VALLEY				0x0022
#define RTP_MIDI_MANU_LONG_TRITON					0x0023
#define RTP_MIDI_MANU_LONG_KTI						0x0024
#define RTP_MIDI_MANU_LONG_BREAKAWAY_TECHNOLOGIES			0x0025
#define RTP_MIDI_MANU_LONG_CAE						0x0026
#define RTP_MIDI_MANU_LONG_HARRISON_SYSTEMS_INC				0x0027
#define RTP_MIDI_MANU_LONG_FUTURE_LAB_MARK_KUO				0x0028
#define RTP_MIDI_MANU_LONG_ROCKTRON_CORP				0x0029
#define RTP_MIDI_MANU_LONG_PIANODISC					0x002a
#define RTP_MIDI_MANU_LONG_CANNON_RESEARCH_GROUP			0x002b
/* #define RTP_MIDI_MANU_LONG_						0x002c */
#define RTP_MIDI_MANU_LONG_RODGERS_INSTRUMENTS_CORP			0x002d
#define RTP_MIDI_MANU_LONG_BLUE_SKY_LOGIC				0x002e
#define RTP_MIDI_MANU_LONG_ENCORE_ELECTRONICS				0x002f
#define RTP_MIDI_MANU_LONG_UPTOWN					0x0030
#define RTP_MIDI_MANU_LONG_VOCE						0x0031
#define RTP_MIDI_MANU_LONG_CTI_AUDIO_INC				0x0032
#define RTP_MIDI_MANU_LONG_SS_RESEARCH					0x0033
#define RTP_MIDI_MANU_LONG_BRODERBUND_SOFTWARE				0x0034
#define RTP_MIDI_MANU_LONG_ALLEN_ORGAN_CO				0x0035
/* #define RTP_MIDI_MANU_LONG_						0x0036 */
#define RTP_MIDI_MANU_LONG_MUSIC_QUEST					0x0037
#define RTP_MIDI_MANU_LONG_APHEX					0x0038
#define RTP_MIDI_MANU_LONG_GALLIEN_KRUEGER				0x0039
#define RTP_MIDI_MANU_LONG_IBM						0x003a
#define RTP_MIDI_MANU_LONG_MARK_OF_THE_UNICORN				0x003b
#define RTP_MIDI_MANU_LONG_HOTZ_INSTRUMENTS_TECH			0x003c
#define RTP_MIDI_MANU_LONG_ETA_LIGHTING					0x003d
#define RTP_MIDI_MANU_LONG_NSI_CORPORATION				0x003e
#define RTP_MIDI_MANU_LONG_ADLIB_INC					0x003f
#define RTP_MIDI_MANU_LONG_RICHMOND_SOUND_DESIGN			0x0040
#define RTP_MIDI_MANU_LONG_MICROSOFT					0x0041
#define RTP_MIDI_MANU_LONG_THE_SOFTWARE_TOOLWORKS			0x0042
#define RTP_MIDI_MANU_LONG_RJMG_NICHE					0x0043
#define RTP_MIDI_MANU_LONG_INTONE					0x0044
#define RTP_MIDI_MANU_LONG_ADVANCED_REMOTE_TECH				0x0045
/* #define RTP_MIDI_MANU_LONG_						0x0046 */
#define RTP_MIDI_MANU_LONG_GT_ELECTRONICS_GROOVE_TUBES			0x0047
#define RTP_MIDI_MANU_LONG_INTERMIDI					0x0048
#define RTP_MIDI_MANU_LONG_TIMELINE_VISTA				0x0049
#define RTP_MIDI_MANU_LONG_MESA_BOOGIE					0x004a
/* #define RTP_MIDI_MANU_LONG_						0x004b */
#define RTP_MIDI_MANU_LONG_SEQUOIA_DEVELOPMENT				0x004c
#define RTP_MIDI_MANU_LONG_STUDIO_ELECTRONICS				0x004d
#define RTP_MIDI_MANU_LONG_EUPHONIX					0x004e
#define RTP_MIDI_MANU_LONG_INTERMIDI2					0x004f
#define RTP_MIDI_MANU_LONG_MIDI_SOLUTIONS				0x0050
#define RTP_MIDI_MANU_LONG_3DO_COMPANY					0x0051
#define RTP_MIDI_MANU_LONG_LIGHTWAVE_RESEARCH				0x0052
#define RTP_MIDI_MANU_LONG_MICROW					0x0053
#define RTP_MIDI_MANU_LONG_SPECTRAL_SYNTHESIS				0x0054
#define RTP_MIDI_MANU_LONG_LONE_WOLF					0x0055
#define RTP_MIDI_MANU_LONG_STUDIO_TECHNOLOGIES				0x0056
#define RTP_MIDI_MANU_LONG_PETERSON_EMP					0x0057
#define RTP_MIDI_MANU_LONG_ATARI					0x0058
#define RTP_MIDI_MANU_LONG_MARION_SYSTEMS				0x0059
#define RTP_MIDI_MANU_LONG_DESIGN_EVENT					0x005a
#define RTP_MIDI_MANU_LONG_WINJAMMER_SOFTWARE				0x005b
#define RTP_MIDI_MANU_LONG_ATT_BELL_LABS				0x005c
/*# define RTP_MIDI_MANU_LONG_						0x005d */
#define RTP_MIDI_MANU_LONG_SYMETRIX					0x005e
#define RTP_MIDI_MANU_LONG_MIDI_THE_WORLD				0x005f
#define RTP_MIDI_MANU_LONG_DESPER_PRODUCTS				0x0060
#define RTP_MIDI_MANU_LONG_MICROS_N_MIDI				0x0061
#define RTP_MIDI_MANU_LONG_ACCORDIANS_INTL				0x0062
#define RTP_MIDI_MANU_LONG_EUPHONICS					0x0063
#define RTP_MIDI_MANU_LONG_MUSONIX					0x0064
#define RTP_MIDI_MANU_LONG_TURTLE_BEACH_SYSTEMS				0x0065
#define RTP_MIDI_MANU_LONG_MACKIE_DESIGNS				0x0066
#define RTP_MIDI_MANU_LONG_COMPUSERVE					0x0067
#define RTP_MIDI_MANU_LONG_BES_TECHNOLOGIES				0x0068
#define RTP_MIDI_MANU_LONG_QRS_MUSIC_ROLLS				0x0069
#define RTP_MIDI_MANU_LONG_P_G_MUSIC					0x006a
#define RTP_MIDI_MANU_LONG_SIERRA_SEMICONDUCTOR				0x006b
#define RTP_MIDI_MANU_LONG_EPIGRAF_AUDIO_VISUAL				0x006c
#define RTP_MIDI_MANU_LONG_ELECTRONICS_DIVERSIFIED			0x006d
#define RTP_MIDI_MANU_LONG_TUNE_1000					0x006e
#define RTP_MIDI_MANU_LONG_ADVANCED_MICRO_DEVICES			0x006f
#define RTP_MIDI_MANU_LONG_MEDIAMATION					0x0070
#define RTP_MIDI_MANU_LONG_SABINE_MUSIC					0x0071
#define RTP_MIDI_MANU_LONG_WOOG_LABS					0x0072
#define RTP_MIDI_MANU_LONG_MIRCOPOLIS					0x0073
#define RTP_MIDI_MANU_LONG_TA_HORNG_MUSICAL_INSTRUMENT			0x0074
#define RTP_MIDI_MANU_LONG_ETEK_LABS_FORTE_TECH				0x0075
#define RTP_MIDI_MANU_LONG_ELECTRO_VOICE				0x0076
#define RTP_MIDI_MANU_LONG_MIDISOFT_CORPORATION				0x0077
#define RTP_MIDI_MANU_LONG_QSOUND_LABS					0x0078
#define RTP_MIDI_MANU_LONG_WESTREX					0x0079
#define RTP_MIDI_MANU_LONG_NVIDIA					0x007a
#define RTP_MIDI_MANU_LONG_ESS_TECHNOLOGY				0x007b
#define RTP_MIDI_MANU_LONG_MEDIATRIX_PERIPHERALS			0x007c
#define RTP_MIDI_MANU_LONG_BROOKTREE_CORP				0x007d
#define RTP_MIDI_MANU_LONG_OTARI_CORP					0x007e
#define RTP_MIDI_MANU_LONG_KEY_ELECTRONICS				0x007f
/* ---break--- */
#define RTP_MIDI_MANU_LONG_SHURE_INCORPORATED				0x0100
#define RTP_MIDI_MANU_LONG_AURA_SOUND					0x0101
#define RTP_MIDI_MANU_LONG_CRYSTAL_SEMICONDUCTOR			0x0102
#define RTP_MIDI_MANU_LONG_CONEXANT_ROCKWELL				0x0103
#define RTP_MIDI_MANU_LONG_SILICON_GRAPHICS				0x0104
#define RTP_MIDI_MANU_LONG_MAUDIO_MIDIMAN				0x0105
#define RTP_MIDI_MANU_LONG_PRESONUS					0x0106
/* #define RTP_MIDI_MANU_LONG_						0x0107 */
#define RTP_MIDI_MANU_LONG_TOPAZ_ENTERPRISES				0x0108
#define RTP_MIDI_MANU_LONG_CAST_LIGHTING				0x0109
#define RTP_MIDI_MANU_LONG_MICROSOFT_CONSUMER_DIVISION			0x010a
#define RTP_MIDI_MANU_LONG_SONIC_FOUNDRY				0x010b
#define RTP_MIDI_MANU_LONG_LINE6_FAST_FORWARD				0x010c
#define RTP_MIDI_MANU_LONG_BEATNIK_INC					0x010d
#define RTP_MIDI_MANU_LONG_VAN_KOEVERING_COMPANY			0x010e
#define RTP_MIDI_MANU_LONG_ALTECH_SYSTEMS				0x010f
#define RTP_MIDI_MANU_LONG_S_S_RESEARCH					0x0110
#define RTP_MIDI_MANU_LONG_VLSI_TECHNOLOGY				0x0111
#define RTP_MIDI_MANU_LONG_CHROMATIC_RESEARCH				0x0112
#define RTP_MIDI_MANU_LONG_SAPPHIRE					0x0113
#define RTP_MIDI_MANU_LONG_IDRC						0x0114
#define RTP_MIDI_MANU_LONG_JUSTONIC_TUNING				0x0115
#define RTP_MIDI_MANU_LONG_TORCOMP_RESEARCH_INC				0x0116
#define RTP_MIDI_MANU_LONG_NEWTEK_INC					0x0117
#define RTP_MIDI_MANU_LONG_SOUND_SCULPTURE				0x0118
#define RTP_MIDI_MANU_LONG_WALKER_TECHNICAL				0x0119
#define RTP_MIDI_MANU_LONG_DIGITAL_HARMONY				0x011a
#define RTP_MIDI_MANU_LONG_INVISION_INTERACTIVE				0x011b
#define RTP_MIDI_MANU_LONG_TSQUARE_DESIGN				0x011c
#define RTP_MIDI_MANU_LONG_NEMESYS_MUSIC_TECHNOLOGY			0x011d
#define RTP_MIDI_MANU_LONG_DBX_PROFESSIONAL_HARMAN_INTL			0x011e
#define RTP_MIDI_MANU_LONG_SYNDYNE_CORPORATION				0x011f
#define RTP_MIDI_MANU_LONG_BITHEADZ					0x0120
#define RTP_MIDI_MANU_LONG_CAKEWALK_MUSIC_SOFTWARE			0x0121
#define RTP_MIDI_MANU_LONG_ANALOG_DEVICES				0x0122
#define RTP_MIDI_MANU_LONG_NATIONAL_SEMICONDUCTOR			0x0123
#define RTP_MIDI_MANU_LONG_BOOM_THEORY					0x0124
#define RTP_MIDI_MANU_LONG_VIRTUAL_DSP_CORPORATION			0x0125
#define RTP_MIDI_MANU_LONG_ANTARES_SYSTEMS				0x0126
#define RTP_MIDI_MANU_LONG_ANGEL_SOFTWARE				0x0127
#define RTP_MIDI_MANU_LONG_ST_LOUIS_MUSIC				0x0128
#define RTP_MIDI_MANU_LONG_LYRRUS_DBA_GVOX				0x0129
#define RTP_MIDI_MANU_LONG_ASHLEY_AUDIO_INC				0x012a
#define RTP_MIDI_MANU_LONG_VARILITE_INC					0x012b
#define RTP_MIDI_MANU_LONG_SUMMIT_AUDIO_INC				0x012c
#define RTP_MIDI_MANU_LONG_AUREAL_SEMICONDUCTOR_INC			0x012d
#define RTP_MIDI_MANU_LONG_SEASOUND_LLC					0x012e
#define RTP_MIDI_MANU_LONG_US_ROBOTICS					0x012f
#define RTP_MIDI_MANU_LONG_AURISIS_RESEARCH				0x0130
#define RTP_MIDI_MANU_LONG_NEARFIELD_MULTIMEDIA				0x0131
#define RTP_MIDI_MANU_LONG_FM7_INC					0x0132
#define RTP_MIDI_MANU_LONG_SWIVEL_SYSTEMS				0x0133
#define RTP_MIDI_MANU_LONG_HYPERACTIVE_AUDIO_SYSTEMS			0x0134
#define RTP_MIDI_MANU_LONG_MIDILITE_CASTE_STUDIO_PROD			0x0135
#define RTP_MIDI_MANU_LONG_RADIKAL_TECHNOLOGIES				0x0136
#define RTP_MIDI_MANU_LONG_ROGER_LINN_DESIGN				0x0137
#define RTP_MIDI_MANU_LONG_TCHELION_VOCAL_TECHNOLOGIES			0x0138
#define RTP_MIDI_MANU_LONG_EVENT_ELECTRONICS				0x0139
#define RTP_MIDI_MANU_LONG_SONIC_NETWORK_INC				0x013a
#define RTP_MIDI_MANU_LONG_REALTIME_MUSIC_SOLUTIONS			0x013b
#define RTP_MIDI_MANU_LONG_APOGEE_DIGITAL				0x013c
#define RTP_MIDI_MANU_LONG_CLASSICAL_ORGANS_INC				0x013d
#define RTP_MIDI_MANU_LONG_MICROTOOLS_INC				0x013e
#define RTP_MIDI_MANU_LONG_NUMARK_INDUSTRIES				0x013f
#define RTP_MIDI_MANU_LONG_FRONTIER_DESIGN_GROUP_LLC			0x0140
#define RTP_MIDI_MANU_LONG_RECORDARE_LLC				0x0141
#define RTP_MIDI_MANU_LONG_STARR_LABS					0x0142
#define RTP_MIDI_MANU_LONG_VOYAGER_SOUND_INC				0x0143
#define RTP_MIDI_MANU_LONG_MANIFOLD_LABS				0x0144
#define RTP_MIDI_MANU_LONG_AVIOM_INC					0x0145
#define RTP_MIDI_MANU_LONG_MIXMEISTER_TECHNOLOGY			0x0146
#define RTP_MIDI_MANU_LONG_NOTATION_SOFTWARE				0x0147
#define RTP_MIDI_MANU_LONG_MERCURIAL_COMMUNICATIONS			0x0148
#define RTP_MIDI_MANU_LONG_WAVE_ARTS					0x0149
#define RTP_MIDI_MANU_LONG_LOGIC_SEQUENCING_DEVICES			0x014a
#define RTP_MIDI_MANU_LONG_AXESS_ELECTRONICS				0x014b
#define RTP_MIDI_MANU_LONG_MUSE_RESEARCH				0x014c
#define RTP_MIDI_MANU_LONG_OPEN_LABS					0x014d
#define RTP_MIDI_MANU_LONG_GUILLEMOT_RD_INC				0x014e
#define RTP_MIDI_MANU_LONG_SAMSON_TECHNOLOGIES				0x014f
#define RTP_MIDI_MANU_LONG_ELECTRONIC_THEATRE_CONTROLS			0x0150
#define RTP_MIDI_MANU_LONG_RESEARCH_IN_MOTION				0x0151
#define RTP_MIDI_MANU_LONG_MOBILEER					0x0152
#define RTP_MIDI_MANU_LONG_SYNTHOGY					0x0153
#define RTP_MIDI_MANU_LONG_LYNX_STUDIO_TECHNOLOGY_INC			0x0154
#define RTP_MIDI_MANU_LONG_DAMAGE_CONTROL_ENGINEERING			0x0155
#define RTP_MIDI_MANU_LONG_YOST_ENGINEERING_INC				0x0156
#define RTP_MIDI_MANU_LONG_BROOKS_FORSMAN_DESIGNS_LLC			0x0157
#define RTP_MIDI_MANU_LONG_MAGNEKEY					0x0158
#define RTP_MIDI_MANU_LONG_GARRITAN_CORP				0x0159
#define RTP_MIDI_MANU_LONG_PLOQUE_ART_ET_TECHNOLOGIE			0x015a
#define RTP_MIDI_MANU_LONG_RJM_MUSIC_TECHNOLOGY				0x015b
#define RTP_MIDI_MANU_LONG_CUSTOM_SOLUTIONS_SOFTWARE			0x015c
#define RTP_MIDI_MANU_LONG_SONARCANA_LLC				0x015d
#define RTP_MIDI_MANU_LONG_CENTRANCE					0x015e
#define RTP_MIDI_MANU_LONG_KESUMO_LLC					0x015f
#define RTP_MIDI_MANU_LONG_STANTON					0x0160
#define RTP_MIDI_MANU_LONG_LIVID_INSTRUMENTS				0x0161
#define RTP_MIDI_MANU_LONG_FIRST_ACT_745_MEDIA				0x0162
#define RTP_MIDI_MANU_LONG_PYGRAPHICS_INC				0x0163
#define RTP_MIDI_MANU_LONG_PANADIGM_INNOVATIONS_LTD			0x0164
#define RTP_MIDI_MANU_LONG_AVEDIS_ZILDJIAN_CO				0x0165
#define RTP_MIDI_MANU_LONG_AUVITAL_MUSIC_CORP				0x0166
#define RTP_MIDI_MANU_LONG_INSPIRED_INSTRUMENTS_INC			0x0167
#define RTP_MIDI_MANU_LONG_CHRIS_GRIGG_DESIGNS				0x0168
#define RTP_MIDI_MANU_LONG_SLATE_DIGITAL_LLC				0x0169
#define RTP_MIDI_MANU_LONG_MIXWARE					0x016a
#define RTP_MIDI_MANU_LONG_SOCIAL_ENTROPY				0x016b
#define RTP_MIDI_MANU_LONG_SOURCE_AUDIO_LLC				0x016c
#define RTP_MIDI_MANU_LONG_RESERVED_016d				0x016d
#define RTP_MIDI_MANU_LONG_RESERVED_016e				0x016e
#define RTP_MIDI_MANU_LONG_RESERVED_016f				0x016f
#define RTP_MIDI_MANU_LONG_AMERICAN_AUDIO_DJ				0x0170
#define RTP_MIDI_MANU_LONG_MEGA_CONTROL_SYSTEMS				0x0171
#define RTP_MIDI_MANU_LONG_KILPATRICK_AUDIO				0x0172
#define RTP_MIDI_MANU_LONG_IKINGDOM_CORP				0x0173
#define RTP_MIDI_MANU_LONG_FRACTAL_AUDIO				0x0174
#define RTP_MIDI_MANU_LONG_NETLOGIC_MICROSYSTEMS			0x0175
#define RTP_MIDI_MANU_LONG_MUSIC_COMPUTING				0x0176
#define RTP_MIDI_MANU_LONG_NEKTAR_TECHNOLOGY_INC			0x0177
#define RTP_MIDI_MANU_LONG_ZENPH_SOUND_INNOVATIONS			0x0178
#define RTP_MIDI_MANU_LONG_DJTECHTOOLS_COM				0x0179
#define RTP_MIDI_MANU_LONG_RESERVED_017a				0x017a

/* European manufacturers */
#define RTP_MIDI_MANU_LONG_DREAM					0x2000
#define RTP_MIDI_MANU_LONG_STRAND_LIGHTING				0x2001
#define RTP_MIDI_MANU_LONG_AMEK_SYSTEMS					0x2002
#define RTP_MIDI_MANU_LONG_CASA_DI_RISPARMIO_DI_LORETO			0x2003
#define RTP_MIDI_MANU_LONG_BOHM_ELECTRONIC				0x2004
#define RTP_MIDI_MANU_LONG_SYNTEC_DIGITAL_AUDIO				0x2005
#define RTP_MIDI_MANU_LONG_TRIDENT_AUDIO				0x2006
#define RTP_MIDI_MANU_LONG_REAL_WORLD_STUDIO				0x2007
#define RTP_MIDI_MANU_LONG_EVOLUTION_SYNTHESIS				0x2008
#define RTP_MIDI_MANU_LONG_YES_TECHNOLOGY				0x2009
#define RTP_MIDI_MANU_LONG_AUDIOMATICA					0x200a
#define RTP_MIDI_MANU_LONG_BONTEMPI_FARFISA				0x200b
#define RTP_MIDI_MANU_LONG_FBT_ELETTRONICA				0x200c
#define RTP_MIDI_MANU_LONG_MIDITEMP					0x200d
#define RTP_MIDI_MANU_LONG_LA_AUDIO_LARKING_AUDIO			0x200e
#define RTP_MIDI_MANU_LONG_ZERO_88_LIGHTING_LIMITED			0x200f
#define RTP_MIDI_MANU_LONG_MICON_AUDIO_ELECTRONICS_GMBH			0x2010
#define RTP_MIDI_MANU_LONG_FOREFRONT_TECHNOLOGY				0x2011
#define RTP_MIDI_MANU_LONG_STUDIO_AUDIO_AND_VIDEO_LTD			0x2012
#define RTP_MIDI_MANU_LONG_KENTON_ELECTRONICS				0x2013
#define RTP_MIDI_MANU_LONG_CELCO_DIVISON_OF_ELECTRONICS			0x2014
#define RTP_MIDI_MANU_LONG_ADB						0x2015
#define RTP_MIDI_MANU_LONG_MARSHALL_PRODUCTS				0x2016
#define RTP_MIDI_MANU_LONG_DDA						0x2017
#define RTP_MIDI_MANU_LONG_BBS						0x2018
#define RTP_MIDI_MANU_LONG_MA_LIGHTING_TECHNOLOGY			0x2019
#define RTP_MIDI_MANU_LONG_FATAR					0x201a
#define RTP_MIDI_MANU_LONG_QSC_AUDIO					0x201b
#define RTP_MIDI_MANU_LONG_ARTISAN_CLASSIC_ORGAN			0x201c
#define RTP_MIDI_MANU_LONG_ORLA_SPA					0x201d
#define RTP_MIDI_MANU_LONG_PINNACLE_AUDIO				0x201e
#define RTP_MIDI_MANU_LONG_TC_ELECTRONICS				0x201f
#define RTP_MIDI_MANU_LONG_DOEPFER_MUSIKELEKTRONIK			0x2020
#define RTP_MIDI_MANU_LONG_CREATIVE_TECHNOLOGY_PTE			0x2021
#define RTP_MIDI_MANU_LONG_MINAMI_SEIYDDO				0x2022
#define RTP_MIDI_MANU_LONG_GOLDSTAR					0x2023
#define RTP_MIDI_MANU_LONG_MIDISOFT_SAS_DI_M_CIMA			0x2024
#define RTP_MIDI_MANU_LONG_SAMICK					0x2025
#define RTP_MIDI_MANU_LONG_PENNY_AND_GILES				0x2026
#define RTP_MIDI_MANU_LONG_ACORN_COMPUTER				0x2027
#define RTP_MIDI_MANU_LONG_LSC_ELECTRONICS				0x2028
#define RTP_MIDI_MANU_LONG_NOVATION_EMS					0x2029
#define RTP_MIDI_MANU_LONG_SAMKYUNG_MECHATRONICS			0x202a
#define RTP_MIDI_MANU_LONG_MEDELI_ELECTRONICS_CO			0x202b
#define RTP_MIDI_MANU_LONG_CHARLIE_LAB_SRL				0x202c
#define RTP_MIDI_MANU_LONG_BLUE_CHIP_MUSIC_TECHNOLOGY			0x202d
#define RTP_MIDI_MANU_LONG_BEE_OH_CORP					0x202e
#define RTP_MIDI_MANU_LONG_LG_SEMICON_AMERICA				0x202f
#define RTP_MIDI_MANU_LONG_TESI						0x2030
#define RTP_MIDI_MANU_LONG_EMAGIC					0x2031
#define RTP_MIDI_MANU_LONG_BEHRINGER_GMBH				0x2032
#define RTP_MIDI_MANU_LONG_ACCESS_MUSIC_ELECTRONICS			0x2033
#define RTP_MIDI_MANU_LONG_SYNOPTIC					0x2034
#define RTP_MIDI_MANU_LONG_HANMESOFT_CORP				0x2035
#define RTP_MIDI_MANU_LONG_TERRATEC_ELECTRONIC_GMBH			0x2036
#define RTP_MIDI_MANU_LONG_PROEL_SPA					0x2037
#define RTP_MIDI_MANU_LONG_IBK_MIDI					0x2038
#define RTP_MIDI_MANU_LONG_IRCAM					0x2039
#define RTP_MIDI_MANU_LONG_PROPELLERHEAD_SOFTWARE			0x203a
#define RTP_MIDI_MANU_LONG_RED_SOUND_SYSTEMS_LTD			0x203b
#define RTP_MIDI_MANU_LONG_ELEKTRON_ESI_AB				0x203c
#define RTP_MIDI_MANU_LONG_SINTEFEX_AUDIO				0x203d
#define RTP_MIDI_MANU_LONG_MAM_MUSIC_AND_MORE				0x203e
#define RTP_MIDI_MANU_LONG_AMSARO_GMBH					0x203f
#define RTP_MIDI_MANU_LONG_CDS_ADVANCED_TECHNOLOGY_BV			0x2040
#define RTP_MIDI_MANU_LONG_TOUCHED_BY_SOUND_GMBH			0x2041
#define RTP_MIDI_MANU_LONG_DSP_ARTS					0x2042
#define RTP_MIDI_MANU_LONG_PHIL_REES_MUSIC_TECH				0x2043
#define RTP_MIDI_MANU_LONG_STAMER_MUSIKANLAGEN_GMBH			0x2044
#define RTP_MIDI_MANU_LONG_MUSICAL_MUNTANER_SA_DBA			0x2045
#define RTP_MIDI_MANU_LONG_CMEXX_SOFTWARE				0x2046
#define RTP_MIDI_MANU_LONG_KLAVIS_TECHNOLOGIES				0x2047
#define RTP_MIDI_MANU_LONG_NOTEHEADS_AB					0x2048
#define RTP_MIDI_MANU_LONG_ALGORITHMIX					0x2049
#define RTP_MIDI_MANU_LONG_SKRYDSTRUP_RD				0x204a
#define RTP_MIDI_MANU_LONG_PROFRESSIONAL_AUDIO_COMPANY			0x204b
#define RTP_MIDI_MANU_LONG_DBTECH_MADWAVES				0x204c
#define RTP_MIDI_MANU_LONG_VERMONA					0x204d
#define RTP_MIDI_MANU_LONG_NOKIA					0x204e
#define RTP_MIDI_MANU_LONG_WAVE_IDEA					0x204f
#define RTP_MIDI_MANU_LONG_HARTMANN_GMBH				0x2050
#define RTP_MIDI_MANU_LONG_LIONS_TRACK					0x2051
#define RTP_MIDI_MANU_LONG_ANALOGUE_SYSTEMS				0x2052
#define RTP_MIDI_MANU_LONG_FOCAL_JMLAB					0x2053
#define RTP_MIDI_MANU_LONG_RINGWAY_ELECTRONICS				0x2054
#define RTP_MIDI_MANU_LONG_FAITH_TECHNOLOGIES_DIGIPLUG			0x2055
#define RTP_MIDI_MANU_LONG_SHOWWORKS					0x2056
#define RTP_MIDI_MANU_LONG_MANIKIN_ELECTRONIC				0x2057
#define RTP_MIDI_MANU_LONG_1_COME_TECH					0x2058
#define RTP_MIDI_MANU_LONG_PHONIC_CORP					0x2059
#define RTP_MIDI_MANU_LONG_LAKE_TECHNOLOGY				0x205a
#define RTP_MIDI_MANU_LONG_SILANSYS_TECHNOLOGIES			0x205b
#define RTP_MIDI_MANU_LONG_WINBOND_ELECTRONICS				0x205c
#define RTP_MIDI_MANU_LONG_CINETIX_MEDIEN_UND_INTERFACE			0x205d
#define RTP_MIDI_MANU_LONG_AG_SOLUTIONI_DIGITALI			0x205e
#define RTP_MIDI_MANU_LONG_SEQUENTIX_MUSIC_SYSTEMS			0x205f
#define RTP_MIDI_MANU_LONG_ORAM_PRO_AUDIO				0x2060
#define RTP_MIDI_MANU_LONG_BE4_LTD					0x2061
#define RTP_MIDI_MANU_LONG_INFECTION_MUSIC				0x2062
#define RTP_MIDI_MANU_LONG_CENTRAL_MUSIC_CO_CME				0x2063
#define RTP_MIDI_MANU_LONG_GENOQS_MACHINES				0x2064
#define RTP_MIDI_MANU_LONG_MEDIALON					0x2065
#define RTP_MIDI_MANU_LONG_WAVES_AUDIO_LTD				0x2066
#define RTP_MIDI_MANU_LONG_JERASH_LABS					0x2067
#define RTP_MIDI_MANU_LONG_DA_FACT					0x2068
#define RTP_MIDI_MANU_LONG_ELBY_DESIGNS					0x2069
#define RTP_MIDI_MANU_LONG_SPECTRAL_AUDIO				0x206a
#define RTP_MIDI_MANU_LONG_ARTURIA					0x206b
#define RTP_MIDI_MANU_LONG_VIXID					0x206c
#define RTP_MIDI_MANU_LONG_C_THRU_MUSIC					0x206d
#define RTP_MIDI_MANU_LONG_YA_HORNG_ELECTRONIC_CO_LTD			0x206e
#define RTP_MIDI_MANU_LONG_SM_PRO_AUDIO					0x206f
#define RTP_MIDI_MANU_LONG_OTO_MACHINES					0x2070
#define RTP_MIDI_MANU_LONG_ELZAB_SA_G_LAB				0x2071
#define RTP_MIDI_MANU_LONG_BLACKSTAR_AMPLIFICATION_LTD			0x2072
#define RTP_MIDI_MANU_LONG_M3I_TECHNOLOGIES_GMBH			0x2073
#define RTP_MIDI_MANU_LONG_GEMALTO					0x2074
#define RTP_MIDI_MANU_LONG_PROSTAGE_SL					0x2075
#define RTP_MIDI_MANU_LONG_TEENAGE_ENGINEERING				0x2076
#define RTP_MIDI_MANU_LONG_TOBIAS_ERICHSEN				0x2077
#define RTP_MIDI_MANU_LONG_NIXER_LTD					0x2078
#define RTP_MIDI_MANU_LONG_HANPIN_ELECTRON_CO_LTD			0x2079
#define RTP_MIDI_MANU_LONG_MIDI_HARDWARE_R_SOWA				0x207a

#define RTP_MIDI_MANU_LONG_BEYOND_MUSIC_INDUSTRIAL_LTD			0x207b
#define RTP_MIDI_MANU_LONG_KISS_BOX_BV					0x207c
#define RTP_MIDI_MANU_LONG_MISA_DIGITAL_TECHNOLOGIES_LTD		0x207d
#define RTP_MIDI_MANU_LONG_AI_MUSICS_TECHNOLOGY_INC			0x207e
#define RTP_MIDI_MANU_LONG_SERATO_INC_LP				0x207f
#define RTP_MIDI_MANU_LONG_LIMEX_MUSIC_HANDLES_GMBH			0x2100
#define RTP_MIDI_MANU_LONG_KYODDAY_TOKAI				0x2101
#define RTP_MIDI_MANU_LONG_MUTABLE_INSTRUMENTS				0x2102
#define RTP_MIDI_MANU_LONG_PRESONUS_SOFTWARE_LTD			0x2103
#define RTP_MIDI_MANU_LONG_XIRING					0x2104
#define RTP_MIDI_MANU_LONG_FAIRLIGHT_INTRUMENTS_PTY_LTD			0x2105
#define RTP_MIDI_MANU_LONG_MUSICOM_LAB					0x2106
#define RTP_MIDI_MANU_LONG_VACO_LOCO					0x2107
#define RTP_MIDI_MANU_LONG_RWA_HONG_KONG_LIMITED			0x2108

/* Japanese Manufacturers */
#define RTP_MIDI_MANU_LONG_CRIMSON_TECHNOLOGY_INC			0x4000
#define RTP_MIDI_MANU_LONG_SOFTBANK_MOBILE_CORP				0x4001
/*#define RTP_MIDI_MANU_LONG_*/
#define RTP_MIDI_MANU_LONG_DM_HOLDINGS_INC				0x4003


#define RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_ON				0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_OFF			0x02
#define RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_VOICE_ALLOCATION_OFF	0x03
#define RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_VOICE_ALLOCATION_ON	0x04


#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_SPECIAL				0x00
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_PUNCH_IN_POINTS			0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_PUNGH_OUT_POINTS			0x02
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_PUNCH_IN_POINTS		0x03
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_PUNCH_OUT_POINTS		0x04
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_START_POINT			0x05
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_STOP_POINT			0x06
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_START_POINT_ADD		0x07
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_STOP_POINT_ADD		0x08
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_EVENT_START_POINT		0x09
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_EVENT_STOP_POINT		0x0a
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_CUE_POINTS			0x0b
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_CUE_POINTS_ADD			0x0c
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_CUE_POINT			0x0d
#define RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_NAME_IN_ADD			0x0e

#define RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINT_TRANSMISSION	0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINTS_REQUEST		0x02

#define RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_UNI				0x00
#define RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_BI				0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_OFF				0x7f

#define RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REQUEST			0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REPLY			0x02

#define RTP_MIDI_SYSEX_COMMON_NRT_FD_HEADER				0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_FD_DATA_PACKET			0x02
#define RTP_MIDI_SYSEX_COMMON_NRT_FD_REQUEST				0x03

#define RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REQUEST			0x00
#define RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REPLY			0x01
#define RTP_MIDI_SYSEX_COMMON_TUNING_NOTE_CHANGE			0x02

#define RTP_MIDI_SYSEX_COMMON_NRT_GM_ON					0x01
#define RTP_MIDI_SYSEX_COMMON_NRT_GM_OFF				0x02

#define RTP_MIDI_SYSEX_COMMON_RT_MTC_FULL_MESSAGE			0x01
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_USER_BITS				0x02

#define RTP_MIDI_SYSEX_COMMON_RT_SCL_EXTENSIONS				0x00
#define RTP_MIDI_SYSEX_COMMON_RT_SC_LIGHTING				0x01
#define RTP_MIDI_SYSEX_COMMON_RT_SC_MOVING_LIGHTS			0x02
#define RTP_MIDI_SYSEX_COMMON_RT_SC_COLOR_CHANGERS			0x03
#define RTP_MIDI_SYSEX_COMMON_RT_SC_STROBES				0x04
#define RTP_MIDI_SYSEX_COMMON_RT_SCL_LASERS				0x05
#define RTP_MIDI_SYSEX_COMMON_RT_SCL_CHASERS				0x06
#define RTP_MIDI_SYSEX_COMMON_RT_SC_SOUND				0x10
#define RTP_MIDI_SYSEX_COMMON_RT_SC_MUSIC				0x11
#define RTP_MIDI_SYSEX_COMMON_RT_SC_CD_PLAYERS				0x12
#define RTP_MIDI_SYSEX_COMMON_RT_SC_EPROM_PLAYBACK			0x13
#define RTP_MIDI_SYSEX_COMMON_RT_SCL_AUDIO_TAPE_MACHINE			0x14
#define RTP_MIDI_SYSEX_COMMON_RT_SC_INTERCOMS				0x15
#define RTP_MIDI_SYSEX_COMMON_RT_SC_AMPLIFIERS				0x16
#define RTP_MIDI_SYSEX_COMMON_RT_SC_AUDIO_EFFECTS			0x17
#define RTP_MIDI_SYSEX_COMMON_RT_SC_EQUALIZERS				0x18
#define RTP_MIDI_SYSEX_COMMON_RT_SC_MACHINERY				0x20
#define RTP_MIDI_SYSEX_COMMON_RT_SC_RIGGING				0x21
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FLYS				0x22
#define RTP_MIDI_SYSEX_COMMON_RT_SC_LIFTS				0x23
#define RTP_MIDI_SYSEX_COMMON_RT_SC_TURNTABLES				0x24
#define RTP_MIDI_SYSEX_COMMON_RT_SC_TRUSSES				0x25
#define RTP_MIDI_SYSEX_COMMON_RT_SC_ROBOTS				0x26
#define RTP_MIDI_SYSEX_COMMON_RT_SC_ANIMATION				0x27
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FLOATS				0x28
#define RTP_MIDI_SYSEX_COMMON_RT_SC_BREAKAWAYS				0x29
#define RTP_MIDI_SYSEX_COMMON_RT_SC_BARGES				0x2a
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO				0x30
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_TAPE_MACHINES			0x31
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_CASSETTE_MACHINES		0x32
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_DISC_PLAYERS			0x33
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_SWITCHERS			0x34
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_EFFECT			0x35
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_CHARACTER_GENERATORS		0x36
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_STIL_STORES			0x37
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_MONITORS			0x38
#define RTP_MIDI_SYSEX_COMMON_RT_SC_PROJECTION				0x40
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FILM_PROJECTORS			0x41
#define RTP_MIDI_SYSEX_COMMON_RT_SC_SLIDE_PROJECTORS			0x42
#define RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_PROJECTORS			0x43
#define RTP_MIDI_SYSEX_COMMON_RT_SC_DISSOLVERS				0x44
#define RTP_MIDI_SYSEX_COMMON_RT_SC_SHUTTER_CONTROLS			0x45
#define RTP_MIDI_SYSEX_COMMON_RT_SC_PROCESS_CONTROL			0x50
#define RTP_MIDI_SYSEX_COMMON_RT_SC_HYDRAULIC_OIL			0x51
#define RTP_MIDI_SYSEX_COMMON_RT_SC_H2O					0x52
#define RTP_MIDI_SYSEX_COMMON_RT_SC_CO2					0x53
#define RTP_MIDI_SYSEX_COMMON_RT_SC_COMPRESSED_AIR			0x54
#define RTP_MIDI_SYSEX_COMMON_RT_SC_NATURAL_GAS				0x55
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FOG					0x56
#define RTP_MIDI_SYSEX_COMMON_RT_SC_SMOKE				0x57
#define RTP_MIDI_SYSEX_COMMON_RT_SC_CRACKED_HAZE			0x58
#define RTP_MIDI_SYSEX_COMMON_RT_SC_PYRO				0x60
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FIREWORKS				0x61
#define RTP_MIDI_SYSEX_COMMON_RT_SC_EXPLOSIONS				0x62
#define RTP_MIDI_SYSEX_COMMON_RT_SC_FLAME				0x63
#define RTP_MIDI_SYSEX_COMMON_RT_SC_SMOKE_POTS				0x64
#define RTP_MIDI_SYSEX_COMMON_RT_SC_ALL_TYPES				0x7f

#define RTP_MIDI_SYSEX_COMMON_RT_NT_BAR_NUMBER				0x01
#define RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_IMMEDIATE		0x02
#define RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_DELAYED		0x42

#define RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_VOLUME			0x01
#define RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_BALANCE			0x02

#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_SPECIAL			0x00
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_PUNCH_IN_POINTS		0x01
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_PUNCH_OUT_POINTS		0x02
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_03			0x03
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_04			0x04
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_START_POINTS		0x05
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_STOP_POINTS		0x06
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_START_POINTS_ADD		0x07
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_STOP_POINTS_ADD		0x08
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_09			0x09
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_0A			0x0a
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_CUE_POINTS			0x0b
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_CUE_POINTS_ADD			0x0c
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_0D			0x0d
#define RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_NAME_ADD			0x0e

/* TODO: Add MMC Commands */

/* TODO: Add MMC Responses */



#define RTP_MIDI_COMMON_MTC_QF_FRAME_LS_NIBBLE				0x00
#define RTP_MIDI_COMMON_MTC_QF_FRAME_MS_NIBBLE				0x01
#define RTP_MIDI_COMMON_MTC_QF_SECONDS_LS_NIBBLE			0x02
#define RTP_MIDI_COMMON_MTC_QF_SECONDS_MS_NIBBLE			0x03
#define RTP_MIDI_COMMON_MTC_QF_MINUTES_LS_NIBBLE			0x04
#define RTP_MIDI_COMMON_MTC_QF_MINUTES_MS_NIBBLE			0x05
#define RTP_MIDI_COMMON_MTC_QF_HOURS_LS_NIBBLE				0x06
#define RTP_MIDI_COMMON_MTC_QF_HOURS_MS_NIBBLE				0x07








#define RTP_MIDI_TREE_NAME_COMMAND					"Command Section"
#define RTP_MIDI_TREE_NAME_COMMAND_SYSEX_MANU				"Manufacturer specific data"
#define RTP_MIDI_TREE_NAME_COMMAND_SYSEX_EDU				"Educational data"
#define RTP_MIDI_TREE_NAME_JOURNAL					"Journal Section"
#define RTP_MIDI_TREE_NAME_SYSTEM_JOURNAL				"System-Journal"
#define RTP_MIDI_TREE_NAME_SYSTEM_CHAPTERS				"System-Chapters"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_J				"System Common 0xF4"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_K				"System Common 0xF5"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_Y				"System Common 0xF9"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_Z				"System Common 0xFD"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_Q					"Sequencer State Commands"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_F					"MIDI Time Code Tape Position"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_X					"System Exclusive"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_X_DATA				"Data field (sysex commands)"
#define RTP_MIDI_TREE_NAME_SJ_CHAPTER_X_INVALID_DATA			"Data field (invalid sysex commands)"
#define RTP_MIDI_TREE_NAME_CHANNEL_JOURNAL				"Channel-Journal"
#define RTP_MIDI_TREE_NAME_CHANNEL_CHAPTERS				"Channel-Chapters"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_P					"Program Change"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_C					"Control Change"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_C_LOGLIST				"Log List"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_C_LOGITEM				"Log Item"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_M					"Parameter System"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_M_LOGLIST				"Log List"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_M_LOGITEM				"Log Item"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_M_LOG_COUNT			"Count"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_W					"Pitch Wheel"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_N					"Note on/off"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_LOGLIST				"Log List"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_LOGITEM				"Note On"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_OCTETS				"Offbit Octets"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_E					"Note Command Extras"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGLIST				"Log List"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGITEM1			"Note Off"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGITEM2			"Note On/Off"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_A					"Poly Aftertouch"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_A_LOGLIST				"Log List"
#define RTP_MIDI_TREE_NAME_CJ_CHAPTER_A_LOGITEM				"Pressure"

/* used to mask the most significant bit, which flags the start of a new midi-command! */
#define RTP_MIDI_COMMAND_STATUS_FLAG					0x80

/* used to mask the lower 7 bits of the single octets that make up the delta-time */
#define RTP_MIDI_DELTA_TIME_OCTET_MASK					0x7f
/* used to mask the most significant bit, which flags the extension of the delta-time */
#define RTP_MIDI_DELTA_TIME_EXTENSION					0x80

#define RTP_MIDI_CS_FLAG_B						0x80
#define RTP_MIDI_CS_FLAG_J						0x40
#define RTP_MIDI_CS_FLAG_Z						0x20
#define RTP_MIDI_CS_FLAG_P						0x10
#define RTP_MIDI_CS_MASK_SHORTLEN					0x0f
#define RTP_MIDI_CS_MASK_LONGLEN					0x0fff

#define RTP_MIDI_CJ_CHAPTER_M_FLAG_J					0x80
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_K					0x40
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_L					0x20
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_M					0x10
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_N					0x08
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_T					0x04
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_V					0x02
#define RTP_MIDI_CJ_CHAPTER_M_FLAG_R					0x01

#define RTP_MIDI_JS_FLAG_S				0x80
#define RTP_MIDI_JS_FLAG_Y				0x40
#define RTP_MIDI_JS_FLAG_A				0x20
#define RTP_MIDI_JS_FLAG_H				0x10
#define RTP_MIDI_JS_MASK_TOTALCHANNELS			0x0f

#define RTP_MIDI_SJ_FLAG_S				0x8000
#define RTP_MIDI_SJ_FLAG_D				0x4000
#define RTP_MIDI_SJ_FLAG_V				0x2000
#define RTP_MIDI_SJ_FLAG_Q				0x1000
#define RTP_MIDI_SJ_FLAG_F				0x0800
#define RTP_MIDI_SJ_FLAG_X				0x0400
#define RTP_MIDI_SJ_MASK_LENGTH				0x03ff

#define RTP_MIDI_SJ_CHAPTER_D_FLAG_S			0x80
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_B			0x40
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_G			0x20
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_H			0x10
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_J			0x08
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_K			0x04
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_Y			0x02
#define RTP_MIDI_SJ_CHAPTER_D_FLAG_Z			0x01

#define RTP_MIDI_SJ_CHAPTER_D_RESET_FLAG_S		0x80
#define RTP_MIDI_SJ_CHAPTER_D_RESET_COUNT		0x7f
#define RTP_MIDI_SJ_CHAPTER_D_TUNE_FLAG_S		0x80
#define RTP_MIDI_SJ_CHAPTER_D_TUNE_COUNT		0x7f
#define RTP_MIDI_SJ_CHAPTER_D_SONG_SEL_FLAG_S		0x80
#define RTP_MIDI_SJ_CHAPTER_D_SONG_SEL_VALUE		0x7f

#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_S		0x8000
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_C		0x4000
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_V		0x2000
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_L		0x1000
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_DSZ		0x0c00
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_LENGTH	0x03ff
#define RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_COUNT		0xff

#define RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_S		0x80
#define RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_C		0x40
#define RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_L		0x20
#define RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_LENGTH	0x1f
#define RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_COUNT	0xff

#define RTP_MIDI_SJ_CHAPTER_Q_FLAG_S			0x80
#define RTP_MIDI_SJ_CHAPTER_Q_FLAG_N			0x40
#define RTP_MIDI_SJ_CHAPTER_Q_FLAG_D			0x20
#define RTP_MIDI_SJ_CHAPTER_Q_FLAG_C			0x10
#define RTP_MIDI_SJ_CHAPTER_Q_FLAG_T			0x80
#define RTP_MIDI_SJ_CHAPTER_Q_MASK_TOP			0x07
#define RTP_MIDI_SJ_CHAPTER_Q_MASK_CLOCK		0x07ffff
#define RTP_MIDI_SJ_CHAPTER_Q_MASK_TIMETOOLS		0xffffff

#define RTP_MIDI_SJ_CHAPTER_F_FLAG_S			0x80
#define RTP_MIDI_SJ_CHAPTER_F_FLAG_C			0x40
#define RTP_MIDI_SJ_CHAPTER_F_FLAG_P			0x20
#define RTP_MIDI_SJ_CHAPTER_F_FLAG_Q			0x10
#define RTP_MIDI_SJ_CHAPTER_F_FLAG_D			0x08
#define RTP_MIDI_SJ_CHAPTER_F_MASK_POINT		0x07
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT0			0xf0000000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT1			0x0f000000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT2			0x00f00000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT3			0x000f0000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT4			0x0000f000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT5			0x00000f00
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT6			0x000000f0
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MT7			0x0000000f
#define RTP_MIDI_SJ_CHAPTER_F_MASK_HR			0xff000000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_MN			0x00ff0000
#define RTP_MIDI_SJ_CHAPTER_F_MASK_SC			0x0000ff00
#define RTP_MIDI_SJ_CHAPTER_F_MASK_FR			0x000000ff

#define RTP_MIDI_SJ_CHAPTER_X_FLAG_S			0x80
#define RTP_MIDI_SJ_CHAPTER_X_FLAG_T			0x40
#define RTP_MIDI_SJ_CHAPTER_X_FLAG_C			0x20
#define RTP_MIDI_SJ_CHAPTER_X_FLAG_F			0x10
#define RTP_MIDI_SJ_CHAPTER_X_FLAG_D			0x08
#define RTP_MIDI_SJ_CHAPTER_X_FLAG_L			0x04
#define RTP_MIDI_SJ_CHAPTER_X_MASK_STA			0x03
#define RTP_MIDI_SJ_CHAPTER_X_MASK_TCOUNT		0xff
#define RTP_MIDI_SJ_CHAPTER_X_MASK_COUNT		0xff

#define RTP_MIDI_CJ_FLAG_S				0x800000
#define RTP_MIDI_CJ_FLAG_H				0x040000
#define RTP_MIDI_CJ_FLAG_P				0x000080
#define RTP_MIDI_CJ_FLAG_C				0x000040
#define RTP_MIDI_CJ_FLAG_M				0x000020
#define RTP_MIDI_CJ_FLAG_W				0x000010
#define RTP_MIDI_CJ_FLAG_N				0x000008
#define RTP_MIDI_CJ_FLAG_E				0x000004
#define RTP_MIDI_CJ_FLAG_T				0x000002
#define RTP_MIDI_CJ_FLAG_A				0x000001
#define RTP_MIDI_CJ_MASK_LENGTH				0x03ff00
#define RTP_MIDI_CJ_MASK_CHANNEL			0x780000
#define RTP_MIDI_CJ_CHANNEL_SHIFT			19

#define RTP_MIDI_CJ_CHAPTER_M_MASK_LENGTH		0x3f

#define RTP_MIDI_CJ_CHAPTER_N_MASK_LENGTH		0x7f00
#define RTP_MIDI_CJ_CHAPTER_N_MASK_LOW			0x00f0
#define RTP_MIDI_CJ_CHAPTER_N_MASK_HIGH			0x000f

#define RTP_MIDI_CJ_CHAPTER_E_MASK_LENGTH		0x7f
#define RTP_MIDI_CJ_CHAPTER_A_MASK_LENGTH		0x7f





static const char rtp_midi_unknown_value_dec[] =			"unknown value: %d";
static const char rtp_midi_unknown_value_hex[] =			"unknown value: 0x%x";

static const value_string rtp_midi_note_values[] = {
	{ 0,								"C-1" },
	{ 1,								"C#-1" },
	{ 2,								"D-1" },
	{ 3,								"D#-1" },
	{ 4,								"E-1" },
	{ 5,								"F-1" },
	{ 6,								"F#-1" },
	{ 7,								"G-1" },
	{ 8,								"G#-1" },
	{ 9,								"A-1" },
	{ 10,								"A#-1" },
	{ 11,								"B-1" },
	{ 12,								"C0" },
	{ 13,								"C#0" },
	{ 14,								"D0" },
	{ 15,								"D#0" },
	{ 16,								"E0" },
	{ 17,								"F0" },
	{ 18,								"F#0" },
	{ 19,								"G0" },
	{ 20,								"G#0" },
	{ 21,								"A0" },
	{ 22,								"A#0" },
	{ 23,								"B0" },
	{ 24,								"C1" },
	{ 25,								"C#1" },
	{ 26,								"D1" },
	{ 27,								"D#1" },
	{ 28,								"E1" },
	{ 29,								"F1" },
	{ 30,								"F#1" },
	{ 31,								"G1" },
	{ 32,								"G#1" },
	{ 33,								"A1" },
	{ 34,								"A#1" },
	{ 35,								"B1" },
	{ 36,								"C2" },
	{ 37,								"C#2" },
	{ 38,								"D2" },
	{ 39,								"D#2" },
	{ 40,								"E2" },
	{ 41,								"F2" },
	{ 42,								"F#2" },
	{ 43,								"G2" },
	{ 44,								"G#2" },
	{ 45,								"A2" },
	{ 46,								"A#2" },
	{ 47,								"B2" },
	{ 48,								"C3" },
	{ 49,								"C#3" },
	{ 50,								"D3" },
	{ 51,								"D#3" },
	{ 52,								"E3" },
	{ 53,								"F3" },
	{ 54,								"F#3" },
	{ 55,								"G3" },
	{ 56,								"G#3" },
	{ 57,								"A3" },
	{ 58,								"A#3" },
	{ 59,								"B3" },
	{ 60,								"C4" },
	{ 61,								"C#4" },
	{ 62,								"D4" },
	{ 63,								"D#4" },
	{ 64,								"E4" },
	{ 65,								"F4" },
	{ 66,								"F#4" },
	{ 67,								"G4" },
	{ 68,								"G#4" },
	{ 69,								"A4" },
	{ 70,								"A#4" },
	{ 71,								"B4" },
	{ 72,								"C5" },
	{ 73,								"C#5" },
	{ 74,								"D5" },
	{ 75,								"D#5" },
	{ 76,								"E5" },
	{ 77,								"F5" },
	{ 78,								"F#5" },
	{ 79,								"G5" },
	{ 80,								"G#5" },
	{ 81,								"A5" },
	{ 82,								"A#5" },
	{ 83,								"B5" },
	{ 84,								"C6" },
	{ 85,								"C#6" },
	{ 86,								"D6" },
	{ 87,								"D#6" },
	{ 88,								"E6" },
	{ 89,								"F6" },
	{ 90,								"F#6" },
	{ 91,								"G6" },
	{ 92,								"G#6" },
	{ 93,								"A6" },
	{ 94,								"A#6" },
	{ 95,								"B6" },
	{ 96,								"C7" },
	{ 97,								"C#7" },
	{ 98,								"D7" },
	{ 99,								"D#7" },
	{ 100,								"E7" },
	{ 101,								"F7" },
	{ 102,								"F#7" },
	{ 103,								"G7" },
	{ 104,								"G#7" },
	{ 105,								"A7" },
	{ 106,								"A#7" },
	{ 107,								"B7" },
	{ 108,								"C8" },
	{ 109,								"C#8" },
	{ 110,								"D8" },
	{ 111,								"D#8" },
	{ 112,								"E8" },
	{ 113,								"F8" },
	{ 114,								"F#8" },
	{ 115,								"G8" },
	{ 116,								"G#8" },
	{ 117,								"A8" },
	{ 118,								"A#8" },
	{ 119,								"B8" },
	{ 120,								"C9" },
	{ 121,								"C#9" },
	{ 122,								"D9" },
	{ 123,								"D#9" },
	{ 124,								"E9" },
	{ 125,								"F9" },
	{ 126,								"F#9" },
	{ 127,								"G9" },
	{ 0,								NULL }
};
static value_string_ext rtp_midi_note_values_ext = VALUE_STRING_EXT_INIT(rtp_midi_note_values);



static const value_string rtp_midi_controller_values[] = {
	{ RTP_MIDI_CTRL_BANK_SELECT_MSB,				"Bank Select (msb)" },
	{ RTP_MIDI_CTRL_MODULATION_WHEEL_OR_LEVER_MSB,			"Modulation Wheel Or Lever (msb)" },
	{ RTP_MIDI_CTRL_BREATH_CONTROLLER_MSB,				"Breath Controller (msb)" },
	{ RTP_MIDI_CTRL_FOOT_CONTROLLER_MSB,				"Foot Controller (msb)" },
	{ RTP_MIDI_CTRL_PORTAMENTO_TIME_MSB,				"Portamento Time (msb)" },
	{ RTP_MIDI_CTRL_DATA_ENTRY_MSB,					"Data Entry (msb)" },
	{ RTP_MIDI_CTRL_CHANNEL_VOLUME_MSB,				"Channel Volume (msb)" },
	{ RTP_MIDI_CTRL_BALANCE_MSB,					"Balance (msb)" },
	{ RTP_MIDI_CTRL_PAN_MSB,					"Pan (msb)" },
	{ RTP_MIDI_CTRL_EXPRESSION_CONTROLLER_MSB,			"Expression Controller (msb)" },
	{ RTP_MIDI_CTRL_EFFECT_CONTROL_1_MSB,				"Effect Control 1 (msb)" },
	{ RTP_MIDI_CTRL_EFFECT_CONTROL_2_MSB,				"Effect Control 2 (msb)" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_1_MSB,		"General Purpose Controller 1 (msb)" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_2_MSB,		"General Purpose Controller 2 (msb)" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_3_MSB,		"General Purpose Controller 3 (msb)" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_4_MSB,		"General Purpose Controller 4 (msb)" },
	{ RTP_MIDI_CTRL_BANK_SELECT_LSB,				"Bank Select (lsb)" },
	{ RTP_MIDI_CTRL_MODULATION_WHEEL_OR_LEVER_LSB,			"Modulation Wheel Or Lever (lsb)" },
	{ RTP_MIDI_CTRL_BREATH_CONTROLLER_LSB,				"Breath Controller (lsb)" },
	{ RTP_MIDI_CTRL_FOOT_CONTROLLER_LSB,				"Foot Controller (lsb)" },
	{ RTP_MIDI_CTRL_PORTAMENTO_TIME_LSB,				"Portamento Time (lsb)" },
	{ RTP_MIDI_CTRL_DATA_ENTRY_LSB,					"Data Entry (lsb)" },
	{ RTP_MIDI_CTRL_CHANNEL_VOLUME_LSB,				"Volume (lsb)" },
	{ RTP_MIDI_CTRL_BALANCE_LSB,					"Balance (lsb)" },
	{ RTP_MIDI_CTRL_PAN_LSB,					"Pan (lsb)" },
	{ RTP_MIDI_CTRL_EXPRESSION_CONTROLLER_LSB,			"Expression Controller (lsb)" },
	{ RTP_MIDI_CTRL_EFFECT_CONTROL_1_LSB,				"Effect Control 1 (lsb)" },
	{ RTP_MIDI_CTRL_EFFECT_CONTROL_2_LSB,				"Effect Control 2 (lsb)" },
	{ RTP_MIDI_CTRL_DAMPER_PEDAL,					"Damper Pedal" },
	{ RTP_MIDI_CTRL_PORTAMENTO_ON_OFF,				"Portamento On/Off" },
	{ RTP_MIDI_CTRL_SUSTENUTO,					"Sustenuto" },
	{ RTP_MIDI_CTRL_SOFT_PEDAL,					"Soft Pedal" },
	{ RTP_MIDI_CTRL_LEGATO_FOOTSWITCH,				"Legato Footswitch" },
	{ RTP_MIDI_CTRL_HOLD_2,						"Hold 2" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_1,				"Sound Controller 1 (default: Sound Variation)" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_2,				"Sound Controller 2 (default: Timbre/Harmonic Intensity)" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_3,				"Sound Controller 3 (default: Release Time)" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_4,				"Sound Controller 4 (default: Attack Time)" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_5,				"Sound Controller 5 (default: Brightness)" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_6,				"Sound Controller 6" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_7,				"Sound Controller 7" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_8,				"Sound Controller 8" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_9,				"Sound Controller 9" },
	{ RTP_MIDI_CTRL_SOUND_CONTROLLER_10,				"Sound Controller 10" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_5,			"General Purpose Controller 5" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_6,			"General Purpose Controller 6" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_7,			"General Purpose Controller 7" },
	{ RTP_MIDI_CTRL_GENERAL_PURPOSE_CONTROLLER_8,			"General Purpose Controller 8" },
	{ RTP_MIDI_CTRL_PORTAMENTO_CONTROL,				"Portamento Control" },
	{ RTP_MIDI_CTRL_EFFECTS_1_DEPTH,				"Effects 1 Depth (formerly: External Effects Depth)" },
	{ RTP_MIDI_CTRL_EFFECTS_2_DEPTH,				"Effects 2 Depth (formerly: Tremolo Depth)" },
	{ RTP_MIDI_CTRL_EFFECTS_3_DEPTH,				"Effects 3 Depth (formerly: Chorus Depth)" },
	{ RTP_MIDI_CTRL_EFFECTS_4_DEPTH,				"Effects 4 Depth (formerly: Celeste (Detune) Depth)" },
	{ RTP_MIDI_CTRL_EFFECTS_5_DEPTH,				"Effects 5 Depth (formerly: Phaser Depth)" },
	{ RTP_MIDI_CTRL_DATA_INCREMENT,					"Data Increment" },
	{ RTP_MIDI_CTRL_DATA_DECREMENT,					"Data Decrement" },
	{ RTP_MIDI_CTRL_NON_REGISTERED_PARAM_LSB,			"Non-Registered Parameter (lsb)" },
	{ RTP_MIDI_CTRL_NON_REGISTERED_PARAM_MSB,			"Non-Registered Parameter (msb)" },
	{ RTP_MIDI_CTRL_REGISTERED_PARAM_LSB,				"Registered Parameter (lsb)" },
	{ RTP_MIDI_CTRL_REGISTERED_PARAM_MSB,				"Registered Parameter (msb)" },
	{ RTP_MIDI_CTRL_ALL_SOUND_OFF,					"All Sound Off" },
	{ RTP_MIDI_CTRL_RESET_ALL_CONTROLLERS,				"Reset All Controllers" },
	{ RTP_MIDI_CTRL_LOCAL_CONTROL_ON_OFF,				"Local Control" },
	{ RTP_MIDI_CTRL_ALL_NOTES_OFF,					"All Notes Off" },
	{ RTP_MIDI_CTRL_OMNI_MODE_OFF,					"Omni Mode Off" },
	{ RTP_MIDI_CTRL_OMNI_MODE_ON,					"Omni Mode On" },
	{ RTP_MIDI_CTRL_MONO_MODE_ON,					"Mono Mode On" },
	{ RTP_MIDI_CTRL_POLY_MODE_ON,					"Poly Mode On" },
	{ 0,								NULL }
};
static value_string_ext rtp_midi_controller_values_ext = VALUE_STRING_EXT_INIT(rtp_midi_controller_values);


static const value_string rtp_midi_manu_short_values[] = {
	{ RTP_MIDI_MANU_SHORT_ISLONG,					"< long id >" },
	{ RTP_MIDI_MANU_SHORT_SEQUENTIAL_CIRCUITS,			"Sequential Circuits" },
	{ RTP_MIDI_MANU_SHORT_BIG_BRIAR_IDP,				"Big Briar / IDP" },
	{ RTP_MIDI_MANU_SHORT_OCTAVE_PLATEAU_VOYETRA,			"Voyetra / Octave-Plateau" },
	{ RTP_MIDI_MANU_SHORT_MOOG,					"Moog" },
	{ RTP_MIDI_MANU_SHORT_PASSPORT_DESIGNS,				"Passport Designs" },
	{ RTP_MIDI_MANU_SHORT_LEXICON,					"Lexicon" },
	{ RTP_MIDI_MANU_SHORT_KURZWEIL,					"Kurzweil" },
	{ RTP_MIDI_MANU_SHORT_FENDER,					"Fender" },
	{ RTP_MIDI_MANU_SHORT_GULBRANSEN,				"Gulbransen" },
	{ RTP_MIDI_MANU_SHORT_AKG_ACOUSTICS,				"AKG Acoustics" },
	{ RTP_MIDI_MANU_SHORT_VOYCE_MUSIC,				"Voyce Music" },
	{ RTP_MIDI_MANU_SHORT_WAVEFRAME,				"Waveframe Corp" },
	{ RTP_MIDI_MANU_SHORT_ADA_SIGNAL_PROCESSORS,			"ADA Signal Processors" },
	{ RTP_MIDI_MANU_SHORT_GARFIELD_ELECTRONICS,			"Garfield Electronics" },
	{ RTP_MIDI_MANU_SHORT_ENSONIQ,					"Ensoniq" },
	{ RTP_MIDI_MANU_SHORT_OBERHEIM_GIBSON_LABS,			"Oberheim / Gibson Labs" },
	{ RTP_MIDI_MANU_SHORT_APPLE_COMPUTERS,				"Apple Computer" },
	{ RTP_MIDI_MANU_SHORT_GREY_MATTER_RESPONSE,			"Grey Matter Response / Simmons" },
	{ RTP_MIDI_MANU_SHORT_DIGIDESIGN,				"DigiDesign" },
	{ RTP_MIDI_MANU_SHORT_PALMTREE_INSTRUMENTS,			"Fairlight / Palmtree Instruments" },
	{ RTP_MIDI_MANU_SHORT_JL_COOPER,				"JL Cooper" },
	{ RTP_MIDI_MANU_SHORT_LOWREY,					"Lowrey" },
	{ RTP_MIDI_MANU_SHORT_LINN_ADAMS_SMITH,				"Linn / Adams-Smith" },
	{ RTP_MIDI_MANU_SHORT_EMU_SYSTEMS,				"Emu Systems" },
	{ RTP_MIDI_MANU_SHORT_HARMONY_SYSTEMS,				"Harmony Systems" },
	{ RTP_MIDI_MANU_SHORT_ART,					"ART" },
	{ RTP_MIDI_MANU_SHORT_BALDWIN,					"Baldwin" },
	{ RTP_MIDI_MANU_SHORT_EVENTIDE,					"Eventide" },
	{ RTP_MIDI_MANU_SHORT_INVENTRONICS,				"Inventronics" },
	{ RTP_MIDI_MANU_SHORT_CLARITY,					"Clarity" },
	{ RTP_MIDI_MANU_SHORT_PASSAC,					"Passac" },
	{ RTP_MIDI_MANU_SHORT_SIEL,					"S.I.E.L." },
	{ RTP_MIDI_MANU_SHORT_SYNTHE_AXE,				"SyntheAxe" },
	{ RTP_MIDI_MANU_SHORT_STEPP,					"Stepp" },
	{ RTP_MIDI_MANU_SHORT_HOHNER,					"Hohner" },
	{ RTP_MIDI_MANU_SHORT_CRUMAR_TWISTER,				"Crumar / Twister" },
	{ RTP_MIDI_MANU_SHORT_SOLTON,					"Solton" },
	{ RTP_MIDI_MANU_SHORT_JELLINGHAUS_MS,				"Jellinghaus MS" },
	{ RTP_MIDI_MANU_SHORT_SOUTHWORK_MUSIC_SYSTEMS,			"Southworks Music Systems" },
	{ RTP_MIDI_MANU_SHORT_PPG,					"PPG" },
	{ RTP_MIDI_MANU_SHORT_JEN,					"JEN" },
	{ RTP_MIDI_MANU_SHORT_SSL,					"SSL (Solid States Logic)" },
	{ RTP_MIDI_MANU_SHORT_AUDIO_VERITRIEB,				"Audio Veritrieb" },
	{ RTP_MIDI_MANU_SHORT_NEVE_HINTON_INSTRUMENTS,			"Neve / Hinton Instruments" },
	{ RTP_MIDI_MANU_SHORT_SOUNDTRACS,				"Soundtracs Ltd" },
	{ RTP_MIDI_MANU_SHORT_ELKA_GENERAL_MUSIC,			"Elka / General Music" },
	{ RTP_MIDI_MANU_SHORT_DYNACORD,					"Dynacord" },
	{ RTP_MIDI_MANU_SHORT_VISCOUNT,					"Viscount" },
	{ RTP_MIDI_MANU_SHORT_DRAWMER,					"Drawmer" },
	{ RTP_MIDI_MANU_SHORT_CLAVIA_DIGITAL_INSTRUMENTS,		"Clavia Digital Instruments" },
	{ RTP_MIDI_MANU_SHORT_AUDIO_ARCHITECTURE,			"Audio Architecture" },
	{ RTP_MIDI_MANU_SHORT_GENERAL_MUSIC_CORP,			"General Music Corp" },
	{ RTP_MIDI_MANU_SHORT_CHEETAH,					"Cheetah Marketing" },
	{ RTP_MIDI_MANU_SHORT_CTM,					"C.T.M" },
	{ RTP_MIDI_MANU_SHORT_SIMMONS_UK,				"Simmons UK" },
	{ RTP_MIDI_MANU_SHORT_SOUNDCRAFT_ELECTRONICS,			"Soundcraft Electronics" },
	{ RTP_MIDI_MANU_SHORT_STEINBERG_GMBH,				"Steinberg GmbH" },
	{ RTP_MIDI_MANU_SHORT_WERSI,					"Wersi" },
	{ RTP_MIDI_MANU_SHORT_AVAB_ELEKTRONIK_AB,			"Avab Electronik Ab" },
	{ RTP_MIDI_MANU_SHORT_DIGIGRAM,					"Digigram" },
	{ RTP_MIDI_MANU_SHORT_WALDORF,					"Waldorf Electronics" },
	{ RTP_MIDI_MANU_SHORT_QUASIMIDI,				"Quasimidi" },
	{ RTP_MIDI_MANU_SHORT_KAWAI,					"Kawai" },
	{ RTP_MIDI_MANU_SHORT_ROLAND,					"Roland" },
	{ RTP_MIDI_MANU_SHORT_KORG,					"Korg" },
	{ RTP_MIDI_MANU_SHORT_YAMAHA,					"Yamaha" },
	{ RTP_MIDI_MANU_SHORT_CASIO,					"Casio" },
	{ RTP_MIDI_MANU_SHORT_MORIDAIRA,				"Moridaira" },
	{ RTP_MIDI_MANU_SHORT_KAMIYA_STUDIO,				"Kamiya Studio" },
	{ RTP_MIDI_MANU_SHORT_AKAI,					"Akai" },
	{ RTP_MIDI_MANU_SHORT_JAPAN_VICTOR,				"Japan Victor" },
	{ RTP_MIDI_MANU_SHORT_MEISOSHA,					"Meisosha" },
	{ RTP_MIDI_MANU_SHORT_HOSHINO_GAKKI,				"Hoshino Gakki" },
	{ RTP_MIDI_MANU_SHORT_FUJITSU,					"Fujitsu" },
	{ RTP_MIDI_MANU_SHORT_SONY,					"Sony" },
	{ RTP_MIDI_MANU_SHORT_NISSHIN_ONPA,				"Nisshin Onpa" },
	{ RTP_MIDI_MANU_SHORT_TEAC,					"TEAC" },
	{ RTP_MIDI_MANU_SHORT_MATSUSHITA_ELECTRIC,			"Matsushita Electric" },
	{ RTP_MIDI_MANU_SHORT_FOSTEX,					"Fostex" },
	{ RTP_MIDI_MANU_SHORT_ZOOM,					"Zoom" },
	{ RTP_MIDI_MANU_SHORT_MIDORI_ELECTRONICS,			"Midori Electronics" },
	{ RTP_MIDI_MANU_SHORT_MATSUSHITA_COMMUNICATION,			"Matsushita Communication" },
	{ RTP_MIDI_MANU_SHORT_SUZUKI,					"Suzuki" },
	{ RTP_MIDI_MANU_SHORT_FUJI,					"Fuji" },
	{ RTP_MIDI_MANU_SHORT_ACOUSTIC_TECHNICAL_LAB,			"Acoustic Technical Lab" },
	{ RTP_MIDI_MANU_SHORT_FAITH,					"Faith" },
	{ RTP_MIDI_MANU_SHORT_INTERNET_CORPORATION,			"Internet Corporation" },
	{ RTP_MIDI_MANU_SHORT_SEEKERS_CO,				"Seekers Co" },
	{ RTP_MIDI_MANU_SHORT_SD_CARD_ASSOCIATION,			"SD Card Assoc" },
	{ RTP_MIDI_MANU_SHORT_EDUCATIONAL_USE,				"Educational Use" },
	{ RTP_MIDI_MANU_SHORT_NON_REALTIME_UNIVERSAL,			"Non-Realtime Universal" },
	{ RTP_MIDI_MANU_SHORT_REALTIME_UNIVERSAL,			"Realtime Universal" },
	{ 0,								NULL }
};
static value_string_ext rtp_midi_manu_short_values_ext = VALUE_STRING_EXT_INIT(rtp_midi_manu_short_values);

static const value_string rtp_midi_manu_long_values[] = {
	/* North American Manufacturers */
	{ RTP_MIDI_MANU_LONG_TIME_WARNER_INTERACTIVE,			"Time Warner Interactive" },
	{ RTP_MIDI_MANU_LONG_ADVANCED_GRAVIS_COMP,			"Advanced Gravis Comp" },
	{ RTP_MIDI_MANU_LONG_MEDIA_VISION,				"Media Vision" },
	{ RTP_MIDI_MANU_LONG_DORNES_RESEARCH_GROUP,			"Dornes Research Group" },
	{ RTP_MIDI_MANU_LONG_KMUSE,					"K-Muse" },
	{ RTP_MIDI_MANU_LONG_STYPHER,					"Stypher" },
	{ RTP_MIDI_MANU_LONG_DIGITAL_MUSIC_CORPORATION,			"Digital Music Corporation" },
	{ RTP_MIDI_MANU_LONG_IOTA_SYSTEMS,				"IOTA Systems" },
	{ RTP_MIDI_MANU_LONG_NEW_ENGLAND_DIGITAL,			"New England Digital" },
	{ RTP_MIDI_MANU_LONG_ARTISYN,					"Artisyn" },
	{ RTP_MIDI_MANU_LONG_IVL_TECHNOLOGIES,				"IVL Technologies" },
	{ RTP_MIDI_MANU_LONG_SOUTHERN_MUSIC_SYSTEMS,			"Southern Music Systems" },
	{ RTP_MIDI_MANU_LONG_LAKE_BUTLER_SOUND_COMPANY,			"Lake Butler Sound Company" },
	{ RTP_MIDI_MANU_LONG_ALESIS,					"Alesis" },
	{ RTP_MIDI_MANU_LONG_SOUND_CREATION,				"Sound Creation" },
	{ RTP_MIDI_MANU_LONG_DOD_ELECTRONICS,				"DOD Electronics" },
	{ RTP_MIDI_MANU_LONG_STUDER_EDITECH,				"Studer-Editech" },
	{ RTP_MIDI_MANU_LONG_SONUS,					"Sonus" },
	{ RTP_MIDI_MANU_LONG_TEMPORAL_ACUITY_PRODUCTS,			"Temporal Acuity Products" },
	{ RTP_MIDI_MANU_LONG_PERFECT_FRETWORKS,				"Perfect Fretworks" },
	{ RTP_MIDI_MANU_LONG_KAT,					"KAT" },
	{ RTP_MIDI_MANU_LONG_OPCODE,					"Opcode" },
	{ RTP_MIDI_MANU_LONG_RANE_CORP,					"Rane Corp" },
	{ RTP_MIDI_MANU_LONG_SPATIAL_SOUND_ANADI_INC,			"Spatial Sound / Anadi Inc" },
	{ RTP_MIDI_MANU_LONG_KMX,					"KMX" },
	{ RTP_MIDI_MANU_LONG_ALLEN_AND_HEATH_BRENNEL,			"Allen & Heath Brenell" },
	{ RTP_MIDI_MANU_LONG_PEAVEY,					"Peavey" },
	{ RTP_MIDI_MANU_LONG_360_SYSTEMS,				"360 Systems" },
	{ RTP_MIDI_MANU_LONG_SPECTRUM_DESIGN_DEVELOPMENT,		"Spectrum Design and Development" },
	{ RTP_MIDI_MANU_LONG_MARQUIS_MUSIC,				"Marquis Music" },
	{ RTP_MIDI_MANU_LONG_ZETA_SYSTEMS,				"Zeta Systems" },
	{ RTP_MIDI_MANU_LONG_AXXES,					"Axxes" },
	{ RTP_MIDI_MANU_LONG_ORBAN,					"Orban" },
	{ RTP_MIDI_MANU_LONG_INDIAN_VALLEY,				"Indian Valley Mfg" },
	{ RTP_MIDI_MANU_LONG_TRITON,					"Triton" },
	{ RTP_MIDI_MANU_LONG_KTI,					"KTI" },
	{ RTP_MIDI_MANU_LONG_BREAKAWAY_TECHNOLOGIES,			"Breakaway Technologies" },
	{ RTP_MIDI_MANU_LONG_CAE,					"CAE" },
	{ RTP_MIDI_MANU_LONG_HARRISON_SYSTEMS_INC,			"Harrison Systems Inc" },
	{ RTP_MIDI_MANU_LONG_FUTURE_LAB_MARK_KUO,			"Future Lab / Mark Kuo" },
	{ RTP_MIDI_MANU_LONG_ROCKTRON_CORP,				"Rocktron Corp" },
	{ RTP_MIDI_MANU_LONG_PIANODISC,					"PianoDisc" },
	{ RTP_MIDI_MANU_LONG_CANNON_RESEARCH_GROUP,			"Cannon Research Group" },
	{ RTP_MIDI_MANU_LONG_RODGERS_INSTRUMENTS_CORP,			"Rodgers Instrument Corp" },
	{ RTP_MIDI_MANU_LONG_BLUE_SKY_LOGIC,				"Blue Sky Logic" },
	{ RTP_MIDI_MANU_LONG_ENCORE_ELECTRONICS,			"Encore Electronics" },
	{ RTP_MIDI_MANU_LONG_UPTOWN,					"Uptown" },
	{ RTP_MIDI_MANU_LONG_VOCE,					"Voce" },
	{ RTP_MIDI_MANU_LONG_CTI_AUDIO_INC,				"CTI Audio, Inc / Music Intel Dev" },
	{ RTP_MIDI_MANU_LONG_SS_RESEARCH,				"S&S Research" },
	{ RTP_MIDI_MANU_LONG_BRODERBUND_SOFTWARE,			"Broderbund Software Inc" },
	{ RTP_MIDI_MANU_LONG_ALLEN_ORGAN_CO,				"Allen Organ Co" },
	{ RTP_MIDI_MANU_LONG_MUSIC_QUEST,				"Music Quest" },
	{ RTP_MIDI_MANU_LONG_APHEX,					"APHEX" },
	{ RTP_MIDI_MANU_LONG_GALLIEN_KRUEGER,				"Gallien Krueger" },
	{ RTP_MIDI_MANU_LONG_IBM,					"IBM" },
	{ RTP_MIDI_MANU_LONG_MARK_OF_THE_UNICORN,			"Mark of the Unicorn" },
	{ RTP_MIDI_MANU_LONG_HOTZ_INSTRUMENTS_TECH,			"Hotz Instruments Technologies" },
	{ RTP_MIDI_MANU_LONG_ETA_LIGHTING,				"ETA Lighting" },
	{ RTP_MIDI_MANU_LONG_NSI_CORPORATION,				"NSI Corporation" },
	{ RTP_MIDI_MANU_LONG_ADLIB_INC,					"PAd Lib, Inc" },
	{ RTP_MIDI_MANU_LONG_RICHMOND_SOUND_DESIGN,			"Richmond Sound Design" },
	{ RTP_MIDI_MANU_LONG_MICROSOFT,					"Microsoft" },
	{ RTP_MIDI_MANU_LONG_THE_SOFTWARE_TOOLWORKS,			"The Software Toolworks" },
	{ RTP_MIDI_MANU_LONG_RJMG_NICHE,				"RJMG / Niche" },
	{ RTP_MIDI_MANU_LONG_INTONE,					"Intone" },
	{ RTP_MIDI_MANU_LONG_ADVANCED_REMOTE_TECH,			"Advanced Remote Tech" },
	{ RTP_MIDI_MANU_LONG_GT_ELECTRONICS_GROOVE_TUBES,		"GT Electronics / Groove Tubes" },
	{ RTP_MIDI_MANU_LONG_INTERMIDI,					"InterMIDI" },
	{ RTP_MIDI_MANU_LONG_TIMELINE_VISTA,				"Timeline Vista" },
	{ RTP_MIDI_MANU_LONG_MESA_BOOGIE,				"Mesa Boogie" },
	{ RTP_MIDI_MANU_LONG_SEQUOIA_DEVELOPMENT,			"Sequoia Development" },
	{ RTP_MIDI_MANU_LONG_STUDIO_ELECTRONICS,			"Studio Electronics" },
	{ RTP_MIDI_MANU_LONG_EUPHONIX,					"Euphonix" },
	{ RTP_MIDI_MANU_LONG_INTERMIDI2,				"InterMIDI" },
	{ RTP_MIDI_MANU_LONG_MIDI_SOLUTIONS,				"MIDI Solutions" },
	{ RTP_MIDI_MANU_LONG_3DO_COMPANY,				"3DO Company" },
	{ RTP_MIDI_MANU_LONG_LIGHTWAVE_RESEARCH,			"Lightwave Research" },
	{ RTP_MIDI_MANU_LONG_MICROW,					"Micro-W" },
	{ RTP_MIDI_MANU_LONG_SPECTRAL_SYNTHESIS,			"Spectral Synthesis" },
	{ RTP_MIDI_MANU_LONG_LONE_WOLF,					"Lone Wolf" },
	{ RTP_MIDI_MANU_LONG_STUDIO_TECHNOLOGIES,			"Studio Technologies" },
	{ RTP_MIDI_MANU_LONG_PETERSON_EMP,				"Peterson EMP" },
	{ RTP_MIDI_MANU_LONG_ATARI,					"Atari" },
	{ RTP_MIDI_MANU_LONG_MARION_SYSTEMS,				"Marion Systems" },
	{ RTP_MIDI_MANU_LONG_DESIGN_EVENT,				"Design Event" },
	{ RTP_MIDI_MANU_LONG_WINJAMMER_SOFTWARE,			"Winjammer Software" },
	{ RTP_MIDI_MANU_LONG_ATT_BELL_LABS,				"AT&T Bell Labs" },
	{ RTP_MIDI_MANU_LONG_SYMETRIX,					"Symetrix" },
	{ RTP_MIDI_MANU_LONG_MIDI_THE_WORLD,				"MIDI the world" },
	{ RTP_MIDI_MANU_LONG_DESPER_PRODUCTS,				"Desper Products" },
	{ RTP_MIDI_MANU_LONG_MICROS_N_MIDI,				"Micros 'n MIDI" },
	{ RTP_MIDI_MANU_LONG_ACCORDIANS_INTL,				"Accodians Intl" },
	{ RTP_MIDI_MANU_LONG_EUPHONICS,					"EuPhonics" },
	{ RTP_MIDI_MANU_LONG_MUSONIX,					"Musonix" },
	{ RTP_MIDI_MANU_LONG_TURTLE_BEACH_SYSTEMS,			"Turtle Beach Systems" },
	{ RTP_MIDI_MANU_LONG_MACKIE_DESIGNS,				"Mackie Designs" },
	{ RTP_MIDI_MANU_LONG_COMPUSERVE,				"Compuserve" },
	{ RTP_MIDI_MANU_LONG_BES_TECHNOLOGIES,				"BES Technologies" },
	{ RTP_MIDI_MANU_LONG_QRS_MUSIC_ROLLS,				"QRS Music Rolls" },
	{ RTP_MIDI_MANU_LONG_P_G_MUSIC,					"P G Music" },
	{ RTP_MIDI_MANU_LONG_SIERRA_SEMICONDUCTOR,			"Sierra Semiconductor" },
	{ RTP_MIDI_MANU_LONG_EPIGRAF_AUDIO_VISUAL,			"EpiGraf Audio Visual" },
	{ RTP_MIDI_MANU_LONG_ELECTRONICS_DIVERSIFIED,			"Electronics Diversified" },
	{ RTP_MIDI_MANU_LONG_TUNE_1000,					"Tune 1000" },
	{ RTP_MIDI_MANU_LONG_ADVANCED_MICRO_DEVICES,			"Advanced Micro Devices" },
	{ RTP_MIDI_MANU_LONG_MEDIAMATION,				"Mediamation" },
	{ RTP_MIDI_MANU_LONG_SABINE_MUSIC,				"Sabine Music" },
	{ RTP_MIDI_MANU_LONG_WOOG_LABS,					"Woog Labs" },
	{ RTP_MIDI_MANU_LONG_MIRCOPOLIS,				"Micropolis" },
	{ RTP_MIDI_MANU_LONG_TA_HORNG_MUSICAL_INSTRUMENT,		"Ta Horng Musical Instrument" },
	{ RTP_MIDI_MANU_LONG_ETEK_LABS_FORTE_TECH,			"e-Tek Labs (Forte Tech)" },
	{ RTP_MIDI_MANU_LONG_ELECTRO_VOICE,				"Electro-Voice" },
	{ RTP_MIDI_MANU_LONG_MIDISOFT_CORPORATION,			"Midisoft Corporation" },
	{ RTP_MIDI_MANU_LONG_QSOUND_LABS,				"Q-Sound Labs" },
	{ RTP_MIDI_MANU_LONG_WESTREX,					"Westrex" },
	{ RTP_MIDI_MANU_LONG_NVIDIA,					"NVidia" },
	{ RTP_MIDI_MANU_LONG_ESS_TECHNOLOGY,				"ESS Technology" },
	{ RTP_MIDI_MANU_LONG_MEDIATRIX_PERIPHERALS,			"MediaTrix Peripherals" },
	{ RTP_MIDI_MANU_LONG_BROOKTREE_CORP,				"Brooktree Corp" },
	{ RTP_MIDI_MANU_LONG_OTARI_CORP,				"Otari Corp" },
	{ RTP_MIDI_MANU_LONG_KEY_ELECTRONICS,				"Key Electronics" },
	{ RTP_MIDI_MANU_LONG_SHURE_INCORPORATED,			"Shure Incorporated" },
	{ RTP_MIDI_MANU_LONG_AURA_SOUND,				"Aura Sound" },
	{ RTP_MIDI_MANU_LONG_CRYSTAL_SEMICONDUCTOR,			"Crystal Semiconductor" },
	{ RTP_MIDI_MANU_LONG_CONEXANT_ROCKWELL,				"Conexant (Rockwell)" },
	{ RTP_MIDI_MANU_LONG_SILICON_GRAPHICS,				"Silicon Graphics" },
	{ RTP_MIDI_MANU_LONG_MAUDIO_MIDIMAN,				"M-Audio (Midiman)" },
	{ RTP_MIDI_MANU_LONG_PRESONUS,					"PreSonus" },
	{ RTP_MIDI_MANU_LONG_TOPAZ_ENTERPRISES,				"Topaz Enterprises" },
	{ RTP_MIDI_MANU_LONG_CAST_LIGHTING,				"Cast Lighting" },
	{ RTP_MIDI_MANU_LONG_MICROSOFT_CONSUMER_DIVISION,		"Microsoft (Consumer Division)" },
	{ RTP_MIDI_MANU_LONG_SONIC_FOUNDRY,				"Sonic Foundry" },
	{ RTP_MIDI_MANU_LONG_LINE6_FAST_FORWARD,			"Line 6 (Fast Forward)" },
	{ RTP_MIDI_MANU_LONG_BEATNIK_INC,				"Beatnik Inc" },
	{ RTP_MIDI_MANU_LONG_VAN_KOEVERING_COMPANY,			"Van Koevering Company" },
	{ RTP_MIDI_MANU_LONG_ALTECH_SYSTEMS,				"DiAltech Systemsgit" },
	{ RTP_MIDI_MANU_LONG_S_S_RESEARCH,				"S&S Research" },
	{ RTP_MIDI_MANU_LONG_VLSI_TECHNOLOGY,				"VLSI Technology" },
	{ RTP_MIDI_MANU_LONG_CHROMATIC_RESEARCH,			"Chromatic Research" },
	{ RTP_MIDI_MANU_LONG_SAPPHIRE,					"Sapphire" },
	{ RTP_MIDI_MANU_LONG_IDRC,					"IDRC" },
	{ RTP_MIDI_MANU_LONG_JUSTONIC_TUNING,				"Justonic Tuning" },
	{ RTP_MIDI_MANU_LONG_TORCOMP_RESEARCH_INC,			"TorComp Research Inc" },
	{ RTP_MIDI_MANU_LONG_NEWTEK_INC,				"NewTek Inc" },
	{ RTP_MIDI_MANU_LONG_SOUND_SCULPTURE,				"Sound Sculpture" },
	{ RTP_MIDI_MANU_LONG_WALKER_TECHNICAL,				"Walker Technical" },
	{ RTP_MIDI_MANU_LONG_DIGITAL_HARMONY,				"Digital Harmony" },
	{ RTP_MIDI_MANU_LONG_INVISION_INTERACTIVE,			"InVision Interactive" },
	{ RTP_MIDI_MANU_LONG_TSQUARE_DESIGN,				"T-Square Design" },
	{ RTP_MIDI_MANU_LONG_NEMESYS_MUSIC_TECHNOLOGY,			"Nemesys Music Technology" },
	{ RTP_MIDI_MANU_LONG_DBX_PROFESSIONAL_HARMAN_INTL,		"DBX Professional (Harman Intl)" },
	{ RTP_MIDI_MANU_LONG_SYNDYNE_CORPORATION,			"Syndyne Corporation" },
	{ RTP_MIDI_MANU_LONG_BITHEADZ,					"Bitheadz" },
	{ RTP_MIDI_MANU_LONG_CAKEWALK_MUSIC_SOFTWARE,			"Cakewalk Music Software" },
	{ RTP_MIDI_MANU_LONG_ANALOG_DEVICES,				"Analog Devices" },
	{ RTP_MIDI_MANU_LONG_NATIONAL_SEMICONDUCTOR,			"National Semiconductor" },
	{ RTP_MIDI_MANU_LONG_BOOM_THEORY,				"Boom Theory" },
	{ RTP_MIDI_MANU_LONG_VIRTUAL_DSP_CORPORATION,			"Virtual DSP Corporation" },
	{ RTP_MIDI_MANU_LONG_ANTARES_SYSTEMS,				"Antares Systems" },
	{ RTP_MIDI_MANU_LONG_ANGEL_SOFTWARE,				"Angel Software" },
	{ RTP_MIDI_MANU_LONG_ST_LOUIS_MUSIC,				"St Louis Music" },
	{ RTP_MIDI_MANU_LONG_LYRRUS_DBA_GVOX,				"Lyrrus dba G-VOX" },
	{ RTP_MIDI_MANU_LONG_ASHLEY_AUDIO_INC,				"Ashley Audio Inc" },
	{ RTP_MIDI_MANU_LONG_VARILITE_INC,				"Vari-Lite Inc" },
	{ RTP_MIDI_MANU_LONG_SUMMIT_AUDIO_INC,				"Summit Audio Inc" },
	{ RTP_MIDI_MANU_LONG_AUREAL_SEMICONDUCTOR_INC,			"Aureal Semiconductor Inc" },
	{ RTP_MIDI_MANU_LONG_SEASOUND_LLC,				"SeaSound LLC" },
	{ RTP_MIDI_MANU_LONG_US_ROBOTICS,				"U.S. Robotics" },
	{ RTP_MIDI_MANU_LONG_AURISIS_RESEARCH,				"Aurisis Research" },
	{ RTP_MIDI_MANU_LONG_NEARFIELD_MULTIMEDIA,			"Nearfield Multimedia" },
	{ RTP_MIDI_MANU_LONG_FM7_INC,					"FM7 Inc" },
	{ RTP_MIDI_MANU_LONG_SWIVEL_SYSTEMS,				"Swivel Systems" },
	{ RTP_MIDI_MANU_LONG_HYPERACTIVE_AUDIO_SYSTEMS,			"Hyperactive Audio Systems" },
	{ RTP_MIDI_MANU_LONG_MIDILITE_CASTE_STUDIO_PROD,		"MidiLite (Castle Studios Productions)" },
	{ RTP_MIDI_MANU_LONG_RADIKAL_TECHNOLOGIES,			"Radikal Technologies" },
	{ RTP_MIDI_MANU_LONG_ROGER_LINN_DESIGN,				"Roger Linn Design" },
	{ RTP_MIDI_MANU_LONG_TCHELION_VOCAL_TECHNOLOGIES,		"TC-Helicon Vocal Technologies" },
	{ RTP_MIDI_MANU_LONG_EVENT_ELECTRONICS,				"Event Electronics" },
	{ RTP_MIDI_MANU_LONG_SONIC_NETWORK_INC,				"Sonic Network Inc" },
	{ RTP_MIDI_MANU_LONG_REALTIME_MUSIC_SOLUTIONS,			"Realtime Music Solutions" },
	{ RTP_MIDI_MANU_LONG_APOGEE_DIGITAL,				"Apogee Digital" },
	{ RTP_MIDI_MANU_LONG_CLASSICAL_ORGANS_INC,			"Classical Organs, Inc" },
	{ RTP_MIDI_MANU_LONG_MICROTOOLS_INC,				"Microtools Inc" },
	{ RTP_MIDI_MANU_LONG_NUMARK_INDUSTRIES,				"Numark Industries" },
	{ RTP_MIDI_MANU_LONG_FRONTIER_DESIGN_GROUP_LLC,			"Frontier Design Group LLC" },
	{ RTP_MIDI_MANU_LONG_RECORDARE_LLC,				"Recordare LLC" },
	{ RTP_MIDI_MANU_LONG_STARR_LABS,				"Starr Labs" },
	{ RTP_MIDI_MANU_LONG_VOYAGER_SOUND_INC,				"Voyager Sound Inc" },
	{ RTP_MIDI_MANU_LONG_MANIFOLD_LABS,				"Manifold Labs" },
	{ RTP_MIDI_MANU_LONG_AVIOM_INC,					"Aviom Inc" },
	{ RTP_MIDI_MANU_LONG_MIXMEISTER_TECHNOLOGY,			"Mixmeister Technology" },
	{ RTP_MIDI_MANU_LONG_NOTATION_SOFTWARE,				"Notation Software" },
	{ RTP_MIDI_MANU_LONG_MERCURIAL_COMMUNICATIONS,			"Mercurial Communications" },
	{ RTP_MIDI_MANU_LONG_WAVE_ARTS,					"Wave Arts" },
	{ RTP_MIDI_MANU_LONG_LOGIC_SEQUENCING_DEVICES,			"Logic Sequencing Devices" },
	{ RTP_MIDI_MANU_LONG_AXESS_ELECTRONICS,				"Axess Electronics" },
	{ RTP_MIDI_MANU_LONG_MUSE_RESEARCH,				"Muse Research" },
	{ RTP_MIDI_MANU_LONG_OPEN_LABS,					"Open Labs" },
	{ RTP_MIDI_MANU_LONG_GUILLEMOT_RD_INC,				"Guillemot R&D Inc" },
	{ RTP_MIDI_MANU_LONG_SAMSON_TECHNOLOGIES,			"Samson Technologies" },
	{ RTP_MIDI_MANU_LONG_ELECTRONIC_THEATRE_CONTROLS,		"Electronic Theatre Controls" },
	{ RTP_MIDI_MANU_LONG_RESEARCH_IN_MOTION,			"Research In Motion" },
	{ RTP_MIDI_MANU_LONG_MOBILEER,					"Mobileer" },
	{ RTP_MIDI_MANU_LONG_SYNTHOGY,					"Synthogy" },
	{ RTP_MIDI_MANU_LONG_LYNX_STUDIO_TECHNOLOGY_INC,		"Lynx Studio Technology Inc" },
	{ RTP_MIDI_MANU_LONG_DAMAGE_CONTROL_ENGINEERING,		"Damage Control Engineering LLC" },
	{ RTP_MIDI_MANU_LONG_YOST_ENGINEERING_INC,			"Yost Engineering Inc" },
	{ RTP_MIDI_MANU_LONG_BROOKS_FORSMAN_DESIGNS_LLC,		"Brooks & Forsman Designs LLC" },
	{ RTP_MIDI_MANU_LONG_MAGNEKEY,					"Magnekey" },
	{ RTP_MIDI_MANU_LONG_GARRITAN_CORP,				"Garritan Corp" },
	{ RTP_MIDI_MANU_LONG_PLOQUE_ART_ET_TECHNOLOGIE,			"Ploque Art et Technologie, Inc" },
	{ RTP_MIDI_MANU_LONG_RJM_MUSIC_TECHNOLOGY,			"RJM Music Technology" },
	{ RTP_MIDI_MANU_LONG_CUSTOM_SOLUTIONS_SOFTWARE,			"Custom Solutions Software" },
	{ RTP_MIDI_MANU_LONG_SONARCANA_LLC,				"Sonarcana LLC" },
	{ RTP_MIDI_MANU_LONG_CENTRANCE,					"Centrance" },
	{ RTP_MIDI_MANU_LONG_KESUMO_LLC,				"Kesumo LLC" },
	{ RTP_MIDI_MANU_LONG_STANTON,					"Stanton" },
	{ RTP_MIDI_MANU_LONG_LIVID_INSTRUMENTS,				"Livid Instruments" },
	{ RTP_MIDI_MANU_LONG_FIRST_ACT_745_MEDIA,			"First Act / 745 Media" },
	{ RTP_MIDI_MANU_LONG_PYGRAPHICS_INC,				"Pygraphics, Inc" },
	{ RTP_MIDI_MANU_LONG_PANADIGM_INNOVATIONS_LTD,			"Panadigm Innovations Ltd" },
	{ RTP_MIDI_MANU_LONG_AVEDIS_ZILDJIAN_CO,			"Avedis Zildjian Co" },
	{ RTP_MIDI_MANU_LONG_AUVITAL_MUSIC_CORP,			"Auvital Music Corp" },
	{ RTP_MIDI_MANU_LONG_INSPIRED_INSTRUMENTS_INC,			"Inspired Instruments Inc" },
	{ RTP_MIDI_MANU_LONG_CHRIS_GRIGG_DESIGNS,			"Chris Grigg Designs" },
	{ RTP_MIDI_MANU_LONG_SLATE_DIGITAL_LLC,				"Slate Digital LLC" },
	{ RTP_MIDI_MANU_LONG_MIXWARE,					"Mixware" },
	{ RTP_MIDI_MANU_LONG_SOCIAL_ENTROPY,				"Social Entropy" },
	{ RTP_MIDI_MANU_LONG_SOURCE_AUDIO_LLC,				"Source Audio LLC" },
	{ RTP_MIDI_MANU_LONG_RESERVED_016d,				"reserved" },
	{ RTP_MIDI_MANU_LONG_RESERVED_016e,				"reserved" },
	{ RTP_MIDI_MANU_LONG_RESERVED_016f,				"reserved" },
	{ RTP_MIDI_MANU_LONG_AMERICAN_AUDIO_DJ,				"American Audio/DJ" },
	{ RTP_MIDI_MANU_LONG_MEGA_CONTROL_SYSTEMS,			"Mega Control Systems" },
	{ RTP_MIDI_MANU_LONG_KILPATRICK_AUDIO,				"Kilpatrick Audio" },
	{ RTP_MIDI_MANU_LONG_IKINGDOM_CORP,				"iKingdom Corp" },
	{ RTP_MIDI_MANU_LONG_FRACTAL_AUDIO,				"Fractal Audio" },
	{ RTP_MIDI_MANU_LONG_NETLOGIC_MICROSYSTEMS,			"NetLogic Microsystems" },
	{ RTP_MIDI_MANU_LONG_MUSIC_COMPUTING,				"Music Computing" },
	{ RTP_MIDI_MANU_LONG_NEKTAR_TECHNOLOGY_INC,			"Nektar Technology Inc" },
	{ RTP_MIDI_MANU_LONG_ZENPH_SOUND_INNOVATIONS,			"Zenph Sound Innovations" },
	{ RTP_MIDI_MANU_LONG_DJTECHTOOLS_COM,				"DJTechTools.com" },
	{ RTP_MIDI_MANU_LONG_RESERVED_017a,				"reserved" },

	/* European Manufacturers */
	{ RTP_MIDI_MANU_LONG_DREAM,					"Dream" },
	{ RTP_MIDI_MANU_LONG_STRAND_LIGHTING,				"Strand Lighting" },
	{ RTP_MIDI_MANU_LONG_AMEK_SYSTEMS,				"Amek Systems" },
	{ RTP_MIDI_MANU_LONG_CASA_DI_RISPARMIO_DI_LORETO,		"Cas Di Risparmio Di Loreto" },
	{ RTP_MIDI_MANU_LONG_BOHM_ELECTRONIC,				"Bohm electronic" },
	{ RTP_MIDI_MANU_LONG_SYNTEC_DIGITAL_AUDIO,			"Syntec Digital Audio" },
	{ RTP_MIDI_MANU_LONG_TRIDENT_AUDIO,				"Trident Audio" },
	{ RTP_MIDI_MANU_LONG_REAL_WORLD_STUDIO,				"Real World Studio" },
	{ RTP_MIDI_MANU_LONG_EVOLUTION_SYNTHESIS,			"Evolution Synthesis" },
	{ RTP_MIDI_MANU_LONG_YES_TECHNOLOGY,				"Yes Technology" },
	{ RTP_MIDI_MANU_LONG_AUDIOMATICA,				"Audiomatica" },
	{ RTP_MIDI_MANU_LONG_BONTEMPI_FARFISA,				"Bontempi / Farfisa" },
	{ RTP_MIDI_MANU_LONG_FBT_ELETTRONICA,				"F.B.T. Elettronica" },
	{ RTP_MIDI_MANU_LONG_MIDITEMP,					"Miditemp" },
	{ RTP_MIDI_MANU_LONG_LA_AUDIO_LARKING_AUDIO,			"LA Audio (Larking Audio)" },
	{ RTP_MIDI_MANU_LONG_ZERO_88_LIGHTING_LIMITED,			"Zero 88 Lighting Limited" },
	{ RTP_MIDI_MANU_LONG_MICON_AUDIO_ELECTRONICS_GMBH,		"Micon Audio Electronics GmbH" },
	{ RTP_MIDI_MANU_LONG_FOREFRONT_TECHNOLOGY,			"Forefront Technology" },
	{ RTP_MIDI_MANU_LONG_STUDIO_AUDIO_AND_VIDEO_LTD,		"Studio Audio and Video Ltd" },
	{ RTP_MIDI_MANU_LONG_KENTON_ELECTRONICS,			"Kenton Electronics" },
	{ RTP_MIDI_MANU_LONG_CELCO_DIVISON_OF_ELECTRONICS,		"Celco Division of Electrosonic" },
	{ RTP_MIDI_MANU_LONG_ADB,					"ADB" },
	{ RTP_MIDI_MANU_LONG_MARSHALL_PRODUCTS,				"Marshall Products" },
	{ RTP_MIDI_MANU_LONG_DDA,					"DDA" },
	{ RTP_MIDI_MANU_LONG_BBS,					"BSS" },
	{ RTP_MIDI_MANU_LONG_MA_LIGHTING_TECHNOLOGY,			"MA Lighting Technology" },
	{ RTP_MIDI_MANU_LONG_FATAR,					"Fatar" },
	{ RTP_MIDI_MANU_LONG_QSC_AUDIO,					"QSC Audio" },
	{ RTP_MIDI_MANU_LONG_ARTISAN_CLASSIC_ORGAN,			"Artisan Classic Organ" },
	{ RTP_MIDI_MANU_LONG_ORLA_SPA,					"Rola Spa" },
	{ RTP_MIDI_MANU_LONG_PINNACLE_AUDIO,				"Pinnacle Audio" },
	{ RTP_MIDI_MANU_LONG_TC_ELECTRONICS,				"TC Electronics" },
	{ RTP_MIDI_MANU_LONG_DOEPFER_MUSIKELEKTRONIK,			"Doepfer Musikelektronik" },
	{ RTP_MIDI_MANU_LONG_CREATIVE_TECHNOLOGY_PTE,			"Creative Technology Pte" },
	{ RTP_MIDI_MANU_LONG_MINAMI_SEIYDDO,				"Minami / Seiyddo" },
	{ RTP_MIDI_MANU_LONG_MIDISOFT_SAS_DI_M_CIMA,			"Goldstar" },
	{ RTP_MIDI_MANU_LONG_SAMICK,					"Samick" },
	{ RTP_MIDI_MANU_LONG_PENNY_AND_GILES,				"Penny and Giles" },
	{ RTP_MIDI_MANU_LONG_ACORN_COMPUTER,				"Acorn Computer" },
	{ RTP_MIDI_MANU_LONG_LSC_ELECTRONICS,				"LSC Electronics" },
	{ RTP_MIDI_MANU_LONG_NOVATION_EMS,				"Novation EMS" },
	{ RTP_MIDI_MANU_LONG_SAMKYUNG_MECHATRONICS,			"Samkyung Mechatronics" },
	{ RTP_MIDI_MANU_LONG_MEDELI_ELECTRONICS_CO,			"Medeli Electronics Co" },
	{ RTP_MIDI_MANU_LONG_CHARLIE_LAB_SRL,				"Charlie Lab SRL" },
	{ RTP_MIDI_MANU_LONG_BLUE_CHIP_MUSIC_TECHNOLOGY,		"Blue Chip Music Technology" },
	{ RTP_MIDI_MANU_LONG_BEE_OH_CORP,				"BEE OH Corp" },
	{ RTP_MIDI_MANU_LONG_LG_SEMICON_AMERICA,			"LG Semicon America" },
	{ RTP_MIDI_MANU_LONG_TESI,					"TESI" },
	{ RTP_MIDI_MANU_LONG_EMAGIC,					"EMAGIC" },
	{ RTP_MIDI_MANU_LONG_BEHRINGER_GMBH,				"Behringer GmbH" },
	{ RTP_MIDI_MANU_LONG_ACCESS_MUSIC_ELECTRONICS,			"Access Music Electronics" },
	{ RTP_MIDI_MANU_LONG_SYNOPTIC,					"Synoptic" },
	{ RTP_MIDI_MANU_LONG_HANMESOFT_CORP,				"Hanmesoft Corp" },
	{ RTP_MIDI_MANU_LONG_TERRATEC_ELECTRONIC_GMBH,			"Terratec Electronic GmbH" },
	{ RTP_MIDI_MANU_LONG_PROEL_SPA,					"Proel SpA" },
	{ RTP_MIDI_MANU_LONG_IBK_MIDI,					"IBK MIDI" },
	{ RTP_MIDI_MANU_LONG_IRCAM,					"IRCAM" },
	{ RTP_MIDI_MANU_LONG_PROPELLERHEAD_SOFTWARE,			"Propellerhead Software" },
	{ RTP_MIDI_MANU_LONG_RED_SOUND_SYSTEMS_LTD,			"Red Sound Systems Ltd" },
	{ RTP_MIDI_MANU_LONG_ELEKTRON_ESI_AB,				"Elektron ESI AB" },
	{ RTP_MIDI_MANU_LONG_SINTEFEX_AUDIO,				"Sintefex Audio" },
	{ RTP_MIDI_MANU_LONG_MAM_MUSIC_AND_MORE,			"MAM (Music and More)" },
	{ RTP_MIDI_MANU_LONG_AMSARO_GMBH,				"Amsaro GmbH" },
	{ RTP_MIDI_MANU_LONG_CDS_ADVANCED_TECHNOLOGY_BV,		"CDS Advanced Technology BV" },
	{ RTP_MIDI_MANU_LONG_TOUCHED_BY_SOUND_GMBH,			"Touched By Sound GmbH" },
	{ RTP_MIDI_MANU_LONG_DSP_ARTS,					"DSP Arts" },
	{ RTP_MIDI_MANU_LONG_PHIL_REES_MUSIC_TECH,			"Phil Rees Music Tech" },
	{ RTP_MIDI_MANU_LONG_STAMER_MUSIKANLAGEN_GMBH,			"Stamer Musikanlagen GmbH" },
	{ RTP_MIDI_MANU_LONG_MUSICAL_MUNTANER_SA_DBA,			"Musical Muntaner S.A. dba Soundart" },
	{ RTP_MIDI_MANU_LONG_CMEXX_SOFTWARE,				"C-Mexx Software" },
	{ RTP_MIDI_MANU_LONG_KLAVIS_TECHNOLOGIES,			"Klavis Technologies" },
	{ RTP_MIDI_MANU_LONG_NOTEHEADS_AB,				"Noteheads AB" },
	{ RTP_MIDI_MANU_LONG_ALGORITHMIX,				"Algorithmix" },
	{ RTP_MIDI_MANU_LONG_SKRYDSTRUP_RD,				"Skrydstrup R&D" },
	{ RTP_MIDI_MANU_LONG_PROFRESSIONAL_AUDIO_COMPANY,		"Professional Audio Company" },
	{ RTP_MIDI_MANU_LONG_DBTECH_MADWAVES,				"DBTECH/MadWaves" },
	{ RTP_MIDI_MANU_LONG_VERMONA,					"Vermona" },
	{ RTP_MIDI_MANU_LONG_NOKIA,					"Nokia" },
	{ RTP_MIDI_MANU_LONG_WAVE_IDEA,					"Wave Idea" },
	{ RTP_MIDI_MANU_LONG_HARTMANN_GMBH,				"Hartmann GmbH" },
	{ RTP_MIDI_MANU_LONG_LIONS_TRACK,				"Lion's Track" },
	{ RTP_MIDI_MANU_LONG_ANALOGUE_SYSTEMS,				"Analogue Systems" },
	{ RTP_MIDI_MANU_LONG_FOCAL_JMLAB,				"Focal-JMlab" },
	{ RTP_MIDI_MANU_LONG_RINGWAY_ELECTRONICS,			"Ringway Electronics" },
	{ RTP_MIDI_MANU_LONG_FAITH_TECHNOLOGIES_DIGIPLUG,		"Faith Technologies (Digiplug)" },
	{ RTP_MIDI_MANU_LONG_SHOWWORKS,					"Showworks" },
	{ RTP_MIDI_MANU_LONG_MANIKIN_ELECTRONIC,			"Manikin Electronic" },
	{ RTP_MIDI_MANU_LONG_1_COME_TECH,				"1 Come Tech" },
	{ RTP_MIDI_MANU_LONG_PHONIC_CORP,				"Phonic Corp" },
	{ RTP_MIDI_MANU_LONG_LAKE_TECHNOLOGY,				"Lake Technology" },
	{ RTP_MIDI_MANU_LONG_SILANSYS_TECHNOLOGIES,			"Silansys Technologies" },
	{ RTP_MIDI_MANU_LONG_WINBOND_ELECTRONICS,			"Winbond Electronics" },
	{ RTP_MIDI_MANU_LONG_CINETIX_MEDIEN_UND_INTERFACE,		"Cinetix Medien und Interface GmbH" },
	{ RTP_MIDI_MANU_LONG_AG_SOLUTIONI_DIGITALI,			"A&G Soluzioni Digitali" },
	{ RTP_MIDI_MANU_LONG_SEQUENTIX_MUSIC_SYSTEMS,			"Sequentix Music Systems" },
	{ RTP_MIDI_MANU_LONG_ORAM_PRO_AUDIO,				"Oram Pro Audio" },
	{ RTP_MIDI_MANU_LONG_BE4_LTD,					"Be4 Ltd" },
	{ RTP_MIDI_MANU_LONG_INFECTION_MUSIC,				"Infection Music" },
	{ RTP_MIDI_MANU_LONG_CENTRAL_MUSIC_CO_CME,			"Central Music Co. (CME)" },
	{ RTP_MIDI_MANU_LONG_GENOQS_MACHINES,				"GenoQs Machines" },
	{ RTP_MIDI_MANU_LONG_MEDIALON,					"Medialon" },
	{ RTP_MIDI_MANU_LONG_WAVES_AUDIO_LTD,				"Waves Audio Ltd" },
	{ RTP_MIDI_MANU_LONG_JERASH_LABS,				"Jerash Labs" },
	{ RTP_MIDI_MANU_LONG_DA_FACT,					"Da Fact" },
	{ RTP_MIDI_MANU_LONG_ELBY_DESIGNS,				"Elby Designs" },
	{ RTP_MIDI_MANU_LONG_SPECTRAL_AUDIO,				"Spectral Audio" },
	{ RTP_MIDI_MANU_LONG_ARTURIA,					"Arturia" },
	{ RTP_MIDI_MANU_LONG_VIXID,					"Vixid" },
	{ RTP_MIDI_MANU_LONG_C_THRU_MUSIC,				"C-Thru Music" },
	{ RTP_MIDI_MANU_LONG_YA_HORNG_ELECTRONIC_CO_LTD,		"Ya Horng Electronic Co LTD" },
	{ RTP_MIDI_MANU_LONG_SM_PRO_AUDIO,				"SM Pro Audio" },
	{ RTP_MIDI_MANU_LONG_OTO_MACHINES,				"OTO MACHINES" },
	{ RTP_MIDI_MANU_LONG_ELZAB_SA_G_LAB,				"ELZAB S.A., G LAB" },
	{ RTP_MIDI_MANU_LONG_BLACKSTAR_AMPLIFICATION_LTD,		"Blackstar Amplification Ltd" },
	{ RTP_MIDI_MANU_LONG_M3I_TECHNOLOGIES_GMBH,			"M3i Technologies GmbH" },
	{ RTP_MIDI_MANU_LONG_GEMALTO,					"Gemalto" },
	{ RTP_MIDI_MANU_LONG_PROSTAGE_SL,				"Prostage SL" },
	{ RTP_MIDI_MANU_LONG_TEENAGE_ENGINEERING,			"Teenage Engineering" },
	{ RTP_MIDI_MANU_LONG_TOBIAS_ERICHSEN,				"Tobias Erichsen" },
	{ RTP_MIDI_MANU_LONG_NIXER_LTD,					"Nixer Ltd" },
	{ RTP_MIDI_MANU_LONG_HANPIN_ELECTRON_CO_LTD,			"Hanpin Electron Co Ltd" },
	{ RTP_MIDI_MANU_LONG_MIDI_HARDWARE_R_SOWA,			"MIDI-hardware R.Sowa" },
	{ RTP_MIDI_MANU_LONG_BEYOND_MUSIC_INDUSTRIAL_LTD,		"Beyond Music Industrial Ltd" },
	{ RTP_MIDI_MANU_LONG_KISS_BOX_BV,				"Kiss Box B.V." },
	{ RTP_MIDI_MANU_LONG_MISA_DIGITAL_TECHNOLOGIES_LTD,		"Misa Digital Technologies Ltd" },
	{ RTP_MIDI_MANU_LONG_AI_MUSICS_TECHNOLOGY_INC,			"AI Musics Technology Inc" },
	{ RTP_MIDI_MANU_LONG_SERATO_INC_LP,				"Serato Inc LP" },
	{ RTP_MIDI_MANU_LONG_LIMEX_MUSIC_HANDLES_GMBH,			"Limex Music Handles GmbH" },
	{ RTP_MIDI_MANU_LONG_KYODDAY_TOKAI,				"Kyodday/Tokai" },
	{ RTP_MIDI_MANU_LONG_MUTABLE_INSTRUMENTS,			"Mutable Instruments" },
	{ RTP_MIDI_MANU_LONG_PRESONUS_SOFTWARE_LTD,			"PreSonus Software Ltd" },
	{ RTP_MIDI_MANU_LONG_XIRING,					"Xiring" },
	{ RTP_MIDI_MANU_LONG_FAIRLIGHT_INTRUMENTS_PTY_LTD,		"Fairlight Instruments Pty Ltd" },
	{ RTP_MIDI_MANU_LONG_MUSICOM_LAB,				"Musicom Lab" },
	{ RTP_MIDI_MANU_LONG_VACO_LOCO,					"VacoLoco" },
	{ RTP_MIDI_MANU_LONG_RWA_HONG_KONG_LIMITED,			"RWA (Hong Kong) Limited" },


	/* Japanese Manufacturers */
	{ RTP_MIDI_MANU_LONG_CRIMSON_TECHNOLOGY_INC,			"Crimson Technology Inc" },
	{ RTP_MIDI_MANU_LONG_SOFTBANK_MOBILE_CORP,			"Softbank Mobile Corp" },
	{ RTP_MIDI_MANU_LONG_DM_HOLDINGS_INC,				"D&M Holdings Inc." },
	{ 0,								NULL }
};
static value_string_ext rtp_midi_manu_long_values_ext = VALUE_STRING_EXT_INIT(rtp_midi_manu_long_values);

static const value_string rtp_midi_channels[] = {
	{ RTP_MIDI_CHANNEL_1,						"Channel 1" },
	{ RTP_MIDI_CHANNEL_2,						"Channel 2" },
	{ RTP_MIDI_CHANNEL_3,						"Channel 3" },
	{ RTP_MIDI_CHANNEL_4,						"Channel 4" },
	{ RTP_MIDI_CHANNEL_5,						"Channel 5" },
	{ RTP_MIDI_CHANNEL_6,						"Channel 6" },
	{ RTP_MIDI_CHANNEL_7,						"Channel 7" },
	{ RTP_MIDI_CHANNEL_8,						"Channel 8" },
	{ RTP_MIDI_CHANNEL_9,						"Channel 9" },
	{ RTP_MIDI_CHANNEL_10,						"Channel 10" },
	{ RTP_MIDI_CHANNEL_11,						"Channel 11" },
	{ RTP_MIDI_CHANNEL_12,						"Channel 12" },
	{ RTP_MIDI_CHANNEL_13,						"Channel 13" },
	{ RTP_MIDI_CHANNEL_14,						"Channel 14" },
	{ RTP_MIDI_CHANNEL_15,						"Channel 15" },
	{ RTP_MIDI_CHANNEL_16,						"Channel 16" },
	{ 0,								NULL }
};



static const char rtp_midi_channel_status_special_off[] =		"Note Off*";
static const char rtp_midi_common_status_sysex_segment_end[] =		"End of Sysex-Segment";
static const char rtp_midi_common_status_sysex_segment_start[] =	"Start of Sysex-Segment";
static const char rtp_midi_common_status_sysex_segment[] =		"Middle Sysex-Segment";
static const char rtp_midi_common_status_sysex_segment_complete[] =	"Full Sysex-Command";
static const char rtp_midi_common_status_sysex_cancel[] =		"Sysex-Cancel";

static const value_string rtp_midi_channel_status[] = {
	/* Channel */
	{ RTP_MIDI_STATUS_CHANNEL_NOTE_OFF,				"Note Off" },
	{ RTP_MIDI_STATUS_CHANNEL_NOTE_ON,				"Note On" },
	{ RTP_MIDI_STATUS_CHANNEL_POLYPHONIC_KEY_PRESSURE,		"Polyphonic Key Pressure" },
	{ RTP_MIDI_STATUS_CHANNEL_CONTROL_CHANGE,			"Control Change" },
	{ RTP_MIDI_STATUS_CHANNEL_PROGRAM_CHANGE,			"Program Change" },
	{ RTP_MIDI_STATUS_CHANNEL_CHANNEL_PRESSURE,			"Channel Pressure" },
	{ RTP_MIDI_STATUS_CHANNEL_PITCH_BEND_CHANGE,			"Pitch Bend Change" },
	{ 0,								NULL }
};

static const value_string rtp_midi_common_status[] = {
	/* Common 0xf0 - 0xf7 */
	{ RTP_MIDI_STATUS_COMMON_SYSEX_START,				"SOX: Start Of System-Exclusive / End of System-Exclusive Segment" },
	{ RTP_MIDI_STATUS_COMMON_MTC_QUARTER_FRAME,			"MTC Quarter Frame" },
	{ RTP_MIDI_STATUS_COMMON_SONG_POSITION_POINTER,			"Song Position Pointer" },
	{ RTP_MIDI_STATUS_COMMON_SONG_SELECT,				"Song Select" },
	{ RTP_MIDI_STATUS_COMMON_UNDEFINED_F4,				"Systems Exclusive Cancel" },			/* not defined, special use in RTP-MIDI */
	{ RTP_MIDI_STATUS_COMMON_UNDEFINED_F5,				"Undefined 0xf5" },			/* not defined */
	{ RTP_MIDI_STATUS_COMMON_TUNE_REQUEST,				"Tune Request" },
	{ RTP_MIDI_STATUS_COMMON_SYSEX_END,				"EOX: Start of System-Exclusive Segment / End of System-Exclusive" },
	/* Realtime 0xf8 - 0xff */
	{ RTP_MIDI_STATUS_COMMON_REALTIME_TIMING_CLOCK,			"RT: Timing Clock" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_MIDI_TICK,			"RT: Midi Tick" },			/* undefined ??? */
	{ RTP_MIDI_STATUS_COMMON_REALTIME_START,			"RT: Start" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_CONTINUE,			"RT: Continue" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_STOP,				"RT: Stop" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_UNDEFINED_FD,			"RT: Undefined 0xfd" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_ACTIVE_SENSING,		"RT: Active sensing" },
	{ RTP_MIDI_STATUS_COMMON_REALTIME_SYSTEM_RESET,			"RT: Systems reset" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_rt_mtc_cue[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_SPECIAL,			"Special" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_PUNCH_IN_POINTS,		"Punch In Points" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_PUNCH_OUT_POINTS,		"Punch Out Points" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_03,			"(Reserved 0x03)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_04,			"(Reserved 0x04)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_START_POINTS,		"Event Start Points" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_STOP_POINTS,		"Event Stop Points" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_START_POINTS_ADD,	"Event Start Points with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_STOP_POINTS_ADD,	"Event Stop Points with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_09,			"(Reserved 0x09)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_0A,			"(Reserved 0x0a)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_CUE_POINTS,			"Cue Points" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_CUE_POINTS_ADD,		"Cue Points with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_RESERVED_0D,			"(Reserved 0x0d)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUE_EVENT_NAME_ADD,		"Event Name in additional Info" },
	{ 0,								NULL }
};


static const value_string rtp_midi_sysex_common_nrt_mtc[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_SPECIAL,			"Special" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_PUNCH_IN_POINTS,		"Punch In Points" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_PUNGH_OUT_POINTS,		"Punch Out Points" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_PUNCH_IN_POINTS,		"Delete Punch In Points" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_PUNCH_OUT_POINTS,	"Delete Punch Out Points" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_START_POINT,		"Event Start Point" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_STOP_POINT,		"Event Stop Point" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_START_POINT_ADD,		"Event Start Point with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_STOP_POINT_ADD,		"Event Stop Point with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_EVENT_START_POINT,	"Delete Event Start Point" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_EVENT_STOP_POINT,	"Delete Event Stop Point" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_CUE_POINTS,			"Cue Points" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_CUE_POINTS_ADD,			"Cue Points with additional Info" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_DELETE_CUE_POINT,		"Delete Cue Point" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC_EVENT_NAME_IN_ADD,		"Event Name in additional Info" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_sd_ext[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINT_TRANSMISSION,	"Loop Point Transmission" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINTS_REQUEST,		"Loop Points Request" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_sd_lt[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_UNI,			"Forwards only (unidirectional)" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_BI,			"Backwards/Forwards (bi-directional)" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LP_OFF,			"Off" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_gi[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REQUEST,		"Identity Request" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REPLY,			"Identity Reply" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_fd[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_FD_HEADER,				"Header" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_FD_DATA_PACKET,			"Data Packet" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_FD_REQUEST,				"Request" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_tuning[] = {
	{ RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REQUEST,		"Bulk Dump Request" },
	{ RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REPLY,			"Bulk Dump Reply" },
	{ RTP_MIDI_SYSEX_COMMON_TUNING_NOTE_CHANGE,			"Note Change" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_gm[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_GM_ON,				"General MIDI System On" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_GM_OFF,				"General MIDI System Off" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_nrt_dls[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_ON,			"DLS Level 1 On" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_OFF,			"DLS Level 1 Off" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_VOICE_ALLOCATION_OFF,	"DLS Level 1 Voice Allocation Off" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_DLS_LEVEL1_VOICE_ALLOCATION_ON,	"DLS Level 1 Voice Allocation On" },
	{ 0,								NULL }
};


static const value_string rtp_midi_sysex_common_rt_mtc[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_FULL_MESSAGE,			"Full Message" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_USER_BITS,			"User Bits" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_rt_show_control[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_SCL_EXTENSIONS,			"Extensions" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_LIGHTING,				"Lighting (General Category)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_MOVING_LIGHTS,			"Moving Lights" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_COLOR_CHANGERS,			"Color Changers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_STROBES,				"Strobes" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SCL_LASERS,				"Lasers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SCL_CHASERS,				"Chasers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_SOUND,				"Sound (General Category)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_MUSIC,				"Music" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_CD_PLAYERS,			"CD Players" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_EPROM_PLAYBACK,			"EPROM Playback" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SCL_AUDIO_TAPE_MACHINE,		"Audio Tape Machine" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_INTERCOMS,			"Intercoms" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_AMPLIFIERS,			"Amplifiers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_AUDIO_EFFECTS,			"Audio Effects Devices" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_EQUALIZERS,			"Equalizers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_MACHINERY,			"Machinery (General Category" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_RIGGING,				"Rigging" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FLYS,				"Flys" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_LIFTS,				"Lifts" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_TURNTABLES,			"Turntables" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_TRUSSES,				"Trusses" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_ROBOTS,				"Robots" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_ANIMATION,			"Animation" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FLOATS,				"Floats" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_BREAKAWAYS,			"Breakaways" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_BARGES,				"Barges" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO,				"Video (General Category" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_TAPE_MACHINES,		"Video Tape Machines" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_CASSETTE_MACHINES,		"Video Cassette Machines" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_DISC_PLAYERS,		"Video Disc Players" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_SWITCHERS,			"Video Switchers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_EFFECT,			"Video Effects" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_CHARACTER_GENERATORS,	"Video Character Generators" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_STIL_STORES,		"Video Still Stores" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_MONITORS,			"Video Monitors" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_PROJECTION,			"Projections (General)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FILM_PROJECTORS,			"Film Projectors" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_SLIDE_PROJECTORS,			"Slide Projectors" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_VIDEO_PROJECTORS,			"Video Projectors" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_DISSOLVERS,			"Dissolvers" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_SHUTTER_CONTROLS,			"Shutter Controls" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_PROCESS_CONTROL,			"Process Control (General)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_HYDRAULIC_OIL,			"Hydraulic Oil" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_H2O,				"H2O" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_CO2,				"CO2" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_COMPRESSED_AIR,			"Compressed Air" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_NATURAL_GAS,			"Natural Gas" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FOG,				"Fog" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_SMOKE,				"Smoke" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_CRACKED_HAZE,			"Cracked Haze" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_PYRO,				"Pyro (General Category)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FIREWORKS,			"Fireworks" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_EXPLOSIONS,			"Explosions" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_FLAME,				"Flame" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_SMOKE_POTS,			"Smoke Pots" },
	{ RTP_MIDI_SYSEX_COMMON_RT_SC_ALL_TYPES,			"All Types" },
	{ 0,								NULL }
};
static value_string_ext rtp_midi_sysex_common_rt_show_control_ext =
	VALUE_STRING_EXT_INIT(rtp_midi_sysex_common_rt_show_control);

static const value_string rtp_midi_sysex_common_rt_notations[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_NT_BAR_NUMBER,			"Bar Number" },
	{ RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_IMMEDIATE,		"Time Signature (Immediate)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_DELAYED,		"Time Signature (Delayed)" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sysex_common_rt_device_control[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_VOLUME,			"Master Volume" },
	{ RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_BALANCE,			"Master Balance" },
	{ 0,								NULL }
};


/* TODO: Add MMC Commands */
static const value_string rtp_midi_sysex_common_rt_mmc_commands[] = {
	{ 0,								NULL }
};

/* TODO: Add MMC Responses */
static const value_string rtp_midi_sysex_common_rt_mmc_responses[] = {
	{ 0,								NULL }
};



static const value_string rtp_midi_sysex_common_nrt[] = {
	{ RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_HEADER,			"Common NRT: Sample Dump Header" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DATA_PACKET,			"Common NRT: Sample Data Packet" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_REQUEST,		"Common NRT: Sample Dump Request" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MTC,				"Common NRT: MIDI Time Code (MTC)" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_EXTENSIONS,		"Common NRT: Sample Dump Extensions" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_INFORMATION,		"Common NRT: General Information" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_FILE_DUMP,				"Common NRT: File Dump" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_MIDI_TUNING_STANDARD,		"Common NRT: Midi Tuning Standard" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_MIDI,			"Common NRT: General MIDI (GM)" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_DOWNLOADABLE_SOUNDS,		"Common NRT: Downloadable Sounds (DLS)" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_END_OF_FILE,			"Common NRT: End Of File" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_WAIT,				"Common NRT: Wait" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_CANCEL,				"Common NRT: Cancel" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_NAK,				"Common NRT: NAK" },
	{ RTP_MIDI_SYSEX_COMMON_NRT_ACK,				"Common NRT: ACK" },
	{ 0,								NULL }
};


static const value_string rtp_midi_sysex_common_rt[] = {
	{ RTP_MIDI_SYSEX_COMMON_RT_MIDI_TIME_CODE,			"Common RT: MIDI Time Code (MTC)" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MIDI_SHOW_CONTROL,			"Common RT: MIDI Show Control" },
	{ RTP_MIDI_SYSEX_COMMON_RT_NOTATION_INFORMATION,		"Common RT: Notation Information" },
	{ RTP_MIDI_SYSEX_COMMON_RT_DEVICE_CONTROL,			"Common RT: Device Control" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MTC_CUEING,				"Common RT: Real Time MTC Cueing" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_COMMAND,	"Common RT: MIDI Machine Control Command" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_RESPONSE,	"Common RT: MIDI Machine Control Response" },
	{ RTP_MIDI_SYSEX_COMMON_RT_MIDI_TUNING_STANDARD,		"Common RT: MIDI Tuning Standard" },
	{ 0,								NULL }
};



static const value_string rtp_midi_js_tot_channels[] = {
	{ 0,								"1 channel" },
	{ 1,								"2 channels" },
	{ 2,								"3 channels" },
	{ 3,								"4 channels" },
	{ 4,								"5 channels" },
	{ 5,								"6 channels" },
	{ 6,								"7 channels" },
	{ 7,								"8 channels" },
	{ 8,								"9 channels" },
	{ 9,								"10 channels" },
	{ 10,								"11 channels" },
	{ 11,								"12 channels" },
	{ 12,								"13 channels" },
	{ 13,								"14 channels" },
	{ 14,								"15 channels" },
	{ 15,								"16 channels" },
	{ 0,								NULL }
};

static const value_string rtp_midi_qft_values[] = {
	{ RTP_MIDI_COMMON_MTC_QF_FRAME_LS_NIBBLE,			"Frame count LS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_FRAME_MS_NIBBLE,			"Frame count MS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_SECONDS_LS_NIBBLE,			"Seconds count LS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_SECONDS_MS_NIBBLE,			"Seconds count MS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_MINUTES_LS_NIBBLE,			"Minutes count LS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_MINUTES_MS_NIBBLE,			"Minutes count MS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_HOURS_LS_NIBBLE,			"Hours count LS nibble" },
	{ RTP_MIDI_COMMON_MTC_QF_HOURS_MS_NIBBLE,			"Hours count MS nibble" },
	{ 0,								NULL }
};

static const value_string rtp_midi_sj_chapter_d_syscom_dsz_values[] = {
	{ 0,								"Command has 0 octets" },
	{ 1,								"Command has 1 octet" },
	{ 2,								"Command has 2 octets" },
	{ 3,								"Command has 3 or more octets" },
	{ 0,								NULL }
};

static const value_string rtp_midi_mtc_fm_type_values[] = {
	{ 0,								"24 Frames/Second" },
	{ 1,								"25 Frames/Second" },
	{ 2,								"30 Frames/Second (drop frame)" },
	{ 3,								"30 Frames/Second (non-drop frame)" },
	{ 0,								NULL }
};

static const true_false_string rtp_midi_cs_flag_b= {
	"Long Header: 2 octets",
	"Short Header: 1 octet"
};

static const true_false_string rtp_midi_cs_flag_j= {
	"Journal present",
	"No Journal present"
};

static const true_false_string rtp_midi_cs_flag_z= {
	"Delta-Time present for first MIDI-command",
	"No Delta-Time for first MIDI-command"
};

static const true_false_string rtp_midi_cs_flag_p= {
	"Status byte was NOT present in original MIDI-command",
	"Status byte was present in original MIDI-command"
};

static const true_false_string rtp_midi_js_flag_y = {
	"System-journal present",
	"System-journal NOT present"
};

static const true_false_string rtp_midi_js_flag_s = {
	"Standard journal behaviour",
	"Journal encodes data in preceding RTP-packet"
};

static const true_false_string rtp_midi_js_flag_a = {
	"Channel-journal present",
	"Channel-journal NOT present"
};

static const true_false_string rtp_midi_js_flag_h = {
	"Enhanced Encoding",
	"Standard Encoding"
};

static const true_false_string rtp_midi_sj_flag_d = {
	"Chapter D journal present",
	"Chapter D journal NOT present"
};

static const true_false_string rtp_midi_sj_flag_v = {
	"Chapter V journal present",
	"Chapter V journal NOT present"
};

static const true_false_string rtp_midi_sj_flag_q = {
	"Chapter Q journal present",
	"Chapter Q journal NOT present"
};

static const true_false_string rtp_midi_sj_flag_f = {
	"Chapter F journal present",
	"Chapter F journal NOT present"
};

static const true_false_string rtp_midi_sj_flag_x = {
	"Chapter X journal present",
	"Chapter X journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_p = {
	"Chapter P journal present",
	"Chapter P journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_c = {
	"Chapter C journal present",
	"Chapter C journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_m = {
	"Chapter M journal present",
	"Chapter M journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_w = {
	"Chapter W journal present",
	"Chapter W journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_n = {
	"Chapter N journal present",
	"Chapter N journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_e = {
	"Chapter E journal present",
	"Chapter E journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_t = {
	"Chapter T journal present",
	"Chapter T journal NOT present"
};

static const true_false_string rtp_midi_cj_flag_a = {
	"Chapter A journal present",
	"Chapter A journal NOT present"
};

static const true_false_string rtp_midi_cj_chapter_c_flag_a = {
	"Alternative Coding",
	"Value Coding"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_p = {
	"Pending field is encoded",
	"Pending field is NOT encoded"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_e = {
	"Initiated transaction is in progress",
	"NO transaction in progress"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_u = {
	"All parameters are RPNs",
	"Parameters may include NRPNs"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_w = {
	"All parameters are NRPNs",
	"Parameters may include RPNs"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_z = {
	"Only LSB parameters",
	"Parameters may have LSB & MSB"
};

static const true_false_string rtp_midi_cj_chapter_m_flag_q = {
	"Pending encodes NRPN MSB",
	"Pending encodes RPN MSB"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_q = {
	"NRPN parameter encoded",
	"RPN parameter encoded"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_j = {
	"ENTRY-MSB field present",
	"ENTRY-MSB field NOT present"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_k = {
	"ENTRY-LSB field present",
	"ENTRY-LSB field NOT present"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_l = {
	"A-Button field present",
	"A-Button field NOT present"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_m = {
	"C-Button field present",
	"C-Button field NOT present"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_n = {
	"Count field present",
	"Count field NOT present"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_t = {
	"Count Tool used",
	"Count Tool NOT used"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_v = {
	"Value Tool used",
	"Value Tool NOT used"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_r = {
	"Reserved-Flag set",
	"Reserved-Flag NOT set"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_x = {
	"Precedes most recent Control Change command for ctrl=121",
	"Does not precede most recent Control Change command for ctrl=121"
};

static const true_false_string rtp_midi_cj_chapter_m_log_flag_g = {
	"negative value",
	"positive value"
};

static const true_false_string rtp_midi_cj_chapter_n_log_flag_y = {
	"Execute Note-On",
	"Skip Note-On"
};

static const true_false_string rtp_midi_cj_chapter_e_log_flag_v = {
	"Encodes Note-Off velocity",
	"Encodes Note-On/Off count"
};

static const true_false_string rtp_midi_cj_chapter_a_log_flag_x = {
	"Precedes most recent Control Change command for ctrl=123-127",
	"Does not precede most recent Control Change command for ctrl=123-127"
};

static const true_false_string rtp_midi_cj_chapter_p_log_flag_x = {
	"Reset all controllers received between MSB and LSB",
	"NO Reset all controllers received between MSB and LSB"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_b = {
	"Reset field present",
	"Reset field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_g = {
	"Tune request field present",
	"Tune request field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_h = {
	"Song select field present",
	"Song select field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_j = {
	"Undefined system common 0xF4 field present",
	"Undefined system common 0xF4 field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_k = {
	"Undefined system common 0xF5 field present",
	"Undefined system common 0xF5 field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_y = {
	"Undefined system common 0xF9 field present",
	"Undefined system common 0xF9 field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_flag_z = {
	"Undefined system common 0xFD field present",
	"Undefined system common 0xFD field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_syscom_flag_c = {
	"Count field present",
	"Count field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_syscom_flag_v = {
	"Value field present",
	"Value field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_d_syscom_flag_l = {
	"Legal field present",
	"Legal field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_q_flag_n = {
	"Active Start / Continue",
	"NO active Start / Stop Continue"
};

static const true_false_string rtp_midi_sj_chapter_q_flag_d = {
	"Song position already reached",
	"Song position yet to be reached"
};

static const true_false_string rtp_midi_sj_chapter_q_flag_c = {
	"Clock field present",
	"Clock field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_q_flag_t = {
	"Timetools field present",
	"Timetools field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_f_flag_c = {
	"Complete field present",
	"Complete field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_f_flag_p = {
	"Partial field present",
	"Partial field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_f_flag_q = {
	"Complete field contains quarter frames",
	"Complete field contains full frame format"
};

static const true_false_string rtp_midi_sj_chapter_f_flag_d = {
	"Tape head is moving reverse",
	"Tape head is moving forward or undefined"
};

static const true_false_string rtp_midi_sj_chapter_x_flag_t = {
	"TCOUNT field present",
	"TCOUNT field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_x_flag_c = {
	"COUNT field present",
	"COUNT field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_x_flag_f = {
	"FIRST field present",
	"FIRST field NOT present"
};

static const true_false_string rtp_midi_sj_chapter_x_flag_d = {
	"DATA field present",
	"DATA field not present"
};

static const true_false_string rtp_midi_sj_chapter_x_flag_l = {
	"List Tool used",
	"Recency Tool used"
};



static int hf_rtp_midi_bflag					= -1;
static int hf_rtp_midi_jflag					= -1;
static int hf_rtp_midi_zflag					= -1;
static int hf_rtp_midi_pflag					= -1;
static int hf_rtp_midi_shortlen					= -1;
static int hf_rtp_midi_longlen					= -1;

static int hf_rtp_midi_sflag					= -1;
static int hf_rtp_midi_yflag					= -1;
static int hf_rtp_midi_aflag					= -1;
static int hf_rtp_midi_hflag					= -1;
static int hf_rtp_midi_totchan					= -1;
static int hf_rtp_midi_check_seq_num				= -1;

static int hf_rtp_midi_deltatime1				= -1;
static int hf_rtp_midi_deltatime2				= -1;
static int hf_rtp_midi_deltatime3				= -1;
static int hf_rtp_midi_deltatime4				= -1;

static int hf_rtp_midi_channel_status				= -1;
static int hf_rtp_midi_common_status				= -1;
static int hf_rtp_midi_channel					= -1;
static int hf_rtp_midi_note					= -1;
static int hf_rtp_midi_velocity					= -1;
static int hf_rtp_midi_pressure					= -1;
static int hf_rtp_midi_controller				= -1;
static int hf_rtp_midi_controller_value				= -1;
static int hf_rtp_midi_program					= -1;
static int hf_rtp_midi_channel_pressure				= -1;
static int hf_rtp_midi_pitch_bend				= -1;
/* static int hf_rtp_midi_pitch_bend_truncated			= -1; */

static int hf_rtp_midi_manu_short				= -1;
static int hf_rtp_midi_manu_long				= -1;

static int hf_rtp_midi_sysjour_toc_s				= -1;
static int hf_rtp_midi_sysjour_toc_d				= -1;
static int hf_rtp_midi_sysjour_toc_v				= -1;
static int hf_rtp_midi_sysjour_toc_q				= -1;
static int hf_rtp_midi_sysjour_toc_f				= -1;
static int hf_rtp_midi_sysjour_toc_x				= -1;
static int hf_rtp_midi_sysjour_len				= -1;

static int hf_rtp_midi_chanjour_sflag				= -1;
static int hf_rtp_midi_chanjour_chan				= -1;
static int hf_rtp_midi_chanjour_hflag				= -1;
static int hf_rtp_midi_chanjour_len				= -1;
static int hf_rtp_midi_chanjour_toc_p				= -1;
static int hf_rtp_midi_chanjour_toc_c				= -1;
static int hf_rtp_midi_chanjour_toc_m				= -1;
static int hf_rtp_midi_chanjour_toc_w				= -1;
static int hf_rtp_midi_chanjour_toc_n				= -1;
static int hf_rtp_midi_chanjour_toc_e				= -1;
static int hf_rtp_midi_chanjour_toc_t				= -1;
static int hf_rtp_midi_chanjour_toc_a				= -1;

static int hf_rtp_midi_cj_chapter_p_sflag			= -1;
static int hf_rtp_midi_cj_chapter_p_program			= -1;
static int hf_rtp_midi_cj_chapter_p_bflag			= -1;
static int hf_rtp_midi_cj_chapter_p_bank_msb			= -1;
static int hf_rtp_midi_cj_chapter_p_xflag			= -1;
static int hf_rtp_midi_cj_chapter_p_bank_lsb			= -1;

static int hf_rtp_midi_cj_chapter_c_sflag			= -1;
static int hf_rtp_midi_cj_chapter_c_length			= -1;
static int hf_rtp_midi_cj_chapter_c_number			= -1;
static int hf_rtp_midi_cj_chapter_c_aflag			= -1;
static int hf_rtp_midi_cj_chapter_c_value			= -1;
static int hf_rtp_midi_cj_chapter_c_tflag			= -1;
static int hf_rtp_midi_cj_chapter_c_alt				= -1;

static int hf_rtp_midi_cj_chapter_m_sflag			= -1;
static int hf_rtp_midi_cj_chapter_m_pflag			= -1;
static int hf_rtp_midi_cj_chapter_m_eflag			= -1;
static int hf_rtp_midi_cj_chapter_m_uflag			= -1;
static int hf_rtp_midi_cj_chapter_m_wflag			= -1;
static int hf_rtp_midi_cj_chapter_m_zflag			= -1;
static int hf_rtp_midi_cj_chapter_m_length			= -1;
static int hf_rtp_midi_cj_chapter_m_qflag			= -1;
static int hf_rtp_midi_cj_chapter_m_pending			= -1;
static int hf_rtp_midi_cj_chapter_m_log_sflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_pnum_lsb		= -1;
static int hf_rtp_midi_cj_chapter_m_log_qflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_pnum_msb		= -1;
static int hf_rtp_midi_cj_chapter_m_log_jflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_kflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_lflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_mflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_nflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_tflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_vflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_rflag			= -1;
static int hf_rtp_midi_cj_chapter_m_log_msb				= -1;
static int hf_rtp_midi_cj_chapter_m_log_msb_x			= -1;
static int hf_rtp_midi_cj_chapter_m_log_msb_entry		= -1;
static int hf_rtp_midi_cj_chapter_m_log_lsb				= -1;
static int hf_rtp_midi_cj_chapter_m_log_lsb_x			= -1;
static int hf_rtp_midi_cj_chapter_m_log_lsb_entry		= -1;
static int hf_rtp_midi_cj_chapter_m_log_a_button_g		= -1;
static int hf_rtp_midi_cj_chapter_m_log_a_button_x		= -1;
static int hf_rtp_midi_cj_chapter_m_log_a_button		= -1;
static int hf_rtp_midi_cj_chapter_m_log_a_button_full	= -1;
static int hf_rtp_midi_cj_chapter_m_log_c_button		= -1;
static int hf_rtp_midi_cj_chapter_m_log_c_button_g		= -1;
static int hf_rtp_midi_cj_chapter_m_log_c_button_r		= -1;
static int hf_rtp_midi_cj_chapter_m_log_c_button_full	= -1;
static int hf_rtp_midi_cj_chapter_m_log_count			= -1;
static int hf_rtp_midi_cj_chapter_m_log_count_x			= -1;
static int hf_rtp_midi_cj_chapter_m_log_count_full		= -1;

static int hf_rtp_midi_cj_chapter_w_sflag			= -1;
static int hf_rtp_midi_cj_chapter_w_first			= -1;
static int hf_rtp_midi_cj_chapter_w_rflag			= -1;
static int hf_rtp_midi_cj_chapter_w_second			= -1;

static int hf_rtp_midi_cj_chapter_n_bflag			= -1;
static int hf_rtp_midi_cj_chapter_n_len				= -1;
static int hf_rtp_midi_cj_chapter_n_low				= -1;
static int hf_rtp_midi_cj_chapter_n_high			= -1;
static int hf_rtp_midi_cj_chapter_n_log_sflag			= -1;
static int hf_rtp_midi_cj_chapter_n_log_notenum			= -1;
static int hf_rtp_midi_cj_chapter_n_log_yflag			= -1;
static int hf_rtp_midi_cj_chapter_n_log_velocity		= -1;
static int hf_rtp_midi_cj_chapter_n_log_octet			= -1;

static int hf_rtp_midi_cj_chapter_e_sflag			= -1;
static int hf_rtp_midi_cj_chapter_e_len				= -1;
static int hf_rtp_midi_cj_chapter_e_log_sflag			= -1;
static int hf_rtp_midi_cj_chapter_e_log_notenum			= -1;
static int hf_rtp_midi_cj_chapter_e_log_vflag			= -1;
static int hf_rtp_midi_cj_chapter_e_log_velocity		= -1;
static int hf_rtp_midi_cj_chapter_e_log_count			= -1;

static int hf_rtp_midi_cj_chapter_t_channel_aftertouch	= -1;
static int hf_rtp_midi_cj_chapter_t_sflag				= -1;
static int hf_rtp_midi_cj_chapter_t_pressure			= -1;

static int hf_rtp_midi_cj_chapter_a_sflag			= -1;
static int hf_rtp_midi_cj_chapter_a_len				= -1;
static int hf_rtp_midi_cj_chapter_a_log_sflag			= -1;
static int hf_rtp_midi_cj_chapter_a_log_notenum			= -1;
static int hf_rtp_midi_cj_chapter_a_log_xflag			= -1;
static int hf_rtp_midi_cj_chapter_a_log_pressure		= -1;

static int hf_rtp_midi_sj_chapter_v					= -1;
static int hf_rtp_midi_sj_chapter_v_sflag			= -1;
static int hf_rtp_midi_sj_chapter_v_count			= -1;

static int hf_rtp_midi_sj_chapter_d_simple_system_commands  = -1;
static int hf_rtp_midi_sj_chapter_d_sflag			= -1;
static int hf_rtp_midi_sj_chapter_d_bflag			= -1;
static int hf_rtp_midi_sj_chapter_d_gflag			= -1;
static int hf_rtp_midi_sj_chapter_d_hflag			= -1;
static int hf_rtp_midi_sj_chapter_d_jflag			= -1;
static int hf_rtp_midi_sj_chapter_d_kflag			= -1;
static int hf_rtp_midi_sj_chapter_d_yflag			= -1;
static int hf_rtp_midi_sj_chapter_d_zflag			= -1;

static int hf_rtp_midi_sj_chapter_d_reset				= -1;
static int hf_rtp_midi_sj_chapter_d_reset_sflag			= -1;
static int hf_rtp_midi_sj_chapter_d_reset_count			= -1;
static int hf_rtp_midi_sj_chapter_d_tune				= -1;
static int hf_rtp_midi_sj_chapter_d_tune_sflag			= -1;
static int hf_rtp_midi_sj_chapter_d_tune_count			= -1;
static int hf_rtp_midi_sj_chapter_d_song_sel			= -1;
static int hf_rtp_midi_sj_chapter_d_song_sel_sflag		= -1;
static int hf_rtp_midi_sj_chapter_d_song_sel_value		= -1;

static int hf_rtp_midi_sj_chapter_d_syscom_sflag		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_cflag		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_vflag		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_lflag		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_dsz			= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_length		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_count		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_value		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_legal		= -1;
static int hf_rtp_midi_sj_chapter_d_syscom_data			= -1;

static int hf_rtp_midi_sj_chapter_d_sysreal_sflag		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_cflag		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_lflag		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_length		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_count		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_legal		= -1;
static int hf_rtp_midi_sj_chapter_d_sysreal_data		= -1;

static int hf_rtp_midi_sj_chapter_q_sflag			= -1;
static int hf_rtp_midi_sj_chapter_q_nflag			= -1;
static int hf_rtp_midi_sj_chapter_q_dflag			= -1;
static int hf_rtp_midi_sj_chapter_q_cflag			= -1;
static int hf_rtp_midi_sj_chapter_q_tflag			= -1;
static int hf_rtp_midi_sj_chapter_q_top				= -1;
static int hf_rtp_midi_sj_chapter_q_clock			= -1;
static int hf_rtp_midi_sj_chapter_q_timetools			= -1;

static int hf_rtp_midi_sj_chapter_f_sflag			= -1;
static int hf_rtp_midi_sj_chapter_f_cflag			= -1;
static int hf_rtp_midi_sj_chapter_f_pflag			= -1;
static int hf_rtp_midi_sj_chapter_f_qflag			= -1;
static int hf_rtp_midi_sj_chapter_f_dflag			= -1;
static int hf_rtp_midi_sj_chapter_f_point			= -1;
static int hf_rtp_midi_sj_chapter_f_complete		= -1;
static int hf_rtp_midi_sj_chapter_f_partial			= -1;
static int hf_rtp_midi_sj_chapter_f_mt0				= -1;
static int hf_rtp_midi_sj_chapter_f_mt1				= -1;
static int hf_rtp_midi_sj_chapter_f_mt2				= -1;
static int hf_rtp_midi_sj_chapter_f_mt3				= -1;
static int hf_rtp_midi_sj_chapter_f_mt4				= -1;
static int hf_rtp_midi_sj_chapter_f_mt5				= -1;
static int hf_rtp_midi_sj_chapter_f_mt6				= -1;
static int hf_rtp_midi_sj_chapter_f_mt7				= -1;
static int hf_rtp_midi_sj_chapter_f_hr				= -1;
static int hf_rtp_midi_sj_chapter_f_mn				= -1;
static int hf_rtp_midi_sj_chapter_f_sc				= -1;
static int hf_rtp_midi_sj_chapter_f_fr				= -1;

static int hf_rtp_midi_sj_chapter_x_sflag			= -1;
static int hf_rtp_midi_sj_chapter_x_tflag			= -1;
static int hf_rtp_midi_sj_chapter_x_cflag			= -1;
static int hf_rtp_midi_sj_chapter_x_fflag			= -1;
static int hf_rtp_midi_sj_chapter_x_dflag			= -1;
static int hf_rtp_midi_sj_chapter_x_lflag			= -1;
static int hf_rtp_midi_sj_chapter_x_sta				= -1;
static int hf_rtp_midi_sj_chapter_x_tcount			= -1;
static int hf_rtp_midi_sj_chapter_x_count			= -1;
static int hf_rtp_midi_sj_chapter_x_first1			= -1;
static int hf_rtp_midi_sj_chapter_x_first2			= -1;
static int hf_rtp_midi_sj_chapter_x_first3			= -1;
static int hf_rtp_midi_sj_chapter_x_first4			= -1;
static int hf_rtp_midi_sj_chapter_x_data			= -1;
static int hf_rtp_midi_sj_chapter_x_invalid_data		= -1;

static int hf_rtp_midi_quarter_frame_type			= -1;
static int hf_rtp_midi_quarter_frame_value			= -1;
/* static int hf_rtp_midi_spp_truncated				= -1; */
static int hf_rtp_midi_spp					= -1;
static int hf_rtp_midi_song_select				= -1;
static int hf_rtp_midi_manu_data				= -1;
static int hf_rtp_midi_edu_data					= -1;
static int hf_rtp_midi_unknown_data				= -1;
static int hf_rtp_midi_sysex_common_non_realtime		= -1;
static int hf_rtp_midi_sysex_common_realtime			= -1;
static int hf_rtp_midi_sysex_common_device_id			= -1;

static int hf_rtp_midi_sysex_common_nrt_mtc			= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_ext			= -1;
static int hf_rtp_midi_sysex_common_nrt_gi			= -1;
static int hf_rtp_midi_sysex_common_nrt_fd			= -1;
static int hf_rtp_midi_sysex_common_tuning			= -1;
static int hf_rtp_midi_sysex_common_nrt_gm			= -1;
static int hf_rtp_midi_sysex_common_nrt_dls			= -1;

static int hf_rtp_midi_sysex_common_rt_mtc			= -1;
static int hf_rtp_midi_sysex_common_rt_sc			= -1;
static int hf_rtp_midi_sysex_common_rt_ni			= -1;
static int hf_rtp_midi_sysex_common_rt_dc			= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_cueing		= -1;
static int hf_rtp_midi_sysex_common_rt_mmc_commands		= -1;
static int hf_rtp_midi_sysex_common_rt_mmc_responses		= -1;

static int hf_rtp_midi_sysex_common_nrt_gi_device_family	= -1;
static int hf_rtp_midi_sysex_common_nrt_gi_device_family_member = -1;
static int hf_rtp_midi_sysex_common_nrt_gi_software_rev		= -1;

static int hf_rtp_midi_sysex_common_nrt_sd_packet_number	= -1;

static int hf_rtp_midi_sysex_common_nrt_sd_header_sn		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_sf		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_sp		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_sl		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_ls		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_le		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_header_lt		= -1;

static int hf_rtp_midi_sysex_common_nrt_sd_packet_count		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_packet_check		= -1;
static int hf_rtp_midi_sysex_common_nrt_sd_packet_data		= -1;

static int hf_rtp_midi_sysex_common_nrt_sd_ext_ln		= -1;

static int hf_rtp_midi_sysex_common_nrt_fd_device_id		= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_type			= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_name			= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_length		= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_packet_num		= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_byte_count		= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_packet_data		= -1;
static int hf_rtp_midi_sysex_common_nrt_fd_checksum		= -1;

static int hf_rtp_midi_sysex_common_tune_program		= -1;
static int hf_rtp_midi_sysex_common_tune_name			= -1;
static int hf_rtp_midi_sysex_common_tune_freq			= -1;
static int hf_rtp_midi_sysex_common_tune_checksum		= -1;
static int hf_rtp_midi_sysex_common_tune_changes		= -1;
static int hf_rtp_midi_sysex_common_tune_note			= -1;

static int hf_rtp_midi_sysex_common_rt_mtc_fm_type		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_fm_hr		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_fm_mn		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_fm_sc		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_fm_fr		= -1;

static int hf_rtp_midi_sysex_common_rt_mtc_ub_u1		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u2		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u3		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u4		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u5		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u6		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u7		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u8		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_ub_u9		= -1;

static int hf_rtp_midi_sysex_common_nrt_mtc_type		= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_hr			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_mn			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_sc			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_fr			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_ff			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_enl			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_enm			= -1;
static int hf_rtp_midi_sysex_common_nrt_mtc_add			= -1;

static int hf_rtp_midi_sysex_common_rt_mtc_cue_enl		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_cue_enm		= -1;
static int hf_rtp_midi_sysex_common_rt_mtc_cue_add		= -1;

static int hf_rtp_midi_sysex_common_rt_ni_bar_num		= -1;
static int hf_rtp_midi_sysex_common_rt_ni_bytes			= -1;
static int hf_rtp_midi_sysex_common_rt_ni_numerator		= -1;
static int hf_rtp_midi_sysex_common_rt_ni_denominator		= -1;
static int hf_rtp_midi_sysex_common_rt_ni_midi_clocks		= -1;
static int hf_rtp_midi_sysex_common_rt_ni_32nds			= -1;

static int hf_rtp_midi_sysex_common_rt_dc_volume		= -1;
static int hf_rtp_midi_sysex_common_rt_dc_balance		= -1;


/* RTP MIDI fields defining a subtree */

static gint ett_rtp_midi					= -1;
static gint ett_rtp_midi_commands				= -1;
static gint ett_rtp_midi_journal				= -1;
static gint ett_rtp_midi_command				= -1;
static gint ett_rtp_midi_systemjournal				= -1;
static gint ett_rtp_midi_channeljournals			= -1;
static gint ett_rtp_midi_systemchapters				= -1;
static gint ett_rtp_midi_sj_chapter_d				= -1;
static gint ett_rtp_midi_sj_chapter_d_field_b			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_g			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_h			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_j			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_k			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_y			= -1;
static gint ett_rtp_midi_sj_chapter_d_field_z			= -1;
static gint ett_rtp_midi_sj_chapter_v				= -1;
static gint ett_rtp_midi_sj_chapter_q				= -1;
static gint ett_rtp_midi_sj_chapter_f				= -1;
static gint ett_rtp_midi_sj_chapter_f_complete			= -1;
static gint ett_rtp_midi_sj_chapter_f_partial			= -1;
static gint ett_rtp_midi_sj_chapter_x				= -1;
static gint ett_rtp_midi_sj_chapter_x_first			= -1;
static gint ett_rtp_midi_sj_chapter_x_data			= -1;
static gint ett_rtp_midi_channeljournal				= -1;
static gint ett_rtp_midi_channelchapters			= -1;
static gint ett_rtp_midi_cj_chapter_p				= -1;
static gint ett_rtp_midi_cj_chapter_c				= -1;
static gint ett_rtp_midi_cj_chapter_c_loglist			= -1;
static gint ett_rtp_midi_cj_chapter_c_logitem			= -1;
static gint ett_rtp_midi_cj_chapter_m				= -1;
static gint ett_rtp_midi_cj_chapter_m_loglist			= -1;
static gint ett_rtp_midi_cj_chapter_m_logitem			= -1;
static gint ett_rtp_midi_cj_chapter_m_log_msb			= -1;
static gint ett_rtp_midi_cj_chapter_m_log_lsb			= -1;
static gint ett_rtp_midi_cj_chapter_m_log_a_button		= -1;
static gint ett_rtp_midi_cj_chapter_m_log_c_button		= -1;
static gint ett_rtp_midi_cj_chapter_m_log_count			= -1;
static gint ett_rtp_midi_cj_chapter_w				= -1;
static gint ett_rtp_midi_cj_chapter_n				= -1;
static gint ett_rtp_midi_cj_chapter_n_loglist			= -1;
static gint ett_rtp_midi_cj_chapter_n_logitem			= -1;
static gint ett_rtp_midi_cj_chapter_n_octets			= -1;
static gint ett_rtp_midi_cj_chapter_e				= -1;
static gint ett_rtp_midi_cj_chapter_e_loglist			= -1;
static gint ett_rtp_midi_cj_chapter_e_logitem			= -1;
static gint ett_rtp_midi_cj_chapter_t				= -1;
static gint ett_rtp_midi_cj_chapter_a				= -1;
static gint ett_rtp_midi_cj_chapter_a_loglist			= -1;
static gint ett_rtp_midi_cj_chapter_a_logitem			= -1;
static gint ett_rtp_midi_sysex_data				= -1;
static gint ett_rtp_midi_sysex_edu				= -1;
static gint ett_rtp_midi_sysex_manu				= -1;
static gint ett_rtp_midi_sysex_common_rt			= -1;
static gint ett_rtp_midi_sysex_common_nrt			= -1;
static gint ett_rtp_midi_sysex_common_tune_note			= -1;


static int proto_rtp_midi			= -1;


static dissector_handle_t	rtp_midi_handle;


void proto_reg_handoff_rtp_midi( void );


/*
 * This decodes the delta-time before a MIDI-command
 */
static int
decodetime(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset)
{
	guint8		octet;
	unsigned int	consumed;
	unsigned int    i;
	unsigned int    deltatime;

	consumed  = 0;
	deltatime = 0;


	/* RTP-MIDI deltatime is "compressed" using only the necessary amount of octets */
	for ( i=0; i < 4; i++ ) {

		octet = tvb_get_guint8( tvb, offset + consumed );
		deltatime = ( deltatime << 7 ) | ( octet & RTP_MIDI_DELTA_TIME_OCTET_MASK );
		consumed++;

		if ( ( octet & RTP_MIDI_DELTA_TIME_EXTENSION ) == 0 ) {
			break;
		}
	}

	switch (consumed) {
		case 1:
		    proto_tree_add_uint ( tree, hf_rtp_midi_deltatime1, tvb, offset, consumed, deltatime );
			break;
		case 2:
		    proto_tree_add_uint ( tree, hf_rtp_midi_deltatime2, tvb, offset, consumed, deltatime );
			break;
		case 3:
		    proto_tree_add_uint ( tree, hf_rtp_midi_deltatime3, tvb, offset, consumed, deltatime );
			break;
		case 4:
		    proto_tree_add_uint ( tree, hf_rtp_midi_deltatime4, tvb, offset, consumed, deltatime );
			break;

	}

	return consumed;
}



/*
 * Here a Note-Off command is decoded.
 */
static int
decode_note_off(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 note;
	guint8		 velocity;
	const gchar	*status_str;
	const gchar	*note_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	note = tvb_get_guint8( tvb, offset );
	note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );
	velocity = tvb_get_guint8( tvb, offset + 1 );

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_velocity, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command , NULL,
				"%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_velocity, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
	}
	return 2;
}


/*
 * Here a Note-On command is decoded.
 */
static int
decode_note_on(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 note;
	guint8		 velocity;
	const gchar	*status_str;
	const gchar	*note_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	note = tvb_get_guint8( tvb, offset );
	note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );
	velocity = tvb_get_guint8( tvb, offset + 1 );

	/* special case velocity=0 for Note-On means Note-Off (to preserve running-status!) */
	if ( velocity == 0 ) {
		status_str = rtp_midi_channel_status_special_off;
	}

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_velocity, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command, NULL,
				"%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_velocity, tvb, offset + 1, 1, ENC_BIG_ENDIAN);
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, n=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, velocity );
	}
	return 2;
}




/*
 * Here polyphonic aftertouch is decoded.
 */
static int
decode_poly_pressure(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 note;
	guint8		 pressure;
	const gchar	*status_str;
	const gchar	*note_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	note = tvb_get_guint8( tvb, offset );
	note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );
	pressure = tvb_get_guint8( tvb, offset + 1 );

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, n=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, pressure );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_pressure, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command, NULL,
				"%s (c=%d, n=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, pressure );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_note, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_pressure, tvb, offset + 1, 1, ENC_BIG_ENDIAN);
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, n=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, pressure );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, n=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, note_str, pressure );
	}
	return 2;
}


/*
 * Here channel aftertouch is decoded.
 */
static int
decode_channel_pressure(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 pressure;
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	pressure = tvb_get_guint8( tvb, offset );

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 1, ett_rtp_midi_command, NULL,
				"%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pressure );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_pressure, tvb, offset, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pressure );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_pressure, tvb, offset, 1, ENC_BIG_ENDIAN);
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pressure );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pressure );
	}
	return 1;
}



/*
 * Here pitch-bend is decoded.
 */
static int
decode_pitch_bend_change(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 octet1;
	guint8		 octet2;
	guint8		 pitch;
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	octet1 = tvb_get_guint8( tvb, offset );
	octet2 = tvb_get_guint8( tvb, offset + 1 );
	pitch = ( octet1 << 7 ) | octet2;

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, pb=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pitch );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_pitch_bend, tvb, offset, 2, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command, NULL,
				"%s (c=%d, pb=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pitch );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_pitch_bend, tvb, offset, 2, ENC_BIG_ENDIAN );
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, pb=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pitch );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, pb=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, pitch );
	}
	return 2;
}




/*
 * Here program_change is decoded.
 */
static int
decode_program_change(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 program;
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	program = tvb_get_guint8( tvb, offset );

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 1, ett_rtp_midi_command, NULL,
				"%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, program );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_program, tvb, offset, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 2, ett_rtp_midi_command, NULL,
				"%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, program );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_program, tvb, offset, 1, ENC_BIG_ENDIAN);
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, program );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, program );
	}

	return 1;
}



/*
 * Here control change is decoded.
 */
static int
decode_control_change(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, unsigned int cmd_count, unsigned int offset, guint8 status, unsigned int rsoffset, gboolean using_rs ) {

	guint8		 controller;
	guint8		 value;
	const gchar	*status_str;
	const gchar	*ctrl_str;
	proto_tree	*command_tree;

	status_str = val_to_str( status >> 4, rtp_midi_channel_status, rtp_midi_unknown_value_hex );
	controller = tvb_get_guint8( tvb, offset );
	ctrl_str = val_to_str_ext( controller, &rtp_midi_controller_values_ext, "Unknown: %d" );
	value = tvb_get_guint8( tvb, offset + 1 );

	if ( using_rs ) {
		command_tree = proto_tree_add_subtree_format( tree, tvb, offset, 2,  ett_rtp_midi_command, NULL,
				"%s (c=%d, ctrl=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, ctrl_str, value );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, rsoffset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_controller, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_controller_value, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
	} else {
		command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command, NULL,
				"%s (c=%d, ctrl=%s, v=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, ctrl_str, value );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_channel, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_controller, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( command_tree, hf_rtp_midi_controller_value, tvb, offset + 1, 1, ENC_BIG_ENDIAN);
	}

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (c=%d, ctrl=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, ctrl_str, value );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (c=%d, ctrl=%s, p=%d)", status_str, ( status & RTP_MIDI_CHANNEL_MASK ) + 1, ctrl_str, value );
	}
	return 2;
}


/*
 * Here a Sysex-Common Non-Realtime Sample Dump Header command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_sd_hdr( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	unsigned int start_offset = offset;

	/* sample number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sn, tvb, offset, 2, ENC_BIG_ENDIAN );
	offset	 += 2;

	/* sample format */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sf, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	/* sample period */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sp, tvb, offset, 3, ENC_BIG_ENDIAN );
	offset	 += 3;

	/* sample length */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sl, tvb, offset, 3, ENC_BIG_ENDIAN );
	offset	 += 3;

	/* loop start */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_ls, tvb, offset, 3, ENC_BIG_ENDIAN );
	offset	 += 3;

	/* loop end */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_le, tvb, offset, 3, ENC_BIG_ENDIAN );
	offset	 += 3;

	/* loop type */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_lt, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Non-Realtime Sample Dump Packet command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_sd_packet( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	unsigned int start_offset = offset;

	/* sample number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_count, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	/* do we have sample-data? */
	if ( data_len - 1 > 1 ) {

		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_data, tvb, offset, data_len - 2, ENC_NA );

		offset	 += ( data_len - 2 );
	}

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_check, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Non-Realtime Sample Dump Request command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_sd_req( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset) {

	/* sample number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sn, tvb, offset, 2, ENC_BIG_ENDIAN );

	return 2;
}


/*
 * Here a Sysex-Common Non-Realtime Sample Dump Extension command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_sd_ext( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_ext, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINT_TRANSMISSION ) {

		/* sample number */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sn, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	 += 2;

		/* loop number */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_ext_ln, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	 += 2;

		/* loop type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_lt, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* loop start */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_ls, tvb, offset, 3, ENC_BIG_ENDIAN );
		offset	 += 3;

		/* loop end */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_le, tvb, offset, 3, ENC_BIG_ENDIAN );
		offset += 3;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_SD_EXT_LOOP_POINTS_REQUEST ) {

		/* sample number */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_header_sn, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	 += 2;

		/* loop number */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_ext_ln, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset += 2;
	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Non-Realtime General Information command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_gi( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	int		 consumed	= 0;

	if ( data_len < 1 )
		return consumed;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_gi, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;
	consumed++;


	/* This is the only sub-command that we know that we can decode,
	 * the Identity Request has already been decoded up to this point. */
	if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REQUEST ) {

		/* nothing more to do... */

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_GI_IDENTITY_REPLY ) {
		guint8 manu_short;

		if ( data_len < 1 ) {
			return consumed;
		}

		/* the manu-id should follow */
		proto_tree_add_item( tree, hf_rtp_midi_manu_short, tvb, offset, 1, ENC_BIG_ENDIAN );
		manu_short = tvb_get_guint8( tvb, offset );
		offset++;
		data_len--;
		consumed++;

		/* possibly escaped to be a long manu-id */
		if ( manu_short == RTP_MIDI_MANU_SHORT_ISLONG ) {
			if ( data_len < 2 ) {
				return consumed;
			}
			proto_tree_add_item( tree, hf_rtp_midi_manu_long, tvb, offset, 2, ENC_BIG_ENDIAN );
			offset	 += 2 ;
			data_len -= 2;
			consumed += 2;
		}

		/* lets see if we can get the device family for this unit */
		if ( data_len < 2 ) {
			return consumed;
		}
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_gi_device_family, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	 += 2 ;
		data_len -= 2;
		consumed += 2;

		/* ...also get the detailed member of this family... */
		if ( data_len < 2 ) {
			return consumed;
		}
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_gi_device_family_member, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	 += 2 ;
		data_len -= 2;
		consumed += 2;

		/* and even the software-revision-level */
		if ( data_len < 4 ) {
			return consumed;
		}
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_gi_software_rev, tvb, offset, 4, ENC_BIG_ENDIAN );
		consumed += 4;
	}

	return consumed;
}

/*
 * Here a Sysex-Common Non-Realtime File Dump command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_fd( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_FD_HEADER ) {

		/* type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_device_id, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		data_len--;

		/* file type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_type, tvb, offset, 4, ENC_ASCII );
		offset	 += 4;
		data_len -= 4;

		/* file length */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_length, tvb, offset, 4, ENC_BIG_ENDIAN );
		offset	 += 4;
		data_len -= 4;

		if ( data_len ) {

			/* file-name */
			proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_name, tvb, offset, data_len, ENC_ASCII );
			offset	 += data_len;
		}

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_FD_DATA_PACKET ) {

		/* packet-num */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_packet_num, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		data_len--;

		/* byte count */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_byte_count, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		data_len--;

		/* do we have file-data? */
		if ( data_len > 1 ) {

			proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_packet_data, tvb, offset, data_len - 1, ENC_NA );

			offset	 += ( data_len - 1 );
		}

		proto_tree_add_checksum(tree, tvb, offset, hf_rtp_midi_sysex_common_nrt_fd_checksum, -1, NULL, pinfo, 0, ENC_BIG_ENDIAN, PROTO_CHECKSUM_NO_FLAGS);
		offset++;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_NRT_FD_REQUEST ) {

		/* type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_device_id, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		data_len--;

		/* file type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_type, tvb, offset, 4, ENC_ASCII );
		offset	 += 4;
		data_len -= 4;

		if ( data_len ) {

			/* file-name */
			proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_fd_name, tvb, offset, data_len, ENC_ASCII );
			offset += data_len;
		}
	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common (Non-)Realtime MIDI Tuning Standard command is decoded.
 * As the code-points do not overlap, both RT and NRT are decoded here...
 */
static unsigned int
decode_sysex_common_tuning( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	proto_tree	*tune_tree;
	const gchar	*note_str;
	guint8		 sub_id;
	unsigned int start_offset = offset;
	unsigned int	 i;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tuning, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REQUEST ) {

		/* type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tune_program, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_TUNING_BULK_DUMP_REPLY ) {

		/* type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tune_program, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* file length */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tune_name, tvb, offset, 16, ENC_ASCII );
		offset	 += 16;

		for ( i=0; i < 128; i++ ) {
			note_str = val_to_str_ext( i, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );

			tune_tree = proto_tree_add_subtree_format(tree, tvb, offset, 3, ett_rtp_midi_sysex_common_tune_note, NULL, "Note: %s", note_str );

			/* frequency */
			proto_tree_add_item( tune_tree, hf_rtp_midi_sysex_common_tune_freq, tvb, offset, 3, ENC_BIG_ENDIAN );
			offset	 += 3;
		}

		/* checksum */
		proto_tree_add_checksum(tree, tvb, offset, hf_rtp_midi_sysex_common_tune_checksum, -1, NULL, pinfo, 0, ENC_BIG_ENDIAN, PROTO_CHECKSUM_NO_FLAGS);
		offset++;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_TUNING_NOTE_CHANGE ) {
		guint8 changes;

		/* type */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tune_program, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* changes */
		changes = tvb_get_guint8( tvb, offset );
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_tune_changes, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		for ( i=0; i < changes; i++ ) {
			guint8 note;

			note = tvb_get_guint8( tvb, offset );

			note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );

			tune_tree = proto_tree_add_subtree_format(tree, tvb, offset, 3, ett_rtp_midi_sysex_common_tune_note, NULL, "Note: %s", note_str );

			/* note */
			proto_tree_add_item( tune_tree, hf_rtp_midi_sysex_common_tune_note, tvb, offset, 1, ENC_BIG_ENDIAN );
			offset++;

			/* frequency */
			proto_tree_add_item( tune_tree, hf_rtp_midi_sysex_common_tune_freq, tvb, offset, 3, ENC_BIG_ENDIAN );
			offset	 += 3;
		}

	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Non-Realtime General MIDI command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_gm( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_gm, tvb, offset, 1, ENC_BIG_ENDIAN );
	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime Downloadable Sounds command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_dls( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_dls, tvb, offset, 1, ENC_BIG_ENDIAN );
	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime End Of File command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_eof( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	if ( data_len < 1 )
		return 0;

	/* we only have a packet-number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_number, tvb, offset, 1, ENC_BIG_ENDIAN );

	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime Wait command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_wait( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	/* we only have a packet-number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_number, tvb, offset, 1, ENC_BIG_ENDIAN );

	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime Cancel command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_cancel( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	/* we only have a packet-number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_number, tvb, offset, 1, ENC_BIG_ENDIAN );

	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime NAK command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_nak( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	/* we only have a packet-number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_number, tvb, offset, 1, ENC_BIG_ENDIAN );

	return 1;
}

/*
 * Here a Sysex-Common Non-Realtime ACK command is decoded.
 */
static unsigned int
decode_sysex_common_nrt_ack( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	/* we only have a packet-number */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_sd_packet_number, tvb, offset, 1, ENC_BIG_ENDIAN );

	return 1;
}

/*
 * here a sysex-common non-realtime midi time code cueing command is decoded.
 * as the codepoints are the same, we decode both realtime and non-realtime here.
 */
static unsigned int
decode_sysex_common_nrt_mtc( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* type and hours */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_type, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_hr, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* minutes */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_mn, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* seconds */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_sc, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* frames */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_fr, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* fractional frames */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_ff, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* event-number (lsb) */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_enl, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* event-number (msb) */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_enm, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	if ( data_len ) {

		/* additional data */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_nrt_mtc_add, tvb, offset, data_len, ENC_NA );
		offset += data_len;
	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Realtime MIDI Time Code Cueing command is decoded.
 * As the codepoints are the same, we decode both realtime and non-realtime here.
 */
static unsigned int
decode_sysex_common_rt_mtc_cue( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_cueing, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* event-number (lsb) */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_cue_enl, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	/* event-number (msb) */
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_cue_enm, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;

	if ( data_len ) {

		/* additional data */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_cue_add, tvb, offset, data_len, ENC_NA );
		offset += data_len;
	}

	return offset-start_offset;
}




/*
 * Here a Sysex-Common Non-Realtime command is decoded.
 */
static unsigned int
decode_sysex_common_nrt( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 common_nrt;
	const gchar	*nrt_str;
	proto_tree	*command_tree;
	int		 consumed	= 0;
	int		 ext_consumed	= 0;

	if ( data_len < 1 )
		return consumed;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_device_id, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;
	consumed++;

	if ( data_len < 1 ) {
		return consumed;
	}

	common_nrt = tvb_get_guint8( tvb, offset );

	nrt_str = val_to_str( common_nrt, rtp_midi_sysex_common_nrt, "Unknown 0x%02x" );

	command_tree = proto_tree_add_subtree(tree, tvb, offset, data_len, ett_rtp_midi_sysex_common_nrt, NULL, nrt_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_sysex_common_non_realtime, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;
	data_len--;
	consumed++;

	switch ( common_nrt ) {
		case RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_HEADER:
			ext_consumed = decode_sysex_common_nrt_sd_hdr( tvb, pinfo, command_tree, offset);
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DATA_PACKET:
			ext_consumed = decode_sysex_common_nrt_sd_packet( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_REQUEST:
			ext_consumed = decode_sysex_common_nrt_sd_req( tvb, pinfo, command_tree, offset);
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_MTC:
			ext_consumed = decode_sysex_common_nrt_mtc( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_SAMPLE_DUMP_EXTENSIONS:
			ext_consumed = decode_sysex_common_nrt_sd_ext( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_INFORMATION:
			ext_consumed = decode_sysex_common_nrt_gi( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_FILE_DUMP:
			ext_consumed = decode_sysex_common_nrt_fd( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_MIDI_TUNING_STANDARD:
			ext_consumed = decode_sysex_common_tuning( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_GENERAL_MIDI:
			ext_consumed = decode_sysex_common_nrt_gm( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_DOWNLOADABLE_SOUNDS:		/* ??? not in Spec */
			ext_consumed = decode_sysex_common_nrt_dls( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_END_OF_FILE:
			ext_consumed = decode_sysex_common_nrt_eof( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_WAIT:
			ext_consumed = decode_sysex_common_nrt_wait( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_CANCEL:
			ext_consumed = decode_sysex_common_nrt_cancel( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_NAK:
			ext_consumed = decode_sysex_common_nrt_nak( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_NRT_ACK:
			ext_consumed = decode_sysex_common_nrt_ack( tvb, pinfo, command_tree, offset, data_len );
			break;
	}

	if ( ext_consumed < 0 ) {
		return -1;
	}

	/* set our pointers correct to move past already decoded data */
	offset	 += ext_consumed;
	data_len -= ext_consumed;
	consumed += ext_consumed;

	if ( data_len > 0 ) {
		proto_tree_add_item( command_tree, hf_rtp_midi_unknown_data, tvb, offset, data_len, ENC_NA );
		consumed += data_len;
	}

	return consumed;
}






/*
 * Here a Sysex-Common Realtime MIDI Time Code command is decoded.
 */
static unsigned int
decode_sysex_common_rt_mtc( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_MTC_FULL_MESSAGE ) {

		/* type and hours */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_fm_type, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_fm_hr, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* minutes */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_fm_mn, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* seconds */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_fm_sc, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* frames */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_fm_fr, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_MTC_USER_BITS ) {

		/* U1 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u1, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U2 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u2, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U3 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u3, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U4 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u4, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U5 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u5, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U6 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u6, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U7 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u7, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U8 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u8, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* U9 */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mtc_ub_u9, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Realtime MIDI Show Control command is decoded.
 */
static unsigned int
decode_sysex_common_rt_sc( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_sc, tvb, offset, 1, ENC_BIG_ENDIAN );

	/* TODO: decode Show Control stuff */
	return 1;
}

/*
 * Here a Sysex-Common Realtime Notation Information command is decoded.
 */
static unsigned int
decode_sysex_common_rt_ni( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_NT_BAR_NUMBER ) {

		/* bar number */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_bar_num, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset += 2;

	} else if ( ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_IMMEDIATE ) ||
			( sub_id == RTP_MIDI_SYSEX_COMMON_RT_NT_TIME_SIGNATURE_DELAYED ) ) {
		int bytes;

		/* bytes */
		bytes = tvb_get_guint8( tvb, offset );
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_bytes, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;

		/* beats / numerator */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_numerator, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		bytes--;

		/* beats / denominator */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_denominator, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		bytes--;

		/* clocks */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_midi_clocks, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		bytes--;

		/* 32nds */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_32nds, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		bytes--;

		while ( bytes > 0 ) {

			/* beats / numerator */
			proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_numerator, tvb, offset, 1, ENC_BIG_ENDIAN );
			offset++;
			bytes--;

			/* beats / denominator */
			proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_ni_denominator, tvb, offset, 1, ENC_BIG_ENDIAN );
			offset++;
			bytes--;
		}
	}

	return offset-start_offset;
}

/*
 * Here a Sysex-Common Realtime Device Control command is decoded.
 */
static unsigned int
decode_sysex_common_rt_dc( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 sub_id;
	unsigned int start_offset = offset;

	if ( data_len < 1 )
		return 0;

	/* first we retrieve the sub-command */
	sub_id = tvb_get_guint8( tvb, offset );
	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_dc, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	if ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_VOLUME ) {

		/* volume */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_dc_volume, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset += 2;

	} else if ( sub_id == RTP_MIDI_SYSEX_COMMON_RT_DC_MASTER_BALANCE ) {

		/* balance */
		proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_dc_balance, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset += 2;

	}

	return offset-start_offset;
}


/*
 * Here a Sysex-Common Realtime MIDI Machine Control command is decoded.
 */
static unsigned int
decode_sysex_common_rt_mmc_command( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mmc_commands, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;

	/* TODO: decode MMC stuff */
	return 1;
}

/*
 * Here a Sysex-Common Realtime MIDI Machine Control response is decoded.
 */
static unsigned int
decode_sysex_common_rt_mmc_response( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {

	if ( data_len < 1 )
		return 0;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_rt_mmc_responses, tvb, offset, 1, ENC_BIG_ENDIAN );

	/* TODO: decode MMC stuff */
	return 1;
}





/*
 * Here a Sysex-Common Realtime command is decoded.
 */
static unsigned int
decode_sysex_common_rt( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	guint8		 common_rt;
	const gchar	*rt_str;
	proto_tree	*command_tree;
	int		 consumed	= 0;
	int		 ext_consumed	= 0;

	if ( data_len < 1 )
		return consumed;

	proto_tree_add_item( tree, hf_rtp_midi_sysex_common_device_id, tvb, offset, 1, ENC_BIG_ENDIAN );
	offset++;
	data_len--;
	consumed++;

	if ( data_len < 1 ) {
		return consumed;
	}

	common_rt = tvb_get_guint8( tvb, offset );
	rt_str = val_to_str( common_rt, rtp_midi_sysex_common_rt, "Unknown 0x%02x" );

	command_tree = proto_tree_add_subtree(tree, tvb, offset, data_len, ett_rtp_midi_sysex_common_rt, NULL, rt_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_sysex_common_realtime, tvb, offset, 1, ENC_BIG_ENDIAN );

	data_len--;
	offset++;
	consumed++;

	switch ( common_rt ) {
		case RTP_MIDI_SYSEX_COMMON_RT_MIDI_TIME_CODE:
			ext_consumed = decode_sysex_common_rt_mtc( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_MIDI_SHOW_CONTROL:
			ext_consumed = decode_sysex_common_rt_sc( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_NOTATION_INFORMATION:
			ext_consumed = decode_sysex_common_rt_ni( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_DEVICE_CONTROL:
			ext_consumed = decode_sysex_common_rt_dc( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_MTC_CUEING:
			ext_consumed = decode_sysex_common_rt_mtc_cue( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_COMMAND:
			ext_consumed = decode_sysex_common_rt_mmc_command( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_MIDI_MACHINE_CONTROL_RESPONSE:
			ext_consumed = decode_sysex_common_rt_mmc_response( tvb, pinfo, command_tree, offset, data_len );
			break;
		case RTP_MIDI_SYSEX_COMMON_RT_MIDI_TUNING_STANDARD:
			ext_consumed = decode_sysex_common_tuning( tvb, pinfo, command_tree, offset, data_len );
			break;
	}

	if ( ext_consumed < 0 ) {
		return -1;
	}

	/* set our pointers correct to move past already decoded data */
	offset	 += ext_consumed;
	data_len -= ext_consumed;
	consumed += ext_consumed;

	if ( data_len > 0 ) {
		proto_tree_add_item( command_tree, hf_rtp_midi_unknown_data, tvb, offset, data_len, ENC_NA );
		consumed += data_len;
	}

	return consumed;
}


/*
 * Here a Sysex-Common Educational command is decoded.
 * According to the MIDI-specification, this should never be found "in the wild".
 * We don't know what this data encodes, so we just dump it.
 */
static unsigned int
decode_sysex_common_educational( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len ) {
	proto_tree	*command_tree;
	int		 consumed	= 0;

	if ( data_len < 1 )
		return consumed;

	command_tree = proto_tree_add_subtree(tree, tvb, offset, data_len, ett_rtp_midi_sysex_edu, NULL, RTP_MIDI_TREE_NAME_COMMAND_SYSEX_EDU );
	proto_tree_add_item( command_tree, hf_rtp_midi_edu_data, tvb, offset, data_len, ENC_NA );

	/* We consumed everything */
	return data_len;
}

/*
 * Here a Sysex-Common Manufacturer command is decoded.
 * We don't know what this data encodes, so we just dump it.
 */
static unsigned int
decode_sysex_common_manufacturer( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int data_len, unsigned int manu_code _U_) {
	proto_tree	*command_tree;
	int		 consumed	= 0;

	if ( data_len < 1 )
		return consumed;

	/* Instead of dumping, at this point we could implement to start manufacturer-specific decoding
	 * of the sysex-data...
	 */

	command_tree = proto_tree_add_subtree(tree, tvb, offset, data_len, ett_rtp_midi_sysex_manu, NULL, RTP_MIDI_TREE_NAME_COMMAND_SYSEX_MANU );

	proto_tree_add_item( command_tree, hf_rtp_midi_manu_data, tvb, offset, data_len, ENC_NA );

	/* We consumed everything */
	return data_len;
}



/*
 * Here a Sysex-Start command is decoded.
 */
static unsigned int
decode_sysex_start(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len ) {
	const gchar	*status_str;
	proto_tree	*command_tree;
	int		 consumed	= 0;
	int		 data_len;
	int		 ext_consumed	= 0;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_SYSEX_END, rtp_midi_common_status, rtp_midi_unknown_value_hex );

	/* we need to parse "away" data until the next command */
	while ( cmd_len ) {
		guint8 octet;
		octet = tvb_get_guint8( tvb, offset + consumed );
		consumed++;

		/*
		 * lets check if we have a "normal" end -
		 * if so, the last status-byte is consumed
		 * as it belongs to the message
		 */
		if ( octet == RTP_MIDI_STATUS_COMMON_SYSEX_END ) {
			status_str = rtp_midi_common_status_sysex_segment_complete;
		} else if ( octet == RTP_MIDI_STATUS_COMMON_SYSEX_START ) {
			status_str = rtp_midi_common_status_sysex_segment_start;
		} else if ( octet == RTP_MIDI_STATUS_COMMON_UNDEFINED_F4 ) {
			status_str = rtp_midi_common_status_sysex_cancel;
		}

		/* Is this command through? */
		if ( octet & RTP_MIDI_COMMAND_STATUS_FLAG ) {
			break;
		}
	}

	/* the unknown data we have does not include the trailing status-command */
	data_len = consumed - 1;

	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 1 + consumed, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );

	/* lets see if we have enough data for the sysex-id */
	if ( data_len > 1 ) {
		guint8  manu_short;
		guint16 manufacturer;
		manufacturer = tvb_get_guint8( tvb, offset );
		proto_tree_add_item( command_tree, hf_rtp_midi_manu_short, tvb, offset, 1, ENC_BIG_ENDIAN );
		manu_short = tvb_get_guint8( tvb, offset );
		offset++;
		data_len--;

		switch( manu_short ) {
			case RTP_MIDI_MANU_SHORT_NON_REALTIME_UNIVERSAL:
				ext_consumed = decode_sysex_common_nrt( tvb, pinfo, command_tree, offset, data_len );
				break;
			case RTP_MIDI_MANU_SHORT_REALTIME_UNIVERSAL:
				ext_consumed = decode_sysex_common_rt( tvb, pinfo, command_tree, offset, data_len );
				break;
			case RTP_MIDI_MANU_SHORT_EDUCATIONAL_USE:
				ext_consumed = decode_sysex_common_educational( tvb, pinfo, command_tree, offset, data_len );
				break;
			case RTP_MIDI_MANU_SHORT_ISLONG:
				/* lets see if we have enough data to decode a long sysex-id and if we have a long id at all */
				if ( data_len > 2 ) {
					manufacturer = tvb_get_ntohs( tvb, offset );
					proto_tree_add_item( command_tree, hf_rtp_midi_manu_long, tvb, offset, 2, ENC_BIG_ENDIAN );
					offset	     +=	2 ;
					data_len     -= 2;
					ext_consumed  = decode_sysex_common_manufacturer( tvb, pinfo, command_tree, offset, data_len, manufacturer );
				}  else {
					ext_consumed  = -1;
				}
				break;
			default:
				ext_consumed = decode_sysex_common_manufacturer( tvb, pinfo, command_tree, offset, data_len, manufacturer );
				break;
		}
	}

	if ( ext_consumed < 0 ) {
		return -1;
	}

	/* set our pointers correct to move past already decoded data */
	offset += ext_consumed;

	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}

	return consumed;
}

/*
 * Here the MIDI-Time-Code (MTC) Quarter Frame command is decoded.
 */
static int
decode_mtc_quarter_frame(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset) {
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_MTC_QUARTER_FRAME, rtp_midi_common_status, rtp_midi_unknown_value_hex );

	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 2, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( command_tree, hf_rtp_midi_quarter_frame_type, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( command_tree, hf_rtp_midi_quarter_frame_value, tvb, offset + 1, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO, ", %s", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}
	return 1;
}

/*
 * Here the Song Position Pointer command is decoded.
 */
static int
decode_song_position_pointer(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset) {
	guint8		 octet1;
	guint8		 octet2;
	guint8		 position;
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_SONG_POSITION_POINTER, rtp_midi_common_status, rtp_midi_unknown_value_hex );
	octet1 = tvb_get_guint8( tvb, offset );
	octet2 = tvb_get_guint8( tvb, offset + 1 );
	position = ( octet1 << 7 ) | octet2;

	command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 3, ett_rtp_midi_command, NULL, "%s (p=%d)", status_str, position );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( command_tree, hf_rtp_midi_spp, tvb, offset, 2, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (p=%d)", status_str, position );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (p=%d)", status_str, position );
	}
	return 2;
}

/*
 * Here a Song-Select command is decoded.
 */
static int
decode_song_select(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset) {
	guint8		 song_nr;
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_SONG_SELECT, rtp_midi_common_status, rtp_midi_unknown_value_hex );
	song_nr = tvb_get_guint8( tvb, offset );

	command_tree = proto_tree_add_subtree_format(tree, tvb, offset - 1, 2, ett_rtp_midi_command, NULL, "%s (s=%d)", status_str, song_nr );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( command_tree, hf_rtp_midi_song_select, tvb, offset, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s (s=%d)", status_str, song_nr );
	} else {
		col_append_fstr(pinfo->cinfo, COL_INFO, "%s (s=%d)", status_str, song_nr );
	}
	return 1;
}

/*
 * Here the undefined common-command 0xf4 is decoded.
 */
static int
decode_undefined_f4(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len ) {
	const gchar	*status_str;
	proto_tree	*command_tree;
	int		 consumed	= 0;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_UNDEFINED_F4, rtp_midi_common_status, rtp_midi_unknown_value_hex );

	/* we need to parse "away" data until the next command */
	while ( cmd_len ) {
		guint8 octet;
		octet = tvb_get_guint8( tvb, offset + consumed );
		consumed++;

		/* Is this command through? */
		if ( octet & RTP_MIDI_COMMAND_STATUS_FLAG ) {
			break;
		}
	}

	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 1 + consumed, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	/* just add the unknown data to the tree */
	proto_tree_add_item( command_tree, hf_rtp_midi_unknown_data, tvb, offset, consumed, ENC_NA );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset + consumed - 1, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}
	return consumed;
}

/*
 * Here the undefined common-command 0xf5 is decoded.
 */
static int
decode_undefined_f5(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len ) {
	const gchar	*status_str;
	proto_tree	*command_tree;
	int		 consumed	= 0;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_UNDEFINED_F5, rtp_midi_common_status, rtp_midi_unknown_value_hex );

	/* we need to parse "away" data until the next command */
	while ( cmd_len ) {
		guint8 octet;
		octet = tvb_get_guint8( tvb, offset + consumed );
		consumed++;
		/* Is this command done? */
		if ( octet & RTP_MIDI_COMMAND_STATUS_FLAG ) {
			break;
		}
	}

	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 1 + consumed, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	/* just add the unknown data to the tree */
	proto_tree_add_item( command_tree, hf_rtp_midi_unknown_data, tvb, offset, consumed, ENC_NA );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset + consumed - 1, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}
	return consumed;
}

/*
 * Here a Tune-Request command is decoded.
 */
static int
decode_tune_request(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len _U_ ) {
	const gchar	*status_str;
	proto_tree	*command_tree;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_TUNE_REQUEST, rtp_midi_common_status, rtp_midi_unknown_value_hex );
	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 1, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );

	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s)", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}
	return 0;
}

/*
 * Here a Sysex-End command is decoded - in RTP-MIDI this has a special semantic, it either starts a segmented Sysex-frame or a Sysex-Cancel
 */
static int
decode_sysex_end(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len ) {
	const gchar	*status_str;
	proto_tree	*command_tree;
	int		 consumed = 0;

	status_str = val_to_str( RTP_MIDI_STATUS_COMMON_SYSEX_END, rtp_midi_common_status, rtp_midi_unknown_value_hex );

	/* we need to parse "away" data until the next command */
	while ( cmd_len ) {
		guint8 octet;
		octet = tvb_get_guint8( tvb, offset + consumed );
		consumed++;

		/*
		 * lets check if we have a "normal" end -
		 * if so, the last status-byte is consumed
		 * as it belongs to the message
		 */
		if ( octet == RTP_MIDI_STATUS_COMMON_SYSEX_END ) {
			status_str = rtp_midi_common_status_sysex_segment_end;
		} else if ( octet == RTP_MIDI_STATUS_COMMON_SYSEX_START ) {
			status_str = rtp_midi_common_status_sysex_segment;
		} else if ( octet == RTP_MIDI_STATUS_COMMON_UNDEFINED_F4 ) {
			status_str = rtp_midi_common_status_sysex_cancel;
		}

		/* Is this command through? */
		if ( octet & RTP_MIDI_COMMAND_STATUS_FLAG ) {
			break;
		}
	}

	command_tree = proto_tree_add_subtree(tree, tvb, offset - 1, 1 + consumed, ett_rtp_midi_command, NULL, status_str );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset - 1, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( command_tree, hf_rtp_midi_unknown_data, tvb, offset, consumed - 1, ENC_NA );
	proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset + consumed - 1, 1, ENC_BIG_ENDIAN );


	if ( cmd_count ) {
		col_append_fstr(pinfo->cinfo, COL_INFO,", %s", status_str );
	} else {
		col_append_str(pinfo->cinfo, COL_INFO, status_str );
	}

	return consumed;
}



/*
 * Here each single MIDI-command is decoded.
 * The Status-octet is read and then the decoding
 * of the individual MIDI-commands is punted to
 * external decoders.
 */

static int
decodemidi(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int cmd_count, unsigned int offset, unsigned int cmd_len, guint8 *runningstatus, unsigned int *rsoffset )
{
	int		 consumed	= 0;
	int		 ext_consumed	= 0;
	guint8		 octet;
	gboolean	 using_rs;


#if 0
	guint8		octet2;
	guint8		octet3;
	guint8		cmd;
	guint8		channel;
	proto_tree	*sysexdata_tree   = NULL;
	gboolean	foundend;
	guint16		wordvalue;
	char		helptext[200];
	char		entry[200];
	char		segtype[200];
	guint8		manu1;
	guint16		manu2;
	guint8		subid1;
	guint8		subid2;
	guint8		sysexchan;
	unsigned int	helpoffset;
#endif

	octet = tvb_get_guint8( tvb, offset );

	/* midi realtime-data -> one octet  -- unlike serial-wired MIDI realtime-commands in RTP-MIDI will
	 * not be intermingled with other MIDI-commands, so we handle this case right here and return */
	if ( octet >= 0xf8 ) {
		proto_tree  *command_tree;
		const gchar *valstr;

		valstr =  val_to_str( octet, rtp_midi_common_status, rtp_midi_unknown_value_hex );
		command_tree = proto_tree_add_subtree(tree, tvb, offset, 1, ett_rtp_midi_command, NULL, valstr );
		proto_tree_add_item( command_tree, hf_rtp_midi_common_status, tvb, offset, 1, ENC_BIG_ENDIAN );

		if ( cmd_count ) {
			col_add_fstr(pinfo->cinfo, COL_INFO,", %s", valstr );
		} else {
			col_add_str(pinfo->cinfo, COL_INFO, valstr );
		}
		return 1;
	}



	/* see if this first octet is a status message */
	if ( ( octet & RTP_MIDI_COMMAND_STATUS_FLAG ) == 0 ) {
		/* if we have no running status yet -> error */
		if ( ( ( *runningstatus ) & RTP_MIDI_COMMAND_STATUS_FLAG ) == 0 ) {
			return -1;
		}
		/* our first octet is "virtual" coming from a preceding MIDI-command,
		 * so actually we have not really consumed anything yet */
		octet = *runningstatus;
		using_rs = TRUE;
	} else {
		/* We have a "real" status-byte */
		using_rs = FALSE;

		/* Let's see how this octet influences our running-status */
		/* if we have a "normal" MIDI-command then the new status replaces the current running-status */
		if ( octet < 0xf0 ) {
			*rsoffset = offset ;
			*runningstatus = octet;
		} else {
			/* system-realtime-commands maintain the current running-status
			 * other system-commands clear the running-status, since we
			 * already handled realtime, we can reset it here */
			*runningstatus = 0;
		}

		/* lets update our pointers */
		consumed++;
		cmd_len--;
		offset++;
	}




	/* non-system MIDI-commands encode the command in the high nibble and the channel
	 * in the low nibble - so we will take care of those cases next */

	if ( octet < 0xf0 ) {
		switch ( octet >> 4 ) {
			case RTP_MIDI_STATUS_CHANNEL_NOTE_OFF:
				ext_consumed = decode_note_off( tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_NOTE_ON:
				ext_consumed = decode_note_on( tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_POLYPHONIC_KEY_PRESSURE:
				ext_consumed = decode_poly_pressure(tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_CONTROL_CHANGE:
				ext_consumed = decode_control_change(tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_PROGRAM_CHANGE:
				ext_consumed = decode_program_change(tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_CHANNEL_PRESSURE:
				ext_consumed = decode_channel_pressure(tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
			case RTP_MIDI_STATUS_CHANNEL_PITCH_BEND_CHANGE:
				ext_consumed = decode_pitch_bend_change(tvb, pinfo, tree, cmd_count, offset, octet, *rsoffset, using_rs );
				break;
		}
		/* external decoder informed us of error -> pass this through */
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		return consumed + ext_consumed;
	}

	/* Here we catch the remaining system-common commands */
	switch ( octet ) {
		case RTP_MIDI_STATUS_COMMON_SYSEX_START:
			ext_consumed =  decode_sysex_start( tvb, pinfo, tree, cmd_count, offset, cmd_len );
			break;
		case RTP_MIDI_STATUS_COMMON_MTC_QUARTER_FRAME:
			ext_consumed =  decode_mtc_quarter_frame( tvb, pinfo, tree, cmd_count, offset);
			break;
		case RTP_MIDI_STATUS_COMMON_SONG_POSITION_POINTER:
			ext_consumed =  decode_song_position_pointer( tvb, pinfo, tree, cmd_count, offset);
			break;
		case RTP_MIDI_STATUS_COMMON_SONG_SELECT:
			ext_consumed =  decode_song_select( tvb, pinfo, tree, cmd_count, offset);
			break;
		case RTP_MIDI_STATUS_COMMON_UNDEFINED_F4:
			ext_consumed =  decode_undefined_f4( tvb, pinfo, tree, cmd_count, offset, cmd_len );
			break;
		case RTP_MIDI_STATUS_COMMON_UNDEFINED_F5:
			ext_consumed =  decode_undefined_f5( tvb, pinfo, tree, cmd_count, offset, cmd_len );
			break;
		case RTP_MIDI_STATUS_COMMON_TUNE_REQUEST:
			ext_consumed =  decode_tune_request( tvb, pinfo, tree, cmd_count, offset, cmd_len );
			break;
		case RTP_MIDI_STATUS_COMMON_SYSEX_END:
			ext_consumed =  decode_sysex_end( tvb, pinfo, tree, cmd_count, offset, cmd_len );
			break;
	}
	/* external decoder informed us of error -> pass this through */
	if ( ext_consumed < 0 ) {
		return ext_consumed;
	}
	return consumed + ext_consumed;
}



/*
 * Here the chapter c of the channel-journal is decoded.
 */
static int
decode_cj_chapter_c( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_cj_chapter_tree;
	proto_tree			*rtp_midi_loglist_tree;
	proto_tree			*loglist_tree;
	unsigned int start_offset = offset;
	guint8				 octet;
	int				 count;
	int				 i;

	octet = tvb_get_guint8( tvb, offset );
	count = octet & 0x7f;

	/* count encoded is n+1 */
	count++;

	rtp_midi_cj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, 1 + 2 * count,
				ett_rtp_midi_cj_chapter_c, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_C );

	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_c_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_c_length, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;

	rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, count * 2,
				ett_rtp_midi_cj_chapter_c_loglist, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_C_LOGLIST );

	for ( i = 0; i < count; i++ ) {

		loglist_tree = proto_tree_add_subtree(rtp_midi_loglist_tree, tvb, offset, 2,
			ett_rtp_midi_cj_chapter_c_logitem, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_C_LOGITEM );

		proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_number, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;

		octet = tvb_get_guint8( tvb, offset );
		proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_aflag, tvb, offset, 1, ENC_BIG_ENDIAN );

		/* do we have a value or alternative coding? */
		if ( octet & 0x80 ) {
			proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_tflag, tvb, offset, 1, ENC_BIG_ENDIAN );
			proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_alt, tvb, offset, 1, ENC_BIG_ENDIAN );
		} else {
			proto_tree_add_item(loglist_tree, hf_rtp_midi_cj_chapter_c_value, tvb, offset, 1, ENC_BIG_ENDIAN );
		}

		offset++;
	}
	return offset-start_offset;
}



/*
 * Here the chapter m of the channel-journal is decoded, possibly the most complex part of the RTP-MIDI stuff ;-)
 */
static int
decode_cj_chapter_m( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_cj_chapter_tree;
	proto_tree			*rtp_midi_loglist_tree;
	proto_tree			*rtp_midi_loglist_item_tree;
	guint16				header;
	guint8				logitemheader;
	int				length;
	int				logitemhdrlen;
	int				logitemlen;
	gboolean			no_pnum_msb;
	unsigned int start_offset = offset;

	/* first we need to get the flags & length of this chapter */
	header = tvb_get_ntohs( tvb, offset );
	length = header & RTP_MIDI_CJ_CHAPTER_M_MASK_LENGTH;
	/* take of length of header */
	length -= 2;

	rtp_midi_cj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, length, ett_rtp_midi_cj_chapter_m, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_M );

	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_sflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_pflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_eflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_uflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_wflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_zflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_length, tvb, offset, 2, ENC_BIG_ENDIAN );

	/* done with header */
	offset	 += 2;

	/* do we have the pending field? */
	if ( header & 0x4000 ) {

		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_qflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_m_pending, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
	}

	/*
	 * lets find out if we need to decode the pnum_msb:
	 * if Z = 1 and either U = 1 or W = 1 we don't
	 */
	no_pnum_msb = ( header & 0x0400 ) && ( ( header & 0x0800 ) || ( header & 0x1000 ) );
	logitemhdrlen = no_pnum_msb ? 2 : 3;

	rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, length, ett_rtp_midi_cj_chapter_m_loglist, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_M_LOGLIST );

	/* lets step through the loglist */
	while ( length > 0 ) {

		if ( no_pnum_msb ) {
			logitemheader = tvb_get_guint8( tvb, offset + 1 );
		} else {
			logitemheader = tvb_get_guint8( tvb, offset + 2 );
		}

		logitemlen = logitemhdrlen;

		/* do we have a msb field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_J ) {
			logitemlen++;
		}

		/* do we have a lsb field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_K ) {
			logitemlen++;
		}

		/* do we have an a-button field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_L ) {
			logitemlen +=2;
		}

		/* do we have a c-button field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_M ) {
			logitemlen +=2;
		}

		/* do we have a count field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_N ) {
			logitemlen++;
		}

		/* now that we have the length of the logitem, we can create the tree... */
		rtp_midi_loglist_item_tree = proto_tree_add_subtree( rtp_midi_loglist_tree, tvb, offset, logitemlen,
						ett_rtp_midi_cj_chapter_m_logitem, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_M_LOGITEM );

		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_pnum_lsb, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		length--;

		/* if we have the msb, we need to decode it */
		if ( !no_pnum_msb ) {
			proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_qflag, tvb, offset, 1, ENC_BIG_ENDIAN );
			proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_pnum_msb, tvb, offset, 1, ENC_BIG_ENDIAN );
			offset++;
			length--;
		}

		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_jflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_kflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_lflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_mflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_nflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_tflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_vflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_loglist_item_tree, hf_rtp_midi_cj_chapter_m_log_rflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
		length--;

		/* do we have a entry-msb field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_J ) {
			static int * const msb_flags[] = {
				&hf_rtp_midi_cj_chapter_m_log_msb_x,
				&hf_rtp_midi_cj_chapter_m_log_msb,
				NULL
				};

			proto_tree_add_bitmask(rtp_midi_loglist_tree, tvb, offset, hf_rtp_midi_cj_chapter_m_log_msb_entry, ett_rtp_midi_cj_chapter_m_log_msb, msb_flags, ENC_NA);

			offset++;
			length--;
		}

		/* do we have a entry-lsb field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_K ) {
			static int * const lsb_flags[] = {
				&hf_rtp_midi_cj_chapter_m_log_lsb_x,
				&hf_rtp_midi_cj_chapter_m_log_lsb,
				NULL
				};

			proto_tree_add_bitmask(rtp_midi_loglist_tree, tvb, offset, hf_rtp_midi_cj_chapter_m_log_lsb_entry, ett_rtp_midi_cj_chapter_m_log_lsb, lsb_flags, ENC_NA);
			offset++;
			length--;
		}

		/* do we have an a-button field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_L ) {
			static int * const button_flags[] = {
				&hf_rtp_midi_cj_chapter_m_log_a_button_g,
				&hf_rtp_midi_cj_chapter_m_log_a_button_x,
				&hf_rtp_midi_cj_chapter_m_log_a_button,
				NULL
				};

			proto_tree_add_bitmask(rtp_midi_loglist_tree, tvb, offset, hf_rtp_midi_cj_chapter_m_log_a_button_full, ett_rtp_midi_cj_chapter_m_log_a_button, button_flags, ENC_BIG_ENDIAN);

			offset	 += 2;
			length	 -= 2;
		}

		/* do we have a c-button field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_M ) {
			static int * const button_flags[] = {
				&hf_rtp_midi_cj_chapter_m_log_c_button_g,
				&hf_rtp_midi_cj_chapter_m_log_c_button_r,
				&hf_rtp_midi_cj_chapter_m_log_c_button,
				NULL
				};

			proto_tree_add_bitmask(rtp_midi_loglist_tree, tvb, offset, hf_rtp_midi_cj_chapter_m_log_c_button_full, ett_rtp_midi_cj_chapter_m_log_c_button, button_flags, ENC_BIG_ENDIAN);

			offset	 += 2;
			length	 -= 2;
		}

		/* do we have a count field? */
		if ( logitemheader & RTP_MIDI_CJ_CHAPTER_M_FLAG_N ) {
			static int * const log_flags[] = {
				&hf_rtp_midi_cj_chapter_m_log_count_x,
				&hf_rtp_midi_cj_chapter_m_log_count,
				NULL
				};

			proto_tree_add_bitmask(rtp_midi_loglist_tree, tvb, offset, hf_rtp_midi_cj_chapter_m_log_count_full, ett_rtp_midi_cj_chapter_m_log_count, log_flags, ENC_BIG_ENDIAN);

			offset++;
			length--;
		}

	}

	return offset-start_offset;
}


/*
 * Here the chapter n of the channel-journal is decoded.
 */
static int
decode_cj_chapter_n( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_cj_chapter_tree;
	proto_tree			*rtp_midi_loglist_tree;
	const gchar			*note_str;
	unsigned int start_offset = offset;
	guint16				 header;
	guint8				 note;
	guint8				 velocity;
	int				 log_count;
	int				 octet_count;
	int				 low;
	int				 high;
	int				 i;

	/* first we need to get the flags & length of this chapter */
	header = tvb_get_ntohs( tvb, offset );
	log_count = ( header & RTP_MIDI_CJ_CHAPTER_N_MASK_LENGTH ) >> 8;
	low = ( header & RTP_MIDI_CJ_CHAPTER_N_MASK_LOW ) >> 4;
	high = header & RTP_MIDI_CJ_CHAPTER_N_MASK_HIGH;

	/* how many offbits octets do we have? */
	if ( low <= high ) {
		octet_count = high - low + 1;
	} else if ( ( low == 15 ) && ( high == 0 ) ) {
		octet_count = 0;
	} else if ( ( low == 15 ) && ( high == 1 ) ) {
		octet_count = 0;
	} else {
		return -1;
	}

	/* special case -> no offbit octets, but 128 note-logs */
	if ( ( log_count == 127 ) && ( low == 15) && ( high == 0 ) ) {
		log_count++;
	}

	rtp_midi_cj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, 2 + ( log_count * 2 ) + octet_count,
					ett_rtp_midi_cj_chapter_n, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_N );

	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_n_bflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_n_len, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_n_low, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_n_high, tvb, offset, 2, ENC_BIG_ENDIAN );

	offset	 += 2;

	if ( log_count > 0 ) {

		rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, log_count * 2,
					ett_rtp_midi_cj_chapter_n_loglist, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_LOGLIST );

		for ( i = 0; i < log_count; i++ ) {

			note = tvb_get_guint8( tvb, offset ) & 0x7f;
			velocity = tvb_get_guint8( tvb, offset + 1 ) & 0x7f;

			note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );

			rtp_midi_loglist_tree = proto_tree_add_subtree_format(rtp_midi_loglist_tree, tvb, offset, 2,
						ett_rtp_midi_cj_chapter_n_logitem, NULL, "%s (n=%s, v=%d)", RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_LOGITEM, note_str, velocity );

			proto_tree_add_item( rtp_midi_loglist_tree, hf_rtp_midi_cj_chapter_n_log_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
			proto_tree_add_item( rtp_midi_loglist_tree, hf_rtp_midi_cj_chapter_n_log_notenum, tvb, offset, 1, ENC_BIG_ENDIAN );

			offset++;

			proto_tree_add_item( rtp_midi_loglist_tree, hf_rtp_midi_cj_chapter_n_log_yflag, tvb, offset, 1, ENC_BIG_ENDIAN );
			proto_tree_add_item( rtp_midi_loglist_tree, hf_rtp_midi_cj_chapter_n_log_velocity, tvb, offset, 1, ENC_BIG_ENDIAN );

			offset++;
		}

	}

	if ( octet_count > 0 ) {

		rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, log_count,
				ett_rtp_midi_cj_chapter_n_octets, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_N_OCTETS );

		for ( i = 0; i < octet_count; i++ ) {

			proto_tree_add_item(rtp_midi_loglist_tree, hf_rtp_midi_cj_chapter_n_log_octet, tvb, offset, 1, ENC_BIG_ENDIAN );

			offset++;
		}
	}

	return offset-start_offset;
}


/*
 * Here the chapter e of the channel-journal is decoded.
 */
static int
decode_cj_chapter_e( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_cj_chapter_tree;
	proto_tree			*rtp_midi_loglist_tree, *log_tree;
	const gchar			*note_str;
	unsigned int start_offset = offset;
	guint8				 header;
	guint8				 note;
	guint8				 count_vel;
	guint8				 octet;
	int				 log_count;
	int				 i;

	/* first we need to get the flags & length of this chapter */
	header = tvb_get_guint8( tvb, offset );
	log_count = header & RTP_MIDI_CJ_CHAPTER_E_MASK_LENGTH;

	/* count is encoded n+1 */
	log_count++;

	rtp_midi_cj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, 1 + ( log_count * 2 ),
				ett_rtp_midi_cj_chapter_e, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_E );

	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_e_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_e_len, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;

	rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, log_count * 2,
				ett_rtp_midi_cj_chapter_e_loglist, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGLIST );

	for ( i = 0; i < log_count; i++ ) {

		note = tvb_get_guint8( tvb, offset ) & 0x7f;
		octet = tvb_get_guint8( tvb, offset + 1 );
		count_vel = octet & 0x7f;

		note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );

		if ( octet & 0x80 ) {
			log_tree = proto_tree_add_subtree_format(rtp_midi_loglist_tree, tvb, offset, 2, ett_rtp_midi_cj_chapter_e_logitem, NULL,
						"%s (n=%s, v=%d)", RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGITEM1, note_str, count_vel );
		} else {
			log_tree = proto_tree_add_subtree_format(rtp_midi_loglist_tree, tvb, offset, 2, ett_rtp_midi_cj_chapter_e_logitem, NULL,
						"%s (n=%s, c=%d)", RTP_MIDI_TREE_NAME_CJ_CHAPTER_E_LOGITEM2, note_str, count_vel );
		}

		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_e_log_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_e_log_notenum, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;

		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_e_log_vflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		if ( octet & 0x80 ) {
			proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_e_log_velocity, tvb, offset, 1, ENC_BIG_ENDIAN );
		} else {
			proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_e_log_count, tvb, offset, 1, ENC_BIG_ENDIAN );
		}

		offset++;
	}

	return offset-start_offset;
}








/*
 * Here the chapter a of the channel-journal is decoded.
 */
static int
decode_cj_chapter_a( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_cj_chapter_tree;
	proto_tree			*rtp_midi_loglist_tree, *log_tree;
	const gchar			*note_str;
	unsigned int start_offset = offset;
	guint8				 header;
	guint8				 note;
	guint8				 pressure;
	int				 log_count;
	int				 i;

	/* first we need to get the flags & length of this chapter */
	header = tvb_get_guint8( tvb, offset );
	log_count = header & RTP_MIDI_CJ_CHAPTER_A_MASK_LENGTH;

	/* count is encoded n+1 */
	log_count++;

	rtp_midi_cj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, 1 + ( log_count * 2 ), ett_rtp_midi_cj_chapter_a, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_A );

	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_a_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_a_len, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;

	rtp_midi_loglist_tree = proto_tree_add_subtree( rtp_midi_cj_chapter_tree, tvb, offset, log_count * 2, ett_rtp_midi_cj_chapter_a_loglist, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_A_LOGLIST );

	for ( i = 0; i < log_count; i++ ) {

		note	 = tvb_get_guint8( tvb, offset ) & 0x7f;
		pressure = tvb_get_guint8( tvb, offset + 1 ) & 0x7f;

		note_str = val_to_str_ext( note, &rtp_midi_note_values_ext, rtp_midi_unknown_value_dec );

		log_tree = proto_tree_add_subtree_format(rtp_midi_loglist_tree, tvb, offset, 2, ett_rtp_midi_cj_chapter_a_logitem, NULL,
				"%s (n=%s, p=%d)", RTP_MIDI_TREE_NAME_CJ_CHAPTER_A_LOGITEM, note_str, pressure );

		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_a_log_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_a_log_notenum, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;

		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_a_log_xflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( log_tree, hf_rtp_midi_cj_chapter_a_log_pressure, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
	}

	return offset-start_offset;
}





/*
 * Here a channel-journal is decoded.
 */
static int
decode_channel_journal( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_chanjournal_tree;
	proto_tree			*rtp_midi_cj_chapters_tree;
	proto_tree			*rtp_midi_cj_chapter_tree;
	guint32				chanflags;
	guint16				chanjourlen;
	int				consumed = 0;
	int				ext_consumed = 0;

	/* first we need to get the flags & length of this channel-journal */
	chanflags = tvb_get_ntoh24( tvb, offset );
	chanjourlen = ( chanflags & RTP_MIDI_CJ_MASK_LENGTH ) >> 8;

	rtp_midi_chanjournal_tree = proto_tree_add_subtree( tree, tvb, offset, chanjourlen, ett_rtp_midi_channeljournal, NULL,
			val_to_str( ( chanflags & RTP_MIDI_CJ_MASK_CHANNEL ) >> RTP_MIDI_CJ_CHANNEL_SHIFT, rtp_midi_channels, rtp_midi_unknown_value_hex ) );

	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_sflag, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_chan, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_hflag, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_len, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_p, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_c, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_m, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_w, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_n, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_e, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_t, tvb, offset, 3, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_chanjournal_tree, hf_rtp_midi_chanjour_toc_a, tvb, offset, 3, ENC_BIG_ENDIAN );

	rtp_midi_cj_chapters_tree = proto_tree_add_subtree( rtp_midi_chanjournal_tree, tvb, offset + 3, chanjourlen - 3,
			ett_rtp_midi_channelchapters, NULL, RTP_MIDI_TREE_NAME_CHANNEL_CHAPTERS );

	/* take care of length of header */
	offset	 += 3;
	consumed += 3;

	/* Do we have a program change chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_P ) {

		rtp_midi_cj_chapter_tree = proto_tree_add_subtree( rtp_midi_cj_chapters_tree, tvb, offset, 3,
				ett_rtp_midi_cj_chapter_p, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_P );

		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_program, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_bflag, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_bank_msb, tvb, offset + 1, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_xflag, tvb, offset + 2, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_p_bank_lsb, tvb, offset + 2, 1, ENC_BIG_ENDIAN );

		offset	 += 3;
		consumed += 3;
	}

	/* Do we have a control chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_C ) {
		ext_consumed = decode_cj_chapter_c( tvb, pinfo, rtp_midi_cj_chapters_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		consumed += ext_consumed;
		offset	 += ext_consumed;

	}

	/* Do we have a parameter changes? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_M ) {
		ext_consumed = decode_cj_chapter_m( tvb, pinfo, rtp_midi_cj_chapters_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		consumed += ext_consumed;
		offset	 += ext_consumed;
	}

	/* Do we have a pitch-wheel chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_W ) {

		rtp_midi_cj_chapter_tree = proto_tree_add_subtree( rtp_midi_cj_chapters_tree, tvb, offset, 2,
						ett_rtp_midi_cj_chapter_w, NULL, RTP_MIDI_TREE_NAME_CJ_CHAPTER_W );

		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_w_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_w_first, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		consumed++;

		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_w_rflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_cj_chapter_tree, hf_rtp_midi_cj_chapter_w_second, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		consumed++;
	}

	/* Do we have a note on/off chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_N ) {
		ext_consumed = decode_cj_chapter_n( tvb, pinfo, rtp_midi_cj_chapters_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		consumed += ext_consumed;
		offset	 += ext_consumed;
	}

	/* Do we have a note command extras chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_E ) {
		ext_consumed = decode_cj_chapter_e( tvb, pinfo, rtp_midi_cj_chapters_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		consumed += ext_consumed;
		offset	 += ext_consumed;
	}

	/* Do we have channel aftertouch chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_T ) {
		static int * const flags_t[] = {
			&hf_rtp_midi_cj_chapter_t_sflag,
			&hf_rtp_midi_cj_chapter_t_pressure,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_cj_chapters_tree, tvb, offset, hf_rtp_midi_cj_chapter_t_channel_aftertouch, ett_rtp_midi_cj_chapter_t, flags_t, ENC_NA);

		offset++;
		consumed++;
	}

	/* Do we have a poly aftertouch chapter? */
	if ( chanflags & RTP_MIDI_CJ_FLAG_A ) {
		ext_consumed = decode_cj_chapter_a( tvb, pinfo, rtp_midi_cj_chapters_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		consumed += ext_consumed;
	}

	/* Make sanity check for consumed data vs. stated length of this channels journal */
	if ( consumed != chanjourlen ) {
		return -1;
	}
	return consumed;
}


/*
 * Here the chapter D F4-field of the system-journal is decoded.
 */
static int
decode_sj_chapter_d_f4( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {

	proto_tree	*rtp_midi_field_tree;
	unsigned int start_offset = offset;
	guint16		 f4flags;
	guint16		 f4length;

	/* Get flags & length */
	f4flags = tvb_get_ntohs( tvb, offset );
	f4length = f4flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_LENGTH;

	/* now we can display our tree, as we now have the full length */
	rtp_midi_field_tree = proto_tree_add_subtree( tree, tvb, offset, f4length, ett_rtp_midi_sj_chapter_d_field_j, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_J );

	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_sflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_cflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_vflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_lflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_dsz, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_length, tvb, offset, 2, ENC_BIG_ENDIAN );

	offset	 += 2;
	f4length -= 2;

	if ( f4flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_C ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_count, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		f4length--;
	}

	if ( f4flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_V ) {

		int valuelen = 0;
		guint8 octet;

		/* variable length field - ends with an octet with MSB set */
		for (;;) {
			octet = tvb_get_guint8( tvb, offset+valuelen );
			valuelen++;
			if ( octet & 0x80 ) {
				break;
			}
		}
		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_value, tvb, offset, valuelen, ENC_NA );
		offset	 += valuelen;
		f4length -= valuelen;
	}

	if ( f4flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_L ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_legal, tvb, offset, f4length, ENC_NA );
		offset	 += f4length;
	}

	/* if we still have data, the length-field was incorrect we dump the data here and abort! */
	if ( f4length > 0 ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_data, tvb, offset, f4length, ENC_NA );
		offset += f4length;
		/* must be a protocol error - since we have a length, we can recover...*/
	}


	return offset-start_offset;
}


/*
 * Here the chapter D F5-field of the system-journal is decoded.
 */
static int
decode_sj_chapter_d_f5( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {

	proto_tree	*rtp_midi_field_tree;
	unsigned int start_offset = offset;
	guint16		 f5flags;
	guint16		 f5length;

	/* Get flags & length */
	f5flags = tvb_get_ntohs( tvb, offset );
	f5length = f5flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_LENGTH;

	/* now we can display our tree, as we now have the full length */
	rtp_midi_field_tree = proto_tree_add_subtree( tree, tvb, offset, f5length, ett_rtp_midi_sj_chapter_d_field_k, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_K );

	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_sflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_cflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_vflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_lflag, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_dsz, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_length, tvb, offset, 2, ENC_BIG_ENDIAN );

	offset	 += 2;
	f5length -= 2;

	if ( f5flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_C ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_count, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		f5length--;
	}

	if ( f5flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_V ) {

		int valuelen = 0;
		guint8 octet;

		/* variable length field - ends with an octet with MSB set */
		for (;;) {
			octet = tvb_get_guint8( tvb, offset+valuelen );
			valuelen++;
			if ( octet & 0x80 ) {
				break;
			}
		}
		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_value, tvb, offset, valuelen, ENC_NA );
		offset	 += valuelen;
		f5length -= valuelen;
	}

	if ( f5flags & RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_L ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_legal, tvb, offset, f5length, ENC_NA );
		offset	 += f5length;
		f5length = 0;
	}

	/* if we still have data, we dump it here - see above! */
	if ( f5length > 0 ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_syscom_data, tvb, offset, f5length, ENC_NA );
		offset += f5length;
		/* must be a protocol error - since we have a length, we can recover...*/
	}

	return offset-start_offset;
}

/*
 * Here the chapter D F9-field of the system-journal is decoded.
 */
static int
decode_sj_chapter_d_f9( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {

	proto_tree	*rtp_midi_field_tree;
	unsigned int start_offset = offset;
	guint8		 f9flags;
	guint8		 f9length;

	/* Get flags & length */
	f9flags = tvb_get_guint8( tvb, offset );
	f9length = f9flags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_LENGTH;

	/* now we can display our tree, as we now have the full length */
	rtp_midi_field_tree = proto_tree_add_subtree( tree, tvb, offset, f9length, ett_rtp_midi_sj_chapter_d_field_y, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_Y );

	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_cflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_lflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_length, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;
	f9length--;

	if ( f9flags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_C ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_count, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		f9length--;
	}

	if ( f9flags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_L ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_legal, tvb, offset, f9length, ENC_NA );
		offset	 += f9length;
		f9length = 0;
	}

	/* if we still have data, the length-field was incorrect we dump the data here and abort! */

	if ( f9length > 0 ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_data, tvb, offset, f9length, ENC_NA );
		offset += f9length;
		/* must be a protocol error - since we have a length, we can recover...*/
	}

	return offset-start_offset;
}


/*
 * Here the chapter D FD-field of the system-journal is decoded.
 */
static int
decode_sj_chapter_d_fd( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {

	proto_tree	*rtp_midi_field_tree;
	unsigned int start_offset = offset;
	guint8		 fdflags;
	guint8		 fdlength;

	/* Get flags & length */
	fdflags  = tvb_get_guint8( tvb, offset );
	fdlength = fdflags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_LENGTH;

	/* now we can display our tree, as we now have the full length */
	rtp_midi_field_tree = proto_tree_add_subtree( tree, tvb, offset, fdlength, ett_rtp_midi_sj_chapter_d_field_z, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_D_FIELD_Z );

	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_cflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_lflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_length, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;
	fdlength--;

	if ( fdflags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_C ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_count, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
		fdlength--;
	}

	if ( fdflags & RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_L ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_legal, tvb, offset, fdlength, ENC_NA );
		offset	 += fdlength;
		fdlength = 0;
	}

	/* if we still have data, the length-field was incorrect we dump the data here and abort! */

	if ( fdlength > 0 ) {

		proto_tree_add_item( rtp_midi_field_tree, hf_rtp_midi_sj_chapter_d_sysreal_data, tvb, offset, fdlength, ENC_NA );

		offset += fdlength;
		/* must be a protocol error - since we have a length, we can recover...*/
	}

	return offset-start_offset;
}



/*
 * Here the chapter D of the channel-journal is decoded.
 */
static int
decode_sj_chapter_d( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_item			*tix;
	proto_tree			*rtp_midi_sj_chapter_tree;
	guint8				header;
	unsigned int start_offset = offset;
	int				ext_consumed;

	static int * const chapter_d_flags[] = {
		&hf_rtp_midi_sj_chapter_d_sflag,
		&hf_rtp_midi_sj_chapter_d_bflag,
		&hf_rtp_midi_sj_chapter_d_gflag,
		&hf_rtp_midi_sj_chapter_d_hflag,
		&hf_rtp_midi_sj_chapter_d_jflag,
		&hf_rtp_midi_sj_chapter_d_kflag,
		&hf_rtp_midi_sj_chapter_d_yflag,
		&hf_rtp_midi_sj_chapter_d_zflag,
		NULL
	};

	/* first we need to get the flags of this chapter */
	header = tvb_get_guint8( tvb, offset );

	/*
	 * we don't know the complete length yet - we know that we have at least the header -> 1.
	 * At the end we set the real length...
	 */
	tix = proto_tree_add_bitmask(tree, tvb, offset, hf_rtp_midi_sj_chapter_d_simple_system_commands, ett_rtp_midi_sj_chapter_d, chapter_d_flags, ENC_NA);
	rtp_midi_sj_chapter_tree = proto_item_add_subtree( tix, ett_rtp_midi_sj_chapter_d );

	/* done with header */
	offset++;

	/* do we have Reset field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_B ) {

		static int * const reset_flags[] = {
			&hf_rtp_midi_sj_chapter_d_reset_sflag,
			&hf_rtp_midi_sj_chapter_d_reset_count,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_d_reset, ett_rtp_midi_sj_chapter_d_field_b, reset_flags, ENC_NA);
		offset++;
	}

	/* do we have Tune request field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_G ) {

		static int * const tune_flags[] = {
			&hf_rtp_midi_sj_chapter_d_tune_sflag,
			&hf_rtp_midi_sj_chapter_d_tune_count,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_d_tune, ett_rtp_midi_sj_chapter_d_field_g, tune_flags, ENC_BIG_ENDIAN);
		offset++;
	}

	/* do we have Song select field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_H ) {

		static int * const song_flags[] = {
			&hf_rtp_midi_sj_chapter_d_song_sel_sflag,
			&hf_rtp_midi_sj_chapter_d_song_sel_value,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_d_song_sel, ett_rtp_midi_sj_chapter_d_field_h, song_flags, ENC_BIG_ENDIAN);
		offset++;
	}

	/* do we have 0xF4 field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_J ) {
		ext_consumed = decode_sj_chapter_d_f4( tvb, pinfo, rtp_midi_sj_chapter_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		offset	 += ext_consumed;
	}

	/* do we have 0xF5 field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_K ) {
		ext_consumed = decode_sj_chapter_d_f5( tvb, pinfo, rtp_midi_sj_chapter_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		offset	 += ext_consumed;
	}

	/* do we have 0xF9 field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_Y ) {
		ext_consumed = decode_sj_chapter_d_f9( tvb, pinfo, rtp_midi_sj_chapter_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		offset	 += ext_consumed;
	}

	/* do we have 0xFD field? */
	if ( header & RTP_MIDI_SJ_CHAPTER_D_FLAG_Z ) {
		ext_consumed = decode_sj_chapter_d_fd( tvb, pinfo, rtp_midi_sj_chapter_tree, offset );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		offset += ext_consumed;
	}


	/* now we know the complete length and set it. */
	proto_item_set_len( tix, offset-start_offset );
	return offset-start_offset;
}

/*
 * Here the chapter Q of the channel-journal is decoded.
 */
static int
decode_sj_chapter_q( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_sj_chapter_tree;
	guint8				header;
	unsigned int start_offset = offset;
	int				len = 1;

	/* first we need to get the flags of this chapter */
	header = tvb_get_guint8( tvb, offset );

	if ( header & RTP_MIDI_SJ_CHAPTER_Q_FLAG_C ) {
		len += 2;
	}
	if ( header & RTP_MIDI_SJ_CHAPTER_Q_FLAG_T ) {
		len += 3;
	}

	/* now we have the length... */
	rtp_midi_sj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, len, ett_rtp_midi_sj_chapter_q, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_Q );

	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_nflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_dflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_cflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_tflag, tvb, offset, 1, ENC_BIG_ENDIAN );

	if ( header & RTP_MIDI_SJ_CHAPTER_Q_FLAG_C ) {

		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_clock, tvb, offset, 3, ENC_BIG_ENDIAN );

		offset	 += 3;

	} else {
		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_top, tvb, offset, 1, ENC_BIG_ENDIAN );

		offset++;
	}

	if ( header & RTP_MIDI_SJ_CHAPTER_Q_FLAG_T ) {

		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_q_timetools, tvb, offset, 3, ENC_BIG_ENDIAN );

		offset += 3;
	}

	return offset-start_offset;
}

/*
 * Here the chapter F of the channel-journal is decoded.
 */
static int
decode_sj_chapter_f( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_sj_chapter_tree;
	guint8				header;
	unsigned int start_offset = offset;
	int				len = 1;

	/* first we need to get the flags of this chapter */
	header = tvb_get_guint8( tvb, offset );

	if ( header & RTP_MIDI_SJ_CHAPTER_F_FLAG_C ) {
		len += 4;
	}
	if ( header & RTP_MIDI_SJ_CHAPTER_F_FLAG_P ) {
		len += 4;
	}

	/* now we have the length... */
	rtp_midi_sj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, len, ett_rtp_midi_sj_chapter_f, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_F );

	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_cflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_pflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_qflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_dflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_f_point, tvb, offset, 1, ENC_BIG_ENDIAN );

	offset++;

	if ( header & RTP_MIDI_SJ_CHAPTER_F_FLAG_C ) {

		if ( header & RTP_MIDI_SJ_CHAPTER_F_FLAG_Q ) {
			static int * const fq_flags[] = {
				&hf_rtp_midi_sj_chapter_f_mt0,
				&hf_rtp_midi_sj_chapter_f_mt1,
				&hf_rtp_midi_sj_chapter_f_mt2,
				&hf_rtp_midi_sj_chapter_f_mt3,
				&hf_rtp_midi_sj_chapter_f_mt4,
				&hf_rtp_midi_sj_chapter_f_mt5,
				&hf_rtp_midi_sj_chapter_f_mt6,
				&hf_rtp_midi_sj_chapter_f_mt7,
				NULL
			};

			proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_f_complete, ett_rtp_midi_sj_chapter_f_complete, fq_flags, ENC_BIG_ENDIAN);
		} else {
			static int * const f_flags[] = {
				&hf_rtp_midi_sj_chapter_f_hr,
				&hf_rtp_midi_sj_chapter_f_mn,
				&hf_rtp_midi_sj_chapter_f_sc,
				&hf_rtp_midi_sj_chapter_f_fr,
				NULL
			};

			proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_f_complete, ett_rtp_midi_sj_chapter_f_complete, f_flags, ENC_BIG_ENDIAN);
		}

		offset	 += 4;
	}

	if ( header & RTP_MIDI_SJ_CHAPTER_F_FLAG_P ) {
		static int * const fp_flags[] = {
			&hf_rtp_midi_sj_chapter_f_mt0,
			&hf_rtp_midi_sj_chapter_f_mt1,
			&hf_rtp_midi_sj_chapter_f_mt2,
			&hf_rtp_midi_sj_chapter_f_mt3,
			&hf_rtp_midi_sj_chapter_f_mt4,
			&hf_rtp_midi_sj_chapter_f_mt5,
			&hf_rtp_midi_sj_chapter_f_mt6,
			&hf_rtp_midi_sj_chapter_f_mt7,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_sj_chapter_tree, tvb, offset, hf_rtp_midi_sj_chapter_f_partial, ett_rtp_midi_sj_chapter_f_partial, fp_flags, ENC_BIG_ENDIAN);
		offset += 4;
	}

	return offset-start_offset;
}

/*
 * Here the chapter X of the channel-journal is decoded.
 */
static int
decode_sj_chapter_x( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset, unsigned int max_length ) {
	proto_tree			*rtp_midi_sj_chapter_tree;
	proto_tree			*rtp_midi_sj_data_tree;
	guint8				header;
	guint8				octet;
	unsigned int			consumed = 0;
	unsigned int			cmdlen   = 0;
	unsigned int			i;

	/* first we need to get the flags of this chapter */
	header = tvb_get_guint8( tvb, offset );

	rtp_midi_sj_chapter_tree = proto_tree_add_subtree( tree, tvb, offset, max_length, ett_rtp_midi_sj_chapter_x, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_X );

	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_tflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_cflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_fflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_dflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_lflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_sta, tvb, offset, 1, ENC_BIG_ENDIAN );

	consumed++;
	offset++;

	if ( header & RTP_MIDI_SJ_CHAPTER_X_FLAG_T ) {

		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_tcount, tvb, offset, 1, ENC_BIG_ENDIAN );

		consumed++;
		offset++;
	}

	if ( header & RTP_MIDI_SJ_CHAPTER_X_FLAG_C ) {

		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_count, tvb, offset, 1, ENC_BIG_ENDIAN );

		consumed++;
		offset++;
	}

	if ( header & RTP_MIDI_SJ_CHAPTER_X_FLAG_F ) {
		unsigned int field    = 0;
		unsigned int fieldlen = 0;

		/* FIRST is "compressed" using only the necessary amount of octets, like delta-time */
		for ( i=0; i < 4; i++ ) {

			/* do we still fit in the dissected packet & in the length restriction of this chapter? */
			if ( ( !( consumed >= max_length ) ) || ( !tvb_bytes_exist( tvb, offset + fieldlen, 1 ) ) ) {
				return -1;
			}

			octet = tvb_get_guint8( tvb, offset + fieldlen );
			field = ( field << 7 ) | ( octet & RTP_MIDI_DELTA_TIME_OCTET_MASK );
			fieldlen++;

			if ( ( octet & RTP_MIDI_DELTA_TIME_EXTENSION ) == 0 ) {
				break;
			}
		}

		switch (fieldlen) {
			case 1:
			    proto_tree_add_uint ( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_first1, tvb, offset, fieldlen, field );
				break;
			case 2:
			    proto_tree_add_uint ( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_first2, tvb, offset, fieldlen, field );
				break;
			case 3:
			    proto_tree_add_uint ( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_first3, tvb, offset, fieldlen, field );
				break;
			case 4:
			    proto_tree_add_uint ( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_first4, tvb, offset, fieldlen, field );
				break;

		}
		consumed += fieldlen;
		offset	 += fieldlen;

	}

	if ( header & RTP_MIDI_SJ_CHAPTER_X_FLAG_D ) {
		rtp_midi_sj_data_tree = proto_tree_add_subtree( rtp_midi_sj_chapter_tree, tvb, offset,  max_length - consumed,
						ett_rtp_midi_sj_chapter_x_data, NULL, RTP_MIDI_TREE_NAME_SJ_CHAPTER_X_DATA );
		while ( consumed < max_length ) {
			octet = tvb_get_guint8( tvb, offset + cmdlen );
			if ( octet & 0x80 ) {
				proto_tree_add_item( rtp_midi_sj_data_tree, hf_rtp_midi_sj_chapter_x_data, tvb, offset, cmdlen, ENC_NA );
				offset += cmdlen;
				cmdlen	= 0;
			} else {
				cmdlen += 1;
			}
			consumed += 1;
		}
		/* unfinished command still to put into tree */
		if ( cmdlen ) {
			proto_tree_add_item( rtp_midi_sj_data_tree, hf_rtp_midi_sj_chapter_x_invalid_data, tvb, offset, cmdlen, ENC_NA );
			offset += cmdlen;
		}
	}

	/* this should not ever enter - we still have data, but flag d was apparently not set...  */
	if ( consumed < max_length ) {

		proto_tree_add_item( rtp_midi_sj_chapter_tree, hf_rtp_midi_sj_chapter_x_invalid_data, tvb, offset, max_length - consumed, ENC_NA );
		consumed = max_length;
	}

	return consumed;
}



/*
 * Here the system-journal is decoded.
 */
static int
decode_system_journal( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, unsigned int offset ) {
	proto_tree			*rtp_midi_sysjournal_tree;
	proto_tree			*rtp_midi_sj_chapters_tree;
	unsigned int start_offset = offset;
	int				ext_consumed = 0;
	guint16				sysjourlen;
	guint16				systemflags;

	systemflags = tvb_get_ntohs( tvb, offset );
	sysjourlen  = systemflags & RTP_MIDI_SJ_MASK_LENGTH;

	rtp_midi_sysjournal_tree = proto_tree_add_subtree( tree, tvb, offset, sysjourlen, ett_rtp_midi_systemjournal, NULL, RTP_MIDI_TREE_NAME_SYSTEM_JOURNAL );

	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_s, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_d, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_v, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_q, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_f, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_toc_x, tvb, offset, 2, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_sysjournal_tree, hf_rtp_midi_sysjour_len, tvb, offset, 2, ENC_BIG_ENDIAN );

	offset	 += 2;

	rtp_midi_sj_chapters_tree = proto_tree_add_subtree( rtp_midi_sysjournal_tree, tvb, offset, sysjourlen - 2,
				ett_rtp_midi_systemchapters, NULL, RTP_MIDI_TREE_NAME_SYSTEM_CHAPTERS );

	/* Do we have a simple system commands chapter? */
	if ( systemflags & RTP_MIDI_SJ_FLAG_D ) {
		offset += decode_sj_chapter_d( tvb, pinfo, rtp_midi_sj_chapters_tree, offset );
	}

	/* Do we have a active sensing chapter? */
	if ( systemflags & RTP_MIDI_SJ_FLAG_V ) {
		static int * const v_flags[] = {
			&hf_rtp_midi_sj_chapter_v_sflag,
			&hf_rtp_midi_sj_chapter_v_count,
			NULL
		};

		proto_tree_add_bitmask(rtp_midi_sj_chapters_tree, tvb, offset, hf_rtp_midi_sj_chapter_v, ett_rtp_midi_sj_chapter_v, v_flags, ENC_BIG_ENDIAN);

		offset++;
	}

	/* Do we have a sequencer state commands chapter? */
	if ( systemflags & RTP_MIDI_SJ_FLAG_Q ) {
		offset += decode_sj_chapter_q( tvb, pinfo, rtp_midi_sj_chapters_tree, offset );
	}

	/* Do we have a MTC chapter? */
	if ( systemflags & RTP_MIDI_SJ_FLAG_F ) {
		offset += decode_sj_chapter_f( tvb, pinfo, rtp_midi_sj_chapters_tree, offset );
	}


	/* Do we have a Sysex chapter? */
	if ( systemflags & RTP_MIDI_SJ_FLAG_X ) {
		ext_consumed = decode_sj_chapter_x( tvb, pinfo, rtp_midi_sj_chapters_tree, offset, sysjourlen - (offset-start_offset) );
		if ( ext_consumed < 0 ) {
			return ext_consumed;
		}
		offset += ext_consumed;
	}


	/* Make sanity check for consumed data vs. stated length of system journal */
	if ( offset-start_offset != sysjourlen ) {
		return -1;
	}
	return offset-start_offset;
}






/*
 * This is the main dissector function for the RTP-MIDI protocol
 * It decodes the wrapper of the command-section and punts
 * decoding of delta-time and the actual MIDI-command itself to
 * external decoders.  Afterwards the journal-section is decoded.
 */

static int
dissect_rtp_midi( tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_ )
{
	proto_item		*ti;
	proto_tree		*rtp_midi_tree;
	unsigned int		 offset = 0;

	guint8			flags;		/* used for command-section and journal-section*/
	unsigned int		cmd_len;
	unsigned int		cmd_count;
	guint8			runningstatus;

	int			consumed;
	unsigned int		rsoffset = 0;

	guint8			totchan;

	unsigned int		i;

	col_set_str( pinfo->cinfo, COL_PROTOCOL, RTP_MIDI_DISSECTOR_SHORTNAME );
	col_clear( pinfo->cinfo, COL_INFO );

	/*
	 * MIDI command section
	 */

	/* RTP-MIDI starts with 4 bits of flags... */
	flags = tvb_get_guint8( tvb, offset );

	ti = proto_tree_add_item( tree, proto_rtp_midi, tvb, 0, -1, ENC_NA );
	rtp_midi_tree = proto_item_add_subtree( ti, ett_rtp_midi);

	/* Decode those four flags */
	proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_bflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_jflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_zflag, tvb, offset, 1, ENC_BIG_ENDIAN );
	proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_pflag, tvb, offset, 1, ENC_BIG_ENDIAN );

	/* ...followed by a length-field of at least 4 bits */
	cmd_len = flags & RTP_MIDI_CS_MASK_SHORTLEN;

	/* see if we have small or large len-field */
	if (flags & RTP_MIDI_CS_FLAG_B)  {
		cmd_len = tvb_get_ntohs( tvb, offset) & RTP_MIDI_CS_MASK_LONGLEN;
		proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_longlen, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset	+= 2;
	} else {
		proto_tree_add_item( rtp_midi_tree, hf_rtp_midi_shortlen, tvb, offset, 1, ENC_BIG_ENDIAN );
		offset++;
	}

	/* if we have a command-section -> dissect it */
	if ( cmd_len ) {
		proto_tree *rtp_midi_commands_tree;

		rtp_midi_commands_tree = proto_tree_add_subtree( rtp_midi_tree, tvb, offset, cmd_len, ett_rtp_midi_commands, NULL, RTP_MIDI_TREE_NAME_COMMAND );

		/* No commands decoded yet */
		cmd_count = 0;

		/* RTP-MIDI-pdus always start with no running status */
		runningstatus = 0;

		/* Multiple MIDI-commands might follow - the exact number can only be discovered by really decoding the commands! */
		while ( cmd_len) {

			/* for the first command we only have a delta-time if Z-Flag is set */
			if ( ( cmd_count ) || ( flags & RTP_MIDI_CS_FLAG_Z ) ) {

				/* Decode a delta-time */
				consumed = decodetime( tvb, pinfo, rtp_midi_commands_tree, offset );

				/* seek to next command and set remaining length */
				offset  += consumed;
				cmd_len -= consumed;
			}

			/* Only decode MIDI-command if there is any data left - it is valid to only have delta-time! */
			if ( cmd_len ) {

				/* Decode a MIDI-command - if 0 is returned something went wrong */
				consumed = decodemidi( tvb, pinfo, rtp_midi_commands_tree, cmd_count, offset, cmd_len, &runningstatus, &rsoffset );
				if ( -1 == consumed ) {
					return offset;
				}

				/* seek to next delta-time and set remaining length */
				offset  += consumed;
				cmd_len -= consumed;

				/* as we have successfully decoded another command, increment count */
				cmd_count++;
			}
		}
	}
	/*
	 * Journal section
	 */

	/* if we have a journal-section -> dissect it */
	if ( flags & RTP_MIDI_CS_FLAG_J ) {
		proto_tree *rtp_midi_journal_tree;

		rtp_midi_journal_tree = proto_tree_add_subtree( rtp_midi_tree, tvb, offset, -1, ett_rtp_midi_journal, NULL, RTP_MIDI_TREE_NAME_JOURNAL );

		/* lets get the main flags from the recovery journal header */
		flags = tvb_get_guint8(     tvb, offset );

		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_sflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_yflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_aflag, tvb, offset, 1, ENC_BIG_ENDIAN );
		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_hflag, tvb, offset, 1, ENC_BIG_ENDIAN );

		/* At the same place we find the total channels encoded in the channel journal */
		totchan = flags & RTP_MIDI_JS_MASK_TOTALCHANNELS;
		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_totchan, tvb, offset, 1, ENC_BIG_ENDIAN);
		offset++;

		/* the checkpoint-sequence-number can be used to see if the recovery journal covers all lost events */
		proto_tree_add_item( rtp_midi_journal_tree, hf_rtp_midi_check_seq_num, tvb, offset, 2, ENC_BIG_ENDIAN );
		offset += 2;


		/* do we have system journal? */
		if ( flags & RTP_MIDI_JS_FLAG_Y ) {
			/* first we need to get the flags & length from the system-journal */
			consumed = decode_system_journal( tvb, pinfo, rtp_midi_journal_tree, offset );

			if ( -1 == consumed ) {
				return offset;
			}

			/* seek to optional channel-journals-section */
			offset += consumed;

		}

		/* do we have channel journal(s)? */
		if ( flags & RTP_MIDI_JS_FLAG_A ) {
			proto_tree *rtp_midi_chanjournals_tree;
			rtp_midi_chanjournals_tree = proto_tree_add_subtree( rtp_midi_journal_tree, tvb, offset, -1, ett_rtp_midi_channeljournals, NULL, RTP_MIDI_TREE_NAME_CHANNEL_JOURNAL );

			/* iterate through all the channels specified in header */
			for ( i = 0; i <= totchan; i++ ) {

				consumed = decode_channel_journal( tvb, pinfo, rtp_midi_chanjournals_tree, offset );

				if ( -1 == consumed ) {
					return offset;
				}

				/* seek to next channel-journal */
				offset += consumed;

			}
		}
	}
	return tvb_captured_length(tvb);
}




void
proto_register_rtp_midi( void )
{
	module_t *rtp_midi_module;

	static hf_register_info hf[] = {
		{
			&hf_rtp_midi_bflag,
			{
				"B-Flag",
				"rtpmidi.b_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cs_flag_b),
				RTP_MIDI_CS_FLAG_B,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_jflag,
			{
				"J-Flag",
				"rtpmidi.j_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cs_flag_j),
				RTP_MIDI_CS_FLAG_J,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_zflag,
			{
				"Z-Flag",
				"rtpmidi.z_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cs_flag_z),
				RTP_MIDI_CS_FLAG_Z,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_pflag,
			{
				"P-Flag",
				"rtpmidi.p_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cs_flag_p),
				RTP_MIDI_CS_FLAG_P,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_shortlen,
			{
				"Command length (short)",
				"rtpmidi.cmd_length_short",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_CS_MASK_SHORTLEN,
				NULL, HFILL
			}
		},
			{
			&hf_rtp_midi_longlen,
			{
				"Command length (long)",
				"rtpmidi.cmd_length_long",
				FT_UINT16,
				BASE_DEC,
				NULL,
				RTP_MIDI_CS_MASK_LONGLEN,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sflag,
			{
				"S-Flag",
				"rtpmidi.s_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_JS_FLAG_S,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_yflag,
			{
				"Y-Flag",
				"rtpmidi.y_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_y),
				RTP_MIDI_JS_FLAG_Y,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_aflag,
			{
				"A-Flag",
				"rtpmidi.a_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_a),
				RTP_MIDI_JS_FLAG_A,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_hflag,
			{
				"H-Flag",
				"rtpmidi.h_flag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_h),
				RTP_MIDI_JS_FLAG_H,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_totchan,
			{
				"Total channels",
				"rtpmidi.total_channels",
				FT_UINT8,
				BASE_DEC,
				VALS(rtp_midi_js_tot_channels),
				RTP_MIDI_JS_MASK_TOTALCHANNELS,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_check_seq_num,
			{
				"Checkpoint Packet Seqnum",
				"rtpmidi.check_Seq_num",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_deltatime1,
			{
				"Delta Time (one octet)",
				"rtpmidi.deltatime_1",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_deltatime2,
			{
				"Delta Time (two octets)",
				"rtpmidi.deltatime_2",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_deltatime3,
			{
				"Delta Time (three octets)",
				"rtpmidi.deltatime_3",
				FT_UINT24,
				BASE_HEX,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_deltatime4,
			{
				"Delta Time (four octets)",
				"rtpmidi.deltatime_4",
				FT_UINT32,
				BASE_HEX,
				NULL,
				0x7f7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_channel_status,
			{
				"Channel Status",
				"rtpmidi.channel_status",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_channel_status),
				0xf0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_common_status,
			{
				"Common Status",
				"rtpmidi.common_status",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_common_status),
				0xff,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_channel,
			{
				"Channel",
				"rtpmidi.channel",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_channels),
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_note,
			{
				"Note",
				"rtpmidi.note",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_note_values_ext,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_velocity,
			{
				"Velocity",
				"rtpmidi.velocity",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_pressure,
			{
				"Pressure",
				"rtpmidi.pressure",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_controller,
			{
				"Controller",
				"rtpmidi.controller",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_controller_values_ext,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_controller_value,
			{
				"Value",
				"rtpmidi.controller_value",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_program,
			{
				"Program",
				"rtpmidi.program",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_channel_pressure,
			{
				"Pressure",
				"rtpmidi.channel_pressure",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_pitch_bend,
			{
				"Pitch Bend",
				"rtpmidi.pitch_bend",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
#if 0
		{
			&hf_rtp_midi_pitch_bend_truncated,
			{
				"Pitch Bend (truncated)",
				"rtpmidi.pitch_bend_truncated",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
#endif
		{
			&hf_rtp_midi_manu_short,
			{
				"Manufacturer (short)",
				"rtpmidi.manufacturer_short",
				FT_UINT8,
				BASE_HEX | BASE_EXT_STRING,
				&rtp_midi_manu_short_values_ext,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_manu_long,
			{
				"Manufacturer (long)",
				"rtpmidi.manufacturer_long",
				FT_UINT16,
				BASE_HEX | BASE_EXT_STRING,
				&rtp_midi_manu_long_values_ext,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_s,
			{
				"System-Journal TOC-S-Flag",
				"rtpmidi.sysjour_toc_s",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_FLAG_S,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_d,
			{
				"System-Journal TOC-D-Flag",
				"rtpmidi.sysjour_toc_d",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_flag_d),
				RTP_MIDI_SJ_FLAG_D,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_v,
			{
				"System-Journal TOC-V-Flag",
				"rtpmidi.sysjour_toc_v",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_flag_v),
				RTP_MIDI_SJ_FLAG_V,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_q,
			{
				"System-Journal TOC-Q-Flag",
				"rtpmidi.sysjour_toc_q",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_flag_q),
				RTP_MIDI_SJ_FLAG_Q,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_f,
			{
				"System-Journal TOC-F-Flag",
				"rtpmidi.sysjour_toc_f",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_flag_f),
				RTP_MIDI_SJ_FLAG_F,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_toc_x,
			{
				"System-Journal TOC-X-Flag",
				"rtpmidi.sysjour_toc_x",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_flag_x),
				RTP_MIDI_SJ_FLAG_X,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysjour_len,
			{
				"System-Journal-Length",
				"rtpmidi.cmd_sysjour_len",
				FT_UINT16,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_MASK_LENGTH,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_sflag,
			{
				"Channel-Journal S-Flag",
				"rtpmidi.chanjour_s",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_js_flag_s),
				0x800000,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_chan,
			{
				"Channel",
				"rtpmidi.chanjour_channel",
				FT_UINT24,
				BASE_HEX,
				VALS(rtp_midi_channels),
				0x780000,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_hflag,
			{
				"Channel-Journal H-Flag",
				"rtpmidi.chanjour_h",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_js_flag_h),
				0x040000,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_len,
			{
				"Channel-Journal-Length",
				"rtpmidi.cmd_chanjour_len",
				FT_UINT24,
				BASE_DEC,
				NULL,
				0x03FF00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_p,
			{
				"Channel-Journal TOC-P-Flag",
				"rtpmidi.chanjour_toc_p",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_p),
				0x000080,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_c,
			{
				"Channel-Journal TOC-C-Flag",
				"rtpmidi.chanjour_toc_c",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_c),
				0x000040,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_m,
			{
				"Channel-Journal TOC-M-Flag",
				"rtpmidi.chanjour_toc_m",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_m),
				0x000020,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_w,
			{
				"Channel-Journal TOC-W-Flag",
				"rtpmidi.chanjour_toc_w",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_w),
				0x000010,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_n,
			{
				"Channel-Journal TOC-N-Flag",
				"rtpmidi.chanjour_toc_n",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_n),
				0x000008,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_e,
			{
				"Channel-Journal TOC-E-Flag",
				"rtpmidi.chanjour_toc_e",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_e),
				0x000004,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_t,
			{
				"Channel-Journal TOC-T-Flag",
				"rtpmidi.chanjour_toc_t",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_t),
				0x000002,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_chanjour_toc_a,
			{
				"Channel-Journal TOC-A-Flag",
				"rtpmidi.chanjour_toc_a",
				FT_BOOLEAN,
				24,
				TFS(&rtp_midi_cj_flag_a),
				0x000001,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_p_sflag,
			{
				"Chapter P S-Flag",
				"rtpmidi.cj_chapter_p_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_p_program,
			{
				"Chapter P Program",
				"rtpmidi.cj_chapter_p_program",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_p_bflag,
			{
				"Chapter P B-Flag",
				"rtpmidi.cj_chapter_p_bflag",
				FT_BOOLEAN,
				8,
				NULL,
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_p_bank_msb,
			{
				"Chapter P Bank-MSB",
				"rtpmidi.cj_chapter_p_bank_msb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_p_xflag,
			{
				"Chapter P X-Flag",
				"rtpmidi.cj_chapter_p_xflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_p_log_flag_x),
				0x80,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_cj_chapter_p_bank_lsb,
			{
				"Chapter P Bank-LSB",
				"rtpmidi.cj_chapter_p_bank_lsb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_sflag,
			{
				"Chapter C S-Flag",
				"rtpmidi.cj_chapter_c_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_length,
			{
				"Chapter C Length",
				"rtpmidi.cj_chapter_c_length",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_number,
			{
				"Chapter C Number",
				"rtpmidi.cj_chapter_c_number",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_controller_values_ext,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_aflag,
			{
				"Chapter C A-Flag",
				"rtpmidi.cj_chapter_c_aflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_c_flag_a),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_tflag,
			{
				"Chapter C T-Flag",
				"rtpmidi.cj_chapter_c_tflag",
				FT_BOOLEAN,
				8,
				NULL,
				0x40,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_value,
			{
				"Chapter C Value",
				"rtpmidi.cj_chapter_c_value",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_c_alt,
			{
				"Chapter C Alt",
				"rtpmidi.cj_chapter_c_alt",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x3f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_sflag,
			{
				"Chapter M S-Flag",
				"rtpmidi.cj_chapter_m_sflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_js_flag_s),
				0x8000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_pflag,
			{
				"Chapter M P-Flag",
				"rtpmidi.cj_chapter_m_pflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_flag_p),
				0x4000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_eflag,
			{
				"Chapter M E-Flag",
				"rtpmidi.cj_chapter_m_eflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_flag_e),
				0x2000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_uflag,
			{
				"Chapter M U-Flag",
				"rtpmidi.cj_chapter_m_uflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_flag_u),
				0x1000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_wflag,
			{
				"Chapter M W-Flag",
				"rtpmidi.cj_chapter_m_wflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_flag_w),
				0x0800,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_zflag,
			{
				"Chapter M Z-Flag",
				"rtpmidi.cj_chapter_m_zflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_flag_z),
				0x0400,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_length,
			{
				"Chapter M Length",
				"rtpmidi.cj_chapter_m_length",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x03ff,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_qflag,
			{
				"Chapter M Q-Flag",
				"rtpmidi.cj_chapter_m_qflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_flag_q),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_pending,
			{
				"Chapter M Pending",
				"rtpmidi.cj_chapter_m_pending",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_sflag,
			{
				"Chapter M Log S-Flag",
				"rtpmidi.cj_chapter_m_log_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_pnum_lsb,
			{
				"Chapter M Log PNUM-LSB",
				"rtpmidi.cj_chapter_m_log_pnum_lsb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_qflag,
			{
				"Chapter M Log Q-Flag",
				"rtpmidi.cj_chapter_m_log_qflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_q),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_pnum_msb,
			{
				"Chapter M Log PNUM-MSB",
				"rtpmidi.cj_chapter_m_log_pnum_msb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_jflag,
			{
				"Chapter M Log J-Flag",
				"rtpmidi.cj_chapter_m_log_jflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_j),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_J,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_kflag,
			{
				"Chapter M Log K-Flag",
				"rtpmidi.cj_chapter_m_log_kflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_k),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_K,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_lflag,
			{
				"Chapter M Log L-Flag",
				"rtpmidi.cj_chapter_m_log_lflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_l),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_L,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_mflag,
			{
				"Chapter M Log M-Flag",
				"rtpmidi.cj_chapter_m_log_mflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_m),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_M,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_nflag,
			{
				"Chapter M Log N-Flag",
				"rtpmidi.cj_chapter_m_log_nflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_n),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_N,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_tflag,
			{
				"Chapter M Log T-Flag",
				"rtpmidi.cj_chapter_m_log_tflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_t),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_T,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_vflag,
			{
				"Chapter M Log V-Flag",
				"rtpmidi.cj_chapter_m_log_vflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_v),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_V,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_rflag,
			{
				"Chapter M Log R-Flag",
				"rtpmidi.cj_chapter_m_log_rflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_r),
				RTP_MIDI_CJ_CHAPTER_M_FLAG_R,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_msb_entry,
			{
				"Entry MSB",
				"rtpmidi.cj_chapter_m_log_msb_entry",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_msb_x,
			{
				"Chapter M Log MSB X-Flag",
				"rtpmidi.cj_chapter_m_log_msb_xflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_x),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_msb,
			{
				"Chapter M Log MSB",
				"rtpmidi.cj_chapter_m_log_msb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_lsb_entry,
			{
				"Entry LSB",
				"rtpmidi.cj_chapter_m_log_lsb_entry",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_lsb_x,
			{
				"Chapter M Log LSB X-Flag",
				"rtpmidi.cj_chapter_m_log_lsb_xflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_x),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_lsb,
			{
				"Chapter M Log LSB",
				"rtpmidi.cj_chapter_m_log_lsb",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_a_button_g,
			{
				"Chapter M Log A-Button G-Flag",
				"rtpmidi.cj_chapter_m_log_a_button_gflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_log_flag_g),
				0x8000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_a_button_x,
			{
				"Chapter M Log A-Button X-Flag",
				"rtpmidi.cj_chapter_m_log_a_button_xflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_log_flag_x),
				0x4000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_a_button,
			{
				"Chapter M Log A-Button",
				"rtpmidi.cj_chapter_m_log_a_button",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x3fff,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_a_button_full,
			{
				"A-Button",
				"rtpmidi.cj_chapter_m_log_a_button_full",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_c_button_g,
			{
				"Chapter M Log C-Button G-Flag",
				"rtpmidi.cj_chapter_m_log_c_button_gflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_log_flag_g),
				0x8000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_c_button_r,
			{
				"Chapter M Log C-Button R-Flag",
				"rtpmidi.cj_chapter_m_log_c_button_rflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_cj_chapter_m_log_flag_r),
				0x4000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_c_button,
			{
				"Chapter M Log C-Button",
				"rtpmidi.cj_chapter_m_log_c_button",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x3fff,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_c_button_full,
			{
				"C-Button",
				"rtpmidi.cj_chapter_m_log_c_button_full",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_count_x,
			{
				"Chapter M Log Count X-Flag",
				"rtpmidi.cj_chapter_m_log_count_xflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_x),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_count,
			{
				"Chapter M Log Count",
				"rtpmidi.cj_chapter_m_log_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_m_log_count_full,
			{
				"Count",
				"rtpmidi.cj_chapter_m_log_count_full",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_w_sflag,
			{
				"Chapter W S-Flag",
				"rtpmidi.cj_chapter_w_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_w_first,
			{
				"Chapter W First",
				"rtpmidi.cj_chapter_w_first",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_w_rflag,
			{
				"Chapter W R-Flag",
				"rtpmidi.cj_chapter_w_rflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_m_log_flag_r),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_w_second,
			{
				"Chapter W Second",
				"rtpmidi.cj_chapter_w_second",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_bflag,
			{
				"Chapter N B-Flag",
				"rtpmidi.cj_chapter_n_bflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_js_flag_s),
				0x8000,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_len,
			{
				"Chapter N Length",
				"rtpmidi.cj_chapter_n_length",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f00,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_low,
			{
				"Chapter N Low",
				"rtpmidi.cj_chapter_n_low",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x00f0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_high,
			{
				"Chapter N High",
				"rtpmidi.cj_chapter_n_high",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x000f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_log_sflag,
			{
				"Chapter N Log S-Flag",
				"rtpmidi.cj_chapter_n_log_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_log_notenum,
			{
				"Chapter N Log Note",
				"rtpmidi.cj_chapter_n_log_note",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_note_values_ext,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_log_yflag,
			{
				"Chapter N Log Y-Flag",
				"rtpmidi.cj_chapter_n_log_yflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_n_log_flag_y),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_log_velocity,
			{
				"Chapter N Log Velocity",
				"rtpmidi.cj_chapter_n_log_velocity",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_n_log_octet,
			{
				"Chapter N Log Octet",
				"rtpmidi.cj_chapter_n_log_octet",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0xff,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_sflag,
			{
				"Chapter E S-Flag",
				"rtpmidi.cj_chapter_e_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_len,
			{
				"Chapter E Length",
				"rtpmidi.cj_chapter_e_length",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_log_sflag,
			{
				"Chapter E Log S-Flag",
				"rtpmidi.cj_chapter_e_log_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_log_notenum,
			{
				"Chapter E Log Note",
				"rtpmidi.cj_chapter_e_log_note",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_note_values_ext,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_log_vflag,
			{
				"Chapter N Log V-Flag",
				"rtpmidi.cj_chapter_n_log_vflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_e_log_flag_v),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_log_velocity,
			{
				"Chapter E Log Velocity",
				"rtpmidi.cj_chapter_e_log_velocity",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_e_log_count,
			{
				"Chapter E Log Count",
				"rtpmidi.cj_chapter_e_log_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_t_channel_aftertouch,
			{
				"Channel Aftertouch",
				"rtpmidi.cj_chapter_t_channel_aftertouch",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_t_sflag,
			{
				"Chapter T S-Flag",
				"rtpmidi.cj_chapter_t_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_t_pressure,
			{
				"Chapter T Pressure",
				"rtpmidi.cj_chapter_t_pressure",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_sflag,
			{
				"Chapter A S-Flag",
				"rtpmidi.cj_chapter_a_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_len,
			{
				"Chapter A Length",
				"rtpmidi.cj_chapter_a_length",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_log_sflag,
			{
				"Chapter A Log S-Flag",
				"rtpmidi.cj_chapter_a_log_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_log_notenum,
			{
				"Chapter A Log Note",
				"rtpmidi.cj_chapter_a_log_note",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_note_values_ext,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_log_xflag,
			{
				"Chapter A Log X-Flag",
				"rtpmidi.cj_chapter_a_log_xflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_cj_chapter_a_log_flag_x),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_cj_chapter_a_log_pressure,
			{
				"Chapter A Log Pressure",
				"rtpmidi.cj_chapter_a_log_pressure",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_v,
			{
				"Active Sensing",
				"rtpmidi.sj_chapter_v",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_v_sflag,
			{
				"Chapter V S-Flag",
				"rtpmidi.sj_chapter_v_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				0x80,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_v_count,
			{
				"Chapter V Count",
				"rtpmidi.sj_chapter_v_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_simple_system_commands,
			{
				"Simple System Commands",
				"rtpmidi.sj_chapter_d_simple_system_commands",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_sflag,
			{
				"Chapter D S-Flag",
				"rtpmidi.sj_chapter_d_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_bflag,
			{
				"Chapter D B-Flag",
				"rtpmidi.sj_chapter_d_bflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_b),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_B,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_gflag,
			{
				"Chapter D G-Flag",
				"rtpmidi.sj_chapter_d_gflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_g),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_G,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_hflag,
			{
				"Chapter D H-Flag",
				"rtpmidi.sj_chapter_d_hflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_h),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_H,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_jflag,
			{
				"Chapter D J-Flag",
				"rtpmidi.sj_chapter_d_jflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_j),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_J,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_kflag,
			{
				"Chapter D K-Flag",
				"rtpmidi.sj_chapter_d_kflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_k),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_K,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_yflag,
			{
				"Chapter D Y-Flag",
				"rtpmidi.sj_chapter_d_yflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_y),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_Y,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_zflag,
			{
				"Chapter D Z-Flag",
				"rtpmidi.sj_chapter_d_zflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_flag_z),
				RTP_MIDI_SJ_CHAPTER_D_FLAG_Z,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_reset,
			{
				"Reset Field",
				"rtpmidi.cj_chapter_d_reset",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_reset_sflag,
			{
				"Chapter D Reset S-Flag",
				"rtpmidi.sj_chapter_d_reset_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_RESET_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_reset_count,
			{
				"Chapter D Reset Count",
				"rtpmidi.cj_chapter_d_reset_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_RESET_COUNT,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_tune,
			{
				"Tune Request Field",
				"rtpmidi.cj_chapter_d_tune",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_tune_sflag,
			{
				"Chapter D Tune Request S-Flag",
				"rtpmidi.sj_chapter_d_tune_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_TUNE_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_tune_count,
			{
				"Chapter D Tune Request Count",
				"rtpmidi.cj_chapter_d_tune_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_TUNE_COUNT,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_song_sel,
			{
				"Song Select Field",
				"rtpmidi.cj_chapter_d_song_sel",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_song_sel_sflag,
			{
				"Chapter D Song Select S-Flag",
				"rtpmidi.sj_chapter_d_song_sel_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_SONG_SEL_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_song_sel_value,
			{
				"Chapter D Song Select Value",
				"rtpmidi.cj_chapter_d_song_sel_value",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_SONG_SEL_VALUE,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_syscom_sflag,
			{
				"Chapter D System Common (F4/F5) S-Flag",
				"rtpmidi.sj_chapter_d_syscom_sflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_d_syscom_cflag,
			{
				"Chapter D System Common (F4/F5) C-Flag",
				"rtpmidi.sj_chapter_d_syscom_cflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_chapter_d_syscom_flag_c),
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_C,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_vflag,
			{
				"Chapter D System Common (F4/F5) V-Flag",
				"rtpmidi.sj_chapter_d_syscom_vflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_chapter_d_syscom_flag_v),
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_V,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_lflag,
			{
				"Chapter D System Common (F4/F5) L-Flag",
				"rtpmidi.sj_chapter_d_syscom_lflag",
				FT_BOOLEAN,
				16,
				TFS(&rtp_midi_sj_chapter_d_syscom_flag_l),
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_FLAG_L,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_dsz,
			{
				"Chapter D System Common (F4/F5) DSZ",
				"rtpmidi.sj_chapter_d_syscom_dsz",
				FT_UINT16,
				BASE_DEC,
				VALS(rtp_midi_sj_chapter_d_syscom_dsz_values),
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_DSZ,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_length,
			{
				"Chapter D System Common (F4/F5) Length",
				"rtpmidi.sj_chapter_d_syscom_len",
				FT_UINT16,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_LENGTH,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_count,
			{
				"Chapter D System Common (F4/F5) Count",
				"rtpmidi.sj_chapter_d_syscom_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_SYSCOM_MASK_COUNT,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_value,
			{
				"Chapter D System Common (F4/F5) Value",
				"rtpmidi.sj_chapter_d_syscom_value",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_legal,
			{
				"Chapter D System Common (F4/F5) Legal - for future extension",
				"rtpmidi.sj_chapter_d_syscom_legal",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_syscom_data,
			{
				"Chapter D System Common (F4/F5) Erroneous Data",
				"rtpmidi.sj_chapter_d_syscom_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_sflag,
			{
				"Chapter D System Realtime (F9/FD) S-Flag",
				"rtpmidi.sj_chapter_d_sysreal_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_S,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_cflag,
			{
				"Chapter D System Realtime (F9/FF) C-Flag",
				"rtpmidi.sj_chapter_d_sysreal_cflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_syscom_flag_c),
				RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_C,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_lflag,
			{
				"Chapter D System Realtime (F9/FD) L-Flag",
				"rtpmidi.sj_chapter_d_sysreal_lflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_d_syscom_flag_l),
				RTP_MIDI_SJ_CHAPTER_D_SYSREAL_FLAG_L,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_length,
			{
				"Chapter D System Realtime (F9/FD) Length",
				"rtpmidi.sj_chapter_d_sysreal_len",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_LENGTH,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_count,
			{
				"Chapter D System Realtime (F9/FD) Count",
				"rtpmidi.sj_chapter_d_sysreal_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_D_SYSREAL_MASK_COUNT,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_legal,
			{
				"Chapter D System Realtime (F9/FD) Legal - for future extension",
				"rtpmidi.sj_chapter_d_sysreal_legal",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_d_sysreal_data,
			{
				"Chapter D System Realtime (F9/FD) Erroneous Data",
				"rtpmidi.sj_chapter_d_sysreal_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_q_sflag,
			{
				"Chapter Q Sequencer State S-Flag",
				"rtpmidi.sj_chapter_q_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_Q_FLAG_S,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_q_nflag,
			{
				"Chapter Q Sequencer State N-Flag",
				"rtpmidi.sj_chapter_q_nflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_q_flag_n),
				RTP_MIDI_SJ_CHAPTER_Q_FLAG_N,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_q_dflag,
			{
				"Chapter Q Sequencer State D-Flag",
				"rtpmidi.sj_chapter_q_dflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_q_flag_d),
				RTP_MIDI_SJ_CHAPTER_Q_FLAG_D,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_q_cflag,
			{
				"Chapter Q Sequencer State C-Flag",
				"rtpmidi.sj_chapter_q_cflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_q_flag_c),
				RTP_MIDI_SJ_CHAPTER_Q_FLAG_C,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_q_tflag,
			{
				"Chapter Q Sequencer State T-Flag",
				"rtpmidi.sj_chapter_q_tflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_q_flag_t),
				RTP_MIDI_SJ_CHAPTER_Q_FLAG_T,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_q_top,
			{
				"Chapter Q Sequencer State Top",
				"rtpmidi.sj_chapter_q_top",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_Q_MASK_TOP,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_q_clock,
			{
				"Chapter Q Sequencer State Clock",
				"rtpmidi.sj_chapter_q_clock",
				FT_UINT24,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_Q_MASK_CLOCK,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_q_timetools,
			{
				"Chapter Q Sequencer State Timetools",
				"rtpmidi.sj_chapter_q_timetools",
				FT_UINT24,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_Q_MASK_TIMETOOLS,
				NULL, HFILL
			}
		},

		{
			&hf_rtp_midi_sj_chapter_f_sflag,
			{
				"Chapter F MTC S-Flag",
				"rtpmidi.sj_chapter_f_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_F_FLAG_S,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_cflag,
			{
				"Chapter F MTC C-Flag",
				"rtpmidi.sj_chapter_f_cflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_f_flag_c),
				RTP_MIDI_SJ_CHAPTER_F_FLAG_C,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_pflag,
			{
				"Chapter F MTC P-Flag",
				"rtpmidi.sj_chapter_f_pflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_f_flag_p),
				RTP_MIDI_SJ_CHAPTER_F_FLAG_P,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_qflag,
			{
				"Chapter F MTC Q-Flag",
				"rtpmidi.sj_chapter_f_qflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_f_flag_q),
				RTP_MIDI_SJ_CHAPTER_F_FLAG_Q,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_dflag,
			{
				"Chapter F MTC D-Flag",
				"rtpmidi.sj_chapter_f_dflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_f_flag_d),
				RTP_MIDI_SJ_CHAPTER_F_FLAG_D,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_point,
			{
				"Chapter F MTC Point",
				"rtpmidi.sj_chapter_f_point",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_POINT,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_complete,
			{
				"Complete field",
				"rtpmidi.sj_chapter_f_complete",
				FT_UINT32,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_partial,
			{
				"Partial field",
				"rtpmidi.sj_chapter_f_partial",
				FT_UINT32,
				BASE_HEX,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt0,
			{
				"Chapter F MTC MT0",
				"rtpmidi.sj_chapter_f_mt0",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt1,
			{
				"Chapter F MTC MT1",
				"rtpmidi.sj_chapter_f_mt1",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT1,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt2,
			{
				"Chapter F MTC MT2",
				"rtpmidi.sj_chapter_f_mt2",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT2,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt3,
			{
				"Chapter F MTC MT3",
				"rtpmidi.sj_chapter_f_mt3",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT3,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt4,
			{
				"Chapter F MTC MT4",
				"rtpmidi.sj_chapter_f_mt4",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT4,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt5,
			{
				"Chapter F MTC MT5",
				"rtpmidi.sj_chapter_f_mt5",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT5,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt6,
			{
				"Chapter F MTC MT6",
				"rtpmidi.sj_chapter_f_mt6",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT6,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mt7,
			{
				"Chapter F MTC MT7",
				"rtpmidi.sj_chapter_f_mt7",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MT7,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_hr,
			{
				"Chapter F MTC Hour",
				"rtpmidi.sj_chapter_f_hr",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_HR,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_mn,
			{
				"Chapter F MTC Minute",
				"rtpmidi.sj_chapter_f_mn",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_MN,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_sc,
			{
				"Chapter F MTC Second",
				"rtpmidi.sj_chapter_f_sc",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_SC,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_f_fr,
			{
				"Chapter F MTC Frame",
				"rtpmidi.sj_chapter_f_fr",
				FT_UINT32,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_F_MASK_FR,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_sflag,
			{
				"Chapter X Sysex S-Flag",
				"rtpmidi.sj_chapter_x_sflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_js_flag_s),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_S,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_tflag,
			{
				"Chapter X Sysex T-Flag",
				"rtpmidi.sj_chapter_x_tflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_x_flag_t),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_T,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_cflag,
			{
				"Chapter X Sysex C-Flag",
				"rtpmidi.sj_chapter_x_cflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_x_flag_c),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_C,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_fflag,
			{
				"Chapter X Sysex F-Flag",
				"rtpmidi.sj_chapter_x_fflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_x_flag_f),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_F,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_dflag,
			{
				"Chapter X Sysex D-Flag",
				"rtpmidi.sj_chapter_x_dflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_x_flag_d),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_D,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_lflag,
			{
				"Chapter X Sysex L-Flag",
				"rtpmidi.sj_chapter_x_lflag",
				FT_BOOLEAN,
				8,
				TFS(&rtp_midi_sj_chapter_x_flag_l),
				RTP_MIDI_SJ_CHAPTER_X_FLAG_L,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_sta,
			{
				"Chapter X Sysex STA",
				"rtpmidi.sj_chapter_x_sta",
				FT_UINT8,
				BASE_HEX,
				NULL,
				RTP_MIDI_SJ_CHAPTER_X_MASK_STA,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_tcount,
			{
				"Chapter X Sysex TCOUT",
				"rtpmidi.sj_chapter_x_tcount",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_X_MASK_TCOUNT,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_count,
			{
				"Chapter X Sysex COUNT",
				"rtpmidi.sj_chapter_x_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				RTP_MIDI_SJ_CHAPTER_X_MASK_COUNT,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_first1,
			{
				"Chapter X Sysex FIRST (one octet)",
				"rtpmidi.sj_chapter_x_first_1",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_first2,
			{
				"Chapter X Sysex FIRST (two octets)",
				"rtpmidi.sj_chapter_x_first_2",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_first3,
			{
				"Chapter X Sysex FIRST (three octets)",
				"rtpmidi.sj_chapter_x_first_3",
				FT_UINT24,
				BASE_HEX,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_first4,
			{
				"Chapter X Sysex FIRST (four octets)",
				"rtpmidi.sj_chapter_x_first_4",
				FT_UINT32,
				BASE_HEX,
				NULL,
				0x7f7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_data,
			{
				"Chapter X Sysex Data",
				"rtpmidi.sj_chapter_x_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sj_chapter_x_invalid_data,
			{
				"Chapter X Invalid Sysex Data",
				"rtpmidi.sj_chapter_x_invalid_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_quarter_frame_type,
			{
				"MTC-quarter-frame-type",
				"rtpmidi.mtc_quarter_frame_type",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_qft_values),
				0x70,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_quarter_frame_value,
			{
				"MTC-quarter-frame-value",
				"rtpmidi.mtc_quarter_frame_value",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_spp,
			{
				"Song Position Pointer",
				"rtpmidi.song_position_pointer",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
#if 0
		{
			&hf_rtp_midi_spp_truncated,
			{
				"Song Position Pointer (truncated)",
				"rtpmidi.song_position_pointer_truncated",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
#endif
		{
			&hf_rtp_midi_song_select,
			{
				"Song Select",
				"rtpmidi.song_select",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_manu_data,
			{
				"Manufacturer specific data",
				"rtpmidi.manu_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_edu_data,
			{
				"Educational command data",
				"rtpmidi.edu_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_unknown_data,
			{
				"Unknown Data",
				"rtpmidi.unknown_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x00,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_realtime,
			{
				"Sysex Common Realtime",
				"rtpmidi.sysex_common_realtime",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_non_realtime,
			{
				"Sysex Common Non-Realtime",
				"rtpmidi.sysex_common_non_realtime",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_device_id,
			{
				"Sysex Common Device ID",
				"rtpmidi.sysex_common_device_id",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_ext,
			{
				"Sysex Common Non-Realtime Sample Dump Extension",
				"rtpmidi.sysex_common_non_realtime_sample_dump_ext",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_sd_ext),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_gi,
			{
				"Sysex Common Non-Realtime General Information",
				"rtpmidi.sysex_common_non_realtime_general_information",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_gi),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd,
			{
				"Sysex Common Non-Realtime File Dump",
				"rtpmidi.sysex_common_non_realtime_file_dump",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_fd),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tuning,
			{
				"Sysex Common (Non-)Realtime Tuning",
				"rtpmidi.sysex_common_tuning",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_tuning),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_gm,
			{
				"Sysex Common Non-Realtime General MIDI (GM)",
				"rtpmidi.sysex_common_non_realtime_gm",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_gm),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_dls,
			{
				"Sysex Common Non-Realtime Downloadable Sounds (DLS)",
				"rtpmidi.sysex_common_non_realtime_dls",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_dls),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc,
			{
				"Sysex Common Realtime MTC",
				"rtpmidi.sysex_common_realtime_mtc",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_mtc),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_sc,
			{
				"Sysex Common Realtime Show Control",
				"rtpmidi.sysex_common_realtime_sc",
				FT_UINT8,
				BASE_HEX | BASE_EXT_STRING,
				&rtp_midi_sysex_common_rt_show_control_ext,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni,
			{
				"Sysex Common Realtime Notation Information",
				"rtpmidi.sysex_common_realtime_ni",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_notations),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_dc,
			{
				"Sysex Common Realtime Device Control",
				"rtpmidi.sysex_common_realtime_dc",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_device_control),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_cueing,
			{
				"Sysex Common Realtime MTC Cueing",
				"rtpmidi.sysex_common_rt_mtc_cue",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_mtc_cue),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc,
			{
				"Sysex Common Non-Realtime MTC",
				"rtpmidi.sysex_common_nrt_mtc",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_nrt_mtc),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mmc_commands,
			{
				"Sysex Common Realtime Machine Control Commands",
				"rtpmidi.sysex_common_realtime_mmc_cmds",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_mmc_commands),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mmc_responses,
			{
				"Sysex Common Realtime Machine Control Responses",
				"rtpmidi.sysex_common_realtime_mmc_resps",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_sysex_common_rt_mmc_responses),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_gi_device_family,
			{
				"Sysex Common Non-Realtime General Information Device Family",
				"rtpmidi.sysex_common_non_realtime_general_information_device_family",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_gi_device_family_member,
			{
				"Sysex Common Non-Realtime General Information Device Family Member",
				"rtpmidi.sysex_common_non_realtime_general_information_device_family_member",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_gi_software_rev,
			{
				"Sysex Common Non-Realtime General Information Software Revision",
				"rtpmidi.sysex_common_non_realtime_general_information_software_revision",
				FT_UINT32,
				BASE_HEX,
				NULL,
				0x7f7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_packet_number,
			{
				"Sysex Common Non-Realtime Sample Dump Packet Number",
				"rtpmidi.sysex_common_non_realtime_sample_dump_packet_number",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_sn,
			{
				"Sysex Common Non-Realtime Sample Dump Sample Number",
				"rtpmidi.sysex_common_non_realtime_sample_dump_sample_number",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_sf,
			{
				"Sysex Common Non-Realtime Sample Dump Sample Format",
				"rtpmidi.sysex_common_non_realtime_sample_dump_sample_format",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_sp,
			{
				"Sysex Common Non-Realtime Sample Dump Sample Period",
				"rtpmidi.sysex_common_non_realtime_sample_dump_sample_period",
				FT_UINT24,
				BASE_DEC,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_sl,
			{
				"Sysex Common Non-Realtime Sample Dump Sample Length",
				"rtpmidi.sysex_common_non_realtime_sample_dump_sample_length",
				FT_UINT24,
				BASE_DEC,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_ls,
			{
				"Sysex Common Non-Realtime Sample Dump Loop Start",
				"rtpmidi.sysex_common_non_realtime_sample_dump_loop_start",
				FT_UINT24,
				BASE_DEC,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_le,
			{
				"Sysex Common Non-Realtime Sample Dump Loop End",
				"rtpmidi.sysex_common_non_realtime_sample_dump_loop_end",
				FT_UINT24,
				BASE_DEC,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_header_lt,
			{
				"Sysex Common Non-Realtime Sample Dump Loop Type",
				"rtpmidi.sysex_common_non_realtime_sample_dump_loop_type",
				FT_UINT8,
				BASE_DEC,
				VALS(rtp_midi_sysex_common_nrt_sd_lt),
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_packet_count,
			{
				"Sysex Common Non-Realtime Sample Dump Running Packet Count",
				"rtpmidi.sysex_common_non_realtime_sample_dump_packet_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_packet_check,
			{
				"Sysex Common Non-Realtime Sample Dump Checksum",
				"rtpmidi.sysex_common_non_realtime_sample_dump_checksum",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_packet_data,
			{
				"Sysex Common Non-Realtime Sample Dump Data Byte",
				"rtpmidi.sysex_common_non_realtime_sample_dump_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_sd_ext_ln,
			{
				"Sysex Common Non-Realtime Sample Dump LP Transmission Loop Number",
				"rtpmidi.sysex_common_non_realtime_sample_dump_lp_trans_ln",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_device_id,
			{
				"Sysex Common Non-Realtime File Dump Device ID",
				"rtpmidi.sysex_common_non_realtime_fd_device_id",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_type,
			{
				"Sysex Common Non-Realtime File Dump File Type",
				"rtpmidi.sysex_common_non_realtime_fd_type",
				FT_STRING,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_name,
			{
				"Sysex Common Non-Realtime File Dump File Name",
				"rtpmidi.sysex_common_non_realtime_fd_name",
				FT_STRING,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_length,
			{
				"Sysex Common Non-Realtime File Dump Length",
				"rtpmidi.sysex_common_non_realtime_fd_length",
				FT_UINT32,
				BASE_DEC,
				NULL,
				0x7f7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_packet_num,
			{
				"Sysex Common Non-Realtime File Packet Number",
				"rtpmidi.sysex_common_non_realtime_fd_packet_num",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_byte_count,
			{
				"Sysex Common Non-Realtime File Byte Count",
				"rtpmidi.sysex_common_non_realtime_fd_byte_count",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_packet_data,
			{
				"Sysex Common Non-Realtime File Packet Data",
				"rtpmidi.sysex_common_non_realtime_fd_packet_data",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_fd_checksum,
			{
				"Sysex Common Non-Realtime File Checksum",
				"rtpmidi.sysex_common_non_realtime_fd_checksum",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_program,
			{
				"Sysex Common (Non-)Realtime Tuning Program",
				"rtpmidi.sysex_common_tune_program",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_name,
			{
				"Sysex Common (Non-)Realtime Tuning Name",
				"rtpmidi.sysex_common_tune_name",
				FT_STRING,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_freq,
			{
				"Sysex Common (Non-)Realtime Tuning Frequency",
				"rtpmidi.sysex_common_tune_freq",
				FT_UINT24,
				BASE_HEX,
				NULL,
				0x7f7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_checksum,
			{
				"Sysex Common (Non-)Realtime Tuning Checksum",
				"rtpmidi.sysex_common_tune_checksum",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_changes,
			{
				"Sysex Common (Non-)Realtime Tuning Changes",
				"rtpmidi.sysex_common_tune_changes",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_tune_note,
			{
				"Sysex Common (Non-)Realtime Tuning Note",
				"rtpmidi.sysex_common_tune_note",
				FT_UINT8,
				BASE_DEC | BASE_EXT_STRING,
				&rtp_midi_note_values_ext,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_fm_type,
			{
				"Sysex Common Realtime MTC Full Message Type",
				"rtpmidi.sysex_common_rt_mtc_fm_type",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_mtc_fm_type_values),
				0x60,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_fm_hr,
			{
				"Sysex Common Realtime MTC Full Message Hour",
				"rtpmidi.sysex_common_rt_mtc_fm_hour",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x1f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_fm_mn,
			{
				"Sysex Common Realtime MTC Full Message Minute",
				"rtpmidi.sysex_common_rt_mtc_fm_minute",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_fm_sc,
			{
				"Sysex Common Realtime MTC Full Message Second",
				"rtpmidi.sysex_common_rt_mtc_fm_second",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_fm_fr,
			{
				"Sysex Common Realtime MTC Full Message Frame",
				"rtpmidi.sysex_common_rt_mtc_fm_frame",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u1,
			{
				"Sysex Common Realtime MTC User Bits U1",
				"rtpmidi.sysex_common_rt_mtc_ub_u1",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u2,
			{
				"Sysex Common Realtime MTC User Bits U2",
				"rtpmidi.sysex_common_rt_mtc_ub_u2",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u3,
			{
				"Sysex Common Realtime MTC User Bits U3",
				"rtpmidi.sysex_common_rt_mtc_ub_u3",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u4,
			{
				"Sysex Common Realtime MTC User Bits U4",
				"rtpmidi.sysex_common_rt_mtc_ub_u4",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u5,
			{
				"Sysex Common Realtime MTC User Bits U5",
				"rtpmidi.sysex_common_rt_mtc_ub_u5",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u6,
			{
				"Sysex Common Realtime MTC User Bits U6",
				"rtpmidi.sysex_common_rt_mtc_ub_u6",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u7,
			{
				"Sysex Common Realtime MTC User Bits U7",
				"rtpmidi.sysex_common_rt_mtc_ub_u7",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u8,
			{
				"Sysex Common Realtime MTC User Bits U8",
				"rtpmidi.sysex_common_rt_mtc_ub_u8",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x0f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_ub_u9,
			{
				"Sysex Common Realtime MTC User Bits U9",
				"rtpmidi.sysex_common_rt_mtc_ub_u9",
				FT_UINT8,
				BASE_HEX,
				NULL,
				0x03,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_type,
			{
				"Sysex Common Non-Realtime MTC Type",
				"rtpmidi.sysex_common_nrt_mtc_type",
				FT_UINT8,
				BASE_HEX,
				VALS(rtp_midi_mtc_fm_type_values),
				0x60,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_hr,
			{
				"Sysex Common Non-Realtime MTC Hour",
				"rtpmidi.sysex_common_nrt_mtc_hour",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x1f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_mn,
			{
				"Sysex Common Non-Realtime MTCMinute",
				"rtpmidi.sysex_common_nrt_mtc_minute",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_sc,
			{
				"Sysex Common Non-Realtime MTC Second",
				"rtpmidi.sysex_common_nrt_mtc_second",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_fr,
			{
				"Sysex Common Non-Realtime MTC Frame",
				"rtpmidi.sysex_common_nrt_mtc_frame",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_ff,
			{
				"Sysex Common Non-Realtime MTC Fractional Frame",
				"rtpmidi.sysex_common_nrt_mtc_fract_frame",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_enl,
			{
				"Sysex Common Non-Realtime MTC Event Number (LSB)",
				"rtpmidi.sysex_common_nrt_mtc_en_lsb",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_enm,
			{
				"Sysex Common Non-Realtime MTC Event Number (MSB)",
				"rtpmidi.sysex_common_nrt_mtc_en_msb",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_nrt_mtc_add,
			{
				"Sysex Common Non-Realtime MTC Additional Info",
				"rtpmidi.sysex_common_nrt_mtc_add",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_cue_enl,
			{
				"Sysex Common Realtime MTC Cue Event Number (LSB)",
				"rtpmidi.sysex_common_rt_mtc_cue_en_lsb",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_cue_enm,
			{
				"Sysex Common Realtime MTC Cue Event Number (MSB)",
				"rtpmidi.sysex_common_rt_mtc_cue_en_msb",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_mtc_cue_add,
			{
				"Sysex Common Realtime MTC Cue Additional Info",
				"rtpmidi.sysex_common_rt_mtc_cue_add",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_bar_num,
			{
				"Sysex Common Realtime NI Bar Number",
				"rtpmidi.sysex_common_rt_ni_barnum",
				FT_UINT16,
				BASE_HEX,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_bytes,
			{
				"Sysex Common Realtime NI Bytes",
				"rtpmidi.sysex_common_rt_ni_bytes",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_numerator,
			{
				"Sysex Common Realtime NI Numerator",
				"rtpmidi.sysex_common_rt_ni_numerator",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_denominator,
			{
				"Sysex Common Realtime NI Denominator",
				"rtpmidi.sysex_common_rt_ni_denominator",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_midi_clocks,
			{
				"Sysex Common Realtime NI MIDI clocks in click",
				"rtpmidi.sysex_common_rt_ni_midiclock",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_ni_32nds,
			{
				"Sysex Common Realtime NI 32nd notes in MIDI quarter note",
				"rtpmidi.sysex_common_rt_ni_32nds",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_dc_volume,
			{
				"Sysex Common Realtime DC Volume",
				"rtpmidi.sysex_common_rt_dc_volume",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		},
		{
			&hf_rtp_midi_sysex_common_rt_dc_balance,
			{
				"Sysex Common Realtime DC Balance",
				"rtpmidi.sysex_common_rt_dc_balance",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x7f7f,
				NULL, HFILL
			}
		}

	};





	static gint *ett[] = {
		&ett_rtp_midi,
		&ett_rtp_midi_commands,
		&ett_rtp_midi_journal,
		&ett_rtp_midi_command,
		&ett_rtp_midi_systemjournal,
		&ett_rtp_midi_channeljournals,
		&ett_rtp_midi_channelchapters,
		&ett_rtp_midi_cj_chapter_p,
		&ett_rtp_midi_cj_chapter_c,
		&ett_rtp_midi_cj_chapter_c_loglist,
		&ett_rtp_midi_cj_chapter_c_logitem,
		&ett_rtp_midi_cj_chapter_m,
		&ett_rtp_midi_cj_chapter_m_loglist,
		&ett_rtp_midi_cj_chapter_m_logitem,
		&ett_rtp_midi_cj_chapter_m_log_msb,
		&ett_rtp_midi_cj_chapter_m_log_lsb,
		&ett_rtp_midi_cj_chapter_m_log_a_button,
		&ett_rtp_midi_cj_chapter_m_log_c_button,
		&ett_rtp_midi_cj_chapter_m_log_count,
		&ett_rtp_midi_cj_chapter_w,
		&ett_rtp_midi_cj_chapter_n,
		&ett_rtp_midi_cj_chapter_n_loglist,
		&ett_rtp_midi_cj_chapter_n_logitem,
		&ett_rtp_midi_cj_chapter_n_octets,
		&ett_rtp_midi_cj_chapter_e,
		&ett_rtp_midi_cj_chapter_e_loglist,
		&ett_rtp_midi_cj_chapter_e_logitem,
		&ett_rtp_midi_cj_chapter_t,
		&ett_rtp_midi_cj_chapter_a,
		&ett_rtp_midi_cj_chapter_a_loglist,
		&ett_rtp_midi_cj_chapter_a_logitem,
		&ett_rtp_midi_systemchapters,
		&ett_rtp_midi_sj_chapter_d,
		&ett_rtp_midi_sj_chapter_d_field_b,
		&ett_rtp_midi_sj_chapter_d_field_g,
		&ett_rtp_midi_sj_chapter_d_field_h,
		&ett_rtp_midi_sj_chapter_d_field_j,
		&ett_rtp_midi_sj_chapter_d_field_k,
		&ett_rtp_midi_sj_chapter_d_field_y,
		&ett_rtp_midi_sj_chapter_d_field_z,
		&ett_rtp_midi_sj_chapter_v,
		&ett_rtp_midi_sj_chapter_q,
		&ett_rtp_midi_sj_chapter_f,
		&ett_rtp_midi_sj_chapter_f_complete,
		&ett_rtp_midi_sj_chapter_f_partial,
		&ett_rtp_midi_sj_chapter_x,
		&ett_rtp_midi_sj_chapter_x_first,
		&ett_rtp_midi_sj_chapter_x_data,
		&ett_rtp_midi_channeljournal,
		&ett_rtp_midi_sysex_data,
		&ett_rtp_midi_sysex_common_rt,
		&ett_rtp_midi_sysex_common_nrt,
		&ett_rtp_midi_sysex_common_tune_note,
		&ett_rtp_midi_sysex_manu,
		&ett_rtp_midi_sysex_edu
	};

	proto_rtp_midi = proto_register_protocol( RTP_MIDI_DISSECTOR_NAME, RTP_MIDI_DISSECTOR_SHORTNAME, RTP_MIDI_DISSECTOR_ABBREVIATION );
	proto_register_field_array( proto_rtp_midi, hf, array_length( hf ) );
	proto_register_subtree_array( ett, array_length( ett ) );

	rtp_midi_module = prefs_register_protocol( proto_rtp_midi, NULL );
	prefs_register_obsolete_preference( rtp_midi_module, "midi_payload_type_value");
	rtp_midi_handle = register_dissector( RTP_MIDI_DISSECTOR_ABBREVIATION, dissect_rtp_midi, proto_rtp_midi );
}



void
proto_reg_handoff_rtp_midi( void )
{
	dissector_add_string("rtp_dyn_payload_type", "rtp-midi", rtp_midi_handle);
	dissector_add_uint_range_with_preference( "rtp.pt", "", rtp_midi_handle );

}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: t
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 noexpandtab:
 * :indentSize=8:tabSize=8:noTabs=false:
 */
