// Copyright 2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"). You may not
// use this file except in compliance with the License. A copy of the
// License is located at
//
// http://aws.amazon.com/apache2.0/
//
// or in the "license" file accompanying this file. This file is distributed
// on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
// either express or implied. See the License for the specific language governing
// permissions and limitations under the License.

// Usage:
// go run rip-gen.go
package main

import (
	"bytes"
	"fmt"
	"go/format"
	"io/ioutil"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"github.com/aws/amazon-ssm-agent/agent/jsonutil"
)

const (
	startMarkerPrefix = "//AUTOGEN_START_"
	endMarkerPrefix   = "//AUTOGEN_END_"
	serviceStatus     = "GA"
)

var serviceNames = []string{"MessageGatewayService"}

var desiredAccessibilityAttributes = []string{"PUBLICLY_ANNOUNCED"}
var avoidedAccessibilityAttributes = []string{"RETAIL", "TEST"}
var secretAccessibilityAttributes = []string{"ISOLATED"}

var ripConfigJsonFilePath = filepath.Join("agent", "rip", "rip_static_config.json")
var ripUtilFilePath = filepath.Join("agent", "rip", "riputil.go")

// As of 03/22/21
var expectedNumOfEndpoints = 27

// rip-gen is a simple program that generates the riputil file,
// riputil file contains information about services endpoints.
func main() {
	var ripConfig AutoGeneratedRipConfigSchema
	if err := jsonutil.UnmarshalFile(ripConfigJsonFilePath, &ripConfig); err != nil {
		fmt.Printf("Error in opening/parsing RIP config file %v\n", err)
		os.Exit(1)
	}
	for _, serviceName := range serviceNames {
		generateEndpoints(serviceName, ripConfig)
	}
}

func generateEndpoints(serviceName string, ripConfig AutoGeneratedRipConfigSchema) {
	var mapEntries []string
	for i, region := range ripConfig.Regions {
		if (allStringsInSlice(desiredAccessibilityAttributes, region.AccessibilityAttributes) ||
			allStringsInSlice(secretAccessibilityAttributes, region.AccessibilityAttributes)) &&
			!anyStringInSlice(avoidedAccessibilityAttributes, region.AccessibilityAttributes) {
			for _, service := range ripConfig.Regions[i].Services {
				if service.Name == serviceName && service.Status == serviceStatus {
					mapEntries = append(mapEntries, "\""+region.RegionName+"\" :\t\""+service.Endpoint+"\",\n")
				}
			}
		}
	}
	if len(mapEntries) < expectedNumOfEndpoints {
		fmt.Printf("Error: Expected at least %d endpoints, got %d\n", expectedNumOfEndpoints, len(mapEntries))
		os.Exit(1)
	}
	sort.Strings(mapEntries)

	var newContent bytes.Buffer
	for _, mapEntry := range mapEntries {
		newContent.WriteString(mapEntry)
	}

	if err := replaceFileContents(newContent.String(), serviceName); err != nil {
		fmt.Printf("Error in updating the riputil file %v\n", err)
	}
}

func replaceFileContents(newContent string, serviceName string) (err error) {
	if oldContent, err := ioutil.ReadFile(ripUtilFilePath); err == nil {
		parts := strings.Split(string(oldContent), startMarkerPrefix+serviceName)
		if len(parts) != 2 {
			fmt.Printf("Error: Cannot generate rip endpoint map code. It is possible that the %s file was manually edited.\n", ripUtilFilePath)
			os.Exit(1)
		}
		oldContentPart1 := parts[0]
		moreParts := strings.Split(string(parts[1]), endMarkerPrefix+serviceName)
		if len(moreParts) != 2 {
			fmt.Printf("Error: Cannot generate rip endpoint map code. It is possible that the %s file was manually edited.\n", ripUtilFilePath)
			os.Exit(1)
		}
		oldContentPart2 := moreParts[1]
		fileNewContent := oldContentPart1 + startMarkerPrefix + serviceName + "\n" + newContent + endMarkerPrefix + serviceName + oldContentPart2
		if newContentFormatted, err := format.Source([]byte(fileNewContent)); err == nil {
			err = ioutil.WriteFile(ripUtilFilePath, newContentFormatted, 0644)
		}
	} else {
		fmt.Printf("Error: cannot open file %s", ripUtilFilePath)
		os.Exit(1)
	}
	return
}

func anyStringInSlice(keyWords []string, searchList []string) bool {
	for _, listItem := range searchList {
		for _, keyword := range keyWords {
			if keyword == listItem {
				return true
			}
		}
	}
	return false
}

func allStringsInSlice(keyWords []string, searchList []string) bool {
	foundCount := 0
	for _, listItem := range searchList {
		for _, keyword := range keyWords {
			if keyword == listItem {
				foundCount++
				break
			}
		}
	}
	return len(keyWords) == foundCount
}

// Auto generated using https://mholt.github.io/json-to-go/
type AutoGeneratedRipConfigSchema struct {
	Regions []struct {
		AccessibilityAttributes []string `json:"accessibility_attributes"`
		RegionName              string   `json:"region_name"`
		Services                []struct {
			Name     string `json:"name"`
			Endpoint string `json:"endpoint"`
			Status   string `json:"status"`
		} `json:"services"`
	} `json:"regions"`
}
