# Copyright (c) 2023, SUSE LLC, All rights reserved.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 3.0 of the License, or (at your option) any later version.
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# You should have received a copy of the GNU Lesser General Public
# License along with this library.

"""Utility functions for the cloud guest registration"""

import base64
import configparser
import glob
import ipaddress
import json
import logging
import os
import pickle
import random
import re
import requests
import site
import socket
import stat
import subprocess
import sys
import time
import toml
import yaml

from collections import namedtuple
from lxml import etree
from pathlib import Path
from requests.auth import HTTPBasicAuth

from cloudregister import smt

AVAILABLE_SMT_SERVER_DATA_FILE_NAME = 'availableSMTInfo_%s.obj'
FRAMEWORK_IDENTIFIER = 'framework_info'
HOSTSFILE_PATH = '/etc/hosts'
NEW_REGISTRATION_MARKER = 'newregistration'
OLD_REGISTRATION_DATA_DIR = '/var/lib/cloudregister/'
REGISTRATION_DATA_DIR = '/var/cache/cloudregister/'
REGISTERED_SMT_SERVER_DATA_FILE_NAME = 'currentSMTInfo.obj'
RMT_AS_SCC_PROXY_MARKER = 'rmt_is_scc_proxy'
REGISTRY_CREDENTIALS_PATH = '/etc/containers/config.json'
PROFILE_LOCAL_PATH = '/etc/profile.local'
REGISTRIES_CONF_PATH = '/etc/containers/registries.conf'
DOCKER_CONFIG_PATH = '/etc/docker/daemon.json'
SUMA_REGISTRY_CONF_PATH = '/etc/uyuni/uyuni-tools.yaml'
BASE_PRODUCT_PATH = '/etc/products.d/baseproduct'
SUSE_REGISTRY = 'registry.suse.com'

requests.packages.urllib3.disable_warnings(
    requests.packages.urllib3.exceptions.InsecureRequestWarning
)


# ----------------------------------------------------------------------------
def add_hosts_entry(smt_server):
    """Add an entry to the /etc/hosts file for the given SMT server"""

    smt_hosts_entry_comment = '\n# Added by SMT registration do not remove, '
    smt_hosts_entry_comment += 'retain comment as well\n'
    smt_ip = smt_server.get_ipv4()
    if has_rmt_ipv6_access(smt_server):
        smt_ip = smt_server.get_ipv6()
    entry = '%s\t%s\t%s\n' % (
        smt_ip,
        smt_server.get_FQDN(),
        smt_server.get_name()
    )
    if smt_server.get_registry_FQDN():
        entry += '%s\t%s\n' % (
            smt_ip,
            smt_server.get_registry_FQDN()
        )

    with open('/etc/hosts', 'a') as hosts_file:
        hosts_file.write(smt_hosts_entry_comment)
        hosts_file.write(entry)
    logging.info('Modified /etc/hosts, added: %s' % entry)


# ----------------------------------------------------------------------------
def add_region_server_args_to_URL(api, cfg):
    """Add arguments from the instance to the given api URL.
       The arguments are generated by a plugin that must provide the
       generateRegionSrvArgs() function.
    """

    mod = __get_framework_plugin(cfg)
    if not mod:
        return api

    regionSrvArgs = __get_region_server_args(mod)
    if regionSrvArgs:
        api += '?' + regionSrvArgs

    return api


# ----------------------------------------------------------------------------
def clean_hosts_file(domain_name):
    """Remove the smt server and registry entries from the /etc/hosts file"""
    if isinstance(domain_name, str):
        domain_name = domain_name.encode()
    new_hosts_content = []
    # Handle entries as bytes,
    # Yes, users put non ascii characters into /etc/hosts
    with open(HOSTSFILE_PATH, 'rb') as hosts_file:
        content = hosts_file.readlines()

    smt_announce_found = None
    for entry in content:
        if b'# Added by SMT' in entry:
            smt_announce_found = True
            continue
        if smt_announce_found:
            if domain_name in entry:
                continue

        new_hosts_content.append(entry)

    # Clean up empty lines at the end of the file such that there is only 1
    # empty line remaining
    try:
        while new_hosts_content[-1] == b'\n':
            new_hosts_content.pop(-1)
        new_hosts_content.append(b'\n')
    except IndexError:
        pass

    with open(HOSTSFILE_PATH, 'wb') as hosts_file:
        for entry in new_hosts_content:
            hosts_file.write(entry)


# ----------------------------------------------------------------------------
def clean_framework_identifier():
    """Remove the framework identification data"""
    framework_file_path = os.path.join(get_state_dir(), FRAMEWORK_IDENTIFIER)
    if os.path.exists(framework_file_path):
        os.unlink(framework_file_path)


# ----------------------------------------------------------------------------
def clean_smt_cache():
    """Clean the disk cache for SMT data"""

    smt_data = glob.glob(os.path.join(get_state_dir(), '*SMTInfo*'))
    for cache_entry in smt_data:
        os.unlink(cache_entry)


# ----------------------------------------------------------------------------
def clear_new_registration_flag():
    """Clear the new registration marker"""
    try:
        os.unlink(os.path.join(get_state_dir(), NEW_REGISTRATION_MARKER))
    except FileNotFoundError:
        pass


# ----------------------------------------------------------------------------
def clear_rmt_as_scc_proxy_flag():
    """Clear the marker that indicates that RMT is used as SCC proxy"""
    try:
        os.unlink(os.path.join(get_state_dir(), RMT_AS_SCC_PROXY_MARKER))
    except FileNotFoundError:
        pass


# ----------------------------------------------------------------------------
def clean_non_free_extensions():
    """Remove/uninstall non free extensions from the system."""
    extensions = get_extensions()
    installed_products = get_installed_products()

    for extension in extensions:
        # The following not free condition checks on
        # the access capability of the extension
        # This is a server side capability
        if not extension.get('free', True):
            triplet = get_product_triplet(extension)
            triplet = '{name}/{version}/{arch}'.format(
                name=triplet.name,
                version=triplet.version,
                arch=triplet.arch
            )
            if triplet in installed_products:
                reg_prod = register_product(
                    registration_target=get_current_smt(),
                    product=triplet,
                    de_register=True
                )
                msg = 'Non free extension {} '.format(triplet)
                if reg_prod.returncode:
                    msg += 'failed to be removed'
                else:
                    msg += 'removed'

                logging.info(msg)


# ----------------------------------------------------------------------------
def get_product_tree(product_file=BASE_PRODUCT_PATH):
    """
    Read product element from baseproduct and return an etree
    """
    if os.path.isfile(product_file):
        with open(product_file, 'r') as product:
            raw_xml = product.read()
            product_entry_point = raw_xml.index('<product')
            return etree.fromstring(raw_xml[product_entry_point:])


# ----------------------------------------------------------------------------
def get_product_triplet(product_tree):
    """
    Extract identifier, version and arch information from
    the given product tree
    """
    product_type = namedtuple(
        'product_type', ['name', 'version', 'arch']
    )
    if type(product_tree) is dict:
        return product_type(
                name=product_tree.get('identifier'),
                version=product_tree.get('version'),
                arch=product_tree.get('arch')
        )

    return product_type(
        name=product_tree.find('name').text,
        version=product_tree.find('version').text,
        arch=product_tree.find('arch').text
    )


# ----------------------------------------------------------------------------
def get_extensions():
    """Get the extensions of the base product on the update server."""
    prod_data = get_product_data()
    return prod_data.get('extensions', [])


# ----------------------------------------------------------------------------
def get_product_data(registration_target=None):
    if not registration_target:
        registration_target = get_current_smt()
    if not registration_target:
        logging.info('No update server set')
        return {}

    base_product = get_product_triplet(get_product_tree())
    headers = {'Accept': 'application/vnd.scc.suse.com.v4+json'}
    query_args = 'identifier=%s&version=%s&arch=%s' % (
        base_product.name, base_product.version, base_product.arch
    )
    user, password = get_credentials(get_credentials_file(registration_target))
    auth_creds = HTTPBasicAuth(user, password)
    url_system_products = 'https://%s/connect/systems/products?%s' % (
        registration_target.get_FQDN(), query_args
    )
    res = requests.get(url_system_products, auth=auth_creds, headers=headers)
    if res.status_code != requests.codes.ok:
        err_msg = 'Unable to obtain product information from server "%s"\n'
        err_msg += '\t%s\n\t%s, exiting.'
        ips = '%s,%s' % (
            registration_target.get_ipv4(), registration_target.get_ipv6()
        )
        err_msg = err_msg % (ips, res.reason, res.content.decode("UTF-8"))
        logging.error(err_msg)
        raise Exception(err_msg)

    return res.json()


# ----------------------------------------------------------------------------
def credentials_files_are_equal(repo_credentials):
    """Compare the base credentials files the the repo header and make
       sure they have the same values."""

    if not repo_credentials or not isinstance(repo_credentials, str):
        return False

    base_credentials_location = '/etc/zypp/credentials.d/'
    target_root = get_zypper_target_root()
    credentials_location = target_root + base_credentials_location
    credentials_base = os.path.join(credentials_location, 'SCCcredentials')
    credentials_header = os.path.join(credentials_location, repo_credentials)
    ref_user, ref_pass = get_credentials(credentials_base)
    repo_user, repo_pass = get_credentials(credentials_header)
    if (ref_user != repo_user) or (ref_pass != repo_pass):
        return False

    return True


# ----------------------------------------------------------------------------
def enable_repository(repo_name):
    """Enable the given repository"""

    cmd = ['zypper', 'mr', '-e', repo_name]
    res = exec_subprocess(cmd)
    if res:
        logging.error('Unable to enable repository %s' % repo_name)


# ----------------------------------------------------------------------------
def exec_subprocess(cmd, return_output=False):
    """Execute the given command as a subprocess (blocking)
       Returns one off:
           - exit code of the command
           - stdout, stderr and exit code
           - -1 indicates an exception"""
    try:
        proc = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        out, err = proc.communicate()
        if return_output:
            return (out, err, proc.returncode)
        return proc.returncode
    except OSError:
        return -1


# ----------------------------------------------------------------------------
def get_register_cmd():
    """Determine which command we need to use to register the system"""

    register_cmd = '/usr/sbin/SUSEConnect'
    if is_transactional_system():
        cmd_name = 'transactional-update'
        for path in ['/sbin/', '/usr/sbin/']:
            exec_path = path + cmd_name
            if os.path.exists(exec_path):
                register_cmd = exec_path
                break

    return register_cmd


# ----------------------------------------------------------------------------
def register_product(
    registration_target, regcode='', email='',
    instance_data_filepath='', product='', de_register=False
):
    """
    Wrapper for SUSEConnect

    Call SUSEConnect or the respective transactional update call
    with the given parameters:

    registration_target:
        SMT registration object
    regcode:
        registration code
    email:
        registration e-mail
    instance_data_filepath:
        CSP related metadata file
    product:
        SUSE product triplet
    """
    register_cmd = get_register_cmd()
    if not (os.path.exists(register_cmd) and os.access(register_cmd, os.X_OK)):
        err_msg = 'No registration executable found'
        print(err_msg, file=sys.stderr)
        logging.error(err_msg)
        sys.exit(1)

    cmd = [register_cmd]
    target_url = registration_target.get_FQDN()

    # distinguish command between standard and read-only system(transactional)
    if 'transactional' in register_cmd:
        cmd += ['register']

    cmd += ['--url', 'https://{0}'.format(target_url)]

    if de_register:
        if not product:
            error_msg = 'De-register the system is not allowed for SUSEConnect'
            print(error_msg)
            logging.error(error_msg)
            sys.exit(1)

        cmd += ['--de-register']

    # SUSE product triplet
    if product:
        cmd += ['--product', product]

    # set path to a metadata file, cloud specific
    if instance_data_filepath:
        cmd += ['--instance-data', instance_data_filepath]

    # set e-mail
    if email:
        cmd += ['--email', email]

    # set registration code for given product
    if regcode:
        cmd += ['--regcode', regcode]

    log_information = ' '.join(cmd)
    if regcode:
        # registration codes should not end up in the log
        log_information = log_information.replace(regcode, 'XXXX')

    logging.info('Registration: {0}'.format(log_information))
    call = subprocess.Popen(
        cmd,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    output, error = call.communicate()
    suseconnect_type = namedtuple(
        'suseconnect_type', ['returncode', 'output', 'error']
    )
    return suseconnect_type(
        returncode=call.returncode,
        output=output.decode(),
        error=error.decode()
    )


# ----------------------------------------------------------------------------
def fetch_smt_data(cfg, proxies, quiet=False):
    """Retrieve the data for the region SMT servers from a remote host"""
    response = None
    if cfg.has_option('server', 'metadata_server'):
        metadata_url = cfg.get('server', 'metadata_server')
        msg = 'Using metadata server "%s" to obtain SMT information'
        logging.info(msg % metadata_url)
        try:
            response = requests.get(
                metadata_url,
                timeout=15.0,
                proxies=proxies
            )
            if response.status_code != 200:
                raise Exception(
                    'Metadata server returned %s' % response.status_code
                )
        except Exception as e:
            logging.error('=' * 20)
            logging.error(str(e))
            logging.error('Unable to obtain update server information, exiting')
            sys.exit(1)
        smt_info = json.loads(response.text)
        expected_entries = ('fingerprint', 'SMTserverIP', 'SMTserverName')
        smt_info_xml = '<regionSMTdata><smtInfo '
        for attr in expected_entries:
            value = smt_info.get(attr)
            if not value:
                logging.error(
                    'Metadata server did not supply a value for "%s"' % attr
                )
                logging.error('Cannot proceed, exiting registration code')
                sys.exit(1)
            smt_info_xml += '%s="%s" ' % (attr, value)
        smt_info_xml += '/></regionSMTdata>'
        smt_data_root = etree.fromstring(smt_info_xml)
    else:
        # Get the API to use
        api = cfg.get('server', 'api')
        logging.info('Using API: %s' % api)
        # Add regionserver arguments
        api = add_region_server_args_to_URL(api, cfg)
        # Get the location of the cert files for the region servers
        cert_dir = cfg.get('server', 'certLocation')
        region_servers = cfg.get('server', 'regionsrv').split(',')
        # sort into ipv4 & ipv6 buckets, randomize, then make a best-effort by
        # trying IPv6 first, if available.
        region_servers_ipv4 = []
        region_servers_ipv6 = []
        region_servers_dns = []
        for srv in region_servers:
            srv_id = srv.strip()
            try:
                ip_addr = ipaddress.ip_address(srv_id)
            except ValueError:
                region_servers_dns.append(srv_id)
                continue
            if isinstance(ip_addr, ipaddress.IPv6Address):
                region_servers_ipv6.append(ip_addr)
            else:
                region_servers_ipv4.append(ip_addr)
        random.shuffle(region_servers_ipv4)
        random.shuffle(region_servers_ipv6)
        if has_ipv6_access():
            region_servers = region_servers_ipv6 + region_servers_ipv4
        else:
            region_servers = region_servers_ipv4
        random.shuffle(region_servers_dns)
        region_servers += region_servers_dns
        # After the network interface is up, i.e. After=network-online
        # is satisfied routing on the framework side may not be setup yet
        # and we may not be able to immediately reach the update
        # infrastructure.
        # As time passes is more likely that the framework completes routing
        # setup for outbound traffic. Therefore we wait less time the longer
        # we are in the process of attempting to access the update
        # infrastructure.
        # Processing delays are handled at two levels; the individual request
        # timeout and the time we wait between the retry of obtaining the
        # update server information. For each timeout the starting number
        # 15 and 20 seconds, respectively is an arbitrary choice. The maximum
        # wait time is dependent on the number of region servers in the
        # framework. The maximum wait time is calculated as follows
        #
        # sumof(num_regions_srvs * 15/i) + sumof(20/j)
        # where i = 1,2,3
        #       j = 1,2
        #
        # For a configuration with 3 region servers the framework has 112.5
        # seconds to setup the routing
        #
        max_attempts = 3
        have_update_server_data = False
        for attempt in range(max_attempts):
            if have_update_server_data:
                break
            retry_cnt = attempt + 1
            request_timeout = 15/retry_cnt
            retry_timeout = int(20/retry_cnt)
            if not quiet:
                logging.info(
                    'Getting update server information, attempt %d' % retry_cnt
                )
            for srv in region_servers:
                srvName = str(srv)
                if not quiet:
                    logging.info('\tUsing region server: %s' % srvName)
                certFile = os.path.join(cert_dir, srvName + '.pem')
                if not os.path.isfile(certFile):
                    logging.info(
                        '\tNo cert found: %s skip this server' % certFile
                    )
                    continue
                try:
                    url = 'https://%s/%s' % (srvName, api)
                    # Per rfc3986 IPv6 addresses in a URI are enclosed in []
                    if isinstance(srv, ipaddress.IPv6Address):
                        url = 'https://[%s]/%s' % (srvName, api)
                    response = requests.get(
                        url,
                        verify=certFile,
                        timeout=request_timeout,
                        proxies=proxies
                    )
                    if response.status_code == 200:
                        have_update_server_data = True
                        break
                    else:
                        logging.error('=' * 20)
                        logging.error(
                            'Server returned: %d' % response.status_code
                        )
                        logging.error('Server error: "%s"' % response.reason)
                        logging.error('=' * 20)
                        if srv == region_servers[-1]:
                            logging.error('\tAll servers reported an error')
                except requests.exceptions.RequestException:
                    if quiet:
                        continue
                    logging.error('\tNo response from: %s' % srvName)
                    if srv == region_servers[-1]:
                        logging.error('\tNone of the servers responded')
                        logging.error('\tAttempted: %s' % region_servers)
            else:
                # No message on the last go around
                if attempt + 1 < max_attempts:
                    log_msg = 'Waiting %d seconds before next attempt'
                    logging.info(log_msg % retry_timeout)
                    time.sleep(retry_timeout)
        else:
            err_msg = 'Request not answered by any server '
            err_msg += 'after %d attempts' % max_attempts
            logging.error(err_msg)
            logging.error('Exiting without registration')
            sys.exit(1)

        smt_data_root = etree.fromstring(response.text)

    return smt_data_root


# ----------------------------------------------------------------------------
def find_equivalent_smt_server(configured_smt, known_smt_servers):
    """Find an SMT server that is equivalent to the currently configured
       SMT server, only consider responsive servers"""
    for update_server in known_smt_servers:
        # Take a shortcut and only compare the IPv4 address to
        # skip the same server in the list. If the IPv4 addresses of
        # a given object are the same and other data is different we have
        # a really big problem.
        if update_server.get_ipv4() == configured_smt.get_ipv4():
            continue
        if (
                update_server.is_equivalent(configured_smt) and
                update_server.is_responsive()
        ):
            return update_server

    return None


# ----------------------------------------------------------------------------
def find_repos(contains_name):
    """Find all repos that contain the given name (case insensitive) in
       the repo name"""
    repo_names = []
    search_for = contains_name.lower()
    repos = glob.glob('/etc/zypp/repos.d/*.repo')
    for repo in repos:
        repo_cfg = get_config(repo)
        for section in repo_cfg.sections():
            cfg_repo_name = repo_cfg.get(section, 'name')
            if search_for in cfg_repo_name.lower():
                repo_names.append(cfg_repo_name)

    return repo_names


# ----------------------------------------------------------------------------
def get_activations():
    """Get the activated products from the update server"""
    update_server = get_smt()
    user, password = get_credentials(get_credentials_file(update_server))
    if not (user and password):
        logging.error(
            'Unable to extract username and password '
            'for "%s"' % update_server.get_FQDN()
        )
        return {}

    auth_creds = HTTPBasicAuth(user, password)

    instance_data = bytes(get_instance_data(get_config()), 'utf-8')
    headers = {'X-Instance-Data': base64.b64encode(instance_data)}

    res = requests.get(
        'https://%s/connect/systems/activations' % update_server.get_FQDN(),
        auth=auth_creds,
        headers=headers
    )

    if res.status_code != 200:
        srv_ipv4 = update_server.get_ipv4()
        srv_ipv6 = update_server.get_ipv6()
        logging.error(
            'Unable to get product info from '
            'update server: "%s"' % str((srv_ipv4, srv_ipv6))
        )
        logging.error('\tReason: "%s"' % res.reason)
        logging.error('\tCode: %d', res.status_code)
        return {}

    return res.json()


# ----------------------------------------------------------------------------
def get_available_smt_servers():
    """Return a list of available SMT servers"""
    available_smt_servers = []
    if not os.path.exists(get_state_dir()):
        return available_smt_servers
    smt_data_files = glob.glob(
        os.path.join(
            get_state_dir(),
            AVAILABLE_SMT_SERVER_DATA_FILE_NAME % '*'
        )
    )
    for smt_data in smt_data_files:
        available_smt_servers.append(get_smt_from_store(smt_data))

    return available_smt_servers


# ----------------------------------------------------------------------------
def get_config(configFile=None):
    """Read configuration file and return a config object"""
    if not configFile:
        configFile = '/etc/regionserverclnt.cfg'

    cfg = configparser.RawConfigParser()
    try:
        parsed = cfg.read(configFile)
    except configparser.Error:
        print('Could not parse configuration file %s' % configFile)
        type, value, tb = sys.exc_info()
        print(format(value))
        sys.exit(1)

    if not parsed:
        print('Error parsing config file: %s' % configFile)
        sys.exit(1)

    return cfg


# ----------------------------------------------------------------------------
def get_credentials(credentials_file):
    """Return username and password extracted from the given file"""
    username = None
    password = None
    if not os.path.exists(credentials_file):
        return (username, password)
    with open(credentials_file) as cred_file:
        credentials = cred_file.readlines()
    known_entries = ('system_token')
    for entry in credentials:
        if entry.startswith('username'):
            username = entry.split('=')[-1].strip()
        elif entry.startswith('password'):
            password = entry.split('=')[-1].strip()
        elif not entry.startswith(known_entries):
            logging.warning('Found unknown entry in '
                            'credentials file "%s"' % entry)

    return (username, password)


# ----------------------------------------------------------------------------
def setup_registry(registry_fqdn, username, password):
    """Set all the necessary parts for the registry,
       returns True if the setup completed, False otherwise."""
    os.makedirs(os.path.dirname(REGISTRY_CREDENTIALS_PATH), exist_ok=True)

    return (
        set_registry_auth_token(registry_fqdn, username, password) and
        set_container_engines_env_vars() and set_registries_conf(registry_fqdn)
    )


# ----------------------------------------------------------------------------
def get_registry_credentials(set_new):
    """Read the registry credentials file
       and return its content or an empty dict."""
    config_json = {}
    status = None
    # status can have 3 values
    # - None, meaning opening the file suceeded
    # - 0, meaning opening the file failed but preserving succeeded
    # - <returncode> of the op or -1, meaning preserving it failed
    file_error = True
    try:
        with open(REGISTRY_CREDENTIALS_PATH, 'r') as cred_json:
            config_json = json.load(cred_json)
    except OSError as error:
        logging.info(str(error))
        action = 'open'
    except json.decoder.JSONDecodeError:
        action = 'parse'
    else:
        file_error = False

    if file_error:
        error_msg = 'Unable to %s existing %s, preserving file as %s.bak' % (
            action, REGISTRY_CREDENTIALS_PATH, REGISTRY_CREDENTIALS_PATH
        )
        if set_new:
            error_msg += ', writing new credentials'
        logging.info(error_msg)
        mv_file_cmd = 'mv -Z {} {}.bak'.format(
            REGISTRY_CREDENTIALS_PATH, REGISTRY_CREDENTIALS_PATH
        ).split()
        status = exec_subprocess(mv_file_cmd)
        message = 'File not preserved.' if status else 'File preserved.'
        logging.info(message)

    return config_json, status


# ----------------------------------------------------------------------------
def write_registry_credentials(content, set_new):
    """Update the registry credentials file with the value of 'content'."""
    try:
        with open(REGISTRY_CREDENTIALS_PATH, 'w') as cred_json_file:
            json.dump(content, cred_json_file)
        action_done = 'added' if set_new else 'removed'
        message = 'Credentials for the registry {} in {}'.format(
            action_done, REGISTRY_CREDENTIALS_PATH
        )
        logging.info(message)
        return True
    except Exception as error:
        action_done = 'add' if set_new else 'remove'
        message = 'Could not {} the registry credentials: {}'.format(
            action_done, error
        )
        logging.error(message)


# ----------------------------------------------------------------------------
def set_registry_auth_token(registry_fqdn, username, password):
    """Set the auth token to access the SUSE registry."""
    config_json = {}
    if os.path.exists(REGISTRY_CREDENTIALS_PATH):
        config_json, preserve_status = get_registry_credentials(set_new=True)
        if preserve_status:
            # there was an error parsing the credentials json file
            # and we could not preserve the file
            # we do not write anything new to the credentials file
            return False

    auth_token = __generate_registry_auth_token(username, password)
    # set the new registry credentials,
    # independently of what that content was,
    # preserving the rest of the dictionary or keys, if any
    registry_credentials = {registry_fqdn: {'auth': auth_token}}
    config_json['auths'] = dict(
        list(config_json.get('auths', {}).items()) +  # old content
        list(registry_credentials.items())            # new content
    )
    updated = write_registry_credentials(content=config_json, set_new=True)
    return updated


# ----------------------------------------------------------------------------
def set_container_engines_env_vars():
    """Set the environment variables needed for the container runtimes
    to find the config file."""
    env_vars = {
        'REGISTRY_AUTH_FILE': REGISTRY_CREDENTIALS_PATH,
        'DOCKER_CONFIG': os.path.dirname(REGISTRY_CREDENTIALS_PATH)
    }
    bashrc_local_lines = []
    if os.path.exists(PROFILE_LOCAL_PATH):
        try:
            with open(PROFILE_LOCAL_PATH, 'r') as bashrc_local:
                bashrc_local_lines = bashrc_local.read()
        except OSError as error:
            logging.info('Could not open %s: %s' % (PROFILE_LOCAL_PATH, error))
            failed = __mv_file_backup(PROFILE_LOCAL_PATH)
            if failed:
                return False

    keys_to_add = ''
    for env_var_key, env_var_value in env_vars.items():
        if env_var_key not in bashrc_local_lines:
            keys_to_add += (
                '{}export {}={}{}'.format(
                    os.linesep, env_var_key, env_var_value, os.linesep
                )
            )

    if keys_to_add:
        return update_bashrc(keys_to_add, 'a')


# ----------------------------------------------------------------------------
def set_registries_conf(registry_fqdn):
    """Add the registry container engines search configuration."""
    container_engines_conf_set = (
        __set_registries_conf_podman(registry_fqdn) and
        __set_registries_conf_docker(registry_fqdn)
    )
    suma_registry_conf_set = True
    if is_suma_instance():
        suma_registry_conf_set = __set_registry_fqdn_suma(registry_fqdn)

    return container_engines_conf_set and suma_registry_conf_set


# ----------------------------------------------------------------------------
def get_registry_conf_file(container_path, container):
    registries_conf = {}
    try:
        with open(container_path, 'r') as registries_conf_file:
            if container == 'podman':
                registries_conf = toml.load(registries_conf_file)
            if container == 'docker':
                registries_conf = json.load(registries_conf_file)
            return registries_conf, False
    except IOError as error:
        logging.info(str(error))
        action = 'open'
    except json.decoder.JSONDecodeError:
        action = 'parse'

    logging.info(
        'Could not %s %s, preserving file as %s.bak' % (
            action, container_path, container_path
        )
    )
    mv_file_cmd = 'mv -Z {} {}.bak'.format(
        container_path, container_path
    ).split()
    failed = exec_subprocess(mv_file_cmd)
    message = 'File not preserved.' if failed else 'File preserved.'
    logging.info(message)
    return {}, failed


# ----------------------------------------------------------------------------
def update_bashrc(content, mode):
    """Update the env vars for the container engines
    with the location of the config file to the bashrc local file."""
    try:
        with open(PROFILE_LOCAL_PATH, mode) as bashrc_file:
            bashrc_file.write(content)
        logging.info('%s updated' % PROFILE_LOCAL_PATH)
        return True
    except OSError as error:
        logging.error('Could not update %s: %s' % (PROFILE_LOCAL_PATH, error))
        failed = __mv_file_backup(PROFILE_LOCAL_PATH)
        return not failed


# ----------------------------------------------------------------------------
def clean_registry_setup():
    """Remove the data previously set to make the registry work."""
    smt = get_smt_from_store(__get_registered_smt_file_path())
    private_registry_fqdn = smt.get_registry_FQDN() if smt else ''
    clean_registry_auth(private_registry_fqdn)
    unset_env_vars()
    clean_registries_conf(private_registry_fqdn)


# ----------------------------------------------------------------------------
def clean_registry_auth(registry_fqdn):
    """Clean the auth token from the config json file."""
    if not os.path.exists(REGISTRY_CREDENTIALS_PATH):
        logging.info('Credentials file does not exist. Nothing to do')
        return True

    config_json, preserve_status = get_registry_credentials(set_new=False)
    if (
        (not config_json or config_json == {'auths': {}}) and
        not preserve_status
    ):
        # credentials file is empty or
        # credentials file is not empty but its content is useless or
        # could not access the file but the backup suceeded
        if preserve_status is None:
            logging.info('JSON content is empty')
            os.unlink(REGISTRY_CREDENTIALS_PATH)
        return True

    if preserve_status:
        return False

    # we could open the credentials file
    # and it is not empty
    # unset the registry credentials
    try:
        if same_registry_auth_content(config_json, registry_fqdn):
            # if the content of the registry auth file is only
            # our auth info, remove the file
            logging.info(
                'Registry authentication config only contains managed content.'
                ' Removing the file %s' % REGISTRY_CREDENTIALS_PATH
            )
            os.unlink(REGISTRY_CREDENTIALS_PATH)
            return True

        if registry_fqdn:
            logging.info(
                'Unsetting the auth entry for %s' % registry_fqdn
            )
        else:
            logging.info('Unsetting the auth entry based on the token')
            auth_token = __generate_registry_auth_token()
            entries = config_json.get('auths', {}).items()
            entry = [entry for entry, auth in entries
                     if auth == auth_token and 'registry' in entry and
                     'susecloud.net' in entry]
            registry_fqdn = ''.join(entry)

        if config_json.get('auths', {}).pop(registry_fqdn, {}):
            # file was not empty or
            # file could not be parsed and the remove cmd did not fail
            # there is content worth updating the credentials file
            # the dictionary has changed since read from the auth config file
            # we write the changed dictionary back to the file
            logging.info('Registry auth entry unset')
            return write_registry_credentials(
                content=config_json, set_new=False
            )
    except AttributeError:
        logging.error('The entry for "auths" key is not a dictionary')
        logging.info(
            'Preserving file %s as %s.bak' % (
                REGISTRY_CREDENTIALS_PATH, REGISTRY_CREDENTIALS_PATH
            )
        )
        mv_file_cmd = 'mv -Z {} {}.bak'.format(
            REGISTRATION_DATA_DIR, REGISTRY_CREDENTIALS_PATH
        )
        status = exec_subprocess(mv_file_cmd)
        message = 'File not preserved.' if status else 'File preserved.'
        logging.info(message)


# ----------------------------------------------------------------------------
def same_registry_auth_content(content, registry_fqdn):
    """Check if the registry auth content contains only SUSE registry info."""
    auth_token = __generate_registry_auth_token()
    one_key = len(content.keys()) == 1
    if registry_fqdn:
        expected_content = {'auths': {registry_fqdn: auth_token}}
        different_auth_token = False
        if one_key:
            different_auth_token = content.get('auths', {}).get(registry_fqdn)

        # if that is True is safe to remove the file
        return content == expected_content or different_auth_token
    elif one_key and content.get('auths'):
        one_auths_key = len(content.get('auths').keys()) == 1
        return one_auths_key and auth_token in content.get('auths').values()


# ----------------------------------------------------------------------------
def unset_env_vars():
    """Remove the registry environment variables."""
    env_vars = ['REGISTRY_AUTH_FILE', 'DOCKER_CONFIG']
    if not os.path.exists(PROFILE_LOCAL_PATH):
        logging.info('%s file does not exist' % PROFILE_LOCAL_PATH)
        # remove the enviroment variables from the env, if present
        return True

    lines, modified, preserved_failed, mv_backup = clean_bashrc_local(env_vars)
    succeeded = True
    if modified:
        # we could access the file and registry env vars were found and removed
        succeeded = update_bashrc(''.join(lines), 'w')
    else:
        # Either we could access the file and no registry env vars were found
        # or we could not access the file and attempted to create a backup
        # if backup scenario, log is already populated
        if not mv_backup:
            # we could access the bashrc local file and
            # no env vars were found
            logging.info(
                'Environment variables not present in %s' % PROFILE_LOCAL_PATH
            )
            succeeded = True
        elif preserved_failed:
            # we could not access the bashrc local file
            # the attempt to create a backup failed
            succeeded = False

    return succeeded


# ----------------------------------------------------------------------------
def clean_bashrc_local(env_vars):
    """
    Clean the registry env vars, if any, from the PROFILE_LOCAL_PATH file

    :returns:
        - bashrc_local_new_lines: list - the new lines after cleaning
        - modified: bool - whether the content of the file was modified
        - preserved_failed: bool - if accessing the file failed,
                                   whether preserve op failed
        - mv_backup: bool: whether the backup file operation suceeded
    """
    bashrc_local_lines = []
    try:
        with open(PROFILE_LOCAL_PATH, 'r') as bashrc_local:
            bashrc_local_lines = bashrc_local.readlines()
    except OSError as error:
        logging.info('Could not open %s: %s' % (PROFILE_LOCAL_PATH, error))
        failed = __mv_file_backup(PROFILE_LOCAL_PATH)
        return [], False, failed, True

    bashrc_local_new_lines = []
    modified = False
    for bashrc_line in bashrc_local_lines:
        for env_var in env_vars:
            if not bashrc_line.startswith('#') and env_var in bashrc_line:
                modified = True
                break
        else:
            bashrc_local_new_lines.append(bashrc_line)

    return bashrc_local_new_lines, modified, False, False


# ----------------------------------------------------------------------------
def clean_registries_conf(registry_fqdn):
    """Clean up the registry content from Podman and/or Docker config files."""
    clean_registries_conf_podman(registry_fqdn)
    clean_registries_conf_docker(registry_fqdn)


# ----------------------------------------------------------------------------
def clean_registries_conf_podman(private_registry_fqdn):
    """Clean up the registry content from the REGISTRIES_CONF_PATH file."""
    if not os.path.exists(REGISTRIES_CONF_PATH):
        return True

    registries_conf, failed = get_registry_conf_file(
        REGISTRIES_CONF_PATH, 'podman'
    )
    if failed:
        return False

    modified_by_us = False
    # Drop from unqualified-search-registries
    unqualified_search_reg = registries_conf.get(
        'unqualified-search-registries', []
    )
    if not private_registry_fqdn:
        private_registry_fqdn = __matches_susecloud(unqualified_search_reg)
    if private_registry_fqdn in unqualified_search_reg:
        unqualified_search_reg.remove(private_registry_fqdn)
        modified_by_us = True

    # Drop from registry
    registry_mirrors = registries_conf.get(
        'registry', []
    )
    mirror_index = 0
    for registry_mirror in registry_mirrors:
        if registry_mirror.get('location', '') == private_registry_fqdn:
            registry_mirrors.pop(mirror_index)
            modified_by_us = True
            break
        mirror_index += 1

    # write registry setup if modified by us
    if modified_by_us:
        registries_conf['unqualified-search-registries'] = \
            unqualified_search_reg
        registries_conf['registry'] = registry_mirrors
        logging.info(
            'SUSE registry information has been removed from {0}'.format(
                REGISTRIES_CONF_PATH
            )
        )
        return write_registries_conf(
            registries_conf, REGISTRIES_CONF_PATH, 'podman'
        )
    return True


# ----------------------------------------------------------------------------
def clean_registries_conf_docker(private_registry_fqdn):
    """Clean up the registry content from the DOCKER_CONFIG_PATH file."""
    if not os.path.exists(DOCKER_CONFIG_PATH):
        return True

    docker_cfg_json, failed = get_registry_conf_file(
        DOCKER_CONFIG_PATH, 'docker'
    )
    if failed:
        logging.info(
            'Unable to read "{0}", cleanup not possible'.format(
                DOCKER_CONFIG_PATH
            )
        )
        return False

    modified_by_us = False
    # Drop registry-mirrors in docker daemon.json
    registry_mirrors = docker_cfg_json.get('registry-mirrors', [])
    if not private_registry_fqdn:
        private_registry_url = __matches_susecloud(registry_mirrors)
    else:
        private_registry_url = 'https://{0}'.format(private_registry_fqdn)
    if private_registry_url in registry_mirrors:
        registry_mirrors.remove(private_registry_url)
        modified_by_us = True

    # write registry setup if modified by us
    if modified_by_us:
        docker_cfg_json['registry-mirrors'] = registry_mirrors
        logging.info(
            'SUSE registry information has been removed from {0}'.format(
                DOCKER_CONFIG_PATH
            )
        )
        return write_registries_conf(
            docker_cfg_json, DOCKER_CONFIG_PATH, 'docker'
        )
    return True


# ----------------------------------------------------------------------------
def write_registries_conf(registries_conf, container_path, container_name):
    """Write registries_conf content to container_path."""
    try:
        if container_name == 'podman':
            with open(container_path, 'w') as registries_conf_file:
                toml.dump(registries_conf, registries_conf_file)
        if container_name == 'docker':
            with open(container_path, 'w') as registries_conf_file:
                json.dump(registries_conf, registries_conf_file)
        logging.info('File %s updated' % container_path)
        return True
    except IOError as error:
        logging.info(str(error))
        action = 'open'
    except TypeError:
        action = 'write'

    logging.error('Could not %s %s' % (action, container_path))


# ----------------------------------------------------------------------------
def get_credentials_file(update_server, service_name=None):
    """Return the credentials filename.
    Credentials are stored per service. If there is a service
    associated with a given repo use those credentials, if not
    use the default credentials associated with the server providing
    the service.
    Note this is based on naming convention. This can only succeed
    after the system is properly registered.
    """
    credentials_file = ''

    target_root = get_zypper_target_root()
    credentials_loc = target_root + '/etc/zypp/credentials.d/'
    credential_names = [
        '*' + update_server.get_FQDN().replace('.', '_'),
        'SCC*'
    ]
    if service_name:
        credential_names.insert(0, '*' + service_name + '*')

    for entry in credential_names:
        cred_files = glob.glob(os.path.join(credentials_loc, entry))
        if not cred_files:
            logging.info('No credentials entry for "%s"' % entry)
            continue
        if len(cred_files) > 1:
            logging.warning(
                'Found multiple credentials for "%s" entry and '
                'hoping for the best' % service_name)
        credentials_file = cred_files[0]
        break

    if not credentials_file:
        logging.error('No matching credentials file found')

    return credentials_file


# ----------------------------------------------------------------------------
def get_hosts(with_loopback=False):
    """
    Read /etc/hosts file into the following structure

    {
        hostname: {
            'address': some,
            'ipv6': true|false,
            'ipv4': true|false
        }
    }
    """
    with open(HOSTSFILE_PATH, 'r') as hosts_fd:
        hostlines = hosts_fd.readlines()
    hostlines = [line.strip() for line in hostlines
                 if not line.startswith('#') and line.strip() != '']
    hosts = {}
    for line in hostlines:
        hostnames = line.split('#')[0].split()[1:]
        hostaddress = line.split('#')[0].split()[0]
        loopback = False
        if hostaddress == '::1' or hostaddress == '127.0.0.1':
            loopback = True
        if not with_loopback and loopback:
            # no loopback addresses wanted
            continue
        for hostname in hostnames:
            # ipv6 vs. ipv4 address detection is based on the following
            # information. All valid IPv6 address strings contain at
            # least 1 colon in the first 6 characters, and no valid
            # IPv4 address strings contain a colon in the first 6
            # characters.
            hosts[hostname] = {
                'address': hostaddress,
                'ipv6': True if ':' in hostaddress[:6] else False,
                'ipv4': False if ':' in hostaddress[:6] else True
            }
    return hosts


# ----------------------------------------------------------------------------
def get_current_smt():
    """Return the data for the current SMT server.
       The current SMT server is the server against which this client
       is registered."""
    smt = get_smt_from_store(__get_registered_smt_file_path())
    if not smt:
        return
    # Verify that this system is also in /etc/hosts and we are in
    # a consistent state
    # Handle entries as bytes,
    # Yes users put non ascii characters into /etc/hosts
    smt_ipv4 = smt.get_ipv4()
    smt_ipv6 = smt.get_ipv6()
    smt_fqdn = smt.get_FQDN()
    # A bit cumbersome to support Python 3.4
    ipv4_search = r'%s\s' % smt_ipv4
    ipv6_search = r'%s\s' % smt_ipv6
    fqdn_search = r'\s%s\s' % smt_fqdn
    with open(HOSTSFILE_PATH, 'rb') as hosts_file:
        hosts = hosts_file.read()
    if (
        not (
            re.search(ipv4_search.encode(), hosts) or
            re.search(ipv6_search.encode(), hosts)
        ) or not
            re.search(fqdn_search.encode(), hosts)
    ):
        os.unlink(__get_registered_smt_file_path())
        return
    if not is_registered(smt_fqdn):
        return

    return smt


# ----------------------------------------------------------------------------
def get_framework_identifier_path():
    """Return the path for the framework identifier file."""
    return os.path.join(get_state_dir(), FRAMEWORK_IDENTIFIER)


# ----------------------------------------------------------------------------
def get_instance_data(config):
    """Run the configured instance data collection command and return
       the result or none."""
    instance_data = b''
    if (
            config.has_section('instance') and
            config.has_option('instance', 'dataProvider')
    ):
        instance_data_cmd = config.get('instance', 'dataProvider')
        cmd = instance_data_cmd.split()[0]
        if cmd != 'none':
            if not cmd.startswith('/'):
                cmd_lookup = exec_subprocess(['which', cmd])
                if cmd_lookup:
                    errMsg = 'Could not find configured dataProvider: %s' % cmd
                    logging.error(errMsg)
            if os.access(cmd, os.X_OK):
                instance_data, errors, returncode = exec_subprocess(
                    instance_data_cmd.split(), True
                )
                if errors:
                    errMsg = 'Data collected from stderr for instance '
                    errMsg += 'data collection "%s"' % errors.decode()
                    logging.error(errMsg)
                if not instance_data:
                    warn_msg = 'Possible issue accessing the metadata '
                    warn_msg += 'service. Metadata is empty, may result '
                    warn_msg += 'in registration failure.'
                    logging.warning(warn_msg)

    # Marker for the server to not return https:// formatted
    # service and repo information
    inst_data = instance_data.decode()

    return inst_data + '<repoformat>plugin:susecloud</repoformat>\n'


# ----------------------------------------------------------------------------
def get_installed_products():
    """Get a list of the installed products"""

    products = []
    # It is possible for users to force a zypper process before we had
    # a chance to set up the repos. We'll wait for the lock for a little while
    for wait_cnt in range(4):
        if is_zypper_running():
            time.sleep(5 * wait_cnt)
        else:
            break
    else:
        errMsg = 'Wait time expired could not acquire zypper lock file'
        logging.error(errMsg)
        return products

    zypper_products_cmd = ["zypper", "--no-remote", "-x", "products"]
    try:
        cmd = subprocess.Popen(zypper_products_cmd, stdout=subprocess.PIPE)
        product_xml = cmd.communicate()
        # Just in case something else started zypper again
        if cmd.returncode != 0:
            errMsg = 'zypper product query returned with zypper code %d'
            logging.error(errMsg % cmd.returncode)
            return products
    except OSError:
        logging.error(
            'Could not get product list %s', ' '.join(zypper_products_cmd)
        )
        return products

    # Determine the base product
    baseProdSet = '/etc/products.d/baseproduct'
    baseprodName = None
    if os.path.islink(baseProdSet):
        baseprod = os.path.realpath(baseProdSet)
        baseprodName = baseprod.split(os.sep)[-1].split('.')[0]
    else:
        errMsg = 'No baseproduct installed system cannot be registered'
        logging.error(errMsg)
        return products

    product_tree = etree.fromstring(product_xml[0].decode())
    for child in product_tree.find("product-list"):
        name = child.attrib['name']
        if name == baseprodName:
            continue
        vers = child.attrib['version']
        arch = child.attrib['arch']
        prod = os.path.join(name, vers, arch)
        if prod not in products:
            products.append(prod)

    return products


# ----------------------------------------------------------------------------
def get_repo_url(repo_name):
    """Return the url for the given repository"""
    repos = glob.glob('/etc/zypp/repos.d/*.repo')
    for repo in repos:
        repo_cfg = get_config(repo)
        for section in repo_cfg.sections():
            if repo_name == repo_cfg.get(section, 'name'):
                return repo_cfg.get(section, 'baseurl')

    return ''


# ----------------------------------------------------------------------------
def get_smt(cache_refreshed=None):
    """Returns an update server that is reachable."""

    available_servers = get_available_smt_servers()
    current_smt = get_current_smt()
    if current_smt:
        if is_registered(current_smt.get_FQDN()):
            alive = current_smt.is_responsive()
            if alive:
                logging.info(
                    'Current update server will be used: '
                    '"%s"' % str(
                        (current_smt.get_ipv4(), current_smt.get_ipv6())
                    )
                )
                return current_smt
            else:
                # The configured server is not responsive:
                # Especially during initial registration we have seen
                # failover due to network instability in the early startup
                # phases of a system. Give the network a chance to sort
                # itself out. We make the wait time shorter for each
                # iteration as the network has had more time to become
                # operational.
                for delay in [5, 3, 1]:
                    time.sleep(delay)
                    msg = 'Waiting for current server to show up for %d s'
                    logging.info(msg % delay)
                    if current_smt.is_responsive():
                        logging.info(
                            'No failover needed, system access recovered'
                        )
                        return current_smt
                # Looks like the target server is really not reachable,
                # let's check if we can find an equivalent server.
                # If this happens during initial registration we need to give
                # the target server time to share its registration data, take
                # another break. The registration sharing timer is set to
                # ~3 seconds
                # We depend on a background process on the update servers
                # to re-sync the databases.
                time.sleep(5)
                new_target = find_equivalent_smt_server(
                    current_smt,
                    available_servers
                )
                if new_target:
                    logging.info(
                        'Using equivalent update server: '
                        '"%s"' % str(
                            (new_target.get_ipv4(), new_target.get_ipv6())
                        )
                    )
                    # Fetch cert for new target server
                    import_smt_cert(new_target)
                    # Verify the new target server has our credentials
                    replace_hosts_entry(current_smt, new_target)
                    credentials_file_path = get_credentials_file(new_target)
                    user, password = get_credentials(credentials_file_path)
                    if not has_smt_access(
                            new_target.get_FQDN(), user, password
                    ):
                        original_smt_ips = str(
                            (current_smt.get_ipv4(), current_smt.get_ipv6())
                        )
                        new_target_ips = str(
                            (new_target.get_ipv4(), new_target.get_ipv6())
                        )
                        msg = 'Sibling update server, %s, does not have '
                        msg += 'system credentials cannot failover. Retaining '
                        msg += 'current, %s, target update server.'
                        msg += 'Try again later.'
                        logging.error(msg % (new_target_ips, original_smt_ips))
                        replace_hosts_entry(new_target, current_smt)
                        return current_smt
                    set_as_current_smt(new_target)
                    return new_target
    else:
        # Try any other update server we might know about
        for server in available_servers:
            if server.is_responsive():
                import_smt_cert(server)
                logging.info(
                    'Found alternate update server: '
                    '"%s"' % str((server.get_ipv4(), server.get_ipv6()))
                )
                # Assume the new server is in the same domain
                clean_hosts_file(server.get_domain_name())
                add_hosts_entry(server)
                set_as_current_smt(server)
                return server

    # No server was found update the cache of known servers and try again
    if not cache_refreshed:
        clean_smt_cache()
        __populate_srv_cache()
        return get_smt(True)


# ----------------------------------------------------------------------------
def get_smt_from_store(smt_store_file_path):
    """Create an SMTinstance from the stored data."""
    if not os.path.exists(smt_store_file_path):
        return None

    smt = None
    with open(smt_store_file_path, 'rb') as smt_file:
        u = pickle.Unpickler(smt_file)
        try:
            smt = u.load()
        except pickle.UnpicklingError:
            pass

    return smt


# ----------------------------------------------------------------------------
def get_state_dir():
    return REGISTRATION_DATA_DIR


# ----------------------------------------------------------------------------
def get_update_server_name_from_hosts(ignore_inconsistent=False):
    """Try and extract the update server name from the /etc/hosts file"""
    if not ignore_inconsistent:
        logging.warning('The system is in an inconsistent state repositories '
                        'definitions, cached update server data, and '
                        'credentials file do not match')
    servers = get_available_smt_servers()
    with open(HOSTSFILE_PATH, 'rb') as hosts_file:
        hosts_content = hosts_file.read()
    for server in servers:
        name = server.get_FQDN().encode()
        if name in hosts_content:
            return name.decode()


# ----------------------------------------------------------------------------
def get_zypper_command():
    """Returns the command line for zypper if zypper is running"""
    zypper_pid = get_zypper_pid()
    zypper_cmd = None
    if zypper_pid:
        with open('/proc/%s/cmdline' % zypper_pid, 'r') as zypper_pid_file:
            zypper_cmd = zypper_pid_file.read()
        zypper_cmd = zypper_cmd.replace('\x00', ' ')

    return zypper_cmd


# ----------------------------------------------------------------------------
def get_zypper_pid():
    """Return the PID of zypper if it is running"""
    pid = ''
    for executable_name in ['zypper', 'Zypp-main']:
        zyppPIDCmd = ['ps', '-C', executable_name, '-o', 'pid=']
        zyppPID = subprocess.Popen(zyppPIDCmd, stdout=subprocess.PIPE)
        pidData = zyppPID.communicate()
        pid = pidData[0].decode().split('\n')[0].strip()
        if pid:
            break

    return pid


# ----------------------------------------------------------------------------
def get_zypper_pid_cache():
    """Return the PID for zypper stored in cache"""
    zypper_pid = 0
    if not os.path.exists(os.path.join(get_state_dir(), 'zypper_pid')):
        return zypper_pid
    with open(get_state_dir() + 'zypper_pid') as zypper_state_file:
        return zypper_state_file.read()


# ----------------------------------------------------------------------------
def get_zypper_target_root():
    """Return the target root if zypper has the --root argument to
       specify a target directory in which to operate.
    """
    zypper_cmd = get_zypper_command()
    target_root = ''
    for root_arg in ('-R', '--root'):
        if zypper_cmd and root_arg in zypper_cmd:
            target_root = zypper_cmd.split(root_arg)[-1].split()[0].strip()
            break

    return target_root


# ----------------------------------------------------------------------------
def has_network_access_by_ip_address(server_ip):
    """Check if we can connect to the given server"""
    try:
        connection = socket.create_connection((server_ip, 443), timeout=2)
    except OSError:
        try:
            ip_ver = 'IPv%d' % ipaddress.ip_address(server_ip).version
        except ValueError:
            ip_ver = 'UNKNOWN'
        logging.info(
            'Skipping %s protocol version, no network configuration' % ip_ver
            )
        return False

    connection.close()
    return True


# ----------------------------------------------------------------------------
def has_rmt_ipv6_access(smt):
    """IPv6 access is possible if we have an SMT server that has an IPv6
       address and it can be accessed over IPv6"""
    if not has_ipv6_access() or not smt.get_ipv6():
        return False
    logging.info('Attempt to access update server over IPv6')
    protocol = 'http'  # Default for backward compatibility
    if https_only(get_config()):
        protocol = 'https'
    try:
        # Per rfc3986 IPv6 addresses in a URI are enclosed in []
        cert_res = requests.get(
            '%s://[%s]/smt.crt' % (protocol, smt.get_ipv6()),
            timeout=3,
            verify=False
        )
    except Exception:
        logging.info('Update server not reachable over IPv6')
        return False
    if cert_res and cert_res.status_code == 200:
        return True


# ----------------------------------------------------------------------------
def has_nvidia_support():
    """Check if the instance has Nvidia capabilities"""
    try:
        pci_info, errors, returncode = exec_subprocess(['lspci'], True)
    except TypeError:
        logging.info(
            'lspci command not found, instance Nvidia support cannot '
            'be determined'
        )
        return False

    if 'NVIDIA' in pci_info.decode():
        logging.info('Instance has Nvidia support')
        return True

    return False


# ----------------------------------------------------------------------------
def has_region_changed(cfg):
    """Check if the region has changed. If no region information is available
       we assume the instance has not moved."""

    framework = __get_system_mfg()
    plugin = __get_framework_plugin(cfg)
    region = 'unknown'
    if plugin:
        region_hint = __get_region_server_args(plugin)
        if region_hint:
            region = region_hint.split('=')[-1]

    if framework == 'unknown' or region == 'unknown':
        # We cannot determine with certainty if anything has changed
        # Assume everything is as it was
        return False

    try:
        with open(get_framework_identifier_path()) as framework_file:
            registered_region = json.loads(
                framework_file.read()
            )
    except Exception:
        return False

    if (
            framework == registered_region.get('framework') and
            region == registered_region.get('region')
    ):
        return False

    return True


# ----------------------------------------------------------------------------
def has_rmt_in_hosts(server):
    """
    Check if an entry for the given update server is in the hosts file
    """
    hosts = get_hosts()
    rmt_server_name = server.get_FQDN()
    if rmt_server_name in hosts.keys():
        return True
    return False


# ----------------------------------------------------------------------------
def has_registry_in_hosts(server):
    """
    Check if an entry for the given registry server is in the hosts file
    """
    hosts = get_hosts()
    registry_server_name = server.get_registry_FQDN()
    if registry_server_name in hosts.keys():
        return True
    return False


# ----------------------------------------------------------------------------
def has_services(smt_server_name):
    """Check if repositories exist."""
    service_files = glob.glob('/etc/zypp/services.d/*.service')
    for service_file in service_files:
        with open(service_file) as svc_file:
            content = svc_file.readlines()
        for entry in content:
            if entry.startswith('url'):
                if (
                    smt_server_name in entry or
                    'plugin:/susecloud' in entry or
                    'plugin:susecloud' in entry
                ):
                    return True
    service_plugins = __get_service_plugins()
    if service_plugins:
        return True

    return False


# ----------------------------------------------------------------------------
def has_smt_access(update_server_fqdn, user, password):
    """Check if the given update server can be accessed with the provided
       credentials."""
    auth_creds = HTTPBasicAuth(user, password)
    api = 'https://%s/connect/systems/products/migrations'
    res = requests.post(api % update_server_fqdn, auth=auth_creds, json={})
    # Anything else bubbles to the top
    if res.reason == 'Unauthorized':
        return False

    return True


# ----------------------------------------------------------------------------
def https_only(config):
    """Return true if the configuration has httpsOnly set to true"""
    if (
            config.has_section('instance') and
            config.has_option('instance', 'httpsOnly')
    ):
        https_setting = config.get('instance', 'httpsOnly')
        if https_setting == 'true':
            return True

    return False


# ----------------------------------------------------------------------------
def import_smtcert_12(smt):
    """Import the SMT certificate on SLES 12"""
    key_chain = '/etc/pki/trust/anchors'
    if not smt.write_cert(key_chain):
        return 0
    if not update_ca_chain(['update-ca-certificates']):
        return 0

    return 1


# ----------------------------------------------------------------------------
def import_smt_cert(smt):
    """Import the SMT certificate for the given server"""
    # 1 step of indirection to allow us to handle different cert import
    # mechanisms per distribution
    import_result = import_smtcert_12(smt)
    if not import_result:
        logging.error('SMT certificate import failed')
        return None
    # Check if the underlying Python packages use certs that are built in
    # bsc#1214801
    for site_path in site.getsitepackages():
        py_pack_certs = glob.glob(os.path.join(site_path, 'certifi/*.pem'))
        if py_pack_certs:
            logging.warning(
                'SMT certificate imported, but "%s" exist. '
                'This may lead to registration failure' % ' '.join(
                    py_pack_certs)
            )

    return 1


# ----------------------------------------------------------------------------
def get_profile_env_var(varname, profile_file):
    shell_command = [
        'bash', '-c', 'source {0}; echo ${1}'.format(
            profile_file, varname
        )
    ]
    variable_content, error, returncode = exec_subprocess(shell_command, True)
    if not error:
        return variable_content.decode().strip()


# ----------------------------------------------------------------------------
def is_registry_registered(registry_server_name):
    """
    Check if the instance is registered for accessing the
    container registry server on the update server
    """
    # A good registry setup exists if we can find the entry in the
    # hosts file and a proper environment definition
    hosts = get_hosts()
    registry_auth_file = get_profile_env_var(
        'REGISTRY_AUTH_FILE', PROFILE_LOCAL_PATH
    )
    if registry_server_name in hosts.keys() \
       and registry_auth_file == REGISTRY_CREDENTIALS_PATH \
       and os.path.exists(registry_auth_file):
        return True
    return False


# ----------------------------------------------------------------------------
def is_new_registration():
    """Indicate whether a new registration is in process based on the
       marker file. Note it is the responsibility of the process to properly
       manage the marker file"""
    return os.path.exists(
        os.path.join(get_state_dir(), NEW_REGISTRATION_MARKER)
    )


# ----------------------------------------------------------------------------
def is_registered(smt_server_name):
    """Check if the instance is already registered"""
    # For a "valid" registration we need to have credentials and a service
    return has_services(smt_server_name) and __has_credentials(smt_server_name)


# ----------------------------------------------------------------------------
def is_registration_supported(cfg):
    """
    Check if a registration process is available
    based on the supported package manager backend

    zypper:
      Indicates a SUSE/SLES system including the registration
      process based on SUSEConnect and /etc/products.d/baseproduct

    dnf:
      Indicates a product of the RHEL family for which we do not
      provide subscription management.
    """
    registration_supported = True
    try:
        package_backend = cfg.get('service', 'packageBackend')
        if package_backend == 'dnf':
            logging.info('Registration for RHEL product family requested')
            logging.info('Exit after repository server hosts entry setup')
            registration_supported = False

        return registration_supported
    except configparser.NoSectionError as e:
        if 'server' not in cfg.sections():
            logging.error('Error accessing the config file: {}'.format(e))
            return False

    return registration_supported


# ----------------------------------------------------------------------------
def is_scc_connected():
    """If any repo url points to suse.com then at least some of the
       repositories come from SCC and the SCCcredentials file is
       expected to have the system credentials for SCC access."""

    repo_files = glob.glob('/etc/zypp/repos.d/*.repo')
    for repo in repo_files:
        repo_cfg = get_config(repo)
        for section in repo_cfg.sections():
            url = repo_cfg.get(section, 'baseurl')
            if url and 'suse.com' in url:
                return True

    return False


# ----------------------------------------------------------------------------
def is_transactional_system():
    """
    Determine if we are running on a transactional update system. This means
    the root filesystem is read only and the transactional-update command
    can be found.
    """
    # Figure out if we are on RO transactional-update system
    output, error, returncode = exec_subprocess(
        ['findmnt', '--noheadings', '--json', '/'], return_output=True
    )
    if returncode != 0:
        logging.warning('Unable to find filesystem information for "/"')
        return False
    else:
        fsinfo = json.loads(output.decode())
        fsdata = fsinfo.get('filesystems')
        if fsdata:
            fsoptions = fsdata[0].get('options')
            # If we are on a RO system we need to use the
            # transactional-update command
            if 'ro' in fsoptions.split(','):
                cmd_name = 'transactional-update'
                for path in ['/sbin/', '/usr/sbin/']:
                    exec_path = path + cmd_name
                    if os.path.exists(exec_path):
                        return True
            else:
                return False
            err_msg = 'transactional-update command not found.'
            err_msg += ' But is required on a RO filesystem for '
            err_msg += 'registration'
            logging.error(err_msg)
            print(err_msg, file=sys.stderr)
            # If we cannot find the transactional-update command a number of
            # follow on errors would be triggered. Exit to avoid triggering
            # these meaningless errors.
            sys.exit(1)


# ----------------------------------------------------------------------------
def is_zypper_running():
    """Check if zypper is running"""
    # Zypper doesn't remove it's pid file, need to consult the process table
    zypper_pid = get_zypper_pid()
    if zypper_pid != '':
        return True

    return False


# ----------------------------------------------------------------------------
def refresh_zypper_pid_cache():
    """Write the current zypper pid to the cache file"""
    zypper_pid = get_zypper_pid()
    with open(os.path.join(get_state_dir(), 'zypper_pid'), 'w') as cache_file:
        cache_file.write(zypper_pid)


# ----------------------------------------------------------------------------
def set_as_current_smt(smt):
    """Store the given SMT as the current SMT server."""
    if not os.path.exists(get_state_dir()):
        os.system('mkdir -p %s' % get_state_dir())
    store_smt_data(__get_registered_smt_file_path(), smt)


# ----------------------------------------------------------------------------
def set_proxy():
    """Set up proxy environment if applicable"""
    existing_http_proxy = os.environ.get('http_proxy')
    existing_https_proxy = os.environ.get('https_proxy')
    if (existing_http_proxy and existing_https_proxy):
        # If the environment variables exist all external functions used
        # by the registration code will honor them, thus we can tell the
        # client that we didn't do anything, which also happens to be true
        logging.info('Using proxy settings from execution environment')
        logging.info('\thttp_proxy: %s' % existing_http_proxy)
        logging.info('\thttps_proxy: %s' % existing_https_proxy)
        return False
    proxy_config_file = '/etc/sysconfig/proxy'
    if not os.path.exists(proxy_config_file):
        return False
    with open(proxy_config_file, 'r') as pc_file:
        proxy_config = pc_file.readlines()
    http_proxy = ''
    https_proxy = ''
    no_proxy = ''
    for entry in proxy_config:
        if 'PROXY_ENABLED' in entry and 'no' in entry:
            return False
        if 'HTTP_PROXY' in entry:
            http_proxy = entry.split('"')[1]
        if 'HTTPS_PROXY' in entry:
            https_proxy = entry.split('"')[1]
        if 'NO_PROXY' in entry:
            no_proxy = entry.split('"')[1]
    os.environ['http_proxy'] = http_proxy
    os.environ['https_proxy'] = https_proxy
    os.environ['no_proxy'] = no_proxy

    return True


# ----------------------------------------------------------------------------
def set_new_registration_flag():
    """Set a marker that this is the beginning of the registration process"""
    Path(get_state_dir() + NEW_REGISTRATION_MARKER).touch()


# ----------------------------------------------------------------------------
def set_rmt_as_scc_proxy_flag():
    """Set a marker that the RMT registration is a proxy for SCC"""
    Path(get_state_dir(), RMT_AS_SCC_PROXY_MARKER).touch()


# ----------------------------------------------------------------------------
def switch_services_to_plugin():
    """Switches a .service based RIS service that points to the update
       infrastructure to the service plugin"""
    service_plugin = '/usr/sbin/cloudguest-repo-service'
    service_plugin_loc = '/usr/lib/zypp/plugins/services/'
    service_files = glob.glob('/etc/zypp/services.d/*.service')
    update_servers = get_available_smt_servers()
    # If we have no available servers there is a risk that we would break
    # the user setup if additional services that do not point to the update
    # infrastructure exists, thus do nothing
    if not update_servers:
        return
    service_targets = []
    for update_server in update_servers:
        service_targets.append(update_server.get_ipv4())
        service_targets.append(update_server.get_ipv6())
        service_targets.append(update_server.get_FQDN())

    for service_file in service_files:
        link_created = None
        cfg = configparser.RawConfigParser()
        try:
            cfg.read(service_file)
        except configparser.Error:
            logging.warning('Unable to parse "%s" skipping' % service_file)
            continue
        # This implementation depends on each config file having all sections
        # point to the same service target. If this is not the case and one
        # of the sections points to a target we recognize then the other
        # service will be dropped
        for section in cfg.sections():
            url = cfg.get(section, 'url')
            if url and not url.startswith('plugin:'):
                for service_target in service_targets:
                    if (
                            url.startswith('http://%s' % service_target) or
                            url.startswith('https://%s' % service_target)
                    ):
                        link_dest = os.path.join(service_plugin_loc, section)
                        # Assume /usr/sbin and /usr/lib/zypp/plugins are on the
                        # same filesystem
                        if os.path.exists(link_dest):
                            os.unlink(link_dest)
                        os.symlink(service_plugin, link_dest)
                        link_created = 1
        if link_created:
            os.unlink(service_file)


# ----------------------------------------------------------------------------
def remove_registration_data():
    """Reset the instance to an unregistered state"""
    clear_rmt_as_scc_proxy_flag()
    smt_data_file = __get_registered_smt_file_path()
    user, password = get_credentials('/etc/zypp/credentials.d/SCCcredentials')
    if not user:
        if not is_new_registration():
            logging.info('No credentials, nothing to do server side')
        return
    auth_creds = HTTPBasicAuth(user, password)
    if os.path.exists(smt_data_file):
        smt = get_smt_from_store(smt_data_file)
        smt_ips = (smt.get_ipv4(), smt.get_ipv6())
        logging.info('Clean current registration server: %s' % str(smt_ips))
        server_name = smt.get_FQDN()
        domain_name = smt.get_domain_name()
        try:
            response = requests.delete(
                'https://%s/connect/systems' % server_name, auth=auth_creds
            )
            if response.status_code == 204:
                logging.info(
                    'System successfully removed from update infrastructure'
                )
            else:
                rmt_check_msg = 'System unknown to update infrastructure, '
                rmt_check_msg += 'continue with local changes'
                logging.info(rmt_check_msg)
        except requests.exceptions.RequestException as e:
            logging.warning('Unable to remove client registration from server')
            logging.warning(e)
            logging.info('Continue with local artifact removal')
        clean_hosts_file(domain_name)
        __remove_repo_artifacts(server_name)
        os.unlink(smt_data_file)
    if is_scc_connected():
        logging.info('Removing system from SCC')
        try:
            response = requests.delete(
                'https://scc.suse.com/connect/systems', auth=auth_creds
            )
            if response.status_code == 204:
                logging.info('System successfully removed from SCC')
            else:
                scc_check_msg = 'System not found in SCC. The system may still'
                scc_check_msg += ' be tracked against your subscription. It is'
                scc_check_msg += ' recommended to investigate the issue. '
                scc_check_msg += 'System user name: "%s". '
                scc_check_msg += 'Local registration artifacts removed.'
                logging.info(scc_check_msg % user)
        except requests.exceptions.RequestException as e:
            scc_except_msg = 'Unable to remove client registration from SCC. '
            scc_except_msg += 'The system is most likely still tracked against'
            scc_except_msg += ' your subscription. Please inform your SCC '
            scc_except_msg += 'administrator that the system with "%s" user '
            scc_except_msg += 'should be removed from SCC. Registration '
            scc_except_msg += 'artifacts removed locally.'
            logging.error(scc_except_msg % user)
            logging.warning(e)
        __remove_repo_artifacts('suse.com')
    else:
        logging.info('No current registration server set.')


# ----------------------------------------------------------------------------
def replace_hosts_entry(current_smt, new_smt):
    clean_hosts_file(current_smt.get_domain_name())
    add_hosts_entry(new_smt)


# ----------------------------------------------------------------------------
def start_logging():
    """Set up logging"""
    log_filename = '/var/log/cloudregister'
    try:
        logging.basicConfig(
            filename=log_filename,
            level=logging.INFO,
            format='%(asctime)s %(levelname)s:%(message)s'
        )
    except IOError:
        print('Could not open log file "', log_filename, '" for writing.')
        sys.exit(1)


# ----------------------------------------------------------------------------
def store_smt_data(smt_data_file_path, smt):
    """Store the given SMT server information to the given file"""
    with open(smt_data_file_path, 'wb') as smt_data:
        os.fchmod(smt_data.fileno(), stat.S_IREAD | stat.S_IWRITE)
        p = pickle.Pickler(smt_data)
        p.dump(smt)


# ----------------------------------------------------------------------------
def switch_smt_repos(smt):
    """Switch all the repositories pointing to the current SMT server to the
       given SMT server."""
    repo_files = glob.glob('/etc/zypp/repos.d/*.repo')
    __replace_url_target(repo_files, smt)


# ----------------------------------------------------------------------------
def switch_smt_service(smt):
    """Switch the existing service to the given SMT server"""
    service_files = glob.glob('/etc/zypp/services.d/*.service*')
    __replace_url_target(service_files, smt)


# ----------------------------------------------------------------------------
def update_ca_chain(cmd_w_args_lst):
    """Update the CA chain using the given command with arguments"""
    logging.info('Updating CA certificates: %s' % cmd_w_args_lst[0])
    retry_attempts = 3
    for attempt in range(retry_attempts):
        if exec_subprocess(cmd_w_args_lst):
            errMsg = 'Certificate update failed attempt %d' % (attempt + 1)
            logging.error(errMsg)
            time.sleep(5)
        else:
            return 1

    return 0


# ----------------------------------------------------------------------------
def update_rmt_cert(server):
    """Import the cert for the given server if it has changed"""
    # We are currently getting the latest cert information, nothing to do
    if is_new_registration():
        return
    target_ipv4 = server.get_ipv4()
    target_ipv6 = server.get_ipv6()
    proxies = None
    if set_proxy():
        proxies = {
            'http_proxy': os.environ.get('http_proxy'),
            'https_proxy': os.environ.get('https_proxy')
        }
    logging.info('Check for cert update')
    region_rmt_server_data = fetch_smt_data(get_config(), proxies, True)
    region_rmt_servers = []
    for child in region_rmt_server_data:
        region_rmt_servers.append(smt.SMT(child, True))
    for region_rmt_server in region_rmt_servers:
        region_ipv4 = region_rmt_server.get_ipv4()
        region_ipv6 = region_rmt_server.get_ipv6()
        if (region_ipv4 == target_ipv4) and (region_ipv6 == target_ipv6):
            if region_rmt_server != server:
                import_smt_cert(region_rmt_server)
                logging.info('Update server cert updated')
                return True
    logging.info('No cert change')
    return False


# ----------------------------------------------------------------------------
def uses_rmt_as_scc_proxy():
    """Check if the RMT registration is used as an SCC proxy"""

    return Path(get_state_dir(), RMT_AS_SCC_PROXY_MARKER).is_file()


# ----------------------------------------------------------------------------
def write_framework_identifier(cfg):
    """Write a file we use as identifier to detect movement of a golden
       image created from a registered instance"""
    identifier = {}
    identifier['framework'] = __get_system_mfg()
    identifier['region'] = 'unknown'
    plugin = __get_framework_plugin(cfg)
    if plugin:
        identifier['plugin'] = plugin.__file__
        region_hint = __get_region_server_args(plugin)
        if region_hint:
            identifier['region'] = region_hint.split('=')[-1]

    with open(get_framework_identifier_path(), 'w') as framework_file:
        framework_file.write(json.dumps(identifier))


# ----------------------------------------------------------------------------
def has_ipv4_access():
    """Check if we have IPv4 network configuration"""
    return has_network_access_by_ip_address('8.8.8.8')


# ----------------------------------------------------------------------------
def has_ipv6_access():
    """Check if we have IPv6 network configuration"""
    return has_network_access_by_ip_address('2001:4860:4860::8888')


# ----------------------------------------------------------------------------
def is_suma_instance():
    """Return True if there is a match for SUMA and SLE-Micro products,
    otherwise False."""
    products = glob.glob('/etc/products.d/*.prod')
    num_matches = 0
    suma_prods = [
        '/etc/products.d/suse-manager-server.prod',
        '/etc/products.d/sle-micro.prod'
    ]
    for product in products:
        if product.lower() in suma_prods:
            num_matches += 1

    return num_matches == len(suma_prods)


# ----------------------------------------------------------------------------
def get_suma_registry_content():
    """Return the content of the SUMA_REGISTRY_CONF_PATH."""
    suma_reg_config_data = {}
    if not os.path.exists(SUMA_REGISTRY_CONF_PATH):
        return suma_reg_config_data, False

    try:
        with open(SUMA_REGISTRY_CONF_PATH, 'r') as suma_reg_config:
            suma_reg_config_data = yaml.safe_load(suma_reg_config)
        return suma_reg_config_data, False
    except IOError as error:
        logging.info(str(error))
        action = 'open'
    except yaml.YAMLError:
        action = 'parse'

    logging.info('Could not %s %s' % (action, SUMA_REGISTRY_CONF_PATH))
    return {}, True


# Private
# ----------------------------------------------------------------------------
def __get_framework_plugin(cfg):
    """Return the configured framework specific plugin module"""
    mod = None
    if cfg.has_section('instance'):
        module = None
        if cfg.has_option('instance', 'instanceArgs'):
            module = cfg.get('instance', 'instanceArgs')
        if module and module != 'none':
            try:
                mod = __import__('cloudregister.%s' % module, fromlist=[''])
            except Exception:
                msg = 'Configured instanceArgs module could not be loaded. '
                msg += 'Continuing without additional arguments.'
                logging.warning(msg)

    return mod


# ----------------------------------------------------------------------------
def __get_referenced_credentials(smt_server_name):
    """Return a list of credential names referenced by repositories"""
    repo_files = glob.glob('/etc/zypp/repos.d/*.repo')
    referenced_credentials = []
    for repo_file in repo_files:
        repo_cfg = get_config(repo_file)
        for section in repo_cfg.sections():
            url = repo_cfg.get(section, 'baseurl')
            if url:
                if (
                        smt_server_name in url or
                        'plugin:/susecloud' in url or
                        'plugin:susecloud' in url
                ):
                    credentials_name = repo_cfg.get(section, 'service')
                    if credentials_name not in referenced_credentials:
                        referenced_credentials.append(credentials_name)

    return referenced_credentials


# ----------------------------------------------------------------------------
def __get_region_server_args(plugin):
    """Returns the region server arguments"""
    region_srv_args = ''
    retry_cnt = 0
    while retry_cnt < 5:
        try:
            region_srv_args = plugin.generateRegionSrvArgs()
        except AttributeError:
            msg = 'Configured and loaded module "%s" does not provide the '
            msg += 'required generateRegionSrvArgs function.'
            logging.error(msg % plugin.__file__)
            return region_srv_args
        if not region_srv_args:
            retry_cnt += 1
            time.sleep(1)
            continue
        logging.info('Region server arguments: ?%s' % region_srv_args)
        break

    return region_srv_args


# ----------------------------------------------------------------------------
def __get_registered_smt_file_path():
    """Return the file path for the SMT infor stored for the registered
       server"""
    return os.path.join(get_state_dir(), REGISTERED_SMT_SERVER_DATA_FILE_NAME)


# ----------------------------------------------------------------------------
def __get_service_plugins():
    """Return the names of the links to the service plugin"""
    plugin_link_names = []
    service_plugins = glob.glob('/usr/lib/zypp/plugins/services/*')
    for service_plugin in service_plugins:
        if os.path.basename(
                str(Path(service_plugin).resolve())
        ) == 'cloudguest-repo-service':
            plugin_link_names.append(service_plugin)

    return plugin_link_names


# ----------------------------------------------------------------------------
def __get_system_mfg():
    """Returns the system manufacturer information"""
    try:
        vendor, error, returncode = exec_subprocess(
            ['dmidecode', '-s', 'system-manufacturer'], True
        )
    except TypeError:
        vendor = b'unknown'

    return vendor.decode().strip()


# ----------------------------------------------------------------------------
def __has_credentials(smt_server_name):
    """Check if a credentials file exists."""
    referenced_credentials = __get_referenced_credentials(smt_server_name)
    system_credentials = glob.glob('/etc/zypp/credentials.d/*')
    for system_credential in system_credentials:
        if os.path.basename(system_credential) == 'SCCcredentials':
            return True
        if os.path.basename(system_credential) in referenced_credentials:
            return True

    return False


# ----------------------------------------------------------------------------
def __populate_srv_cache():
    """Populate the registration server cache"""
    proxies = None
    if set_proxy():
        proxies = {
            'http_proxy': os.environ.get('http_proxy'),
            'https_proxy': os.environ.get('https_proxy')
        }
    cfg = get_config()
    logging.info('Populating server cache')
    region_smt_data = fetch_smt_data(cfg, proxies, True)
    cnt = 1
    for child in region_smt_data:
        update_server = smt.SMT(child)
        server_cache_file_name = AVAILABLE_SMT_SERVER_DATA_FILE_NAME % cnt
        store_smt_data(
            os.path.join(
                get_state_dir(),
                server_cache_file_name
            ),
            update_server
        )
        cnt += 1


# ----------------------------------------------------------------------------
def __remove_credentials(smt_server_name):
    """Remove the server generated credentials"""
    referenced_credentials = __get_referenced_credentials(smt_server_name)
    # Special files that may exist but may not be referenced
    referenced_credentials += ['SCCcredentials', 'NCCcredentials']
    system_credentials = glob.glob('/etc/zypp/credentials.d/*')
    for system_credential in system_credentials:
        if os.path.basename(system_credential) in referenced_credentials:
            logging.info('Removing credentials: %s' % system_credential)
            os.unlink(system_credential)

    return 1


# ----------------------------------------------------------------------------
def __remove_repo_artifacts(repo_server_name):
    """Remove the artifacts related to repository handling for a registration
    """

    __remove_credentials(repo_server_name)
    __remove_repos(repo_server_name)
    __remove_service(repo_server_name)

    if os.path.exists('/etc/SUSEConnect'):
        os.unlink('/etc/SUSEConnect')


# ----------------------------------------------------------------------------
def __remove_repos(smt_server_name):
    """Remove the repositories for the given server"""
    repo_files = glob.glob('/etc/zypp/repos.d/*')
    for repo_file in repo_files:
        repo_cfg = get_config(repo_file)
        for section in repo_cfg.sections():
            url = repo_cfg.get(section, 'baseurl')
            if url:
                if (
                        smt_server_name in url or
                        'plugin:/susecloud' in url or
                        'plugin:susecloud' in url
                ):
                    logging.info(
                        'Removing repo: %s' % os.path.basename(repo_file)
                    )
                    os.unlink(repo_file)

    return 1


# ----------------------------------------------------------------------------
def __remove_service(smt_server_name):
    """Remove the services pointing to the update infrastructure"""
    service_files = glob.glob('/etc/zypp/services.d/*')
    for service_file in service_files:
        service_cfg = get_config(service_file)
        for section in service_cfg.sections():
            url = service_cfg.get(section, 'url')
            if url:
                if (
                        smt_server_name in url or
                        'plugin:/susecloud' in url or
                        'plugin:susecloud' in url
                ):
                    logging.info('Removing service: %s'
                                 % os.path.basename(service_file))
                    os.unlink(service_file)

    service_plugins = __get_service_plugins()
    for service_plugin in service_plugins:
        os.unlink(service_plugin)

    return 1


# ----------------------------------------------------------------------------
def __replace_url_target(config_files, new_smt):
    """Switch the url of the current SMT server for the given SMT server"""
    current_smt = get_current_smt()
    current_service_server = current_smt.get_FQDN()
    for config_file in config_files:
        with open(config_file, 'r') as cfg_file:
            content = cfg_file.read()
        if current_service_server in content:
            with open(config_file, 'w') as new_config:
                new_config.write(content.replace(
                    current_service_server,
                    new_smt.get_FQDN())
                )


# ----------------------------------------------------------------------------
def __generate_registry_auth_token(username=None, password=None):
    if not (username and password):
        username, password = get_credentials(
            '/etc/zypp/credentials.d/SCCcredentials'
        )

    return base64.b64encode('{username}:{password}'.format(
        username=username,
        password=password
    ).encode()).decode()


# ----------------------------------------------------------------------------
def __mv_file_backup(filename):
    message = ('Preserving file as %s.bak' % filename)
    logging.info(message)
    mv_file_cmd = 'mv -Z {} {}.bak'.format(filename, filename).split()
    failed = exec_subprocess(mv_file_cmd)
    message = 'File not preserved' if failed else 'File preserved'
    logging.info(message)
    return failed


# ----------------------------------------------------------------------------
def __set_registries_conf_podman(private_registry_fqdn):
    """Set the registry search order for Podman."""
    registries_conf = {}
    if os.path.exists(REGISTRIES_CONF_PATH):
        registries_conf, failed = get_registry_conf_file(
            REGISTRIES_CONF_PATH, 'podman'
        )
        if failed:
            return False

    modified_by_us = False
    # Setup unqualified-search-registries
    unqualified_search_reg = registries_conf.get(
        'unqualified-search-registries', []
    )
    if private_registry_fqdn not in unqualified_search_reg:
        # susecloud registry not added, always place it first
        unqualified_search_reg.insert(0, private_registry_fqdn)
        modified_by_us = True

    if SUSE_REGISTRY not in unqualified_search_reg:
        # the suse registry search is provided by the libcontainers-common
        # package. For the case when we cannot find any suse registry we
        # append it after the susecloud registry
        private_registry_index = unqualified_search_reg.index(
            private_registry_fqdn
        )
        unqualified_search_reg.insert(private_registry_index + 1, SUSE_REGISTRY)
        modified_by_us = True

    # Setup registry mirror
    registry_mirrors = registries_conf.get(
        'registry', []
    )
    mirror_found = False
    for registry_mirror in registry_mirrors:
        if registry_mirror.get('location', '') == private_registry_fqdn:
            mirror_found = True
            break
    if not mirror_found:
        registry_mirrors.append(
            {'location': private_registry_fqdn, 'insecure': False}
        )
        modified_by_us = True

    # write registry setup if modified by us
    if modified_by_us:
        registries_conf['unqualified-search-registries'] = \
            unqualified_search_reg
        registries_conf['registry'] = registry_mirrors
        logging.info(
            'Content for {0} has changed, updating the file'.format(
                REGISTRIES_CONF_PATH
            )
        )
        return write_registries_conf(
            registries_conf, REGISTRIES_CONF_PATH, 'podman'
        )


# ----------------------------------------------------------------------------
def __set_registries_conf_docker(private_registry_fqdn):
    suse_registry_url = 'https://{0}'.format(SUSE_REGISTRY)
    private_registry_url = 'https://{0}'.format(private_registry_fqdn)
    docker_cfg_json = {}
    registry_mirrors = []
    os.makedirs(os.path.dirname(DOCKER_CONFIG_PATH), exist_ok=True)
    if os.path.exists(DOCKER_CONFIG_PATH):
        docker_cfg_json, failed = get_registry_conf_file(
            DOCKER_CONFIG_PATH, 'docker'
        )
        if failed:
            return False

    modified_by_us = False
    # Setup registry-mirrors in docker daemon.json
    registry_mirrors = docker_cfg_json.get('registry-mirrors', [])
    if private_registry_url not in registry_mirrors:
        # susecloud registry not added, always place it first
        registry_mirrors.insert(0, private_registry_url)
        modified_by_us = True

    if suse_registry_url not in registry_mirrors:
        # the suse registry search is provided by the libcontainers-common
        # package. For the case when we cannot find any suse registry we
        # append it after the susecloud registry
        #
        # Note: docker search is disabled for Docker server side !
        private_registry_index = registry_mirrors.index(
            private_registry_url
        )
        registry_mirrors.insert(
            private_registry_index + 1, suse_registry_url
        )
        modified_by_us = True

    # write registry setup if modified by us
    if modified_by_us:
        docker_cfg_json['registry-mirrors'] = registry_mirrors
        return write_registries_conf(
            docker_cfg_json, DOCKER_CONFIG_PATH, 'docker'
        )


# ----------------------------------------------------------------------------
def __set_registry_fqdn_suma(private_registry_fqdn):
    """Set the registry FQDN for SUMa SUMA_REGISTRY_CONF_PATH file."""
    suma_yaml_content = {}
    os.makedirs(os.path.dirname(SUMA_REGISTRY_CONF_PATH), exist_ok=True)
    suma_yaml_content, failed_to_read = get_suma_registry_content()
    if failed_to_read:
        return False

    if suma_yaml_content.get('registry', {}) != private_registry_fqdn:
        # registry value does not match the update server registry FQDN
        # or there is no registry entry
        # set the content
        suma_yaml_content.update({'registry': private_registry_fqdn})
        return __write_suma_conf(suma_yaml_content)

    # the registry value inside the file has the update server registry FQDN
    return True


# ----------------------------------------------------------------------------
def __write_suma_conf(updated_content):
    """Update the SUMA SUMA_REGISTRY_CONF_PATH file with the new content."""
    try:
        with open(SUMA_REGISTRY_CONF_PATH, 'w') as suma_config:
            yaml.dump(updated_content, suma_config, default_flow_style=False)
        logging.info('%s updated' % SUMA_REGISTRY_CONF_PATH)
        return True
    except IOError as error:
        logging.info(str(error))
        action = 'open'
    except yaml.YAMLError:
        action = 'parse'

    message = 'Could not %s %s' % (action, SUMA_REGISTRY_CONF_PATH)
    logging.info(message)


# ----------------------------------------------------------------------------
def __matches_susecloud(elements):
    for element in elements:
        if 'registry-' in element and 'susecloud.net' in element:
            return element
    return ''
