/*
 *	This file implements a netcf interface
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *	See the GNU Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with this library; if not, see
 *		<http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <limits.h>

#include <netcontrol/netcf.h>
#include <sutils.h>
#include <nutils.h>
#include <tutils.h>
#include <handle.h>
#include <syntax.h>
#include <logging.h>
#include <unistd.h>
#include <pthread.h>

/*
 * With factor > 0, adjust expire to last refresh time * factor
 * (e.g. if refresh took 0.5s, we omit refresh for 0,5s * factor)
 . except it has been forced.
 * When factor is 0, the fixed expire time (bellow) is used.
 */
#define NETCONTROL_NETCF_REFRESH_FACTOR		2
#define NETCONTROL_NETCF_REFRESH_EXPIRE		{ 1, 0 }

/*
 * Define to enable basic call profiling debug log
 */
#define NETCONTROL_NETCF_PROFILING		1

/*
 * Define to enable ref counter debug log
 */
#undef NETCONTROL_NETCF_DEBUG_REF


/*
 * Define to enable xml dumps to debug log
 */
#undef NETCONTROL_NETCF_DEBUG_XML


/* === data structures === */
struct netcf_ifinfo {
	unsigned int		factor;
	struct timeval		expire;
	struct timeval		modified;

	nc_handle_t *		config;
	nc_handle_t *		system;
};

struct netcf {
	unsigned int		users;

	int			error;
	char *			errstr;

	struct netcf_ifinfo 	ifinfo;
	nc_syntax_t *		syntax;
};

struct netcf_if {
	struct netcf *		netcf;
	char *			name;
	nc_stringbuf_t		mac;
};

/* === private prototypes === */
static int			nc_ncf_list_interfaces(struct netcf *ncf, unsigned int flags,
				                      char **names, unsigned int nnames);
static nc_interface_t *		nc_ncf_lookup(struct netcf *ncf, nc_handle_t *nh, const char *name);
static int			nc_ncf_refresh(struct netcf *ncf, int force);

static struct netcf *		nc_ncf_acquire(struct netcf *ncf);
static unsigned int		nc_ncf_release(struct netcf *ncf);

static struct netcf_if *	nc_ncf_if_new(struct netcf *ncf, nc_interface_t *ifp);
static char *			nc_ncf_if_xml(struct netcf *ncf, nc_handle_t *nh,
						nc_interface_t *ifp);
static void			nc_ncf_if_free(struct netcf_if *nif);

static const char *		nc_ncf_strerror(int error);
static void			nc_ncf_error_clear (struct netcf *ncf, nc_handle_t *nh);
static void			nc_ncf_error_report(struct netcf *ncf, nc_handle_t *nh,
							int error, const char *fmt, ...);
static const char *		nc_ncf_flags2str(unsigned int flags);


#ifdef	NETCONTROL_NETCF_PROFILING
#define nc_debug_netcf_started(fmt, args...)				\
	struct timeval started;						\
	nc_get_time(&started);						\
	nc_debug_netcf(fmt, ##args);
#define nc_debug_netcf_finished(fmt, args...)	 			\
	do {								\
		struct timeval elapsed;					\
		nc_get_time_elapsed(&started, &elapsed);		\
		nc_debug_netcf(fmt " [%ld.%06lds]", ##args,		\
				elapsed.tv_sec, elapsed.tv_usec);	\
	} while(0)
#else
#define nc_debug_netcf_started(fmt, args...)		{}
#define nc_debug_netcf_finished(started, fmt, args...)	{}
#endif

static inline void
nc_ncf_init_expire(struct netcf_ifinfo *nii)
{
	static const struct timeval expire = NETCONTROL_NETCF_REFRESH_EXPIRE;

	nii->factor = NETCONTROL_NETCF_REFRESH_FACTOR;
	nii->expire = expire;
}

/* === public functions === */
int
ncf_init(struct netcf **netcf, const char *root)
{
	nc_service_t *service = NULL;
	struct netcf *ncf = NULL;

	nc_debug_netcf_started("netcf=%p, root='%s'",
			(void *)netcf, root ? root : "");

	if (!netcf)
		goto failure;
	*netcf = NULL;

	if (getuid() != 0) {
		/*
		** - All ifcfg files are readable by root only (passwd)
		** - /proc/vlan/config is readable by root only
		** - Every modifying action need root permissions
		*/
		nc_error("UID root permissions needed to read config and some states");
		goto failure;
	}

	ncf = calloc(1, sizeof(*ncf));
	if (!ncf)
		goto failure;
	ncf->users = 1;

	service = nc_service_detect(root);
	if (!service)
		goto failure;

	nc_ncf_init_expire(&ncf->ifinfo);

	ncf->ifinfo.system = nc_indirect_open("/system", root, service);
	if (!ncf->ifinfo.system)
		goto failure;

	ncf->ifinfo.config = nc_indirect_open("/config", root, service);
	if(!ncf->ifinfo.config)
		goto failure;

	ncf->syntax = nc_syntax_new("netcf", NULL);
	if(!ncf->syntax)
		goto failure;

	/*
	** we ignore NETCF_DATADIR & NETCF_DEBUG env vars.
	*/
	*netcf = ncf;
	nc_service_free(service);
	nc_debug_netcf_finished("netcf=>%p", (void *)ncf);
	return 0;

failure:
	nc_service_free(service);
	ncf_close(ncf);
	nc_debug_netcf_finished("netcf=>%p", (void *)ncf);
	return -1;
}


int
ncf_close(struct netcf *ncf)
{
	if (ncf) {
		if (ncf->users > 1) {
			nc_ncf_error_report(ncf, NULL, NETCF_EINUSE,
					"netcf instance %p is referenced in %u interfaces",
					(void *)ncf, ncf->users - 1);
			return -1;
		}
		nc_ncf_release(ncf);
	}
	return 0;
}


int
ncf_error(struct netcf *ncf, const char **errmsg, const char **details)
{
	int error = 0;

	nc_debug_netcf_started("ncf=%p, errmsg=%p, details=%p",
			(void *)ncf, errmsg, details);
	if (ncf) {

		error = ncf->error;
		if (errmsg)
			*errmsg = nc_ncf_strerror(error);
		if (details)
			*details = ncf->errstr;
		ncf->error = 0;
	}
	nc_debug_netcf_finished("ncf=%p, error=>%d",
			(void *)ncf, error);
	return error;
}


int
ncf_num_of_interfaces(struct netcf *ncf, unsigned int flags)
{
	int count;

	nc_debug_netcf_started("ncf=%p, flags=0x%x (%s)",
			(void *)ncf, flags, nc_ncf_flags2str(flags));

	nc_ncf_error_clear(ncf, ncf ? ncf->ifinfo.config : NULL);
	count = nc_ncf_list_interfaces(ncf, flags, NULL, 0);

	nc_debug_netcf_finished("ncf=%p, flags=0x%x (%s), count=>%d",
			(void *)ncf, flags, nc_ncf_flags2str(flags), count);
	return count;
}


int
ncf_list_interfaces(struct netcf *ncf, int maxnames, char **names, unsigned int flags)
{
	int count;

	nc_debug_netcf_started("ncf=%p, maxnames=%d, flags=0x%x (%s)",
			(void *)ncf, maxnames, flags, nc_ncf_flags2str(flags));
	if(names && maxnames > 0) {
		memset(names, 0, maxnames * sizeof(*names));
	} else {
		maxnames = 0;
		names = NULL;
	}

	nc_ncf_error_clear(ncf, ncf ? ncf->ifinfo.config : NULL);
	count = nc_ncf_list_interfaces(ncf, flags, names, maxnames);
	nc_debug_netcf_finished("ncf=%p, flags=0x%x (%s), count=>%d",
			(void *)ncf, flags, nc_ncf_flags2str(flags), count);
	return count;
}


struct netcf_if *
ncf_lookup_by_name(struct netcf *ncf, const char *name)
{
	struct netcf_if *nif = NULL;
	nc_interface_t *ifp;

	nc_debug_netcf_started("ncf=%p, name=%s", (void *)ncf, name);

	nc_ncf_error_clear(ncf, ncf ? ncf->ifinfo.config : NULL);
	if (!ncf || !(ifp = nc_ncf_lookup(ncf, ncf->ifinfo.config, name))) {
		nc_ncf_error_report(ncf, ncf ? ncf->ifinfo.config : NULL, NETCF_ENOENT,
			"Unable to find configuration for interface '%s'", name);
		goto done;
	}

	if(nc_syntax_supported_inteface(ncf->syntax, ncf->ifinfo.config, ifp)) {
		nc_ncf_error_report(ncf, ncf ? ncf->ifinfo.config : NULL, NETCF_ENOENT,
			"Interface '%s' (type) is not supported", name);
		goto done;
	}

	if (!(nif = nc_ncf_if_new(ncf, ifp))) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_ENOMEM,
			"Unable to allocate netcf interface '%s'", name);
		goto done;
	}

done:
	nc_debug_netcf_finished("ncf=%p, name=%s, nif=>%p",
			(void *)ncf, name, nif);
	return nif;
}

int
ncf_lookup_by_mac_string(struct netcf *ncf, const char *mac, int maxifaces,
			 struct netcf_if **ifaces)
{
	nc_interface_t *cfg, *sys;
	unsigned int pos;
	int count = 0, i;

	nc_debug_netcf_started("ncf=%p, mac=%s, maxifaces=%d, ifaces=%p",
			(void *)ncf, mac, maxifaces, ifaces);

	nc_ncf_error_clear(ncf, ncf ? ncf->ifinfo.config : NULL);
	if(!ncf || nc_string_empty(mac)) {
		nc_ncf_error_report(ncf, NULL, NETCF_EOTHER,
			"Invalid mac adress argument '%s'", mac);
		goto failure;
	}

	if(ifaces && maxifaces > 0) {
		memset(ifaces, 0, maxifaces * sizeof(*ifaces));
	} else {
		ifaces = NULL;
		maxifaces = 0;
	}

	if (nc_ncf_refresh(ncf, 0) < 0)
		goto failure;

	for(cfg = nc_interface_first(ncf->ifinfo.config, &pos); cfg;
	    cfg = nc_interface_next(ncf->ifinfo.config, &pos)) {
		nc_stringbuf_t nsb = NC_STRINGBUF_INIT;

		if(cfg->invalid)
			continue;

		if(nc_syntax_supported_inteface(ncf->syntax, ncf->ifinfo.config, cfg) != 0) {
			nc_info("skipping syntax unsupported interface %s", cfg->name);
			continue;
		}

		if(!nc_hw_address_format(&cfg->hwaddr, &nsb)) {
			sys = nc_interface_by_name(ncf->ifinfo.system, cfg->name);
			if(!sys || !nc_hw_address_format(&sys->hwaddr, &nsb))
				continue;
		}

		if (strcasecmp(mac, nsb.string)) {
			nc_stringbuf_destroy(&nsb);
			continue;
		}

		if(count < maxifaces && ifaces) {
			ifaces[count] = nc_ncf_if_new(ncf, cfg);
			if (ifaces[count]) {
				nc_stringbuf_move(&ifaces[count]->mac, &nsb);
			} else {
				nc_stringbuf_destroy(&nsb);
				nc_ncf_error_report(ncf, NULL, NETCF_ENOMEM,
						"Unable to create netcf interface for '%s'",
						cfg->name);
				goto failure;
			}
		} else {
			nc_stringbuf_destroy(&nsb);
		}
		count++;
	}
	nc_debug_netcf_finished("ncf=%p, mac=%s, maxifaces=%d, ifaces=%p, count=>%d",
			(void *)ncf, mac, maxifaces, ifaces, count);
	return count;

failure:
	nc_debug_netcf_finished("ncf=%p, mac=%s, maxifaces=%d, ifaces=%p, count=>%d (%d)",
			(void *)ncf, mac, maxifaces, ifaces, -1, count);
	if (ifaces && count > 0) {
		for (i = 0; i < maxifaces; ++i) {
			if (ifaces[i]) {
				nc_ncf_if_free(ifaces[i]);
				ifaces[i] = NULL;
			}
		}
	}
	return -1;
}

#ifdef NETCONTROL_NETCF_DEBUG_XML
static inline void
nc_ncf_parsed_xml(const char *str, void *arg)
{
	const char *pfx = (const char *)arg;
	nc_debug_netcf("%s%s", pfx ? pfx : "", str);
}
#endif

struct netcf_if *
ncf_define(struct netcf *ncf, const char *xml_data)
{
	FILE           *memstream;
	nc_interface_t *cfg, *ifp;
	xml_node_t     *xml_root;
	xml_node_t     *ifnode;
	struct netcf_if *nif = NULL;

#ifdef NETCONTROL_NETCF_DEBUG_XML
	nc_debug_netcf_started("ncf=%p, xml=%s", (void *)ncf, xml_data);
#else
	nc_debug_netcf_started("ncf=%p, xml=[...]", (void *)ncf);
#endif
	if (!ncf || nc_string_empty(xml_data))
		goto failure;

	nc_ncf_error_clear(ncf, ncf->ifinfo.config);

	memstream = fmemopen((char *) xml_data, strlen(xml_data), "r");
	if (!memstream) {
		nc_ncf_error_report(ncf, NULL, NETCF_ENOMEM,
			"Unable to open memstream buffer: %m");
		goto failure;
	}

	xml_root = xml_node_scan(memstream);
	fclose(memstream);
	if (!xml_root) {
		nc_ncf_error_report(ncf, NULL, NETCF_EXMLPARSER,
			"Unable to parse xml data");
		goto failure;
	}

	if (nc_string_eq(xml_root->name, "interface")) {
		ifnode = xml_root;
	} else {
		ifnode = xml_node_get_child(xml_root, "interface");
	}
	if (!ifnode) {
		nc_ncf_error_report(ncf, NULL, NETCF_EXMLINVALID,
			"Unable to find interface node in xml data");
		xml_node_free(xml_root);
		goto failure;
	}

#ifdef NETCONTROL_NETCF_DEBUG_XML
	nc_debug_netcf("parsed xml config node dump:");
	xml_node_print_fn(ifnode, nc_ncf_parsed_xml, NULL);
#endif

	if (nc_ncf_refresh(ncf, 0) < 0) {
		xml_node_free(xml_root);
		goto failure;
	}

	cfg = nc_syntax_xml_to_interface(ncf->syntax, ncf->ifinfo.config, ifnode);
	xml_node_free(xml_root);
	if (!cfg) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_EXMLINVALID,
			"Cannot to interpret xml interface configuration");
		goto failure;
	}

	if (nc_interfaces_configure(ncf->ifinfo.config, cfg) < 0) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_EOTHER,
			"Configuration of interface '%s' failed", cfg->name);

		nc_ncf_refresh(ncf, 1);
		nc_interface_free(cfg);
		goto failure;
	}

	if (nc_ncf_refresh(ncf, 1) < 0) {
		nc_interface_free(cfg);
		goto failure;
	}

	ifp = nc_interface_by_name(ncf->ifinfo.config, cfg->name);
	if (!ifp) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_EOTHER,
			"Unable to re-read interface '%s' configuration", cfg->name);
		nc_interface_free(cfg);
		goto failure;
	}
	nc_interface_free(cfg);

#if 1	/* regression test */
	ifnode = nc_syntax_xml_from_interface(ncf->syntax, ncf->ifinfo.config, ifp);
	if (!ifnode) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_EOTHER,
			"Unable to re-parse interface '%s' confguration", ifp->name);
		goto failure;
	}
	xml_node_free(ifnode);
#endif

	nif = nc_ncf_if_new(ncf, ifp);
	if (!nif) {
		nc_ncf_error_report(ncf, ncf->ifinfo.config, NETCF_ENOMEM,
			"Failed to construct netcf interface '%s'", ifp->name);
		goto failure;
	}

	nc_debug_netcf_finished("ncf=%p, nif=>%p (name=%s)", (void *)ncf, nif, nif->name);
	return nif;

failure:
	nc_debug_netcf_finished("ncf=%p, nif=>%p", (void *)ncf, nif);
	return NULL;
}


void
ncf_if_free(struct netcf_if *nif)
{
	const char *name = nif ? nif->name : NULL;
	nc_debug_netcf("nif=%p (%s)", nif, name);
	nc_ncf_if_free(nif);
}


const char *
ncf_if_name(struct netcf_if *nif)
{
	const char *name = nif ? nif->name : NULL;
	nc_debug_netcf("nif=%p, name=>%s", nif, name);
	return name;
}


const char *
ncf_if_mac_string(struct netcf_if *nif)
{
	const char *name = nif ? nif->name : NULL;
	const char *mac = NULL;
	nc_interface_t *ifp;

	if(nif && nif->netcf) {
		if (nif->mac.string)
			mac = nif->mac.string;
		else
		if ((ifp = nc_interface_by_name(nif->netcf->ifinfo.config, nif->name)) &&
		    nc_hw_address_format(&ifp->hwaddr, &nif->mac))
			mac = nif->mac.string;
		else
		if ((ifp = nc_interface_by_name(nif->netcf->ifinfo.system, nif->name)) &&
		    nc_hw_address_format(&ifp->hwaddr, &nif->mac))
			mac = nif->mac.string;
	}
	nc_debug_netcf("nif=%p (%s) mac=>%s", nif, name, mac);
	return mac;
}

char *
ncf_if_xml_desc(struct netcf_if *nif)
{
	nc_interface_t *ifp;
	nc_handle_t *nh;
	char *xml = NULL;

	nc_debug_netcf_started("nif=%p (%s)", (void *)nif, nif ? nif->name : NULL);
	if(!nif || !nif->netcf)
		goto report;

	nh = nif->netcf->ifinfo.config;
	nc_ncf_error_clear(nif->netcf, nh);
	if (!(ifp = nc_ncf_lookup(nif->netcf, nh, nif->name))) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_ENOENT,
			"Unable to find configuration for interface '%s'", nif->name);
		goto report;
	}

	xml = nc_ncf_if_xml(nif->netcf, nh, ifp);
#ifdef NETCONTROL_NETCF_DEBUG_XML
	nc_debug_netcf("desc xml for '%s':\n%s", nif->name, xml);
#endif
	nc_ncf_error_clear(nif->netcf, nh);

report:
	nc_debug_netcf_finished("nif=%p (%s), xml=>%p",
			(void *)nif, nif ? nif->name : NULL, xml);
	return xml;
}

char *
ncf_if_xml_state(struct netcf_if *nif)
{
	nc_interface_t *ifp;
	nc_handle_t *nh;
	char *xml = NULL;

	nc_debug_netcf_started("nif=%p (%s)", (void *)nif, nif ? nif->name : NULL);
	if(!nif || !nif->netcf)
		goto report;

	nh = nif->netcf->ifinfo.system;
	nc_ncf_error_clear(nif->netcf, nh);
	if (!(ifp = nc_ncf_lookup(nif->netcf, nh, nif->name))) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_ENOENT,
			"Unable to find system state for interface '%s'", nif->name);
		goto report;
	}

	xml = nc_ncf_if_xml(nif->netcf, nh, ifp);
#ifdef NETCONTROL_NETCF_DEBUG_XML
	nc_debug_netcf("state xml for %s:\n%s", nif->name, xml);
#endif
	nc_ncf_error_clear(nif->netcf, nh);

report:
	nc_debug_netcf_finished("nif=%p (%s), xml=>%p",
		(void *)nif, nif ? nif->name : NULL, xml);
	return xml;
}


int
ncf_if_up(struct netcf_if *nif)
{
	nc_interface_t *ifp;
	nc_handle_t *nh;
	int ret = -1;

	nc_debug_netcf_started("nif=%p (%s)", (void *)nif, nif ? nif->name : NULL);
	if (!nif || !nif->netcf)
		goto report;

	nh = nif->netcf->ifinfo.config;
	nc_ncf_error_clear(nif->netcf, nh);
	if (!(ifp = nc_ncf_lookup(nif->netcf, nh, nif->name))) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_ENOENT,
			"Unable to find configuration for interface '%s'", nif->name);
		goto report;
	}

	if (nc_interface_start(nh, ifp)) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_EEXEC,
			"Unable to start interface '%s'", nif->name);
		goto report;
	}

	ret = 0;
	nc_ncf_error_clear(nif->netcf, nh);

report:
	nc_debug_netcf_finished("nif=%p (%s), ret=>%d", (void *)nif, nif ? nif->name : NULL, ret);
	return ret;
}

int
ncf_if_down(struct netcf_if *nif)
{
	nc_interface_t *ifp;
	nc_handle_t *nh;
	int ret = -1;

	nc_debug_netcf_started("nif=%p (%s)", (void *)nif, nif ? nif->name : NULL);
	if (!nif || !nif->netcf)
		goto report;

	nh = nif->netcf->ifinfo.config;
	nc_ncf_error_clear(nif->netcf, nh);
	if (!(ifp = nc_ncf_lookup(nif->netcf, nh, nif->name))) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_ENOENT,
			"Unable to find system state for interface '%s'", nif->name);
		goto report;
	}

	if(nc_interface_stop(nh, ifp)) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_EEXEC,
			"Unable to stop interface '%s'", nif->name);
		goto report;
	}

	ret = 0;
	nc_ncf_error_clear(nif->netcf, nh);

report:
	nc_debug_netcf_finished("nif=%p (%s), ret=>%d", (void *)nif, nif ? nif->name : NULL, ret);
	return ret;
}

int
ncf_if_status(struct netcf_if *nif, unsigned int *pflags)
{
	unsigned int flags = 0;
	nc_interface_t *ifp;
	nc_handle_t *nh;
	int ret = -1;

	nc_debug_netcf_started("nif=%p (%s) flags=%p",
			(void *)nif, nif ? nif->name : NULL,
			(void *)pflags);
	if (!nif || !nif->netcf || !pflags)
		goto report;

	nh = nif->netcf->ifinfo.system;
	nc_ncf_error_clear(nif->netcf, nh);
	if ((ifp = nc_ncf_lookup(nif->netcf, nh, nif->name)))
		nc_interface_status(nh, ifp);
	if (nc_interface_link_is_up(ifp)) {
		flags = NETCF_IFACE_ACTIVE;
	} else {
		flags = NETCF_IFACE_INACTIVE;
	}
	*pflags = flags;

	ret = 0;
	nc_ncf_error_clear(nif->netcf, nh);

report:
	nc_debug_netcf_finished("nif=%p (%s), ret=>%d, flags=>%p[0x%x (%s)]",
			(void *)nif, nif ? nif->name : NULL, ret,
			(void *)pflags, flags, nc_ncf_flags2str(flags));
	return ret;
}

int
ncf_if_undefine(struct netcf_if *nif)
{
	nc_interface_t *ifp;
	nc_handle_t *nh;
	int ret = -1;

	nc_debug_netcf_started("nif=%p (%s)", (void *)nif, nif ? nif->name : NULL);
	if (!nif || !nif->netcf)
		goto report;

	nh = nif->netcf->ifinfo.config;
	nc_ncf_error_clear(nif->netcf, nh);
	if (!(ifp = nc_ncf_lookup(nif->netcf, nh, nif->name))) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_ENOENT,
			"Unable to find configuration for interface '%s'", nif->name);
		goto report;
	}

	if (nc_interfaces_delete(nh, ifp)) {
		nc_ncf_error_report(nif->netcf, nh, NETCF_EEXEC,
			"Unable to delete configuration for interface '%s'", nif->name);
		goto report;
	}

	ret = 0;
	nc_ncf_error_clear(nif->netcf, nh);
report:
	nc_debug_netcf_finished("nif=%p (%s), ret=>%d", (void *)nif, nif ? nif->name : NULL, ret);
	return ret;
}


/* === private helpers === */
static const char *
nc_ncf_strerror(int error)
{
	switch (error) {
	case NETCF_NOERROR:
		return "no error";
	case NETCF_EINTERNAL:
		return "internal error";
	case NETCF_EOTHER:
		return "other error";
	case NETCF_ENOMEM:
		return "allocation failed";
	case NETCF_EXMLPARSER:
		return "XML parser choked";
	case NETCF_EXMLINVALID:
		return "XML invalid in some form";
	case NETCF_ENOENT:
		return "Required entry in a tree is missing";
	case NETCF_EEXEC:
		return "external program execution failed";
	case NETCF_EINUSE:
		return "cannot close netcf, instances still in use";
	case NETCF_EXSLTFAILED:
		return "XSLT transformation failed";
	case NETCF_EFILE:
		return "some file access failed";
	case NETCF_EIOCTL:
		return "an ioctl call failed";
	case NETCF_ENETLINK:
		return "netlink error";
	default:
		return "error code not handled";
	}
}

static void
nc_ncf_error_clear(struct netcf *ncf, nc_handle_t *nh)
{
	if(ncf) {
		ncf->error = 0;
		nc_string_free(&ncf->errstr);
		if(nh) {
			nc_error_detail_clear(nh);
		}
	}
}

static void
nc_ncf_error_report(struct netcf *ncf, nc_handle_t *nh,
			int error, const char *fmt, ...)
{
	if(ncf) {
		/* Hmm... no details for success */
		nc_string_free(&ncf->errstr);
		if((ncf->error = error) == NETCF_NOERROR)
			return;

		/* When handle provided and has details, use them */
		if(nh) {
			nc_error_detail_move(nh, &ncf->errstr);
			if(ncf->errstr)
				return;
		}

		/* Otherwise, format the provided message */
		if(fmt) {
			nc_stringbuf_t  buf = NC_STRINGBUF_INIT;
			va_list         ap;     

			va_start(ap, fmt);
			nc_stringbuf_vprintf(&buf, fmt, ap);
			va_end(ap);

			ncf->errstr = buf.string;
		}
	}
}

static const char *
nc_ncf_flags2str(unsigned int flags)
{
	switch (flags) {
		case NETCF_IFACE_INACTIVE|NETCF_IFACE_ACTIVE:
			return "inactive|active";
		case NETCF_IFACE_INACTIVE:
			return "inactive";
		case NETCF_IFACE_ACTIVE:
			return "active";
		default:
			return "invalid";
	}
}

static unsigned int
ncf_interface_names_pos(char **names, unsigned int nnames,
			unsigned int first, const char *name)
{
	unsigned int pos;

	for (pos = first; pos < nnames; ++pos) {
		if (nc_string_eq(names[pos], name))
			return pos;
	}
	return -1U;
}

static int
ncf_interface_names_add(char **names, unsigned int nnames,
			unsigned int pos, const char *name)
{
	if (pos >= nnames || !names || !name)
		return -1;

	if (ncf_interface_names_pos(names, nnames, 0, name) != -1U)
		return 1;

	return nc_string_dup(&names[pos], name);
}

static int /* bool */
nc_ncf_interace_usable(nc_syntax_t *syntax, nc_handle_t *nh, nc_interface_t *ifp)
{
	if (!ifp || ifp->invalid)
		return 0;

	if (ifp->parent)
		return 0;

	if (nc_syntax_supported_inteface(syntax, nh, ifp) != 0)
		return 0;

	return 1;
}

static int
nc_ncf_list_interfaces(struct netcf *ncf, unsigned int flags, char **names, unsigned int nnames)
{
	/*
	 * This driver supports to configure "system interfaces".
	 * The flags request a join of _configured_ interfaces in
	 * an inactive (unstarted) and/or active (started) state.
	 * That is, interfaces without an existing (ifcfg) config
	 * like virbrX, vnetX, ... set up in other ways (manually,
	 * or by libvirt itself) are considered.
	 * The config handle provides interface configs, system
	 * handle (all, incl. unrelated) existing interfaces in
	 * the kernel.
	 *
	 *   config     :  a b c -
	 *   system     :  - b c d
	 *   --------------------
	 *   "running"  :  - - + +
	 *   --------------------
	 *   1: inactive:  + + - -
	 *   2:  active :  - - + -
	 *   3:    both :  + + + -
	 */
	unsigned int ifaces = 0, pos;
	nc_interface_t *cfg, *sys;

	if (nc_ncf_refresh(ncf, 0) < 0)
		return -1;

	/* check all interfaces with existing config */
	for (cfg = nc_interface_first(ncf->ifinfo.config, &pos); cfg;
	     cfg = nc_interface_next(ncf->ifinfo.config, &pos)) {
		int active = 0;
#if 0
		if ((sys = nc_interface_by_name(ncf->system.handle, cfg->name))) {
			/* without type (specific elements) in the config, we may
			 * not know the config type, e.g. non-vitual interfaces
			 * like ethernet may not contain any such hint and we'd
			 * not expose them as type=ethernet (netcf requires one).
			 * we assign type of the existing system interface then.
			 */
			if (cfg->type == NC_IFTYPE_UNKNOWN)
				cfg->type = sys->type;
		}
#endif
		if (!nc_ncf_interace_usable(ncf->syntax, ncf->ifinfo.config, cfg))
			continue;

		if ((sys = nc_interface_by_name(ncf->ifinfo.system, cfg->name)))
			nc_interface_status(ncf->ifinfo.system, sys);
		active = nc_interface_link_is_up(sys);

		if (((flags & NETCF_IFACE_ACTIVE) && active) ||
		    ((flags & NETCF_IFACE_INACTIVE) && !active)) {
			if (!nnames || /* just count ... or also collect: */
			    !ncf_interface_names_add(names, nnames, ifaces, cfg->name))
				ifaces++;
		}
	}

	return ifaces;
}

static struct netcf *
nc_ncf_acquire(struct netcf *ncf)
{
	if (ncf) {
		assert(ncf->users);
		if (!ncf->users || ncf->users == UINT_MAX)
			return NULL;

		ncf->users++;
#ifdef NETCONTROL_NETCF_DEBUG_REF
		nc_debug_netcf("ncf=%p aquired (refs %u)",
				(void *)ncf, ncf->users);
#endif
	}
	return ncf;
}

static unsigned int
nc_ncf_release(struct netcf *ncf)
{
	if (ncf) {
		assert(ncf->users);

		ncf->users--;
		if (ncf->users == 0) {
			if (ncf->ifinfo.system)
				nc_close(ncf->ifinfo.system);
			if (ncf->ifinfo.config)
				nc_close(ncf->ifinfo.config);
			if (ncf->syntax)
				nc_syntax_free(ncf->syntax);
			nc_string_free(&ncf->errstr);

			memset(ncf, 0, sizeof(*ncf));
			free(ncf);
#ifdef NETCONTROL_NETCF_DEBUG_REF
			nc_debug_netcf("ncf=%p destroyed", (void *)ncf);
#endif
			return 0;
		} else {
#ifdef NETCONTROL_NETCF_DEBUG_REF
			nc_debug_netcf("ncf=%p released (refs %u)",
					(void *)ncf, ncf->users);
#endif
			return ncf->users;
		}
	}
	return 0;
}

static void
nc_ncf_if_free(struct netcf_if *nif)
{
	if (nif) {
		nc_ncf_release(nif->netcf);
		nc_string_free(&nif->name);
		nc_stringbuf_destroy(&nif->mac);
		memset(nif, 0, sizeof(*nif));
		free(nif);
	}
}

static struct netcf_if *
nc_ncf_if_new(struct netcf *ncf, nc_interface_t *ifp)
{
	struct netcf_if *nif;

	if (!ifp || nc_string_empty(ifp->name))
		return NULL;

	if (!nc_ncf_acquire(ncf))
		return NULL;

	nif = calloc(1, sizeof(*nif));
	if(nif) {
		nif->netcf = ncf; /* acquired above */
		nc_string_dup(&nif->name, ifp->name);
		nc_stringbuf_init(&nif->mac);
	} else {
		nc_ncf_release(ncf);
	}
	return nif;
}

static char *
nc_ncf_if_xml(struct netcf *ncf, nc_handle_t *nh, nc_interface_t *ifp)
{
	FILE   *mem_stream = NULL;
	char   *out_string = NULL;
	size_t  out_size = 0;
	xml_node_t *xml = NULL;

	xml = nc_syntax_xml_from_interface(ncf->syntax, nh, ifp);
	if(!xml) {
		nc_ncf_error_report(ncf, nh, NETCF_EOTHER,
			"Failed to generate xml representation of interface '%s'",
			(ifp && ifp->name ? ifp->name : NULL));
		goto error;
	}

	mem_stream = open_memstream(&out_string, &out_size);
	if(!mem_stream) {
		nc_ncf_error_report(ncf, NULL, NETCF_EOTHER,
			"Failed to open memory stream for xml representation of interface '%s'",
			(ifp && ifp->name ? ifp->name : NULL));
		goto error;
	}

	if (xml_node_print(xml, mem_stream) < 0) {
		nc_ncf_error_report(ncf, NULL, NETCF_EOTHER,
			"Failed to write interface '%s' xml representation to memory stream",
			(ifp && ifp->name ? ifp->name : NULL));
		goto error;
	}

	fclose(mem_stream);
	xml_node_free(xml);

	return out_string;

error:
	if(mem_stream)
		fclose(mem_stream);
	if(xml)
		xml_node_free(xml);
	return NULL;
}

static nc_interface_t *
nc_ncf_lookup(struct netcf *ncf, nc_handle_t *nh, const char *name)
{
	if (!nh || !name || nc_ncf_refresh(ncf, 0) < 0)
		return NULL;

	return nc_interface_by_name(nh, name);
}

static inline int /* bool */
ncf_nii_is_expired(struct netcf_ifinfo *nii, struct timeval *now, struct timeval *age)
{
	if (nc_get_time_delta(&nii->modified, now, age))
		return 1;

	return timercmp(age, &nii->expire, >);
}

static inline void
ncf_nii_set_expire(struct netcf_ifinfo *nii, const struct timeval *elapsed)
{
	if (!nii || !nii->factor || !elapsed)
		return;

	nii->expire.tv_sec = elapsed->tv_sec * nii->factor;
	nii->expire.tv_usec = elapsed->tv_usec * nii->factor;
	if (nii->expire.tv_usec >= 1000000L) {
		nii->expire.tv_sec += nii->expire.tv_usec / 1000000L;
		nii->expire.tv_usec = nii->expire.tv_usec % 1000000L;
	}
}

static inline void
ncf_nii_set_modified(struct netcf_ifinfo *nii, const struct timeval *modified)
{
	if (!nii || !modified)
		return;

	nii->modified = *modified;
}

static int
nc_ncf_refresh(struct netcf *ncf, int force)
{
	struct timeval beg, end, age;
	pthread_t tid = pthread_self();
	int ret = 0;

	if (ncf) {
		nc_get_time(&beg);
		if (!force && !ncf_nii_is_expired(&ncf->ifinfo, &beg, &age)) {
#if 0			/* this causes a lot of messages/sec */
			nc_debug_netcf("ncf=%p[%lu], age %ld.%06lds -- omitted",
					(void *)ncf, (unsigned long int)tid,
					age.tv_sec, age.tv_usec);
#endif
			return 1;
		} else {
			nc_debug_netcf("ncf=%p[%lu], age %ld.%06lds -- %sstart",
					(void *)ncf, (unsigned long int)tid,
					age.tv_sec, age.tv_usec,
					force ? "forced " : "");
		}

		if (nc_interfaces_refresh(ncf->ifinfo.config) < 0) {
			nc_error("config interface tree refresh failed");
			ret = -1;
		}
		if (nc_interfaces_refresh(ncf->ifinfo.system) < 0) {
			nc_error("system interface tree refresh failed");
			ret = -1;
		}

		nc_get_time(&end);
		nc_get_time_delta(&beg, &end, &age);
		ncf_nii_set_expire(&ncf->ifinfo, &age);
		ncf_nii_set_modified(&ncf->ifinfo, &end);
		nc_debug_netcf("ncf=%p[%lu], performed in %ld.%lds",
			       (void *)ncf, (unsigned long int)tid,
			       age.tv_sec, age.tv_usec);

		return ret;
	}
	return -1;
}

