/*
 * suse-blinux
 * driver for all Papenmeier devices
 * Author: marco Skambraks <marco@suse.de>
 * SuSE Linux AG, Nuernberg
 * http://www.blinux.suse.de
 *
 * special thanks to the brltty team 
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 */

/* 
 * This Driver was written as a project in the
 * (Technical High School, Department for electrical engineering, Vienna, Austria)
 * by:
 *   Tibor Becker
 *   Michael Burger
 *   Herbert Gruber
 *   Heimo Schn
 *   Teacher: August Hrandl <hoerandl@elina.htlw1.ac.at>
 *
 * modified/rebuild by Marco Skambraks <marco@suse.de>
 * SuSE Linux AG, Nuernberg
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <string.h>

#include "brlconf.h"
/*#include "../brl.h"*/
#include "../brl_driver.h"

#include "../sbllog.h"

#define READ_WAIT 10            /* select timeout for incoming data in ms */
#define CMD_ERR 0

char DefDev[] = BRLDEV;         /* default braille device */

int brl_fd = -1;                /* file descriptor for Braille display */

struct termios oldtio;          /* old terminal settings */

static unsigned char *prevline;

static unsigned char *prev;

static unsigned int brl_dbg = 0;

void allocmem ();

void init_table ();

/* special table for papenmeier */
#define B1 1
#define B2 2
#define B3 4
#define B4 8
#define B5 16
#define B6 32
#define B7 64
#define B8 128

unsigned char pm_ones[11] = { B1 + B5 + B4, B2, B2 + B5,
  B2 + B1, B2 + B1 + B4, B2 + B4,
  B2 + B5 + B1, B2 + B5 + B4 + B1, B2 + B5 + B4,
  B5 + B1, B1 + B2 + B4 + B5
};

char pm_tens[11] = { B8 + B6 + B3, B7, B7 + B8,
  B7 + B3, B7 + B3 + B6, B7 + B6,
  B7 + B8 + B3, B7 + B8 + B3 + B6, B7 + B8 + B6,
  B8 + B3, B3 + B6 + B7 + B8
};

/* create bits for number 0..99 - special for papenmeier */
int pm_num (int x)
{
/*  return pm_tens[(x / 10) % 10] | pm_ones[x % 10];  */
  if (x)
    return 255;
  return 255;
}

/* status cell   tens: line number    ones: no/all bits set */
int pm_stat (int line, int on)
{
  if (on)
    return pm_tens[line % 10] | pm_ones[10];
  else
    return pm_tens[line];
}

void brl_debug (unsigned int dbg)
{
  brl_dbg = dbg;
}

void initbrl (brldim * brl, const char *dev)
{
  brldim res;                   /* return result */

  struct termios newtio;        /* new terminal settings */

  brl->usb = res.usb = 0;
  brl->x = res.x = BRLCOLS;     /* initialise size of display */
  brl->y = res.y = BRLROWS;
  res.disp = NULL;              /* clear pointers */

  /* Now open the Braille display device for random access */
  if (!dev)
    dev = DefDev;

  brl_fd = open (dev, O_RDWR | O_NOCTTY | O_NONBLOCK);
  if (brl_fd < 0)
   {
     closebrl (brl);
     brl->brl_fd = -1;
     brl->disp = NULL;
     return;
   }

  tcgetattr (brl_fd, &oldtio);  /* save current settings */

  /* Set bps, flow control and 8n1, enable reading */
  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;

  /* Ignore bytes with parity errors and make terminal raw and dumb */
  newtio.c_iflag = IGNPAR;
  newtio.c_oflag = 0;           /* raw output */
  newtio.c_lflag = 0;           /* don't echo or generate signals */
  newtio.c_cc[VMIN] = 0;        /* set nonblocking read */
  newtio.c_cc[VTIME] = 0;
  tcflush (brl_fd, TCIFLUSH);   /* clean line */
  tcsetattr (brl_fd, TCSANOW, &newtio); /* activate new settings */
  /* Allocate space for buffers */
  res.disp = (unsigned char *) malloc (res.x * res.y);
  allocmem ();
  init_table ();
  res.brl_fd = brl_fd;

  *brl = res;
  return;
}

void closebrl (brldim * brl)
{
  if (brl->disp != NULL)
    free (brl->disp);
  if (brl_fd >= 0)
   {
     tcsetattr (brl_fd, TCSANOW, &oldtio);      /* restore terminal settings */
     close (brl_fd);
     brl->brl_fd = -1;
     brl_fd = -1;
   }

  brl->disp = NULL;
}

struct brlinfo identbrl (const char *name, const char *dev)
{
  struct brlinfo retinfo;

  int devcnt;

  devcnt = sizeof (fhpdevs) / sizeof (struct brlinfo);
  retinfo.cols = 0;
  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (name, fhpdevs[devnr].name))
     {
       printf (" %s on %s\n", FULLNAME, dev);
       sbl_log (" %s on %s\n", FULLNAME, dev);
       retinfo = fhpdevs[devnr];
       break;
     }

  return retinfo;
}

int write_to_braille (int offset, int size, const unsigned char *data)
{
  unsigned char BrlHead[] = { cSTX,
    cIdSend,
    0,
    0,
    0,
    0,
  };
  unsigned char BrlTrail[] = { cETX };

  BrlHead[2] = offset / 256;
  BrlHead[3] = offset % 256;
  BrlHead[5] = (sizeof (BrlHead) + sizeof (BrlTrail) + size) % 256;

  if (write (brl_fd, BrlHead, sizeof (BrlHead)) < 0)
    return -1;
  else if (write (brl_fd, data, size) < 0)
    return -1;
  else if (write (brl_fd, BrlTrail, sizeof (BrlTrail)) < 0)
    return -1;
  return 0;
}

void init_table ()
{
  unsigned char line[BRLCOLS];

  unsigned char spalte[PMSC];

  int i;

  // dont use the internal table for the status column 
  for (i = 0; i < PMSC; i++)
    spalte[i] = 1;              /* 1 = no table */
  write_to_braille (offsetTable + offsetVertical, PMSC, spalte);

  // dont use the internal table for the line
  for (i = 0; i < BRLCOLS; i++)
    line[i] = 1;                // 1 = no table
  write_to_braille (offsetTable + offsetHorizontal, BRLCOLS, line);
}

void setbrlstat (const unsigned char *s)
{
  static unsigned char str[30];

  const unsigned char highbits[10] = { B2 + B4 + B5, B1, B1 + B2, B1 + B4,
    B1 + B4 + B5, B1 + B5, B1 + B2 + B4, B1 + B2 + B4 + B5, B1 + B2 + B5,
    B2 + B4
  };
  const unsigned char lowbits[10] =
    { B6 + B7 + B8, B3, B3 + B7, B3 + B6, B3 + B6 + B8,
    B3 + B8, B3 + B6 + B7, B3 + B6 + B7 + B8, B3 + B7 + B8, B6 + B7
  };

  if (PMSC == 0)
    return;
  if (memcmp (s, prevline, PMSC) != 0)
   {
     memcpy (prevline, s, PMSC);
     memcpy (str, s, PMSC);
     if (PMSC != 2)
      {
        str[0] = pm_tens[(int) str[0] / 10] | pm_ones[(int) str[0] % 10];
        str[1] = pm_tens[(int) str[1] / 10] | pm_ones[(int) str[1] % 10];
        str[2] = pm_tens[(int) str[2] / 10] | pm_ones[(int) str[2] % 10];
      }
     else
      {
        int left = str[0], right = str[1];

        str[0] = highbits[left / 10] | lowbits[right / 10];
        str[1] = highbits[left % 10] | lowbits[right % 10];
      }

     write_to_braille (offsetVertical, PMSC, str);
   }
}

/*
 * change bits for the papenmeier terminal
 *                             1 2           1 4
 * dot number -> bit number    3 4   we get  2 5 
 *                             5 6           3 6
 *                             7 8           7 8
 */
unsigned char change_bits[] = {
  0x00, 0x01, 0x08, 0x09, 0x02, 0x03, 0x0a, 0x0b,
  0x10, 0x11, 0x18, 0x19, 0x12, 0x13, 0x1a, 0x1b,
  0x04, 0x05, 0x0c, 0x0d, 0x06, 0x07, 0x0e, 0x0f,
  0x14, 0x15, 0x1c, 0x1d, 0x16, 0x17, 0x1e, 0x1f,
  0x20, 0x21, 0x28, 0x29, 0x22, 0x23, 0x2a, 0x2b,
  0x30, 0x31, 0x38, 0x39, 0x32, 0x33, 0x3a, 0x3b,
  0x24, 0x25, 0x2c, 0x2d, 0x26, 0x27, 0x2e, 0x2f,
  0x34, 0x35, 0x3c, 0x3d, 0x36, 0x37, 0x3e, 0x3f,
  0x40, 0x41, 0x48, 0x49, 0x42, 0x43, 0x4a, 0x4b,
  0x50, 0x51, 0x58, 0x59, 0x52, 0x53, 0x5a, 0x5b,
  0x44, 0x45, 0x4c, 0x4d, 0x46, 0x47, 0x4e, 0x4f,
  0x54, 0x55, 0x5c, 0x5d, 0x56, 0x57, 0x5e, 0x5f,
  0x60, 0x61, 0x68, 0x69, 0x62, 0x63, 0x6a, 0x6b,
  0x70, 0x71, 0x78, 0x79, 0x72, 0x73, 0x7a, 0x7b,
  0x64, 0x65, 0x6c, 0x6d, 0x66, 0x67, 0x6e, 0x6f,
  0x74, 0x75, 0x7c, 0x7d, 0x76, 0x77, 0x7e, 0x7f,
  0x80, 0x81, 0x88, 0x89, 0x82, 0x83, 0x8a, 0x8b,
  0x90, 0x91, 0x98, 0x99, 0x92, 0x93, 0x9a, 0x9b,
  0x84, 0x85, 0x8c, 0x8d, 0x86, 0x87, 0x8e, 0x8f,
  0x94, 0x95, 0x9c, 0x9d, 0x96, 0x97, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa8, 0xa9, 0xa2, 0xa3, 0xaa, 0xab,
  0xb0, 0xb1, 0xb8, 0xb9, 0xb2, 0xb3, 0xba, 0xbb,
  0xa4, 0xa5, 0xac, 0xad, 0xa6, 0xa7, 0xae, 0xaf,
  0xb4, 0xb5, 0xbc, 0xbd, 0xb6, 0xb7, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc8, 0xc9, 0xc2, 0xc3, 0xca, 0xcb,
  0xd0, 0xd1, 0xd8, 0xd9, 0xd2, 0xd3, 0xda, 0xdb,
  0xc4, 0xc5, 0xcc, 0xcd, 0xc6, 0xc7, 0xce, 0xcf,
  0xd4, 0xd5, 0xdc, 0xdd, 0xd6, 0xd7, 0xde, 0xdf,
  0xe0, 0xe1, 0xe8, 0xe9, 0xe2, 0xe3, 0xea, 0xeb,
  0xf0, 0xf1, 0xf8, 0xf9, 0xf2, 0xf3, 0xfa, 0xfb,
  0xe4, 0xe5, 0xec, 0xed, 0xe6, 0xe7, 0xee, 0xef,
  0xf4, 0xf5, 0xfc, 0xfd, 0xf6, 0xf7, 0xfe, 0xff
};

void writebrl (brldim * brl)
{
  int i;

  for (i = 0; i < BRLCOLS; i++)
    brl->disp[i] = change_bits[brl->disp[i]];

  if (memcmp (prev, brl->disp, BRLCOLS) != 0)
   {
     memcpy (prev, brl->disp, BRLCOLS);
     if (write_to_braille (offsetHorizontal, BRLCOLS, prev) < 0)
       closebrl (brl);
   }
}

/* ------------------------------------------------------------ */

/* some makros */
/* read byte */
#define READ(OFFS) \
  if (read(brl_fd,buf+OFFS,1) != 1) \
      return EOF;                   \

/* read byte and check value */
#define READ_CHK(OFFS, VAL)  \
    { READ(OFFS);            \
      if (buf[OFFS] != VAL)  \
        return EOF;      \
    }

int readbrl (int *xx)
{
  unsigned char buf[20];

  unsigned int i, l;

  fd_set select_set;

  struct timeval timeout;

  READ_CHK (0, cSTX);           /* STX - Start */
  usleep (5000);

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;

  READ_CHK (1, cIdReceive);     /* 'K' */

  usleep (5000);

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;
  READ (2);                     /* code - 2 bytes */
  usleep (5000);

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;
  READ (3);

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;
  READ (4);                     /* length - 2 bytes */

  FD_ZERO (&select_set);
  FD_SET (brl_fd, &select_set);
  timeout.tv_sec = 0;
  timeout.tv_usec = READ_WAIT * 1000;
  if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
    return EOF;
  READ (5);
  l = 0x100 * buf[4] + buf[5];  /* Data count */
  if (l > sizeof (buf))
    return CMD_ERR;
  for (i = 6; i < l; i++)
   {
     usleep (5000);
     FD_ZERO (&select_set);
     FD_SET (brl_fd, &select_set);
     timeout.tv_sec = 0;
     timeout.tv_usec = READ_WAIT * 1000;
     if (select (brl_fd + 1, &select_set, NULL, NULL, &timeout) <= 0)
       return EOF;

     READ (i);                  /* Data */
   }

  if (buf[l - 1] != cETX)       /* ETX - End */
    return CMD_ERR;

  *xx = buf[6];

  if (brl_dbg)
    sbl_log ("brlkey: %i %i %i %i %i %i %i %i %i %i %i\n",
             buf[0], buf[1], buf[2], buf[3], buf[4], buf[5], buf[6], buf[7],
             buf[8], buf[9], buf[10]);
  if (buf[1] != 'K')
    sbl_log ("ukey: %d ", buf[1]);

  {
    int Kod = 0x100 * buf[2] + buf[3];

    if (brl_dbg)
      sbl_log ("brlcode: %x \n", Kod);

    if (Kod >= 0x003 && Kod <= 0x2a)
      return Kod / 3;
    if (buf[6] == 1 && Kod >= 0x300)
      return (Kod - 0x300) / 3 + 100;

    return EOF;
  }

}

void allocmem ()
{
  prevline = (unsigned char *) malloc (PMSC);
  prev = (unsigned char *) malloc (BRLCOLS + 1);
}
