# -*- coding: utf-8 -*-
# Copyright 2024 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import os

# try/except added for compatibility with python < 3.8
try:
    from unittest import mock
    from unittest.mock import AsyncMock  # pragma: NO COVER
except ImportError:  # pragma: NO COVER
    import mock

import math

from google.api_core import gapic_v1, grpc_helpers, grpc_helpers_async, path_template
from google.api_core import api_core_version, client_options
from google.api_core import exceptions as core_exceptions
import google.auth
from google.auth import credentials as ga_credentials
from google.auth.exceptions import MutualTLSChannelError
from google.iam.v1 import iam_policy_pb2  # type: ignore
from google.iam.v1 import options_pb2  # type: ignore
from google.iam.v1 import policy_pb2  # type: ignore
from google.oauth2 import service_account
from google.protobuf import field_mask_pb2  # type: ignore
from google.protobuf import timestamp_pb2  # type: ignore
from google.type import expr_pb2  # type: ignore
import grpc
from grpc.experimental import aio
from proto.marshal.rules import wrappers
from proto.marshal.rules.dates import DurationRule, TimestampRule
import pytest

from google.cloud.iam_admin_v1.services.iam import (
    IAMAsyncClient,
    IAMClient,
    pagers,
    transports,
)
from google.cloud.iam_admin_v1.types import iam


def client_cert_source_callback():
    return b"cert bytes", b"key bytes"


# If default endpoint is localhost, then default mtls endpoint will be the same.
# This method modifies the default endpoint so the client can produce a different
# mtls endpoint for endpoint testing purposes.
def modify_default_endpoint(client):
    return (
        "foo.googleapis.com"
        if ("localhost" in client.DEFAULT_ENDPOINT)
        else client.DEFAULT_ENDPOINT
    )


# If default endpoint template is localhost, then default mtls endpoint will be the same.
# This method modifies the default endpoint template so the client can produce a different
# mtls endpoint for endpoint testing purposes.
def modify_default_endpoint_template(client):
    return (
        "test.{UNIVERSE_DOMAIN}"
        if ("localhost" in client._DEFAULT_ENDPOINT_TEMPLATE)
        else client._DEFAULT_ENDPOINT_TEMPLATE
    )


def test__get_default_mtls_endpoint():
    api_endpoint = "example.googleapis.com"
    api_mtls_endpoint = "example.mtls.googleapis.com"
    sandbox_endpoint = "example.sandbox.googleapis.com"
    sandbox_mtls_endpoint = "example.mtls.sandbox.googleapis.com"
    non_googleapi = "api.example.com"

    assert IAMClient._get_default_mtls_endpoint(None) is None
    assert IAMClient._get_default_mtls_endpoint(api_endpoint) == api_mtls_endpoint
    assert IAMClient._get_default_mtls_endpoint(api_mtls_endpoint) == api_mtls_endpoint
    assert (
        IAMClient._get_default_mtls_endpoint(sandbox_endpoint) == sandbox_mtls_endpoint
    )
    assert (
        IAMClient._get_default_mtls_endpoint(sandbox_mtls_endpoint)
        == sandbox_mtls_endpoint
    )
    assert IAMClient._get_default_mtls_endpoint(non_googleapi) == non_googleapi


def test__read_environment_variables():
    assert IAMClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        assert IAMClient._read_environment_variables() == (True, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "false"}):
        assert IAMClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            IAMClient._read_environment_variables()
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
    )

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        assert IAMClient._read_environment_variables() == (False, "never", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        assert IAMClient._read_environment_variables() == (False, "always", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "auto"}):
        assert IAMClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            IAMClient._read_environment_variables()
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
    )

    with mock.patch.dict(os.environ, {"GOOGLE_CLOUD_UNIVERSE_DOMAIN": "foo.com"}):
        assert IAMClient._read_environment_variables() == (False, "auto", "foo.com")


def test__get_client_cert_source():
    mock_provided_cert_source = mock.Mock()
    mock_default_cert_source = mock.Mock()

    assert IAMClient._get_client_cert_source(None, False) is None
    assert IAMClient._get_client_cert_source(mock_provided_cert_source, False) is None
    assert (
        IAMClient._get_client_cert_source(mock_provided_cert_source, True)
        == mock_provided_cert_source
    )

    with mock.patch(
        "google.auth.transport.mtls.has_default_client_cert_source", return_value=True
    ):
        with mock.patch(
            "google.auth.transport.mtls.default_client_cert_source",
            return_value=mock_default_cert_source,
        ):
            assert (
                IAMClient._get_client_cert_source(None, True)
                is mock_default_cert_source
            )
            assert (
                IAMClient._get_client_cert_source(mock_provided_cert_source, "true")
                is mock_provided_cert_source
            )


@mock.patch.object(
    IAMClient, "_DEFAULT_ENDPOINT_TEMPLATE", modify_default_endpoint_template(IAMClient)
)
@mock.patch.object(
    IAMAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(IAMAsyncClient),
)
def test__get_api_endpoint():
    api_override = "foo.com"
    mock_client_cert_source = mock.Mock()
    default_universe = IAMClient._DEFAULT_UNIVERSE
    default_endpoint = IAMClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=default_universe
    )
    mock_universe = "bar.com"
    mock_endpoint = IAMClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=mock_universe
    )

    assert (
        IAMClient._get_api_endpoint(
            api_override, mock_client_cert_source, default_universe, "always"
        )
        == api_override
    )
    assert (
        IAMClient._get_api_endpoint(
            None, mock_client_cert_source, default_universe, "auto"
        )
        == IAMClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        IAMClient._get_api_endpoint(None, None, default_universe, "auto")
        == default_endpoint
    )
    assert (
        IAMClient._get_api_endpoint(None, None, default_universe, "always")
        == IAMClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        IAMClient._get_api_endpoint(
            None, mock_client_cert_source, default_universe, "always"
        )
        == IAMClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        IAMClient._get_api_endpoint(None, None, mock_universe, "never") == mock_endpoint
    )
    assert (
        IAMClient._get_api_endpoint(None, None, default_universe, "never")
        == default_endpoint
    )

    with pytest.raises(MutualTLSChannelError) as excinfo:
        IAMClient._get_api_endpoint(
            None, mock_client_cert_source, mock_universe, "auto"
        )
    assert (
        str(excinfo.value)
        == "mTLS is not supported in any universe other than googleapis.com."
    )


def test__get_universe_domain():
    client_universe_domain = "foo.com"
    universe_domain_env = "bar.com"

    assert (
        IAMClient._get_universe_domain(client_universe_domain, universe_domain_env)
        == client_universe_domain
    )
    assert (
        IAMClient._get_universe_domain(None, universe_domain_env) == universe_domain_env
    )
    assert IAMClient._get_universe_domain(None, None) == IAMClient._DEFAULT_UNIVERSE

    with pytest.raises(ValueError) as excinfo:
        IAMClient._get_universe_domain("", None)
    assert str(excinfo.value) == "Universe Domain cannot be an empty string."


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc"),
    ],
)
def test__validate_universe_domain(client_class, transport_class, transport_name):
    client = client_class(
        transport=transport_class(credentials=ga_credentials.AnonymousCredentials())
    )
    assert client._validate_universe_domain() == True

    # Test the case when universe is already validated.
    assert client._validate_universe_domain() == True

    if transport_name == "grpc":
        # Test the case where credentials are provided by the
        # `local_channel_credentials`. The default universes in both match.
        channel = grpc.secure_channel(
            "http://localhost/", grpc.local_channel_credentials()
        )
        client = client_class(transport=transport_class(channel=channel))
        assert client._validate_universe_domain() == True

        # Test the case where credentials do not exist: e.g. a transport is provided
        # with no credentials. Validation should still succeed because there is no
        # mismatch with non-existent credentials.
        channel = grpc.secure_channel(
            "http://localhost/", grpc.local_channel_credentials()
        )
        transport = transport_class(channel=channel)
        transport._credentials = None
        client = client_class(transport=transport)
        assert client._validate_universe_domain() == True

    # TODO: This is needed to cater for older versions of google-auth
    # Make this test unconditional once the minimum supported version of
    # google-auth becomes 2.23.0 or higher.
    google_auth_major, google_auth_minor = [
        int(part) for part in google.auth.__version__.split(".")[0:2]
    ]
    if google_auth_major > 2 or (google_auth_major == 2 and google_auth_minor >= 23):
        credentials = ga_credentials.AnonymousCredentials()
        credentials._universe_domain = "foo.com"
        # Test the case when there is a universe mismatch from the credentials.
        client = client_class(transport=transport_class(credentials=credentials))
        with pytest.raises(ValueError) as excinfo:
            client._validate_universe_domain()
        assert (
            str(excinfo.value)
            == "The configured universe domain (googleapis.com) does not match the universe domain found in the credentials (foo.com). If you haven't configured the universe domain explicitly, `googleapis.com` is the default."
        )

        # Test the case when there is a universe mismatch from the client.
        #
        # TODO: Make this test unconditional once the minimum supported version of
        # google-api-core becomes 2.15.0 or higher.
        api_core_major, api_core_minor = [
            int(part) for part in api_core_version.__version__.split(".")[0:2]
        ]
        if api_core_major > 2 or (api_core_major == 2 and api_core_minor >= 15):
            client = client_class(
                client_options={"universe_domain": "bar.com"},
                transport=transport_class(
                    credentials=ga_credentials.AnonymousCredentials(),
                ),
            )
            with pytest.raises(ValueError) as excinfo:
                client._validate_universe_domain()
            assert (
                str(excinfo.value)
                == "The configured universe domain (bar.com) does not match the universe domain found in the credentials (googleapis.com). If you haven't configured the universe domain explicitly, `googleapis.com` is the default."
            )

    # Test that ValueError is raised if universe_domain is provided via client options and credentials is None
    with pytest.raises(ValueError):
        client._compare_universes("foo.bar", None)


@pytest.mark.parametrize(
    "client_class,transport_name",
    [
        (IAMClient, "grpc"),
        (IAMAsyncClient, "grpc_asyncio"),
    ],
)
def test_iam_client_from_service_account_info(client_class, transport_name):
    creds = ga_credentials.AnonymousCredentials()
    with mock.patch.object(
        service_account.Credentials, "from_service_account_info"
    ) as factory:
        factory.return_value = creds
        info = {"valid": True}
        client = client_class.from_service_account_info(info, transport=transport_name)
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        assert client.transport._host == ("iam.googleapis.com:443")


@pytest.mark.parametrize(
    "transport_class,transport_name",
    [
        (transports.IAMGrpcTransport, "grpc"),
        (transports.IAMGrpcAsyncIOTransport, "grpc_asyncio"),
    ],
)
def test_iam_client_service_account_always_use_jwt(transport_class, transport_name):
    with mock.patch.object(
        service_account.Credentials, "with_always_use_jwt_access", create=True
    ) as use_jwt:
        creds = service_account.Credentials(None, None, None)
        transport = transport_class(credentials=creds, always_use_jwt_access=True)
        use_jwt.assert_called_once_with(True)

    with mock.patch.object(
        service_account.Credentials, "with_always_use_jwt_access", create=True
    ) as use_jwt:
        creds = service_account.Credentials(None, None, None)
        transport = transport_class(credentials=creds, always_use_jwt_access=False)
        use_jwt.assert_not_called()


@pytest.mark.parametrize(
    "client_class,transport_name",
    [
        (IAMClient, "grpc"),
        (IAMAsyncClient, "grpc_asyncio"),
    ],
)
def test_iam_client_from_service_account_file(client_class, transport_name):
    creds = ga_credentials.AnonymousCredentials()
    with mock.patch.object(
        service_account.Credentials, "from_service_account_file"
    ) as factory:
        factory.return_value = creds
        client = client_class.from_service_account_file(
            "dummy/file/path.json", transport=transport_name
        )
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        client = client_class.from_service_account_json(
            "dummy/file/path.json", transport=transport_name
        )
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        assert client.transport._host == ("iam.googleapis.com:443")


def test_iam_client_get_transport_class():
    transport = IAMClient.get_transport_class()
    available_transports = [
        transports.IAMGrpcTransport,
    ]
    assert transport in available_transports

    transport = IAMClient.get_transport_class("grpc")
    assert transport == transports.IAMGrpcTransport


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc"),
        (IAMAsyncClient, transports.IAMGrpcAsyncIOTransport, "grpc_asyncio"),
    ],
)
@mock.patch.object(
    IAMClient, "_DEFAULT_ENDPOINT_TEMPLATE", modify_default_endpoint_template(IAMClient)
)
@mock.patch.object(
    IAMAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(IAMAsyncClient),
)
def test_iam_client_client_options(client_class, transport_class, transport_name):
    # Check that if channel is provided we won't create a new one.
    with mock.patch.object(IAMClient, "get_transport_class") as gtc:
        transport = transport_class(credentials=ga_credentials.AnonymousCredentials())
        client = client_class(transport=transport)
        gtc.assert_not_called()

    # Check that if channel is provided via str we will create a new one.
    with mock.patch.object(IAMClient, "get_transport_class") as gtc:
        client = client_class(transport=transport_name)
        gtc.assert_called()

    # Check the case api_endpoint is provided.
    options = client_options.ClientOptions(api_endpoint="squid.clam.whelk")
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(transport=transport_name, client_options=options)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host="squid.clam.whelk",
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT is
    # "never".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(transport=transport_name)
            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                ),
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT is
    # "always".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(transport=transport_name)
            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=client.DEFAULT_MTLS_ENDPOINT,
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT has
    # unsupported value.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            client = client_class(transport=transport_name)
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
    )

    # Check the case GOOGLE_API_USE_CLIENT_CERTIFICATE has unsupported value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            client = client_class(transport=transport_name)
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
    )

    # Check the case quota_project_id is provided
    options = client_options.ClientOptions(quota_project_id="octopus")
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id="octopus",
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )
    # Check the case api_endpoint is provided
    options = client_options.ClientOptions(
        api_audience="https://language.googleapis.com"
    )
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience="https://language.googleapis.com",
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,use_client_cert_env",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc", "true"),
        (IAMAsyncClient, transports.IAMGrpcAsyncIOTransport, "grpc_asyncio", "true"),
        (IAMClient, transports.IAMGrpcTransport, "grpc", "false"),
        (IAMAsyncClient, transports.IAMGrpcAsyncIOTransport, "grpc_asyncio", "false"),
    ],
)
@mock.patch.object(
    IAMClient, "_DEFAULT_ENDPOINT_TEMPLATE", modify_default_endpoint_template(IAMClient)
)
@mock.patch.object(
    IAMAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(IAMAsyncClient),
)
@mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "auto"})
def test_iam_client_mtls_env_auto(
    client_class, transport_class, transport_name, use_client_cert_env
):
    # This tests the endpoint autoswitch behavior. Endpoint is autoswitched to the default
    # mtls endpoint, if GOOGLE_API_USE_CLIENT_CERTIFICATE is "true" and client cert exists.

    # Check the case client_cert_source is provided. Whether client cert is used depends on
    # GOOGLE_API_USE_CLIENT_CERTIFICATE value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        options = client_options.ClientOptions(
            client_cert_source=client_cert_source_callback
        )
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(client_options=options, transport=transport_name)

            if use_client_cert_env == "false":
                expected_client_cert_source = None
                expected_host = client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                )
            else:
                expected_client_cert_source = client_cert_source_callback
                expected_host = client.DEFAULT_MTLS_ENDPOINT

            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=expected_host,
                scopes=None,
                client_cert_source_for_mtls=expected_client_cert_source,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case ADC client cert is provided. Whether client cert is used depends on
    # GOOGLE_API_USE_CLIENT_CERTIFICATE value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        with mock.patch.object(transport_class, "__init__") as patched:
            with mock.patch(
                "google.auth.transport.mtls.has_default_client_cert_source",
                return_value=True,
            ):
                with mock.patch(
                    "google.auth.transport.mtls.default_client_cert_source",
                    return_value=client_cert_source_callback,
                ):
                    if use_client_cert_env == "false":
                        expected_host = client._DEFAULT_ENDPOINT_TEMPLATE.format(
                            UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                        )
                        expected_client_cert_source = None
                    else:
                        expected_host = client.DEFAULT_MTLS_ENDPOINT
                        expected_client_cert_source = client_cert_source_callback

                    patched.return_value = None
                    client = client_class(transport=transport_name)
                    patched.assert_called_once_with(
                        credentials=None,
                        credentials_file=None,
                        host=expected_host,
                        scopes=None,
                        client_cert_source_for_mtls=expected_client_cert_source,
                        quota_project_id=None,
                        client_info=transports.base.DEFAULT_CLIENT_INFO,
                        always_use_jwt_access=True,
                        api_audience=None,
                    )

    # Check the case client_cert_source and ADC client cert are not provided.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        with mock.patch.object(transport_class, "__init__") as patched:
            with mock.patch(
                "google.auth.transport.mtls.has_default_client_cert_source",
                return_value=False,
            ):
                patched.return_value = None
                client = client_class(transport=transport_name)
                patched.assert_called_once_with(
                    credentials=None,
                    credentials_file=None,
                    host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                        UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                    ),
                    scopes=None,
                    client_cert_source_for_mtls=None,
                    quota_project_id=None,
                    client_info=transports.base.DEFAULT_CLIENT_INFO,
                    always_use_jwt_access=True,
                    api_audience=None,
                )


@pytest.mark.parametrize("client_class", [IAMClient, IAMAsyncClient])
@mock.patch.object(IAMClient, "DEFAULT_ENDPOINT", modify_default_endpoint(IAMClient))
@mock.patch.object(
    IAMAsyncClient, "DEFAULT_ENDPOINT", modify_default_endpoint(IAMAsyncClient)
)
def test_iam_client_get_mtls_endpoint_and_cert_source(client_class):
    mock_client_cert_source = mock.Mock()

    # Test the case GOOGLE_API_USE_CLIENT_CERTIFICATE is "true".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        mock_api_endpoint = "foo"
        options = client_options.ClientOptions(
            client_cert_source=mock_client_cert_source, api_endpoint=mock_api_endpoint
        )
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source(
            options
        )
        assert api_endpoint == mock_api_endpoint
        assert cert_source == mock_client_cert_source

    # Test the case GOOGLE_API_USE_CLIENT_CERTIFICATE is "false".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "false"}):
        mock_client_cert_source = mock.Mock()
        mock_api_endpoint = "foo"
        options = client_options.ClientOptions(
            client_cert_source=mock_client_cert_source, api_endpoint=mock_api_endpoint
        )
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source(
            options
        )
        assert api_endpoint == mock_api_endpoint
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "never".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
        assert api_endpoint == client_class.DEFAULT_ENDPOINT
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "always".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
        assert api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "auto" and default cert doesn't exist.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.mtls.has_default_client_cert_source",
            return_value=False,
        ):
            api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
            assert api_endpoint == client_class.DEFAULT_ENDPOINT
            assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "auto" and default cert exists.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.mtls.has_default_client_cert_source",
            return_value=True,
        ):
            with mock.patch(
                "google.auth.transport.mtls.default_client_cert_source",
                return_value=mock_client_cert_source,
            ):
                (
                    api_endpoint,
                    cert_source,
                ) = client_class.get_mtls_endpoint_and_cert_source()
                assert api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT
                assert cert_source == mock_client_cert_source

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT has
    # unsupported value.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            client_class.get_mtls_endpoint_and_cert_source()

        assert (
            str(excinfo.value)
            == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
        )

    # Check the case GOOGLE_API_USE_CLIENT_CERTIFICATE has unsupported value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            client_class.get_mtls_endpoint_and_cert_source()

        assert (
            str(excinfo.value)
            == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
        )


@pytest.mark.parametrize("client_class", [IAMClient, IAMAsyncClient])
@mock.patch.object(
    IAMClient, "_DEFAULT_ENDPOINT_TEMPLATE", modify_default_endpoint_template(IAMClient)
)
@mock.patch.object(
    IAMAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(IAMAsyncClient),
)
def test_iam_client_client_api_endpoint(client_class):
    mock_client_cert_source = client_cert_source_callback
    api_override = "foo.com"
    default_universe = IAMClient._DEFAULT_UNIVERSE
    default_endpoint = IAMClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=default_universe
    )
    mock_universe = "bar.com"
    mock_endpoint = IAMClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=mock_universe
    )

    # If ClientOptions.api_endpoint is set and GOOGLE_API_USE_CLIENT_CERTIFICATE="true",
    # use ClientOptions.api_endpoint as the api endpoint regardless.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.requests.AuthorizedSession.configure_mtls_channel"
        ):
            options = client_options.ClientOptions(
                client_cert_source=mock_client_cert_source, api_endpoint=api_override
            )
            client = client_class(
                client_options=options,
                credentials=ga_credentials.AnonymousCredentials(),
            )
            assert client.api_endpoint == api_override

    # If ClientOptions.api_endpoint is not set and GOOGLE_API_USE_MTLS_ENDPOINT="never",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with GDU as the api endpoint.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        client = client_class(credentials=ga_credentials.AnonymousCredentials())
        assert client.api_endpoint == default_endpoint

    # If ClientOptions.api_endpoint is not set and GOOGLE_API_USE_MTLS_ENDPOINT="always",
    # use the DEFAULT_MTLS_ENDPOINT as the api endpoint.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        client = client_class(credentials=ga_credentials.AnonymousCredentials())
        assert client.api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT

    # If ClientOptions.api_endpoint is not set, GOOGLE_API_USE_MTLS_ENDPOINT="auto" (default),
    # GOOGLE_API_USE_CLIENT_CERTIFICATE="false" (default), default cert source doesn't exist,
    # and ClientOptions.universe_domain="bar.com",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with universe domain as the api endpoint.
    options = client_options.ClientOptions()
    universe_exists = hasattr(options, "universe_domain")
    if universe_exists:
        options = client_options.ClientOptions(universe_domain=mock_universe)
        client = client_class(
            client_options=options, credentials=ga_credentials.AnonymousCredentials()
        )
    else:
        client = client_class(
            client_options=options, credentials=ga_credentials.AnonymousCredentials()
        )
    assert client.api_endpoint == (
        mock_endpoint if universe_exists else default_endpoint
    )
    assert client.universe_domain == (
        mock_universe if universe_exists else default_universe
    )

    # If ClientOptions does not have a universe domain attribute and GOOGLE_API_USE_MTLS_ENDPOINT="never",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with GDU as the api endpoint.
    options = client_options.ClientOptions()
    if hasattr(options, "universe_domain"):
        delattr(options, "universe_domain")
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        client = client_class(
            client_options=options, credentials=ga_credentials.AnonymousCredentials()
        )
        assert client.api_endpoint == default_endpoint


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc"),
        (IAMAsyncClient, transports.IAMGrpcAsyncIOTransport, "grpc_asyncio"),
    ],
)
def test_iam_client_client_options_scopes(
    client_class, transport_class, transport_name
):
    # Check the case scopes are provided.
    options = client_options.ClientOptions(
        scopes=["1", "2"],
    )
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=["1", "2"],
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,grpc_helpers",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc", grpc_helpers),
        (
            IAMAsyncClient,
            transports.IAMGrpcAsyncIOTransport,
            "grpc_asyncio",
            grpc_helpers_async,
        ),
    ],
)
def test_iam_client_client_options_credentials_file(
    client_class, transport_class, transport_name, grpc_helpers
):
    # Check the case credentials file is provided.
    options = client_options.ClientOptions(credentials_file="credentials.json")

    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file="credentials.json",
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


def test_iam_client_client_options_from_dict():
    with mock.patch(
        "google.cloud.iam_admin_v1.services.iam.transports.IAMGrpcTransport.__init__"
    ) as grpc_transport:
        grpc_transport.return_value = None
        client = IAMClient(client_options={"api_endpoint": "squid.clam.whelk"})
        grpc_transport.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host="squid.clam.whelk",
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,grpc_helpers",
    [
        (IAMClient, transports.IAMGrpcTransport, "grpc", grpc_helpers),
        (
            IAMAsyncClient,
            transports.IAMGrpcAsyncIOTransport,
            "grpc_asyncio",
            grpc_helpers_async,
        ),
    ],
)
def test_iam_client_create_channel_credentials_file(
    client_class, transport_class, transport_name, grpc_helpers
):
    # Check the case credentials file is provided.
    options = client_options.ClientOptions(credentials_file="credentials.json")

    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file="credentials.json",
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )

    # test that the credentials from file are saved and used as the credentials.
    with mock.patch.object(
        google.auth, "load_credentials_from_file", autospec=True
    ) as load_creds, mock.patch.object(
        google.auth, "default", autospec=True
    ) as adc, mock.patch.object(
        grpc_helpers, "create_channel"
    ) as create_channel:
        creds = ga_credentials.AnonymousCredentials()
        file_creds = ga_credentials.AnonymousCredentials()
        load_creds.return_value = (file_creds, None)
        adc.return_value = (creds, None)
        client = client_class(client_options=options, transport=transport_name)
        create_channel.assert_called_with(
            "iam.googleapis.com:443",
            credentials=file_creds,
            credentials_file=None,
            quota_project_id=None,
            default_scopes=("https://www.googleapis.com/auth/cloud-platform",),
            scopes=None,
            default_host="iam.googleapis.com",
            ssl_credentials=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.ListServiceAccountsRequest,
        dict,
    ],
)
def test_list_service_accounts(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountsResponse(
            next_page_token="next_page_token_value",
        )
        response = client.list_service_accounts(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.ListServiceAccountsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListServiceAccountsPager)
    assert response.next_page_token == "next_page_token_value"


def test_list_service_accounts_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        client.list_service_accounts()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountsRequest()


def test_list_service_accounts_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.ListServiceAccountsRequest(
        name="name_value",
        page_token="page_token_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        client.list_service_accounts(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountsRequest(
            name="name_value",
            page_token="page_token_value",
        )


@pytest.mark.asyncio
async def test_list_service_accounts_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountsResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.list_service_accounts()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountsRequest()


@pytest.mark.asyncio
async def test_list_service_accounts_async(
    transport: str = "grpc_asyncio", request_type=iam.ListServiceAccountsRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountsResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.list_service_accounts(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.ListServiceAccountsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListServiceAccountsAsyncPager)
    assert response.next_page_token == "next_page_token_value"


@pytest.mark.asyncio
async def test_list_service_accounts_async_from_dict():
    await test_list_service_accounts_async(request_type=dict)


def test_list_service_accounts_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ListServiceAccountsRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        call.return_value = iam.ListServiceAccountsResponse()
        client.list_service_accounts(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_list_service_accounts_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ListServiceAccountsRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountsResponse()
        )
        await client.list_service_accounts(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_list_service_accounts_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountsResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.list_service_accounts(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_list_service_accounts_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.list_service_accounts(
            iam.ListServiceAccountsRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_list_service_accounts_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountsResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountsResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.list_service_accounts(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_list_service_accounts_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.list_service_accounts(
            iam.ListServiceAccountsRequest(),
            name="name_value",
        )


def test_list_service_accounts_pager(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
                next_page_token="abc",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[],
                next_page_token="def",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                ],
                next_page_token="ghi",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
            ),
            RuntimeError,
        )

        metadata = ()
        metadata = tuple(metadata) + (
            gapic_v1.routing_header.to_grpc_metadata((("name", ""),)),
        )
        pager = client.list_service_accounts(request={})

        assert pager._metadata == metadata

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, iam.ServiceAccount) for i in results)


def test_list_service_accounts_pages(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
                next_page_token="abc",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[],
                next_page_token="def",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                ],
                next_page_token="ghi",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
            ),
            RuntimeError,
        )
        pages = list(client.list_service_accounts(request={}).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.asyncio
async def test_list_service_accounts_async_pager():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
                next_page_token="abc",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[],
                next_page_token="def",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                ],
                next_page_token="ghi",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
            ),
            RuntimeError,
        )
        async_pager = await client.list_service_accounts(
            request={},
        )
        assert async_pager.next_page_token == "abc"
        responses = []
        async for response in async_pager:  # pragma: no branch
            responses.append(response)

        assert len(responses) == 6
        assert all(isinstance(i, iam.ServiceAccount) for i in responses)


@pytest.mark.asyncio
async def test_list_service_accounts_async_pages():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_accounts),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
                next_page_token="abc",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[],
                next_page_token="def",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                ],
                next_page_token="ghi",
            ),
            iam.ListServiceAccountsResponse(
                accounts=[
                    iam.ServiceAccount(),
                    iam.ServiceAccount(),
                ],
            ),
            RuntimeError,
        )
        pages = []
        # Workaround issue in python 3.9 related to code coverage by adding `# pragma: no branch`
        # See https://github.com/googleapis/gapic-generator-python/pull/1174#issuecomment-1025132372
        async for page_ in (  # pragma: no branch
            await client.list_service_accounts(request={})
        ).pages:
            pages.append(page_)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        iam.GetServiceAccountRequest,
        dict,
    ],
)
def test_get_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount(
            name="name_value",
            project_id="project_id_value",
            unique_id="unique_id_value",
            email="email_value",
            display_name="display_name_value",
            etag=b"etag_blob",
            description="description_value",
            oauth2_client_id="oauth2_client_id_value",
            disabled=True,
        )
        response = client.get_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.GetServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


def test_get_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        client.get_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountRequest()


def test_get_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.GetServiceAccountRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        client.get_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.get_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountRequest()


@pytest.mark.asyncio
async def test_get_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.GetServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.get_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.GetServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


@pytest.mark.asyncio
async def test_get_service_account_async_from_dict():
    await test_get_service_account_async(request_type=dict)


def test_get_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccount()
        client.get_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        await client.get_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_get_service_account_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.get_service_account(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_get_service_account_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_service_account(
            iam.GetServiceAccountRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_service_account_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.get_service_account(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_get_service_account_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.get_service_account(
            iam.GetServiceAccountRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.CreateServiceAccountRequest,
        dict,
    ],
)
def test_create_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount(
            name="name_value",
            project_id="project_id_value",
            unique_id="unique_id_value",
            email="email_value",
            display_name="display_name_value",
            etag=b"etag_blob",
            description="description_value",
            oauth2_client_id="oauth2_client_id_value",
            disabled=True,
        )
        response = client.create_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.CreateServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


def test_create_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        client.create_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountRequest()


def test_create_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.CreateServiceAccountRequest(
        name="name_value",
        account_id="account_id_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        client.create_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountRequest(
            name="name_value",
            account_id="account_id_value",
        )


@pytest.mark.asyncio
async def test_create_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.create_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountRequest()


@pytest.mark.asyncio
async def test_create_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.CreateServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.create_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.CreateServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


@pytest.mark.asyncio
async def test_create_service_account_async_from_dict():
    await test_create_service_account_async(request_type=dict)


def test_create_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccount()
        client.create_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_create_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        await client.create_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_create_service_account_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.create_service_account(
            name="name_value",
            account_id="account_id_value",
            service_account=iam.ServiceAccount(name="name_value"),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].account_id
        mock_val = "account_id_value"
        assert arg == mock_val
        arg = args[0].service_account
        mock_val = iam.ServiceAccount(name="name_value")
        assert arg == mock_val


def test_create_service_account_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.create_service_account(
            iam.CreateServiceAccountRequest(),
            name="name_value",
            account_id="account_id_value",
            service_account=iam.ServiceAccount(name="name_value"),
        )


@pytest.mark.asyncio
async def test_create_service_account_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.create_service_account(
            name="name_value",
            account_id="account_id_value",
            service_account=iam.ServiceAccount(name="name_value"),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].account_id
        mock_val = "account_id_value"
        assert arg == mock_val
        arg = args[0].service_account
        mock_val = iam.ServiceAccount(name="name_value")
        assert arg == mock_val


@pytest.mark.asyncio
async def test_create_service_account_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.create_service_account(
            iam.CreateServiceAccountRequest(),
            name="name_value",
            account_id="account_id_value",
            service_account=iam.ServiceAccount(name="name_value"),
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.ServiceAccount,
        dict,
    ],
)
def test_update_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount(
            name="name_value",
            project_id="project_id_value",
            unique_id="unique_id_value",
            email="email_value",
            display_name="display_name_value",
            etag=b"etag_blob",
            description="description_value",
            oauth2_client_id="oauth2_client_id_value",
            disabled=True,
        )
        response = client.update_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.ServiceAccount()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


def test_update_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        client.update_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ServiceAccount()


def test_update_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.ServiceAccount(
        name="name_value",
        project_id="project_id_value",
        unique_id="unique_id_value",
        email="email_value",
        display_name="display_name_value",
        description="description_value",
        oauth2_client_id="oauth2_client_id_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        client.update_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ServiceAccount(
            name="name_value",
            project_id="project_id_value",
            unique_id="unique_id_value",
            email="email_value",
            display_name="display_name_value",
            description="description_value",
            oauth2_client_id="oauth2_client_id_value",
        )


@pytest.mark.asyncio
async def test_update_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.update_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ServiceAccount()


@pytest.mark.asyncio
async def test_update_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.ServiceAccount
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.update_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.ServiceAccount()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


@pytest.mark.asyncio
async def test_update_service_account_async_from_dict():
    await test_update_service_account_async(request_type=dict)


def test_update_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ServiceAccount()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccount()
        client.update_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_update_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ServiceAccount()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        await client.update_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.PatchServiceAccountRequest,
        dict,
    ],
)
def test_patch_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccount(
            name="name_value",
            project_id="project_id_value",
            unique_id="unique_id_value",
            email="email_value",
            display_name="display_name_value",
            etag=b"etag_blob",
            description="description_value",
            oauth2_client_id="oauth2_client_id_value",
            disabled=True,
        )
        response = client.patch_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.PatchServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


def test_patch_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        client.patch_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.PatchServiceAccountRequest()


def test_patch_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.PatchServiceAccountRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        client.patch_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.PatchServiceAccountRequest()


@pytest.mark.asyncio
async def test_patch_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.patch_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.PatchServiceAccountRequest()


@pytest.mark.asyncio
async def test_patch_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.PatchServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccount(
                name="name_value",
                project_id="project_id_value",
                unique_id="unique_id_value",
                email="email_value",
                display_name="display_name_value",
                etag=b"etag_blob",
                description="description_value",
                oauth2_client_id="oauth2_client_id_value",
                disabled=True,
            )
        )
        response = await client.patch_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.PatchServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccount)
    assert response.name == "name_value"
    assert response.project_id == "project_id_value"
    assert response.unique_id == "unique_id_value"
    assert response.email == "email_value"
    assert response.display_name == "display_name_value"
    assert response.etag == b"etag_blob"
    assert response.description == "description_value"
    assert response.oauth2_client_id == "oauth2_client_id_value"
    assert response.disabled is True


@pytest.mark.asyncio
async def test_patch_service_account_async_from_dict():
    await test_patch_service_account_async(request_type=dict)


def test_patch_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.PatchServiceAccountRequest()

    request.service_account.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccount()
        client.patch_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "service_account.name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_patch_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.PatchServiceAccountRequest()

    request.service_account.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.patch_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.ServiceAccount())
        await client.patch_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "service_account.name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.DeleteServiceAccountRequest,
        dict,
    ],
)
def test_delete_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.delete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_delete_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        client.delete_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountRequest()


def test_delete_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.DeleteServiceAccountRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        client.delete_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_delete_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.delete_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountRequest()


@pytest.mark.asyncio
async def test_delete_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.DeleteServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.delete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_delete_service_account_async_from_dict():
    await test_delete_service_account_async(request_type=dict)


def test_delete_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        call.return_value = None
        client.delete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_delete_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.delete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_delete_service_account_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.delete_service_account(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_delete_service_account_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.delete_service_account(
            iam.DeleteServiceAccountRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_delete_service_account_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.delete_service_account(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_delete_service_account_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.delete_service_account(
            iam.DeleteServiceAccountRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.UndeleteServiceAccountRequest,
        dict,
    ],
)
def test_undelete_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.UndeleteServiceAccountResponse()
        response = client.undelete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.UndeleteServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.UndeleteServiceAccountResponse)


def test_undelete_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        client.undelete_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteServiceAccountRequest()


def test_undelete_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.UndeleteServiceAccountRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        client.undelete_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteServiceAccountRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_undelete_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.UndeleteServiceAccountResponse()
        )
        response = await client.undelete_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteServiceAccountRequest()


@pytest.mark.asyncio
async def test_undelete_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.UndeleteServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.UndeleteServiceAccountResponse()
        )
        response = await client.undelete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.UndeleteServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.UndeleteServiceAccountResponse)


@pytest.mark.asyncio
async def test_undelete_service_account_async_from_dict():
    await test_undelete_service_account_async(request_type=dict)


def test_undelete_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UndeleteServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        call.return_value = iam.UndeleteServiceAccountResponse()
        client.undelete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_undelete_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UndeleteServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.undelete_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.UndeleteServiceAccountResponse()
        )
        await client.undelete_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.EnableServiceAccountRequest,
        dict,
    ],
)
def test_enable_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.enable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.EnableServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_enable_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        client.enable_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountRequest()


def test_enable_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.EnableServiceAccountRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        client.enable_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_enable_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.enable_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountRequest()


@pytest.mark.asyncio
async def test_enable_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.EnableServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.enable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.EnableServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_enable_service_account_async_from_dict():
    await test_enable_service_account_async(request_type=dict)


def test_enable_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.EnableServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        call.return_value = None
        client.enable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_enable_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.EnableServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.enable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.DisableServiceAccountRequest,
        dict,
    ],
)
def test_disable_service_account(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.disable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.DisableServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_disable_service_account_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        client.disable_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountRequest()


def test_disable_service_account_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.DisableServiceAccountRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        client.disable_service_account(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_disable_service_account_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.disable_service_account()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountRequest()


@pytest.mark.asyncio
async def test_disable_service_account_async(
    transport: str = "grpc_asyncio", request_type=iam.DisableServiceAccountRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.disable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.DisableServiceAccountRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_disable_service_account_async_from_dict():
    await test_disable_service_account_async(request_type=dict)


def test_disable_service_account_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DisableServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        call.return_value = None
        client.disable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_disable_service_account_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DisableServiceAccountRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.disable_service_account(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.ListServiceAccountKeysRequest,
        dict,
    ],
)
def test_list_service_account_keys(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountKeysResponse()
        response = client.list_service_account_keys(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.ListServiceAccountKeysRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ListServiceAccountKeysResponse)


def test_list_service_account_keys_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        client.list_service_account_keys()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountKeysRequest()


def test_list_service_account_keys_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.ListServiceAccountKeysRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        client.list_service_account_keys(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountKeysRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_list_service_account_keys_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountKeysResponse()
        )
        response = await client.list_service_account_keys()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListServiceAccountKeysRequest()


@pytest.mark.asyncio
async def test_list_service_account_keys_async(
    transport: str = "grpc_asyncio", request_type=iam.ListServiceAccountKeysRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountKeysResponse()
        )
        response = await client.list_service_account_keys(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.ListServiceAccountKeysRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ListServiceAccountKeysResponse)


@pytest.mark.asyncio
async def test_list_service_account_keys_async_from_dict():
    await test_list_service_account_keys_async(request_type=dict)


def test_list_service_account_keys_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ListServiceAccountKeysRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        call.return_value = iam.ListServiceAccountKeysResponse()
        client.list_service_account_keys(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_list_service_account_keys_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.ListServiceAccountKeysRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountKeysResponse()
        )
        await client.list_service_account_keys(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_list_service_account_keys_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountKeysResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.list_service_account_keys(
            name="name_value",
            key_types=[iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED],
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].key_types
        mock_val = [iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED]
        assert arg == mock_val


def test_list_service_account_keys_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.list_service_account_keys(
            iam.ListServiceAccountKeysRequest(),
            name="name_value",
            key_types=[iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED],
        )


@pytest.mark.asyncio
async def test_list_service_account_keys_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_service_account_keys), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListServiceAccountKeysResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListServiceAccountKeysResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.list_service_account_keys(
            name="name_value",
            key_types=[iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED],
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].key_types
        mock_val = [iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED]
        assert arg == mock_val


@pytest.mark.asyncio
async def test_list_service_account_keys_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.list_service_account_keys(
            iam.ListServiceAccountKeysRequest(),
            name="name_value",
            key_types=[iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED],
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.GetServiceAccountKeyRequest,
        dict,
    ],
)
def test_get_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey(
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
            private_key_data=b"private_key_data_blob",
            public_key_data=b"public_key_data_blob",
            key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
            key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
            disabled=True,
        )
        response = client.get_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.GetServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


def test_get_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        client.get_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountKeyRequest()


def test_get_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.GetServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        client.get_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.get_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_get_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.GetServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.get_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.GetServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


@pytest.mark.asyncio
async def test_get_service_account_key_async_from_dict():
    await test_get_service_account_key_async(request_type=dict)


def test_get_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccountKey()
        client.get_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey()
        )
        await client.get_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_get_service_account_key_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.get_service_account_key(
            name="name_value",
            public_key_type=iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE,
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].public_key_type
        mock_val = iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE
        assert arg == mock_val


def test_get_service_account_key_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_service_account_key(
            iam.GetServiceAccountKeyRequest(),
            name="name_value",
            public_key_type=iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE,
        )


@pytest.mark.asyncio
async def test_get_service_account_key_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.get_service_account_key(
            name="name_value",
            public_key_type=iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE,
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].public_key_type
        mock_val = iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE
        assert arg == mock_val


@pytest.mark.asyncio
async def test_get_service_account_key_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.get_service_account_key(
            iam.GetServiceAccountKeyRequest(),
            name="name_value",
            public_key_type=iam.ServiceAccountPublicKeyType.TYPE_X509_PEM_FILE,
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.CreateServiceAccountKeyRequest,
        dict,
    ],
)
def test_create_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey(
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
            private_key_data=b"private_key_data_blob",
            public_key_data=b"public_key_data_blob",
            key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
            key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
            disabled=True,
        )
        response = client.create_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.CreateServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


def test_create_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        client.create_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountKeyRequest()


def test_create_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.CreateServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        client.create_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_create_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.create_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_create_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.CreateServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.create_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.CreateServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


@pytest.mark.asyncio
async def test_create_service_account_key_async_from_dict():
    await test_create_service_account_key_async(request_type=dict)


def test_create_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccountKey()
        client.create_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_create_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey()
        )
        await client.create_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_create_service_account_key_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.create_service_account_key(
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].private_key_type
        mock_val = iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
        assert arg == mock_val
        arg = args[0].key_algorithm
        mock_val = iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
        assert arg == mock_val


def test_create_service_account_key_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.create_service_account_key(
            iam.CreateServiceAccountKeyRequest(),
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
        )


@pytest.mark.asyncio
async def test_create_service_account_key_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.create_service_account_key(
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].private_key_type
        mock_val = iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
        assert arg == mock_val
        arg = args[0].key_algorithm
        mock_val = iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
        assert arg == mock_val


@pytest.mark.asyncio
async def test_create_service_account_key_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.create_service_account_key(
            iam.CreateServiceAccountKeyRequest(),
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.UploadServiceAccountKeyRequest,
        dict,
    ],
)
def test_upload_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ServiceAccountKey(
            name="name_value",
            private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
            key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
            private_key_data=b"private_key_data_blob",
            public_key_data=b"public_key_data_blob",
            key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
            key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
            disabled=True,
        )
        response = client.upload_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.UploadServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


def test_upload_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        client.upload_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UploadServiceAccountKeyRequest()


def test_upload_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.UploadServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        client.upload_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UploadServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_upload_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.upload_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UploadServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_upload_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.UploadServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey(
                name="name_value",
                private_key_type=iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE,
                key_algorithm=iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024,
                private_key_data=b"private_key_data_blob",
                public_key_data=b"public_key_data_blob",
                key_origin=iam.ServiceAccountKeyOrigin.USER_PROVIDED,
                key_type=iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED,
                disabled=True,
            )
        )
        response = await client.upload_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.UploadServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.ServiceAccountKey)
    assert response.name == "name_value"
    assert (
        response.private_key_type == iam.ServiceAccountPrivateKeyType.TYPE_PKCS12_FILE
    )
    assert response.key_algorithm == iam.ServiceAccountKeyAlgorithm.KEY_ALG_RSA_1024
    assert response.private_key_data == b"private_key_data_blob"
    assert response.public_key_data == b"public_key_data_blob"
    assert response.key_origin == iam.ServiceAccountKeyOrigin.USER_PROVIDED
    assert response.key_type == iam.ListServiceAccountKeysRequest.KeyType.USER_MANAGED
    assert response.disabled is True


@pytest.mark.asyncio
async def test_upload_service_account_key_async_from_dict():
    await test_upload_service_account_key_async(request_type=dict)


def test_upload_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UploadServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        call.return_value = iam.ServiceAccountKey()
        client.upload_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_upload_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UploadServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.upload_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ServiceAccountKey()
        )
        await client.upload_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.DeleteServiceAccountKeyRequest,
        dict,
    ],
)
def test_delete_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.delete_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_delete_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        client.delete_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountKeyRequest()


def test_delete_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.DeleteServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        client.delete_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_delete_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.delete_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_delete_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.DeleteServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.delete_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_delete_service_account_key_async_from_dict():
    await test_delete_service_account_key_async(request_type=dict)


def test_delete_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        call.return_value = None
        client.delete_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_delete_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.delete_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_delete_service_account_key_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.delete_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_delete_service_account_key_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.delete_service_account_key(
            iam.DeleteServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_delete_service_account_key_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.delete_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.delete_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_delete_service_account_key_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.delete_service_account_key(
            iam.DeleteServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.DisableServiceAccountKeyRequest,
        dict,
    ],
)
def test_disable_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.disable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.DisableServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_disable_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        client.disable_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountKeyRequest()


def test_disable_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.DisableServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        client.disable_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_disable_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.disable_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DisableServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_disable_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.DisableServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.disable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.DisableServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_disable_service_account_key_async_from_dict():
    await test_disable_service_account_key_async(request_type=dict)


def test_disable_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DisableServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        call.return_value = None
        client.disable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_disable_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DisableServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.disable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_disable_service_account_key_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.disable_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_disable_service_account_key_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.disable_service_account_key(
            iam.DisableServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_disable_service_account_key_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.disable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.disable_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_disable_service_account_key_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.disable_service_account_key(
            iam.DisableServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.EnableServiceAccountKeyRequest,
        dict,
    ],
)
def test_enable_service_account_key(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        response = client.enable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.EnableServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


def test_enable_service_account_key_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        client.enable_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountKeyRequest()


def test_enable_service_account_key_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.EnableServiceAccountKeyRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        client.enable_service_account_key(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountKeyRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_enable_service_account_key_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.enable_service_account_key()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.EnableServiceAccountKeyRequest()


@pytest.mark.asyncio
async def test_enable_service_account_key_async(
    transport: str = "grpc_asyncio", request_type=iam.EnableServiceAccountKeyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        response = await client.enable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.EnableServiceAccountKeyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert response is None


@pytest.mark.asyncio
async def test_enable_service_account_key_async_from_dict():
    await test_enable_service_account_key_async(request_type=dict)


def test_enable_service_account_key_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.EnableServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        call.return_value = None
        client.enable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_enable_service_account_key_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.EnableServiceAccountKeyRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        await client.enable_service_account_key(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_enable_service_account_key_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.enable_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_enable_service_account_key_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.enable_service_account_key(
            iam.EnableServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_enable_service_account_key_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.enable_service_account_key), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = None

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(None)
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.enable_service_account_key(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_enable_service_account_key_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.enable_service_account_key(
            iam.EnableServiceAccountKeyRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.SignBlobRequest,
        dict,
    ],
)
def test_sign_blob(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignBlobResponse(
            key_id="key_id_value",
            signature=b"signature_blob",
        )
        response = client.sign_blob(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.SignBlobRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.SignBlobResponse)
    assert response.key_id == "key_id_value"
    assert response.signature == b"signature_blob"


def test_sign_blob_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        client.sign_blob()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignBlobRequest()


def test_sign_blob_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.SignBlobRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        client.sign_blob(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignBlobRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_sign_blob_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignBlobResponse(
                key_id="key_id_value",
                signature=b"signature_blob",
            )
        )
        response = await client.sign_blob()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignBlobRequest()


@pytest.mark.asyncio
async def test_sign_blob_async(
    transport: str = "grpc_asyncio", request_type=iam.SignBlobRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignBlobResponse(
                key_id="key_id_value",
                signature=b"signature_blob",
            )
        )
        response = await client.sign_blob(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.SignBlobRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.SignBlobResponse)
    assert response.key_id == "key_id_value"
    assert response.signature == b"signature_blob"


@pytest.mark.asyncio
async def test_sign_blob_async_from_dict():
    await test_sign_blob_async(request_type=dict)


def test_sign_blob_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.SignBlobRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        call.return_value = iam.SignBlobResponse()
        client.sign_blob(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_sign_blob_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.SignBlobRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignBlobResponse()
        )
        await client.sign_blob(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_sign_blob_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignBlobResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.sign_blob(
            name="name_value",
            bytes_to_sign=b"bytes_to_sign_blob",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].bytes_to_sign
        mock_val = b"bytes_to_sign_blob"
        assert arg == mock_val


def test_sign_blob_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.sign_blob(
            iam.SignBlobRequest(),
            name="name_value",
            bytes_to_sign=b"bytes_to_sign_blob",
        )


@pytest.mark.asyncio
async def test_sign_blob_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_blob), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignBlobResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignBlobResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.sign_blob(
            name="name_value",
            bytes_to_sign=b"bytes_to_sign_blob",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].bytes_to_sign
        mock_val = b"bytes_to_sign_blob"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_sign_blob_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.sign_blob(
            iam.SignBlobRequest(),
            name="name_value",
            bytes_to_sign=b"bytes_to_sign_blob",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.SignJwtRequest,
        dict,
    ],
)
def test_sign_jwt(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignJwtResponse(
            key_id="key_id_value",
            signed_jwt="signed_jwt_value",
        )
        response = client.sign_jwt(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.SignJwtRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.SignJwtResponse)
    assert response.key_id == "key_id_value"
    assert response.signed_jwt == "signed_jwt_value"


def test_sign_jwt_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        client.sign_jwt()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignJwtRequest()


def test_sign_jwt_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.SignJwtRequest(
        name="name_value",
        payload="payload_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        client.sign_jwt(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignJwtRequest(
            name="name_value",
            payload="payload_value",
        )


@pytest.mark.asyncio
async def test_sign_jwt_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignJwtResponse(
                key_id="key_id_value",
                signed_jwt="signed_jwt_value",
            )
        )
        response = await client.sign_jwt()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.SignJwtRequest()


@pytest.mark.asyncio
async def test_sign_jwt_async(
    transport: str = "grpc_asyncio", request_type=iam.SignJwtRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.SignJwtResponse(
                key_id="key_id_value",
                signed_jwt="signed_jwt_value",
            )
        )
        response = await client.sign_jwt(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.SignJwtRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.SignJwtResponse)
    assert response.key_id == "key_id_value"
    assert response.signed_jwt == "signed_jwt_value"


@pytest.mark.asyncio
async def test_sign_jwt_async_from_dict():
    await test_sign_jwt_async(request_type=dict)


def test_sign_jwt_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.SignJwtRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        call.return_value = iam.SignJwtResponse()
        client.sign_jwt(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_sign_jwt_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.SignJwtRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.SignJwtResponse())
        await client.sign_jwt(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_sign_jwt_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignJwtResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.sign_jwt(
            name="name_value",
            payload="payload_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].payload
        mock_val = "payload_value"
        assert arg == mock_val


def test_sign_jwt_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.sign_jwt(
            iam.SignJwtRequest(),
            name="name_value",
            payload="payload_value",
        )


@pytest.mark.asyncio
async def test_sign_jwt_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.sign_jwt), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.SignJwtResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.SignJwtResponse())
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.sign_jwt(
            name="name_value",
            payload="payload_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val
        arg = args[0].payload
        mock_val = "payload_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_sign_jwt_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.sign_jwt(
            iam.SignJwtRequest(),
            name="name_value",
            payload="payload_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.GetIamPolicyRequest,
        dict,
    ],
)
def test_get_iam_policy(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy(
            version=774,
            etag=b"etag_blob",
        )
        response = client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.GetIamPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)
    assert response.version == 774
    assert response.etag == b"etag_blob"


def test_get_iam_policy_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        client.get_iam_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.GetIamPolicyRequest()


def test_get_iam_policy_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam_policy_pb2.GetIamPolicyRequest(
        resource="resource_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        client.get_iam_policy(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.GetIamPolicyRequest(
            resource="resource_value",
        )


@pytest.mark.asyncio
async def test_get_iam_policy_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )
        response = await client.get_iam_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.GetIamPolicyRequest()


@pytest.mark.asyncio
async def test_get_iam_policy_async(
    transport: str = "grpc_asyncio", request_type=iam_policy_pb2.GetIamPolicyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )
        response = await client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.GetIamPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)
    assert response.version == 774
    assert response.etag == b"etag_blob"


@pytest.mark.asyncio
async def test_get_iam_policy_async_from_dict():
    await test_get_iam_policy_async(request_type=dict)


def test_get_iam_policy_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.GetIamPolicyRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        call.return_value = policy_pb2.Policy()
        client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_iam_policy_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.GetIamPolicyRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())
        await client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


def test_get_iam_policy_from_dict_foreign():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()
        response = client.get_iam_policy(
            request={
                "resource": "resource_value",
                "options": options_pb2.GetPolicyOptions(requested_policy_version=2598),
            }
        )
        call.assert_called()


def test_get_iam_policy_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.get_iam_policy(
            resource="resource_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val


def test_get_iam_policy_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_iam_policy(
            iam_policy_pb2.GetIamPolicyRequest(),
            resource="resource_value",
        )


@pytest.mark.asyncio
async def test_get_iam_policy_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.get_iam_policy(
            resource="resource_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_get_iam_policy_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.get_iam_policy(
            iam_policy_pb2.GetIamPolicyRequest(),
            resource="resource_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.SetIamPolicyRequest,
        dict,
    ],
)
def test_set_iam_policy(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy(
            version=774,
            etag=b"etag_blob",
        )
        response = client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.SetIamPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)
    assert response.version == 774
    assert response.etag == b"etag_blob"


def test_set_iam_policy_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        client.set_iam_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.SetIamPolicyRequest()


def test_set_iam_policy_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam_policy_pb2.SetIamPolicyRequest(
        resource="resource_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        client.set_iam_policy(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.SetIamPolicyRequest(
            resource="resource_value",
        )


@pytest.mark.asyncio
async def test_set_iam_policy_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )
        response = await client.set_iam_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.SetIamPolicyRequest()


@pytest.mark.asyncio
async def test_set_iam_policy_async(
    transport: str = "grpc_asyncio", request_type=iam_policy_pb2.SetIamPolicyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )
        response = await client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.SetIamPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)
    assert response.version == 774
    assert response.etag == b"etag_blob"


@pytest.mark.asyncio
async def test_set_iam_policy_async_from_dict():
    await test_set_iam_policy_async(request_type=dict)


def test_set_iam_policy_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.SetIamPolicyRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        call.return_value = policy_pb2.Policy()
        client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_set_iam_policy_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.SetIamPolicyRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())
        await client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


def test_set_iam_policy_from_dict_foreign():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()
        response = client.set_iam_policy(
            request={
                "resource": "resource_value",
                "policy": policy_pb2.Policy(version=774),
                "update_mask": field_mask_pb2.FieldMask(paths=["paths_value"]),
            }
        )
        call.assert_called()


def test_set_iam_policy_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.set_iam_policy(
            resource="resource_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val


def test_set_iam_policy_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.set_iam_policy(
            iam_policy_pb2.SetIamPolicyRequest(),
            resource="resource_value",
        )


@pytest.mark.asyncio
async def test_set_iam_policy_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.set_iam_policy(
            resource="resource_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_set_iam_policy_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.set_iam_policy(
            iam_policy_pb2.SetIamPolicyRequest(),
            resource="resource_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.TestIamPermissionsRequest,
        dict,
    ],
)
def test_test_iam_permissions(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse(
            permissions=["permissions_value"],
        )
        response = client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.TestIamPermissionsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam_policy_pb2.TestIamPermissionsResponse)
    assert response.permissions == ["permissions_value"]


def test_test_iam_permissions_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        client.test_iam_permissions()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.TestIamPermissionsRequest()


def test_test_iam_permissions_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam_policy_pb2.TestIamPermissionsRequest(
        resource="resource_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        client.test_iam_permissions(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.TestIamPermissionsRequest(
            resource="resource_value",
        )


@pytest.mark.asyncio
async def test_test_iam_permissions_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse(
                permissions=["permissions_value"],
            )
        )
        response = await client.test_iam_permissions()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam_policy_pb2.TestIamPermissionsRequest()


@pytest.mark.asyncio
async def test_test_iam_permissions_async(
    transport: str = "grpc_asyncio",
    request_type=iam_policy_pb2.TestIamPermissionsRequest,
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse(
                permissions=["permissions_value"],
            )
        )
        response = await client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam_policy_pb2.TestIamPermissionsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam_policy_pb2.TestIamPermissionsResponse)
    assert response.permissions == ["permissions_value"]


@pytest.mark.asyncio
async def test_test_iam_permissions_async_from_dict():
    await test_test_iam_permissions_async(request_type=dict)


def test_test_iam_permissions_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.TestIamPermissionsRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()
        client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_test_iam_permissions_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.TestIamPermissionsRequest()

    request.resource = "resource_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse()
        )
        await client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource_value",
    ) in kw["metadata"]


def test_test_iam_permissions_from_dict_foreign():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()
        response = client.test_iam_permissions(
            request={
                "resource": "resource_value",
                "permissions": ["permissions_value"],
            }
        )
        call.assert_called()


def test_test_iam_permissions_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.test_iam_permissions(
            resource="resource_value",
            permissions=["permissions_value"],
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val
        arg = args[0].permissions
        mock_val = ["permissions_value"]
        assert arg == mock_val


def test_test_iam_permissions_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.test_iam_permissions(
            iam_policy_pb2.TestIamPermissionsRequest(),
            resource="resource_value",
            permissions=["permissions_value"],
        )


@pytest.mark.asyncio
async def test_test_iam_permissions_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.test_iam_permissions(
            resource="resource_value",
            permissions=["permissions_value"],
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].resource
        mock_val = "resource_value"
        assert arg == mock_val
        arg = args[0].permissions
        mock_val = ["permissions_value"]
        assert arg == mock_val


@pytest.mark.asyncio
async def test_test_iam_permissions_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.test_iam_permissions(
            iam_policy_pb2.TestIamPermissionsRequest(),
            resource="resource_value",
            permissions=["permissions_value"],
        )


@pytest.mark.parametrize(
    "request_type",
    [
        iam.QueryGrantableRolesRequest,
        dict,
    ],
)
def test_query_grantable_roles(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.QueryGrantableRolesResponse(
            next_page_token="next_page_token_value",
        )
        response = client.query_grantable_roles(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.QueryGrantableRolesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.QueryGrantableRolesPager)
    assert response.next_page_token == "next_page_token_value"


def test_query_grantable_roles_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        client.query_grantable_roles()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryGrantableRolesRequest()


def test_query_grantable_roles_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.QueryGrantableRolesRequest(
        full_resource_name="full_resource_name_value",
        page_token="page_token_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        client.query_grantable_roles(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryGrantableRolesRequest(
            full_resource_name="full_resource_name_value",
            page_token="page_token_value",
        )


@pytest.mark.asyncio
async def test_query_grantable_roles_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryGrantableRolesResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.query_grantable_roles()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryGrantableRolesRequest()


@pytest.mark.asyncio
async def test_query_grantable_roles_async(
    transport: str = "grpc_asyncio", request_type=iam.QueryGrantableRolesRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryGrantableRolesResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.query_grantable_roles(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.QueryGrantableRolesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.QueryGrantableRolesAsyncPager)
    assert response.next_page_token == "next_page_token_value"


@pytest.mark.asyncio
async def test_query_grantable_roles_async_from_dict():
    await test_query_grantable_roles_async(request_type=dict)


def test_query_grantable_roles_flattened():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.QueryGrantableRolesResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.query_grantable_roles(
            full_resource_name="full_resource_name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].full_resource_name
        mock_val = "full_resource_name_value"
        assert arg == mock_val


def test_query_grantable_roles_flattened_error():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.query_grantable_roles(
            iam.QueryGrantableRolesRequest(),
            full_resource_name="full_resource_name_value",
        )


@pytest.mark.asyncio
async def test_query_grantable_roles_flattened_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.QueryGrantableRolesResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryGrantableRolesResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.query_grantable_roles(
            full_resource_name="full_resource_name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].full_resource_name
        mock_val = "full_resource_name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_query_grantable_roles_flattened_error_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.query_grantable_roles(
            iam.QueryGrantableRolesRequest(),
            full_resource_name="full_resource_name_value",
        )


def test_query_grantable_roles_pager(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )

        metadata = ()
        pager = client.query_grantable_roles(request={})

        assert pager._metadata == metadata

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, iam.Role) for i in results)


def test_query_grantable_roles_pages(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        pages = list(client.query_grantable_roles(request={}).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.asyncio
async def test_query_grantable_roles_async_pager():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        async_pager = await client.query_grantable_roles(
            request={},
        )
        assert async_pager.next_page_token == "abc"
        responses = []
        async for response in async_pager:  # pragma: no branch
            responses.append(response)

        assert len(responses) == 6
        assert all(isinstance(i, iam.Role) for i in responses)


@pytest.mark.asyncio
async def test_query_grantable_roles_async_pages():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_grantable_roles),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryGrantableRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        pages = []
        # Workaround issue in python 3.9 related to code coverage by adding `# pragma: no branch`
        # See https://github.com/googleapis/gapic-generator-python/pull/1174#issuecomment-1025132372
        async for page_ in (  # pragma: no branch
            await client.query_grantable_roles(request={})
        ).pages:
            pages.append(page_)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        iam.ListRolesRequest,
        dict,
    ],
)
def test_list_roles(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.ListRolesResponse(
            next_page_token="next_page_token_value",
        )
        response = client.list_roles(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.ListRolesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListRolesPager)
    assert response.next_page_token == "next_page_token_value"


def test_list_roles_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        client.list_roles()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListRolesRequest()


def test_list_roles_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.ListRolesRequest(
        parent="parent_value",
        page_token="page_token_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        client.list_roles(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListRolesRequest(
            parent="parent_value",
            page_token="page_token_value",
        )


@pytest.mark.asyncio
async def test_list_roles_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListRolesResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.list_roles()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.ListRolesRequest()


@pytest.mark.asyncio
async def test_list_roles_async(
    transport: str = "grpc_asyncio", request_type=iam.ListRolesRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.ListRolesResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.list_roles(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.ListRolesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListRolesAsyncPager)
    assert response.next_page_token == "next_page_token_value"


@pytest.mark.asyncio
async def test_list_roles_async_from_dict():
    await test_list_roles_async(request_type=dict)


def test_list_roles_pager(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.ListRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )

        metadata = ()
        pager = client.list_roles(request={})

        assert pager._metadata == metadata

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, iam.Role) for i in results)


def test_list_roles_pages(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_roles), "__call__") as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.ListRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        pages = list(client.list_roles(request={}).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.asyncio
async def test_list_roles_async_pager():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_roles), "__call__", new_callable=mock.AsyncMock
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.ListRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        async_pager = await client.list_roles(
            request={},
        )
        assert async_pager.next_page_token == "abc"
        responses = []
        async for response in async_pager:  # pragma: no branch
            responses.append(response)

        assert len(responses) == 6
        assert all(isinstance(i, iam.Role) for i in responses)


@pytest.mark.asyncio
async def test_list_roles_async_pages():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_roles), "__call__", new_callable=mock.AsyncMock
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                    iam.Role(),
                ],
                next_page_token="abc",
            ),
            iam.ListRolesResponse(
                roles=[],
                next_page_token="def",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                ],
                next_page_token="ghi",
            ),
            iam.ListRolesResponse(
                roles=[
                    iam.Role(),
                    iam.Role(),
                ],
            ),
            RuntimeError,
        )
        pages = []
        # Workaround issue in python 3.9 related to code coverage by adding `# pragma: no branch`
        # See https://github.com/googleapis/gapic-generator-python/pull/1174#issuecomment-1025132372
        async for page_ in (  # pragma: no branch
            await client.list_roles(request={})
        ).pages:
            pages.append(page_)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        iam.GetRoleRequest,
        dict,
    ],
)
def test_get_role(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.Role(
            name="name_value",
            title="title_value",
            description="description_value",
            included_permissions=["included_permissions_value"],
            stage=iam.Role.RoleLaunchStage.BETA,
            etag=b"etag_blob",
            deleted=True,
        )
        response = client.get_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.GetRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


def test_get_role_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        client.get_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetRoleRequest()


def test_get_role_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.GetRoleRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        client.get_role(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetRoleRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_role_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.get_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.GetRoleRequest()


@pytest.mark.asyncio
async def test_get_role_async(
    transport: str = "grpc_asyncio", request_type=iam.GetRoleRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.get_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.GetRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


@pytest.mark.asyncio
async def test_get_role_async_from_dict():
    await test_get_role_async(request_type=dict)


def test_get_role_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        call.return_value = iam.Role()
        client.get_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_role_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.GetRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_role), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.Role())
        await client.get_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.CreateRoleRequest,
        dict,
    ],
)
def test_create_role(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.Role(
            name="name_value",
            title="title_value",
            description="description_value",
            included_permissions=["included_permissions_value"],
            stage=iam.Role.RoleLaunchStage.BETA,
            etag=b"etag_blob",
            deleted=True,
        )
        response = client.create_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.CreateRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


def test_create_role_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        client.create_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateRoleRequest()


def test_create_role_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.CreateRoleRequest(
        parent="parent_value",
        role_id="role_id_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        client.create_role(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateRoleRequest(
            parent="parent_value",
            role_id="role_id_value",
        )


@pytest.mark.asyncio
async def test_create_role_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.create_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.CreateRoleRequest()


@pytest.mark.asyncio
async def test_create_role_async(
    transport: str = "grpc_asyncio", request_type=iam.CreateRoleRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.create_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.CreateRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


@pytest.mark.asyncio
async def test_create_role_async_from_dict():
    await test_create_role_async(request_type=dict)


def test_create_role_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateRoleRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        call.return_value = iam.Role()
        client.create_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_create_role_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.CreateRoleRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.create_role), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.Role())
        await client.create_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.UpdateRoleRequest,
        dict,
    ],
)
def test_update_role(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.Role(
            name="name_value",
            title="title_value",
            description="description_value",
            included_permissions=["included_permissions_value"],
            stage=iam.Role.RoleLaunchStage.BETA,
            etag=b"etag_blob",
            deleted=True,
        )
        response = client.update_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.UpdateRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


def test_update_role_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        client.update_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UpdateRoleRequest()


def test_update_role_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.UpdateRoleRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        client.update_role(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UpdateRoleRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_update_role_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.update_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UpdateRoleRequest()


@pytest.mark.asyncio
async def test_update_role_async(
    transport: str = "grpc_asyncio", request_type=iam.UpdateRoleRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.update_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.UpdateRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


@pytest.mark.asyncio
async def test_update_role_async_from_dict():
    await test_update_role_async(request_type=dict)


def test_update_role_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UpdateRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        call.return_value = iam.Role()
        client.update_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_update_role_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UpdateRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.update_role), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.Role())
        await client.update_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.DeleteRoleRequest,
        dict,
    ],
)
def test_delete_role(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.Role(
            name="name_value",
            title="title_value",
            description="description_value",
            included_permissions=["included_permissions_value"],
            stage=iam.Role.RoleLaunchStage.BETA,
            etag=b"etag_blob",
            deleted=True,
        )
        response = client.delete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


def test_delete_role_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        client.delete_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteRoleRequest()


def test_delete_role_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.DeleteRoleRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        client.delete_role(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteRoleRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_delete_role_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.delete_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.DeleteRoleRequest()


@pytest.mark.asyncio
async def test_delete_role_async(
    transport: str = "grpc_asyncio", request_type=iam.DeleteRoleRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.delete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.DeleteRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


@pytest.mark.asyncio
async def test_delete_role_async_from_dict():
    await test_delete_role_async(request_type=dict)


def test_delete_role_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        call.return_value = iam.Role()
        client.delete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_delete_role_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.DeleteRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.delete_role), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.Role())
        await client.delete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.UndeleteRoleRequest,
        dict,
    ],
)
def test_undelete_role(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.Role(
            name="name_value",
            title="title_value",
            description="description_value",
            included_permissions=["included_permissions_value"],
            stage=iam.Role.RoleLaunchStage.BETA,
            etag=b"etag_blob",
            deleted=True,
        )
        response = client.undelete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.UndeleteRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


def test_undelete_role_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        client.undelete_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteRoleRequest()


def test_undelete_role_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.UndeleteRoleRequest(
        name="name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        client.undelete_role(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteRoleRequest(
            name="name_value",
        )


@pytest.mark.asyncio
async def test_undelete_role_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.undelete_role()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.UndeleteRoleRequest()


@pytest.mark.asyncio
async def test_undelete_role_async(
    transport: str = "grpc_asyncio", request_type=iam.UndeleteRoleRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.Role(
                name="name_value",
                title="title_value",
                description="description_value",
                included_permissions=["included_permissions_value"],
                stage=iam.Role.RoleLaunchStage.BETA,
                etag=b"etag_blob",
                deleted=True,
            )
        )
        response = await client.undelete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.UndeleteRoleRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.Role)
    assert response.name == "name_value"
    assert response.title == "title_value"
    assert response.description == "description_value"
    assert response.included_permissions == ["included_permissions_value"]
    assert response.stage == iam.Role.RoleLaunchStage.BETA
    assert response.etag == b"etag_blob"
    assert response.deleted is True


@pytest.mark.asyncio
async def test_undelete_role_async_from_dict():
    await test_undelete_role_async(request_type=dict)


def test_undelete_role_field_headers():
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UndeleteRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        call.return_value = iam.Role()
        client.undelete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_undelete_role_field_headers_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam.UndeleteRoleRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.undelete_role), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(iam.Role())
        await client.undelete_role(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.parametrize(
    "request_type",
    [
        iam.QueryTestablePermissionsRequest,
        dict,
    ],
)
def test_query_testable_permissions(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.QueryTestablePermissionsResponse(
            next_page_token="next_page_token_value",
        )
        response = client.query_testable_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.QueryTestablePermissionsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.QueryTestablePermissionsPager)
    assert response.next_page_token == "next_page_token_value"


def test_query_testable_permissions_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        client.query_testable_permissions()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryTestablePermissionsRequest()


def test_query_testable_permissions_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.QueryTestablePermissionsRequest(
        full_resource_name="full_resource_name_value",
        page_token="page_token_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        client.query_testable_permissions(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryTestablePermissionsRequest(
            full_resource_name="full_resource_name_value",
            page_token="page_token_value",
        )


@pytest.mark.asyncio
async def test_query_testable_permissions_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryTestablePermissionsResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.query_testable_permissions()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryTestablePermissionsRequest()


@pytest.mark.asyncio
async def test_query_testable_permissions_async(
    transport: str = "grpc_asyncio", request_type=iam.QueryTestablePermissionsRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryTestablePermissionsResponse(
                next_page_token="next_page_token_value",
            )
        )
        response = await client.query_testable_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.QueryTestablePermissionsRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.QueryTestablePermissionsAsyncPager)
    assert response.next_page_token == "next_page_token_value"


@pytest.mark.asyncio
async def test_query_testable_permissions_async_from_dict():
    await test_query_testable_permissions_async(request_type=dict)


def test_query_testable_permissions_pager(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                    iam.Permission(),
                ],
                next_page_token="abc",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[],
                next_page_token="def",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                ],
            ),
            RuntimeError,
        )

        metadata = ()
        pager = client.query_testable_permissions(request={})

        assert pager._metadata == metadata

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, iam.Permission) for i in results)


def test_query_testable_permissions_pages(transport_name: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                    iam.Permission(),
                ],
                next_page_token="abc",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[],
                next_page_token="def",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                ],
            ),
            RuntimeError,
        )
        pages = list(client.query_testable_permissions(request={}).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.asyncio
async def test_query_testable_permissions_async_pager():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                    iam.Permission(),
                ],
                next_page_token="abc",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[],
                next_page_token="def",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                ],
            ),
            RuntimeError,
        )
        async_pager = await client.query_testable_permissions(
            request={},
        )
        assert async_pager.next_page_token == "abc"
        responses = []
        async for response in async_pager:  # pragma: no branch
            responses.append(response)

        assert len(responses) == 6
        assert all(isinstance(i, iam.Permission) for i in responses)


@pytest.mark.asyncio
async def test_query_testable_permissions_async_pages():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_testable_permissions),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                    iam.Permission(),
                ],
                next_page_token="abc",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[],
                next_page_token="def",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                ],
                next_page_token="ghi",
            ),
            iam.QueryTestablePermissionsResponse(
                permissions=[
                    iam.Permission(),
                    iam.Permission(),
                ],
            ),
            RuntimeError,
        )
        pages = []
        # Workaround issue in python 3.9 related to code coverage by adding `# pragma: no branch`
        # See https://github.com/googleapis/gapic-generator-python/pull/1174#issuecomment-1025132372
        async for page_ in (  # pragma: no branch
            await client.query_testable_permissions(request={})
        ).pages:
            pages.append(page_)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        iam.QueryAuditableServicesRequest,
        dict,
    ],
)
def test_query_auditable_services(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_auditable_services), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.QueryAuditableServicesResponse()
        response = client.query_auditable_services(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.QueryAuditableServicesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.QueryAuditableServicesResponse)


def test_query_auditable_services_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_auditable_services), "__call__"
    ) as call:
        client.query_auditable_services()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryAuditableServicesRequest()


def test_query_auditable_services_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.QueryAuditableServicesRequest(
        full_resource_name="full_resource_name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_auditable_services), "__call__"
    ) as call:
        client.query_auditable_services(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryAuditableServicesRequest(
            full_resource_name="full_resource_name_value",
        )


@pytest.mark.asyncio
async def test_query_auditable_services_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_auditable_services), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryAuditableServicesResponse()
        )
        response = await client.query_auditable_services()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.QueryAuditableServicesRequest()


@pytest.mark.asyncio
async def test_query_auditable_services_async(
    transport: str = "grpc_asyncio", request_type=iam.QueryAuditableServicesRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.query_auditable_services), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.QueryAuditableServicesResponse()
        )
        response = await client.query_auditable_services(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.QueryAuditableServicesRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.QueryAuditableServicesResponse)


@pytest.mark.asyncio
async def test_query_auditable_services_async_from_dict():
    await test_query_auditable_services_async(request_type=dict)


@pytest.mark.parametrize(
    "request_type",
    [
        iam.LintPolicyRequest,
        dict,
    ],
)
def test_lint_policy(request_type, transport: str = "grpc"):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.lint_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam.LintPolicyResponse()
        response = client.lint_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        request = iam.LintPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.LintPolicyResponse)


def test_lint_policy_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.lint_policy), "__call__") as call:
        client.lint_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.LintPolicyRequest()


def test_lint_policy_non_empty_request_with_auto_populated_field():
    # This test is a coverage failsafe to make sure that UUID4 fields are
    # automatically populated, according to AIP-4235, with non-empty requests.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc",
    )

    # Populate all string fields in the request which are not UUID4
    # since we want to check that UUID4 are populated automatically
    # if they meet the requirements of AIP 4235.
    request = iam.LintPolicyRequest(
        full_resource_name="full_resource_name_value",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.lint_policy), "__call__") as call:
        client.lint_policy(request=request)
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.LintPolicyRequest(
            full_resource_name="full_resource_name_value",
        )


@pytest.mark.asyncio
async def test_lint_policy_empty_call_async():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.lint_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.LintPolicyResponse()
        )
        response = await client.lint_policy()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == iam.LintPolicyRequest()


@pytest.mark.asyncio
async def test_lint_policy_async(
    transport: str = "grpc_asyncio", request_type=iam.LintPolicyRequest
):
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.lint_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam.LintPolicyResponse()
        )
        response = await client.lint_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        request = iam.LintPolicyRequest()
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam.LintPolicyResponse)


@pytest.mark.asyncio
async def test_lint_policy_async_from_dict():
    await test_lint_policy_async(request_type=dict)


def test_credentials_transport_error():
    # It is an error to provide credentials and a transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    with pytest.raises(ValueError):
        client = IAMClient(
            credentials=ga_credentials.AnonymousCredentials(),
            transport=transport,
        )

    # It is an error to provide a credentials file and a transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    with pytest.raises(ValueError):
        client = IAMClient(
            client_options={"credentials_file": "credentials.json"},
            transport=transport,
        )

    # It is an error to provide an api_key and a transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    options = client_options.ClientOptions()
    options.api_key = "api_key"
    with pytest.raises(ValueError):
        client = IAMClient(
            client_options=options,
            transport=transport,
        )

    # It is an error to provide an api_key and a credential.
    options = client_options.ClientOptions()
    options.api_key = "api_key"
    with pytest.raises(ValueError):
        client = IAMClient(
            client_options=options, credentials=ga_credentials.AnonymousCredentials()
        )

    # It is an error to provide scopes and a transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    with pytest.raises(ValueError):
        client = IAMClient(
            client_options={"scopes": ["1", "2"]},
            transport=transport,
        )


def test_transport_instance():
    # A client may be instantiated with a custom transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    client = IAMClient(transport=transport)
    assert client.transport is transport


def test_transport_get_channel():
    # A client may be instantiated with a custom transport instance.
    transport = transports.IAMGrpcTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    channel = transport.grpc_channel
    assert channel

    transport = transports.IAMGrpcAsyncIOTransport(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    channel = transport.grpc_channel
    assert channel


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.IAMGrpcTransport,
        transports.IAMGrpcAsyncIOTransport,
    ],
)
def test_transport_adc(transport_class):
    # Test default credentials are used if not provided.
    with mock.patch.object(google.auth, "default") as adc:
        adc.return_value = (ga_credentials.AnonymousCredentials(), None)
        transport_class()
        adc.assert_called_once()


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
    ],
)
def test_transport_kind(transport_name):
    transport = IAMClient.get_transport_class(transport_name)(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    assert transport.kind == transport_name


def test_transport_grpc_default():
    # A client should use the gRPC transport by default.
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
    )
    assert isinstance(
        client.transport,
        transports.IAMGrpcTransport,
    )


def test_iam_base_transport_error():
    # Passing both a credentials object and credentials_file should raise an error
    with pytest.raises(core_exceptions.DuplicateCredentialArgs):
        transport = transports.IAMTransport(
            credentials=ga_credentials.AnonymousCredentials(),
            credentials_file="credentials.json",
        )


def test_iam_base_transport():
    # Instantiate the base transport.
    with mock.patch(
        "google.cloud.iam_admin_v1.services.iam.transports.IAMTransport.__init__"
    ) as Transport:
        Transport.return_value = None
        transport = transports.IAMTransport(
            credentials=ga_credentials.AnonymousCredentials(),
        )

    # Every method on the transport should just blindly
    # raise NotImplementedError.
    methods = (
        "list_service_accounts",
        "get_service_account",
        "create_service_account",
        "update_service_account",
        "patch_service_account",
        "delete_service_account",
        "undelete_service_account",
        "enable_service_account",
        "disable_service_account",
        "list_service_account_keys",
        "get_service_account_key",
        "create_service_account_key",
        "upload_service_account_key",
        "delete_service_account_key",
        "disable_service_account_key",
        "enable_service_account_key",
        "sign_blob",
        "sign_jwt",
        "get_iam_policy",
        "set_iam_policy",
        "test_iam_permissions",
        "query_grantable_roles",
        "list_roles",
        "get_role",
        "create_role",
        "update_role",
        "delete_role",
        "undelete_role",
        "query_testable_permissions",
        "query_auditable_services",
        "lint_policy",
    )
    for method in methods:
        with pytest.raises(NotImplementedError):
            getattr(transport, method)(request=object())

    with pytest.raises(NotImplementedError):
        transport.close()

    # Catch all for all remaining methods and properties
    remainder = [
        "kind",
    ]
    for r in remainder:
        with pytest.raises(NotImplementedError):
            getattr(transport, r)()


def test_iam_base_transport_with_credentials_file():
    # Instantiate the base transport with a credentials file
    with mock.patch.object(
        google.auth, "load_credentials_from_file", autospec=True
    ) as load_creds, mock.patch(
        "google.cloud.iam_admin_v1.services.iam.transports.IAMTransport._prep_wrapped_messages"
    ) as Transport:
        Transport.return_value = None
        load_creds.return_value = (ga_credentials.AnonymousCredentials(), None)
        transport = transports.IAMTransport(
            credentials_file="credentials.json",
            quota_project_id="octopus",
        )
        load_creds.assert_called_once_with(
            "credentials.json",
            scopes=None,
            default_scopes=("https://www.googleapis.com/auth/cloud-platform",),
            quota_project_id="octopus",
        )


def test_iam_base_transport_with_adc():
    # Test the default credentials are used if credentials and credentials_file are None.
    with mock.patch.object(google.auth, "default", autospec=True) as adc, mock.patch(
        "google.cloud.iam_admin_v1.services.iam.transports.IAMTransport._prep_wrapped_messages"
    ) as Transport:
        Transport.return_value = None
        adc.return_value = (ga_credentials.AnonymousCredentials(), None)
        transport = transports.IAMTransport()
        adc.assert_called_once()


def test_iam_auth_adc():
    # If no credentials are provided, we should use ADC credentials.
    with mock.patch.object(google.auth, "default", autospec=True) as adc:
        adc.return_value = (ga_credentials.AnonymousCredentials(), None)
        IAMClient()
        adc.assert_called_once_with(
            scopes=None,
            default_scopes=("https://www.googleapis.com/auth/cloud-platform",),
            quota_project_id=None,
        )


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.IAMGrpcTransport,
        transports.IAMGrpcAsyncIOTransport,
    ],
)
def test_iam_transport_auth_adc(transport_class):
    # If credentials and host are not provided, the transport class should use
    # ADC credentials.
    with mock.patch.object(google.auth, "default", autospec=True) as adc:
        adc.return_value = (ga_credentials.AnonymousCredentials(), None)
        transport_class(quota_project_id="octopus", scopes=["1", "2"])
        adc.assert_called_once_with(
            scopes=["1", "2"],
            default_scopes=("https://www.googleapis.com/auth/cloud-platform",),
            quota_project_id="octopus",
        )


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.IAMGrpcTransport,
        transports.IAMGrpcAsyncIOTransport,
    ],
)
def test_iam_transport_auth_gdch_credentials(transport_class):
    host = "https://language.com"
    api_audience_tests = [None, "https://language2.com"]
    api_audience_expect = [host, "https://language2.com"]
    for t, e in zip(api_audience_tests, api_audience_expect):
        with mock.patch.object(google.auth, "default", autospec=True) as adc:
            gdch_mock = mock.MagicMock()
            type(gdch_mock).with_gdch_audience = mock.PropertyMock(
                return_value=gdch_mock
            )
            adc.return_value = (gdch_mock, None)
            transport_class(host=host, api_audience=t)
            gdch_mock.with_gdch_audience.assert_called_once_with(e)


@pytest.mark.parametrize(
    "transport_class,grpc_helpers",
    [
        (transports.IAMGrpcTransport, grpc_helpers),
        (transports.IAMGrpcAsyncIOTransport, grpc_helpers_async),
    ],
)
def test_iam_transport_create_channel(transport_class, grpc_helpers):
    # If credentials and host are not provided, the transport class should use
    # ADC credentials.
    with mock.patch.object(
        google.auth, "default", autospec=True
    ) as adc, mock.patch.object(
        grpc_helpers, "create_channel", autospec=True
    ) as create_channel:
        creds = ga_credentials.AnonymousCredentials()
        adc.return_value = (creds, None)
        transport_class(quota_project_id="octopus", scopes=["1", "2"])

        create_channel.assert_called_with(
            "iam.googleapis.com:443",
            credentials=creds,
            credentials_file=None,
            quota_project_id="octopus",
            default_scopes=("https://www.googleapis.com/auth/cloud-platform",),
            scopes=["1", "2"],
            default_host="iam.googleapis.com",
            ssl_credentials=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )


@pytest.mark.parametrize(
    "transport_class", [transports.IAMGrpcTransport, transports.IAMGrpcAsyncIOTransport]
)
def test_iam_grpc_transport_client_cert_source_for_mtls(transport_class):
    cred = ga_credentials.AnonymousCredentials()

    # Check ssl_channel_credentials is used if provided.
    with mock.patch.object(transport_class, "create_channel") as mock_create_channel:
        mock_ssl_channel_creds = mock.Mock()
        transport_class(
            host="squid.clam.whelk",
            credentials=cred,
            ssl_channel_credentials=mock_ssl_channel_creds,
        )
        mock_create_channel.assert_called_once_with(
            "squid.clam.whelk:443",
            credentials=cred,
            credentials_file=None,
            scopes=None,
            ssl_credentials=mock_ssl_channel_creds,
            quota_project_id=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )

    # Check if ssl_channel_credentials is not provided, then client_cert_source_for_mtls
    # is used.
    with mock.patch.object(transport_class, "create_channel", return_value=mock.Mock()):
        with mock.patch("grpc.ssl_channel_credentials") as mock_ssl_cred:
            transport_class(
                credentials=cred,
                client_cert_source_for_mtls=client_cert_source_callback,
            )
            expected_cert, expected_key = client_cert_source_callback()
            mock_ssl_cred.assert_called_once_with(
                certificate_chain=expected_cert, private_key=expected_key
            )


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
        "grpc_asyncio",
    ],
)
def test_iam_host_no_port(transport_name):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        client_options=client_options.ClientOptions(api_endpoint="iam.googleapis.com"),
        transport=transport_name,
    )
    assert client.transport._host == ("iam.googleapis.com:443")


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
        "grpc_asyncio",
    ],
)
def test_iam_host_with_port(transport_name):
    client = IAMClient(
        credentials=ga_credentials.AnonymousCredentials(),
        client_options=client_options.ClientOptions(
            api_endpoint="iam.googleapis.com:8000"
        ),
        transport=transport_name,
    )
    assert client.transport._host == ("iam.googleapis.com:8000")


def test_iam_grpc_transport_channel():
    channel = grpc.secure_channel("http://localhost/", grpc.local_channel_credentials())

    # Check that channel is used if provided.
    transport = transports.IAMGrpcTransport(
        host="squid.clam.whelk",
        channel=channel,
    )
    assert transport.grpc_channel == channel
    assert transport._host == "squid.clam.whelk:443"
    assert transport._ssl_channel_credentials == None


def test_iam_grpc_asyncio_transport_channel():
    channel = aio.secure_channel("http://localhost/", grpc.local_channel_credentials())

    # Check that channel is used if provided.
    transport = transports.IAMGrpcAsyncIOTransport(
        host="squid.clam.whelk",
        channel=channel,
    )
    assert transport.grpc_channel == channel
    assert transport._host == "squid.clam.whelk:443"
    assert transport._ssl_channel_credentials == None


# Remove this test when deprecated arguments (api_mtls_endpoint, client_cert_source) are
# removed from grpc/grpc_asyncio transport constructor.
@pytest.mark.parametrize(
    "transport_class", [transports.IAMGrpcTransport, transports.IAMGrpcAsyncIOTransport]
)
def test_iam_transport_channel_mtls_with_client_cert_source(transport_class):
    with mock.patch(
        "grpc.ssl_channel_credentials", autospec=True
    ) as grpc_ssl_channel_cred:
        with mock.patch.object(
            transport_class, "create_channel"
        ) as grpc_create_channel:
            mock_ssl_cred = mock.Mock()
            grpc_ssl_channel_cred.return_value = mock_ssl_cred

            mock_grpc_channel = mock.Mock()
            grpc_create_channel.return_value = mock_grpc_channel

            cred = ga_credentials.AnonymousCredentials()
            with pytest.warns(DeprecationWarning):
                with mock.patch.object(google.auth, "default") as adc:
                    adc.return_value = (cred, None)
                    transport = transport_class(
                        host="squid.clam.whelk",
                        api_mtls_endpoint="mtls.squid.clam.whelk",
                        client_cert_source=client_cert_source_callback,
                    )
                    adc.assert_called_once()

            grpc_ssl_channel_cred.assert_called_once_with(
                certificate_chain=b"cert bytes", private_key=b"key bytes"
            )
            grpc_create_channel.assert_called_once_with(
                "mtls.squid.clam.whelk:443",
                credentials=cred,
                credentials_file=None,
                scopes=None,
                ssl_credentials=mock_ssl_cred,
                quota_project_id=None,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )
            assert transport.grpc_channel == mock_grpc_channel
            assert transport._ssl_channel_credentials == mock_ssl_cred


# Remove this test when deprecated arguments (api_mtls_endpoint, client_cert_source) are
# removed from grpc/grpc_asyncio transport constructor.
@pytest.mark.parametrize(
    "transport_class", [transports.IAMGrpcTransport, transports.IAMGrpcAsyncIOTransport]
)
def test_iam_transport_channel_mtls_with_adc(transport_class):
    mock_ssl_cred = mock.Mock()
    with mock.patch.multiple(
        "google.auth.transport.grpc.SslCredentials",
        __init__=mock.Mock(return_value=None),
        ssl_credentials=mock.PropertyMock(return_value=mock_ssl_cred),
    ):
        with mock.patch.object(
            transport_class, "create_channel"
        ) as grpc_create_channel:
            mock_grpc_channel = mock.Mock()
            grpc_create_channel.return_value = mock_grpc_channel
            mock_cred = mock.Mock()

            with pytest.warns(DeprecationWarning):
                transport = transport_class(
                    host="squid.clam.whelk",
                    credentials=mock_cred,
                    api_mtls_endpoint="mtls.squid.clam.whelk",
                    client_cert_source=None,
                )

            grpc_create_channel.assert_called_once_with(
                "mtls.squid.clam.whelk:443",
                credentials=mock_cred,
                credentials_file=None,
                scopes=None,
                ssl_credentials=mock_ssl_cred,
                quota_project_id=None,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )
            assert transport.grpc_channel == mock_grpc_channel


def test_key_path():
    project = "squid"
    service_account = "clam"
    key = "whelk"
    expected = "projects/{project}/serviceAccounts/{service_account}/keys/{key}".format(
        project=project,
        service_account=service_account,
        key=key,
    )
    actual = IAMClient.key_path(project, service_account, key)
    assert expected == actual


def test_parse_key_path():
    expected = {
        "project": "octopus",
        "service_account": "oyster",
        "key": "nudibranch",
    }
    path = IAMClient.key_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_key_path(path)
    assert expected == actual


def test_service_account_path():
    project = "cuttlefish"
    service_account = "mussel"
    expected = "projects/{project}/serviceAccounts/{service_account}".format(
        project=project,
        service_account=service_account,
    )
    actual = IAMClient.service_account_path(project, service_account)
    assert expected == actual


def test_parse_service_account_path():
    expected = {
        "project": "winkle",
        "service_account": "nautilus",
    }
    path = IAMClient.service_account_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_service_account_path(path)
    assert expected == actual


def test_common_billing_account_path():
    billing_account = "scallop"
    expected = "billingAccounts/{billing_account}".format(
        billing_account=billing_account,
    )
    actual = IAMClient.common_billing_account_path(billing_account)
    assert expected == actual


def test_parse_common_billing_account_path():
    expected = {
        "billing_account": "abalone",
    }
    path = IAMClient.common_billing_account_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_common_billing_account_path(path)
    assert expected == actual


def test_common_folder_path():
    folder = "squid"
    expected = "folders/{folder}".format(
        folder=folder,
    )
    actual = IAMClient.common_folder_path(folder)
    assert expected == actual


def test_parse_common_folder_path():
    expected = {
        "folder": "clam",
    }
    path = IAMClient.common_folder_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_common_folder_path(path)
    assert expected == actual


def test_common_organization_path():
    organization = "whelk"
    expected = "organizations/{organization}".format(
        organization=organization,
    )
    actual = IAMClient.common_organization_path(organization)
    assert expected == actual


def test_parse_common_organization_path():
    expected = {
        "organization": "octopus",
    }
    path = IAMClient.common_organization_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_common_organization_path(path)
    assert expected == actual


def test_common_project_path():
    project = "oyster"
    expected = "projects/{project}".format(
        project=project,
    )
    actual = IAMClient.common_project_path(project)
    assert expected == actual


def test_parse_common_project_path():
    expected = {
        "project": "nudibranch",
    }
    path = IAMClient.common_project_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_common_project_path(path)
    assert expected == actual


def test_common_location_path():
    project = "cuttlefish"
    location = "mussel"
    expected = "projects/{project}/locations/{location}".format(
        project=project,
        location=location,
    )
    actual = IAMClient.common_location_path(project, location)
    assert expected == actual


def test_parse_common_location_path():
    expected = {
        "project": "winkle",
        "location": "nautilus",
    }
    path = IAMClient.common_location_path(**expected)

    # Check that the path construction is reversible.
    actual = IAMClient.parse_common_location_path(path)
    assert expected == actual


def test_client_with_default_client_info():
    client_info = gapic_v1.client_info.ClientInfo()

    with mock.patch.object(transports.IAMTransport, "_prep_wrapped_messages") as prep:
        client = IAMClient(
            credentials=ga_credentials.AnonymousCredentials(),
            client_info=client_info,
        )
        prep.assert_called_once_with(client_info)

    with mock.patch.object(transports.IAMTransport, "_prep_wrapped_messages") as prep:
        transport_class = IAMClient.get_transport_class()
        transport = transport_class(
            credentials=ga_credentials.AnonymousCredentials(),
            client_info=client_info,
        )
        prep.assert_called_once_with(client_info)


@pytest.mark.asyncio
async def test_transport_close_async():
    client = IAMAsyncClient(
        credentials=ga_credentials.AnonymousCredentials(),
        transport="grpc_asyncio",
    )
    with mock.patch.object(
        type(getattr(client.transport, "grpc_channel")), "close"
    ) as close:
        async with client:
            close.assert_not_called()
        close.assert_called_once()


def test_transport_close():
    transports = {
        "grpc": "_grpc_channel",
    }

    for transport, close_name in transports.items():
        client = IAMClient(
            credentials=ga_credentials.AnonymousCredentials(), transport=transport
        )
        with mock.patch.object(
            type(getattr(client.transport, close_name)), "close"
        ) as close:
            with client:
                close.assert_not_called()
            close.assert_called_once()


def test_client_ctx():
    transports = [
        "grpc",
    ]
    for transport in transports:
        client = IAMClient(
            credentials=ga_credentials.AnonymousCredentials(), transport=transport
        )
        # Test client calls underlying transport.
        with mock.patch.object(type(client.transport), "close") as close:
            close.assert_not_called()
            with client:
                pass
            close.assert_called()


@pytest.mark.parametrize(
    "client_class,transport_class",
    [
        (IAMClient, transports.IAMGrpcTransport),
        (IAMAsyncClient, transports.IAMGrpcAsyncIOTransport),
    ],
)
def test_api_key_credentials(client_class, transport_class):
    with mock.patch.object(
        google.auth._default, "get_api_key_credentials", create=True
    ) as get_api_key_credentials:
        mock_cred = mock.Mock()
        get_api_key_credentials.return_value = mock_cred
        options = client_options.ClientOptions()
        options.api_key = "api_key"
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(client_options=options)
            patched.assert_called_once_with(
                credentials=mock_cred,
                credentials_file=None,
                host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                ),
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )
